(function(){
'use strict';

const app = angular.module('dataiku.shaker');

app.component('partitionSummary', {
    templateUrl: "/templates/datasets/fragments/partition-summary.html",
    bindings: {
        selection: "<",
        partitioned: "<",
        projectKey: "<",
        datasetName: "<",
    }
});

app.service("ElasticSearchQueryUtils", function() {
    return {
        escapeColumn: (columnName) => {
            // Taken from https://www.elastic.co/guide/en/elasticsearch/reference/current/query-dsl-query-string-query.html#_reserved_characters
            // Selected text is quoted, so only '"' is escaped. Special character in field names need to be escaped
            return columnName.replace(/[&|=-\s+<>!^\\~*?:()[\]{}"]/g, m => `\\${m}`);
        },
        getShakerHeaderOptions: () => {
            return {
                showName: true,
                showStorageType: false,
                showMeaning: false,
                showProgressBar: false,
                showHeaderSeparator: true,
            };
        },
        setShakerCommonParameters: (scope) => {
            scope.table = null;
            scope.scriptId = "__pristine__";
            scope.shakerWithSteps = false;
            scope.shakerReadOnlyActions = true;
            scope.searchableDataset = true;
            scope.shakerWritable = false;
            scope.setPageHeightWidthFromChunk = true;
            scope.highlightTagName = "MARK";
        }
    };
});


app.component('savedQuery', {
    templateUrl: "/templates/datasets/fragments/saved-es-query.html",
    bindings: {
        searchQuery: "<",
        projectKey: "<",
        inputDatasetSmartName: "<",
        partitioned: "<",
        replaceQuery: "<",
        canWriteProjectContent: "<",
        inDashboard: "<"
    },
    require: {
        apiErrorContext: "^apiErrorContext"
    },
    controller: function($scope, DataikuAPI, CreateModalFromTemplate, Dialogs) {
        const ctrl = this;
        ctrl.keys = Object.keys;

        ctrl.$onInit = () => {
            ctrl.refreshList();
            ctrl.selectedSavedQuery = null;
        };

        ctrl.$doCheck = () => {
            if (ctrl.selectedSavedQuery !== null && ctrl.searchQuery && !angular.equals(ctrl.selectedSavedQuery.query, ctrl.searchQuery)) {
                ctrl.selectedSavedQuery = null;  // reset selected query if search was modified
            }
        }

        ctrl.filteredSavedQueries = () => {
            if (ctrl.selectedSavedQuery === null) {
                return ctrl.savedQueries;
            }
            return ctrl.savedQueries.filter(q => q.name !== ctrl.selectedSavedQuery.name);
        }

        ctrl.refreshList = () => {
            DataikuAPI.explores.getInteractiveSearchSavedQueries(ctrl.projectKey, ctrl.inputDatasetSmartName)
            .then(({data}) => {
                ctrl.savedQueries = data;
            })
            .catch(ctrl.apiErrorContext.setError);
        }

        ctrl.deleteQuery = (savedQuery, clearSelected) => {
            Dialogs.confirmInfoMessages($scope, 'Confirm query deletion', null, `Are you sure you want to delete query '${sanitize(savedQuery.name)}'?`, false).then(function() {
                DataikuAPI.explores.deleteInteractiveSearchSavedQuery(ctrl.projectKey, ctrl.inputDatasetSmartName, savedQuery.name)
                    .then(() => {
                        ctrl.refreshList();
                        // Unselect selected query if it was the one deleted
                        if (ctrl.selectedSavedQuery !== null && savedQuery.name === ctrl.selectedSavedQuery.name) {
                            ctrl.selectedSavedQuery = null;
                        }
                    })
                    .catch(ctrl.apiErrorContext.setError);
            });
        };

        ctrl.replaceAndUpdateQuery = (savedQuery) => {
            ctrl.selectedSavedQuery = savedQuery;
            ctrl.replaceQuery(angular.copy(savedQuery.query));  // copy here to prevent modifying the saved query
        }

        ctrl.openSaveQueryModal = (queryToEdit) => {
            CreateModalFromTemplate("/templates/datasets/fragments/save-es-query-modal.html", $scope, null, function(newScope) {
                let oldQueryName = null;
                if (queryToEdit) {
                    newScope.modalTitle = "Edit query";
                    newScope.data = angular.copy(ctrl.selectedSavedQuery);
                    oldQueryName = queryToEdit.name;
                } else {
                    newScope.modalTitle = "Save query";
                    newScope.data = {name: "", description: "", query: angular.copy(ctrl.searchQuery)};
                }
                newScope.partitioned = ctrl.partitioned;

                newScope.isQueryNameUnique = function (v) {
                    if (v === null){
                        return true;
                    }
                    for (const query of ctrl.savedQueries) {
                        if (query.name === v && oldQueryName !== query.name) {
                            return false;
                        }
                    }
                    return true;
                };

                newScope.saveQuery = function() {
                    DataikuAPI.explores.saveInteractiveSearchQuery(ctrl.projectKey, ctrl.inputDatasetSmartName, newScope.data, oldQueryName)
                    .then(() => {
                        newScope.dismiss();
                        ctrl.refreshList();
                    }).then(() => {
                        ctrl.selectedSavedQuery = newScope.data;
                    })
                    .catch(setErrorInScope.bind(newScope));
                }
            });
        }
    }
});


app.controller("_ShakerExploreCommonElasticController", function($scope, ElasticSearchQueryUtils, Assert) {
    Assert.inScope($scope, 'shakerHooks');
    $scope.shakerHooks.setColumnMeaning = function(column, newMeaning){
    };

    $scope.shakerHooks.getSetColumnStorageTypeImpact = function(column, newType){
        return null;
    };

    $scope.shakerHooks.setColumnStorageType = function(column, newType, actions){
    };

    $scope.shakerHooks.updateColumnDetails = function(column) {
    };

    /* ********************* Main ******************* */

    // Set base context and call baseInit
    ElasticSearchQueryUtils.setShakerCommonParameters($scope);
});


app.controller("_InteractiveSearchController", function($scope, $stateParams, MonoFuture, WT1, ElasticSearchQueryUtils) {
    $scope.interactiveSearchQuery = {
        queryString: $stateParams.queryString || "",
        datasetSelection: { selectedPartitions: [], latestPartitionsN: 1, partitionSelectionMethod: "ALL" }
    };

    $scope.isDirty = function() {
        return !angular.equals($scope.previousInteractiveSearchQuery, $scope.interactiveSearchQuery);
    }

    $scope.resetSearch = function() {
        $scope.interactiveSearchQuery.queryString = "";
        $scope.searchInteractive()
    }

    $scope.replaceQuery = function(query) {
        $scope.interactiveSearchQuery = query;
        $scope.searchInteractive()
    }

    // If editing interactively from a recipe, consider we have input a query and enable search
    $scope.previousInteractiveSearchQuery = $stateParams.queryString ? {} : angular.copy($scope.interactiveSearchQuery);

    $scope.searchInteractive = function(wt1Origin) {
        $scope.previousInteractiveSearchQuery = angular.copy($scope.interactiveSearchQuery);

        $scope.refreshTable(true);
        if (wt1Origin) {
            WT1.event($scope.searchWT1EventName, {origin: wt1Origin});
        }
    }

    const quoteContent = function(string) {
        // Content is quoted to prevent problems with special characters, so we need to escape '"' in the text
        return '"' + string.replace(/"/g, "\\\"") + '"';
    }

    const searchInteractiveFilterNoEscapeContent = function(columnName, content, wt1Origin) {
        columnName = ElasticSearchQueryUtils.escapeColumn(columnName);
        if ($scope.interactiveSearchQuery.queryString) {
            $scope.interactiveSearchQuery.queryString = "(" + $scope.interactiveSearchQuery.queryString + ") AND " + columnName + ":"  + content;
        } else {
            $scope.interactiveSearchQuery.queryString = columnName + ":" + content
        }
        $scope.searchInteractive(wt1Origin);
    }

    $scope.searchInteractiveFilter = function(columnName, content) {
        content = quoteContent(content);
        searchInteractiveFilterNoEscapeContent(columnName, content, "selection");
    }

    // Filtering
    $scope.addDateRangeToInteractiveFilter = function(columnName, fromDate, toDate) {
        let content;
        if (!fromDate && !toDate) {
            return;
        }
        if (fromDate && !toDate) {
            content = `[${fromDate.toISOString()} TO *]`;
        } else if (!fromDate && toDate) {
            content = `[* TO ${toDate.toISOString()}]`;
        } else {
            content = `[${fromDate.toISOString()} TO ${toDate.toISOString()}]`;
        }

        searchInteractiveFilterNoEscapeContent(columnName, content, "date-filter");
    }

    $scope.appendTextValuesToInteractiveFilter = function (columnName, textValues) {
        textValues = textValues.filter(c => c.length > 0);
        if (textValues.length == 0) {
            return;
        }

        textValues = textValues.map(quoteContent);
        let content = textValues.join(" OR ");
        if (textValues.length > 1) {
            content = "(" + content + ")";
        }
        searchInteractiveFilterNoEscapeContent(columnName, content, "text-filter");
    }

    // Sorting
    // override default sortDirection from the shaker to easily stay compatible
    $scope.sortDirection = function(column) {
        if (!$scope.interactiveSearchQuery.sortParameters || $scope.interactiveSearchQuery.sortParameters[0].column !== column) {
            return null;
        }
        return $scope.interactiveSearchQuery.sortParameters[0].order === "asc";
    }

    $scope.isColumnTypeSortable = function(type) {
        return ["date", "dateonly", "datetimenotz", "tinyint", "smallint", "int", "bigint", "float", "double", "long", "boolean"].includes(type);
    }

    // override default addSort from the shaker to easily stay compatible
    $scope.addSort = function(column) {
        if (!$scope.interactiveSearchQuery.sortParameters || $scope.interactiveSearchQuery.sortParameters[0].column !== column) {
            $scope.interactiveSearchQuery.sortParameters = [{ "column": column, "order": "asc"}];
        } else if ($scope.interactiveSearchQuery.sortParameters[0].order === "asc") {
            $scope.interactiveSearchQuery.sortParameters[0].order = "desc";
        } else {
            $scope.interactiveSearchQuery.sortParameters[0].order = "asc";
        }
        $scope.searchInteractive();
    }

    $scope.clearSortInteractive = () => {
        $scope.interactiveSearchQuery.sortParameters = null;
        $scope.searchInteractive();
    }

    $scope.checkQueryInput = (event) => {
        if (event.originalEvent.code === "Enter") {
            event.preventDefault();
            event.target.blur();
            $scope.searchInteractive("click");
        }
    }

    $scope.resetTextareaSize = (textareaInput) => {
        if (!textareaInput) {
            textareaInput = document.getElementById("interactive-search-input");
        }
        textareaInput.style.height = "";
        textareaInput.style.overflowY = "hidden";  // Do not display scroll bar for a single line
    }

    let textareaInputOriginalHeight = null;
    let textareaInput = null;
    $scope.checkTextareaSize = () => {
        if (textareaInputOriginalHeight === null) {
            textareaInput = document.getElementById("interactive-search-input");
            textareaInputOriginalHeight = textareaInput.scrollHeight;
        }
        textareaInput.style.overflowY = "hidden";  // Do not display scroll bar for a single line
        textareaInput.style.height = "1px";
        if (textareaInput.scrollHeight > textareaInputOriginalHeight * 1.5) {  // More than one line of query
            textareaInput.style.height = Math.min(textareaInput.scrollHeight, 100) + "px";  // Recompute height based on input
            textareaInput.style.overflowY = "auto";
        } else {
            $scope.resetTextareaSize(textareaInput);
        }
    }

    const monoFuture = MonoFuture($scope);

    $scope.monoFuturizedSearch = monoFuture.wrap($scope.searchAPI);

    $scope.shakerState.onDataset = true;

    $scope.shakerHooks.isMonoFuturizedRefreshActive = monoFuture.active;

    $scope.shakerHooks.shakerForQuery = function(){
        const queryObj = angular.copy($scope.shaker);
        if ($scope.isRecipe) {
            queryObj.recipeSchema = $scope.recipeOutputSchema;
        }
        queryObj.contextProjectKey = $stateParams.projectKey; // quick 'n' dirty, but there are too many call to bother passing the projectKey through them
        return queryObj;
    }

    $scope.shakerHooks.updateColumnWidth = function(name, width) {
        $scope.shaker.columnWidthsByName[name] = width;
        $scope.autoSaveAutoRefresh();
    };

    $scope.getQuerySyntaxError = function() {
        if (!$scope.table || $scope.table.warnings.totalCount == 0 || !('INPUT_ELASTICSEARCH_BAD_QUERY' in $scope.table.warnings.warnings)) {
            return null;
        }
        // Only return first message
        return $scope.table.warnings.warnings['INPUT_ELASTICSEARCH_BAD_QUERY']['stored'][0]['message'];
    }

    $scope.shakerHooks.fetchDetailedAnalysis = function(setAnalysis, handleError, columnName, alphanumMaxResults, fullSamplePartitionId, withFullSampleStatistics) {
        // Do nothing
    };

    $scope.shakerHooks.fetchClusters = function(setClusters, columnName, setBased, radius, timeOut, blockSize) {
        // Do nothing
    };

    $scope.shakerHooks.fetchTextAnalysis = function(setTextAnalysis, columnName, textSettings) {
        // Do nothing
    };

    $scope.toSuccessAPI = ({result}) => {
        $scope.table.warnings = result.warnings; // bubble up refresh warnings
        return {data: result}; // mimic http response, in a data field;
    };
});


app.directive("shakerOnDatasetElastic", function() {
    return {
        scope: true,
        controller: function ($scope, $stateParams, DataikuAPI, FutureWatcher, $controller, $q, WT1, TopNav, DatasetErrorCta, DatasetUtils, ElasticSearchQueryUtils, SmartId) {
            $scope.searchAPI = DataikuAPI.shakers.searchElasticDataset;
            $controller("_InteractiveSearchController", {$scope: $scope});

            const batchSize = 128;  // number of rows to fetch in a single "refresh" call
            $scope.shakerHooks.getRefreshTablePromise = function(filtersOnly, filterRequest) {
                const ret = $scope.monoFuturizedSearch($stateParams.projectKey, $scope.inputDatasetProjectKey, $scope.inputDatasetName,
                        $scope.shakerHooks.shakerForQuery(), $scope.interactiveSearchQuery, 0, batchSize, batchSize);
                return $scope.refreshNoSpinner ? ret.noSpinner() : ret;
            };

            $scope.shakerHooks.getTableChunk = function(firstRow, nbRows, firstCol, nbCols, filterRequest) {
                return DataikuAPI.shakers.searchElasticDataset($stateParams.projectKey, $scope.inputDatasetProjectKey, $scope.inputDatasetName,
                                                       $scope.shakerHooks.shakerForQuery(), $scope.interactiveSearchQuery,
                                                       firstRow, $scope.table.totalKeptRows, batchSize)
                .then(({data}) => {
                    if (data.hasResult) {
                        return $scope.toSuccessAPI(data);
                    }
                    return FutureWatcher.watchJobId(data.jobId).then(({data}) => $scope.toSuccessAPI(data));
                });
            }


            $controller("_ShakerExploreCommonElasticController", {$scope: $scope});
            $scope.projectKey = $stateParams.projectKey;

            if ($scope.insight) { // On a dashboard
                /* ********************* Callbacks for shakerExploreBase ******************* */
                $scope.shakerHooks.saveForAuto = function() {
                    return $q((resolve, reject) => {
                        try {
                            $scope.insight.params.shakerScript = $scope.getShakerData();
                            if ($scope.hook && $scope.hook.onSave) {
                                $scope.hook.onSave($scope.insight.params);
                            }

                            resolve();
                        } catch (error) {
                            reject(error);
                        }
                    });
                };

                $scope.setSpinnerPosition = function() {}

                /* ********************* Main ******************* */

                // Set base context and call baseInit

                WT1.event("dashboard-dataset-search-open");
                $scope.searchWT1EventName = "dashboard-dataset-search-action-search";
                $scope.refreshNoSpinner = false;  // Display spinner for initial search

                const resolvedDataset = SmartId.resolve($scope.insight.params.datasetSmartName);
                $scope.inputDatasetProjectKey = resolvedDataset.projectKey;
                $scope.inputDatasetName = resolvedDataset.id;
                $scope.inputDatasetSmartName = $scope.insight.params.datasetSmartName;


                $scope.shaker = $scope.insight.params.shakerScript;
                $scope.shaker.origin = "DATASET_EXPLORE";
                if ($scope.origInsight) {
                    $scope.origInsight.params.shakerScript.origin = "DATASET_EXPLORE";
                }

                $scope.shakerState.writeAccess = true;
                if ($scope.insight.params.shakerState) {
                    Object.entries($scope.insight.params.shakerState).forEach(([key, value]) => $scope.shakerState[key] = value);
                }

                $scope.shaker.$headerOptions = ElasticSearchQueryUtils.getShakerHeaderOptions();


                $scope.shaker.coloring.highlightSearchMatches = true;

                $scope.fixupShaker();
                if ($scope.origInsight) {
                    $scope.fixupShaker($scope.origInsight.params.shakerScript);
                }

                $scope.insight.$tileFilteringInformation = {};

                $scope.$watch('shakerState.activeView', (nv, ov) => {
                    if (nv === 'table' && ov) {
                        $scope.refreshTable(false);
                    }
                });
            } else { // On a dataset
                /* ********************* Callbacks for shakerExploreBase ******************* */
                $scope.shakerHooks.saveForAuto = function() {
                    var deferred = $q.defer();
                    resetErrorInScope($scope);
                    var shakerData = $scope.getShakerData();

                    if ($scope.isRecipe) {
                        throw "Should not call this for a recipe";
                    } else {
                        DataikuAPI.explores.saveScript($stateParams.projectKey, $scope.inputDatasetSmartName,
                            shakerData).success(function(data){
                            $scope.originalShaker = shakerData;
                            deferred.resolve();
                        }).error(setErrorInScope.bind($scope));
                    }
                    return deferred.promise;
                };


                TopNav.setLocation(TopNav.TOP_FLOW, 'datasets', TopNav.TABS_DATASET, "search");

                if ($stateParams.datasetFullName) {  // On a foreign (exposed) dataset
                    const loc = DatasetUtils.getLocFromFull($stateParams.datasetFullName);
                    $scope.inputDatasetProjectKey = loc.projectKey;
                    $scope.inputDatasetName = loc.name;
                    $scope.inputDatasetSmartName = $stateParams.datasetFullName;
                } else {  // dataset local to project
                    $scope.inputDatasetProjectKey = $stateParams.projectKey;
                    $scope.inputDatasetName = $stateParams.datasetName;
                    $scope.inputDatasetSmartName = $stateParams.datasetName;
                }

                WT1.event("dataset-search-open");
                $scope.searchWT1EventName = "dataset-search-action-search";

                //For datasetErrorCTA directive (CTA in case of error while loading dataset sample)
                $scope.updateUiState = DatasetErrorCta.getupdateUiStateFunc($scope);
                $scope.$watch("datasetFullInfo", _ => $scope.updateUiState($scope.shakerState.runError), true);
                $scope.$watch("shakerState", _ => $scope.updateUiState($scope.shakerState.runError), true);
                $scope.$watch("table", _ => $scope.updateUiState($scope.shakerState.runError));


                // Load shaker, set the necessary stuff in scope and call the initial refresh
                DataikuAPI.explores.getScript($stateParams.projectKey, $scope.inputDatasetSmartName).success(function(shaker) {
                    $scope.shaker = shaker;
                    $scope.shaker.$headerOptions = ElasticSearchQueryUtils.getShakerHeaderOptions();

                    $scope.shaker.coloring.highlightSearchMatches = true;
                    $scope.shaker.origin = "DATASET_EXPLORE";
                    $scope.fixupShaker();
                    $scope.requestedSampleId = null;
                    $scope.refreshTable(false);
                }).error(setErrorInScope.bind($scope));
            }
        }
    }
});

app.directive("shakerOnSearchNotebook", function() {
    return {
        scope: true,
        controller: function ($scope, $stateParams, DataikuAPI, FutureWatcher, $controller, ElasticSearchQueryUtils, DatasetErrorCta, WT1, $q, $rootScope) {
            $scope.searchAPI = DataikuAPI.searchNotebooks.search;
            $controller("_InteractiveSearchController", {$scope: $scope});

            const batchSize = 128;  // number of rows to fetch in a single "refresh" call
            $scope.shakerHooks.getRefreshTablePromise = function(filtersOnly, filterRequest) {
                const index = $scope.notebookCell.$resolvedIndex;
                $scope.notebookCell.query = $scope.interactiveSearchQuery.queryString;
                $scope.notebookCell.lastExecutionTimestamp = Date.now();
                const ret = $scope.monoFuturizedSearch($stateParams.projectKey, $scope.notebookConnection, index, $scope.searchNotebookSchema,
                                                       $scope.interactiveSearchQuery, 0, batchSize, batchSize)
                return $scope.refreshNoSpinner ? ret.noSpinner() : ret;
            };

            $scope.shakerHooks.getTableChunk = function(firstRow, nbRows, firstCol, nbCols, filterRequest) {
                const index = $scope.notebookCell.$resolvedIndex;
                return DataikuAPI.searchNotebooks.search($stateParams.projectKey, $scope.notebookConnection, index, $scope.searchNotebookSchema,
                                                         $scope.interactiveSearchQuery, firstRow, $scope.table.totalKeptRows, batchSize)
                .then(({data}) => {
                    if (data.hasResult) {
                        return $scope.toSuccessAPI(data);
                    }
                    return FutureWatcher.watchJobId(data.jobId).then(({data}) => $scope.toSuccessAPI(data));
                });
            }

            $scope.$watch("selectedField", (nv, ov) => {
                if (!nv || !nv.name) {
                    return;
                }
                const columnName = ElasticSearchQueryUtils.escapeColumn(nv.name);

                if ($scope.interactiveSearchQuery.queryString) {
                    $scope.interactiveSearchQuery.queryString = "(" + $scope.interactiveSearchQuery.queryString + ") AND " + columnName + ":";
                } else {
                    $scope.interactiveSearchQuery.queryString = columnName + ":";
                }
            });

            $scope.$watch("notebookCell.searchScope", (nv, ov) => {
                if (!nv) {
                    return;
                }
                $scope.interactiveSearchQuery.queryString = $scope.notebookCell.query || "";
                $scope.searchInteractive("load-notebook-cell");
            });



            $controller("_ShakerExploreCommonElasticController", {$scope: $scope});
            $scope.setSpinnerPosition = function(position){
                $rootScope.spinnerPosition = position;
            };

            $scope.shakerHooks.saveForAuto = function() {
                var deferred = $q.defer();
                resetErrorInScope($scope);
                deferred.resolve();
                return deferred.promise;
            };

            $scope.projectKey = $stateParams.projectKey;

            WT1.event("search-notebook-open");
            $scope.searchWT1EventName = "search-notebook-action-search";

            //For datasetErrorCTA directive (CTA in case of error while loading dataset sample)

            $scope.updateUiState = DatasetErrorCta.getupdateUiStateFunc($scope);
            $scope.$watch("shakerState", _ => $scope.updateUiState($scope.shakerState.runError), true);
            $scope.$watch("table", _ => $scope.updateUiState($scope.shakerState.runError));


            // Load shaker, set the necessary stuff in scope and call the initial refresh
            $scope.shaker = {
                steps: [],
                coloring: {}
            };

            $scope.shaker.$headerOptions = ElasticSearchQueryUtils.getShakerHeaderOptions();

            $scope.shaker.$isColumnMenuDisabled = (columnName) => {
                return columnName === "__index"; // added on search notebooks
            }

            $scope.shaker.coloring.highlightSearchMatches = true;
            $scope.shaker.origin = "SEARCH_NOTEBOOK";
            $scope.fixupShaker();
            $scope.requestedSampleId = null;
            $scope.refreshTable(false);
        }
    }
});

}());
