(function() {
    'use strict';

    angular.module('dataiku.charts')
        .directive('continuousColorLegend', continuousColorLegend);

    /**
     * (!) This directive previously was in static/dataiku/js/simple_report/common/legends.js
     */
    function continuousColorLegend(Fn, D3ChartAxes) {
        return {
            scope: true,
            templateUrl: '/static/dataiku/js/simple_report/directives/legends/continuous-color-legend/continuous-color-legend.directive.html',
            link: function($scope, element, attrs) {

                const placement = element.closest('.pivot-charts').attr('legend-placement');

                $scope.gradientError = false;
                $scope.$watch(attrs.legend, function(nv, ov) {
                    $scope.draw($scope.$eval(attrs.legend));
                });

                const svg = d3.select(element[0]).select('svg'),
                    $svg = element.find('svg'),
                    gradient = svg.select('linearGradient');

                let vertical, orient,
                    barWidth = Math.max(0, $svg.width() - 10),
                    barHeight = Math.max(0, $svg.height() - 20),
                    axisX = 5,
                    axisY = 10,
                    rectX = 0;
                switch (placement) {
                    case 'OUTER_RIGHT':
                        vertical = true;
                        barWidth = 15;
                        axisX = 15;
                        orient = 'right';
                        break;
                    case 'OUTER_LEFT':
                        vertical = true;
                        barWidth = 15;
                        orient = 'left';
                        break;
                    case 'OUTER_TOP':
                    case 'OUTER_BOTTOM':
                    default: // sidebar or inner
                        vertical = false;
                        $svg.height(45);
                        orient = 'bottom';
                        axisY = 15;
                        barHeight = 15;
                        break;
                }

                if (vertical) {
                    gradient.attr('x2', '0%').attr('y2', '100%');
                }

                svg.select('rect')
                    .attr('width', barWidth)
                    .attr('height', barHeight)
                    .attr('y', vertical ? 10 : 0)
                    .attr('x', vertical ? 0 : 5);

                const setStyle = function(style, ticks) {
                    const fontSize = style && style.fontSize ? `${style.fontSize}px` : '13px';
                    const fontColor = style && style.fontColor ? style.fontColor : 'inherit';
                    ticks.forEach(tick => {
                        tick.style.fontSize = fontSize;
                        tick.style.fill = fontColor;
                    });
                };

                $scope.draw = function(legend) {
                    if (!legend) {
                        if (svg.select('rect').attr('width') == 0) {
                            $scope.gradientError = true;
                        }
                        return;
                    }

                    $scope.gradientError = false;
                    const isCondFormatting = legend.type === 'CONDITIONAL_FORMATTING';

                    if (isCondFormatting) {
                        barWidth = 185;
                        // set width && height here bc legend can be initialized with empty attrs.legend
                        svg.select('rect')
                            .attr('width', barWidth)
                            .attr('height', 10)
                        ;
                    }

                    const axisScale = legend.scale.innerScale.copy();
                    if (legend.scale.diverging) {
                        axisScale.domain([axisScale.invert(0), axisScale.invert(1)]).range([0, 1]);
                    }

                    axisScale.range(axisScale.range().map(x => vertical ? (barHeight - x * barHeight) : x * barWidth)).interpolate(d3.interpolate);
                    const axis = d3.svg.axis().orient(orient).scale(axisScale);

                    // Arbitrary value to convert pixels to tick number
                    const pixelDivisor = 50;
                    axis.tickValues(axisScale.ticks(Math.min(10, vertical ? barHeight / pixelDivisor : barWidth / pixelDivisor)).concat(axisScale.domain()));
                    axis.tickFormat(legend.formatter);
                    const axisG = svg.select('g.axis');
                    axisG.selectAll('*').remove();
                    axisG.call(axis).select('path.domain').remove();

                    const ticks = axisG.selectAll('g.tick')[0];
                    ticks.sort((a, b) => a.__data__ - b.__data__);
                    setStyle($scope.chart ? $scope.chart.def.legendFormatting : {}, ticks);

                    // Anchor to start & end mostleft & mostright labels
                    if (!vertical) {
                        d3.select(ticks[0]).select('text').style('text-anchor', 'start');
                        d3.select(ticks[ticks.length - 1]).select('text').style('text-anchor', 'end');
                    }

                    if (isCondFormatting) {
                        // keep only first and last tick for conditional formatting
                        ticks.length > 1 && ticks.slice(1, -1).forEach(tick => tick.remove());
                        d3.select(ticks[0]).select('line').remove();
                        d3.select(ticks[0]).select('text').text('Min');

                        d3.select(ticks[ticks.length - 1]).select('line').remove();
                        d3.select(ticks[ticks.length - 1]).select('text').text('Max');
                    } else {
                        D3ChartAxes.sanitizeTicksDisplay(ticks);
                    }

                    const colors = legend.scale.outerScale.range();
                    const colorStops = [];
                    const numStops = legend.scale.quantizationMode === 'NONE' ? colors.length - 1 : colors.length;

                    if (legend.scale.quantizationMode !== 'QUANTILES') {
                        colors.forEach(function(c, i) {
                            colorStops.push({
                                color: c,
                                offset: i * 100 / numStops
                            });

                            if (legend.scale.quantizationMode !== 'NONE') {
                                colorStops.push({
                                    color: c,
                                    offset: (i + 1) * (100 / numStops)
                                });
                            }
                        });
                    } else {
                        const thresholds = legend.scale.outerScale.quantiles();
                        colors.forEach(function(c, i) {
                            colorStops.push({
                                color: c,
                                offset: (i === 0 ? 0 : thresholds[i - 1] * 100)
                            });
                            colorStops.push({
                                color: c,
                                offset: (i === colors.length - 1 ? 100 : thresholds[i] * 100)
                            });
                        });
                    }

                    // In the vertical scale, we want the first stop at the bottom
                    if (vertical) {
                        colorStops.forEach(function(stop) {
                            stop.offset = (100 - (stop.offset));
                        });
                        colorStops.reverse();
                    }

                    /*
                     * This was used to display the color palette with a log/square/square root gradient instead of a linear gradient,
                     * but instead we display a linear gradient and let d3 put the ticks at the right places
                     * if (scale.mode == 'LINEAR') {
                     *  points = legend.scale.domain();
                     * } else {
                     *  var NUM_STOPS = 100;
                     *  var range = axisScale.range();
                     *  var step = (domain[domain.length-1] - domain[0])/NUM_STOPS;
                     *  for (var i = 0; i < NUM_STOPS; i++) {
                     *      points.push(domain[0] + step*i);
                     *  }
                     * }
                     */

                    const padding = 5;
                    const margin = 15;
                    gradient[0][0].replaceChildren([]);
                    gradient.selectAll('stop').data(colorStops)
                        .enter().append('stop')
                        .attr('offset', stop => stop.offset + '%')
                        .attr('stop-color', Fn.prop('color'))
                        .attr('stop-opacity', 1);

                    if (vertical) {
                        const maxWidth = d3.max(axisG.selectAll('g.tick')[0].map(function(itm) {
                            return itm.getBoundingClientRect().width;
                        })) || 0;

                        $svg.css('width', maxWidth + margin);
                    } else {
                        const collapsedColorGroupHeight = 22;
                        const defaultHeight = isCondFormatting ? collapsedColorGroupHeight : 0;

                        const maxHeight = d3.max(axisG.selectAll('g.tick')[0].map(function(itm) {
                            return itm.getBoundingClientRect().height;
                        })) || defaultHeight;

                        $svg.css('height', maxHeight + margin + padding);
                        $svg.css('background-color', 'rgb(255, 255, 255, 0.6)');
                    }

                    if (placement == 'OUTER_LEFT') {
                        rectX = $svg.width() - 15;
                        axisX = rectX;
                    }

                    const axisGY = (vertical || isCondFormatting) ? axisY : axisY + padding;
                    const svgY = vertical ? 0 : padding;
                    axisG.attr('transform', 'translate(' + axisX + ',' + axisGY + ')');
                    svg.select('rect').attr('transform', 'translate(' + rectX + ', ' + svgY + ')');
                };
            }
        }
    }

})();
