// @ts-check
(function() {
    'use strict';

    // @ts-ignore
    angular.module('dataiku.directives.simple_report').factory('ChartAxesUtils', chartAxesUtils);

    /**
     * ChartAxesUtils service
     * Utils to compute chart axes (agnostic, used for both d3 and echarts)
     * (!) This service previously was in static/dataiku/js/simple_report/common/axes.js
     * @typedef {import("../../../../../../../../../server/src/frontend/src/app/features/simple-report/services/chart-axis/chart-axis.type").ChartAxesUtilsReturnType} ChartAxesUtilsReturnType
     * @returns {ChartAxesUtilsReturnType}
     */
    function chartAxesUtils(ChartDataUtils, ChartUADimension, ChartDimension, ChartsStaticData, ChartFeatures, ChartYAxisPosition, AxisTicksConfigMode, AxisTicksFormatting, AxisTitleFormatting, ColumnAvailability, translate, CHART_VARIANTS) {

        const AUTO_EXTENT_MODE = 'AUTO';
        const MANUAL_EXTENT_MODE = 'MANUAL';

        function getManualExtentAtIndex(customExtent, index) {
            if (customExtent == null || customExtent.$autoExtent === undefined || (index < 0 && index > 1)) {
                return null;
            } // extent index can only be 0 or 1
            return customExtent.manualExtent[index] === null ? customExtent.$autoExtent[index] : customExtent.manualExtent[index];
        }

        function getManualExtentMin(customExtent) {
            return getManualExtentAtIndex(customExtent, 0);
        }

        function getManualExtentMax(customExtent) {
            return getManualExtentAtIndex(customExtent, 1);
        }

        /**
         * Get manual extent to apply (non-null) from customExtent
         * @param customExtent
         * @returns {[number, number]} [min, max]
         */
        function getManualExtent(customExtent) {
            return [getManualExtentMin(customExtent), getManualExtentMax(customExtent)];
        }

        /**
         * Checks if the user defined custom extent crops the initial extent
         * @param axesFormatting
         * @returns Returns True if initial extent is cropped, False otherwise.
         */
        const isCroppedExtent = axesFormatting => {
            const isCropped = axesFormatting.some(formatting => {
                if (formatting.customExtent.$autoExtent && formatting.customExtent.editMode === MANUAL_EXTENT_MODE) {
                    const manualExtent = getManualExtent(formatting.customExtent);
                    return formatting.customExtent.$autoExtent[0] < manualExtent[0] || formatting.customExtent.$autoExtent[1] > manualExtent[1];
                }
            });
            return isCropped;

        };

        /** @type {ChartAxesUtilsReturnType} */
        const svc = {
            shouldIncludeZero(chartDef, yAxisId) {
                const yAxisFormatting = this.getFormattingForYAxis(chartDef.yAxesFormatting, yAxisId);
                return ChartFeatures.canIncludeZero(chartDef.type) && yAxisFormatting && !svc.isManualMode(yAxisFormatting.customExtent) && yAxisFormatting.includeZero;
            },

            isNumerical(axisSpec) {
                return axisSpec.dimension && axisSpec.dimension.isA === 'ua' ? ChartUADimension.isTrueNumerical(axisSpec.dimension) : ChartDimension.isTrueNumerical(axisSpec.dimension);
            },

            isContinuousDate(axisSpec) {
                return axisSpec.dimension && axisSpec.dimension.isA === 'ua' ? ChartUADimension.isDate(axisSpec.dimension) : ChartDimension.isTimeline(axisSpec.dimension);
            },

            /**
             * Initialize CustomExtent with given initial extent, and compute the extent to apply in return
             * @param customExtent
             * customExtent (for user to define custom y and x axis range), has below parameters:
             *   - editMode: ("AUTO" or "MANUAL"); // the mode to use to define extent (auto computed or manually defined)
             *   - $autoExtent: [minExtent, maxExtent]; // array of 2 floats: initial min and max values auto-detected (used in AUTO editMode)
             *   - manualExtent: [minExtent, maxExtent]; // array of 2 floats or null: user custom min and max values, if null min or max are auto computed (used in MANUAL editMode)
             * @param initialExtent
             * @param isPercentScale
             * @returns new extent [minValue, maxValue] to apply based on user custom extent
             */
            initCustomAxisExtent: function(customExtent, initialExtent, isPercentScale) {
                const coeff = isPercentScale ? 100 : 1; // to format input between [0, 100] when in percentScale
                // @ts-ignore
                if (_.isFinite(initialExtent[0]) && _.isFinite(initialExtent[1])) {
                    customExtent.$autoExtent = [initialExtent[0] * coeff, initialExtent[1] * coeff];
                }

                if (customExtent.editMode === ChartsStaticData.AUTO_EXTENT_MODE) {
                    customExtent.manualExtent = [null, null];
                    return initialExtent;
                } else {
                    const manualExtent = getManualExtent(customExtent);
                    return [manualExtent[0] / coeff, manualExtent[1] / coeff];
                }
            },

            getDimensionExtent(chartData, axisSpec, ignoreLabels) {
                let extent = axisSpec.extent;

                if (!extent) {
                    if (axisSpec.type === 'UNAGGREGATED') {
                        extent = axisSpec.extent || ChartDataUtils.getUnaggregatedAxisExtent(axisSpec.dimension, axisSpec.data);
                    } else {
                        extent = ChartDataUtils.getAxisExtent(chartData, axisSpec.name, axisSpec.dimension, { ignoreLabels, initialExtent: axisSpec.customExtent && axisSpec.customExtent.$autoExtent });
                    }
                }

                if (axisSpec.customExtent) {
                    const transientExtent = svc.initCustomAxisExtent(axisSpec.customExtent, [extent.min, extent.max], axisSpec.isPercentScale);
                    extent = Object.assign(extent, { min: transientExtent[0], max: transientExtent[1] });
                }

                // Override min and max with pre-defined interval if requested.
                if (axisSpec.initialInterval) {
                    extent.min = axisSpec.initialInterval.min;
                    extent.max = axisSpec.initialInterval.max;
                }

                return extent;
            },

            getMeasureExtent(chartData, axisSpec, isLogScale, includeZero, computePercentScale, otherAxesIndexes = []) {
                let extent = axisSpec.extent;

                if (!extent) {
                    if (axisSpec.measureIdx === undefined) {
                        return null;
                    }
                    extent = ChartDataUtils.getMeasureExtent(chartData, axisSpec.measureIdx, true, null, axisSpec.measure[0]?.function, otherAxesIndexes);
                }

                if (extent[0] == Infinity) {
                    return null; // No values -> no axis
                }

                // Adjust extent if needed
                if (axisSpec.customExtent) {
                    extent = svc.initCustomAxisExtent(axisSpec.customExtent, extent, computePercentScale ? axisSpec.isPercentScale : false);
                }

                if (includeZero) {
                    const isZeroInRange = extent[0] > 0 != extent[1] > 0;
                    if (!isZeroInRange) {
                        extent[0] = Math.min(extent[0], 0);
                        extent[1] = Math.max(extent[1], 0);
                    }
                }

                if (isLogScale) {
                    const logScaleExtent = svc.fixNumericalLogScaleExtent(axisSpec, { min: extent[0] }, includeZero);
                    extent[0] = logScaleExtent.min;
                }

                return extent;
            },

            /**
             * A log axis scale cannot contain 0.
             * For user convenience, when includeZero is checked in auto range mode we allow the zero to be replaced by 1 automatically.
             *
             * @param   axisSpec        - Axis specification
             * @param {Boolean}     includeZero     - True to force inclusion of zero in extent
             * @param {Number}      currentMinValue        - Current known minimal value for the axis range.
             */
            getProperMinValueWhenInLogScale(axisSpec, includeZero, currentMinValue) {
                const shouldReplace = !svc.isManualMode(axisSpec.customExtent) && includeZero && currentMinValue === 0;
                return shouldReplace ? 1 : currentMinValue;
            },

            fixNumericalLogScaleExtent(axisSpec, extent, includeZero) {
                extent.min = svc.getProperMinValueWhenInLogScale(axisSpec, includeZero, extent.min);
                if (extent.min <= 0) {
                    // @ts-ignore
                    throw new ChartIAE('Cannot represent value 0 nor negative values on a log scale. Please disable log scale.');
                }
                return extent;
            },

            fixUnbinnedNumericalExtent(axisSpec, extent) {
                /*
                 * If the data is based on range bands AND we are going to use the linear scale
                 * to place the bands, then the linear scale must be refitted to give space for the bands
                 * (this corresponds to the COLUMN charts with 'Use raw values' mode)
                 * Same thing for scatter plot
                 */
                if (ChartDimension.isUngroupedNumerical(axisSpec.dimension) || axisSpec.type === 'UNAGGREGATED') {
                    const nbVals = extent.values.length;
                    const interval = extent.max - extent.min;
                    // Add 10% margin when not many bars, 5% margin else
                    const additionalPct = nbVals > 10 ? 5.0 : 10.0;
                    extent.min = extent.min - (interval * additionalPct) / 100;
                    extent.max = extent.max + (interval * additionalPct) / 100;
                }
                return extent;
            },

            includeZero(extent) {
                extent.min = Math.min(extent.min, 0);
                extent.max = Math.max(extent.max, 0);
                return extent;
            },

            computeYAxisID(orientation = 'left', index = 0, dimension) {
                let id = `y_${orientation}_${index}`;
                if (dimension) {
                    id += `_${dimension}`;
                }
                return id;
            },

            getFormattingForYAxis(yAxesFormatting, id = ChartsStaticData.LEFT_AXIS_ID) {
                return yAxesFormatting.find(v => v.id === id);
            },

            isYAxisLogScale(yAxesFormatting, id = ChartsStaticData.LEFT_AXIS_ID) {
                const formatting = this.getFormattingForYAxis(yAxesFormatting, id);
                return (formatting || {}).isLogScale;
            },

            getYAxisNumberFormatting(yAxesFormatting, id = ChartsStaticData.LEFT_AXIS_ID) {
                const formatting = this.getFormattingForYAxis(yAxesFormatting, id);
                return ((formatting || {}).axisValuesFormatting || {}).numberFormatting;
            },

            getYAxisCustomExtent(yAxesFormatting, id = ChartsStaticData.LEFT_AXIS_ID) {
                const formatting = this.getFormattingForYAxis(yAxesFormatting, id);
                return (formatting || {}).customExtent;
            },

            setYAxisCustomExtent(yAxesFormatting, newCustomExtent, id = ChartsStaticData.LEFT_AXIS_ID) {
                const index = yAxesFormatting.map(x => x.id).indexOf(id);
                yAxesFormatting[index].customExtent = newCustomExtent;
            },

            contatenateMeasuresNames(measures) {
                return measures.map(m => ColumnAvailability.getAggregatedLabel(m)).join(', ');
            },

            /**
             * getXAxisTitle is used to get correct x axis title (it is used by d3 and echarts, so it should stay agnostic)
             * @param xAxisSpec
             * @param chartDef
             */
            getXAxisTitle: function(xAxisSpec, chartDef) {
                if (!xAxisSpec || !chartDef || !ChartFeatures.canShowXAxisTitle(chartDef)) {
                    return;
                }

                let titleText = null;

                if (!!chartDef.xAxisFormatting.axisTitle && chartDef.xAxisFormatting.axisTitle.length > 0) {
                    titleText = chartDef.xAxisFormatting.axisTitle;
                } else if (xAxisSpec.dimension !== undefined) {
                    titleText = xAxisSpec.dimension.column;
                } else if (xAxisSpec.measure !== undefined) {
                    return this.contatenateMeasuresNames(xAxisSpec.measure);
                } else if (chartDef.genericDimension0.length === 1 || chartDef.genericHierarchyDimension.length === 1) {
                    titleText = (ChartDimension.getGenericDimension(chartDef) || {}).column;
                }

                return titleText;
            },

            /**
             * getYAxisTitle is used to get correct y axis title (it is used by d3 and echarts, so it should stay agnostic)
             * @param yAxisSpec
             * @param chartDef
             */
            getYAxisTitle: function(yAxisSpec, chartDef, axisFormatting) {
                if (!yAxisSpec) {
                    return;
                }

                let titleText = null;
                // @ts-ignore
                if (axisFormatting && !_.isNil(axisFormatting.axisTitle) && axisFormatting.axisTitle.length > 0) {
                    titleText = axisFormatting.axisTitle;
                } else if (yAxisSpec.dimension !== undefined) {
                    titleText = yAxisSpec.dimension.column;
                } else if (yAxisSpec.measure !== undefined) {
                    return ColumnAvailability.getAggregatedLabel(yAxisSpec.measure[0]);
                } else if (chartDef != null) {
                    const measuresOnAxis = svc.getMeasuresDisplayedOnAxis(yAxisSpec.position, chartDef.genericMeasures);
                    if (measuresOnAxis.length >= 1) {
                        return this.contatenateMeasuresNames(measuresOnAxis);
                    }
                }

                return titleText;
            },

            getMeasuresDisplayedOnAxis: function(axisPosition, measures) {
                const displayAxis = axisPosition === ChartYAxisPosition.LEFT ? 'axis1' : 'axis2';
                return measures.filter(measure => measure.displayAxis === displayAxis);
            },

            getYAxisSpecs: function(axisSpecs) {
                const ySpecs = axisSpecs && Object.keys(axisSpecs)
                    .filter((key) => key.toLowerCase().includes('y'))
                    .reduce((object, key) => {
                        return Object.assign(object, {
                            [key]: axisSpecs[key]
                        });
                    }, {});
                return ySpecs;
            },

            getYAxisName: function(id) {
                if (id.includes('right')) {
                    return translate('CHARTS.SHARED.RIGHT_AXIS', 'Right axis');
                } else {
                    return translate('CHARTS.SHARED.LEFT_AXIS', 'Left axis');
                }
            },

            isAxisDisplayed: function(genericMeasures, id) {
                const isAxisDisplayed = id.includes('right') ? svc.isRightAxisDisplayed : svc.isLeftAxisDisplayed;
                return id != null ? isAxisDisplayed(genericMeasures) : false;
            },

            isLeftAxisDisplayed: function(genericMeasures) {
                return genericMeasures != null && genericMeasures.some(measure => measure.displayAxis === 'axis1');
            },

            isRightAxisDisplayed: function(genericMeasures) {
                return genericMeasures != null && genericMeasures.some(measure => measure.displayAxis === 'axis2');
            },

            setNumberOfBinsToDimensions: function(chartData, chartDef, axisSpecs) {
                const possibleAxes = { x: 'x', y: 'y_left_0' };
                for (const [dataId, specId] of Object.entries(possibleAxes)) {
                    const axisLabels = chartData.getAxisLabels(dataId);
                    if (axisLabels && axisSpecs[specId] && axisSpecs[specId].dimension) {
                        axisSpecs[specId].dimension.$numberOfBins = axisLabels.length;
                        axisSpecs[specId].dimension.$isInteractiveChart = ChartDimension.isInteractiveChart(chartDef);
                        axisSpecs[specId].dimension.$forceOneTickPerBin = chartDef.variant === CHART_VARIANTS.waterfall;
                    }
                }
            },

            /**
             * Checks if a chart is cropped compared to its initial x and y extents
             * @param   chartDef
             * @returns  True if chart is cropped, False otherwise
             */
            isCroppedChart: function(chartDef) {
                return isCroppedExtent(chartDef.yAxesFormatting) || isCroppedExtent([chartDef.xAxisFormatting]);
            },

            /**
             * Computes the ratio: custom extent / initial extent
             * `customExtent.$initialExtent` keep track of the initial range of an axis before it was altered by another component (reference line e.g.)
             * @param   customExtent
             * @returns  result of initialExtentWidth / manualExtentWidth
             */
            getCustomExtentRatio: function(customExtent) {
                const activeExtent = customExtent.$autoExtent && customExtent.$initialExtent ? customExtent.$initialExtent : customExtent.$autoExtent;
                if (activeExtent) {
                    const initialExtentWidth = activeExtent[1] - activeExtent[0];
                    if (customExtent.editMode === ChartsStaticData.MANUAL_EXTENT_MODE) {
                        const manualExtent = getManualExtent(customExtent);
                        const manualExtentWidth = manualExtent[1] - manualExtent[0];
                        return parseFloat('' + initialExtentWidth / manualExtentWidth);
                    } else if (customExtent.$autoExtent && customExtent.$initialExtent) {
                        const autoExtentWidth = customExtent.$autoExtent[1] - customExtent.$autoExtent[0];
                        return parseFloat('' + initialExtentWidth / autoExtentWidth);
                    }
                }

                return 1;
            },

            /**
             * Check if custom extent is valid
             * @param axesFormatting
             * @returns {boolean} True if valid, False otherwise
             */
            isCustomExtentValid: function(axesFormatting) {
                return axesFormatting.every(formatting => {
                    if (formatting.customExtent && formatting.customExtent.editMode === MANUAL_EXTENT_MODE) {
                        const [min, max] = getManualExtent(formatting.customExtent);
                        return min <= max;
                    }
                    return true;
                });
            },

            resetCustomExtents: function(axesFormatting) {
                if (!axesFormatting) {
                    return;
                }
                axesFormatting.forEach(formatting => {
                    if (formatting && formatting.customExtent) {
                        formatting.customExtent.editMode = AUTO_EXTENT_MODE;
                    }
                });
            },

            isManualMode: function(customExtent) {
                return customExtent && customExtent.editMode
                    && customExtent.editMode === ChartsStaticData.MANUAL_EXTENT_MODE;
            },

            getLeftYAxis: function(yAxes) {
                const leftAxis = yAxes.filter(axis => svc.isLeftYAxis(axis));
                return leftAxis.length ? leftAxis[0] : null;
            },

            getRightYAxis: function(yAxes) {
                const rightAxis = yAxes.filter(axis => svc.isRightYAxis(axis));
                return rightAxis.length ? rightAxis[0] : null;
            },

            isLeftYAxis: function(axis) {
                return !!axis && axis.position === ChartYAxisPosition.LEFT;
            },

            isRightYAxis: function(axis) {
                return !!axis && axis.position === ChartYAxisPosition.RIGHT;
            },

            initYAxesFormatting: function(id) {
                return {
                    id,
                    axisTitleFormatting: { ...AxisTitleFormatting },
                    showAxisTitle: true,
                    displayAxis: true,
                    axisValuesFormatting: {
                        axisTicksFormatting: { ...AxisTicksFormatting },
                        numberFormatting: { ...ChartsStaticData.DEFAULT_NUMBER_FORMATTING_OPTIONS }
                    },
                    ticksConfig: { mode: AxisTicksConfigMode.INTERVAL },
                    customExtent: { ...ChartsStaticData.DEFAULT_CUSTOM_EXTENT },
                    isLogScale: false
                };
            },

            getManualExtentMin,
            getManualExtentMax,
            getManualExtent
        };

        return svc;
    }
})();
