/* jshint loopfunc: true*/
(function(){
    'use strict';

    const app = angular.module('dataiku.charts');

    // (!) This service previously was in static/dataiku/js/simple_report/maps/geometry.js
    app.factory('GeometryMapChart', function(ChartFormatting, _MapCharts, ChartLegendUtils, ChartPropertiesService, ColorUtils, ChartColorUtils, UaChartsCommon, _GeometryCommon, ChartLabels, ChartUADimension, $timeout, BuiltinMapBackgrounds) {
        return function($container, chartDef, data, chartHandler) {

            const elt = _MapCharts.getOrCreateMapContainer($container);
            const geoJsons = data.geoJsons.map(unparsedGeo => JSON.parse(unparsedGeo));
            const layers = {};
            const colorCaches = {};
            const colorScales = {};
            const singleColors = {};

            chartHandler.legendsWrapper.deleteLegends();
            const displayableGeoLayers = _GeometryCommon.getDisplayableLayers(chartDef.geoLayers);

            for (let geometryIndex = 0; geometryIndex < displayableGeoLayers.length; geometryIndex++) {
                const geoLayer = displayableGeoLayers[geometryIndex];
                const geometry = geoLayer.geometry[0];
                colorCaches[geometryIndex] = {};
                // Build color scale
                const hasUAColor = _GeometryCommon.hasUAColor(geoLayer);
                if (hasUAColor) {
                    colorScales[geometryIndex] = _GeometryCommon.makeColorScale(geoLayer, data, chartHandler, geometryIndex);
                } else {
                    singleColors[geometryIndex] = _GeometryCommon.makeSingleColor(geoLayer);
                }

                // Build legend. Can we make some of this common with other UA color scales ?
                const isLegendDiscrete = (ChartUADimension.isAlphanumLike(geoLayer.uaColor[0]) || ChartUADimension.isDiscreteDate(geoLayer.uaColor[0]));
                if (hasUAColor && isLegendDiscrete) {
                    const baseFadeColor = BuiltinMapBackgrounds.getBackgrounds().find(b => b.id === chartDef.mapOptions.tilesLayer).fadeColor || '#333';
                    const fadeColor = ColorUtils.toRgba(baseFadeColor, .5 * geoLayer.colorOptions.transparency);
                    const colorTokenName = ChartColorUtils.getPaletteName(geometryIndex);

                    const legend = {
                        type: 'COLOR_DISCRETE',
                        label: geometry.column,
                        items: data.values[colorTokenName].str.sortedMapping.map(function(_, rowIndex) {
                            const color = colorScales[geometryIndex](rowIndex);
                            return {
                                label: {
                                    ...data.values[colorTokenName].str.sortedMapping[rowIndex],
                                    colorId: ChartColorUtils.getColorId(geoLayer.genericMeasures, { data }, rowIndex, geometryIndex)
                                },
                                color,
                                focusFn: function(){
                                    layers[geometryIndex].setStyle(function(feature) {
                                        if (data.values[colorTokenName].str.data[feature.properties.idx] === rowIndex) {
                                            return {
                                                color,
                                                opacity: 1,
                                                weight: chartDef.strokeWidth
                                            };
                                        } else {
                                            return {
                                                color: fadeColor,
                                                opacity: 1,
                                                weight: chartDef.strokeWidth };
                                        }
                                    });
                                },
                                unfocusFn : function() {
                                    layers[geometryIndex].setStyle(function(feature) {
                                        const c = singleColors[geometryIndex] || _GeometryCommon.makeColor(geoLayer, data, feature.properties.idx, colorScales[geometryIndex], singleColors[geometryIndex], colorCaches[geometryIndex], geometryIndex);
                                        return {
                                            color: c,
                                            opacity: 1,
                                            weight: chartDef.strokeWidth
                                        };
                                    });
                                },
                                focused : false
                            };
                        })
                    };
                    chartHandler.legendsWrapper.pushLegend(legend);

                } else if (colorScales[geometryIndex]) {
                    const legend = ChartLegendUtils.createContinuousLegend(geoLayer.uaColor[0], colorScales[geometryIndex]);
                    legend.label = geometry.column;
                    if (ChartUADimension.isDateRange(geoLayer.uaColor[0])) {
                        legend.formatter = ChartFormatting.getForDate();
                    } else {
                        legend.formatter = ChartFormatting.getForLegend(geoLayer.uaColor[0], colorScales[geometryIndex].innerScale.domain());
                    }
                    chartHandler.legendsWrapper.pushLegend(legend);
                } else {
                    if (chartDef.geoLayers.length < 3) {
                        chartHandler.legendsWrapper.deleteLegends();
                    } else {
                        const legend = ChartLegendUtils.getSingleColorLegend(singleColors[geometryIndex], geometry.column);
                        chartHandler.legendsWrapper.pushLegend(legend);
                    }
                }
            }

            ChartLegendUtils.drawLegend(chartDef, chartHandler, $container).then(function() {

                _MapCharts.adjustLegendPlacement(chartDef, $container);

                const map = _MapCharts.createMapIfNeeded(elt, chartHandler.chartSpecific, chartDef);
                if (elt.data('leaflet-data-layer')) {
                    map.removeLayer(elt.data('leaflet-data-layer'));
                }
                _MapCharts.repositionMap(map, elt, data);

                function onEachFeatureClosure(geometryIndex, geoLayer){
                    return function onEachFeature(feature, layer) {
                        if (!ChartPropertiesService.showTooltip(chartDef.tooltipOptions, chartHandler.chartTileProperties)) {
                            return;
                        }

                        let html = '';

                        if (_GeometryCommon.hasUAColor(geoLayer)) {
                            const colorTokenName = ChartColorUtils.getPaletteName(geometryIndex);
                            if (colorTokenName in data.values){
                                html += sanitize(ChartLabels.getUaLabel(geoLayer.uaColor[0])) + ': <strong>' + sanitize(_GeometryCommon.formattedColorVal(chartDef, data, feature.properties.idx, colorTokenName, geometryIndex)) +'</strong><br />';
                            }
                        }

                        chartDef.uaTooltip.forEach(function(ua, j) {
                            html += sanitize(ChartLabels.getUaLabel(ua)) + ': <strong>' + sanitize(UaChartsCommon.formattedVal(data.values['tooltip_' + j], ua, feature.properties.idx)) + '</strong><br/>';
                        });

                        if (html.length) {
                            //Classname is a locator for the integration tests - Leaflet API doesn't allow us to add an id
                            layer.bindPopup(html, { className: 'qa-chart-tooltip' });
                        }
                    };
                }

                function getLayer(geoJsons, geoLayer, geometryIndex){
                    const layerSingleColor = _GeometryCommon.makeSingleColor(geoLayer);
                    return L.geoJson(geoJsons.features, {
                        style: function(feature) {
                            const c = _GeometryCommon.makeColor(
                                geoLayer,
                                data,
                                feature.properties.idx,
                                colorScales[geometryIndex],
                                layerSingleColor,
                                colorCaches[geometryIndex],
                                geometryIndex
                            );
                            return {
                                color: c,
                                opacity: 1,
                                weight: chartDef.strokeWidth,
                                fillOpacity: chartDef.fillOpacity
                            };
                        },
                        onEachFeature : onEachFeatureClosure(geometryIndex, geoLayer),
                        pointToLayer: function(feature, latlng) {
                            const c = _GeometryCommon.makeColor(
                                geoLayer,
                                data,
                                feature.properties.idx,
                                colorScales[geometryIndex],
                                layerSingleColor,
                                colorCaches[geometryIndex],
                                geometryIndex
                            );

                            const geojsonMarkerOptions = {
                                radius: 5,
                                fillColor: c,
                                color: c,
                                weight: chartDef.strokeWidth,
                                opacity: 1,
                                fillOpacity: 1
                            };
                            return L.circleMarker(latlng, geojsonMarkerOptions);
                        }
                    });
                }

                function plot(geometryIndex){
                    map.addLayer(layers[geometryIndex]);
                    if (map.$justCreated) {
                        $timeout(function() {
                            map.fitBounds(layers[geometryIndex]);
                        });
                    }
                    elt.data('leaflet-data-layer', layers[geometryIndex]);
                }

                // delete all the current geometry layers
                for (const displayedGeo of chartHandler.displayedGeometries){
                    map.removeLayer(displayedGeo.geoJsonLayer);
                }
                const newGeometries = [];
                for (let geometryIndex = 0; geometryIndex < displayableGeoLayers.length; geometryIndex++) {
                    const newGeoLayer = displayableGeoLayers[geometryIndex];
                    layers[geometryIndex] = getLayer(
                        geoJsons[geometryIndex],
                        newGeoLayer,
                        geometryIndex
                    );
                    plot(geometryIndex);
                    newGeometries[geometryIndex] = { geometry: angular.copy(newGeoLayer[geometryIndex]), geoJsonLayer: layers[geometryIndex] };
                }
                chartHandler.displayedGeometries= newGeometries;

            }).finally(function(){
                /*
                 * Signal to the callee handler that the chart has been loaded.
                 * Dashboards use it to determine when all insights are completely loaded.
                 */
                if (typeof(chartHandler.loadedCallback) === 'function') {
                    chartHandler.loadedCallback();
                }
            });

        };
    });

})();
