/* jshint loopfunc: true*/
(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    // (!) This service previously was in static/dataiku/js/simple_report/maps/scatter.js
    app.factory('GridMapChart', function(ChartFormatting, ChartPropertiesService, _MapCharts, ChartLabels, UaChartsCommon, ChartLegendUtils, ChartColorScales) {
        return function($container, chartDef, data, chartHandler) {

            const elt = _MapCharts.getOrCreateMapContainer($container);
            const map = _MapCharts.createMapIfNeeded(elt, chartHandler.chartSpecific, chartDef);
            if (elt.data('leaflet-data-layer')) {
                map.removeLayer(elt.data('leaflet-data-layer'));
            }
            _MapCharts.repositionMap(map, elt, data);

            // Build color scale
            const hasColor = chartDef.colorMeasure.length;
            let colorScale, resultingColor;
            if (hasColor) {
                colorScale = ChartColorScales.continuousColorScale(chartDef.colorOptions, data.aggregations.color.min, data.aggregations.color.max, data.aggregations.color.tensor, undefined, chartHandler.getChartTheme());
                colorScale.type = 'MEASURE';
            } else {
                resultingColor = UaChartsCommon.makeSingleColor(chartDef);
            }

            ChartLegendUtils.initLegend(chartDef, null, chartHandler.legendsWrapper, colorScale);

            if (colorScale) {
                chartHandler.legendsWrapper.getLegend(0).formatter = ChartFormatting.getForLegend(chartDef.colorMeasure[0], colorScale.innerScale.domain());
            }

            ChartLegendUtils.drawLegend(chartDef, chartHandler, $container).then(function() {
                _MapCharts.adjustLegendPlacement(chartDef, $container);

                const layerGroup = L.layerGroup();

                data.cellMinLat.forEach(function(x, i) {
                    const minLat = data.cellMinLat[i];
                    const minLon = data.cellMinLon[i];
                    const maxLat = minLat + data.gridLatDeg;
                    const maxLon = minLon + data.gridLonDeg;

                    const c = hasColor ? colorScale(data.aggregations.color.tensor[i]) : resultingColor;

                    const rect = L.rectangle([[minLat, minLon], [maxLat, maxLon]], {
                        stroke: false,
                        fill: true,
                        fillColor: c,
                        fillOpacity: 1
                    });

                    if (ChartPropertiesService.showTooltip(chartDef.tooltipOptions, chartHandler.chartTileProperties)) {
                        let html = '';

                        html += 'Lon: <strong>' + ChartFormatting.getForIsolatedNumber()(minLon + (maxLon - minLon) / 2) + '</strong><br />';
                        html += 'Lat: <strong>' + ChartFormatting.getForIsolatedNumber()(minLat + (maxLat - minLat) / 2) + '</strong><br />';

                        if (hasColor) {
                            const colorFormatter = ChartFormatting.getForIsolatedNumber(chartDef.colorMeasure[0]);
                            html += ChartLabels.getShortMeasureLabel(chartDef.colorMeasure[0])
                                + ': <strong>' + sanitize(colorFormatter(data.aggregations.color.tensor[i])) + '</strong>';
                        }

                        if (chartDef.tooltipMeasures.length > 0) {
                            html += '<hr/>';
                        }

                        chartDef.tooltipMeasures.forEach(function(measure, j) {
                            const tooltipFormatter = ChartFormatting.getForIsolatedNumber(measure);
                            html += `${ChartLabels.getLongMeasureLabel(measure)}: <strong>${sanitize(tooltipFormatter(data.aggregations['tooltip_' + j].tensor[i]))}</strong><br/>`;
                        });

                        //Classname is a locator for the integration tests - Leaflet API doesn't allow us to add an id
                        rect.bindPopup(html, { className: 'qa-chart-tooltip' });
                    }

                    layerGroup.addLayer(rect);
                });
                layerGroup.addTo(map);

                elt.data('leaflet-data-layer', layerGroup);
            }).finally(function() {
                /*
                 * Signal to the callee handler that the chart has been loaded.
                 * Dashboards use it to determine when all insights are completely loaded.
                 */
                if (typeof (chartHandler.loadedCallback) === 'function') {
                    chartHandler.loadedCallback();
                }
            });
        };
    });

})();
