
(function() {
    'use strict';

    angular.module('dataiku.shared').factory('BuiltinMapBackgrounds', builtinMapBackgrounds);

    /**
     * This file declares the builtin map backgrounds stored in window.dkuMapBackgrounds
     * Plugins can add their own map backgrounds with window.dkuMapBackgrounds.addCustom & addMapbox
     * (!) This file previously was in static/dataiku/js/simple_report/common/builtin-map-backgrounds.js
     */
    function builtinMapBackgrounds($window, translate, $rootScope) {
        const backgrounds = []

        function _addCustom({background, atTheEnd = true, withLog = false}) {

            const existingIndex = backgrounds.findIndex(b => b.id === background.id);

            if (existingIndex !== -1) {
                // eslint-disable-next-line no-console
                withLog && console.warn('Map background \'' + background.id + '\' already exists, it will be overriden.'); /*@console*/ // NOSONAR: OK to use console.
                backgrounds.splice(existingIndex, 1, background);
            } else {
                atTheEnd ? backgrounds.push(background): backgrounds.unshift(background);
            }
        }

        function addCustom(background) {
            _addCustom({background, atTheEnd: true, withLog: true});
        }

        function addMapbox(mapId, label, accessToken) {
            let name;
            if (!label) {
                name = mapId.split('.')[mapId.split('.').length - 1];
                name = name.charAt(0).toUpperCase() + name.slice(1).toLowerCase();
            } else {
                name = label;
            }

            addCustom({
                id: mapId,
                name: name,
                category: 'Mapbox',
                getTileLayer: function() {
                    return L.tileLayer('https://api.mapbox.com/v4/' + mapId + '/{z}/{x}/{y}.png?access_token=' + accessToken, {
                        attribution: '© <a href="https://www.mapbox.com/feedback/">Mapbox</a> © <a href="http://www.openstreetmap.org/copyright">OpenStreetMap</a>'
                    });
                }
            });
        }

        function addWMS(id, name, category, wmsURL, layerId) {
            addCustom({
                id: id,
                name: name,
                category: category,
                getTileLayer: function() {
                    return L.tileLayer.wms(wmsURL, {
                        layers: layerId
                    });
                }
            });
        }

        function _getDefaultMapBackgrounds() {
            return [
                {
                    'id': 'cartodb-positron',
                    'name': translate('BUILTIN_MAP_BACKGROUNDS.CARTODB_POSITRON', 'Black & White (light)'),
                    'getTileLayer': function() {
                        return new L.tileLayer(
                            'http://{s}.basemaps.cartocdn.com/light_all/{z}/{x}/{y}.png', {
                                attribution: '&copy; <a href="http://www.openstreetmap.org/copyright">OpenStreetMap</a> contributors, &copy; <a href="http://cartodb.com/attributions">CartoDB</a>'
                            }
                        );
                    },
                    'fadeColor': '#333'
                },
                {
                    'id': 'cartodb-dark',
                    'name': translate('BUILTIN_MAP_BACKGROUNDS.CARTODB_DARK', 'Black & White (dark)'),
                    'getTileLayer': function() {
                        return new L.tileLayer('http://{s}.basemaps.cartocdn.com/dark_all/{z}/{x}/{y}.png', {
                            attribution: '&copy; <a href="http://www.openstreetmap.org/copyright">OpenStreetMap</a> contributors, &copy; <a href="http://cartodb.com/attributions">CartoDB</a>'
                        });
                    },
                    'fadeColor': '#EEE'
                }
            ];
        }

        const dkuMapBackgrounds = {};
        dkuMapBackgrounds.addCustom = addCustom;
        dkuMapBackgrounds.addMapbox = addMapbox;
        dkuMapBackgrounds.addWMS = addWMS;
        dkuMapBackgrounds.backgrounds = backgrounds;
        $window.dkuMapBackgrounds = dkuMapBackgrounds;

        let isBound = false

        function bindOnTranslateChangeSuccess() {
            if (!isBound) {
                $rootScope.$on('$translateChangeSuccess', function () {
                    const defaultBackgrounds = _getDefaultMapBackgrounds();
                    defaultBackgrounds.forEach(defaultBackground => _addCustom({
                        background: defaultBackground,
                        atTheEnd: false,
                        withLog: false,
                    }))
                });
                isBound = true;
            }
        }

        return {
            getBackgrounds: () => [...dkuMapBackgrounds.backgrounds],
            /*
             *   The following is used in plugins to add new map backgrounds, don't rename those methods
             */
            bindOnTranslateChangeSuccess,
            addCustom,
            addMapbox,
            addWMS
        };
    }
})();
