(function() {
    'use strict';

    angular.module("dataiku.shared").factory("PromptChatService", PromptChatService);

    function PromptChatService($dkuSanitize, $rootScope, Logger, CreateModalFromTemplate, CodeMirrorSettingService, ClipboardUtils, ConnectionsService, PromptUtils, ReasoningEffortCompletionSetting) {
        const svc = {};

        svc.buildCurrentChatBranch = function(messages, lastMessageId, settingsComparator = compareLlmSettings) {
            let currentBranch = [];
            if (!messages || Object.keys(messages).length === 0) {
                return currentBranch;
            }
            let currentMessage = messages[lastMessageId];
            while (currentMessage && currentMessage.parentId) {
                currentBranch.push(currentMessage);
                try {
                    if (currentMessage.message.uiRepresentation){
                        currentMessage.message.$markdownContent = currentMessage.message.uiRepresentation;
                    } else if (currentMessage.message.content){
                        currentMessage.message.$markdownContent = $dkuSanitize(marked(currentMessage.message.content, {
                            breaks: true
                        }));
                    }
                } catch(e) {
                    Logger.error('Error parsing markdown HTML, switching to plain text', e);
                    currentMessage.message.$markdownContent = null;
                }

                if (currentMessage.error && currentMessage.llmError) {
                    currentMessage.credentialsError = ConnectionsService.getCredentialsError(currentMessage.llmError);
                }

                currentMessage = messages[currentMessage.parentId];
            }
            let previousAssistantMessage;
            currentBranch =  currentBranch.reverse();
            Object.entries(currentBranch).forEach(([key, message]) => {
                if (message.message && ["assistant", "toolValidationRequests"].includes(message.message.role) && message.$customSettings != null){
                    if (previousAssistantMessage){
                        message.$changedSettings = settingsComparator(message, previousAssistantMessage);
                        message.$contextChanged = !angular.equals(message.context, previousAssistantMessage.context);
                    }
                    previousAssistantMessage = message;
                }
            });
            return currentBranch;
        };

        svc.enrichChatMessages = function(messages, preEnrichedMessages) {
            if (!messages) return {};

            const enrichedMessages = angular.copy(messages);
            Object.entries(enrichedMessages).forEach(([key, message]) => {
                message.statusClass = getChatOutputClass(message);
                if (preEnrichedMessages && preEnrichedMessages[message.id]?.$customSettings) {
                    message.$customSettings = preEnrichedMessages[message.id].$customSettings;
                }
                const parentId = message.parentId;
                if (parentId) {
                    const parentMessage = enrichedMessages[parentId];

                    if (!parentMessage.childrenIds) {
                        parentMessage.childrenIds = [];
                    }
                    parentMessage.childrenIds[message.version] = message.id;

                    if (message.message && message.message.toolValidationResponses &&
                        parentMessage && parentMessage.message.toolValidationRequests ) {
                        const tvrMap = new Map();
                        parentMessage.message.toolValidationRequests.forEach(tvreq => tvrMap.set(tvreq.id, tvreq));
                        message.message.toolValidationResponses.forEach(tvresp => {
                                const tvreq = tvrMap.get(tvresp.validationRequestId);
                                tvreq.validated = tvresp.validated;
                                tvreq.arguments = tvresp.arguments;
                            }
                        )
                    }
                }
            });
            return enrichedMessages;
        };

        svc.openConversationHistoryModal = function($scope, chatMessages, lastMessageId, llmSettings, chatSystemMessage) {
            CreateModalFromTemplate("/templates/promptstudios/conversation-history-modal.html", $scope, null, function (modalScope) {
                modalScope.uiState = {
                    activeTab: "json",
                };

                modalScope.pythonReadOnlyOptions = CodeMirrorSettingService.get('text/x-python');
                modalScope.pythonReadOnlyOptions["readOnly"] = "nocursor";
                modalScope.jsonReadOnlyOptions = CodeMirrorSettingService.get('application/json');
                modalScope.jsonReadOnlyOptions["readOnly"] = "nocursor";

                function getJsonMessages() {
                    let messages = svc.buildCurrentChatBranch(chatMessages, lastMessageId)
                        .filter(item => !item.error)
                        .map(item => {
                            const { $markdownContent, ...rest } = item.message;
                            return rest;
                        });
                    if (chatSystemMessage) {
                        let systemMessage = {
                            "role": "system",
                            "content": chatSystemMessage
                        };
                        messages.unshift(systemMessage);
                    }
                    return messages;
                };

                function getPythonMessages() {
                    let messages = getJsonMessages();

                    let pythonMessages = "import dataiku\n";
                    pythonMessages += `LLM_ID = "${$scope.activeLLM.id}"\n`;
                    pythonMessages += `llm = dataiku.api_client().get_default_project().get_llm(LLM_ID)\n`;
                    pythonMessages += `# Create and run a completion query with your chat history\n`;
                    pythonMessages += `chat = llm.new_completion()\n\n`;

                    if (llmSettings['reasoningEffort'] && PromptUtils.isReasoningModel($scope.activeLLM)) {
                        pythonMessages += `chat.settings["reasoningEffort"] = "${llmSettings['reasoningEffort']}"\n`
                        if (llmSettings['reasoningEffort'] === "CUSTOM" && llmSettings['customReasoningEffort']) {
                            pythonMessages += `chat.settings["customReasoningEffort"] = "${llmSettings['customReasoningEffort']}"\n`
                        }
                    }
                    if (llmSettings['temperature']) {
                        pythonMessages += `chat.settings["temperature"] = ${llmSettings['temperature']}\n`
                    }
                    if (llmSettings['topK']) {
                        pythonMessages += `chat.settings["topK"] = ${llmSettings['topK']}\n`
                    }
                    if (llmSettings['topP']) {
                        pythonMessages += `chat.settings["topP"] = ${llmSettings['topP']}\n`
                    }
                    if (llmSettings['maxOutputTokens']) {
                        pythonMessages += `chat.settings["maxOutputTokens"] = ${llmSettings['maxOutputTokens']}\n`
                    }
                    if (llmSettings['stopSequences'] && llmSettings['stopSequences'].length > 0) {
                        pythonMessages += `chat.settings["stopSequences"] = ${llmSettings['stopSequences']}\n`
                    }
                    if (llmSettings['presencePenalty']) {
                        pythonMessages += `chat.settings["presencePenalty"] = ${llmSettings['presencePenalty']}\n`
                    }
                    if (llmSettings['frequencyPenalty']) {
                        pythonMessages += `chat.settings["frequencyPenalty"] = ${llmSettings['frequencyPenalty']}\n`
                    }
                    if (llmSettings['responseFormat']) {
                        pythonMessages += `chat.with_json_output()\n`
                    }

                    const pythoniseBoolean = function(s) {
                        s = s.replaceAll(`"allowEditingInputs":false`, `"allowEditingInputs":False`)
                        return s.replaceAll(`"allowEditingInputs":true`, `"allowEditingInputs":True`);
                    }

                    pythonMessages += "\n";
                    messages.forEach((message) => {
                        if (message["role"] === "memoryFragment") {
                            pythonMessages += `chat.with_memory_fragment(${JSON.stringify(message["memoryFragment"])})\n`;
                        } else if (message["role"] === "toolValidationRequests") {
                            pythonMessages += `chat.with_tool_validation_requests(${pythoniseBoolean(JSON.stringify(message["toolValidationRequests"]))})\n`;
                        } else if (message["role"] === "toolValidationResponses") {
                            for (const tvresp of message["toolValidationResponses"]) {
                                const validated = tvresp.validated ? "True" : "False";
                                pythonMessages += `chat.with_tool_validation_response(${JSON.stringify(tvresp.validationRequestId)}, ${validated}`
                                if (tvresp.arguments != null) {
                                    pythonMessages += `, ${JSON.stringify(tvresp.arguments)}`;
                                } 
                                pythonMessages += `)\n`;
                            }
                        } else {
                            pythonMessages += `chat.with_message(${JSON.stringify(message["content"])}, role=${JSON.stringify(message["role"])})\n`;
                        }
                    });

                    pythonMessages += "\n";
                    pythonMessages += `response = chat.execute()`;
                    return pythonMessages;
                }

                modalScope.jsonMessages = getJsonMessages();
                modalScope.pythonMessages = getPythonMessages();

                modalScope.copyConversationHistory = function () {
                    if (modalScope.uiState.activeTab === "json") {
                        ClipboardUtils.copyToClipboard(JSON.stringify(modalScope.jsonMessages, null, 2), "Conversation copied to clipboard");
                    }
                    if (modalScope.uiState.activeTab === "python") {
                        ClipboardUtils.copyToClipboard(modalScope.pythonMessages, "Conversation copied to clipboard");
                    }
                }
            });
        };

        const logs = {};
        svc.getLog = function(sessionId) {
            return logs[sessionId];
        };

        svc.setLog = (sessionId, log) => {
            logs[sessionId] = log;
        };

        svc.clearLogs = () => {
            Object.keys(logs).forEach(sessionId => delete logs[sessionId]);
        };

        const chatCache = {};
        svc.getChat = (sessionId) => {
            return chatCache[sessionId];
        };

        svc.setChat = (sessionId, lastMessageId, messages) => {
            chatCache[sessionId] = {
                ...(chatCache[sessionId] || {}),
                lastMessageId,
                messages
            };
        };

        svc.setChatContext = (sessionId, context) => {
            chatCache[sessionId] = {
                ...(chatCache[sessionId] || {}),
                context
            };
        };

        svc.resetChat = (sessionId) => {
            svc.setChat(sessionId);
            svc.setChatContext(sessionId, null);
        }

        svc.getChatContext = (sessionId) => {
            return chatCache[sessionId]?.context;
        };

        svc.getChatContextInitializeIfNeeded = (sessionId) => {
            let context = chatCache[sessionId]?.context;
            if (!context) {
                context = {
                    // Generate a random conversation ID
                    "conversationId":  'conv-' + Math.random().toString(36).substring(2, 15) + Math.random().toString(36).substring(2, 15),
                    "dkuOnBehalfOf": $rootScope.appConfig.login
                }
                svc.setChatContext(sessionId, context);
            }
            return context;
        };

        svc.upsertChatContext = (sessionId, contextUpsert) => {
            let previousContext = svc.getChatContext(sessionId);
            const mergedContext = { ...previousContext };

            for (const key in contextUpsert) {
                if (Object.prototype.hasOwnProperty.call(contextUpsert, key)) {
                    const upsertValue = contextUpsert[key];

                    if (upsertValue != null) {
                        mergedContext[key] = upsertValue;
                    } else if (Object.prototype.hasOwnProperty.call(mergedContext, key)) {
                        delete mergedContext[key];
                    }
                }
            }

            svc.setChatContext(sessionId, mergedContext);
        };


        svc.cleanChatMesssages = function(messages) {
            const cleanedMessages = {};
            Object.entries(messages).forEach(([key, message]) => {
                const { statusClass, credentialsError, $changedSettings, $customSettings, $contextChanged, message: { $markdownContent, ...restMessage } = {}, ...rest } = message;
                cleanedMessages[key] = {
                    ...rest,
                    ...(message.message && { message: restMessage })
                };
            });
            return cleanedMessages;
        };

        svc.compareCompletionSettings = (newCompletionSettings, oldCompletionSettings, changes) => {
            const settings = {
                "temperature": "Temperature",
                "topK": "TopK",
                "topP": "TopP",
                "maxOutputTokens": "Max Output Tokens",
                "frequencyPenalty": "Frequency Penalty",
                "presencePenalty": "Presence Penalty"
            };

            Object.entries(settings).forEach(([key, label]) => {
                compareAndPush(newCompletionSettings?.[key], oldCompletionSettings?.[key], label, changes);
            });

            compareAndPush(newCompletionSettings.responseFormat?.type, oldCompletionSettings.responseFormat?.type, "Response Format", changes);
            compareAndPushReasoning(newCompletionSettings, oldCompletionSettings, changes);
        };

        return svc;

        function compareAndPush(newVal, oldVal, label, changes) {
            if (newVal !== oldVal) {
                if(["System message", "Response Format"].includes(label)){
                    changes.push(`${label} has changed`);
                }
                else{
                    changes.push(`${label} has changed from ${oldVal ?? "not defined"} to ${newVal ?? "not defined"}`);
                }
            }
        }

            function compareAndPushReasoning(newSettings, oldSettings, changes){
                let newReasoningEffort = ReasoningEffortCompletionSetting.find(function(reasoningEffort) {
                    return reasoningEffort.rawValue === newSettings.reasoningEffort;
                });
                let oldReasoningEffort = ReasoningEffortCompletionSetting.find(function(reasoningEffort) {
                    return reasoningEffort.rawValue === oldSettings.reasoningEffort;
                });

                if (newReasoningEffort !== oldReasoningEffort) {
                    if (newReasoningEffort.rawValue === "CUSTOM") {
                        changes.push(`Reasoning Effort has changed from ${oldReasoningEffort.displayName} to custom value ${newSettings.customReasoningEffort}`);
                    } else if (oldReasoningEffort.rawValue === "CUSTOM") {
                        changes.push(`Reasoning Effort has changed from custom value ${oldSettings.customReasoningEffort} to ${newReasoningEffort.displayName}`);
                    } else {
                        changes.push(`Reasoning Effort has changed from ${oldReasoningEffort.displayName} to ${newReasoningEffort.displayName}`);
                    }
                }
                if (oldReasoningEffort.rawValue === "CUSTOM" && newReasoningEffort.rawValue === "CUSTOM"
                    && oldSettings.customReasoningEffort !== newSettings.customReasoningEffort) {
                    changes.push(`Reasoning Effort has changed from custom value ${oldSettings.customReasoningEffort} to custom value ${newSettings.customReasoningEffort}`);
                }
            }

        function compareLlmSettings(newMessage, oldMessage){
            let changes = [];

            compareAndPush(newMessage.llmStructuredRef.id, oldMessage.llmStructuredRef.id, "Model", changes);
            compareAndPush(newMessage.systemMessage, oldMessage.systemMessage, "System message", changes);
            svc.compareCompletionSettings(newMessage.completionSettings, oldMessage.completionSettings, changes);
            return changes;
        }

        function getChatOutputClass(message) {
            let status = '';
            if (message.error) {
                status = 'invalid'
            } else {
                status = 'valid';
            }
            return 'prompt-chat__assistant-message--' + status;
        }
    }
})();
