(function(root, factory) {
    if (typeof define === 'function' && define.amd) {
    // AMD. Register as an anonymous module.
        define([], factory);
    } else if (typeof exports === 'object') {
    /*
     * Node. Does not work with strict CommonJS, but
     * only CommonJS-like environments that support module.exports,
     * like Node.
     */
        module.exports = factory();
    } else {
    // Browser globals (root is window)
        root.GridList = factory();
    }
})(this, function() {
    const GridList = function(items, options) {
    /**
     * A GridList manages the two-dimensional positions from a list of items,
     * within a virtual matrix.
     *
     * The GridList's main function is to convert the item positions from one
     * grid size to another, maintaining as much of their order as possible.
     *
     * The GridList's second function is to handle collisions when moving an item
     * over another.
     *
     * The positioning algorithm places items in columns. Starting from left to
     * right, going through each column top to bottom.
     *
     * The size of an item is expressed using the number of cols and rows it
     * takes up within the grid (w and h)
     *
     * w = number of cols
     * h = number of rows
     *
     * The position of an item is express using the col and row position within
     * the grid (x and y)
     *
     * x = col
     * y = row
     *
     * An item is an object of structure:
     * {
     *   w: 3, h: 1,
     *   x: 0, y: 1
     * }
     */

        this._options = options;
        for (const k in this.defaults) {
            if (!this._options.hasOwnProperty(k)) {
                this._options[k] = this.defaults[k];
            }
        }

        this.items = items;

        this._adjustSizeOfFullWidthItems();

        this.generateGrid();

        this.positionPositionlessElements();

        this._sortItemsByPosition();

        for (const item of this.items) {
            this._resolveCollisions(item);
        }
    };

    GridList.cloneItems = function(sourceItems, destinationItems) {
    /**
     * Clone items with a deep level of one. Items are not referenced but their
     * properties are
     */
        let i;
        let k;
        if (destinationItems === undefined) {
            destinationItems = [];
        } else {
            destinationItems.splice(0, destinationItems.length);
        }
        for (i = 0; i < sourceItems.length; i++) {
            /*
             * XXX: this is good because we don't want to lose item reference, but
             * maybe we should clear their properties since some might be optional
             */
            if (!destinationItems[i]) {
                destinationItems[i] = {};
            }
            for (k in sourceItems[i]) {
                destinationItems[i][k] = sourceItems[i][k];
            }
        }
        return destinationItems;
    };

    GridList.prototype = {
        defaults: {
            lanes: 5,
            direction: 'horizontal'
        },

        /**
         * Illustates grid as text-based table, using a number identifier for each
         * item. E.g.
         *
         *  #|  0  1  2  3  4  5  6  7  8  9 10 11 12 13
         *  --------------------------------------------
         *  0| 00 02 03 04 04 06 08 08 08 12 12 13 14 16
         *  1| 01 -- 03 05 05 07 09 10 11 11 -- 13 15 --
         *
         * Warn: Does not work if items don't have a width or height specified
         * besides their position in the grid.
         */
        toString: function() {
            const widthOfGrid = this.grid.length;
            let output = '\n #|';
            let border = '\n --';
            let item;
            let i;
            let j;

            // Render the table header
            for (i = 0; i < widthOfGrid; i++) {
                output += ' ' + this._padNumber(i, ' ');
                border += '---';
            }
            output += border;

            // Render table contents row by row, as we go on the y axis
            for (i = 0; i < this._options.lanes; i++) {
                output += '\n' + this._padNumber(i, ' ') + '|';
                for (j = 0; j < widthOfGrid; j++) {
                    output += ' ';
                    item = this.grid[j][i];
                    output += item ? this._padNumber(this.items.indexOf(item), '0') : '--';
                }
            }
            output += '\n';
            return output;
        },

        generateGrid: function() {
            /**
             * Build the grid structure from scratch, with the current item positions
             */
            let i;
            this._resetGrid();
            for (i = 0; i < this.items.length; i++) {
                this._markItemPositionToGrid(this.items[i]);
            }
        },

        toggleGapMode(allowDirectionalGaps) {
            if (allowDirectionalGaps === this._options.allowDirectionalGaps) {
                return;
            }
            this._options.allowDirectionalGaps = allowDirectionalGaps;
            if (!this._options.allowDirectionalGaps) {
                this.fillDirectionalGaps();
            }
        },

        resizeGrid: function(lanes) {
            if (lanes === this._options.lanes) {
                return;
            }
            let previousLanes = this._options.lanes;
            if (this.itemSizeAndPositionSnapshot != null) {
                this._restoreItemSizeAndPositionFromSnapshot();
                previousLanes = this.itemSizeAndPositionSnapshot.lanes;
            }
            this._options.lanes = lanes;
            // Here were adjusting the items size and position to fit the new grid even if it might output collisions
            this._adjustItemsForLaneChange(previousLanes);

            this._resetGrid();
            this._sortItemsByPosition();
            this._resolveItemCollisionsAfterLaneChange();
            if (!this._options.allowDirectionalGaps) {
                this.fillDirectionalGaps();
            }
        },

        captureItemSizeAndPositionSnapshot: function() {
            this.itemSizeAndPositionSnapshot = {
                lanes: this._options.lanes,
                positionAndSizeById: new Map()
            };
            this.items.forEach(item => this.itemSizeAndPositionSnapshot.positionAndSizeById.set(item.id, {
                x: item.x,
                y: item.y,
                w: item.w,
                h: item.h,
                minWidth: item.minWidth,
                minHeight: item.minHeight
            }));
        },

        positionPositionlessElements: function() {
            let currentColumn = 0;
            this._sortItemsByPosition();

            for (let i = 0; i < this.items.length; i++) {
                const item = this.items[i];
                let position = this._getItemPosition(item);

                if (!this._isANumber(position.x) || !this._isANumber(position.y)) {
                    this._updateItemPosition(item, this.findPositionForItem(item, { x: currentColumn, y: 0 }));
                    position = this._getItemPosition(item);
                }

                // New items should never be placed to the left of previous items
                currentColumn = Math.max(currentColumn, position.x);
            }
        },

        findPositionForItem: function(item, start, fixedRow) {
            /*
             * This method has two options for the position we want for the item:
             * - Starting from a certain row/column number and only looking for
             *   positions to its right
             * - Accepting positions for a certain row number only (use-case: items
             *   being shifted to the left/right as a result of collisions)
             *
             * @param {Object<x:Number, y:Number, w:Number, h:Number} item
             * @param {Object<x:Number, y:Number} start Position from which to start
             *     the search.
             * @param {Number} [fixedRow] If provided, we're going to try to find a
             *     position for the new item on it. If doesn't fit there, we're going
             *     to put it on the first row.
             *
             * @returns {Number[2]} x and y.
             */

            let x, y, position;

            /*
             * Start searching for a position from the horizontal position of the item
             */
            for (x = start.x; x < this.grid.length; x++) {
                if (fixedRow !== undefined) {
                    position = [x, fixedRow];

                    if (this._itemFitsAtPosition(item, position)) {
                        return position;
                    }
                } else {
                    for (y = start.y; y < this._options.lanes; y++) {
                        position = [x, y];

                        if (this._itemFitsAtPosition(item, position)) {
                            return position;
                        }
                    }
                    for (y = start.y; y >= 0; y--) {
                        position = [x, y];

                        if (this._itemFitsAtPosition(item, position)) {
                            return position;
                        }
                    }
                }
            }

            // If we've reached this point, we need to start a new column
            const newCol = this.grid.length;
            let newRow = 0;

            if (fixedRow !== undefined && this._itemFitsAtPosition(item, [newCol, fixedRow])) {
                newRow = fixedRow;
            }

            return [newCol, newRow];
        },

        moveItemToPosition: function(item, newPosition) {
            const previousItemPosition = this._getItemPosition(item);
            const position = this._getItemPosition({
                x: newPosition[0],
                y: newPosition[1],
                w: item.w,
                h: item.h
            });
            this._updateItemPosition(item, [position.x, position.y]);
            this._resolveCollisions(item, previousItemPosition);
            if (!this._options.allowDirectionalGaps) {
                this.fillDirectionalGaps();
            }
        },

        moveAndResizeItem: function(item, newItem) {
            const position = this._getItemPosition({
                x: newItem.x,
                y: newItem.y,
                h: newItem.w,
                w: newItem.h
            });

            this._updateItemPosition(item, [position.x, position.y]);

            this._updateItemSize(item, position.w, position.h);

            this._resolveCollisions(item);

            if (!this._options.allowDirectionalGaps) {
                this.fillDirectionalGaps();
            }
        },

        resizeItem: function(item, size) {
            /**
             * Resize an item and resolve collisions.
             *
             * @param {Object} item A reference to an item that's part of the grid.
             * @param {Object} size
             * @param {Number} [size.w=item.w] The new width.
             * @param {Number} [size.h=item.h] The new height.
             */

            const width = size.w || item.w,
                height = size.h || item.h;

            this._updateItemSize(item, width, height);

            this._resolveCollisions(item);
            if (!this._options.allowDirectionalGaps) {
                this.fillDirectionalGaps();
            }
        },

        deleteItem: function(item) {
            const index = this._getItemIndex(item);

            if (index <= -1) {
                return;
            }

            if (this._isANumber(item.x) && this._isANumber(item.y)) {
                this._deleteItemPositionFromGrid(item);
            }

            this.items.splice(index, 1);

            if (!this._options.allowDirectionalGaps) {
                this.fillDirectionalGaps();
                return;
            }

            if (item.isDisplacing) {
                const itemsUnder = this.getItemsBelowOrOverlapping(item);
                this._shiftItems(itemsUnder, -item.h);
            }

        },

        getItemsBelowOrOverlapping: function(item) {
            this._sortItemsByPosition();
            return this.items.filter(item_ => {
                const isBelow = this._isBelowItem(item, item_);
                return item_.id !== item.id && isBelow;
            });
        },

        addItem: function(item) {
            if (this._isANumber(item.x) && this._isANumber(item.y)) {
                this.solveCollisionWithNewItemPositionning(item);
            }
            this.items.push(item);
            this.positionPositionlessElements();
            if (!this._options.allowDirectionalGaps) {
                this.fillDirectionalGaps();
            }
        },

        fillDirectionalGaps: function() {
            this._sortItemsByPosition();
            for (const item of this.items) {
                const position = this._getItemPosition(item);
                const newPosition = this._findPositionForItemFillingHorizontalGap(item, position);
                if (position.x !== newPosition[0]) {
                    this._updateItemPosition(item, newPosition);
                }
            }
            this._removeOverflowingColumns();
        },

        hasDirectionalGaps: function() {
            this._sortItemsByPosition();
            for (const item of this.items) {
                const position = this._getItemPosition(item);
                const newPosition = this._findPositionForItemFillingHorizontalGap(item, position);
                if (position.x !== newPosition[0]) {
                    return true;
                }
            }
            return false;
        },

        compactItems: function() {
            this.fillDirectionalGaps();
            this._sortItemsByPosition();
            this._resetGrid();

            // Locked items must be positioned first if allowing gaps
            if (this._options.allowDirectionalGaps) {
                for (const item of this.items) {
                    if (item.locked) {
                        const lockedPosition = this._getItemPosition(item);
                        this._updateItemPosition(item, [lockedPosition.x, lockedPosition.y]);
                    }
                }
            }

            for (const item of this.items) {
                if (item.locked && this._options.allowDirectionalGaps) {
                    continue;
                }
                // Resolve position for unlocked items
                const newPosition = this.findPositionForItem(item, { x: 0 , y: 0 });
                this._updateItemPosition(item, newPosition);
            }
        },

        _solveCollisionDisplacingItem(newItem) {
            const itemsUnder = this.getItemsBelowOrOverlapping(newItem);
            const maxOffset = itemsUnder.reduce((offset, itemUnder) => {
                return Math.max(
                    Math.max(
                        newItem.y - itemUnder.y,
                        0
                    ) + newItem.h,
                    offset
                );
            }, 0);

            this._shiftItems(itemsUnder, maxOffset);
            const newPosition = this._getItemPosition(newItem);
            // Set the new item at its desired position
            this._updateItemPosition(newItem, [newPosition.x, newPosition.y]);
        },

        solveCollisionWithNewItemPositionning(newItem){
            if (newItem.isDisplacing) {
                this._solveCollisionDisplacingItem(newItem);
                return;
            }

            // If new item is colliding with a locked item, we move the new item down
            for (const item of this.items) {
                if (item.locked && this._itemsAreColliding(newItem, item)) {
                    newItem.x = undefined;
                    newItem.y = undefined;
                    return;
                };
            }

            /*
             * If new item is colliding with an unlocked item or the new item is colliding with a lock item and has overrideLockItems,
             * we move the unlock items down
             */
            for (const item of this.items) {
                if (!item.locked && this._itemsAreColliding(newItem, item)) {
                    this._deleteItemPositionFromGrid(item);
                    this._setItemPosition(item, [undefined, undefined]);
                }
            }

            // Set the new item at its desired position
            const position = this._getItemPosition(newItem);
            this._updateItemPosition(newItem, [position.x, position.y]);
        },

        getChangedItems: function(initialItems, idAttribute) {
            /**
             * Compare the current items against a previous snapshot and return only
             * the ones that changed their attributes in the meantime. This includes both
             * position (x, y) and size (w, h)
             *
             * Since both their position and size can change, the items need an
             * additional identifier attribute to match them with their previous state
             */
            const changedItems = [];

            for (let i = 0; i < initialItems.length; i++) {
                const item = this._getItemByAttribute(idAttribute, initialItems[i][idAttribute]);

                if (item) {
                    if (
                        item.x !== initialItems[i].x ||
            item.y !== initialItems[i].y ||
            item.w !== initialItems[i].w ||
            item.h !== initialItems[i].h
                    ) {
                        changedItems.push(item);
                    }
                }
            }

            return changedItems;
        },

        _isANumber: function(number) {
            return typeof number === 'number' && !Number.isNaN(number);
        },

        _sortItemsByPosition: function() {
            this.items.sort(
                function(item1, item2) {
                    const position1 = this._getItemPosition(item1),
                        position2 = this._getItemPosition(item2);

                    // Try to preserve columns.
                    if (position1.x != position2.x) {
                        return position1.x - position2.x;
                    }

                    if (position1.y != position2.y) {
                        return position1.y - position2.y;
                    }

                    if (item1.isDisplacing && !item2.isDisplacing) {
                        return -1;
                    }

                    if (!item1.isDisplacing && item2.isDisplacing) {
                        return 1;
                    }

                    // The items are placed on the same position.
                    return 0;
                }.bind(this)
            );
        },

        _resolveItemCollisionsAfterLaneChange: function() {
            for (const item of this.items) {
                const position = this._getItemPosition(item);
                this._ensureColumns(position.x + position.w);
                let newPosition = this.findPositionForItem(item, position);
                // If the item is not fitting and thus added to a new column, we try reducing it size by 1 to see if it fits
                if (newPosition[1] === 0 && position.y !== 0) {
                    const minHeight = this._getItemMinHeight(item);
                    this._setItemSize(item, { w: position.w, h: Math.max(minHeight, position.h - 1) });
                    const testPosition = this.findPositionForItem(item, position);
                    if (testPosition[1] !== 0) {
                        newPosition = testPosition;
                    } else {
                        this._setItemSize(item, { w: position.w, h: position.h });
                    }
                }
                this._updateItemPosition(item, newPosition);
            }
        },

        _adjustItemsForLaneChange: function(previousLanes) {
            this._adjustSizeOfFullWidthItems();

            if (this._options.lanes >= previousLanes) {
                return;
            }

            if (!this._hasOverflowingItems()) {
                return;
            }

            const { minLaneIndex, maxLaneIndex } = this._getMinAndMaxLaneIndexUsedByItems();
            const minLanesRequired = maxLaneIndex - minLaneIndex;
            const canShiftItemsToFit = this._options.lanes >= minLanesRequired;
            if (canShiftItemsToFit) {
                const offset = this._options.lanes - minLaneIndex - minLanesRequired;
                this._shiftAllItems(0, offset);
                return;
            }

            this._resizeItemsProportionally(minLaneIndex, minLanesRequired);
        },

        _resizeItemsProportionally: function(minLaneIndex, minLanesRequired) {
            // Let's remove potential blank space on the left before resizing
            if (minLaneIndex > 0) {
                this._shiftAllItems(0, -minLaneIndex);
            }
            const delta = this._options.lanes / minLanesRequired;

            // Round item position and size to new lanes
            this.items.forEach(item => {
                const position = this._getItemPosition(item);
                const minHeight = this._getItemMinHeight(item);
                const newSize = {
                    w: position.w,
                    h: Math.min(this._options.lanes, Math.max(minHeight, Math.round(position.h * delta)))
                };
                this._setItemSize(item, newSize);

                const newPosition = [
                    position.x,
                    Math.floor(position.y * delta)
                ];
                this._setItemPosition(item, newPosition);
            });
        },

        _restoreItemSizeAndPositionFromSnapshot: function() {
            this.items.forEach(item => {
                const ref = this.itemSizeAndPositionSnapshot.positionAndSizeById.get(item.id) || item;
                item.w = ref.w;
                item.h = ref.h;
                item.x = ref.x;
                item.y = ref.y;
            });
        },

        _shiftItems: function(items, dx = 0, dy = 0) {
            items.forEach(item => {
                const position = this._getItemPosition(item);
                this._updateItemPosition(item, [position.x + dx, position.y + dy]);
            });
        },

        _shiftAllItems: function(dx = 0, dy = 0) {
            this._shiftItems(this.items, dx, dy);
        },

        _hasOverflowingItems: function() {
            return this.items.some(item => this._isItemOverflowing(item));
        },

        _getMinAndMaxLaneIndexUsedByItems: function() {
            return this.items.reduce((acc, item) => {
                const position = this._getItemPosition(item);
                acc.minLaneIndex = Math.min(acc.minLaneIndex, position.y);
                acc.maxLaneIndex = Math.max(acc.maxLaneIndex, position.y + position.h);
                return acc;
            }, { minLaneIndex: Infinity, maxLaneIndex: -Infinity });
        },

        _adjustSizeOfFullWidthItems: function() {
            /**
             * Some items can have 100% height or 100% width. Those dimensions are
             * expressed as 0. We need to ensure a valid width and height for each of
             * those items as the number of items per lane.
             */

            for (let i = 0; i < this.items.length; i++) {
                const item = this.items[i];
                /*
                 * This can happen only the first time items are checked.
                 * We need the property to have a value for all the items so that the
                 * `cloneItems` method will merge the properties properly. If we only set
                 * it to the items that need it then the following can happen:
                 *
                 * cloneItems([{id: 1, autoSize: true}, {id: 2}],
                 *            [{id: 2}, {id: 1, autoSize: true}]);
                 *
                 * will result in
                 *
                 * [{id: 1, autoSize: true}, {id: 2, autoSize: true}]
                 */
                if (item.autoSize === undefined) {
                    item.autoSize = item.w === 0 || item.h === 0;
                }
                if (item.autoSize) {
                    if (this._options.direction === 'horizontal') {
                        item.h = this._options.lanes;
                    } else {
                        item.w = this._options.lanes;
                    }
                }
            }
        },

        _resetGrid: function() {
            this.grid = [];
        },

        _itemFitsAtPosition: function(item, newPosition) {
            /**
             * Check that an item wouldn't overlap with another one if placed at a
             * certain position within the grid
             */

            const position = this._getItemPosition(item);
            let x, y;

            // No coordonate can be negative
            if (newPosition[0] < 0 || newPosition[1] < 0) {
                return false;
            }

            // Make sure the item isn't larger than the entire grid
            if (newPosition[1] + position.h > this._options.lanes) {
                return false;
            }

            /*
             * Make sure the position doesn't overlap with an already positioned
             * item.
             */
            for (x = newPosition[0]; x < newPosition[0] + position.w; x++) {
                const col = this.grid[x];

                // Surely a column that hasn't even been created yet is available
                if (!col) {
                    continue;
                }

                for (y = newPosition[1]; y < newPosition[1] + position.h; y++) {
                    /*
                     * Any space occupied by an item can continue to be occupied by the
                     * same item.
                     */
                    if (col[y] && col[y] !== item) {
                        return false;
                    }
                }
            }

            return true;
        },

        _updateItemPosition: function(item, position) {
            if (this._isANumber(item.x) && this._isANumber(item.y)) {
                this._deleteItemPositionFromGrid(item);
            }

            this._setItemPosition(item, position);
            this._adjustSizeOfOverflowingItem(item);
            this._markItemPositionToGrid(item);
        },

        _adjustSizeOfOverflowingItem: function(item) {
            if (this._isItemOverflowing(item)) {
                const position = this._getItemPosition(item);
                const newHeight = this._options.lanes - position.y;
                this._setItemSize(item, { ...position, h: newHeight });
            }
        },

        _isItemOverflowing: function(item) {
            const position = this._getItemPosition(item);
            return position.h + position.y > this._options.lanes;
        },

        _updateItemSize: function(item, width, height) {
            /**
             * @param {Object} item A reference to a grid item.
             * @param {Number} width The new width.
             * @param {Number} height The new height.
             */

            if (this._isANumber(item.x) && this._isANumber(item.y)) {
                this._deleteItemPositionFromGrid(item);
            }

            const initialHeight = item.h;

            item.w = width;
            item.h = height;

            if (item.isDisplacing) {
                const itemsUnder = this.getItemsBelowOrOverlapping(item);
                this._shiftItems(itemsUnder, (height - initialHeight ));
            }

            this._markItemPositionToGrid(item);
        },

        _markItemPositionToGrid: function(item) {
            /**
             * Mark the grid cells that are occupied by an item. This prevents items
             * from overlapping in the grid
             */

            const position = this._getItemPosition(item);
            let x, y;

            // Ensure that the grid has enough columns to accomodate the current item.
            this._ensureColumns(position.x + position.w);

            for (x = position.x; x < position.x + position.w; x++) {
                for (y = position.y; y < position.y + position.h; y++) {
                    this.grid[x][y] = item;
                }
            }
        },

        _deleteItemPositionFromGrid: function(item) {
            const position = this._getItemPosition(item);
            let x, y;

            for (x = position.x; x < position.x + position.w; x++) {
                /*
                 * It can happen to try to remove an item from a position not generated
                 * in the grid, probably when loading a persisted grid of items. No need
                 * to create a column to be able to remove something from it, though
                 */
                if (!this.grid[x]) {
                    continue;
                }

                for (y = position.y; y < position.y + position.h; y++) {
                    /*
                     * Don't clear the cell if it's been occupied by a different widget in
                     * the meantime (e.g. when an item has been moved over this one, and
                     * thus by continuing to clear this item's previous position you would
                     * cancel the first item's move, leaving it without any position even)
                     */
                    if (this.grid[x][y] == item) {
                        this.grid[x][y] = null;
                    }
                }
            }
        },

        _ensureColumns: function(N) {
            /**
             * Ensure that the grid has at least N columns available.
             */
            let i;
            for (i = 0; i < N; i++) {
                if (!this.grid[i]) {
                    this.grid.push(new GridCol(this._options.lanes));
                }
            }
        },

        _removeOverflowingColumns: function() {
            for (let i = this.grid.length - 1; i >= 0; i--) {
                if (this.grid[i].length === 0 || this.grid[i].every(row => row == null)) {
                    this.grid.splice(i, 1);
                } else {
                    break;
                }
            }
        },

        _getItemsCollidingWithItem: function(item) {
            const collidingItems = [];
            for (let i = 0; i < this.items.length; i++) {
                if (item != this.items[i] && this._itemsAreColliding(item, this.items[i])) {
                    collidingItems.push(i);
                }
            }
            return collidingItems;
        },

        _itemsAreColliding: function(item1, item2) {
            const position1 = this._getItemPosition(item1),
                position2 = this._getItemPosition(item2);

            return this._arePositionsColliding(position1, position2);
        },

        _isBelowItem: function(referenceItem, candidateItem) {
            const referencePosition = this._getItemPosition(referenceItem);
            const candidatePosition = this._getItemPosition(candidateItem);

            const candidateIsAtTheLeft = candidatePosition.x + candidatePosition.w > referencePosition.x;

            const candidateTopHasVerticalColliding = (
                candidatePosition.y + candidatePosition.h - referencePosition.y > 0
                && referencePosition.y + referencePosition.h - candidatePosition.y - candidatePosition.h > 0
            );

            const candidateBottomHasVerticalColliding = (
                referencePosition.y + referencePosition.h - candidatePosition.y > 0
                && candidatePosition.y + candidatePosition.h - referencePosition.y - referencePosition.h >= 0
            );

            const candidateHasVerticalColliding = (
                candidateTopHasVerticalColliding
                || candidateBottomHasVerticalColliding
            );

            return candidateIsAtTheLeft && candidateHasVerticalColliding;
        },

        _arePositionsColliding: function(position1, position2) {
            return !(
                position2.x >= position1.x + position1.w ||
        position2.x + position2.w <= position1.x ||
        position2.y >= position1.y + position1.h ||
        position2.y + position2.h <= position1.y
            );
        },

        _resolveCollisions: function(item, previousItemPosition = null) {
            if (!this._tryToResolveCollisionsLocally(item, previousItemPosition)) {
                this._pullItemsToLeft(item);
            }
        },

        _tryToSwapWithRightCollidingItemWhenMovingItemRight: function(item, previousItemPosition, collidingItemIndex) {
            const collidingItemPosition = this._getItemPosition(this.items[collidingItemIndex]);
            const wasItemOnTheLeftOfCollidingItem = previousItemPosition.x < collidingItemPosition.x;
            if (!wasItemOnTheLeftOfCollidingItem) {
                return false;
            }

            const gridClone = new GridList([], this._options);
            GridList.cloneItems(this.items, gridClone.items);
            gridClone.generateGrid();

            const itemIndex = this._getItemIndex(item);

            const itemClone = gridClone.items[itemIndex];
            const collidingItem = gridClone.items[collidingItemIndex];

            gridClone._updateItemPosition(itemClone, [previousItemPosition.x, previousItemPosition.y]);
            const itemPosition = gridClone._getItemPosition(itemClone);
            this._updateItemPosition(collidingItem, [itemPosition.x, collidingItemPosition.y]);
            this._updateItemPosition(itemClone, [collidingItemPosition.x + collidingItemPosition.w, itemPosition.y]);
            if (gridClone._getItemsCollidingWithItem(itemClone).length === 0 && gridClone._getItemsCollidingWithItem(collidingItem).length === 0) {
                GridList.cloneItems(gridClone.items, this.items);
                this.generateGrid();
                return true;
            }
            return false;
        },

        _tryToResolveCollisionsLocally: function(item, previousItemPosition = null) {
            /**
             * Attempt to resolve the collisions after moving a an item over one or more
             * other items within the grid, by shifting the position of the colliding
             * items around the moving one. This might result in subsequent collisions,
             * in which case we will revert all position permutations. To be able to
             * revert to the initial item positions, we create a virtual grid in the
             * process
             */
            const collidingItems = this._getItemsCollidingWithItem(item);
            if (!collidingItems.length) {
                return true;
            }

            /*
             * In non-allowDirectionalGaps mode, items are compacted toward the left (or toward the top in vertical mode).
             * This makes it difficult to move a tile to the right (or downward in vertical mode) if it immediately
             * collides with another one.
             *
             * When such a collision occurs, and the dragged item comes from the left (or from above in vertical mode),
             * this method attempts to resolve the collision by swapping the dragged item with the colliding one.
             * The swap is only performed if it results in a valid grid state with no further collisions in a virtual grid.
             */
            if (collidingItems.length === 1 && previousItemPosition != null && !this._options.allowDirectionalGaps) {
                if (this._tryToSwapWithRightCollidingItemWhenMovingItemRight(item, previousItemPosition, collidingItems[0])){
                    return true;
                };
            }

            const gridClone = new GridList([], this._options);
            const itemIndex = this._getItemIndex(item);
            let leftOfItem;
            let rightOfItem;
            let aboveOfItem;
            let belowOfItem;

            GridList.cloneItems(this.items, gridClone.items);
            gridClone.generateGrid();

            for (let i = 0; i < collidingItems.length; i++) {
                const itemClone = gridClone.items[itemIndex];
                const collidingItem = gridClone.items[collidingItems[i]];
                const collidingPosition = this._getItemPosition(collidingItem);

                /*
                 * Checking if collidingItem is still colliding with item b/c _itemFitsAtPositionWhithShiftingRight
                 * may resolve several collisions simultaneously, for instance if we move 03 to
                 * position 00 in following example
                 *
                 *  #|  0  1  2  3  4
                 *  -------------------
                 *  0| 01 02 -- 03 03
                 *  1| -- 02 -- 03 03
                 *
                 */
                const ciCollidingItems = gridClone._getItemsCollidingWithItem(collidingItem);
                if (ciCollidingItems.indexOf(itemIndex) == -1) {
                    continue;
                }

                /*
                 * We use a simple algorithm for moving items around when collisions occur:
                 * In this prioritized order, we try to move a colliding item around the
                 * moving one:
                 * 1. to its left side
                 * 2. above it
                 * 3. under it
                 * 4. to its right side
                 */
                const position = this._getItemPosition(item);

                leftOfItem = [position.x - collidingPosition.w, collidingPosition.y]; // visually above if grid is vertical
                rightOfItem = [position.x + position.w, collidingPosition.y]; // visually below if grid is vertical
                aboveOfItem = [collidingPosition.x, position.y - collidingPosition.h]; // visually left if grid is vertical
                belowOfItem = [collidingPosition.x, position.y + position.h]; // visually right if grid is vertical

                if (gridClone._itemFitsAtPosition(collidingItem, belowOfItem)) {
                    gridClone._updateItemPosition(collidingItem, belowOfItem);
                } else if (gridClone._itemFitsAtPosition(collidingItem, aboveOfItem)) {
                    gridClone._updateItemPosition(collidingItem, aboveOfItem);
                } else if (gridClone._itemFitsAtPositionWhithMultipleVerticalShifts(collidingItem, itemClone, 'down')) {
                    /*
                     *  visually right if grid is vertical
                     * nothing to do here
                     */
                } else if (gridClone._itemFitsAtPositionWhithMultipleVerticalShifts(collidingItem, itemClone, 'up')) {
                    /*
                     *  visually right if grid is vertical
                     * nothing to do here
                     */
                } else if (gridClone._itemFitsAtPosition(collidingItem, rightOfItem)) {
                    gridClone._updateItemPosition(collidingItem, rightOfItem);
                } else if (gridClone._itemFitsAtPosition(collidingItem, leftOfItem)) {
                    gridClone._updateItemPosition(collidingItem, leftOfItem);
                } else {
                    /*
                     * Collisions failed, we must use the pullItemsToLeft method to arrange
                     * the other items around this item with fixed position. This is our
                     * plan B for when local collision resolving fails.
                     */
                    return false;
                }
            }

            /*
             * If we reached this point it means we managed to resolve the collisions
             * from one single iteration, just by moving the colliding items around. So
             * we accept this scenario and marge the brached-out grid instance into the
             * original one
             */
            GridList.cloneItems(gridClone.items, this.items);
            this.generateGrid();
            return true;
        },

        _itemFitsAtPositionWhithMultipleVerticalShifts: function(collidingItem, justMovedItem, direction) {
            const verticalMove = direction == 'up' ? -1 : 1;

            //Creating a virtual grid in the process
            const gridClone = new GridList([], this._options);

            GridList.cloneItems(this.items, gridClone.items);
            gridClone.generateGrid();
            const collidingItemClone = gridClone.items[this._getItemIndex(collidingItem)];
            const justMovedItemClone = gridClone.items[this._getItemIndex(justMovedItem)];

            // Return true if item is on the set of columns occupied by justMoveItemPosition
            const isItemOnRightPerimeter = function(item) {
                const position = gridClone._getItemPosition(item);
                return (
                    position.x >= justMoveItemPosition.x &&
          position.x + position.w - 1 <= justMoveItemPosition.x + justMoveItemPosition.w - 1
                );
            };

            // Return true if shifting item to the right won't create a new line on the grid
            const wontGetOutOfBundaries = function(item) {
                const position = gridClone._getItemPosition(item);
                return position.y + position.h - 1 + verticalMove < gridClone._options.lanes && position.y + verticalMove >= 0;
            };

            /*
             * Shift an item below (and recursively any other item colliding after the shift) if:
             * - The item is on the right perimeter
             * - The item won't get out of bundaries
             * - Any other shift necessary to resolve resulting collision respects those two previous conditions
             */
            const shiftItemVertically = function(item) {
                //Looping out conditions
                if (!isItemOnRightPerimeter(item) || !wontGetOutOfBundaries(item) || item.locked) {
                    return false;
                }

                //Shifting below
                const oldPosition = gridClone._getItemPosition(item);

                const newPosition = {
                    x: oldPosition.x,
                    y: oldPosition.y + verticalMove,
                    w: item.w,
                    h: item.h
                };

                gridClone._updateItemPosition(item, [newPosition.x, newPosition.y]);

                //Resolving collisions
                const collidingItems = gridClone._getItemsCollidingWithItem(item);
                for (let i = 0; i < collidingItems.length; i++) {
                    const ci = gridClone.items[collidingItems[i]];
                    if (ci != justMovedItemClone && !shiftItemVertically(ci)) {
                        return false;
                    }
                }

                //No exit condition satisfied nor nested shifting impossible ?
                return true;
            };

            //Getting positions
            const collidingPosition = gridClone._getItemPosition(collidingItemClone);
            const justMoveItemPosition = gridClone._getItemPosition(justMovedItemClone);

            //Shifting to the right as many times as necessary to resolve original conflict
            let nbShiftsNecessary;
            if (verticalMove == 1) {
                nbShiftsNecessary = justMoveItemPosition.y + justMoveItemPosition.h - collidingPosition.y;
            } else {
                nbShiftsNecessary = collidingPosition.y + collidingPosition.h - justMoveItemPosition.y;
            }

            for (let i = 0; i < nbShiftsNecessary; i++) {
                if (!shiftItemVertically(collidingItemClone)) {
                    return false;
                }
            }

            /*
             * If we reached this point it means we managed to shift collidingItem enough times
             * to resolve original collision and that any other resulting collision was solved
             * using the same method without violating the two following conditions :
             * - Not creating new column
             * - Not affecting any item outside of the set of lines occupied by justMovedItem
             */
            GridList.cloneItems(gridClone.items, this.items);
            this.generateGrid();
            return true;
        },

        _pullItemsToLeft: function(fixedItem) {
            /**
             * Build the grid from scratch, by using the current item positions and
             * pulling them as much to the left as possible, removing as space between
             * them as possible.
             *
             * If a "fixed item" is provided, its position will be kept intact and the
             * rest of the items will be layed around it.
             */

            // Start a fresh grid with the fixed item already placed inside
            this._sortItemsByPosition();
            this._resetGrid();

            // Start the grid with the fixed item as the first positioned item
            if (fixedItem) {
                const fixedPosition = this._getItemPosition(fixedItem);
                this._updateItemPosition(fixedItem, [fixedPosition.x, fixedPosition.y]);
            }

            for (const item of this.items) {
                if (item.locked) {
                    // Locked items must be positioned first
                    const lockedPosition = this._getItemPosition(item);
                    this._updateItemPosition(item, [lockedPosition.x, lockedPosition.y]);
                }
            }

            for (const item of this.items) {
                if (!item.locked && fixedItem !== item) {
                    // Resolve position for unlocked items
                    const position = this._getItemPosition(item);
                    const newPosition = this.findPositionForItem(item, { x: position.x , y: 0 }, position.y);
                    this._updateItemPosition(item, newPosition);
                }
            }
        },

        _findLeftMostPositionForItem: function(item) {
            /**
             * When pulling items to the left, we need to find the leftmost position for
             * an item, with two considerations in mind:
             * - preserving its current row
             * - preserving the previous horizontal order between items
             */

            let tail = 0;
            const position = this._getItemPosition(item);

            for (let i = 0; i < this.grid.length; i++) {
                for (let j = position.y; j < position.y + position.h; j++) {
                    const otherItem = this.grid[i][j];

                    if (!otherItem) {
                        continue;
                    }

                    const otherPosition = this._getItemPosition(otherItem);

                    if (this.items.indexOf(otherItem) < this.items.indexOf(item)) {
                        tail = otherPosition.x + otherPosition.w;
                    }
                }
            }

            return tail;
        },

        _findPositionForItemFillingHorizontalGap: function(item, start) {
            let currentX = start.x;
            for (let i = currentX; i >= 0; i--) {
                if (this._itemFitsAtPosition(item, [i, start.y])) {
                    currentX = i;
                } else {
                    break;
                }
            }
            return [currentX, start.y];
        },

        _getItemByAttribute: function(key, value) {
            for (let i = 0; i < this.items.length; i++) {
                if (this.items[i][key] === value) {
                    return this.items[i];
                }
            }
            return null;
        },

        _padNumber: function(nr, prefix) {
            // Currently works for 2-digit numbers (<100)
            return nr >= 10 ? nr : prefix + nr;
        },

        _getItemPosition: function(item) {
            /**
             * If the direction is vertical we need to rotate the grid 90 deg to the
             * left. Thus, we simulate the fact that items are being pulled to the top.
             *
             * Since the items have widths and heights, if we apply the classic
             * counter-clockwise 90 deg rotation
             *
             *     [0 -1]
             *     [1  0]
             *
             * then the top left point of an item will become the bottom left point of
             * the rotated item. To adjust for this, we need to subtract from the y
             * position the height of the original item - the width of the rotated item.
             *
             * However, if we do this then we'll reverse some actions: resizing the
             * width of an item will stretch the item to the left instead of to the
             * right; resizing an item that doesn't fit into the grid will push the
             * items around it instead of going on a new row, etc.
             *
             * We found it better to do a vertical flip of the grid after rotating it.
             * This restores the direction of the actions and greatly simplifies the
             * transformations.
             */

            if (this._options.direction === 'horizontal') {
                return item;
            } else {
                return {
                    x: item.y,
                    y: item.x,
                    w: item.h,
                    h: item.w
                };
            }
        },

        _getItemMinHeight: function(item) {
            if (this._options.direction === 'horizontal') {
                return item.minHeight || 1;
            } else {
                return item.minWidth || 1;
            }
        },

        _getItemIndex: function(item) {
            return this.items.indexOf(item);
        },

        _setItemPosition: function(item, position) {
            /**
             * See _getItemPosition.
             */

            if (this._options.direction === 'horizontal') {
                item.x = position[0];
                item.y = position[1];
            } else {
                /*
                 * We're supposed to subtract the rotated item's height which is actually
                 * the non-rotated item's width.
                 */
                item.x = position[1];
                item.y = position[0];
            }
        },

        _setItemSize: function(item, dimensions) {
            if (this._options.direction === 'horizontal') {
                item.h = dimensions.h;
                item.w = dimensions.w;
            } else {
                item.h = dimensions.w;
                item.w = dimensions.h;
            }
        }
    };

    const GridCol = function(lanes) {
        for (let i = 0; i < lanes; i++) {
            this.push(null);
        }
    };

    // Extend the Array prototype
    GridCol.prototype = [];

    // This module will have direct access to the GridList class
    return GridList;
});
