import traceback, logging

from dataiku.base.utils import watch_stdin, get_json_friendly_error
from dataiku.base.socket_block_link import JavaLink, parse_javalink_args
from pathlib import Path
try:
    from dataiku_business_app.upgrade import upgrade_instance
except ImportError as e:
    raise Exception("Unable to import Business Application upgrade_instance function. Make sure the Business Application is defining an upgrade_instance function: " + str(e))

# socket-based connection to backend
def serve(port, secret, server_cert=None):
    link = JavaLink(port, secret, server_cert=server_cert)
    # initiate connection
    link.connect()
    # get work to do
    command = link.read_json()
    try:
        # get task and dispatch work to exporter
        task = command["task"]
        result = {}
        if task == "upgrade":
            old_instance_path = Path(command["oldInstanceDirectory"])
            new_instance_path = Path(command["newInstanceDirectory"])
            print(f"Config: old_instance_path={old_instance_path}, new_instance_path={new_instance_path}")

            upgrade_instance_result = upgrade_instance(old_instance_path, new_instance_path)

            result["success"] = upgrade_instance_result.get("success", True)
            result["messages"] = upgrade_instance_result.get("messages", [])

            # Temporary fix while Business Applications are upgraded to only use "messages"
            for msg in upgrade_instance_result.get("info_messages") or []:
                result["messages"].append({
                    "severity": "INFO",
                    "title": "Upgrade script info",
                    "details": msg,
                })
            for msg in upgrade_instance_result.get("warning_messages") or []:
                result["messages"].append({
                    "severity": "WARNING",
                    "title": "Upgrade script warning",
                    "details": msg,
                })
            error_message = upgrade_instance_result.get("error_message")
            if error_message:
                result["messages"].append({
                    "severity": "ERROR",
                    "title": "Upgrade script error",
                    "details": error_message,
                })

            link.send_json(result)
        else:
            raise Exception("Unexpected task %s" % task)
    except:
        traceback.print_exc()
        link.send_string('') # mark failure
        link.send_json(get_json_friendly_error())
    finally:
        # done
        link.close()

if __name__ == "__main__":
    logging.basicConfig(level=logging.DEBUG, format='%(asctime)s %(levelname)s %(message)s')
    watch_stdin()
    port, secret, server_cert = parse_javalink_args()
    serve(port, secret, server_cert=server_cert)

