# -*- coding: utf-8 -*-
from dataiku.doctor.docgen.renderer.document_handler import DocumentHandler
from dataiku.doctor.docgen.renderer.renderer_helpers import RendererHelpers
from dataiku.doctor.docgen.common.placeholder import TablePlaceholder
from dataiku.doctor.docgen.extractor.docx_parser import DocxParser
from dataiku.doctor.docgen.renderer.widget import Table, Text, PuppeteerContent
from dataiku.doctor.docgen.renderer.puppeteer_extractor import MultiExtractor
import logging

logger = logging.getLogger(__name__)


class TreeRenderer(object):
    """
    This is the renderer for the generic tree workflow.
    It works in a single reccursive process.
    Placeholders are read sequentially and replaced by their value if simple, removed / kept if conditional, or repeated if foreach
    When a placeholder is nested (if ou foreach), its content is resolved just after the container is resolved
    """

    def __init__(self, project_key):
        self.docx_parser = DocxParser()
        self.helpers = RendererHelpers()
        self.multi_extractor = MultiExtractor(project_key)

    def __get_placeholder_context__(self, placeholder):
        placeholder_type = placeholder["type"]

        if placeholder_type == "JSON_TABLE":
            return Table(placeholder["value"])
        elif placeholder_type == "JSON_TEXT":
            return Text(placeholder["value"])
        elif placeholder_type == "EXPORT":
            if "exportId" not in placeholder: # exportId missing is the marker for a failed export
                logger.warning("Unable to replace %s because export failed" % (placeholder["name"]))
                return None
            extractor = self.multi_extractor.get_extractor(placeholder["exportId"], placeholder["exportType"])
            # the second argument of PuppeteerContent actually doesn't matter for SINGLE_EXPORT, we use the name for log purpose
            return PuppeteerContent(extractor, placeholder["name"])
        else:
            logger.error("Unsupported resolved placeholder type: %s for %s" % (placeholder_type, placeholder["name"]))

        return None


    def __resolve_foreach__(self, document, placeholder, resolved_placeholder, font_reference):
        elements_to_repeat = DocumentHandler.between(document, placeholder)
        
        # we interate in reverse order because we insert content just after the opening tag at each loop
        for resolved_iter in reversed(resolved_placeholder['children']): 
            resolved_iter = resolved_iter['children'] # unnest the VIRTUAL container to get the resolved content

            # copy all elements. reversed because we insert each element after the opening tag of the foreach tag
            inserted_elements = [DocumentHandler.copy_element_after_paragraph(placeholder.start.paragraph, el) for el in reversed(elements_to_repeat)]
            inserted_elements.reverse() # put inserted elements back in normal order

            # parse placeholders in new inserted elements
            inserted_placeholders_flat = self.docx_parser.parse_text_and_tables(inserted_elements)
            inserted_placeholders = self.docx_parser.build_placeholders_tree(inserted_placeholders_flat)

            # resolve new placeholders with current iteration values
            self.__resolve_placeholder_tree_sub__(document, resolved_iter, inserted_placeholders, font_reference)
        
        # remove original template
        for el in elements_to_repeat:
            DocumentHandler.delete_element(el)
                

    def __resolve_placeholder_tree_sub__(self, document, resolved_placeholders, placeholders_tree, font_reference):
        # we resolve starting by the end in order to preserve the placeholder references
        i = len(placeholders_tree) - 1
        while i >= 0:
            placeholder = placeholders_tree[i]
            resolved_placeholder = resolved_placeholders[i]
            resolved_type = resolved_placeholder["type"]

            if resolved_type == "REMOVE":
                self.helpers.replace_placeholder(document, placeholder, None, font_reference)

            elif resolved_type == "STRIP": # can only happen for IF blocks so it is guaranteed to be a NestedPlaceholder
                self.__resolve_placeholder_tree_sub__(document, resolved_placeholder["children"], placeholder.children, font_reference)
                self.helpers.strip_block_placeholder(placeholder)

            elif resolved_type == "VIRTUAL": # can only happen for TablePlaceholder (because the only other occurrence of VIRTUAL is inside the foreach loop)
                if isinstance(placeholder, TablePlaceholder):
                    cell = placeholder.table.cell(placeholder.row_idx, placeholder.col_idx)
                    self.__resolve_placeholder_tree_sub__(cell, resolved_placeholder["children"], placeholder.placeholders, font_reference)

            elif resolved_type == "REPEAT": # can only happen for FOREACH blocks so it is guaranteed to be a NestedPlaceholder
                self.__resolve_foreach__(document, placeholder, resolved_placeholder, font_reference)
                self.helpers.strip_block_placeholder(placeholder)

            else: # these are the classic one-to-one replacements JSON_TABLE, JSON_TEXT (and in the future maybe PUPPETEER)
                widget = self.__get_placeholder_context__(resolved_placeholder)
                self.helpers.replace_placeholder(document, placeholder, widget, font_reference)
            
            i -= 1

    def __resolve_tree_sub__(self, document, section, resolved_ptr, resolved_placeholders):
        """
        Wrapper around __resolve_placeholder_tree_sub__ that extract the right placeholders & resolved results, and update the resolved_ptr
        """
        placeholders = self.docx_parser.extract_placeholder_tree(section)
        resolved = resolved_placeholders[resolved_ptr:resolved_ptr+len(placeholders)]
        resolved_ptr += len(placeholders)
        self.__resolve_placeholder_tree_sub__(section, resolved, placeholders, document)
        return resolved_ptr
    
    def resolve_placeholders(self, document, resolved_placeholders):
        """
        Resolve the placeholders in the document, including the headers and the footers.
        """
        # Important assumption about order of operations here:
        # The order in which we parse / resolve main part, header and footers MUST match the order used during the parsing phase.

        logger.debug(resolved_placeholders)

        # resolved_ptr tracks which placeholders have been consumed by the resolving process already 

        # we resolve the main text part with the first chunk of resolved data
        resolved_ptr = self.__resolve_tree_sub__(document, document, 0, resolved_placeholders)

        # resolve each section based on it's own placeholder tree
        for section_number in range(len(document.sections)):
            # resolve the header then the footer for this section
            resolved_ptr = self.__resolve_tree_sub__(document, document.sections[section_number].header, resolved_ptr, resolved_placeholders)
            resolved_ptr = self.__resolve_tree_sub__(document, document.sections[section_number].footer, resolved_ptr, resolved_placeholders)

        return document
