import logging

import numpy as np
import pandas as pd

from dataiku.doctor.timeseries.models.statistical.base_estimator import DkuStatisticalEstimator
from dataiku.doctor.timeseries.utils import ModelForecast

logger = logging.getLogger(__name__)

class DkuStatsforecastBaseEstimator(DkuStatisticalEstimator):

    def __init__(
            self,
            frequency,
            time_variable,
            prediction_length,
            target_variable,
            timeseries_identifier_columns,
            monthly_day_alignment=None,
    ):
        super(DkuStatsforecastBaseEstimator, self).__init__(
            frequency=frequency,
            time_variable=time_variable,
            prediction_length=prediction_length,
            target_variable=target_variable,
            timeseries_identifier_columns=timeseries_identifier_columns,
            monthly_day_alignment=monthly_day_alignment,
        )

    def _build_statsforecast_model(self):
        raise NotImplementedError()

    def _get_statsforecast_model_name(self):
        raise NotImplementedError()

    def _fit_single(self, target_values, date_values=None, external_features_values=None):
        """Fit one time series"""

        # Avoid top level import
        from statsforecast import StatsForecast
        from statsforecast.utils import ConformalIntervals

        statsforecast_df = pd.DataFrame({
            # statsforecast library supports training using multiple timeseries identifiers, but under the hood it will train one model per timeseries
            # to keep it simple and similar to other statistical models within DSS we manage each model on our side and use a static value for mandatory unique_id
            # https://github.com/dataiku/dip/pull/41066#discussion_r2450744127
            "unique_id": np.full(len(date_values), 1),
            "ds": date_values,
            "y": target_values,
        })

        statsforecast_model = self._build_statsforecast_model()

        model = StatsForecast(models=[statsforecast_model], freq=self.frequency)

        model.fit(df=statsforecast_df)

        return model

    def _forecast_single_timeseries(
            self,
            trained_model,
            past_target_values,
            past_date_values,
            quantiles,
            past_external_features_values,
            future_external_features_values,
            fit_before_predict,
            prediction_length
    ):

        if fit_before_predict:
            # instantiate and fit model with the same hyperparameters as the trained model used during training
            trained_model = self._fit_single(past_target_values, past_date_values, past_external_features_values)
        else:
            if not np.array_equal(trained_model.ga.data.flatten(), past_target_values):
                logger.warning(
                    "Predicting a Croston model with a different target than the one used during training"
                )

        result = trained_model.predict(h=prediction_length)

        return {
            ModelForecast.FORECAST_VALUES: result[self._get_statsforecast_model_name()].to_numpy(),
            ModelForecast.QUANTILES_FORECASTS: np.zeros(shape=(len(quantiles), prediction_length)),
        }

    def get_fitted_values_and_residuals(self, identifier, df_of_identifier, min_scoring_size):
        trained_model = self.trained_models[identifier]
        fitted_values = trained_model.fitted_[0, 0].model_["fitted"]
        residuals = df_of_identifier[self.target_variable].reset_index(drop=True) - fitted_values

        return pd.Series(fitted_values), residuals
