import pandas as pd

from dataiku.base.folder_context import build_folder_context
from dataiku.core import doctor_constants
from dataiku.doctor.timeseries.models import TimeseriesForecastingAlgorithm
from dataiku.doctor.timeseries.preparation.preprocessing import get_external_features
from dataiku.doctor.timeseries.score.scoring_handler import resample_for_scoring
from dataiku.doctor.timeseries_interactive.interactive_scenarios_computer import InteractiveScoringComputer

from dataiku.doctor.timeseries_interactive.interactive_scenarios_handler import TimeseriesInteractiveScenariosHandler
from dataiku.doctor.utils.split import load_train_set


def _prepare_interactive_scoring_data(split_desc, core_params, preprocessing_folder, model_folder, split_folder):
    model_folder_context = build_folder_context(model_folder)
    preprocessing_folder_context = build_folder_context(preprocessing_folder)
    preprocessing_params = preprocessing_folder_context.read_json("rpreprocessing_params.json")
    split_folder_context = build_folder_context(split_folder)
    modeling_params = model_folder_context.read_json("rmodeling_params.json")
    resolved_params = model_folder_context.read_json("actual_params.json")["resolved"]

    full_df = load_train_set(core_params, preprocessing_params, split_desc, "full",
                             split_folder_context, use_diagnostics=False)

    algorithm = TimeseriesForecastingAlgorithm.build(modeling_params["algorithm"])

    external_features = get_external_features(preprocessing_params, algorithm)

    full_df = resample_for_scoring(
        full_df,
        split_desc["schema"],
        preprocessing_params[doctor_constants.TIMESERIES_SAMPLING],
        core_params,
        preprocessing_params,
        external_features,
        algorithm.EXTERNAL_FEATURES_COMPATIBILITY.supports_past_only_external_features(),
        separate_external_features=True,
    )

    return model_folder_context, preprocessing_folder_context, preprocessing_params, modeling_params, resolved_params, full_df, external_features


def create_scenario(split_desc, core_params, preprocessing_folder, model_folder,
                                                    split_folder, computation_parameters):
    """
    Create a scenario for a given timeseries identifier
    """
    computation_parameters = computation_parameters or {}
    target_timeseries_identifier = computation_parameters.get('timeseriesIdentifier')
    source_scenario_id = computation_parameters.get('scenarioId')
    start_date_str = computation_parameters.get('startDate')
    scenario_name = computation_parameters.get('scenarioName')

    model_folder_context, _, preprocessing_params, _, resolved_params, full_df, external_features = \
        _prepare_interactive_scoring_data(split_desc, core_params, preprocessing_folder, model_folder, split_folder)

    if len(external_features) == 0:
        raise ValueError("The model doesn't use any external features.")

    # We retrieve only identifiers that we trained on
    trained_models_identifiers = list(model_folder_context.read_json("forecasts.json.gz")["perTimeseries"].keys())
    if target_timeseries_identifier in trained_models_identifiers:
        interactive_scenarios_handler = TimeseriesInteractiveScenariosHandler(model_folder_context, resolved_params,
                                                                              preprocessing_params, core_params,
                                                                              full_df)
        interactive_scenarios_handler.create_scenario(target_timeseries_identifier, source_scenario_id,
                                                      pd.to_datetime(start_date_str), scenario_name)
        interactive_scenarios_handler.dump_identifiers_mapping()
        return interactive_scenarios_handler.get_scenarios_with_metadata(target_timeseries_identifier)
    else:
        raise Exception("No model for timeseries identifier %s." % target_timeseries_identifier)

def compute_scenarios_forecasts(split_desc, core_params, preprocessing_folder, model_folder,
                                                   split_folder, computation_parameters):
    """
    Compute scenarios of a given timeseries identifier
    """
    model_folder_context, preprocessing_folder_context, preprocessing_params, modeling_params, resolved_params, full_df, external_features = \
        _prepare_interactive_scoring_data(split_desc, core_params, preprocessing_folder, model_folder, split_folder)
    computer = InteractiveScoringComputer(core_params, model_folder_context, preprocessing_folder_context, preprocessing_params, modeling_params, resolved_params, full_df, external_features)
    identifier = computation_parameters['timeseriesIdentifier']
    return computer.compute_forecasts(identifier)
