from typing import Optional
from dataiku.eda.types import Literal

import numpy as np

from dataiku.eda.computations.immutable_data_frame import ImmutableDataFrame
from dataiku.eda.filtering.filter import Filter
from dataiku.eda.types import MissingFilterModel


class MissingFilter(Filter):
    def __init__(self, column: str, name: Optional[str] = None):
        self.column = column
        self.name = name

    @staticmethod
    def get_type() -> Literal["missing"]:
        return "missing"

    @staticmethod
    def build(params: MissingFilterModel) -> 'MissingFilter':
        return MissingFilter(params["column"], params.get('name'))

    def apply(self, idf: ImmutableDataFrame, inverse: bool = False) -> ImmutableDataFrame:
        mask = np.isnan(idf.float_col(self.column))
        if inverse:
            mask = ~mask
        return idf[mask]

    def serialize(self) -> MissingFilterModel:
        return {"type": self.get_type(), "column": self.column, "name": self.name}
