from typing import Optional

import pandas as pd

from dataiku.llm.types import CompletionSettings


def mean_or_none(pd_series) -> Optional[float]:
    mean = pd_series.mean(skipna=True)
    return None if pd.isna(mean) else mean


def median_or_none(pd_series) -> Optional[float]:
    median = pd_series.median(skipna=True)
    return None if pd.isna(median) else median


def p95_or_none(pd_series: pd.Series) -> Optional[float]:
    if pd_series.dropna().empty:  # dropna includes both NaN and None
        # quantile will be nan in this case- we'd rather have None for serialization reasons
        return None

    # quantile will ignore NaN/None, so long as there is at least one non-NaN/None value
    return float(pd_series.quantile(0.95))


def get_llm_args(completion_settings: CompletionSettings):
    llm_args = {}
    if "temperature" in completion_settings:
        llm_args["temperature"] = completion_settings["temperature"]
    if "maxOutputTokens" in completion_settings:
        llm_args["max_tokens"] = completion_settings["maxOutputTokens"]
    if "topK" in completion_settings:
        llm_args["top_k"] = completion_settings["topK"]
    if "topP" in completion_settings:
        llm_args["top_p"] = completion_settings["topP"]
    return llm_args

