import dataikuapi
import sys
import json
from pathlib import Path

FM_URL = "http://localhost:10000"

def read_credentials(credentials_json_file):
    """
    Read credentials previously created with the fmcli and stored in a JSON file.
    """
    credentials_file_path = Path(credentials_json_file)
    if not credentials_file_path.is_file():
        raise Error("The provided credentials file path does not match an existing JSON file")

    try:
        with open(credentials_json_file) as credentials_file:
            credentials_json = json.load(credentials_file)
        if not "id" in credentials_json or not "secret" in credentials_json:
            raise Error("The provided credentials file has incorrect JSON format")
    except:
        raise Error("The provided credentials file has incorrect JSON format")

    api_key = credentials_json["id"]
    api_secret = credentials_json["secret"]

    return api_key, api_secret

def read_data(data_json_file):
    """
    Read content of the JSON file containing the credentials data.
    """
    data_file_path = Path(data_json_file)
    if not data_file_path.is_file():
        raise Error("The provided data file path does not match an existing JSON file")

    try:
        with open(data_json_file) as data_file:
            return json.load(data_file)
    except:
        raise Error("The provided credentials file has incorrect JSON format")

def is_populated(field_name, data_json):
    """
    Check whether or not a field in a JSON is populated.
    """
    return field_name in data_json and data_json[field_name] and data_json[field_name] != "null"

def perform_install_aws(api_key, api_secret, data_json):
    """
    Perform initial setup for AWS.
    """
    print("Setting credentials information for AWS")
    print(json.dumps(data_json))
    fm_client = dataikuapi.FMClientAWS(FM_URL, api_key, api_secret)

    if is_populated("cmkId", data_json):
        cloud_credentials = fm_client.get_cloud_credentials()
        cloud_credentials.set_cmk_key(data_json["cmkId"])
        cloud_credentials.save()

    roles_update_occurred = False
    try :
        default_instance_template = fm_client.get_instance_template("ist-default")

        if is_populated("awsStartupRoleArn", data_json):
            default_instance_template.ist_data["startupInstanceProfileArn"] = data_json["awsStartupRoleArn"]
            roles_update_occurred = True
        
        if is_populated("awsRuntimeRoleArn", data_json):
            default_instance_template.ist_data["runtimeInstanceProfileArn"] = data_json["awsRuntimeRoleArn"]
            roles_update_occurred = True
        
        if roles_update_occurred:
            default_instance_template.save()
    except HTTPError:
        print("An error occurred during the configuration of the default instance template")
        raise

def perform_install_azure(api_key, api_secret, data_json):
    """
    Perform initial setup for Azure.
    """
    fm_client = dataikuapi.FMClientAzure(FM_URL, api_key, api_secret)

    cloud_credentials = fm_client.get_cloud_credentials()
    update_occurred = False

    if is_populated("azureSubscription", data_json):
        cloud_credentials.cloud_credentials["azureSubscription"] = data_json["azureSubscription"]
        update_occurred = True

    if is_populated("azureTenant", data_json):
        cloud_credentials.cloud_credentials["azureTenantId"] = data_json["azureTenant"]
        update_occurred = True

    if is_populated("azureEnvironment", data_json):
        cloud_credentials.cloud_credentials["azureEnvironment"] = data_json["azureEnvironment"]
        update_occurred = True

    if update_occurred:
        cloud_credentials.save()

def perform_install_gcp(api_key, api_secret, data_json):
    """
    Perform initial setup for GCP.
    """
    fm_client = dataikuapi.FMClientGCP(FM_URL, api_key, api_secret)

    roles_update_occurred = False
    try:
        default_instance_template = fm_client.get_instance_template("ist-default")

        if is_populated("gcpStartupServiceAccount", data_json):
            default_instance_template.ist_data["startupServiceAccount"] = data_json["gcpStartupServiceAccount"]
            roles_update_occurred = True
        
        if roles_update_occurred:
            default_instance_template.save()
    except HTTPError:
        print("An error occurred during the configuration of the default instance template")
        raise

def perform_setup(cloud, credentials_json_file, data_json_file):
    """
    Perform initial setup.
    """
    data_json = read_data(data_json_file)
    api_key, api_secret = read_credentials(credentials_json_file)

    if cloud == "aws":
        perform_install_aws(api_key, api_secret, data_json)
    elif cloud == "azure":
        perform_install_azure(api_key, api_secret, data_json)
    elif cloud == "gcp":
        perform_install_gcp(api_key, api_secret, data_json)
    else:
        raise Error("Unknown provided cloud")

if __name__ == "__main__":
    credentials_json_file = sys.argv[1]
    cloud = sys.argv[2]
    data_json_file = sys.argv[3]

    print("FM API initialization script called")
    print(f"Credentials JSON file: {credentials_json_file}")
    print(f"Cloud: {cloud}")
    print(f"Data JSON file: {data_json_file}")

    perform_setup(cloud, credentials_json_file, data_json_file)
