#
# Supervisor configuration file generation
#
from six.moves import configparser
import os, sys
from os import path as osp

import base

#
# Returns a configparser object initialized with the generic sections
#
def defaultConfig(dipHome, installConfig):
    runDir = '%(ENV_DIP_HOME)s/run'
    config = configparser.RawConfigParser()

    config.add_section('supervisord')
    config.set('supervisord', 'directory', runDir)
    config.set('supervisord', 'pidfile', '%s/supervisord.pid' % runDir)
    config.set('supervisord', 'logfile', '%s/supervisord.log' % runDir)
    if base.is_os_windows():
        # This makes DSS run as fast as it can, while Supervisor does not tax CPU as much as 0 delay
        config.set('supervisord', 'delaysecs', '0.00001')

    config.add_section('rpcinterface:supervisor')
    config.set('rpcinterface:supervisor', 'supervisor.rpcinterface_factory',
        'supervisor.rpcinterface:make_main_rpcinterface')

    password = dipHome.get_supervisor_key()

    if base.is_os_windows():
        config.add_section('inet_http_server')
        config.set('inet_http_server', 'port', '127.0.0.1:%s' % os.environ['DKU_SUPERVISORCTL_PORT']) # Replace by a generated port derived from base port ?
        config.set('inet_http_server', 'username', 'dataiku')
        config.set('inet_http_server', 'password', password)
    else:
        config.add_section('unix_http_server')
        config.set('unix_http_server', 'file', '%s/svd.sock' % runDir)
        config.set('unix_http_server', 'username', 'dataiku')
        config.set('unix_http_server', 'password', password)

    config.add_section('supervisorctl')
    if base.is_os_windows():
        config.set('supervisorctl', 'serverurl', 'http://127.0.0.1:%s' % os.environ['DKU_SUPERVISORCTL_PORT'])
    else:
        config.set('supervisorctl', 'serverurl', 'unix://%s/svd.sock' % runDir)
    config.set('supervisorctl', 'username', 'dataiku')
    config.set('supervisorctl', 'password', password)

    return config

#
# Adds a managed program to supervisor config "config"
#
def addChild(config, programName, installConfig, startSecs=5, stopAsGroup=True, stopSignal=None, stopWaitSecs=None):
    sectionName = 'program:%s' % programName
    config.add_section(sectionName)

    redirect_log_to_stdout = False
    if programName in ['apimain']:
        redirect_to_supervisor_stdout = installConfig.getOption('logs', 'redirect_to_supervisor_stdout')
        redirect_log_to_stdout = redirect_to_supervisor_stdout is not None and redirect_to_supervisor_stdout == 'true'

    if installConfig.getNodeType() == "fm":
        config.set(sectionName, 'command', '%(ENV_DIP_HOME)s/bin/fm run %(program_name)s')
    else:
        if base.is_os_windows():
            # We launch the wrapper to launch Powershell scripts
            config.set(sectionName, 'command', '%(ENV_DIP_HOME)s/bin/dss.cmd run %(program_name)s')
        else:
            config.set(sectionName, 'command', '%(ENV_DIP_HOME)s/bin/dss run %(program_name)s')

    config.set(sectionName, 'stdout_logfile', '%(ENV_DIP_HOME)s/run/%(program_name)s.log')
    logfile_maxbytes = installConfig.getOption('logs', 'logfiles_maxbytes')
    if logfile_maxbytes is not None:
        config.set(sectionName, 'stdout_logfile_maxbytes', logfile_maxbytes)
    else:
        config.set(sectionName, 'stdout_logfile_maxbytes', "100MB")
    config.set(sectionName, 'redirect_stderr', 'true')
    config.set(sectionName, 'autorestart', 'true')
    config.set(sectionName, 'startsecs', startSecs)

    if redirect_log_to_stdout:
        config.set(sectionName, 'stdout_events_enabled', 'true')

        config.add_section('eventlistener:stdout')
        config.set('eventlistener:stdout', 'command',
                   "'%(ENV_DKUPYTHONBIN)s' '%(ENV_DKUINSTALLDIR)s/python/deploymentsupport/supervisor_stdout/supervisor_stdout.py'")
        config.set('eventlistener:stdout', 'buffer_size', '100')
        config.set('eventlistener:stdout', 'events', 'PROCESS_LOG')
        config.set('eventlistener:stdout', 'result_handler', 'deploymentsupport.supervisor_stdout.supervisor_stdout:event_handler')

    logfile_backups = installConfig.getOption('logs', 'logfiles_backups')
    if logfile_backups is not None:
        config.set(sectionName, 'stdout_logfile_backups', logfile_backups)

    if stopAsGroup:
        config.set(sectionName, 'stopasgroup', 'true')
    else:
        config.set(sectionName, 'killasgroup', 'true')
    if stopSignal and not base.is_os_windows(): # Stop signal is not really supported on Windows
        config.set(sectionName, 'stopsignal', stopSignal)
    if config.has_option('supervisord', 'logfile_maxbytes'):
        config.set(sectionName, 'logfile_maxbytes', config.get('supervisord', 'logfile_maxbytes'))
    if config.has_option('supervisord', 'logfile_backups'):
        config.set(sectionName, 'logfile_backups', config.get('supervisord', 'logfile_backups'))
    
    if stopWaitSecs:
        config.set(sectionName, 'stopwaitsecs', stopWaitSecs)

def addBackend(config, installConfig):
    addChild(config, 'backend', installConfig, startSecs=10)

def addNginx(config, installConfig):
    addChild(config, 'nginx', installConfig, stopAsGroup=False, stopSignal='QUIT')

def addIPython(config, installConfig):
    addChild(config, 'ipython', installConfig)

def addAPIMain(config, installConfig):
    addChild(config, 'apimain', installConfig, startSecs=10)

def addFMMain(config, installConfig):
    addChild(config, 'fmmain', installConfig, startSecs=10)

def addGovernServer(config, installConfig):
    addChild(config, 'governserver', installConfig, startSecs=10)

def addCollectd(config, installConfig):
    addChild(config, 'collectd', installConfig)

def addEventServer(config, installConfig):
    addChild(config, 'eventserver', installConfig)

def addStories(config, installConfig):
    addChild(config, 'stories', installConfig, startSecs=60, stopWaitSecs=15)

#
# Generates the supervisor configuration corresponding to dipHome
# Assumes the environment variables in env-default have been sourced
#
def generate_supervisor_config(dipHome):
    installConfig = dipHome.get_install_config()
    config = defaultConfig(dipHome, installConfig)

    nodeType = installConfig.getNodeType()
    if nodeType == 'design':
        addBackend(config, installConfig)
        addIPython(config, installConfig)
        addNginx(config, installConfig)
        if osp.exists(osp.join(dipHome.path, "stories")):
            addStories(config, installConfig)
    elif nodeType == "automation":
        addBackend(config, installConfig)
        addIPython(config, installConfig)
        addNginx(config, installConfig)
    elif nodeType == 'api':
        addAPIMain(config, installConfig)
        addNginx(config, installConfig)
    elif nodeType == 'fm':
        addFMMain(config, installConfig)
        addNginx(config, installConfig)
    elif nodeType == 'govern':
        addGovernServer(config, installConfig)
        addNginx(config, installConfig)
    else:
        raise Exception("Node type not supported: %s" % nodeType)

    if installConfig.getBoolOption("collectd","enabled",False):
        addCollectd(config, installConfig)

    if installConfig.getBoolOption("eventserver", "enabled", False):
        addEventServer(config, installConfig)

    if installConfig.getOption('supervisord', 'kill_supervisord_if_child_dies_agressive') is not None:
        config.add_section("eventlistener:supervisord-watchdog")
        config.set("eventlistener:supervisord-watchdog", "events", "PROCESS_STATE_FATAL")
        config.set("eventlistener:supervisord-watchdog", "command",
            "'%(ENV_DKUPYTHONBIN)s' '%(ENV_DKUINSTALLDIR)s/scripts/_kill-supervisord-if-child-dies-agressive.py'")
        config.set("eventlistener:supervisord-watchdog", 'stderr_logfile', '%(ENV_DIP_HOME)s/run/supervisord-watchdog.log')

    return config

#
# Prints the supervisor configuration on standard output
#
if __name__ == "__main__":
    generate_supervisor_config(base.DipHome(os.environ["DIP_HOME"])).write(sys.stdout)
