# coding: utf-8
from __future__ import unicode_literals

import argparse
import logging
import os
import pickle
import socket
import ssl
import tempfile

from dataiku.base.block_link import BlockInput
from dataiku.base.block_link import BlockOutput

"""
Implement block-link protocol over socket
"""

logger = logging.getLogger(__name__)


class AbstractSocketBlockLink(BlockInput, BlockOutput):
    def __init__(self, connection_timeout):
        BlockInput.__init__(self, self)
        BlockOutput.__init__(self, self)
        self.connection_timeout = connection_timeout

    def read(self, size):
        """
        Read from the socket (required by BlockInput)
        """
        return self.get_socket().recv(size)

    def write(self, data):
        """
        Write to the socket (required by BlockOutput)
        """
        return self.get_socket().sendall(data)

    def get_socket(self):
        raise NotImplementedError

    def close(self):
        raise NotImplementedError

    def negotiate_pickle_protocol(self):
        self.send_int(pickle.HIGHEST_PROTOCOL)
        peer_pickle_protocol_version = self.read_int()
        logger.info("Highest pickle protocol version: '%s'. Peer highest protocol version: '%s'" % (str(pickle.HIGHEST_PROTOCOL), str(peer_pickle_protocol_version)))
        self.pickle_protocol_version = min(self.pickle_protocol_version, peer_pickle_protocol_version)
        logger.info("Using pickle protocol version: '%s'." % (str(self.pickle_protocol_version)))

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.close()

    @staticmethod
    def _properly_close_socket(s):
        """
        Calling shutdown() seems necessary to unblock reader/writer threads waiting on this socket
        (and calling close() is not enough)
        """
        if s is None:
            return
        try:
            s.shutdown(socket.SHUT_RDWR)
        except socket.error:
            # shutdown() may fail if socket already closed
            pass
        s.close()


class SocketBlockLinkClient(AbstractSocketBlockLink):
    """
    Client-side link
    """

    def __init__(self, host, port, secret, connection_timeout=60, server_cert_path=None):
        super(SocketBlockLinkClient, self).__init__(connection_timeout)
        self.host = host
        self.port = port
        self.socket = None
        self.secret = secret
        self.server_cert_path = server_cert_path

    def connect(self):
        ip_addr = socket.gethostbyname(self.host)
        host_with_resolved_addr = "%s (%s)" % (self.host, ip_addr) if self.host != ip_addr else self.host
        logger.info("Connecting to %s at port %s" % (host_with_resolved_addr, self.port))
        sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        sock.setsockopt(socket.IPPROTO_TCP, socket.TCP_NODELAY, 1)
        sock.setsockopt(socket.SOL_SOCKET, socket.SO_KEEPALIVE, 1)
        sock.settimeout(self.connection_timeout)
        sock.connect((ip_addr, self.port))
        if self.server_cert_path is not None:
            context = ssl.create_default_context(capath=self.server_cert_path)
            context.load_verify_locations(self.server_cert_path)
            self.socket = context.wrap_socket(sock, server_hostname=self.host)
            sock.close()
        else:
            self.socket = sock
        self.send_string(self.secret)
        logger.info("Connected to %s at port %s" % (host_with_resolved_addr, self.port))

        # Link has been establish: disable timeouts
        self.socket.settimeout(None)

    def close(self):
        if self.socket is None:
            return

        self._properly_close_socket(self.socket)
        self.socket = None
        logger.info("Client closed")

    def get_socket(self):
        if self.socket is None:
            raise IOError('Not connected to server')

        return self.socket


class SocketBlockLinkServer(AbstractSocketBlockLink):
    """
    Server-side link
    """

    def __init__(self, secret, timeout=60, host=None):
        super(SocketBlockLinkServer, self).__init__(timeout)
        self.secret = secret
        self.serversocket = None
        self.host = host or socket.gethostname()
        self.socket = None

    def listen(self):
        """
        Start listening for a connection on a (randomly picked) port
        """
        logger.info("Starting server...")
        self.serversocket = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        self.serversocket.setsockopt(socket.IPPROTO_TCP, socket.TCP_NODELAY, 1)
        self.serversocket.setsockopt(socket.SOL_SOCKET, socket.SO_KEEPALIVE, 1)
        self.serversocket.settimeout(self.connection_timeout)
        self.serversocket.bind((self.host, 0))
        host, port = self.serversocket.getsockname()
        self.serversocket.listen(1)

        logger.info("Server is listening on %s:%s" % (host, port))
        return port

    def close(self):
        """Close the socket"""
        if self.socket is None and self.serversocket is None:
            return

        logger.info("Closing server...")
        self._properly_close_socket(self.socket)
        self.socket = None
        self._properly_close_socket(self.serversocket)
        self.serversocket = None
        logger.info("Server closed")

    def accept(self):
        """
        Wait until client is connected
        """
        logger.info("Waiting for client to connect...")
        (self.socket, _) = self.serversocket.accept()
        received_secret = self.read_string()
        if self.secret != received_secret:
            raise IOError("Invalid secret from {}".format(self.socket.getsockname()))
        logger.info("Client {} is connected".format(self.socket.getsockname()))

        # Link has been establish: disable timeouts
        self.socket.settimeout(None)

    def get_socket(self):
        if self.socket is None:
            raise IOError('Client is not connected')

        return self.socket


class JavaLink(SocketBlockLinkClient):
    """
    Connection with DSS backend or JEK
    """

    def __init__(self, port, secret, server_cert=None):
        server_cert_path = None
        if server_cert is None or len(server_cert) == 0:
            logger.info("no cert found")
        elif server_cert == "NA":
            logger.info("cert is NA")
        else:
            logger.info("cert found")
            with tempfile.NamedTemporaryFile(prefix="encrypted-rpc-cert-", delete=False) as f:
                if server_cert.find("b64:") == 0:
                    server_cert = server_cert[4:]
                    import base64
                    server_cert = base64.b64decode(server_cert).decode("utf8")
                f.write(server_cert.encode("utf-8"))
                server_cert_path = f.name
        dss_host = os.getenv("DKU_BACKEND_HOST", "localhost")
        super(JavaLink, self).__init__(dss_host, port, secret, server_cert_path=server_cert_path)


def parse_javalink_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('port', type=int)
    parser.add_argument('secret', type=str)
    parser.add_argument('--server-cert', type=str)
    parsed = parser.parse_args()
    return parsed.port, parsed.secret, parsed.server_cert
