import logging

import dataiku
from dataiku.core import intercom
import functools
import logging
import json
from datetime import datetime
from dateutil.relativedelta import relativedelta

# noinspection PyPep8Naming
class SparkLike:

    def __init__(self):
        # keep a map of sessions, indexed by DSS connection name, because they can be costly to instantiate,
        # time-wise and resource-wide
        self.__session_by_connection_names = {}
        self._dialect = None
        self._connection_type = None

    def get_session(self, connection_name, project_key=None):
        """Return session configured to read on the supplied DSS connection."""
        if connection_name not in self.__session_by_connection_names:
            self.__session_by_connection_names[connection_name] = self.create_session(connection_name, project_key)

        return self.__session_by_connection_names[connection_name]

    def create_session(self, connection_name, project_key=None):
        """Creates a new session configured to read on the supplied DSS connection."""
        connection_info = self._get_connection_info(connection_name, project_key)
        return self._create_session(connection_name, connection_info, project_key)

    def _create_session(self, connection_name, connection_info, project_key=None):
        return None
        
    def _get_connection_info(self, connection_name, project_key):
        connection = dataiku.api_client().get_connection(connection_name)
        connection_info = connection.get_info(project_key)
        if connection_info is None or connection_info["type"] != self._connection_type:
            raise ValueError("The connection is not a %s connection" % self._connection_type)
        connection_params = connection_info["resolvedParams"]
        if "useURL" in connection_params and connection_params["useURL"]:
            # split what is defined in the URL to the fields in the params
            params = self._split_jdbc_url(connection_params.get("url"))
            for k in params:
                if isinstance(params[k], list) and isinstance(connection_params[k], list):
                    connection_params[k] = connection_params[k] + params[k]
                else:
                    connection_params[k] = params[k]
        return connection_info
        
    def _get_connection_param(self, connection_params, param_name, property_name):
        if param_name in connection_params and connection_params[param_name].strip():
            return connection_params[param_name]
        for prop in connection_params["properties"]:
            if prop["name"] is not None and prop["name"].lower() == property_name.lower():
                if "value" in prop and prop["value"].strip():
                    logging.info("Connection %s not found in the parameters but found in the JDBC properties. Using value %s" % (param_name, prop["value"]))
                    return prop["value"]
                break
        return None

    def _split_jdbc_url(self, url):
        return {} # return a connection params dict

    def _get_dataframe_from_table(self, session, full_table_name, dataset_params, partitions_filters):
        df = session.table(full_table_name)
        if len(partitions_filters) > 0:
            return df.filter(self._combine_partition_filters(partitions_filters))
        else:
            return df
    def _get_dataframe_from_sql(self, session, query):
        return session.sql(query)

    def _get_union_all(self, dfs):
        return functools.reduce(lambda df1, df2: df1.unionAll(df2), dfs)
    def _combine_dimension_filters(self, filters):
        return functools.reduce(lambda f1, f2: f1 & f2, filters)
    def _combine_partition_filters(self, filters):
        return functools.reduce(lambda f1, f2: f1 | f2, filters)

    def get_dataframe(self, dataset, session=None):
        """Return a DataFrame configured to read the table that is underlying the specified dataset."""
        dataset_config = dataset.get_config()
        dataset_params = dataset_config["params"]
        if session is None:
            session = self.get_session(dataset_params["connection"], dataset.project_key)

        if dataset_config["type"] != self._connection_type:
            raise ValueError("Dataset is not of type %s" % self._connection_type)

        dataset_info = dataset.get_location_info()["info"]
        logging.debug("Retrieved dataset info: %s " % dataset_info)
        dss_schema = dataset.read_schema(raise_if_empty=False)
        if dataset_params["mode"] == "table":
            full_table_name = self._get_table_full_name(dataset_info, dataset_config["projectKey"])
            partitions_filters = []
            if hasattr(dataset, "read_partitions") and dataset.read_partitions:
                # Build the filter that correspond to the requested partitions (if any)
                dimensions = dataset_config["partitioning"]["dimensions"]

                for partition in dataset.read_partitions:
                    dimensions_filters = []
                    for index, partition_id in enumerate(partition.split('|')):
                        dimensions_filters.append(self._build_partition_filter_item(dataset, dss_schema, dimensions[index], partition_id))
                    if len(dimensions_filters) > 0:
                        partitions_filters.append(self._combine_dimension_filters(dimensions_filters))

            result = self._get_dataframe_from_table(session, full_table_name, dataset_params, partitions_filters)
        else:
            if hasattr(dataset, "read_partitions") and dataset.read_partitions:
                # Build the filter that correspond to the requested partitions (if any)
                expanded_queries_df = []
                for partition in dataset.read_partitions:
                    expand_sql_request = {
                        "projectKey": dataset.project_key,
                        "datasetName": dataset.short_name,
                        "partition": partition
                    }
                    expanded_query = intercom.jek_or_backend_json_call("datasets/expand-sql-query-for-partition/", data=expand_sql_request)
                    expanded_queries_df.append(self._get_dataframe_from_sql(session, expanded_query))

                result = self._get_union_all(expanded_queries_df)
            else:
                result = self._get_dataframe_from_sql(session, dataset_info["query"])

        return result

    def write_with_schema(self, dataset, df, force_direct_write=False, dropAndCreate=False):
        """Writes this dataset (or its target partition, if applicable) from
        a single dataframe.

        This variant replaces the schema of the output dataset with the schema
        of the dataframe.

        :param df: input dataframe.
        :param dataset: Output dataset to write.
        :param force_direct_write: Force writing the dataframe using the direct API into the dataset even if they don't come from the same DSS connection.
        :param dropAndCreate: drop and recreate the dataset.
        """
        return self.write_dataframe(dataset, df, True, force_direct_write, dropAndCreate)

    def _get_schema_from_dataframe(self, df):
        return df.schema
    def _write_dataframe_to_table(self, df, full_table_name):
        df.write.mode("append").saveAsTable(full_table_name)

    def write_dataframe(self, dataset, df, infer_schema=False, force_direct_write=False, dropAndCreate=False):
        """Writes this dataset (or its target partition, if applicable) from
        a single dataframe.

        This variant only edit the schema if infer_schema is True, otherwise you must
        take care to only write dataframes that have a compatible schema.
        Also see "write_with_schema".

        :param df: input dataframe.
        :param dataset: Output dataset to write.
        :param infer_schema: infer the schema from the dataframe.
        :param force_direct_write: Force writing the dataframe using the direct API into the dataset even if they don't come from the same DSS connection.
        :param dropAndCreate:  if infer_schema and this parameter are both set to True, clear and recreate the dataset structure.
        """
        
        self._check_dataframe_type(df)
        df = self._prepare_dataframe_for_write(df)

        df_connection_name = df._session.dss_connection_name if hasattr(df._session, "dss_connection_name") else None
        dataset_config = dataset.get_config()
        dataset_info = dataset.get_location_info()["info"]
        logging.debug("Retrieved dataset info: %s " % dataset_info)

        # If the destination dataset is of the right type
        # And it uses the same DSS connection as the one used to create the DataFrame
        # Then we directly write into the destination table using direct API, else we raise errors.
        if dataset_config["type"] != self._connection_type:
            raise ValueError("Cannot use direct API to direct write the dataframe into a dataset that is not stored on %s. Use dataset.write_dataframe(df.toPandas()) to write using a degraded mode if the dataframe is not too large." % self._connection_type)
        if dataset_info["connectionName"] != df_connection_name and not force_direct_write:
            raise ValueError("Cannot use direct API to direct write the dataframe into the dataset as they are on different DSS connections. Use force_direct_write=True to ignore this check or call dataset.write_dataframe(df.toPandas()) to write using a degraded mode if the dataframe is not too large.")

        logging.info("Writing dataframe directly with direct API as the destination dataset is based on the same connection as the dataframe.")
        write_mode = dataset.spec_item["appendMode"] and "APPEND" or "OVERWRITE"
        qualified_table_id = self._get_table_full_name(dataset_info, dataset_config["projectKey"])
        partitioning = dataset_config["partitioning"] if "partitioning" in dataset_config else {}
        dimensions = partitioning["dimensions"] if "dimensions" in partitioning else []

        dss_schema = dataset.read_schema(raise_if_empty=False)
        # For partitioned datasets, add columns for missing partitions
        if len(dimensions) > 0:
            logging.info("Checking for missing partitioning columns")
            column_names, column_fields = self._dialect._get_components_from_df_schema(self._get_schema_from_dataframe(df))
            for index, dimension in enumerate(dimensions):
                partition_name = dimension["name"]
                partition_col = self._get_schema_column(dss_schema, partition_name)
                if partition_col is None:
                    logging.warn("Dataset '%s' does not contain the partitioning column %s. Schema retrieved: %s "
                                 % (dataset.name, partition_name, dss_schema))
                    raise Exception("Dataset '%s' does not contain the partitioning column %s " % (dataset.name, partition_name))
                if not (partition_name in column_names):
                    logging.debug("Adding missing partitioning column %s" % partition_name)
                    # add column with a "fallback" type (ie: string). Don't look at the value, the real one from
                    # the actual partition id is set below
                    df = self._do_with_column(df, partition_name, self._dialect.python_string_to_sql_literal('', partition_col["type"]))

        # Once new partition columns have potentially been added:
        # Infer & write schema
        if infer_schema:
            dataset.write_schema(self._dialect.get_dss_schema_from_df_schema(self._get_schema_from_dataframe(df)), dropAndCreate, origin='sparklike')
            dss_schema = dataset.read_schema(raise_if_empty=False)

        if len(dimensions) > 0:
            # at this stage, there won't be missing columns in the schemas (neither dataset nor dataframe)
            partition_ids = dataset.writePartition.split('|')
            for index, dimension in enumerate(dimensions):
                partition_name = dimension["name"]
                partition_col = self._get_schema_column(dss_schema, partition_name)
                partition_id = partition_ids[index]
                if dimension["type"] == "time":
                    partition_id = self._compute_time_partition_id(dimension["params"]["period"], partition_id, partition_col["type"])
                #print("partition written %s -> %s" % (json.dumps(partition_col), partition_id))
                logging.debug("Setting value of partitioning column %s with value %s" % (partition_name, partition_id))
                partition_value = self._dialect.python_string_to_sql_literal(partition_id, partition_col["type"], partition_col.get("originalType"))
                df = self._do_with_column(df, partition_name, partition_value)

        prepare_sql_request = {
            "projectKey": dataset.project_key,
            "datasetName": dataset.short_name,
            "partition": dataset.writePartition,
            "writeMode": write_mode
        }
        logging.info("Preparing SQL table for write with %s" % prepare_sql_request)
        intercom.jek_or_backend_void_call("datasets/prepare-sql-table-for-write/", data=prepare_sql_request)

        # cast columns to their final types
        # important: AFTER prepare-sql-table-for-write so that the table exists
        df = self._cast_to_target_types(df, dss_schema, qualified_table_id)

        logging.info("Storing dataframe in table with INSERT INTO")
        self._write_dataframe_to_table(df, qualified_table_id)

    def _check_dataframe_type(self, df):
        """Check if the dataframe is of the correct type"""
        pass

    def _prepare_dataframe_for_write(self, df):
        """hook for last minute changes"""
        return df

    def _do_with_column(self, df, column_name, column_value):
        """Add or set a column in the dataframe"""
        return df
        
    def _cast_to_target_types(self, df, dss_schema, qualified_table_id):
        return df

    def _build_eq_partition_filter(self, col_name, value):
        return (self._dialect._column_name_to_sql_column(col_name) == value)
    def _build_ge_and_lt_partition_filter(self, col_name, value_low, value_hig):
        return (
                    (self._dialect._column_name_to_sql_column(col_name) >= value_low)
                  &
                    (self._dialect._column_name_to_sql_column(col_name) < value_hig)
                )

    def _build_partition_filter_item(self, dataset, schema, dimension, partition_id):
        dimension_col = self._get_schema_column(schema, dimension["name"])
        if dimension["type"] == "value":
            if dimension_col is None:
                # not very legit IMHO, but dkusnowpark had this
                partition_id_type = 'string'
                partition_id_original_type = None
            else:
                partition_id_type = dimension_col["type"]
                partition_id_original_type = dimension_col.get("originalType")
            partition_id_expr = self._dialect.python_string_to_sql_literal(partition_id, partition_id_type, partition_id_original_type)
            partition_filter_item = self._build_eq_partition_filter(dimension["name"], partition_id_expr)
        elif dimension["type"] == "time":
            period = dimension["params"]["period"]
            if dimension_col is None:
                logging.warn("Dataset '%s' does not contain the partitioning column %s. Schema retrieved: %s"
                             % (dataset.name, dimension["name"], schema))
                raise Exception("Dataset '%s' does not contain the partitioning column %s " % (dataset.name, dimension["name"]))
            partition_id = self._compute_time_partition_id(period, partition_id, dimension_col["type"])
            #print("partition_id=%s" % partition_id)
            
            partition_id_type = dimension_col["type"]
            partition_id_original_type = dimension_col.get("originalType")
            #print("partition_id_type=%s" % partition_id_type)
            #print("partition_id_original_type=%s" % partition_id_original_type)
            partition_id_expr = self._dialect.python_string_to_sql_literal(partition_id, partition_id_type, partition_id_original_type)
            if partition_id_type == 'date' or partition_id_type == 'dateonly' or partition_id_type == 'datetimenotz':
                # partition_id is now an ISO8061 timestamp => make a filter for "x == this_partition" as a "x >= a and x < a+1" (like we do in the java side)
                format_to_use = "%Y-%m-%d"
                if partition_id_type == 'date':
                    format_to_use = format_to_use + "T%H:%M:%S.%fZ"
                elif partition_id_type == 'datetimenotz':
                    format_to_use = format_to_use + " %H:%M:%S.%f"
                partition_id_plus_1 = datetime.strftime(datetime.strptime(partition_id, format_to_use) + self._compute_time_partition_granularity(period), format_to_use)
                partition_id_plus_1_expr = self._dialect.python_string_to_sql_literal(partition_id_plus_1, partition_id_type, partition_id_original_type)

                #print("partition_id_plus_1=%s" % partition_id_plus_1)
                partition_filter_item = self._build_ge_and_lt_partition_filter(dimension["name"], partition_id_expr, partition_id_plus_1_expr)
            else:
                # not handled as date, so do a simple equality test
                partition_filter_item = self._build_eq_partition_filter(dimension["name"], partition_id_expr)
        else:
            raise ValueError("Invalid partition dimension type: %s" % dimension["type"])
        return partition_filter_item


    @staticmethod
    def _get_schema_column(dss_schema, column_name):
        if dss_schema is not None:
            for column in dss_schema:
                if column["name"] == column_name:
                    return column
        return None

    @staticmethod
    def _compute_time_partition_id(period, partition_id, dss_type):
        if dss_type == 'date' or dss_type == 'dateonly' or dss_type == 'datetimenotz':
            time_suffix = ""
            if dss_type == 'date':
                time_suffix = "T00:00:00.000Z"
            elif dss_type == 'datetimenotz':
                time_suffix = " 00:00:00.000"
            if period == "HOUR":
                if dss_type == 'dateonly':
                    raise ValueError("Partition dimension at HOUR granularity for date only column")
                if len(partition_id) != 13:
                    raise ValueError("Partition id %s doesn't conform to format YYYY-MM-DD-HH" % partition_id)
                if dss_type == 'date':
                    return partition_id[0:10] + "T" + partition_id[11:] + ":00:00.000Z"
                else:
                    return partition_id[0:10] + " " + partition_id[11:] + ":00:00.000"
            elif period == "DAY":
                return partition_id + time_suffix
            elif period == "MONTH":
                return partition_id + "-01" + time_suffix
            elif period == "YEAR":
                return partition_id + "-01-01" + time_suffix
            else:
                raise ValueError("Invalid time partition period: %s" % period)
        else:
            return partition_id
            
    @staticmethod
    def _compute_time_partition_granularity(period):
        if period == "HOUR":
            return relativedelta(hours=1)
        elif period == "DAY":
            return relativedelta(days=1)
        elif period == "MONTH":
            return relativedelta(months=1)
        elif period == "YEAR":
            return relativedelta(years=1)
        else:
            raise ValueError("Invalid time partition period: %s" % period)

    def _get_table_full_name(self, dataset_info, project_key):
        connection_name = dataset_info["connectionName"]
        connection_params = self._get_connection_info(connection_name, project_key)["resolvedParams"]
        table = dataset_info["table"]
        schema = self._get_table_schema(dataset_info.get("schema"), connection_params)
        catalog = self._get_table_catalog(dataset_info.get("catalog"), connection_params)
        
        parts = []
        if catalog is not None and len(catalog) > 0:
            parts.append(catalog)
            # when there's a catalog, put schema always
            if schema is None or len(schema) == 0:
                if self._dialect.allow_empty_schema_after_catalog():
                    parts.append('') 
                else:
                    raise ValueError("Empty schema is not supported if catalog is set")
            else:
                parts.append(schema) 
        else:
            if schema is not None and len(schema) > 0:
                parts.append(schema)
        parts.append(table)
        
        quoted_parts = [self._dialect.quote_identifier(part) for part in parts]
        return '.'.join(quoted_parts)

    def _get_table_schema(self, schema, connection_params):
        if schema and schema.strip():
            return schema
        return None

    def _get_table_catalog(self, catalog, connection_params):
        if catalog and catalog.strip():
            return catalog
        return None

