/*
 * Decompiled with CFR 0.152.
 */
package com.dataiku.dip.server.services;

import com.dataiku.dip.DSSTempUtils;
import com.dataiku.dip.coremodel.InfoMessage;
import com.dataiku.dip.exceptions.CodedException;
import com.dataiku.dip.exceptions.DKUSecurityException;
import com.dataiku.dip.exceptions.UnauthorizedException;
import com.dataiku.dip.futures.FuturePayload;
import com.dataiku.dip.futures.FutureResponse;
import com.dataiku.dip.futures.FutureService;
import com.dataiku.dip.futures.SimpleFutureThread;
import com.dataiku.dip.security.AuthCtx;
import com.dataiku.dip.security.DSSAuthCtx;
import com.dataiku.dip.security.UrlRedactionUtils;
import com.dataiku.dip.server.services.GitReferencesService;
import com.dataiku.dip.server.services.IJupyterService;
import com.dataiku.dip.server.services.JupyterService;
import com.dataiku.dip.server.services.TransactionService;
import com.dataiku.dip.transactions.git.GitCodes;
import com.dataiku.dip.transactions.git.GitModel;
import com.dataiku.dip.transactions.git.cli.GitRemoteCommands;
import com.dataiku.dip.transactions.git.jgit.GitLocalCommands;
import com.dataiku.dip.transactions.ifaces.RWTransaction;
import com.dataiku.dip.transactions.ifaces.Transaction;
import com.dataiku.dip.util.AutoDelete;
import com.dataiku.dip.utils.DKULogger;
import com.dataiku.dip.utils.DKUtils;
import com.dataiku.dip.utils.ErrorContext;
import com.dataiku.dip.utils.JSON;
import com.dataiku.dip.utils.SmartLogTail;
import com.dataiku.dip.utils.StringTransmogrifier;
import com.dataiku.dss.shadelib.com.google.gson.annotations.Expose;
import com.dataiku.dss.shadelib.org.apache.commons.io.FileUtils;
import com.dataiku.dss.shadelib.org.apache.commons.io.FilenameUtils;
import com.google.common.annotations.VisibleForTesting;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.reflect.TypeToken;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.net.URISyntaxException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;
import org.apache.commons.lang.StringUtils;
import org.eclipse.jgit.api.Git;
import org.eclipse.jgit.api.Status;
import org.eclipse.jgit.api.errors.GitAPIException;
import org.eclipse.jgit.revwalk.RevCommit;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

@Service
public class JupyterGitService {
    @Autowired
    private IJupyterService service;
    @Autowired
    private TransactionService transactionService;
    @Autowired
    private FutureService futureService;
    @Autowired
    private GitReferencesService gitReferencesService;
    public static final String NOTEBOOK_EXTENSION = ".ipynb";
    private static DKULogger logger = DKULogger.getLogger((String)"dku.jupyter.git.service");

    public List<NotebookImportGitDTO> retrieveNotebooksFromGit(File gitDir, List<String> ipynbPaths) throws IOException {
        ArrayList<NotebookImportGitDTO> notebooks = new ArrayList<NotebookImportGitDTO>(ipynbPaths.size());
        for (String ipynbPath : ipynbPaths) {
            if (ipynbPath.contains(".ipynb_checkpoints")) continue;
            File ipynbFile = JupyterGitService.getFileInGit(gitDir, ipynbPath);
            JupyterService.NotebookSafeForWritingNew notebookObj = null;
            NotebookImportGitDTO notebookImportGitDTO = new NotebookImportGitDTO();
            notebookImportGitDTO.name = FilenameUtils.removeExtension((String)ipynbFile.getName());
            String path = gitDir.toPath().relativize(ipynbFile.getParentFile().toPath()).toString();
            notebookImportGitDTO.path = path != null && path.length() > 1 ? File.separator + path : "";
            try (InputStreamReader reader = new InputStreamReader((InputStream)new FileInputStream(ipynbFile), StandardCharsets.UTF_8);){
                notebookObj = (JupyterService.NotebookSafeForWritingNew)JSON.gson().fromJson((Reader)reader, JupyterService.NotebookSafeForWritingNew.class);
            }
            catch (Exception e) {
                logger.warn((Object)("Failed to read Jupyter notebook from file " + ipynbPath));
            }
            if (notebookObj != null) {
                notebookImportGitDTO.isValid = true;
                notebookImportGitDTO.nbFormat = notebookObj.nbformat;
                notebookImportGitDTO.nbFormatMinor = notebookObj.nbformat_minor;
                JsonObject kernelSpec = JupyterService.safeGetMetadataJsonObject(notebookObj, "kernelspec");
                if (kernelSpec != null && kernelSpec.has("language")) {
                    notebookImportGitDTO.language = kernelSpec.get("language").getAsString();
                } else {
                    logger.warn((Object)("Unable to detect language of notebook " + ipynbPath));
                }
            } else {
                logger.warn((Object)("Unable to parse notebook " + ipynbPath + ". It will be flagged as invalid."));
            }
            notebooks.add(notebookImportGitDTO);
        }
        return notebooks;
    }

    static File getFileInGit(File gitDir, String ipynbPath) {
        return new File(gitDir, ipynbPath);
    }

    public List<PreparedNotebookToImport> prepareImportNotebooks(AuthCtx user, String repository, String login, String password, String ref, File gitDir, List<NotebookImportGitDTO> notebooksToPrepare, GitRemoteCommands git) throws IOException, DKUSecurityException, CodedException {
        ArrayList<PreparedNotebookToImport> notebooks = new ArrayList<PreparedNotebookToImport>(notebooksToPrepare.size());
        for (NotebookImportGitDTO notebook : notebooksToPrepare) {
            JupyterService.NotebookSafeForWritingNew notebookObj;
            this.service.checkLicenceForLanguage(user, notebook.language);
            File file = JupyterGitService.getFileInGit(gitDir, notebook.getFullPath());
            if (!file.exists()) {
                logger.warnV("Trying to import a non existing notebook:%s", new Object[]{file.toString()});
                continue;
            }
            try (InputStreamReader reader = new InputStreamReader((InputStream)new FileInputStream(file), StandardCharsets.UTF_8);){
                notebookObj = (JupyterService.NotebookSafeForWritingNew)JSON.gson().fromJson((Reader)reader, JupyterService.NotebookSafeForWritingNew.class);
            }
            if (notebookObj.nbformat < 4) {
                logger.warn((Object)String.format("Trying to import a notebook:%s with a format: %d not supported", notebook.getFullPath(), notebookObj.nbformat));
                continue;
            }
            GitModel.GitReference gitReference = new GitModel.GitReference();
            gitReference.remote = repository;
            gitReference.remoteLogin = login;
            gitReference.remotePassword = password;
            gitReference.checkout = StringUtils.isNotBlank((String)ref) ? ref : git.getCurrentBranch();
            gitReference.remotePath = notebook.getFullPath();
            if (JupyterGitService.getFileInGit(gitDir, gitReference.remotePath).exists()) {
                gitReference.lastHash = git.getHashOfFile(notebook.getFullPath());
                gitReference.lastTimestamp = git.getLastCommitTimestamp(notebook.getFullPath());
            }
            JsonObject dkuMetadata = new JsonObject();
            dkuMetadata.add("gitReference", (JsonElement)JSON.toJsonObject((Object)this.gitReferencesService.encryptProjectGitReference(gitReference)));
            notebookObj.metadata.add("dkuGit", (JsonElement)dkuMetadata);
            notebooks.add(new PreparedNotebookToImport(notebook.name.replaceAll("[#\\.\\*/\\\\]", "_"), notebookObj));
        }
        return notebooks;
    }

    public void importNotebooks(AuthCtx user, String projectKey, List<PreparedNotebookToImport> prepareImportNotebooks) throws IOException {
        if (prepareImportNotebooks == null) {
            return;
        }
        StringTransmogrifier st2 = null;
        for (PreparedNotebookToImport prepareImportNotebook : prepareImportNotebooks) {
            RWTransaction t = this.transactionService.beginWriteAsLoggedInUser(user);
            try {
                if (st2 == null) {
                    st2 = new StringTransmogrifier();
                    for (JupyterService.JupyterNotebookListEntry e : this.service.listUnsafe(DSSAuthCtx.newNone(), projectKey)) {
                        st2.addAlreadyTransmogrified(e.name);
                    }
                }
                String nbName = st2.transmogrify(prepareImportNotebook.name);
                this.service.installNotebookFromFile(user, projectKey, nbName, null, prepareImportNotebook.notebook);
                this.service.updateNotebookFromFile(user, projectKey, nbName, prepareImportNotebook.notebook, null);
                t.commitV("Uploaded Jupyter notebook: '%s'", new Object[]{nbName});
            }
            finally {
                if (t == null) continue;
                t.close();
            }
        }
    }

    public Map<String, List<JupyterService.JupyterNotebookListEntry>> groupNotebooksByRepository(String projectKey, Collection<String> notebooks) throws IOException {
        HashMap<String, List<JupyterService.JupyterNotebookListEntry>> notebooksPerRepo = new HashMap<String, List<JupyterService.JupyterNotebookListEntry>>();
        for (String notebookName : notebooks) {
            JupyterService.JupyterNotebookListEntry notebook;
            try (Transaction ignored = this.transactionService.beginRead();){
                notebook = this.service.getMandatory(projectKey, notebookName, true);
            }
            if (notebook.gitReference == null) {
                logger.warn((Object)("Trying to push a non git notebook:" + notebookName));
                continue;
            }
            String key = notebook.gitReference.remote + notebook.gitReference.checkout;
            notebooksPerRepo.computeIfAbsent(key, k -> new ArrayList()).add(notebook);
        }
        return notebooksPerRepo;
    }

    public Map<String, List<NotebookWithRemoteHashFromConflict>> groupNotebooksByRepository(String projectKey, List<NotebookNameWithRemoteHashFromConflictDTO> notebooksWithHash) throws IOException {
        HashMap<String, List<NotebookWithRemoteHashFromConflict>> notebooksWithHashPerRepo = new HashMap<String, List<NotebookWithRemoteHashFromConflict>>();
        for (NotebookNameWithRemoteHashFromConflictDTO entry : notebooksWithHash) {
            JupyterService.JupyterNotebookListEntry notebook;
            try (Transaction ignored = this.transactionService.beginRead();){
                notebook = this.service.getMandatory(projectKey, entry.notebookName, true);
            }
            if (notebook.gitReference == null) {
                logger.warn((Object)("Trying to push a non git notebook:" + entry.notebookName));
                continue;
            }
            String key = notebook.gitReference.remote + notebook.gitReference.checkout;
            notebooksWithHashPerRepo.computeIfAbsent(key, k -> new ArrayList()).add(new NotebookWithRemoteHashFromConflict(notebook, entry.remoteHashFileDuringConflict));
        }
        return notebooksWithHashPerRepo;
    }

    public NotebooksPushGitDTO checkConflicts(AuthCtx user, Map<String, List<JupyterService.JupyterNotebookListEntry>> notebooksPerRepo, boolean checkForPull) throws CodedException, InterruptedException, UnauthorizedException, IOException, URISyntaxException {
        NotebooksPushGitDTO notebookStatus = new NotebooksPushGitDTO();
        for (Map.Entry<String, List<JupyterService.JupyterNotebookListEntry>> entry : notebooksPerRepo.entrySet()) {
            List<JupyterService.JupyterNotebookListEntry> notebookListEntries = entry.getValue();
            if (notebookListEntries == null || notebookListEntries.isEmpty()) continue;
            String tmpGitFolderName = "jupyter-git-exports-" + UrlRedactionUtils.cleanHttpUrls((String)entry.getKey());
            AutoDelete gitDir = DSSTempUtils.getTempFolder((String)tmpGitFolderName, (String)String.valueOf(System.currentTimeMillis()));
            try {
                GitRemoteCommands gitRemote = this.createGitRemote(gitDir);
                GitModel.GitReference gitReference = notebookListEntries.get((int)0).gitReference;
                gitRemote.shallowClone(user, this.gitReferencesService.getGitReferenceRemoteURL(gitReference), gitReference.checkout, this.getOldestDate(notebookListEntries));
                for (JupyterService.JupyterNotebookListEntry listEntry : notebookListEntries) {
                    String remoteHash = gitRemote.getHashOfFile(listEntry.gitReference.remotePath);
                    NotebookPushGitDTO notebookToPush = new NotebookPushGitDTO(listEntry.name, JupyterGitService.getName(listEntry.gitReference.remotePath), listEntry.gitReference.remote, listEntry.gitReference.checkout, listEntry.language, remoteHash);
                    if (StringUtils.isBlank((String)remoteHash)) {
                        notebookStatus.noLongerOnRemoteNotebooks.add(notebookToPush);
                        continue;
                    }
                    if (!checkForPull && this.areHashesCorrectForPush(gitRemote, listEntry, remoteHash) || checkForPull && JupyterGitService.hasNotBeenModifiedAfterGitInteraction(listEntry)) {
                        notebookToPush.selected = true;
                        notebookStatus.nonConflictingNotebooks.add(notebookToPush);
                        continue;
                    }
                    notebookStatus.conflictingNotebooks.add(notebookToPush);
                }
            }
            finally {
                if (gitDir == null) continue;
                gitDir.close();
            }
        }
        return notebookStatus;
    }

    GitRemoteCommands createGitRemote(AutoDelete gitDir) {
        return new GitRemoteCommands((File)gitDir);
    }

    private boolean areHashesCorrectForPush(GitRemoteCommands gitRemote, JupyterService.JupyterNotebookListEntry listEntry, String remoteHash) throws CodedException {
        if (!Objects.equals(listEntry.gitReference.lastHash, remoteHash)) {
            List<String> commits = gitRemote.listCommits(new Date(this.getLastTimestamp(listEntry)));
            if (commits.size() > 1) {
                return Objects.equals(commits.get(0), listEntry.gitReference.lastHash);
            }
            return false;
        }
        return true;
    }

    private Date getOldestDateForNotebooksWithHash(List<NotebookWithRemoteHashFromConflict> notebookListEntries) {
        Long oldestTimestamp = Long.MAX_VALUE;
        if (!notebookListEntries.isEmpty()) {
            oldestTimestamp = notebookListEntries.stream().map(n -> this.getLastTimestamp(n.notebook)).min(Long::compareTo).orElse(Long.MAX_VALUE);
        }
        if (oldestTimestamp != Long.MAX_VALUE) {
            return new Date(oldestTimestamp);
        }
        return null;
    }

    private Date getOldestDate(List<JupyterService.JupyterNotebookListEntry> notebookListEntries) {
        Long oldestTimestamp = Long.MAX_VALUE;
        if (!notebookListEntries.isEmpty()) {
            oldestTimestamp = notebookListEntries.stream().map(this::getLastTimestamp).min(Long::compareTo).orElse(Long.MAX_VALUE);
        }
        if (oldestTimestamp != Long.MAX_VALUE) {
            return new Date(oldestTimestamp);
        }
        return null;
    }

    private Long getLastTimestamp(JupyterService.JupyterNotebookListEntry entry) {
        if (entry == null || entry.gitReference == null || entry.gitReference.lastTimestamp == null) {
            return Long.MAX_VALUE;
        }
        return entry.gitReference.lastTimestamp;
    }

    public InfoMessage.InfoMessages pushNotebooks(AuthCtx user, String projectKey, Map<String, List<NotebookWithRemoteHashFromConflict>> notebooksPerRepoWithHash, String commitMessage, DKUtils.SmartLogTailBuilder logTailBuilder) throws CodedException, InterruptedException, UnauthorizedException, IOException, GitAPIException, URISyntaxException {
        NotebooksReportsDTO outputs = new NotebooksReportsDTO();
        for (List<NotebookWithRemoteHashFromConflict> notebookListEntries : notebooksPerRepoWithHash.values()) {
            if (notebookListEntries == null || notebookListEntries.isEmpty()) continue;
            GitModel.GitReference gitReference = notebookListEntries.get((int)0).notebook.gitReference;
            NotebooksReportsDTO pushReport = this.pushAllNotebooksIntoAGit(user, notebookListEntries, gitReference, projectKey, commitMessage, logTailBuilder);
            outputs.merge(pushReport);
        }
        return outputs.toPushInfoMessages();
    }

    private NotebooksReportsDTO pushAllNotebooksIntoAGit(AuthCtx user, List<NotebookWithRemoteHashFromConflict> notebookListEntries, GitModel.GitReference gitReference, String projectKey, String commitMessage, DKUtils.SmartLogTailBuilder logTailBuilder) throws CodedException, InterruptedException, UnauthorizedException, IOException, GitAPIException, URISyntaxException {
        String tmpGitFolderName = "jupyter-git-exports-" + UrlRedactionUtils.cleanHttpUrls((String)gitReference.remote) + gitReference.checkout;
        try (AutoDelete gitDir = DSSTempUtils.getTempFolder((String)tmpGitFolderName, (String)String.valueOf(System.currentTimeMillis()));){
            GitRemoteCommands gitRemote = this.createGitRemote(gitDir);
            gitRemote.shallowClone(user, this.gitReferencesService.getGitReferenceRemoteURL(gitReference), gitReference.checkout, this.getOldestDateForNotebooksWithHash(notebookListEntries));
            String gitName = UrlRedactionUtils.cleanHttpUrls((String)gitReference.remote) + "-" + gitReference.checkout;
            NotebooksReportsDTO notebooksPushReportsDTO = this.commitBeforePush(notebookListEntries, projectKey, (File)gitDir, user, gitName, commitMessage, gitRemote);
            if (!notebooksPushReportsDTO.successNotebooks.isEmpty()) {
                GitRemoteCommands.GitCommandResult pushResult = gitRemote.pushWithLogTail(user, "origin", gitReference.checkout, false, logTailBuilder);
                if (!pushResult.commandSucceeded) {
                    notebooksPushReportsDTO.setNotebooksAsInvalid(pushResult.messages.report());
                }
                Map<String, JupyterService.JupyterNotebookListEntry> notebooksMap = notebookListEntries.stream().collect(Collectors.toMap(n -> n.notebook.name, n -> n.notebook));
                try (RWTransaction rwTransaction = this.transactionService.beginWriteAsLoggedInUser(user);){
                    for (NotebooksReportDTO successNotebook : notebooksPushReportsDTO.successNotebooks) {
                        JupyterService.JupyterNotebookListEntry listEntry = notebooksMap.getOrDefault(successNotebook.notebookName, null);
                        if (listEntry == null) continue;
                        this.service.updateGitInformation(listEntry, successNotebook.commitHash, successNotebook.commitTimestamp);
                    }
                    rwTransaction.commit("Updated hashes after push to remote");
                }
            }
            NotebooksReportsDTO notebooksReportsDTO = notebooksPushReportsDTO;
            return notebooksReportsDTO;
        }
    }

    public InfoMessage.InfoMessages pullNotebooks(AuthCtx user, String projectKey, Map<String, List<JupyterService.JupyterNotebookListEntry>> notebooksPerRepo) throws IOException, InterruptedException, CodedException, UnauthorizedException, URISyntaxException {
        NotebooksReportsDTO outputs = new NotebooksReportsDTO();
        for (Map.Entry<String, List<JupyterService.JupyterNotebookListEntry>> entry : notebooksPerRepo.entrySet()) {
            List<JupyterService.JupyterNotebookListEntry> notebookListEntries = entry.getValue();
            if (notebookListEntries == null || notebookListEntries.isEmpty()) continue;
            GitModel.GitReference gitReference = notebookListEntries.get((int)0).gitReference;
            NotebooksReportsDTO gitOutputs = new NotebooksReportsDTO();
            String tmpGitFolderName = "jupyter-git-pull-" + UrlRedactionUtils.cleanHttpUrls((String)entry.getKey());
            try (AutoDelete gitDir = DSSTempUtils.getTempFolder((String)tmpGitFolderName, (String)String.valueOf(System.currentTimeMillis()));){
                GitRemoteCommands gitRemote = this.createGitRemote(gitDir);
                gitRemote.shallowClone(user, this.gitReferencesService.getGitReferenceRemoteURL(gitReference), gitReference.checkout);
                String gitName = UrlRedactionUtils.cleanHttpUrls((String)gitReference.remote) + "-" + gitReference.checkout;
                for (JupyterService.JupyterNotebookListEntry listEntry : notebookListEntries) {
                    File fileInGit = JupyterGitService.getFileInGit((File)gitDir, listEntry.gitReference.remotePath);
                    if (!fileInGit.exists()) {
                        logger.warnV("Trying to pull:%s that does not exist anymore on the remote", new Object[]{listEntry.gitReference.remotePath});
                        gitOutputs.warningNotebooks.add(this.createMessageNotebookNotExists(listEntry.name, gitName));
                        continue;
                    }
                    JupyterService.NotebookSafeForWritingNew notebookSafeForWritingNew = (JupyterService.NotebookSafeForWritingNew)JSON.parseFile((File)fileInGit, JupyterService.NotebookSafeForWritingNew.class);
                    String remoteHash = gitRemote.getHashOfFile(listEntry.gitReference.remotePath);
                    Long commitTimestamp = gitRemote.getLastCommitTimestamp(listEntry.gitReference.remotePath);
                    try {
                        RWTransaction rwTransaction = this.transactionService.beginWriteAsLoggedInUser(user);
                        try {
                            this.service.updateNotebookFromFile(user, projectKey, listEntry.name, notebookSafeForWritingNew, null);
                            this.service.updateGitInformation(listEntry, remoteHash, commitTimestamp);
                            rwTransaction.commitV("%s updated from:%s to:%s", new Object[]{listEntry.name, listEntry.gitReference.lastHash, remoteHash});
                            gitOutputs.successNotebooks.add(new NotebooksReportDTO(gitName, listEntry.name));
                        }
                        finally {
                            if (rwTransaction == null) continue;
                            rwTransaction.close();
                        }
                    }
                    catch (Exception e) {
                        gitOutputs.setNotebooksAsInvalid(e.getMessage());
                    }
                }
            }
            outputs.merge(gitOutputs);
        }
        return outputs.toPullInfoMessages();
    }

    public void editGitNotebookReference(AuthCtx user, String projectKey, String notebook, GitModel.GitReference reference) throws IOException, InterruptedException, CodedException, UnauthorizedException {
        ErrorContext.check((boolean)reference.remotePath.endsWith(NOTEBOOK_EXTENSION), (String)"remotePath must contains the name of the notebook ending with .ipynb");
        if (StringUtils.isBlank((String)reference.checkout)) {
            reference.checkout = "master";
        }
        if (reference.remotePath.startsWith("/")) {
            reference.remotePath = reference.remotePath.substring(1);
        }
        reference.lastTimestamp = null;
        reference.lastHash = null;
        try (RWTransaction rwTransaction = this.transactionService.beginWriteAsLoggedInUser(user);){
            JupyterService.JupyterNotebookListEntry listEntry = this.service.getMandatory(projectKey, notebook, true);
            this.service.updateGitInformation(listEntry, reference);
            rwTransaction.commitV("Update git reference of the notebook:%s", new Object[]{notebook});
        }
    }

    public void unlinkGitNotebookReference(AuthCtx user, String projectKey, String notebook) throws IOException {
        try (RWTransaction rwTransaction = this.transactionService.beginWriteAsLoggedInUser(user);){
            JupyterService.JupyterNotebookListEntry listEntry = this.service.getOrNullWithGitInformation(projectKey, notebook);
            this.service.unlinkGitInformation(listEntry);
            rwTransaction.commitV("Remove git reference of the notebook:%s", new Object[]{notebook});
        }
    }

    public FutureResponse<NotebooksPushGitDTO> checkConflictsAsync(AuthCtx user, final String projectKey, final List<String> notebooks, final boolean checkForPull) throws Exception {
        SimpleFutureThread<NotebooksPushGitDTO> futureConflictsChecksThread = new SimpleFutureThread<NotebooksPushGitDTO>(user){

            public FuturePayload getPayload() {
                return FuturePayload.newSimple((String)"check_conflicts_notebooks", (String)"Checks conflicts for notebooks");
            }

            @Override
            protected NotebooksPushGitDTO compute() throws Exception {
                Map<String, List<JupyterService.JupyterNotebookListEntry>> notebooksByRepository = JupyterGitService.this.groupNotebooksByRepository(projectKey, notebooks);
                return JupyterGitService.this.checkConflicts(this.owner, notebooksByRepository, checkForPull);
            }
        };
        return this.futureService.runFuture(futureConflictsChecksThread, 50L, new TypeToken<FutureResponse<NotebooksPushGitDTO>>(){});
    }

    public FutureResponse<List<NotebookImportGitDTO>> listNotebooksFromGitAsync(AuthCtx user, final String repository, final String login, final String password, final String ref) throws Exception {
        SimpleFutureThread<List<NotebookImportGitDTO>> futureListNotebooksThread = new SimpleFutureThread<List<NotebookImportGitDTO>>(user){

            public FuturePayload getPayload() {
                return FuturePayload.newSimple((String)"list_git_notebooks", (String)"List notebooks on remote git");
            }

            @Override
            protected List<NotebookImportGitDTO> compute() throws Exception {
                try (AutoDelete gitDir = DSSTempUtils.getTempFolder((String)"jupyter-git-list", (String)Long.toString(System.currentTimeMillis()));){
                    GitRemoteCommands git = JupyterGitService.this.createGitRemote(gitDir);
                    String repositoryUrl = JupyterGitService.this.gitReferencesService.mkRemoteURL(repository, login, password);
                    git.shallowClone(this.owner, repositoryUrl, ref);
                    List<String> ipynbFiles = git.listFiles("*.ipynb");
                    List<NotebookImportGitDTO> list = JupyterGitService.this.retrieveNotebooksFromGit((File)gitDir, ipynbFiles);
                    return list;
                }
            }
        };
        return this.futureService.runFuture(futureListNotebooksThread, 50L, new TypeToken<FutureResponse<List<NotebookImportGitDTO>>>(){});
    }

    public FutureResponse<Void> importNotebooksAsync(AuthCtx user, final String projectKey, final String repository, final String login, final String password, final String ref, final List<NotebookImportGitDTO> notebooks) throws Exception {
        SimpleFutureThread<Void> futureImportNotebooksThread = new SimpleFutureThread<Void>(user){

            public FuturePayload getPayload() {
                return FuturePayload.newSimple((String)"import_git_notebooks", (String)"Import notebooks from a remote git");
            }

            @Override
            protected Void compute() throws Exception {
                try (AutoDelete tmpDir = DSSTempUtils.getTempFolder((String)"jupyter-git-imports", (String)Long.toString(System.currentTimeMillis()));){
                    GitRemoteCommands git = JupyterGitService.this.createGitRemote(tmpDir);
                    git.shallowClone(this.owner, JupyterGitService.this.gitReferencesService.mkRemoteURL(repository, login, password), ref);
                    List<PreparedNotebookToImport> preparedNotebookToImports = JupyterGitService.this.prepareImportNotebooks(this.owner, repository, login, password, ref, (File)tmpDir, notebooks, git);
                    JupyterGitService.this.importNotebooks(this.owner, projectKey, preparedNotebookToImports);
                }
                return null;
            }
        };
        return this.futureService.runFuture(futureImportNotebooksThread, 50L, new TypeToken<FutureResponse<Void>>(){});
    }

    public FutureResponse<InfoMessage.InfoMessages> pushNotebooksAsync(AuthCtx user, final String projectKey, final List<NotebookNameWithRemoteHashFromConflictDTO> notebooksWithHash, final String message) throws Exception {
        SimpleFutureThread<InfoMessage.InfoMessages> futurePushNotebooksThread = new SimpleFutureThread<InfoMessage.InfoMessages>(user){
            DKUtils.SmartLogTailBuilder logTailBuilder;
            {
                super(owner);
                this.logTailBuilder = new DKUtils.SmartLogTailBuilder();
            }

            public FuturePayload getPayload() {
                return FuturePayload.newSimple((String)"push_git_notebooks", (String)"Push notebooks to a remote git");
            }

            @Override
            protected InfoMessage.InfoMessages compute() throws Exception {
                Map<String, List<NotebookWithRemoteHashFromConflict>> notebooksPerRepoWithHash = JupyterGitService.this.groupNotebooksByRepository(projectKey, notebooksWithHash);
                return JupyterGitService.this.pushNotebooks(this.owner, projectKey, notebooksPerRepoWithHash, message, this.logTailBuilder);
            }

            public SmartLogTail getLog() {
                return this.logTailBuilder.get();
            }
        };
        return this.futureService.runFuture(futurePushNotebooksThread, 50L, new TypeToken<FutureResponse<InfoMessage.InfoMessages>>(){});
    }

    public FutureResponse<InfoMessage.InfoMessages> pullNotebooksAsync(AuthCtx user, final String projectKey, final List<String> notebooks) throws Exception {
        SimpleFutureThread<InfoMessage.InfoMessages> futurePullNotebooksThread = new SimpleFutureThread<InfoMessage.InfoMessages>(user){

            public FuturePayload getPayload() {
                return FuturePayload.newSimple((String)"pull_git_notebooks", (String)"Pull notebooks from a remote git");
            }

            @Override
            protected InfoMessage.InfoMessages compute() throws Exception {
                Map<String, List<JupyterService.JupyterNotebookListEntry>> notebooksPerRepo = JupyterGitService.this.groupNotebooksByRepository(projectKey, notebooks);
                return JupyterGitService.this.pullNotebooks(this.owner, projectKey, notebooksPerRepo);
            }
        };
        return this.futureService.runFuture(futurePullNotebooksThread, 50L, new TypeToken<FutureResponse<InfoMessage.InfoMessages>>(){});
    }

    @VisibleForTesting
    NotebooksReportsDTO commitBeforePush(List<NotebookWithRemoteHashFromConflict> notebookListEntries, String projectKey, File gitDir, AuthCtx user, String gitName, String commitMessage, GitRemoteCommands gitRemoteCommands) throws IOException, GitAPIException, CodedException {
        NotebooksReportsDTO notebooksPushReportsDTO = new NotebooksReportsDTO();
        GitModel.GitAuthor gitAuthor = new GitModel.GitAuthor(user);
        for (NotebookWithRemoteHashFromConflict listEntry : notebookListEntries) {
            String hashOfFile;
            File fileInGit = JupyterGitService.getFileInGit(gitDir, listEntry.notebook.gitReference.remotePath);
            if (fileInGit.exists() && !Objects.equals(hashOfFile = gitRemoteCommands.getHashOfFile(fileInGit.getPath()), listEntry.remoteHashFileDuringConflict)) {
                notebooksPushReportsDTO.errorNotebooks.add(new NotebooksReportDTO(gitName, listEntry.notebook.name).withMessage("The remote file has changed since you have checked the conflict, please try again"));
                continue;
            }
            File file = this.transactionService.resolve(this.service.getNotebookFile(projectKey, listEntry.notebook.name));
            FileUtils.copyFile((File)file, (File)fileInGit);
            this.service.removeGitInformation(fileInGit);
            RevCommit commit = JupyterGitService.commitToGit(gitAuthor, fileInGit, gitDir, commitMessage);
            if (commit != null) {
                notebooksPushReportsDTO.successNotebooks.add(new NotebooksReportDTO(gitName, listEntry.notebook.name, commit.getName(), Integer.toUnsignedLong(commit.getCommitTime()) * 1000L));
                continue;
            }
            notebooksPushReportsDTO.infoNotebooks.add(this.createMessageNotebookWithoutChanges(listEntry.notebook.name, gitName));
        }
        return notebooksPushReportsDTO;
    }

    private NotebooksReportDTO createMessageNotebookNotExists(String notebookName, String gitName) {
        NotebooksReportDTO notebooksReportDTO = new NotebooksReportDTO();
        notebooksReportDTO.notebookName = notebookName;
        notebooksReportDTO.message = "File does not exist anymore on the remote git";
        notebooksReportDTO.gitId = gitName;
        return notebooksReportDTO;
    }

    private NotebooksReportDTO createMessageNotebookWithoutChanges(String notebookName, String gitName) {
        NotebooksReportDTO notebooksReportDTO = new NotebooksReportDTO();
        notebooksReportDTO.notebookName = notebookName;
        notebooksReportDTO.gitId = gitName;
        return notebooksReportDTO;
    }

    @VisibleForTesting
    static RevCommit commitToGit(GitModel.GitAuthor author, File fileInGit, File gitDir, String message) throws GitAPIException, IOException {
        CommitMessagePlaceholders commitMessagePlaceholders = new CommitMessagePlaceholders().withPlaceHolder("%filename%", fileInGit.getName()).withPlaceHolder("%author%", author.name);
        String commitMessage = commitMessagePlaceholders.getCommitMessage(StringUtils.isBlank((String)message) ? "Update %filename%" : message);
        try (Git git = Git.open((File)gitDir);){
            GitLocalCommands gitLocalCommands = new GitLocalCommands(git);
            gitLocalCommands.disableSymlinksIfNeeded();
            gitLocalCommands.disableHooksIfNeeded();
            String pattern = gitDir.toPath().relativize(fileInGit.toPath()).toString();
            git.add().addFilepattern(pattern).call();
            Status status = git.status().addPath(pattern).call();
            if (status.getChanged().isEmpty() && status.getAdded().isEmpty()) {
                RevCommit revCommit = null;
                return revCommit;
            }
            RevCommit revCommit = git.commit().setSign(Boolean.valueOf(false)).setMessage(commitMessage).setAuthor(author.name, author.getEmailOrName()).call();
            return revCommit;
        }
    }

    private static boolean hasNotBeenModifiedAfterGitInteraction(JupyterService.JupyterNotebookListEntry listEntry) {
        return listEntry.lastModifiedOn - listEntry.lastGitInteraction <= 1000L;
    }

    private static String getName(String remotePath) {
        File file = new File(remotePath);
        return file.getName().replace(NOTEBOOK_EXTENSION, "");
    }

    public static class NotebookImportGitDTO {
        String path;
        String name;
        String language;
        int nbFormat;
        int nbFormatMinor;
        boolean isValid;

        public String getFullPath() {
            Object nameWithExtension = this.name;
            String pathWithoutSep = this.path;
            if (!this.name.endsWith(JupyterGitService.NOTEBOOK_EXTENSION)) {
                nameWithExtension = (String)nameWithExtension + JupyterGitService.NOTEBOOK_EXTENSION;
            }
            if (pathWithoutSep.startsWith(File.separator)) {
                pathWithoutSep = pathWithoutSep.substring(File.separator.length());
            }
            if (File.separator.equals(pathWithoutSep) || "".equals(pathWithoutSep)) {
                return nameWithExtension;
            }
            return pathWithoutSep + File.separator + (String)nameWithExtension;
        }
    }

    public static class PreparedNotebookToImport {
        String name;
        JupyterService.NotebookSafeForWritingNew notebook;

        public PreparedNotebookToImport(String name, JupyterService.NotebookSafeForWritingNew notebookObj) {
            this.name = name;
            this.notebook = notebookObj;
        }
    }

    public static class NotebookNameWithRemoteHashFromConflictDTO {
        public String notebookName;
        public String remoteHashFileDuringConflict;

        public NotebookNameWithRemoteHashFromConflictDTO(String notebookName, String remoteHashFileDuringConflict) {
            this.notebookName = notebookName;
            this.remoteHashFileDuringConflict = remoteHashFileDuringConflict;
        }
    }

    public static class NotebookWithRemoteHashFromConflict {
        public JupyterService.JupyterNotebookListEntry notebook;
        public String remoteHashFileDuringConflict;

        public NotebookWithRemoteHashFromConflict(JupyterService.JupyterNotebookListEntry notebook, String remoteHashFileDuringConflict) {
            this.notebook = notebook;
            this.remoteHashFileDuringConflict = remoteHashFileDuringConflict;
        }
    }

    public static class NotebooksPushGitDTO {
        List<NotebookPushGitDTO> nonConflictingNotebooks = new ArrayList<NotebookPushGitDTO>();
        List<NotebookPushGitDTO> conflictingNotebooks = new ArrayList<NotebookPushGitDTO>();
        List<NotebookPushGitDTO> noLongerOnRemoteNotebooks = new ArrayList<NotebookPushGitDTO>();
    }

    public static class NotebookPushGitDTO {
        String notebookName;
        String gitUrl;
        String gitBranch;
        String language;
        String remoteNotebookName;
        boolean selected;
        String remoteHashFileDuringConflict;

        public NotebookPushGitDTO(String notebookName, String remoteNotebookName, String gitUrl, String gitBranch, String language, String remoteHashFileDuringConflict) {
            this.notebookName = notebookName;
            this.remoteNotebookName = remoteNotebookName;
            this.gitUrl = UrlRedactionUtils.sanitizeHttpUrls((String)gitUrl);
            this.gitBranch = gitBranch;
            this.language = language;
            this.remoteHashFileDuringConflict = remoteHashFileDuringConflict;
        }
    }

    public static class NotebooksReportsDTO {
        List<NotebooksReportDTO> successNotebooks = new ArrayList<NotebooksReportDTO>();
        List<NotebooksReportDTO> infoNotebooks = new ArrayList<NotebooksReportDTO>();
        List<NotebooksReportDTO> warningNotebooks = new ArrayList<NotebooksReportDTO>();
        List<NotebooksReportDTO> errorNotebooks = new ArrayList<NotebooksReportDTO>();

        public void setNotebooksAsInvalid(String errorMessage) {
            for (NotebooksReportDTO reportToConvert : this.successNotebooks) {
                reportToConvert.message = errorMessage;
                this.errorNotebooks.add(reportToConvert);
            }
            for (NotebooksReportDTO reportToConvert : this.infoNotebooks) {
                reportToConvert.message = errorMessage;
                this.errorNotebooks.add(reportToConvert);
            }
            for (NotebooksReportDTO reportToConvert : this.warningNotebooks) {
                reportToConvert.message = errorMessage;
                this.errorNotebooks.add(reportToConvert);
            }
            this.successNotebooks.clear();
            this.infoNotebooks.clear();
            this.warningNotebooks.clear();
        }

        public void merge(NotebooksReportsDTO otherRapport) {
            this.successNotebooks.addAll(otherRapport.successNotebooks);
            this.infoNotebooks.addAll(otherRapport.infoNotebooks);
            this.warningNotebooks.addAll(otherRapport.warningNotebooks);
            this.errorNotebooks.addAll(otherRapport.errorNotebooks);
        }

        public InfoMessage.InfoMessages toPushInfoMessages() {
            InfoMessage message;
            InfoMessage.InfoMessages infoMessages = new InfoMessage.InfoMessages();
            for (NotebooksReportDTO report : this.errorNotebooks) {
                message = InfoMessage.error((InfoMessage.MessageCode)GitCodes.INFO_GIT_PUSH_NOTEBOOK_ERROR, (String)(report.notebookName + "\n(" + report.gitId + "):\n" + report.message));
                infoMessages.addMessage(message);
            }
            for (NotebooksReportDTO report : this.warningNotebooks) {
                message = InfoMessage.warning((InfoMessage.MessageCode)GitCodes.INFO_GIT_PUSH_NOTEBOOK_ERROR, (String)(report.notebookName + "\n(" + report.gitId + "):\n" + report.message));
                infoMessages.addMessage(message);
            }
            for (NotebooksReportDTO report : this.infoNotebooks) {
                message = InfoMessage.info((InfoMessage.MessageCode)GitCodes.INFO_GIT_PUSH_NOTEBOOK_NO_MODIFICATION, (String)(report.notebookName + "\n(" + report.gitId + ")"));
                infoMessages.addMessage(message);
            }
            for (NotebooksReportDTO report : this.successNotebooks) {
                message = InfoMessage.success((InfoMessage.MessageCode)GitCodes.INFO_GIT_PUSH_NOTEBOOK_OK, (String)(report.notebookName + "\n(" + report.gitId + ")"));
                infoMessages.addMessage(message);
            }
            return infoMessages;
        }

        public InfoMessage.InfoMessages toPullInfoMessages() {
            InfoMessage message;
            InfoMessage.InfoMessages infoMessages = new InfoMessage.InfoMessages();
            for (NotebooksReportDTO report : this.errorNotebooks) {
                message = InfoMessage.error((InfoMessage.MessageCode)GitCodes.INFO_GIT_PULL_NOTEBOOK_ERROR, (String)(report.notebookName + "\n(" + report.gitId + "):\n" + report.message));
                infoMessages.addMessage(message);
            }
            for (NotebooksReportDTO report : this.warningNotebooks) {
                message = InfoMessage.warning((InfoMessage.MessageCode)GitCodes.INFO_GIT_PULL_NOTEBOOK_ERROR, (String)(report.notebookName + "\n(" + report.gitId + "):\n" + report.message));
                infoMessages.addMessage(message);
            }
            for (NotebooksReportDTO report : this.infoNotebooks) {
                message = InfoMessage.info((InfoMessage.MessageCode)GitCodes.INFO_GIT_PULL_NOTEBOOK_NO_MODIFICATION, (String)(report.notebookName + "\n(" + report.gitId + ")"));
                infoMessages.addMessage(message);
            }
            for (NotebooksReportDTO report : this.successNotebooks) {
                message = InfoMessage.success((InfoMessage.MessageCode)GitCodes.INFO_GIT_PULL_NOTEBOOK_OK, (String)(report.notebookName + "\n(" + report.gitId + ")"));
                infoMessages.addMessage(message);
            }
            return infoMessages;
        }
    }

    public static class NotebooksReportDTO {
        String message;
        String gitId;
        String notebookName;
        @Expose(serialize=false, deserialize=false)
        String commitHash;
        @Expose(serialize=false, deserialize=false)
        Long commitTimestamp;

        public NotebooksReportDTO() {
        }

        public NotebooksReportDTO(String gitId, String notebookName) {
            this(gitId, notebookName, null, null);
        }

        public NotebooksReportDTO(String gitId, String notebookName, String commitHash, Long commitTimestamp) {
            this.gitId = gitId;
            this.notebookName = notebookName;
            this.commitHash = commitHash;
            this.commitTimestamp = commitTimestamp;
        }

        public NotebooksReportDTO withMessage(String message) {
            this.message = message;
            return this;
        }
    }

    public static class CommitMessagePlaceholders {
        Map<String, String> placeholders = new HashMap<String, String>();

        public CommitMessagePlaceholders withPlaceHolder(String placeHolder, String replacement) {
            Object pattern = placeHolder;
            if (!((String)pattern).endsWith("%")) {
                pattern = (String)pattern + "%";
            }
            if (!((String)pattern).startsWith("%")) {
                pattern = "%" + (String)pattern;
            }
            this.placeholders.put((String)pattern, replacement);
            return this;
        }

        public String getCommitMessage(String messageWithPlaceholder) {
            String commitMessage = messageWithPlaceholder;
            for (Map.Entry<String, String> entry : this.placeholders.entrySet()) {
                commitMessage = commitMessage.replaceAll(entry.getKey(), entry.getValue());
            }
            return commitMessage;
        }
    }
}

