/*
 * Decompiled with CFR 0.152.
 */
package com.dataiku.dip.transactions.git.cli;

import com.dataiku.common.server.SerializedError;
import com.dataiku.dip.DKUApp;
import com.dataiku.dip.coremodel.InfoMessage;
import com.dataiku.dip.coremodel.SimpleKeyValue;
import com.dataiku.dip.dao.GeneralSettingsDAO;
import com.dataiku.dip.exceptions.CodedException;
import com.dataiku.dip.exceptions.UnauthorizedException;
import com.dataiku.dip.futures.FutureProgress;
import com.dataiku.dip.futures.FutureProgressState;
import com.dataiku.dip.security.AuthCtx;
import com.dataiku.dip.security.UrlRedactionUtils;
import com.dataiku.dip.transactions.TransactionContext;
import com.dataiku.dip.transactions.git.GitCodes;
import com.dataiku.dip.transactions.git.GitModel;
import com.dataiku.dip.transactions.git.cli.GitConfigurationRulesHelper;
import com.dataiku.dip.transactions.git.jgit.GitLocalCommands;
import com.dataiku.dip.transactions.git.jgit.JGitWrapper;
import com.dataiku.dip.utils.DKULogger;
import com.dataiku.dip.utils.DKUtils;
import com.dataiku.dip.utils.ExceptionUtils;
import com.dataiku.dip.utils.JSON;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Scanner;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import org.apache.commons.lang.StringUtils;

public class GitRemoteCommands {
    public static final String DEFAULT_BRANCH_NAME = "main";
    public static final String DEFAULT_ORIGIN = "origin";
    public static final String DKU_CORE_FS_PULL_WITH_TAGS = "dku.core.fs.pullWithTags";
    public static final String DKU_CORE_FS_PUSH_WITH_TAGS = "dku.core.fs.pushWithTags";
    private final File rootFolder;
    private final SimpleDateFormat dateformat = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSSZ");
    private static final DKULogger logger = DKULogger.getLogger((String)"dku.gitcli");

    public GitRemoteCommands(File rootFolder) {
        this.rootFolder = rootFolder;
    }

    public List<String> listRefs(AuthCtx authCtx, String repositoryUrl) throws CodedException {
        ArrayList gitListRemotes = Lists.newArrayList((Object[])new String[]{"ls-remote", "--heads", "--tags", "-q", "--exit-code", repositoryUrl});
        String output = this.gitExecWithRemoteURLAndSecurityCheck(authCtx, repositoryUrl, gitListRemotes, GitCodes.ERR_GIT_LIST_REFS_FAILED, "Branches and tags could not be fetched from remote Git", new Object[0]);
        Pattern refPattern = Pattern.compile("(refs/(?:heads|tags)/.+)(?<!\\^\\{})$");
        ArrayList<String> refs = new ArrayList<String>();
        try (Scanner scanner = new Scanner(output);){
            Matcher branchMatcher = refPattern.matcher("");
            while (scanner.hasNextLine()) {
                String line = scanner.nextLine();
                branchMatcher.reset(line);
                if (!branchMatcher.find()) continue;
                refs.add(branchMatcher.group(1));
            }
        }
        return refs;
    }

    public static String getFullBranchName(String branchName) {
        return branchName.startsWith("refs/") ? branchName : "refs/heads/" + branchName;
    }

    public GitCommandResult pullRebase(AuthCtx authCtx, String remoteName, String branchName, String hash, DKUtils.SmartLogTailBuilder logTailBuilder, boolean revertOnFail) {
        GitAuthCtxWithRemoteName ctx = new GitAuthCtxWithRemoteName(authCtx, remoteName);
        ArrayList cmd = Lists.newArrayList((Object[])new String[]{"pull", "--rebase"});
        if (StringUtils.isNotBlank((String)remoteName)) {
            cmd.add(remoteName);
        }
        boolean withTags = DKUApp.getParams().getBoolParam(DKU_CORE_FS_PULL_WITH_TAGS, true);
        if (StringUtils.isNotBlank((String)branchName)) {
            Preconditions.checkArgument((boolean)StringUtils.isNotBlank((String)remoteName), (Object)"Cannot specify the git branch name without the git remote name");
            cmd.add(GitRemoteCommands.getFullBranchName(branchName));
            if (withTags) {
                cmd.add("--tags");
            }
        } else if (StringUtils.isNotBlank((String)hash)) {
            Preconditions.checkArgument((boolean)StringUtils.isNotBlank((String)remoteName), (Object)"Cannot specify the hash  without the git remote name");
            cmd.add(hash);
        } else if (withTags) {
            logTailBuilder.appendLine("# Git fetch tags");
            ArrayList cmdPullTags = Lists.newArrayList((Object[])new String[]{"fetch", "--tags"});
            this.gitExecWithLogTailWithRemoteNameAndSecurityCheck(ctx, cmdPullTags, logTailBuilder, GitCodes.INFO_GIT_PULL_OK, "Successfully pulled", GitCodes.ERR_GIT_PULL_FAILED, String.format("Git fetch tags failed on %s", StringUtils.defaultIfBlank((String)remoteName, (String)DEFAULT_ORIGIN)));
        }
        GitCommandResult commandResult = this.gitExecWithLogTailWithRemoteNameAndSecurityCheck(ctx, cmd, logTailBuilder, GitCodes.INFO_GIT_PULL_OK, "Successfully pulled", GitCodes.ERR_GIT_PULL_FAILED, String.format("Git pull failed on %s - %s", StringUtils.defaultIfBlank((String)remoteName, (String)DEFAULT_ORIGIN), StringUtils.defaultIfBlank((String)branchName, (String)DEFAULT_BRANCH_NAME)));
        if (!commandResult.commandSucceeded && !GitRemoteCommands.isRemoteGitPermissionError(commandResult.commandError) && revertOnFail) {
            try {
                ArrayList cmdAbort = Lists.newArrayList((Object[])new String[]{"rebase", "--abort"});
                this.gitExecWithoutRemoteNameNorSecurityCheck(true, cmdAbort, GitCodes.ERR_LOCAL_GIT_FAILED_UNSPECIFIED, "Git rebase abort failed", new Object[0]);
                logTailBuilder.appendLine("# Git pull failed on merging");
                logTailBuilder.appendLine("# Rebasing was aborted, discard your changes or perform the merge outside of DSS");
            }
            catch (Exception exception) {
                // empty catch block
            }
        }
        return commandResult;
    }

    public GitCommandResult pullRebase(AuthCtx authCtx, String remoteName, String branchName, DKUtils.SmartLogTailBuilder logTailBuilder) {
        return this.pullRebase(authCtx, remoteName, branchName, null, logTailBuilder, true);
    }

    public GitCommandResult fetch(AuthCtx authCtx, String remoteName, DKUtils.SmartLogTailBuilder logTailBuilder) {
        GitAuthCtxWithRemoteName ctx = new GitAuthCtxWithRemoteName(authCtx, remoteName);
        ArrayList cmd = Lists.newArrayList((Object[])new String[]{"fetch"});
        if (StringUtils.isNotBlank((String)remoteName)) {
            cmd.add(remoteName);
        }
        return this.gitExecWithLogTailWithRemoteNameAndSecurityCheck(ctx, cmd, logTailBuilder, GitCodes.INFO_GIT_FETCH_OK, "Successfully fetched", GitCodes.ERR_GIT_FETCH_FAILED, "Git fetch failed");
    }

    public GitCommandResult merge(AuthCtx authCtx, String remoteName, String branchToMerge, DKUtils.SmartLogTailBuilder logTailBuilder) {
        ArrayList cmd = Lists.newArrayList((Object[])new String[]{"merge"});
        if (StringUtils.isNotBlank((String)remoteName)) {
            cmd.add(remoteName + "/" + branchToMerge);
        } else {
            cmd.add(branchToMerge);
        }
        return this.gitExecWithLogTailWithoutRemoteNameNorSecurityCheck(true, cmd, logTailBuilder, GitCodes.INFO_GIT_MERGED_OK, "Successfully merged", GitCodes.ERR_GIT_MERGE_FAILED, "Git merged failed");
    }

    public GitCommandResult deleteBranch(String branchName, DKUtils.SmartLogTailBuilder logTailBuilder) {
        ArrayList cmd = Lists.newArrayList((Object[])new String[]{"branch", "-D", branchName});
        return this.gitExecWithLogTailWithoutRemoteNameNorSecurityCheck(true, cmd, logTailBuilder, GitCodes.INFO_GIT_MERGED_OK, "Successfully merged", GitCodes.ERR_GIT_MERGE_FAILED, "Git merged failed");
    }

    public GitCommandResult reset(AuthCtx authCtx, String remoteName, String branchToMerge, DKUtils.SmartLogTailBuilder logTailBuilder) {
        GitAuthCtxWithRemoteName ctx = new GitAuthCtxWithRemoteName(authCtx, remoteName);
        ArrayList cmd = Lists.newArrayList((Object[])new String[]{"reset", "--hard"});
        if (StringUtils.isNotBlank((String)remoteName)) {
            cmd.add(remoteName + "/" + branchToMerge);
        } else {
            cmd.add(branchToMerge);
        }
        return this.gitExecWithLogTailWithRemoteNameAndSecurityCheck(ctx, cmd, logTailBuilder, GitCodes.INFO_GIT_RESET_OK, "Successfully reset", GitCodes.ERR_GIT_RESET_FAILED, "Git reset failed");
    }

    public void setUpstream(AuthCtx authCtx, String remoteName, String branchName) throws CodedException {
        GitAuthCtxWithRemoteName ctx = new GitAuthCtxWithRemoteName(authCtx, remoteName);
        ArrayList cmd = Lists.newArrayList((Object[])new String[]{"branch", "--set-upstream-to"});
        assert (StringUtils.isNotBlank((String)branchName));
        String remoteOrigin = StringUtils.defaultIfBlank((String)remoteName, (String)DEFAULT_ORIGIN);
        cmd.add(String.format("%s/%s", remoteOrigin, branchName));
        cmd.add(branchName);
        this.gitExecWithRemoteNameAndSecurityCheck(ctx, cmd, GitCodes.ERR_GIT_BRANCH_UPSTREAM_FAILED, "Git branch failed (%s - %s)", remoteOrigin, StringUtils.defaultIfBlank((String)branchName, (String)"no branch specified"));
    }

    public GitCommandResult pushWithLogTail(AuthCtx authCtx, String remoteName, String branchName, boolean setUpstream, DKUtils.SmartLogTailBuilder logTailBuilder) {
        GitAuthCtxWithRemoteName ctx = new GitAuthCtxWithRemoteName(authCtx, remoteName);
        List<String> cmd = GitRemoteCommands.getPushCmd(remoteName, branchName, setUpstream);
        if (DKUApp.getParams().getBoolParam(DKU_CORE_FS_PUSH_WITH_TAGS, true)) {
            cmd.add("--follow-tags");
        }
        return this.gitExecWithLogTailWithRemoteNameAndSecurityCheck(ctx, cmd, logTailBuilder, GitCodes.INFO_GIT_PUSH_OK, "Successfully pushed", GitCodes.ERR_GIT_PUSH_FAILED, String.format("Git push failed on %s - %s", StringUtils.defaultIfBlank((String)remoteName, (String)DEFAULT_ORIGIN), StringUtils.defaultIfBlank((String)branchName, (String)DEFAULT_BRANCH_NAME)));
    }

    public String push(AuthCtx authCtx, String remoteName, String branchName, boolean setUpstream) throws CodedException {
        GitAuthCtxWithRemoteName ctx = new GitAuthCtxWithRemoteName(authCtx, remoteName);
        List<String> cmd = GitRemoteCommands.getPushCmd(remoteName, branchName, setUpstream);
        return this.gitExecWithRemoteNameAndSecurityCheck(ctx, cmd, GitCodes.ERR_GIT_PUSH_FAILED, "Git push failed on %s - %s", StringUtils.defaultIfBlank((String)remoteName, (String)DEFAULT_ORIGIN), StringUtils.defaultIfBlank((String)branchName, (String)DEFAULT_BRANCH_NAME));
    }

    private static List<String> getPushCmd(String remoteName, String branchName, boolean setUpstream) {
        ArrayList cmd = Lists.newArrayList((Object[])new String[]{"push", "-v"});
        if (setUpstream) {
            cmd.add("--set-upstream");
        }
        if (StringUtils.isNotBlank((String)remoteName)) {
            cmd.add(remoteName);
        }
        if (StringUtils.isNotBlank((String)branchName)) {
            cmd.add(GitRemoteCommands.getFullBranchName(branchName));
        }
        return cmd;
    }

    public String deleteRemoteBranch(AuthCtx authCtx, String remoteName, String branchName) throws CodedException {
        Preconditions.checkNotNull((Object)branchName);
        GitAuthCtxWithRemoteName ctx = new GitAuthCtxWithRemoteName(authCtx, remoteName);
        String remoteOrigin = StringUtils.defaultIfBlank((String)remoteName, (String)DEFAULT_ORIGIN);
        List<String> cmd = GitRemoteCommands.getDeleteRemoteBranchCmd(remoteOrigin, branchName);
        return this.gitExecWithRemoteNameAndSecurityCheck(ctx, cmd, GitCodes.ERR_GIT_PUSH_FAILED, "Git push (remote branch delete) failed on %s - %s", remoteOrigin, branchName);
    }

    private static List<String> getDeleteRemoteBranchCmd(String remoteName, String branchName) {
        return Lists.newArrayList((Object[])new String[]{"push", remoteName, "--delete", branchName});
    }

    public String fetch(AuthCtx authCtx, String remoteName, String branchName) throws CodedException {
        GitAuthCtxWithRemoteName ctx = new GitAuthCtxWithRemoteName(authCtx, remoteName);
        ArrayList cmd = Lists.newArrayList((Object[])new String[]{"fetch", "-v"});
        if (StringUtils.isNotBlank((String)remoteName)) {
            cmd.add(remoteName);
        }
        if (StringUtils.isNotBlank((String)branchName)) {
            cmd.add(branchName);
        }
        return this.gitExecWithRemoteNameAndSecurityCheck(ctx, cmd, GitCodes.ERR_GIT_FETCH_FAILED, "Git fetch failed (%s - %s)", StringUtils.defaultIfBlank((String)remoteName, (String)DEFAULT_ORIGIN), StringUtils.defaultIfBlank((String)branchName, (String)"no branch specified"));
    }

    public String addRemoteRepository(AuthCtx authCtx, String remoteName, String remoteUrl, boolean remoteFetch) throws CodedException, UnauthorizedException {
        ArrayList cmd = Lists.newArrayList((Object[])new String[]{"remote", "add", remoteName, remoteUrl});
        if (remoteFetch) {
            cmd.add(3, "-f");
            return this.gitExecWithRemoteURLAndSecurityCheck(authCtx, remoteUrl, cmd, GitCodes.ERR_GIT_ADD_REMOTE_FAILED, "An error happened while adding the following remote: %s (%s), and then fetching it", remoteName, remoteUrl);
        }
        if (StringUtils.isNotBlank((String)remoteUrl) && remoteUrl.startsWith("/")) {
            logger.info((Object)("Url '" + remoteUrl + "' is local, not checking if access to the remote git is allowed for the user"));
        } else {
            GitConfigurationRulesHelper.checkIfGitRemoteURLAllowed(authCtx, remoteUrl);
        }
        return this.gitExecWithoutRemoteNameNorSecurityCheck(true, cmd, GitCodes.ERR_GIT_ADD_REMOTE_FAILED, "An error happened while adding the following remote: %s (%s), without fetching it", remoteName, remoteUrl);
    }

    public String changeRemoteURL(AuthCtx authCtx, String remoteName, String remoteUrl) throws CodedException, UnauthorizedException {
        GitConfigurationRulesHelper.checkIfGitRemoteURLAllowed(authCtx, remoteUrl);
        ArrayList cmd = Lists.newArrayList((Object[])new String[]{"remote", "set-url", remoteName, remoteUrl});
        return this.gitExecWithoutRemoteNameNorSecurityCheck(true, cmd, GitCodes.ERR_GIT_REMOTE_SET_URL_FAILED, "An error happened while setting the URL on the following remote: %s (%s)", remoteName, remoteUrl);
    }

    public String removeRemoteRepository(String remoteName) throws CodedException {
        ArrayList cmd = Lists.newArrayList((Object[])new String[]{"remote", "rm", remoteName});
        return this.gitExecWithoutRemoteNameNorSecurityCheck(true, cmd, GitCodes.ERR_GIT_RM_REMOTE_FAILED, "An error happened while removing the following remote: %s", remoteName);
    }

    public String checkoutBranch(String branchName) throws CodedException {
        ArrayList cmd = Lists.newArrayList((Object[])new String[]{"checkout", branchName});
        return this.gitExecWithoutRemoteNameNorSecurityCheck(true, cmd, GitCodes.ERR_GIT_CHECKOUT_FAILED, "Git checkout failed on '%s'", branchName);
    }

    public GitCommandResult checkoutBranch(String branchName, DKUtils.SmartLogTailBuilder logTailBuilder) {
        ArrayList cmd = Lists.newArrayList((Object[])new String[]{"checkout", branchName});
        return this.gitExecWithLogTailWithoutRemoteNameNorSecurityCheck(true, cmd, logTailBuilder, GitCodes.INFO_GIT_CHECKOUT_BRANCH_OK, String.format("Successfully switched to '%s'", branchName), GitCodes.ERR_GIT_CHECKOUT_FAILED, String.format("Git checkout failed on '%s'", branchName));
    }

    public String createAndCheckoutBranch(String branchName, @Nullable String commitHash) throws CodedException {
        ArrayList cmd = Lists.newArrayList((Object[])new String[]{"checkout", "-b", branchName});
        if (!StringUtils.isBlank((String)commitHash)) {
            cmd.add(commitHash);
        }
        return this.gitExecWithoutRemoteNameNorSecurityCheck(true, cmd, GitCodes.ERR_GIT_BRANCH_CREATION_FAILED, "Git branch creation failed on '%s'", branchName);
    }

    public String getHashFor(String ref) throws CodedException {
        ArrayList cmd = Lists.newArrayList((Object[])new String[]{"rev-parse", ref});
        return this.gitExecWithoutRemoteNameNorSecurityCheck(true, cmd, GitCodes.ERR_GIT_REV_PARSE_FAILED, "Git rev-parse failed on '%s'", ref).trim();
    }

    public String getHashOfFile(String path) throws CodedException {
        ArrayList logCmd = Lists.newArrayList((Object[])new String[]{"log", "-n", "1", "--pretty=format:%H", "--", path});
        return this.gitExecWithoutRemoteNameNorSecurityCheck(true, logCmd, GitCodes.ERR_GIT_LOG_FAILED, "Git log failed on '%s'", path).trim();
    }

    public long getLastCommitTimestamp(String path) throws CodedException {
        ArrayList logCmd = Lists.newArrayList((Object[])new String[]{"log", "-n", "1", "--pretty=format:%at", "--", path});
        return Long.parseLong(this.gitExecWithoutRemoteNameNorSecurityCheck(true, logCmd, GitCodes.ERR_GIT_LOG_FAILED, "Git log failed on '%s'", path).trim()) * 1000L;
    }

    public String getCurrentBranch() throws CodedException {
        ArrayList cmd = Lists.newArrayList((Object[])new String[]{"rev-parse", "--abbrev-ref", "HEAD"});
        return this.gitExecWithoutRemoteNameNorSecurityCheck(true, cmd, GitCodes.ERR_GIT_REV_PARSE_FAILED, "Git rev-parse failed on HEAD", new Object[0]).trim();
    }

    public void clone(AuthCtx authCtx, String repositoryUrl, String checkout, String folderName, boolean copyHistory) throws CodedException, InterruptedException, UnauthorizedException {
        GitConfigurationRulesHelper.checkIfGitRemoteURLAllowed(authCtx, repositoryUrl);
        ArrayList cmd = Lists.newArrayList((Object[])new String[]{"clone", repositoryUrl, "-q", "--progress"});
        if (!copyHistory) {
            cmd.add("--depth");
            cmd.add("1");
        }
        if (StringUtils.isNotBlank((String)checkout)) {
            cmd.add("--single-branch");
            cmd.add("--branch");
            cmd.add(GitLocalCommands.shortenRefName((String)checkout));
        }
        cmd.add(folderName);
        try {
            this.gitExecCloneInFutureThread(authCtx, repositoryUrl, cmd, this.rootFolder);
        }
        catch (Exception ex) {
            if (Thread.currentThread().isInterrupted()) {
                throw new InterruptedException(ExceptionUtils.getMessageWithCauses((Throwable)ex));
            }
            throw new CodedException((InfoMessage.MessageCode)GitCodes.ERR_GIT_CLONE_FAILED, "Git clone was unsuccessful on '" + UrlRedactionUtils.sanitizeHttpUrls((String)repositoryUrl) + "'", (Throwable)ex);
        }
    }

    public void shallowClone(AuthCtx authCtx, String repositoryUrl, String checkout, Date since) throws UnauthorizedException, CodedException, InterruptedException {
        ArrayList gitClone;
        GitConfigurationRulesHelper.checkIfGitRemoteURLAllowed(authCtx, repositoryUrl);
        ArrayList gitCheckout = null;
        Pattern hashPattern = Pattern.compile("^[0-9a-f]{7,}$");
        if (StringUtils.isNotBlank((String)checkout) && hashPattern.matcher(checkout).matches()) {
            gitClone = Lists.newArrayList((Object[])new String[]{"clone", "-q", "--progress", repositoryUrl, "."});
            gitCheckout = Lists.newArrayList((Object[])new String[]{"checkout", checkout});
        } else {
            gitClone = Lists.newArrayList((Object[])new String[]{"clone", "-q", "--progress"});
            if (since != null) {
                gitClone.add("--single-branch");
                gitClone.add("--shallow-since=" + this.dateformat.format(since));
            } else {
                gitClone.add("--depth");
                gitClone.add("1");
            }
            if (StringUtils.isNotBlank((String)checkout)) {
                gitClone.add("-b");
                gitClone.add(GitLocalCommands.shortenRefName((String)checkout));
            }
            gitClone.add(repositoryUrl);
            gitClone.add(".");
        }
        try {
            this.gitExecCloneInFutureThread(authCtx, repositoryUrl, gitClone, this.rootFolder);
        }
        catch (Exception ex) {
            if (Thread.currentThread().isInterrupted()) {
                throw new InterruptedException(ExceptionUtils.getMessageWithCauses((Throwable)ex));
            }
            if (since != null) {
                logger.info((Object)"It seems like your git has failed to shallow-since, we are retrying without the option");
                this.simulateShallowSinceWithOldGit(authCtx, repositoryUrl, checkout, since);
            }
            throw new CodedException((InfoMessage.MessageCode)GitCodes.ERR_GIT_CLONE_FAILED, "Git clone was unsuccessful on '" + UrlRedactionUtils.sanitizeHttpUrls((String)repositoryUrl) + "'", (Throwable)ex);
        }
        if (gitCheckout != null) {
            try {
                ProcessBuilder pb = this.getProcessBuilderWithRemoteURLAndSecurityCheck(authCtx, repositoryUrl, gitCheckout);
                pb.directory(this.rootFolder);
                JGitWrapper.execAndGetOutputWithErrorInException((ProcessBuilder)pb, (String)"Git checkout failed");
            }
            catch (Exception ex) {
                throw new CodedException((InfoMessage.MessageCode)GitCodes.ERR_GIT_CHECKOUT_FAILED, "Git checkout failed on '" + checkout + "'", (Throwable)ex);
            }
        }
    }

    private void simulateShallowSinceWithOldGit(AuthCtx authCtx, String repositoryUrl, String checkout, Date since) throws InterruptedException, CodedException, UnauthorizedException {
        this.clone(authCtx, repositoryUrl, checkout, ".", true);
        if (since != null) {
            String numberStr = this.numbersOfCommitsSince("HEAD", since);
            int number = Integer.parseInt(numberStr);
            ArrayList cmd = Lists.newArrayList((Object[])new String[]{"fetch", "--depth=" + number});
            this.gitExecWithoutRemoteNameNorSecurityCheck(true, cmd, GitCodes.ERR_GIT_FETCH_FAILED, "Git fetch --depth failed", "");
        }
    }

    private String numbersOfCommitsSince(String ref, Date since) throws CodedException {
        ArrayList cmd = Lists.newArrayList((Object[])new String[]{"rev-list", ref, "--since=" + this.dateformat.format(since), "--count"});
        return this.gitExecWithoutRemoteNameNorSecurityCheck(true, cmd, GitCodes.ERR_GIT_REV_LIST_FAILED, "Git rev-list failed on '%s'", ref).trim();
    }

    public void shallowClone(AuthCtx authCtx, String repositoryUrl, String checkout) throws CodedException, InterruptedException, UnauthorizedException {
        this.shallowClone(authCtx, repositoryUrl, checkout, null);
    }

    public List<String> listFiles(String pattern) throws CodedException {
        Preconditions.checkArgument((boolean)StringUtils.isNotBlank((String)pattern), (Object)"Project key is not specified");
        ArrayList gitLs = Lists.newArrayList((Object[])new String[]{"-c", "core.quotepath=off", "ls-files", "-z", pattern});
        String data = this.gitExecWithoutRemoteNameNorSecurityCheck(true, gitLs, GitCodes.ERR_GIT_LIST_FILES_FAILED, String.format("Git ls-files failed on:%s with:%s pattern", this.rootFolder, pattern), new Object[0]);
        if (StringUtils.isBlank((String)data)) {
            return Collections.emptyList();
        }
        return Arrays.asList(data.split("\u0000"));
    }

    public Set<String> listFilesUnmerged(String subPath) throws CodedException {
        ArrayList gitLs = Lists.newArrayList((Object[])new String[]{"--no-pager", "diff", "--name-only", "--diff-filter=U"});
        String data = this.gitExecWithoutRemoteNameNorSecurityCheck(true, gitLs, GitCodes.ERR_GIT_LIST_FILES_FAILED, String.format("Git ls-files conflict on:%s ", this.rootFolder), new Object[0]);
        if (StringUtils.isBlank((String)data)) {
            return new HashSet<String>();
        }
        if (StringUtils.isNotBlank((String)subPath)) {
            return Arrays.stream(data.split("\n")).map(val -> val.replaceFirst(subPath + File.separator, "")).collect(Collectors.toSet());
        }
        return new HashSet<String>(Arrays.asList(data.split("\n")));
    }

    public List<String> listCommits(Date date) throws CodedException {
        String dateFormatted = this.dateformat.format(date);
        ArrayList cmd = Lists.newArrayList((Object[])new String[]{"log", "--after=" + dateFormatted, "--before=" + dateFormatted, "--pretty=format:%H"});
        String output = this.gitExecWithoutRemoteNameNorSecurityCheck(true, cmd, GitCodes.ERR_GIT_LOG_FAILED, "Git log date failed", "").trim();
        if (StringUtils.isBlank((String)output)) {
            return Collections.emptyList();
        }
        return Arrays.asList(output.split("\n"));
    }

    private GitCommandResult gitExecWithLogTailWithRemoteNameAndSecurityCheck(GitAuthCtxWithRemoteName ctx, List<String> gitCommand, DKUtils.SmartLogTailBuilder logTailBuilder, GitCodes infoCode, String infoMsg, GitCodes errorCode, String errorMsg) {
        GitCommandResult ret = new GitCommandResult();
        try {
            String remoteURL = this.getRemoteURLAndCheckIfItMatchesWhitelist(ctx);
            ProcessBuilder pb = this.getProcessBuilderWithRemoteURLAndSecurityCheck(ctx.authCtx, remoteURL, gitCommand);
            pb.directory(this.rootFolder);
            logTailBuilder.appendLine("$ " + StringUtils.join(pb.command(), (char)' '));
            DKUtils.execAndLogWithErrorInException((ProcessBuilder)pb, (DKUtils.SmartLogTailBuilder)logTailBuilder);
            ret.messages.withInfo((InfoMessage.MessageCode)infoCode, infoMsg);
            ret.commandSucceeded = true;
        }
        catch (Exception ex) {
            ret.messages.withFatalV((InfoMessage.MessageCode)errorCode, errorMsg + ": %s", new Object[]{ExceptionUtils.getMessageWithCauses((Throwable)ex)});
            ret.commandError = new SerializedError((Throwable)ex, !DKUApp.hideErrorStacks(), !DKUApp.hideErrorStacks(), !DKUApp.hideLogTails());
            ret.commandSucceeded = false;
        }
        return ret;
    }

    private GitCommandResult gitExecWithLogTailWithoutRemoteNameNorSecurityCheck(boolean iHaveCheckedThatIAmNotDoingRemoteAccess, List<String> gitCommand, DKUtils.SmartLogTailBuilder logTailBuilder, GitCodes infoCode, String infoMsg, GitCodes errorCode, String errorMsg) {
        GitCommandResult ret = new GitCommandResult();
        try {
            ProcessBuilder pb = this.getProcessBuilderWithoutRemoteURLNorSecurityCheck(iHaveCheckedThatIAmNotDoingRemoteAccess, gitCommand);
            pb.directory(this.rootFolder);
            logTailBuilder.appendLine("$ " + StringUtils.join(pb.command(), (char)' '));
            DKUtils.execAndLogThrows((ProcessBuilder)pb, (DKUtils.SmartLogTailBuilder)logTailBuilder);
            ret.messages.withInfo((InfoMessage.MessageCode)infoCode, infoMsg);
            ret.commandSucceeded = true;
        }
        catch (Exception ex) {
            ret.messages.withFatalV((InfoMessage.MessageCode)errorCode, errorMsg + ": " + ExceptionUtils.getMessageWithCauses((Throwable)ex), new Object[0]);
            ret.commandError = new SerializedError((Throwable)ex, !DKUApp.hideErrorStacks(), !DKUApp.hideErrorStacks(), !DKUApp.hideLogTails());
            ret.commandSucceeded = false;
        }
        return ret;
    }

    private String gitExecWithRemoteNameAndSecurityCheck(GitAuthCtxWithRemoteName ctx, List<String> gitCommand, GitCodes errorGitCode, String errorMessage, Object ... errorFormat) throws CodedException {
        try {
            String remoteURL = this.getRemoteURLAndCheckIfItMatchesWhitelist(ctx);
            ProcessBuilder pb = this.getProcessBuilderWithRemoteURLAndSecurityCheck(ctx.authCtx, remoteURL, gitCommand);
            pb.directory(this.rootFolder);
            logger.info((Object)("Executing " + UrlRedactionUtils.sanitizeHttpUrls((String)JSON.json(gitCommand)) + " in " + String.valueOf(this.rootFolder)));
            return new String(JGitWrapper.execAndGetOutputWithErrorInException((ProcessBuilder)pb, null), StandardCharsets.UTF_8);
        }
        catch (Exception ex) {
            throw new CodedException((InfoMessage.MessageCode)errorGitCode, UrlRedactionUtils.sanitizeHttpUrls((String)String.format(errorMessage, errorFormat)), (Throwable)ex);
        }
    }

    private String gitExecWithRemoteURLAndSecurityCheck(AuthCtx authCtx, String remoteURL, List<String> gitCommand, GitCodes errorGitCode, String errorMessage, Object ... errorFormat) throws CodedException {
        try {
            ProcessBuilder pb = this.getProcessBuilderWithRemoteURLAndSecurityCheck(authCtx, remoteURL, gitCommand);
            pb.directory(this.rootFolder);
            logger.info((Object)("Executing " + UrlRedactionUtils.sanitizeHttpUrls((String)JSON.json(gitCommand)) + " in " + String.valueOf(this.rootFolder)));
            return new String(JGitWrapper.execAndGetOutputWithErrorInException((ProcessBuilder)pb, null), StandardCharsets.UTF_8);
        }
        catch (Exception ex) {
            throw new CodedException((InfoMessage.MessageCode)errorGitCode, UrlRedactionUtils.sanitizeHttpUrls((String)String.format(errorMessage, errorFormat)), (Throwable)ex);
        }
    }

    private String gitExecWithoutRemoteNameNorSecurityCheck(boolean iHaveCheckedThatIAmNotDoingRemoteAccess, List<String> gitCommand, GitCodes errorGitCode, String errorMessage, Object ... errorFormat) throws CodedException {
        try {
            ProcessBuilder pb = this.getProcessBuilderWithoutRemoteURLNorSecurityCheck(iHaveCheckedThatIAmNotDoingRemoteAccess, gitCommand);
            pb.directory(this.rootFolder);
            logger.info((Object)("Executing " + UrlRedactionUtils.sanitizeHttpUrls((String)JSON.json(gitCommand)) + " in " + String.valueOf(this.rootFolder)));
            return new String(JGitWrapper.execAndGetOutputWithErrorInException((ProcessBuilder)pb, null), StandardCharsets.UTF_8);
        }
        catch (Exception ex) {
            throw new CodedException((InfoMessage.MessageCode)errorGitCode, UrlRedactionUtils.sanitizeHttpUrls((String)String.format(errorMessage, errorFormat)), (Throwable)ex);
        }
    }

    private ProcessBuilder getProcessBuilderWithoutRemoteURLNorSecurityCheck(boolean iHaveCheckedThatIAmNotDoingRemoteAccess, List<String> gitCommand) {
        assert (iHaveCheckedThatIAmNotDoingRemoteAccess);
        TransactionContext.warnAttachedTransaction();
        ArrayList<String> finalCommand = new ArrayList<String>();
        finalCommand.add("git");
        this.addCoreSafetiesToGitCommand(finalCommand);
        finalCommand.addAll(gitCommand);
        logger.info((Object)("Will execute Git command: " + UrlRedactionUtils.sanitizeHttpUrls((String)StringUtils.join(finalCommand, (String)" "))));
        ProcessBuilder pb = new ProcessBuilder(finalCommand);
        pb.environment().put("GIT_ASKPASS", "echo");
        return pb;
    }

    private ProcessBuilder getProcessBuilderWithRemoteURLAndSecurityCheck(AuthCtx authCtx, String remoteURL, List<String> gitCommand) throws UnauthorizedException {
        TransactionContext.warnAttachedTransaction();
        GeneralSettingsDAO.GitEnforcedConfigurationRule rule = null;
        if (!remoteURL.startsWith("/")) {
            rule = GitConfigurationRulesHelper.getMatchingRuleOrThrow(authCtx, remoteURL);
            if (!rule.allowGit) {
                throw new UnauthorizedException("You are not allowed to use remote Git commands", "git-denied");
            }
        } else {
            logger.info((Object)("Url '" + remoteURL + "' is local, not checking if access to the remote git is allowed for the user"));
        }
        ArrayList<String> finalCommand = new ArrayList<String>();
        finalCommand.add("git");
        this.addCoreSafetiesToGitCommand(finalCommand);
        if (rule != null) {
            for (SimpleKeyValue skv : rule.gitConfigurationOptions) {
                finalCommand.add("-c");
                finalCommand.add(skv.key + "=" + skv.value);
            }
            if (rule.dssControlsSSHCommand) {
                finalCommand.add("-c");
                finalCommand.add("core.sshCommand=ssh -o StrictHostKeyChecking=yes");
            }
        }
        finalCommand.addAll(gitCommand);
        logger.info((Object)("Will execute Git command: " + UrlRedactionUtils.sanitizeHttpUrls((String)StringUtils.join(finalCommand, (String)" "))));
        ProcessBuilder pb = new ProcessBuilder(finalCommand);
        pb.environment().put("GIT_ASKPASS", "echo");
        GitModel.GitAuthor author = new GitModel.GitAuthor(authCtx);
        if (StringUtils.isNotBlank((String)author.name)) {
            pb.environment().put("GIT_AUTHOR_NAME", author.name);
            pb.environment().put("GIT_COMMITTER_NAME", author.name);
        }
        if (StringUtils.isNotBlank((String)author.email)) {
            pb.environment().put("GIT_AUTHOR_EMAIL", author.email);
            pb.environment().put("GIT_COMMITTER_EMAIL", author.email);
        }
        if (rule != null && StringUtils.isNotBlank((String)rule.alternateHomeDir)) {
            pb.environment().put("HOME", rule.alternateHomeDir);
        }
        return pb;
    }

    private void addCoreSafetiesToGitCommand(List<String> gitCommand) {
        if (DKUApp.getParams().getBoolParam("dku.core.fs.disableSymlinksInGit", true)) {
            gitCommand.add("-c");
            gitCommand.add("core.symlinks=false");
        }
        if (DKUApp.getParams().getBoolParam("dku.core.fs.disableGitHooks", true)) {
            gitCommand.add("-c");
            gitCommand.add("core.hooksPath=/dev/null");
        }
    }

    private void gitExecCloneInFutureThread(AuthCtx authCtx, String remoteURL, final List<String> gitCommand, File dir) throws IOException, InterruptedException, UnauthorizedException {
        ProcessBuilder pb = this.getProcessBuilderWithRemoteURLAndSecurityCheck(authCtx, remoteURL, gitCommand);
        pb.directory(dir);
        DKUtils.execAndLogWithErrorInException((ProcessBuilder)pb, (DKUtils.SmartLogTailBuilder)new DKUtils.SmartLogTailBuilder(){
            final Matcher percentMatcher = Pattern.compile("([^:]+): {1,3}(\\d{1,3})%").matcher("");
            String lastMessage = null;

            public synchronized void appendLine(String line) {
                super.appendLine(line);
                try {
                    this.percentMatcher.reset(line);
                    if (this.percentMatcher.find()) {
                        if (this.lastMessage == null || !this.lastMessage.equals(this.percentMatcher.group(1))) {
                            FutureProgress.popState();
                            this.lastMessage = this.percentMatcher.group(1);
                            FutureProgress.pushState((String)this.lastMessage, (double)100.0, (FutureProgressState.StateUnit)FutureProgressState.StateUnit.NONE);
                        }
                        FutureProgress.updateState((double)Integer.parseInt(this.percentMatcher.group(2)));
                    }
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    throw new RuntimeException("Git command (" + StringUtils.join((Collection)gitCommand, (String)" ") + ") was aborted", e);
                }
            }
        });
    }

    public String getRemoteURLAndCheckIfItMatchesWhitelist(GitAuthCtxWithRemoteName ctx) throws UnauthorizedException, IOException, InterruptedException {
        assert (ctx.remoteName != null);
        String remoteUrl = this.getRemoteUrl(ctx.authCtx, ctx.remoteName);
        if (StringUtils.isBlank((String)remoteUrl)) {
            throw new UnauthorizedException("Remote URL is null, cannot check security", "no-remote-url");
        }
        if (!remoteUrl.startsWith("/")) {
            GitConfigurationRulesHelper.checkIfGitRemoteURLAllowed(ctx.authCtx, remoteUrl);
        }
        return remoteUrl;
    }

    private String getRemoteUrl(AuthCtx authCtx, String remoteName) throws IOException, InterruptedException {
        Preconditions.checkNotNull((Object)authCtx);
        Preconditions.checkNotNull((Object)remoteName);
        ArrayList gitCommand = Lists.newArrayList((Object[])new String[]{"config", "--get", "remote." + remoteName + ".url"});
        ProcessBuilder pb = this.getProcessBuilderWithoutRemoteURLNorSecurityCheck(true, gitCommand);
        pb.directory(this.rootFolder);
        logger.info((Object)("Executing " + UrlRedactionUtils.sanitizeHttpUrls((String)JSON.json((Object)gitCommand)) + " in " + String.valueOf(this.rootFolder)));
        String output = new String(JGitWrapper.execAndGetOutputWithErrorInException((ProcessBuilder)pb, null), StandardCharsets.UTF_8);
        return StringUtils.deleteWhitespace((String)output.split("\n")[0]);
    }

    private static boolean isRemoteGitPermissionError(SerializedError commandError) {
        return commandError != null && GitCodes.ERR_GIT_URL_DOES_NOT_MATCH_WHITELIST.getCode().equals(commandError.code);
    }

    public static class GitAuthCtxWithRemoteName {
        private final AuthCtx authCtx;
        private final String remoteName;

        public GitAuthCtxWithRemoteName(AuthCtx authCtx, String remoteName) {
            this.authCtx = authCtx;
            this.remoteName = StringUtils.defaultIfBlank((String)remoteName, (String)GitRemoteCommands.DEFAULT_ORIGIN);
        }
    }

    public static class GitCommandResult {
        public InfoMessage.InfoMessages messages = new InfoMessage.InfoMessages();
        public boolean commandSucceeded;
        public SerializedError commandError;
    }
}

