import json
import logging
import os
import uuid
from typing import Any, Optional, List, Dict, Union

from google.cloud import aiplatform, storage
from google.cloud.aiplatform.matching_engine.matching_engine_index_endpoint import (
    Namespace,
    NumericNamespace,
)
from google.oauth2.service_account import Credentials
from langchain_core.documents import Document
from langchain_google_vertexai.vectorstores import vectorstores

from dataiku.core.vector_stores.dku_vector_store import DkuRemoteVectorStore, logger
from dataiku.core.vector_stores.vector_store_document_filter import VectorStoreDocumentFilter, ColumnType
from dataiku.langchain.metadata_generator import DKU_SECURITY_TOKENS_META
from dataiku.llm.types import RetrievableKnowledge
from dataikuapi.dss.admin import DSSConnection
from dataikuapi.dss.langchain import DKUEmbeddings


class VertexAiVectorStore(DkuRemoteVectorStore):

    def __init__(self, kb: RetrievableKnowledge, exec_folder: str):
        self.gcs_credentials: Optional[Credentials] = None
        self.gcs_project_id: Optional[str] = None
        self.gcs_bucket_name: Optional[str] = None
        self.gcs_path_prefix: Optional[str] = None  # prefix for index storage path_from_bucket
        self.vertex_region: Optional[str] = None
        self.vertex_shard_size: Optional[str] = None
        self.storage_client: Optional[storage.Client] = None

        super().__init__(kb, exec_folder)
        self.vector_size = self.get_vector_size()  # required to create the index dynamically
        self.document_filter = VertexAIVectorStoreDocumentFilter(self.metadata_column_type)


    def set_vertex_region_from_bucket(self) -> None:
        # Vertex index must be on the same location than the GCP bucket (=>Retrieve the bucket region to store the index in the same region).
        assert self.storage_client is not None, "Vertex client not initialized"
        bucket = self.storage_client.get_bucket(self.gcs_bucket_name)
        self.vertex_region = bucket.location.lower()

    def set_vertex_shard_size(self, custom_properties: List) -> None:
        """ Choose Vertex shard_size : 'SHARD_SIZE_SMALL' by default if not overriden by a custom property
            :param custom_properties: custom properties of the associated GCS connection.
        """
        self.vertex_shard_size = "SHARD_SIZE_SMALL"
        supported_sizes = ["SHARD_SIZE_SMALL", "SHARD_SIZE_MEDIUM", "SHARD_SIZE_LARGE"]

        for custom_property in custom_properties:
            if custom_property["name"] == "indexShardSize":
                if custom_property["value"] not in supported_sizes:
                    raise Exception("Unsupported custom shard size for Vertex index, supported values are: {}, please edit your connection properties accordingly"
                                    .format(supported_sizes))
                self.vertex_shard_size = custom_property["value"]
                logger.info("Setting custom index shard size from connection property: {}".format(self.vertex_shard_size))
                break

    def build_index_storage_path_from_bucket(self, index_name: str) -> str:
        if self.gcs_path_prefix:
            return "{}/vertexIndexStorage/{}".format(self.gcs_path_prefix, index_name)
        else:
            return "vertexIndexStorage/{}".format(index_name)

    def init_connection(self, connection: DSSConnection) -> None:
        connection_infos = connection.get_info()
        connection_params = connection_infos['params']
        connection_resolved_params = connection_infos.get_resolved_params()  # todo test if accessible from non-admin with read details

        self.gcs_project_id = connection_params.get("projectId")
        self.gcs_bucket_name = connection_params.get("chbucket") or connection_params.get("defaultManagedBucket")
        self.gcs_path_prefix = connection_params.get("chroot", connection_params.get("defaultManagedPath"))

        if not self.gcs_project_id:
            raise ValueError("A Google cloud project id must be selected in the GCS connection")  # todo have a warning in the kb form as well

        if not self.gcs_bucket_name:
            raise ValueError("Either a bucket restriction or a default bucket must be selected in the GCS connection")

        if connection_resolved_params["authType"] == "KEYPAIR":
            logger.info("Initiating the connection to Vertex AI vector store with Api Key")
            json_creds = json.loads(connection_resolved_params["appSecretContent"])
            self.gcs_credentials = Credentials.from_service_account_info(json_creds)
        else:
            raise Exception("Unsupported authorization type {} for Google Cloud Storage connection with Knowledge banks".format(connection_params["authType"]))

        self.storage_client = storage.Client(self.gcs_project_id, self.gcs_credentials)
        self.set_vertex_region_from_bucket()
        self.set_vertex_shard_size(connection_params.get("dkuProperties", []))
        aiplatform.init(project=self.gcs_project_id, location=self.vertex_region, staging_bucket=self.gcs_bucket_name, credentials=self.gcs_credentials)

    def clear_index(self) -> None:
        refs = self._load_remote_resources_references()

        if not refs.get("MatchingEngineIndexEndpointName") and not refs.get("MatchingEngineIndexName"):
            logger.info("No existing remote resources to clear - skipping")
            return

        try:
            if refs.get("MatchingEngineIndexEndpointName"):
                my_index_endpoint = aiplatform.MatchingEngineIndexEndpoint(refs.get("MatchingEngineIndexEndpointName"))
                my_index_endpoint.delete(force=True, sync=True)  # all deployed indexes on this endpoint will be undeployed first then the endpoint is deleted.

            if refs.get("MatchingEngineIndexName"):
                # there are existing remote resources to delete:
                my_index = aiplatform.MatchingEngineIndex(refs.get("MatchingEngineIndexName"))
                my_index.delete(sync=True)

                path_from_bucket = self.build_index_storage_path_from_bucket(refs.get("MatchingEngineIndexName", ""))
                logger.info("Deleting associated index storage ('{}')".format(path_from_bucket))
                if self.storage_client is None:
                    raise ValueError("Storage client not initialized, cannot delete index storage")
                bucket = self.storage_client.bucket(self.gcs_bucket_name)
                bucket.delete_blobs(blobs=list(bucket.list_blobs(prefix=path_from_bucket)))

        except Exception as e:
            # don't fail the whole job for a cleaning method:
            logger.warning("Some managed remote resources could not be deleted, maybe they do not exist anymore? Skipping the clearing task. ({})".format(
                str(e))
            )
        finally:
            # We do not wish to retry the cleaning (if it failed it's most probably due to resources to have been manually removed, retrying won't help).
            refs.update({
                "MatchingEngineIndexName": None,
                "MatchingEngineIndexEndpointName": None
            })
            self._dump_remote_resources_references(refs)

    def _ensure_initialized(self, allow_creation: bool) -> Dict:
        remote_resources = self._load_remote_resources_references()
        if remote_resources.get("MatchingEngineIndexName") and remote_resources.get("MatchingEngineIndexEndpointName"):
            try:  # try to load resources to see if they still exist on Vertex side as well (if not we need to recreate those)
                aiplatform.MatchingEngineIndexEndpoint(remote_resources["MatchingEngineIndexEndpointName"])
                aiplatform.MatchingEngineIndex(remote_resources["MatchingEngineIndexName"])
                logger.info("No need to create remote resources, found existing MatchingEngineIndex.name='{}' deployed on MatchingEngineIndexEndpoint.name='{}'"
                            .format(remote_resources["MatchingEngineIndexName"], remote_resources["MatchingEngineIndexEndpointName"]))
                return remote_resources
            except Exception as e:
                logger.warning("Remote resources don't exist on Vertex side anymore.")

        if not allow_creation:
            raise Exception("Vertex vector store resources don't exist or are not ready yet, you may need to run the embedding recipe.")

        logger.info("Creating new Vertex AI MatchingEngineIndex / MatchingEngineIndexEndpoint resources")
        vertex_index = aiplatform.MatchingEngineIndex.create_brute_force_index(
            display_name=self.index_name,  # /!\ vertex vertex_index.name is unique, display_name isn't.
            dimensions=self.vector_size,
            distance_measure_type="DOT_PRODUCT_DISTANCE",
            index_update_method="STREAM_UPDATE",
            shard_size=self.vertex_shard_size
        )
        vertex_index_endpoint = aiplatform.MatchingEngineIndexEndpoint.create(display_name=f"{self.index_name}-endpoint", public_endpoint_enabled=True)
        # save newly created resources to ensure future accesses (display_names aren't unique, unique identifiers (names) are only accessible at creation)
        remote_resources.update({
            "MatchingEngineIndexName": vertex_index.name,
            "MatchingEngineIndexEndpointName": vertex_index_endpoint.name
        })
        self._dump_remote_resources_references(remote_resources)

        # The deployed_index_id must be unique within the GCP project it is created in & only underscores letters & numbers are accepted:
        deployed_index_id = "deployed_index_{}".format(str(uuid.uuid4()).replace("-", "_"))
        vertex_index_endpoint.deploy_index(index=vertex_index, deployed_index_id=deployed_index_id)
        return remote_resources

    def get_db(self, embeddings: DKUEmbeddings, allow_creation: bool = False, **kwargs: Any) -> vectorstores.VectorSearchVectorStore:
        remote_resources = self._ensure_initialized(allow_creation)  # for vertex ai, langchain doesn't create remote resource if not already existing.

        # VectorSearchVectorStore originally expects a path to a google credential file. For security reason and to ease the containerized case we don't want to
        # dump the google credentials to a file and handle its access.
        # the following is a monkey patch of VectorSearchSDKManager to use its 'credentials' argument (not used in VectorSearchVectorStore by default)
        # this allows to use inlined json creds rather than a filepath.

        class VectorSearchSDKManagerWrapper(vectorstores.VectorSearchSDKManager):
            def __init__(_self, *, project_id: str, region: str, credentials: Optional[Credentials]=None, credentials_path: Optional[str]=None):
                if credentials is not None or credentials_path is not None:
                    super().__init__(project_id=project_id, region=region, credentials=credentials, credentials_path=credentials_path)
                else:
                    # add a way to give google Credentials since credentials is unused for now by VectorSearchVectorStore
                    super().__init__(project_id=project_id, region=region, credentials=self.gcs_credentials, credentials_path=None)

        vectorstores.VectorSearchSDKManager = VectorSearchSDKManagerWrapper

        db = vectorstores.VectorSearchVectorStore.from_components(
            project_id=self.gcs_project_id,
            region=self.vertex_region,
            gcs_bucket_name=self.gcs_bucket_name,
            index_id=remote_resources["MatchingEngineIndexName"],
            endpoint_id=remote_resources["MatchingEngineIndexEndpointName"],
            embedding=embeddings,
            stream_update=True,
            credentials_path=None
        )
        # monkey patch on langchain API to ensure each index write into a different subfolder (that can be cleared on demand):
        path_from_bucket = self.build_index_storage_path_from_bucket(remote_resources["MatchingEngineIndexName"])
        db._document_storage._prefix = path_from_bucket  # todo add test to check it does write on the expected folder to make sure this monkey patch resist to libs bumps
        logger.info("Configured the index to point on GCS storage: {}".format(path_from_bucket))
        return db


class VertexAIVectorStoreDocumentFilter(VectorStoreDocumentFilter):
    """ Vertex AI support for filtering

    See https://cloud.google.com/vertex-ai/docs/vector-search/filtering
    and https://python.langchain.com/docs/integrations/vectorstores/google_vertex_ai_vector_search/
    """

    def _coerce_value(self, clause: Dict) -> Union[float, List[float], str, List[str]]:
        column_type = self._get_column_type(clause)
        if column_type in [ColumnType.Decimal, ColumnType.Integer]:
            return float(clause["value"]) if type(clause["value"]) != list else [float(v) for v in clause["value"]]
        else:
            return clause["value"]

    SIMPLE_OPERATOR_MAPPING = {
        "EQUALS": "EQUAL",
        "NOT_EQUALS": "NOT_EQUAL",
        "LESS_THAN": "LESS",
        "LESS_OR_EQUAL": "LESS_EQUAL",
        "GREATER_THAN": "GREATER",
        "GREATER_OR_EQUAL": "GREATER_EQUAL",
    }

    def convert(self, clause: Dict) -> List[Union[Namespace, NumericNamespace]]:
        if clause["operator"] in self.SIMPLE_OPERATOR_MAPPING:
            operator = self.SIMPLE_OPERATOR_MAPPING[clause["operator"]]
            column_type = self._get_column_type(clause)
            if column_type == ColumnType.String:
                if operator == "EQUAL":
                    return [Namespace(name=clause["column"], allow_tokens=[self._coerce_value(clause)])]
                elif operator == "NOT_EQUAL":
                    return [Namespace(name=clause["column"], deny_tokens=[self._coerce_value(clause)])]
                else:
                    raise Exception(f"The {operator} operator can't be used with strings on VertexAI")
            elif column_type in [ColumnType.Decimal, ColumnType.Integer]:
                return [NumericNamespace(name=clause["column"], value_float=self._coerce_value(clause), op=operator)]
            else:
                raise self._unsupported_type_for_operator(column_type, clause["operator"])
        elif clause["operator"] == "IN_ANY_OF":
            column_type = self._get_column_type(clause)
            if column_type == ColumnType.String:
                return [Namespace(name=clause["column"], allow_tokens=self._coerce_value(clause))]
            elif column_type in [ColumnType.Decimal, ColumnType.Integer]:
                numbers = self._coerce_value(clause)
                if len(numbers) == 1:
                    return [NumericNamespace(name=clause["column"], value_float=numbers[0], op="EQUAL")]
                else:
                    raise Exception("The IN_ANY_OF operator can't be used with numbers on VertexAI")
            else:
                raise self._unsupported_type_for_operator(column_type, clause["operator"])
        elif clause["operator"] == "IN_NONE_OF":
            column_type = self._get_column_type(clause)
            if column_type == ColumnType.String:
                return [Namespace(name=clause["column"], deny_tokens=self._coerce_value(clause))]
            elif column_type in [ColumnType.Decimal, ColumnType.Integer]:
                return [NumericNamespace(name=clause["column"], value_float=value, op="NOT_EQUAL") for value in self._coerce_value(clause)]
            else:
                raise self._unsupported_type_for_operator(column_type, clause["operator"])
        elif clause["operator"] == "AND":
            filters = []
            for subclause in clause["clauses"]:
                filters.extend(self.convert(subclause))
            return filters
        else:
            # OR and CONTAINS are not supported
            raise Exception("Unsupported filter operator for VertexAI: %s" % clause["operator"])

    def add_filter(self, search_kwargs: Dict, simple_filter: Dict) -> None:
        """ Add string filters and numeric filters
        """
        if "filter" not in search_kwargs:
            search_kwargs["filter"] = []
        if "numeric_filter" not in search_kwargs:
            search_kwargs["numeric_filter"] = []

        filters = self.convert(simple_filter)
        for filter_ in filters:
            if isinstance(filter_, Namespace):
                search_kwargs["filter"].append(filter_)
            elif isinstance(filter_, NumericNamespace):
                search_kwargs["numeric_filter"].append(filter_)
            else:
                raise Exception("Unexpected filter type for VertexAI", type(filter_))

    def add_security_tokens_to_document(self, document: Document) -> Document:
        """ Add a field with the list of security tokens, not used by add_security_filter() but it still is useful info
        """
        if DKU_SECURITY_TOKENS_META in document.metadata:
            document.metadata[DKU_SECURITY_TOKENS_META] = json.loads(document.metadata[DKU_SECURITY_TOKENS_META])
            logging.info("Updated metadata to %s" % self._sanitize_metadata_for_print(document.metadata))
        return document

    def add_security_filter(self, search_kwargs: Dict, caller_security_tokens: List) -> None:
        raise Exception("Document-level security is not supported by VertexAI")
