
(function () {
    "use strict";

    function RequestCenterService($q, DataikuAPI, WT1, Logger, StateUtils) {
        function search(facets, offset, limit) {
            return DataikuAPI.requests.listRequests(facets, offset, limit).then((result) => {
                return result.data;
            });
        }

        function getAllRequests(){
            return search({}, 0, 0);
        }

        function loadRequest(requestId) {
            return $q((resolve, reject) => {
                let request;
                let requester;
                let selectedPermission;
                let selectedUserOrGroup;
                let requestingUserGroups;
                let closer;
                DataikuAPI.requests.getRequest(requestId).then((response) => {
                    request = response.data;
                    request.requestDetails = request.requestDetails || {};
                    selectedPermission = request.requestType;

                    const promises = [];
                    const needTrialStatus = request.requestType === 'INSTANCE_ACCESS';
                    promises.push(DataikuAPI.profile.get(request.requesterLogin, needTrialStatus).catch((e) => { return { data: { login: request.requesterLogin }}}));
                    if (request.closingUser) {
                        promises.push(DataikuAPI.profile.get(request.closingUser).catch((e) => { return { data: { login: request.closingUser }}}));
                    }
                    $q.all(promises).then((responses) => {
                        const requesterResponse = responses[0];
                        const closerResponse = responses[1];
                        requester = requesterResponse.data;
                        if (requester.enabled) {
                            selectedUserOrGroup = buildUserOrGroupObject(requesterResponse.data.login, "USER");
                        }
                        if (requester.groups) {
                            requestingUserGroups = [
                                ...requester.groups.map((name) => buildUserOrGroupObject(name, "GROUP")),
                            ];
                            if (selectedUserOrGroup) {
                                requestingUserGroups.push(selectedUserOrGroup);
                            }
                        }
                        if (closerResponse) {
                            closer = closerResponse.data;
                        }
                        resolve({
                            request,
                            selectedPermission,
                            requester,
                            requestingUserGroups,
                            selectedUserOrGroup,
                            closer,
                        });
                    }, reject);
                }, reject);
            });
        }

        function listUsersDisplayNames() {
            return DataikuAPI.security.listUsers().then((response) => {
                return response.data.reduce((users, user) => {
                    return { ...users, ...{ [user.login]: user.displayName } };
                }, {});
            });
        }

        function listProjectsNames() {
            return DataikuAPI.projects.list().then((response) => {
                return response.data.reduce((projects, project) => {
                    return { ...projects, ...{ [project.projectKey]: project.name } };
                }, {});
            });
        }

        function listApplicationsNames() {
            return DataikuAPI.apps.listTemplates().then((response) => {
                return response.data.items.reduce((apps, app) => {
                    return { ...apps, ...{ [app.appId]: app.label } };
                }, {});
            });
        }

        function listPlugins() {
            return DataikuAPI.plugins.list().then((response) => {
                return response.data.plugins;
            });
        }

        function buildUserOrGroupObject(name, type) {
            return { name: name, type: type };
        }

        function getPluginObjectLink(objectId, pluginsList ) {
            if (pluginsList) {
                for (let i = 0; i < pluginsList.length; i++) {
                    let plugin = pluginsList[i];
                    if (plugin.id === objectId && plugin.installed) {
                        let updateAvailable = plugin.storeDesc && plugin.storeDesc.storeVersion > plugin.installedDesc.desc.version;
                        return updateAvailable ? StateUtils.href.pluginStore(objectId) : StateUtils.href.pluginSummary(objectId);
                    }
                }
            }
            return StateUtils.href.pluginStore(objectId);
        }

        function getCodeEnvObjectLink(envName, envLang) {
            return StateUtils.href.codeEnvEdit(envName, envLang);
        }

        function getCodeEnvLangLabel(envName, envLang){
            let prefix = "";
            if (envName && envName.startsWith("INTERNAL_")) {
                prefix = "internal ";
            }
            let lowerCaseLang = envLang.toLowerCase();
            return prefix + lowerCaseLang[0].toUpperCase() + lowerCaseLang.slice(1);
        }

        function getObjectLink(projectKey, objectType, objectId, pluginsList, envLang) {
            switch(objectType) {
                case "DATASET":
                    return "projects.project.datasets.dataset.explore({projectKey: '" + projectKey + "', datasetName: '" + objectId + "'})";
                case "SAVED_MODEL":
                    return "projects.project.savedmodels.savedmodel.versions({projectKey: '" + projectKey + "', smId: '" + objectId + "'})";
                case "MODEL_EVALUATION_STORE":
                    return "projects.project.modelevaluationstores.modelevaluationstore.evaluations({projectKey: '" + projectKey + "', mesId: '" + objectId + "'})";
                case "MANAGED_FOLDER":
                    return "projects.project.managedfolders.managedfolder.view({projectKey: '" + projectKey + "', odbId: '" + objectId + "'})";
                case "PLUGIN":
                    return getPluginObjectLink(objectId, pluginsList);
                case "CODE_ENV":
                    return getCodeEnvObjectLink(objectId, envLang);
            }
        }

        function numberOfPendingRequests(data) {
            if (data && data.aggregations && data.aggregations.STATUS) {
                const pendingFacet =  data.aggregations.STATUS.find((agg) => {
                    return agg.key === "PENDING";
                });
                return pendingFacet ? pendingFacet.value : 0;
            }
            return 0;
        }

        function hasRequests(data) {
            if (data && data.aggregations && data.aggregations.STATUS) {
                return true;
            }
            return false;
        }

        /**
         * 
         * @param {*} requestEnabled - the local setting
         * @param {*} accessRequestsMode - the instance setting
         */
        function onProjectAccessRequestsSettingChanged(requestEnabled, accessRequestsMode){
            if(requestEnabled === "ENABLED"){
                WT1.event('requests-center-request-type-enabled', { requestType: 'project-access' });
            } else if (requestEnabled === "DISABLED") {
                WT1.event('requests-center-request-type-disabled', { requestType: 'project-access' });
            } else if (requestEnabled === "INHERIT") {
                WT1.event("requests-center-request-type-inherit", {
                    requestType: "project-access",
                    inheritedValue: accessRequestsMode === "ENABLED_BY_DEFAULT" ? "enabled" : "disabled",
                });
            } else {
                Logger.error(`Unknown project access request setting: ${requestEnabled} expected 'ENABLED', 'DISABLED', 'INHERIT'`);
            }
        }

        /**
         * 
         * @param {*} applicationId - can't be infered by WT1 service
         * @param {*} requestEnabled - the local setting
         * @param {*} accessRequestsMode - the instance setting
         */
        function onApplicationAccessRequestsSettingChanged(applicationId, requestEnabled, accessRequestsMode){
            const hashedApplicationId = md5(applicationId);
            if(requestEnabled === "ENABLED"){
                WT1.event('requests-center-request-type-enabled', { requestType: 'application-access', applicationh: hashedApplicationId });
            } else if (requestEnabled === "DISABLED"){
                WT1.event('requests-center-request-type-disabled', { requestType: 'application-access', applicationh: hashedApplicationId });
            } else if (requestEnabled === "INHERIT") {
                WT1.event("requests-center-request-type-inherit", {
                    requestType: "application-access",
                    applicationh: hashedApplicationId,
                    inheritedValue: accessRequestsMode === "ENABLED_BY_DEFAULT" ? "enabled" : "disabled",
                });
            } else {
                Logger.error(`Uknown application execute request setting: ${requestEnabled} expected 'ENABLED', 'DISABLED', 'INHERIT'`);
            }
        }

        /**
         * 
         * @param {*} requestEnabled - the local setting
         * @param {*} accessRequestsMode - the instance setting
         */
        function onObjectAccessRequestsSettingChanged(requestEnabled, accessRequestsMode) {
            if (requestEnabled === "ENABLED") {
              WT1.event("requests-center-request-type-enabled", { requestType: "object-share" });
            } else if (requestEnabled === "DISABLED") {
              WT1.event("requests-center-request-type-disabled", { requestType: "object-share" });
            } else if (requestEnabled === "INHERIT") {
              WT1.event("requests-center-request-type-inherit", {
                requestType: "object-share",
                inheritedValue: accessRequestsMode === "ENABLED_BY_DEFAULT" ? "enabled" : "disabled",
              });
            } else {
                Logger.error(`Unknown object sharing request setting: ${requestEnabled} expected 'ENABLED', 'DISABLED', 'INHERIT'`);
            }
        }

        // in the future we may want to switch to passing an object for all the conditional values
        function onRequestSent(objectType, projectKey, objectId, requestMessage, requestId, targetProjectKey, wt1Context) {
            WT1.tryEvent('requests-center-request-create', () => {
                let payload = {};
                switch (objectType) {
                    case 'PROJECT':
                        payload = {
                            requestType: 'project-access',
                            objectType: 'Project',
                            targetProjecth: md5(projectKey),
                        }
                        break;

                    case 'APP':
                        payload = {
                            requestType: 'app-instantiate',
                        }
                        break;

                    case 'PLUGIN':
                        payload = {
                            requestType: 'plugin-install',
                        }
                        break;

                    case 'CODE_ENV':
                        payload = {
                            requestType: 'code-env-install',
                        }
                        break;

                    case 'INSTANCE':
                        payload = {
                            requestType: 'instance-access-request'
                        }
                        break;

                    case 'PROFILE':
                        payload = {
                            requestType: 'profile-upgrade-request'
                        }
                        break;

                    default:
                        payload = {
                            requestType: 'object-share',
                            originProjecth: md5(projectKey),
                            targetProjecth: md5(targetProjectKey),
                            objectType: objectType.toLowerCase(),
                            objecth: md5(`${projectKey}.${objectId}`),
                        }
                }

                return {
                    ...payload,
                    messageRequest: requestMessage && requestMessage.length ? true : false,
                    requesth: md5(requestId),
                    ...(wt1Context || {}),
                };
            });
        }

        function onRequestManaged(request, requestResponse, requestAction){
            let payload = {};
            switch (request.objectType) {
                case 'PROJECT':
                    payload = {
                      requestType: "project-access",
                      objectType: "Project",
                      permissionLevel: requestAction === "approved" ? requestResponse.selectedPermission : null,
                      userGroup:
                        requestAction === "approved" ? requestResponse.selectedUserOrGroup.type === "GROUP" : undefined,
                      userIndividual:
                        requestAction === "approved" ? requestResponse.selectedUserOrGroup.type === "USER" : undefined,
                        targetProjecth: md5(request.objectProjectKey),
                    };
                    break;

                case 'APP':
                    payload = {
                      requestType: "app-instantiate",
                      objectType: "App",
                      userGroup:
                        requestAction === "approved" ? requestResponse.selectedUserOrGroup.type === "GROUP" : undefined,
                      userIndividual:
                        requestAction === "approved" ? requestResponse.selectedUserOrGroup.type === "USER" : undefined,
                      applicationh: md5(request.objectId),
                    };
                    break;
                case 'PLUGIN':
                    payload = {
                      requestType: "plugin-processed",
                    };
                    break;

                case 'CODE_ENV':
                    payload = {
                      requestType: "code-env-processed",
                    };
                    break;
                case 'INSTANCE':
                    payload = {
                        requestType: "instance-access",
                    };
                    if (requestAction === "approved") {
                        payload["selectedUserProfile"] = requestResponse.selectedUserProfile;
                        payload["selectedProfileType"] = requestResponse.selectedProfileType;
                    }
                    break;
                case 'PROFILE':
                    payload = {
                        requestType: "profile-upgrade",
                    };
                    if (requestAction === "approved") {
                        payload["selectedUserProfile"] = requestResponse.selectedUserProfile;
                    }
                    break;    
                default:
                    payload = {
                        requestType: 'managed-sharing',
                        objectType: request.objectType.toLowerCase(),
                        originProjecth: md5(request.objectProjectKey),
                        objecth: md5(`${request.objectProjectKey}.${request.objectId}`),
                        targetProjecth: md5(request.requestDetails.sharingTargetProject)
                    }
            }
            payload = {...payload, ...{
                requestAction,
                messageRequest: request.requestMessage && request.requestMessage.length ? true : false,
                requesth: md5(request.requestId),
            }}
            WT1.event('requests-center-request-manage', payload);
        }

        const WT1Events = {
            onProjectAccessRequestsSettingChanged,
            onApplicationAccessRequestsSettingChanged,
            onObjectAccessRequestsSettingChanged,
            onRequestSent,
            onRequestManaged
        }


        const checkPendingProfileRequestStatus = (scope) => {
            DataikuAPI.requests.getLatestRequestForCurrentUser("", "PROFILE", "")
            .then(response => {
                scope.hasPendingProfileRequest = response.data.status === "PENDING";
            }, error => {
                scope.hasPendingProfileRequest = false;
                if (error.status !== 404) {
                    setErrorInScope.bind(scope)(error);
                }
            });
        }

        return {
            search,
            loadRequest,
            getObjectLink,
            listUsersDisplayNames,
            listProjectsNames,
            listApplicationsNames,
            buildUserOrGroupObject,
            numberOfPendingRequests,
            WT1Events,
            hasRequests,
            getAllRequests,
            listPlugins,
            getCodeEnvLangLabel,
            checkPendingProfileRequestStatus
        };
    }

    angular.module("dataiku.inbox").service("RequestCenterService", RequestCenterService);
})();
