(function() {
'use strict';

const app = angular.module('dataiku.agenttools', ['dataiku.services', 'dataiku.filters']);

app.controller('AgentToolController', function ($scope, $state, $stateParams, $q, $timeout, WT1, DataikuAPI, PromptUtils, TopNav, DatasetUtils, FutureProgressModal,
    $rootScope, FutureWatcher, Debounce, CreateModalFromTemplate, TaggingService, Dialogs, localStorageService, SmartId, translate, ClipboardUtils, AgentToolService,
    SavedModelsService, PluginsService, AgentToolCodeTemplates, ActiveProjectKey, Expressions, CodeBasedEditorUtils, DKUtils) {
    $scope.uiState = {};

    $scope.editorOptions = CodeBasedEditorUtils.editorOptions('text/x-python', $scope, true);
    CodeBasedEditorUtils.registerBroadcastSelectionHandler($scope.editorOptions);
    CodeBasedEditorUtils.setRecipeScript($scope.script);

    let savedAgentTool;
    $scope.set = function(agentTool) {
        $scope.agentToolTypeUiData = AgentToolService.getAgentToolTypesUiData(agentTool);
        $scope.agentTool = agentTool;
        savedAgentTool = angular.copy(agentTool);
        $timeout(function() {
            $scope.$broadcast('agentToolSetEvent');
        }, 0);
    };

    $scope.isDirty = function() {
        return !angular.equals(savedAgentTool, $scope.agentTool);
    };

    checkChangesBeforeLeaving($scope, $scope.isDirty);

    $scope.save = function() {
        if (!$scope.isDirty()) {
            return $q.when('agent tool not dirty');
        }

        const deferred = $q.defer();
        const saveAfterConflictCheck = function() {
            DataikuAPI.agentTools.save($scope.agentTool).success(function(data) {
                WT1.event(
                    'agent-tool-save', {
                        agentToolType: data.type,
                    });
                $scope.agentTool = data;
                savedAgentTool = angular.copy($scope.agentTool);
                $scope.$broadcast('agentToolSavedEvent');
                deferred.resolve('agent tool saved');
            }, () => deferred.reject())
        };

        DataikuAPI.agentTools.checkSaveConflict($scope.agentTool).success(function(conflictResult) {
            if (!conflictResult.canBeSaved) {
                Dialogs.openConflictDialog($scope, conflictResult).then(
                    function(resolutionMethod) {
                        if (resolutionMethod === 'erase') {
                            saveAfterConflictCheck();
                        } else if (resolutionMethod === 'ignore') {
                            deferred.reject();
                            DKUtils.reloadState();
                        }
                    }
                );
            } else {
                saveAfterConflictCheck();
            }
        }).error(setErrorInScope.bind($scope));

        return deferred.promise;
    };

    $scope.saveIgnoringQuickTestQuery = function() {
        if (!$scope.isDirty()) {
            return $q.when('agent tool not dirty');
        }

        let frankenTool = angular.copy($scope.agentTool);
        frankenTool["quickTestQuery"] = savedAgentTool["quickTestQuery"];
        if (angular.equals(frankenTool, savedAgentTool)) {
            return $q.when('only difference is the quicktest query, not saving');
        }

        return $scope.save();
    };

    $scope.datasetColumns = [];
    $scope.updateInputDatasetColumns = function() {
        if (!$scope.agentTool || !$scope.agentTool.params || !$scope.agentTool.params.datasetRef) return;
        const dataset = SmartId.resolve($scope.agentTool.params.datasetRef, $stateParams.projectKey);
        DataikuAPI.datasets.get(dataset.projectKey, dataset.id, $stateParams.projectKey).then(response => {
            $scope.datasetColumns = response.data.schema.columns.map(column => column.name);
            $scope.datasetSchema = response.data.schema;
        });
    };

    $scope.showToolDescriptionModal = function(agentTool){
        DataikuAPI.agentTools.getDescriptor(agentTool.projectKey, agentTool.id).success(descriptor => {
            $scope.toolDescriptionModal = {
                // remove the global description that was appended by the back-end in get-descriptor to show it in another field
                internalDescription: descriptor.description.replace("\n\n" + agentTool.additionalDescriptionForLLM, ""),
                globalDescription: agentTool.additionalDescriptionForLLM,
                agentSpecificDescription: undefined,
            }
            CreateModalFromTemplate('/templates/savedmodels/agents/agent-tool-description-modal.html', $scope);
        }).error(setErrorInScope.bind($scope));
    }

    function load() {
        DatasetUtils.listDatasetsUsabilityForAny($stateParams.projectKey).success(data => {
            $scope.availableDatasets = data;
        }).error(setErrorInScope.bind($scope));

        DataikuAPI.agentTools.get($stateParams.projectKey, $stateParams.agentToolId).success(function(data) {
            $scope.set(data);

            TopNav.setItem(TopNav.ITEM_AGENT_TOOL, $stateParams.agentToolId, {
                name : $scope.agentTool.name,
                agentToolType: $scope.agentTool.type
            });
            TopNav.setPageTitle($scope.agentTool.name + " - Agent Tool");

            WT1.event("agent-tool-load", {
                agentToolType: $scope.agentTool.type,
            });

            if ($scope.agentTool.type.indexOf("Custom") == 0) {
                $scope.customAgentToolDesc = $rootScope.appConfig.customAgentTools.findLast(c => c.agentType === $scope.agentTool.type.substring(7));
                $scope.pluginDesc = PluginsService.getOwnerPluginDesc($scope.customAgentToolDesc);
            }

            if ($scope.agentTool.type == 'VectorStoreSearch') {
                // keep in sync with retrievable-knowledge.js
                let kbDefaults = {
                    maxDocuments: 4,
                    searchType: "SIMILARITY",
                    similarityThreshold: 0.5,
                    mmrK: 20,
                    mmrDiversity: .25,
                    useAdvancedReranking: false,
                    rrfRankConstant: 60,
                    rrfRankWindowSize: 4,
                };
                if ($scope.agentTool.params === undefined) $scope.agentTool.params = [];
                Object.keys(kbDefaults).forEach((field) => {
                    if (!$scope.agentTool.params[field])
                        $scope.agentTool.params[field] = kbDefaults[field];
                });

                $scope.knowledgeBankSchema = {
                    "columns" : []
                };
                $scope.knowledgeBankSchemaColumnNames = [];

                $scope.getSelectableKBColumns = function() {
                    return angular.copy($scope.knowledgeBankSchemaColumnNames);
                };

                $scope.$watch("agentTool.params.knowledgeBankRef", Debounce().withScope($scope).withDelay(50,200).wrap(function(){
                    if (!($scope.agentTool && $scope.agentTool.params && $scope.agentTool.params.knowledgeBankRef)) return;

                    DataikuAPI.retrievableknowledge.get($stateParams.projectKey, $scope.agentTool.params.knowledgeBankRef).success(retrievableKnowledge => {
                        $scope.knowledgeBank = retrievableKnowledge;
                        $scope.knowledgeBankSchema.columns =  $scope.knowledgeBank.metadataColumnsSchema || [];
                        $scope.knowledgeBankSchemaColumnNames = $scope.knowledgeBankSchema.columns.map((column) => column.name);
                    }).error(setErrorInScope.bind($scope));

                    DataikuAPI.retrievableknowledge.getFullInfo(ActiveProjectKey.get(), SmartId.create($scope.agentTool.params.knowledgeBankRef, $stateParams.projectKey)).then(function({data}){
                        $scope.rkFullInfo = data;
                    }).catch(setErrorInScope.bind($scope));

                    DataikuAPI.retrievableknowledge.getCurrentVersionInfo($stateParams.projectKey, $scope.agentTool.params.knowledgeBankRef).then(function({data}) {
                        $scope.embeddingRecipeDesc = data.embeddingRecipeParams;
                    }).catch(setErrorInScope.bind($scope));
                }));

                $scope.getParentRecipeHref = function () {
                    if ($scope.rkFullInfo && $scope.rkFullInfo.creatingRecipe) {
                        return $state.href('projects.project.recipes.recipe', { projectKey: $scope.rkFullInfo.creatingRecipe.projectKey, recipeName: $scope.rkFullInfo.creatingRecipe.name });
                    }
                    return null;
                };
            }

            if ($scope.agentTool.type == 'LLMMeshLLMQuery') {
                $scope.updateLLMSettings = function() {
                    if ($scope.agentTool && $scope.agentTool.params && $scope.agentTool.params.llmId && $scope.availableLLMs) {
                        $scope.activeLLM = $scope.availableLLMs.find(l => l.id === $scope.agentTool.params.llmId);
                        $scope.temperatureRange = PromptUtils.getTemperatureRange($scope.activeLLM);
                        if (!$scope.agentTool.params.completionSettings) {
                            $scope.agentTool.params.completionSettings = {};
                        }
                    }
                };
                DataikuAPI.pretrainedModels.listAvailableLLMs($stateParams.projectKey, 'GENERIC_COMPLETION')
                    .success(function (data) {
                        $scope.availableLLMs = data.identifiers;
                        $scope.updateLLMSettings();
                    })
                    .error(setErrorInScope.bind($scope));
            }

            if ($scope.agentTool.type == "DataikuReporter") {
                $scope.integrationTypes = [
                    { "id" : "slack-direct", "label" : translate("SCENARIO.INTEGRATIONS.TYPES.SLACK", "Slack") },
                    { "id" : "msft-teams-direct", "label" : translate("SCENARIO.INTEGRATIONS.TYPES.TEAMS", "Microsoft Teams") },
                    { "id": "mail-direct", "label": translate("SCENARIO.INTEGRATIONS.TYPES.MAIL", "Mail") },
                ];
                $scope.variableTypes = [
                    { id: 'TOOL_INPUT',    label:'Tool input',        description: 'Provided by the LLM'},
                    { id: 'QUERY_CONTEXT', label: 'Context variable', description: 'Provided by the context'},
                    { id: 'DSS_VARIABLE',  label: 'DSS variable',     description: 'Instance/project variable'},
                    { id: 'FORMULA',       label: 'Custom formula',   description: 'GREL formula'}
                ];
                if (!$scope.agentTool.params.variables) {
                    $scope.agentTool.params.variables = []
                }
                const fetchDssVariables = function () {
                    const promise = DataikuAPI.flow.recipes.generic.getVariablesWithProjectKey($scope.agentTool.projectKey);
                    promise.success(function(data) {
                        $scope.dssVariables = data;
                        $scope.dssVariablesOptions =  Object.entries(data).map(([k, v]) => {return k;});
                    }).error(setErrorInScope.bind($scope));
                };
                fetchDssVariables();

                $scope.editFormula = function(formulaVariable, variableIndex) {
                    CreateModalFromTemplate("/templates/savedmodels/agents/reporter-tool-formula-editor-modal.html", $scope, 'ToolReporterFormulaEditorController', function (modalScope) {
                        modalScope.formulaVariable = angular.copy(formulaVariable);
                        modalScope.agentTool = $scope.agentTool;
                        modalScope.availableColumns = [
                            {name: 'context', type: 'string', comment: 'Context coming from the LLM query as JSON string (eg: parseJson(context)["FIELD"])'}
                        ].concat($scope.agentTool.params.variables.slice(0, variableIndex).map((v) => {
                            return {name: v.id, type: 'string', comment: v.toolInputDescription || ''};
                        }));
                        modalScope.dssVariables = $scope.dssVariables;
                        modalScope.closeCallback = function(newFormulaVariable) {
                            if (newFormulaVariable) {
                                formulaVariable.expression = newFormulaVariable.expression;
                            }
                        };
                    }, true, 'static', true);
                }
            }

            if ($scope.agentTool.type == "ClassicalPredictionModelPredict") {
                DataikuAPI.savedmodels.listWithAccessible($stateParams.projectKey).success(function (data) {
                    $scope.availableSavedModels = data
                        .filter(sm => SavedModelsService.isVisualMLModel(sm)
                            && sm.miniTask.taskType == 'PREDICTION'
                            && ['BINARY_CLASSIFICATION', 'REGRESSION', 'MULTICLASS'].includes(sm.miniTask.predictionType)
                        );

                    $scope.availableSavedModels.forEach(sm => {
                        if (sm.projectKey == $stateParams.projectKey) {
                            sm.ref = sm.id;
                        } else {
                            sm.ref = sm.projectKey + "." + sm.id;
                        }
                    });
                }).error(setErrorInScope.bind($scope));
            }

            if ($scope.agentTool.type == "InlinePython") {
                if (!$scope.agentTool.params.code) {  // TODO @inline-python-tool Maybe there's a better way to get the first load time?
                    $scope.agentTool.params.code = AgentToolCodeTemplates["empty-template"].codeSample
                    $scope.save();
                }

                TopNav.setLocation(TopNav.TOP_AGENT_TOOLS, 'agent tools', TopNav.TABS_NONE, "design");
            }

            if ($scope.agentTool.type == "ApiEndpoint") {
                DataikuAPI.apideployer.client.listEndpoints().success(function(data) {
                    $scope.availableApiEndpoints = data.map(item => {
                        const id = item.deploymentId + " &gt; " + item.endpointId;

                        let html = `<div style='padding: 8px 0; border-bottom: 1px solid #ccc; white-space: normal; max-width: 100%; overflow-wrap: break-word;'>`;
                        html += `<div style='font-weight: bold; font-size: 18px;'>${id}</div>`;

                        if (!item.hasUsePermission) {
                            html += `<div style='margin-top: 4px;'>No authorization to query this endpoint through the deployer</div>`;
                        }
                        if (!!item.description) {
                            html += `<div style='margin-top: 4px;'>${item.description}</div>`;

                        }
                        html += `<br/>`;
                        html += `<div style='font-style: italic;'>Infrastructure : ${item.infrastructureId}</div>`;
                        html += `<div style='font-style: italic;'>Deployment : ${item.deploymentId}</div>`;
                        html += `</div>`;

                        return {
                            id,
                            type: item.type,
                            deploymentId: item.deploymentId,
                            endpointId: item.endpointId,
                            html,
                            description: item.description || 'No description available for this endpoint',
                            disabled: !(item.hasOpenAPIDocumentation && item.hasUsePermission)
                        }
                    });

                    $scope.$watch("uiState.apiEndpoint", function(nv, ov) {
                        if (nv && nv.length) {
                            const endpoint =  $scope.availableApiEndpoints.find(endpoint => endpoint.id === nv);
                            if (endpoint) {
                                $scope.agentTool.params.deploymentId = endpoint.deploymentId;
                                $scope.agentTool.params.endpointId = endpoint.endpointId;
                                $scope.agentTool.params.$description = endpoint.deploymentId;
                                $scope.uiState = {
                                    ...$scope.uiState,
                                    hasError: false,
                                    persistedApiEndpointName: undefined
                                }
                            }

                            if (nv !== ov) {
                                let defaultInputQuery;
                                if (["PY_FUNCTION", "R_FUNCTION"].includes(endpoint.type)) {
                                    defaultInputQuery = {
                                        "data": {
                                            "<PARAM1_NAME>": "<PARAM1_VALUE>",
                                            "<PARAM2_NAME>": "<PARAM2_VALUE>"
                                        }
                                    }
                                }
                                else if ("DATASET_LOOKUP" === endpoint.type) {
                                    defaultInputQuery = {
                                        "data": {
                                            "data": {}
                                        }
                                    };
                                }
                                else {
                                    defaultInputQuery = {
                                        "data": {
                                            "features": {
                                                "<FEATURE1_NAME>": "<FEATURE1_VALUE>",
                                                "<FEATURE2_NAME>": "<FEATURE2_VALUE>"
                                            }
                                        }
                                    };
                                }
                                $scope.agentTool.quickTestQuery = {"input": defaultInputQuery, "context": {}};
                            }
                        }
                    });

                    if ($scope.agentTool.params.deploymentId && $scope.agentTool.params.endpointId) {
                        const persistedEndpoint = $scope.availableApiEndpoints.find(endpoint => endpoint.deploymentId === $scope.agentTool.params.deploymentId && endpoint.endpointId === $scope.agentTool.params.endpointId);
                        if (persistedEndpoint) {
                            $scope.uiState = {
                                ...$scope.uiState,
                                apiEndpoint: persistedEndpoint.id
                            }
                        } else {
                            $scope.uiState = {
                                ...$scope.uiState,
                                apiEndpoint: null,
                                hasError: true,
                                persistedApiEndpointName: $scope.agentTool.params.deploymentId + " > " + $scope.agentTool.params.endpointId
                            }
                        }
                    } else {
                        $scope.uiState = {
                            ...$scope.uiState,
                            apiEndpoint: null // Going from undefined to null to trigger a cycle detection
                        }
                    }
                }).error(setErrorInScope.bind($scope));
            }
        }).error(setErrorInScope.bind($scope));
    }

    /* *********************** Initialization code **************** */

    TopNav.setLocation(TopNav.TOP_AGENT_TOOLS, 'agent tools', TopNav.TABS_NONE, "design");
    TopNav.setItem(TopNav.ITEM_AGENT_TOOL, $stateParams.agentToolId);

    load();
});

app.controller("AgentToolsListController", function($scope, $controller, $stateParams, DataikuAPI, $state, TopNav, CreateModalFromTemplate, AgentToolService, $rootScope) {
    $controller('_TaggableObjectsListPageCommon', {$scope: $scope});

    $scope.sortBy = [
        { value: 'name', label: 'Name' },
        { value: '-lastModifiedOn', label: 'Last modified'}
    ];
    $scope.selection = $.extend({
        filterQuery: {
            userQuery: '',
            tags: [],
            interest: {
                starred: '',
            },
            inputDatasetSmartName: []
        },
        filterParams: {
            userQueryTargets: ["name", "tags"],
            propertyRules: {tag: 'tags'},
        },
        orderQuery: "-lastModifiedOn",
        orderReversed: false
    }, $scope.selection || {});

    $scope.list = function() {
        DataikuAPI.agentTools.listHeads($stateParams.projectKey).success(function(data) {
            $scope.listItems = data;
            $scope.restoreOriginalSelection();
        }).error(setErrorInScope.bind($scope));

        DataikuAPI.agentTools.list($stateParams.projectKey).success(function(data) {
            $scope.agentsUiData = data.reduce((acc, agent) => {
                acc[agent.id] = AgentToolService.getAgentToolTypesUiData(agent);
                return acc;
            }, {});
        }).error(setErrorInScope.bind($scope));
    };
    TopNav.setLocation(TopNav.TOP_AGENT_TOOLS, TopNav.LEFT_AGENT_TOOLS, TopNav.TABS_NONE, null);
    TopNav.setNoItem();
    $scope.list();

    /* Tags handling */

    $scope.$on('selectedIndex', function(){
        // an index has been selected, we unselect the multiselect
        $scope.$broadcast('clearMultiSelect');
    });
    /* Specific actions */
    $scope.goToItem = function(data) {
        $state.go("projects.project.agenttools.agenttool", {projectKey : $stateParams.projectKey, agentToolId : data.id});
    };

    $scope.newAgentTool = function() {
        CreateModalFromTemplate('/templates/agent-tools/new-agent-tool-modal.html', $rootScope, 'NewAgentToolController');
    };

    if ($stateParams.createTool) {
        $scope.newAgentTool();
        $state.go($state.current, { createTool: null }, { notify: false, reload: false });
    }
});

app.controller('NewAgentToolController', function ($scope, $state, $stateParams, DataikuAPI, AgentToolService, WT1) {
    $scope.selection = $.extend(
        {
            filterParams: {
                userQueryTargets: ['label', 'description'],
            },
        },
        $scope.selection || {}
    );

    $scope.newAgentTool = {
        name: '',
        id: '',
    };
    $scope.defaultName = '';

    $scope.agentToolTypes = AgentToolService.listAgentToolTypesUiData();

    $scope.create = function () {
        if (!$scope.newAgentTool.name) {
            $scope.newAgentTool.name = $scope.defaultName;
        }
        const quickTestQuery = {"input": AgentToolService.getAgentToolQuickTestQueryForType($scope.selection.selectedObject.type), "context": {}};
        DataikuAPI.agentTools
            .create($stateParams.projectKey, { ...$scope.newAgentTool, type: $scope.selection.selectedObject.type, quickTestQuery: quickTestQuery })
            .success(function (data) {
                WT1.event(
                    'agent-tool-create', {
                        agentToolType: $scope.selection.selectedObject.type,
                    });
                $state.go('projects.project.agenttools.agenttool', { agentToolId: data.id });
                $scope.dismiss();
            })
            .error(setErrorInScope.bind($scope));
    };

    $scope.$watch('selection.selectedObject', function (selectedObject) {
        if (selectedObject) {
            $scope.defaultName = selectedObject.label;
        }
    });
});

app.component("agentToolQuicktest", {
    bindings: {
        projectKey: "<",
        agentToolId: "<",
        agentToolType: "<",
        quickTestQuery: "=",
        saveTool: '&',
        datasetSchema: "<",
    },
    templateUrl : '/templates/agent-tools/quicktest.html',
    controller: function($scope, $timeout, DataikuAPI, AgentToolService, ClipboardUtils, CodeMirrorSettingService, WT1) {
        const $ctrl = this;
        $ctrl.codeMirrorSettingService = CodeMirrorSettingService;
        $ctrl.quickTestResponse = undefined;
        $ctrl.quickTestEnabled = true;
        $scope.activeTab = 'response';
        $ctrl.hasCustomQuickTestQueryResolved;

        $ctrl.loadDescriptor = function() {
            DataikuAPI.agentTools.getDescriptor($ctrl.projectKey, $ctrl.agentToolId).success(data => {
                $ctrl.descriptor = data;
                $ctrl.descriptorError = undefined;
            }).error((data, status, headers, _, statusText) => {
                if (data instanceof Error) {
                    throw data;
                }
                const apiError = getErrorDetails(data, status, headers, statusText);
                $ctrl.descriptor = undefined;
                $ctrl.descriptorError = apiError;

                /* Report to WT1 */
                window.APIErrorLogger.error("API error", apiError);
                window.DKUErrorReporting.reportBackendAPIError(apiError);
            });

        };

        $ctrl.quickTest = function() {
            if (!$ctrl.quickTestEnabled) {
                return;
            }
            $ctrl.quickTestEnabled = false;
            $ctrl.saveTool().then(() => {
                DataikuAPI.agentTools.test($ctrl.projectKey, $ctrl.agentToolId, $ctrl.quickTestQuery).success((data) => {
                    WT1.event(
                        'agent-tool-test-run', {
                            agentToolType: $ctrl.agentToolType,
                        });
                    $ctrl.quickTestResponse = data;
                    $scope.activeTab = 'response';
                }).error(setErrorInScope.bind($scope)
                ).finally(() => {
                    $ctrl.quickTestEnabled = true;
                });
            });
        };

        $ctrl.checkCustomQuickTestQuery = function() {
            if (!$ctrl.agentToolType) {
                $ctrl.hasCustomQuickTestQueryResolved = false;
                return;
            }

            if ($ctrl.agentToolType.startsWith('Custom_')) {
                // Assume plugins have sample query
                $ctrl.hasCustomQuickTestQueryResolved = true;
                return;
            }

            DataikuAPI.agentTools.hasSampleQuery($ctrl.projectKey, $ctrl.agentToolId)
                .then((response) => {
                    $ctrl.hasCustomQuickTestQueryResolved = response.data;
                })
                .catch((error) => {
                    $ctrl.hasCustomQuickTestQueryResolved = false;
                    setErrorInScope.bind($scope)(error);
                });
        };

        $ctrl.loadQuickTestQuery = function() {
            $ctrl.saveTool().then(() => {
                DataikuAPI.agentTools.loadSampleQuery($ctrl.projectKey, $ctrl.agentToolId).success((data) => {
                    $ctrl.quickTestQuery = {"input": data, "context": {}};
                }).error(setErrorInScope.bind($scope));
            });
        }

        $ctrl.copyTraceToClipboard = function(value) {
            $scope.traceCopied = true;
            ClipboardUtils.copyToClipboard(JSON.stringify($ctrl.quickTestResponse.fullTrace));
            $timeout(() => { $scope.traceCopied = false}, 5000);
        };

        $scope.$on('agentToolSavedEvent', () => {
            // only reload the descriptor if previously loaded manually
            if ($ctrl.descriptor) {
                $ctrl.loadDescriptor();
            }
        });

        $scope.$on('agentToolSetEvent', () => {
            if ($ctrl.agentToolType && !$ctrl.quickTestQuery) {
                $ctrl.quickTestQuery = {"input": AgentToolService.getAgentToolQuickTestQueryForType($ctrl.agentToolType), "context": {}};
            }
            $ctrl.checkCustomQuickTestQuery();
        });
    },
});

app.directive('agentToolRightColumnSummary', function($controller, $state, $stateParams, $rootScope, DataikuAPI, CreateModalFromTemplate, QuickView,
    ActiveProjectKey, ActivityIndicator, TopNav, WT1, GlobalProjectActions) {

    return {
        templateUrl :'/templates/agent-tools/right-column-summary.html',

        link : function(scope) {
            $controller('_TaggableObjectsMassActions', {$scope: scope});

            scope.$stateParams = $stateParams;
            scope.QuickView = QuickView;

            scope.getSmartName = function (projectKey, name) {
                if (projectKey == ActiveProjectKey.get()) {
                    return name;
                } else {
                    return projectKey + '.' + name;
                }
            }

            scope.refreshData = function() {
                const projectKey = scope.selection.selectedObject.projectKey;
                const name = scope.selection.selectedObject.name;

                DataikuAPI.agentTools.getFullInfo(ActiveProjectKey.get(), scope.selection.selectedObject.id).then(function({data}){
                    if (!scope.selection.selectedObject || scope.selection.selectedObject.projectKey != projectKey || scope.selection.selectedObject.name != name) {
                        return;
                    }
                    data.realName = data.name;
                    scope.agentToolData = data;
                    scope.agentTool = data.agentTool;
                    scope.selection.selectedObject.interest = data.interest;
                    scope.selection.selectedObject.agentToolType = scope.agentTool.type;
                }).catch(setErrorInScope.bind(scope));
            };

            scope.copy = function() {
                if (!scope.agentTool) return;

                DataikuAPI.agentTools.copy(ActiveProjectKey.get(), scope.agentTool.id).then(function({data}) {
                    if ($state.includes('projects.project.agenttools.agenttool')) {
                        $state.transitionTo("projects.project.agenttools.agenttool",{
                            projectKey: $stateParams.projectKey,
                            agentToolId: data.id
                        });
                    } else {
                        scope.list();
                    }
                }).catch(setErrorInScope.bind(scope));
            };

            function save() {
                return DataikuAPI.agentTools.save(scope.agentTool, {summaryOnly: true})
                .success(function() {
                    ActivityIndicator.success("Saved");
                }).error(setErrorInScope.bind(scope));
            }

            scope.$on("objectSummaryEdited", save);

            scope.$watch("selection.selectedObject",function() {
                if(scope.selection.selectedObject != scope.selection.confirmedItem) {
                    scope.agentTool = null;
                    scope.objectTimeline = null;
                }
            });

            scope.$watch("selection.confirmedItem", function(nv) {
                if (!nv) return;
                if (!nv.projectKey) {
                    nv.projectKey = ActiveProjectKey.get();
                }
                scope.refreshData();
            });

            scope.deleteAgentTool = function() {
                GlobalProjectActions.deleteTaggableObject(scope, 'AGENT_TOOL', scope.selection.selectedObject.id, scope.selection.selectedObject.name);
            };


            /** Custom fields from plugins **/
            scope.editCustomFields = function() {
                if (!scope.agentTool) {
                    return;
                }
                let modalScope = angular.extend(scope, {objectType: 'AGENT_TOOL', objectName: scope.agentTool.name, objectCustomFields: scope.agentTool.customFields});
                CreateModalFromTemplate("/templates/taggable-objects/custom-fields-edit-modal.html", modalScope).then(function(newCustomFields) {
                    WT1.event('custom-fields-save', {objectType: 'AGENT_TOOL'});
                    const oldCustomFields = angular.copy(scope.agentTool.customFields);
                    scope.agentTool.customFields = newCustomFields;
                    return save().then(function() {
                            $rootScope.$broadcast('customFieldsSaved', TopNav.getItem(), scope.agentTool.customFields);
                        }, function() {
                            scope.agentTool.customFields = oldCustomFields;
                        });
                });
            };

            const customFieldsListener = $rootScope.$on('customFieldsSaved', scope.refreshData);
            scope.$on("$destroy", customFieldsListener);
        }
    }
});


app.controller("AgentToolPageRightColumnActions", function($controller, $scope, $rootScope, DataikuAPI, $stateParams, ActiveProjectKey) {

    $controller('_TaggableObjectPageRightColumnActions', {$scope: $scope});

    $scope.selection = {};

    DataikuAPI.agentTools.get(ActiveProjectKey.get(), $stateParams.agentToolId).success((data) => {
        $scope.selection = {
            selectedObject : data, confirmedItem : data,
        };
        $scope.selection.selectedObject.nodeType = "AGENT_TOOL";
    }).error(setErrorInScope.bind($scope));

    $scope.renameObjectAndSave = function(newName) {
        $scope.selection.selectedObject.name = newName;
        return DataikuAPI.agentTools.save($scope.selection.selectedObject);
    };

    /** Interests (star, watch) **/
    function updateUserInterests() {
        DataikuAPI.interests.getForObject($rootScope.appConfig.login, "AGENT_TOOL", ActiveProjectKey.get(), $scope.selection.selectedObject.id).success(function(data) {
            $scope.selection.selectedObject.interest = data;
            $scope.agentToolData.interest = data;
        }).error(setErrorInScope.bind($scope));
    }

    const interestsListener = $rootScope.$on('userInterestsUpdated', updateUserInterests);
    $scope.$on("$destroy", interestsListener);
});

app.directive("toolIntegrationParams", function(translate, DataikuAPI, CodeMirrorSettingService) {
    return {
        scope : {
            hook : '=',
            form : '=',
        },
        templateUrl : '/templates/agent-tools/integrations/tool-integration-params.html',
        link : function($scope) {
            $scope.translate = translate;
            $scope.codeMirrorSettingService = CodeMirrorSettingService;
            $scope.warningUnusableChannelMessage = 'You are not authorized to use this channel. The tool will fail when used by users without this authorization';

            DataikuAPI.agentTools.listEmailTemplates().success(function(data){
                $scope.emailTemplates = data;
            }).error(setErrorInScope.bind($scope));

            $scope.$watch("hook.type", function() {
                $scope.hook.configuration = $scope.hook.configuration || {};
                if ($scope.hook.type) {
                    let hook = $scope.hook;
                    let hookConf = hook.configuration;
                    switch (hook.type) {
                        case "slack-direct":
                            hookConf.message = hookConf.message || "${message}"
                            hookConf.messageMode = hookConf.messageMode || "TEXT_MESSAGE"
                            break;

                        case "mail-direct":
                            hookConf.subject = hookConf.subject || "Tool report";
                            hookConf.messageSource = hookConf.messageSource || "INLINE";
                            hookConf.templateName = hookConf.templateName || "default.ftl";
                            hookConf.templateFormat = hookConf.templateFormat || "FREEMARKER";
                            hookConf.sendAsHTML = hookConf.sendAsHTML ?? true;
                            break;

                        case "msft-teams-direct":
                            hookConf.message = hookConf.message || "${message}"
                            hookConf.webhookType = hookConf.webhookType || "WORKFLOWS";
                            break;
                    }
                }
            });
        }
    }
});

app.controller("ToolReporterFormulaEditorController", function($scope, $element, DataikuAPI, Expressions) {

    $scope.validate = function() {
        return !$scope.validationPending && $scope.status && $scope.status.ok;
    };

    $scope.getColumns = function() {
        return $scope.availableColumns;
    };
    $scope.getSchema = function() {
        return {columns: $scope.availableColumns};
    };

    $scope.onExpressionChange = function() {

    };
    $scope.clickedOnColumnName = function(colName) {
        let valueToAdd;
        if (colName.match(/^[a-z0-9_]+$/i)) {
            valueToAdd = colName;
        } else {
            valueToAdd = `val('${colName}')`;
        }
        $scope.addFormulaElement(valueToAdd);
    };

    $scope.clickedOnVariableName = function(varName) {
        $scope.addFormulaElement('variables["' + varName + '"]');
    };

    $scope.addFormulaElement = function(code) {
        // replace selection and focuses editor
        var cm = $('.CodeMirror', $element).get(0).CodeMirror;
        cm.replaceSelection(code);
        cm.focus();
    };

    $scope.validationPending = true;
    $scope.validateExpression = function() {
        $scope.validationPending = false;
        $scope.validationInProgress = $scope.validationInProgress ? $scope.validationInProgress+1 : 1;
        Expressions.validateExpression($scope.formulaVariable.expression, $scope.getSchema(), null)
            .success(function(data) {
                $scope.status = $scope.status || {};
                $.extend($scope.status, data);
                $scope.status.validated = true;
                $scope.validationInProgress--;
            })
            .error(function(data) {
                $scope.validationInProgress--;
                setErrorInScope.bind($scope);
            });
    };

    $scope.onError = function(e) {
    };
    $scope.cancel = function() {
        $scope.closeCallback(null);
        $scope.dismiss();
    };
    $scope.confirm = function() {
        $scope.closeCallback($scope.formulaVariable);
        $scope.dismiss();
    };
});

app.controller("ToolReporterMessageController", function($scope, CodeMirrorSettingService) {

    $scope.editorOptions = CodeMirrorSettingService.get('text/plain', {onLoad: function(cm) {$scope.codeMirror = cm;}});

    $scope.htmlEditorOptions = CodeMirrorSettingService.get('text/html', {onLoad: function(cm) {$scope.codeMirror = cm;}});

    $scope.prefillMessageAsHTML = function() {
        if ($scope.hook.configuration.message === undefined) {
            $scope.hook.configuration.message = "<html><head><style type=\"text/css\"> * { font-family: Verdana, sans-serif; } table { width: 100%; border-collapse: collapse; border: 1px solid #ddd; } td { padding: 10px; text-align: left; border: 1px solid #ddd; } th { background-color: #f2f2f2; padding: 10px; text-align: left; border: 1px solid #ddd; }</style></head>\n" +
                "    <body>\n" +
                "        <h1>Tool report</h1>\n" +
                "        <p>${message}</p>\n" +
                "        \n" +
                "    </body>\n" +
                "</html>";
        }
    };
});

app.directive('agentToolInstructionsForm', function() {
        return {
            restrict: 'E',
            scope: {
                agentTool: '=',
                showToolDescriptionModal: '&'
            },
            templateUrl: '/templates/agent-tools/agent-tool-instructions-form.html'
        };
    });

app.component("agentToolSelector", {
    bindings: {
        availableAgentTools: "<",
        selectedAgentTool: "=",
        onChange: "<",
        defaultOpened: '<',
        disableAlertWarning: '<',
        placeholder: '<', // Optional, defaults to "Nothing selected"
    },
    template: `
<basic-select ng-if="$ctrl.availableAgentTools"
          items="$ctrl.availableAgentTools | orderBy: 'name'"
          ng-model="$ctrl.selectedAgentTool"
          bind-value="id"
          bind-label="name"
          group-by-fn="$ctrl.niceAgentToolType"
          searchable="true"
          placeholder="{{ $ctrl.placeholder }}"
          invalidate-on-ghosts="true"
          default-opened="$ctrl.defaultOpened"
          ghost-items-tooltip="Agent tool not available. It has been disabled, retired, or is not supported for this feature. Choose another agent tool or contact your administrator."
          custom-ng-select-class="ng-dropdown-panel--500 
                                  ng-select--custom-dropdown-height 
                                  ng-select--custom-x-overflow
                                  ng-select--custom-reset-default-padding"></basic-select>
<div class="mbot0 mtop8 alert alert-warning" ng-if="!$ctrl.isAvailable && !$ctrl.disableAlertWarning">
Agent tool not available. It has been disabled, retired, or is not supported for this feature. Choose another Agent tool or contact your administrator. <strong>{{$ctrl.selectedAgentTool.id}}</strong>
</div>`,
    controller: function ($scope, $filter) {
        const $ctrl = this;

        $scope.$watch(
            function () {
                return $ctrl.selectedAgentTool;
            },
            function (newVal, oldVal) {
                $ctrl.updateAvailability();
                if (newVal !== oldVal && $ctrl.onChange) {
                    $ctrl.onChange();
                }
            }
        );

        $ctrl.$onChanges = () => {
            if (typeof $ctrl.placeholder === 'undefined' || $ctrl.placeholder === null) {
                $ctrl.placeholder = "Nothing selected";
            }

            $ctrl.updateAvailability();
        };

        $ctrl.isAvailable = true;
        $ctrl.updateAvailability = () => {
            if ($ctrl.availableAgentTools && $ctrl.selectedAgentTool) {
                $ctrl.isAvailable = !!$ctrl.availableAgentTools.find((l) => l.id === $ctrl.selectedAgentTool);
            }
        };

        $ctrl.niceAgentToolType = (agentToolOption) => {
            return $filter('niceAgentToolType')(agentToolOption.type);
        };

    },
});

app.filter("niceAgentToolType", function(AgentToolService){
    return function(rawType) {
        const uiData = AgentToolService.getAgentToolTypesUiDataForType(rawType);
        if (uiData) {
            return uiData.label;
        } else {
            return rawType;
        }
    };
});

})();
