(function(){
'use strict';

var app = angular.module('dataiku.analysis.mlcore');


/**
 * Controllers, services and directives for the views of a single model
 * of a MLTask
 */

/**
 * Injected into all controllers that display a single PMLTask model. Handles:
 *   - the global nav handle to switch between PMLTask models
 *   - setting the top nav item
 *   - retrieving the distributions of the features
 */
app.controller("_PMLModelBaseController", function($scope, $controller, $q, DataikuAPI, $stateParams, PMLFilteringService, CreateModalFromTemplate,
                                                   MLRoutingService, $state, ExportModelDatasetService, CachedAPICalls){
    $controller("_ModelUtilsController", {$scope:$scope});
    $controller("_MLModelBaseController", {$scope:$scope});
    $controller("exportModelController", {$scope:$scope});

    $scope.isMLBackendType = function(backendType) {
        return $scope.modelData
            && $scope.modelData.coreParams
            && $scope.modelData.coreParams.backendType === backendType;
    }

    $scope.getPredictionDesignTabPrefix = () => MLRoutingService.getPredictionDesignTabPrefix($scope);

    $scope.isClassicalPrediction = function() {
        if (!$scope.modelData) return;
        return ["BINARY_CLASSIFICATION", "REGRESSION", "MULTICLASS"].includes($scope.modelData.coreParams.prediction_type);
    };

    $scope.isTimeseriesPrediction = function() {
        if (!$scope.modelData) return;
        return $scope.modelData.coreParams.prediction_type === 'TIMESERIES_FORECAST'
    };

    $scope.isDeepHubPrediction = function() {
        if (!$scope.modelData) return;
        return $scope.isMLBackendType("DEEP_HUB");
    };

    $scope.isCausalPrediction = function() {
        if (!$scope.modelData) return;
        return ["CAUSAL_REGRESSION", "CAUSAL_BINARY_CLASSIFICATION"].includes($scope.modelData.coreParams.prediction_type);
    };

    $scope.isDeepNeuralNetworkPrediction = function() {
        if ($scope.modelData && $scope.modelData.modeling && $scope.modelData.modeling.algorithm) {
            const algoName = $scope.modelData.modeling.algorithm;
            if (["DEEP_NEURAL_NETWORK_REGRESSION", "DEEP_NEURAL_NETWORK_CLASSIFICATION"].includes(algoName)) {
                return true;
            }
        }
        return false;
    }

    function updateCMGIfDefined() {
        // This is to initialize the cost matrix gain in the frontend when relevant (i.e. in model report, but not in predicted data).
        // TODO: The current implementation is awful by relying on shared scope between controllers, where the `updateCMG` is only defined
        // if another controller (_PredictionModelReportController) was already initialized before.
        // Let's clean that in the future!
        if ($scope.updateCMG) {
            $scope.updateCMG();
        }
    }

    $scope.fetchModelIfNotInScope = function() {
        if ($scope.modelData) return $q.when(null);
        else {
            const treatment = $stateParams.treatment || null; // For causal predictions with multi-valued treatments
            return DataikuAPI.ml.prediction.getModelDetails($stateParams.fullModelId, treatment).success(function(data){
                // replace only if absent or different, else may have been enriched, e.g. in _PredictionModelReportController
                if (!$scope.modelData || $scope.modelData.fullModelId !== data.fullModelId) {
                    $scope.modelData = data;
                    updateCMGIfDefined();
                    CachedAPICalls.pmlDiagnosticsDefinition.then(pmlDiagnosticsDefinition => {
                        $scope.diagnosticsDefinition = pmlDiagnosticsDefinition($scope.modelData.coreParams.backendType, $scope.modelData.coreParams.prediction_type);
                    });
                }
            });
        }
    }

    $scope.mlTasksContext.delete = function() {
        $scope.deleteTrainedAnalysisModel();
    }

    $scope.fetchModelIfNotInScope().then(function () {
        const algoName = $scope.modelData.modeling.algorithm;
        $scope.mlTasksContext.noPredicted = /^(MLLIB|SPARKLING|VERTICA|PYTHON_ENSEMBLE|SPARK_ENSEMBLE|KERAS)/i.test(algoName);
        $scope.mlTasksContext.noExport =
            $scope.mlTasksContext.noPredicted ||
            algoName === "CUSTOM_PLUGIN" ||
            $scope.isCausalPrediction() ||
            $scope.isDeepHubPrediction() ||
            $scope.isTimeseriesPrediction()||
            $scope.isDeepNeuralNetworkPrediction();
    });

    $scope.mlTasksContext.deploy = function(){
        $scope.fetchModelIfNotInScope().then(function(){
            $scope.deploymentOrigin = "lab-model-page";
             CreateModalFromTemplate("/templates/analysis/prediction/model/deploy-modal.html",
                $scope,"AnalysisPredictionDeployController");
        });
    }
    $scope.mlTasksContext.exportNotebook = function(){
        $scope.fetchModelIfNotInScope().then(function(){
            CreateModalFromTemplate("/templates/analysis/mlcommon/export-notebook-modal.html",
            $scope,"AnalysisPredictionExportNotebookController");
        });
    }

    $scope.mlTasksContext.exportTrainTestSets = function() {
        ExportModelDatasetService.exportTrainTestSets($scope, $scope.mlTasksContext.model.fullModelId);
    };

    $scope.mlTasksContext.exportTrainTestSetsForbiddenReason = function() {
        return ExportModelDatasetService.exportTrainTestSetsForbiddenReason($scope.mlTasksContext.model, $scope.canWriteProject());
    };

    $scope.mlTasksContext.exportPredictedData = function() {
        ExportModelDatasetService.exportPredictedData($scope, $scope.mlTasksContext.model.fullModelId, $scope.table.totalRows);
    };

    $scope.mlTasksContext.exportPredictedDataForbiddenReason = function() {
        return ExportModelDatasetService.exportPredictedDataForbiddenReason($scope.mlTasksContext.model);
    };

    $scope.mlTasksContext.showExportPredictedData = function() {
        if ($scope.table) {
            return true;
        } else {
            return false;
        }
    };

    DataikuAPI.analysis.pml.getTaskStatus($stateParams.projectKey, $stateParams.analysisId, $stateParams.mlTaskId).success(function(data){
        if (!$scope.mlTasksContext.activeMetric) {
            $scope.mlTasksContext.activeMetric = data.headSessionTask.modeling.metrics.evaluationMetric;
        }
    });

    DataikuAPI.analysis.pml.getModelSnippets($stateParams.projectKey, $stateParams.analysisId, $stateParams.mlTaskId).success(function(data){
        $scope.mlTasksContext.models = Object.values(data).filter(function(m){
            return m.trainInfo.state == "DONE" && m.fullModelId != $stateParams.fullModelId;
        });
        $scope.mlTasksContext.models.sort(function(a, b) {
            var stardiff = (0+b.userMeta.starred) - (0+a.userMeta.starred)
            if (stardiff != 0) return stardiff;
            return b.sessionDate - a.sessionDate;
        });
        $scope.mlTasksContext.models.forEach(function(m){
            m.mainMetric = PMLFilteringService.getMetricFromSnippet(m, $scope.mlTasksContext.activeMetric);
            m.mainMetricStd = PMLFilteringService.getMetricStdFromSnippet(m, $scope.mlTasksContext.activeMetric);
        });
    }).error(setErrorInScope.bind($scope));
});

app.controller("exportModelController", function($scope, MLExportService) {
    $scope.mlTasksContext.showExportModel = function() {
        if (!$scope.modelData) return false;
        return MLExportService.showExportModel($scope.appConfig);
    };
    $scope.mlTasksContext.mayExportModel = function(type) {
        if (!$scope.modelData) return false;
        return MLExportService.mayExportModel($scope.appConfig, $scope.modelData, type);
    };
    $scope.mlTasksContext.downloadDocForbiddenReason = function() {
        if (!$scope.modelData) return null;
        return MLExportService.downloadDocForbiddenReason($scope.appConfig, $scope.modelData);
    };
    $scope.mlTasksContext.downloadDoc = function() {
        if (!$scope.modelData) return false;
        return MLExportService.downloadDoc($scope)
    }
    $scope.mlTasksContext.exportModelModal = function() {
        if (!$scope.modelData) return null;
        MLExportService.exportModelModal($scope, $scope.modelData, true);
    }
    $scope.mlTasksContext.disableExportModelModalReason = function() {
        if (!$scope.modelData) return null;
        return MLExportService.disableExportModelModalReason($scope.modelData)
    }
});


/**
 * Controller for displaying results screen of a prediction model
 * in a PMLTask
 */
app.controller("PMLModelReportController", function($scope, $controller, TopNav, WebAppsService, MLRoutingService, $state) {
    TopNav.setLocation(TopNav.TOP_ANALYSES, TopNav.LEFT_ANALYSES, "PREDICTION-ANALYSIS-MODEL", "report");

    $controller("_PMLModelBaseController",{$scope:$scope});
    $controller("_PredictionModelReportController",{$scope:$scope});


    const baseRoute = "projects.project.analyses.analysis.ml.predmltask.model.report";
    $scope.$on('$stateChangeStart', function (event, toState, toParams, fromState, fromParams) {
        if (!$scope.modelData) {
            return;
        }
        MLRoutingService.goToRouteSuffixIfBase($state, toState, toParams, fromState, fromParams, baseRoute, MLRoutingService.getPredictionReportSummaryTab($scope.isDeepHubPrediction(), false), event);
    });

    $scope.fetchModelIfNotInScope()
    .then(function() {
        if ($state.current.name !== baseRoute) {
            return;
        }
        // redirect to summary page when arriving on "report"
        $state.go(`.${MLRoutingService.getPredictionReportSummaryTab($scope.isDeepHubPrediction(), false)}`, null, {location: "replace"});
    })
    .then(function() {
        const contentType = `${$scope.modelData.coreParams.taskType}/${$scope.modelData.coreParams.backendType}/${$scope.modelData.modeling.algorithm}`.toLowerCase();
        $scope.modelSkins = WebAppsService.getSkins(
            'ANALYSIS', '',
            { predictionType: $scope.modelData.coreParams.prediction_type, backendType: $scope.modelData.coreParams.backendType, contentType },
            $scope.staticModelSkins
        );
    });
});

app.controller("PMLPartModelReportController", function($scope, $controller, DataikuAPI, $state, $stateParams, $location, MLRoutingService,
    FullModelLikeIdUtils, ModelDataUtils){
    $scope.ModelDataUtils = ModelDataUtils;
    const fullModelId = $stateParams.fullModelId || $scope.fullModelId;

    const setScopeSnippets = (partSnippets) => {
        if (partSnippets.fullModelId === fullModelId) {
            $scope.currentPartitionedModel = partSnippets;
        } else {
            $scope.currentPartitionedModel = Object.values(partSnippets.partitions.summaries)
                .find(m => m.snippet.fullModelId === fullModelId)
                .snippet;
            $scope.currentPartitionName = $scope.currentPartitionedModel.partitionName;
        }

        $scope.partitionedModelSnippets = partSnippets;

        if ($scope.mlTasksContext) {
            $scope.mlTasksContext.noPredicted = !$scope.isOnModelPartition();
            $scope.mlTasksContext.noExport = true;
            $scope.mlTasksContext.partitionName = $scope.currentPartitionName;
        }
        if ($scope.smContext) {
            $scope.smContext.partitionName = $scope.currentPartitionName;
        }
    };

    $scope.goToBaseModel = function() {
        $state.go('^.' + MLRoutingService.getPredictionReportSummaryTab(false, false), {fullModelId : $scope.partitionedModelSnippets.fullModelId});
    };

    $scope.goToPartitionedModel = function(partitionName) {
        if (partitionName) {
            // Switch between partitions, stay on the same tab
            const partitionFmi = $scope.partitionedModelSnippets.partitions.summaries[partitionName].snippet.fullModelId;
            $state.go('.', {fullModelId: partitionFmi})
        } else {
            // Switch from overall model to partitioned, go to the first one done
            const firstPartitionFmi = Object.values($scope.partitionedModelSnippets.partitions.summaries)
                .find(summary => summary.state.endsWith('DONE')).snippet.fullModelId;
            $state.go('.', {fullModelId: firstPartitionFmi});
        }
    };

    $scope.showPartitions = function() {
        return !$scope.insight;
    }

    $scope.isOnModelPartition = function (){
        return $scope.currentPartitionedModel && $scope.currentPartitionedModel.partitionName;
    };

    $scope.isOnPartitionedBaseModel = function () {
        return !$scope.currentPartitionedModel || !$scope.currentPartitionedModel.partitionName;
    };

    /* In analysis */
    $scope.$watch('mlTasksContext.models', function(models) {
        if (!models) {
            return;
        }
        DataikuAPI.analysis.pml.getPartitionedModelSnippets(fullModelId)
            .then((result) => setScopeSnippets(result.data), setErrorInScope.bind($scope));
    });

    /* In saved model */
    $scope.$watch('versionsContext.currentVersion', function(currentVersion) {
        if (!$scope.versionsContext || !$scope.versionsContext.versions) {
            return;
        }

        const baseFullModelId = FullModelLikeIdUtils.getBase(fullModelId);
        const baseVersion = $scope.versionsContext.versions
            .find(m => m.snippet.fullModelId === baseFullModelId);

        if (!$scope.versionsContext.currentVersion || !$scope.versionsContext.currentVersion.snippet) {
            $scope.versionsContext.currentVersion = Object.values(baseVersion.snippet.partitions.summaries)
                .find(m => m.snippet.fullModelId === fullModelId);
        }

        if (baseVersion) {
            setScopeSnippets(baseVersion.snippet);
        } else {
            setScopeSnippets(currentVersion.snippet);
        }
    });
});

app.controller("AnalysisPredictionDeployController", function($scope, $controller, DataikuAPI, $state, $stateParams,
    Assert, TopNav, DatasetUtils, Dialogs, PMLSettings, WT1){
    $scope.onSelectTrain = function(){
        $scope.uiState.selectedMode = 'train';
        $scope.formData = {
            optimizationBehaviour: "REDO",
            thresholdBehaviour: "REDO"
        };

        function autoSetModelName() {
            if ($scope.formData.modelName || !$scope.formData.trainDatasetSmartName) {
                return;
            }

            switch ($scope.modelData.coreParams.prediction_type) {
                case "DEEP_HUB_IMAGE_CLASSIFICATION":
                    $scope.formData.modelName = `Image classification on ${$scope.formData.trainDatasetSmartName}`;
                    break;
                case "DEEP_HUB_IMAGE_OBJECT_DETECTION":
                    $scope.formData.modelName = `Object detection on ${$scope.formData.trainDatasetSmartName}`;
                    break;
                case "CAUSAL_REGRESSION":
                case "CAUSAL_BINARY_CLASSIFICATION": {

                    $scope.formData.modelName = `Predict effect of ${$scope.getTreatmentVariable()} on ${$scope.getTargetVariable()}`;
                    break;
                }
                default: {
                    const target = $scope.getTargetVariable();
                    const predictionType = $scope.getPredictionType();
                    const predTypeStrings = PMLSettings.task.predictionTypes.find(e => e.type === predictionType);
                    const partitionedPart = $scope.isPartitionedModel() ? ', partitioned' : '';
                    const predTypeNamePart = predTypeStrings?` (${predTypeStrings.shortName}${partitionedPart})`:"";
                    $scope.formData.modelName = `Predict ${target}${predTypeNamePart}`;
                    break;
                }
            }
        }

        $scope.$watch("trainDatasetSmartName", autoSetModelName);

        var splitParams = $scope.modelData.splitDesc.params;
        if (splitParams.ttPolicy == 'SPLIT_SINGLE_DATASET') {
            $scope.formData.trainDatasetSmartName = $scope.analysisCoreParams.inputDatasetSmartName;
        } else if (splitParams.ttPolicy == 'EXPLICIT_FILTERING_TWO_DATASETS') {
            $scope.formData.trainDatasetSmartName = splitParams.eftdTrain.datasetSmartName;
            $scope.formData.testDatasetSmartName = splitParams.eftdTest.datasetSmartName;
        } else if (splitParams.ttPolicy == 'EXPLICIT_FILTERING_SINGLE_DATASET') {
            $scope.formData.trainDatasetSmartName = $scope.analysisCoreParams.inputDatasetSmartName;
        } else {
            throw "Unhandled split mode";
        }

        $scope.formData.managedFolderSmartId = $scope.modelData.coreParams.managedFolderSmartId;
        $scope.isManagedFolderAvailable = function() {
            return !!$scope.modelData.coreParams.managedFolderSmartId;
        }

        $scope.deployTrain = function() {
            var options = {
                redoOptimization: $scope.formData.optimizationBehaviour === "REDO",
                redoThresholdOptimization: $scope.formData.thresholdBehaviour === "REDO",
                fixedThreshold: $scope.modelData.userMeta.activeClassifierThreshold
            };
            DataikuAPI.analysis.pml.deployTrain($stateParams.fullModelId,
                $scope.formData.trainDatasetSmartName, $scope.formData.testDatasetSmartName,
                $scope.formData.modelName,
                $scope.formData.managedFolderSmartId,
                options
            ).success(function(data){
                WT1.event("analysis-pml-deploy-model-to-the-flow", { from: $scope.deploymentOrigin })
                $scope.dismiss();
                $state.go("projects.project.flow");
            }).error(setErrorInScope.bind($scope));
        };
    };
    $scope.onSelectRedeployTrain = function() {
        $scope.uiState.selectedMode = 'redeploy-train';
        $scope.formData = {
            redeployTrainRecipeName: $scope.redeployables.length === 1
                ? $scope.redeployables[0].recipeName : null,
            redeployTrainActivate: true,
            optimizationBehaviour: "REDO",
            thresholdBehaviour: "REDO"
        };
        $scope.redeployTrain = function() {
            var options = {
                redoOptimization: $scope.formData.optimizationBehaviour === "REDO",
                redoThresholdOptimization: $scope.formData.thresholdBehaviour === "REDO",
                fixedThreshold: $scope.modelData.userMeta.activeClassifierThreshold
            };
            DataikuAPI.analysis.pml.redeployTrain($stateParams.fullModelId,
                $scope.formData.redeployTrainRecipeName,
                $scope.formData.redeployTrainActivate,
                options
            ).success(function(data){
                WT1.event("analysis-pml-redeploy-model-to-the-flow", { from: $scope.deploymentOrigin })
                var go = $state.go.bind($state, "projects.project.flow");
                var parentScope = $scope.$parent;
                $scope.dismiss();
                if (data.schemaChanged) {
                    let warningMessage;
                    if ($scope.isTimeseriesPrediction()) {
                        warningMessage = "The updated model has a different preparation script schema, custom metric configuration or time series quantile configuration compared to the previous version.\n"
                    } else {
                        warningMessage = "The updated model has a different preparation script schema or custom metric configuration compared to the previous version.\n"
                    }
                    Dialogs.ackMarkdown(parentScope, "Schema changed", warningMessage +
                        "This change may affect the output schema of any downstream scoring and evaluation recipes."
                    ).then(go);
                } else {
                    go();
                }
            }).error(setErrorInScope.bind($scope));
        };
    };
    $scope.suggestRedeployTrain = function(redeployables) {
        $scope.uiState.selectedMode = 'can-redeploy';
        $scope.redeployables = redeployables;
        $scope.canRedeploy = true;
    };

    function main(){
        $scope.uiState = {};
        Assert.inScope($scope, "modelData");
        Assert.inScope($scope, "analysisCoreParams");

        DataikuAPI.analysis.pml.listRedeployableTrain($stateParams.fullModelId).success(function(data) {
            if (data && data.length) {
                $scope.suggestRedeployTrain(data);
            } else {
                $scope.onSelectTrain();
            }
        }).error($scope.onSelectTrain);

        DatasetUtils.listDatasetsUsabilityForAny($stateParams.projectKey).success(function(data){
            $scope.availableDatasets = data;
        });
    }

    main();
});


app.controller("DownloadModelDocumentationController", function($scope, DataikuAPI, WT1, FutureWatcher, ProgressStackMessageBuilder) {

    $scope.radio = { type: "default" };
    $scope.newTemplate = {};
    $scope.renderingInProgress = false;
    $scope.renderingDone = false;
    $scope.downloaded = false;
    $scope.errorOccured = false;
    $scope.hasDesignChangesOccurred = false;
    $scope.data = undefined;

    // The model data is stored in $scope.modelData for analysis and in $scope.smContext.model for savedmodels
    let fullModelId = (($scope.modelData) ? $scope.modelData : $scope.smContext.model).fullModelId;

    // Compute design changes before starting MDG to avoid wasting the user's time on generating documents 
    DataikuAPI.ml.prediction.getPreDocGenInfoMessages(fullModelId).success((data) => {
        $scope.data = data;
    }).error(setErrorInScope.bind($scope));

    $scope.export = (templateType) => {
        if (templateType === "custom") {
            WT1.event("render-model-documentation", {type: "custom"});
            DataikuAPI.ml.prediction.docGenCustom($scope.newTemplate.file, fullModelId, (e) => {
                // You can use here Math.round(e.loaded * 100 / e.total) to compute and display to the user the progress percentage of the template upload
            })
            .then(watchJobId)
            .catch((error) => {
                setErrorInScope2.call($scope, error);
            });
        } else {
            WT1.event("render-model-documentation", {type: "default"});
            DataikuAPI.ml.prediction.docGenDefault(fullModelId)
            .success(watchJobId)
            .error(setErrorInScope.bind($scope));
        }

        function watchJobId(initialResponse) {
            $scope.initialResponse = angular.fromJson(initialResponse);
            $scope.data = undefined;

            FutureWatcher.watchJobId($scope.initialResponse.jobId)
            .success(function(response) {
                let exportId = response.result.exportId;
                $scope.data = response.result.data;
                $scope.text = "The model documentation is ready.";
                if ($scope.data.maxSeverity === 'WARNING') {
                    $scope.text += " Be aware that the placeholders which couldn't be resolved are not shown in the model documentation.";
                } else if ($scope.data.maxSeverity === 'ERROR') {
                    $scope.text = "";
                    $scope.errorOccured = true;
                }
                
                // When an error occured it means the generation failed to produce the documentation, there is nothing to download
                if (!$scope.errorOccured) {
                    $scope.modelDocumentationURL = DataikuAPI.savedmodels.getModelDocumentationExportURL(exportId);
                }
                $scope.renderingInProgress = false;
                $scope.renderingDone = true;
            }).update(function(response) {
                $scope.futureResponse = response;
                $scope.percentage =  ProgressStackMessageBuilder.getPercentage(response.progress);
                $scope.stateLabels = ProgressStackMessageBuilder.build(response.progress, true);
            }).error(function(response, status, headers) {
                setErrorInScope.bind($scope)(response, status, headers);
            });
        }

        $scope.percentage = 100;
        $scope.stateLabels = "Uploading the template";
        $scope.renderingInProgress = true;
    };

    $scope.download = function() {
        downloadURL($scope.modelDocumentationURL);
        WT1.event("download-model-documentation");
        $scope.downloaded = true;
    };

    $scope.abort = function() {
        DataikuAPI.futures.abort($scope.initialResponse.jobId).error(setErrorInScope.bind($scope));
        $scope.dismiss();
        WT1.event("abort-model-documentation-rendering");
    }
});


app.controller("AnalysisPredictionExportNotebookController", function($scope, $controller, DataikuAPI, $state, $stateParams, Assert, TopNav) {
    Assert.inScope($scope, "modelData");
    Assert.inScope($scope, "analysisCoreParams");

    $scope.formData = {};

    var cp = $scope.modelData.coreParams;
    $scope.formData.notebookName = "Predict " + cp.target_variable + " in " +
        $scope.analysisCoreParams.inputDatasetSmartName.replace(/\./g, '_');

    $scope.createNotebook = function() {
        DataikuAPI.analysis.pml.createNotebook($stateParams.fullModelId, $scope.formData.notebookName)
        .success(function(data){
            $scope.dismiss();
            $state.go("projects.project.notebooks.jupyter_notebook", {notebookId : data.id});
        }).error(setErrorInScope.bind($scope));
    }
});


app.controller("PredictionScatterPlotController", function($scope){
    // remove duplicates in the scatter plot data - prevent d3 voronoi issue https://github.com/d3/d3/issues/1908
    $scope.$watch("modelData.perf.scatterPlotData", function(nv){
        if (!nv) { return }
        var hashTbl = {};
        for (var i=nv.x.length-1;i>=0;i--) {
            var key = nv.x[i] + '#' + nv.y[i];
            if (hashTbl[key]) { nv.x.splice(i,1) ; nv.y.splice(i,1) }
            else { hashTbl[key] = true }
        }
        $scope.spd = nv;

        // Signal to Puppeteer that the content of the element has been loaded and is thus available for content extraction
        $scope.puppeteerHook_elementContentLoaded = true;
    });
});


app.directive("analysisPredictionPredictedTable", function($q, Assert, MonoFuture, WT1, TopNav) {
    return {
        scope: true,
        priority: 1,
        controller: function($scope, $stateParams, $state, DataikuAPI, $controller) {
            $controller("_PMLModelBaseController", {$scope});
            WT1.event("pml-model-prediction-open");
            TopNav.setLocation(TopNav.TOP_ANALYSES, TopNav.LEFT_ANALYSES, "PREDICTION-ANALYSIS-MODEL", "predictedtable");
            Assert.inScope($scope, "loadMLTask");
            $scope.loadMLTask();

            DataikuAPI.ml.prediction.getModelDetails($stateParams.fullModelId)
                .success(function(data) {
                    if ($scope.mlTasksContext) $scope.mlTasksContext.model = data;
                    // Signal to Puppeteer that the content of the element has been loaded and is thus available for content extraction
                    $scope.puppeteerHook_elementContentLoaded = true;
                })
                .error(setErrorInScope.bind($scope));
        }
    }
});


app.directive('analysisPredictionPredictedCharts', function(Logger, DataikuAPI, WT1, TopNav, ChartNavigationService) {
return {
        scope: true,
        controller: function ($scope, $stateParams, $state, $controller) {
            var main = function(){
                WT1.event("analysis-pml-charts-open");
                TopNav.setLocation(TopNav.TOP_ANALYSES, TopNav.LEFT_ANALYSES, "PREDICTION-ANALYSIS-MODEL", "charts");
                $controller("_PMLModelBaseController",{$scope:$scope});

                ChartNavigationService.bindCurrentChartWithUrl();

                $scope.$on('$destroy', () => {
                    ChartNavigationService.unbindCurrentChartWithUrl()
                });

                $scope.$watchGroup(
                    ChartNavigationService.getCurrentChartWatchers($scope),
                    function () {
                        ChartNavigationService.updateCurrentChart($scope.charts, $scope.currentChart.index)
                    }
                );

                DataikuAPI.analysis.mlcommon.getCurrentSettings($stateParams.projectKey, $stateParams.analysisId, $stateParams.mlTaskId).success(function(data){
                    $scope.mlTaskDesign = data;
                    $scope.shaker = data.predictionDisplayScript;
                    $scope.charts = data.predictionDisplayCharts;
                    $scope.currentChart.index = ChartNavigationService.getChartIndexFromId($scope.charts.map(chart => chart.def), $stateParams.chartId);

                    $scope.onSettingsLoaded();
                }).error(setErrorInScope.bind($scope));

                DataikuAPI.ml.prediction.getModelDetails($stateParams.fullModelId).success(function(data){
                    if ($scope.mlTasksContext) $scope.mlTasksContext.model = data;
                    // Signal to Puppeteer that the content of the element has been loaded and is thus available for content extraction
                    $scope.puppeteerHook_elementContentLoaded = true;
                });
            }
            main();
        }
    };
});


})();
