(function() {
    'use strict';

    const app = angular.module('dataiku.analysis.mlcore');

    app.constant("SINGLE_TIMESERIES_IDENTIFIER", "__single_timeseries_identifier");

    app.controller("TimeseriesPMLTaskBaseController", function($scope, $controller, DataikuAPI, TimeseriesForecastingUtils, VisualMlCodeEnvCompatibility,
                                                               AlgorithmsSettingsService, FeatureFlagsService, CachedAPICalls) {
        $controller("_PMLTrainSessionController", { $scope: $scope });
        $scope.legacyAlgorithms = ["gluonts_mqcnn_timeseries", "gluonts_transformer_timeseries", "gluonts_deepar_timeseries", "gluonts_simple_feed_forward_timeseries"];

        $scope.deferredAfterInitMlTaskDesign.then(() => CachedAPICalls.pmlGuessPolicies)
            .then(pmlGuessPolicies => {
                $scope.guessPolicies = $scope.prepareGuessPolicies(pmlGuessPolicies["timeseries-forecasting"]);
            }).then(() => {
            $scope.setAlgorithms($scope.mlTaskDesign);
            $scope.setSelectedAlgorithm(AlgorithmsSettingsService.getDefaultAlgorithm(
                $scope.mlTaskDesign,
                $scope.base_algorithms[$scope.mlTaskDesign.backendType]
            ));
        })
            .catch(setErrorInScope.bind($scope));

        $scope.hasExternalFeatures = per_feature => !!$scope.mlTaskFeatures(per_feature, ['INPUT']).length;

        $scope.listAlgosWithoutExternalFeatures = function(algos) {
            if (!$scope.mlTaskDesign || !$scope.hasExternalFeatures($scope.mlTaskDesign.preprocessing.per_feature)) return [];

            return algos.filter(algo => $scope.mlTaskDesign.modeling[algo.algKey]
                && $scope.mlTaskDesign.modeling[algo.algKey].enabled
                && TimeseriesForecastingUtils.ALGOS_WITHOUT_EXTERNAL_FEATURES.keys.includes(algo.algKey));
        };

        $scope.listAlgosSlowOnMultipleTimeseries = function(algos) {
            if (!$scope.mlTaskDesign || !$scope.mlTaskDesign.timeseriesIdentifiers.length) return [];

            return algos.filter(algo => $scope.mlTaskDesign.modeling[algo.algKey]
                && $scope.mlTaskDesign.modeling[algo.algKey].enabled
                && TimeseriesForecastingUtils.ALGOS_SLOW_ON_MULTIPLE_TIMESERIES.includes(algo.algKey));
        };

        $scope.listAlgosIncompatibleWithMs = function(algos) {
            if ($scope.mlTaskDesign?.timestepParams?.timeunit !== "MILLISECOND") return [];
            return algos.filter(algo => $scope.mlTaskDesign.modeling[algo.algKey]?.enabled
                && TimeseriesForecastingUtils.ALGOS_INCOMPATIBLE_WITH_MS.includes(algo.algKey));
        };

        $scope.displayProphetCodeEnvWarning = function(algorithm) {
            if (!$scope.mlTaskDesign
                || !$scope.mlTaskDesign.modeling[algorithm.algKey].enabled
                || algorithm.algKey !== "prophet_timeseries") {
                return false;
            }

            const envCompat = VisualMlCodeEnvCompatibility.getCodeEnvCompat($scope.mlTaskDesign.envSelection, $scope.codeEnvsCompat);
            const isEnvProphetCompatible = envCompat && envCompat.prophet && envCompat.prophet.compatible;
            return !isEnvProphetCompatible;
        };

        $scope.displayGluontsCodeEnvWarning = function(algorithm) {
            const torch_algs = ["trivial_identity_timeseries", "seasonal_naive_timeseries", "gluonts_npts_timeseries", "gluonts_simple_feed_forward_timeseries", "gluonts_deepar_timeseries", "gluonts_transformer_timeseries", "gluonts_mqcnn_timeseries"];
            if (!$scope.isAlgoEnabledAndInList(algorithm.algKey, torch_algs)) {
                return false;
            }

            const envCompat = VisualMlCodeEnvCompatibility.getCodeEnvCompat($scope.mlTaskDesign.envSelection, $scope.codeEnvsCompat);
            const isEnvGluontsTorchCompatible = envCompat && envCompat.gluonts && envCompat.gluonts.compatible;
            return !isEnvGluontsTorchCompatible;
        };

        $scope.displayPmdarimaCodeEnvWarning = function(algorithm) {
            if (!$scope.mlTaskDesign
                || !$scope.mlTaskDesign.modeling[algorithm.algKey].enabled
                || algorithm.algKey !== "autoarima_timeseries") {
                return false;
            }

            const envCompat = VisualMlCodeEnvCompatibility.getCodeEnvCompat($scope.mlTaskDesign.envSelection, $scope.codeEnvsCompat);
            const isEnvArimaCompatible = envCompat && envCompat.pmdarima && envCompat.pmdarima.compatible;
            return !isEnvArimaCompatible;
        };

        $scope.displayStatsmodelCodeEnvWarning = function(algorithm) {

            const statsmodelsAlgs = ["arima_timeseries", "ets_timeseries", "seasonal_loess_timeseries"];
            if (!$scope.isAlgoEnabledAndInList(algorithm.algKey, statsmodelsAlgs)) {
                return false;
            }

            const envCompat = VisualMlCodeEnvCompatibility.getCodeEnvCompat($scope.mlTaskDesign.envSelection, $scope.codeEnvsCompat);
            const isEnvStatsmodelCompatible = envCompat && envCompat.statsmodel && envCompat.statsmodel.compatible;
            return !isEnvStatsmodelCompatible;
        };




        $scope.displayMXNetAlgLegacyWarning = function(algorithm) {
            return $scope.isAlgoEnabledAndInList(algorithm.algKey, $scope.legacyAlgorithms);
        };

        $scope.isAlgoEnabledAndInList = function(algKey, algList) {
            return $scope.mlTaskDesign
                && $scope.mlTaskDesign.modeling[algKey].enabled
                && algList.includes(algKey);
        }

        $scope.displayMXNetAlgCodeEnvWarning = function(algorithm) {
            if (!$scope.isAlgoEnabledAndInList(algorithm.algKey, $scope.legacyAlgorithms)) {
                return false;
            }

            const envCompat = VisualMlCodeEnvCompatibility.getCodeEnvCompat($scope.mlTaskDesign.envSelection, $scope.codeEnvsCompat);
            const isEnvMXNetCompatible = envCompat && envCompat.mxnetTimeseries && envCompat.mxnetTimeseries.compatible;
            return !isEnvMXNetCompatible;
        };

        $scope.displayGluontsTorchAlgCodeEnvWarning = function(algorithm) {
            const torch_algs = ["gluonts_torch_deepar_timeseries", "gluonts_torch_simple_feed_forward_timeseries"];
            if (!$scope.isAlgoEnabledAndInList(algorithm.algKey, torch_algs)) {
                return false;
            }

            const envCompat = VisualMlCodeEnvCompatibility.getCodeEnvCompat($scope.mlTaskDesign.envSelection, $scope.codeEnvsCompat);
            const isEnvGluontsTorchCompatible = envCompat && envCompat.torchTimeseries && envCompat.torchTimeseries.compatible;
            return !isEnvGluontsTorchCompatible;
        };
    });

    app.controller("TimeseriesPMLTaskDesignController", function($scope, $controller, CreateModalFromTemplate, $timeout, TimeseriesForecastingUtils, TimeseriesForecastingCustomTrainTestFoldsUtils) {
        $controller("_TabularPMLTaskDesignController", { $scope: $scope });

        $scope.prettyTimeUnit = TimeseriesForecastingUtils.prettyTimeUnit;
        $scope.prettyTimeSteps = TimeseriesForecastingUtils.prettyTimeSteps;
        $scope.plurifiedTimeUnits = TimeseriesForecastingUtils.plurifiedTimeUnits;
        $scope.timeseriesImputeMethods = function(featureType, interpolation, extrapolation) {
            return TimeseriesForecastingUtils.TIMESERIES_IMPUTE_METHODS.filter(imputeMethod =>
                (featureType !== undefined ? imputeMethod.featureTypes.includes(featureType) : true) &&
                (interpolation !== undefined ? imputeMethod.interpolation === interpolation : true) &&
                (extrapolation !== undefined ? imputeMethod.extrapolation === extrapolation : true)
            )
        }
        $scope.timeseriesImputeMethodsDescriptions = function(featureType, interpolation, extrapolation) {
            return $scope.timeseriesImputeMethods(featureType, interpolation, extrapolation).map(imputeMethod => imputeMethod.description);
        }
        $scope.duplicateTimestampsHandlingMethods = TimeseriesForecastingUtils.DUPLICATE_TIMESTAMPS_HANDLING_METHODS;

        $scope.getWeekDayName = TimeseriesForecastingUtils.getWeekDayName;
        $scope.getDayNumber = TimeseriesForecastingUtils.getDayNumber;
        $scope.daysInMonth = Array.from({ length: 31 }, (_, index) => index + 1);
        $scope.getMonthName = TimeseriesForecastingUtils.getMonthName;
        $scope.getQuarterName = TimeseriesForecastingUtils.getQuarterName;
        $scope.getHalfYearName = TimeseriesForecastingUtils.getHalfYearName;

        $scope.togglePartitioningSection = function() {
            $scope.partitioningSectionOpened = !$scope.partitioningSectionOpened;
        }

        $scope.$watch('mlTaskDesign', function(nv) {
            if (!nv) return;
            $scope.uiState.timeVariable = nv.timeVariable;
            $scope.uiState.timeseriesIdentifiers = [].concat(nv.timeseriesIdentifiers);
            $scope.uiState.multipleTimeSeries = nv.timeseriesIdentifiers[0] !== undefined;
            $scope.uiState.predictionLength = nv.predictionLength;
            $scope.uiState.timestepParams = Object.assign({}, nv.timestepParams);
            $scope.setDefaultMonthlyAlignment(nv.timestepParams, false);
            $scope.setDefaultUnitAlignment(nv.timestepParams, false);

            $scope.partitioningSectionOpened = $scope.mlTaskDesign.partitionedModel && $scope.mlTaskDesign.partitionedModel.enabled;

            $scope.uiState.numberOfHorizonsInTest = Math.floor($scope.mlTaskDesign.evaluationParams.testSize / $scope.mlTaskDesign.predictionLength);

            $scope.uiState.hyperparamSearchStrategies = [["GRID", "Grid search"],
                ["RANDOM", "Random search"],
                ["BAYESIAN", "Bayesian search"]];

            $scope.uiState.crossValidationModes = [["TIME_SERIES_SINGLE_SPLIT", "Simple split"],
                ["TIME_SERIES_KFOLD", "K-fold cross-validation"]];

            $scope.uiState.preprocessingPerFeature = $scope.mlTaskFeatures(nv.preprocessing.per_feature, ['INPUT', 'REJECT']);
            $scope.featuresWithDateMeaning = Object.fromEntries(
                Object.entries($scope.mlTaskDesign.preprocessing.per_feature).filter(function(entry) {
                    const feature = entry[1];
                    return ['Date', 'DateOnly', 'DatetimeNoTz'].indexOf(feature.state.recordedMeaning) >= 0;
                })
            );
        });

        $scope.onChangeTimeVariable = function() {
            if (!$scope.uiState.timeVariable) return;
            if ($scope.dirtySettings()) {
                $scope.saveSettings();
            }
            CreateModalFromTemplate("/templates/analysis/prediction/change-core-params-modal.html", $scope, "PMLChangeBasicParamsModal", function(newScope) {
                newScope.paramKey = "timeVariable";
            });
        };

        $scope.changeTimeSeriesType = function(multiple) {
            $scope.uiState.multipleTimeSeries = multiple;
            if (multiple || !$scope.mlTaskDesign.timeseriesIdentifiers.length) return;

            // If 'Single time series' is selected, remove time series identifiers
            $scope.uiState.timeseriesIdentifiers = [];
            onChangeTimeseriesIdentifiers();
        };

        function haveTimeseriesIdentifiersChanged() {
            return $scope.mlTaskDesign.timeseriesIdentifiers.length !== $scope.uiState.timeseriesIdentifiers.length
                || $scope.mlTaskDesign.timeseriesIdentifiers.some(function(elem, idx) {
                    return elem !== $scope.uiState.timeseriesIdentifiers[idx];
                });
        }

        function onChangeTimeseriesIdentifiers() {
            if ($scope.dirtySettings()) {
                $scope.saveSettings();
            }
            CreateModalFromTemplate("/templates/analysis/prediction/change-core-params-modal.html", $scope, "PMLChangeBasicParamsModal", function(newScope) {
                newScope.onCloseCallback = function() {
                    $scope.uiState.preprocessingPerFeature = $scope.mlTaskFeatures($scope.mlTaskDesign.preprocessing.per_feature, ['INPUT', 'REJECT']);
                    $scope.uiState.multipleTimeSeries = !!$scope.uiState.timeseriesIdentifiers.length;

                };
                newScope.getUIStateParam = param => [].concat(param);

                newScope.paramKey = "timeseriesIdentifiers";
            });
        };

        $scope.newQuantileIsValid = function(newQuantile) {
            const inRange = newQuantile > 0 && newQuantile < 1;
            if (!inRange) return false;

            return parseFloat(newQuantile.toFixed(4)) === newQuantile;
        };

        $scope.$on("$stateChangeStart", function(e) {
            // Prevent to change state if the identifiers have been modified, so that
            // the user first interact with the keep settings/redetect modal
            if (haveTimeseriesIdentifiersChanged() || hasNumberOfTimeunitsChanged() || hasPredictionLengthChanged()) {
                e.preventDefault();
            }
        });

        $scope.onChangeTimeseriesIdentifiers = function() {
            // the timeout prevents the modal to be closed if the dropdown was closed via 'esc'
            $timeout(function() {
                if (haveTimeseriesIdentifiersChanged()) {
                    onChangeTimeseriesIdentifiers();
                }
            }, 100);
        };

        function onChangeTimestepParams() {
            if ($scope.dirtySettings()) {
                $scope.saveSettings();
            }
            CreateModalFromTemplate("/templates/analysis/prediction/change-core-params-modal.html", $scope, "PMLChangeBasicParamsModal", function(newScope) {
                newScope.getUIStateParam = param => Object.assign({}, param);
                newScope.paramKey = "timestepParams";
            });
        }

        function hasPredictionLengthChanged() {
            if (!$scope.uiState.predictionLength) return;
            return $scope.mlTaskDesign.predictionLength !== $scope.uiState.predictionLength;
        }

        $scope.onChangePredictionLength = function() {
            if (!hasPredictionLengthChanged()) return;

            if ($scope.dirtySettings()) {
                $scope.saveSettings();
            }

            const numberOfHorizonsInTest = $scope.uiState.numberOfHorizonsInTest;
            CreateModalFromTemplate("/templates/analysis/prediction/change-core-params-modal.html", $scope, "PMLChangeBasicParamsModal", function(newScope) {
                newScope.onCloseCallback = function() {
                    $scope.uiState.numberOfHorizonsInTest = numberOfHorizonsInTest;
                    $scope.updateTimeseriesTestSize();
                };
                newScope.paramKey = "predictionLength";
            });
        };

        $scope.setDefaultMonthlyAlignment = function(params, force) {
            if (!params.monthlyAlignment || force) {
                params.monthlyAlignment = 31;
            }
        }

        $scope.setDefaultUnitAlignment = function(params, force) {
            if (!params.unitAlignment || force) {
                if (params.timeunit === "QUARTER") {
                    params.unitAlignment = 3;
                } else if (params.timeunit === "HALF_YEAR") {
                    params.unitAlignment = 6;
                } else if (params.timeunit === "YEAR") {
                    params.unitAlignment = 12;
                }
            }
        }

        $scope.onChangeTimestepUnit = function() {
            $scope.setDefaultMonthlyAlignment($scope.uiState.timestepParams, true);
            $scope.setDefaultUnitAlignment($scope.uiState.timestepParams, true);
            onChangeTimestepParams();
        }

        $scope.onChangeNumberOfTimeunits = function() {
            if (!hasNumberOfTimeunitsChanged()) return;
            onChangeTimestepParams();
        };

        function hasNumberOfTimeunitsChanged() {
            if (!$scope.uiState.timestepParams.numberOfTimeunits) return;
            return $scope.uiState.timestepParams.numberOfTimeunits !== $scope.mlTaskDesign.timestepParams.numberOfTimeunits;
        };

        $scope.hasRole = function(feature, roles) {
            const role = $scope.mlTaskDesign.preprocessing.per_feature[feature].role;
            return roles.includes(role);
        };

        $scope.updateTimeseriesTestSize = function() {
            $scope.mlTaskDesign.evaluationParams.testSize = $scope.uiState.predictionLength * $scope.uiState.numberOfHorizonsInTest;
        };
        
        $scope.initTimeseriesFixedFoldUIState = function() {
            if (!$scope.mlTaskDesign.customTrainTestIntervals.length) {
                $scope.mlTaskDesign.customTrainTestIntervals.push({
                    train: [new Date().toISOString(), new Date().toISOString()],
                    test: [new Date().toISOString(), new Date().toISOString()]
                });
            }
            const interval = $scope.mlTaskDesign.customTrainTestIntervals[0];
            $scope.uiState.fixedFoldInterval = {
                train: [new Date(interval['train'][0]), new Date(interval['train'][1])],
                test: [new Date(interval['test'][0]), new Date(interval['test'][1])]
            }
        };
        
        $scope.validateFixedFolds = function() {
            return TimeseriesForecastingCustomTrainTestFoldsUtils.validateCustomTrainTestFold(
                $scope.mlTaskDesign.timestepParams, 
                $scope.mlTaskDesign.customTrainTestIntervals[0],
                $scope.mlTaskDesign.predictionLength
            );
        };

        $scope.propagateFixedIntervalChangeFn = function(intervalIdx, fold, boundary) {
            $timeout(function() {
                if (!$scope.mlTaskDesign.customTrainTestIntervals.length > intervalIdx) return;
                $scope.mlTaskDesign.customTrainTestIntervals[intervalIdx][fold][boundary] =
                    $scope.uiState.fixedFoldInterval[fold][boundary] ? moment($scope.uiState.fixedFoldInterval[fold][boundary]).format("YYYY-MM-DD HH:mm:ss.SSS") : undefined;
            });
        };

        $scope.resamplingMethodToGraph = function(method, prefix) {
            switch (method) {
                case "NEAREST":
                    return prefix + "-nearest";
                case "PREVIOUS":
                    return prefix + "-previous";
                case "NEXT":
                    return prefix + "-next";
                case "STAIRCASE":
                    return prefix + "-staircase";
                case "LINEAR":
                    if (prefix === "left") return "left-next";
                    return prefix + "-linear";
                case "QUADRATIC":
                    return prefix + "-quadratic";
                case "CUBIC":
                    return prefix + "-cubic";
                case "CONSTANT":
                    return prefix + "-constant";
                case "PREVIOUS_NEXT":
                    if (prefix === "left") return "left-next";
                    return "right-previous";
                case "NO_EXTRAPOLATION":
                    return prefix + "-dont";
            }
        };

        function isAutoArimaSearchExplicit() {
            if ($scope.mlTaskDesign.modeling.gridSearchParams.strategy == "GRID") {
                return $scope.mlTaskDesign.modeling.autoarima_timeseries.m.gridMode == 'EXPLICIT';
            } else {
                return $scope.mlTaskDesign.modeling.autoarima_timeseries.m.randomMode == 'EXPLICIT';
            }
        };

        $scope.isAutoArimaSeasonal = function() {
            if (isAutoArimaSearchExplicit()) {
                return Math.max(...$scope.mlTaskDesign.modeling.autoarima_timeseries.m.values) > 1;
            } else {
                return $scope.mlTaskDesign.modeling.autoarima_timeseries.m.range.max > 1;
            }
        };

        $scope.isAutoArimaSeasonLengthTooHigh = function() {
            if (isAutoArimaSearchExplicit()) {
                return Math.max(...$scope.mlTaskDesign.modeling.autoarima_timeseries.m.values) > 12;
            } else {
                return $scope.mlTaskDesign.modeling.autoarima_timeseries.m.range.max > 12;
            }
        };

        $scope.isTrendValid = function() {
            // this function should be kept in sync with ArimaMeta's isTrendValid method
            let sum = $scope.mlTaskDesign.modeling.arima_timeseries.d +
                $scope.mlTaskDesign.modeling.arima_timeseries.D;
            switch($scope.mlTaskDesign.modeling.arima_timeseries.trend) {
                case "n":
                    return true;
                case "c":
                case "ct":
                    return sum <= 0;
                case "t":
                    return sum <= 1;
                default:
                    return false;
            }
        };

        $scope.getAutoArimaMaxOrderMinValue = function() {
            let minValue = $scope.mlTaskDesign.modeling.autoarima_timeseries.start_p + $scope.mlTaskDesign.modeling.autoarima_timeseries.start_q + 1;
            if ($scope.isAutoArimaSeasonal()) {
                minValue += $scope.mlTaskDesign.modeling.autoarima_timeseries.start_P + $scope.mlTaskDesign.modeling.autoarima_timeseries.start_Q;
            }
            return minValue;
        }

        $scope.isOdd = number => number % 2 === 1;

        $scope.isPositive = number => number > 0;
    });

    app.directive("forecastExplanation", function(TimeseriesForecastingUtils) {
        return {
            scope: {
                predictionLength: '<',
                gapSize: '<',
                numberOfHorizonsInTest: '<',
                timeUnit: '<',
                numberOfTimeunits: '<',
                loadedStateField: '<'
            },
            restrict: 'A',
            templateUrl: '/templates/analysis/prediction/timeseries/forecasting-schema.html',
            link: function($scope, element) {
                const MAX_NB_OF_HORIZONS_IN_SCHEMA = 5;
                $scope.prettyTimeUnit = TimeseriesForecastingUtils.prettyTimeUnit;

                // Indicate to puppeteer that this content is ready for extraction
                const puppeteerSelectorName = $scope.loadedStateField;
                element.attr(puppeteerSelectorName, true)
                $scope[puppeteerSelectorName] = true;

                $scope.forecastingSchemaHasEllipsis = function() {
                    if (!$scope.predictionLength) return;
                    return $scope.numberOfHorizonsInTest > MAX_NB_OF_HORIZONS_IN_SCHEMA;
                };

                $scope.isEllipsedFromForecastingSchema = function(index) {
                    return $scope.forecastingSchemaHasEllipsis() && index == Math.floor(MAX_NB_OF_HORIZONS_IN_SCHEMA / 2);
                }

                $scope.maxNumberOfHorizonsInSchema = function() {
                    if (!$scope.numberOfHorizonsInTest) return;
                    return Math.min($scope.numberOfHorizonsInTest, MAX_NB_OF_HORIZONS_IN_SCHEMA);
                };

                $scope.getHorizonTickLabel = function(index) {
                    if ($scope.forecastingSchemaHasEllipsis() && index >= Math.floor(MAX_NB_OF_HORIZONS_IN_SCHEMA / 2)) {
                        index = $scope.numberOfHorizonsInTest - (MAX_NB_OF_HORIZONS_IN_SCHEMA - index);
                    }
                    ;
                    const nbHorizons = index + 1;
                    const nbTimeUnitsInHorizon = $scope.predictionLength * $scope.numberOfTimeunits;
                    return nbHorizons * nbTimeUnitsInHorizon;
                };
            }
        };
    })

    app.controller("TimeseriesPMLTaskPreTrainModal", function($scope, $controller, $stateParams, DataikuAPI, Logger, WT1) {
        $controller("PMLTaskPreTrainModal", { $scope });
        $controller("_TabularPMLTaskPreTrainBase", { $scope });

        function redactSensitiveInformation(eventContent) {
            // For now, there are no:
            // - custom metrics
            // - custom code algorithms (keras, or custom python or mllib)
            // - custom feature selection code
            // - feature generation manual interactions

            const redacted = dkuDeepCopy(eventContent, $scope.SettingsService.noDollarKey); // don't want to delete actual values in scope

            if (redacted.timeseriesSamplingParams) {
                delete redacted.timeseriesSamplingParams.numericalInterpolateConstantValue;
                delete redacted.timeseriesSamplingParams.numericalExtrapolateConstantValue;
                delete redacted.timeseriesSamplingParams.categoricalConstantValue;
                redacted.timeseriesSamplingParams = JSON.stringify(redacted.timeseriesSamplingParams)
            }

            return redacted;
        }

        $scope.algosWithoutExternalFeatures = $scope.listAlgosWithoutExternalFeatures($scope.base_algorithms[$scope.mlTaskDesign.backendType]);
        $scope.algosSlowOnMultipleTimeseries = $scope.listAlgosSlowOnMultipleTimeseries($scope.base_algorithms[$scope.mlTaskDesign.backendType]);

        // The settings can be updated from inside the modal (if you activate the GPU training),
        // so we save first before training
        $scope.train = () => $scope.saveSettings().then($scope._doTrainThenResolveModal);

        $scope._doTrain = function() {
            try {
                const algorithms = {};
                $.each($scope.mlTaskDesign.modeling, function(alg, params) {
                    if (params.enabled) {
                        algorithms[alg] = params;
                    }
                });

                let wt1Content = redactSensitiveInformation({
                    backendType: $scope.mlTaskDesign.backendType,
                    taskType: $scope.mlTaskDesign.taskType,
                    predictionType: $scope.mlTaskDesign.predictionType,
                    guessPolicy: $scope.mlTaskDesign.guessPolicy,
                    algorithms: JSON.stringify(algorithms),
                    predictionLength: $scope.mlTaskDesign.predictionLength,
                    gapSize: $scope.mlTaskDesign.evaluationParams.gapSize,
                    testSize: $scope.mlTaskDesign.evaluationParams.testSize,
                    nbTimeseriesIdentifiers: $scope.mlTaskDesign.timeseriesIdentifiers.length,
                    nbExternalFeatures: Object.values($scope.mlTaskDesign.preprocessing.per_feature).filter((feature) => feature.role === "INPUT").length,
                    quantiles: JSON.stringify($scope.mlTaskDesign.quantilesToForecast),
                    timestepParams: JSON.stringify($scope.mlTaskDesign.timestepParams),
                    timeseriesSamplingParams: $scope.mlTaskDesign.preprocessing.timeseriesSampling,
                    isPartitioned: $scope.mlTaskDesign.partitionedModel && $scope.mlTaskDesign.partitionedModel.enabled,
                    nFoldsEvaluation: $scope.mlTaskDesign.splitParams.kfold ? $scope.mlTaskDesign.splitParams.nFolds : null,
                    hasSessionName: !!$scope.uiState.userSessionName,
                    hasSessionDescription: !!$scope.uiState.userSessionDescription,
                    gridSearchParams: JSON.stringify($scope.mlTaskDesign.modeling.gridSearchParams),
                    evaluationMetric: $scope.mlTaskDesign.modeling.metrics.evaluationMetric,
                    runsOnKubernetes: $scope.hasSelectedK8sContainer(),
                    usesCustomTrainTestSplit: $scope.mlTaskDesign.customTrainTestSplit,
                    customTrainTestIntervals: $scope.mlTaskDesign.customTrainTestSplit ? JSON.stringify($scope.mlTaskDesign.customTrainTestIntervals) : null,
                });

                WT1.event("prediction-train", wt1Content);
            } catch (e) {
                Logger.error('Failed to report mltask info', e);
            }
            return DataikuAPI.analysis.pml.trainStart($stateParams.projectKey, $stateParams.analysisId, $stateParams.mlTaskId,
                $scope.uiState.userSessionName, $scope.uiState.userSessionDescription, $scope.uiState.forceRefresh, true).error(setErrorInScope.bind($scope));
        };

        $scope.displayMessages = function() {
            return $scope.preTrainStatus.messages.length || $scope.algosWithoutExternalFeatures.length || $scope.algosSlowOnMultipleTimeseries.length;
        };
    });

    app.controller("TimeseriesPMLTaskResultController", function($scope, $controller, PMLSettings, PMLFilteringService) {
        $scope.algorithmCategories = PMLSettings.algorithmCategories("TIMESERIES_FORECAST");
        $scope.metricMap = PMLFilteringService.metricMap;
        $controller("_TabularPMLTaskResultController", { $scope });

        $scope.uiState.tsSessionDetailView = 'METRICS';

        let metrics;
        let highestMetricValues;
        let default_displayed_metrics;
        $scope.$watch("selection.sessionModels", function(nv, ov) {
            if (!nv || nv.length == 0) return;

            metrics = $scope.timeseriesEvaluationMetrics.map(metric => $scope.metricMap[metric[0]]);
            default_displayed_metrics = ["RMSE", "SMAPE", "MAPE"];
            if (!default_displayed_metrics.includes(nv[0].evaluationMetric)) {
                default_displayed_metrics.push(nv[0].evaluationMetric)
            }

            $scope.displayableMetrics = $scope.timeseriesEvaluationMetrics.map(function(metric) {
                return {
                    name: metric[0],
                    $displayed: default_displayed_metrics.includes(metric[0])
                }
            });

            highestMetricValues = metrics.reduce(function(res, metric) {
                res[metric] = Math.max(...$scope.selection.sessionModels.map(model => model[metric] || 0));
                return res;
            }, {});

            // add all unique custom metrics and get their max values
            let customMetrics = new Set();
            nv.filter(model => model.customMetricsResults).forEach(function(model) {
                model.customMetricsResults.filter(customMetricResult => customMetricResult.didSucceed).forEach(function(customMetricResult) {
                    customMetrics.add(customMetricResult.metric.name)
                    if (!(customMetricResult.metric.name in highestMetricValues) || customMetricResult.value > highestMetricValues[customMetricResult.metric.name]) {
                        highestMetricValues[customMetricResult.metric.name] = customMetricResult.value;
                    }
                });
            });

            customMetrics.forEach(function(customMetric) {
                $scope.displayableMetrics.push({
                    name: customMetric,
                    $displayed: default_displayed_metrics.includes(customMetric)
                });
                metrics.push(customMetric)
            })

            const newSessionId = (nv[0] || {}).sessionId;
            const oldSessionId = (ov && ov[0] || {}).sessionId;

            // Do not change the tab if we are still on the same session
            if (newSessionId === oldSessionId) return;

            if ($scope.isSessionRunning(newSessionId) && $scope.anySessionModelNeedsHyperparameterSearch()) {
                // Switch to HP search chart when starting a training or switching to the session
                // currently being trained
                $scope.uiState.tsSessionDetailView = 'HP_SEARCH';
            } else {
                // Otherwise always show metrics when switching between sessions
                $scope.uiState.tsSessionDetailView = 'METRICS';
            }
        });


        let sessionIdThatWasRunning = null;
        $scope.$on("mlTaskStatusRefresh", function() {
            const currentSessionId = (($scope.selection && $scope.selection.sessionModels && $scope.selection.sessionModels[0]) || {}).sessionId;
            if (!currentSessionId) return;

            if (!sessionIdThatWasRunning && $scope.mlTaskStatus.training && $scope.isSessionRunning(currentSessionId)) {
                sessionIdThatWasRunning = currentSessionId;
            }

            if (currentSessionId === sessionIdThatWasRunning && !$scope.mlTaskStatus.training) {
                // Only switch back to METRICS at the end of the training, if we are currently
                // viewing the session being trained
                $scope.uiState.tsSessionDetailView = 'METRICS';
                sessionIdThatWasRunning = null;
            }
        });

        $scope.anySessionModelHasMetrics = function() {
            return ($scope.selection.sessionModels || []).some(function(model) {
                return metrics.some(metric => model[metric]);
            })
        };

        $scope.getModelMetric = function(model, metricName) {
            if (metricName in $scope.metricMap) {
                return model[$scope.metricMap[metricName]];
            } else {
                const customMetric = model.customMetricsResults.find(customMetric => metricName == customMetric.metric.name);
                if (customMetric) return customMetric.value;
            }
        }

        $scope.getModelMetricBarRatioHeight = function(model, metricName) {
            const metric = $scope.metricMap[metricName] || metricName;
            const metricValueForModel = $scope.getModelMetric(model, metricName);
            if (!metricValueForModel) return 0;
            const highestMetricValue = highestMetricValues && highestMetricValues[metric];
            if (!highestMetricValue) return 0;
            return metricValueForModel / highestMetricValue;
        };

        $scope.selectMetric = function(metricIdx) {
            $scope.displayableMetrics[metricIdx].$displayed = !$scope.displayableMetrics[metricIdx].$displayed;
        };

        $scope.nbDisplayedMetrics = function() {
            return $scope.displayableMetrics.filter(metric => metric.$displayed).length;
        };

        $scope.$watch('uiState.currentMetric', function(nv) {
            if (!nv || !$scope.displayableMetrics) return;
            const currentMetric = $scope.displayableMetrics.find(metric => metric.name === $scope.uiState.currentMetric);
            if (currentMetric) currentMetric.$displayed = true;
        });
    });

    app.controller('ExportTimeseriesDataController', function($scope, ExportUtils) {
        $scope.$on('timeseriesTableEvent', function(event, timeseriesTable) {
            $scope.tableHeaders = timeseriesTable.headers
            $scope.allTableRows = timeseriesTable.rows
        });

        function getSingleTsData(withStats) {
            return $scope.allTableRows.map(row => {
                let newRow = [];
                row.forEach(x => {
                    // Export the displayValue when there is no rawValue
                    newRow.push(x.rawValue || x.displayValue);

                    // add statistics after each coefficient value
                    if (withStats) {
                        if (x.rawStderr !== undefined) {
                            newRow.push(x.rawStderr);
                        }
                        if (x.rawPvalue !== undefined) {
                            newRow.push(x.rawPvalue);
                        }
                        if (x.rawTvalue !== undefined) {
                            newRow.push(x.rawTvalue);
                        }
                    }
                });
                return newRow;
            });
        }

        function getMultipleTsData() {
            return $scope.allTableRows.map(row => {
                let newRow = [];
                for (const header of $scope.tableHeaders) {
                    if (header.rawField) { // handle fields with different display value to actual value
                        newRow.push(row[header.rawField].rawValue);
                    } else {
                        newRow.push(row[header.field]);
                    }
                }
                return newRow;
            });
        }

        $scope.exportTimeseriesData = function(notAgGridRow, label, valueType, exportPerFold, withStats) {
            let nbTimeseriesIdentifiers = $scope.modelData.coreParams.timeseriesIdentifiers.length;
            const columns = [];
            valueType = valueType || "double";  // Use double by default for metrics and coefficients
            exportPerFold = exportPerFold || false;
            withStats = withStats || false;

            // if export per fold, we have 3 extra columns that we want to be considered string type
            nbTimeseriesIdentifiers = exportPerFold ? nbTimeseriesIdentifiers + 3 : nbTimeseriesIdentifiers;
            $scope.tableHeaders.forEach(function(column, i) {
                if (notAgGridRow) {
                    columns.push({ name: column.rawName || column.displayName, type: i < nbTimeseriesIdentifiers ? "string" : valueType });
                } else {
                    columns.push({ name: column.headerName || column.field, type: i < nbTimeseriesIdentifiers ? "string" : valueType });
                }
            });

            let data;
            if (notAgGridRow) {
                data = getSingleTsData(withStats);
            } else {
                data = getMultipleTsData();
            }

            ExportUtils.exportUIData($scope, {
                name: label + " of model " + $scope.modelData.userMeta.name,
                columns: columns,
                data: data
            }, `Export model ${label}`);
        }

        $scope.noModelCoefficientExplanation = function() {
            // with a null order arima, there is no coefficient to show
            let isArimaAndOrderIsZero = false;
            isArimaAndOrderIsZero = $scope.modelData.actualParams.resolved.algorithm === "ARIMA" &&
                ["p", "q", "P", "Q"].every(order => {
                    return $scope.modelData.actualParams.resolved.arima_timeseries_params[order] === 0;
                });
            if (isArimaAndOrderIsZero) {
                return "No coefficients for the selected model since p, q, P and Q are equal to 0"
            }
        }
    });

    app.controller('InformationCriteriaController', function($scope, DataikuAPI, FutureProgressModal) {
        $scope.retrieveInformationCriteria = function(fullModelId) {
            DataikuAPI.ml.prediction.getInformationCriteria(fullModelId)
                .then(function(res) {
                    FutureProgressModal.show($scope, res.data, "Retrieve information criteria").then(function(informationCriteria) {
                        if (informationCriteria) {
                            // will be undefined if computation was aborted
                            $scope.modelData.iperf.informationCriteria = informationCriteria;
                        }
                    });
                }).catch(setErrorInScope.bind($scope));
        };
    });

    app.component("multipleTsInformationCriteria", {
        bindings: {
            informationCriteria: '<',
            timeseriesIdentifierColumns: '<',
            fullModelId: '<',
            insightId: '<',
        },
        template: `
            <table-manager
                class="h100 model-info-page__table-manager"
                get-rows-for-identifier="$ctrl.getRowsForIdentifier"
                get-headers="$ctrl.getHeaders"
                full-model-id="{{$ctrl.fullModelId}}"
                insight-id="$ctrl.insightId"
                unparsed-timeseries-identifiers="$ctrl.unparsedTimeseriesIdentifiers"
                timeseries-identifier-columns="$ctrl.timeseriesIdentifierColumns"
            ></table-manager>
        `,
        controller: function($scope, $stateParams, PerTimeseriesService, InformationCriteriaUtils) {
            const $ctrl = this;

            $ctrl.$onInit = () => {
                $ctrl.featureColumns = $ctrl.informationCriteria.map(ic => {
                    return {
                        field: ic.displayName,
                        rawField: ic.displayName, // used by us for export
                        valueFormatter: p => getDisplayValueFromInformationCriteria(p, ic)
                    };
                });
                $ctrl.unparsedTimeseriesIdentifiers = Object.keys($ctrl.informationCriteria[0].values); // if an identifier is lacking the first ic, it won't appear at all
            }

            $ctrl.getHeaders = function($scope) {
                return PerTimeseriesService.initTimeseriesIdentifierTableColumns($scope.timeseriesIdentifierColumns, $ctrl.featureColumns);
            };

            function getDisplayValueFromInformationCriteria(params, ic) {
                return params.data[ic.displayName].displayValue
            }

            $ctrl.getRowsForIdentifier = function(parsedTimeseriesIdentifier, unparsedTimeseriesIdentifier, uiState) {
                const row = {...parsedTimeseriesIdentifier};
                $ctrl.informationCriteria.forEach(function(criteria) {
                    const criteriaValue = unparsedTimeseriesIdentifier in criteria.values ? criteria.values[unparsedTimeseriesIdentifier] : "";

                    row[criteria["displayName"]] = {
                        displayValue: InformationCriteriaUtils.formatValue(criteriaValue),
                        rawValue: InformationCriteriaUtils.formatValue(criteriaValue, false)
                    };
                });
                return [row];
            };
        }
    });

    app.component("multipleTsAutoarimaOrders", {
        bindings: {
            postTrain: '<',
            timeseriesIdentifierColumns: '<',
            fullModelId: '<',
            insightId: '<',
        },
        template: `
            <table-manager
                class="h100 model-info-page__table-manager"
                get-rows-for-identifier="$ctrl.getRowsForIdentifier"
                get-headers="$ctrl.getHeaders"
                full-model-id="{{$ctrl.fullModelId}}"
                insight-id="$ctrl.insightId"
                unparsed-timeseries-identifiers="$ctrl.unparsedTimeseriesIdentifiers"
                timeseries-identifier-columns="$ctrl.timeseriesIdentifierColumns"
            ></table-manager>
        `,
        controller: function($scope, $stateParams, AutoArimaOrdersService, PerTimeseriesService) {
            const $ctrl = this;
            $ctrl.$onInit = () => {
                $ctrl.resolvedAlgoValues = $ctrl.postTrain.auto_arima_timeseries_params;
                $ctrl.autoArimaRelevantParams = AutoArimaOrdersService.ordersToDisplay($ctrl.resolvedAlgoValues);
                $ctrl.unparsedTimeseriesIdentifiers = Object.keys($ctrl.resolvedAlgoValues.p)
            };

            $ctrl.getHeaders = function($scope) {
                return PerTimeseriesService.initTimeseriesIdentifierTableColumns($scope.timeseriesIdentifierColumns, $ctrl.autoArimaRelevantParams);
            };

            $ctrl.getRowsForIdentifier = function(parsedTimeseriesIdentifier, unparsedTimeseriesIdentifier, uiState) {
                const row = {...parsedTimeseriesIdentifier};
                $ctrl.autoArimaRelevantParams.forEach(function(algoParam) {
                    row[algoParam.displayName] = String($ctrl.resolvedAlgoValues[algoParam.fieldName][unparsedTimeseriesIdentifier]);
                });
                return [row];
            };
        },
    });

    app.service('InformationCriteriaUtils', function() {
        this.formatValue = function(informationCriteriaValue, round=true) {
            if (["+∞", "-∞"].includes(informationCriteriaValue)) {
                return informationCriteriaValue
            }
            if (informationCriteriaValue.value !== "") {
                return round ? String(informationCriteriaValue.toFixed(4)) : String(informationCriteriaValue);
            }
            return informationCriteriaValue;
        };
    });

    app.component("singleTsInformationCriteria", {
        bindings: {
            informationCriteria: '<'
        },
        templateUrl: "/templates/ml/prediction-model/timeseries/single-ts-information-criteria.html",
        controller: function($scope, SINGLE_TIMESERIES_IDENTIFIER, InformationCriteriaUtils) {
            const $ctrl = this;
            $ctrl.$onInit = () => {
                $ctrl.headers = $ctrl.informationCriteria.map(criteria => {
                    return { displayName: criteria.displayName }
                });
                $ctrl.singleRow = $ctrl.informationCriteria.map(function(criteria) {
                    return {
                        displayValue: InformationCriteriaUtils.formatValue(criteria.values[SINGLE_TIMESERIES_IDENTIFIER]),
                        rawValue: InformationCriteriaUtils.formatValue(criteria.values[SINGLE_TIMESERIES_IDENTIFIER], false)
                    }
                });
                $scope.$emit('timeseriesTableEvent', { headers: $ctrl.headers, rows: [$ctrl.singleRow] });
            };
        },
    });

    app.component("singleTsAutoarimaOrders", {
        bindings: {
            postTrain: '<'
        },
        templateUrl: "/templates/ml/prediction-model/timeseries/single-ts-autoarima-orders.html",
        controller: function($scope, SINGLE_TIMESERIES_IDENTIFIER, AutoArimaOrdersService) {
            const $ctrl = this;
            $ctrl.$onInit = () => {
                $ctrl.SINGLE_TIMESERIES_IDENTIFIER = SINGLE_TIMESERIES_IDENTIFIER;
                $ctrl.resolvedAutoArimaValues = $ctrl.postTrain.auto_arima_timeseries_params;
                $ctrl.autoArimaRelevantParams = AutoArimaOrdersService.ordersToDisplay($ctrl.resolvedAutoArimaValues);
                const singleRow = $ctrl.autoArimaRelevantParams.map(function(order) {
                    return { displayValue: $ctrl.resolvedAutoArimaValues[order.displayName][SINGLE_TIMESERIES_IDENTIFIER] }
                });
                $scope.$emit('timeseriesTableEvent', { headers: $ctrl.autoArimaRelevantParams, rows: [singleRow] });
            };
        },
    });

    app.service("AutoArimaOrdersService", function() {
        return {
            ordersToDisplay
        };

        function ordersToDisplay(resolvedAutoArimaValues) {
            const autoArimaRelevantParams = [
                { field: 'p', displayName: 'p', fieldName: 'p', helper: 'Auto-regressive model order' },
                { field: 'q', displayName: 'q', fieldName: 'q', helper: 'Moving-average model order' },
            ];
            if (!resolvedAutoArimaValues.stationary) {
                autoArimaRelevantParams.push({ field: 'd', displayName: 'd', fieldName: 'd', helper: 'Differencing order' });
            }
            const isSeasonalAutoARIMA = resolvedAutoArimaValues.m > 1;
            if (isSeasonalAutoARIMA) {
                autoArimaRelevantParams.push({ field: 'P', displayName: 'P', fieldName: 'P', helper: 'Seasonal auto-regressive model order' });
                autoArimaRelevantParams.push({ field: 'Q', displayName: 'Q', fieldName: 'Q', helper: 'Seasonal moving-average model order' });
                if (!resolvedAutoArimaValues.stationary) {
                    autoArimaRelevantParams.push({ field: 'D', displayName: 'D', fieldName: 'D', helper: 'Seasonal differencing order' });
                }
            }
            return autoArimaRelevantParams;
        }
    });

    app.component("modelCoefficientsExplanation", {
        bindings: {
            algorithm: '<',
        },
        templateUrl: "/templates/ml/prediction-model/timeseries/model-coefficients-explanation.html",
        controller: function() {
            const $ctrl = this;

            const algorithmTitles = {
                "ETS": "ETS coefficients",
                "SEASONAL_LOESS": "Seasonal trend coefficients",
                "AUTO_ARIMA": "Seasonal ARIMA coefficients",
                "ARIMA": "Seasonal ARIMA coefficients",
                "PROPHET": "Prophet coefficients"
            }

            $ctrl.$onInit = () => {
                $ctrl.title = algorithmTitles[$ctrl.algorithm]
            };

            $ctrl.foldableToggle = function() {
                $ctrl.foldableOpen = !$ctrl.foldableOpen;
            };
        },
    });

    app.component("singleTsModelCoefficients", {
        bindings: {
            modelCoefficients: '<',
            algorithm: '<',
        },
        templateUrl: "/templates/ml/prediction-model/timeseries/single-ts-model-coefficients.html",
        controller: function($scope, SINGLE_TIMESERIES_IDENTIFIER, PerTimeseriesService) {
            const $ctrl = this;
            $ctrl.$onInit = () => {
                $ctrl.modelCoefficientsHeader = PerTimeseriesService.initModelCoefficientsHeader($ctrl.modelCoefficients, false, false, false);
                $ctrl.canDisplayStatisticalValues = ["AUTO_ARIMA", "ARIMA", "ETS", "SEASONAL_LOESS"].includes($ctrl.algorithm);
                $ctrl.hasStderrs = $ctrl.modelCoefficients.some(coeff => coeff.stderrs !== undefined && Object.keys(coeff.stderrs).length > 0);
                $ctrl.hasPvalues = $ctrl.modelCoefficients.some(coeff => coeff.pvalues !== undefined && Object.keys(coeff.pvalues).length > 0);
                $ctrl.hasTvalues = $ctrl.modelCoefficients.some(coeff => coeff.tvalues !== undefined && Object.keys(coeff.tvalues).length > 0);
                $ctrl.singleRow = $ctrl.modelCoefficients.map(coeff => {
                    const coefficientHasStderr = coeff.stderrs && coeff.stderrs[SINGLE_TIMESERIES_IDENTIFIER] !== undefined;
                    const coefficientHasPvalue = coeff.pvalues && coeff.pvalues[SINGLE_TIMESERIES_IDENTIFIER] !== undefined;
                    const coefficientHasTvalue = coeff.tvalues && coeff.tvalues[SINGLE_TIMESERIES_IDENTIFIER] !== undefined;
                    return {
                        displayValue: String(coeff.values[SINGLE_TIMESERIES_IDENTIFIER].toFixed(4)),
                        rawValue: coeff.values[SINGLE_TIMESERIES_IDENTIFIER],
                        displayStderr: coefficientHasStderr ? String(coeff.stderrs[SINGLE_TIMESERIES_IDENTIFIER].toFixed(4)) : "",
                        rawStderr: coefficientHasStderr ? coeff.stderrs[SINGLE_TIMESERIES_IDENTIFIER] : ($ctrl.hasStderrs ? "" : undefined),
                        displayPvalue: coefficientHasPvalue ? String(coeff.pvalues[SINGLE_TIMESERIES_IDENTIFIER].toFixed(4)) : "",
                        rawPvalue: coefficientHasPvalue ? coeff.pvalues[SINGLE_TIMESERIES_IDENTIFIER] : ($ctrl.hasPvalues ? "" : undefined),
                        displayTvalue: coefficientHasTvalue ? String(coeff.tvalues[SINGLE_TIMESERIES_IDENTIFIER].toFixed(4)) : "",
                        rawTvalue: coefficientHasTvalue ? coeff.tvalues[SINGLE_TIMESERIES_IDENTIFIER] : ($ctrl.hasTvalues ? "" : undefined),
                    }
                });
                $scope.$emit('timeseriesTableEvent', {
                    headers: PerTimeseriesService.initModelCoefficientsHeader($ctrl.modelCoefficients, $ctrl.hasStderrs, $ctrl.hasPvalues, $ctrl.hasTvalues),
                    rows: [$ctrl.singleRow]
                });
            };
        },
    });

    app.component("multipleTsModelCoefficients", {
        bindings: {
            modelCoefficients: '<',
            postTrain: '<',
            timeseriesIdentifierColumns: '<',
            fullModelId: '<',
            insightId: '<',
        },
        template: `
            <table-manager
                class="h100 model-info-page__table-manager"
                get-rows-for-identifier="$ctrl.getRowsForIdentifier"
                get-headers="$ctrl.getHeaders"
                full-model-id="{{$ctrl.fullModelId}}"
                insight-id="$ctrl.insightId"
                unparsed-timeseries-identifiers="$ctrl.unparsedTimeseriesIdentifiers"
                set-flags="$ctrl.setFlags"
                timeseries-identifier-columns="$ctrl.timeseriesIdentifierColumns"
                hide-columns-with-field="$ctrl.getColumnsToHide"
            ></table-manager>
        `,
        controller: function($scope, $stateParams, PerTimeseriesService) {
            const $ctrl = this;

            $ctrl.$onInit = function() {
                $ctrl.unparsedTimeseriesIdentifiers = PerTimeseriesService.retrieveUnparsedTimeseriesIdentifiers($ctrl.postTrain, $ctrl.modelCoefficients);
            }

            $ctrl.getColumnsToHide = function(uiState) {
                const fieldsToHide = [];

                if (!uiState.timeseriesTableStatisticsDisplay.includes("stderr")) {
                    fieldsToHide.push("stderr")
                }
                if (!uiState.timeseriesTableStatisticsDisplay.includes("p-value")) {
                    fieldsToHide.push("pValue")
                }
                if (!uiState.timeseriesTableStatisticsDisplay.includes("t-stat")) {
                    fieldsToHide.push("tStat")
                }
                return fieldsToHide;
            }

            $ctrl.setFlags = function(uiState) {
                uiState.canDisplayStatisticalValues = ["AUTO_ARIMA", "ARIMA", "ETS", "SEASONAL_LOESS"].includes($ctrl.postTrain.algorithm);
                uiState.hasStderrs = $ctrl.modelCoefficients.some(coeff => coeff.stderrs !== undefined && Object.keys(coeff.stderrs).length > 0);
                uiState.hasPvalues = $ctrl.modelCoefficients.some(coeff => coeff.pvalues !== undefined && Object.keys(coeff.pvalues).length > 0);
                uiState.hasTvalues = $ctrl.modelCoefficients.some(coeff => coeff.tvalues !== undefined && Object.keys(coeff.tvalues).length > 0);
                uiState.isModelCoefficient = true;
                uiState.hasStatistics = uiState.hasStderrs || uiState.hasPvalues || uiState.hasTvalues;
                uiState.timeseriesTableStatisticsDisplay = [];
            }

            $ctrl.getHeaders = function($scope) {
                const modelCoefficientsHeader = PerTimeseriesService.initModelCoefficientsHeader($ctrl.modelCoefficients, true, true, true);
                return PerTimeseriesService.initTimeseriesIdentifierTableColumns($scope.timeseriesIdentifierColumns, modelCoefficientsHeader);
            }

            $ctrl.getRowsForIdentifier = function(parsedTimeseriesIdentifier, unparsedTimeseriesIdentifier, uiState) {
                const row = {...parsedTimeseriesIdentifier};
                $ctrl.modelCoefficients.forEach(function(coeff) {
                    const coefficientValue = unparsedTimeseriesIdentifier in coeff.values ? coeff.values[unparsedTimeseriesIdentifier] : "";

                    const displayName = coeff["displayName"];

                    row[displayName] = coefficientValue !== "" ? String(coefficientValue.toFixed(4)) : "";

                    if (uiState.hasStderrs) {
                        const hasStderr = coeff.stderrs[unparsedTimeseriesIdentifier] !== undefined;
                        row[displayName + " stderr"] = hasStderr ? String(coeff.stderrs[unparsedTimeseriesIdentifier].toFixed(4)) : "";
                    }

                    if (uiState.hasPvalues) {
                        const hasPvalue = coeff.pvalues[unparsedTimeseriesIdentifier] !== undefined;
                        row[displayName + " p-value"] = hasPvalue ? String(coeff.pvalues[unparsedTimeseriesIdentifier].toFixed(4)) : "";
                    }

                    if (uiState.hasTvalues) {
                        const hasTvalue = coeff.tvalues[unparsedTimeseriesIdentifier] !== undefined;
                        row[displayName + " t-stat"] = hasTvalue ? String(coeff.tvalues[unparsedTimeseriesIdentifier].toFixed(4)) : "";
                    }
                });

                return [row];
            };
        },
    });

    app.component("granularTimeseriesMetrics", {
        bindings: {
            timeseriesIdentifierColumns: '<',
            fullModelId: '<',
            fullModelEvaluationId: '<?',
            insightId: '<',
            isKfold: '<',
            isCurrentlyDisplayingKfold: '=' // only used to signal to parent current display status
        },
        template: `
            <table-manager
                class="h100 model-info-page__table-manager"
                get-rows-for-identifier="$ctrl.getRowsForIdentifier"
                get-headers="$ctrl.getHeaders"
                full-model-id="{{$ctrl.fullModelId}}"
                unparsed-timeseries-identifiers="$ctrl.unparsedTimeseriesIdentifiers"
                set-flags="$ctrl.setFlags"
                insight-id="$ctrl.insightId"
                timeseries-identifier-columns="$ctrl.timeseriesIdentifierColumns"
                on-ui-change="$ctrl.onUiChange"
                get-additional-filters="$ctrl.getAdditionalFilters"
                get-pinned-rows="$ctrl.calculateAverageRow"
            ></table-manager>
        `,
        controller: function($scope, $filter, DataikuAPI, PMLSettings, PMLFilteringService, PerTimeseriesService, $stateParams) {
            const $ctrl = this;
            $ctrl.unparsedTimeseriesIdentifiers = [];
            $ctrl.foldHeaders = [
                {
                    headerName: "Fold ID",
                    field: "foldId",
                    filter: 'agNumberColumnFilter',
                    type: 'leftAligned', // ag-grid for content layout
                    valueType: "int", // us to consume when populating rows
                    pinned:"left"
                },
                {
                    headerName: "Start Date",
                    field: "startDate",
                    type: "date"
                },
                {
                    headerName: "End Date",
                    field: "endDate",
                    type: "date"
                }
            ];
            $ctrl.showPerFold = false;
            $ctrl.hasIdentifiers = true;
            $ctrl.isGranularMetrics = true;
            $ctrl.couldNotLocatePerFoldMetrics = false;
            $ctrl.customMetricHeaders = [];

            function getDisplayValueFromMetricNode(params) {
                const rawField = params.colDef.rawField;
                return params.data[rawField].displayValue
            }


            $ctrl.$onInit = function() {
                const PER_TIMESERIES_EVALUATION_METRICS = ["MSE", "MAPE", "MASE", "SMAPE", "MAE", "MSIS"];
                const timeseriesEvaluationMetrics = PMLSettings.taskF().timeseriesEvaluationMetrics;

                $ctrl.metricNameMap = timeseriesEvaluationMetrics
                    .filter(metric => PER_TIMESERIES_EVALUATION_METRICS.includes(metric[0]))
                    .map(function(metric) {
                        return {
                            headerName: PMLSettings.names.evaluationMetrics[metric[0]], // used by ag-grid for the column name
                            field: PMLFilteringService.metricMap[metric[0]] + ".rawValue", // used by ag-grid for the column value
                            rawField: PMLFilteringService.metricMap[metric[0]], // used by us to export
                            rawName: metric[0],
                            filter: 'agNumberColumnFilter',
                            valueFormatter: p => getDisplayValueFromMetricNode(p)
                        };
                    });

                if ($ctrl.timeseriesIdentifierColumns.length === 0) {
                    $ctrl.showPerFold = true;
                    $ctrl.isCurrentlyDisplayingKfold = $ctrl.showPerFold;
                    $ctrl.hasIdentifiers = false;
                }

                if ($ctrl.showPerFold) {
                    $ctrl.getPerFoldData();
                } else {
                    $ctrl.getPerTimeseriesData();
                }
            };

            $ctrl.setFlags = function(uiState) {
                uiState.showPerFold = $ctrl.showPerFold;
                uiState.hasIdentifiers = $ctrl.hasIdentifiers;
                uiState.isGranularMetrics = $ctrl.isGranularMetrics;
                uiState.couldNotLocatePerFoldMetrics = $ctrl.couldNotLocatePerFoldMetrics;
                uiState.isKfold = $ctrl.isKfold;
                uiState.timeseriesTableFoldFilters = [];
                uiState.foldIds = $ctrl.foldIds;
            };

            $ctrl.onUiChange = function(uiState) {
                if ($ctrl.showPerFold !== uiState.showPerFold) {
                    $ctrl.showPerFold = uiState.showPerFold;
                    $ctrl.isCurrentlyDisplayingKfold = $ctrl.showPerFold;

                    if ($ctrl.showPerFold) {
                        $ctrl.getPerFoldData();
                    } else {
                        $ctrl.getPerTimeseriesData();
                    }
                }
            }

            $ctrl.getAdditionalFilters = function(uiState) {
                if (uiState.showPerFold) {
                    return {foldId: uiState.timeseriesTableFoldFilters.map(x => Number(x))} // we tell ag-grid that foldId is a number, so we have to convert here
                }
                return {};
            }

            $ctrl.getPerFoldData = function() {
                if (!$ctrl.perFoldData) {
                    DataikuAPI.ml.prediction.getKfoldPerfs($ctrl.fullModelId).then(function(response) {
                        $ctrl.perFoldData = response.data.perFoldMetrics;
                        $ctrl.foldIds = (Object.values($ctrl.perFoldData)[0].map((x) => x["foldId"]));
                        $ctrl.unparsedTimeseriesIdentifiers = Object.keys($ctrl.perFoldData);
                    }).catch(response => {
                        if (response.status === 404) {
                            $ctrl.unparsedTimeseriesIdentifiers = [];
                            $ctrl.couldNotLocatePerFoldMetrics = true;
                        } else {
                            setErrorInScope.bind($scope);
                        }
                    });
                } else {
                    $ctrl.unparsedTimeseriesIdentifiers = Object.keys($ctrl.perFoldData);
                }
            }

            $ctrl.getPerTimeseriesData = function() {
                if (!$ctrl.perTimeseriesMetrics) {
                    if ($ctrl.fullModelEvaluationId) {
                            DataikuAPI.modelevaluations.getPerTimeseriesMetrics($ctrl.fullModelEvaluationId).then(({ data }) => {
                                $ctrl.perTimeseriesMetrics = data.perTimeseries;
                                $ctrl.unparsedTimeseriesIdentifiers = Object.keys($ctrl.perTimeseriesMetrics);
                            }).catch(setErrorInScope.bind($scope));
                        } else {
                            DataikuAPI.ml.prediction.getPerTimeseriesMetrics($ctrl.fullModelId).then(({ data }) => {
                                $ctrl.perTimeseriesMetrics = data.perTimeseries;
                                $ctrl.unparsedTimeseriesIdentifiers = Object.keys($ctrl.perTimeseriesMetrics);
                            }).catch(setErrorInScope.bind($scope));
                        }
                } else {
                    $ctrl.unparsedTimeseriesIdentifiers = Object.keys($ctrl.perTimeseriesMetrics);
                }
            }

            $ctrl.getCustomMetricHeader = function(foundMetric) {
                const customMetricResults = foundMetric.customMetricsResults || [];
                return customMetricResults.map(x => {
                    return {
                        field: "CUSTOM_METRIC" + sanitize(x.metric.name),
                        rawField: "CUSTOM_METRIC" + sanitize(x.metric.name),
                        rawName: x.metric.name,
                        headerName: sanitize(x.metric.name),
                        valueFormatter: p => getDisplayValueFromMetricNode(p)
                    }
                });
            }

            $ctrl.getHeaders = function($scope) {
                if ($scope.uiState.showPerFold) {
                    $ctrl.customMetricHeaders = $ctrl.getCustomMetricHeader($ctrl.perFoldData[Object.keys($ctrl.perFoldData)[0]][0].metrics)
                    return PerTimeseriesService.initTimeseriesIdentifierTableColumns($ctrl.timeseriesIdentifierColumns, [...$ctrl.foldHeaders, ...$ctrl.metricNameMap, ...$ctrl.customMetricHeaders]);
                } else {
                    $ctrl.customMetricHeaders = $ctrl.getCustomMetricHeader($ctrl.perTimeseriesMetrics[Object.keys($ctrl.perTimeseriesMetrics)[0]])
                    return PerTimeseriesService.initTimeseriesIdentifierTableColumns($ctrl.timeseriesIdentifierColumns, [...$ctrl.metricNameMap, ...$ctrl.customMetricHeaders]);
                }
            };

            $ctrl.getRowsForIdentifier = function(parsedTimeseriesIdentifier, unparsedTimeseriesIdentifier, uiState) {
                if (uiState.showPerFold) {
                    const rows = [];
                    const perTsMetrics = $ctrl.perFoldData[unparsedTimeseriesIdentifier];
                    for (let foldMetric of perTsMetrics) {
                        const row = {...parsedTimeseriesIdentifier};
                        $ctrl.foldHeaders.forEach(function(header) {
                            if (header.valueType === "int") {
                                row[header.field] = Number(foldMetric[header.field]); // necessary for filtering to work
                            } else {
                                row[header.field] = foldMetric[header.field];
                            }
                        });

                        const metrics = foldMetric.metrics;

                        $ctrl.metricNameMap.filter(metricHeader => !metricHeader.isCustom).forEach(function(metricHeader) {
                            row[metricHeader.rawField] = {
                                rawValue: metrics[metricHeader.rawField],
                                displayValue: PerTimeseriesService.createMetricRow(metrics[metricHeader.rawField], metrics[metricHeader.rawField + 'std'], metricHeader.rawName)
                            }
                        });
                        if ($ctrl.customMetricHeaders.length > 0) {
                            foldMetric["metrics"].customMetricsResults.forEach(function(customMetric) {
                                row["CUSTOM_METRIC" + sanitize(customMetric.metric.name)] = {
                                    rawValue: customMetric.value,
                                    displayValue: PerTimeseriesService.createMetricRow(customMetric.value, customMetric.valuestd, null),
                                    rawStd: customMetric.valuestd
                                }
                            })
                        }
                        rows.push(row);
                    }
                    return rows;
                } else {
                    const row = {...parsedTimeseriesIdentifier};
                    const data = $ctrl.perTimeseriesMetrics[unparsedTimeseriesIdentifier];
                    $ctrl.metricNameMap.filter(metricHeader => !metricHeader.isCustom).forEach(function(metricHeader) {
                        row[metricHeader.rawField] = {
                            rawValue: data[metricHeader.rawField],
                            rawStd: data[metricHeader.rawField + 'std'],
                            displayValue: PerTimeseriesService.createMetricRow(data[metricHeader.rawField], data[metricHeader.rawField + 'std'], metricHeader.rawName)
                        }
                    });

                    if ($ctrl.customMetricHeaders.length > 0) {
                        data.customMetricsResults.forEach(function(customMetric) {
                            row["CUSTOM_METRIC" + sanitize(customMetric.metric.name)] = {
                                rawValue: customMetric.value,
                                displayValue: PerTimeseriesService.createMetricRow(customMetric.value, customMetric.valuestd, null),
                                rawStd: customMetric.valuestd
                            }
                        })
                    }
                    return [row];
                }
            };

            $ctrl.calculateAverageRow = function(gridOptions, headers){

                if (!gridOptions || !headers) return [];

                const numericColumns = getNumericColumns(headers);

                if (numericColumns.length === 0 || gridOptions.api.getDisplayedRowCount() === 0) return [];

                let columnValues = {}
                numericColumns.forEach(col => (columnValues[col.rawField] = {values:[], rawName: col.rawName}));

                // Retrieve each values displayed in each numerical column
                gridOptions.api.forEachNodeAfterFilter(node => {
                    numericColumns.forEach(col => {
                        if (node.data[col.rawField]){
                            columnValues[col.rawField].values.push(node.data[col.rawField].rawValue)
                        }
                    });
                });

                let averageRow = {}

                // Compute average datas for each columns
                for (const [key, columns] of Object.entries(columnValues)) {

                    const validValues = columns.values.filter(val => val !== undefined);

                    if (validValues.length === 0) continue;

                    const mean = validValues.reduce((sum, val) => sum + val, 0) / validValues.length;
                    const variance = validValues.reduce((sum, val) => sum + Math.pow(val - mean, 2), 0) / validValues.length;
                    const stdDev = Math.sqrt(variance);

                    averageRow[key] = {
                        rawValue: mean,
                        rawStd: stdDev,
                        displayValue: PerTimeseriesService.createMetricRow(mean, stdDev / 2, columns.rawName, 2),
                    }
                }
                return [averageRow]
            }

            function getNumericColumns(headers) {
                return headers
                    .filter(col => col.type === "rightAligned")
                    .map(col => ({
                        rawField: col.rawField,
                        rawName: col.rawName
                    }));
            }
        }
    });

    app.service("PerTimeseriesService", function($filter, SINGLE_TIMESERIES_IDENTIFIER) {
        const COLUMN_WIDTH = 160;

        return {
            removeDuplicatesAndSortIdentifierValuesForFilterDropdowns, addIdentifierValues,
            initTimeseriesIdentifierTableColumns, initTimeseriesIdentifiersValues,
            initModelCoefficientsHeader, retrieveUnparsedTimeseriesIdentifiers, createMetricRow, shouldDisplayTimeseries
        };

        function initTimeseriesIdentifiersValues(timeseriesIdentifierColumns) {
            const allTimeseriesIdentifierValuesMap = {};
            if (!timeseriesIdentifierColumns || !timeseriesIdentifierColumns.length) return allTimeseriesIdentifierValuesMap;
            timeseriesIdentifierColumns.forEach(function(identifierColumn) {
                allTimeseriesIdentifierValuesMap[identifierColumn] = [];
            });
            return allTimeseriesIdentifierValuesMap;
        }

        function initTimeseriesIdentifierTableColumns(timeseriesIdentifierColumns, specificColHeaders) {
            const tableHeaders = [];
            timeseriesIdentifierColumns.forEach(function(identifierColumn, _) {
                tableHeaders.push({
                    field: identifierColumn,
                    pinned: 'left',
                    menuTabs: ["generalMenuTab"],
                });
            });

            specificColHeaders.forEach(function(header, _) {
                let new_header = { type: 'rightAligned', ...header }
                tableHeaders.push(new_header);
            });

            tableHeaders.forEach(dict => {
                if (!dict.headerName) { // we explicitly set headerName for the metrics columns, therefore we don't want to override
                    dict.headerName = dict.field; // `field` gets auto-capitalised by ag-grid, which causes problems for arima coefficients - headerName is not modified
                }
            });

            return tableHeaders;
        }

        function initModelCoefficientsHeader(coefficients, withStderrs, withPvalues, withTvalues) {
            const modelCoefficientsHeader = [];
            const showStderrs = withStderrs ? coefficients.some((coeff) => coeff.stderrs !== undefined && Object.keys(coeff.stderrs).length > 0) : false;
            const showPvalues = withPvalues ? coefficients.some((coeff) => coeff.pvalues !== undefined && Object.keys(coeff.pvalues).length > 0) : false;
            const showTvalues = withTvalues ? coefficients.some((coeff) => coeff.tvalues !== undefined && Object.keys(coeff.tvalues).length > 0) : false;
            coefficients.forEach(function(coeff) {
                const displayName = coeff.displayName;
                if (coeff.isExternalFeature) {
                    // Replace ":" by "_" when exporting data
                    modelCoefficientsHeader.push({
                        field: sanitize(displayName),
                        displayName: $filter("mlFeature")(displayName, true),
                        rawName: displayName.replace(/:/g, "_"),
                        headerClass:["monospace-column", "ag-right-aligned-header"]
                    });
                } else {
                    modelCoefficientsHeader.push({ field: displayName, displayName: displayName });
                }

                // add columns for statistics after each coefficient value
                // use short names for statistical values to fit table headers, they are still exported with their full name
                const shortNames = {
                    "smoothing_level": "sl",
                    "smoothing_trend": "st",
                    "initial_level": "il",
                    "initial_trend": "it",
                }
                let shortName = displayName;
                if (shortNames[displayName] !== undefined) {
                    shortName = shortNames[displayName];
                }

                // add columns for statistics after each coefficient value
                if (showStderrs) {
                    modelCoefficientsHeader.push({ field: displayName + " stderr", displayName: shortName + " stderr", rawName: displayName + " stderr", stderr:true });
                }
                if (showPvalues) {
                    modelCoefficientsHeader.push({ field: displayName + " p-value", displayName: shortName + " p-value", rawName: displayName + " p-value", pValue:true });
                }
                if (showTvalues) {
                    modelCoefficientsHeader.push({ field: displayName + " t-stat", displayName: shortName + " t-stat", rawName: displayName + " t-stat", tStat:true });
                }
            });
            return modelCoefficientsHeader;
        }

        function retrieveUnparsedTimeseriesIdentifiers(postTrain, modelCoefficients) {
            // Use model coefficients that are present for all time series to retrieve all time series identifiers.
            // Indeed some coefficients can be defined only for a subset of time series, hence are not good candidates to retrieve all time series identifiers.
            const algorithm = postTrain.algorithm;
            if (algorithm == "AUTO_ARIMA") return Object.keys(postTrain.auto_arima_timeseries_params.p);
            let coeffAlwaysSet;
            if (algorithm == "ARIMA") coeffAlwaysSet = modelCoefficients[0]; // any coefficient can work
            if (["SEASONAL_LOESS", "ETS"].includes(algorithm)) coeffAlwaysSet = modelCoefficients.find(x => x.displayName === "smoothing_level");
            if (algorithm == "PROPHET") coeffAlwaysSet = modelCoefficients.find(x => x.displayName === "k");
            if (coeffAlwaysSet) return Object.keys(coeffAlwaysSet.values);
            return [];
        }

        function addIdentifierValues(parsedTimeseriesIdentifier, timeseriesIdentifierColumns, allTimeseriesIdentifierValuesMap) {
            timeseriesIdentifierColumns.forEach(function(identifierColumn) {
                const identifierValue = parsedTimeseriesIdentifier[identifierColumn];
                allTimeseriesIdentifierValuesMap[identifierColumn].push(identifierValue);
            });
        }

        function removeDuplicatesAndSortIdentifierValuesForFilterDropdowns(allTimeseriesIdentifierValuesMap) {
            Object.keys(allTimeseriesIdentifierValuesMap).forEach(function(identifierColumn) {
                // remove duplicates and sort a copy of the array, to trigger update of dropdown values in basic-select
                const sortedValues = [...new Set(allTimeseriesIdentifierValuesMap[identifierColumn])].sort();
                allTimeseriesIdentifierValuesMap[identifierColumn] = sortedValues;
            });
        }

        function createMetricRow(metricValue, metricValueStd, metricRawName, precision=5) {
            const metricDisplayValue = $filter("mlMetricFormat")(metricValue, metricRawName, precision, metricValueStd, true);
            // this is a copy of what is done in the stripHtml method of the showTooltipOnTextOverflow directive
            // no need to sanitize as the only outside value is the metric value that is always a float or null
            return new DOMParser().parseFromString(metricDisplayValue, 'text/html').body.innerText || "";
        }

        function shouldDisplayTimeseries(unparsedTimeseriesIdentifier, filters) {
            if (unparsedTimeseriesIdentifier === SINGLE_TIMESERIES_IDENTIFIER) return true;
            const parsedTimeseriesIdentifier = JSON.parse(unparsedTimeseriesIdentifier);
            for (let filter in filters) {
                if (!filters[filter].length) continue;
                if (!filters[filter].includes(parsedTimeseriesIdentifier[filter])) {
                    return false;
                }
            }
            return true;
        }
    });


    app.controller("TimeseriesInteractiveScoringAnalysis", function($scope, PMLSettings, DataikuAPI, PerTimeseriesService, TimeseriesForecastingUtils, FutureProgressModal) {

        $scope.supportsExternalFeatures = false;

        $scope.$watch('modelData', function() {
            if (!$scope.modelData) return;

            if (!TimeseriesForecastingUtils.ALGOS_WITHOUT_EXTERNAL_FEATURES.names.includes($scope.modelData.modeling.algorithm)){
                if (Object.values($scope.modelData.preprocessing.per_feature).some(f => f.role === "INPUT")) {
                    $scope.supportsExternalFeatures = true;
                    $scope.createScenario();
                }
            }
        });

        $scope.createScenario = function() {
            $scope.loading = true;
            DataikuAPI.ml.prediction.createInteractiveScoringScenario($scope.modelData.fullModelId)
                .then(function (res) {
                    FutureProgressModal.show($scope, res.data, "Create timeseries scenarios").then(function (result) {
                        if (result) {
                            $scope.scenarios = result;
                        }
                    });
                }).catch(setErrorInScope.bind($scope)).finally(() => {
                $scope.loading = false;
            });
        }

        $scope.getScenario = function() {
            DataikuAPI.ml.prediction.getInteractiveScoringScenario($scope.modelData.fullModelId).success(function(data) {
                $scope.scenarios = data
            }).error(setErrorInScope.bind($scope));
        }

        $scope.getScenariosExternalFeatures = function() {
            return Object.values($scope.scenarios)?.[0]?.scenario?.[0]?.externalFeatures 
                ? Object.keys(Object.values($scope.scenarios)[0].scenario[0].externalFeatures)
                : [];
        };
    });

    app.controller("TimeseriesPMLReportForecastController", function($scope, PMLSettings, DataikuAPI, SINGLE_TIMESERIES_IDENTIFIER, PerTimeseriesService, $stateParams) {
        $scope.algosWithoutQuantiles = PMLSettings.algorithmCategories("TIMESERIES_FORECAST")["Baseline Models"];
        const isDashboardTile = !!$stateParams.dashboardId;
        $scope.objectId = isDashboardTile ? `insightId.${$scope.insight.id}` : `fullModelId.${$scope.modelData.fullModelId}`;
        $scope.timeseriesGraphFilters = {};
        $scope.allDisplayedTimeseries = {};
        $scope.quantiles = {
            lower: null,
            upper: null
        };

        let forecasts;
        const deregister = $scope.$watch('modelData', function() {
            if (!$scope.modelData) return;

            let getForecastPromise;
            if ($scope.fullModelEvaluationId) {
                getForecastPromise = DataikuAPI.modelevaluations.getForecasts($scope.fullModelEvaluationId);
            } else {
                getForecastPromise = DataikuAPI.ml.prediction.getForecasts($scope.modelData.fullModelId);
            }
            getForecastPromise.success(function(data) {
                forecasts = data;
                const algo = $scope.modelData.actualParams?.resolved?.algorithm || $scope.modelData.modeling.algorithm;
                if (!$scope.algosWithoutQuantiles.includes(algo) && $scope.modelData.coreParams.quantilesToForecast.length > 1) {
                    $scope.quantiles.lower = Math.min(...$scope.modelData.coreParams.quantilesToForecast);
                    $scope.quantiles.upper = Math.max(...$scope.modelData.coreParams.quantilesToForecast);
                }

                const unparsedTimeseriesArray = Object.keys(forecasts.perTimeseries);
                if (unparsedTimeseriesArray.length > 1) {
                    $scope.allTimeseriesIdentifierValuesMap = PerTimeseriesService.initTimeseriesIdentifiersValues($scope.modelData.coreParams.timeseriesIdentifiers);
                    unparsedTimeseriesArray.forEach(function(unparsedTimeseriesIdentifier) {
                        const parsedTimeseriesIdentifier = JSON.parse(unparsedTimeseriesIdentifier);
                        PerTimeseriesService.addIdentifierValues(parsedTimeseriesIdentifier, $scope.modelData.coreParams.timeseriesIdentifiers, $scope.allTimeseriesIdentifierValuesMap);
                    });
                    PerTimeseriesService.removeDuplicatesAndSortIdentifierValuesForFilterDropdowns($scope.allTimeseriesIdentifierValuesMap);
                }

                $scope.updateDisplayedTimeseries();

                deregister();
            }).error(setErrorInScope.bind($scope));
        });

        $scope.updateDisplayedTimeseries = function() {
            if (!forecasts) return;
            for (const unparsedTimeseriesIdentifier in forecasts.perTimeseries) {
                delete $scope.allDisplayedTimeseries[unparsedTimeseriesIdentifier];
                if (PerTimeseriesService.shouldDisplayTimeseries(unparsedTimeseriesIdentifier, $scope.timeseriesGraphFilters)) {
                    $scope.allDisplayedTimeseries[unparsedTimeseriesIdentifier] = forecasts.perTimeseries[unparsedTimeseriesIdentifier];
                }
            }
        };

        $scope.$on('timeseriesIdentifiersFiltersUpdated', function(event, filters) {
            $scope.timeseriesGraphFilters = filters;
            $scope.updateDisplayedTimeseries();
        });

        $scope.displayQuantilesSelector = function() {
            const algo = $scope.modelData.actualParams?.resolved?.algorithm || $scope.modelData.modeling.algorithm;
            return (
                !$scope.algosWithoutQuantiles.includes(algo) &&
                $scope.modelData.coreParams.quantilesToForecast.length > 1
            );
        }

        $scope.showNotAllTimeseriesDisplayedWarning = function() {
            if ($scope.modelData
                && $scope.modelData.iperf
                && $scope.modelData.iperf.totalNbOfTimeseries
                && forecasts
                && forecasts.perTimeseries) {
                $scope.nbOfTimeseriesInForecastCharts = Object.keys(forecasts.perTimeseries).length
                return $scope.nbOfTimeseriesInForecastCharts < $scope.modelData.iperf.totalNbOfTimeseries;
            }
            return false;
        }
    });

    app.controller('TimeseriesPMLReportResamplingController', function($scope, Assert, TimeseriesForecastingUtils) {
        Assert.inScope($scope, 'modelData');
        $scope.timestepParams = $scope.modelData.coreParams.timestepParams;
        $scope.timeseriesSampling = $scope.modelData.preprocessing.timeseriesSampling;
        $scope.numericalInterpolateMethod = TimeseriesForecastingUtils.TIMESERIES_IMPUTE_METHODS.find(
            obj => obj.value === $scope.timeseriesSampling.numericalInterpolateMethod
        );
        $scope.numericalExtrapolateMethod = TimeseriesForecastingUtils.TIMESERIES_IMPUTE_METHODS.find(
            obj => obj.value === $scope.timeseriesSampling.numericalExtrapolateMethod
        );
        $scope.categoricalImputeMethod = TimeseriesForecastingUtils.TIMESERIES_IMPUTE_METHODS.find(
            obj => obj.value === $scope.timeseriesSampling.categoricalImputeMethod
        );
        $scope.duplicateTimestampsHandlingMethod = TimeseriesForecastingUtils.DUPLICATE_TIMESTAMPS_HANDLING_METHODS.find(
            obj => obj.value === $scope.timeseriesSampling.duplicateTimestampsHandlingMethod
        );
        $scope.prettyTimeSteps = TimeseriesForecastingUtils.prettyTimeSteps;
        $scope.prettySelectedDate = TimeseriesForecastingUtils.prettySelectedDate;
        $scope.getWeekDayName = TimeseriesForecastingUtils.getWeekDayName;
    });

    /**
     * Directive for the forecasting charts (used in model reports & snippets)
     *
     * @param {object} forecasts - the different series (actual, backtest, future)
     * @param {string} timeseries - name of the time series
     * @param {object} quantiles - object with properties regarding the quantiles:
     *                             > display: whether to display the quantiles
     *                             > lower & upper: the lower & upper quantiles to display
     *                             (only for reports, where the user can change the default)
     * @param {string} target - (optional) name of the target var. Only used for reports.
     * @param {string} time - (optional) name of the time var. Only used for reports.
     * @param {object} yAxisRange - (optional) object with properties 'min' & 'max' to define
     *                            the range of the y axis. Only used for snippets, to enforce
     *                            a similar range across the different models.
     * @param {boolean} disableInsideZoom - (optional) whether to enable zooming in the chart.
     *                                    Only used for reports (zoom is never available in snippets).
     */
    app.directive("timeseriesForecastingGraphs", function($filter, Debounce) {
        return {
            scope: {
                forecasts: '<',
                timeseries: '<',
                quantiles: '<',
                horizon: '<',
                gapSize: '<',
                target: '<?',
                time: '<?',
                yAxisRange: '<?',
                disableInsideZoom: '<?',
                loadedStateField: '<'
            },
            template: `<div class="h100">
            <div block-api-error />
            <ng2-lazy-echart [options]="chartOptions" ng-if="chartOptions" (chart-init)="onChartInit($event)"></ng2-lazy-echart>
        </div>`,
            restrict: 'A',
            link: function(scope, _elem, attrs) {

                // Allow MDG/puppeteer to find this element, and know when it has loaded
                const puppeteerSelectorName = scope.loadedStateField;
                if (puppeteerSelectorName) {
                    _elem.attr(puppeteerSelectorName, true)

                    const setPuppeteerField = function() {
                        scope[puppeteerSelectorName] = true;
                    };

                    // we debounce here as the echart render event actually gets called multiple times during the animation process -
                    // if we mark ready on the first fire, the screenshot will often be not the fully rendered graph
                    var debouncedSetPuppeteerField = Debounce().withDelay(50, 200).wrap(setPuppeteerField);

                    scope.onChartInit = (echart) => {
                        echart.on('rendered', () => {
                            debouncedSetPuppeteerField();
                        });
                    };
                }

                // eslint-disable-next-line no-prototype-builtins
                const largeContainer = attrs.hasOwnProperty("largeContainer");

                const COLORS = [
                    { line: "#1F77B4", area: "#AEC7E8" },
                    { line: "#FF7F0E", area: "#FFBB78" },
                    { line: "#2CA02C", area: "#98DF8A" },
                    { line: "#D62728", area: "#FF9896" },
                    { line: "#9467BD", area: "#C5B0D5" },
                    { line: "#8C564B", area: "#C49C94" },
                    { line: "#E377C2", area: "#F7B6D2" },
                    { line: "#7F7F7F", area: "#C7C7C7" },
                    { line: "#BCBD22", area: "#DBDB8D" },
                    { line: "#17BECF", area: "#9EDAE5" },
                    { line: "#393B79", area: "#5254A3" },
                    { line: "#637939", area: "#8CA252" },
                    { line: "#6B6ECF", area: "#9C9EDE" },
                    { line: "#843C39", area: "#AD494A" },
                    { line: "#7B4173", area: "#A55194" },
                ];

                function createSeries(data, color, name, quantileAsMarkline) {
                    return {
                        type: 'line',
                        symbol: 'circle',
                        emphasis: { scale: false },
                        // zlevel is 0 by default; if quantiles are displayed as marklines,
                        // with zlevel = 1 the data point is drawn in front instead of hidden
                        zlevel: quantileAsMarkline ? 1 : 0,
                        data,
                        name,
                        color,
                    };
                }

                function createQuantileSeries(lowerQuantileSeriesData, upperQuantileSeriesData, color, name, singleDataPoint) {
                    const lowerQuantileSeries = {
                        type: 'line',
                        data: lowerQuantileSeriesData,
                        lineStyle: { opacity: 0 },
                        stack: name,
                        symbol: 'none',
                    };

                    const upperQuantileSeries = {
                        type: 'line',
                        data: upperQuantileSeriesData,
                        lineStyle: { opacity: 0 },
                        areaStyle: { color },
                        stack: name,
                        stackStrategy: 'all',
                        symbol: 'none',
                        emphasis: { disabled: true },
                        name,
                        color,
                    };

                    if (singleDataPoint) {
                        // Quantile series have one data point, so series-line.stack does not work,
                        // we will use a markline instead.

                        // When singleDataPoint is true, lower quantile (resp. upper quantile) is an array of
                        // one data point, i.e. lowerQuantileSeriesData = [ [ date, y ] ]. So we retrieve the
                        // first and only value in the array & then spread it to retrieve date (x) & y.
                        const [date, lowerY, _, upperY] = [
                            ...lowerQuantileSeriesData[0],
                            ...upperQuantileSeriesData[0]
                        ];
                        upperQuantileSeries.markLine = {
                            symbol: 'none',
                            silent: true,
                            lineStyle: { type: 'solid', width: 4, color },
                            data: [[
                                { xAxis: date, yAxis: lowerY },
                                { xAxis: date, yAxis: upperY },
                            ]],
                        };
                    }
                    return [lowerQuantileSeries, upperQuantileSeries];
                }

                // eslint-disable-next-line no-prototype-builtins
                const displayFuture = attrs.hasOwnProperty("displayFuture") && scope.forecasts.futureForecast;
                const toFixed = (number, precision) => parseFloat(number.toFixed(precision));

                function updateGraph() {
                    if (!scope.forecasts) return;

                    if (puppeteerSelectorName) {
                        scope[puppeteerSelectorName] = false;
                    }

                    const folds = new Set(scope.forecasts.foldId);
                    const nrFolds = folds.size;
                    // Nb. folds always divide nb. timestamps (nb.timestamps in backtest folds = test size * nrFolds)
                    const backtestSize = scope.forecasts.forecast.length / nrFolds;

                    let quantilesToDisplay;
                    if (scope.quantiles && scope.quantiles.display) {
                        let lowerQuantileData;
                        let upperQuantileData;
                        if (angular.isNumber(scope.quantiles.lower)) {
                            lowerQuantileData = scope.forecasts.quantiles.find((elem) => elem.quantile === scope.quantiles.lower);
                        }
                        if (angular.isNumber(scope.quantiles.upper)) {
                            upperQuantileData = scope.forecasts.quantiles.find((elem) => elem.quantile === scope.quantiles.upper);
                        }

                        if (!lowerQuantileData || !upperQuantileData) {
                            const sortedQuantiles = scope.forecasts.quantiles.sort((a, b) => a.quantile - b.quantile);
                            lowerQuantileData = lowerQuantileData || sortedQuantiles[0];
                            upperQuantileData = upperQuantileData || sortedQuantiles[sortedQuantiles.length - 1];
                        }

                        if (upperQuantileData.quantile > lowerQuantileData.quantile) {
                            const lowestQuantileValue = scope.yAxisRange && scope.yAxisRange.min || Math.min(
                                ...lowerQuantileData.forecast, ...(displayFuture ? lowerQuantileData.futureForecast : [])
                            );

                            quantilesToDisplay = {
                                legend: `[${toFixed(lowerQuantileData.quantile, 3)}, ${toFixed(upperQuantileData.quantile, 3)}] interval`,
                                lower: {
                                    value: lowerQuantileData.quantile,
                                    backTestData: lowerQuantileData.forecast.map(function(data, idx) {
                                        return [scope.forecasts.forecastTime[idx], data];
                                    })
                                },
                                upper: {
                                    value: upperQuantileData.quantile,
                                    backTestData: upperQuantileData.forecast.map(function(data, idx) {
                                        if (backtestSize === 1) {
                                            // Quantiles are marklines; simply use upper quantile value
                                            return [scope.forecasts.forecastTime[idx], data];
                                        }
                                        // Quantiles are areas (using series-line.stack); use the
                                        // difference between lower & upper quantile values
                                        return [scope.forecasts.forecastTime[idx], data - lowerQuantileData.forecast[idx]];
                                    })
                                }
                            }

                            if (displayFuture) {
                                quantilesToDisplay.lower.forecastData = lowerQuantileData.futureForecast.map(function(data, idx) {
                                    return [scope.forecasts.futureTime[idx], data];
                                });
                                quantilesToDisplay.upper.forecastData = upperQuantileData.futureForecast.map(function(data, idx) {
                                    if (scope.forecasts.futureForecast.length === 1) {
                                        // Quantiles are marklines; simply use upper quantile value
                                        return [scope.forecasts.futureTime[idx], data];
                                    }
                                    // Quantiles are areas (using series-line.stack); use the
                                    // difference between lower & upper quantile values
                                    return [scope.forecasts.futureTime[idx], data - lowerQuantileData.futureForecast[idx]];
                                });
                            }
                        }
                    }

                    let actualData;
                    if (largeContainer) {
                        actualData = scope.forecasts.groundTruth.map(function(data, idx) {
                            return [scope.forecasts.groundTruthTime[idx], data];
                        });
                        if (scope.forecasts.futureForecastContext != undefined) {
                            let forecastContextData = scope.forecasts.futureForecastContext.map(function(data, idx) {
                                return [scope.forecasts.futureForecastContextTime[idx], data];
                            });
                            // Future forecast context contains dataset data whenever the evaluation fold and the forecast
                            // periods are discontiguous. We provide at most 100 context timesteps. If it exceeds that value,
                            // we therefore need to add a line break in the series display.
                            if (scope.forecasts.futureForecastContext.length >= 100) { 
                                actualData = [...actualData, [undefined, undefined], ...forecastContextData];
                            } else {
                                actualData = [...actualData, ...forecastContextData];
                            }
                        }
                    } else {
                        // Do not display all past data in snippets (to make graphs more readable)
                        // Only timestamps starting from the first timestamp with backtest data + two more extra timestamps in case we have too few data points
                        const timeWindowLength = scope.forecasts.groundTruthTime.filter(date => date >= scope.forecasts.forecastTime[0]).length + 2;
                        actualData = scope.forecasts.groundTruth.slice(-timeWindowLength).map(function(data, idx) {
                            return [
                                scope.forecasts.groundTruthTime[idx + scope.forecasts.groundTruthTime.length - timeWindowLength],
                                data
                            ];
                        });
                    }

                    const backtestData = scope.forecasts.forecast.map(function(data, idx) {
                        return [scope.forecasts.forecastTime[idx], data];
                    });

                    const forecastData = (scope.forecasts.futureForecast || []).map(function(data, idx) {
                        return [scope.forecasts.futureTime[idx], data];
                    });

                    const quantilesAsMarklineForBacktest = backtestSize === 1 && !!quantilesToDisplay;
                    const quantilesAsMarklineForForecast = displayFuture && scope.forecasts.futureForecast.length === 1 && !!quantilesToDisplay;
                    const series = [createSeries(actualData, '#666666', 'Actual', quantilesAsMarklineForBacktest)];

                    const legend = [{ name: "Actual", itemStyle: { opacity: 0 } }];

                    for (let fold of folds) {
                        const firstIdx = scope.forecasts.foldId.indexOf(fold);
                        const lastIdx = scope.forecasts.foldId.lastIndexOf(fold);
                        const suffix = nrFolds > 1 ? (' (fold ' + (fold + 1) + ')') : '';
                        const color = COLORS[fold % COLORS.length];

                        const foldData = backtestData.slice(firstIdx, lastIdx + 1);
                        series.push(createSeries(
                            foldData,
                            color.line,
                            "Backtest" + suffix,
                            quantilesAsMarklineForBacktest
                        ));
                        legend.push({ name: "Backtest" + suffix, itemStyle: { opacity: 0 } });

                        if (quantilesToDisplay) {
                            const quantileAreaName = quantilesToDisplay.legend + suffix;
                            const lowerQuantileFoldData = quantilesToDisplay.lower.backTestData.slice(firstIdx, lastIdx + 1);
                            const upperQuantileFoldData = quantilesToDisplay.upper.backTestData.slice(firstIdx, lastIdx + 1);

                            series.push(...createQuantileSeries(
                                lowerQuantileFoldData, upperQuantileFoldData, color.area, quantileAreaName, quantilesAsMarklineForBacktest
                            ));

                            legend.push({ name: quantileAreaName, icon: 'roundRect' });
                        }
                    }

                    if (displayFuture) {
                        const color = COLORS[nrFolds % COLORS.length];
                        series.push(createSeries(
                            forecastData,
                            color.line,
                            'Forecast',
                            quantilesAsMarklineForForecast
                        ));
                        legend.push({ name: "Forecast", itemStyle: { opacity: 0 } });

                        if (quantilesToDisplay) {
                            const quantileAreaName = quantilesToDisplay.legend + " (forecast)";
                            series.push(...createQuantileSeries(
                                quantilesToDisplay.lower.forecastData, quantilesToDisplay.upper.forecastData,
                                color.area, quantileAreaName, quantilesAsMarklineForForecast
                            ));
                            legend.push({ name: quantileAreaName, symbol: 'line' });
                        }
                    }

                    function tooltipContentForSeries(seriesName, displayedValue, color) {
                        return `<i class='icon-circle mright8' style='color: ${color}'></i>
                        ${seriesName}: ${displayedValue}<br/>`;
                    };

                    function getStepAndHorizonTooltipLine(timestepIdx, displayHorizonNb, displayGap = true) {
                        let tooltipLine = "<span class='text-debug'>"
                        if (displayHorizonNb) {
                            const horizonNb = Math.floor(timestepIdx / scope.horizon) + 1;
                            tooltipLine += `Horizon ${horizonNb} - `;
                        }

                        const stepNb = timestepIdx % scope.horizon + 1;
                        const isGapStep = stepNb <= scope.gapSize;
                        tooltipLine += `Step ${stepNb}${isGapStep && displayGap ? ' (ignored for evaluation)' : ''}</span><br/>`;
                        return tooltipLine;
                    }

                    function getQuantileIntervalTooltipLine(lowerQuantilesSeries, upperQuantilesSeries, quantilesAsMarkline) {
                        if (quantilesAsMarkline) {
                            return `[${toFixed(lowerQuantilesSeries.data[1], 3)}, ${toFixed(upperQuantilesSeries.data[1], 3)}]`
                        }
                        return `[${toFixed(lowerQuantilesSeries.data[1], 3)}, ${toFixed(lowerQuantilesSeries.data[1] + upperQuantilesSeries.data[1], 3)}]`;
                    }

                    scope.chartOptions = {
                        title: {
                            show: largeContainer,
                            textAlign: "left",
                            text: $filter('displayTimeseriesName')(scope.timeseries),
                            textStyle: { fontWeight: 'bold', fontSize: 16 }
                        },
                        legend: {
                            show: largeContainer,
                            selectedMode: false,
                            padding: 8,
                            data: legend,
                            type: "scroll",
                            y: 16,
                            x: "center",
                            width: "75%",
                            textStyle: { lineOverflow: "truncate", width: 184, overflow: "truncate", lineHeight: 32 }
                        },
                        tooltip: {
                            trigger: 'axis',
                            textStyle: { fontSize: 13 },
                            formatter: function(params) {
                                // In the tooltip there can either be:
                                // - only one series (actual or forecast)
                                // - two series (actual + backtest)
                                // - three series (forecast + lower quantile + upper quantile)
                                // - four series (actual + backtest + lower quantile + upper quantile)
                                const [firstSeries, secondSeries, thirdSeries, fourthSeries] = params;

                                let tooltipContent = `<span class="font-weight-bold">${firstSeries.data[0]}</span><br/>`;

                                let lowerQuantilesSeries, upperQuantilesSeries;
                                if (fourthSeries) { // actual + backtest + lower quantile + upper quantile
                                    const testSizeHasMultipleHorizons = backtestSize / scope.horizon > 1;
                                    tooltipContent += getStepAndHorizonTooltipLine(secondSeries.dataIndex, testSizeHasMultipleHorizons)
                                    tooltipContent += tooltipContentForSeries(firstSeries.seriesName, toFixed(firstSeries.data[1], 3), firstSeries.color);
                                    tooltipContent += tooltipContentForSeries(secondSeries.seriesName, toFixed(secondSeries.data[1], 3), secondSeries.color);

                                    [lowerQuantilesSeries, upperQuantilesSeries] = [thirdSeries, fourthSeries];
                                    tooltipContent += tooltipContentForSeries(
                                        "Backtest interval", getQuantileIntervalTooltipLine(lowerQuantilesSeries, upperQuantilesSeries, quantilesAsMarklineForBacktest), upperQuantilesSeries.color
                                    );
                                } else if (thirdSeries) { // forecast + lower quantile + upper quantile
                                    tooltipContent += getStepAndHorizonTooltipLine(secondSeries.dataIndex, false, false)
                                    tooltipContent += tooltipContentForSeries(firstSeries.seriesName, toFixed(firstSeries.data[1], 3), firstSeries.color);
                                    [lowerQuantilesSeries, upperQuantilesSeries] = [secondSeries, thirdSeries];
                                    tooltipContent += tooltipContentForSeries(
                                        "Forecast interval", getQuantileIntervalTooltipLine(lowerQuantilesSeries, upperQuantilesSeries, quantilesAsMarklineForForecast), upperQuantilesSeries.color
                                    );
                                } else if (secondSeries) { // actual + backtest
                                    const testSizeHasMultipleHorizons = backtestSize / scope.horizon > 1;
                                    tooltipContent += getStepAndHorizonTooltipLine(secondSeries.dataIndex, testSizeHasMultipleHorizons)
                                    tooltipContent += tooltipContentForSeries(firstSeries.seriesName, toFixed(firstSeries.data[1], 3), firstSeries.color);
                                    tooltipContent += tooltipContentForSeries(secondSeries.seriesName, toFixed(secondSeries.data[1], 3), secondSeries.color);
                                } else {
                                    // only firstSeries: actual or forecast
                                    tooltipContent += tooltipContentForSeries(firstSeries.seriesName, toFixed(firstSeries.data[1], 3), firstSeries.color);
                                }

                                return tooltipContent;
                            }
                        },
                        textStyle: { fontFamily: 'SourceSansPro' },
                        toolbox: {
                            show: largeContainer,
                            top: 16,
                            right: 64,
                            feature: {
                                restore: {
                                    title: 'Reset zoom',
                                    emphasis: {
                                        iconStyle: { textPosition: 'top' }
                                    }
                                },
                            }
                        },
                        xAxis: {
                            axisTick: { show: largeContainer },
                            axisLine: { onZero: false, show: largeContainer },
                            axisLabel: {
                                rotate: 45,
                                show: largeContainer,
                                formatter: {
                                    year: '{MMM} {dd}, {yyyy}',
                                    month: '{MMM} {dd}, {yyyy}',
                                    day: '{MMM} {dd}, {yyyy}',
                                    hour: '{MMM} {dd}, {yyyy} \n{HH}:{mm}',
                                    minute: '{MMM} {dd}, {yyyy} \n{HH}:{mm}',
                                    second: '{MMM} {dd}, {yyyy} \n{HH}:{mm}:{ss}',
                                    millisecond: '{MMM} {dd}, {yyyy} \n{HH}:{mm}:{ss} {SSS}'
                                }
                            },
                            name: scope.time,
                            nameGap: 72,
                            nameLocation: "middle",
                            type: "time",
                            nameTextStyle: { overflow: "truncate", width: 264, fontWeight: 'bold' }
                        },
                        yAxis: {
                            axisLabel: { formatter: val => toFixed(val, 3) },
                            axisTick: { show: largeContainer },
                            scale: true,
                            name: scope.target,
                            nameGap: 48,
                            nameLocation: "middle",
                            type: "value",
                            nameTextStyle: { overflow: "truncate", width: 320, fontWeight: 'bold' }
                        },
                        animation: largeContainer,
                        series
                    };

                    if (largeContainer) {
                        const nrGroundTruthValues = scope.forecasts.groundTruthTime.length;
                        const pastGroundTruthTime = scope.forecasts.groundTruthTime.filter(date => date < scope.forecasts.forecastTime[0]);

                        const firstDate = pastGroundTruthTime[Math.max(0, pastGroundTruthTime.length - backtestSize)];
                        const lastDate = displayFuture ? scope.forecasts.futureTime[scope.forecasts.futureTime.length - 1] : scope.forecasts.groundTruthTime[nrGroundTruthValues - 1];
                        const minInterval = new Date(scope.forecasts.groundTruthTime[1]) - new Date(scope.forecasts.groundTruthTime[0]);

                        const padToLengthTwo = dateElem => dateElem.toString().padStart(2, '0');

                        scope.chartOptions.grid = { bottom: 136, right: 64, left: 72 };
                        scope.chartOptions.dataZoom = [
                            {
                                type: 'inside',
                                minValueSpan: 3 * minInterval,
                                startValue: new Date(firstDate),
                                endValue: new Date(lastDate),
                                disabled: scope.disableInsideZoom
                            },
                            {
                                type: "slider",
                                startValue: new Date(firstDate),
                                endValue: new Date(lastDate),
                                left: 64,
                                right: 64,
                                textStyle: { fontSize: 11 },
                                labelFormatter: function(timestamp) {
                                    const date = new Date(timestamp);
                                    const year = date.getFullYear();
                                    const month = padToLengthTwo(date.getMonth() + 1);
                                    const day = padToLengthTwo(date.getDate());
                                    const hour = padToLengthTwo(date.getHours());
                                    const minute = padToLengthTwo(date.getMinutes());
                                    const second = padToLengthTwo(date.getSeconds());
                                    return `${year}-${month}-${day}\n${hour}:${minute}:${second}`;
                                }
                            }
                        ];
                    } else {
                        scope.chartOptions.grid = { top: 8, bottom: 8, right: 8, left: 8 };
                    }

                    if (scope.yAxisRange) {
                        scope.chartOptions.yAxis.min = scope.yAxisRange.min;
                        scope.chartOptions.yAxis.max = scope.yAxisRange.max;
                        scope.chartOptions.yAxis.interval = (scope.yAxisRange.max - scope.yAxisRange.min) / 5;
                    }
                }

                scope.$watchGroup(["yAxisRange", "disableInsideZoom", "quantiles"], function() {
                    // For quantiles: We only need a shallow $watch (which is what $watchGroup provides),
                    // not a $watchCollection (~ deep watch), because the quantiles object is defined
                    // in the html -> Creates a new object when properties change.
                    updateGraph();
                });
            }
        };
    });

    app.filter('displayTimeseriesName', function(SINGLE_TIMESERIES_IDENTIFIER) {
        return function(unparsedName) {
            if (unparsedName === SINGLE_TIMESERIES_IDENTIFIER) {
                return "Time series";
            }
            return Object.entries(JSON.parse(unparsedName)).map(entry => entry.join(": ")).join(", ");
        }
    });

    app.component("timeseriesSplitSchema", {
        bindings: {
            kfold: '<',
            nFolds: '<',
            predictionLength: '<',
            foldOffset: '<',
            equalDurationFolds: '<',
            splitType: '@',
            customTrainTestSplit: '<',
        },
        templateUrl: '/templates/analysis/prediction/timeseries/split-schema.html',
    });

    app.component("timeseriesSplitSchemaBar", {
        bindings: {
            nFolds: '<',
            predictionLength: '<',
            foldOffset: '<',
            equalDurationFolds: '<',
            splitType: '<',
            foldId: '<',
            customTrainTestSplit: '<',
        },
        templateUrl: '/templates/analysis/prediction/timeseries/split-schema-bar.html',
        controller: function($attrs) {
            const ctrl = this;
            ctrl.kfold = $attrs.hasOwnProperty('kfold');
            ctrl.equalDurationFolds = $attrs.hasOwnProperty('equalDurationFolds');

            ctrl.getNbEvaluationSetToSubtractLeft = function() {
                // Gives the length in number of evaluation set of the ignored part to the left of the dataset for each bar
                if (ctrl.customTrainTestSplit && ctrl.splitType !== 'HP_SEARCH') return 1; // When split type is not HP_SEARCH, the full bar width is constrained with doctor-explanation__hyperparameters-progress--timeseries-custom-train-test-split.
                if (!ctrl.kfold || !ctrl.equalDurationFolds) return 0;

                return (ctrl.foldOffset ? 2 : 1) * (ctrl.foldId - 1);
            };

            ctrl.getNbEvaluationSetToSubtractRight = function() {
                // Gives the length in number of evaluation set of the ignored part to the right of the dataset for each bar
                if (ctrl.customTrainTestSplit && ctrl.splitType !== 'HP_SEARCH') return 1; // When split type is not HP_SEARCH, the full bar width is constrained with doctor-explanation__hyperparameters-progress--timeseries-custom-train-test-split.
                if (!ctrl.kfold || ctrl.foldId === ctrl.nFolds) return 0;

                if (ctrl.foldId === 1) {
                    return (ctrl.foldOffset ? 2 : 1) * (ctrl.nFolds - 1);
                }

                if (ctrl.foldId === ctrl.nFolds - 1) {
                    return ctrl.foldOffset ? 2 : 1;
                }
            };

            ctrl.getMinWidth = function() {
                // Only matters for first bar; ensures the first train set does not shrink too much as nfolds grows
                if (ctrl.splitType === 'HP_SEARCH' && ctrl.foldOffset) return `calc(16px + var(--doctor-evaluation-forecast-width))`;
                return "16px";
            };
        }
    });

    app.factory("TimeseriesForecastingUtils", function($filter) {
        const TIME_UNITS = {
            MILLISECOND: "Millisecond",
            SECOND: "Second",
            MINUTE: "Minute",
            HOUR: "Hour",
            DAY: "Day",
            BUSINESS_DAY: "Business day",
            WEEK: "Week",
            MONTH: "Month",
            QUARTER: "Quarter",
            HALF_YEAR: "Half year",
            YEAR: "Year"
        };

        const TIMESERIES_IMPUTE_METHODS = [
            {
                displayName: "Nearest",
                value: "NEAREST",
                description: "Value of the nearest date between previous and next",
                featureTypes: ['num'],
                interpolation: true,
                extrapolation: false
            },
            {
                displayName: "Previous",
                value: "PREVIOUS",
                description: "Value of the previous date",
                featureTypes: ['num', 'non-num'],
                interpolation: true,
                extrapolation: false
            },
            {
                displayName: "Next",
                value: "NEXT",
                description: "Value of the next date",
                featureTypes: ['num', 'non-num'],
                interpolation: true,
                extrapolation: false
            },
            {
                displayName: "Staircase",
                value: "STAIRCASE",
                description: "Average between previous and next dates values",
                featureTypes: ['num'],
                interpolation: true,
                extrapolation: false
            },
            {
                displayName: "Linear",
                value: "LINEAR",
                description: "Time-sensitive average between previous and next dates values",
                featureTypes: ['num'],
                interpolation: true,
                extrapolation: true
            },
            {
                displayName: "Quadratic",
                value: "QUADRATIC",
                description: "Quadratic interpolation of previous and next dates values",
                featureTypes: ['num'],
                interpolation: true,
                extrapolation: true
            },
            {
                displayName: "Cubic",
                value: "CUBIC",
                description: "Cubic interpolation of previous and next dates values",
                featureTypes: ['num'],
                interpolation: true,
                extrapolation: true
            },
            {
                displayName: "Constant",
                value: "CONSTANT",
                description: "Constant value",
                featureTypes: ['num', 'non-num'],
                interpolation: true,
                extrapolation: true
            },

            {
                displayName: "Previous or next",
                value: "PREVIOUS_NEXT",
                description: "Previous date (for future extrapolated values), or next date (for past values)",
                featureTypes: ['num', 'non-num'],
                extrapolation: true
            },
            { displayName: "No extrapolation", value: "NO_EXTRAPOLATION", description: "No extrapolation", featureTypes: ['num'], extrapolation: true },

            { displayName: "No value", value: "NULL", description: "No extrapolated value", featureTypes: ['non-num'] },
            { displayName: "Most common", value: "MOST_COMMON", description: "Most common value", featureTypes: ['non-num'] }
        ];

        const DUPLICATE_TIMESTAMPS_HANDLING_METHODS = [
            { displayName: "Fail on conflicting duplicates", value: "FAIL_IF_CONFLICTING" },
            { displayName: "Drop all conflicting duplicates", value: "DROP_IF_CONFLICTING" },
            { displayName: "Use mean (resp. mode) of numerical (resp. categorical) columns", value: "MEAN_MODE" }
        ];

        // To be synced with:
        // java -> com/dataiku/dip/analysis/model/prediction/TimeseriesForecastingModelDetails.java L.43
        // doctor -> dataiku/doctor/timeseries/models/__init__.py L.34
        const ALGOS_WITHOUT_EXTERNAL_FEATURES = {
            keys: [
                "trivial_identity_timeseries",
                "seasonal_naive_timeseries",
                "seasonal_loess_timeseries",
                "gluonts_simple_feed_forward_timeseries",
                "gluonts_torch_simple_feed_forward_timeseries",
                "gluonts_torch_deepar_timeseries"
            ],
            names: [
                "TRIVIAL_IDENTITY_TIMESERIES",
                "SEASONAL_NAIVE",
                "SEASONAL_LOESS",
                "GLUONTS_SIMPLE_FEEDFORWARD",
                "GLUONTS_TORCH_SIMPLE_FEEDFORWARD",
                "GLUONTS_TORCH_DEEPAR"
            ]
        };

        const ALGOS_SLOW_ON_MULTIPLE_TIMESERIES = [
            "autoarima_timeseries",
            "arima_timeseries",
            "ets_timeseries",
            "seasonal_loess_timeseries",
            "prophet_timeseries",
        ];

        const ALGOS_INCOMPATIBLE_WITH_MS = [
            "gluonts_transformer_timeseries",
            "gluonts_deepar_timeseries",
            "gluonts_npts_timeseries"
        ]

        const service = {
            TIMESERIES_IMPUTE_METHODS,
            DUPLICATE_TIMESTAMPS_HANDLING_METHODS,
            ALGOS_WITHOUT_EXTERNAL_FEATURES,
            ALGOS_SLOW_ON_MULTIPLE_TIMESERIES,
            ALGOS_INCOMPATIBLE_WITH_MS,
            prettyTimeUnit,
            prettyTimeSteps,
            plurifiedTimeUnits,
            getDayNumber,
            getWeekDayName,
            getMonthName,
            getQuarterName,
            getHalfYearName,
            prettySelectedDate,
        };

        return service;

        function prettyTimeUnit(timeunit) {
            if (!timeunit) return;
            return TIME_UNITS[timeunit].toLowerCase()
        };

        function prettyTimeSteps(timeSteps, timeunit) {
            return `${timeSteps} ${$filter('plurify')(prettyTimeUnit(timeunit), timeSteps)}`;
        };

        function getDayNumber(index) {
            switch (index) {
                case 0:
                case 31:
                    return "Last";
                case 1:
                    return "First";
                default:
                    return index.toString();
            }
        }

        function getDayName(index) {
            switch (index) {
                case 0:
                case 31:
                    return "Last day";
                case 1:
                    return "First day";
                case 21:
                    return index + "st";
                case 2:
                case 22:
                    return index + "nd";
                case 3:
                case 23:
                    return index + "rd";
                default:
                    return index + "th";
            }
        }

        function getWeekDayName(index) {
            return getDayLabels(index - 1);
        }

        function getMonthName(index) {
            if (index === 0) {
                index = 12;
            }
            const date = new Date(2024, index - 1);
            return date.toLocaleString('en', { month: 'long' });
        }

        function getQuarterName(index) {
            if (index === 0) {
                index = 3;
            }
            return [
                "Jan, Apr, Jul, Oct",
                "Feb, May, Aug, Nov",
                "Mar, Jun, Sep, Dec",
            ][index - 1];
        }

        function getHalfYearName(index) {
            if (index === 0) {
                index = 6;
            }
            return [
                "January and July",
                "February and August",
                "March and September",
                "April and October",
                "May and November",
                "June and December",
            ][index - 1];
        }

        function prettySelectedDate(timeunit, monthlyAlignement, unitAlignment) {
            // this should be the same logic as prettySelectedDate in TimeSeriesUtil.java
            let day = getDayName(monthlyAlignement);
            let period = "the month"
            if (timeunit === "QUARTER") {
                period = getQuarterName(unitAlignment);
            } else if (timeunit === "HALF_YEAR") {
                period = getHalfYearName(unitAlignment);
            } else if (timeunit === "YEAR") {
                period = getMonthName(unitAlignment);
            }
            return day + ' of ' + period;
        }

        function plurifiedTimeUnits(timeSteps) {
            const timeUnits = Object.assign({}, TIME_UNITS);
            angular.forEach(TIME_UNITS, function(displayUnit, rawUnit) {
                timeUnits[rawUnit] = $filter('plurify')(displayUnit, timeSteps);
            });
            return timeUnits;
        };
    });
    
    app.factory("TimeseriesForecastingCustomTrainTestFoldsUtils", function(TimeseriesForecastingUtils) {
        function isTestIntervalTooSmall(interval, predictionLength, timeunit) {
            const testFoldDuration = moment.duration(moment(interval['test'][1]).diff(moment(interval['test'][0])));
            switch (timeunit) {
                case "MILLISECOND": return predictionLength > testFoldDuration.asMilliseconds();
                case "SECOND": return predictionLength > testFoldDuration.asSeconds();
                case "MINUTE": return predictionLength > testFoldDuration.asMinutes();
                case "HOUR": return predictionLength > testFoldDuration.asHours();
                case "DAY":
                case "BUSINESS_DAY": {
                    return predictionLength > Math.round(testFoldDuration.asDays());
                }
                case "WEEK": return (predictionLength * 7) > Math.round(testFoldDuration.asDays());
                case "MONTH": return predictionLength > Math.round(testFoldDuration.asMonths());
                case "QUARTER": return (predictionLength * 3) > Math.round(testFoldDuration.asMonths());
                case "HALF_YEAR": return (predictionLength * 6) > Math.round(testFoldDuration.asMonths());
                case "YEAR": return predictionLength > Math.round(testFoldDuration.asYears());
            }
        }

        function validateDate(date, condition, errorMessage) {
            const dateObj = new Date(date);
            if (!condition(dateObj)) {
                return errorMessage;
            }
        }
        
        function validateCustomTrainTestFold(timestepParams, interval, predictionLength) {
            const { timeunit, endOfWeekDay, unitAlignment } = timestepParams;
            if (interval['train'][0] > interval['train'][1]) {
                return "Train interval start date must be before the end date";
            }
            if (interval['test'][0] > interval['test'][1]) {
                return "Test interval start date must be before the end date";
            }
            if (interval['train'][1] > interval['test'][0]) {
                return "Test interval start date must be after the end of the train interval";
            }
            if (["WEEK", "BUSINESS_DAY"].includes(timeunit)) {
                const targetDays = timeunit === "WEEK" ? [endOfWeekDay] : [2, 3, 4, 5, 6];
                const targetDayString = timeunit === "WEEK" ? TimeseriesForecastingUtils.getWeekDayName(endOfWeekDay) : "business day";
                const condition = d => targetDays.includes(d.getDay() + 1);
                const errorMessage =
                    validateDate(interval['train'][0], condition, "Train start is not a " + targetDayString + ".") ||
                    validateDate(interval['train'][1], condition, "Train end is not a " + targetDayString + ".") ||
                    validateDate(interval['test'][0], condition, "Test start is not a " + targetDayString + ".") ||
                    validateDate(interval['test'][1], condition, "Test end is not a " + targetDayString + ".");
                if (errorMessage) return errorMessage;
            }

            if (timeunit === "QUARTER") {
                if (timestepParams.unitAlignment) {
                    const unitAlignmentStartMonth = timestepParams.unitAlignment - 1;
                    const validMonths = [unitAlignmentStartMonth, unitAlignmentStartMonth + 3, unitAlignmentStartMonth + 6, unitAlignmentStartMonth + 9]
                    const condition = d => validMonths.includes(d.getMonth());
                    const quarterName = TimeseriesForecastingUtils.getQuarterName(unitAlignment);
                    const errorMessage =
                        validateDate(interval['train'][0], condition, "Train start is not one of " + quarterName + ".") ||
                        validateDate(interval['train'][1], condition, "Train end is not one of " + quarterName + ".") ||
                        validateDate(interval['test'][0], condition, "Test start is not one of " + quarterName + ".") ||
                        validateDate(interval['test'][1], condition, "Test end is not one of " + quarterName + ".");
                    if (errorMessage) return errorMessage;
                }
            }

            if (timeunit === "HALF_YEAR") {
                if (unitAlignment) {
                    const unitAlignmentStartMonth = unitAlignment - 1;
                    const validMonths = [unitAlignmentStartMonth, unitAlignmentStartMonth + 6]
                    const condition = d => validMonths.includes(d.getMonth());
                    const halfYearName = TimeseriesForecastingUtils.getHalfYearName(unitAlignment);
                    const errorMessage =
                        validateDate(interval['train'][0], condition, "Train start is not one of " + halfYearName + ".") ||
                        validateDate(interval['train'][1], condition, "Train end is not one of " + halfYearName + ".") ||
                        validateDate(interval['test'][0], condition, "Test start is not one of " + halfYearName + ".") ||
                        validateDate(interval['test'][1], condition, "Test end is not one of " + halfYearName + ".");
                    if (errorMessage) return errorMessage;
                }
            }

            if (isTestIntervalTooSmall(interval, predictionLength, timeunit)) {
                return `Test interval is too small (Minimum of ${TimeseriesForecastingUtils.prettyTimeSteps(predictionLength, timeunit)})`
            }
        }
        
        const service = {
            validateCustomTrainTestFold
        }
        
        return service;
    });
    
    app.component("periodDatePicker", {
        bindings: {
            timestepParams: '<',
            updateFn: '&',
            date: '<'
        },
        templateUrl: '/templates/analysis/prediction/timeseries/period-date-picker.html',
        controller: function($timeout, $element, TimeseriesForecastingUtils) {
            const $ctrl = this;
            
            $ctrl.MONTHS = ['January', 'February', 'March', 'April', 'May', 'June', 'July', 'August', 'September', 'October', 'November', 'December']
            
            $ctrl.$onInit = () => {
                $ctrl.hoursOptions = getHoursOptions();
                $ctrl.quartersOptions = getQuartersOptions();
                $ctrl.halfYearsOptions = getHalfYearsOptions();
                $ctrl.refreshYearsOptions();
                $ctrl.uiState = {
                    year: $ctrl.date.getFullYear(),
                    month: $ctrl.MONTHS[$ctrl.date.getMonth()],
                    day: $ctrl.date.getDate(),
                    hour: $ctrl.date.getHours(),
                    minute: $ctrl.date.getMinutes(),
                    second: $ctrl.date.getSeconds(),
                    millisecond: $ctrl.date.getMilliseconds(),
                    inputTimeStep: getTimeInputStep(),
                    quarter: $ctrl.MONTHS[$ctrl.date.getMonth()],
                    halfYear: $ctrl.MONTHS[$ctrl.date.getMonth()],
                    dateHolder: initDateHolder($ctrl.date, $ctrl.timestepParams.timeunit)
                };
            }
            
            $ctrl.initDatePicker = () => {
                $element.find('.datepicker').datepicker($ctrl.getDatePickerConfig());
                $element.find('.datepicker').datepicker("setDate", $ctrl.date);
            }
            
            $ctrl.showDatePicker = () => {
                $element.find('.datepicker').datepicker("show");
            }
            
            function initDateHolder(date, timeUnit) { 
                // Since we use the `step` param on the time input, this triggers native validation errors if the date
                // contains some milliseconds/seconds if the timestep doesn't match
                const uiDateHolder = new Date(date);
                switch (timeUnit) {
                    case 'SECOND': {
                        uiDateHolder.setMilliseconds(0);
                        break;
                    }
                    case 'MINUTE': {
                        uiDateHolder.setMilliseconds(0);
                        uiDateHolder.setSeconds(0);
                        break;
                    }
                    default:
                        break;
                }
                return uiDateHolder;
            } 
            
            $ctrl.getDatePickerConfig = function() {
                let isChangingMonthYear = false; // Lock to avoid infinite recursion performing the updates
                const currentYear = $ctrl.date.getFullYear();
                let datePickerSettings = {
                    changeMonth: true,
                    changeYear: true,
                    yearRange: (currentYear - 100) + ":" + (currentYear + 100),
                    showButtonPanel: true,
                    dateFormat: "yy-mm-dd",
                    onClose: function(raw, obj) {
                        if (isChangingMonthYear) return;
                        $ctrl.date.setYear(obj.selectedYear);
                        $ctrl.date.setMonth(obj.selectedMonth); // selectedMonth is (0-11) :facepalm:
                        $ctrl.date.setDate(obj.selectedDay);
                        $ctrl.uiState.year = obj.selectedYear;
                        $ctrl.uiState.month = obj.selectedMonth;
                        $ctrl.uiState.day = obj.selectedDay;
                        $ctrl.propagateUpdate();
                    },
                    onChangeMonthYear(newYear, newMonth, inst) {
                        if (isChangingMonthYear) return; 
                        $ctrl.date.setYear(newYear);
                        $ctrl.date.setMonth(newMonth - 1); // New month is (1-12)
                        $ctrl.uiState.year = newYear;
                        $ctrl.uiState.month = newMonth - 1;
                        if (inst && inst.input) {
                            isChangingMonthYear = true;
                            inst.input.datepicker("setDate", $ctrl.date);
                            isChangingMonthYear = false;
                        }
                        $ctrl.propagateUpdate();
                    },
                    onUpdateDatepicker: function() {
                        $('.ui-datepicker-prev').append("<i class='dku-icon-arrow-left-16'></i>");
                        $('.ui-datepicker-next').append("<i class='dku-icon-arrow-right-16'></i>");
                    }
                }
                
                switch ($ctrl.timestepParams.timeunit) {
                    case 'MILLISECOND':
                    case 'SECOND':
                    case 'MINUTE':
                    case 'HOUR':
                    case 'DAY':
                        break;
                    case 'BUSINESS_DAY':
                        datePickerSettings = {
                            beforeShowDay: function (date) {
                                // Disables days
                                var day = date.getDay()
                                return [day != 6 && day != 0]
                            },
                            ...datePickerSettings,
                        }
                        break;
                    case 'WEEK':
                        datePickerSettings = {
                            beforeShowDay: function (date) {
                                // Disables days
                                var day = date.getDay()
                                return [day == $ctrl.timestepParams.endOfWeekDay - 1]
                            },
                            showWeek: true,
                            ...datePickerSettings,
                        }
                        break;
                    default: return undefined;
                }
                return datePickerSettings;
            }
            
            function getHoursOptions() {
                const hours = [];
                for (let h = 0; h < 24; h++) {
                    hours.push(h);
                }
                return hours;
            }
            
            function getTimeInputStep() {
                switch ($ctrl.timestepParams.timeunit) {
                    case 'MILLISECOND': return .001;
                    case 'SECOND': return 1;
                    case 'MINUTE': return 60;
                    default: return undefined;
                }
            }
            
            $ctrl.refreshYearsOptions = function() {
                const currentYear = $ctrl.date.getFullYear();
                const startYear = Math.max(1, currentYear - 500); // Prevent year 0 or negative years
                const endYear = currentYear + 500;
                $ctrl.yearsOptions = Array.from({length: endYear - startYear + 1}, (_, i) => startYear + i);
            }
            
            function getQuartersOptions() {
                if ($ctrl.timestepParams.timeunit !== "QUARTER") return undefined;
                switch ($ctrl.timestepParams.unitAlignment) {
                    case 1: return ["January", "April", "July", "October"];
                    case 2: return ["February", "May", "August", "November"];
                    case 3: return ["March", "June", "September", "December"];
                }
            }
            
            function getHalfYearsOptions() {
                if ($ctrl.timestepParams.timeunit !== "HALF_YEAR") return undefined;
                switch ($ctrl.timestepParams.unitAlignment) {
                    case 1: return ["January", "July"];
                    case 2: return ["February", "August"];
                    case 3: return ["March", "September"];
                    case 4: return ["April", "October"];
                    case 5: return ["May", "November"];
                    case 6: return ["June", "December"];
                    default: return undefined;
                }
            }
            
            $ctrl.getViewModeString = function() {
                switch ($ctrl.timestepParams.timeunit) {
                    case 'MILLISECOND': {
                        const dayString = TimeseriesForecastingUtils.getWeekDayName($ctrl.date.getDay() + 1).slice(0, 3);
                        const dateString = `${dayString}, ${$ctrl.date.getDate()} ${$ctrl.MONTHS[$ctrl.date.getMonth()]} ${$ctrl.date.getFullYear()}`
                        const timeString = `${$ctrl.date.toTimeString().split(" ")[0]}`
                        return `${dateString} ${timeString}.${$ctrl.date.getMilliseconds().toString().padStart(3, "0")}`;
                    }
                    case 'SECOND':
                    case 'MINUTE':
                    case 'HOUR': {
                        const dayString = TimeseriesForecastingUtils.getWeekDayName($ctrl.date.getDay() + 1).slice(0, 3);
                        const dateString = `${dayString}, ${$ctrl.date.getDate()} ${$ctrl.MONTHS[$ctrl.date.getMonth()]} ${$ctrl.date.getFullYear()}`
                        const timeString = `${$ctrl.date.toTimeString().split(" ")[0]}`
                        return `${dateString} ${timeString}`;
                    }
                    case 'DAY':
                    case 'BUSINESS_DAY':
                    case 'WEEK':
                        return $ctrl.date.toDateString();
                    case 'MONTH':
                    case 'QUARTER':
                    case 'HALF_YEAR':
                        return $ctrl.MONTHS[$ctrl.date.getMonth()] + ' ' + $ctrl.date.getFullYear();
                    case 'YEAR':
                        return $ctrl.date.getFullYear();
                }
            }
            
            function getTimeInputStringValue(date, timeUnit) {
                let stringValue;
                switch (timeUnit) {
                    case 'SECOND': {
                        stringValue = date.toTimeString().split(" ")[0];
                        break;
                    }
                    case 'MINUTE': {
                        stringValue = date.toTimeString().split(" ")[0].split(":").slice(0,2).join(":");
                        break;
                    }
                    default:
                        stringValue = date.toTimeString().split(" ")[0] + "." + date.getMilliseconds().toString().padStart(3, '0');
                        break;
                }
                return stringValue;
            }
            
            $ctrl.setValue = function() {
                if (!$ctrl.uiState.dateHolder) return;
                const stringValue = getTimeInputStringValue($ctrl.uiState.dateHolder, $ctrl.timestepParams.timeunit);
                $ctrl.date.setSeconds($ctrl.uiState.dateHolder.getSeconds());
                $ctrl.date.setHours($ctrl.uiState.dateHolder.getHours());
                $ctrl.date.setMinutes($ctrl.uiState.dateHolder.getMinutes());
                $ctrl.date.setMilliseconds($ctrl.uiState.dateHolder.getMilliseconds());
                // This is required to not break native time-input validation depending on the `step` input value.
                $timeout(function() { $element.find('input[type="time"]').val(stringValue);});
                $ctrl.propagateUpdate();
            }
            
            $ctrl.propagateUpdate = function() {
                $ctrl.setZeros()
                $ctrl.updateFn();
            }

            $ctrl.setZeros = function() {
                switch ($ctrl.timestepParams.timeunit) {
                    case 'HOUR': {
                        $ctrl.date.setMilliseconds(0);
                        $ctrl.date.setSeconds(0);
                        $ctrl.date.setMinutes(0);
                        break;
                    }
                    case 'DAY':
                    case 'BUSINESS_DAY':
                    case 'WEEK':
                    case 'MONTH':
                    case 'QUARTER':
                    case 'HALF_YEAR':
                    case 'YEAR': {
                        $ctrl.date.setMilliseconds(0);
                        $ctrl.date.setSeconds(0);
                        $ctrl.date.setMinutes(0);
                        $ctrl.date.setHours(0);
                        break;
                    }
                    default: break;
                }
            }
        }
    })

})();
