(function() {
    'use strict';

    const app = angular.module('dataiku.dashboards.insights');

    app.controller('InsightEditController', function($scope, Dialogs, TopNav) {
        TopNav.setLocation(TopNav.TOP_DASHBOARD, 'insights', null, 'edit');
        if ($scope.insight) {
            TopNav.setPageTitle($scope.insight.name + ' - Insight');
        }

        Dialogs.saveChangesBeforeLeaving($scope, $scope.isDirty, $scope.saveInsight, $scope.revertChanges, 'This insight has unsaved changes.');
        Dialogs.checkChangesBeforeLeaving($scope, $scope.isDirty);
    });

    app.directive('insightEditGoToView', function($state) {
        return {
            link: function() {
                $state.go('projects.project.dashboards.insights.insight.view', { location: 'replace', inherit: true });
            }
        };
    });

    app.controller('NewInsightModalController', function($scope, $controller, $stateParams, $q, $filter,
				   DataikuAPI, INSIGHT_TYPES, DashboardUtils, DSSVisualizationThemeUtils, $rootScope) {

        $scope.DashboardUtils = DashboardUtils;
        $scope.insightTypes = INSIGHT_TYPES;
        $scope.displayableInsightType = null;

        $scope.uiState = $scope.uiState || {};
        $scope.uiState.modalTab = 'new';
        $scope.filter = {};

        // Load insight list
        DataikuAPI.dashboards.insights.listWithAccessState($stateParams.projectKey)
            .success(function(data) {
                $scope.insights = data.insights;
                filterInsights();
                $scope.insightAccessMap = angular.extend($scope.insightAccessMap || {}, data.insightAccessData);
            }).error(setErrorInScope.bind($scope));

        $scope.$watch('filter', function(nv, ov) {
            filterInsights();
        }, true);

        $scope.$watch('insight.type', function(nv, ov) {
            filterInsights();
        });

        function filterInsights() {
            $scope.filteredInsights = $filter('filter')($scope.insights, { type: $scope.insight.type });
            $scope.noInsightOfSelectedType = !$scope.filteredInsights || $scope.filteredInsights.length == 0;
            $scope.filteredInsights = $filter('filter')($scope.filteredInsights, { name: $scope.filter.q });
            if ($scope.filter.sourceId) {
                $scope.filteredInsights = $scope.filteredInsights.filter(function(insight) {
                    return DashboardUtils.getInsightSourceId(insight) == $scope.filter.sourceId;
                });
            }
        }

        function isFiltersColumnsSelectionStep() {
            return $scope.displayableInsightType && $scope.displayableInsightType.toLowerCase() === 'filters' && $scope.uiState.source;
        };

        // Insights types that share the same create-form directive
        const INSIGHT_TYPE_GROUPS = {
            'scenario_last_runs': 'scenario',
            'scenario_run_button': 'scenario'
        };

        $scope.getInsightTypeGroup = function(insightType) {
            return INSIGHT_TYPE_GROUPS[insightType] || insightType;
        };

        $scope.simpleTileTypes = {
            text: { name: 'Text', desc: 'Zone of text' },
            image: { name: 'Image', desc: 'Upload an image' },
            iframe: { name: 'Web Content', desc: 'Embedded web page' },
            group: { name: 'Group', desc: 'Tile that can contain others' }
        };

        $scope.setDashboardCreationId = function(dashboardId) {
            $scope.insight.dashboardCreationId = dashboardId;
        };

        $scope.getModalTitle = function() {
            if ($scope.inDashboard) {
                const insightType = $scope.displayableInsightType ? $scope.displayableInsightType.toLowerCase() : 'tile';
                return insightType === 'filters' ? 'Add filters to slide' : 'Add ' + insightType + ' to dashboard';
            }
            return $scope.displayableInsightType ? 'New ' + $scope.displayableInsightType + ' insight' : 'New insight';
        };

        $scope.getModalStepHeaderTitle = function() {
            return $scope.displayableInsightType.toLowerCase() === 'filters' ?
                'Select origin of the source dataset' : 'Type selection';
        };

        $scope.isFiltersSourceSelectionStep = function() {
            return $scope.displayableInsightType && $scope.displayableInsightType.toLowerCase() === 'filters' && !$scope.uiState.source;
        };

        $scope.resetModal = function() {
            if (isFiltersColumnsSelectionStep() && $scope.filtersCompatibleTiles.length) {
                $scope.uiState.source = null;
                return; // return to filters source selection
            }

            const insightTheme = $scope.dashboard ? $scope.dashboard.theme : $rootScope.appConfig.selectedDSSVisualizationTheme;
            $scope.insight = {
                projectKey: $stateParams.projectKey,
                params: {
                    theme: insightTheme
                }
            };

            $scope.displayableInsightType = null;

            $scope.filter = {};

            $scope.hook = {
                //Can be overwritten by {{insightType}}InsightCreateForm directives
                beforeSave: function(resolve, reject) {
                    resolve();
                },

                //Can be overwritten by {{insightType}}InsightCreateForm directives
                afterSave: function(resolve, reject) {
                    resolve();
                },

                defaultName: null,
                sourceObject: {},
                setErrorInModaleScope : function(data, status, headers, config, statusText) {
                    setErrorInScope.bind($scope)(data, status, headers, config, statusText);
                }
            };
        };

        $scope.resetModal();

        $scope.returnSimpleTile = function(tileType) {
            $scope.resolveModal({
                tileType: tileType
            });
        };

        $scope.selectType = function(type) {
            $scope.insight.type = type;
            $scope.canPickExistingInsight = DashboardUtils.canListInsight($scope.insight);
            $scope.displayableInsightType = DashboardUtils.getInsightHandler(DashboardUtils.getInsightTypeGroup(type)).name;
            if (type === 'static_file') {
                $scope.pointerMode.isPointerMode = true;
            }
        };

        function beforeSavePromise() {
            const deferred = $q.defer();
            $scope.hook.beforeSave(deferred.resolve, deferred.reject);
            return deferred.promise;
        }

        function afterSavePromise() {
            const deferred = $q.defer();
            $scope.hook.afterSave(deferred.resolve, deferred.reject);
            return deferred.promise;
        }

        $scope.create = function() {
            if (!$scope.insight.name) {
                $scope.insight.name = $scope.hook.defaultName;
            }

            beforeSavePromise().then(
                function() {
                    function save() {
                        DataikuAPI.dashboards.insights.save($scope.insight)
                            .error(setErrorInScope.bind($scope))
                            .success(function(insightId) {
                                $scope.insight.id = insightId;
                                if ($scope.hook.sourceObject && !$scope.hook.noReaderAuth) {
                                    $scope.insight.isReaderAccessible = $scope.hook.sourceObject.isReaderAccessible;
                                } else {
                                    $scope.insight.isReaderAccessible = true;
                                }
                                $scope.resolveModal({ insight: $scope.insight, redirect: $scope.hook.redirect });

                                const insightIsAPointer = $scope.$modalScope.pointerMode && $scope.$modalScope.pointerMode.isPointerMode || false;
                                const createdFromDashboard = !!$scope.$modalScope.inDashboard;
                                if (!insightIsAPointer) {
                                    DashboardUtils.sendWT1InsightCreation($scope.insight.type, {
                                        triggeredFrom: createdFromDashboard ? 'dashboard-new-tile-modal' : 'insight-list-new-insight-modal'
                                    });
                                }

                                afterSavePromise().then(
                                    function() {
                                        //nothing specific to do in case of success
                                    },
                                    function(data, status, headers, config, statusText) {
                                        setErrorInScope.bind($scope)(data, status, headers, config, statusText);
                                    }
                                );
                            });
                    }

                    if ($scope.hook.addSourceToReaderAuthorizations) {
                        const neededReaderAuthorization = DashboardUtils.getNeededReaderAuthorization($scope.insight);

                        DataikuAPI.projects.addReaderAuthorizations($stateParams.projectKey, [neededReaderAuthorization])
                            .success(function() {
                                $scope.hook.sourceObject.isReaderAccessible = true;
                                save();

                                // For model evaluation insights, the referenced SM must be authorized also
                                if ($scope.insight.type === 'model-evaluation_report') {
                                    DataikuAPI.modelevaluationstores.getLatestEvaluationDetails($scope.insight.projectKey, $scope.insight.params.mesSmartId)
                                        .success(function(data) {
                                            if (data.evaluation && data.evaluation.hasModel) {
                                                const smResolved = resolveObjectSmartId(data.evaluation.modelRef.smId);
                                                const ref = {
                                                    objectType: 'SAVED_MODEL',
                                                    objectId: smResolved.id
                                                };
                                                if (smResolved.projectKey) {
                                                    ref.projectKey = smResolved.projectKey;
                                                }
                                                DataikuAPI.projects.addReaderAuthorizations($stateParams.projectKey, [{ objectRef: ref, modes: ['READ'] }])
                                                    .success(function() {
                                                        save();
                                                    })
                                                    .error(setErrorInScope.bind($scope));
                                            }
                                        })
                                        .error(setErrorInScope.bind($scope));
                                }
                            })
                            .error(setErrorInScope.bind($scope));
                    } else {
                        save();
                    }
                },
                function(argArray) {
                    if (argArray) {
                        setErrorInScope.bind($scope).apply(null, argArray);
                    }
                }
            );
        };
    });

    app.directive('insightSourceInfo', function(DashboardUtils) {
        return {
            templateUrl: 'templates/dashboards/insights/insight-source-info.html',
            link: function($scope, element, attrs) {

                if (!$scope.inDashboard) {
                    return;
                }

                function updateMatches() {
                    if (!$scope.insights) {
                        return;
                    }
                    const handler = DashboardUtils.getInsightHandler($scope.insight.type);
                    $scope.matching = $scope.insights.filter(function(insight) {
                        return $scope.getInsightTypeGroup(insight.type) == $scope.getInsightTypeGroup($scope.insight.type)
							&& handler.getSourceId(insight) == handler.getSourceId($scope.insight)
							&& (handler.sourceType || (handler.getSourceType(insight) == handler.getSourceType($scope.insight)));
                    });
                }

                $scope.go = function() {
                    const handler = DashboardUtils.getInsightHandler($scope.insight.type);
                    $scope.filter.sourceId = handler.getSourceId($scope.insight);
                    $scope.filter.sourceType = handler.sourceType || handler.getSourceType($scope.insight);
                    $scope.uiState.modalTab = 'existing';
                };

                $scope.$watch('insight', updateMatches, true);
            }
        };
    });


    app.controller('CopyInsightModalController', function($scope, DataikuAPI, ActivityIndicator, StateUtils, $stateParams, DashboardUtils) {
        $scope.init = function(insight) {
            $scope.insight = insight;
            $scope.insight.newName = 'Copy of ' + insight.name;
        };

        $scope.copy = function() {
            DataikuAPI.dashboards.insights.copy($stateParams.projectKey, [$scope.insight.id], [$scope.insight.newName])
                .error(setErrorInScope.bind($scope))
                .success(function(data) {
                    const insightCopy = data[0];
                    const href = StateUtils.href.insight(insightCopy.id, insightCopy.projectKey, { name: insightCopy.name });
                    ActivityIndicator.success($scope.insight.name + ' copied into ' + insightCopy.name + ', <a href=\'' + href + '\' >view insight</a>.', 5000);
                    $scope.resolveModal();
                    DashboardUtils.sendWT1InsightCreation($scope.insight.type, {
                        triggeredFrom: 'insight-list-copy-tile-modal'
                    });
                });
        };
    });


    app.controller('MoveCopyTileModalController', function($scope, DataikuAPI, $controller, StateUtils, $rootScope, $timeout, TileUtils, $stateParams, DashboardUtils) {
        $controller('MultiPinInsightModalController', { $scope:$scope });

        $scope.keepOriginal = true;
        $scope.pointerMode = {
            mode: false
        };

        const initCopy = $scope.init;
        $scope.init = function(insight, tileParams) {
            if ($scope.insight) {
                initCopy(insight, tileParams);
            } else {
                //creating new tile
                $scope.newTile = TileUtils.copyNonInsightTile($scope.tile);
                //listing dashboards
                $scope.listDashboards($scope.insight);
            }
        };

        /*
         * Methods used if copying/moving tile to other dashboard
         */
        $scope.addPinningOrder = function() {
            for (let i = 0; i < $scope.dashboards.length; i++) {
                if ($scope.dashboards[i].id == $scope.dashboard.id) {
                    $scope.pinningOrder = {
                        dashboard: $scope.dashboards[i],
                        page: $scope.dashboards[i].pages[$scope.uiState.currentPageIdx],
                        useInsightTheme: false
                    };
                    $scope.pinningOrders = [$scope.pinningOrder];
                    break;
                }
            }
        };

        $scope.multiPinCallback = function() {
            removeOriginalIfNeeded();
            $timeout(function() {
                const pin = $scope.pinningOrders[0];
                const options = {
                    name: pin.dashboard.name,
                    tab: 'edit',
                    pageId: pin.page.id
                };
                StateUtils.go.dashboard(pin.dashboard.id, $stateParams.projectKey, options).then(function() {
                    const unregister = $rootScope.$on('dashboardSyncModelsDone', () => {
                        unregister();
                        $rootScope.$broadcast('dashboardSelectLastTile');
                    });
                });
            });
        };

        /*
         * Methods used if copying/moving the tile in the current dashboard
         */

        function moveCopyTile(destinationPageId) {
            let destinationPage = null;
            for (let i=0; i<$scope.dashboard.pages.length; i++) {
                if ($scope.dashboard.pages[i].id == destinationPageId || $scope.dashboard.pages[i].$$hashKey == destinationPageId) {
                    destinationPage = $scope.dashboard.pages[i];
                    break;
                }
            }

            function moveCopyTileFront(insightId) {
                const copyTile = angular.copy($scope.tile);
                delete copyTile.$added;
                delete copyTile.$tileId;
                delete copyTile.box.top;
                delete copyTile.box.left;

                if (insightId) {
                    copyTile.insightId = insightId;
                }


                DashboardUtils.sendWT1TileCreation(copyTile.tileType, {
                    insightType: copyTile.insightType,
                    reuseExistingInsight: $scope.pointerMode.mode,
                    triggeredFrom: 'move-or-copy-tile-modal'
                });

                destinationPage.grid.tiles.push(copyTile);
                removeOriginalIfNeeded();
                $scope.uiState.currentPageIdx = $scope.dashboard.pages.indexOf(destinationPage);
                $scope.uiState.selectedTile = copyTile;
            }

            if ($scope.tile.tileType == 'INSIGHT' && $scope.keepOriginal && !$scope.pointerMode.mode) {
                DataikuAPI.dashboards.insights.copy($stateParams.projectKey, [$scope.tile.insightId], [$scope.insightName], $scope.dashboard.id)
                    .error(setErrorInScope.bind($scope))
                    .success(function(data) {
                        const insightCopy = data[0];
                        $scope.insightsMap[insightCopy.id] = insightCopy;
                        DashboardUtils.sendWT1InsightCreation(insightCopy.type, {
                            triggeredFrom: 'move-or-copy-tile-modal'
                        });
                        moveCopyTileFront(insightCopy.id);
                        $scope.dismiss();
                    });
            } else {
                moveCopyTileFront();
                $scope.dismiss();
            }
        }

        function removeOriginalIfNeeded() {
            if (!$scope.keepOriginal) {
                $scope.removeTileFromDashboardPage($scope.tile.$tileId);
            }
        }

        /*
         * Forms methods
         */

        $scope.validate = function() {
            if ($scope.pinningOrder.dashboard.id == $scope.dashboard.id) {
                moveCopyTile($scope.pinningOrder.page.id || $scope.pinningOrder.page.$$hashKey);
            } else {
                $scope.sendPinningOrders();
            }
        };

        $scope.checkConsistency = function() {
            if ($scope.keepOriginal) {
                $scope.pointerMode.mode = false;
            }
        };

        $scope.getPagesList = function() {
            if (!$scope.pinningOrder) {
                return [];
            }

            if (!$scope.keepOriginal) {
                return $scope.pinningOrder.dashboard.pages.filter(function(page) {
                    return $scope.page.id != page.id;
                });
            } else {
                return $scope.pinningOrder.dashboard.pages;
            }
        };
    });


    app.controller('_MultiPinInsightModalCommonController', function($scope, $timeout, DataikuAPI, TileUtils, DatasetUtils, $stateParams, DashboardUtils, CreateAndPinInsight, translate) {
        $scope.existingPinningList = [];
        $scope.pinningOrders = [];
        $scope.initiated = false;
        $scope.hasSampledChartInsightSelected = false;
        $scope.dashboardNamePlaceholder = DashboardUtils.getDashboardDefaultName();
        $scope.dummyPageModel = { ...CreateAndPinInsight.getPageTemplate, title: translate('INSIGHTS.MODAL.DUMMY_PAGE_PLACEHOLDER', 'Page 1') };
        // Warning: Sets are not supported by AngularJS! Do not use it in AngularJS scope, templates, or inside methods like `angular.equals`, JSON.stringify...
        const sampledChartInsightSelected = new Set();

        /*
         * Form initialization
         */

        $scope.init = function(insight, tileParams, payload) {
            //create new tile
            $scope.insight = insight;
            $scope.newTile = $scope.initNewTile(insight, tileParams);
            $scope.payload = payload;

            // list dashboards where we could copy it
            $scope.listDashboards(insight);
        };

        $scope.initNewTile = function(insight, tileParams) {
            const newTile = TileUtils.newInsightTile(insight);
            if (tileParams) {
                angular.extend(newTile.tileParams, tileParams);
            }

            if ($scope.tile) {
                newTile.box = angular.copy($scope.tile.box);
            }

            newTile.box.left = -1;
            newTile.box.top = -1;

            return newTile;
        };

        function getChartSampleMetadataRequest(chartInsight){
            const inputDatasetLoc = DatasetUtils.getLocFromSmart($stateParams.projectKey, chartInsight.params.datasetSmartName);
            const dataSpec = {
                datasetProjectKey : inputDatasetLoc.projectKey,
                datasetName : inputDatasetLoc.name,
                script: angular.copy($scope.shaker),
                copySelectionFromScript: chartInsight.params.copySelectionFromScript,
                sampleSettings : chartInsight.params.refreshableSelection,
                engineType : chartInsight.params.engineType
            };
            dataSpec.script.origin = 'DATASET_EXPLORE';
            const sampleId = chartInsight.params.summary && chartInsight.params.summary.requiredSampleId || null;
            return { dataSpec, sampleId };
        }

        $scope.insightSelectionChange = function(insightDataItem){
            if (insightDataItem.selected && insightDataItem.sampleMetadata && insightDataItem.sampleMetadata.sampleIsWholeDataset === false){
                sampledChartInsightSelected.add(insightDataItem);
            } else {
                sampledChartInsightSelected.delete(insightDataItem);
            }
            $scope.hasSampledChartInsightSelected = sampledChartInsightSelected.size > 0;
        };

        $scope.fetchChartInsightsMetaData = function(chartInsights){
            const chartsData = chartInsights.map(chartInsight => getChartSampleMetadataRequest(chartInsight));
            DataikuAPI.shakers.charts.getSamplesMetaData($stateParams.projectKey, chartsData)
                .error(setErrorInScope.bind($scope))
                .success(function(samplesMetadata) {
                    samplesMetadata = samplesMetadata || [];
                    $scope.insightData.items.forEach((item, i) => {
                        item.sampleMetadata = samplesMetadata.length > i ? samplesMetadata[i] : {};
                        $scope.insightSelectionChange(item);
                    });
                });
        };

        $scope.listDashboards = function(insight){
            DataikuAPI.dashboards.listEditable($stateParams.projectKey)
                .error(setErrorInScope.bind($scope))
                .success(function(data) {
                    $scope.dashboards = data.dashboards;
                    $scope.allDashboardsCount = data.allDashboardsCount;
                    //listing where insight got already pinned
                    if (insight && insight.id) {
                        $scope.dashboards.forEach(function(dashboard, index) {
                            dashboard.pages.forEach(function(page, pageIndex) {
                                page.index = pageIndex;
                                page.grid.tiles.forEach(function(tile) {
                                    if (tile.insightId == insight.id) {
                                        $scope.existingPinningList.push({
                                            'dashboard': dashboard,
                                            'page': page
                                        });
                                    }
                                });
                            });
                            // Use local state of current dashboard to take into account slides being added / removed
                            if ($scope.dashboard && $scope.dashboard.id === dashboard.id) {
                                $scope.dashboards[index] = $scope.dashboard;
                            }
                        });
                    }
                    if ($scope.dashboards.length > 0) {
                        $scope.addPinningOrder();
                    }
                    $scope.dashboardOptions = [
                        ...$scope.dashboards,
                        CreateAndPinInsight.getCreateDashboardOption()
                    ];
                    $scope.pageOptionsByDashboardId = Object.fromEntries($scope.dashboards.map(({ id, pages }) => ([ id, [...pages, CreateAndPinInsight.getCreatePageOption()]])));
                    $scope.initiated = true;
                });
        };

        $scope.onDashboardSelected = function(pinningOrder, index) {
            // If the option that has been selected is the "Create new dashboard one".
            if (!pinningOrder.dashboard.id) {
                const dashboardToCreate = CreateAndPinInsight.getDashboardTemplate();
                pinningOrder.dashboard = dashboardToCreate;
                pinningOrder.page = dashboardToCreate.pages[0];
                //  Focus the last "Create new dashboard" input.
                $timeout(() => document.querySelector(`.multi-pin-insight-table .dashboard-and-slide-selection:nth-child(${index + 1}) .dashboard-and-slide-selection__dashboard .dashboard-and-slide-selection__create-new-item`).focus(), 100);
            } else {
                pinningOrder.page = pinningOrder.dashboard.pages[0];
            }
        };

        $scope.onPageSelected = function(pinningOrder, index) {
            if (pinningOrder.page.id) {
                return;
            }
            const newPage = CreateAndPinInsight.getPageTemplate();
            pinningOrder.page = newPage;
            $timeout(() => {
                const inputElement = document.querySelector(`.multi-pin-insight-table tr:nth-child(${index + 1}) .dashboard-and-slide-selection__page .dashboard-and-slide-selection__create-new-item`);
                if (!_.isNil(inputElement)) {
                    inputElement.focus();
                }
            });
        };

        /*
         * PinningOrder CRUD
         */

        $scope.addPinningOrder = function() {
            const projectHasDashboards = $scope.dashboards && $scope.dashboards.length > 0;
            $scope.pinningOrders.push({
                dashboard: projectHasDashboards ? $scope.dashboards[0] : CreateAndPinInsight.getDashboardTemplate(),
                page: projectHasDashboards ? $scope.dashboards[0].pages[0] : CreateAndPinInsight.getPageTemplate(),
                useInsightTheme: !$scope.dashboards[0].theme
            });
        };

        $scope.removePinningOrder = function(index) {
            $scope.pinningOrders.splice(index, 1);
        };

        $scope.getLightPinningOrders = function(pinningOrders = $scope.pinningOrders) {
            const lightPinningOrders = [];
            pinningOrders.forEach(function(pinningOrder) {
                lightPinningOrders.push({
                    dashboardId: pinningOrder.dashboard.id,
                    pageId: pinningOrder.page.id,
                    useInsightTheme: pinningOrder.useInsightTheme
                });
            });
            return lightPinningOrders;
        };

        /*
         * UI
         */

        const pagesLabels = {};
        $scope.getPageLabel = function(dashboard, page) {
            if (page.id && pagesLabels[page.id]) {
                return pagesLabels[page.id];
            }
            let pageLabel = '';
            if (page.title) {
                pageLabel = page.title;
            } else {
                pageLabel = 'Page ' + (dashboard.pages.indexOf(page) + 1);
            }
            if (page.id) {
                pagesLabels[page.id] = pageLabel;
            }
            return pageLabel;
        };
    });


    app.controller('MultiPinInsightModalController', function($scope, DataikuAPI, ActivityIndicator, CreateAndPinInsight, $controller, $stateParams, DSSVisualizationThemeUtils, DashboardUtils) {
        $controller('_MultiPinInsightModalCommonController', { $scope:$scope });

        const initCopy = $scope.init;
        $scope.init = function(insight, tileParams) {
            initCopy(insight, tileParams);
            $scope.pointerMode = {
                mode: insight.type == 'static_file'
            };
        };

        $scope.sendPinningOrders = function() {
            CreateAndPinInsight.createPinningOrdersDashboardAndPages($scope.pinningOrders, setErrorInScope.bind($scope)).subscribe((updatedPinningOrders) => {
                const lightPinningOrders = $scope.getLightPinningOrders(updatedPinningOrders);

                const newTiles = Array.from({ length: updatedPinningOrders.length }, () => _.cloneDeep($scope.newTile));

                for (let tileIndex = 0; tileIndex < newTiles.length; tileIndex++) {
                    DSSVisualizationThemeUtils.applyToTile(newTiles[tileIndex], updatedPinningOrders[tileIndex].dashboard.theme);
                }

                DataikuAPI.dashboards.multiPin($stateParams.projectKey, $scope.insight.id, newTiles, lightPinningOrders, $scope.pointerMode.mode)
                    .error(setErrorInScope.bind($scope))
                    .success(function() {
                        ActivityIndicator.success('Saved!');
                        lightPinningOrders.forEach(() => {
                            if (!$scope.pointerMode.mode) {
                                DashboardUtils.sendWT1InsightCreation($scope.insight.type, {
                                    triggeredFrom: 'insight-list-add-to-dashboard-modal'
                                });
                            }
                            DashboardUtils.sendWT1TileCreation('INSIGHT', {
                                insightType: $scope.insight.type,
                                reuseExistingInsight: $scope.pointerMode.mode,
                                triggeredFrom: 'insight-list-add-to-dashboard-modal'
                            });
                        });
                        if ($scope.multiPinCallback && typeof($scope.multiPinCallback)==='function') {
                            $scope.multiPinCallback();
                        }
                        $scope.resolveModal();
                    });
            });
        };
    });

    app.controller('_CreateAndPinInsightModalCommonController', function($scope, DataikuAPI, ActivityIndicator, TileUtils, $controller, $timeout, StateUtils, $rootScope, DashboardUtils, $stateParams, WT1, CreateAndPinInsight, DSSVisualizationThemeUtils) {
        $controller('_MultiPinInsightModalCommonController', { $scope:$scope });

        $scope.missingReaderAuthorizations = [];
        $scope.addReaderAuthorization = $scope.projectSummary.canManageDashboardAuthorizations;

        $scope.authorize = function(insights) {
            const neededReaderAuthorizations = insights.map(_ => DashboardUtils.getNeededReaderAuthorization(_));

            DataikuAPI.projects.checkReaderAuthorizations($stateParams.projectKey, neededReaderAuthorizations)
                .error(setErrorInScope.bind($scope))
                .success(function(data) {
                    $scope.missingReaderAuthorizations = data;
                });
        };

        $scope.sendCreateAndPinOrders = function(insights, newTiles, payloads) {
            function save() {
                CreateAndPinInsight.createPinningOrdersDashboardAndPages($scope.pinningOrders, setErrorInScope.bind($scope)).subscribe((updatedPinningOrders) => {
                    const lightPinningOrders = $scope.getLightPinningOrders(updatedPinningOrders);

                    const tileList = Array.from({ length: updatedPinningOrders.length }, () => _.cloneDeep(newTiles));

                    for (let pinningOrdersIndex = 0; pinningOrdersIndex < updatedPinningOrders.length; pinningOrdersIndex++) {
                        for (let insightIndex = 0; insightIndex < insights.length; insightIndex++) {
                            DashboardUtils.sendWT1TileCreation('INSIGHT', {
                                insightType: insights[insightIndex].type,
                                triggeredFrom: 'publish-to-dashboard-modal'
                            });
                            DSSVisualizationThemeUtils.applyToTile(tileList[pinningOrdersIndex][insightIndex], updatedPinningOrders[pinningOrdersIndex].dashboard.theme);
                        }
                    }

                    DataikuAPI.dashboards.insights.createAndPin($stateParams.projectKey, insights, tileList, lightPinningOrders, payloads)
                        .error(setErrorInScope.bind($scope))
                        .success(function(data) {
                            insights.forEach(insight => {
                                if (insight.type === 'chart') {
                                    WT1.event('chart-publish', {
                                        chartId: `${insight.projectKey.dkuHashCode()}.${insight.params.datasetSmartName.dkuHashCode()}.${insight.params.def.name.dkuHashCode()}`,
                                        chartType: insight.params.def.type,
                                        chartVariant: insight.params.def.variant
                                    });
                                }
                                DashboardUtils.sendWT1InsightCreation(insight.type, {
                                    triggeredFrom: 'publish-to-dashboard-modal'
                                });
                            });


                            ActivityIndicator.success('Saved!');
                            if (updatedPinningOrders.length === 0) {
                                StateUtils.go.insight(data[0], $stateParams.projectKey, { name: insights[0].name });
                            } else {
                                const pin = updatedPinningOrders[0];
                                const options = {
                                    name: pin.dashboard.name,
                                    tab: 'edit',
                                    pageId: pin.page.id
                                };
                                StateUtils.go.dashboard(pin.dashboard.id, $stateParams.projectKey, options).then(function() {
                                    // Save the new dashboard layout to avoid dirty state
                                    const unregister = $rootScope.$on('dashboardSyncModelsDone', () => {
                                        unregister();
                                        $rootScope.$broadcast('dashboardSelectLastTile');
                                        $timeout(() => $rootScope.$broadcast('saveDashboard'));
                                    });
                                });
                            }
                            $scope.resolveModal();
                        });
                });
            }

            if ($scope.addReaderAuthorization && $scope.missingReaderAuthorizations) {
                DataikuAPI.projects.addReaderAuthorizations($stateParams.projectKey, $scope.missingReaderAuthorizations)
                    .success(save)
                    .error(setErrorInScope.bind($scope));
            } else {
                save();
            }
        };
    });

    app.controller('CreateAndPinInsightModalController', function($scope, DataikuAPI, ActivityIndicator, TileUtils, $controller, $timeout, StateUtils, $rootScope, DashboardUtils, $stateParams) {
        $controller('_CreateAndPinInsightModalCommonController', { $scope:$scope });

        const initCopy = $scope.init;
        $scope.init = function(insight, tileParams, payload) {
            initCopy(insight, tileParams, payload);
            $scope.authorize([insight]);
            $scope.hasSampledInsight = false;
        };

        const sendOrders = $scope.sendCreateAndPinOrders;
        $scope.sendCreateAndPinOrders = function() {
            sendOrders([$scope.insight], [$scope.newTile], [$scope.payload]);
        };
    });

    app.controller('CreateAndPinInsightsModalController', function($scope, DataikuAPI, ActivityIndicator, TileUtils, $controller, $timeout, ChartsStaticData, StateUtils, $rootScope, DashboardUtils, $stateParams, CreateAndPinInsight) {
        $controller('_CreateAndPinInsightModalCommonController', { $scope });

        $scope.insights = [];
        $scope.newTiles = [];
        $scope.insightData = [];
        $scope.warningBadge = ChartsStaticData.WARNING_BADGE;

        $scope.init = function(insights, tileParams) {
            //create new tile
            insights.forEach(insight => {
                $scope.insights.push(insight);
                $scope.newTiles.push($scope.initNewTile(insight, tileParams));
            });

            const areChartInsights = insights.length && insights[0].type === 'chart';
            if (areChartInsights) {
                $scope.fetchChartInsightsMetaData(insights);
            }
            // list dashboards where we could copy it
            $scope.listDashboards($scope.insights[0]);
            $scope.authorize(insights);
        };

        const sendOrders = $scope.sendCreateAndPinOrders;
        $scope.sendCreateAndPinOrders = function() {
            const insights = $scope.insights.filter((_, i) => $scope.insightData.items[i].selected);
            const newTiles = $scope.newTiles.filter((_, i) => $scope.insightData.items[i].selected);

            sendOrders(insights, newTiles);
        };

        $scope.canCreate = function() {
            return $scope.insightData.items && $scope.insightData.items.some(_ => _.selected);
        };
    });
})();
