(function() {
    'use strict';

    const app = angular.module('dataiku.dashboards.insights');

    app.constant('ModelEvaluationReportInsightHandler', {
        name: 'Model evaluation report',
        desc: 'Full report of a model evaluation',
        icon: 'icon-model-evaluation-store',
        color: 'model-evaluation-store',
        getSourceId: function(insight) {
            return insight.params.mesSmartId;
        },
        sourceType: 'MODEL_EVALUATION_STORE',
        hasEditTab: false,
        defaultTileParams: {
            displayMode: 'summary'
        },
        defaultTileDimensions: [24, 12]
    });

    app.controller('ModelEvaluationReportViewCommon', function($scope, DataikuAPI, $controller, FullModelLikeIdUtils, WebAppsService, $state, $stateParams) {
        $scope.readOnly = true;
        $scope.noUrlChange = true;

        $scope.isClassicalPrediction = function() {
            if (!$scope.modelData) {
                return false;
            }
            return ['BINARY_CLASSIFICATION', 'REGRESSION', 'MULTICLASS'].includes($scope.modelData.coreParams.prediction_type);
        };

        $scope.getModelEvaluation = function(onLoadError) {
            const p = DataikuAPI.modelevaluationstores
                .get($scope.insight.projectKey, $scope.insight.params.mesSmartId)
                .success(function(data) {
                    $scope.modelEvaluationStore = data;
                    $scope.readOnly = true;
                    $scope.uiState = {};
                    $scope.noSetLoc = true;
                    $scope.mesContext = {};
                    $scope.versionsContext = {};
                    const getDetailsP = DataikuAPI.modelevaluationstores
                        .getLatestEvaluationDetails($scope.insight.projectKey, $scope.insight.params.mesSmartId)
                        .success(function(data) {
                            /*
                             * In the case of dashboards, the state does not contain the mesId nor
                             * the evaluationId (and you can see that it’s not in the URL), so we have
                             * to rely on the scope in those cases
                             */
                            $scope.$modelEvaluation = data;
                            $scope.mesId = $scope.insight.params.mesSmartId;
                            $scope.evaluationId = data.evaluation.ref.evaluationId;
                            $scope.modelData = data.details;
                            const isLLMEvaluation = $scope.$modelEvaluation.evaluation && $scope.$modelEvaluation.evaluation.type === 'llm';
                            if (isLLMEvaluation) {
                                $controller('_LLMPredictionModelReportController', { $scope });
                            } else {
                                $controller('_PredictionModelReportController', { $scope });
                            }
                            $controller('ModelEvaluationStoreBaseEvaluationController', { $scope });
                        })
                        .error(setErrorInScope.bind($scope));
                    if ($scope.noSpinner) {
                        getDetailsP.noSpinner();
                    }
                })
                .error(function(data, status, headers, config, statusText) {
                    if (typeof onLoadError === 'function') {
                        onLoadError(data, status, headers, config, statusText);
                    }
                });

            if ($scope.noSpinner) {
                p.noSpinner();
            }
        };
    });

    app.directive('modelEvaluationReportInsightTile', function($controller, DashboardUtils, TileLoadingState, SavedModelsService) {
        return {
            templateUrl: '/templates/dashboards/insights/model-evaluation_report/model-evaluation_report_tile.html',
            scope: {
                insight: '=',
                tile: '=',
                hook: '='
            },
            link: function($scope) {
                $scope.load = function(resolve, reject) {
                    $scope.loading = true;
                    $scope.onLoadSuccess = DashboardUtils.setLoaded.bind([$scope, resolve]);
                    $scope.onLoadError = DashboardUtils.setError.bind([$scope, reject]);
                    $scope.noSpinner = true;
                    $scope.getModelEvaluation($scope.onLoadError);
                };
                $scope.hook.loadPromises[$scope.tile.$tileId] = $scope.load;
                $scope.hook.reloadPromises[$scope.tile.$tileId] = $scope.load;

                $controller('ModelEvaluationReportViewCommon', { $scope: $scope });

                $scope.$watch('modelData', function(nv) {
                    if (!nv) {
                        return;
                    }
                    $scope.insight.$modelData = nv;
                });

                $scope.$watch('evaluationDetails', function(nv) {
                    if (!nv) {
                        return;
                    }
                    $scope.insight.$evaluationDetails = nv;
                    $scope.insight.$isLLMEvaluation = nv.evaluation && nv.evaluation.type === 'llm';
                });

                if ($scope.tile.autoLoad) {
                    $scope.hook.loadStates[$scope.tile.$tileId] = TileLoadingState.WAITING;
                }
            }
        };
    });

    app.controller('ModelEvaluationReportInsightController', function($scope, $controller, $stateParams, DataikuAPI, $state, MLRoutingService) {
        const routePrefix = 'projects.project.dashboards.insights.insight.view.modelevaluations';

        $controller('ModelEvaluationReportViewCommon', { $scope: $scope });

        $scope.isLLMEvaluation = function() {
            return $scope.modelEvaluation && $scope.modelEvaluation.type === 'llm';
        };

        const goToSummary = () => {
            goToTab(MLRoutingService.getPredictionReportSummaryTab(false, $scope.isLLMEvaluation()), null);
        };

        const goToTab = (tab, params) => {
            $state.go(`${routePrefix}.prediction.${tab}`, params, { location: 'replace' });
        };

        $scope.$on('$stateChangeStart', function(event, toState, toParams, fromState, fromParams) {
            if (!$scope.modelData) {
                return;
            }
            MLRoutingService.goToRouteSuffixIfBase(
                $state,
                toState,
                toParams,
                fromState,
                fromParams,
                'projects.project.dashboards.insights.insight.view',
                `modelevaluations.prediction.${MLRoutingService.getPredictionReportSummaryTab(false, $scope.isLLMEvaluation())}`,
                event
            );
        });

        $scope.getModelEvaluation(setErrorInScope.bind($scope));
        $scope.$watch('evaluationDetails', function(nv, ov) {
            if (!nv) {
                return;
            }
            if ($state.current.name === routePrefix) {
                goToSummary();
            }

            if ($scope.originDashboardStateParams) {
                DataikuAPI.dashboards.getFullInfo($stateParams.projectKey, $scope.originDashboardStateParams.dashboardId).success(function(data) {
                    const tile = data.dashboard.pages
                        .find(page => page.id === $scope.originDashboardStateParams.pageId)
                        .grid.tiles.find(tile => tile.insightId === $stateParams.insightId);
                    const pane = MLRoutingService.dashboardTileToSavedModelPane(tile, false, $scope);
                    goToTab(pane.route, pane.routeParams);
                });
            }
        });

        $scope.onLoadError = function(data, status, headers, config, statusText) {
            setErrorInScope.bind($scope)(data, status, headers, config, statusText);
        };
    });

    app.directive('modelEvaluationReportInsightView', function($controller, $stateParams, DataikuAPI, $state) {
        return {
            scope: true,
            link: function($scope, element, attrs) {
                if ($state.current.name === 'projects.project.dashboards.insights.insight.view') {
                    $state.go('.modelevaluations', null, { location: 'replace' });
                }
            }
        };
    });

    app.directive('llmModelEvaluationReportInsightTileParams', function($controller, $timeout, DataikuAPI, FullModelLikeIdUtils, $q, ModelDataUtils, $stateParams) {
        return {
            templateUrl: '/templates/dashboards/insights/model-evaluation_report/llm-model-evaluation_report_tile_params.html',
            scope: {
                tileParams: '=',
                insight: '='
            },
            link: function($scope, $element, attrs) {
                $scope.ModelDataUtils = ModelDataUtils;

                $scope.$watch('insight.$evaluationDetails', function(nv) {
                    if (!nv) {
                        return;
                    }
                    $scope.evaluationDetails = nv;

                    $controller('ModelEvaluationReportViewCommon', { $scope: $scope });

                    $controller('_LLMPredictionModelReportController', { $scope: $scope });

                    $timeout(function() {
                        $element.find('.view-select').selectpicker('refresh');
                    });

                    $scope.onDisplayModeExtendedChange = function() {
                        if (!$scope.tileParams || !$scope.tileParams.$displayModeExtended) {
                            return;
                        }
                        const displayModeExtended = $scope.tileParams.$displayModeExtended;
                        $scope.tileParams.displayMode = displayModeExtended;
                    };

                    const deregister = $scope.$watch('tileParams', function(nv) {
                        if (!nv) {
                            return;
                        }
                        $scope.tileParams.$displayModeExtended = $scope.tileParams.displayMode || 'llm-summary';
                        deregister();
                    });
                });
            }
        };
    });


    app.directive('modelEvaluationReportInsightTileParams', function($controller, $timeout, DataikuAPI, FullModelLikeIdUtils, $q, ModelDataUtils, $stateParams) {
        return {
            templateUrl: '/templates/dashboards/insights/model-evaluation_report/model-evaluation_report_tile_params.html',
            scope: {
                tileParams: '=',
                insight: '='
            },
            link: function($scope, $element, attrs) {
                $scope.ModelDataUtils = ModelDataUtils;

                const setDefaultOverridesMetricsDisplayMode = function() {
                    $scope.tileParams.advancedOptions = $scope.tileParams.advancedOptions || {};
                    $scope.tileParams.advancedOptions.overridesMetrics = $scope.tileParams.advancedOptions.overridesMetrics || {};
                    $scope.tileParams.advancedOptions.overridesMetrics.displayMode =
                        $scope.tileParams.advancedOptions.overridesMetrics.displayMode || 'sankey';
                };

                $scope.$watch('insight.$modelData', function(nv) {
                    if (!nv) {
                        return;
                    }
                    $scope.modelData = nv;
                    $scope.fullModelId = $scope.insight.$modelData.fullModelId;

                    $controller('ModelEvaluationReportViewCommon', { $scope: $scope });

                    if (!($scope.tileParams.advancedOptions && $scope.tileParams.advancedOptions.interactiveScoring) && $scope.fullModelId) {
                        Promise.all([
                            DataikuAPI.ml.prediction.getColumnImportance($scope.fullModelId),
                            DataikuAPI.ml.prediction.getSplitDesc($scope.fullModelId),
                            DataikuAPI.ml.prediction.getPreparationScript($scope.fullModelId),
                            DataikuAPI.ml.prediction.getInputDatasetSchema($scope.fullModelId).catch(e => e)
                        ])
                            .then(([columnImportanceResp, splitDescResp, preparationScriptResp, inputDatasetSchemaResp]) => {
                                let featuresOrder;
                                if (columnImportanceResp.data) {
                                    const importances = columnImportanceResp.data.importances;
                                    const columns = columnImportanceResp.data.columns;
                                    featuresOrder = columns.sort((c1, c2) => importances[columns.indexOf(c2)] - importances[columns.indexOf(c1)]);
                                } else {
                                    const perFeature = $scope.modelData.preprocessing.per_feature;
                                    const inputColumns = Object.keys(perFeature).filter(featureName => perFeature[featureName].role === 'INPUT');
                                    featuresOrder = splitDescResp.data.schema.columns.map(c => c.name).filter(c => inputColumns.includes(c));
                                }
                                const hasPreparationSteps = preparationScriptResp.data.steps.some(step => !step.disabled);
                                if (hasPreparationSteps) {
                                    if (inputDatasetSchemaResp.data.columns) {
                                        const preScriptFeatures = inputDatasetSchemaResp.data.columns.map(col => col.name);
                                        if (columnImportanceResp.data) {
                                            featuresOrder.push(...preScriptFeatures.filter(f => !featuresOrder.includes(f)));
                                        } else {
                                            featuresOrder = [...preScriptFeatures, ...featuresOrder.filter(f => !preScriptFeatures.includes(f))];
                                        }
                                    } else if (inputDatasetSchemaResp.status !== 404) {
                                        setErrorInScope.call($scope, inputDatasetSchemaResp);
                                    }
                                }
                                $scope.tileParams.advancedOptions = {
                                    ...$scope.tileParams.advancedOptions,
                                    interactiveScoring: {
                                        featuresOrder
                                    }
                                };
                            })
                            .catch(setErrorInScope.bind($scope));
                    }

                    $controller('_PredictionModelReportController', { $scope: $scope });

                    $timeout(function() {
                        $element.find('.view-select').selectpicker('refresh');
                    });


                    $scope.onDisplayModeExtendedChange = function() {
                        if (!$scope.tileParams || !$scope.tileParams.$displayModeExtended) {
                            return;
                        }
                        const displayModeExtended = $scope.tileParams.$displayModeExtended;
                        $scope.tileParams.displayMode = displayModeExtended;
                        if (displayModeExtended === 'tabular-overrides_metrics') {
                            setDefaultOverridesMetricsDisplayMode();
                        }
                    };
                    const deregister = $scope.$watch('tileParams', function(nv) {
                        if (!nv) {
                            return;
                        }
                        $scope.tileParams.$displayModeExtended = $scope.tileParams.displayMode || 'summary';
                        deregister();
                    });

                    if ($scope.tileParams.displayMode === 'tabular-overrides_metrics') {
                        setDefaultOverridesMetricsDisplayMode();
                    }
                    setDefaultImportanceDisplayMode($scope.hasVariableImportance(), $scope.hasGlobalExplanations());
                });

                if ($scope.tileParams.displayMode === 'tabular-overrides_metrics') {
                    setDefaultOverridesMetricsDisplayMode();
                }

                const setDefaultImportanceDisplayMode = function(hasVariableImportance, hasGlobalExplanations) {
                    let importanceDisplayMode = 'globalExplanations';
                    const tileParamsAdvancedOptions = $scope.tileParams.advancedOptions;
                    if (hasVariableImportance) {
                        if (!tileParamsAdvancedOptions || !tileParamsAdvancedOptions.featureImportance) {
                            if (!hasGlobalExplanations) {
                                importanceDisplayMode = 'variableImportance';
                            }
                        } else {
                            importanceDisplayMode = tileParamsAdvancedOptions.featureImportance.importanceDisplayMode;
                        }
                    }
                    const advancedOptions = $scope.tileParams.advancedOptions || { featureImportance: {} };
                    $scope.tileParams.advancedOptions = {
                        ...advancedOptions,
                        featureImportance: {
                            ...advancedOptions.featureImportance,
                            importanceDisplayMode
                        }
                    };
                    setDefaultGraphType();
                };

                const setDefaultGraphType = function() {
                    let graphType = 'absoluteFeatureImportance';
                    const tileParamsAdvancedOptions = $scope.tileParams.advancedOptions;
                    if (tileParamsAdvancedOptions && tileParamsAdvancedOptions.featureImportance.graphType) {
                        graphType = tileParamsAdvancedOptions.featureImportance.graphType;
                    }

                    $scope.tileParams.advancedOptions = {
                        ...tileParamsAdvancedOptions,
                        featureImportance: {
                            ...tileParamsAdvancedOptions.featureImportance,
                            graphType
                        }
                    };
                };
            }
        };
    });

    app.directive('modelEvaluationReportInsightCreateForm', function(DataikuAPI) {
        return {
            templateUrl: '/templates/dashboards/insights/model-evaluation_report/model-evaluation_report_create_form.html',
            scope: true,
            link: function($scope, element, attrs) {
                $scope.hook.defaultName = 'Model Evaluation table';
                $scope.$watch('hook.sourceObject', function(nv) {
                    if (!nv || !nv.label) {
                        return;
                    }
                    $scope.hook.defaultName = nv.label + ' table';
                });
            }
        };
    });
})();
