(function() {
'use strict';

/** This module provides global Git capabilities that are shared between projects, plugins, git references, ... */
const app = angular.module('dataiku.git', []);

app.controller("CommitObjectModalController", function($scope, $stateParams, Assert, DataikuAPI) {
    $scope.uiState = {
        activeTab: "message"
    };

    function fetch() {
        Assert.inScope($scope, 'object');
        DataikuAPI.git.prepareObjectCommit($stateParams.projectKey,
        	$scope.object.objectType,
        	$scope.object.objectId).success(function(data){
            $scope.preparationData = data;
        }).error(setErrorInScope.bind($scope))
    }

    $scope.$watch("object", function(nv, ov){
        if (!nv) {
            return;
        }
        fetch();
    });

    $scope.commit = function() {
    	Assert.inScope($scope, 'object');
    	DataikuAPI.git.commitObject($stateParams.projectKey,
        	$scope.object.objectType,
        	$scope.object.objectId, $scope.uiState.message).success(function(data){
        	$scope.dismiss();
        }).error(setErrorInScope.bind($scope))
    };
});

app.controller("_gitLogControllerBase", function($scope, DataikuGitAPI, element, objectType, $document, $state, 
                                                 $timeout, CreateModalFromTemplate, $stateParams, Dialogs, $filter, 
                                                 DKUtils, WT1, ActivityIndicator, FullGitSupportService) {
    let $element = $(element[0]);
    let $line = $element.find('.line-selected');

    $scope.compare = {};

    $scope.withTags = () => $scope.logEntries 
                            && $scope.logEntries.length > 0
                            && 'tags' in $scope.logEntries[0]
                            && !$scope.noTagEdit;

    $scope.setCompareFrom = function(day, commit) {
        $scope.compare.from = {day: day, commit: commit};
        $scope.compare.to = null;
        $timeout(function() { $document.on('click', $scope.exitCompare); });
    };

    $scope.exitCompare = function() {
        safeApply($scope, function() { $scope.compare = {}; });
        $document.off('click', $scope.exitCompare)
    };

    $scope.$on('$destroy', function() {
        $document.off('click', $scope.exitCompare)
    });

    $scope.setCompareTo = function(day, commit) {
        if (!$scope.compare.from) return;
        $scope.compare.to = {day: day, commit: commit};
    };

    $scope.dayInCompareRange = function(day) {
        if (!$scope.compare.from) return true;
        if (!$scope.compare.to) return $scope.compare.from.day == day;
        return day >= $scope.compare.top.day && day <= $scope.compare.bottom.day;
    };

    $scope.commitInCompareRange = function(day, commit) {
        if (!$scope.compare.from) return true;
        if (!$scope.compare.to) return $scope.compare.from.day == day && $scope.compare.from.commit == commit;

        var afterTop = day > $scope.compare.top.day || (day == $scope.compare.top.day) && commit > $scope.compare.top.commit;
        var beforeBottom = day < $scope.compare.bottom.day || (day == $scope.compare.bottom.day) && commit < $scope.compare.bottom.commit;
        return afterTop && beforeBottom;
    };

    $scope.clickIsOnAnchor = function($event) {
        return $event.target && $event.target.tagName && $event.target.tagName.toLowerCase() == 'a';
    };
    $scope.openCompareModal = function() {
        if (!$scope.compare.from || !$scope.compare.to) return;

        var commitFrom = $scope.days[$scope.compare.bottom.day].changes[$scope.compare.bottom.commit].commitId;
        var commitTo = $scope.days[$scope.compare.top.day].changes[$scope.compare.top.commit].commitId;

        if (commitFrom === commitTo) return $scope.exitCompare();

        CreateModalFromTemplate("/templates/git/git-compare-modal.html", $scope, null, function(newScope) {
            DataikuGitAPI.getRevisionsDiff(commitFrom, commitTo).success(function (data) {
                newScope.diff = data;
            }).error(setErrorInScope.bind($scope));
        }).then($scope.exitCompare, $scope.exitCompare);
    };

    $scope.isElementA = function($event) {
        return $event.target.tagName.toLowerCase() == 'a';
    };
    $scope.openDiffModal = function(commitId, tags, redirectAfterRevertCallback) {
        CreateModalFromTemplate("/templates/git/git-diff-modal.html", $scope, null, function(newScope){
            // You can't revert an object/project to its last change (#7271)

            DataikuGitAPI.getCommitDiff(commitId).success(function(data) {
                newScope.commit = data;
                newScope.tags = tags;

                if (DataikuGitAPI.createBranchFromCommit) {
                    newScope.createBranch = function() {
                        newScope.dismiss();
                        DataikuGitAPI.createBranchFromCommit(commitId);
                    };
                }

                newScope.withTags = $scope.withTags();

                if (newScope.withTags) {
                    newScope.addTag = function() {
                        newScope.dismiss();
                        $scope.addTag(commitId);
                    };
                    newScope.removeTag = function(tagName) {
                        newScope.dismiss();
                        $scope.removeTag(tagName);
                    };
                }
                if (newScope.commitRevertable) {
                    newScope.revertCommit = function(){
                        newScope.dismiss();
                        $scope.revertSingleCommit(commitId);
                    };
                }
                const isRevertable = newScope.objectRevertable && commitId !== $scope.logEntries[0].commitId;
                if (isRevertable) {
                    newScope.revertTo = function(){
                        newScope.dismiss();
                        if (objectType === 'PROJECT') {
                            $scope.revertProjectToHash(commitId);
                        } else {
                            $scope.revertObjectToHash(commitId, redirectAfterRevertCallback);
                        }
                    };
                }
            }).error(setErrorInScope.bind($scope));
        }, true);
    };

    $scope.revertObjectToHash = function(hash, redirectAfterRevertCallback) {
        DataikuGitAPI.getDSSVersionForASpecificCommit(hash).success(function(dssVersionForCommitResponse){ 
            let msg = "Are you sure you want to revert " + $filter("taggableObjectRef")($scope.objectRef) + " ?";
            if ($scope.objectRef.projectKey) {
                msg += "<br/>Reverting a single object can lead to inconsistent state. It is recommended to prefer reverting the complete project.";
            }
            msg = addWarningOnRevertIfNeeded(msg, dssVersionForCommitResponse)

            Dialogs.confirm($scope, "Revert: " + $filter("taggableObjectRef")($scope.objectRef), msg).then(function(){
                DataikuGitAPI.revertObjectToRevision(hash).success(function(data){
                    if(redirectAfterRevertCallback){
                        redirectAfterRevertCallback();
                    } else{
                        DKUtils.reloadState();
                    }
                }).error(setErrorInScope.bind($scope));
            })
        }).error(setErrorInScope.bind($scope));
    };

    $scope.revertProjectToHash = function(hash) {
        DataikuGitAPI.getDSSVersionForASpecificCommit(hash).success(function(dssVersionForCommitResponse){ 
            let msg = $scope.projectZone
                ? `<div>Are you sure you want to revert Project ${$scope.projectZone} to this revision ?</div>Reverting ${$scope.projectZone} can lead to inconsistent state. It is recommended to revert the complete project.`
                : "Are you sure you want to revert Project to this revision ?";

            msg = addWarningOnRevertIfNeeded(msg, dssVersionForCommitResponse)

            Dialogs.confirm($scope, "Revert Project " + ($scope.projectZone || ''), msg)
                .then(function () {
                    DataikuGitAPI.revertProjectToRevision(hash)
                        .success(function (data) {
                            Dialogs.infoMessagesDisplayOnly($scope, "Revert results", data)
                                .then(function () {
                                    DKUtils.reloadState();
                                }
                                )
                        })
                        .error(setErrorInScope.bind($scope));
                })
        }).error(setErrorInScope.bind($scope));
        
    };

    $scope.revertSingleCommit = function(hash) {
        DataikuGitAPI.getDSSVersionForASpecificCommit(hash).success(function(dssVersionForCommitResponse){ 
            let msg = `
    <div class='mbot8'>Are you sure you want to revert this revision ?</div>
    Reverting a single revision can lead to an inconsistent state. It is recommended to prefer reverting the complete ${objectType}
    `;
            msg = addWarningOnRevertIfNeeded(msg, dssVersionForCommitResponse)

            Dialogs.confirm($scope, "Revert " + hash, msg)
                .then(function () {
                    DataikuGitAPI.revertSingleCommit(hash)
                        .success(function (data) {
                            Dialogs.infoMessagesDisplayOnly($scope, "Revert results", data)
                                .then(function () {
                                    DKUtils.reloadState();
                                })
                        }).error(setErrorInScope.bind($scope));
                })
        }).error(setErrorInScope.bind($scope));
    };

    const addWarningOnRevertIfNeeded = function(msg, dssVersionForCommitResponse) {
    
        if (dssVersionForCommitResponse.commitFromPreviousDSSVersion) {
            msg = `<div class="alert alert-warning alert-with-icon"><i class="dku-icon-warning-outline-48"></i>Warning: Reverting this commit is potentially dangerous.<br/>
                    This commit is associated with DSS version ${dssVersionForCommitResponse.dssVersion}, and a migration occurred after this version or on this commit. This may break configuration files and render the project unusable.</div></div>` + msg;
        }
        
        return msg;
    };

    if (DataikuGitAPI.addTag) {
        $scope.addTag = function(reference='HEAD') {
            FullGitSupportService.addTagModal($scope, reference, tag => DataikuGitAPI.addTag(tag.reference, tag.name, tag.message))
                                .then(() => DKUtils.reloadState());
        };
    }

    if (DataikuGitAPI.removeTag) {
        $scope.removeTag = function(tagName){
            const msg = `Are you sure you want to remove the tag '${tagName}' ?`;
            Dialogs.confirm($scope, `Remove tag '${tagName}'`, msg).then(function(){
                DataikuGitAPI.removeTag(tagName)
                    .then(function(){
                        ActivityIndicator.success(`Tag '${tagName}' removed`, 5000);
                        WT1.event("git-remove-tag", {tagName, objectType});
                        DKUtils.reloadState();
                    })
                    .catch(setErrorInScope.bind($scope));
            });
        };
    }

    $scope.$watch('compare', function(compare) {
        if (!compare || !compare.from || !compare.to) return;

        var days = $element.find('.day');

        if (compare.to.day > compare.from.day || (compare.to.day == compare.from.day) && compare.to.commit > compare.from.commit) {
            compare.bottom = compare.to;
            compare.top = compare.from;
        } else {
            compare.bottom = compare.from;
            compare.top = compare.to;
        }

        var topEl = days.eq(compare.top.day).find('.commit-log-entry').eq(compare.top.commit);
        var bottomEl = days.eq(compare.bottom.day).find('.commit-log-entry').eq(compare.bottom.commit);

        $line.css('top', topEl.position().top + topEl.height()/2);
        $line.css('bottom', $line.parent().height() - bottomEl.position().top - bottomEl.height()/2);
    }, true);

    $scope.$watch("logEntries", function(nv) {
        $scope.days = [];

        if (!nv) return;
        var currentDay = {changes:[]};

        nv.forEach(function(change) {
            var date = new Date(change.timestamp).setHours(0,0,0,0);
            if (date == currentDay.date) {
                currentDay.changes.push(change);
            } else {
                if (currentDay.changes.length) $scope.days.push(currentDay);
                currentDay = {changes: [change], date: date};
            }
        });

        if(currentDay.changes.length) $scope.days.push(currentDay);
    });
});

app.directive("gitLog", function($controller, DataikuAPI, $stateParams) {
    return {
        templateUrl: "/templates/git/git-log.html",
        scope: {
            logEntries: '=',
            firstStatus: '=',
            lastStatus: '=',
            objectRevertable: '=',
            objectRef: '=',
            projectZone: '=',
            commitRevertable: '=',
            noCommitDiff: '=',
            noAuthorLink: '=',
            createBranchFromCommit: '=',
            scrollToCommitId: '=',
            highlightCommitId: '=',
            noTagEdit: '=',
            redirectAfterRevertCallback: '=?'
        },
        link: function ($scope, element) {
            const objectRef = $scope.objectRef 
                ? $scope.objectRef
                : { //assuming 'project' by default
                    projectKey: $stateParams.projectKey,
                    id: $stateParams.projectKey,
                    type: 'PROJECT'
                };
            const projectGitAPI = {
                getDSSVersionForASpecificCommit: (hash) => DataikuAPI.projects.git.getDSSVersionForASpecificCommit($stateParams.projectKey, hash),
                getRevisionsDiff: (commitFrom, commitTo) => DataikuAPI.git.getRevisionsDiff($stateParams.projectKey, commitFrom, commitTo, objectRef),
                getCommitDiff: (commitId) => DataikuAPI.git.getCommitDiff($stateParams.projectKey, objectRef, commitId),
                revertObjectToRevision: (hash) => DataikuAPI.git.revertObjectToRevision(objectRef.projectKey, objectRef.type, objectRef.id, hash),
                revertProjectToRevision: (hash) => DataikuAPI.git.revertProjectToRevision($stateParams.projectKey, hash, $scope.projectZone),
                revertSingleCommit: (hash) => DataikuAPI.git.revertSingleCommit($stateParams.projectKey, $scope.objectRef, hash),
                createBranchFromCommit: $scope.createBranchFromCommit,
                removeTag: (tagName) => DataikuAPI.git.removeTag(objectRef.projectKey, objectRef.type, objectRef.id, tagName),
                addTag: (tagRef, tagName, tagMessage) => DataikuAPI.git.addTag(objectRef.projectKey, objectRef.type, objectRef.id, tagRef, tagName, tagMessage),
            };

            $controller('_gitLogControllerBase', {
                $scope: $scope, 
                element: element, 
                DataikuGitAPI: projectGitAPI, 
                objectType: objectRef.type
            });
        }
    }
});

app.directive("gitMergeRequestLog", function($controller, DataikuAPI, $stateParams) {
    return {
        templateUrl: "/templates/git/git-log.html",
        scope: {
            logEntries: '=',
            noAuthorLink: '=',
            mergeRequest: '='
        },
        link: function ($scope, element) {
            $scope.objectRevertable = false;
            $scope.createBranchFromCommit = false;
            $scope.commitRevertable = false;
            $scope.noTagEdit = true;
            
            const mergeRequestGitAPI = {
                getRevisionsDiff: (commitFrom, commitTo) => DataikuAPI.git.mergeRequests.getRevisionsDiff($stateParams.projectKey, $scope.mergeRequest.id, $scope.mergeRequest.type, commitFrom, commitTo),
                getCommitDiff: (commitId) => DataikuAPI.git.mergeRequests.getCommitDiff($stateParams.projectKey, $scope.mergeRequest.id, commitId)
            };

            $controller('_gitLogControllerBase', {
                $scope: $scope, 
                element: element, 
                DataikuGitAPI: mergeRequestGitAPI, 
                objectType: $scope.mergeRequest.type
            });
        }
    }
});

app.directive("gitDiff", function(DiffFormatter) {
    return {
        templateUrl : "/templates/git/git-diff.html",
        scope: {
            diffEntries : '='
        },
        link: function($scope, element, attrs) {
            $scope.showAll = function() {
                $scope.diffEntries.forEach(function(file) {
                    file.rendered = DiffFormatter.formatChange(file.fileChange);
                    file.shown = true;
                });
            };

            $scope.hideAll = function() {
                $scope.diffEntries.forEach(function(file) {
                    file.shown = false;
                });
            };

            $scope.toggle = function toggle(file) {
                file.shown = !file.shown;
                if (file.shown && !file.rendered) {
                    file.rendered = DiffFormatter.formatChange(file.fileChange);
                }
            };
        }
    }
});

app.directive("objectGitHistory", function($stateParams, DataikuAPI) {
    return {
        scope: {
            objectType: '@',
            objectId: '=',
            projectKey: '=?', // defaults to $stateParams.projectKey
            objectRevertable: '=?',
            commitRevertable: '=?',
            projectZone: '=?',
            redirectAfterRevertCallback: '=?',
        },
        templateUrl : "/templates/git/object-git-history.html",
        link: function($scope) {
            const PAGE_SIZE = 20;
            $scope.hasMore = true;

            $scope.objectRef = {
                projectKey: $scope.projectKey || $stateParams.projectKey,
                type: $scope.objectType,
                id: $scope.objectId
            };

            $scope.$watch('objectId', function(nv) {
                $scope.objectRef.id = nv;
            });

            $scope.loadMore = function () {
                if ($scope.hasMore && !$scope.loading) {
                    $scope.loading = true;
                    DataikuAPI.git.getObjectLog($scope.projectKey || $stateParams.projectKey, $scope.objectType, $scope.objectId || $stateParams.webAppId || $stateParams.smId, $scope.nextCommit, PAGE_SIZE, $scope.projectZone).success(function(data){
                        $scope.logEntries = ($scope.logEntries || []).concat(data.logEntries);
                        $scope.nextCommit = data.nextCommit;
                        if (!$scope.nextCommit) {
                            $scope.hasMore = false;
                        }
                        $scope.loading = false;
                    }).error(function(e) {
                        $scope.loading = false;
                        setErrorInScope.bind($scope);
                    });
                }
            };
        }
    };
});

/**
 * This service provides common support for parts of DSS that support "full-repository push-pull" interaction, i.e
 *   - projects
 *   - plugins in development
 */ 
app.service("FullGitSupportService", function(DataikuAPI, $state, $stateParams, CreateModalFromTemplate,
                                              Dialogs, FutureProgressModal, DKUtils, DKUConstants, $filter,
                                              WT1, ActivityIndicator) {

    var svc = {};

    svc.getFullStatus = function($scope, apiPromise, cb) {
        return apiPromise.then(function(resp) {
            $scope.gitStatus = resp.data;
            $scope.gitStatus.remoteOrigin = resp.data.remotes.find(r => r.name === 'origin');
            $scope.gitStatus.hasRemoteOrigin = !$.isEmptyObject($scope.gitStatus.remoteOrigin);
            $scope.gitStatus.hasTrackingCount = !$.isEmptyObject($scope.gitStatus.trackingCount);

            if (cb) { cb() }
        },
        setErrorInScope.bind($scope));
    };

    svc.getBranches = function($scope, apiPromise) {
        apiPromise.then(function(resp) {
            $scope.gitBranches = resp.data.sort();
            $scope.gitBranchesFiltered = $scope.gitBranches;
        }, setErrorInScope.bind($scope));
    };

    svc.fetch = function($scope, apiPromise) {
        apiPromise.then(function(resp) {
            FutureProgressModal.show($scope, resp.data, "Fetching Git changes").then(function(result){
                if (result) {
                    Dialogs.infoMessagesDisplayOnly($scope, "Fetch result", result.messages, result.futureLog, true).then(function() {
                        // We're dismissing the first modal when it has succeeded.
                        if (result.commandSucceeded) {
                            DKUtils.reloadState();
                        }
                    }, null);
                }
            });
        }, setErrorInScope.bind($scope));
    };

    svc.editRemote = function($scope, saveCallback) {
        let url = "";
        let action = "Add";
        if ($scope.gitStatus.remoteOrigin && $scope.gitStatus.remoteOrigin.url) {
            url = $scope.gitStatus.remoteOrigin.url;
            action = "Edit";
        }
        Dialogs.prompt($scope, action + " remote origin", "Remote URL", url, { placeholder: "git@github.com:user/repo.git"}).then(function(newURL) {
            if (!newURL || newURL === url) {
                return;
            }
            saveCallback("origin", newURL);
        });
    };

    svc.removeRemote = function($scope, saveCallback) {
        Dialogs.confirm($scope, "Remove remote origin",
            "Are you sure you want to unlink this local repository from the remote repository?").then(function() {
            saveCallback("origin");
        });
    };

    svc.pull = function($scope, apiPromise) {
        apiPromise.then(function(resp) {
            FutureProgressModal.show($scope, resp.data, "Pulling Git changes").then(function(result){
                if (result) {
                    CreateModalFromTemplate("/templates/git/pull-result-modal.html", $scope, null, function(newScope) {
                        newScope.DKUConstants = DKUConstants;
                        newScope.data = result.messages;
                        newScope.log = result.futureLog;
                        // for the moment, merge request are supported only for projects,
                        // and only ProjectVersionControlController has the fn `openCreateMergeRequestModal`.
                        newScope.hasConflicts = $scope.openCreateMergeRequestModal && result.messages.messages
                            && result.messages.messages.find(msg => msg.code === 'ERR_GIT_PULL_FAILED')
                            && result.futureLog.lines.find(line => line.startsWith('CONFLICT '));
                    }).then(createMergeRequest => {
                        if (createMergeRequest) {
                            const branch = $scope.gitStatus.remoteOrigin.name + '/' + $scope.gitStatus.currentBranch;
                            $scope.openCreateMergeRequestModal('Resolve conflict with ' + branch, branch);
                        } else {
                            DKUtils.reloadState();
                        }
                    });
                }
            });
        }, setErrorInScope.bind($scope));
    };

    svc.push = function($scope, apiPromise) {
        apiPromise.then(function(resp) {
            FutureProgressModal.show($scope, resp.data, "Pushing Git changes").then(function(result){
                if (result) {
                    Dialogs.infoMessagesDisplayOnly($scope, "Push result", result.messages, result.futureLog, true).then(function() {
                        // We're dismissing the first modal when it has succeeded.
                        if (result.commandSucceeded) {
                            $scope.getGitFullStatus();
                        }
                    }, null);
                }
            });
        }, setErrorInScope.bind($scope));
    };

    svc.switchToBranch = function($scope, apiPromise) {
        apiPromise.then(function() {
            $state.reload();
        }, setErrorInScope.bind($scope));
    };

    svc.deleteBranches = function($scope, callback) {
        CreateModalFromTemplate("/templates/git/delete-branches-modal.html", $scope, null, function (newScope) {
            newScope.selectedGitBranches = [];
            newScope.deleteOptions = { remoteDelete: false, forceDelete : false };
            newScope.deleteBranches = function () {
                callback(newScope, newScope.selectedGitBranches, newScope.deleteOptions);
            };
        });
    };
    
    svc.addTagModal = function($scope, reference, apiAddTagFn) {
        return CreateModalFromTemplate("/templates/git/add-tag-modal.html", $scope, null, function(newScope){
            newScope.tag = {
                reference: reference,
                name: '',
                message: null
            };
            newScope.editorOptions = {
                mode : 'text/plain',
                lineNumbers : false,
                matchBrackets : false
            };
            newScope.addTag = () => {
                apiAddTagFn(newScope.tag).then(() => {
                    const tag = newScope.tag;
                    ActivityIndicator.success(`Tag '${tag.name}' added`, 5000);
                    WT1.event("git-add-tag", {tag, reference, objectType: $scope.objectType || 'PROJECT', objectId: $scope.objectId || $stateParams.projectKey});
                    newScope.resolveModal(tag);
                })
                .catch(setErrorInScope.bind(newScope));
            };
        }, false);
    };

    svc.createMergeRequestModal = ($scope, objectType, objectId, defaultBranch, defaultTitle, defaultBranchToMerge) => {
        return CreateModalFromTemplate("/templates/git/create-merge-request-modal.html", $scope, null, modalScope => {
            modalScope.mergeRequest = {
                title: defaultTitle || '',
                description: '',
                baseBranch: defaultBranch,
                branchToMerge: '',
                branchToMergeType: '',
            };
            modalScope.editorOptions = {
                mode : 'text/plain',
                lineNumbers : false,
                matchBrackets : false
            };
            modalScope.branches = [];
            modalScope.selectedBranch = null;

            const isProjectMerge = objectType === 'PROJECT';

            function getProjectKeyFromBranch(branch) {
                return branch.indexOf("refs/projects/") === 0
                    ? branch.split('/')[2]
                    : objectId;
            }
            
            modalScope.createMergeRequest = () => {
                if (modalScope.selectedBranch 
                    && modalScope.mergeRequest.title 
                    && modalScope.mergeRequest.baseBranch !== modalScope.selectedBranch.branch) {

                    modalScope.mergeRequest.branchToMerge = modalScope.selectedBranch.branch;
                    modalScope.mergeRequest.branchToMergeType = modalScope.selectedBranch.type;
                    if (isProjectMerge) {
                        const baseProject = getProjectKeyFromBranch(modalScope.mergeRequest.baseBranch);
                        modalScope.mergeRequest.projectToMerge = getProjectKeyFromBranch(modalScope.mergeRequest.branchToMerge);
                        modalScope.mergeRequest.type = 'PROJECT';
                        let futureModalScope;
                        DataikuAPI.git.mergeRequests
                            .create(baseProject, modalScope.mergeRequest)
                            .success(response => {
                                FutureProgressModal
                                    .show(modalScope, response, "Create merge request", scope => futureModalScope = scope, 'static', 'false', true)
                                    .then(result => {
                                        WT1.event("git-merge-request_create", {
                                            id: result.mergeRequest.id,
                                            mergeType: result.mergeRequest.branchToMergeType,
                                            nbChangedFiles: result.diff.changedFiles,
                                            nbCommits: result.log.logEntries.length,
                                            nbConflictingFiles: result.status.conflicting.length
                                        });

                                        modalScope.resolveModal(result.mergeRequest);
                                        modalScope.dismiss();
                                    }, (...args) => {
                                        const {data} = args[0];
                                        if (data && data.code === 'ERR_GIT_MERGE_CREATE_MERGE_REQUEST_NO_CHANGES') {
                                            futureModalScope && futureModalScope.dismiss();
                                            Dialogs.ack(modalScope, "Create merge request result", data.title);
                                        } else {
                                            setErrorInScope.bind($scope)(...args);
                                        }
                                    });
                            })
                            .error(setErrorInScope.bind(modalScope));
                    }
                }
            };

            // init modal
            if (isProjectMerge) {
                modalScope.mergeRequest.projectToMerge = objectId;
                DataikuAPI.git.mergeRequests.listBranches(objectId)
                    .success(branchesByType => {
                        let branches = [];
                        for(let type in branchesByType) {
                            branches = branches.concat(branchesByType[type]
                                .filter(branch => branch != modalScope.mergeRequest.baseBranch)
                                .map(branch => { return {type, branch}}));
                        }
                        modalScope.branches = branches;
                        if (defaultBranchToMerge) {
                            modalScope.selectedBranch = modalScope.branches.find(b => b.branch.localeCompare(defaultBranchToMerge) === 0);
                        }
                    })
                    .error(setErrorInScope.bind(modalScope));
            }

        });
    };

    return svc;
})


app.controller("GitMergeRequestController", function($scope, $rootScope, $state, $stateParams, $timeout,
                                                     DataikuAPI, TopNav, Dialogs, CodeMirrorSettingService, ActivityIndicator, 
                                                     ProjectActionsService, FutureProgressModal, CreateModalFromTemplate, WT1) {
    const CONFLICT_MARK_START = '<<<<<';
    const CONFLICT_MARK_END = '>>>>>';
    const CONFLICT_HIGHLIGHT_CSS = 'conflict-line';
    const projectKey = $stateParams.projectKey;
    $scope.mergeRequestId = $stateParams.mergeRequestId;
    TopNav.setLocation(TopNav.TOP_MORE, "version-control", "NONE", null);
    TopNav.setItem(TopNav.ITEM_PROJECT, projectKey);
    
    $scope.uiState = {
        tab: 'overview'
    };
    $scope.mergeRequestInfo = {};
    $scope.mergeRequest = {};
    $scope.editableFiles = [];
    $scope.activeFile = null;
    $scope.editorOptions = null;
    $scope.events = {days: []};

    let editor = null;
    let searchCursor = null;
    let files = {};

    $scope.isReadyToMerge = () => $scope.mergeRequestInfo.status &&
                                  $scope.mergeRequestInfo.status.conflicting.length === 0;
                                  
    function closeFile() {
        editor = null;
        searchCursor = null;
        $scope.editorOptions = null;
        $scope.activeFile = null;
    }

    function resetEditor() {
        closeFile();
        files = {};
    }

    function setupMergeRequest(mergeRequestInfo) {
        $scope.mergeRequestInfo = mergeRequestInfo;
        $scope.mergeRequest = {...mergeRequestInfo.mergeRequest};
        $scope.isOpen = mergeRequestInfo.mergeRequest.status === 'OPENED';
        $scope.isMerged = mergeRequestInfo.mergeRequest.status === 'MERGED';
        // compute editable files
        $scope.nbConflictingFiles = $scope.mergeRequestInfo.status.conflicting.length;
        const editableFilesMap = {};
        // init with initial conflicting files considering they're resolved
        mergeRequestInfo.mergeRequest.initialConflictingFiles.forEach(f =>
            editableFilesMap[f] = {
                filename: f.split('/').at(-1),
                filepath: f,
                resolved: true,
                state: 'resolved',
                nbConflicts: 0
            }
        );
        // override with still conflicting files
        $scope.mergeRequestInfo.status.conflicting.forEach(f =>
            editableFilesMap[f] = {
                filename: f.split('/').at(-1),
                filepath: f,
                resolved: false,
                state: 'conflicting',
                nbConflicts: mergeRequestInfo.nbConflictsPerFile[f] >= 0 ? mergeRequestInfo.nbConflictsPerFile[f] : '?'
            }
        );
        $scope.editableFiles = Object.values(editableFilesMap);
        // compute events by day
        let currentDay = {events: [], date: null};
        const lastidx = mergeRequestInfo.mergeRequest.events.length - 1;
        $scope.eventsByDay = mergeRequestInfo.mergeRequest.events.reduce((eventsByDay, event, idx) => {
            const date = new Date(event.creationDate).setHours(0,0,0,0);
            if (date === currentDay.date) {
                currentDay.events.push(event);
            } else {
                if (currentDay.events.length > 0) {
                    eventsByDay.push(currentDay);
                }
                currentDay = {events: [event], date: date};
            }
            if (idx === lastidx) {
                eventsByDay.push(currentDay);
            }
            return eventsByDay;
        }, []);
        // open default conflicting file
        if ($scope.nbConflictingFiles === 0) {
            closeFile();
        } else if (!$scope.activeFile || !$scope.editableFiles.find(f => f.filepath === $scope.activeFile.path)) {
            closeFile();
            $scope.openFile($scope.editableFiles[0]);
        }
    }
    
    function loadMergeRequest() {
        DataikuAPI.git.mergeRequests
            .get(projectKey, $scope.mergeRequestId)
            .success(setupMergeRequest)
            .error(setErrorInScope.bind($scope));
    }

    function resetEditorAndLoadMergeRequest() {
        resetEditor();
        loadMergeRequest();
    }

    function backToPreviousPage(){
        $state.go('projects.project.version-control', {commitId: null, branch: null});
    }

    $scope.setTab = tab => {
        $scope.uiState.tab = tab;
        if (tab === 'conflicts' && editor) {
            $timeout(() => editor.refresh());
        }
    };

    $scope.isUpdatable = () => $scope.mergeRequestInfo.mergeRequest
        && ($scope.mergeRequest.description !== $scope.mergeRequestInfo.mergeRequest.description || $scope.mergeRequest.title !== $scope.mergeRequestInfo.mergeRequest.title)
        && $scope.mergeRequest.title.trim().length > 0;

    $scope.update = () => {
        WT1.event("git-merge-request_update", {id: $scope.mergeRequestId});
        DataikuAPI.git.mergeRequests
            .update(projectKey, $scope.mergeRequest)
            .success(response => 
                FutureProgressModal
                    .show($scope, response, "Updating...", null, 'static', 'false', true)
                    .then((infoMessages) => {
                        if (infoMessages && !infoMessages.success) { // undefined in case of abort
                            Dialogs.infoMessagesDisplayOnly($scope, "Update failed", infoMessages, infoMessages.futureLog);
                        } else {
                            loadMergeRequest();
                        }
                    }, setErrorInScope.bind($scope))
            )
            .error(setErrorInScope.bind($scope));
    };

    function callCancelMerge() {
        WT1.event("git-merge-request_cancel", {id: $scope.mergeRequestId});
        DataikuAPI.git.mergeRequests
            .delete(projectKey, $scope.mergeRequestId)
            .success(backToPreviousPage)
            .error(setErrorInScope.bind($scope));
    }

    $scope.cancelMerge = () => {
        resetErrorInScope($scope);
        Dialogs.confirm($scope, "Delete merge request", "All conflict resolutions, if any, will be lost. Are you sure?")
            .then(callCancelMerge, angular.noop);
    };

    function mergeSuccess() {
        const projectCurrentBranch = $rootScope.topNav.projectCurrentBranch;
        if ($scope.uiState.tab === 'conflicts') {
            $scope.uiState.tab = 'overview';
        }
        CreateModalFromTemplate("/templates/git/merge-request-success-modal.html", $scope, null, (newScope) => {
            newScope.mergeRequest = $scope.mergeRequest;
            newScope.allowUpdatingBranchToMerge = $scope.mergeRequestInfo.allowUpdatingBranchToMerge;
            newScope.canDeleteBranchToMerge = $scope.mergeRequest.projectToMerge !== projectKey || $scope.mergeRequest.branchToMerge !== projectCurrentBranch;
        }).then((action) => {
            if (action === 'delete') {
                if ($scope.mergeRequest.branchToMergeType === 'PROJECT') {
                    const projectKeyToDelete = $scope.mergeRequest.projectToMerge;
                    DataikuAPI.projects
                        .getSummary(projectKeyToDelete)
                        .success(function(data) {
                            const projectSummary = data.object;
                            ProjectActionsService.deleteThisProject($scope, projectSummary, projectKeyToDelete, 'projects.project.version-control', {projectKey});
                        })
                        .error(setErrorInScope.bind($scope));
                } else {
                    DataikuAPI.projects.git.deleteBranches(
                        $scope.mergeRequest.projectToMerge,
                        [$scope.mergeRequest.branchToMerge],
                        {
                            remoteDelete: true,
                            forceDelete: true
                        }
                    ).then(backToPreviousPage, setErrorInScope.bind($scope))
                }
            } else if (action === 'refresh') {
                DataikuAPI.git.mergeRequests.refreshMergedBranch(
                    projectKey,
                    $scope.mergeRequest.id
                ).success(response => 
                    FutureProgressModal
                        .show($scope, response, "Refreshing...", null, 'static', 'false', true)
                        .then(
                            (infoMessages) => {
                                if (infoMessages && !infoMessages.success) { // undefined in case of abort
                                    Dialogs.infoMessagesDisplayOnly($scope, "Refresh failed", infoMessages, infoMessages.futureLog);
                                    resetEditorAndLoadMergeRequest();
                                } else {
                                    backToPreviousPage();
                                }
                            },
                            setErrorInScope.bind($scope)
                        )
                ).error(setErrorInScope.bind($scope));
            } else {
                backToPreviousPage();
            }
        }, backToPreviousPage);
    }

    function mergeErrorWithNewConflicts(text, logs) {
        CreateModalFromTemplate("/templates/git/failed-merge-request-modal.html", $scope, null, (newScope) => {
            newScope.text = text;
            newScope.log = logs;
        }).then((cancelMergeRequest) => {
            if (cancelMergeRequest) {
                callCancelMerge();
            } else {
                resetEditorAndLoadMergeRequest();
            }
        }, resetEditorAndLoadMergeRequest);
    }

    $scope.merge = () => {
        resetErrorInScope($scope);
        if ($scope.isReadyToMerge()) {
            WT1.event("git-merge-request_action_merge", {id: $scope.mergeRequestId});
            let futureModalScope;
            DataikuAPI.git.mergeRequests
                .merge(projectKey, $scope.mergeRequestId)
                .success(response => 
                    FutureProgressModal
                        .show($scope, response, "Merge...", scope => futureModalScope = scope, 'static', 'false', true)
                        .then(
                            (infoMessages) => {
                                if (infoMessages && !infoMessages.success) { // undefined in case of abort
                                    const msg = infoMessages.messages ? infoMessages.messages[0] : null;
                                    if (msg && msg.code === 'WARNING_GIT_MERGE_MERGING_BASE_CONFLICTS') {
                                        futureModalScope && futureModalScope.dismiss();
                                        mergeErrorWithNewConflicts(msg.title, infoMessages.futureLog);
                                    } else {
                                        Dialogs.infoMessagesDisplayOnly($scope, "Merge failed", infoMessages, infoMessages.futureLog);
                                    }
                                } else {
                                    futureModalScope && futureModalScope.dismiss();
                                    mergeSuccess();
                                }
                            }, 
                            setErrorInScope.bind($scope)
                        )
                )
                .error(setErrorInScope.bind($scope));
        }
    };

    function selectSyntaxicColoration(fileName, mimeType) {
        if (mimeType == 'application/sql' ) {
            return 'text/x-sql'; // codemirror prefers this one
        }
        if (/.*\.java$/.test(fileName)) {
            return 'text/x-java';
        }
        return mimeType;
    };

    function highlightConflicts(cm) {
        const doc = cm.getDoc();
        let inConflict = false;
        doc.eachLine(lh => {
            if (inConflict) {
                doc.addLineClass(lh, 'gutter', CONFLICT_HIGHLIGHT_CSS);
                inConflict = !lh.text.startsWith(CONFLICT_MARK_END);
            } else {
                inConflict = lh.text.startsWith(CONFLICT_MARK_START);
                if (inConflict || lh.text.startsWith('||||') || lh.text.startsWith('====') || lh.text.startsWith(CONFLICT_MARK_END)) {
                    doc.addLineClass(lh, 'gutter', CONFLICT_HIGHLIGHT_CSS);
                } else {
                    doc.removeLineClass(lh, 'gutter', CONFLICT_HIGHLIGHT_CSS);
                }
            }
        });
    }

    function getSearchCursor() {
        if (!searchCursor && editor) {
            searchCursor = editor.getSearchCursor(CONFLICT_MARK_START);
        }
        return searchCursor;
    }

    function clearFileCache(file) {
        if (file) {
            if (file.path in files) {
                delete files[file.path];
            }
        } else {
            files = {};
        }
    }

    function setupFile(file) {
        files[file.path] = file;
        $scope.activeFile = file;
        searchCursor = null;
        if (file.hasData) {
            if (!file.originalData) {
                file.originalData = file.data;
            }
            const mimeType = selectSyntaxicColoration(file.name, file.mimeType);
            $scope.editorOptions = CodeMirrorSettingService.get(mimeType, {
                onLoad: function(codeMirror) {
                    editor = codeMirror;
                    editor.on('change', highlightConflicts);
                }
            });
            $scope.editorOptions.readOnly = !$scope.isOpen;
            if (editor) {
                $timeout(() => editor.scrollTo(0, file.scrollTop || 0));
            }
        } else {
            $scope.editorOptions = null;
        }
    }

    checkChangesBeforeLeaving($scope, () => $scope.isUpdatable() || Object.keys(files).some($scope.isDirtyFile));

    $scope.openFile = file => {
        resetErrorInScope($scope);
        const filepath = file.filepath;
        const state = file.state;
        if (!$scope.activeFile || filepath !== $scope.activeFile.path) {
            if ($scope.activeFile) {
                $scope.activeFile.scrollTop = $('.CodeMirror-scroll').scrollTop();
            }
            if (filepath in files) {
                setupFile(files[filepath]);
            } else {
                DataikuAPI.git.mergeRequests
                    .getFile(projectKey, $scope.mergeRequestId, filepath)
                    .success(f => {
                        f.state = state;
                        setupFile(f);
                    })
                    .error(setErrorInScope.bind($scope));
            }
        }
    };

    $scope.saveFile = () => {
        resetErrorInScope($scope);
        return new Promise(success => {
            DataikuAPI.git.mergeRequests
                .saveFile(projectKey, $scope.mergeRequestId, $scope.activeFile.path, $scope.activeFile.data)
                .success(() => {
                    $scope.activeFile.originalData = $scope.activeFile.data;
                    ActivityIndicator.success(`file '${$scope.activeFile.path}' saved`, 5000);
                    success();
                })
                .error(setErrorInScope.bind($scope));
        });
    };

    $scope.isEditable = () =>
        $scope.isOpen && 
        $scope.activeFile &&
        $scope.activeFile.hasData;

    $scope.isConflicting = () =>
        $scope.isEditable() &&
        $scope.activeFile.state === 'conflicting';

    $scope.isResolvable = () =>
        $scope.isConflicting() &&
        $scope.activeFile.data.indexOf(CONFLICT_MARK_START) === -1;

    $scope.isDirtyFile = filepath =>
        files[filepath] &&
        files[filepath].hasData &&
        files[filepath].originalData !== files[filepath].data;

    function markFilesAsResolved(strategy='MANUAL') {
        resetErrorInScope($scope);
        WT1.event("git-merge-request_action_resolve-file", {
            id: $scope.mergeRequestId,
            strategy: strategy,
        });
        DataikuAPI.git.mergeRequests
            .markFileAsResolved(projectKey, $scope.mergeRequestId, $scope.activeFile.path, strategy)
            .success(mergeRequestInfo => {
                ActivityIndicator.success(`file '${$scope.activeFile.path}' resolved`, 5000);
                clearFileCache($scope.activeFile);
                setupMergeRequest(mergeRequestInfo);
            })
            .error(setErrorInScope.bind($scope));
    }

    $scope.markAsResolved = () => {
        if ($scope.isDirtyFile($scope.activeFile.path)) {
            $scope.saveFile().then(markFilesAsResolved);
        } else {
            markFilesAsResolved();
        }
    };

    $scope.resolveWithStrategy = markFilesAsResolved;

    $scope.resolveAllWithStrategy = (strategy) => {
        resetErrorInScope($scope);
        WT1.event("git-merge-request_action_resolve-all-files", {
            id: $scope.mergeRequestId,
            strategy: strategy,
        });
        DataikuAPI.git.mergeRequests
            .markAllFilesAsResolved(projectKey, $scope.mergeRequestId, strategy)
            .success(mergeRequestInfo => {
                ActivityIndicator.success(`All files resolved`, 5000);
                clearFileCache();
                setupMergeRequest(mergeRequestInfo);
            })
            .error(setErrorInScope.bind($scope));
    };
    
    $scope.goToPreviousConflict = () => {
        if (getSearchCursor()) {
            editor.focus();
            if (searchCursor.findPrevious()) {
                editor.setSelection(searchCursor.from(), searchCursor.to(), {scroll: true});
            } else {
                // re-find from end of file
                searchCursor = editor.getSearchCursor(CONFLICT_MARK_START, {line: editor.getDoc().lastLine(), ch: 0});
                if (searchCursor.findPrevious()) {
                    editor.setSelection(searchCursor.from(), searchCursor.to(), {scroll: true});
                }
            }
        }
    };

    $scope.goToNextConflict = () => {
        if (getSearchCursor()) {
            editor.focus();
            if (searchCursor.findNext()) {
                editor.setSelection(searchCursor.from(), searchCursor.to(), {scroll: true});
            } else {
                // re-find from start of file
                searchCursor = editor.getSearchCursor(CONFLICT_MARK_START, {line: editor.getDoc().firstLine(), ch: 0});
                if (searchCursor.findNext()) {
                    editor.setSelection(searchCursor.from(), searchCursor.to(), {scroll: true});
                }
            }
        }
    };

    resetEditorAndLoadMergeRequest();
});

})();
