(function(){
'use strict';

const app = angular.module('dataiku.ml.report');

app.service('PerformanceMetricsDataComposer', function($filter, PartitionedModelsService, BinaryClassificationModelsService, ModelDataUtils) {
    function _withCustomMetrics(metricsData, customMetricsResults) {
        if (!customMetricsResults) {
            return metricsData;
        }
        customMetricsResults.forEach(metricResult => {
            const metric = metricResult.metric;
            metricsData.push({
                name: 'CUSTOM',
                displayName: metric.name,
                info: `Description: ${metric.description || 'N/A'} – ${metric.greaterIsBetter ? 'Greater' : 'Lower'} values are better`,
                value: metricResult.value,
                std: metricResult.valuestd,
                isCustom: true,
                code: metric.metricCode,
                didSucceed: metricResult.didSucceed,
                error: metricResult.error,
            });
            if (metricResult.worstValue != null) {
                metricsData.push({
                    name: 'CUSTOM',
                    displayName: "Worst " + metric.name,
                    info: `Worst (${metric.greaterIsBetter ? 'Minimum' : 'Maximum'}) value of ${metric.name} across all time series.` ,
                    value: metricResult.worstValue,
                    isCustom: true,
                    code: metric.metricCode,
                    didSucceed: metricResult.didSucceed,
                    error: metricResult.error,
                });
            }
        });
        return metricsData;
    }

    function _getMetricDataWithAggregationExplanation(metricData, modelData) {
        if (!ModelDataUtils.isPartitionedBaseModel(modelData)) {
            return metricData;
        }
        metricData.info = PartitionedModelsService.getAggregationExplanation(
            metricData.name,
            metricData.displayName,
            metricData.isCustom,
            modelData.coreParams.prediction_type === 'TIMESERIES_FORECAST',
        ) + ` — ${metricData.info}`;
        return metricData;
    }

    function _getFormattedValue(metricData, modelData) {
        const precision = null; // the default heuristic is fine
        const isRegression = modelData.coreParams.prediction_type === 'REGRESSION';
        const canUseScientificNotation = isRegression && !metricData.isCustom;
        const isKfold = modelData.trainInfo.kfold;
        const approximateInterval = isKfold ? metricData.std : false;
        return $filter('mlMetricFormat')(
            metricData.value,
            metricData.name,
            precision,
            approximateInterval,
            canUseScientificNotation
        );
    }

    function _getPolishedMetricsDataForGivenPerf(getMetricsDataFunction, modelData, perf) {
        const hasProbasIfRequired = metricData => ModelDataUtils.hasProbas(modelData) || !metricData.needsProbability;
        const withFormattedValue = metricData => ({ formattedValue: _getFormattedValue(metricData, modelData), ...metricData });
        const withAggregationExplanation = metricData => _getMetricDataWithAggregationExplanation(metricData, modelData);
        return getMetricsDataFunction(modelData, perf)
            .filter(hasProbasIfRequired)
            .map(withFormattedValue)
            .map(withAggregationExplanation);
    }

    function _getPolishedMetricsData(getMetricsDataFunction, modelData, perfVariants) {
        const metricsVariantData = perfVariants.map(function(variant) {
            if (!variant.perf) {
                return [];
            }
            return _getPolishedMetricsDataForGivenPerf(getMetricsDataFunction, modelData, variant.perf);
        });

        function isEvaluationMetric(modelingMetrics, metricName) {
            if (modelingMetrics.evaluationMetric !== 'CUSTOM') {
                return modelingMetrics.evaluationMetric === metricName;
            }
            return modelingMetrics.customEvaluationMetricName === metricName;
        }

        function isThresholdOptimizationMetric(modelingMetrics, metricName) {
            return modelingMetrics.thresholdOptimizationMetric === metricName;
        }

        const firstVariantData = metricsVariantData[0];

        return {
            columns: perfVariants.map(variant => variant.label),
            rows: firstVariantData.map((metric, i) => {
                const metricDesc = {
                    name: metric.name,
                    displayName: metric.displayName,
                    info: metric.info,
                    isCustom: metric.isCustom,
                    code: metric.code,
                    isEvaluationMetric: isEvaluationMetric(modelData.modeling.metrics, metric.name === "CUSTOM" ? metric.displayName : metric.name),
                    isThresholdOptimizationMetric: isThresholdOptimizationMetric(modelData.modeling.metrics,metric.name),
                    isOneVsAll: !!metric.isOneVsAll && !metric.isCustom,
                };
                return {
                    variants: metricsVariantData.map(metricVariantData => metricVariantData[i]),
                    ...metricDesc,
                };
            }),
        };
    }

    // The additional (and optional) parameters yAxisMax yAxisMin are used only by Learning Curves.
    // yAxisMax/yAxisMin values are not "real min/max" but "good looking one".
    // Note: these are soft boundaries. If some values are out of these limits, thy will be displayed and the yaxis limit will adapt.
    // for more infos, see component learningCurvesPlot > buildChartOptions > yAxis
    function _getRegressionMetricsData(modelData, perf) {
        const metrics = perf.metrics;
        const metricsData = [
            {
                name: 'EVS',
                displayName: 'Explained Variance Score',
                info: 'Best possible score is 1.0, lower values are worse',
                value: metrics.evs,
                std: metrics.evsstd,
                yAxisMax: 1, // values could be negative
            },
            {
                name: 'MAPE',
                displayName: 'Mean Absolute Percentage Error (MAPE)',
                info: 'Average of the absolute value of the relative regression error',
                value: metrics.mape,
                std: metrics.mapestd
            },
            {
                name: 'MAE',
                displayName: 'Mean Absolute Error (MAE)',
                info: 'Average of the absolute value of the regression error',
                value: metrics.mae,
                std: metrics.maestd
            },
            {
                name: 'MSE',
                displayName: 'Mean Squared Error (MSE)',
                info: 'Average of the squares of the errors',
                value: metrics.mse,
                std: metrics.msestd
            },
            {
                name: 'RMSE',
                displayName: 'Root Mean Squared Error (RMSE)',
                info: 'Square root of the MSE',
                value: metrics.rmse,
                std: metrics.rmsestd
            },
            {
                name: 'RMSLE',
                displayName: 'Root Mean Squared Logarithmic Error (RMSLE)',
                info: 'Root of the average of the squares of the natural log of the regression error',
                value: metrics.rmsle,
                std: metrics.rmslestd
            },
            {
                name: 'R2',
                displayName: 'R2 Score',
                info: '(Coefficient of determination) regression score function',
                value: metrics.r2,
                std: metrics.r2std,
                yAxisMax: 1,
                yAxisMin: 0,
            },
            {
                name: 'PEARSON',
                displayName: 'Pearson Coefficient',
                info: 'Correlation coefficient between actual and predicted values. +1 = perfect correlation, 0 =  no correlation, -1 = perfect anti-correlation',
                value: metrics.pearson,
                std: metrics.pearsonstd,
                yAxisMax: 1,
                yAxisMin: 0,
            }
        ];
        return _withCustomMetrics(metricsData, metrics.customMetricsResults);
    }

    function _getMulticlassMetricsData(modelData, perf) {
        const metrics = perf.metrics;
        const metricsData = [
            {
                name: 'ACCURACY',
                displayName: 'Accuracy',
                info: 'Proportion of correctly-classified observations',
                value: metrics.accuracy,
                std: metrics.accuracystd,
                yAxisMax: 1,
                yAxisMin: 0,
                isOneVsAll: false,
            },
            {
                name: 'PRECISION',
                displayName: 'Precision',
                info: 'Unweighted average of precision for all classes (Proportion of predicted X that were indeed X)',
                value: metrics.precision,
                std: metrics.precisionstd,
                yAxisMax: 1,
                yAxisMin: 0,
                isOneVsAll: true,
            },
            {
                name: 'RECALL',
                displayName: 'Recall',
                info: 'Unweighted average of recall for all classes (Proportion of actual class X found by the classifier)',
                value: metrics.recall,
                std: metrics.recallstd,
                yAxisMax: 1,
                yAxisMin: 0,
                isOneVsAll: true,
            },
            {
                name: 'F1',
                displayName: 'F1-score',
                info: 'Harmonic mean between Precision and Recall',
                value: metrics.f1,
                std: metrics.f1std,
                yAxisMax: 1,
                yAxisMin: 0,
                isOneVsAll: true,
            },
            {
                name: 'HAMMINGLOSS',
                displayName: 'Hamming Loss',
                info: 'Fraction of labels that are incorrectly predicted (the lower the better)',
                value: metrics.hammingLoss,
                std: metrics.hammingLossstd,
                yAxisMax: 1,
                yAxisMin: 0,
                isOneVsAll: false,
            },
            {
                name: 'ROC_AUC',
                displayName: 'ROC MAUC',
                info: 'Area under the ROC — From 0.5 (random model) to 1 (perfect model)',
                value: metrics.mrocAUC,
                std: metrics.mrocAUCstd,
                needsProbability: true,
                yAxisMax: 1,
                yAxisMin: 0.5,
                isOneVsAll: true,
            },
            {
                name: 'AVERAGE_PRECISION',
                displayName: 'Average Precision',
                info: 'Summarizes a precision-recall curve - 1 (perfect model), positive-rate (random model)',
                value: metrics.averagePrecision,
                std: metrics.averagePrecisionstd,
                needsProbability: true,
                yAxisMax: 1,
                yAxisMin: 0,
                isOneVsAll: true,
            },
            {
                name: 'LOG_LOSS',
                displayName: 'Log Loss',
                info: 'Error metric that takes into account the predicted probabilities (the lower the better)'
                    + (ModelDataUtils.hasCalibration(modelData) ? ' — Calibration might change the log loss as it modifies the predicted probabilities' : ''),
                value: metrics.logLoss,
                std: metrics.logLossstd,
                needsProbability: true,
                yAxisMin: 0,
                isOneVsAll: false,
            }
        ];
        if (!ModelDataUtils.isPartitionedBaseModel(modelData)) {
            metricsData.push({
                name: 'CALIBRATION_LOSS',
                displayName: 'Calibration Loss',
                info: 'Average distance between calibration curve and diagonal — From 0 (perfectly calibrated) up to 0.5',
                value: metrics.mcalibrationLoss,
                std: metrics.mcalibrationLossstd,
                needsProbability: true,
                yAxisMax: 0.5,
                yAxisMin: 0,
                isOneVsAll: true,
            });
        }
        return _withCustomMetrics(metricsData, metrics.customMetricsResults);
    }

    function _getThresholdIndependentBinaryClassifMetricsData(modelData, perf) {
        const metrics = perf.tiMetrics;
        const metricsData = []
        metricsData.push({
                name: 'ROC_AUC',
                displayName: 'ROC AUC',
                info: 'Area under the ROC — From 0.5 (random model) to 1 (perfect model)',
                value: metrics.auc,
                std: metrics.aucstd,
                needsProbability: true,
                yAxisMax: 1,
                yAxisMin: 0.5,
        });
        if (!ModelDataUtils.isPartitionedBaseModel(modelData)) {
            metricsData.push({
                name: 'CUMULATIVE_LIFT',
                displayName: $filter("mlMetricName")("CUMULATIVE_LIFT", modelData.modeling.metrics),
                info: 'Ratio between the true positive rate of this model and the one of a random model',
                value: metrics.lift,
                std: metrics.liftstd,
                needsProbability: true
            });
        }
        metricsData.push({
                name: 'AVERAGE_PRECISION',
                displayName: 'Average Precision',
                info: 'Summarizes a precision-recall curve - 1 (perfect model), positive-rate (random model)',
                value: metrics.averagePrecision,
                std: metrics.averagePrecisionstd,
                needsProbability: true,
                yAxisMax: 1,
                yAxisMin: 0,
        });
        metricsData.push({
                name: 'LOG_LOSS',
                displayName: 'Log Loss',
                info: 'Error metric that takes into account the predicted probabilities (the lower the better)'
                    + (ModelDataUtils.hasCalibration(modelData) ? ' — Calibration might change the log loss as it modifies the predicted probabilities' : ''),
                value: metrics.logLoss,
                std: metrics.logLossstd,
                needsProbability: true,
                yAxisMin: 0,
        });
        if (!ModelDataUtils.isPartitionedBaseModel(modelData)) {
            metricsData.push({
                name: 'CALIBRATION_LOSS',
                displayName: 'Calibration Loss',
                info: 'Average distance between calibration curve and diagonal — From 0 (perfectly calibrated) up to 0.5',
                value: metrics.calibrationLoss,
                std: metrics.calibrationLossstd,
                needsProbability: true,
                yAxisMax: 0.5,
                yAxisMin: 0,
            });
        }
        return _withCustomMetrics(metricsData, metrics.customMetricsResults);
    }

    function _getThresholdDependentBinaryClassifMetricsData(modelData, perf) {
        const metrics = BinaryClassificationModelsService.findCutData(perf, modelData.userMeta.activeClassifierThreshold);
        const metricsData = [
            {
                name: 'ACCURACY',
                displayName: 'Accuracy',
                info: 'Proportion of correct predictions (positive and negative) in the test set',
                value: metrics.Accuracy,
                std: metrics.accuracystd,
                yAxisMax: 1,
                yAxisMin: 0,
            },
            {
                name: 'PRECISION',
                displayName: 'Precision',
                info: 'Proportion of positive predictions that were indeed positive (in the test set)',
                value: metrics.Precision,
                std: metrics.precisionstd,
                yAxisMax: 1,
                yAxisMin: 0,
            },
            {
                name: 'RECALL',
                displayName: 'Recall',
                info: 'Proportion of actual positive values found by the classifier',
                value: metrics.Recall,
                std: metrics.recallstd,
                yAxisMax: 1,
                yAxisMin: 0,
            },
            {
                name: 'F1',
                displayName: 'F1-score',
                info: 'Harmonic mean between Precision and Recall',
                value: metrics['F1-Score'],
                std: metrics.f1std,
                yAxisMax: 1,
                yAxisMin: 0,
            },
            {
                name: 'COST_MATRIX',
                displayName: 'Cost Matrix Gain',
                info: `Average gain per record that the test set ` +
                    `(${ modelData.trainInfo.kfold ? modelData.trainInfo.fullRows : modelData.trainInfo.testRows } rows) ` +
                    `would yield given the specified gain for each outcome. Specified gains: TP = ${modelData.headTaskCMW.tpGain}, ` +
                    `TN = ${modelData.headTaskCMW.tnGain}, FP = ${modelData.headTaskCMW.fpGain}, FN = ${modelData.headTaskCMW.fnGain}.`,
                value: metrics.cmg
            },
            {
                name: 'HAMMINGLOSS',
                displayName: 'Hamming Loss',
                info: 'Fraction of labels that are incorrectly predicted (the lower the better)',
                value: metrics.hammingLoss,
                std: metrics.hammingLossstd,
                yAxisMax: 1,
                yAxisMin: 0,
            },
            {
                name: 'MCC',
                displayName: 'Matthews Correlation Coefficient',
                info: 'Correlation coefficient between actual and predicted values. +1 = perfect, 0 = no correlation, -1 = perfect anti-correlation',
                value: metrics.mcc,
                std: metrics.mccstd,
                yAxisMax: 1,
                yAxisMin: 0,
            }
        ];
        return _withCustomMetrics(metricsData, metrics.customMetricsResults).map((metricData) => {return {...metricData, isThresholdDependent: true}});
    }

    function _getTimeSeriesMetricsModelEvaluationData(modelData, perf) {
        const metricsData = _getTimeSeriesMetricsData(modelData, perf);
        const metrics = perf.aggregatedMetrics;
        metricsData.push(
            {
                name: 'WORST_MASE',
                displayName: 'Worst MASE',
                info: 'Worst (maximum) of the MASE metrics across all identifiers',
                value: metrics.worstMase
            },
            {
                name: 'WORST_MAPE',
                displayName: 'Worst MAPE',
                info: 'Worst (maximum) of the MAPE metrics across all identifiers',
                value: metrics.worstMape
            },
            {
                name: 'WORST_SMAPE',
                displayName: 'Worst SMAPE',
                info: 'Worst (maximum) of the SMAPE metrics across all identifiers',
                value: metrics.worstSmape
            },
            {
                name: 'WORST_MAE',
                displayName: 'Worst MAE',
                info: 'Worst (maximum) of the MAE metrics across all identifiers',
                value: metrics.worstMae
            },
            {
                name: 'WORST_MSE',
                displayName: 'Worst MSE',
                info: 'Worst (maximum) of the MSE metrics across all identifiers',
                value: metrics.worstMse
            },
            {
                name: 'WORST_MSIS',
                displayName: 'Worst MSIS',
                info: 'Worst (maximum) of the MSIS metrics across all identifiers',
                value: metrics.worstMsis
            }
        )
        return metricsData;
    }

    function _getTimeSeriesMetricsData(modelData, perf) {
        const metrics = perf.aggregatedMetrics;
        const metricsData = [
            {
                name: 'MASE',
                displayName: 'Mean Absolute Scaled Error (MASE)',
                info: 'Ratio between the mean absolute error of the forecast values and the mean absolute error of one-step naive forecast values',
                value: metrics.mase,
                std: metrics.masestd,
            },
            {
                name: 'MAPE',
                displayName: 'Mean Absolute Percentage Error (MAPE)',
                info: 'Average of the absolute value of the relative forecast error',
                value: metrics.mape,
                std: metrics.mapestd,
            },
            {
                name: 'SMAPE',
                displayName: 'Symmetric MAPE',
                info: 'Average of the absolute forecast error divided by the half-sum of the actual value and the forecast value',
                value: metrics.smape,
                std: metrics.smapestd
            },
            {
                name: 'MAE',
                displayName: 'Mean Absolute Error (MAE)',
                info: 'Average of the absolute value of the forecast error',
                value: metrics.mae,
                std: metrics.maestd
            },
            {
                name: 'MEAN_ABSOLUTE_QUANTILE_LOSS',
                displayName: 'Mean Absolute Quantile Loss',
                info: 'Average of the quantile losses, where a quantile loss is the sum of the absolute quantile forecast error weighted by the quantile value',
                value: metrics.meanAbsoluteQuantileLoss,
                std: metrics.meanAbsoluteQuantileLossstd
            },
            {
                name: 'MEAN_WEIGHTED_QUANTILE_LOSS',
                displayName: 'Mean Weighted Quantile Loss',
                info: 'Mean Absolute Quantile Loss divided by the sum of the actual values',
                value: metrics.meanWeightedQuantileLoss,
                std: metrics.meanWeightedQuantileLossstd
            },
            {
                name: 'MSE',
                displayName: 'Mean Squared Error (MSE)',
                info: 'Average of the squares of the forecast errors',
                value: metrics.mse,
                std: metrics.msestd
            },
            {
                name: 'RMSE',
                displayName: 'Root Mean Squared Error (RMSE)',
                info: 'Square root of the MSE',
                value: metrics.rmse,
                std: metrics.rmsestd
            },
            {
                name: 'MSIS',
                displayName: 'Mean Scaled Interval Score (MSIS)',
                info: 'Average of the squares of the forecast errors',
                value: metrics.msis,
                std: metrics.msisstd
            },
            {
                name: 'ND',
                displayName: 'Normalized Deviation',
                info: 'Average of the squares of the forecast errors',
                value: metrics.nd,
                std: metrics.ndstd
            }
        ];
        return _withCustomMetrics(metricsData, metrics.customMetricsResults);
    }

    function _getCausalMetricsData(modelData, metrics) {
        const metricsData = [
            {
                name: 'AUUC',
                displayName: 'Area Under the Uplift Curve',
                info: 'Area between the cumulative uplift curve and the random assignment line, see \'Uplift charts\'',
                value: metrics.auuc,
                std: metrics.auucstd
            },
            {
                name: 'QINI',
                displayName: 'Qini Score',
                info: 'Area between the Qini curve and the random assignment line, see \'Uplift charts\'',
                value: metrics.qini,
                std: metrics.qinistd
            },
            {
                name: 'NET_UPLIFT',
                displayName: $filter('mlMetricName')('NET_UPLIFT', modelData.modeling.metrics),
                info: 'Value of the cumulative uplift curve subtracted by the value of the random assignment line for a given % of the test population, see \'Uplift charts\'',
                value: metrics.netUplift,
                std: metrics.netUpliftstd
            }
        ];
        return _withCustomMetrics(metricsData, metrics.customMetricsResults);
    }

    function _getOverrideVariants(modelData, withAndWithoutOverrides) {
        if (withAndWithoutOverrides) {
            return [
                { label: 'no overrides', perf: modelData.perfWithoutOverrides },
                { label: 'with overrides', perf: modelData.perf },
            ];
        }
        return [{ label: '__single_variant__', perf: modelData.perf }];
    }

    return {
        getRegressionMetricsData: (modelData, withAndWithoutOverrides=false) => {
            let metricsAndVariants = _getOverrideVariants(modelData, withAndWithoutOverrides);
            return _getPolishedMetricsData(_getRegressionMetricsData, modelData, metricsAndVariants);
        },
        getMulticlassMetricsData: (modelData, withAndWithoutOverrides=false) => {
            let metricsAndVariants = _getOverrideVariants(modelData, withAndWithoutOverrides);
            return _getPolishedMetricsData(_getMulticlassMetricsData, modelData, metricsAndVariants);
        },
        getThresholdIndependentBinaryClassifMetricsData: (modelData, withAndWithoutOverrides=false) => {
            let metricsAndVariants = _getOverrideVariants(modelData, withAndWithoutOverrides);
            return _getPolishedMetricsData(_getThresholdIndependentBinaryClassifMetricsData, modelData, metricsAndVariants);
        },
        getThresholdDependentBinaryClassifMetricsData: (modelData, withAndWithoutOverrides=false) => {
            let metricsAndVariants = _getOverrideVariants(modelData, withAndWithoutOverrides);
            return _getPolishedMetricsData(_getThresholdDependentBinaryClassifMetricsData, modelData, metricsAndVariants);
        },
        getTimeSeriesMetricsData: (modelData) => _getPolishedMetricsData(
            _getTimeSeriesMetricsData, modelData, [{label: '__single_variant__', perf: modelData.perf }]
        ),
        getTimeSeriesMetricsModelEvaluationData: (modelData) => _getPolishedMetricsData(
            _getTimeSeriesMetricsModelEvaluationData, modelData, [{label: '__single_variant__', perf: modelData.perf }]
        ),
        getCausalMetricsData: (modelData) => _getPolishedMetricsData(
            _getCausalMetricsData, modelData,
            [{ label: 'raw', perf: modelData.perf.causalPerf.raw }, { label: 'normalized by absolute value of ATE', perf: modelData.perf.causalPerf.normalized }]
        ),
    };
});

app.component('detailedMetricsTable', {
    bindings: {
        metrics: '<',
        modelData: '<',
        puppeteerHookName: '@?'
    },
    templateUrl: '/templates/ml/prediction-model/detailed-metrics/detailed-metrics-table.html',
    controller: function($scope, $element, ModelDataUtils) {
        this.$onInit = () => {
            if (this.puppeteerHookName) {
                $element.children().first().attr(this.puppeteerHookName, '');
                $scope[this.puppeteerHookName] = true; // used to tell puppeteer that the component is ready
            }

            $scope.isWeighted = ModelDataUtils.areMetricsWeighted(this.modelData);
        };
    }
});

app.component('customMetricDetailsButton', {
    bindings: { metricData: '<' },
    templateUrl: '/templates/ml/prediction-model/detailed-metrics/custom-metric-details-button.html'
});

app.component('classificationMetricsExplanation', {
    bindings: { isOnPartitionedBaseModel: '<',
                classAveragingMethod: '<',
                modelData: '<',
                hasOptimalThreshold: '<',
                evaluationDetails: '<'
            },
    templateUrl: '/templates/ml/prediction-model/detailed-metrics/classification-metrics-explanation.html',
});

app.component('bcDetailedMetrics', {
    bindings: {
        modelData: '<',
        hasOptimalThreshold: '<',
        evaluationDetails: '<',
        withAndWithoutOverrides: '@?'
    },
    templateUrl: '/templates/ml/prediction-model/bc_detailed_metrics.html',
    controller: function($scope, PerformanceMetricsDataComposer, ModelDataUtils) {
        this.$onInit = () => {
            $scope.isOnPartitionedBaseModel = ModelDataUtils.isPartitionedBaseModel(this.modelData);
            const updateMetricsData = () => {
                $scope.tiMetricsData = PerformanceMetricsDataComposer.getThresholdIndependentBinaryClassifMetricsData(this.modelData, this.withAndWithoutOverrides);
                $scope.tdMetricsData = PerformanceMetricsDataComposer.getThresholdDependentBinaryClassifMetricsData(this.modelData, this.withAndWithoutOverrides);
            }

            const thresholdUpdateChecker = ModelDataUtils.createThresholdUpdateChecker();
            this.$doCheck = () => thresholdUpdateChecker.executeIfUpdated(this.modelData, updateMetricsData);
        };
    }
});

app.component('mcDetailedMetrics', {
    bindings: {
        modelData: '<',
        evaluationDetails: '<',
        withAndWithoutOverrides: '@?'
    },
    templateUrl: '/templates/ml/prediction-model/mc_detailed_metrics.html',
    controller: function($scope, PerformanceMetricsDataComposer, ModelDataUtils) {
        this.$onInit = () => {
            $scope.metricsData = PerformanceMetricsDataComposer.getMulticlassMetricsData(this.modelData, this.withAndWithoutOverrides);

            $scope.oneVsAllMetricsData = {...$scope.metricsData};
            $scope.oneVsAllMetricsData.rows = $scope.metricsData.rows.filter(metric => !!metric.isOneVsAll);
            $scope.notOneVsAllMetricsData = {...$scope.metricsData};
            $scope.notOneVsAllMetricsData.rows  = $scope.metricsData.rows.filter(metric => !metric.isOneVsAll);

            $scope.isOnPartitionedBaseModel = ModelDataUtils.isPartitionedBaseModel(this.modelData);
            $scope.classAveragingMethod = this.modelData.modeling.metrics.classAveragingMethod;
    }
}});

app.component('rDetailedMetrics', {
    bindings: {
        modelData: '<',
        evaluationDetails: '<',
        withAndWithoutOverrides: '@?'
    },
    templateUrl: '/templates/ml/prediction-model/r_detailed_metrics.html',
    controller: function($scope, PerformanceMetricsDataComposer, ModelDataUtils) {
        this.$onInit = () => {
            $scope.metricsData = PerformanceMetricsDataComposer.getRegressionMetricsData(this.modelData, this.withAndWithoutOverrides);
            $scope.isOnPartitionedBaseModel = ModelDataUtils.isPartitionedBaseModel(this.modelData);
        };
    }
});

app.component('tsDetailedMetrics', {
    bindings: { modelData: '<',  evaluationDetails: '<' },
    templateUrl: '/templates/ml/prediction-model/ts_detailed_metrics.html',
    controller: function($scope, PerformanceMetricsDataComposer, ModelDataUtils) {
        this.$onInit = () => {
            if (this.evaluationDetails) {
                $scope.metricsData = PerformanceMetricsDataComposer.getTimeSeriesMetricsModelEvaluationData(this.modelData);
            } else {
                $scope.metricsData = PerformanceMetricsDataComposer.getTimeSeriesMetricsData(this.modelData);
            }
            $scope.isOnPartitionedBaseModel = ModelDataUtils.isPartitionedBaseModel(this.modelData);
        };
    }
});

app.component('causalDetailedMetrics', {
    bindings: { modelData: '<' },
    templateUrl: '/templates/ml/prediction-model/causal_detailed_metrics.html',
    controller: function($scope, PerformanceMetricsDataComposer) {
        this.$onInit = () => {
            $scope.testAte = this.modelData.perf.causalPerf.testATE;
            $scope.metricsData = PerformanceMetricsDataComposer.getCausalMetricsData(this.modelData);
        };
    }
});

app.component('assertionsMetrics', {
    bindings: { modelData: '<', evaluationDetails: '<' },
    templateUrl: '/templates/ml/prediction-model/detailed-metrics/assertions-metrics.html',
    controller: function($scope, BinaryClassificationModelsService, ModelDataUtils, SavedModelsService, ModelEvaluationUtils) {
        const updateAssertionResult = () => {
            // used for initialization in multiclass & regression and for initialization and update for binary classification
            if (!$scope.modelData) return;
            let assertionsMetrics;

            if ($scope.isBinaryClassification) {
                $scope.currentCutData = BinaryClassificationModelsService.findCutData($scope.modelData.perf, $scope.modelData.userMeta.activeClassifierThreshold);
                if ($scope.modelData.perf.perCutData.assertionsMetrics) {
                    assertionsMetrics = $scope.modelData.perf.perCutData.assertionsMetrics[$scope.currentCutData.index];
                }
            } else {
                assertionsMetrics = $scope.modelData.perf.metrics && $scope.modelData.perf.metrics.assertionsMetrics;
            }
            if (assertionsMetrics) {
                // Reordering assertion results as {assertionName: {... assertion results ...}} for ease of use
                $scope.assertionsResult = {hasAnyDroppedRows: false};
                for (let assertionMetric of assertionsMetrics.perAssertion) {
                    $scope.assertionsResult[assertionMetric.name] = assertionMetric;
                    let nbDropped = assertionMetric.nbDroppedRows;
                    if (nbDropped > 0) {
                        $scope.assertionsResult.hasAnyDroppedRows = true;
                    }
                    let nbMatchingRows = assertionMetric.nbMatchingRows;
                    let nbPassing = Math.round(assertionMetric.validRatio * (nbMatchingRows - nbDropped));
                    let nbFailing = Math.round(nbMatchingRows - nbDropped - nbPassing);
                    $scope.assertionsResult[assertionMetric.name].nbPassing = nbPassing;
                    $scope.assertionsResult[assertionMetric.name].nbFailing = nbFailing;
                    $scope.assertionsResult[assertionMetric.name].passingPercentage = Math.round(100 * nbPassing / nbMatchingRows);
                    $scope.assertionsResult[assertionMetric.name].failingPercentage = Math.round(100 * nbFailing / nbMatchingRows);
                    $scope.assertionsResult[assertionMetric.name].droppedPercentage = Math.round(100 * nbDropped / nbMatchingRows);
                }
            }
        }

        $scope.getAssertionsDisabledReason = () => {
            const canHaveAssertions = ModelDataUtils.getAlgorithm(this.modelData)
                && !ModelDataUtils.isEnsemble(this.modelData)
                && this.modelData.backendType === 'PY_MEMORY'
                && !SavedModelsService.isExternalMLflowModel(this.modelData);

            if (!!this.evaluationDetails && !!this.evaluationDetails.evaluation) {
                return 'No Assertions were configured';
            }
            if (canHaveAssertions) {
                return 'No assertions were configured. Set them in Design > Debugging.'
            }
            if (ModelDataUtils.isEnsemble(this.modelData)) {
                return 'Assertions not available for ensemble models';
            }
            if (SavedModelsService.isMLflowModel(this.modelData)) {
                return 'Assertions not available for MLflow models';
            }
            if (SavedModelsService.isProxyModel(this.modelData)) {
                return 'Assertions not available for External Models';
            }
            if (!this.modelData.backendType) {
                return ModelEvaluationUtils.hasNoAssociatedModelText(this.evaluationDetails);
            }
            if (this.modelData.backendType !== 'PY_MEMORY') {
                return 'Assertions are only available for in-memory python backend';
            }
        };

        this.$onInit = () => {
            $scope.modelData = this.modelData;
            $scope.isOnPartitionBaseModel = ModelDataUtils.isPartitionedBaseModel(this.modelData);
            $scope.isRegression = ModelDataUtils.isRegression(this.modelData);
            $scope.isClassification = ModelDataUtils.isClassification(this.modelData);
            $scope.isBinaryClassification = ModelDataUtils.isBinaryClassification(this.modelData);

            if ($scope.isBinaryClassification) {
                const thresholdUpdateChecker = ModelDataUtils.createThresholdUpdateChecker();
                this.$doCheck = () => thresholdUpdateChecker.executeIfUpdated(this.modelData, updateAssertionResult);
            }
            updateAssertionResult();
        }

    }
});

// in a model, go to report tab, performance, metrics and assertions, there is a compute learning curve button
app.component('computeLearningCurveForm', { // For now it's lie, it's not a form
    bindings: {
        computeCurveFn: '<',
        alreadyComputedNumberOfPoints: '<',
    },
    template: `
        <div class="learning-curves-form__wrapper">
            <button class="btn btn--primary btn--contained" ng-click="$ctrl.onComputeButtonClick(numberOfPoints)">
                Compute learning curves
            </button>
            <button class="btn btn--secondary btn--outline" ng-click="$ctrl.openModal()">
                <i class="icon-gear mright0"></i>
            </button>
        </div>
    `,
    controller: function ($scope, CreateModalFromTemplate) {
        const $ctrl = this;
        $scope.numberOfPoints = 4;

        $ctrl.$onChanges = function() {
            setNumberOfPoints()
        }

        function setNumberOfPoints() {
            if ($ctrl.alreadyComputedNumberOfPoints)  {
                $scope.numberOfPoints = $ctrl.alreadyComputedNumberOfPoints;
                $scope.currentNumberOfPoints = $ctrl.alreadyComputedNumberOfPoints;
            }
        }

        $ctrl.onComputeButtonClick = function() {
            $ctrl.computeCurveFn($scope.numberOfPoints);
        }

        $ctrl.openModal = function() {
            CreateModalFromTemplate("/templates/ml/prediction-model/detailed-metrics/learning-curves-modal.html", $scope, null, function(newScope) {
                newScope.save = () => {
                    $scope.numberOfPoints = newScope.numberOfPoints;
                    newScope.dismiss();
                }
            }, false, false, true)
        };
    }
});

app.component('learningCurves', {
    bindings: {
        modelData: '<',
        activeClassifierThreshold: '<', // passed through to learningCurvesPlot,
    },
    templateUrl: '/templates/ml/prediction-model/detailed-metrics/learning-curves.html',
    controller: function ($scope, DataikuAPI, FutureProgressModal, PerformanceMetricsDataComposer, ModelDataUtils, PMLSettings, WT1) {
        const $ctrl = this;
        $ctrl.learningCurvePoints = [];
        $ctrl.availableMetrics = [];
        $ctrl.modelData = {};
        // The reduction ratio could be calculated in the doctor and stored in the learning_curve.json file
        $ctrl.trainRows = 0; // train size
        $ctrl.kfold = false;
        $ctrl.showCurve = false;
        $ctrl.showEmptyState = false;
        $ctrl.displayedTrainPoints = [];
        $ctrl.displayedTestPoints = [];

        $ctrl.$onInit = () => {
            $ctrl.predictionType = $ctrl.modelData.coreParams.prediction_type;
            $ctrl.kfold = $ctrl.modelData.trainInfo.kfold;
            $ctrl.fullModelId = $ctrl.modelData.fullModelId;
            if ($ctrl.kfold) {
                $ctrl.trainRows = $ctrl.modelData.trainInfo.fullRows;
            } else {
                $ctrl.trainRows = $ctrl.modelData.trainInfo.trainRows;
            }
            getExistingLearningCurveData();
        };

        $ctrl.$onChanges = () => {
            if ($ctrl.metric && $ctrl.metric.isThresholdDependent) {
                setLearningCurvesAndMetrics($ctrl.rawLearningCurveResult);
            }
        }

        /**
         * Computes the metricsData (as done in the detailed metrics above the learning curves)
         * This allows not to duplicate the code
         */
        function getMetricsData(perf) {
            const fakeModelData = angular.copy({
                ...$ctrl.modelData,
                perf,
            }); //getting a deepcopy so that we can modify it for cmg for example
            switch ($ctrl.predictionType) {
                case "BINARY_CLASSIFICATION":
                    fakeModelData.userMeta.activeClassifierThreshold = $ctrl.activeClassifierThreshold;
                    fakeModelData.perf.perCutData.cmg = ModelDataUtils.computeCmgForGivenPerf($ctrl.modelData, perf);
                    return [
                        ...PerformanceMetricsDataComposer.getThresholdIndependentBinaryClassifMetricsData(fakeModelData, false).rows,
                        ...PerformanceMetricsDataComposer.getThresholdDependentBinaryClassifMetricsData(fakeModelData, false).rows,
                    ];
                case "MULTICLASS":
                    return PerformanceMetricsDataComposer.getMulticlassMetricsData(fakeModelData, false).rows;
                case "REGRESSION":
                    return PerformanceMetricsDataComposer.getRegressionMetricsData(fakeModelData, false).rows;
            }
        };

        /**
         * Gets the list of metrics from a curve point
         * the learningCurvePoint needs to have already computed metricsData from getMetricsData()
         */
        function getAvailableMetrics(learningCurvePoint) {
            const rawMetrics = learningCurvePoint.train_metrics;
            return rawMetrics.map((metric) => {return {
                name: metric.name,
                displayName: metric.displayName,
                isThresholdDependent: !!metric.variants[0].isThresholdDependent,
                isEvaluationMetric: metric.isEvaluationMetric,
                lowerBetter: PMLSettings.sort.lowerBetter.includes(metric.name),
                yAxisMax: metric.variants[0].yAxisMax,
                yAxisMin: metric.variants[0].yAxisMin,
            }})
        }

        /**
         * Turns raw learning curves into the current metric points.
         * needs to be updated if the threshold is changing.
         */
        function setLearningCurvesAndMetrics(rawLearningCurveResult) {
            if (!rawLearningCurveResult || !rawLearningCurveResult.perf_points) {
                return;
            }
            const learningCurvePoints = [];
            for (const point of rawLearningCurveResult.perf_points) {
                learningCurvePoints.push({
                    ... point,
                    train_metrics: getMetricsData(point.train_metrics),
                    test_metrics: getMetricsData(point.test_metrics),
                });
            }
            $ctrl.learningCurvePoints = learningCurvePoints;
            if (!($ctrl.availableMetrics && $ctrl.availableMetrics.length)) {
                $ctrl.availableMetrics = getAvailableMetrics(learningCurvePoints[0]);
            }
            if (!$ctrl.metric && $ctrl.availableMetrics) {
                const evaluationMetric = $ctrl.availableMetrics.find((metric) => metric.isEvaluationMetric);
                $ctrl.metric = evaluationMetric ? evaluationMetric : $ctrl.availableMetrics[0];
            }
            $ctrl.setCurrentMetricPoints();
            $ctrl.alreadyComputedNumberOfPoints = $ctrl.learningCurvePoints.length;
            $ctrl.showEmptyState = false;
        }

        function getMetricScore(point, metricName, metricDisplayName, isTrainScore) {
            const pointMetrics = isTrainScore ? point.train_metrics : point.test_metrics;
            // Need to have name and displayName corresponding because all custom metrics have CUSTOM as name
            const metric = pointMetrics.find(metric => metric.name === metricName && metric.displayName === metricDisplayName);
            if (metric && metric.variants && metric.variants.length) {
                return metric.variants[0].value;
            }
        }

        /**
         * Changes the currentMetricPoints to the selected metric.
         * Needs to be called when changin metric to display.
         */
        $ctrl.setCurrentMetricPoints = function() {
            if ($ctrl.metric && $ctrl.metric.name) {
                $ctrl.currentMetricPoints = $ctrl.learningCurvePoints.map((point) => {
                    return {
                        trainSize: point.train_size,
                        testSize: point.test_size,
                        trainScore: getMetricScore(point, $ctrl.metric.name, $ctrl.metric.displayName, true),
                        testScore: getMetricScore(point, $ctrl.metric.name, $ctrl.metric.displayName, false),
                        trainTime: point.train_time,
                    }
                });
            }
        }

        const getExistingLearningCurveData = () => {
            DataikuAPI.ml.prediction.getLearningCurves($ctrl.fullModelId).success((rawLearningCurveResult) => {
                if (rawLearningCurveResult && rawLearningCurveResult.perf_points && rawLearningCurveResult.perf_points.length) {
                    $ctrl.rawLearningCurveResult = rawLearningCurveResult;
                    setLearningCurvesAndMetrics($ctrl.rawLearningCurveResult);
                } else {
                    $ctrl.showEmptyState = true;
                }
            }).error((data, status, headers) => {
                $ctrl.showEmptyState = true;
                setErrorInScope.bind($scope)(data, status, headers);
            });
        };

        $ctrl.computeLearningCurves = (numberOfPoints) => {
            DataikuAPI.ml.prediction.learningCurvesStart($ctrl.fullModelId, numberOfPoints).success((result) => {
                FutureProgressModal.show($scope, result, "Computing learning curve").then((rawLearningCurveResult) => {
                    $ctrl.rawLearningCurveResult = rawLearningCurveResult;
                    setLearningCurvesAndMetrics($ctrl.rawLearningCurveResult);
                });
            }).error(setErrorInScope.bind($scope));
            WT1.event("learning-curves-compute", {
                predictionType: $ctrl.predictionType,
            });
        };
    }
});

app.component('learningCurvesPlot', {
    bindings: {
        points: '<', // { trainSize: 0, trainScore: 0, testSize: 0, testScore: 0, trainTime: 0 }[]
        yAxisMin: '<',
        yAxisMax: '<',
        initialTrainRows: '<',
        kfold: '<',
    },
    template: `
        <div>
            <div block-api-error />
            <ng2-lazy-echart [options]="$ctrl.chartOptions" ng-if="$ctrl.chartOptions"></ng2-lazy-echart>
        </div>
    `,
    controller: function () {
        const $ctrl = this;
        $ctrl.initialTrainRows = 0;
        $ctrl.kfold = false;

        function getSizeRatio(trainSize, testSize, initialTrainSize, kfold) {
            if (kfold) {
                // SC-151406: handle this properly when doing suport for kfold
                return 100 * trainSize / (initialTrainSize - testSize);
            } else {
                return 100 * trainSize / initialTrainSize;
            }
        }

        $ctrl.$onChanges = function () {
            if ($ctrl.points && $ctrl.points.length) {
                $ctrl.chartOptions = buildChartOptions(
                    $ctrl.points,
                    $ctrl.initialTrainRows,
                    $ctrl.yAxisMin,
                    $ctrl.yAxisMax,
                );
            }
        };

        function timeFormatter(seconds) {
            if (seconds < 1) {
                const miliseconds = Math.round(seconds * 1000)
                return `${miliseconds}ms`;
            }
            if (seconds < 10) {
                return `${seconds.toFixed(3)}s`;
            } if (seconds < 60) {
                return `${Math.round(seconds)}s`;
            }
            else {
                const minutes = Math.floor(seconds / 60);
                const remaining_seconds = Math.round(seconds % 60);
                if (seconds === 0) {
                    return `${minutes}min`;
                }
               return `${minutes}min ${remaining_seconds}s`;
            }
        }

        // could be refactored not to be copy-pasted
        function getLegendMarker(color) {
            return `<span style="display:inline-block;margin-right:5px;border-radius:10px;width:10px;height:10px;background-color:${color};"></span>`; // no need to sanitize colors since they are hard coded
        }

        function buildChartOptions(dataPoints, initialTrainSize, yAxisMin, yAxisMax) {
            const trainColour = "#4CAF50"; //success-green-base
            // const trainIntervalColor = "#F49D55"; //warning orange-lighten-1
            const testColour = "#3B99FC"; //digital-blue-base
            // const testIntervalColor = "#BDD8ED"; //numeric-blue-lighten-1
            // SC-151406: interval colors used when kfold support done

            const trainData = dataPoints.map(({trainSize, trainScore}) => [trainSize, trainScore]);
            const testData = dataPoints.map(({trainSize, testScore}) => [trainSize, testScore]);

            return {
                textStyle: { fontFamily: 'SourceSansPro' },
                tooltip: {
                    trigger: 'item',
                    borderColor: '#999999',
                    textStyle: { color: '#000' },
                    formatter: (params) => {
                        const point = $ctrl.points[params.dataIndex];
                        const trainSize = point.trainSize;
                        const trainRatio = getSizeRatio(point.trainSize, point.testSize, initialTrainSize, $ctrl.kfold);
                        return `
                        <span>Model trained using ${sanitize(trainSize)} rows (${sanitize(trainRatio.toFixed(0))}% of the train dataset)</span>
                        <br/>
                        Train time: ${sanitize(timeFormatter(point.trainTime))}
                        <br/>
                        ${getLegendMarker(trainColour)}Train Score: ${sanitize(point.trainScore)}
                        <br/>
                        ${getLegendMarker(testColour)}Test Score: ${sanitize(point.testScore)}
                    `;
                    }
                },
                grid: {
                    top: 32,
                    right: 32,
                    left: 32,
                    bottom: 24,
                    height: 330,
                    containLabel: true,
                },
                xAxis: {
                    type: 'value',
                    scale: false,
                    name: 'Training examples',
                    nameLocation: 'center',
                    nameGap: 24,
                    min: 0,
                    max: "dataMax",
                    axisLine: { lineStyle: { color: '#000' } },
                },
                yAxis: [
                    {
                        type: 'value',
                        scale: true,
                        position: 'left',
                        name: 'Score',
                        nameLocation: 'center',
                        nameGap: 40,
                        axisLine: { lineStyle: { color: '#000' } },
                        min: function(value) {
                            if (typeof yAxisMin !== "number") {
                                return; // will automatically determine the min
                            }
                            if (yAxisMin <= value.min) { // The values to display are in the limits set for the metric
                                return yAxisMin;
                            }
                            else { // some value(s) are outside the expected limits: we let Echarts manage the y axis limit
                                return;
                            }
                        },
                        max: function(value) {
                            if (typeof yAxisMax !== "number") {
                                return; // will automatically determine the max
                            }
                            if (yAxisMax >= value.max) {
                                return yAxisMax;
                            }
                            else { // some value(s) are outside the expected limits: we let Echarts manage the y axis limit
                                return;
                            }
                        },
                    }
                ],
                legend: {
                    show: true,
                    right: 32,
                    icon: 'circle',
                },
                series: [
                    {
                        data: trainData,
                        color: trainColour,
                        name: 'train',
                        type: 'line',
                        yAxisIndex: 0,
                    },
                    {
                        data: testData,
                        color: testColour,
                        name: 'test',
                        type: 'line',
                        symbolSize: 4,
                        yAxisIndex: 0,
                    },
                ]
            };
        }
    }
});

})();
