(function() {
  'use strict';

const app = angular.module('dataiku.admin.config', []);

app.service('AppConfigService', function($rootScope) {
    const appConfig = $rootScope.appConfig;
    const deploymentMode = appConfig.deploymentMode;

    this.isDeploymentCloud = function() {
        return deploymentMode === 'CLOUD';
    }

    this.getRaw = function() {
        return appConfig;
    }

    this.get = function(key) {
        return appConfig[key];
    }
});

}()); 
;
(function(){
    'use strict';

    var services = angular.module('dataiku.services');

    services.factory('Diagnostics', function ($timeout, DataikuAPI, WT1, MessengerUtils) {
        function getLatest(success) {
            DataikuAPI.admin.diagnostics.getLatest().success(success);
        }
        function downLoadLatest () {
            var url = "/dip/api/admin/diagnostics/get-results";
            diagnosisDownloader.attr('src', url);
            $('body').append(diagnosisDownloader);
        }
        var diagnosisDownloader = $('<iframe>').attr('id', 'diagnosis-downloader');

        return {
            getLatest: getLatest, // only fetches metadata
            downLoadLatest: downLoadLatest
        };
    });

    services.factory('Logs', function(DataikuAPI) {

        function download(logFileName) {
            var url = "/dip/api/admin/logs/get-files?name=";
            if (logFileName) {
                url += logFileName;
            }
            logsDownloader.attr('src', url);
            $('body').append(logsDownloader);
        }

        function downloadCluster(clusterId, logName) {
            var url = "/dip/api/clusters/stream-log?clusterId=" + encodeURIComponent(clusterId) + "&logName=" + encodeURIComponent(logName);
            logsDownloader.attr('src', url);
            $('body').append(logsDownloader);
        }

        function downloadPod(clusterId, podName, namespace) {
            var url = "/dip/api/clusters/k8s/monitoring/stream-pod-log?clusterId=" + encodeURIComponent(clusterId) + "&podName=" + encodeURIComponent(podName) + "&namespace=" + encodeURIComponent(namespace);
            logsDownloader.attr('src', url);
            $('body').append(logsDownloader);
        }

        function downloadAll() {
            download(null)
        }

        function list() {
            return DataikuAPI.admin.logs.list();
        }

        function cat(logFileName) {
            return DataikuAPI.admin.logs.get(logFileName);
        }

        var logsDownloader = $('<iframe>').attr('id', 'logs-downloader');

        return {
            list: list,
            cat: cat,
            download: download,
            downloadAll: downloadAll,
            downloadCluster: downloadCluster,
            downloadPod: downloadPod
        };
    });
})();
;
(function () {
'use strict';

var app = angular.module('dataiku.admin', []);

app.controller("AdminGeneralSettingsController", function (
    $scope,
    $state,
    $stateParams,
    $timeout,
    $q,
    translate,
    DataikuAPI,
    WT1,
    ActivityIndicator,
    TopNav,
    CodeMirrorSettingService,
    TaggingService,
    FutureProgressModal,
    Dialogs,
    $rootScope,
    $filter,
    GlobalProjectActions,
    $anchorScroll,
    DeployerUtils,
    FutureWatcher,
    CreateModalFromTemplate,
    Assert,
    localStorageService,
    ProjectFolderService,
    PathUtils,
    PromiseService,
    $window,
    FeatureFlagsService,
    $http
) {
    if ($state.is('admin.general')) {
        $state.go('admin.general.themes');
    }

    // to be used by sub-controllers
    $scope.setErrorInGeneralSettingsControllerScope = setErrorInScope.bind($scope);

    $scope.hasUrlSuffix = DeployerUtils.hasUrlSuffix;

    $scope.httpMethods = ['GET', 'POST', 'OPTIONS', 'PUT', 'DELETE', 'HEAD'];

    $scope.virtualWebAppBackendSettingsModes = [{id:"USE_DEFAULT", label:"Run as local processes"}, {id:"EXPLICIT", label:"Run in container"}];

    $scope.globalVariables = {}
    TopNav.setLocation(TopNav.DSS_HOME, "administration");

    $scope.welcomeEmailTest = {
        testing: false,
        success: false,
        error: null,
    };
    $scope.welcomeEmailTest.recipient = $scope.welcomeEmailTest.recipient || $rootScope.appConfig.user.email || "";

    $scope.testWelcomeEmail = function() {
        $scope.welcomeEmailTest.testing = true;
        $scope.welcomeEmailTest.success = false;
        $scope.welcomeEmailTest.error = null;
        DataikuAPI.admin.testWelcomeEmail(
            $scope.welcomeEmailTest.recipient,
            $scope.generalSettings.welcomeEmailSettings,
            $scope.generalSettings.notifications,
            $scope.generalSettings.studioMailAddress,
            $scope.channels,
        )
            .success(function(result) {
                $scope.welcomeEmailTest.success = true;
            })
            .error(function(error) {
                $scope.welcomeEmailTest.error = error;
            })
            .finally(function() {
                $scope.welcomeEmailTest.testing = false;
            });
    };


    // ** Default folder rules stuff

    // all users virtual group - for catch-all / fall back folder rule
    const ALL_USERS_GROUP_DISPLAY = 'All Users';
    const ALL_USERS_GROUP_INTERNAL = '$$ALL_USERS$$';

    $scope.getProjectFolderName = (item) => item.name;

    $scope.browseFolder = folderIds => {
        // Use last id in path
        $scope.destination = PathUtils.makeNLNT(folderIds).split('/').pop();
        // browse-path expects a success-error promise so we need to wrap with qToHttp for now (catch() does not return a monkey-patched promise)
        return PromiseService.qToHttp(ProjectFolderService.getBrowseNode($scope.destination)
            .catch(setErrorInScope.bind($scope)));
    };

    // Cache a promise for root folder as we use it for every new folder rule
    const ROOT_FOLDER_PROMISE = ProjectFolderService.getFolderWithPath('ROOT');

    function setFolderOnRuleFromPromise(rule, promise) {
        promise.then((folder) => {
            if (rule.folderId === "ROOT") {
                folder.pathElts = '/';
            }
            rule.$folder = folder;
        }).catch(setErrorInScope.bind($scope));
    }

    // Adds the $folder object which gives the folder-path-input enough info to start browsing
    // and gives us the path info to show in the form (for 'ROOT' it is just '/')
    // If the folder was deleted we display "(missing)" but allow the user to browse from root to replace it
    function setFolderOnRule(rule) {
        setFolderOnRuleFromPromise(rule, rule.folderId === "ROOT" ? ROOT_FOLDER_PROMISE : ProjectFolderService.getFolderFallbackRoot(rule.folderId, "(missing)"));
    }

    //insert new rules just above All Users group
    $scope.addFolderRulePosition = function() {
        let allGroupIndex = $scope.generalSettings.defaultFolderSettings.rules.findIndex( (rule) => rule.group === ALL_USERS_GROUP_INTERNAL);
        return allGroupIndex === -1 ? $scope.generalSettings.defaultFolderSettings.rules.length : allGroupIndex;
    }

    $scope.folderRuleTemplate = function() {
        const rule = { group : ""};
        rule.folderId = "ROOT";
        setFolderOnRule(rule);
        return rule;
    }

    let groupsPreparedForDefaultFolderUI = [];
    // Exposing groups in a callback for reliable use via transcopes
    $scope.getGroupsPreparedForDefaultFolderUI = () => groupsPreparedForDefaultFolderUI;

    function addMissingGroupsToGroupOptions(rule) {
        if (groupsPreparedForDefaultFolderUI.every(g => g.id != rule.group)) {
            const displayName = rule.group + " (missing)";
            const missingGroup = { display : displayName, id : rule.group};
            groupsPreparedForDefaultFolderUI.unshift(missingGroup);
        }
    }

    // When we retrieve the folder settings from the backend, add things to make the UI work and handle edge cases
    $scope.processFolderSettings = function(defaultFolderSettings) {
        $scope.allGroupsPromise.then((rawGroups) => {
            groupsPreparedForDefaultFolderUI = rawGroups.map((g) => ({ display : g, id : g}));
            groupsPreparedForDefaultFolderUI.push({display : ALL_USERS_GROUP_DISPLAY, id : ALL_USERS_GROUP_INTERNAL});

            if (defaultFolderSettings && defaultFolderSettings.rules) {
                defaultFolderSettings.rules.forEach(rule => {
                    addMissingGroupsToGroupOptions(rule);
                    // TODO replace this with a call to a new backend operation which does getFolderSummariesFallbackRoot on an array
                    setFolderOnRule(rule);
                });
            }
         });
    }

    // ** end of folder rules stuff



    $scope.alationRegister = {
    }
    $scope.urlWithProtocolAndHost = $window.urlWithProtocolAndHost();

    $scope.editorOptions = CodeMirrorSettingService.get('text/plain', {
        onLoad: function(cm) {
            $scope.codeMirror = cm;
        }
    });
    $scope.htmlEditorOptions = CodeMirrorSettingService.get('text/html', {
        onLoad: function(cm) {
            $scope.codeMirror = cm;
        }
    });

    DataikuAPI.security.listUsers().success(function(data) {
        $scope.allUsers = data.sort((a, b) => a.displayName.localeCompare(b.displayName));
        $scope.allUsersLogin = data.map(user => '@' + user.login);
    }).error(setErrorInScope.bind($scope));

    $scope.allGroupsPromise = DataikuAPI.security.listGroups(false).error(setErrorInScope.bind($scope))
        .then(function({data}) {
            $scope.allGroups = data.sort((a, b) => a.localeCompare(b));
            return $scope.allGroups;
        });

    DataikuAPI.admin.connections.list().success(function(data) {
        $scope.connectionsByName = data;
        $scope.connections = Object.values(data);
    }).error(setErrorInScope.bind($scope));

    $scope.isAzureConnectionOAuth = function (connectionName) {
        return connectionName in $scope.connectionsByName
        && $scope.connectionsByName[connectionName].type == "Azure"
        && $scope.connectionsByName[connectionName].params.authType == "OAUTH2_APP";
    }

    var savedGeneralSettings, savedGlobalVariables, savedChannels;
    $scope.dirtySettings = function () {
        return !angular.equals($scope.generalSettings, savedGeneralSettings) ||
            !angular.equals($scope.globalVariables.asJSON, savedGlobalVariables) ||
            !angular.equals($scope.channels, savedChannels);
    };

    function allowedTransitionsFn(data) {
        return (data.toState && data.fromState && data.toState.name.startsWith('admin.general') && data.fromState.name.startsWith('admin.general'));
    }
    checkChangesBeforeLeaving($scope, $scope.dirtySettings, null, allowedTransitionsFn);

    $scope.hasBoundaryWhiteSpace = function (stringValue) {
        const leadingOrTrailingWhitespace = /^\s+|\s+$/;
        return leadingOrTrailingWhitespace.test(stringValue);
    };

    $scope.anyHaveBoundaryWhiteSpace = function (stringValues) {
        for(let value of stringValues) {
            if ($scope.hasBoundaryWhiteSpace(value)) {
                return true;
            }
        }
       return false;
    };

    // Hackish : to check for the existence of a nodes directory, we test to see if the deployer is registered in the nodes directory.
    // only use this in a context where you already know that the deployer is active.
    $scope.deployerAndNodesDirectoryEnabled = $scope.appConfig.nodesDirectoryManagedDeployerServer || $scope.appConfig.nodesDirectoryManagedDeployerClient;

    DataikuAPI.unifiedMonitoring.deployer.getRemoteDesignNodesUrls().success(function(data) {
        $scope.designNodes = data;
    }).error(setErrorInScope.bind($scope));

    $scope.load = function () {

        var promises = [];
        promises.push(DataikuAPI.admin.getGeneralSettings());
        promises.push(DataikuAPI.admin.getGlobalVariables());
        promises.push(DataikuAPI.admin.integrationChannels.list());
        promises.push(DataikuAPI.admin.getThemes());

        $scope.promises = $q.all(promises).then(
            function (values) {
                //general settings
                $scope.generalSettings = values[0].data;
                savedGeneralSettings = angular.copy($scope.generalSettings);
                $scope.savedGeneralSettings = angular.copy($scope.generalSettings); // allow to check local dirtiness
                $scope.processFolderSettings($scope.generalSettings.defaultFolderSettings);

                //global variables
                $scope.globalVariables.asJSON = values[1].data;
                savedGlobalVariables = angular.copy($scope.globalVariables.asJSON);
                //messaging channels
                $scope.channels = values[2].data;
                savedChannels = angular.copy($scope.channels);
                //themes
                $scope.themes = values[3].data;

                // fixup customFieldsPluginComponentOrder
                $scope.generalSettings.customFieldsPluginComponentOrder = $scope.generalSettings.customFieldsPluginComponentOrder
                    .filter(ref1 => $scope.appConfig.customFieldsPluginComponentRefs.find(ref2 => ref2.pluginId == ref1.pluginId && ref2.componentId == ref1.componentId));
                for (let i = 0; i < $scope.appConfig.customFieldsPluginComponentRefs.length; i++) {
                    const ref1 = $scope.appConfig.customFieldsPluginComponentRefs[i];
                    const existingRef = $scope.generalSettings.customFieldsPluginComponentOrder.find(ref2 => ref2.pluginId == ref1.pluginId && ref2.componentId == ref1.componentId);
                    if (!existingRef) {
                        $scope.generalSettings.customFieldsPluginComponentOrder.push(ref1);
                    }
                }
                // fixup customPolicyHooksPluginComponentOrder
                $scope.generalSettings.customPolicyHooksPluginComponentOrder = $scope.generalSettings.customPolicyHooksPluginComponentOrder
                    .filter(ref1 => $scope.appConfig.customPolicyHooksPluginComponentRefs.find(ref2 => ref2.pluginId == ref1.pluginId && ref2.componentId == ref1.componentId));
                for (let i = 0; i < $scope.appConfig.customPolicyHooksPluginComponentRefs.length; i++) {
                    const ref1 = $scope.appConfig.customPolicyHooksPluginComponentRefs[i];
                    const existingRef = $scope.generalSettings.customPolicyHooksPluginComponentOrder.find(ref2 => ref2.pluginId == ref1.pluginId && ref2.componentId == ref1.componentId);
                    if (!existingRef) {
                        $scope.generalSettings.customPolicyHooksPluginComponentOrder.push(ref1);
                    }
                }
            },
            function (errors) {
                setErrorInScope.bind($scope);
            }
        ).then(() => {
            // If URL contains a hash, wait for the settings to be loaded before scrolling to the anchor
            $timeout($anchorScroll);
        });
    };

    $scope.$watchCollection("channels", function (nv) {
        if (nv) {
            $scope.mailChannels = $scope.channels.filter(function (channel) {
                return ["aws-ses-mail", "microsoft-graph-mail", "smtp"].includes(channel.type);
            });
            if ($scope.generalSettings && $scope.mailChannels.map(c => c.id).indexOf($scope.generalSettings.notifications.emailChannelId) < 0) {
                $scope.generalSettings.notifications.emailChannelId = void 0;
            }
        }
    });

    $scope.noneUsersCallToActionBehaviorChoices = [];

    $scope.$watch("generalSettings.licensingSettings.trialBehavior", function(nv , ov) {
        if (nv == "DISABLED") {
            $scope.noneUsersCallToActionBehaviorChoices =[
                ['DISPLAY_MESSAGE', 'None, only display a message'],
                ['ALLOW_REQUEST_ACCESS', 'User can request access']
            ]
            if ($scope.generalSettings.licensingSettings.noneUsersCallToActionBehavior === 'ALLOW_START_TRIAL') {
                $scope.generalSettings.licensingSettings.noneUsersCallToActionBehavior = 'ALLOW_REQUEST_ACCESS';
            }
        } else {
            $scope.noneUsersCallToActionBehaviorChoices =[
                ['DISPLAY_MESSAGE', 'None, only display a message'],
                ['ALLOW_START_TRIAL', 'User can start a trial'],
                ['ALLOW_REQUEST_ACCESS', 'User can request access']
            ]
        }
    });

    $scope.availableUserProfilesForTrials = window.dkuAppConfig.licensing.userProfiles.filter(p => p != "NONE").map(p => [p,p]);

    $scope.load();

    // optionally scroll to a specific place of the page (use scroll-to-me directive)
    // after the promises because data may changes scroll position
    if($stateParams.scrollTo) {
        $scope.promises.then(() => {
            $scope.scrollToMe = $stateParams.scrollTo;
        });
    }

    $scope.autofillStudioUrl = function() {
        $scope.generalSettings.studioExternalUrl = urlWithProtocolAndHost();
    };

    function removeEmptyGitConfigurationOptions() {
        $scope.generalSettings.git.enforcedConfigurationRules.forEach((configRule) => {
            configRule.gitConfigurationOptions = configRule.gitConfigurationOptions.filter(option => !option.$invalid);
        });
    }

    $scope.invalidTabs = new Set();

    $scope.$on("$stateChangeStart", function(event, toState, toParams, fromState) {
        // We do not set 'Resource control' tab as invalid to avoid weird UI behavior. For this tab, a ng-model is not
        // changed if the new input value is not valid. Hence if a user exits the 'Resource control' tab with some
        // invalid fields and then switch back to it, the fields will no longer be invalid, which can be confusing.
        if ($scope.adminGeneralIndexForm.$invalid && fromState.name !== 'admin.general.limits') {
            $scope.invalidTabs.add(fromState.name);
        }
        if ($scope.adminGeneralIndexForm.homeArticlesListInvalid) {
            $scope.invalidTabs.add('admin.general.help');
        }
        if ($scope.adminGeneralIndexForm.homepagePromotedContentInvalid) {
            $scope.invalidTabs.add('admin.general.homepage');
        }
        $timeout(function() {
            $scope.invalidTabs.delete(toState.name);
        });
    });

    $scope.isAdminGeneralIndexFormInvalid = function() {
        return $scope.adminGeneralIndexForm.$invalid || $scope.invalidTabs.size || $scope.adminGeneralIndexForm.homeArticlesListInvalid || $scope.adminGeneralIndexForm.homepagePromotedContentInvalid;
    }

    function fetchGlobalTagsIfChanged() {
        if (!angular.equals($scope.generalSettings.globalTagsCategories, savedGeneralSettings.globalTagsCategories)) {
            TaggingService.fetchGlobalTags(true);
        }
    }

    function checkForDuplicateNames (list, type) {
        let names = [];

        list.forEach(function(element) {
            if (!element.name) {
                throw({message: "Found empty " + type + " name"});
            }
            if (names.includes(element.name)) {
                throw({message: "Found duplicate " + type + " names: " + element.name});
            }
            names.push(element.name);
        });
    }

    $scope.saveGeneralSettings = function() {
        if (savedGeneralSettings.udrMode != $scope.generalSettings.udrMode) {
            WT1.event("usage-reporting-mode-changed", { mode: $scope.generalSettings.udrMode });
        }
        return DataikuAPI.admin.saveGeneralSettings($scope.generalSettings).success(function(data) {
            fetchGlobalTagsIfChanged();
            savedGeneralSettings = angular.copy($scope.generalSettings);
            $scope.savedGeneralSettings = angular.copy($scope.generalSettings); // allow to check local dirtiness
            if (data.anyMessage) {
                Dialogs.infoMessagesDisplayOnly($scope, "Save warnings", data);
            }
            $scope.reloadAppConfig(); // Reload $rootScope.appConfig & friends so that hasGlobalProxy (among other) is correctly (re-)initialized
        }).error(setErrorInScope.bind($scope));
    };

    $scope.updateGlobalTags = function() {
        if (angular.equals($scope.generalSettings.globalTagsCategories, savedGeneralSettings.globalTagsCategories)) return;
        let updatedGlobalTagsMap = {};
        let update = false;
        const toDelete = ['isEdited', 'isNew', 'originalTagName', 'removeUsage'];
        //Map the global tags with their original- and updated name to update the tag on object across the instance
        $scope.generalSettings.globalTagsCategories.forEach(function(category, index) {
            category.globalTags.forEach(function(tag, idx) {
                let originalCategoryName = savedGeneralSettings.globalTagsCategories[index] && savedGeneralSettings.globalTagsCategories[index].name;
                if (originalCategoryName && (tag.originalTagName || originalCategoryName != category.name)) {
                    let oldGlobalTagName = `${originalCategoryName}:${tag.originalTagName || tag.name}`
                    updatedGlobalTagsMap[oldGlobalTagName] = {color: tag.color, updatedTagName: `${category.name}:${tag.name}`, globalTagsCategory: category.name, removeUsage: tag.removeUsage};
                    update = true;
                }
                var it = $scope.generalSettings.globalTagsCategories[index].globalTags[idx];
                toDelete.forEach(k => delete it[k]);
            });
            delete category.isNew;
        });
        if (update) DataikuAPI.admin.globalTags.updateGlobalTags(updatedGlobalTagsMap);
        return updatedGlobalTagsMap;
    };

    $scope.invalidateConfigCache = function() {
        var options = {type: 'text'};
        $scope.cacheInvalidationError = {};
        Dialogs.prompt($scope, "Invalidate cache", "Path to invalidate", "", options)
               .then(function(path) {
                   DataikuAPI.admin.invalidateConfigCache(path).error(setErrorInScope.bind($scope.cacheInvalidationError));
                });
    };

    $scope.disablePermissionsByEmail = () => {
        return $scope.generalSettings && $scope.generalSettings.security && $scope.generalSettings.security.enableEmailAndDisplayNameModification;
    };

    const getModelCacheDetails = () => DataikuAPI.admin.getModelCacheDetails().success((futureRes) => FutureWatcher.watchJobId(futureRes.jobId).success((res) => $scope.modelCacheDetails = res.result));
    getModelCacheDetails();
    $scope.clearModelCache = function() {
        Dialogs.confirmSimple($scope, "Clear model cache?").then(function() {
            DataikuAPI.admin.clearModelCache().error(setErrorInScope.bind($scope))
            .success(function (resp) {
                FutureProgressModal.show($scope, resp, "Clearing model cache").then(function(result) {
                    ActivityIndicator.success("Cleared model cache", 5000);
                    getModelCacheDetails();
                });
            });
        });
    };

    $scope.openDeleteModelModal = function (modelKey) {
        Dialogs.confirmSimple($scope, "Delete " + modelKey + " ?").then(function() {
            DataikuAPI.admin.deleteModelFromCache(modelKey).error(setErrorInScope.bind($scope))
            .success(function (resp) {
                getModelCacheDetails();
            });
        });
    };

    $scope.exportModel = function (modelKey) {
        downloadURL("/dip/api/admin/model-cache/export?modelKey=" + modelKey);
    };

    $scope.openImportModelModal = function() {
        $scope.importData = {
            uploading: false
        };
        CreateModalFromTemplate("/templates/import-file.html", $scope, null, function(modalScope) {
            modalScope.fileName = "Model";
            const parentScope = $scope.$parent.$parent;
            modalScope.import = function() {
                Assert.trueish($scope.importData.files, "No model file");
                $scope.importData.uploading = true;
                DataikuAPI.admin.importModel($scope.importData.files).then(function (data) {
                    modalScope.dismiss();
                    FutureProgressModal.show(parentScope, JSON.parse(data), "Model import", undefined, 'static', false).then(function(result) {
                        Dialogs.infoMessagesDisplayOnly(parentScope, "Creation result", result.messages, result.futureLog, undefined, 'static', false);
                        getModelCacheDetails();
                    });
                }, function(payload) {
                    setErrorInScope.bind($scope)(JSON.parse(payload.response), payload.status, function(h) {return payload.getResponseHeader(h)});
                });
            };
        });
    }

    $scope.fetchOpenIDConfig = function(openIdWellKnown) {
        return DataikuAPI.admin.fetchOpenIDConfig(openIdWellKnown).success(function (config) {
            $scope.generalSettings.ssoSettings.openIDParams.issuer = config.issuer;
            $scope.generalSettings.ssoSettings.openIDParams.authorizationEndpoint = config.authorization_endpoint;
            $scope.generalSettings.ssoSettings.openIDParams.tokenEndpoint = config.token_endpoint;
            $scope.generalSettings.ssoSettings.openIDParams.jwksUri = config.jwks_uri;
        }).error(setErrorInScope.bind($scope));
    }

    $scope.computeWellKnowFromIssuer = function() {
        let issuer = $scope.generalSettings.ssoSettings.openIDParams.issuer;
        $scope.openIdWellKnown = issuer ?
            issuer + '/.well-known/openid-configuration' :
            "";
    }

    /**
    *   Save the generalSettings, global variables and integrations channels and perform all checks prior to that
    *   Needs to return a promise and shouldn't just return, as cmd+s serves a fallback to force save
    */
    $scope.save = function () {
        try {
            removeEmptyGitConfigurationOptions();

            checkForDuplicateNames($scope.generalSettings.containerSettings.executionConfigs, "container configuration");
            checkForDuplicateNames($scope.generalSettings.sparkSettings.executionConfigs, "Spark configuration");
            checkForDuplicateNames($scope.generalSettings.globalTagsCategories, "global category");

            const updatedGlobalTagsMap = $scope.updateGlobalTags();
            const gv = $scope.globalVariables.asJSON || '{}';
            return $scope.saveGeneralSettings().then(function () {
                return DataikuAPI.admin.integrationChannels.saveAll($scope.channels).success(function (data) {
                    $scope.channels = data;
                    savedChannels = angular.copy(data);
                    return DataikuAPI.admin.saveGlobalVariables(gv).success(function () {
                        savedGlobalVariables = angular.copy($scope.globalVariables.asJSON);
                        $scope.$broadcast('generalSettingsSaved', updatedGlobalTagsMap);
                        if ($scope.isAdminGeneralIndexFormInvalid()) {
                            const allInvalidTabs = new Set();
                            $scope.invalidTabs.forEach(tab => allInvalidTabs.add($state.get(tab).pageTitle()));
                            if ($scope.adminGeneralIndexForm.$invalid) {
                                allInvalidTabs.add($state.current.pageTitle());
                            }
                            if ($scope.adminGeneralIndexForm.homepagePromotedContentInvalid) {
                                allInvalidTabs.add($state.get('admin.general.homepage').pageTitle());
                            }
                            if ($scope.adminGeneralIndexForm.homeArticlesListInvalid) {
                                allInvalidTabs.add($state.get('admin.general.help').pageTitle());
                            }
                            const warningMessage = "Saved with some invalid fields in tab" +
                                ($scope.invalidTabs.size + $scope.adminGeneralIndexForm.$invalid > 1 ? "s '" : " '") +
                                [...allInvalidTabs].join("', '") + "'";
                            ActivityIndicator.warning(warningMessage);
                        } else {
                            ActivityIndicator.success("Saved!");
                        }
                        // special cases: flags that need to be in appConfig for the corresponding options to be available
                        // in the frontend: impalaEnabled, etc
                        // note : settings are not sent back from the backend, but that's fine because they are saved as is
                        $scope.appConfig.impalaEnabled = $scope.generalSettings.impalaSettings.enabled && $scope.appConfig.hadoopEnabled;
                        $scope.appConfig.pluginDevExplicitCommit = $scope.generalSettings.pluginDevExplicitCommit;
                        $scope.appConfig.npsSurveyEnabled = $scope.generalSettings.npsSurveyEnabled;
                        $scope.appConfig.nodeName = $scope.generalSettings.nodeName;
                        $scope.appConfig.helpIntegrationEnabled = $scope.generalSettings.helpIntegrationEnabled;
                        $scope.appConfig.openTicketsFromOpalsEnabled = $scope.generalSettings.openTicketsFromOpalsEnabled;
                        $scope.appConfig.projectStatusList = $scope.generalSettings.projectStatusList;
                        if (!$rootScope.featureFlagEnabled('homepageRedesign')) {
                            // TODO @homepage cleanup when removing feature flag
                            $scope.appConfig.homeMessages = $scope.generalSettings.homeMessages;
                        }
                        $scope.appConfig.apiDeployerStages = $scope.generalSettings.apiDeployerServerSettings.stages;
                        $scope.appConfig.projectVisibility = $scope.generalSettings.projectVisibility;
                        $scope.appConfig.appVisibility = $scope.generalSettings.appVisibility;
                        $scope.appConfig.studioForgotPasswordUrl = $scope.generalSettings.studioForgotPasswordUrl;
                        $scope.appConfig.autoAcceptSchemaChangeAtEndOfFlow = $scope.generalSettings.autoAcceptSchemaChangeAtEndOfFlow;
                        $scope.appConfig.uiCustomization = $scope.generalSettings.uiCustomizationSettings;
                        $scope.appConfig.quickSharingElementsEnabled = $scope.generalSettings.quickSharingElementsEnabled;
                        $scope.appConfig.pluginInstallRequestsEnabled = $scope.generalSettings.pluginInstallRequestsEnabled;
                        $scope.appConfig.opalsEnabled = $scope.generalSettings.opalsEnabled;
                        $scope.appConfig.codeEnvInstallRequestsEnabled = $scope.generalSettings.codeEnvInstallRequestsEnabled;
                    }).error(setErrorInScope.bind($scope));
                }).error(setErrorInScope.bind($scope));
            }).catch(setErrorInScope.bind($scope));
        } catch (err) {
            ActivityIndicator.error("Invalid format: "+err.message);
        }
    };

    $scope.registerAlationOpener = function(){
        $scope.save().then(function(){
            DataikuAPI.connections.registerAlationOpener($scope.alationRegister.alationAPIToken).success(function(){
                ActivityIndicator.success("Alation registration done")
            }).error(setErrorInScope.bind($scope));
        })
    }

    DataikuAPI.connections.getNames('SQL')
        .success(function (data) { $scope.sqlConnections = data; })
        .error(setErrorInScope.bind($scope));

    $scope.testLdapSettings = function () {
        $scope.ldapTesting = true;
        $scope.ldapTestResult = null;
        DataikuAPI.admin.testLdapSettings($scope.generalSettings.ldapSettings).success(function (data) {
            $scope.ldapTestResult = data;
        })
            .error(setErrorInScope.bind($scope))
            .finally(() => $scope.ldapTesting = false);
    };


    $scope.testLdapGetUserDetails = function (userName) {
        $scope.ldapUserTesting = true;
        $scope.ldapTestUserDetails = null;
        DataikuAPI.admin.testLdapGetUserDetails({
            settings:$scope.generalSettings.ldapSettings,
            username:userName
        }).success(function (data) {
            $scope.ldapTestUserDetails = data;
        })
            .error(setErrorInScope.bind($scope))
            .finally(() => $scope.ldapUserTesting = false);
    };

    $scope.testAzureADSettings = function (login, email) {
        $scope.azureADTesting = true;
        $scope.azureADTestResult = null;
        DataikuAPI.admin.testAzureADSettings(
        {
            "azureADSettings": $scope.generalSettings.azureADSettings,
            "userIdentity": {
                "login": login || "",
                "email": email || ""
            }
        }).success(function (data) {
            $scope.azureADTestResult = data;
            if ($scope.azureADTestResult.connectionOK) {
                $scope.azureADTestResult.userAttributes = JSON.stringify($scope.azureADTestResult.userAttributes, null, 2);
            }
        })
            .error(setErrorInScope.bind($scope))
            .finally(() => $scope.azureADTesting = false);
    };

    $scope.governIntegrationTest = function() {
        DataikuAPI.admin.govern.test().success(function (data) {
            $scope.governIntegrationTestResult = data;
        }).error(setErrorInScope.bind($scope));
    };

    $scope.governIntegrationParams = { projectKey : '' };
    $scope.governIntegrationFullSync = function() {
        DataikuAPI.admin.govern.sync($scope.governIntegrationParams.projectKey).success(function (data) {
            FutureProgressModal.show($scope, data, "Synchronizing DSS Items on Dataiku Govern", null, 'static', false, true).then(function(result){
                if (result) {
                    $scope.governIntegrationFullSyncResult = {
                        status: 'SUCCESS',
                        data: result
                    };
                }
            })
        }).error(function(data, status, headers) {
                $scope.governIntegrationFullSyncResult = {
                    status: 'ERROR',
                    error: getErrorDetails(data, status, headers)
                };
        });
    };

    $scope.governIntegrationFullDeployerSync = function() {
        DataikuAPI.admin.govern.deployerSync().success(function (data) {
            FutureProgressModal.show($scope, data, "Synchronizing DSS Deployer Items on Dataiku Govern", null, 'static', false, true).then(function(result){
                if (result) {
                    $scope.governIntegrationFullDeployerSyncResult = {
                        status: 'SUCCESS',
                        data: result
                    };
                }
            })
        }).error(function(data, status, headers) {
                $scope.governIntegrationFullDeployerSyncResult = {
                    status: 'ERROR',
                    error: getErrorDetails(data, status, headers)
                };
        });
    };

    $scope.checkDatastoryIntegration = function() {
        DataikuAPI.admin.datastory.check().success(function (data) {
            $scope.datastoryIntegrationCheckResult = data;
        }).error(setErrorInScope.bind($scope));
    };

    $scope.addLDAPGroupProfile = function () {
        var groupProfiles = $scope.generalSettings.ldapSettings.groupProfiles

        if (!groupProfiles || !$.isArray(groupProfiles)) {
            $scope.generalSettings.ldapSettings.groupProfiles = []
        }
        $scope.generalSettings.ldapSettings.groupProfiles.push({key: '', value: $scope.generalSettings.ldapSettings.userProfile});
    };

    $scope.deleteLDAPGroupProfile = function (index) {
        $scope.generalSettings.ldapSettings.groupProfiles.splice(index, 1)
    };

    $scope.addSSOGroupProfile = function () {
        var groupProfiles = $scope.generalSettings.ssoSettings.groupProfiles

        if (!groupProfiles || !$.isArray(groupProfiles)) {
            $scope.generalSettings.ssoSettings.groupProfiles = []
        }
        $scope.generalSettings.ssoSettings.groupProfiles.push({key: '', value: $scope.generalSettings.ssoSettings.userProfile});
    };

    $scope.deleteSSOGroupProfile = function (index) {
        $scope.generalSettings.ssoSettings.groupProfiles.splice(index, 1)
    };

    $scope.addAzureADGroupProfile = function () {
        var groupProfiles = $scope.generalSettings.azureADSettings.groupProfiles

        if (!groupProfiles || !$.isArray(groupProfiles)) {
            $scope.generalSettings.azureADSettings.groupProfiles = []
        }
        $scope.generalSettings.azureADSettings.groupProfiles.push({key: '', value: $scope.generalSettings.azureADSettings.userProfile});
    };

    $scope.deleteAzureADGroupProfile = function (index) {
        $scope.generalSettings.azureADSettings.groupProfiles.splice(index, 1)
    };

    $scope.addCustomAuthGroupProfile = function () {
        var groupProfiles = $scope.generalSettings.customAuthSettings.groupProfiles

        if (!groupProfiles || !$.isArray(groupProfiles)) {
            $scope.generalSettings.customAuthSettings.groupProfiles = []
        }
        $scope.generalSettings.customAuthSettings.groupProfiles.push({key: '', value: $scope.generalSettings.customAuthSettings.userProfile});
    };

    $scope.deleteCustomAuthGroupProfile = function (index) {
        $scope.generalSettings.customAuthSettings.groupProfiles.splice(index, 1)
    };

    $scope.getChannelTypeLabel = function (type) {
        if (!type) {
            return "Unknown";
        } else if (type === 'msft-teams') {
            return "Microsoft Teams";
        } else if (type === 'google-chat') {
            return "Google Chat";
        } else if (type === 'aws-ses-mail') {
            return "Mail (via Amazon SES)";
        } else if(type === 'microsoft-graph-mail') {
            return "Mail (via Microsoft 365 with OAuth)"
        } else if (type === 'smtp') {
            return "Mail (SMTP)";
        } else {
            return type.charAt(0).toUpperCase() + type.slice(1);
        }
    };

    $scope.addChannel = function (type) {
        var definition = {
            type : type,
            configuration : {
                sessionProperties: []
            },
            permissions: [{
                group: ALL_USERS_GROUP_INTERNAL,
                canUse: true,
            }],
            $creation : true //flag to allow id edition only on creation
        };
        if (type === 'slack' || type === 'webhook' || type === 'twilio' || type === 'msft-teams' || type === 'google-chat' ) {
            definition.configuration.useProxy = true;
        }
        if (type === 'slack') {
            definition.configuration.mode = 'WEBHOOK';
        }
        if (type === 'msft-teams') {
            definition.configuration.webhookType = 'WORKFLOWS';
        }
        $scope.channels.push(definition)
    };

    $scope.removeChannel = function (channel) {
        var index = $scope.channels.indexOf(channel);
        if (index >= 0) {
            $scope.channels.splice(index, 1);
        }
    };

    DataikuAPI.admin.clusters.listAccessible('HADOOP').success(function(data){
        $scope.clusters = [{id:null, name:'No override'}].concat(data);
    }).error(setErrorInScope.bind($scope));

    DataikuAPI.admin.clusters.listAccessible('KUBERNETES').success(function(data){
        $scope.k8sClusters = [{id:null, name:'No override'}].concat(data);
    }).error(setErrorInScope.bind($scope));

    $scope.DEPLOYER_MODES = [
        ["DISABLED", "Disabled"],
        ["LOCAL", "Local"],
        ["REMOTE", "Remote"]
    ]
    $scope.DEPLOYER_MODES_DESCRIPTIONS = [
        "Disable ability to publish models and projects on the Deployer",
        "Use this DSS instance as Deployer",
        "Publish models and projects on a remote Deployer"
    ]

    $scope.listOpalsUnsupportedWhiteLabelingSettings = function (whiteLabeling) {
        if (whiteLabeling === undefined) {
            return [];
        }

        let unsupportedSettings;

        if (whiteLabeling.productLongName !== "Dataiku DSS") {
            unsupportedSettings = ["productLongName"];
        } else {
            unsupportedSettings = [];
        }

        const stringSettings = [
            "referenceDocRootUrl",
            "aboutModalTitle",
            "aboutModalLogoUrl",
            "aboutModalText",
            "getHelpModalTitle",
            "getHelpModalText",
        ];

        const booleanSettings = [
            "contextualHelpSearchEnabled",
        ];

        return unsupportedSettings
            .concat(stringSettings.filter(s => whiteLabeling[s]))
            .concat(booleanSettings.filter(s => s in whiteLabeling && !whiteLabeling[s]));
    };

    $scope.syncOnboardingOpals = function() {
        $scope.generalSettings.onboardingExperience = $scope.generalSettings.opalsEnabled;
    }

    $scope.pushBaseImages = function(){
        if($scope.dirtySettings()) {
            $scope.save().then($scope.pushBaseImages_NoCheckDirty);
        } else {
            $scope.pushBaseImages_NoCheckDirty();
        }
    }

    $scope.pushBaseImages_NoCheckDirty = function(){
        DataikuAPI.admin.containerExec.pushBaseImages().success(function(data) {
            // FutureProgressModal.show($scope, data, "Pushing base images");
            FutureProgressModal.show($scope, data, "Pushing base images").then(function(result){
                if (result) { // undefined in case of abort
                    Dialogs.infoMessagesDisplayOnly($scope, "Push result", result.messages, result.futureLog);
                }
            })
        }).error(setErrorInScope.bind($scope));
    }
    const formatDefaults = function(defaults) {
        // Remap missing choice from options set to empty string for proper display
        // NB: No worries, backend does properly the reverse operation
        if (!defaults.cudaVersion) {
            defaults.cudaVersion = '';
        }
        if(!defaults.distrib) {
            defaults.distrib = '';
        }
        return defaults;
    };
    $scope.buildBaseImage = function(buildType, buildOptions) {
        $scope.saveGeneralSettings().then(() => DataikuAPI.admin.containerExec.getBaseImageDefaults(buildType))
        .then(({data}) => DataikuAPI.admin.containerExec.buildBaseImage(buildType, buildOptions || formatDefaults(data)))
        .then(({data}) => {
            FutureProgressModal.show($scope, data, "Building image").then(function(result){
                if (result) {
                    Dialogs.infoMessagesDisplayOnly($scope, "Image built", result, result.futureLog, true);
                }
            });
        }).catch(setErrorInScope.bind($scope));
    };
    $scope.buildCDEPluginsImage = () => {
        $scope.saveGeneralSettings().then(() => DataikuAPI.admin.containerExec.buildBaseImage("CDE_PLUGINS"))
        .then(({data}) => {
            FutureProgressModal.show($scope, data, "Building image").then(function(result){
                if (result) {
                    Dialogs.infoMessagesDisplayOnly($scope, "Image built", result, result.futureLog, true);
                }
            });
        }).catch(setErrorInScope.bind($scope));
    }
    $scope.setupBuildBaseImage = function(buildType) {
        DataikuAPI.admin.containerExec.getBaseImageDefaults(buildType).then(({data}) => {
            CreateModalFromTemplate("/templates/admin/fragments/build-base-image-modal.html", $scope, "BuildBaseImageModalController", function(modalScope) {
                modalScope.availableDistributions = _.concat([["", "Use default"]], data.AVAILABLE_DISTRIBUTIONS.map(x => [x, x.charAt(0).toUpperCase() + x.slice(1)])); // uppercase first letter
                modalScope.availableCudaVersions = _.concat([["", "No CUDA"]], data.AVAILABLE_CUDA_VERSIONS.map(x => [x, "CUDA " + x]));
                modalScope.buildType = buildType;
                modalScope.buildOptions = formatDefaults(data);
                modalScope.buildBaseImage = $scope.buildBaseImage;
            });
        })
        .catch(setErrorInScope.bind($scope));
    };

    $scope.fillInBoilerplateCustomWelcomeEmail = function() {
        if ($scope.generalSettings != null && $scope.generalSettings.welcomeEmailSettings != null && $scope.generalSettings.welcomeEmailSettings.sendAsHTML &&
                !$scope.generalSettings.welcomeEmailSettings.useDefaultTemplate && !$scope.generalSettings.welcomeEmailSettings.message) {
            $http.get('templates/admin/general/inline-boilerplate-custom-welcome-email.html').then(function(response) {
                $scope.generalSettings.welcomeEmailSettings.message = response.data;
            });
        }
    }

    $scope.datasetTypesOptions = (() => {
        const datasetName = $filter('datasetTypeToName');
        const rawDatasetsTiles = GlobalProjectActions.getAllDatasetByTilesNoFilter(); // call without scope is valid because we won't use the clickCallbacks fields.
        const res = [];
        rawDatasetsTiles.forEach(tile => {
            const allTypes = tile.types.concat(tile.types2 || [])
            allTypes.forEach(type => {
                if(type === undefined) return; // ignore the ones hidden by feature-flags.
                res.push({
                    tile: tile.title,
                    value: type.type,
                    displayName: type.label !== undefined ? type.label : datasetName(type.type),
                });
            });
        });

        // this is a special case, the 'search_and_import' behavior doesn't exist in the new dataset page.
        res.push({
            tile: 'Import existing',
            value: 'search_and_import',
            displayName: translate('PROJECT.ACTIONS.SEARCH_AND_IMPORT', 'Search and import\u2026'),
        });
        return res;
    })();


    $scope.highlightedDatasets = {
        // this function returns the list of choices for a list item, eg the full list minus the items already selected elsewhere
        getOptions: (currentSelected) => {
            return $scope.datasetTypesOptions.filter(datasetTypeOption => {
                return datasetTypeOption.value === currentSelected
                    || $scope.generalSettings.uiCustomizationSettings.highlightedDatasets.indexOf(datasetTypeOption.value) === -1;
            });
        },
        // convert the config format into the data format for editable-list
        init: () => {
            $scope.highlightedDatasets.selection = $scope.generalSettings.uiCustomizationSettings.highlightedDatasets.map(type => ({type}));
        },
        // reverse convert the editable-list into the config (and remove empty values)
        onChange: (nv) => {
            $scope.generalSettings.uiCustomizationSettings.highlightedDatasets = nv.map(({type}) => type).filter(x => x);
        },
    };

    // Init the highlightedDatasets values as soon as config is loaded
    $scope.promises.then($scope.highlightedDatasets.init);

    // find available LLMs
    DataikuAPI.codeAssistant.listAvailableLLMs().success(function(data){
        $scope.availableLLMs = data.identifiers;
    }).error(setErrorInScope.bind($scope));

    $scope.openAIServicesTermsOfUseModal = function() {
        CreateModalFromTemplate("/templates/admin/general/ai-services-terms-of-use-modal.html", $scope, null, function(modalScope) {
            modalScope.uiState = {
                termsAccepted: false
            }
            modalScope.confirm = function() {
                $scope.generalSettings.aiDrivenAnalyticsSettings.dataikuAIServicesTermsOfUseAccepted = true;
                $scope.generalSettings.aiDrivenAnalyticsSettings.dataikuAIServicesTermsOfUseAcceptedOn = moment().toISOString(true);
                $scope.generalSettings.aiDrivenAnalyticsSettings.dataikuAIServicesTermsOfUseAcceptedBy = $rootScope.appConfig.login;
                modalScope.dismiss();
            }
        })
    }

    $scope.llmCostLimitingEnabled = window.dkuAppConfig.llmSettings.costLimitingEnabled;
    $scope.llmRateLimitingEnabled = window.dkuAppConfig.llmSettings.rateLimitingEnabled;
});

app.controller("AdminGenAIController", function ($scope, $state, DataikuAPI, TopNav) {
    TopNav.setLocation(TopNav.DSS_HOME, "administration");

    $scope.useInternalCodeEnvLabel = "Use internal code env";

    DataikuAPI.codeenvs.listNames('PYTHON').then(function({data}) {
        $scope.codeEnvs = data;
        $scope.codeEnvItemsListWithDefault = [{"label": $scope.useInternalCodeEnvLabel, "value": undefined}].concat(data.map(codeEnv => ({"label": codeEnv, "value": codeEnv})));
    }).catch(setErrorInScope.bind($scope));

    DataikuAPI.pretrainedModels.listAvailableConnectionLLMs("TEXT_EMBEDDING_EXTRACTION").success(function(data){
        $scope.availableEmbeddingLLMs = data["identifiers"];
    }).error(setErrorInScope.bind($scope));

    DataikuAPI.pretrainedModels.listAvailableConnectionLLMs("GENERIC_COMPLETION").success(function(data){
        $scope.availableCompletionLLMs = data["identifiers"];
    }).error(setErrorInScope.bind($scope));

    DataikuAPI.pretrainedModels.listAvailableConnectionLLMs("IMAGE_INPUT").success(function(data){
        $scope.availableVLMs = data["identifiers"];
        $scope.addNoneOption();
    }).error(setErrorInScope.bind($scope));

    $scope.addNoneOption = function() {
        $scope.availableVLMs.unshift({ // make sure to have it as 1st option
            id: "DSS_NO_SELECTION",
            friendlyName: "No VLM (text-only extraction)",
            type: "" // no group
        });
    }

    let piiDetectionInternalCodeEnvChecked = false;
    DataikuAPI.codeenvs.checkDSSInternalCodeEnv("PII_DETECTION_CODE_ENV").then(function({data}) {
        if (Object.keys(data).length > 0) {
            $scope.piiDetectionInternalCodeEnv = data.value;
        }
        piiDetectionInternalCodeEnvChecked = true;
    }).catch(setErrorInScope.bind($scope));

    let ragInternalCodeEnvChecked = false;
    DataikuAPI.codeenvs.checkDSSInternalCodeEnv("RAG_CODE_ENV").then(function({data}) {
        if (Object.keys(data).length > 0) {
            $scope.ragInternalCodeEnv = data.value;
        }
        ragInternalCodeEnvChecked = true;
    }).catch(setErrorInScope.bind($scope));

    let documentExtractionCodeEnvChecked = false;
    DataikuAPI.codeenvs.checkDSSInternalCodeEnv("DOCUMENT_EXTRACTION_CODE_ENV").then(function({data}) {
        if (Object.keys(data).length > 0) {
            $scope.docextractionInternalCodeEnv = data.value;
        }
        documentExtractionCodeEnvChecked = true;
    }).catch(setErrorInScope.bind($scope));

    $scope.piiDetectionInternalCodeEnvExists = function() {
        return piiDetectionInternalCodeEnvChecked && $scope.piiDetectionInternalCodeEnv != null;
    }

    $scope.ragInternalCodeEnvExists = function() {
        return ragInternalCodeEnvChecked && $scope.ragInternalCodeEnv != null;
    }

    $scope.piiDetectionCodeEnvIsInternal = function () {
        return $scope.generalSettings && $scope.generalSettings.generativeAISettings && (
            $scope.generalSettings.generativeAISettings.presidioBasedPIIDetectionCodeEnv == null || (
                $scope.piiDetectionInternalCodeEnvExists() && $scope.generalSettings.generativeAISettings.presidioBasedPIIDetectionCodeEnv == $scope.piiDetectionInternalCodeEnv.envName
            )
        );
    }

    $scope.ragCodeEnvIsInternal = function () {
        return $scope.generalSettings && $scope.generalSettings.generativeAISettings && (
            $scope.generalSettings.generativeAISettings.defaultRetrievableKnowledgeCodeEnv == null || (
                $scope.ragInternalCodeEnvExists() && $scope.generalSettings.generativeAISettings.defaultRetrievableKnowledgeCodeEnv == $scope.ragInternalCodeEnv.envName
            )
        );
    }

    $scope.showPiiDetectionCodeEnvWarning = function () {
        return piiDetectionInternalCodeEnvChecked && $scope.generalSettings && $scope.generalSettings.generativeAISettings && (
                (!$scope.piiDetectionInternalCodeEnvExists()) || (!$scope.piiDetectionCodeEnvIsInternal())
            );
    }

    $scope.showRagCodeEnvWarning = function () {
        return ragInternalCodeEnvChecked && $scope.generalSettings && $scope.generalSettings.generativeAISettings && (
                (!$scope.ragInternalCodeEnvExists()) || (!$scope.ragCodeEnvIsInternal())
            );
    }

    $scope.fixupNullishPiiDetectionCodeEnvForDirtynessCheck = function() {
        // if the presidioBasedPIIDetectionCodeEnv is unset in the backend, it will be initialized as non-set (~undefined) in the frontend
        // when selecting "Use internal code env", the code env selector forces this value to null
        // we need to fix this to avoid the dirtyness check failing due to null !== undefined
        if ($scope.generalSettings.generativeAISettings.presidioBasedPIIDetectionCodeEnv == null) {
            $scope.generalSettings.generativeAISettings.presidioBasedPIIDetectionCodeEnv = undefined;
        }
    }

    $scope.fixupNullishEmbedDocumentTextExtractionCodeEnvForDirtynessCheck = function() {
        if ($scope.generalSettings.generativeAISettings.embedDocumentsRecipeSettings.textExtractionCodeEnv == null) {
            $scope.generalSettings.generativeAISettings.embedDocumentsRecipeSettings.textExtractionCodeEnv = undefined;
        }
    }

    $scope.fixupNullishRagCodeEnvForDirtynessCheck = function() {
        // if the defaultRetrievableKnowledgeCodeEnv is unset in the backend, it will be initialized as non-set (~undefined) in the frontend
        // when selecting "Use internal code env", the code env selector forces this value to null
        // we need to fix this to avoid the dirtyness check failing due to null !== undefined
        if ($scope.generalSettings.generativeAISettings.defaultRetrievableKnowledgeCodeEnv == null) {
            $scope.generalSettings.generativeAISettings.defaultRetrievableKnowledgeCodeEnv = undefined;
        }
    }

    $scope.internalCodeEnvsHRef = function() {
        if ($scope.appConfig.isAutomation) {
            return $state.href("admin.codeenvs-automation.internal");
        } else {
            return $state.href("admin.codeenvs-design.internal");
        }
    }

    $scope.availableTraceExplorerWebApps = [];
    $scope.$watch('generalSettings.generativeAISettings.llmTraceSettings.traceExplorerDefaultWebApp.projectKey', function (newProjectKey, oldProjectKey) {
        if (newProjectKey) {
            DataikuAPI.taggableObjects.listAccessibleObjects(newProjectKey, 'WEB_APP').then(function (response) {
                $scope.availableTraceExplorerWebApps = response.data.filter((object) => object.subtype === 'webapp_traces-explorer_traces-explorer');
            });
        }

        if (newProjectKey === oldProjectKey) {
            return;
        }

        if ($scope.generalSettings.generativeAISettings.llmTraceSettings.traceExplorerDefaultWebApp.webAppId && oldProjectKey) {
            $scope.generalSettings.generativeAISettings.llmTraceSettings.traceExplorerDefaultWebApp.webAppId = null;
        }
        $scope.availableTraceExplorerWebApps = [];
    });

});

app.controller("AdminCostLimitingController", function ($scope, $interval, $state, DataikuAPI, Dialogs, CreateModalFromTemplate, ClipboardUtils) {
    const FALLBACK_ID = "DKU-FALLBACK-QUOTA";
    $scope.getSortExpression = function(expression) {
        let fnExpression = {
            'isFiltered': $scope.isFiltered,
            'getTotalCostSpent': $scope.getTotalCostSpent,
            'getCostProgress': $scope.getCostProgress,
            'periodicityAsMinutes': $scope.periodicityAsMinutes,
        }[expression];
        return fnExpression ? fnExpression : expression;
    };
    $scope.selection = {
        orderQuery: "", orderReversed: false, filterQuery: "",
    }; // used for sorting table

    const llmConnectionTypes = ['Anthropic', 'AzureLLM', 'AzureOpenAI', 'Bedrock', 'Cohere', 'DatabricksLLM', 'HuggingFaceLocal', 'mistral', 'OpenAI',
                                    'SageMaker-GenericLLM', 'SnowflakeCortex', 'StabilityAI', 'VertexAILLM'];
    DataikuAPI.admin.connections.list().success(function(data) {
        $scope.llmConnections = Object.values(data)
            .filter(c => llmConnectionTypes.includes(c.type))
            .map(c => ({label: c.name, value: c.name}))
            .sort((a, b) => a.label.localeCompare(b.label));
    }).error(setErrorInScope.bind($scope));

    DataikuAPI.projects.list().success(function (projects) {
        $scope.availableProjects = projects
            .map(p => ({label: p.name + " (" + p.projectKey + ")", value: p.projectKey}))
            .sort((a, b) => a.label.localeCompare(b.label));
    }).error(setErrorInScope.bind($scope));

    DataikuAPI.security.listUsers().success(function(data) {
        $scope.allUsers = data
            .map(u => ({label: u.displayName + " (" + u.login + ")", value: u.login}))
            .sort((a, b) => a.label.localeCompare(b.label));
    }).error(setErrorInScope.bind($scope));

    const allCostLimitingPeriodicities = [
        {"label": "Calendar Year", "value": "YEAR", "allowRollingPeriods": false, "asMinutes": 365 * 24 * 60},
        {"label": "Calendar Quarter", "value": "QUARTER", "allowRollingPeriods": false, "asMinutes": 90 * 24 * 60},
        {"label": "Calendar Month", "value": "MONTH", "allowRollingPeriods": false, "asMinutes": 30 * 24 * 60},
        {"label": "Rolling Day(s)", "value": "DAY", "allowRollingPeriods": true, "asMinutes": 24 * 60},
        {"label": "Rolling Minute(s)", "value": "MINUTE", "allowRollingPeriods": true, "asMinutes": 1},
    ]
    $scope.costLimitingPeriodicities = allCostLimitingPeriodicities.filter(p => window.dkuAppConfig.llmSettings.costLimitingPeriodicities.includes(p.value));
    const periodicityMap = $scope.costLimitingPeriodicities.reduce((acc, p) => { acc[p.value] = p; return acc }, {});

    $scope.onCostLimitingPeriodicityChange = function(quota) {
        const periodicity = $scope.costLimitingPeriodicities.find(p => p.value === quota.periodicity);
        if (periodicity && !periodicity.allowRollingPeriods) {
            quota.rollingPeriods = 1;
        }
    };

    $scope.shouldDisplayRollingPeriods = function(quota) {
        const periodicity = periodicityMap[quota.periodicity]
        return periodicity && periodicity.allowRollingPeriods;
    };

    $scope.getPeriodicitySummary = function(quota) {
        const periodicity = periodicityMap[quota.periodicity]
        if (!periodicity) return "";
        if (periodicity.allowRollingPeriods) {
            return quota.rollingPeriods + " " + periodicity.label;
        } else {
            return periodicity.label;
        }
    };

    $scope.periodicityAsMinutes = function(quota) {
        return quota.rollingPeriods * periodicityMap[quota.periodicity].asMinutes;
    };

    $scope.costLimitingQuotaFilterSchema = {
            'columns': [
                {
                    name: 'project',
                    type: 'enum',
                    $$groupKey: 'DSS items'
                },
                {
                    name: 'user',
                    type: 'enum',
                    $$groupKey: 'DSS items'
                },
                {
                    name: 'provider',
                    type: 'enum',
                    $$groupKey: 'DSS items'
                },
                {
                    name: 'connection',
                    type: 'enum',
                    $$groupKey: 'DSS items'
                },
                {
                    name: 'llmId',
                    type: 'string',
                    $$groupKey: 'Advanced fields'
                }, 
                {
                    name: 'projectKey',
                    type: 'string',
                    $$groupKey: 'Advanced fields'
                },
                {
                    name: 'userLogin',
                    type: 'string',
                    $$groupKey: 'Advanced fields'
                },
                {
                    name: 'connectionName',
                    type: 'string',
                    $$groupKey: 'Advanced fields'
                },
            ] 
        };

    $scope.providers = [
        {
            'label': 'Anthropic',
            'value': 'Anthropic'
        },
        {
            'label': 'Azure LLM',
            'value': 'AzureLLM'
        },
        {
            'label': 'Azure OpenAI',
            'value': 'AzureOpenAI'
        },
        {
            'label': 'AWS Bedrock',
            'value': 'Bedrock'
        },
        {
            'label': 'Cohere',
            'value': 'Cohere'
        },
        // {
        //     'label': 'Databricks Mosaic AI',
        //     'value': 'DatabricksLLM'
        // },
        // {
        //     'label': 'Local Hugging Face',
        //     'value': 'HuggingFaceLocal'
        // },
        {
            'label': 'Mistral AI',
            'value': 'MistralAI'
        },
        {
            'label': 'OpenAI',
            'value': 'OpenAI'
        },
        {
            'label': 'Python Agents',
            'value': 'python-agent'
        },
        // {
        //     'label': 'Amazon SageMaker LLM',
        //     'value': 'SageMaker-GenericLLM'
        // },
        // {
        //     'label': 'Snowflake Cortex',
        //     'value': 'SnowflakeCortex'
        // },
        {
            'label': 'Stability AI',
            'value': 'StabilityAI'
        },
        // Visual agents can't report direct costs atm
        // {
        //     'label': 'Visual Agents',
        //     'value': 'tools-using-agent'
        // },
        {
            'label': 'Vertex AI',
            'value': 'VertexAILLM'
        },
    ];
    const _allCustomLLMPlugins = [...$scope.appConfig.customPythonLLMs, ...$scope.appConfig.customJavaLLMs];
    // Append Custom LLM plugins
    $scope.providers = $scope.providers.concat(
            Array.from(new Set(_allCustomLLMPlugins.map(llm => llm.ownerPluginId)))
            .map(pluginID => $scope.appConfig.loadedPlugins.find(plugin => plugin.id === pluginID))
            .filter(plugin => plugin) // remove any `undefined` entries from plugins that could not be found
            .map(p => ({label: "LLM Plugin: " + p.label, value: "CustomLLM:" + p.id}))
            .sort((a,b) => a.label.localeCompare(b.label))
    );
    // Append Agent plugins
    $scope.providers = $scope.providers.concat(
        Array.from(new Set($scope.appConfig.customAgents.map(agent => agent.ownerPluginId)))
            .map(pluginID => $scope.appConfig.loadedPlugins.find(plugin => plugin.id === pluginID))
            .filter(plugin => plugin) // remove any `undefined` entries from plugins that could not be found
            .map(p => ({label: "Plugin Agent: " + p.label, value: "plugin-agent:" + p.id}))
            .sort((a,b) => a.label.localeCompare(b.label))
    );

    $scope.getCostLimitingEnumValues = function(columnName) {
        if (columnName == 'provider') {
            return $scope.providers;
        } else if (columnName == 'project') {
            return $scope.availableProjects;
        } else if (columnName == 'connection') {
            return $scope.llmConnections;
        } else if (columnName == 'user') {
            return $scope.allUsers;
        }
        return null;
    }

    $scope.getThresholdAmount = function(quota, reportingAction) {
        if (quota.costLimit && reportingAction.threshold) {
            const thresholdAmount = quota.costLimit * reportingAction.threshold / 100.;
            const formattedAmount = (Math.round(thresholdAmount * 100) / 100).toFixed(2);
            return '$' + formattedAmount;
        }
        return '';
    }
    $scope.getThresholdsLabel = function(quota) {
        return quota.reportingActions.map((action) => action.threshold + "%").join(", ");
    };
    $scope.getReportingTargetsAsString = function(quota) {
        return "Will notify " + quota.reportingTargets.map((t) => t.email).join(", ");
    };

    $scope.quotaHasReportingActions = function(quota) {
        return (quota.reportingActions && quota.reportingActions.length) || quota.blockingLimit;
    };
    $scope.quotaHasRecipients = function(quota) {
        if (!quota.reportingTargets || quota.reportingTargets.length == 0) {
            return false;
        }
        return quota.reportingTargets.filter((t) => t.email && t.email.length > 0).length > 0;
    };
    $scope.validateReportingActions = function(quota) {
        if ($scope.quotaHasReportingActions(quota)) {
            return quota.reportingActions.every((action) => !action.$invalid);
        }
        return true;
    };
    $scope.reorderReportingActions = function(quota) {
        if ($scope.quotaHasReportingActions(quota)) {
            quota.reportingActions.sort((a, b) => parseFloat(a.threshold) - parseFloat(b.threshold));
        }
    };

    $scope.hasReportingActionAndNoQuotaAmount = function(quota) {
        return quota.costLimit === 0 && quota.reportingActions.length;
    }
    $scope.hasNoMailChannelAvailable = function() {
        return $scope.channels && !$scope.mailChannels.length;
    };
    $scope.hasNoMailChannelSelected = function(quota) {
        return $scope.mailChannels.length && $scope.quotaHasReportingActions(quota) && !quota.emailChannelId;
    };
    $scope.hasReportingActionsButNotRecipient = function(quota) {
        return $scope.quotaHasReportingActions(quota) && !$scope.quotaHasRecipients(quota);
    };
    $scope.getNotificationsIssues = function(quota) {
        if ($scope.hasNoMailChannelAvailable())
            return "You must define an email channel in the 'Notifications & Integrations > Messaging channels' section to send alerts.";
        if ($scope.hasNoMailChannelSelected(quota))
            return "You must define a channel for alerts to be sent.";
        if ($scope.hasReportingActionsButNotRecipient(quota))
            return "You must add recipients for alerts to be sent.";
        if ( $scope.hasReportingActionAndNoQuotaAmount(quota))
            return "You must define a quota amount > 0 for alerts to be sent.";
        return "";
    };
    $scope.globalQuotaDescription = "This quota will be applied to all queries.";
    $scope.describeFallbackScope = function() {
        return $scope.generalSettings.generativeAISettings.costLimitingSettings.quotas.length ? "This quota will be applied for queries not matching any other quota." : $scope.globalQuotaDescription;
    }

    $scope.costCounters = {};
    let updateCostCounters = function(nospinner) {
        DataikuAPI.admin.costLimiting.getCounters(nospinner).success(function(data){
            $scope.costCounters = data;
        }).error(setErrorInScope.bind($scope));
    };
    updateCostCounters(false);
    const costCounterRefreshIntervalMs = 10000; // 10 seconds
    const costCounterRefresher = $interval(() => updateCostCounters(true), costCounterRefreshIntervalMs);
    $scope.$on('$destroy', function() {
        $interval.cancel(costCounterRefresher);
    });

    $scope.getTotalCostSpent = function(quota) {
        if ($scope.costCounters[quota.id]) {
            return $scope.costCounters[quota.id].accruedCost;
        } else {
            return 0;
        }
    };
    $scope.getAllowedQueries = function(quota) {
        if ($scope.costCounters[quota.id]) {
            return $scope.costCounters[quota.id].allowedQueries;
        } else {
            return 0;
        }
    };
    $scope.getBlockedQueries = function(quota) {
        if ($scope.costCounters[quota.id]) {
            return $scope.costCounters[quota.id].blockedQueries;
        } else {
            return 0;
        }
    };
    $scope.getCostProgress = function(quota) {
        var progress = $scope.getTotalCostSpent(quota);
        if (quota.costLimit === 0) return 100;
        // avoid infinity result when dividing small values
        if (progress < 0.01) return 0;
        return 100 * (progress / quota.costLimit);
    };
    $scope.getProgressColorClass = function(quota) {
        var progress = $scope.getCostProgress(quota);
        if (quota.costLimit === 0 && !quota.blockingLimit) {
            return "gauge--disabled";
        } else if (progress < 75) {
            return "gauge--success";
        } else if (progress < 100) {
            return "gauge--warning";
        } else {
            return "gauge--error";
        }
    };
    $scope.isQuotaBlocked = function(quota) {
        var progress = $scope.getCostProgress(quota);
        return quota.blockingLimit && progress >= 100;
    };
    $scope.isCustomQuota = function(quota) {
        return quota.id !== FALLBACK_ID;
    }

    const randomAlphaNumeric = function(length) {
        let s = '';
        while (s.length < length) {
            s += Math.random().toString(36).slice(2);
        }
        return s.slice(0, length);
    };
    const generateRandomQuotaId = function () {
        return randomAlphaNumeric(8);
    }
    const fillMissingId = function(quota) {
        if (!quota.id) {
            quota.id = generateRandomQuotaId();
        }
        return quota;
    }
    $scope.dirtyCostLimitingSettings = function() {
        return $scope.savedGeneralSettings && !angular.equals($scope.generalSettings.generativeAISettings.costLimitingSettings, $scope.savedGeneralSettings.generativeAISettings.costLimitingSettings);
    };
    const pushNewQuota = function(quota) {
        $scope.generalSettings.generativeAISettings.costLimitingSettings.quotas.push(fillMissingId(quota));
    };
    const duplicateQuota = function(quota) {
        let newQuota = angular.copy(quota);
        newQuota.id = generateRandomQuotaId();
        newQuota.name = "Copy of " + newQuota.name;
        return newQuota;
    };
    const deleteQuota = function(quota) {
        // reassign the value so the change is detected and triggers an update of the table
        $scope.generalSettings.generativeAISettings.costLimitingSettings.quotas = $scope.generalSettings.generativeAISettings.costLimitingSettings.quotas.filter((q) => q.id != quota.id);
    };

    $scope.customQuotasAllowed = function() {
        return $scope.appConfig.licensedFeatures.advancedLLMMeshAllowed;
    };
    $scope.getCustomQuotasAllowedMessage = function() {
        return !$scope.customQuotasAllowed() ? 'Custom quotas require Advanced LLM Mesh, which is not enabled in your license.' : '';
    };
    const openQuotaModal = function(quota, title, action, confirmCallback) {
        CreateModalFromTemplate("/templates/admin/cost-limiting/quota-modal.html", $scope, null, function(modalScope) {
                modalScope.quota = fillMissingId(quota);
                modalScope.title = title;
                modalScope.action = action;
                modalScope.validate = function(quota) {
                    return (!$scope.isCustomQuota(quota) || quota.name) // mandatory name for only for custom quotas 
                        && (quota.costLimit || quota.costLimit === 0)
                        && quota.periodicity && quota.rollingPeriods
                        && $scope.validateReportingActions(quota);
                };
                modalScope.confirm = function(quota) {
                    confirmCallback(quota);
                    modalScope.dismiss();
                };
            });
    };
    $scope.openNewQuotaModal = function() {
        let newQuota = {
              id: generateRandomQuotaId(),
              filter: {
                  enabled: false,
                  distinct: false,
                  uiData: {
                      conditions: [],
                      mode: '&&'
                  }
              },
              periodicity: "MONTH",
              rollingPeriods: 1,
              reportingActions: [],
              reportingTargets: [],
           };
        openQuotaModal(newQuota, "Create a quota", "CREATE", pushNewQuota);
    };
    $scope.openEditQuotaModal = function(currentQuota, title) {
        openQuotaModal(angular.copy(currentQuota), title || "Update a quota", "UPDATE", (quota) => angular.copy(quota, currentQuota));
    };
    $scope.openDuplicateQuotaModal = function(quota) {
        openQuotaModal(duplicateQuota(quota), "Duplicate a quota", "Duplicate", pushNewQuota);
    };
    $scope.openDeleteSingleQuotaDialog = function(quota) {
        Dialogs.confirmInfoMessages($scope, 'Confirm quota deletion', null, '<p>Are you sure you want to delete quota: <strong>'+sanitize(quota.name)+'</strong> (id:&nbsp;<em>'+sanitize(quota.id)+'</em>)</p>', false).then(function() {
            deleteQuota(quota);
        }).catch(() => {}); // needed to avoid exception on abort
    };
    $scope.openDeleteQuotasDialog = function(quotas) {
        Dialogs.confirmInfoMessages($scope, 'Confirm quotas deletion', null, '<p>Are you sure you want to delete the following quotas:</p><ul>'
            + quotas.map((quota) => "<li><strong>"+sanitize(quota.name)+"</strong> (id:&nbsp;<em>"+sanitize(quota.id)+"</em>)</li>").join("") + "</ul>", false)
        .then(function() {
            quotas.forEach((quota) => deleteQuota(quota));
        }).catch(() => {}); // needed to avoid exception on abort
    };
    $scope.duplicateAndPushQuotas = function(quotas) {
        quotas.forEach((quota) => {
            $scope.generalSettings.generativeAISettings.costLimitingSettings.quotas.push(duplicateQuota(quota));
        });
    };
    $scope.copyToClipboard = function(value) {
        ClipboardUtils.copyToClipboard(value);
    };
});

app.controller("AdminLLMRateLimitingController", function ($scope, $state, DataikuAPI, Dialogs, CreateModalFromTemplate, localStorageService) {
    $scope.baselineSettings = {};
    $scope.allModelsByProvider = {};
    $scope.filteredModelsByProvider = {};
    $scope.applicableSettings = {};
    $scope.selection = {"orderQuery": "id"}; // apply model sort by default
    const FALLBACK_RPM = 1000; // default value in case we can't get a RPM, not expected to happen

    // Compute applicable settings by merging baseline settings and user settings
    function buildApplicableSettings() {
        let userSettings = $scope.generalSettings.generativeAISettings.rateLimitingSettings;
        let applicableSettings = angular.copy($scope.baselineSettings);

        // keep in sync with LLMRateLimitingSettingsService.buildApplicableProvidersSettings() on java side
        // Note that we do not filter out user settings with the managed flag so the user can retrieve its updated settings later
        for (let providerId in applicableSettings) {
            let userProviderSettings = userSettings.perProviderSettings[providerId] || {"perModelConfigs": {}};
            let applicableProviderSettings = applicableSettings[providerId];

            applicableProviderSettings.completionDefault = userProviderSettings.completionDefault || applicableProviderSettings.completionDefault;
            applicableProviderSettings.embeddingDefault = userProviderSettings.embeddingDefault || applicableProviderSettings.embeddingDefault
            applicableProviderSettings.imageGenerationDefault = userProviderSettings.imageGenerationDefault || applicableProviderSettings.imageGenerationDefault
            for (let modelId in userProviderSettings.perModelConfigs) {
                applicableProviderSettings.perModelConfigs[modelId] = userProviderSettings.perModelConfigs[modelId];
            }
        }
        return applicableSettings;
    }

    // Recompute generalSettings.generativeAISettings.rateLimitingSettings when the local model is updated
    function configIsDirty(defaultConfig, config) {
        return defaultConfig == null || defaultConfig.managed != config.managed || defaultConfig.requestsPerMinute != config.requestsPerMinute;
    }
    $scope.$watch("applicableSettings", function() {
        if (!$scope.generalSettings) return; // nothing to do until generalSettings are available
        let currentCustomSettings = {perProviderSettings: {}};
        // Only initialize provider config when needed to avoid dirty state when nothing was updated by the user
        const initializeProviderIfNeeded = (providerId) => {
            if (!currentCustomSettings.perProviderSettings[providerId]) {
                currentCustomSettings.perProviderSettings[providerId] = {perModelConfigs: {}};
            }
        };
        $scope.applicableSettings.providerList.forEach((provider) => {
            provider.purposes.forEach((purpose) => {
               if (configIsDirty(getDefaultConfigForPurpose(provider.id, purpose.id), purpose.config)) {
                   initializeProviderIfNeeded(provider.id);
                   currentCustomSettings.perProviderSettings[provider.id][purpose.id] = purpose.config;
               }
            });
        });
        for (let providerId in $scope.applicableSettings.modelConfigs) {
            $scope.applicableSettings.modelConfigs[providerId].forEach((model) => {
                if(configIsDirty(getDefaultConfigForModel(providerId, model.id), model.config)) {
                   initializeProviderIfNeeded(providerId);
                   currentCustomSettings.perProviderSettings[providerId].perModelConfigs[model.id] = model.config;
                }
            });
        }
        $scope.generalSettings.generativeAISettings['rateLimitingSettings'] = currentCustomSettings;
    }, true);

    function loadBaselineSettings() {
        // Load default settings from backend and convert applicableSettings as lists that are better suited for sorting and filtering the table rows
        DataikuAPI.admin.rateLimiting.getDefaults().success(function(data) {
            $scope.baselineSettings = data.baselineSettings;
            $scope.allModelsByProvider = data.models;
            let applicableSettings = buildApplicableSettings();

            // keep a reference of applicableSettings for direct access
            // .perProvider field is kept to avoid collisions with .providerList and .modelConfigs
            $scope.applicableSettings.perProvider = applicableSettings;

            // Convert as list so we can sort and filter providers
            $scope.applicableSettings.providerList = [];
            for (let providerId in applicableSettings) {
                let perProviderSettings = applicableSettings[providerId];

                let perPurposeConfigsList = [];
                ["completionDefault", "embeddingDefault", "imageGenerationDefault"].forEach((purposeId) => {
                    if (perProviderSettings[purposeId]) {
                        perPurposeConfigsList.push({id: purposeId, config: perProviderSettings[purposeId]});
                    }
                });
                $scope.applicableSettings.providerList.push({
                    id: providerId,
                    purposes: perPurposeConfigsList,
                    hasModelSortIdx: Object.keys($scope.allModelsByProvider).includes(providerId) ? 1 : 0
                });
            }
            $scope.applicableSettings.providerList = $scope.applicableSettings.providerList.sort((a, b) => {
                return b.hasModelSortIdx - a.hasModelSortIdx || a.id.localeCompare(b.id);
            });

            // Convert applicable settings as a map of lists so we can filter models
            $scope.applicableSettings.modelConfigs = {};
            for (let providerId in applicableSettings) {
                let perProviderSettings = applicableSettings[providerId];
                $scope.applicableSettings.modelConfigs[providerId] = [];
                for (let modelId in perProviderSettings.perModelConfigs) {
                    $scope.applicableSettings.modelConfigs[providerId].push({
                        id: modelId,
                        config: perProviderSettings.perModelConfigs[modelId],
                    });
                }
                filterSuggestedModelsByProvider(providerId);
            }

            initExpandStatus();
        }).error(setErrorInScope.bind($scope));
    }
    // avoid race condition
    let unwatchGeneralSettings = $scope.$watch("generalSettings", function() {
        if ($scope.generalSettings) {
            loadBaselineSettings();
            unwatchGeneralSettings();
        }
    });

    function filterSuggestedModelsByProvider(providerId) {
        if ($scope.allModelsByProvider[providerId]) {
            $scope.filteredModelsByProvider[providerId] = $scope.allModelsByProvider[providerId].filter((modelId) => !$scope.modelAlreadyExists(providerId, modelId));
        }
    }
    function getDefaultConfigForPurpose(providerId, purposeId) {
        return $scope.baselineSettings[providerId][purposeId || "completionDefault"];
    }
    function getDefaultConfigForModel(providerId, modelId) {
        return $scope.baselineSettings[providerId].perModelConfigs[modelId];
    }
    function getDefaultApplicableRPMForProvider(providerId) {
        let providerConfig = $scope.applicableSettings.perProvider[providerId]["completionDefault"] || {};
        return providerConfig.managed ? getDefaultConfigForPurpose(providerId).requestsPerMinute : providerConfig.requestsPerMinute;
    }

    $scope.getDefaultRPMForPurpose = function(providerId, purposeId) {
        let defaultConfig = getDefaultConfigForPurpose(providerId, purposeId) || {};
        return defaultConfig.requestsPerMinute || FALLBACK_RPM;
    }
    $scope.getDefaultRPM = function(providerId, modelId) {
        if (modelId) {
            let defaultModel = getDefaultConfigForModel(providerId, modelId);
            return (defaultModel && defaultModel.requestsPerMinute)
                || getDefaultApplicableRPMForProvider(providerId)
                || 0;
        } else {
            let defaultProvider = getDefaultConfigForPurpose(providerId);
            return (defaultProvider && defaultProvider.requestsPerMinute) || FALLBACK_RPM;
        }
    };
    $scope.modelAlreadyExists = function(providerId, modelId) {
        return $scope.applicableSettings.modelConfigs[providerId].some((model) => model.id == modelId);
    };
    $scope.addModelRateLimit = function(provider) {
        const providerId = provider.id;
        const newModelId = provider.newModelId;
        $scope.applicableSettings.modelConfigs[providerId].push({
            id: newModelId,
            config: {
                managed: false,
                requestsPerMinute: $scope.getDefaultRPM(providerId, newModelId)
            }
        });
        filterSuggestedModelsByProvider(providerId);
        provider.newModelId = "";
    };
    $scope.isDeleteModelAllowed = function(providerId, modelId) {
        return !getDefaultConfigForModel(providerId, modelId);
    };
    function deleteModelConfig(providerId, modelId) {
        $scope.applicableSettings.modelConfigs[providerId] = $scope.applicableSettings.modelConfigs[providerId].filter((model) => model.id != modelId);
        filterSuggestedModelsByProvider(providerId);
    };
    $scope.openDeleteSingleConfigDialog = function(providerId, modelId) {
        Dialogs.confirmInfoMessages($scope, 'Confirm rate limit deletion for '+sanitize(providerId), null, '<p>Are you sure you want to delete the rate limit settings for model: <strong>'+sanitize(modelId)+'</strong></p>', false).then(function() {
            deleteModelConfig(providerId, modelId);
        }).catch(() => {}); // needed to avoid exception on abort
    };
    $scope.getAddRateLimitDisallowedMessage = function(providerId, newModelId) {
        if (!newModelId) {
            return "Model is empty";
        }
        if ($scope.modelAlreadyExists(providerId, newModelId)) {
            return "Settings are already set for this model";
        }
        return "";
    };
    $scope.getSuggestedModelsForProvider = function(providerId) {
        return $scope.filteredModelsByProvider[providerId];
    };
    const purposeLabels = {
        "completionDefault": "Other completion models",
        "embeddingDefault": "Other embedding models",
        "imageGenerationDefault": "Other image generation models"
    };
    $scope.labelForPurpose = function(purposeId) {
        return purposeLabels[purposeId];
    };

    function matchFilter(value, filter) {
        return value.toLowerCase().includes(filter);
    }
    $scope.filterProviders = function(filter) {
        filter = filter && filter.toLowerCase();
        return function(provider) {
            return !filter || matchFilter(provider.id, filter)
            || $scope.applicableSettings.modelConfigs[provider.id].some((model) => matchFilter(model.id, filter));
        };
    };
    $scope.filterModels = function(providerId, filter) {
        filter = filter && filter.toLowerCase();
        return (model) => !filter || matchFilter(providerId, filter) || matchFilter(model.id, filter);
    };

    const localStorageProviderExpandKey = "rateLimitingSettingsProvidersExpand";
    $scope.expandStatus = localStorageService.get(localStorageProviderExpandKey) || {};
    function initExpandStatus() {
        if (!Object.keys($scope.expandStatus).length) {
            let defaultExpandStatus = {};
            Object.keys($scope.allModelsByProvider).forEach((provider) => defaultExpandStatus[provider] = true);
            $scope.expandStatus = defaultExpandStatus;
        }
    }
    $scope.showProvider = function(providerId) {
        return $scope.expandStatus[providerId];
    };
    $scope.toogleShowProvider = function(providerId) {
        $scope.expandStatus[providerId] = !$scope.expandStatus[providerId];
        localStorageService.set(localStorageProviderExpandKey, $scope.expandStatus);
    }
});

app.controller("AdminVariablesController", function ($scope, $state, $stateParams, DataikuAPI, WT1, TopNav) {
    TopNav.setLocation(TopNav.DSS_HOME, "administration");
    $scope.executeVariablesUpdate = function () {
        $scope.save().then(function () {
            DataikuAPI.admin.executeVariablesUpdate().success(function () {
                $state.transitionTo( $state.current, angular.copy($stateParams), { reload: true, inherit: true, notify: true } );
            }).error(setErrorInScope.bind($scope));
        });
    }
});


app.controller("AdminThemeController", function ($scope, $rootScope, TopNav) {
    TopNav.setLocation(TopNav.DSS_HOME, "administration");

    $scope.getThemeThumbnailUrl = function (theme) {
        var uri = $scope.getThemeUri(theme);
        return uri + theme.thumbnail;
    };

    $scope.setCurrentTheme = function (theme) {
        // saving theme content in app config
        if (theme) {
            $scope.generalSettings.themeId = theme.id;
        } else {
            delete $scope.generalSettings.themeId;
        }
        $scope.saveGeneralSettings().then(function () {
            // visualy setting theme
            $rootScope.appConfig.theme = theme;
            $scope.setTheme(theme);
        }).catch(setErrorInScope.bind($scope));
    };

    $scope.removeHomeMessage = function (homeMessage) {
        var index = $scope.generalSettings.homeMessages.indexOf(homeMessage);
        if (index != -1) {
            $scope.generalSettings.homeMessages.splice(index, 1);
        }
    };
});

app.controller("DeleteGlobalCategoryModalController", function ($scope, $rootScope, $controller) {

    function remove(index) {
        let items = $scope.generalSettings.globalTagsCategories[index].globalTags;
        items.forEach(function(item) {
            item.originalTagName = item.name;
            item.name = "";
            item.removeUsage = true;
        });
        $scope.updateGlobalTags();
        deleteKeep(index);
    };

    function deleteKeep(index) {
        $scope.generalSettings.globalTagsCategories.splice(index, 1);
        // setTimeout : when the form was invalid but is valid after deleting the category
        // give time to update or it will fail on checkFormValidity
        setTimeout(() => $scope.save());
    };

    function reassign(index, reassignTo) {
        if (reassignTo === undefined) return;
        $scope.generalSettings.globalTagsCategories[index].name = $scope.generalSettings.globalTagsCategories[reassignTo].name;
        $scope.updateGlobalTags();
        const tagsName = $scope.generalSettings.globalTagsCategories[reassignTo].globalTags.map(it => it.name);
        var mergedGlobalTags = angular.copy($scope.generalSettings.globalTagsCategories[reassignTo].globalTags);
        $scope.generalSettings.globalTagsCategories[index].globalTags.forEach(function(tag) {
            if (!tagsName.includes(tag.name)){
                mergedGlobalTags.push(tag);
            }
        });
        $scope.generalSettings.globalTagsCategories[reassignTo].globalTags = mergedGlobalTags;
        deleteKeep(index);
    }

    $scope.doDeleteGlobalCategory = function() {
        switch ($scope.deletionMode) {
            case 'remove':
                remove($scope.index);
                break;
            case 'reassign':
                reassign($scope.index, $scope.reassignTo);
                break;
            case 'keep':
                deleteKeep($scope.index);
                break;
        }
        $scope.dismiss();
    };

});

app.controller("MergeGlobalTagsModalController", function ($scope, $rootScope, $controller) {
    $scope.colors = window.dkuColorPalettes.discrete.find(palette => palette.id === "dku_font").colors;
    $scope.doMergeTags = function() {
        $scope.items.forEach(function(item) {
            item.originalTagName = item.name;
            item.name = $scope.outputTag.name;
            item.color = $scope.outputTag.color;
        });
        if (!$scope.generalSettings.globalTagsCategories[$scope.index].globalTags.find(it => (it.originalTagName || it.name) === $scope.outputTag.name)) {
            $scope.items.shift();
        }
        $scope.updateGlobalTags();
        let indexes = $scope.items.map(it => it.$idx);
        $scope.generalSettings.globalTagsCategories[$scope.index].globalTags = $scope.generalSettings.globalTagsCategories[$scope.index].globalTags.filter((it, index) => !indexes.includes(index));
        $scope.save();
        $scope.dismiss();
    }
});

app.controller("DeleteGlobalTagsModalController", function ($scope, $rootScope) {

    function remove(items) {
        items.forEach(function(item) {
            item.originalTagName = item.name;
            item.name = "";
            item.removeUsage = true;
        });
        $scope.updateGlobalTags();
        deleteKeep(items);
    };

    function deleteKeep(items) {
        let indexes = items.map(it => it.$idx);
        $scope.generalSettings.globalTagsCategories[$scope.index].globalTags = $scope.generalSettings.globalTagsCategories[$scope.index].globalTags.filter((it, index) => !indexes.includes(index));
        // setTimeout : when the form was invalid but is valid after deleting the tag(s)
        // give time to update or it will fail on checkFormValidity
        setTimeout(() => $scope.save());
    };

    function reassign(items, reassignTo) {
        if (reassignTo === undefined) return;
        items.forEach(function(item) {
            item.originalTagName = item.name;
            item.name = reassignTo;
        });
        $scope.updateGlobalTags();
        deleteKeep(items);
    }

    $scope.doDeleteGlobalTags = function() {
        switch ($scope.deletionMode) {
            case 'remove':
                remove($scope.items);
                break;
            case 'reassign':
                reassign($scope.items, $scope.reassignTo);
                break;
            case 'keep':
                deleteKeep($scope.items);
                break;
        }
        $scope.dismiss();
    };
});

app.controller("GlobalTagsController", function ($scope, $element, $timeout, TaggingService, CreateModalFromTemplate, TAGGABLE_TYPES, DataikuAPI) {
    $scope.colors = window.dkuColorPalettes.discrete.find(palette => palette.id === "dku_font").colors;
    $scope.taggableTypes = [ // Must be kept in sync with ITaggingService.TaggableType
        'PROJECT',
        'ANALYSIS',
        'SAVED_MODEL',
        'DATASET',
        'RECIPE',
        'LABELING_TASK',
        'MANAGED_FOLDER',
        'STREAMING_ENDPOINT',
        'FLOW_ZONE',
        'SQL_NOTEBOOK',
        'SEARCH_NOTEBOOK',
        'JUPYTER_NOTEBOOK',
        'STATISTICS_WORKSHEET',
        'SCENARIO',
        'DASHBOARD',
        'INSIGHT',
        'ARTICLE',
        'WEB_APP',
        'CODE_STUDIO',
        'REPORT',
        'LAMBDA_SERVICE',
        'RETRIEVABLE_KNOWLEDGE',
        'PROMPT_STUDIO',
        'AGENT_TOOL',
        'DATA_COLLECTION'
    ];
    
    $scope.computeItemTemplate = function() {
        return {name: '', color: TaggingService.getDefaultColor(Math.random().toString(36).slice(2)), isNew: true};
    };

    $scope.addCategory = function() {
        $scope.generalSettings.globalTagsCategories.push({name: '', globalTags: [], appliesTo: $scope.taggableTypes, isNew: true});
        //focus new category name
        $timeout(function() {
            const focusable = $element.find('input[ng-model="category.name"]').last();
            if (focusable) {
                focusable.focus()
            }
        });
    };

    $scope.calcGlobalTagUsage = function() {
        DataikuAPI.catalog.search("", {scope:['dss'], _type:['ALL']}, true).then((data) => {
            $scope.tagUsageMap = data.data.aggregations["tag.raw"].agg.buckets.reduce(function(map, obj) {
                map[obj.key] = obj.doc_count;
                return map;
            }, {});
        });
    };
    $scope.calcGlobalTagUsage();

    $scope.$on('generalSettingsSaved', $scope.calcGlobalTagUsage);

    $scope.getGlobalTagUsage = function(category, item) {
        if (!$scope.tagUsageMap) return 0;
        let tagName = item.originalTagName || item.name;
        return $scope.tagUsageMap[`${category}:${tagName}`];
    };

    $scope.linkToCatalog = function(category, item) {
        const tagName = item.originalTagName || item.name;
        const globalTagName = (`${category}:${tagName}`).replace(/\\/g,'%5C').replace(/\//g,'~2F').replace(/&/g, '%5C&');
        return `/search-dss-items/scope=all&tag.raw=${globalTagName}`;
    };

    $scope.updateItem = function(category, originalTagName, item) {
        if (!item.isEdited) {
            item.isEdited = true;
            item.originalTagName = originalTagName;
        }
    }

    $scope.mergeTags = function(index, items) {
        CreateModalFromTemplate("/templates/global-tags/merge-global-tags-modal.html", $scope, "MergeGlobalTagsModalController", function(modalScope) {
            modalScope.index = index;
            modalScope.items = items;

            modalScope.outputTag = angular.copy(items[0]);
        });
    };

    $scope.deleteGlobalCategory = function(index, categoryName) {
        if ($scope.generalSettings.globalTagsCategories[index].isNew) {
            $scope.generalSettings.globalTagsCategories.splice(index, 1);
            return;
        }
        CreateModalFromTemplate("/templates/global-tags/delete-global-category-modal.html", $scope, "DeleteGlobalCategoryModalController", function(modalScope) {
            modalScope.deletionMode = "remove";
            modalScope.index = index;
            modalScope.categoryName = categoryName;
            modalScope.assignableCategoryList = $scope.generalSettings.globalTagsCategories.filter(cat => cat.name != categoryName).map((cat, idx) => idx >= index ? idx + 1 : idx);
        });
    };

    $scope.deleteGlobalTags = function(index, items) {
        CreateModalFromTemplate("/templates/global-tags/delete-global-tags-modal.html", $scope, "DeleteGlobalTagsModalController", function(modalScope) {
            modalScope.modalTitle = `Delete global tag${items.length > 1 ? 's' : ''}`;
            modalScope.deletionMode = "remove";
            modalScope.index = index;
            modalScope.items = items;
            modalScope.assignableTagsList = $scope.generalSettings.globalTagsCategories[index].globalTags.filter(it => !items.includes(it)).map(it => it.name);
        });
    };
});

app.controller("ProjectStatusController", function ($scope, $rootScope, TopNav, DKUConstants, TaggingService) {
    $scope.defaultTagColor = TaggingService.getDefaultColor;
    $scope.colors = window.dkuColorPalettes.discrete.find(palette => palette.id === "dku_font").colors;
    $scope.newStatus = {
        name: '',
        defaultColor: TaggingService.getDefaultColor('')
    };

    //Isolating the archived status so that it does not get deleted nor reorder
    $scope.projectStatusList = [];
    $scope.promises.then(function (values) {
        $scope.projectStatusList = angular.copy($scope.generalSettings.projectStatusList);
        var archivedStatusIndex =  $scope.projectStatusList.findIndex(function (status) {
            return status.name ==  DKUConstants.ARCHIVED_PROJECT_STATUS;
        });
        var archivedStatus = archivedStatusIndex > -1 ? $scope.projectStatusList.splice(archivedStatusIndex, 1) : [];

        //On local projectStatusList change recomputing generalSettings.projectStatusList with the archived status at its end
        $scope.$watch('projectStatusList', function () {
            $scope.generalSettings.projectStatusList = $scope.projectStatusList.concat(archivedStatus);
        }, true);
    });
});

app.controller("BrandColorsController", function ($scope, ColorUtils) {
    $scope.newBrandColor = {
        defaultColor: '#FFFFFF'
    };

    $scope.brandColors = [];
    $scope.brandColorsPalette = ColorUtils.getDiscretePaletteColors('dku_font');
    $scope.promises.then(function () {
      // The scope of the controller is not accessible from the content projected in EditableList.
      // To be able to access the palette, we need to reference it in each item of the palette.
      $scope.brandColors = $scope.generalSettings.brandColors.map(brandColor => ({
          ...brandColor,
          palette: $scope.brandColorsPalette
      }));

      $scope.$watch('brandColors', function () {
          // We don't want to keep the palette in each color entry when saving, so we manually remove them and sync the controller and global models.
          $scope.generalSettings.brandColors = $scope.brandColors.map(brandColor => {
              const cleanedBrandColor = { ...brandColor };
              delete cleanedBrandColor.palette;
              return cleanedBrandColor;
          });
      }, true);
    });
});


app.controller("AdminLicensingController", function ($scope, $state, CreateModalFromTemplate, TopNav, DataikuAPI) {
    TopNav.setLocation(TopNav.DSS_HOME, "administration");
    $scope.$state = $state;

    $scope.enterLicense = function () {
        CreateModalFromTemplate("/templates/admin/enter-license.html", $scope, "EnterLicenseController");
    };

    $scope.fetchTrialUsersStatus = function() {
        DataikuAPI.admin.getTrialStatus().success(function (data) {
            $scope.trialStatus = data;
        }).error(setErrorInScope.bind($scope));
    }

    $scope.getLimits = function () {
        if ($scope.isDSSAdmin()) {
            DataikuAPI.admin.getLimitsStatus().success(function (data) {
                $scope.limits = data;
            }).error(setErrorInScope.bind($scope));
        }
    };

    $scope.getLimits();
});


app.controller("EnterLicenseController", function ($scope, $state, $rootScope, DataikuAPI, Assert, TopNav, WT1) {
    TopNav.setLocation(TopNav.DSS_HOME, "administration");
    Assert.inScope($scope, "appConfig");

    $scope.existingKey = {}

    $scope.reloadMe = function () {
        $scope.dismiss();
        location.reload();
    };

    $scope.setLicense = function () {
        WT1.event("license-update");
        DataikuAPI.registration.setOfflineLicense($scope.existingKey.license).success(function (data) {
            $scope.registrationSuccessful = {};
        }).error(setErrorInScope.bind($scope));
    };
});


//Integration channels to report info outside of dss:


app.controller("AwsChannelController", function ($scope, $state, $stateParams, DataikuAPI) {

});

app.controller("MicrosoftGraphChannelController", function ($scope, $state, $stateParams, DataikuAPI) {

});

app.controller("SMTPChannelController", function ($scope, $state, $stateParams, DataikuAPI) {

});


app.controller("SlackChannelController", function ($scope, $state, $stateParams, DataikuAPI) {

});


app.controller("TeamsChannelController", function ($scope, $state, $stateParams, DataikuAPI) {
    if ($scope.channel.configuration.useProxy === undefined) {
        $scope.channel.configuration.useProxy = false;
    }
});

app.controller("GoogleChatChannelController", function ($scope, $state, $stateParams, DataikuAPI) {
});

app.controller("WebhookChannelController", function ($scope, $state, $stateParams, DataikuAPI) {

});


app.controller("ShellChannelController", function ($scope, $state, $stateParams, DataikuAPI, TopNav, CodeMirrorSettingService) {
    TopNav.setLocation(TopNav.DSS_HOME, "administration");
    $scope.channel.configuration.type = $scope.channel.configuration.type || 'COMMAND';
    $scope.editorOptions = CodeMirrorSettingService.get('text/x-sh', {onLoad: function(cm) {$scope.cm = cm;}});

    $scope.shellSenderTypes = [{type:'FILE',name:'Script file'}, {type:'COMMAND',name:'Command'}];

    $scope.getCommandLine = function () {
        if ( $scope.channel.configuration.type == 'FILE' ) {
            if ( $scope.channel.configuration.command ) {
                return "sh -c " + $scope.channel.configuration.command + " script_file";
            } else {
                return "sh script_file";
            }
        } else {
            return $scope.channel.configuration.command;
        }
    };
});


app.controller("DatasetChannelController", function ($scope, $state, $stateParams, DataikuAPI, TopNav) {
    TopNav.setLocation(TopNav.DSS_HOME, "administration");

    DataikuAPI.projects.list().success(function (data) {
        $scope.projectsList = data;
    }).error(setErrorInScope.bind($scope));

    $scope.$watch('channel.configuration.projectKey', function () {
        if ($scope.channel.configuration.projectKey == null ) return;
        DataikuAPI.datasets.list($scope.channel.configuration.projectKey).success(function (data) {
            $scope.datasetsList = data;
        }).error(setErrorInScope.bind($scope));
    });
});

app.controller("ContainerSettingsController", function($scope, DataikuAPI, Dialogs, Assert, TopNav, FutureProgressModal, CodeMirrorSettingService) {
    TopNav.setLocation(TopNav.DSS_HOME, "administration");
    Assert.inScope($scope, "appConfig");
    Assert.inScope($scope, "addLicInfo");
    Assert.inScope($scope, "generalSettings");

    $scope.codeMirrorSettingService = CodeMirrorSettingService;
    $scope.settings = $scope.generalSettings.containerSettings;

    const getConfigurationsByWorkloadType = (configs, workloadType) => {
        if (configs === null){
            return [];
        }
        return [{value: "", label: "Nothing selected"}, ...configs.filter(ec => [workloadType, "ANY"].includes(ec.workloadType) && !!ec.name).map(ec => ({value: ec.name, label: ec.name}))];
    }

    const setConfigurationsByWorkloadType = () => {
        $scope.userCodeContainerConfigs = getConfigurationsByWorkloadType($scope.settings.executionConfigs, "USER_CODE");
        $scope.visualRecipesContainerConfigs = getConfigurationsByWorkloadType($scope.settings.executionConfigs, "VISUAL_RECIPES");
    }

    setConfigurationsByWorkloadType();
    $scope.$watch("settings.executionConfigs", setConfigurationsByWorkloadType, true)

    $scope.getNewContainerConfig = function() {
        return {
            type: 'KUBERNETES',
            usableBy: 'ALL', allowedGroups: [], workloadType: 'ANY',
            dockerNetwork: 'host',
            dockerResources: [],
            kubernetesResources: {
                memRequestMB: -1, memLimitMB: -1,
                cpuRequest: -1, cpuLimit: -1,
                customRequests: [], customLimits: [],
                hostPathVolumes: []
            },
            properties: []
        };
    };

    DataikuAPI.security.listGroups(false)
        .success(data => {
            if (data) {
                data.sort();
            }
            $scope.allGroups = data;
        })
        .error(setErrorInScope.bind($scope));

    $scope.isBaseImageNameSuspicious = function(baseImage) {
        return /^(?:[\w-_]+\.)+\w+(?::\d+)?\//.test(baseImage);
    };

    $scope.installJupyterSupport = function(){
        DataikuAPI.admin.containerExec.installJupyterSupport().success(function(data) {
            FutureProgressModal.show($scope, data, "(Re)Installing kernels").then(function(result){
                if (result) { // undefined in case of abort
                    Dialogs.infoMessagesDisplayOnly($scope, "Update result", result.messages, result.futureLog);
                }
            })
        }).error(setErrorInScope.bind($scope));
    }

    $scope.removeJupyterSupport = function(){
        DataikuAPI.admin.containerExec.removeJupyterSupport().success(function(data) {
            FutureProgressModal.show($scope, data, "Removing kernels").then(function(result){
                if (result) { // undefined in case of abort
                    Dialogs.infoMessagesDisplayOnly($scope, "Update result", result.messages, result.futureLog);
                }
            })
        }).error(setErrorInScope.bind($scope));
    }

    $scope.applyK8SPolicies = function(){
        if($scope.dirtySettings()) {
            $scope.save().then($scope.applyK8SPolicies_NoCheckDirty);
        } else {
            $scope.applyK8SPolicies_NoCheckDirty();
        }
    }
    $scope.applyK8SPolicies_NoCheckDirty = function() {
        DataikuAPI.admin.containerExec.applyK8SPolicies().success(function(data) {
            FutureProgressModal.show($scope, data, "Applying Kubernetes policies").then(function(result){
                if (result) {
                    Dialogs.infoMessagesDisplayOnly($scope, "Policies applied", result, result.futureLog, true);
                }
            });
        }).error(setErrorInScope.bind($scope));
    }
});

app.controller("BuildBaseImageModalController", function ($scope) {
    $scope.doBuild = function() {
        $scope.dismiss();
        $scope.buildBaseImage($scope.buildType, $scope.buildOptions);
    };
});


app.controller('AdminHelpSettingsController', function($scope, AccessibleObjectsCacheService) {
    // Cache for object-picker objects
    $scope.getAccessibleObjects = AccessibleObjectsCacheService.createCachedGetter('READ', $scope.setErrorInGeneralSettingsControllerScope);

    $scope.adminGeneralIndexForm.homeArticlesListInvalid = false;
    $scope.articlesAreValid = () => {
        if ($scope.generalSettings && $scope.generalSettings.personalHomePages.articles) {
            for (let elt of $scope.generalSettings.personalHomePages.articles) {
                if (!Object.keys(elt).includes("projectKey") || !Object.keys(elt).includes("id")) {
                    $scope.adminGeneralIndexForm.homeArticlesListInvalid = true;
                    return false;
                }
            }
        }
        $scope.adminGeneralIndexForm.homeArticlesListInvalid = false;
        return true;
    }
});

app.controller('AdminHomepageSettingsController', function($scope, AccessibleObjectsCacheService, Ng1PromotedContentAdminService, $q) {
    // Cache for object-picker objects
    $scope.getAccessibleObjects = AccessibleObjectsCacheService.createCachedGetter('READ', $scope.setErrorInGeneralSettingsControllerScope);

    $scope.adminGeneralIndexForm.homepagePromotedContentInvalid = false;
    $scope.promotedContentIsValid = () => {
        if ($scope.generalSettings && $scope.generalSettings.personalHomePages && $scope.generalSettings.personalHomePages.promotedContent) {
            for (let elt of $scope.generalSettings.personalHomePages.promotedContent) {
                if (!$scope.isItemValid(elt)) {
                    $scope.adminGeneralIndexForm.homepagePromotedContentInvalid = true;
                    return false;
                }
            }
        }
        $scope.adminGeneralIndexForm.homepagePromotedContentInvalid = false;
        return true;
    }

    const TYPES_WITHOUT_PROJECT_KEY = ['WORKSPACE', 'DATA_COLLECTION', 'PROJECT', 'APP', 'ADMIN_MESSAGE', 'LINK'];
    const TYPES_WITHOUT_OBJECT = ['ADMIN_MESSAGE', 'LINK'];

    $scope.isItemValid = function(item) {
        if (!item) {
            return false;
        }
        if (!Object.keys(item).includes("type")) {
            return false;
        }
        if (!TYPES_WITHOUT_OBJECT.includes(item.type) && !Object.keys(item).includes("id")) {
            return false;
        }
        if (!TYPES_WITHOUT_PROJECT_KEY.includes(item.type) && !Object.keys(item).includes("projectKey")) {
            return false;
        }
        if (!$scope.getItemDisplayName(item)) {
            // This ensures that the item exists and has not been deleted
            return false;
        }
        return true;
    }

    $scope.invalidItemReason = function(item) {
        if (!item) {
            return "Content is not defined";
        }
        if (!Object.keys(item).includes("type")) {
            return "Type is missing";
        }
        if (!TYPES_WITHOUT_OBJECT.includes(item.type) && !Object.keys(item).includes("id")) {
            return "Id is missing";
        }
        if (!TYPES_WITHOUT_PROJECT_KEY.includes(item.type) && !Object.keys(item).includes("projectKey")) {
            return "Project key is missing";
        }
        if (!$scope.getItemDisplayName(item)) {
            // This ensures that the item exists and has not been deleted
            return "Item does not exist, or has an empty display name";
        }
        return "";
    }

    $scope.editPromotedContent = function(itemToEdit) {
        return Ng1PromotedContentAdminService.openPromotedContentAdminModal({
            promotedContent: itemToEdit,
            getAccessibleObjectsCached: $scope.getAccessibleObjects,
            mode: 'EDIT'
        }).then((output) => output && output.promotedContent);
    }

    $scope.addPromotedContent = function() {
        $q.when(Ng1PromotedContentAdminService.openPromotedContentAdminModal({
            getAccessibleObjectsCached: $scope.getAccessibleObjects,
            mode: 'ADD'
        })).then((output) => {
            if (output && output.promotedContent) {
                $scope.generalSettings.personalHomePages.promotedContent = [
                    ...$scope.generalSettings.personalHomePages.promotedContent,
                    output.promotedContent
                ];
            }
        });
    }

    $scope.getProjectName = function(item) {
        if (!item) {
            return "";
        }
        if (TYPES_WITHOUT_PROJECT_KEY.includes(item.type)) {
            return "";
        }
        const accessibleProject = $scope.getAccessibleObjects('PROJECT').find((project) => project.id === item.projectKey);
        return accessibleProject && accessibleProject.object && accessibleProject.object.name;
    }

    $scope.getItemDisplayName = function(item) {
        if (!item) {
            return "";
        }
        let accessibleItem;
        if (TYPES_WITHOUT_OBJECT.includes(item.type)) {
            return item.customTitle
        }
        if (TYPES_WITHOUT_PROJECT_KEY.includes(item.type)) {
            accessibleItem = $scope.getAccessibleObjects(item.type).find((it) => it.id === item.id);
        } else {
            accessibleItem = $scope.getAccessibleObjects(item.type, item.projectKey).find((it) => it.id === item.id);
        }
        if (accessibleItem) {
            if (item.type === 'PROJECT') {
                // Projects label contain both the project name and the project key, which we do not want here
                return accessibleItem.object && accessibleItem.object.name;
            } else {
                return accessibleItem.label;
            }
        }
        return "";
    }
});

app.controller('AdminOtherSecurityController', function($scope, DataikuAPI) {
    $scope.updateExistingApiKeysLifetime = function() {
        if($scope.dirtySettings()) {
            $scope.save().then($scope.updateExistingApiKeysLifetime_NoCheckDirty);
        } else {
            $scope.updateExistingApiKeysLifetime_NoCheckDirty();
        }
    }

    $scope.updateExistingApiKeysLifetime_NoCheckDirty = function() {
        DataikuAPI.admin.apiKeys.updateApiKeysLifetime().error(setErrorInScope.bind($scope));
    }
});

    app.controller("AdminAiServicesSettingsController", function($scope, DataikuAPI) {
        DataikuAPI.webapps.listAllBackendsStates().success(function(data) {
            $scope.backends = data
              .filter(backend => backend.backendState && backend.backendState.hasExposedEndpoint)
              .map(backend => ({
                  ...backend,
                  webAppReference: backend.projectKey + "." + backend.webAppId
              }));
        }).error(setErrorInScope.bind($scope));

        $scope.webAppSearch = function(term, webapp) {
            return webapp.webAppReference.toLowerCase().includes(term.toLowerCase()) || webapp.webAppName.includes(term.toLowerCase());
        };

        $scope.defaultAiServerValue = function(localAiIndividualSetting) {
            if (!$scope.generalSettings.aiDrivenAnalyticsSettings.dataikuAIServicesTermsOfUseAccepted) {
                $scope.generalSettings.localAIServerSettings[localAiIndividualSetting + "UseLocal"] = true;
            }
        };

        $scope.updateTelemetryEnabled = function(aiAssistant) {
            // ngChange triggers when the change start so before the value of localAIServerSettings[aiAssistant + "UseLocal"] gets updated. So if it was false we know it will change to true.
            if(!$scope.generalSettings.localAIServerSettings[aiAssistant + "UseLocal"]) {
                $scope.generalSettings.aiDrivenAnalyticsSettings[aiAssistant + "TelemetryEnabled"] = false;
            }
        };

        $scope.$watch('generalSettings.localAIServerSettings.storiesAIUseLocal', function (newVal) {
            if (newVal === true) {
                $scope.generalSettings.storiesVocalEnabled = false;
            }
        });

        $scope.$watchGroup([
        'generalSettings.aiDrivenAnalyticsSettings.storiesAIEnabled',
            function () {
                return $scope.featureFlagEnabled && $scope.featureFlagEnabled('storiesVocal');
            }
        ], function ([aiEnabled, flagEnabled]) {
            if (!(aiEnabled && flagEnabled)) {
                $scope.generalSettings.storiesVocalEnabled = false;
            }
        });
    });

    app.component("integrationChannelPermissions", { // integration-channel-permissions
            bindings: {
                permissions: "<", // IntegrationChannel.PermissionItem[], edited in place
                allUsers: "<",    // readonly UIUser[]
                allGroups: "<",   // readonly string[]
            },
            templateUrl: "templates/admin/general/integration-channel-permissions.html",
            controller: function integrationChannelPermissionsController($scope) {
                const $ctrl = this;

                function updateUnassignedGroups() {
                    $ctrl.unassignedGroupIds = ['$$ALL_USERS$$', ...$ctrl.allGroups].filter(group => !$ctrl.permissions.some(perm => perm.group === group));
                }
                function updateUnassignedUsers() {
                    $ctrl.unassignedUsers = $ctrl.allUsers.filter(user => !$ctrl.permissions.some(perm => perm.user === user.login));
                }

                $ctrl.$onChanges = (changes) => {
                    if(changes.allUsers) {
                        $ctrl.userLoginToDisplayName = new Map($ctrl.allUsers.map(user => [user.login, user.displayName]))
                    }
                    if(changes.allGroups) {
                        $ctrl.groupIdToDisplayName = new Map($ctrl.allGroups.map(group => [group, group]))
                        $ctrl.groupIdToDisplayName.set('$$ALL_USERS$$', 'All Users')
                    }
                    if(changes.permissions || changes.allUsers) {
                        updateUnassignedUsers();
                    }
                    if(changes.permissions || changes.allGroups) {
                        updateUnassignedGroups();
                    }
                }

                $ctrl.onUsersAdded = (users) => {
                    users.forEach(login => {
                        $ctrl.permissions.push({ user: login, canUse: true });
                    });
                    updateUnassignedUsers();
                    $scope.$applyAsync(); // trigger angularjs change detection
                }

                $ctrl.onGroupsAdded = (groups) => {
                    groups.forEach(group => {
                        $ctrl.permissions.push({ group, canUse: true });
                    });
                    updateUnassignedGroups();
                    $scope.$applyAsync(); // trigger angularjs change detection
                }

                $ctrl.removePermission = (index) => {
                    const removed = $ctrl.permissions.splice(index, 1)[0];
                    if(removed.user) {
                        updateUnassignedUsers();
                    } else {
                        updateUnassignedGroups();
                    }
                }

            }
        });
})();

;
(function () {
    'use strict';
    var app = angular.module('dataiku.credentials', []);

    app.factory("CredentialDialogs", function(CreateModalFromTemplate, $q, $state, $timeout, $window, DataikuAPI) {

        function isConnectionCredential(credential) {
            return credential && ['CONNECTION', 'VIRTUAL_CONNECTION', 'DATABRICKS_INTEGRATION'].includes(credential.requestSource);
        }

        function enterCredential($scope, credential) {
            let deferred = $q.defer();

            resetErrorInScope($scope);

            if (credential.type === "SINGLE_FIELD" || credential.type === "BASIC") {
                CreateModalFromTemplate("/templates/profile/edit-connection-credential-modal.html", $scope, null, function (newScope) {
                    newScope.credential = credential;
                    newScope.plugin = credential.pluginCredentialRequestInfo;
                    newScope.credential.password = "";
                    newScope.passwordFieldTitle = credential.type === 'BASIC' ? "Password" : "Credentials";
                    newScope.isConnectionCredential = isConnectionCredential(credential);

                    newScope.confirm = function() {
                        let apiCall;
                        if (isConnectionCredential(credential)) {
                            apiCall = DataikuAPI.profile.setBasicConnectionCredential(newScope.credential.connection,
                                          newScope.credential.user, newScope.credential.password);
                        } else {
                            apiCall = DataikuAPI.profile.pluginCredentials.setBasicCredential(newScope.plugin.pluginId,
                                          newScope.plugin.paramSetId, newScope.plugin.presetId, newScope.plugin.paramName,
                                          newScope.credential.user, newScope.credential.password);
                        }

                        return apiCall.success(function () {
                                newScope.dismiss();
                                deferred.resolve(false);
                            }).error(deferred.reject);
                    }
                });
            } else if (credential.type === "AZURE_OAUTH_DEVICECODE") {
                CreateModalFromTemplate("/templates/profile/edit-azure-oauth-connection-credential-modal.html", $scope, null, function (newScope) {
                    newScope.uiState = {
                        step: "STARTUP"
                    }
                    newScope.credential = credential;

                    newScope.startStep1 = function() {
                        const state = {
                            name: $state.current.name,
                            params: $state.params
                         }
                        DataikuAPI.profile.connectionCredentials
                            .azureOAuthDeviceCodeDanceStep1(
                                urlWithProtocolAndHost(),
                                JSON.stringify(state),
                                newScope.credential)
                            .success(function (data) {
                                newScope.uiState.step = "STEP1_COMPLETE";
                                newScope.uiState.deviceCode = data;
                            }).error(setErrorInScope.bind($scope));
                    }
                    newScope.startStep2 = function() {
                        const state = {
                            name: $state.current.name,
                            params: $state.params
                         }
                        DataikuAPI.profile.connectionCredentials
                            .azureOAuthDeviceCodeDanceStep2(
                                urlWithProtocolAndHost(),
                                JSON.stringify(state),
                                newScope.credential)
                            .success(function() {
                                newScope.dismiss();
                                deferred.resolve(false);
                            }).error(setErrorInScope.bind($scope));
                    }

                    newScope.startStep1();
                });
            } else if (credential.type === "OAUTH_REFRESH_TOKEN") {
                CreateModalFromTemplate("/templates/profile/edit-oauth-connection-credential-modal.html", $scope, null, function (newScope) {
                    newScope.credential = credential;
                    newScope.plugin = credential.pluginCredentialRequestInfo;
                    newScope.isConnectionCredential = isConnectionCredential(credential);

                    newScope.confirm = function() {
                        const state = {
                           name: $state.current.name,
                           params: $state.params
                        }
                        return DataikuAPI.profile.startOAuth2AuthorizationFlow(
                            urlWithProtocolAndHost(),
                            JSON.stringify(state),
                            newScope.credential)
                        .success(function (data) {
                            // Redirect to begin authorization process
                            $window.location.href = data;
                            deferred.resolve(true);
                        }).error(setErrorInScope.bind($scope));
                    }
                });
            }
            return deferred.promise;
        }

        function oauth2Success(scope, credential, plugin) {
            return CreateModalFromTemplate("/templates/profile/oauth2-callback-success-modal.html", scope, null, function (newScope) {
                newScope.credential = credential;
                newScope.plugin = plugin;
                newScope.isConnectionCredential = isConnectionCredential(newScope.credential);
            });
        }

        function oauth2Error(scope, data, credential, plugin, error) {
            return CreateModalFromTemplate("/templates/profile/oauth2-callback-error-modal.html", scope, null, function (newScope) {
                newScope.credential = credential;
                newScope.plugin = plugin;
                newScope.error = error;
                newScope.isConnectionCredential = isConnectionCredential(newScope.credential);
                newScope.tryAgain = enterCredential.bind(this, newScope, newScope.credential);
            });
        }

        return {
            enterCredential: enterCredential,
            oauth2Success: oauth2Success,
            oauth2Error: oauth2Error
        };
    });

    app.controller('OAuth2CallbackController', function($scope, $rootScope, $timeout, $location, $state, DataikuAPI, CredentialDialogs) {
        const code = $location.search().code;
        const state = $location.search().state;
        const error = $location.search().error;
        const errorDescription = $location.search().error_description;
        const errorUri = $location.search().error_uri;

        $location.search("code", null);
        $location.search("state", null);
        $location.search("error", null);
        $location.search("error_description", null);
        $location.search("error_uri", null);

        if (code || error || state) {
            DataikuAPI.profile.exchangeAuthorizationCode(code, state, error, errorDescription, errorUri)
                .success(function(data) {
                    $timeout(() => data.isSuccess
                        ? CredentialDialogs.oauth2Success($rootScope, data.authorizationRequestInfo.credential, data.authorizationRequestInfo.credential.pluginCredentialRequestInfo)
                        : CredentialDialogs.oauth2Error($rootScope, data, data.authorizationRequestInfo.credential, data.authorizationRequestInfo.credential.pluginCredentialRequestInfo, data)
                    , 100);
                    const userCurrentState = JSON.parse(data.authorizationRequestInfo.userCurrentState);
                    $state.go(userCurrentState.name, userCurrentState.params);
                })
                .error(setErrorInScope.bind($scope));
        }
    });

}());
;
(function () {
    'use strict';

    var app = angular.module('dataiku.connections', ['dataiku.credentials']);

    /* Base controller used both for ConnectionsList and hive indexing (as a lot of layout and actions are common) */
    app.controller("_ConnectionsListBaseController", function ($scope, $rootScope, TopNav, DataikuAPI, Dialogs, $timeout, $filter, $state, ConnectionUtils, CreateModalFromTemplate) {
        $scope.noTags = true;
        $scope.noStar = true;
        $scope.noWatch = true;
        $scope.canIndexConnections = true;
        $scope.canCreateConnection = $state.is('admin.connections.list');
        $scope.noDelete = !$state.is('admin.connections.list');
        $scope.selection = $.extend({
            filterQuery: {
                userQuery: '',
                type: []
            },
            filterParams: {
                userQueryTargets: ["name", "type"],
                exactMatch: ["type"]
            },
            orderQuery: "creationTag.lastModifiedOn",
            orderReversed: false,
        }, $scope.selection || {});
        $scope.sortBy = [
            {value: 'name', label: 'Name'},
            {value: 'type', label: 'Type'},
            {value: 'creationTag.lastModifiedOn', label: 'Creation date'}
        ];
        $scope.uiState = $scope.uiState || {};
        $scope.uiState.newConnectionQuery = '';
        $scope.connectionGroups = buildConnectionList();

        TopNav.setLocation(TopNav.DSS_HOME, "administration");
        let pollPromise;

        function processRunningJobsRequest(data) {
            let doRunningJobsRemain = false;

            $scope.listItems.forEach(conn => {
                if (conn.name in data) {
                    conn.indexingMetadata = data[conn.name];
                    if (conn.indexingMetadata.currentJobId) {
                        doRunningJobsRemain = true;
                    }
                } else if (conn.indexingMetadata && conn.metadata.currentJobId) {
                    delete conn.indexingMetadata.currentJobId;
                }
            });
            return doRunningJobsRemain;
        }

        $scope.pollRunningJobs = function () {
            $timeout.cancel(pollPromise);
            DataikuAPI.admin.connections.listRunningJobs().success(function (data) {
                let doRunningJobsRemain = processRunningJobsRequest(data);
                if (doRunningJobsRemain) {
                    pollPromise = $timeout(() => {
                        $scope.pollRunningJobs();
                    }, 5000);
                }
            });
        }

        /** Order connections by their nice connection type rather than the raw type */
        $scope.niceConnectionTypeComparator = (a,b) => {
            return $filter("connectionTypeToNameForList")(a.value).localeCompare($filter("connectionTypeToNameForList")(b.value));
        }

        /** Adds or removes the specified connection type from the filter depending on whether it's already there */
        $scope.setConnectionTypeFilter = (connectionType) => {
            const index = $scope.selection.filterQuery.type.indexOf(connectionType);
            if (index > -1) {
                $scope.selection.filterQuery.type.splice(index, 1);;
            } else {
                $scope.selection.filterQuery.type.push(connectionType);
            }
        }

        $scope.isIndexable = connection => ConnectionUtils.isIndexable(connection);

        $scope.list = function () {
            var func = $state.is('admin.connections.hiveindexing') ? DataikuAPI.admin.connections.listHiveVirtual : DataikuAPI.admin.connections.list;
            func().success(function (data) {
                $scope.connections = data;
                $scope.listItems = $.map(data, function (v, k) {
                    return v;
                });
                $scope.connectionTypes = [...new Set($scope.listItems.map(it => it.type))];
                $scope.canIndexAllConnections = $scope.listItems.length > 0;
                if ($scope.listItems.find(c => c.indexingMetadata && c.indexingMetadata.currentJobId)) {
                    $scope.pollRunningJobs();
                }
            }).error(setErrorInScope.bind($scope));
        };
        $scope.$on("$destroy", function () {
            pollPromise && $timeout.cancel(pollPromise);
        });

        $scope.$on("indexConnectionEvent", (event, connectionName) => {
            createIndexConnectionsModal([connectionName]);
        });

        $scope.list();

        $scope.deleteSelected = function (name) {
            var selectedConnectionsNames = $scope.selection.selectedObjects.map(function (c) {
                return c.name;
            });
            Dialogs.confirm($scope, 'Connection deletion', 'Are you sure you want to delete this connection ?').then(function () {
                DataikuAPI.admin.connections.delete(selectedConnectionsNames).success(function (data) {
                    $scope.list();
                }).error(setErrorInScope.bind($scope));
            });
        };

        let createIndexConnectionsModal = function (connectionNames) {
            const newScope = $scope.$new();
            newScope.selectedConnections = connectionNames;
            CreateModalFromTemplate("/templates/admin/index-connections-modal.html", newScope, 'IndexConnectionsModalController');
        };

        $scope.indexSelectedConnections = function () {
            createIndexConnectionsModal($scope.selection.selectedObjects.map(function (c) {
                return c.name;
            }));
        };

        $scope.isIndexationRunning = function () {
            return $scope.listItems && $scope.listItems.find(c => {
                    return c && c.indexingMetadata && c.indexingMetadata.currentJobId;
                });
        };

        $scope.abortIndexation = function () {
            DataikuAPI.admin.connections.abortIndexation()
                .success(function (data) {
                    processRunningJobsRequest(data);
                })
                .error(setErrorInScope.bind($scope));
        };

        $scope.indexAllConnections = function () {
            CreateModalFromTemplate("/templates/admin/index-connections-modal.html", $scope, 'IndexConnectionsModalController');
        };

        $scope.focusNewConnectionFilter = function() {
            $timeout(() => {
                document.querySelector('#new-connection-query').focus();
            })
        }

        $scope.clearNewConnectionQuery = function($event) {
            $scope.uiState.newConnectionQuery = '';
            $scope.focusNewConnectionFilter();
            $event.stopPropagation();
        }

        $scope.filterNewConnectionGroups = function(connectionGroup) {
            return connectionGroup.connections.some(connection => connectionFound(connection, $scope.uiState.newConnectionQuery));
        }

        $scope.filterNewConnections = function(connection) {
            return connectionFound(connection, $scope.uiState.newConnectionQuery);
        }

        function connectionFound(connection, query) {
            query = query?.toLowerCase() || '';
            return (connection.title || '').toLowerCase().includes(query) || (connection.type || '').toLowerCase().includes(query) || (connection.category || '').toLowerCase().includes(query);
        }

        function buildConnectionList() {
            const categories = [
                {
                    title: 'SQL Databases',
                    connections: [
                        { type: 'Snowflake' },
                        { type: 'Databricks' },
                        { type: 'Redshift', title: 'Amazon Redshift' },
                        { type: 'BigQuery', title: 'Google BigQuery' },
                        { type: 'Synapse', title: 'Azure Synapse' },
                        { type: 'FabricWarehouse', title: 'MS Fabric Warehouse' },
                        { type: 'PostgreSQL' },
                        { type: 'MySQL' },
                        { type: 'SQLServer', title: 'MS SQL Server' },
                        { type: 'Oracle' },
                        { type: 'Teradata' },
                        ...($rootScope.featureFlagEnabled('lakebase') ? [{ type: 'DatabricksLakebase', title: 'Databricks Lakebase' }] : []),
                        { type: 'AlloyDB', title: 'Google AlloyDB' },
                        { type: 'Athena' },
                        { type: 'Greenplum' },
                        { type: 'Vertica' },
                        { type: 'SAPHANA' },
                        { type: 'Netezza', title: 'IBM Netezza' },
                        { type: 'Trino'},
                        { type: 'TreasureData' },
                        { type: 'Denodo' },
                        ...($rootScope.featureFlagEnabled('kdbplus') ? [{ type: 'KDBPlus', title: 'KDB+' }] : []),
                        { type: 'JDBC', title: 'Other SQL databases' },
                    ]
                }, {
                    title: 'File-based',
                    connections: [
                        ...($scope.appConfig.canAccessCloudDataikerAdminCapabilities && $scope.appConfig.admin ? [{ type: 'Filesystem', title: 'Server Filesystem' }] : []),
                        { type: 'FTP' },
                        { type: 'SSH', title: 'SCP/SFTP' },
                    ]
                }, {
                    title: 'Cloud Storages & Hadoop',
                    connections: [
                        { type: 'EC2', title: 'Amazon S3' },
                        { type: 'Azure', title: 'Azure Blob Storage' },
                        { type: 'GCS', title: 'Google Cloud Storage' },
                        ...($scope.appConfig.canAccessCloudDataikerAdminCapabilities ? [{ type: 'HDFS', title: 'Hadoop HDFS' }] : []),
                        { type: 'SharePointOnline', title: 'SharePoint Online' },
                    ]
                }, {
                    title: 'NoSQL',
                    connections: [
                        { type: 'MongoDB' },
                        { type: 'Cassandra' },
                        { type: 'ElasticSearch' },
                        ...($rootScope.featureFlagEnabled('DynamoDB') ? [{ type: 'DynamoDB' }] : []),
                    ]
                }, {
                    title: 'Managed Model Deployment Infrastructures',
                    connections: [
                        { type: 'SageMaker', title: 'Amazon SageMaker' },
                        { type: 'AzureML', title: 'Azure Machine Learning' },
                        { type: 'VertexAIModelDeployment', title: 'Google Vertex AI' },
                        { type: 'DatabricksModelDeployment', title: 'Databricks Managed Model Deployment' },
                    ]
                }, {
                    title: 'LLM Mesh',
                    connections: [
                        { type: 'OpenAI' },
                        { type: 'AzureOpenAI', title: 'Azure OpenAI' },
                        { type: 'AzureLLM', title: 'Azure LLM' },
                        { type: 'Cohere' },
                        { type: 'Anthropic' },
                        { type: 'VertexAILLM', title: 'Vertex Generative AI' },
                        { type: 'Bedrock', title: 'AWS Bedrock' },
                        { type: 'DatabricksLLM', title: 'Databricks Mosaic AI' },
                        { type: 'SnowflakeCortex', title: 'Snowflake Cortex' },
                        { type: 'MistralAI' },
                        { type: 'HuggingFaceLocal', title: 'Hugging Face' },
                        { type: 'SageMaker-GenericLLM', title: 'SageMaker LLM' },
                        { type: 'StabilityAI' },
                        { type: 'CustomLLM', title: 'Custom LLM' },
                        { type: 'Pinecone' },
                        { type: 'AzureAISearch', title: 'Azure AI Search' },
                    ]
                }, {
                    title: 'Other',
                    connections: [
                        { type: 'Twitter' },
                        ...($scope.appConfig.streamingEnabled ? [{ type: 'Kafka' }] : []),
                        ...($scope.appConfig.streamingEnabled ? [{ type: 'SQS' }] : []),
                    ]
                }
            ];

            categories.forEach(({ title, connections }) => connections.forEach(connection => connection.category = title));

            return categories;
        }
    });

    app.controller("ConnectionsController", function ($scope, $controller, TopNav, DataikuAPI, Dialogs, $timeout, $stateParams, ConnectionUtils, CreateModalFromTemplate) {
        $controller("_ConnectionsListBaseController", { $scope: $scope });
    });

    app.controller("ConnectionsHiveIndexingController", function ($scope, $controller, TopNav, DataikuAPI, Dialogs, $timeout, $stateParams, ConnectionUtils, CreateModalFromTemplate) {
        $controller("_ConnectionsListBaseController", { $scope: $scope });
    });

    app.controller("IndexConnectionsModalController", function ($scope, $state, $stateParams, TopNav, DataikuAPI, $timeout, FutureProgressModal, Dialogs) {
        $scope.indexationMode = 'scan';
        $scope.loading = true;

        DataikuAPI.admin.connections.listProcessableConnections($scope.type, $scope.selectedConnections).success(function (response) {
            $timeout(() => {
                $scope.loading = false;
                $scope.processableConnections = response;
                $scope.$digest();
            })
        }).error(setErrorInScope.bind($scope));

        $scope.canStartIndexation = function () {
            return $scope.processableConnections && (
                    $scope.indexationMode === 'index' && $scope.processableConnections.indexableConnections.length ||
                    $scope.indexationMode === 'scan' && $scope.processableConnections.scannableConnections.length
                );
        };
        $scope.startIndexation = function () {
            let indexationFunction;
            let connectionsToProcess;

            if ($scope.indexationMode === 'index') {
                indexationFunction = DataikuAPI.admin.connections.index;
                connectionsToProcess = $scope.processableConnections.indexableConnections;
            } else if ($scope.indexationMode === 'scan') {
                indexationFunction = DataikuAPI.admin.connections.scan;
                connectionsToProcess = $scope.processableConnections.scannableConnections;
            }

            indexationFunction(connectionsToProcess).success(function (data) { // NOSONAR: OK this call is indeed a method call.
                var parentScope = $scope.$parent.$parent;
                $scope.pollRunningJobs();
                FutureProgressModal.show(parentScope, data, "Indexing", newScope => newScope.tellToCloseWindow = true).then(function(result){
                    if (result) {
                        Dialogs.infoMessagesDisplayOnly(parentScope, "Indexing result", result);
                    }
                });
                $scope.dismiss();
            }).error(setErrorInScope.bind($scope));
        };
    });

    app.controller("ConnectionController", function ($scope, $rootScope, $state, $stateParams, Assert, TopNav, DataikuAPI, SqlConnectionNamespaceService) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");

        $scope.connectionParamsForm = {};

        $scope.availableCatalogs = null;
        $scope.availableSchemasMap = {};

        // Angular guys do not want to support the validation of an Angular input inside an AngularJS form, so handling validity through booleans.
        // See https://github.com/angular/angular/issues/9213.
        $scope.areAdvancedPropertiesInvalid = false;
        $scope.areAdvancedConnectionPropertiesInvalid = false;
        $scope.areJobLabelsInvalid = false;

        $scope.setAdvancedPropertiesValidity = function(isValid) {
            $scope.areAdvancedPropertiesInvalid = !isValid;
        };

        $scope.setAdvancedConnectionPropertiesValidity = function(isValid) {
            $scope.areAdvancedConnectionPropertiesInvalid = !isValid;
        };

        $scope.setJobLabelsValidity = function(isValid) {
            $scope.areJobLabelsInvalid = !isValid;
        };

        $scope.isConnectionParamsFormInvalid = function() {
            return $scope.connectionParamsForm.$invalid
                || $scope.areAdvancedPropertiesInvalid
                || $scope.areAdvancedConnectionPropertiesInvalid
                || $scope.areJobLabelsInvalid;
        };

        DataikuAPI.admin.connections.list().success(function (data) {
            $scope.connections = data;
        }).error(setErrorInScope.bind($scope));

        DataikuAPI.security.listGroups(false).success(function (data) {
            if (data) {
                data.sort();
            }
            $scope.allGroups = data;
        });

        function $canHaveProxy(connectionType) {
            return $scope.appConfig.hasGlobalProxy &&
                ['ElasticSearch', 'HTTP', 'FTP', 'EC2', 'GCS', 'Twitter', 'BigQuery', 'Snowflake', 'Azure', 'SAPHANA', 'SQS', 'Databricks', 'AzureML', 'VertexAIModelDeployment', 'SageMaker', 'DatabricksModelDeployment', 'SharePointOnline'].indexOf(connectionType) >= 0;
        }

        $scope.isFsProviderizable = function (t) {
            return ['Filesystem', 'FTP', 'SSH', 'HDFS', 'Azure', 'GCS', 'EC2', 'SharePointOnline', 'DatabricksVolume'].indexOf(t) >= 0;
        };

        $scope.canBeUsedAsVectorStore = function (t) {
            return ['ElasticSearch', 'AzureAISearch', 'Pinecone', 'GCS'].indexOf(t) >= 0;
        };

        $scope.getKBDisabledReason = function() {
            if (!$scope.connection.allowWrite){
                $scope.connection.allowKnowledgeBanks = false; // not necessary since it's already done in the allowWrite watcher but for consistency
                return "Write permissions on the connection are required to use knowledge banks.";
            }
            // No incompatibility found
            return null;
        }

        function isNonDataConnection(connectionType) {
            const managedModelDeploymentInfra = ['SageMaker','AzureML','VertexAIModelDeployment','DatabricksModelDeployment'];
            const llmConnections = ["Anthropic", "AzureOpenAI", "Bedrock", "Cohere", "CustomLLM", "DatabricksLLM", "HuggingFaceInferenceAPI", "HuggingFaceLocal",
                 "MosaicML", "OpenAI", "SageMaker-GenericLLM", "SnowflakeCortex", "VertexAILLMConnection"]; // keep in sync w/ connection inheriting from AbstractLLMConnection (connections/AbstractLLMConnection.java)
            return llmConnections.includes(connectionType) || managedModelDeploymentInfra.includes(connectionType);
        }

        Assert.trueish($stateParams.connectionName || $stateParams.type, "no $stateParams.connectionName and no $stateParams.type");
        if ($stateParams.connectionName) {
            $scope.creation = false;
            DataikuAPI.admin.connections.get($stateParams.connectionName).success(function (data) {
                $scope.savedConnection = angular.copy(data);
                $scope.connection = data;
                $scope.connection.$canHaveProxy = $canHaveProxy(data.type);
                $scope.loadDone = true;
                SqlConnectionNamespaceService.setTooltips($scope, data.type);
                $scope.isPersonalConnection = data.owner != null;
                if($scope.isPersonalConnection) {
                    $scope.usableBy = {
                        "policy": data.usableBy === "ALLOWED" && _.isEmpty(data.allowedGroups) ? "CREATOR" : data.usableBy
                    };
                } else {
                    $scope.usableBy = {
                        "policy": data.usableBy
                    };
                }
            }).error(setErrorInScope.bind($scope));
        } else if ($stateParams.type) {
            $scope.creation = true;
            $scope.savedConnection = null;
            $scope.isPersonalConnection = !$rootScope.appConfig.admin;
            $scope.usableBy = {
                "policy": ($scope.isPersonalConnection) ? "CREATOR" : "ALL"
            };
            $scope.connection = {
                "type": $stateParams.type,
                "params": {
                    namingRule: {}
                },
                "credentialsMode": "GLOBAL",
                "allowMirror": ($stateParams.type == "Vertica" || $stateParams.type == "ElasticSearch"),
                "usableBy": "ALL",
                "$canHaveProxy": $canHaveProxy($stateParams.type),
                "useGlobalProxy": $stateParams.type == 'ElasticSearch' ? false : $canHaveProxy($stateParams.type)
            };

            /* Per connection defaults */
            if ($scope.connection.type == "BigQuery") {
                $scope.connection.params.properties = [
                    { "name": "Timeout", "value": 180, "secret": false }
                ]
                $scope.connection.params.driverMode = "DRIVERLESS";
                $scope.connection.params.authType = "KEYPAIR";
                $scope.connection.params.forbidPartitionsWriteToNonPartitionedTable = true;
            } else if ($scope.connection.type == "Redshift") {
                $scope.connection.params.driverMode = "MANAGED_LEGACY_POSTGRESQL";
                $scope.connection.params.redshiftAuthenticationMode = "USER_PASSWORD";
            } else if ($scope.connection.type == "Greenplum") {
                $scope.connection.params.driverMode = "MANAGED_LEGACY_POSTGRESQL";
            } else if ($scope.connection.type == "PostgreSQL" || $scope.connection.type === "AlloyDB"
                    || $scope.connection.type == "Snowflake" ||  $scope.connection.type == "Databricks" ||  $scope.connection.type == "DatabricksLakebase") {
                $scope.connection.params.driverMode = "MANAGED";
            } else if ($scope.connection.type == "Trino") {
                $scope.connection.params.driverMode = "MANAGED";
                $scope.connection.params.authType = "PASSWORD";
            } else if ($scope.connection.type == "Teradata") {
                $scope.connection.params.properties = [
                    { "name": "CHARSET", "value": "UTF8", "secret": false }
                ]
            } else if ($scope.connection.type == "SSH") {
                $scope.connection.params.port = 22;
            } else if ($scope.connection.type == "HDFS") {
                $scope.connection.params.hiveSynchronizationMode = 'KEEP_IN_SYNC';
            } else if ($scope.connection.type == "EC2") {
                $scope.connection.params.credentialsMode = "KEYPAIR";
                $scope.connection.params.switchToRegionFromBucket = true;
            } else if ($scope.connection.type == "Synapse") {
                // The first option is here to keep compatibility with SQLServer. It should not be used.
                $scope.connection.params.azureDWH = true;
                $scope.connection.params.autocommitMode = true;
            } else if ($scope.connection.type == "GCS") {
                $scope.connection.params.authType = "KEYPAIR";
            } else if ($scope.connection.type == "FabricWarehouse") {
                $scope.connection.params.grantType = "AUTHORIZATION_CODE";
                $scope.connection.params["azureOAuthLoginEnabled"] = true;
                $scope.connection.params["kerberosLoginEnabled"] = false;
                $scope.connection.credentialsMode = "GLOBAL";
                $scope.connection.params["port"] = 1433;
                $scope.connection.params["refreshTokenRotation"] = false;
            } else if ($scope.connection.type == "Oracle") {
                $scope.connection.params.maxIdentifierSize = 30;
            } else if ($scope.connection.type == "Vertica") {
                $scope.connection.params.usePkce = true;
            } else if ($scope.connection.type == "TreasureData") {
                $scope.connection.params.region = "US";
                $scope.connection.params.db = "td-presto";
                $scope.connection.params.driverMode = "MANAGED";
            }

            $scope.connection.allowManagedFolders = $scope.isFsProviderizable($scope.connection.type);
            $scope.connection.allowKnowledgeBanks = $scope.canBeUsedAsVectorStore($scope.connection.type);
            $scope.connection.allowManagedDatasets = $scope.connection.allowWrite = !isNonDataConnection($scope.connection.type);

            $scope.loadDone = true;
        }
        $scope.customFieldsMap = $rootScope.appConfig.customFieldsMap["CONNECTION"];

        $scope.isConnectionNameUnique = function (v) {
            if (v == null) return true;
            if ($scope.connections == null) return true;
            return !$scope.connections.hasOwnProperty(v);
        };

        $scope.isConnectionNameValid = function () {
            return $scope.connection && $scope.connection.name && $scope.connection.name.length;
        };

        function usableByToConnection(connection, usableByPolicy) {
            const tmpConnection = angular.copy(connection);
            switch (usableByPolicy) {
                case "CREATOR":
                    tmpConnection.usableBy = "ALLOWED";
                    tmpConnection.allowedGroups = [];
                    break;
                default:
                    tmpConnection.usableBy = usableByPolicy;
            }

            return tmpConnection;
        }

        $scope.connectionDirty = function () {
            return !angular.equals(usableByToConnection($scope.connection, $scope.usableBy.policy), $scope.savedConnection);
        };

        $scope.saveConnection = function () {
            if ($scope.isConnectionParamsFormInvalid()) {
                return;
            }
            if ($scope.connection.params.regionOrEndpoint != null && $scope.connection.params.regionOrEndpoint.toLowerCase().startsWith('https://')) {
                $scope.connection.params.switchToRegionFromBucket = false;
            }
            const tmpConnection = usableByToConnection($scope.connection, $scope.usableBy.policy);
            DataikuAPI.admin.connections.save(tmpConnection, $scope.creation).success(function (data) {
                $scope.savedConnection = tmpConnection;
                // reset available schemas everytime to permit the user to refresh schemas if needed.
                // ideally we should only refresh it if basic params, advanced params or credentials have changed but it is a bit complicated to check
                $scope.availableCatalogs = null;
                $scope.availableSchemasMap = {};
                $state.transitionTo("admin.connections.edit", {connectionName: $scope.connection.name});
            }).error(setErrorInScope.bind($scope));
        };

        $scope.$watch('connection.allowWrite', function (a) {
            if (!a && $scope.connection) {
                $scope.connection.allowManagedDatasets = false;
                $scope.connection.allowManagedFolders = false;
                $scope.connection.allowKnowledgeBanks = false;
            }
        });

        $scope.canSwitchToRegionFromBucket = function(endpoint) {
            return !endpoint || !endpoint.toLowerCase().startsWith('https://');
        };

        $scope.fetchCatalogs = function(connectionName, origin, connectionType, inputBtnName) {
            SqlConnectionNamespaceService.listSqlCatalogs(connectionName, $scope, origin, connectionType, inputBtnName);
        };

        $scope.fetchSchemas = function(connectionName, catalog, origin, connectionType, inputBtnName) {
            SqlConnectionNamespaceService.listSqlSchemas(connectionName, $scope, catalog, origin, connectionType, inputBtnName);
        };

        $scope.$on('$destroy', function() {
            SqlConnectionNamespaceService.abortListSqlSchemas($scope);
            SqlConnectionNamespaceService.abortListSqlCatalogs($scope);
        });
        
    });

    app.controller("SQLConnectionController", function ($scope, $controller, DataikuAPI, $timeout, $rootScope, TopNav) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");

        $scope.supportsWriteSQLComment = function(type) {
            return ["Snowflake", "Databricks", "PostgreSQL", "Oracle", "Redshift", "MySQL", "BigQuery"].includes(type)
        }

        $scope.supportsWriteSQLCommentInCreateTableStatement = (type) => {
            return ["Snowflake", "BigQuery", "Databricks", "MySQL"].includes(type)
        }

        if (!$scope.connection.params.properties) {
            $scope.connection.params.properties = [];
        }

        if ($scope.creation) {
            const dp = $rootScope.appConfig && $rootScope.appConfig.namingRulesSettings && $rootScope.appConfig.namingRulesSettings.defaultPrefixForTables ?
                    $rootScope.appConfig.namingRulesSettings.defaultPrefixForTables : "${projectKey}_";
            $scope.connection.params.namingRule.tableNameDatasetNamePrefix = dp;
            $scope.connection.params.namingRule.writeDescriptionsAsSQLComment = $scope.supportsWriteSQLCommentInCreateTableStatement($scope.connection.type);
        }

        if (!$scope.connection.params.dkuProperties) {
            $scope.connection.params.dkuProperties = [];
        }

        $scope.warnings = {
            noVariableInTable: false,
        };

        if (!$scope.connection.customBasicConnectionCredentialProviderParams) {
            $scope.connection.customBasicConnectionCredentialProviderParams = [];
        }

        $scope.checkForHttpInHostUrl = (host) => host && (host.startsWith('http://') || host.startsWith('https://'));

        // Force teradata timezone to GMT on connection creation
        if ($scope.connection.type === "Teradata" && $scope.creation) {
            $scope.connection.params.defaultAssumedTzForUnknownTz = "GMT";
            $scope.connection.params.defaultAssumedDbTzForUnknownTz = "GMT";
        }

        $scope.$watch("connection.params", function (nv, ov) {
            $scope.warnings.noVariableInTable = false;
            // Snowflake and BigQuery don't support global Oauth yet ch63879
            if ($scope.connection.type=="BigQuery") {
                if (nv.authType=="OAUTH") {
                    $scope.connection.credentialsMode = "PER_USER";
                } else if (nv.authType=="KEYPAIR") {
                    $scope.connection.credentialsMode = "GLOBAL";
                }
            }
            if ($scope.connection.type=="Databricks" && nv.authType=="OAUTH2_APP" && $scope.connection.params.authType != $scope.savedConnection.params.authType) {
                $scope.connection.credentialsMode = "PER_USER";
            }
            if ($scope.connection.type=="Trino" && nv.authType=="OAUTH2" && $scope.connection.params.authType != $scope.savedConnection.params.authType) {
                $scope.connection.credentialsMode = "PER_USER";
            }
            if (!nv) return;
            if (!$scope.connection.allowManagedDatasets) return;

            if ((!nv.namingRule.tableNameDatasetNamePrefix || nv.namingRule.tableNameDatasetNamePrefix.indexOf("${") == -1) &&
                (!nv.namingRule.tableNameDatasetNameSuffix || nv.namingRule.tableNameDatasetNameSuffix.indexOf("${") == -1) &&
                (!nv.namingRule.schemaName || nv.namingRule.schemaName.indexOf("${") == -1)) {
                $scope.warnings.noVariableInTable = true;
            }
        }, true);
        $scope.$watch("connection.credentialsMode", function (nv, ov) {
            if ($scope.connection.type=="BigQuery") {
                if (nv=="GLOBAL") {
                    if ($scope.connection.params.authType == "OAUTH") {
                        $scope.connection.params.authType = "KEYPAIR";
                    }
                } else if (nv=="PER_USER") {
                    $scope.connection.params.authType = "OAUTH";
                }
            }
        });


        $scope.uiState = {};

        $scope.testConnection = function () {
            if (!$scope.connectionParamsForm || $scope.connectionParamsForm.$valid) {
                DataikuAPI.admin.connections.testSQL($scope.connection, null).success(function (data) {
                    $scope.testResult = data;
                }).error(setErrorInScope.bind($scope));
            }
        };

        $scope.getCatalog = function () {
            if ($scope.connection.type === "BigQuery") {
                return $scope.connection.params.projectId;
            }
            if ($scope.connection.type === "Snowflake") {
                return $scope.connection.params.db;
            }
            if ($scope.connection.type === "Databricks") {
                return $scope.connection.params.defaultCatalog;
            }
            if ($scope.connection.type === "SQLServer") {
                return $scope.connection.params.db;
            }
            if ($scope.connection.type === "MySQL") {
                return $scope.connection.params.db;
            }
            return null;
        }

        $scope.$watch("connection", function (nv, ov) {
            if (nv != null) {
                if (!$scope.connection.params.properties) {
                    $scope.connection.params.properties = [];
                }
            }
        });

        $scope.warnAboutSearchPath = function () {
            if ($scope.connection.params.schemaSearchPath) {
                if ($scope.connection.params.namingRule.schemaName) {
                    return false;
                }
                if ($scope.connection.params.schemaSearchPath.indexOf(',public,') > 0) { // NOSONAR: OK to ignore 0 index.
                    return false;
                }
                if ($scope.connection.params.schemaSearchPath.endsWith(',public')) {
                    return false;
                }
                if ($scope.connection.params.schemaSearchPath.startsWith('public,')) {
                    return false;
                }
                if ($scope.connection.params.schemaSearchPath == 'public') {
                    return false;
                }
                return true;
            } else {
                // no schema search path => don't care
                return false;
            }
        };

        $scope.connectionHasConceptOfDefaultCatalogAndSchema = function() {
            if (!$scope.connection) return false;

            /* Keep in sync with SQLUtils.resolveCatalogFromConnectionDefault / SQLUtils.resolveSchemaFromConnectionDefault */
            return ["MySQL", "Snowflake", "BigQuery", "SQLServer", "Databricks"].indexOf($scope.connection.type) >= 0;
        }

         $scope.dialects = [
            {"value":"","label":"Default"},
            {"value":"MySQLDialect","label":"MySQL < 8.0"},
            {"value":"MySQL8Dialect","label":"MySQL >= 8.0"},
            {"value":"PostgreSQLDialect","label":"PostgreSQL"},
            {"value":"OracleSQLDialect","label":"Oracle"},
            {"value":"SQLServerSQLDialect","label":"SQL Server"},
            {"value":"SynapseSQLDialect","label":"Azure Synapse"},
            {"value":"FabricWarehouseSQLDialect","label":"MS Fabric Warehouse"},
            {"value":"GreenplumSQLDialect","label":"Greenplum < 5.0"},
            {"value":"Greenplum5SQLDialect","label":"Greenplum >= 5.0"},
            {"value":"TeradataSQLDialect","label":"Teradata"},
            {"value":"VerticaSQLDialect","label":"Vertica"},
            {"value":"RedshiftSQLDialect","label":"Redshift"},
            {"value":"SybaseIQSQLDialect","label":"Sybase IQ"},
            {"value":"AsterDataSQLDialect","label":"Aster Data"},
            {"value":"NetezzaSQLDialect","label":"IBM Netezza"},
            {"value":"BigQuerySQLDialect","label":"Google BigQuery"},
            {"value":"SAPHANASQLDialect","label":"SAP HANA"},
            {"value":"ExasolSQLDialect","label":"Exasol"},
            {"value":"SnowflakeSQLDialect","label":"Snowflake"},
            {"value":"DatabricksSQLDialect","label":"Databricks"},
            {"value":"DB2SQLDialect","label":"IBM DB2"},
            {"value":"H2SQLDialect","label":"H2 < 2.0"},
            {"value":"H2V2SQLDialect","label":"H2 >= 2.0"},
            {"value":"ImpalaSQLDialect","label":"Impala"},
            {"value":"HiveSQLDialect","label":"Hive"},
            {"value":"PrestoSQLDialect","label":"Presto"},
            {"value":"TrinoSQLDialect","label":"Trino"},
            {"value":"AthenaSQLDialect","label":"Athena"},
            {"value":"SparkSQLDialect","label":"SparkSQL (via JDBC)"},
            {"value":"SqreamSQLDialect","label":"SQream"},
            {"value":"YellowbrickSQLDialect","label":"Yellowbrick"},
            {"value":"DremioSQLDialect","label":"Dremio"},
            {"value":"DenodoSQLDialect","label":"Denodo"}
        ];
        if ($rootScope.featureFlagEnabled("kdbplus")) {
            $scope.dialects.push({"value":"KDBSQLDialect","label":"KDB+"});
        }
        $rootScope.appConfig.customDialects.forEach(function(d) {
            $scope.dialects.push({"value":d.dialectType, "label":d.desc.meta.label || d.id})
        });
    });

    app.controller("PostgreSQLConnectionController", function ($scope, $controller, DataikuAPI) {
        $controller('SQLConnectionController', {$scope: $scope});

        $scope.testConnection = function () {
            if (!$scope.connectionParamsForm || $scope.connectionParamsForm.$valid) {
                $scope.testing = true;
                DataikuAPI.admin.connections.testPostgreSQL($scope.connection).success(function (data) {
                    $scope.testing = false;
                    $scope.testResult = data;
                }).error(function (a, b, c) {
                    $scope.testing = false;
                    setErrorInScope.bind($scope)(a, b, c)
                });
            }
        }
    });

    app.controller("FilesystemConnectionController", function ($scope, $controller, TopNav) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");

        $scope.notTestable = true;

        if (!$scope.connection.params.dkuProperties) {
            $scope.connection.params.dkuProperties = [];
        }
    });

    app.controller("HDFSConnectionController", function ($scope, $controller, $rootScope, TopNav, $stateParams, DataikuAPI, FutureProgressModal, Dialogs) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");

        $scope.isExtraHadoopConfigurationInvalid = false;

        $scope.setExtraHadoopConfigurationValidity = function(isValid) {
            $scope.isExtraHadoopConfigurationInvalid = !isValid;
        }

        // Overriding Connection Common
        $scope.isConnectionParamsFormInvalid = function() {
            return $scope.connectionParamsForm.$invalid || $scope.isExtraHadoopConfigurationInvalid || $scope.areAdvancedConnectionPropertiesInvalid;
        }

        $scope.notTestable = true;

        if ($scope.creation) {
            const dpp = $rootScope.appConfig && $rootScope.appConfig.namingRulesSettings && $rootScope.appConfig.namingRulesSettings.defaultPrefixForPath ?
                                $rootScope.appConfig.namingRulesSettings.defaultPrefixForPath : "${projectKey}/";
            const dpt = $rootScope.appConfig && $rootScope.appConfig.namingRulesSettings && $rootScope.appConfig.namingRulesSettings.defaultPrefixForTables ?
                                $rootScope.appConfig.namingRulesSettings.defaultPrefixForTables : "${projectKey}_";
            $scope.connection.params.namingRule.hdfsPathDatasetNamePrefix = dpp;
            $scope.connection.params.namingRule.tableNameDatasetNamePrefix = dpt;
        }

        if (!$scope.connection.customBasicConnectionCredentialProviderParams) {
            $scope.connection.customBasicConnectionCredentialProviderParams = [];
        }

        if (!$scope.connection.params.dkuProperties) {
            $scope.connection.params.dkuProperties = [];
        }

        $scope.warnings = {
            noVariableInPath: false,
            noVariableInHive: false,
        };

        $scope.$watch("connection.params", function (nv, ov) {
            $scope.warnings.noVariableInPath = false;
            $scope.warnings.noVariableInHive = false;

            if (!nv) return;
            if (!$scope.connection.allowManagedDatasets) return;


            if ((!nv.namingRule.hdfsPathDatasetNamePrefix || nv.namingRule.hdfsPathDatasetNamePrefix.indexOf("${") == -1) &&
                (!nv.namingRule.hdfsPathDatasetNameSuffix || nv.namingRule.hdfsPathDatasetNameSuffix.indexOf("${") == -1)) {
                $scope.warnings.noVariableInPath = true;
            }

            if ((!nv.namingRule.tableNameDatasetNamePrefix || nv.namingRule.tableNameDatasetNamePrefix.indexOf("${") == -1) &&
                (!nv.namingRule.tableNameDatasetNameSuffix || nv.namingRule.tableNameDatasetNameSuffix.indexOf("${") == -1) &&
                (!nv.namingRule.hiveDatabaseName || nv.namingRule.hiveDatabaseName.indexOf("${") == -1)) {
                $scope.warnings.noVariableInHive = true;
            }
        }, true);

        $scope.resyncPermissions = function () {
            DataikuAPI.admin.connections.hdfs.resyncPermissions($stateParams.connectionName).success(function (data) {
                FutureProgressModal.show($scope, data, "Permissions update").then(function (result) {
                    Dialogs.infoMessagesDisplayOnly($scope, "Update result", result);
                })
            }).error(setErrorInScope.bind($scope));
        }

        $scope.resyncRootPermissions = function () {
            DataikuAPI.admin.connections.hdfs.resyncRootPermissions($stateParams.connectionName).success(function (data) {
                FutureProgressModal.show($scope, data, "Permissions update");
            }).error(setErrorInScope.bind($scope));
        }

        DataikuAPI.projects.list().success(function (data) {
            $scope.projectsList = data;
            if (data.length) {
                $scope.massImportTargetProjectKey = data[0].projectKey;
                $scope.massImportTargetProjectName = data[0].name;
            }
            $scope.$watch("massImportTargetProjectKey", function () {
                var filteredProjects = $scope.projectsList.filter(function (project) {
                    return project.projectKey == $scope.massImportTargetProjectKey;
                });
                if (filteredProjects && filteredProjects.length) {
                    $scope.massImportTargetProjectName = filteredProjects[0].name;
                } else {
                    $scope.massImportTargetProjectName = null;
                }
            })

        }).error(setErrorInScope.bind($scope));
    });


    app.controller("EC2ConnectionController", function ($scope, $controller, DataikuAPI, TopNav, $rootScope) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");

        $scope.testConnection = function () {
            if (!$scope.connectionParamsForm || $scope.connectionParamsForm.$valid) {
                $scope.testing = true;
                if ($scope.connection.params.regionOrEndpoint != null && $scope.connection.params.regionOrEndpoint.toLowerCase().startsWith('https://')) {
                    $scope.connection.params.switchToRegionFromBucket = false;
                }
                DataikuAPI.admin.connections.testEC2($scope.connection).success(function (data) {
                    $scope.testing = false;
                    $scope.testResult = data;
                }).error(function (a, b, c) {
                    $scope.testing = false;
                    setErrorInScope.bind($scope)(a, b, c)
                });
            }
        }

        if ($scope.creation) {
            $scope.connection.params["defaultManagedPath"] = "/dataiku";
            /* On this connection, null prefix defaults to ${projectKey}/, we don't set it explicitly if not in the naming rule settings */
            if ($rootScope.appConfig && $rootScope.appConfig.namingRulesSettings && $rootScope.appConfig.namingRulesSettings.defaultPrefixForPath) {
                $scope.connection.params.namingRule.pathDatasetNamePrefix = $rootScope.appConfig.namingRulesSettings.defaultPrefixForPath;
            }
        }
        if (!$scope.connection.params["hdfsInterface"]) {
            $scope.connection.params["hdfsInterface"] = "S3A";  // Default value
        }
        if (!$scope.connection.params.customAWSCredentialsProviderParams) {
            $scope.connection.params.customAWSCredentialsProviderParams = [];
        }
        if (!$scope.connection.customBasicConnectionCredentialProviderParams) {
            $scope.connection.customBasicConnectionCredentialProviderParams = [];
        }
        if (!$scope.connection.params.dkuProperties) {
            $scope.connection.params.dkuProperties = [];
        }
    });

    app.controller("DatabricksVolumeConnectionController", function ($scope, $controller, DataikuAPI, TopNav) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");

        $scope.testConnection = function () {
            if (!$scope.connectionParamsForm || $scope.connectionParamsForm.$valid) {
                $scope.testing = true;
                DataikuAPI.admin.connections.testDatabricksVolume($scope.connection).success(function (data) {
                    $scope.testing = false;
                    $scope.testResult = data;
                }).error(function (a, b, c) {
                    $scope.testing = false;
                    setErrorInScope.bind($scope)(a, b, c)
                });
            }
        }

        if ($scope.creation) {
            $scope.connection.params["defaultManagedPath"] = "/dataiku";
        }
        if (!$scope.connection.params.dkuProperties) {
            $scope.connection.params.dkuProperties = [];
        }
    });

    app.controller("SageMakerConnectionController", function ($scope, DataikuAPI, TopNav) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");

        $scope.testConnection = function () {
            $scope.testing = true;
            $scope.testResult = null;
            if ($scope.connection.params.regionOrEndpoint != null && $scope.connection.params.regionOrEndpoint.toLowerCase().startsWith('https://')) {
                $scope.connection.params.switchToRegionFromBucket = false;
            }
            DataikuAPI.admin.connections.testSageMaker($scope.connection).success(function (data) {
                $scope.testing = false;
                $scope.testResult = data;
            }).error(function (a, b, c) {
                $scope.testing = false;
                setErrorInScope.bind($scope)(a, b, c)
            });
        }

        if ($scope.creation) {
            $scope.connection.params.credentialsMode = "STS_ASSUME_ROLE";
        }

    });

    app.controller("GCSConnectionController", function ($scope, $controller, $rootScope, DataikuAPI, TopNav) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");

        $scope.testConnection = function () {
            if (!$scope.connectionParamsForm || $scope.connectionParamsForm.$valid) {
                $scope.testing = true;
                DataikuAPI.admin.connections.testGCS($scope.connection).success(function (data) {
                    $scope.testing = false;
                    $scope.testResult = data;
                }).error(function (a, b, c) {
                    $scope.testing = false;
                    setErrorInScope.bind($scope)(a, b, c)
                });
            }
        }

        $scope.$watch("connection.params", function (nv, ov) {
            // GCS doesn't support global Oauth yet
            if (nv.authType=="OAUTH") {
               $scope.connection.credentialsMode = "PER_USER";
            } else {
               $scope.connection.credentialsMode = "GLOBAL";
            }
        }, true);

        if ($scope.creation) {
            $scope.connection.params["defaultManagedPath"] = "/dataiku";
            /* On this connection, null prefix defaults to ${projectKey}/, we don't set it explicitly if not in the naming rule settings */
            if ($rootScope.appConfig && $rootScope.appConfig.namingRulesSettings && $rootScope.appConfig.namingRulesSettings.defaultPrefixForPath) {
                $scope.connection.params.namingRule.pathDatasetNamePrefix = $rootScope.appConfig.namingRulesSettings.defaultPrefixForPath;
            }
        }

        if (!$scope.connection.params.dkuProperties) {
            $scope.connection.params.dkuProperties = [];
        }
    });

    app.controller("SharePointOnlineConnectionController", function ($scope, $controller, DataikuAPI, TopNav) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");

        $scope.testConnection = function () {
            $scope.testing = true;
            DataikuAPI.admin.connections.testSharePointOnline($scope.connection).success(function (data) {
                $scope.testing = false;
                $scope.testResult = data;
            }).error(function (a, b, c) {
                $scope.testing = false;
                setErrorInScope.bind($scope)(a, b, c)
            });
        }

        if ($scope.creation) {
            $scope.connection.params = {};
            $scope.connection.params["defaultManagedPath"] = "/dataiku/";
            $scope.connection.params.authType = "OAUTH2_APP";
            $scope.connection.credentialsMode = "PER_USER";
            $scope.connection.params.credentialsMode = "PER_USER";
            $scope.connection.allowManagedDatasets = true;
            $scope.connection.allowManagedFolders = true;
            $scope.connection.allowWrite = true;
            $scope.connection.params.scopes = "User.Read Files.ReadWrite.All Sites.ReadWrite.All Sites.Manage.All offline_access";
            $scope.connection.params.authorizationEndpoint = "https://login.microsoftonline.com/common/oauth2/v2.0/authorize";
            $scope.connection.params.tokenEndpoint = "https://login.microsoftonline.com/common/oauth2/v2.0/token";
        }

        if (!$scope.connection.params.properties) {
            $scope.connection.params.properties = [];
        }

        if (!$scope.connection.params.dkuProperties) {
            $scope.connection.params.dkuProperties = [];
        }

        if (!$scope.connection.credentialsMode) {
            $scope.connection.credentialsMode = "PER_USER";
        }
        if ($scope.connection.authType=="PASSWORD") {
            $scope.connection.credentialsMode = "GLOBAL";
        }


        $scope.$watch("connection.credentialsMode", function (nv, ov) {
            if (!nv || nv === ov) return;
        
            if (nv === "GLOBAL") {
                $scope.connection.params.scopes = "https://graph.microsoft.com/.default";
            } else {
                $scope.connection.params.scopes = "User.Read Files.ReadWrite.All Sites.ReadWrite.All Sites.Manage.All offline_access";
            }
        }, true);
    });

    app.controller("VertexAIModelDeploymentConnectionController", function ($scope, $controller, DataikuAPI, TopNav) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");

        $scope.testConnection = function () {
            $scope.testing = true;
            $scope.testResult = null;
            DataikuAPI.admin.connections.testVertexAIModelDeployment($scope.connection).success(function (data) {
                $scope.testing = false;
                $scope.testResult = data;
            }).error(function (a, b, c) {
                $scope.testing = false;
                setErrorInScope.bind($scope)(a, b, c)
            });
        }

        $scope.$watch("connection.params", function (nv, ov) {
            // GCS doesn't support global Oauth yet
            if (nv.authType=="OAUTH") {
               $scope.connection.credentialsMode = "PER_USER";
            } else {
               $scope.connection.credentialsMode = "GLOBAL";
            }
        }, true);

        if ($scope.creation) {
            $scope.connection.params.authType = "KEYPAIR";
            $scope.connection.params.dkuProperties = [];
        }
    });

    app.controller("DatabricksModelDeploymentConnectionController", function ($scope, $controller, DataikuAPI, TopNav) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");

        $scope.testConnection = function () {
            $scope.testing = true;
            $scope.testResult = null;
            DataikuAPI.admin.connections.testDatabricksModelDeployment($scope.connection).success(function (data) {
                $scope.testing = false;
                $scope.testResult = data;
            }).error(function (a, b, c) {
                $scope.testing = false;
                setErrorInScope.bind($scope)(a, b, c)
            });
        }

        $scope.$watch("connection.params", function (nv, ov) {
            // ###TODO check this
            // Databricks doesn't support global Oauth yet
            if (nv.authType=="OAUTH2_APP") {
               $scope.connection.credentialsMode = "PER_USER";
            } else {
               $scope.connection.credentialsMode = "GLOBAL";
            }
        }, true);

        $scope.$watch("connection.credentialsMode", function (nv, ov) {
            if (nv=="GLOBAL") {
                $scope.connection.params.authType = "PERSONAL_ACCESS_TOKEN";
            } else if (nv=="PER_USER") {
                $scope.connection.params.authType = "OAUTH2_APP";
            }
        });

        if ($scope.creation) {
            $scope.connection.params.authType = "PERSONAL_ACCESS_TOKEN";
            $scope.connection.params.dkuProperties = [];
        }
    });

    app.controller("AzureConnectionController", function ($scope, $controller, $rootScope, DataikuAPI, TopNav) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");

        $scope.testConnection = function () {
            if (!$scope.connectionParamsForm || $scope.connectionParamsForm.$valid) {
                $scope.testing = true;
                DataikuAPI.admin.connections.testAzure($scope.connection).success(function (data) {
                    $scope.testing = false;
                    $scope.testResult = data;
                }).error(function (a, b, c) {
                    $scope.testing = false;
                    setErrorInScope.bind($scope)(a, b, c)
                });
            }
        }

        if ($scope.creation) {
            $scope.connection.params["defaultManagedPath"] = "/dataiku";
            $scope.connection.params["defaultManagedContainer"] = "dataiku";
            $scope.connection.params["useSSL"] = true;
            $scope.connection.params["authType"] = "SHARED_KEY";

            /* On this connection, null prefix defaults to ${projectKey}/, we don't set it explicitly if not in the naming rule settings */
            if ($rootScope.appConfig && $rootScope.appConfig.namingRulesSettings && $rootScope.appConfig.namingRulesSettings.defaultPrefixForPath) {
                $scope.connection.params.namingRule.pathDatasetNamePrefix = $rootScope.appConfig.namingRulesSettings.defaultPrefixForPath;
            }
        }

        if (!$scope.connection.params.dkuProperties) {
            $scope.connection.params.dkuProperties = [];
        }
    });


    app.controller("AzureMLConnectionController", function ($scope, $controller, DataikuAPI, TopNav) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");

        $scope.testConnection = function () {
            $scope.testing = true;
            $scope.testResult = null;
            DataikuAPI.admin.connections.testAzureML($scope.connection).success(function (data) {
                $scope.testing = false;
                $scope.testResult = data;
            }).error(function (a, b, c) {
                $scope.testing = false;
                setErrorInScope.bind($scope)(a, b, c)
            });
        }

        if ($scope.creation) {
            $scope.connection.params["authType"] = "OAUTH2_APP";
        }

        if (!$scope.connection.params.dkuProperties) {
            $scope.connection.params.dkuProperties = [];
        }
    });


    app.controller("ElasticSearchConnectionController", function ($scope, $controller, DataikuAPI, TopNav, $rootScope) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");

        $scope.availableAuthTypes = [
            ['NONE', 'None', 'No authentication needed', null],
            ['PASSWORD', 'Simple', 'Provide a user and a password', null],
            ['OAUTH2_APP', 'OAuth', 'Use an authentication service supporting OAuth v2.0', null],
            ['AWS_KEYPAIR', 'AWS keypair', 'AccessId + SecretId. Supports only OpenSearch.', 'KEYPAIR'],
            ['AWS_ENVIRONMENT', 'AWS Environment', 'Use credentials from environment variables, or ~/.aws/credentials file, or instance profile. Supports only OpenSearch.', 'ENVIRONMENT'],
            ['AWS_STS', 'AWS STS with AssumeRole', 'Assume a role, with master credentials coming from the environment. Supports only OpenSearch.', 'STS_ASSUME_ROLE'],
            ['AWS_CUSTOM', 'AWS custom provider', 'Use a third-party authentication provider. Supports only OpenSearch.','CUSTOM_PROVIDER']];
        $scope.availableAuthTypesDesc = $scope.availableAuthTypes.map((x) => x[2]);
        $scope.availableAWSAuthTypesDesc = $scope.availableAuthTypes.reduce((acc, x) => {acc[x[0]] = x[3]; return acc }, {});

        $scope.availableAWSServiceTypes = [
            ['OPENSEARCH_SERVERLESS', 'OpenSearch Serverless', 'You are connecting to a AWS serverless instance of OpenSearch.'],
            ['OPENSEARCH_HOSTED', 'Managed OpenSearch', 'You are connecting to a managed OpenSearch instance hosted on AWS.']];
        $scope.availableAWSServiceTypesDesc = $scope.availableAWSServiceTypes.map((x) => x[2]);

        $scope.connectionParamsForm = {};
        if ($scope.creation) {
            $scope.connection.params["host"] = "localhost";
            $scope.connection.params["port"] = 9200;
            $scope.connection.params["dialect"] = 'ES_7';
            $scope.connection.params["connectionLimit"] = 8;
            $scope.connection.params["oauth"] = {};
            $scope.connection.params["aws"] = {"service": "OPENSEARCH_SERVERLESS"};
            $scope.connection.params["authType"] = "NONE";

            const dp = $rootScope.appConfig && $rootScope.appConfig.namingRulesSettings && $rootScope.appConfig.namingRulesSettings.defaultPrefixForTables ?
                                $rootScope.appConfig.namingRulesSettings.defaultPrefixForTables : "${projectKey}_";

            $scope.connection.params.namingRule.indexNameDatasetNamePrefix = dp;
        }

        $scope.isConnectionParamsFormInvalid = function() {
            return $scope.connectionParamsForm.$invalid;
        }

        $scope.checkForHttpInHostUrl = (host) => host && (host.startsWith('http://') || host.startsWith('https://'));

        $scope.testConnection = function () {
            if (!$scope.connectionParamsForm || $scope.connectionParamsForm.$valid) {
                DataikuAPI.admin.connections.testElasticSearch($scope.connection, null).success(function (data) {
                    if (data.dialect && data.dialect !== $scope.connection.params.dialect) {
                        data.dialectMismatch = true;
                    }
                    $scope.testResult = data;
                }).error(setErrorInScope.bind($scope));
            }
        };

        $scope.warnings = {
            noVariableInIndex: false,
        };

        $scope.isIndexNameAllowed = function (v) {
            if (!v) {
                return true;
            }
            // See https://www.elastic.co/guide/en/elasticsearch/reference/current/regexp-syntax.html#regexp-reserved-characters for reserved characters
            return v.match(/^(?:[^.?+*|{}[\]()"\\#@&<>~ ]|\$\{[^}]*\})+$/) !== null;
        };

        $scope.getKBDisabledReason = function() {

            if($scope.connection.params.dialect !== "ES_7") {
                $scope.connection.allowKnowledgeBanks = false;
                return "Knowledge banks require at least ElasticSearch version 7 or OpenSearch.";
            }
            if($scope.connection.params.authType == "OAUTH2_APP") {
                $scope.connection.allowKnowledgeBanks = false;
                return "Knowledge banks are not supported with OAuth authentication.";
            }
            if (!$scope.connection.allowWrite){
                $scope.connection.allowKnowledgeBanks = false; // not necessary since it's already done in the allowWrite watcher but for consistency
                return "Write permissions on the connection are required to use knowledge banks.";
            }
            // No incompatibility found
            return null;
        }

        $scope.fixDialectMismatch = function() {
            if($scope.testResult && $scope.testResult.dialect != $scope.connection.params.dialect) {
                $scope.connection.params.dialect = $scope.testResult.dialect;
                $scope.testConnection();
            }
        };

        $scope.$watch("connection.params", function (nv, ov) {
            $scope.warnings.noVariableInIndex = false;
            $scope.connection.params.aws['credentialsMode'] = $scope.availableAWSAuthTypesDesc[$scope.connection.params.authType];
            if ((!nv.namingRule.indexNameDatasetNamePrefix || nv.namingRule.indexNameDatasetNamePrefix.match(/^.*\$\{[^}]*\}.*$/) === null) &&
                (!nv.namingRule.indexNameDatasetNameSuffix || nv.namingRule.indexNameDatasetNameSuffix.match(/^.*\$\{[^}]*\}.*$/) === null)) {
                $scope.warnings.noVariableInIndex = true;
            }
        }, true);
    });

    app.controller("TwitterConnectionController", function ($scope, $controller, DataikuAPI, TopNav) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");

        $scope.connection.allowWrite = false;
        $scope.connection.allowManagedDatasets = false;
        $scope.connection.allowMirror = false;

        $scope.testConnection = function () {
            $scope.testing = true;
            DataikuAPI.admin.connections.testTwitter($scope.connection).success(function (data) {
                $scope.testResult = data;
            }).error(setErrorInScope.bind($scope))
                ["finally"](function () {
                $scope.testing = false;
            });
        }

        $scope.clearForm = function () {
            $scope.connection.params.api_key = "";
            $scope.connection.params.api_secret = "";
            $scope.connection.params.token_key = "";
            $scope.connection.params.token_secret = "";
            $scope.verified = false;
        }

        DataikuAPI.connections.getTwitterConfig().success(function (data) {
            var activeConnection = data.connection;
            $scope.isActive = ($scope.connection.name === activeConnection);
            $scope.isRunning = (data.running.length > 0);
        }).error(function () {
            setErrorInScope.bind($scope);
            $scope.isActive = false;
        });

        $scope.setActiveConnection = function (name) {
            DataikuAPI.admin.connections.setActiveTwitterConnection(name).success(function () {
                $scope.isActive = true;
            }).error(setErrorInScope.bind($scope));
        }

        DataikuAPI.connections.getNames('Twitter').success(function (data) {
            $scope.displaySetActive = (data.length > 1);
        }).error(setErrorInScope.bind($scope));
    });

    app.controller("_LLMConnectionController", function($scope, DataikuAPI, $state) {
        $scope.openAiConnections = [];
        $scope.huggingFaceLocalConnections = [];
        if ($scope.connection.params.imageAuditManagedFolderRef) {
            const chunks = $scope.connection.params.imageAuditManagedFolderRef.split('.');
            $scope.imageStorage = {
                projectKey: chunks[0],
                managedFolderId: chunks[1],
            };
        } else {
            $scope.imageStorage = {
                projectKey: null,
                managedFolderId: null,
            };
        }

        $scope.showFineTuningSettings = function() {
            return $scope.appConfig.licensedFeatures.advancedLLMMeshAllowed;
        }

        DataikuAPI.pretrainedModels.listAvailableConnectionLLMs("GENERIC_COMPLETION").success(function(data){
            $scope.availableCompletionLLMs = data["identifiers"];
        }).error(setErrorInScope.bind($scope));


        $scope.showFineTuningSettingsIfConnectionAllowsIt = function(allowFinetuning) {
            return $scope.showFineTuningSettings() && allowFinetuning;
        }

        $scope.$watch("imageStorage.projectKey", function(nv, ov) {
            if (nv !== ov) {
                // 1 - set the folder id to null in case the new project got a managed folder with the same name,
                //     forcing the user to select a managed folder for the newly selected project
                // 2 - setting managedFolderId to an empty string '' will trigger the other watch below,
                //     hence invalidating the folder ref, forcing the user to select a new folder to get rid of the warning
                $scope.imageStorage.managedFolderId = '';
                $scope.connection.params.imageAuditManagedFolderRef = $scope.imageStorage.projectKey + ".";
             }
        });

        $scope.$watch("imageStorage.managedFolderId", function(nv, ov) {
            if (nv !== ov) {
                $scope.connection.params.imageAuditManagedFolderRef = $scope.imageStorage.projectKey + "." + $scope.imageStorage.managedFolderId;
            }
        });

        const deregister = $scope.$watch("connection.params", function(nv) {
            if (!nv) return;

            if ($scope.creation) {
                $scope.connection.params.auditingMode = "METADATA_ONLY";

                $scope.connection.params.guardrailsPipelineSettings = {"guardrails":[]};

                $scope.connection.params.cachingEnabled = false;
                $scope.connection.params.embeddingsCachingEnabled = true;

                $scope.connection.params.networkSettings = {
                    queryTimeoutMS: 60000,
                    maxRetries: 3,
                    initialRetryDelayMS: 3000,
                    retryDelayScalingFactor: 2.0,
                }
            }

            deregister();
        });
    });

    // Keep in sync with CustomOpenAIModelType (dip/connections/OpenAIConnection.java)
    app.constant("OpenAiModelTypes", [
        { rawValue: 'COMPLETION_CHAT', displayName: 'Chat completion' },
        { rawValue: 'COMPLETION_CHAT_MULTIMODAL', displayName: 'Chat completion (multimodal)' },
        { rawValue: 'COMPLETION_CHAT_NO_SYSTEM_PROMPT', displayName: 'Chat completion (no system message)' },
        { rawValue: 'COMPLETION_SIMPLE', displayName: 'Simple completion (legacy)' },
        { rawValue: 'TEXT_EMBEDDING_EXTRACTION', displayName: 'Embedding' },
        { rawValue: 'IMAGE_GENERATION', displayName: 'Image Generation' } // Can't really hide this under FF due to usage of app.constant :/
    ]);

    // Keep in sync with CustomOpenAIModelType (dip/connections/OpenAIConnection.java:OpenAIConnectionParams.OpenAIMaxTokensAPIMode)
    // See SC-217279
    app.constant('OpenAiMaxTokensAPIModes', [
        { rawValue: 'AUTO', displayName: 'Auto' },
        { rawValue: 'MODERN', displayName: 'Modern', description: 'Use max_completion_tokens parameter' },
        { rawValue: 'LEGACY', displayName: 'Legacy', description: 'Use max_tokens parameter (deprecated by OpenAI)' },
    ]);

    // Keep in sync with AzureOpenAIConnection (dip/connections/OpenAIConnection.java:AzureOpenAIConnection.AzureOpenAIMaxTokensAPIMode)
    app.constant('AzureOpenAiMaxTokensAPIModes', [
        { rawValue: 'MODERN', displayName: 'Modern (gpt5*, gpt4*, o*)', description: 'Use max_completion_tokens parameter' },
        { rawValue: 'LEGACY', displayName: 'Legacy (gpt3* or earlier)', description: 'Use max_tokens parameter' },
    ]);

    // Keep in sync with OpenAIImageHandling (dip/connections/OpenAIImageHandling.java)
    app.constant('OpenAIImageHandlingModes', [
        { rawValue: 'DALL_E_3', displayName: 'Dall-E 3' },
        { rawValue: 'GPT_IMAGE_1', displayName: 'GPT Image 1' },
    ]);

    app.controller("OpenAiConnectionController", function ($scope, $controller, TopNav, DataikuAPI, OpenAiModelTypes, OpenAiMaxTokensAPIModes, OpenAIImageHandlingModes) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");
        $controller("_LLMConnectionController", {$scope});

        if ($scope.creation) {
            $scope.connection.params.allowFinetuning = false;
            $scope.connection.params["allowGPT5"] = true;
            $scope.connection.params["allowGPT5Mini"] = true;
            $scope.connection.params["allowGPT5Nano"] = true;
            $scope.connection.params["allowGPT5Chat"] = false;
            $scope.connection.params["allowGPT41"] = false;
            $scope.connection.params["allowGPT41Mini"] = false;
            $scope.connection.params["allowGPT41Nano"] = false;
            $scope.connection.params["allowGPT4oMini"] = false;
            $scope.connection.params["allowGPT35Turbo"] = false;
            $scope.connection.params["allowO3"] = false;
            $scope.connection.params["allowO4Mini"] = false;
            $scope.connection.params["allowEmbedding3Small"] = true;
            $scope.connection.params.maxParallelism = 8;
            $scope.connection.params.maxTokensAPIMode = 'AUTO';
        }

        $scope.testConnection = function () {
            $scope.testing = true;
            DataikuAPI.admin.connections.testOpenAi($scope.connection).success(function (data) {
                $scope.testResult = data;
            }).error(setErrorInScope.bind($scope))
                ["finally"](function () {
                $scope.testing = false;
            });
        };

        $scope.modelTypes = OpenAiModelTypes;
        $scope.maxTokensAPIModes = OpenAiMaxTokensAPIModes;
        $scope.imageHandlingModes = OpenAIImageHandlingModes;
    });

    app.controller("AzureOpenAiConnectionController", function ($scope, $controller, TopNav, DataikuAPI, OpenAiModelTypes, AzureOpenAiMaxTokensAPIModes, OpenAIImageHandlingModes) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");
        $controller("_LLMConnectionController", {$scope});

        if ($scope.creation) {
            $scope.connection.params.allowFinetuning = false;
            $scope.connection.params["availableDeployments"] = [];
            $scope.connection.params.maxParallelism = 8;
        }

        $scope.azureResourceURLFormat = "https://RESOURCE_NAME.openai.azure.com/openai"

        DataikuAPI.admin.connections.list().success(function(data) {
            $scope.azureMLConnections = Object.values(data).filter(c => c.type === 'AzureML').map(c => c.name);
        });

        $scope.testConnection = function () {
            $scope.testing = true;
            DataikuAPI.admin.connections.testAzureOpenAi($scope.connection).success(function (data) {
                $scope.testResult = data;
            }).error(setErrorInScope.bind($scope))
                ["finally"](function () {
                $scope.testing = false;
            });
        };

        $scope.deploymentTypes = OpenAiModelTypes;
        $scope.imageHandlingModes = OpenAIImageHandlingModes;

        $scope.usePromptAndCompletionCosts = function(deployment) {
            return ['COMPLETION_CHAT', 'COMPLETION_CHAT_MULTIMODAL', 'COMPLETION_CHAT_NO_SYSTEM_PROMPT', 'COMPLETION_SIMPLE'].includes(deployment.deploymentType)
        };

        $scope.useEmbeddingCosts = function(deployment) {
            return 'TEXT_EMBEDDING_EXTRACTION' === deployment.deploymentType;
        };

        const chatModels = [
            { rawValue: '', displayName: 'Custom pricing' },
            { rawValue: 'gpt-5', displayName: 'GPT 5', promptCost: .00125, completionCost: .010 },
            { rawValue: 'gpt-5-mini', displayName: 'GPT 5 mini', promptCost: .00025, completionCost: .002 },
            { rawValue: 'gpt-5-nano', displayName: 'GPT 5 nano', promptCost: .00005, completionCost: .0004 },
            { rawValue: 'gpt-5-chat', displayName: 'GPT 5 Chat', promptCost: .00125, completionCost: .010 },
            { rawValue: 'o4-mini', displayName: 'o4-mini', promptCost: .0011, completionCost: 0.0044 },
            { rawValue: 'o3', displayName: 'o3', promptCost: .01, completionCost: 0.04 },
            { rawValue: 'o3-mini', displayName: 'o3-mini', promptCost: .0011, completionCost: .0044 },
            { rawValue: 'o1', displayName: 'o1', promptCost: .015, completionCost: .06 },
            { rawValue: 'o1-mini', displayName: 'o1-mini', promptCost: .0011, completionCost: .0044 },
            { rawValue: 'gpt-4.1', displayName: 'GPT 4.1', promptCost: .002, completionCost: .008 },
            { rawValue: 'gpt-4.1-mini', displayName: 'GPT 4.1-mini', promptCost: .0004, completionCost: .0016 },
            { rawValue: 'gpt-4.1-nano', displayName: 'GPT 4.1-nano', promptCost: .0001, completionCost: .0004 },
            { rawValue: 'gpt-4o', displayName: 'GPT 4o', promptCost: .0025, completionCost: .01 },
            { rawValue: 'gpt-4o-mini', displayName: 'GPT 4o-mini', promptCost: .00015, completionCost: .0006 },
            { rawValue: 'gpt-4', displayName: 'GPT 4', promptCost: .03, completionCost: .06 },
            { rawValue: 'gpt-4-32k', displayName: 'GPT 4 (large context)', promptCost: .06, completionCost: .12 },
            { rawValue: 'gpt-3.5-turbo', displayName: 'GPT 3.5 Chat Turbo', promptCost: .0015, completionCost: .002 },
            { rawValue: 'gpt-3.5-turbo-16k', displayName: 'GPT 3.5 Turbo - large context' , promptCost: .003, completionCost: .004 }
        ];

        const simpleCompletionModels = [
            { rawValue: 'text-davinci-003', displayName: 'GPT 3 Text Davinci (text-davinci-003)', promptCost: .02, completionCost: .02 },
            { rawValue: 'babbage-002', displayName: 'GPT 3 Text Curie (text-curie-001)', promptCost: 0.002, completionCost: .002 },
            { rawValue: 'babbage-002', displayName: 'GPT 3 Text Babbage (text-babbage-001)', promptCost: 0.002, completionCost: .0005 },
            { rawValue: 'babbage-002', displayName: 'GPT 3 Text Ada (text-ada-001)', promptCost: 0.002, completionCost: .0004 },
            { rawValue: '', displayName: 'Custom pricing' },
        ];

        const embeddingModels = [
            { rawValue: 'text-embedding-ada-002', displayName: 'Embedding (Ada 002)', embeddingCost: 0.0001 },
            { rawValue: 'text-embedding-3-small', displayName: 'Embedding (v3 Small)', embeddingCost: 0.00002 },
            { rawValue: 'text-embedding-3-large', displayName: 'Embedding (v3 Large)', embeddingCost: 0.00013 },
            { rawValue: '', displayName: 'Custom pricing' },
        ];

        $scope.getOpenAIModels = function(deploymentType) {
            switch (deploymentType) {
            case 'COMPLETION_CHAT':
            case 'COMPLETION_CHAT_MULTIMODAL':
            case 'COMPLETION_CHAT_NO_SYSTEM_PROMPT':
                return chatModels;
            case 'COMPLETION_SIMPLE':
                return simpleCompletionModels;
            case 'TEXT_EMBEDDING_EXTRACTION':
                return embeddingModels;
            case 'IMAGE_GENERATION':
                return [];
            }
            // should not happen
            return [{ rawValue: '', displayName: 'Custom pricing' }];
        };

        $scope.onDeploymentTypeChange = function(deployment) {
            if (deployment.deploymentType !== 'IMAGE_GENERATION') {
                deployment.imageHandlingMode = null;
            } else if (!deployment.imageHandlingMode) {
                deployment.imageHandlingMode = 'DALL_E_3';
            }
            const availableModelsForPricing = $scope.getOpenAIModels(deployment.deploymentType);
            const openAIModel = availableModelsForPricing.find(model => model.rawValue === deployment.underlyingModelName);
            if (openAIModel) {
                deployment.underlyingModelName = openAIModel.rawValue;
            } else {
                deployment.underlyingModelName = '';
            }
        };

        $scope.onModelPricingChange = function(deployment) {
            const availableModelsForPricing = $scope.getOpenAIModels(deployment.deploymentType);
            const openAIModel = (availableModelsForPricing.find(model => model.rawValue === deployment.underlyingModelName) || {});
            deployment.promptCost = openAIModel.promptCost;
            deployment.completionCost = openAIModel.completionCost;
            deployment.embeddingCost = openAIModel.embeddingCost;
        };

        $scope.connection.params.availableDeployments.forEach($scope.onDeploymentTypeChange);
        $scope.maxTokensAPIModes = AzureOpenAiMaxTokensAPIModes;
    });

    app.controller("AzureLLMConnectionController", function ($scope, $controller, TopNav, DataikuAPI, OpenAiModelTypes) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");
        $controller("_LLMConnectionController", {$scope});

        $scope.uiState = {
            azureMLDeplConnections: null,
        };

        if ($scope.creation) {
            $scope.connection.params = {
                defaultKey: null,
                maxParallelism: 8,
                customModels:  [{
                    id: null,
                    displayName: null,
                    modelType: "COMPLETION_CHAT",
                    targetURI: null,
                    key: null,
                    promptCost: null,
                    completionCost: null,
                    embeddingCost: null
                }]
            }
        }

        $scope.modelTypes = OpenAiModelTypes.filter(mt => ['COMPLETION_CHAT', 'COMPLETION_SIMPLE', 'COMPLETION_CHAT_MULTIMODAL', 'COMPLETION_CHAT_NO_SYSTEM_PROMPT', 'TEXT_EMBEDDING_EXTRACTION'].includes(mt.rawValue));

        $scope.testConnection = function () {
            $scope.testing = true;
            DataikuAPI.admin.connections.testAzureMLGenericLLM($scope.connection).success(function (data) {
                $scope.testResult = data;
            }).error(setErrorInScope.bind($scope))
                ["finally"](function () {
                $scope.testing = false;
            });
        };

        $scope.usePromptAndCompletionCosts = function(model) {
            return ['COMPLETION_CHAT', 'COMPLETION_CHAT_MULTIMODAL','COMPLETION_SIMPLE'].includes(model.modelType)
        };

        $scope.useEmbeddingCosts = function(model) {
            return ['TEXT_EMBEDDING_EXTRACTION'].includes(model.modelType);
        };
    });

    app.controller("CohereConnectionController", function ($scope, $controller, TopNav, DataikuAPI) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");
        $controller("_LLMConnectionController", {$scope});

        if ($scope.creation) {
            $scope.connection.params["allowCohereCommand"] = false;
            $scope.connection.params["allowCohereCommandLight"] = false;
            $scope.connection.params["allowCohereCommandR"] = true;
            $scope.connection.params["allowCohereCommandRPlus"] = true;
            $scope.connection.params.maxParallelism = 1;
        }

        $scope.testConnection = function () {
            $scope.testing = true;
            DataikuAPI.admin.connections.testCohere($scope.connection).success(function (data) {
                $scope.testResult = data;
            }).error(setErrorInScope.bind($scope))
                ["finally"](function () {
                $scope.testing = false;
            });
        }
    });

    app.controller("StabilityAIConnectionController", function ($scope, $controller, TopNav, DataikuAPI) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");
        $controller("_LLMConnectionController", {$scope});

        $scope.notTestable = true;

        if ($scope.creation) {
            $scope.connection.params.maxParallelism = 1;
        }

        $scope.testConnection = function () {
            // TODO @llm-img
        }
    });

    app.controller("MistralAiConnectionController", function ($scope, $controller, TopNav, DataikuAPI) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");
        $controller("_LLMConnectionController", {$scope});

        if ($scope.creation) {
            $scope.connection.params["allowMistralSmall"] = true;
            $scope.connection.params["allowMistralMedium"] = true;
            $scope.connection.params["allowMistralLarge"] = true;

            $scope.connection.params["allowMistralEmbed"] = true;
            $scope.connection.params.maxParallelism = 4;
        }

        $scope.testConnection = function () {
            $scope.testing = true;
            DataikuAPI.admin.connections.testMistralAi($scope.connection).success(function (data) {
                $scope.testResult = data;
            }).error(setErrorInScope.bind($scope))
                ["finally"](function () {
                $scope.testing = false;
            });
        };
    });

    app.controller("AnthropicConnectionController", function ($scope, $controller, TopNav, DataikuAPI) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");
        $controller("_LLMConnectionController", {$scope});

        if ($scope.creation) {
            $scope.connection.params["allowClaudeV2"] = false;
            $scope.connection.params["allowClaudeV3Opus"] = false;
            $scope.connection.params["allowClaudeV4Opus"] = true;
            $scope.connection.params["allowClaudeV3Sonnet"] = false;
            $scope.connection.params["allowClaudeV35Sonnet"] = false;
            $scope.connection.params["allowClaudeV35SonnetV2"] = false;
            $scope.connection.params["allowClaudeV37Sonnet"] = false;
            $scope.connection.params["allowClaudeV4Sonnet"] = true;
            $scope.connection.params["allowClaudeV3Haiku"] = false;
            $scope.connection.params["allowClaudeV35Haiku"] = true;
            $scope.connection.params.maxParallelism = 2;
        }

        $scope.testConnection = function () {
            $scope.testing = true;
            DataikuAPI.admin.connections.testAnthropic($scope.connection).success(function (data) {
                $scope.testResult = data;
            }).error(setErrorInScope.bind($scope))
                ["finally"](function () {
                $scope.testing = false;
            });
        }
    });

    app.service('BedrockConnectionService', function() {
        // Keep in sync with the backend CustomBedrockModelType (dip/connections/BedrockConnection.java)
        this.BedrockModelTypes = [
            { rawValue: 'TEXT_MODEL', displayName: 'Chat completion' },
            { rawValue: 'MULTIMODAL_MODEL', displayName: 'Chat completion (multimodal)' },
            { rawValue: 'TEXT_EMBEDDING_EXTRACTION', displayName: 'Text embedding' },
            { rawValue: 'IMAGE_GENERATION', displayName: 'Image generation' },
            { rawValue: 'TEXT_IMAGE_EMBEDDING_EXTRACTION', displayName: 'Multimodal embedding' },
        ];

        // Keep in sync with the backend GenericLLMHandling (dip/llm/online/sagemakergeneric/GenericLLMHandling.java)
        // Commented out the modes that are not applicable for Bedrock
        this.BedrockHandlingModes = {
            TEXT_MODEL: [
                { rawValue: 'GENERIC_CONVERSE', displayName: 'Generic Converse API Model' },
                { rawValue: 'AMAZON_NOVA', displayName: 'Amazon Nova' },
                { rawValue: 'AMAZON_TITAN', displayName: 'Amazon Titan' },
                { rawValue: 'ANTHROPIC_CLAUDE_CHAT', displayName: 'Anthropic Claude Chat' },
                { rawValue: 'ANTHROPIC_CLAUDE', displayName: 'Anthropic Claude (legacy completion API)' },
                { rawValue: 'AI21_J2', displayName: 'AI21 Jurassic 2' },
                //{ rawValue: 'AI21_SUMMARIZE', displayName: 'AI21 Summarize'},
                { rawValue: 'COHERE_COMMAND_CHAT', displayName: 'Cohere Command Chat' },
                { rawValue: 'COHERE_COMMAND', displayName: 'Cohere Command (completion API)' },
                //{ rawValue: 'HUGGING_FACE', displayName: 'Hugging Face'},
                { rawValue: 'META_LLAMA_2_BEDROCK', displayName: 'Meta Llama 2' },
                { rawValue: 'META_LLAMA_3_BEDROCK', displayName: 'Meta Llama 3' },
                //{ rawValue: 'META_LLAMA_2_SAGEMAKER', displayName: 'Meta Llama 2'},
                { rawValue: 'MISTRAL_AI_CHAT', displayName: 'MistralAI Chat' },
                { rawValue: 'MISTRAL_AI', displayName: 'MistralAI (text completion API)' },
                //{ rawValue: 'FULLY_CUSTOM', displayName: 'Fully Custom Handling'},
            ],
            MULTIMODAL_MODEL: [
                { rawValue: 'GENERIC_CONVERSE', displayName: 'Generic Converse API Model' },
                { rawValue: 'AMAZON_NOVA', displayName: 'Amazon Nova' },
                { rawValue: 'ANTHROPIC_CLAUDE_CHAT', displayName: 'Anthropic Claude Chat' },
            ],
            TEXT_EMBEDDING_EXTRACTION: [
                { rawValue: 'AMAZON_TITAN_TEXT_EMBEDDING', displayName: 'Text embedding Amazon Titan' },
                { rawValue: 'COHERE_EMBED', displayName: 'Text embedding Cohere' },
            ],
            TEXT_IMAGE_EMBEDDING_EXTRACTION: [
               { rawValue: "AMAZON_TITAN_TEXT_IMAGE_EMBEDDING", displayName: "Multimodal embedding Amazon Titan" },
            ],
            IMAGE_GENERATION: [
                { rawValue: "AMAZON_TITAN", displayName: "Amazon Titan Image Generator G1" },
                { rawValue: "STABILITYAI_STABLE_DIFFUSION_10", displayName: "Stability AI SDXL 1.0" },
                { rawValue: "STABILITYAI_STABLE_IMAGE_CORE", displayName: "Stability AI Stable Image Core" },
                { rawValue: "STABILITYAI_STABLE_DIFFUSION_3", displayName: "Stability AI SD3" },
                { rawValue: "STABILITYAI_STABLE_IMAGE_ULTRA", displayName: "Stability AI Stable Image Ultra" },
            ]
        }

    })

    app.controller("BedrockConnectionController", function($scope, $controller, TopNav, DataikuAPI, BedrockConnectionService) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");
        $controller("_LLMConnectionController", {$scope});

        if ($scope.creation) {
            $scope.connection.params.allowFinetuning = false;

            /* AWS - Completion */
            $scope.connection.params.allowAWSNovaPro = true;
            $scope.connection.params.allowAWSNovaLite = false;
            $scope.connection.params.allowAWSNovaMicro = false;
            $scope.connection.params.allowAWSTitanTextPremierV1 = false;
            $scope.connection.params.allowAWSTitanTextLiteV1 = false;
            $scope.connection.params.allowAWSTitanTextExpressV1 = false;
            $scope.connection.params.allowAWSTitanLarge = false;

            /* Anthropic */
            $scope.connection.params.allowAnthropicClaude4Sonnet = true;
            $scope.connection.params.allowAnthropicClaude4Opus = true;
            $scope.connection.params.allowAnthropicClaude37Sonnet = false;
            $scope.connection.params.allowAnthropicClaude35SonnetV2 = false;
            $scope.connection.params.allowAnthropicClaude35Sonnet = false;
            $scope.connection.params.allowAnthropicClaude3Sonnet = false;
            $scope.connection.params.allowAnthropicClaude35Haiku = true;
            $scope.connection.params.allowAnthropicClaude3Haiku = false;
            $scope.connection.params.allowAnthropicClaude3Opus = false;
            $scope.connection.params.allowAnthropicClaudeV2 = false;
            $scope.connection.params.allowAnthropicClaudeInstantV1 = false;

            /* AI21 */
            $scope.connection.params.allowAI21Jurassic2Ultra = false;
            $scope.connection.params.allowAI21Jurassic2Mid = false;

            /* Cohere - Completion */
            $scope.connection.params.allowCohereCommandRPlus = true;
            $scope.connection.params.allowCohereCommandR = false;

            /* Meta */
            $scope.connection.params.allowMetaLlama33_70BInstruct = true;
            $scope.connection.params.allowMetaLlama31_8BInstruct = false;
            $scope.connection.params.allowMetaLlama31_70BInstruct = true;
            $scope.connection.params.allowMetaLlama31_405BInstruct = false;
            $scope.connection.params.allowMetaLlama38BInstruct = false;
            $scope.connection.params.allowMetaLlama370BInstruct = false;

            /* Mistral */
            $scope.connection.params.allowMistral7BInstruct = false;
            $scope.connection.params.allowMixtral8X7BInstruct = false;
            $scope.connection.params.allowMistralSmall = false;
            $scope.connection.params.allowMistralLarge = false;
            $scope.connection.params.allowMistralLarge2 = true;

            /* AWS - Embedding */
            $scope.connection.params.allowAWSTitanEmbedTextV2 = true;
            $scope.connection.params.allowAWSTitanEmbedText = false;
            $scope.connection.params.allowAWSTitanMultimodalEmbedV1 = false;

            /* Cohere - Embedding */
            $scope.connection.params.allowCohereEmbedEnglish = false;
            $scope.connection.params.allowCohereEmbedMultilingual = false;

            /* AWS - Image generation */
            $scope.connection.params.allowAWSTitanImageGeneratorV1 = false;

            /* Stability AI */
            $scope.connection.params.allowStabilityAICore = false;
            $scope.connection.params.allowStabilityAIDiffusion3Large = true;
            $scope.connection.params.allowStabilityAIUltra = false;
            $scope.connection.params.allowStabilityAIDiffusion10 = false;

            /* DeepSeek */
            $scope.connection.params.allowDeepSeekR1 = false;

            $scope.connection.params.inferenceProfile = null;
            $scope.connection.params.maxParallelism = 8;
        }

        $scope.s3Connections = [];
        DataikuAPI.admin.connections.list().success(function(data) {
            $scope.s3Connections = Object.values(data).filter(c => c.type === 'EC2').map(c => c.name);
        });

        $scope.testConnection = function () {
            $scope.testing = true;
            DataikuAPI.admin.connections.testBedrock($scope.connection).success(function (data) {
                $scope.testResult = data;
            }).error(setErrorInScope.bind($scope))
                ["finally"](function () {
                $scope.testing = false;
            });
        };

        $scope.modelTypes = BedrockConnectionService.BedrockModelTypes;
        $scope.handlingModes = BedrockConnectionService.BedrockHandlingModes;
        $scope.TEXT_MODEL = 'TEXT_MODEL';
        $scope.MULTIMODAL_MODEL = 'MULTIMODAL_MODEL';

        $scope.customModelTypeChanged = function(customModel) {
            if (!customModel.modelType) return;

            customModel.handlingMode = undefined;
        };

        $scope.customModelHandlingChanged = function(customModel) {
            if (!customModel.handlingMode) return;

            if (['GENERIC_CONVERSE', 'AMAZON_NOVA'].includes(customModel.handlingMode)) {
                customModel.useConverseAPI = true;
            }
        };
    });

    app.controller("MosaicMLConnectionController", function ($scope, $controller, TopNav, DataikuAPI) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");
        $controller("_LLMConnectionController", {$scope});

        if ($scope.creation) {
            $scope.connection.params.allowMPT7BInstruct = true;
            $scope.connection.params.allowMPT30BInstruct = true;
            $scope.connection.params.allowLLAMA270BChat = true;
            $scope.connection.params.maxParallelism = 1;
        }

        $scope.testConnection = function () {
            $scope.testing = true;
            DataikuAPI.admin.connections.testMosaicML($scope.connection).success(function (data) {
                $scope.testResult = data;
            }).error(setErrorInScope.bind($scope))
                ["finally"](function () {
                $scope.testing = false;
            });
        }
    });

    app.controller("SageMakerGenericLLMConnectionController", function ($scope, $controller, TopNav, DataikuAPI) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");
        $controller("_LLMConnectionController", {$scope});

        // Keep in sync with CustomSageMakerModelType (dip/connections/SageMakerGenericLLMConnection/java)
        $scope.modelTypes =  [
            { rawValue: 'TEXT_COMPLETION', displayName: 'Text completion' },
            { rawValue: 'SUMMARIZATION', displayName: 'Summarization' },
            { rawValue: "TEXT_EMBEDDING", displayName: "Text embedding" }
    ];

        const modelHandlingModesCompletionSummarization = [
            ['AI21_J2', 'AI21 Jurassic 2'],
            ['AI21_SUMMARIZE', 'AI21 Summarize'],
            ['COHERE_COMMAND', 'Cohere Command'],
            ['HUGGING_FACE', 'Hugging Face'],
            ['META_LLAMA_2_SAGEMAKER', 'Meta Llama 2'],
            ['FULLY_CUSTOM', 'Fully Custom Handling']
        ];

        const modelHandlingModesTextEmbeddings = [
            ['COHERE_EMBED', 'Cohere Embed']
        ]

        $scope.modelHandlingModes = {
            'TEXT_COMPLETION': modelHandlingModesCompletionSummarization,
            'SUMMARIZATION':  modelHandlingModesCompletionSummarization,
            'TEXT_EMBEDDING': modelHandlingModesTextEmbeddings
        };

        $scope.useFullyCustomHandling = function() {
            if (!$scope.connection || !$scope.connection.params || !$scope.connection.params.sageMakerModel) return false;
            return $scope.connection.params.sageMakerModel.handling === 'FULLY_CUSTOM';
        }

        if ($scope.creation) {
            $scope.connection.params.maxParallelism = 8;
            $scope.connection.params.sageMakerModel= {
                friendlyNameShort: "Custom SageMaker LLM Endpoint",
                modelType: 'TEXT_COMPLETION',
                customHeaders: [],
                handling: null
            }
            $scope.connection.params.customQuery =
`{
    "prompt": __PROMPT__,
    "parameters": {
        "top_k": __TOPK__,
        "top_p": __TOPP__,
        "temperature": __TEMPERATURE__,
        "max_tokens": __MAX_TOKENS__,
        "frequency_penalty": __FREQUENCY_PENALTY__,
        "presence_penalty": __PRESENCE_PENALTY__,
        "logit_bias": __LOGIT_BIAS__
    }
}`;
            $scope.connection.params.responseJsonPath = "$.response";
        }

        $scope.$watch("connection.params.sageMakerModel.modelType", function(nv, ov) {
            if (ov && (nv !== ov)) {
                $scope.connection.params.sageMakerModel.handling = null;
            }
        });

        $scope.sageMakerConnections = [];
        DataikuAPI.admin.connections.list().success(function(data) {
            $scope.sageMakerConnections = Object.values(data).filter(c => c.type === 'SageMaker').map(c => c.name);
            if ($scope.sageMakerConnections.length == 1) {
                $scope.connection.params.sageMakerConnection = $scope.sageMakerConnections[0];
            }
        });

        $scope.testConnection = function () {
            $scope.testing = true;
            DataikuAPI.admin.connections.testSageMakerGenericLLM($scope.connection).success(function (data) {
                $scope.testResult = data;
            }).error(setErrorInScope.bind($scope))
                ["finally"](function () {
                $scope.testing = false;
            });
        }
    });

    // Keep in sync with VertexModelType (dip/connections/VertexAILLMConnection.java)
     app.constant("VertexModelTypes", [
        { rawValue: 'GEMINI_CHAT', displayName: 'Chat Completion' },
        { rawValue: 'TEXT_EMBEDDING_EXTRACTION', displayName: 'Embedding' },
        { rawValue: 'TEXT_IMAGE_EMBEDDING_EXTRACTION', displayName: 'Embedding multimodal' },
        { rawValue: 'IMAGE_GENERATION', displayName: 'Image Generation' }  // Can't really hide this under FF due to usage of app.constant :/
    ]);
    app.controller("VertexAILLMConnectionController", function ($scope, $controller, TopNav, DataikuAPI, VertexModelTypes) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");
        $controller("_LLMConnectionController", {$scope});

        if ($scope.creation) {
            $scope.connection.params.allowGeminiFlash25 = true;
            $scope.connection.params.allowGeminiPro25 = true;
            $scope.connection.params.allowGeminiFlash20 = false;
            $scope.connection.params.allowGeminiFlashLite20 = false;
            $scope.connection.params.allowGeminiFlash20ThinkingExp = false;
            $scope.connection.params.allowGeminiFlash15 = false;
            $scope.connection.params.allowGeminiPro15 = false;
            $scope.connection.params.allowGeminiFlash20Exp = false;

            
            $scope.connection.params.allowGeminiTextEmb = true;
            $scope.connection.params.allowTextEmb005 = true;
            $scope.connection.params.allowTextEmb = false;
            $scope.connection.params.allowTextMultilangEmb = true;
            $scope.connection.params.allowMultimodalEmb = true;

            $scope.connection.params.allowImagen3 = false;
            $scope.connection.params.allowImagen3Fast = false;

            $scope.connection.params.authType = "KEYPAIR";
            $scope.connection.params.dkuProperties = [];
            $scope.connection.params.maxParallelism = 8;
        }

        $scope.testConnection = function () {
            $scope.testing = true;
            DataikuAPI.admin.connections.testVertexAILLM($scope.connection).success(function (data) {
                $scope.testResult = data;
            }).error(setErrorInScope.bind($scope))
                ["finally"](function () {
                $scope.testing = false;
            });
        }

        $scope.$watch("connection.params", function (nv, ov) {
            // GCS doesn't support global Oauth yet
            if (nv.authType=="OAUTH") {
               $scope.connection.credentialsMode = "PER_USER";
            } else {
               $scope.connection.credentialsMode = "GLOBAL";
            }
        }, true);

        $scope.modelTypes = VertexModelTypes;

    });


 // Keep in sync with DatabricksLLMModelType (dip/connections/DatabricksLLMConnection.java)
    app.constant("DatabricksLLMModelTypes", [
        { rawValue: 'CHAT', displayName: 'Chat Completion' },
        { rawValue: 'TEXT_EMBEDDING', displayName: 'Embedding' }
    ]);
    app.controller("DatabricksLlmConnectionController", function ($scope, $controller, TopNav, DataikuAPI, DatabricksLLMModelTypes) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");
        $controller("_LLMConnectionController", {$scope});

        if ($scope.creation) {
            $scope.connection.params.allowClaude3_7_Sonnet= true;
            $scope.connection.params.allowLlama4_Maverick = false;
            $scope.connection.params.allowLlama3_3_70BChat = true;
            $scope.connection.params.allowLlama3_1_405BChat = false;
            $scope.connection.params.allowBGELargeEn = true;

            $scope.connection.params.maxParallelism = 1;
        }

        $scope.databricksModelDeplConnections = [];
        DataikuAPI.admin.connections.list().success(function(data) {
            $scope.databricksModelDeplConnections = Object.values(data).filter(c => c.type === 'DatabricksModelDeployment').map(c => c.name);
        }).error(setErrorInScope.bind($scope));

        $scope.testConnection = function () {
            $scope.testing = true;
            DataikuAPI.admin.connections.testDatabricksLLM($scope.connection).success(function (data) {
                $scope.testResult = data;
            }).error(setErrorInScope.bind($scope))
                ["finally"](function () {
                $scope.testing = false;
            });
        }

        $scope.modelTypes = DatabricksLLMModelTypes;
    });

// Keep in sync with VertexModelType (dip/connections/SnowflakeCortexLLMConnection.java)
    app.constant("SnowflakeCortexLLMModelTypes", [
        { rawValue: 'CHAT_COMPLETION', displayName: 'Chat Completion' },
        { rawValue: 'TEXT_EMBEDDING', displayName: 'Embedding' }
    ]);
    app.controller("SnowflakeCortexConnectionController", function ($scope, $controller, TopNav, DataikuAPI, SnowflakeCortexLLMModelTypes) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");
        $controller("_LLMConnectionController", {$scope});

        $scope.uiState = $scope.uiState || {};

        if ($scope.creation) {
            $scope.connection.params.allowLlama33_70BChat = true;
            $scope.connection.params.allowLlama31_70BChat = false;
            $scope.connection.params.allowLlama32_3BChat = true;
            $scope.connection.params.allowLlama2_70BChat = false;
            $scope.connection.params.allowMixtral8x7B = false;
            $scope.connection.params.allowMistral_7B = false;
            $scope.connection.params.allowMistral_Large2 = true;
            $scope.connection.params.allowMistral_Large = false;
            $scope.connection.params.allowGemma_7B = true;
            $scope.connection.params.allowSnowflakeArctic = true;
            $scope.connection.params.allowDeepSeekR1 = true;
            $scope.connection.params.allowClaude35Sonnet = true;
            $scope.connection.params.allowLlama4_Maverick = true;

            $scope.connection.params.allowSnowflakeArcticEmbedM = true;
            $scope.connection.params.allowE5BaseV2 = true;
            $scope.connection.params.allowNVEmbedQA4 = true;
            $scope.connection.params.maxParallelism = 8;
        }

        function computeSelectedConnection() {
            if ($scope.connection.params.snowflakeConnection) {
                $scope.uiState.selectedSnowflakeConnection = $scope.snowflakeConnections.find(sc => sc.name == $scope.connection.params.snowflakeConnection);
            } else {
                $scope.uiState.selectedSnowflakeConnection = null;
            }
        }

        $scope.snowflakeConnections = [];
        DataikuAPI.admin.connections.list().success(function(data) {
            $scope.snowflakeConnections = Object.values(data).filter(c => c.type === 'Snowflake');
            $scope.snowflakeConnectionNames = $scope.snowflakeConnections.map(c => c.name);
            computeSelectedConnection();
        }).error(setErrorInScope.bind($scope));

        $scope.$watch("connection.params.snowflakeConnection", () => {
            computeSelectedConnection();
        })

        $scope.testConnection = function () {
            $scope.testing = true;
            DataikuAPI.admin.connections.testSnowflakeCortex($scope.connection).success(function (data) {
                $scope.testResult = data;
            }).error(setErrorInScope.bind($scope))
                ["finally"](function () {
                $scope.testing = false;
            });
        }

        $scope.modelTypes = SnowflakeCortexLLMModelTypes;
    });

    app.service('HuggingFaceLocalConnectionService', function() {
        const GUARDED_MODEL_PRESETS = [
            "LLAMA_2_7B_CHAT",
            "LLAMA_2_13B_CHAT",
            "LLAMA_3_8B_INSTRUCT",
            "LLAMA_GUARD2",
            "LLAMA_GUARD3_1B",
            "LLAMA_GUARD3_8B",
            "PROMPT_GUARD",
            "MISTRAL_7B_INSTRUCT",
            "MISTRAL_7B_INSTRUCT_V2",
            "MISTRAL_7B_INSTRUCT_V3",
            "MISTRAL_NEMO_12B_INSTRUCT",
            "MIXTRAL_8X7B_INSTRUCT",
            "LLAMA_3_1_8B_INSTRUCT",
            "LLAMA_3_1_70B_INSTRUCT",
            "GEMMA_2B_INSTRUCT",
            "GEMMA_7B_INSTRUCT",
            "GEMMA_2_2B_INSTRUCT",
            "GEMMA_2_9B_INSTRUCT",
            "LLAMA_3_2_3B_INSTRUCT",
            "LLAMA_3_2_11B_VISION_INSTRUCT",
            "LLAMA_3_3_70B_INSTRUCT",
        ];

        this.isGuardedModel = (presetId) => {
            return GUARDED_MODEL_PRESETS.includes(presetId);
        }

        const HF_PURPOSES = {
            GENERIC_COMPLETION: {
                categoryName: "Text generation models",
                handlingModes: [
                    { rawValue: "TEXT_GENERATION_LLAMA_2", displayName: "Llama 2/3 model" },
                    { rawValue: "TEXT_GENERATION_DOLLY", displayName: "Dolly model" },
                    { rawValue: "TEXT_GENERATION_MISTRAL", displayName: "Mistral model" },
                    { rawValue: "TEXT_GENERATION_ZEPHYR", displayName: "Zephyr model" },
                    { rawValue: "TEXT_GENERATION_FALCON", displayName: "Falcon model" },
                    { rawValue: "TEXT_GENERATION_MPT", displayName: "MPT model" },
                    { rawValue: "TEXT_GENERATION_GEMMA", displayName: "Gemma model" },
                    { rawValue: "TEXT_GENERATION_PHI_3", displayName: "Phi-3 model" },
                ],
            },
            TEXT_EMBEDDING_EXTRACTION: {
                categoryName: "Text embedding models",
                handlingModes: [
                    { rawValue: "TEXT_EMBEDDING", displayName: "Text embedding" },
                ],
            },
            IMAGE_GENERATION: {
                categoryName: "Image generation models",
                handlingModes: [
                    { rawValue: "IMAGE_GENERATION_DIFFUSION", displayName: "Image generation model" },
                ],
            },

            IMAGE_EMBEDDING_EXTRACTION: {
                categoryName: "Image embedding models",
                handlingModes: [
                    { rawValue: "IMAGE_EMBEDDING", displayName: "Image embedding" },
                ],
            },
            TOXICITY_DETECTION: {
                categoryName: "Toxicity detection models",
                handlingModes: [
                    { rawValue: "TEXT_CLASSIFICATION_TOXICITY", displayName: "Toxicity detection (Bert models)" },
                    { rawValue: "TEXT_GENERATION_LLAMA_GUARD", displayName: "Toxicity detection (Llama Guard models)" },
                ],
            },
            PROMPT_INJECTION_DETECTION: {
                categoryName: "Prompt injection detection models",
                handlingModes: [
                    { rawValue: "TEXT_CLASSIFICATION_PROMPT_INJECTION", displayName: "Prompt injection" },
                ],
            },
            SENTIMENT_ANALYSIS: {
                categoryName: "Text classification models: Sentiment analysis",
                handlingModes: [
                    { rawValue: "TEXT_CLASSIFICATION_SENTIMENT", displayName: "Sentiment analysis" },
                ],
            },
            EMOTION_ANALYSIS: {
                categoryName: "Text classification models: Emotion analysis",
                handlingModes: [
                    { rawValue: "TEXT_CLASSIFICATION_EMOTIONS", displayName: "Emotion analysis" },
                ],
            },
            CLASSIFICATION_WITH_OTHER_MODEL_PROVIDED_CLASSES: {
                categoryName: "Text classification models: Other use cases",
                handlingModes: [
                    { rawValue: "TEXT_CLASSIFICATION_OTHER", displayName: "Text classification (other use cases)" },
                ],
            },
            CLASSIFICATION_WITH_USER_PROVIDED_CLASSES: {
                categoryName: "Text classification models: Custom classes (aka zero-shot)",
                handlingModes: [
                    { rawValue: "ZSC_GENERIC", displayName: "Zero-shot classification" },
                ],
            },
            SUMMARIZATION: {
                categoryName: "Text summarization models",
                handlingModes: [
                    { rawValue: "SUMMARIZATION_ROBERTA", displayName: "Summarization (Roberta models)" },
                    { rawValue: "SUMMARIZATION_GENERIC", displayName: "Summarization (generic)" },
                ],
            },
        };

        this.getPossibleHandlingModes = (purpose) => {
            return HF_PURPOSES[purpose]['handlingModes'];
        }

        this.getPurposes = () => {
            return HF_PURPOSES;
        }

        this.getNoPresetCustomModel = function(purpose) {
            const handlingMode = this.getPossibleHandlingModes(purpose)[0].rawValue;
            return { displayName: '', huggingFaceId: '', id: '', handlingMode: handlingMode, quantizationMode: 'NONE', enabled: true, containerSelection: {containerMode: 'INHERIT'} };
        }
    });

    app.controller("HuggingFaceLocalConnectionController", function ($scope, $state, $controller, $interval, TopNav, DataikuAPI, HuggingFaceLocalConnectionService) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");
        $controller("_LLMConnectionController", {$scope});

        $scope.notTestable = true;

        if ($scope.creation) {
            $scope.connection.params.useDSSModelCache = true;
            $scope.connection.params.enableReserveCapacity = true;
            $scope.connection.params.allowFinetuning = false;
            $scope.connection.params.models = [];
            $scope.connection.params.containerSelection = {containerMode: "NONE"};
        }

        DataikuAPI.admin.getGeneralSettings().then(function({data}) {
            $scope.globalMaxKernels = data.generativeAISettings.huggingFaceLocalSettings.maxConcurrentKernels
        }).catch(setErrorInScope.bind($scope));

        DataikuAPI.admin.clusters.listAccessible('KUBERNETES').success(function(data){
            $scope.k8sClusters = [{id:undefined, name:'Inherit instance default'}].concat(data);
        }).error(setErrorInScope.bind($scope));

        $scope.useInternalCodeEnvLabel = "Use internal code env";

        DataikuAPI.codeenvs.listNames('PYTHON').then(function({data}) {
            $scope.codeEnvItemsListWithDefault = [{"label": $scope.useInternalCodeEnvLabel, "value": undefined}].concat(data.map(codeEnv => ({"label": codeEnv, "value": codeEnv})));
        }).catch(setErrorInScope.bind($scope));

        let hfLocalInternalCodeEnvChecked = false;
        DataikuAPI.codeenvs.checkDSSInternalCodeEnv("HUGGINGFACE_LOCAL_CODE_ENV").then(function({data}) {
            if (Object.keys(data).length > 0) {
                $scope.hfLocalInternalCodeEnv = data.value;
                if ($scope.creation && $scope.hfLocalCodeEnv != null) {
                    $scope.connection.params.codeEnvName = $scope.hfLocalInternalCodeEnv.envName;
                }
            }
            hfLocalInternalCodeEnvChecked = true;
        }).catch(setErrorInScope.bind($scope));

        $scope.hfLocalInternalCodeEnvExists = function() {
            return hfLocalInternalCodeEnvChecked && $scope.hfLocalInternalCodeEnv != null;
        }

        $scope.hfLocalCodeEnvIsInternal = function () {
            return $scope.connection.params.codeEnvName == null || (
                $scope.hfLocalInternalCodeEnvExists() && $scope.connection.params.codeEnvName == $scope.hfLocalInternalCodeEnv.envName
            );
        }

        $scope.showHFLocalCodeEnvWarning = function () {
            return hfLocalInternalCodeEnvChecked && (
                (!$scope.hfLocalInternalCodeEnvExists()) || (!$scope.hfLocalCodeEnvIsInternal())
            );
        }

        $scope.internalCodeEnvsHRef = function() {
            if ($scope.appConfig.isAutomation) {
                return $state.href("admin.codeenvs-automation.internal");
            } else {
                return $state.href("admin.codeenvs-design.internal");
            }
        }

        $scope.fixupNullishForDirtynessCheck = function() {
            // if the connection.params.codeEnvName is unset in the backend, it will be initialized as non-set (~undefined) in the frontend
            // when selecting "Use internal code env", the code env selector forces this value to null
            // we need to fix this to avoid the dirtyness check failing due to null !== undefined
            if ($scope.connection.params.codeEnvName == null) {
                $scope.connection.params.codeEnvName = undefined;
            }
        }

        $scope.presetModels = null;
        $scope.facets = null;
        $scope.presets = null;
        $scope.facetFamilyDescriptions = [];
        DataikuAPI.admin.connections.listHuggingFacePresets().success(function(config) {
            $scope.presets = config.presets;
            $scope.presetModels = config.presets.map(p => p.model);
            // add default preset models for new connections
            if ($scope.creation) {
                for (const preset of config.presets) {
                    if (preset.includeInNewConnections) {
                        const newModel = _.cloneDeep(preset.model)
                        if (HuggingFaceLocalConnectionService.isGuardedModel(preset.id)) {
                            // apiKey is always blank initially on new connection creation
                            newModel.enabled = false;
                        }
                        $scope.connection.params.models.push(newModel);
                    }
                }
            }

            $scope.facets = config.facets;
        }).catch(setErrorInScope.bind($scope));

        // Used in "connection-name-test-sav.html"
        $scope.shouldShowExplicitUnsavedWarning = true;

        $scope.hasAtLeastOneReservedCapacity = function() {
            return $scope.connection.params.models.some(model => $scope.isValidHFModel(model) && model.enabled && model.minKernelCount > 0)
        };

        $scope.isValidHFModel = function(model) {
            return model.id && model.huggingFaceId && model.handlingMode
        };

        $scope.updateStatus = function() {
            if (!$scope.creation && $scope.connection.name) {
                DataikuAPI.admin.connections.getHfKernelStatus($scope.connection.name).success(function(status) {
                    $scope.kernelsStatus = status;
                }).catch(setErrorInScope.bind($scope));
            }
        }

        $scope.kernelsStatus = null;
        $scope.updateStatus(); // First call
        const cancelStatusUpdate = $interval(() => {
            $scope.updateStatus();
        }, 5000);

        $scope.$on('$destroy', () => {
            if (cancelStatusUpdate) {
                $interval.cancel(cancelStatusUpdate);
            }
        });

        checkChangesBeforeLeaving($scope, $scope.connectionDirty);
    });

    app.controller("HuggingFaceAddModelFromPresetController", function ($scope, Debounce) {
        $scope.filteredPresets = $scope.presets;
        $scope.selectedPreset = null;
        $scope.filterFacets = angular.copy($scope.facets);
        $scope.uiState = $scope.uiState || {};
        $scope.uiState.facetFilter = '';

        const facetFilterNames = {
            family: 'All families',
            mainUsagePurpose: 'All model types'
        };
        for (const [facetKey, facet] of Object.entries($scope.filterFacets)) {
            facet.values.unshift({name: facetFilterNames[facetKey], id: ''});
        }
        if ($scope.filterFacets || $scope.filterFacets['family']) {
            $scope.facetFamilyDescriptions = ($scope.filterFacets['family'].values || []).map(facet => facet.description);
        }

        $scope.filterPresets = () => {
            $scope.filteredPresets = $scope.presets.filter(preset => {
                for (const [facetKey, facetValue] of Object.entries($scope.selectedFacets)) {
                    if (facetValue !== '' && (preset.facets[facetKey] === undefined || !preset.facets[facetKey].includes(facetValue))) {
                        return false; // facet mismatch, filtering out
                    }
                }
                return preset.model && (preset.model.displayName || '').toLowerCase().includes($scope.uiState.facetFilter.toLowerCase());
            });
        }
        $scope.filterPresets();

        $scope.mainPresetModelPurpose = (preset) => {
            return $scope.filterFacets["mainUsagePurpose"].values.filter(v => preset.facets.mainUsagePurpose && v.id === preset.facets.mainUsagePurpose[0])[0];
        };

        $scope.selectPreset = (preset) => {
            $scope.selectedPreset = preset;
        };

        $scope.addModel = (preset) => {
            $scope.appendModel(preset.model, preset.facets['mainUsagePurpose'][0]);
            $scope.resolveModal();
        };

        $scope.resetFilters = () => {
            $scope.filteredPresets = $scope.presets;
            $scope.uiState.facetFilter = '';
            $scope.selectedFacets = {
                family: '',
                mainUsagePurpose: ''
            };
        };

        $scope.isFiltering = () => {
            return $scope.filteredPresets.length !== $scope.presets.length;
        }

        $scope.$watch('uiState.facetFilter', Debounce().withDelay(100, 200).wrap($scope.filterPresets));
    });

    app.component("hfModelsTable", {
        templateUrl: "/templates/admin/connection-huggingface-local-models-table.html",
        bindings: {
            models: '<',
            kernelsStatus: '<',
            presetModels: '<',
            apiKey: '<',
            supportsLlmFineTuning: '<',
            connectionName: '<',
            presets: '<',
            facets: '<',
            updateStatus: '<',
        },
        controller: function($scope, HuggingFaceLocalConnectionService, CreateModalFromTemplate, Dialogs, $sce) {
            const kernelsStatusWrapper = {kernelsStatus: this.kernelsStatus};

            this.purposes = HuggingFaceLocalConnectionService.getPurposes();

            this.$onChanges = () => {
                kernelsStatusWrapper.kernelsStatus = this.kernelsStatus;
            };

            this.isHandlingModeSupportedForPurpose = (purpose, handlingMode) => {
                const handlingModes = HuggingFaceLocalConnectionService.getPossibleHandlingModes(purpose);
                const supportedHandlingModes = handlingModes.map(mode => mode.rawValue);
                return supportedHandlingModes.includes(handlingMode);
            }

            this.addCustomModel = function(purpose) {
                const newModel = HuggingFaceLocalConnectionService.getNoPresetCustomModel(purpose);
                this.openEditModal(newModel, purpose, "Edit new model", "Add");
            };

            this.appendModel = (model, purpose) => {
                const clonedModel = _.cloneDeep(model);
                this.openEditModal(clonedModel, purpose, "Edit new model", "Add");
            };

            this.addModelFromPresets = function (usagePurpose = null) {
                const newScope = $scope.$new();
                newScope.presets = this.presets;
                newScope.facets = this.facets;
                newScope.appendModel = this.appendModel;

                newScope.selectedFacets = {
                    family: '',
                    mainUsagePurpose: usagePurpose
                };

                CreateModalFromTemplate("/templates/admin/connection-huggingface-local-add-model.html", newScope, 'HuggingFaceAddModelFromPresetController');
            };

            this.delete = (idx) => {
                const dialogScope = $scope.$new();
                const dialogModels = this.models;
                Dialogs.confirmAlert(dialogScope, "Delete model", "Are you sure you want to delete this model?").then(function() {
                    dialogModels.splice(idx, 1);
                }, function() {
                    // Dialog closed
                });
            };

            this._getCustomModelWarningMessages = (customModel, checkDuplicatedModelId) => {
                const warnings = [];
                if (checkDuplicatedModelId(customModel)) {
                    warnings.push("Duplicate model id.");
                }
                if (HuggingFaceLocalConnectionService.isGuardedModel(customModel.presetId) && !this.apiKey && customModel.enabled) {
                    warnings.push("This model is gated and cannot be used without an access token.");
                }
                if (warnings.length == 0) {
                    return null;
                }
                if (warnings.length == 1) {
                    return warnings[0];
                }
                return warnings.map(w => "• " + w).join("<br>");
            }

            this.getCustomModelWarningMessages = (customModel) => {
                return this._getCustomModelWarningMessages(customModel, (model) => {
                    const sameIdModels = this.models.filter(m => m.id == model.id);
                    return sameIdModels.length > 1;
                })
            };

            this.getHumanStatus = (modelId) => {
                if (this.kernelsStatus) {
                    const modelStatus = this.kernelsStatus['kernels'].filter(kernel => kernel.modelId === modelId);

                    if (modelStatus.some(kernel => kernel.state === "READY")) {
                        return $sce.trustAsHtml('<span style="color: green">Running</span>');
                    }

                    const lastDeadKernel = modelStatus.findLast(k => k.state === "DEAD");
                    if (lastDeadKernel && lastDeadKernel.deathReason && lastDeadKernel.deathReason.includes("FAIL")) {
                        return $sce.trustAsHtml('<span style="color: red">Error</span>');
                    }

                    if (modelStatus.some(kernel => kernel.state === "STARTING")) {
                        return $sce.trustAsHtml('<span style="color: orange">Starting</span>');
                    }

                    if (modelStatus.some(kernel => ["SENTENCED", "DYING"].includes(kernel.state))) {
                        return $sce.trustAsHtml('<span style="color: orange">Stopping</span>');
                    }

                    if (lastDeadKernel) {
                        return `Stopped ${moment.unix(lastDeadKernel.diedAtTime.seconds + lastDeadKernel.diedAtTime.nanos / 1e9).fromNow()}`;
                    }

                    return "&mdash;"
                }
            };

            this.getEnrichedLLMId = (model) => {
                return `huggingfacelocal:${this.connectionName}:${model.id}`
            }

            this.openStatusModal = (model) => {
                const newScope = $scope.$new();
                newScope.model = model;
                newScope.connectionName = this.connectionName;
                newScope.kernelsStatusWrapper = kernelsStatusWrapper;
                newScope.updateStatus = this.updateStatus;
                CreateModalFromTemplate("/templates/admin/connection-huggingface-local-custom-status-modal.html", newScope, "HuggingFaceModelStatusController");
            };

            this.openEditModal = (model, purpose, titleText = "Edit model", confirmText = "Ok") => {
                const newScope = $scope.$new();
                newScope.model = model;
                newScope.models = this.models;
                newScope.purpose = purpose;
                newScope.presetModels = this.presetModels;
                newScope.supportsLlmFineTuning = this.supportsLlmFineTuning;
                newScope.titleText = titleText;
                newScope.confirmText = confirmText;
                newScope._getCustomModelWarningMessages = this._getCustomModelWarningMessages;
                CreateModalFromTemplate("/templates/admin/connection-huggingface-local-edit-modal.html", newScope, "HuggingFaceModelEditController");
            }
        },
    });

    app.controller("HuggingFaceModelEditController", function($scope, HuggingFaceLocalConnectionService, Dialogs) {
        const NULL_PRESET_ID = "NONE";
        
        $scope.quantizationModes = [
            { rawValue: "NONE", displayName: "None (recommended)" },
            { rawValue: "Q_8BIT", displayName: "8 bit" },
            { rawValue: "Q_4BIT", displayName: "4 bit" },
        ];
        $scope.enforceEagerModes = [
            { rawValue: "AUTO", displayName: "Auto (recommended)" },
            { rawValue: true, displayName: "Enforce eager mode" },
            { rawValue: false, displayName: "Enable CUDA graph" },
        ];
        $scope.trustRemoteCodeModes = [
            { rawValue: "AUTO", displayName: "Auto (recommended)" },
            { rawValue: true, displayName: "Yes" },
            { rawValue: false, displayName: "No" },
        ];

        $scope.supportedHandlingModes = HuggingFaceLocalConnectionService.getPossibleHandlingModes($scope.purpose);
        const supportedHandlingModesRawValues = $scope.supportedHandlingModes.map(mode => mode.rawValue);
        const supportedPresetModels = _.cloneDeep(($scope.presetModels || []).filter(model => supportedHandlingModesRawValues.includes(model.handlingMode)));

        $scope.tempModel = _.cloneDeep($scope.model);

        function applyDefaultValuesForEmptyFields() {
            $scope.tempModel.enforceEager = $scope.tempModel.enforceEager ?? "AUTO";
            $scope.tempModel.trustRemoteCode = $scope.tempModel.trustRemoteCode ?? "AUTO";
            $scope.tempModel.presetId = $scope.tempModel.presetId ?? NULL_PRESET_ID;
        }
        applyDefaultValuesForEmptyFields();

        // deprecated preset ids are considered as no preset id
        const isDeprecatedPresetId = !supportedPresetModels.some(model => model.presetId === $scope.tempModel.presetId);
        const nonePresetId = isDeprecatedPresetId ? $scope.tempModel.presetId : NULL_PRESET_ID;
        $scope.presetIdOptions = [
            { presetId: nonePresetId, displayName: "None" },
            ...supportedPresetModels
        ];

        $scope.checkDuplicatedModelId = () => {
            const sameIdModels = $scope.models.filter(m => m.id == $scope.tempModel.id);

            // If we're editing a model (and not adding a new one), and the id is the same as the original model (we didn't change it),
            // we check if there is ANOTHER one with the same id. Otherwise, we check if there is 1 or more with the same id
            if ($scope.models.includes($scope.model) && $scope.model.id === $scope.tempModel.id) {
                return sameIdModels.length > 1;
            };

            return sameIdModels.length > 0;
        };

        $scope.getCustomModelWarningMessages = () => {
            return $scope._getCustomModelWarningMessages($scope.tempModel, $scope.checkDuplicatedModelId)
        };

        $scope.applyPreset = function() {
            const presetModel = supportedPresetModels.find(o => o.presetId === $scope.tempModel.presetId);
            const newModel = _.cloneDeep(presetModel ?? HuggingFaceLocalConnectionService.getNoPresetCustomModel($scope.purpose));
            // refresh the "None" presetId option which may have changed for models that used to have a deprecated presetId
            $scope.presetIdOptions = [
                { presetId: NULL_PRESET_ID, displayName: "None" },
                ...supportedPresetModels
            ];
            
            const settingsToPreserve = [
                "enabled",
                "containerSelection",
                "cudaVisibleDevices",
                "minKernelCount",
                "maxKernelCount"
            ];
            settingsToPreserve.forEach(k => newModel[k] = $scope.tempModel[k]);
            
            angular.copy(newModel, $scope.tempModel);
            applyDefaultValuesForEmptyFields();
        };

        $scope.hasInferenceSettings = function() {
            return ['GENERIC_COMPLETION', 'TEXT_EMBEDDING_EXTRACTION'].includes($scope.purpose) || $scope.tempModel.handlingMode === 'IMAGE_GENERATION_DIFFUSION';
        };

        $scope.resetModel = function() {
            const inferenceText = $scope.hasInferenceSettings() ? ' and inference' : '';
            Dialogs.confirm($scope, 'Reset model', `Are you sure you want to reset this model ? It will overwrite the model${inferenceText} settings.`).then(function () {
                $scope.applyPreset();
            }, function () {
                // Dialog closed
            });
        };

        $scope.fixupNullValuesForTextInput = function(attr) {
            $scope.tempModel[attr] = $scope.tempModel[attr] ?? '';
        };

        $scope.saveModel = function() {
            if (!$scope.models.includes($scope.model)) {
                $scope.models.push($scope.model);
            }
            if ($scope.tempModel.dtype === "") $scope.tempModel.dtype = undefined;
            if ($scope.tempModel.enforceEager === "AUTO") $scope.tempModel.enforceEager = undefined;
            if ($scope.tempModel.trustRemoteCode === "AUTO") $scope.tempModel.trustRemoteCode = undefined;
            if ($scope.tempModel.presetId === NULL_PRESET_ID) $scope.tempModel.presetId = undefined;

            // fixup nullish values for connection dirtyness check
            Object.keys($scope.tempModel).forEach(k => $scope.tempModel[k] = $scope.tempModel[k] ?? undefined);

            angular.copy($scope.tempModel, $scope.model);
        }
    });

    app.controller("HuggingFaceModelStatusController", function ($scope, DataikuAPI) {
        $scope.getStateColor = (kernel) => {
            switch (kernel.state) {
                case "READY": return "green";
                case "STARTING":
                case "SENTENCED":
                case "DYING": return "orange";
                case "DEAD": {
                    if (kernel.deathReason && kernel.deathReason.includes("FAIL")) {
                        return "red";
                    } else {
                        return "inherit";
                    }
                }
            }
        };

        $scope.getStateText = (kernel) => {
            let humanReadableDeathReason = null;
            let failedMessage = null;
            switch (kernel.deathReason) {
                // See KernelPool.java#DeathReason enum
                case "STRATEGY": {
                    humanReadableDeathReason = "because it isn't needed anymore";
                    break;
                }
                case "PER_KERNEL_MAX_LIMIT": {
                    humanReadableDeathReason = "to comply with the per-model max limit";
                    break;
                }
                case "GLOBAL_MAX_LIMIT": {
                    humanReadableDeathReason = "to comply with the global model instance limit";
                    break;
                }
                case "ROOM_FOR_RESERVED_CAPACITY": {
                    humanReadableDeathReason = "to make room for reserved capacity for another model instance";
                    break;
                }
                case "ROOM_FOR_REQUEST": {
                    humanReadableDeathReason = "to make room for a request that had no model instance running";
                    break;
                }
                case "OUTDATED": {
                    humanReadableDeathReason = "because model settings have changed";
                    break;
                }
                case "USER_REQUEST": {
                    humanReadableDeathReason = "because it was requested to stop by a user";
                    break;
                }
                case "DEBUG": {
                    humanReadableDeathReason = "for debugging reasons";
                    break;
                }
                case "FAIL_START":
                    failedMessage = "Model instance failed to start. Check the logs for more information.";
                    break;
                case "FAIL_RUNNING": {
                    failedMessage = "Model instance failed while it was running. Check the logs for more information.";
                    break;
                }
                case "UNKNOWN":
                default:
                    humanReadableDeathReason = "for an unknown reason";
            }

            switch (kernel.state) {
                case "STARTING": {
                    return `Model instance is starting (${moment.unix(kernel.startingAtTime.seconds + kernel.startingAtTime.nanos / 1e9).fromNow()})`;
                }
                case "READY": {
                    if (kernel.nbActiveRequests > 0) {
                        return `Processing ${kernel.nbActiveRequests} requests (started ${moment.unix(kernel.readyAtTime.seconds + kernel.readyAtTime.nanos / 1e9).fromNow()})`;
                    }
                    else {
                        return `Ready to accept requests (started ${moment.unix(kernel.readyAtTime.seconds + kernel.readyAtTime.nanos / 1e9).fromNow()})`;
                    }
                }
                case "SENTENCED": {
                    return (failedMessage ?? `Model instance is scheduled to shutdown ${humanReadableDeathReason}, after completing the ${kernel.nbActiveRequests} ongoing requests`) + ` (${moment.unix(kernel.sentencedAtTime.seconds + kernel.sentencedAtTime.nanos / 1e9).fromNow()})`;
                }
                case "DYING": {
                    return (failedMessage ?? `Model instance is shutting down ${humanReadableDeathReason}`)+ ` (${moment.unix(kernel.sentencedAtTime.seconds + kernel.sentencedAtTime.nanos / 1e9).fromNow()})`;
                }
                case "DEAD": {
                    return (failedMessage ?? `Model instance was stopped ${humanReadableDeathReason}`) + ` (${moment.unix(kernel.diedAtTime.seconds + kernel.diedAtTime.nanos / 1e9).fromNow()})`;
                };
                default:
                    return kernel.state;
            }
        };

        $scope.stoppedKernels = [];

        $scope.killKernel = (kernel) => {
            if ($scope.stopShouldBeDisabled(kernel)) return;

            DataikuAPI.admin.connections.killHFKernel(kernel.id).then(() => {
                $scope.updateStatus();
            });

            $scope.stoppedKernels.push(kernel.id);
        }

        $scope.stopShouldBeDisabled = (kernel) => {
            return $scope.stoppedKernels.includes(kernel.id) || (kernel.state != "STARTING" && kernel.state != "READY");
        }

        $scope.logs = {};
        this.fetchedStoppedKernelLogs = [];
        const updateLogs = () => {
            if ($scope.modelKernelsStatus) {
                $scope.modelKernelsStatus.forEach(kernel => {
                    if (kernel.state !== "DEAD" || !this.fetchedStoppedKernelLogs.includes(kernel.id)) {
                        DataikuAPI.admin.connections.getHfKernelLogs(kernel.id).success(function(data) {
                            $scope.logs[kernel.id] = data;
                        }).catch(setErrorInScope.bind($scope));

                        if (kernel.state === "DEAD") {
                            this.fetchedStoppedKernelLogs.push(kernel.id)
                        }
                    }
                });
            }
        }

        $scope.$watch('kernelsStatusWrapper.kernelsStatus', (status) => {
            $scope.modelKernelsStatus = status['kernels'].filter(kernel => kernel.modelId === $scope.model.id);
            $scope.runningKernelsStatus = $scope.modelKernelsStatus.filter(k => k.state !== "DEAD");
            $scope.deadKernelsStatus = $scope.modelKernelsStatus.filter(k => k.state === "DEAD").reverse();
            $scope.graveyardTimeout = status['graveyardTimeoutInS'];
            updateLogs();
        })
    });

    app.controller("HuggingFaceInferenceAPIConnectionController", function ($scope, $controller, TopNav, DataikuAPI) { // TODO @llm :implement HF API inference
        TopNav.setLocation(TopNav.DSS_HOME, "administration");
        $controller("_LLMConnectionController", {$scope});

        if ($scope.creation) {
            $scope.connection.params.maxParallelism = 2;
        }

        $scope.testConnection = function () {
            $scope.testing = true;
            DataikuAPI.admin.connections.testHuggingFace($scope.connection).success(function (data) {
                $scope.testResult = data;
            }).error(setErrorInScope.bind($scope))
                ["finally"](function () {
                $scope.testing = false;
            });
        }
    });

    app.controller("CustomLLMConnectionController", function ($scope, $controller, $rootScope, TopNav, DataikuAPI, PluginConfigUtils) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");
        $controller("_LLMConnectionController", {$scope});

        $scope._allCustomLLMPlugins = [...$scope.appConfig.customPythonLLMs, ...$scope.appConfig.customJavaLLMs];

        const idToType = {}
        $scope.appConfig.customPythonLLMs.forEach(llm => {
            idToType[llm.ownerPluginId] = "python"
        });
        $scope.appConfig.customJavaLLMs.forEach(llm => {
            idToType[llm.ownerPluginId] = "java"
        });

        $scope.customLLMPlugins = Array.from(new Set($scope._allCustomLLMPlugins.map(llm => llm.ownerPluginId)))
            .map(pluginID => $scope.appConfig.loadedPlugins.find(plugin => plugin.id === pluginID))
            .filter(plugin => plugin) // remove any `undefined` entries from plugins that could not be found
            .map(plugin => {
                plugin['type'] = idToType[plugin['id']] ?? "unknown";
                return plugin;
            })
            .sort((a,b) => a.label.localeCompare(b.label));

        $scope.selectedPlugin = $scope.customLLMPlugins.find(plugin => plugin.id === $scope.connection.params.pluginID);

        const cachedLLMDefinitions = {};

        $scope.$watch("selectedPlugin", function(nv, ov) {
            if (!nv) {
                $scope.connection.params.pluginID = "";
                $scope.pluginDesc = {};
                $scope.llmOptions = [];
            } else {
                $scope.connection.params.pluginID = $scope.selectedPlugin.id;
                $scope.pluginDesc = $scope.selectedPlugin;
                $scope.llmOptions = $scope._allCustomLLMPlugins
                                        .filter(llm => llm.ownerPluginId === nv.id)
                                        .sort((a,b) => a.desc.meta.label.localeCompare(b.desc.meta.label));
            }
            cachedLLMDefinitions[ov && ov.id] = $scope.connection.params.models;
            $scope.connection.params.models = cachedLLMDefinitions[nv && nv.id] || [];
            for (const model of $scope.connection.params.models) {
                $scope.onModelTypeChanged(model);
            }
        });

        let previousType;
        $scope.onModelTypeChanged = function(model) {
            model.$cachedCustomConfig = model.$cachedCustomConfig || {};
            model.$cachedCustomConfig[previousType || model.type] = angular.copy(model.customConfig);
            previousType = model.type

            if (model.$cachedCustomConfig[model.type]) {
                // Retrieve cached custom config for the newly selected type, if any
                model.customConfig = model.$cachedCustomConfig[model.type];
            }

            const loadedDesc = $scope.llmOptions.find(llm => llm.llmType == model.type);
            if (loadedDesc) {
                model.$desc = angular.copy(loadedDesc.desc);

                for (let paramName in model.customConfig) {  // Remove previous params that are not in the new model type
                    if (!model.$desc.params.some(x => x.name === paramName)) {
                        delete model.customConfig[paramName];
                    }
                }

                // This method sets default value for the relevant fields (based on the desc of the plugin) in model.customConfig, only if the fields are
                // already not defined
                PluginConfigUtils.setDefaultValues(model.$desc.params, model.customConfig);
            }
        };

        $scope.addModel = function() {
            DataikuAPI.humanId.create($scope.connection.params.models.map(model => model.id)).then(function({data}) {
                $scope.connection.params.models.push({
                    id: data.id,
                    capability: 'TEXT_COMPLETION',
                    type: undefined,
                    customConfig: {}
                });
            });
        };

        $scope.testConnection = function () {
            $scope.testing = true;
            DataikuAPI.admin.connections.testCustomLLM($scope.connection).success(function (data) {
                $scope.testResult = data;
            }).error(setErrorInScope.bind($scope))
                ["finally"](function () {
                $scope.testing = false;
            });
        }

        // Keep in sync with the backend Capability class (dip/connections/CustomLLMConnection.java)
        $scope.capabilities = [
            ['TEXT_COMPLETION', 'Chat completion'],
            ['TEXT_COMPLETION_MULTIMODAL', 'Chat completion (multimodal)'],
            ['TEXT_EMBEDDING', 'Text embedding'],
            ['IMAGE_GENERATION', 'Image generation'],
            ['TEXT_IMAGE_EMBEDDING_EXTRACTION', 'Multimodal embedding'],
        ];
    });

    app.controller("PineconeConnectionController", function ($scope, TopNav, DataikuAPI) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");

        if ($scope.creation) {
            $scope.connection.params["version"] = "POST_APRIL_2024";
        }

        $scope.testConnection = function () {
            $scope.testing = true;
            DataikuAPI.admin.connections.testPinecone($scope.connection).success(function (data) {
                $scope.testResult = data;
            }).error(
                setErrorInScope.bind($scope)
            ).finally(function () {
                $scope.testing = false;
            });
        };
    });

    app.controller("AzureAISearchConnectionController", function ($scope, TopNav, DataikuAPI) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");

        $scope.azureResourceURLFormat = "https://RESOURCE_NAME.search.windows.net"

        $scope.testConnection = function () {
            $scope.testing = true;
            DataikuAPI.admin.connections.testAzureAISearch($scope.connection).success(function (data) {
                $scope.testResult = data;
            }).error(
                setErrorInScope.bind($scope)
            ).finally(function () {
                $scope.testing = false;
            });
        };
    });

    app.controller("KafkaConnectionController", function ($scope, $controller, DataikuAPI, TopNav, CodeMirrorSettingService) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");
        
        $scope.codeMirrorSettingService = CodeMirrorSettingService;
        
        $scope.securityModes = [
                                        {id:'NONE', label:'No security protocol'},
                                        {id:'KERBEROS', label:'Kerberos'},
                                        {id:'SASL', label:'Generic Sasl'},
                                        {id:'CUSTOM', label:'Custom (using properties)'}
                                    ];

        $scope.testConnection = function () {
            $scope.testing = true;
            $scope.testResult = null;
            DataikuAPI.admin.connections.testKafka($scope.connection).success(function (data) {
                $scope.testResult = data;
            }).error(
                setErrorInScope.bind($scope)
            ).finally(function () {
                $scope.testing = false;
            });
        };
        
        $scope.testKsql = function() {
            $scope.testingKsql = true;
            $scope.testKsqlResult = null;
            DataikuAPI.admin.connections.testKsql($scope.connection).success(function (data) {
                $scope.testKsqlResult = data;
            }).error(
                setErrorInScope.bind($scope)
            ).finally(function () {
                $scope.testingKsql = false;
            });
        };
    });

    app.controller("SQSConnectionController", function ($scope, DataikuAPI, TopNav) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");
        
        $scope.testConnection = function () {
            $scope.testing = true;
            $scope.testResult = null;
            DataikuAPI.admin.connections.testSQS($scope.connection).success(function (data) {
                $scope.testResult = data;
            }).error(
                setErrorInScope.bind($scope)
            ).finally(function () {
                $scope.testing = false;
            });
        }
    });


    app.controller("MongoDBConnectionController", function ($scope, $controller, DataikuAPI, TopNav) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");

        $scope.checkForHttpInHostUrl = (host) => host && (host.startsWith('http://') || host.startsWith('https://'));

        if ($scope.creation) {
            $scope.connection.params["useURI"] = false;
            $scope.connection.params["uri"] = "mongodb://HOST:27017/DB";
        }

        var sequenceId = 0;
        $scope.testConnection = function () {
            if (!$scope.connectionParamsForm || $scope.connectionParamsForm.$valid) {
                $scope.testing = true;
                $scope.testResult = null;
                DataikuAPI.admin.connections.testMongoDB($scope.connection, ++sequenceId).success(function (data) {
                    if (data.sequenceId != sequenceId) {
                        // Too late! Another call was triggered
                        return;
                    }
                    $scope.testing = false;
                    $scope.testResult = data;
                }).error(setErrorInScope.bind($scope));
            }
        };

        // TODO - test on arrival - connection form not valid soon enough ???
    });

    app.controller("DynamoDBConnectionController", function ($scope, $controller, DataikuAPI, TopNav) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");

        if ($scope.creation) {
            $scope.connection.params["regionOrEndpoint"] = "eu-west-3";
            $scope.connection.params["mode"] = "WEBSERVICE";
            $scope.connection.params["port"] = 8000;
            $scope.connection.params["hostname"] = "localhost";
            $scope.connection.params["rwCapacityMode"] = "ON_DEMAND";
            $scope.connection.params["readCapacity"] = 1;
            $scope.connection.params["writeCapacity"] = 1;
        }
        var sequenceId = 0;
        $scope.testConnection = function () {
            if (!$scope.connectionParamsForm || $scope.connectionParamsForm.$valid) {
                $scope.testing = true;
                $scope.testResult = null;
                DataikuAPI.admin.connections.testDynamoDB($scope.connection, ++sequenceId).success(function (data) {
                    if (data.sequenceId != sequenceId) {
                        // Too late! Another call was triggered
                        return;
                    }
                    $scope.testing = false;
                    $scope.testResult = data;
                }).error(setErrorInScope.bind($scope));
            }
        };
     });


    app.controller("CassandraConnectionController", function ($scope, $controller, DataikuAPI, TopNav) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");

        addDatasetUniquenessCheck($scope, DataikuAPI);

        $scope.checkForHttpInHostsUrl = (hosts) => hosts && hosts.split(',').some(host => host.startsWith('http://') || host.startsWith('https://'));

        $scope.testConnection = function () {
            if (!$scope.connectionParamsForm || $scope.connectionParamsForm.$valid) {
                $scope.testing = true;
                $scope.testResult = null;
                DataikuAPI.admin.connections.testCassandra($scope.connection).success(function (data) {
                    $scope.testing = false;
                    $scope.testResult = data;
                }).error(setErrorInScope.bind($scope));
            }
        };

        if (!$scope.connection.customBasicConnectionCredentialProviderParams) {
            $scope.connection.customBasicConnectionCredentialProviderParams = [];
        }

        // TODO - test on arrival
    });

    app.controller("FTPConnectionController", function ($scope, $controller, DataikuAPI, TopNav) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");

        if ($scope.creation) {
            $scope.connection.params.passive = true;
            $scope.connection.allowManagedDatasets = false;
        }
        $scope.connection.allowMirror = false;

        $scope.notTestable = true;

        if (!$scope.connection.params.dkuProperties) {
            $scope.connection.params.dkuProperties = [];
        }
    });

    app.controller("SSHConnectionController", function ($scope, $controller, DataikuAPI, TopNav) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");

        $scope.connection.allowMirror = false;
        $scope.notTestable = true;

        if ($scope.creation) {
            $scope.connection.allowManagedDatasets = false;
            $scope.connection.params["usePublicKey"] = false;
        }

        if (!$scope.connection.params.dkuProperties) {
            $scope.connection.params.dkuProperties = [];
        }
    });

    app.component('credentialsErrorHandler', {
        templateUrl: '/templates/admin/credentials-error-handler.html',
        bindings: {
            error: '<',
            short: '<?'
        },
        controller: function($scope, ActivityIndicator, CredentialDialogs, DataikuAPI) {
            const $ctrl = this;
            $scope.error = $ctrl.error;
            $scope.$watch('$ctrl.error', function(error) {
                $scope.short = $ctrl.short ? $ctrl.short : false;
                $scope.error = $ctrl.error;
                if(error && error.code && error.payload && (error.payload.connectionName || error.payload.pluginId)) {
                    DataikuAPI.profile
                        .listConnectionCredentials()
                        .then(({data: {credentials}}) => {
                            $scope.credential = credentials.find(credential =>
                                (error.payload.connectionName && error.payload.connectionName === credential.connection) ||
                                (error.payload.pluginId && credential.pluginCredentialRequestInfo &&
                                    error.payload.pluginId === credential.pluginCredentialRequestInfo.pluginId &&
                                    error.payload.paramSetId === credential.pluginCredentialRequestInfo.paramSetId &&
                                    error.payload.presetId === credential.pluginCredentialRequestInfo.presetId &&
                                    error.payload.paramName === credential.pluginCredentialRequestInfo.paramName
                                )
                            );
                            if($scope.credentials) {
                                $scope.plugin = $scope.credential.pluginCredentialRequestInfo;
                            }
                        });
                } else {
                    $scope.credential = null;
                }
            });

            $scope.needConnection = function() {
                return $scope.credential &&
                    ($scope.credential.type === 'OAUTH_REFRESH_TOKEN' ||
                    $scope.credential.type === 'AZURE_OAUTH_DEVICECODE');
            }

            $scope.enterCredential = function() {
                CredentialDialogs.enterCredential($scope, $scope.credential)
                    .then(function(redirect) {
                    if(!redirect) {
                        ActivityIndicator.success("Credential saved");
                        $scope.credentialEntered = true;
                    }
                });
            }

            $scope.isConnectionCredential = function() {
                return $ctrl.error && $ctrl.error.payload && $ctrl.error.payload.connectionName;
            }
        }
    });

}());

;
(function(){
'use strict';

var app = angular.module('dataiku.admin.security', []);

app.controller("AdminSecurityController", function(){
});


app.directive("projectGrantItem", function(){
    return {
        template : `
            <ul class="grant-matrix">
                <li><abbr ng-show="grant.item.readProjectContent" class="grant grant--readProjectContent" title="Read project content">RPC</abbr></li>
                <li><abbr ng-show="grant.item.writeProjectContent" class="grant grant--writeProjectContent" title="Write project content">WPC</abbr></li>
                <li><abbr ng-show="grant.item.shareToWorkspaces" class="grant grant--shareToWorkspaces" title="Publish to workspaces">PW</abbr></li>
                <li><abbr ng-show="grant.item.publishToDataCollections" class="grant grant--publishToDataCollections" title="Publish to Data Collections">PDC</abbr></li>
                <li><abbr ng-show="grant.item.readDashboards" class="grant grant--readDashboards" title="Read dashboard">RD</abbr></li>
                <li><abbr ng-show="grant.item.writeDashboards" class="grant grant--writeDashboards" title="Write dashboards">WD</abbr></li>
                <li ng-if="showLegacyPermissions"><abbr ng-show="grant.item.moderateDashboards" class="grant grant--moderateDashboards" title="Moderate dashboards">MD</abbr></li>
                <li ng-if="showLegacyPermissions"><abbr ng-show="grant.item.manageAdditionalDashboardUsers" class="grant grant--manageAdditionalDashboardUsers" title="Manage dashboard users">MDU</abbr></li>
                <li><abbr ng-show="grant.item.manageDashboardAuthorizations" class="grant grant--manageDashboardAuthorizations" title="Manage authorized objects">MAO</abbr></li>
                <li><abbr ng-show="grant.item.manageExposedElements" class="grant grant--manageExposedElements" title="Manage shared objects">MSO</abbr></li>
                <li><abbr ng-show="grant.item.runScenarios" class="grant grant--runScenarios" title="Run scenarios">RS</abbr></li>
                <li><abbr ng-show="grant.item.executeApp" class="grant grant--executeApp" title="Execute app">EA</abbr></li>
                <li><abbr ng-show="grant.item.editPermissions" class="grant grant--editPermissions" title="Edit permissions">EP</abbr></li>
                <li><abbr ng-show="grant.item.admin" class="grant grant--admin" title="Admin">A</abbr></li>
            </ul>
        `,
        scope : {
            grant : '=',
            showLegacyPermissions : '='
        }
    }
});


app.directive("authorizationMatrixTable", function(){
    return {
        scope : true,
        link : function($scope) {
            $scope.hover = {
                col : null
            }
        }
    }
});

/**
 * users selection modal for the authorization matrix
 */
app.controller("AdminSecurityAuthorizationMatrixUserController", function($scope) {
    // create a map because we need to save the index of the selected users in the authorization matrix not the whole object
    const loginIndexMap = {};
    $scope.authorizationMatrix.perUser.users.forEach((user, userIdx) => {
        loginIndexMap[user.login] = userIdx;
    });
    $scope.localSelectedUsers = $scope.selectedUsers.map((idx) => $scope.authorizationMatrix.perUser.users[idx]);

    $scope.deselectUser = function(userIndex) {
        // we use filter instead of splice so changes gets detected by dku-bs-select
        $scope.localSelectedUsers = $scope.localSelectedUsers.filter((_, idx) => idx !== userIndex);
    }

    $scope.save = function() {
        const localSelectedIndexes = $scope.localSelectedUsers.map((user) => loginIndexMap[user.login]);
        $scope.saveUsers(localSelectedIndexes, $scope.unselectedUsers);
        $scope.dismiss();
    }
});

/**
 * groups selection modal for the authorization matrix
 */
app.controller("AdminSecurityAuthorizationMatrixGroupController", function($scope) {
    // create a map because we need to save the index of the selected groups in the authorization matrix not the whole object
    const groupIndexMap = {};
    $scope.authorizationMatrix.perGroup.groups.forEach((group, groupIdx) => {
        groupIndexMap[group] = groupIdx;
    });
    $scope.localSelectedGroups = $scope.selectedGroups.map((idx) => $scope.authorizationMatrix.perGroup.groups[idx]);

    $scope.deselectGroup = function(groupIndex) {
        // we use filter instead of splice so changes gets detected by dku-bs-select
        $scope.localSelectedGroups = $scope.localSelectedGroups.filter((_, idx) => idx !== groupIndex);
    }

    $scope.save = function(group) {
        const localSelectedIndexes = $scope.localSelectedGroups.map((group) => groupIndexMap[group]);
        $scope.saveGroups(localSelectedIndexes, $scope.unselectedGroups);
        $scope.dismiss();
    }
});

app.factory("AdminSecurityAuthorizationMatrixExportService", () => {
    function isUserContext(context) {
        return context === "USERS";
    }

    function globalAuthorizationTitleKeyPair(context) {
        return [
            ["Global admin", "admin"],
            ["Manage user-defined meanings", "mayManageUDM"],
            ["Create projects", "mayCreateProjects"],
            ["Create workspaces", "mayCreateWorkspaces"],
            ["Share to workspaces", "mayShareToWorkspaces"],
            ["Create Data Collections", "mayCreateDataCollections"],
            ["Publish to Data Collections", "mayPublishToDataCollections"],
            [isUserContext(context) ? "Manage own code envs" : "Create code envs", "mayCreateCodeEnvs"],
            ["Manage all code envs", "mayManageCodeEnvs"],
            [isUserContext(context) ? "Manage own cluster" : "Create clusters", "mayCreateClusters"],
            ["Manage all clusters", "mayManageClusters"],
            [isUserContext(context) ? "Manage own Code Studio template" : "Create Code Studio templates", "mayCreateCodeStudioTemplates"],
            ["Manage all Code Studio templates", "mayManageCodeStudioTemplates"],
            ["Develop plugins", "mayDevelopPlugins"],
            ["Edit lib folders", "mayEditLibFolders"],
            ["Create user connections", "mayCreateAuthenticatedConnections"],
            ["Write unisolated code", "mayWriteUnsafeCode"],
            ["Write isolated code", "mayWriteSafeCode"],
            ["Create published API services", "mayCreatePublishedAPIServices"],
            ["Create published projects", "mayCreatePublishedProjects"],
            ["May write in root project folder", "mayWriteInRootProjectFolder"],
            ["May create active web content", "mayCreateActiveWebContent"],
            ["Create Data Collections", "mayCreateDataCollections"],
            ["Publish to Data Collections", "mayPublishToDataCollections"],
        ];
    }

    function projectAuthorizationTitleKeyPair() {
        return [
            ["Read project content", "readProjectContent"],
            ["Write project content", "writeProjectContent"],
            ["Run scenarios", "runScenarios"],
            ["Share to workspaces", "shareToWorkspaces"],
            ["Read dashboard", "readDashboards"],
            ["Write dashboards", "writeDashboards"],
            ["Moderate dashboards", "moderateDashboards"],
            ["Manage authorized objects", "manageDashboardAuthorizations"],
            ["Manage shared objects", "manageExposedElements"],
            ["Manage dashboard users", "manageAdditionalDashboardUsers"],
            ["Share to Data Collections", "publishToDataCollections"],
            ["Execute app", "executeApp"],
            ["Admin", "admin"],
        ];
    }

    function createHeader(context) {
        let headers = [
            { name: "Permission", type: "string" },
            { name: "Type", type: "string" },
            { name: "Name", type: "string" }
        ];
        if(isUserContext(context)) {
            headers.push(
                { name: "User", type: "string" },
                { name: "Login", type: "string" }
            );
        } else {
            headers.push({ name: "Group", type: "string" });
        }
        headers.push({ name: "Value", type: "string" });
        return headers;
    }

    function createInstanceAuthorization(element, index, rawAuthorizationMatrix, context) {
        const selector = isUserContext(context) ? "perUser" : "perGroup";
        return globalAuthorizationTitleKeyPair(context).map((authorization) => {
            let result = [
                authorization[0],
                "GLOBAL",
                null
            ];
            if(isUserContext(context)) {
                result.push(element.displayName, element.login);
            } else {
                result.push(element);
            }
            result.push(`${rawAuthorizationMatrix[selector][authorization[1]][index] ? 1 : 0}`);
            return result;
        });
    }

    function createProjectAuthorization(element, elementIndex, rawAuthorizationMatrix, context) {
        const selector = isUserContext(context) ? "perUser" : "perGroup";
        return rawAuthorizationMatrix[selector].projectsGrants.map((projectGrants) => {
            return projectAuthorizationTitleKeyPair().map((authorization) => {
                let result = [
                    authorization[0],
                    "PROJECT",
                    projectGrants.projectKey
                ];
                if(isUserContext(context)) {
                    result.push(element.displayName, element.login);
                } else {
                    result.push(element);
                }
                result.push(projectGrants.grants[elementIndex]
                    ? `${projectGrants.grants[elementIndex].item[authorization[1]] ? 1 : 0}`
                    : "0");
                return result;
            });
        });
    }

    function createElement(rawAuthorizationMatrix, context) {
        return function (element, index) {
            return [
                ...createInstanceAuthorization(element, index, rawAuthorizationMatrix, context),
                ...createProjectAuthorization(element, index, rawAuthorizationMatrix, context).flat(),
            ];
        };
    }

    function createData(rawAuthorizationMatrix, context) {
        return rawAuthorizationMatrix[isUserContext(context) ? "perUser" : "perGroup"][
            isUserContext(context) ? "users" : "groups"
        ].map(createElement(rawAuthorizationMatrix, context)).flat();
    }

    return function (rawAuthorizationMatrix, context) {
        return {
            name: `${isUserContext(context) ? 'users' : 'groups'}-matrix`,
            description: "The matrix",
            columns: createHeader(context),
            data: createData(rawAuthorizationMatrix, context),
        };
    };
});

app.controller("AdminSecurityAuthorizationMatrixController", function($scope, DataikuAPI, TopNav, CreateModalFromTemplate, ExportUtils, AdminSecurityAuthorizationMatrixExportService, $filter) {
    TopNav.setLocation(TopNav.DSS_HOME, "administration");
    $scope.formData = {};
    $scope.selectedUsers = [];
    $scope.selectedGroups = [];
    $scope.userPagination = {
        selectedPage: 0,
        pageSize: 50,
    }
    $scope.groupPagination = {
        selectedPage: 0,
        pageSize: 50,
    }
    $scope.uiState = {
        showPermissionsBy: 'USERS',
        query: '',
        showLegacyPermissions: false,
    }
    $scope.authorizationMatrix = {
        perUser: {
            projectsGrants: []
        },
        perGroup: {
            projectsGrants: [],
        }
    }

    function createRangeOfTenOrLessFromList(list){
        return Array.from({length:Math.min(list.length, 10)}, (_,k)=>k);
    }

    function isUserSelected(index){
        return $scope.selectedUsers.includes(index);
    }

    function isGroupSelected(index){
        return $scope.selectedGroups.includes(index);
    }

    $scope.saveUsers = function(selectedUsers, unselectedUsers){
        $scope.selectedUsers = selectedUsers;
        $scope.unselectedUsers = unselectedUsers;
    }

    $scope.showUserModal = function(){
        CreateModalFromTemplate("/templates/admin/security/authorization-matrix-user.html", $scope, "AdminSecurityAuthorizationMatrixUserController")
    }

    $scope.saveGroups = function(selectedGroups, unselectedGroups){
        $scope.selectedGroups = selectedGroups;
        $scope.unselectedGroups = unselectedGroups;
    }

    $scope.showGroupModal = function(){
        CreateModalFromTemplate("/templates/admin/security/authorization-matrix-group.html", $scope, "AdminSecurityAuthorizationMatrixGroupController")
    }

    DataikuAPI.security.getAuthorizationMatrix().success(function(data){
        $scope.authorizationMatrix = data;
        $scope.selectedUsers = createRangeOfTenOrLessFromList(data.perUser.users);
        $scope.selectedGroups = createRangeOfTenOrLessFromList(data.perGroup.groups);
        $scope.unselectedUsers = data.perUser.users.filter((_, index) => !isUserSelected(index));
        $scope.unselectedGroups = data.perGroup.groups.filter((_, index) => !isGroupSelected(index));
    }).error(setErrorInScope.bind($scope));

    $scope.onGroupPageSelect = function(event){
        $scope.groupPagination.selectedPage = event.pageIndex;
    }

    $scope.onUserPageSelect = function(event){
        $scope.userPagination.selectedPage = event.pageIndex;
    }

    function resetPaginationSelectedPageIfNeeded(filteredOutput, paginationSettings) {
        if (Math.ceil(filteredOutput.length / paginationSettings.pageSize) <= paginationSettings.selectedPage) {
            paginationSettings.selectedPage = 0;
        }
    }

    $scope.$watchCollection(() => [$scope.authorizationMatrix.perGroup.projectsGrants, $scope.uiState.query], ([projectsGrantsGroup, filterQuery]) => {
        $scope.filteredGroup = $filter('filter')(projectsGrantsGroup, filterQuery);
        resetPaginationSelectedPageIfNeeded($scope.filteredGroup, $scope.groupPagination);
    })

    $scope.$watchCollection(() => [$scope.authorizationMatrix.perUser.projectsGrants, $scope.uiState.query], ([projectsGrantsUser, filterQuery]) => {
        $scope.filteredUser = $filter('filter')(projectsGrantsUser, filterQuery);
        resetPaginationSelectedPageIfNeeded($scope.filteredUser, $scope.userPagination);
    })

    $scope.uiState = $scope.uiState || {};
    $scope.uiState.showPermissionsBy = $scope.uiState.showPermissionsBy || "USERS";

    $scope.export = function() {
        ExportUtils.exportUIData(
            $scope,
            AdminSecurityAuthorizationMatrixExportService($scope.authorizationMatrix, $scope.uiState.showPermissionsBy),
            `Export ${$scope.uiState.showPermissionsBy === 'USERS' ? 'users' : 'groups'} matrix`,
            { downloadOnly: true, hideAdvancedParameters:true }
        );
    };
});

// we can't create a component here due to the fact that <user-authorization> is not
// a valid html element
app.directive("authorizations", function () {
  return {
    restrict: "A",
    scope: { title: "@", selectedElements: "<", allElements: "<", key: "@" },
    template: `
            <th class="table-matrix__title table-matrix__title--grant">{{title}}</th>
            <td ng-repeat="elementIndex in selectedElements"
                ng-class="{'global-grant': allElements[key][elementIndex]}">
                <i class="icon-ok-sign" ng-if="allElements[key][elementIndex]" />
                <span class="sr-only" ng-if="allElements[key][elementIndex]">Enabled</span>
            </td>
    `,
  };
});

app.component('paginationInfo', {
    bindings: {
        selectedPage: '<',
        pageSize: '<',
        listSize: '<',
    },
    controller: function(){
        this.$onChanges = () => {
            this.firstVisibleElement = this.selectedPage * this.pageSize + 1;
            this.lastVisibleElement = Math.min((this.selectedPage + 1) * this.pageSize, this.listSize);
        }
    },
    template: `
        <span class="pagination__label">Projects &nbsp;{{$ctrl.firstVisibleElement}}-{{$ctrl.lastVisibleElement}} of {{$ctrl.listSize}}</span>
    `
});

app.component('discoverabilitySettings', {
    bindings: {
        objectType: '<',
        requestType: '<',
        requestDocumentation: '<',
        visibilitySettings: '<'
    },
    templateUrl: '/templates/admin/security/discoverability-settings-dropdowns.html'
});

app.controller("UsersController", function($scope, $filter, DataikuAPI, Dialogs, TopNav, Logger, CreateModalFromTemplate, ListFilter, WT1, FutureProgressModal) {
	TopNav.setLocation(TopNav.DSS_HOME, "administration");

    $scope.canFetchExternalUsers = false;
    DataikuAPI.admin.users.getFetchableSourceTypes().success(function(data) {
        $scope.canFetchExternalUsers = (data || []).length > 0;
    }).error(setErrorInScope.bind($scope));

    // filter and pagination of the user table
    $scope.selection = { orderQuery: "login", orderReversed: false };
    $scope.pagination = new ListFilter.Pagination([], 50);
    $scope.isOutOfTrials = false;

    function arrayToHtmlList(a) {
        return a.map(e => `<li>${sanitize(e)}</li>`).join('');
    }

    function formatUsers(users) {
        return users.map(u => `${u.login}: ${u.displayName}`);
    }

	function refreshList() {
	    const lastActivity = true;
        DataikuAPI.admin.users.list(lastActivity).success(function(data) {
            $scope.users = data;
            refreshPage();
        }).error(setErrorInScope.bind($scope));
    };

    function refreshPage() {
        $scope.pagination.updateAndGetSlice($scope.selection.filteredObjects);
    };

    $scope.switchToTrial = function (login) {
        const targetLoginH = md5(login.toLowerCase());

        function startUserTrial(selectedItem) {
            WT1.event("admin-user-trial-start", {
                "forLoginh": targetLoginH,
                "chosenUserProfile": selectedItem["title"]
            });

            DataikuAPI.admin.users.switchToTrial(login, selectedItem["title"]).success(function(data) {
                WT1.event("admin-user-trial-start-success", {
                    "forLoginh": targetLoginH,
                    "chosenUserProfile": selectedItem["title"]
                });
                refreshList();
            }).error(function(a, b, c) {
                setErrorInScope.bind($scope)(a, b, c);
                WT1.event("admin-user-trial-start-failed", {
                    "forLoginh": targetLoginH,
                    "chosenUserProfile": selectedItem["title"]
                });
            });
        }

        DataikuAPI.admin.getTrialStatus().success(function(data) {
            const trialLicenses = window.dkuAppConfig.licensing.userProfiles
                .filter(profile => profile !== "NONE")
                .map(profile => {
                    const profileTrialStatus = data.allocations.filter(elt => elt.userProfile === profile);
                    const remainingTokensForProfile = profileTrialStatus.length > 0 && profileTrialStatus[0].mode === "TOTAL_CREDIT" ?
                        profileTrialStatus[0].remainingCredit
                        : 0;
                    const selectable = remainingTokensForProfile > 0 || profileTrialStatus.length && profileTrialStatus[0].mode !== "TOTAL_CREDIT";
                    return ({
                        "title": profile,
                        "desc": "Start a trial as profile " + profile,
                        selectable,
                        unselectableReason: !selectable ? `No more trial tokens available for the "${profile}" profile. Please select another profile or reach out to Dataiku to request additional trial tokens.` : ''
                    });
                });

            const selectableTrialLicences = trialLicenses.filter(item => item.selectable);

            if (selectableTrialLicences.length === 0) {
                $scope.isOutOfTrials = true;
                return;
            }
            $scope.isOutOfTrials = false;

            Dialogs.select($scope, "Start trial for " + login,
                "Please select the user profile for which a trial will be started",
                trialLicenses,
                selectableTrialLicences[0] || null,
                {
                    alert: "Dataiku needs to store the user's login in order to grant the personal trial license. Additionally, Dataiku may contact the user during the trial, for onboarding purposes.",
                    alertLevel: "info"
                }
            )
                .then(startUserTrial);
        }).error(setErrorInScope.bind($scope));
    }


    function getConversionOptions(userProfile) {
        let ret = [
            {"title": "Remove access", "desc": "User profile will be changed to NONE"},
            {"title": "Convert to normal user", "desc": "User will become a normal user, with the user profile " + userProfile}
        ]
        if (window.dkuAppConfig && window.dkuAppConfig.licensing && window.dkuAppConfig.licensing.userProfiles &&
            window.dkuAppConfig.licensing.userProfiles.indexOf("AI_CONSUMER") >= 0) {
            ret.push({"title": "Convert to AI Consumer", "desc": "User will become an AI Consumer user"})
        } else {
            ret.push({"title": "Convert to Reader", "desc": "User will become a Reader user"})
        }
        return ret;
    }

    $scope.convertExpiredTrial = function(user) {
        const login = user.login;
        const targetLoginH = md5(login.toLowerCase());
        const items = getConversionOptions(user.userProfile)
        Dialogs.select($scope, "Convert expired trial for " + login,
            login + " had a trial, but it expired. What do you want to do next?",
            items, items[0]).then(function(selectedItem){

            let action = null;
            if (selectedItem === items[0]) {
                action = "SWITCH_TO_NONE";
            } else if (selectedItem === items[1]) {
                action = "SWITCH_TO_REGULAR";
            } else if (selectedItem === items[2]) {
                action = "SWITCH_TO_READER";
            }
            WT1.event("admin-user-trial-convert-expired", {"forLoginh": targetLoginH, "action": action});

            DataikuAPI.admin.users.convertFromTrial(login, action).success(function(data) {
                refreshList();
            }).error(setErrorInScope.bind($scope));
        });
    }

    $scope.convertActiveTrial = function(user) {
        const login = user.login;
        const targetLoginH = md5(login.toLowerCase());
        const items = getConversionOptions(user.userProfile);
        Dialogs.select($scope, "Convert trial for " + login,
            login + " is currently running a trial. What do you want to do next?",
            items, items[0]).then(function(selectedItem){

            let action = null;
            if (selectedItem === items[0]) {
                action = "SWITCH_TO_NONE";
            } else if (selectedItem === items[1]) {
                action = "SWITCH_TO_REGULAR";
            } else if (selectedItem === items[2]) {
                action = "SWITCH_TO_READER";
            }

            WT1.event("admin-user-trial-convert-not-expired", {"forLoginh": targetLoginH, "action": action});

            DataikuAPI.admin.users.convertFromTrial(login, action).success(function(data) {
                refreshList();
            }).error(setErrorInScope.bind($scope));
        });
    }

    $scope.canDoDisableEnableMassAction = function (selectedUsers, activate) {
        return selectedUsers.some(u => u.enabled !== activate);
    };

    $scope.activateDeactivateUsers = function (users, activate) {
        event.preventDefault();
        if (!$scope.canDoDisableEnableMassAction(users, activate)) {
            return;
        }
        users = users.filter(u => u.enabled !== activate);

        const title = `Confirm user${users.length > 1 ? 's' : ''} ${activate ? 'activation' : 'deactivation'}`;
        const loginsText = arrayToHtmlList(formatUsers(users));
        const text = `Are you sure you want to ${activate ? 'enable' : 'disable'} the following user${users.length > 1 ? 's' : ''}<ul>${loginsText}</ul>`;
        const logins = users.map(u => u.login);

        if (activate) {
            Dialogs.confirmPositive($scope, title, text).then(() => {
                DataikuAPI.admin.users.enableOrDisable(logins, true).success(() => {
                    refreshList();
                    logins.forEach((login) => {
                        WT1.event('user-enable', {"forLoginh": md5(login.toLowerCase())});
                    });
                }).error(setErrorInScope.bind($scope));
            });
        } else {
            DataikuAPI.admin.users.prepareDisable(logins).success(data => {
                Dialogs.confirmInfoMessages($scope, title, data, text, false).then(() => {
                    DataikuAPI.admin.users.enableOrDisable(logins, false).success(() => {
                        refreshList();
                        logins.forEach((login) => {
                            WT1.event('user-disable', {"forLoginh": md5(login.toLowerCase())});
                        });
                    }).error(setErrorInScope.bind($scope));
                });
            }).error(setErrorInScope.bind($scope));
        }
    };

    $scope.deleteUsers = function(selectedUsers) {
        const loginsText = arrayToHtmlList(formatUsers(selectedUsers));
        const text = `Are you sure you want to delete the following users<ul>${loginsText}</ul>`;

        const logins = selectedUsers.map(u => u.login);
        DataikuAPI.admin.users.prepareDelete(logins).success(function(data) {
            Dialogs.confirmInfoMessages($scope, 'Confirm users deletion', data, text, false).then(function() {
                DataikuAPI.admin.users.delete(logins).success(function(data) {
                    refreshList();
                    logins.forEach((login) => {
                        WT1.event('user-delete', {"forLoginh": md5(login.toLowerCase())});
                    });
                }).error(setErrorInScope.bind($scope));
            });
        }).error(setErrorInScope.bind($scope));
    };

    $scope.deleteUser = function(user) {
        DataikuAPI.admin.users.prepareDelete([user.login]).success(function(data) {
            Dialogs.confirmInfoMessages($scope, 'Confirm user deletion', data, 'Are you sure you want to delete user "'+sanitize(user.login) + '"?', false).then(function() {
                DataikuAPI.admin.users.delete([user.login]).success(function(data) {
                    refreshList();
                    WT1.event('user-delete', {"forLoginh": md5(user.login.toLowerCase())});
                }).error(setErrorInScope.bind($scope));
            });
        }).error(setErrorInScope.bind($scope));
    };

    $scope.openAssignUsersToGroupModal = function(users, groups) {
        CreateModalFromTemplate("/templates/admin/security/assign-users-groups-modal.html", $scope, null, function(newScope) {
            newScope.users = users;
            const newGroups = {};

            groups.forEach(group => {
                const empty = ! users.some(u => u.groups.includes(group));
                const full = users.every(u => u.groups.includes(group));
                newGroups[group] = {
                    name: group,
                    selected: full,
                    originallyAssigned: !empty,
                    indeterminate: !empty && !full
                };
            });

            newScope.groups = newGroups;

            let newAssignedGroups = {};
            newScope.assignGroup = function(group) {
                if (!group.selected && group.originallyAssigned) {
                    newAssignedGroups[group.name] = false;
                } else if (group.selected && !group.originallyAssigned) {
                    newAssignedGroups[group.name] = true;
                } else if (group.selected && group.indeterminate) {
                    newAssignedGroups[group.name] = true;
                } else {
                    delete newAssignedGroups[group.name];
                }
            };

            newScope.wereGroupsChanged = function() {
                return Object.keys(newAssignedGroups).length > 0;
            };


            newScope.assignGroups = function(users) {
                const groupsToAdd = Object.keys(newAssignedGroups).filter(k => newAssignedGroups[k]);
                const groupsToRemove = Object.keys(newAssignedGroups).filter(k => ! newAssignedGroups[k]);


                const loginsText = arrayToHtmlList(formatUsers(users));
                let text = `The following users <ul>${loginsText}</ul>`;
                if (groupsToAdd.length > 0) {
                    const groupsToAddText = arrayToHtmlList(groupsToAdd);
                    text += `will be added to groups <ul>${groupsToAddText}</ul>`;
                }
                if (groupsToRemove.length > 0) {
                    const groupsToRemoveText = arrayToHtmlList(groupsToRemove);
                    text += `will be removed from groups <ul>${groupsToRemoveText}</ul>`;
                }

                const logins = users.map(u => u.login);
                DataikuAPI.admin.users.prepareAssignUsersGroups(logins, groupsToAdd, groupsToRemove).success(function(data) {
                    Dialogs.confirmInfoMessages($scope, 'Confirm reassigning users to groups', data, text, false).then(function() {
                        newScope.dismiss();  // close first modal
                        Logger.info("Adding users", logins, "to group", groupsToAdd, " and removing from groups", groupsToRemove);
                        DataikuAPI.admin.users.assignUsersGroups(logins, groupsToAdd, groupsToRemove).success(function(data) {
                            refreshList();
                        }).error(setErrorInScope.bind($scope));
                    });
                }).error(setErrorInScope.bind($scope));
            };
        });
    };

    $scope.openAssignUsersToProfileModal = function(users) {
        CreateModalFromTemplate("/templates/admin/security/assign-users-profile-modal.html", $scope, null, function(newScope) {
            newScope.users = users;
            newScope.newProfile = null;
            newScope.assignProfile = function(users, newProfile) {
                const usersByProfile = new Map();
                const usersToUpdate = users.filter(u => u.userProfile !== newProfile);
                usersToUpdate.forEach((user) => {
                    if (user.userProfile !== newProfile) { // Skip users that already have the correct profile
                        const usersForProfile = usersByProfile.get(user.userProfile);
                        if (usersForProfile) {
                            usersForProfile.push(user);
                        } else {
                            usersByProfile.set(user.userProfile, [user]);
                        }
                    }
                });
                let text = "";
                usersByProfile.forEach((affectedUsers, profile) => {
                    const loginsText = arrayToHtmlList(formatUsers(affectedUsers));
                    const niceProfile = $filter('niceProfileName')(profile);
                    const niceNewProfile = $filter('niceProfileName')(newProfile);
                    text += `The following users will have their profile changed from <strong>${niceProfile}</strong> to <strong>${niceNewProfile}</strong>: <ul class="mbot8">${loginsText}</ul></p>`;
                });

                Dialogs.confirmInfoMessages($scope, 'Confirm changing profile for users', null, text, false).then(() => {
                    newScope.dismiss();  // close first modal
                    const logins = usersToUpdate.map(u => u.login);
                    DataikuAPI.admin.users.assignUsersProfile(logins, newProfile).success(() => {
                        refreshList();
                    }).error(setErrorInScope.bind($scope));
                });
            };
        });
    };

    // Used to change the page displayed when clicking on number or arrows
    $scope.$watch('pagination.page', () => {
        $scope.pagination.update();
    });

    // Filter/sorting is done by filtered-multi-select-rows, just refresh the pagination
    $scope.$watch('selection.filteredObjects', () => {
        refreshPage();
    });

    refreshList();
    DataikuAPI.security.listGroups(true).success(function(data) {
        if (data) {
            data.sort();
        }
        $scope.groups = data;
    }).error(setErrorInScope.bind($scope));

    // Sync all users that comes from external sources (LDAP, custom user supplier)
    $scope.syncAll = function() {
        DataikuAPI.admin.users.syncAll().success(function(data) {
            FutureProgressModal.show($scope, data, "Syncing users...", null, 'static', false, true).then(function(result) {
                Dialogs.infoMessagesDisplayOnly($scope, "Sync result", result, result.futureLog);
                refreshList();
            })
        }).error(setErrorInScope.bind($scope));
    }

    // The "sync all" action is possible only if at least one user source type is syncable
    $scope.canSyncAll = false;
    DataikuAPI.admin.users.getSyncableSourceTypes().success(function(data) {
        $scope.canSyncAll = data.length > 0;
    }).error(setErrorInScope.bind($scope));
});


app.controller("UserController", function($scope, $state, $stateParams, DataikuAPI, $route, TopNav, Dialogs, ActivityIndicator, WT1, FutureProgressModal) {
	TopNav.setLocation(TopNav.DSS_HOME, "administration");
    let savedUser;
    $scope.user = {
            groups: [],
            login:'',
            sourceType: 'LOCAL',
            displayName:'',
            userProfile : 'DATA_SCIENTIST',
            //codeAllowed : true,
            password:'',
            trialTokenStrategy: 'NO_ATTEMPT'
    };
    if ($scope.appConfig && $scope.appConfig.licensing && $scope.appConfig.licensing.userProfiles) {
        $scope.user.userProfile = $scope.appConfig.licensing.userProfiles[0];
    }

    if ($stateParams.login) {
        $scope.creation = false;
        fetchUser();
    } else {
        $scope.creation = true;
        DataikuAPI.security.listGroups(true).success(function(data) {
            if (data) {
                data.sort();
            }
            $scope.allGroups = data;
        }).error(setErrorInScope.bind($scope));
    }

    function fetchUser() {
        DataikuAPI.security.listGroups(true).success(function(data) {
            if (data) {
                data.sort();
            }
            $scope.allGroups = data;
            DataikuAPI.admin.users.get($stateParams.login).success(function(data) {
                $scope.user = data;
                savedUser = angular.copy(data);
            }).error(setErrorInScope.bind($scope));
        }).error(setErrorInScope.bind($scope));
    }

    $scope.prepareSaveUser = function() {
        if ($scope.creation) {
            $scope.saveUser();
        } else {
            DataikuAPI.admin.users.prepareUpdate($scope.user).success(function(data) {
                Dialogs.confirmInfoMessages($scope,
                    'Confirm user edition', data, 'Are you sure you want to edit user "'+sanitize($scope.user.login) + '"?', true
                ).then($scope.saveUser);
            }).error(setErrorInScope.bind($scope));
        }
    };

    const buildUserWT1Params = (user) => {
        const params = {
            "forLoginh": md5(user.login.toLowerCase()),
            "type": user.sourceType,
            "chosenUserProfile": user.userProfile
        };
        if (user.email && user.email != "") {
            params.forEmailh = md5(user.email.toLowerCase());
        }
        return params;
    };

    $scope.saveUser = function() {
        if ($scope.creation) {
            DataikuAPI.admin.users.create($scope.user).success(function(data) {
                WT1.event('user-create', buildUserWT1Params($scope.user));
                Dialogs.infoMessagesDisplayOnly($scope, "Warnings during user creation", data).then(function(){
                    $state.go("admin.security.users.list");
                });
            }).error(setErrorInScope.bind($scope));
        } else {
            DataikuAPI.admin.users.update($scope.user).success(function(data) {
                WT1.event('user-edit', buildUserWT1Params($scope.user));
                $state.go("admin.security.users.list");
            }).error(setErrorInScope.bind($scope));
        }
    };

    $scope.userIsDirty = function() {
        return !angular.equals(savedUser, $scope.user);
    };

    var getGeneralSettings = function() {
    	DataikuAPI.admin.getGeneralSettings().success(function(gs) {
            $scope.generalSettings = gs;
    	}).error(setErrorInScope.bind($scope));
    }

    $scope.$watch('user', function() {
        if (! $scope.user.adminProperties) {
            $scope.user.adminProperties = {};
        }
        if (! $scope.user.userProperties) {
            $scope.user.userProperties = {};
        }
    });
    // Init
    getGeneralSettings();

    // Sync the user with its source (LDAP, custom user supplier)
    $scope.syncUser = function() {
        DataikuAPI.admin.users.sync($stateParams.login).success(function(data) {
            FutureProgressModal.show($scope, data, "Syncing user...", null, 'static', false, true).then(function(result) {
                Dialogs.infoMessagesDisplayOnly($scope, "Sync result", result, result.futureLog);
                fetchUser();
            })
        }).error(setErrorInScope.bind($scope));
    };

    // Syncing is possible only for syncable source types and not in creation mode
    if (!$scope.creation) {
        DataikuAPI.admin.users.getSyncableSourceTypes().success(function(data) {
            $scope.syncableSourceTypes = data;
        }).error(setErrorInScope.bind($scope));
    }
});


app.controller("GroupsController", function($scope, $state, $stateParams, DataikuAPI, $route, $modal, $q, Dialogs, TopNav, CreateModalFromTemplate) {
	TopNav.setLocation(TopNav.DSS_HOME, "administration");
	$scope.usersByGroup = {};
    // Populate UI
    var loadGroups = function() {
        DataikuAPI.security.listGroupsFull().success(function(groups) {
            DataikuAPI.security.listUsers().success(function(users) {
                for (const group of groups) {
                    group.userCount = 0;
                    $scope.usersByGroup[group.name] = [];
                    for (const userDesc of users) {
                        if (userDesc.groups.includes(group.name)) {
                            group.userCount++;
                            $scope.usersByGroup[group.name].push(userDesc);
                        }
                    }
                }
                $scope.groups = groups;
            }).error(setErrorInScope.bind($scope));
        }).error(setErrorInScope.bind($scope));
    };

    // Delete a group
    $scope.deleteGroup = function(group) {
        DataikuAPI.security.prepareDeleteGroup(group.name).success(function(data) {
            Dialogs.confirmInfoMessages($scope, 'Delete group', data, 'Are you sure you want to delete group "' + sanitize(group.name) + '" ?', false).then(function () {
                DataikuAPI.security.deleteGroup(group.name).success(function (data) {
                    loadGroups();
                }).error(setErrorInScope.bind($scope));
            });
        }).error(setErrorInScope.bind($scope));
    };

    $scope.showGroupMembers = (groupName) => {
        CreateModalFromTemplate("/templates/admin/security/group-members-modal.html", $scope, null, function (newScope) {
            newScope.groupName = groupName;
            newScope.usersInGroup = $scope.usersByGroup[groupName];

            newScope.matchesQuery = (object) => {
                if (!object) return false;
                if (!newScope.filterQuery) return true;
                if (object.login && object.login.toLowerCase().includes(newScope.filterQuery.toLowerCase())) {
                    return true;
                }
                if (object.displayName && object.displayName.toLowerCase().includes(newScope.filterQuery.toLowerCase())) {
                    return true;
                }
                return false;
            }
        });
    };

    // Init
    loadGroups();
});


app.controller("GroupController",function($scope, $state, $stateParams, DataikuAPI, $route, TopNav, Dialogs) {
    TopNav.setLocation(TopNav.DSS_HOME, "administration");
    let savedGroup;
    if ($stateParams.name) {
        $scope.creation = false;
        DataikuAPI.security.getGroup($stateParams.name).success(function(data) {
            $scope.group = data;
            savedGroup = angular.copy(data);
        }).error(setErrorInScope.bind($scope));
    } else {
        $scope.creation = true;
        $scope.group = {
            sourceType: 'LOCAL',
            mayWriteSafeCode : true,
            mayWriteInRootProjectFolder: true,
            mayCreateProjects: true,
            mayCreateActiveWebContent: true,
            mayShareToWorkspaces: true,
            mayCreateDataCollections: true,
            mayPublishToDataCollections: true,
        };
    }

    $scope.prepareSaveGroup = function() {
        if ($scope.creation) {
            $scope.saveGroup();
        } else {
            DataikuAPI.security.prepareUpdateGroup($scope.group).success(function(data) {
                Dialogs.confirmInfoMessages($scope, 'Confirm user edition', data, 'Are you sure you want to edit group "'+sanitize($scope.group.name) + '"?', true)
                    .then($scope.saveGroup);
            }).error(setErrorInScope.bind($scope));
        }
    };

    // Create or update a group
    $scope.saveGroup = function() {
        if ($scope.creation) {
            DataikuAPI.security.createGroup($scope.group).success(function(data) {
                $state.go("admin.security.groups.list");
            }).error(setErrorInScope.bind($scope));
        } else {
            DataikuAPI.security.updateGroup($scope.group).success(function(data) {
                $state.go("admin.security.groups.list");
            }).error(setErrorInScope.bind($scope));
        }
    };

    $scope.groupIsDirty = function() {
        return !angular.equals(savedGroup, $scope.group);
    };

    var getGeneralSettings = function() {
        DataikuAPI.admin.getGeneralSettings().success(function(gs) {
            $scope.generalSettings = gs;
        }).error(setErrorInScope.bind($scope));
    }

    // Force the required state of external group mappings
    $scope.$watch('[group.sourceType, group.ldapGroupNames, group.azureADGroupNames, group.ssoGroupNames, group.customGroupNames]', () => {
        const invalidGroupMappingRequiredState =
            !$scope.group ||
            ($scope.group.sourceType === 'LDAP' && ($scope.group.ldapGroupNames || []).length === 0) ||
            ($scope.group.sourceType === 'AZURE_AD' && ($scope.group.azureADGroupNames || []).length === 0) ||
            ($scope.group.sourceType === 'LOCAL_NO_AUTH' && ($scope.group.ssoGroupNames || []).length === 0) ||
            ($scope.group.sourceType === 'CUSTOM' && ($scope.group.customGroupNames || []).length === 0);
        $scope.groupDescriptionForm.$setValidity('ldapGroupMappingRequired', !invalidGroupMappingRequiredState);
    });

    // Init
    getGeneralSettings();
});

app.controller("ExternalUsersController", function($scope, DataikuAPI, Dialogs, ListFilter, FutureProgressModal, FutureWatcher) {
    // filter and pagination of the user table
    $scope.selection = { orderQuery: "userAttributes.login", orderReversed: false };
    $scope.pagination = new ListFilter.Pagination([], 50);

    $scope.fetchUsers = function() {
        let groupName = $scope.groupName ? $scope.groupName : $scope.groupNameSelect;
        DataikuAPI.admin.users.fetchUsers($scope.userSourceType, {login: $scope.login, email: $scope.email, groupName: groupName}).success(function(initialResponse) {
            FutureProgressModal.show($scope, initialResponse, "Fetching users...", null, 'static', false, true).then(function(result) {
                $scope.users = result.users;
                refreshPage();
                if(result.unusedFilters && result.unusedFilters.length > 0) {
                    Dialogs.ack($scope, "⚠️ Warning: Incompatible source type", "This source type did not use the following filters: " + result.unusedFilters.join(", "));
                }
            });
        }).error(setErrorInScope.bind($scope));
    };

    function fetchUsersNoModal() {
        let groupName = $scope.groupName ? $scope.groupName : $scope.groupNameSelect;
        DataikuAPI.admin.users.fetchUsers($scope.userSourceType, {login: $scope.login, groupName: groupName}).success(function(initialResponse) {
            FutureWatcher.watchJobId(initialResponse.jobId).success(function(data) {
                $scope.users = data.result.users;
                refreshPage();
            }).error(setErrorInScope.bind($scope));
        }).error(setErrorInScope.bind($scope));
    }

    $scope.provisionUsers = function(selectedUsers) {
        let usersToSync = selectedUsers.filter(u => u.status === 'NOT_PROVISIONED');
        if (usersToSync.length > 0) {
            DataikuAPI.admin.users.provisionUsers($scope.userSourceType, usersToSync).success(function(data) {
                FutureProgressModal.show($scope, data, "Provision users...", null, 'static', false, true).then(function(result) {
                    let title = "Provisioning result";
                    if (result.maxSeverity !== 'INFO') {
                        let summaryMessage = result.messages.find(m => m.code === 'INFO_SECURITY_SUPPLIER_PROVISIONING_SUMMARY');
                        if(summaryMessage) {
                            const index = result.messages.indexOf(summaryMessage);
                            result.messages.splice(index, 1);
                            title = summaryMessage.details;
                        }
                    }
                    Dialogs.infoMessagesDisplayOnly($scope, title, result, result.futureLog);
                    fetchUsersNoModal();
                })
            }).error(setErrorInScope.bind($scope));
        } else {
            let msg = `<div>All selected users are already provisioned</div>`;
            Dialogs.error($scope, `Failed to provision selected users.`, msg);
        }
    }

    $scope.syncUsers = function(selectedUsers) {
        let usersToSync = selectedUsers.filter(u => u.status === 'UNSYNCED').map(u => { return u.userAttributes.login});
        if (usersToSync.length > 0) {
            DataikuAPI.admin.users.syncUsers(usersToSync).success(function(data) {
                FutureProgressModal.show($scope, data, "Syncing users...", null, 'static', false, true).then(function(result) {
                    Dialogs.infoMessagesDisplayOnly($scope, "Sync result", result, result.futureLog);
                    fetchUsersNoModal();
                })
            }).error(setErrorInScope.bind($scope));
        } else {
            let msg = `<div>All selected users cannot be synced (not yet provisionned or already synced)</div>`;
            Dialogs.error($scope, `Failed to sync selected users.`, msg);
        }
    }

    $scope.fetchGroups = function() {
        $scope.enableGroups = true;
        $scope.enableEmail = $scope.userSourceType == "AZURE_AD";
        if ($scope.userSourceType == "LDAP") {
            $scope.enableGroups = $scope.generalSettings.ldapSettings.enableGroups;
        }
        $scope.users = [];
        refreshPage();
        if ($scope.enableGroups) {
            DataikuAPI.admin.users.fetchGroups($scope.userSourceType).success(function(data) {
                FutureProgressModal.show($scope, data, "Fetching groups for filtering...", null, 'static', false, true).then(function(result) {
                    $scope.groups = result;
                })
            }).error(setErrorInScope.bind($scope));
        }
    }

    // Used to change the page displayed when clicking on number or arrows
    $scope.$watch('pagination.page', () => {
        $scope.pagination.update();
    });

    // Filter/sorting is done by filtered-multi-select-rows, just refresh the pagination
    $scope.$watch('selection.filteredObjects', () => {
        refreshPage();
    });

    function refreshPage() {
        $scope.pagination.updateAndGetSlice($scope.selection.filteredObjects);
    };

    var getGeneralSettings = function() {
        DataikuAPI.admin.getGeneralSettings().success(function(gs) {
            $scope.generalSettings = gs;
        }).error(setErrorInScope.bind($scope));
    }

    // Init
    getGeneralSettings();

    $scope.fetchableSourceTypes = [];
    DataikuAPI.admin.users.getFetchableSourceTypes().success(function(data) {
        $scope.fetchableSourceTypes = data;
    }).error(setErrorInScope.bind($scope));
});

app.directive("globalPermissionsEditor", function() {
    return {
        scope: {
            'permissions': '='
        },
        templateUrl: '/templates/admin/security/global-permissions-editor.html',
        link: function($scope) {
            $scope.$watch("permissions", function(nv, ov) {
                if (!nv) return;
                /* Handle implied permissions */
                nv.$mayCreateProjectsFromMacrosDisabled = false;
                nv.$mayCreateProjectsFromTemplatesDisabled = false;

                if (nv.mayCreateProjects || nv.admin) {
                    nv.$mayCreateProjectsFromMacrosDisabled = true;
                    nv.$mayCreateProjectsFromTemplatesDisabled = true;
                }
            }, true);
        }
    }
});

app.controller("AdminSecurityAuditBufferController",function($scope, $state, $stateParams, DataikuAPI, $route, TopNav, Dialogs) {
    TopNav.setLocation(TopNav.DSS_HOME, "administration");

    $scope.uiState = {
        includeAllCalls: false
    }

    $scope.refreshList = function(){
        DataikuAPI.security.getAuditBuffer($scope.uiState.includeAllCalls).success(function(data) {
            $scope.auditBuffer = data;
        }).error(setErrorInScope.bind($scope));
    }

    $scope.refreshList();
});

app.filter('auditBufferEventDetails', function(Logger) {
    return function(obj) {
        var sa = []
        Object.keys(obj).forEach(function(x) {
            if (x != "callPath" && x != "msgType" && x != "authSource" && x != "authUser"
                && x != "clientIP" && x != "originalIP") {
                    let v = obj[x];
                    if (typeof v === 'object'){
                        try {
                            v = JSON.stringify(v);
                        } catch (e) {
                            Logger.debug("could not stringify key: " + x);
                        }
                    }
                    sa.push(x + ": " + v);
            }
        });
        return sa.join(", ");
    };
});


app.controller("GlobalPublicAPIKeysController", function ($scope, $state, DataikuAPI, CreateModalFromTemplate, Dialogs, TopNav) {
    TopNav.setLocation(TopNav.DSS_HOME, "administration");
    $scope.refreshApiKeysList = function () {
        DataikuAPI.admin.publicApi.listGlobalKeys().success(function (data) {
            $scope.apiKeys = data;
        }).error(setErrorInScope.bind($scope));
    };

    $scope.refreshApiKeysList();

    $scope.deleteGlobalKey = function (keyId) {
        Dialogs.confirm($scope, "Remove API key", "Are you sure you want to remove this API key?").then(function () {
            DataikuAPI.admin.publicApi.deleteGlobalKey(keyId).success(function (data) {
                $scope.refreshApiKeysList();
            }).error(setErrorInScope.bind($scope));
        });
    };

    $scope.viewQRCode = function (key) {
        CreateModalFromTemplate("/templates/admin/security/api-key-qrcode-modal.html", $scope, null, function (newScope) {
            newScope.apiKeyQRCode = JSON.stringify({
                k : key.key,
                u : $scope.appConfig.dssExternalURL
            });
        });
    };

    $scope.isExpired = function (key) {
        return key.expiresOn != 0 && key.expiresOn < new Date().getTime();
    };

});


app.controller("EditGlobalPublicAPIKeyController", function ($scope, $state, DataikuAPI, TopNav, $stateParams, CreateModalFromTemplate, ClipboardUtils) {
    TopNav.setLocation(TopNav.DSS_HOME, "administration");
    if ($stateParams.id) {
        $scope.creation = false;
        DataikuAPI.admin.publicApi.getGlobalKey($stateParams.id).success(function(data) {
            $scope.apiKey = data;
        }).error(setErrorInScope.bind($scope));
    } else {
        var sampleProjectsPrivileges = {
            '__YOUR__PROJECTEY__' : {
                admin: false,
                readProjectContent: true,
                writeProjectContent: false,
                shareToWorkspaces: true,
                exportDatasetsData: true,
                readDashboards: true,
                writeDashboard: false,
                moderateDashboards: false,
                runScenarios: false,
                manageDashboardAuthorizations: true,
                manageExposedElements: false,
                manageAdditionalDashboardUsers: false,
                executeApp: false
            }
        };

        var sampleProjectFoldersPrivileges = {
            '__YOUR__PROJECT_FOLDER_ID__': {
                admin: false,
                writeContents: false,
                read: true
            }
        };

        $scope.creation = true;
        $scope.apiKey = {
            label : "New key",
            globalPermissions : {admin: true},
            projects : sampleProjectsPrivileges,
            projectFolders: sampleProjectFoldersPrivileges
        };
    }

    $scope.create = function () {
        DataikuAPI.admin.publicApi.createGlobalKey($scope.apiKey).success(function (data) {
            CreateModalFromTemplate("/templates/admin/security/new-api-key-modal.html", $scope, null, function(newScope) {
                newScope.hashedApiKeysEnabled = $scope.appConfig.hashedApiKeysEnabled;
                newScope.key = data;
                newScope.uiSref = "admin.security.globalapi.list";

                newScope.copyKeyToClipboard = function() {
                    ClipboardUtils.copyToClipboard(data.key, 'Copied to clipboard.');
                };

                newScope.viewQRCode = function() {
                    CreateModalFromTemplate("/templates/admin/security/api-key-qrcode-modal.html", $scope, null, function (newScope) {
                        newScope.apiKeyQRCode = JSON.stringify({
                            k : data.key,
                            u : $scope.appConfig.dssExternalURL
                        });
                    });
                };

                newScope.$on("$destroy",function() {
                    $state.go(newScope.uiSref);
                });
            });
        }).error(setErrorInScope.bind($scope));
    };

    $scope.save = function () {
        DataikuAPI.admin.publicApi.saveGlobalKey($scope.apiKey).success(function (data) {
            $state.go("admin.security.globalapi.list");
        }).error(setErrorInScope.bind($scope));
    };
});

app.controller("AdminPersonalPublicAPIKeysController", function ($scope, $state, translate, DataikuAPI, CreateModalFromTemplate, Dialogs, TopNav) {
    TopNav.setLocation(TopNav.DSS_HOME, "administration");
    $scope.refreshApiKeysList = function () {
        DataikuAPI.admin.publicApi.listPersonalKeys().success(function (data) {
            $scope.apiKeys = data;
        }).error(setErrorInScope.bind($scope));
    };

    $scope.isExpired = function (key) {
        return key.expiresOn != 0 && key.expiresOn < new Date().getTime();
    };

    $scope.refreshApiKeysList();

    $scope.editPersonalAPIKey = function (apiKey) {
        CreateModalFromTemplate("/templates/admin/security/personal-api-key-modal.html", $scope, null, function (newScope) {
            newScope.apiKey = {
                ...apiKey
            };
            newScope.creation = false;
        });
    };

    $scope.deletePersonalAPIKey = function (keyId) {
        Dialogs.confirm($scope,  translate("PROFILE.API_KEYS.REMOVE_DIALOG.TITLE", "Remove API key"), translate("PROFILE.API_KEYS.REMOVE_DIALOG.MESSAGE", "Are you sure you want to remove this API key?")).then(function () {
            DataikuAPI.admin.publicApi.deletePersonalKey(keyId).success(function (data) {
                $scope.refreshApiKeysList();
            }).error(setErrorInScope.bind($scope));
        });
    };

    $scope.viewQRCode = function (key) {
        CreateModalFromTemplate("/templates/admin/security/api-key-qrcode-modal.html", $scope, null, function (newScope) {
            newScope.apiKeyQRCode = JSON.stringify({
                k : key.key,
                u : $scope.appConfig.dssExternalURL
            });
        });
    };
});

app.controller("EditPersonalAPIKeyModalController", function($scope, DataikuAPI, CreateModalFromTemplate, ClipboardUtils, WT1) {
    $scope.create = function(){
        WT1.event("user-profile-create-API-key", {});
        DataikuAPI.profile.createPersonalAPIKey($scope.apiKey).success(function(data) {
            $scope.refreshApiKeysList();
            CreateModalFromTemplate("/templates/admin/security/new-api-key-modal.html", $scope, null, function(newScope) {
                newScope.hashedApiKeysEnabled = $scope.appConfig.hashedApiKeysEnabled;
                newScope.key = data;

                newScope.copyKeyToClipboard = function() {
                    ClipboardUtils.copyToClipboard(data.key, 'Copied to clipboard.');
                    $scope.dismiss();
                };

                newScope.viewQRCode = function() {
                    CreateModalFromTemplate("/templates/admin/security/api-key-qrcode-modal.html", $scope, null, function (newScope) {
                        newScope.apiKeyQRCode = JSON.stringify({
                            k : data.key,
                            u : $scope.appConfig.dssExternalURL
                        });
                    });
                };

                newScope.$on("$destroy", function(){
                    $scope.dismiss();
                });
            });
        }).error(setErrorInScope.bind($scope));
    };

    $scope.save = function(apiKey) {
        WT1.event("user-profile-edit-API-key", {});
        DataikuAPI.profile.editPersonalAPIKey(apiKey).success(function(data) {
            $scope.dismiss();
            $scope.refreshApiKeysList();
        }).error(setErrorInScope.bind($scope));
    };
});

})();

;
(function(){
'use strict';


var app = angular.module('dataiku.admin.maintenance', []);


app.constant("TAIL_STATUS", {
    DEBUG: 0,
    INFO: 1,
    WARNING: 2,
    ERROR: 3
});


app.controller("AdminScheduledTasksController", function($scope, $rootScope, $state, DataikuAPI, ActivityIndicator, TopNav){
	TopNav.setLocation(TopNav.DSS_HOME, "administration");
	$scope.refresh = function() {
        DataikuAPI.admin.scheduledTasks.getStatus().success(function(data){
            $scope.status = data;
        }).error(setErrorInScope.bind($scope));
    };

    $scope.fireTask = function(task){
        DataikuAPI.admin.scheduledTasks.fire(task.jobGroup, task.jobName).success(function(data){
            ActivityIndicator.success("Task fired");
            $scope.refresh();
        }).error(setErrorInScope.bind($scope));
    };

    $scope.refresh();
});


app.controller("AdminMaintenanceInfoController", function($scope, DataikuAPI, TopNav) {
    TopNav.setLocation(TopNav.DSS_HOME, "administration");

    DataikuAPI.admin.getInstanceInfo().success(function(data){
        $scope.data = data;
    });
});


app.controller("AdminProfilingController", function($scope, DataikuAPI, TopNav) {
    TopNav.setLocation(TopNav.DSS_HOME, "administration");

    $scope.uiState = {};

    $scope.loadProfilesList = function() {
        $scope.uiState.loadingProfilesList = true;
        DataikuAPI.admin.profiling.listProfiles().success(function(data) {
            $scope.uiState.loadingProfilesList = false;
            $scope.profiles = data;
        });
    };

    $scope.downloadProfile = function(profile) {
        downloadURL(DataikuAPI.admin.profiling.downloadProfile(profile.id));
    };

    $scope.deleteProfile = function(profile) {
        DataikuAPI.admin.profiling.deleteProfile(profile.id).success(function() {
            $scope.loadProfilesList();
        });
    };

    $scope.refreshProfilerSettings = function() {
        DataikuAPI.admin.profiling.getConfig().success(function(profilerSettings) {
            $scope.profilerSettings = profilerSettings;
            $scope.originalProfilerSettings = angular.copy(profilerSettings);
        }).error(setErrorInScope.bind($scope));
    }

    $scope.saveProfilerSettings = function() {
        return DataikuAPI.admin.profiling.saveConfig($scope.profilerSettings).success(function() {
            $scope.refreshProfilerSettings();
            $scope.loadProfilesList();
        });
    };

    $scope.isDirty = function() {
        return $scope.originalProfilerSettings 
            && $scope.profilerSettings
            && !angular.equals($scope.originalProfilerSettings, $scope.profilerSettings);
    };

    $scope.loadProfilesList();
    $scope.refreshProfilerSettings();
});


app.controller("AdminLogsController", function($scope, $state, $rootScope, $window, $timeout,
               Logs, Diagnostics, DataikuAPI, ActivityIndicator, TopNav, TAIL_STATUS) {
    TopNav.setLocation(TopNav.DSS_HOME, "administration");
    //save JS data to file
    var saveData = (function () {
        var a = document.createElement("a");
        a.style.display = "none";
        document.body.appendChild(a);
        return function (data, fileName) {
            var blob = new Blob([data], {type: "octet/stream"}),
                url = window.URL.createObjectURL(blob);
            a.href = url;
            a.download = fileName;
            a.click();

            //give Firefox time...
            setTimeout(function(){
                window.URL.revokeObjectURL(url);
            }, 1000);
        };
    }());

    $scope.uiState = {
        active:  'logs',
        dState: null
    };

    $scope.TAIL_STATUS = TAIL_STATUS;

    $scope.loadLogsList = function() {
        $scope.uiState.loadingLogsList = true;
        Logs.list().success(function(data) {
            $scope.uiState.loadingLogsList = false;
            $scope.logs = data;
        });
    };

    $scope.loadLog = function(log) {
        $scope.uiState.currentLog = log;
        $scope.uiState.loadingLog = log;
        Logs.cat(log.name).success(function(data) {
            $scope.uiState.loadingLog = null;
            $scope.logData = data;
            $scope.logDataHTML = smartLogTailToHTML(data.tail, false);
            $timeout(function(){
                var content = $('.log-container .scrollable')[0];
                content.scrollTop = content.scrollHeight;
            })
        });
    };

    $scope.reloadLog = function() {
        if ($scope.uiState.currentLog) {
            $scope.loadLog($scope.uiState.currentLog);
        }
    };

    $scope.downloadExtract = function() {
        var text = $scope.logData.tail.lines.join('\n');
        var filename = 'extractof_'+$scope.uiState.currentLog.name;
        saveData(text, filename);
    };

    $scope.downloadCurrentLogFile = function() {
        if ($scope.uiState.currentLog) {
            Logs.download($scope.uiState.currentLog.name);
        }
    };


    $scope.downloadBackendLog = function() {
        Logs.download("backend.log");
    };


    $scope.downloadAllLogFiles = function() {
        Logs.downloadAll();
    };
    $scope.loadLogsList();

});

app.controller("AdminDiagnosticsController", function($scope, $state, $rootScope, $window, $timeout,
               Logs, Diagnostics, DataikuAPI, ActivityIndicator, TopNav, TAIL_STATUS, FutureProgressModal) {

    $scope.now = new Date().getTime()

    $scope.options = {
        includeConfigDir: true,
        includeBackendStacks: true,
        includeDockerImagesListing: true,
        includeFullLogs: false,
        includeFullDataDirListing: true
    };

    $scope.getLatestDiagnosis = function () {
        Diagnostics.getLatest(function(data) {
            if (data.exists) {
                $scope.latestDiagnosis = data;
            }
        });
    }

    $scope.downloadLatestDiagnosis = function () {
        Diagnostics.downLoadLatest();
    }

    $scope.runDiagnosis = function() {
        DataikuAPI.admin.diagnostics.run($scope.options).success(function(data) {
            FutureProgressModal.show($scope, data, "Running diagnosis...", null, 'static', false, true).then(function(result) {
                if (result) {
                    $scope.downloadLatestDiagnosis();
                }
                $scope.getLatestDiagnosis();
                    
            })
        }).error(setErrorInScope.bind($scope));
    }

    $scope.getLatestDiagnosis();
});

app.controller("AdminSanityCheckController", function($scope, $sce, DataikuAPI, TopNav, FutureProgressModal, WT1) {
    TopNav.setLocation(TopNav.DSS_HOME, "administration");

    const getLevel = function(message) {
        if(message.isFatal) return 0;
        if(message.severity === 'ERROR') return 1;
        if(message.severity === 'WARNING') return 2;
        if(message.severity === 'INFO') return 3;
        return 4;
    }

    const sortMessages = function (a, b) {
        return getLevel(a) !== getLevel(b) ?
            getLevel(a) - getLevel(b) :
            a.title.localeCompare(b.title);
    };

    const sortCodes = function (a, b) {
        return a.label.localeCompare(b.label);
    };

    $scope.now = new Date().getTime()
    $scope.exclusions = [];

    DataikuAPI.admin.sanityCheck.getRunningJob().success(function(data) {
        FutureProgressModal.showPeekOnlyIfRunning($scope, data.jobId, "Running sanity check...").then(function(result) {
            DataikuAPI.admin.sanityCheck.getLatestRun().success(function(result) {
                result.analysisResult.messages.sort(sortMessages)
                $scope.latestSanityCheck = result;
            }).error(setErrorInScope.bind($scope));
        })
    }).error(setErrorInScope.bind($scope));

    $scope.getCodes = function(customOptionsCallback) {
        Promise.all([DataikuAPI.admin.sanityCheck.getCodes(), DataikuAPI.admin.sanityCheck.getExclusions()])
            .then(([{data: codes}, {data: exclusions}]) => {
                codes.filter(code => exclusions.includes(code.value)).forEach(code => code.selected = true);
                codes.sort(sortCodes);
                customOptionsCallback(codes);
            }).catch(setErrorInScope.bind($scope));
    };

    $scope.runSanityCheck = function() {
        WT1.event('instance-sanity-check-run', {exclusions: $scope.exclusions});
        DataikuAPI.admin.sanityCheck.run($scope.exclusions).success(function(data) {
            FutureProgressModal.show($scope, data, "Running sanity check...", null, 'static', false, true).then(function(result) {
                result.messages.sort(sortMessages)
                $scope.latestSanityCheck = {
                    analysisResult: result,
                    epochTimestamp: new Date().getTime()
                 };
            })
        }).error(setErrorInScope.bind($scope));
    }

    $scope.resultContainsMessages = function() {
        return _.get($scope, "latestSanityCheck.analysisResult.messages", []).length > 0;
    }

    $scope.hasResults = function() {
        return _.get($scope, "latestSanityCheck.epochTimestamp", 0) !== 0;
    }

    $scope.downloadLatestAnalysis = function() {
        let data = {};
        angular.copy($scope.latestSanityCheck, data);
        delete data.analysisResult["futureLog"];

        const file = new Blob([JSON.stringify(data)], {type: 'application/json'});
        let a = document.createElement('a');
        a.href = URL.createObjectURL(file);
        a.download = "sanity_check.json";
        a.click();
    }
});

})();
;
(function() {
'use strict';

const app = angular.module('dataiku.admin.codeenvs.common', ['dataiku.logs']);

app.directive('codeEnvSecurityPermissions', function(DataikuAPI, $rootScope, PermissionsService) {
    return {
        restrict : 'A',
        templateUrl : '/templates/admin/code-envs/common/security-permissions.html',
        replace : true,
        scope : {
                codeEnv  : '='
        },
        link : function($scope, element, attrs) {
            $scope.appConfig = $rootScope.appConfig;
            $scope.ui = {};

            function makeNewPerm(){
                $scope.newPerm = {
                    update: true,
                    delete: true,
                    use: true
                }
            }
            makeNewPerm();

            const fixupPermissions = function() {
                if (!$scope.codeEnv) return;
                /* Handle implied permissions */
                $scope.codeEnv.permissions.forEach(function(p) {
                    p.$updateDisabled = false;
                    p.$manageUsersDisabled = false;
                    p.$useDisabled = false;
                    if ($scope.codeEnv.usableByAll) {
                        p.$useDisabled = true;
                    }
                    if (p.update) {
                        p.$useDisabled = true;
                    }
                    if (p.manageUsers) {
                        p.$useDisabled = true;
                        p.$updateDisabled = true;
                    }
                });
            };
            DataikuAPI.security.listGroups(false).success(function(allGroups) {
                if (allGroups) {
                    allGroups.sort();
                }
                $scope.allGroups = allGroups;
                DataikuAPI.security.listUsers().success(function(data) {
                    $scope.allUsers = data.sort((a, b) => a.displayName.localeCompare(b.displayName));
                    $scope.allUsersLogin = data.map(user => '@' + user.login);
                }).error(setErrorInScope.bind($scope));
                $scope.unassignedGroups = PermissionsService.buildUnassignedGroups($scope.codeEnv, $scope.allGroups);
            }).error(setErrorInScope.bind($scope));

            $scope.$watch("codeEnv.owner", function() {
                $scope.ui.ownerLogin = $scope.codeEnv.owner;
            });
            $scope.addPermission = function() {
                $scope.codeEnv.permissions.push($scope.newPerm);
                makeNewPerm();
            };

            $scope.$watch("codeEnv.usableByAll", function(nv, ov) {
                fixupPermissions();
            })
            $scope.$watch("codeEnv.permissions", function(nv, ov) {
                if (!nv) return;
                $scope.unassignedGroups = PermissionsService.buildUnassignedGroups($scope.codeEnv, $scope.allGroups);
                fixupPermissions();
            }, true)
            $scope.$watch("codeEnv.permissions", function(nv, ov) {
                if (!nv) return;
                $scope.unassignedGroups = PermissionsService.buildUnassignedGroups($scope.codeEnv, $scope.allGroups);
                fixupPermissions();
            }, false)
            $scope.unassignedGroups = PermissionsService.buildUnassignedGroups($scope.codeEnv, $scope.allGroups);
            fixupPermissions();

            // Ownership mgmt
            $scope.$watch("ui.ownerLogin", function() {
                PermissionsService.transferOwnership($scope, ($scope.codeEnv || {}).desc, "code env");
            });
        }
    };
});

app.directive('codeEnvContainers', function (DataikuAPI, $rootScope) {
    return {
        restrict: 'A',
        templateUrl: '/templates/admin/code-envs/common/code-env-containers.html',
        replace: true,
        scope: {
            codeEnv: '=',
            envLang: '=',
            deploymentMode: '='
        },
        controller: function ($scope) {
            $scope.appConfig = $rootScope.appConfig;
            $scope.addLicInfo = $rootScope.addLicInfo;

            DataikuAPI.codeenvs.getUserAccessibleCodeEnvSettings().then(({data}) => {
                $scope.enableCodeEnvResources = data.enableCodeEnvResources;
            });

            let _mode = "NONE";
            if ($scope.codeEnv.allContainerConfs) {
                _mode = "ALL"
            } else if (!$scope.codeEnv.allContainerConfs && $scope.codeEnv.containerConfs.length !== 0) {
                _mode = "ALLOWED";
            }
            let _sparkMode = "NONE";
            if ($scope.codeEnv.allSparkKubernetesConfs) {
                _sparkMode = "ALL";
            } else if (!$scope.codeEnv.allSparkKubernetesConfs && $scope.codeEnv.sparkKubernetesConfs.length !== 0) {
                _sparkMode = "ALLOWED";
            }
            let _codeStudioMode = $scope.codeEnv.rebuildDependentCodeStudioTemplates;

            $scope.containerSelection = function (newMode) {
                if (!arguments.length) {
                    return _mode;
                }

                _mode = newMode;

                switch (newMode) {
                    case "NONE":
                        $scope.codeEnv.allContainerConfs = false;
                        $scope.codeEnv.containerConfs = [];
                        break;
                    case "ALLOWED":
                        $scope.codeEnv.allContainerConfs = false;
                        break;
                    case "ALL":
                        $scope.codeEnv.allContainerConfs = true;
                        break;
                }
            };
            $scope.sparkKubernetesSelection = function (newMode) {
                if (!arguments.length) {
                    return _sparkMode;
                }

                _sparkMode = newMode;

                switch (newMode) {
                    case "NONE":
                        $scope.codeEnv.allSparkKubernetesConfs = false;
                        $scope.codeEnv.sparkKubernetesConfs = [];
                        break;
                    case "ALLOWED":
                        $scope.codeEnv.allSparkKubernetesConfs = false;
                        break;
                    case "ALL":
                        $scope.codeEnv.allSparkKubernetesConfs = true;
                        break;
                }
            };

            $scope.codeStudioRebuildTemplateSelection = function (newMode) {
                if (!arguments.length) {
                    return _codeStudioMode;
                }

                _codeStudioMode = newMode;
                $scope.codeEnv.rebuildDependentCodeStudioTemplates = _codeStudioMode;
            };

            $scope.removeOutdatedContainerConfs = function() {
                $scope.codeEnv.containerConfs = $scope.codeEnv.containerConfs.filter(o => $scope.outdatedContainerConfs.indexOf(o) === -1);
            };

            $scope.removeOutdatedSparkKubernetesConfs = function() {
                $scope.codeEnv.sparkKubernetesConfs = $scope.codeEnv.sparkKubernetesConfs.filter(o => $scope.outdatedSparkKubernetesConfs.indexOf(o) === -1);
            };

            $scope.containsMismatchedPythonHooks = function() {
                if ($scope.codeEnv && $scope.codeEnv.desc && $scope.codeEnv.desc.predefinedContainerHooks && $scope.codeEnv.desc.pythonInterpreter) {
                    return $scope.codeEnv.desc.predefinedContainerHooks.some(hook =>
                        hook.type &&
                        hook.type !== $scope.codeEnv.desc.pythonInterpreter + "_SUPPORT" &&
                        /^PYTHON\d{2,3}_SUPPORT$/.test(hook.type));
                }

                return false;
            };

            DataikuAPI.containers.listNames(null, "USER_CODE")
                .success(data => {
                    $scope.containerNames = data;
                    $scope.outdatedContainerConfs = $scope.codeEnv.containerConfs.filter(o => $scope.containerNames.indexOf(o) === -1)

                    $scope.$watch("containerNames && codeEnv.containerConfs", function(nv, ov) {
                        $scope.outdatedContainerConfs = $scope.codeEnv.containerConfs.filter(o => $scope.containerNames.indexOf(o) === -1)
                    });
                })
                .error(setErrorInScope.bind($scope));
            DataikuAPI.containers.listSparkNames()
                .success(data => {
                    $scope.sparkKubernetesNames = data;
                    $scope.outdatedSparkKubernetesConfs = $scope.codeEnv.sparkKubernetesConfs.filter(o => $scope.sparkKubernetesNames.indexOf(o) === -1)

                    $scope.$watch("sparkKubernetesNames && codeEnv.sparkKubernetesConfs", function(nv, ov) {
                        $scope.outdatedSparkKubernetesConfs = $scope.codeEnv.sparkKubernetesConfs.filter(o => $scope.sparkKubernetesNames.indexOf(o) === -1)
                    });
                })
                .error(setErrorInScope.bind($scope));

            $scope.cacheLocations = [
                ['BEGINNING', 'Beginning'],
                ['AFTER_START_DOCKERFILE', 'After initial hooks'],
                ['AFTER_PACKAGES', 'After installation of packages'],
                ['AFTER_AFTER_PACKAGES_DOCKERFILE', 'After post-installation hooks'],
                ['END', 'At end'],
                ['NONE', 'None']
            ];

            $scope.runtimeAdditionTypes = [
                ['SYSTEM_LEVEL_CUDA_122_CUDNN_897', 'GPU support for Visual Machine Learning'],
                ['CUDA_SUPPORT_FOR_TORCH2_WITH_PYPI_NVIDIA_PACKAGES', 'GPU support for Torch 2'],
                ['BASIC_GPU_ENABLING', 'GPU support for Torch 1 (with +cuXXX variant)'],
                ['PYTHON36_SUPPORT', 'Install Python 3.6'],
                ['PYTHON37_SUPPORT', 'Install Python 3.7'],
                ['PYTHON38_SUPPORT', 'Install Python 3.8'],
                ['SYSTEM_LEVEL_CUDA_112_CUDNN_811', '[Deprecated] GPU support with CUDA 11.2'],
            ];

            $scope.runtimeAdditionTypesDescriptions = [
                'Visual Machine Learning, Visual Deep Learning and Visual Time Series Forecasting',
                'Prompt studios, LLM recipes & Knowledge banks (for local Hugging Face models)',
                'Object detection, image classification',
                'Add support for Python 3.6',
                'Add support for Python 3.7',
                'Add support for Python 3.8',
                '',
            ];
        }
    };
});

app.directive('codeEnvResources', ['DataikuAPI', 'Dialogs', 'CreateModalFromTemplate', function (DataikuAPI, Dialogs, CreateModalFromTemplate) {
    return {
        restrict: 'A',
        templateUrl: '/templates/admin/code-envs/common/code-env-resources.html',
        replace: true,
        scope: {
            codeEnv: '=',
            envName: '=',
            editorOptions: '=',
            nodeType: '@',
            canUpdateCodeEnv: '=',
            deploymentMode: '='
        },
        controller: function($scope) {
            $scope.showResourcesSize = false;
            $scope.resourcesCanSelectFn = () => true;
            $scope.resourcesBrowserPath = "/";

            $scope.registerExecuteResourcesBrowseFn = function(fnToRegister) {
                $scope.executeResourcesBrowse = fnToRegister;
            }

            $scope.getResourcesEnvVars = function() {
                switch ($scope.nodeType) {
                    case "DESIGN":
                        DataikuAPI.admin.codeenvs.design.getResourcesEnvVars("PYTHON", $scope.envName).then((response) => {
                            $scope.resourcesEnvironmentVariables = response.data.variables;
                        });
                        break;
                    case "AUTOMATION":
                        DataikuAPI.admin.codeenvs.automation.getResourcesEnvVars("PYTHON", $scope.envName, $scope.codeEnv.versionId).then((response) => {
                            $scope.resourcesEnvironmentVariables = response.data.variables;
                        });
                        break;
                }
            }

            $scope.$on('refreshCodeEnvResources', function (event, opt) {
                if(opt.envName === $scope.envName && (!opt.versionId || opt.versionId === $scope.codeEnv.versionId)){
                    $scope.getResourcesEnvVars();
                    $scope.executeResourcesBrowse($scope.resourcesBrowserPath);
                }
            });

            $scope.resourcesBrowseFn = function (relativePath) {
                switch ($scope.nodeType) {
                    case "DESIGN":
                        return DataikuAPI.admin.codeenvs.design.browseResources("PYTHON", $scope.envName, relativePath, $scope.showResourcesSize);
                    case "AUTOMATION":
                        return DataikuAPI.admin.codeenvs.automation.browseResources("PYTHON", $scope.envName, $scope.codeEnv.versionId, relativePath, $scope.showResourcesSize);
                }
            }

            $scope.computeSize = function () {
                switch ($scope.nodeType) {
                    case "DESIGN":
                        DataikuAPI.admin.codeenvs.design.browseResources("PYTHON", $scope.envName, "/", true).then((response) => {
                            $scope.resourcesSize = response.data.size;
                            $scope.showResourcesSize = true;
                            $scope.executeResourcesBrowse($scope.resourcesBrowserPath); // to refresh browser
                        });
                        break;
                    case "AUTOMATION":
                        DataikuAPI.admin.codeenvs.automation.browseResources("PYTHON", $scope.envName, $scope.codeEnv.versionId, "/", true).then((response) => {
                            $scope.resourcesSize = response.data.size;
                            $scope.showResourcesSize = true;
                            $scope.executeResourcesBrowse($scope.resourcesBrowserPath); // to refresh browser
                        });
                        break;
                }
            }

            $scope.clearResources = function () {
                Dialogs.confirm($scope, 'Clear resources', 'Are you sure you want to clear the resources?').then(function() {
                    switch ($scope.nodeType) {
                        case "DESIGN":
                            DataikuAPI.admin.codeenvs.design.clearResources("PYTHON", $scope.envName).then(() => {
                                $scope.showResourcesSize = false;
                                $scope.executeResourcesBrowse("/"); // to refresh browser
                            });
                            break;
                        case "AUTOMATION":
                            DataikuAPI.admin.codeenvs.automation.clearResources("PYTHON", $scope.envName, $scope.codeEnv.versionId).then(() => {
                                $scope.showResourcesSize = false;
                                $scope.executeResourcesBrowse("/"); // to refresh browser
                            });
                            break;
                    }
                });
            }

            $scope.openUploadResourcesModal = function(){
                CreateModalFromTemplate("/templates/admin/code-envs/common/upload-resources-modal.html", $scope, "AdminCodeEnvsUploadResourcesController")
            }

            $scope.canBeUpdated = function() {
                return $scope.canUpdateCodeEnv && ['DESIGN_MANAGED', 'PLUGIN_MANAGED', 'AUTOMATION_SINGLE', 'AUTOMATION_VERSIONED'].includes($scope.deploymentMode);
            };

            DataikuAPI.codeenvs.getUserAccessibleCodeEnvSettings().then(({data}) => {
                $scope.enableCodeEnvResources = data.enableCodeEnvResources;
            });

            $scope.getResourcesEnvVars();
        }
    };
}]);

app.controller("AdminCodeEnvsUploadResourcesController", ['$scope', 'Assert', 'DataikuAPI', function($scope, Assert, DataikuAPI) {
    $scope.resourcesToUpload = false;
    $scope.overwrite = false;

    $scope.upload = function() {
        Assert.trueish($scope.resourcesToUpload, "No resource to upload");

        const parentScope = $scope.$parent.$parent;
        switch (parentScope.nodeType) {
            case "DESIGN":
                DataikuAPI.admin.codeenvs.design.uploadResources("PYTHON", parentScope.envName, parentScope.resourcesBrowserPath, $scope.resourcesToUpload, $scope.overwrite).then(() => {
                    $scope.dismiss();
                }).catch(setErrorInScope.bind($scope)).finally(() => {
                    parentScope.executeResourcesBrowse(parentScope.resourcesBrowserPath); // to refresh browser
                });
                break;
            case "AUTOMATION":
                DataikuAPI.admin.codeenvs.automation.uploadResources("PYTHON", parentScope.envName, parentScope.codeEnv.versionId, parentScope.resourcesBrowserPath, $scope.resourcesToUpload, $scope.overwrite).then(() => {
                    $scope.dismiss();
                }).catch(setErrorInScope.bind($scope)).finally(() => {
                    parentScope.executeResourcesBrowse(parentScope.resourcesBrowserPath); // to refresh browser
                });
                break;
        }
    }
}]);


app.component('dssInternalCodeEnv', {
    templateUrl: "/templates/admin/code-envs/common/dss-internal-code-env.html",
    bindings: {
        internalCodeEnvType: '@',
        codeEnvDescription: '@'
    },
    controller: function($scope, DataikuAPI, FutureProgressModal, Dialogs, $state, $filter, $rootScope, CodeEnvService) {
        const $ctrl = this;

        $ctrl.appConfig = $rootScope.appConfig;
        $ctrl.mayCreateCodeEnvs = $rootScope.mayCreateCodeEnvs;
        $ctrl.supportedPythonInterpreters = [];
        $ctrl.codeEnvConfig = {
            pythonInterpreter: null
        };

        $ctrl.$onInit =  () => {
            $ctrl.checkAndShowCodeEnv();
            $ctrl.fetchSupportedPythonInterpreters();
            $ctrl.fetchDefaultAvailableInterpreter();
        };

        $ctrl.fetchSupportedPythonInterpreters = function() {
            const enrichInterpretersDescription = pythonInterpreters => DataikuAPI.codeenvs.getSupportedInterpreters($ctrl.internalCodeEnvType)
            .then(function({data}) {
                $ctrl.supportedPythonInterpreters = pythonInterpreters.filter(interpreter => data.includes(interpreter[0]))
            })
            .catch(setErrorInScope.bind($scope))
            CodeEnvService.getPythonInterpreters().then(enrichInterpretersDescription);
        }

        $ctrl.fetchDefaultAvailableInterpreter = function() {
            DataikuAPI.codeenvs.getDSSInternalDefaultAvailableInterpreter($ctrl.internalCodeEnvType)
            .then(function({data}) {
                $ctrl.codeEnvConfig.pythonInterpreter = data;
            })
            .catch(setErrorInScope.bind($scope))
        }

        $ctrl.checkAndShowCodeEnv = function () {
            DataikuAPI.codeenvs.checkDSSInternalCodeEnv($ctrl.internalCodeEnvType)
            .then(function({data}) {
                $ctrl.codeEnvExists = Object.keys(data).length > 0;
                if ($ctrl.codeEnvExists) {
                    $ctrl.codeEnv = data.value;
                    if ($ctrl.appConfig.isAutomation) {
                        $ctrl.codeEnvHref = $state.href("admin.codeenvs-automation.python-edit", { envName: $ctrl.codeEnv.envName });
                    } else {
                        $ctrl.codeEnvHref = $state.href("admin.codeenvs-design.python-edit", { envName: $ctrl.codeEnv.envName });
                    }

                    if ($ctrl.internalCodeEnvType === "PII_DETECTION_CODE_ENV") {
                        DataikuAPI.codeenvs.checkInternalCodeEnvUsedForPII().then(function({data}) {
                            $ctrl.showPiiDetectionCodeEnvWarning = !data;
                        }).catch(setErrorInScope.bind($scope));
                    }
                    if ($ctrl.internalCodeEnvType === "RAG_CODE_ENV") {
                        DataikuAPI.codeenvs.checkInternalCodeEnvUsedForRAG().then(function({data}) {
                            $ctrl.showRagCodeEnvWarning = !data;
                        }).catch(setErrorInScope.bind($scope));
                    }
                    if ($ctrl.internalCodeEnvType === "HUGGINGFACE_LOCAL_CODE_ENV") {
                        DataikuAPI.codeenvs.checkInternalCodeEnvUsedForHF().then(function({data}) {
                            $ctrl.hfLocalConnectionsWithWrongCodeEnv = data
                            .map(name => ({"name": name, "href": $state.href("admin.connections.edit", { connectionName: name })}));
                        }).catch(setErrorInScope.bind($scope));
                    }

                    $ctrl.outdated = $ctrl.codeEnv.outdatedItems.length > 0;
                    $ctrl.showOutdatedRequirementsWarning = $ctrl.codeEnv.outdatedItems.includes("REQUIREMENTS");
                    $ctrl.showOutdatedContainerImagesWarning = $ctrl.codeEnv.outdatedItems.includes("CONTAINER_IMAGES");
                }
            })
            .catch(setErrorInScope.bind($scope))
        };

        $ctrl.createCodeEnv = function () {
            DataikuAPI.codeenvs.createForDSSInternal($ctrl.internalCodeEnvType, $ctrl.codeEnvConfig.pythonInterpreter)
            .then(function(resp) {
                FutureProgressModal.show($scope, resp.data, $filter('capitalize')($ctrl.codeEnvDescription + " code environment"), undefined, 'static', false).then(function(result) {
                    if (result) {
                        Dialogs.infoMessagesDisplayOnly($scope, "Creation result", result.messages, result.futureLog, undefined, 'static', false);
                    }
                    $ctrl.checkAndShowCodeEnv();
                });
            })
            .catch(setErrorInScope.bind($scope));
        };

        $ctrl.updateCodeEnv = function () {
            DataikuAPI.codeenvs.updateForDSSInternal($ctrl.internalCodeEnvType)
            .then(function(resp) {
                FutureProgressModal.show($scope, resp.data, $filter('capitalize')($ctrl.codeEnvDescription + " code environment"), undefined, 'static', false).then(function(result) {
                    if (result) {
                        Dialogs.infoMessagesDisplayOnly($scope, "Update result", result.messages, result.futureLog, undefined, 'static', false);
                    }
                    $ctrl.checkAndShowCodeEnv();
                });
            })
            .catch(setErrorInScope.bind($scope));
        };
    }
});


app.component('codeEnvUsage', {
    templateUrl: '/templates/admin/code-envs/common/code-env-usage.html',
    bindings: {
        envLang: '@',
        envName: '@',
    },
    controller: function($scope, DataikuAPI, StateUtils, FullModelLikeIdUtils, Logger, $filter) {
        const $ctrl = this;
        
        //keys based on EnvUsage enum from CodeEnvModel.java
        const getUsageLink = {
            PROJECT: (_objectId, projectKey) => StateUtils.href.project(projectKey, {page: 'settings', selectedTab: 'code-envs'}),
            RECIPE: StateUtils.href.recipe,
            NOTEBOOK: StateUtils.href.jupyterNotebook,
            PLUGIN: StateUtils.href.plugin,
            SCENARIO: (objectId, projectKey) => StateUtils.href.scenario(objectId, projectKey, {tab: 'settings'}),
            SCENARIO_STEP: (objectId, projectKey) => StateUtils.href.scenario(objectId, projectKey, {tab: 'steps'}),
            SCENARIO_TRIGGER: (objectId, projectKey) => StateUtils.href.scenario(objectId, projectKey, {tab: 'settings'}),
            DATASET_METRIC: (objectId, projectKey) => StateUtils.href.dataset(objectId, projectKey, {tab: 'settings'}),
            DATASET_CHECK: (objectId, projectKey) => StateUtils.href.dataset(objectId, projectKey, {tab: 'settings'}),
            DATASET: (objectId, projectKey) => StateUtils.href.dataset(objectId, projectKey, {tab: 'settings'}),
            WEBAPP: (objectId, projectKey) => StateUtils.href.webapp(objectId, projectKey, {tab: 'edit'}),
            REPORT: (objectId, projectKey) => StateUtils.href.report(objectId, projectKey, {tab: 'edit'}),
            RETRIEVABLE_KNOWLEDGE: (objectId, projectKey) => StateUtils.href.retrievableKnowledge(objectId, projectKey, {tab: 'settings'}),
            API_SERVICE_ENDPOINT: StateUtils.href.lambdaService,
            SAVED_MODEL: (objectId, projectKey) => {
                try {
                    const obj = FullModelLikeIdUtils.parse(objectId);
                    if (obj.savedModelId) {
                        return StateUtils.href.savedModel(obj.savedModelId, projectKey);
                    }
                } catch(e) {
                    // do not generate link
                }
                Logger.error("code env usage - unsupported saved model id", objectId, projectKey);
                return null;
            },
            MODEL: (objectId, projectKey) => {
                try {
                    const obj = FullModelLikeIdUtils.parse(objectId);
                    if (obj.analysisId) {
                        return StateUtils.href.analysis(obj.analysisId, projectKey, {tab: 'ml.list'});
                    }
                } catch(e) {
                    if (objectId) {
                        return StateUtils.href.analysis(objectId, projectKey, {tab: 'ml.list'});
                    }
                }
                Logger.error("code env usage - unsupported model id", objectId, projectKey);
                return null;
            },
            CODE_STUDIO_TEMPLATE: StateUtils.href.codeStudioTemplate,
        };
        
        $ctrl.uiState = {
            filters: {
                query: '',
                resetable: false,
                types: [],
                projects: []
            },
            options: {
                types: [],
                projects: []
            },
            filteredUsages: [],
            filteredProjectsCount: 0
        };

        $ctrl.filterUsages = () => {
            if ($ctrl.accessibleUsages) {
                const qry = $ctrl.uiState.filters.query.trim().toLocaleLowerCase();
                const nbSelectedTypes = $ctrl.uiState.filters.types.length;
                const nbSelectedProjects = $ctrl.uiState.filters.projects.length;
                $ctrl.uiState.filteredUsages = $ctrl.accessibleUsages
                    //query filter
                    .filter(usage => !qry
                                  || usage.envUsage.toLocaleLowerCase().includes(qry)
                                  || usage.projectLabel.toLocaleLowerCase().includes(qry)
                                  || (usage.objectId && usage.objectId.toLocaleLowerCase().includes(qry)))
                    //type filter
                    .filter(usage => !nbSelectedTypes || $ctrl.uiState.filters.types.indexOf(usage.envUsage) >= 0)
                    //project filter
                    .filter(usage => !nbSelectedProjects || $ctrl.uiState.filters.projects.indexOf(usage.projectLabel) >= 0);
                $ctrl.uiState.filters.resetable = nbSelectedTypes > 0 || nbSelectedProjects > 0 || qry.length;
                $ctrl.uiState.filteredProjectsCount = $ctrl.uiState.filteredUsages.reduce(
                    (projects, usage) => {
                        projects.add(usage.projectKey)
                        return projects;
                    },
                    new Set()
                ).size;
            }
        };
        
        $ctrl.resetFilters = function(){
            $ctrl.uiState.filters.query = '';
            $ctrl.uiState.filters.types = [];
            $ctrl.uiState.filters.projects = [];
            $ctrl.filterUsages();
        };

        function makeOptions(items, field){
            //dedup item on fields, and generate a label with $itemCount
            return items
                .reduce((dedup, item) => {
                    const dedupedItem = dedup.find(dedupItem => dedupItem.id === item[field]);
                    if (dedupedItem) {
                        dedupedItem.count++;
                    } else {
                        dedup.push({
                            id: item[field],
                            count: 1
                        });
                    }
                    return dedup;
                }, [])
                .map(dedup => {
                    dedup.label = `${dedup.id} (${dedup.count})`;
                    return dedup
                });
        }

        $ctrl.getUsage = function () {
            DataikuAPI.admin.codeenvs.design
                .listUsages($ctrl.envLang, $ctrl.envName, false)
                .success(function(usagesList){
                    $ctrl.inaccessibleObjectsCount = usagesList.filter(usage => !usage.accessible).length;
                    $ctrl.accessibleUsages = usagesList
                        .filter(usage => usage.accessible)
                        .map(usage => {
                            usage.projectLabel = usage.projectKey === '__DKU_ANY_PROJECT__' ? 'All' : usage.projectKey;
                            usage.href = getUsageLink[usage.envUsage](usage.objectId, usage.projectKey);
                            if (usage.envUsage === 'PROJECT') {
                                usage.linkLabel = 'Go to project settings';
                            } else if (usage.envUsage === 'SAVED_MODEL') {
                                usage.linkLabel = 'Go to Saved models versions';
                            } else if (usage.envUsage === 'MODEL') {
                                usage.linkLabel = 'Go to Analysis page';
                            } else if (usage.href) {
                                usage.linkLabel = 'Go to ' + $filter('typeToName')(usage.envUsage);
                            }
                            return usage;
                        });
                    $ctrl.uiState.options.types = makeOptions($ctrl.accessibleUsages, 'envUsage');
                    $ctrl.uiState.options.projects = makeOptions($ctrl.accessibleUsages, 'projectLabel');
                    $ctrl.filterUsages();
                })
                .error(setErrorInScope.bind($scope));
        };
    }
});

}());

;
(function() {
'use strict';

var app = angular.module('dataiku.admin.codeenvs.design', []);

app.controller("AdminCodeEnvsDesignController", function($scope, $rootScope, TopNav, DataikuAPI, Dialogs, FutureProgressModal, CreateModalFromTemplate, ActivityIndicator, Deprecation, CodeEnvsDesignService) {
    $scope.canCreateCodeEnv = $rootScope.mayCreateCodeEnvs;

    $scope.openDeleteEnvModal = function(codeEnv){
        var newScope = $scope.$new();
        newScope.codeEnv = codeEnv;
        // modal appears when usages are ready
        CreateModalFromTemplate("/templates/admin/code-envs/common/delete-env-modal.html", newScope, "AdminCodeEnvsDesignDeleteController");
    }
    $scope.isExportable = function(codeEnv) {
        return codeEnv && ['PLUGIN_MANAGED', 'PLUGIN_NON_MANAGED', 'DSS_INTERNAL'].indexOf(codeEnv.deploymentMode) < 0;
    };
    $scope.exportEnv = function(envLang, envName) {
        ActivityIndicator.success("Exporting code env ...");
        downloadURL(DataikuAPI.admin.codeenvs.design.getExportURL(envLang, envName));
    };

    $scope.getEnvDiagnostic = function(envLang, envName) {
        ActivityIndicator.success("Generating code env diagnostic ...");
        downloadURL(DataikuAPI.admin.codeenvs.design.getDiagnosticURL(envLang, envName));
    };

    $scope.isPythonDeprecated = function (pythonInterpreter) {
        return Deprecation.isPythonDeprecated(pythonInterpreter);
    };

    $scope.openCRANEditHelp = function() {Dialogs.ackMarkdown($scope, "R packages", CodeEnvsDesignService.CRANEditHelpString);}
    $scope.openRCondaSpecEditHelp = function() {Dialogs.ackMarkdown($scope, "Conda packages", CodeEnvsDesignService.rCondaSpecEditHelpString);}
    $scope.openPipRequirementsEditHelp = function() {Dialogs.ackMarkdown($scope, "Pip requirements", CodeEnvsDesignService.pipRequirementsEditHelpString);}
    $scope.openPyCondaSpecEditHelp = function() {Dialogs.ackMarkdown($scope, "Conda packages", CodeEnvsDesignService.pyCondaSpecEditHelpString);}
});

app.controller("AdminCodeEnvsDesignListController", function($scope, $controller, CodeEnvsDesignService, TopNav, DataikuAPI, Dialogs, CreateModalFromTemplate, $state, RequestCenterService, CodeEnvService) {
    $controller("AdminCodeEnvsDesignController", {$scope:$scope});
    TopNav.setLocation(TopNav.DSS_HOME, "administration");
    $scope.refreshList = function() {
        return DataikuAPI.admin.codeenvs.design.list().success(function(data) {
            for (const codeEnv of data) {
                codeEnv["languageDisplayName"] = codeEnv["pythonInterpreter"] || codeEnv["envLang"]
            }
            $scope.codeEnvs = data;
        }).error(setErrorInScope.bind($scope));
    };

    CodeEnvsDesignService.getUserAccessibleSettings().then(({data}) => {
        $scope.enableCodeEnvResources = data.enableCodeEnvResources;
        $scope.useConda = data.useConda;
    });

    $scope.refreshList();

    $scope.canRequestCodeEnvInstall = $scope.appConfig.codeEnvInstallRequestsEnabled;

    $scope.isAdmin = $scope.appConfig.admin;

    $scope.openNewPythonEnvModal = function(){
        CreateModalFromTemplate("/templates/admin/code-envs/design/new-python-env-modal.html", $scope, "AdminCodeEnvsDesignNewPythonController")
    }

    $scope.openNewREnvModal = function(){
        CreateModalFromTemplate("/templates/admin/code-envs/design/new-R-env-modal.html", $scope, "AdminCodeEnvsDesignNewRController")
    }
    $scope.openImportEnvModal = function(){
        CreateModalFromTemplate("/templates/admin/code-envs/design/import-env-modal.html", $scope, "AdminCodeEnvsDesignImportController")
    }
    $scope.actionAfterDeletion = function() {
        $scope.refreshList();
    };
    $scope.goToEditIfExists = function(envName) {
        const env = $scope.codeEnvs.find(e => e.envName === envName);
        if(env && env.envLang === 'R') {
            $state.go("admin.codeenvs-design.r-edit", { envName });
        } else if(env && env.envLang === 'PYTHON'){
            $state.go("admin.codeenvs-design.python-edit", { envName });
        }
    };

    $scope.openNewEnvRequestModal = function(envLang) {
        if (envLang === "PYTHON") {
            $scope.requestedEnv = CodeEnvsDesignService.getDefaultPythonEnv();
        } else if (envLang === "R") {
            $scope.requestedEnv = CodeEnvsDesignService.getDefaultREnv();
        }

        $scope.requestedEnv.conda = $scope.useConda;

        $scope.refreshList();

        $scope.isCodeEnvNameUnique = function(value){
            if ($scope.codeEnvs) {
               for(let k in $scope.codeEnvs) {
                  if($scope.codeEnvs[k].envName === value) {
                     return false;
                  }
               }
            }
            return true;
        }

        $scope.codeEnvModal = {$modalTab: 'GENERAL'};
        $scope.nextTab = function() {
            if ($scope.codeEnvModal.$modalTab==="GENERAL") {$scope.codeEnvModal.$modalTab="PACKAGES";}
        }

        $scope.openCRANEditHelp = function() {Dialogs.ackMarkdown($scope, "R packages", CodeEnvsDesignService.CRANEditHelpString);}
        $scope.openRCondaSpecEditHelp = function() {Dialogs.ackMarkdown($scope, "Conda packages", CodeEnvsDesignService.rCondaSpecEditHelpString);}
        $scope.openPipRequirementsEditHelp = function() {Dialogs.ackMarkdown($scope, "Pip requirements", CodeEnvsDesignService.pipRequirementsEditHelpString);}
        $scope.openPyCondaSpecEditHelp = function() {Dialogs.ackMarkdown($scope, "Conda packages", CodeEnvsDesignService.pyCondaSpecEditHelpString);}

        $scope.codeEnvResourcesEditorOptions = $scope.codeMirrorSettingService.get("text/x-python");
        $scope.specPackageListEditorOptionsPip = $scope.codeMirrorSettingService.get("text/plain", {onLoad: function(cm){$scope.codeMirrorPip = cm;}});
        $scope.specPackageListEditorOptionsConda = $scope.codeMirrorSettingService.get("text/plain", {onLoad: function(cm){$scope.codeMirrorConda = cm;}});
        $scope.updateModalPackages = function() {
            // Fetch mandatory packages list
            DataikuAPI.codeenvs.getMandatoryPackages(envLang, $scope.requestedEnv).then(({data}) => {
                $scope.requestedEnv.mandatoryPackageList = data;
            });
            //Fetch package presets
            if (envLang == "PYTHON") {
                CodeEnvsDesignService.fetchPackagePresets({desc: $scope.requestedEnv}, $scope)
            }
        }

        $scope.requestedEnv.mandatoryPackageList = $scope.updateModalPackages();
        $scope.requestedEnv.specPackageList = "";
        $scope.requestedEnv.specCondaEnvironment = "";

        $scope.insertAdditionalPackages = (editorType) => {
            const newScope = $scope.$new();

            newScope.packageListTypes = $scope.packageListTypes;
            newScope.packageListValues = $scope.packageListValues;
            newScope.selectedPackages = $scope.packageListTypes[0][0];
            newScope.codeMirrorPip = $scope.codeMirrorPip;
            newScope.codeMirrorConda = $scope.codeMirrorConda;
            const editorSettings = $scope.codeMirrorSettingService.get('text/plain');
            newScope.insertReadOnlyOptions = {
                ...editorSettings,
                readOnly: true,
                lineNumbers: false,
                foldGutter: false,
            };

            CreateModalFromTemplate("/templates/admin/code-envs/design/add-additional-packages-modal.html", newScope, null, (scope) => {
                // We must define this function here (rather than in newScope
                // itself) because scope.dismiss() is not available at this stage.
                // It is added when the actual modal scope is built from newScope.
                scope.insertPackages = () => {
                    const selectedContent = scope.packageListValues[scope.selectedPackages];
                    CodeEnvsDesignService.replacePackageListEditorContent(selectedContent, editorType, newScope.codeMirrorPip, newScope.codeMirrorConda);
                    scope.dismiss();
                };
            });
        };

        if (envLang === "PYTHON") {
            CreateModalFromTemplate("/templates/admin/code-envs/design/new-python-env-request-modal.html", $scope, "AdminCodeEnvsDesignNewPythonController", (modalScope) => {
                   modalScope.requestedEnv = $scope.requestedEnv;

                   modalScope.$watch('[requestedEnv.pythonInterpreter, requestedEnv.installCorePackages, requestedEnv.corePackagesSet, requestedEnv.installJupyterSupport]',function(nv, ov) {
                        $scope.updateModalPackages();
                   })
                   modalScope.cancelRequest = function() {
                       modalScope.$destroy();
                       modalScope.dismiss();
                   }
                   modalScope.sendRequest = function() {
                        DataikuAPI.requests.createCodeEnvRequest("INSTALL_CODE_ENV", $scope.requestedEnv.envName, envLang, JSON.stringify($scope.requestedEnv) ,$scope.requestedEnv.specPackageList, $scope.requestedEnv.specCondaEnvironment, $scope.codeEnvModal.message, 'MANUAL').success(function(data){
                            RequestCenterService.WT1Events.onRequestSent("CODE_ENV", null, $scope.requestedEnv.envName, $scope.codeEnvModal.message, data.id);
                        }).error(setErrorInScope.bind($scope));
                        modalScope.$destroy();
                        modalScope.dismiss();
                   }

                   modalScope.updateDefaultCorePackageSet = function() {
                        DataikuAPI.codeenvs.getDefaultCorePackageSet($scope.requestedEnv.pythonInterpreter).success(function(data){
                            $scope.requestedEnv.corePackagesSet = data;
                        }).error(setErrorInScope.bind($scope));
                   }

                   modalScope.updatePythonInterpreters = function(){
                        if (modalScope.requestedEnv.conda === true) {
                            modalScope.requestPythonInterpreters = CodeEnvService.pythonCondaInterpreters;
                        } else {
                            modalScope.requestPythonInterpreters = CodeEnvService.pythonInterpreters.filter(i => i[0] !== "CUSTOM");
                            CodeEnvService.getPythonInterpreters().then((interpreters) => {modalScope.requestPythonInterpreters = interpreters.filter(i => i[0] !== "CUSTOM")});
                        }
                   }
                   modalScope.updatePythonInterpreters();
               });
        } else if (envLang === "R") {
            CreateModalFromTemplate("/templates/admin/code-envs/design/new-R-env-request-modal.html", $scope, "AdminCodeEnvsDesignNewRController", (modalScope) => {
                modalScope.requestedEnv = $scope.requestedEnv;
                modalScope.$watch('[requestedEnv.installCorePackages, requestedEnv.corePackagesSet, requestedEnv.installJupyterSupport]',function(nv, ov) {
                    $scope.updateModalPackages();
                })
                modalScope.cancelRequest = function() {
                    modalScope.$destroy();
                    modalScope.dismiss();
                }
                modalScope.sendRequest = function() {
                    DataikuAPI.requests.createCodeEnvRequest("INSTALL_CODE_ENV", $scope.requestedEnv.envName, envLang, JSON.stringify($scope.requestedEnv) ,$scope.requestedEnv.specPackageList, $scope.requestedEnv.specCondaEnvironment, $scope.codeEnvModal.message, 'MANUAL').success(function(data){
                        RequestCenterService.WT1Events.onRequestSent("CODE_ENV", null, $scope.requestedEnv.envName, $scope.codeEnvModal.message, data.id);
                    }).error(setErrorInScope.bind($scope));
                    modalScope.$destroy();
                    modalScope.dismiss();
                }
            });
        }
    }
});

app.controller("AdminCodeEnvsDesignDeleteController", function($scope, TopNav, DataikuAPI, Dialogs, FutureProgressModal, $q) {
    $scope.delete = function() {
        var parentScope = $scope.$parent;
        DataikuAPI.admin.codeenvs.design.delete($scope.codeEnv.envLang, $scope.codeEnv.envName).success(function(data){
            $scope.dismiss();
            FutureProgressModal.show(parentScope, data, "Env deletion").then(function(result){
                const infoModalClosed = result
                    ? Dialogs.infoMessagesDisplayOnly(parentScope, "Deletion result", result.messages, result.futureLog)
                    : $q.resolve();
                infoModalClosed.then(() => $scope.actionAfterDeletion());
            });
        }).error(setErrorInScope.bind($scope));

    };
});

app.controller("AdminCodeEnvsDesignChangePythonSuccessfulController", function($scope, $state, CodeEnvService, $stateParams) {
    const parentScope = $scope.$parent.$parent;
    $scope.goToEnv = function(envName) {
         $state.go("admin.codeenvs-design.python-edit", {envName});
    }

    $scope.installPackages = function (envName) {
        CodeEnvService.updateEnv(parentScope, envName, true, false, false, "PYTHON", function() {
            if ($stateParams.envName != envName) {
                $scope.goToEnv(envName);
            } else {
                $scope.uiState.active = 'actual';
            }
        })
    }
});

app.controller("AdminCodeEnvsDesignChangePythonController", function($scope, TopNav, DataikuAPI, Dialogs, FutureProgressModal, $q, $state, CreateModalFromTemplate, CodeEnvService, Deprecation) {
    $scope.inplace = true;
    $scope.supportedDeprecatedPythons = ['PYTHON36', 'PYTHON37', 'PYTHON38'];

    $scope.pythonInterpreters = CodeEnvService.pythonInterpreters;
    CodeEnvService.getPythonInterpreters().then((interpreters) => {$scope.pythonInterpreters = interpreters});
    $scope.insertReadOnlyOptions = {
        ...$scope.codeMirrorSettingService.get('text/plain'),
        readOnly: true,
        lineNumbers: true,
        foldGutter: true,
    };
    $scope.insertReadWriteOptions = {
        ...$scope.codeMirrorSettingService.get('text/plain'),
        lineNumbers: true,
        foldGutter: true,
    };

    const isDeprecatedPythonInHooks = function(hooks, pythonInterpreter) {
        return $scope.supportedDeprecatedPythons.includes(pythonInterpreter) && hooks.some(hook => hook.type === pythonInterpreter + "_SUPPORT");
    };

    const parentScope = $scope.$parent.$parent;
    $scope.$watch("newEnv.pythonInterpreter", function(nv) {
        if (!$scope.nameChangedByUser) {
            $scope.newEnv.envName = $scope.oldEnv.envName + "_" + $scope.newEnv.pythonInterpreter.toLowerCase();
        }

        $scope.newEnv.pythonInterpreterFriendlyName = CodeEnvService.pythonInterpreterFriendlyName($scope.newEnv.pythonInterpreter);
        DataikuAPI.admin.codeenvs.design.updatePythonPackagePresets($scope.newEnv.specPackageList, $scope.newEnv.pythonInterpreter).success(function(data) {
            $scope.newEnv.specPackageList = data;
        }).error(setErrorInScope.bind($scope));
        if (!['PYTHON27', 'PYTHON35', 'PYTHON36', 'PYTHON37', 'CUSTOM'].includes($scope.newEnv.pythonInterpreter) && $scope.newEnv.corePackagesSet == "LEGACY_PANDAS023") {
            $scope.previousCorePackagesSet = $scope.newEnv.corePackagesSet;
            $scope.newEnv.corePackagesSet = "PANDAS13";
        } else if ($scope.newEnv.pythonInterpreter == 'PYTHON27') {
            $scope.previousCorePackagesSet = $scope.newEnv.corePackagesSet;
            $scope.newEnv.corePackagesSet = "LEGACY_PANDAS023";
        }

        $scope.oldEnvContainsDeprecatedPythonCRA = function(newPythonInterpreter) {
            if (!$scope.oldEnv || !$scope.oldEnv.desc ||
                    !$scope.oldEnv.desc.predefinedContainerHooks ||
                    !newPythonInterpreter ||
                    !$scope.supportedDeprecatedPythons.includes($scope.oldEnv.desc.pythonInterpreter)) {
                return false;
            }

            if ($scope.supportedDeprecatedPythons.includes(newPythonInterpreter)) {
                return isDeprecatedPythonInHooks($scope.oldEnv.desc.predefinedContainerHooks, $scope.oldEnv.desc.pythonInterpreter) &&
                    !isDeprecatedPythonInHooks($scope.oldEnv.desc.predefinedContainerHooks, newPythonInterpreter);
            } else if (!$scope.supportedDeprecatedPythons.includes(newPythonInterpreter)) {
                return isDeprecatedPythonInHooks($scope.oldEnv.desc.predefinedContainerHooks, "PYTHON36") ||
                    isDeprecatedPythonInHooks($scope.oldEnv.desc.predefinedContainerHooks, "PYTHON37") ||
                    isDeprecatedPythonInHooks($scope.oldEnv.desc.predefinedContainerHooks, "PYTHON38");
            }
        };

        $scope.newEnvRequiresDeprecatedPythonCRA = function (newPythonInterpreter) {
            if (!$scope.oldEnv || !$scope.oldEnv.desc || !$scope.oldEnv.desc.predefinedContainerHooks || !newPythonInterpreter) {
                return false;
            }

            return !$scope.supportedDeprecatedPythons.includes($scope.oldEnv.desc.pythonInterpreter) &&
                !isDeprecatedPythonInHooks($scope.oldEnv.desc.predefinedContainerHooks, newPythonInterpreter);
        }
    });

    $scope.create = function() {
        DataikuAPI.admin.codeenvs.design.create("PYTHON", $scope.newEnv).success(function(data){
            $scope.dismiss();
            FutureProgressModal.show(parentScope, data, "Env creation", undefined, 'static', false).then(function(result){
                const modalClosed = result
                    ? Dialogs.infoMessagesDisplayOnly(parentScope, "Creation result", result.messages, result.futureLog, undefined, 'static', false)
                    : $q.resolve();
                modalClosed.then(() => {
                    DataikuAPI.admin.codeenvs.design.get("PYTHON", $scope.newEnv.envName).success(function(data) {
                        let packageList = $scope.newEnv.specPackageList;
                        $scope.newEnv = data;
                        $scope.newEnv.specPackageList = packageList;
                        $scope.newEnv.resourcesInitScript = $scope.oldEnv.resourcesInitScript;
                        DataikuAPI.admin.codeenvs.design.save("PYTHON", $scope.newEnv.envName, $scope.newEnv).success(function() {
                            CreateModalFromTemplate("/templates/admin/code-envs/design/change-python-modal-successful.html", parentScope, "AdminCodeEnvsDesignChangePythonSuccessfulController", function (newScope) {
                                newScope.env = $scope.newEnv;
                                newScope.inplace = $scope.inplace;
                                newScope.env.pythonInterpreterFriendlyName = CodeEnvService.pythonInterpreterFriendlyName(newScope.env.desc.pythonInterpreter);
                                newScope.previousCorePackagesSet = $scope.previousCorePackagesSet;
                            })
                        }).error(setErrorInScope.bind($scope));
                    }).error(setErrorInScope.bind($scope));
                });
            });

        }).error(setErrorInScope.bind(parentScope));
    }
    $scope.update = function(){
        $scope.dismiss();

        if ($scope.newEnv.removeContainerRuntimeAdditions && !$scope.supportedDeprecatedPythons.includes($scope.newEnv.pythonInterpreter)) {
            var filteredHooks = $scope.oldEnv.desc.predefinedContainerHooks.filter(hook => hook.type !== "PYTHON36_SUPPORT" && hook.type !== "PYTHON37_SUPPORT" && hook.type !== "PYTHON38_SUPPORT");
            $scope.oldEnv.desc.predefinedContainerHooks = filteredHooks;
        } else if ($scope.newEnv.updateContainerRuntimeAdditions && $scope.supportedDeprecatedPythons.includes($scope.newEnv.pythonInterpreter)) {
            var updatedHooks = $scope.oldEnv.desc.predefinedContainerHooks.filter(hook => hook.type !== ($scope.oldEnv.desc.pythonInterpreter + "_SUPPORT"));
            updatedHooks.push({ type: $scope.newEnv.pythonInterpreter + "_SUPPORT" });
            $scope.oldEnv.desc.predefinedContainerHooks = updatedHooks;
        }

        $scope.oldEnv.desc.pythonInterpreter = $scope.newEnv.pythonInterpreter;
        $scope.oldEnv.desc.customInterpreter = $scope.newEnv.customInterpreter;
        $scope.oldEnv.desc.corePackagesSet = $scope.newEnv.corePackagesSet;
        $scope.oldEnv.specPackageList = $scope.newEnv.specPackageList;
        DataikuAPI.admin.codeenvs.design.save("PYTHON", $scope.oldEnv.envName, $scope.oldEnv).success(function(data) {
            DataikuAPI.admin.codeenvs.design.recreate("PYTHON", $scope.oldEnv.envName).success(function(data) {
                FutureProgressModal.show(parentScope, data, "Change python interpreter", undefined, 'static', false).then(function(result){
                    const modalClosed = result
                        ? Dialogs.infoMessagesDisplayOnly(parentScope, "Migration result", result.messages, result.futureLog, undefined, 'static', false)
                        : $q.resolve();
                    if (!result.messages.error) {
                        modalClosed.then(() => {
                            CreateModalFromTemplate("/templates/admin/code-envs/design/change-python-modal-successful.html", parentScope, "AdminCodeEnvsDesignChangePythonSuccessfulController", function (newScope) {
                                $scope.oldEnv.desc.corePackagesSet = $scope.newEnv.corePackagesSet;
                                newScope.env = $scope.oldEnv;
                                newScope.inplace = $scope.inplace;
                                newScope.env.pythonInterpreterFriendlyName = CodeEnvService.pythonInterpreterFriendlyName(newScope.env.desc.pythonInterpreter);
                                newScope.previousCorePackagesSet = $scope.previousCorePackagesSet;
                            })
                        });
                    }
                });
            CodeEnvService.refreshEnv(parentScope);
            }).error(setErrorInScope.bind($scope));
        }).error(setErrorInScope.bind(parentScope));
    }
});

app.controller("AdminCodeEnvsDesignNewPythonController", function($scope, CodeEnvsDesignService, TopNav, DataikuAPI, Dialogs, FutureProgressModal, $q, Deprecation, CodeEnvService) {

    $scope.newEnv = CodeEnvsDesignService.getDefaultPythonEnvNoCorePackage();

    $scope.deploymentModes = [
        ["DESIGN_MANAGED", "Managed by DSS (recommended)"],
        ["DESIGN_NON_MANAGED", "Non-managed path"],
        ["EXTERNAL_CONDA_NAMED", "Named external Conda env"]
    ]

    CodeEnvsDesignService.getUserAccessibleSettings().then(({data}) => {
        $scope.enableCodeEnvResources = data.enableCodeEnvResources;
    })

    $scope.$watch("newEnv.conda", function(nv) {
        if (nv === true) {
            $scope.pythonInterpreters = CodeEnvService.pythonCondaInterpreters;
        } else if (nv === false) {
            $scope.pythonInterpreters = CodeEnvService.pythonInterpreters;
            CodeEnvService.getPythonInterpreters().then((interpreters) => {$scope.pythonInterpreters = interpreters});
        }
    });
    $scope.$watch("newEnv.deploymentMode", $scope.isPythonDeprecated);
    $scope.create = function(){
        var parentScope = $scope.$parent.$parent;
        DataikuAPI.admin.codeenvs.design.create("PYTHON", $scope.newEnv).success(function(data){
            $scope.dismiss();
            FutureProgressModal.show(parentScope, data, "Env creation", undefined, 'static', false).then(function(result){
                const modalClosed = result
                    ? Dialogs.infoMessagesDisplayOnly(parentScope, "Creation result", result.messages, result.futureLog, undefined, 'static', false)
                    : $q.resolve();

                const refreshed = parentScope.refreshList();

                $q.all([modalClosed, refreshed]).then(() => {
                    parentScope.goToEditIfExists(result && result.envName);
                });
            });
        }).error(setErrorInScope.bind($scope));
    }
    $scope.isPythonDeprecated = function() {
        return $scope.newEnv && ($scope.newEnv.deploymentMode === "DESIGN_MANAGED") && Deprecation.isPythonDeprecated($scope.newEnv.pythonInterpreter);
    }
});

app.controller("AdminCodeEnvsDesignNewRController", function($scope, CodeEnvsDesignService, TopNav, DataikuAPI, Dialogs, FutureProgressModal, $q) {

    $scope.newEnv = CodeEnvsDesignService.getDefaultREnv();

    $scope.deploymentModes = [
        ["DESIGN_MANAGED", "Managed by DSS (recommended)"],
        ["DESIGN_NON_MANAGED", "Non-managed path"],
        ["EXTERNAL_CONDA_NAMED", "Named external Conda env"]
    ]

    CodeEnvsDesignService.getUserAccessibleSettings().then(({data}) => {
        $scope.enableCodeEnvResources = data.enableCodeEnvResources;
    })

    $scope.create = function(){
        var parentScope = $scope.$parent.$parent;
        DataikuAPI.admin.codeenvs.design.create("R", $scope.newEnv).success(function(data){
            $scope.dismiss();
            FutureProgressModal.show(parentScope, data, "Env creation", undefined, 'static', false).then(function(result){
                const modalClosed = result
                    ? Dialogs.infoMessagesDisplayOnly(parentScope, "Creation result", result.messages, result.futureLog, undefined, 'static', false)
                    : $q.resolve();

                const refreshed = parentScope.refreshList();

                $q.all([modalClosed, refreshed]).then(() => {
                    parentScope.goToEditIfExists(result && result.envName);
                });
            });
        }).error(setErrorInScope.bind($scope));
    }
});

app.controller("AdminCodeEnvsDesignCreateFromDraftController", function($scope, $controller, $state, $stateParams, Assert, TopNav, DataikuAPI, FutureProgressModal, CreateModalFromTemplate, Dialogs, Logs, $q, CodeEnvService) {
    $controller("AdminCodeEnvsDesignListController", {$scope:$scope});

    $scope.loadDraftCodeEnv = function() {
        return DataikuAPI.admin.codeenvs.design.getDraft($stateParams.draftId).success(function(data) {
            $scope.draftId = data.config.id;
            $scope.newEnv = data.env;
            $scope.newEnv.envName = data.config.envName;
            $scope.envLang = data.config.envLang;
            $scope.openDraftModal();
        }).error(setErrorInScope.bind($scope));
    };
    $scope.loadDraftCodeEnv();

    $scope.isAdmin = $scope.appConfig.admin;

    $scope.openDraftModal = function(){
        if ($scope.envLang === "PYTHON") {
            CreateModalFromTemplate("/templates/admin/code-envs/design/new-python-env-from-draft-modal.html", $scope, "AdminCodeEnvsDesignNewPythonController", (modalScope) => {
                modalScope.requestedEnv = $scope.newEnv;

                if (modalScope.requestedEnv.conda === true) {
                   modalScope.draftPythonInterpreters = CodeEnvService.pythonCondaInterpreters;
                } else {
                   modalScope.draftPythonInterpreters = CodeEnvService.pythonInterpreters;
                   CodeEnvService.getPythonInterpreters().then((interpreters) => {modalScope.draftPythonInterpreters = interpreters});
                }

                modalScope.createEnvFromDraft = function() {
                    DataikuAPI.admin.codeenvs.design.createFromDraft($scope.draftId, "PYTHON", $scope.newEnv).success(function(data){
                        modalScope.dismiss();
                        handleEnvCreationResult(data);
                    }).error(setErrorInScope.bind($scope));
                };
            });
        } else if ($scope.envLang === "R") {
            CreateModalFromTemplate("/templates/admin/code-envs/design/new-R-env-from-draft-modal.html", $scope, "AdminCodeEnvsDesignNewRController", (modalScope) => {
                modalScope.requestedEnv = $scope.newEnv;
                modalScope.createEnvFromDraft = function() {
                    DataikuAPI.admin.codeenvs.design.createFromDraft($scope.draftId, "R", $scope.newEnv).success(function(data){
                        modalScope.dismiss();
                        handleEnvCreationResult(data);
                    }).error(setErrorInScope.bind($scope));
                }
            });
        }
    }

    function handleEnvCreationResult(data){
        FutureProgressModal.show($scope, data, "Env creation", undefined, 'static', false).then(function(result){
            const modalClosed = result
                ? Dialogs.infoMessagesDisplayOnly($scope, "Creation result", result.messages, result.futureLog, undefined, 'static', false)
                : $q.resolve();
            const refreshed = $scope.refreshList();

            $q.all([modalClosed, refreshed]).then(() => {
                $scope.goToEditIfExists(result && result.envName);
            });
        });
    }
});


app.controller("AdminCodeEnvsDesignImportController", function($scope, $state, $stateParams, Assert, TopNav, DataikuAPI, FutureProgressModal, Dialogs, Logs, $q) {
    $scope.newEnv = {}

    $scope.import = function() {
        Assert.trueish($scope.newEnv.file, "No code env file");

        const parentScope = $scope.$parent.$parent;
        DataikuAPI.admin.codeenvs.design.import($scope.newEnv.file).then(function(data) {
            $scope.dismiss();
            FutureProgressModal.show(parentScope, JSON.parse(data), "Env import", undefined, 'static', false).then(function(result) {
                const modalClosed = result
                    ? Dialogs.infoMessagesDisplayOnly(parentScope, "Creation result", result.messages, result.futureLog, undefined, 'static', false)
                    : $q.resolve();

                const refreshed = parentScope.refreshList();

                $q.all([modalClosed, refreshed]).then(() => {
                    parentScope.goToEditIfExists(result && result.envName);
                });
            });
        }, function(payload) {
            setErrorInScope.bind($scope)(JSON.parse(payload.response), payload.status, function(h) {return payload.getResponseHeader(h)});
        });
    }
});


app.controller("_AdminCodeEnvsDesignEditController", function ($scope, $controller, $state, $stateParams, TopNav, DataikuAPI, ActivityIndicator, CreateModalFromTemplate, FutureProgressModal, Dialogs, Logs, CodeEnvService) {
    $controller("AdminCodeEnvsDesignController", { $scope });
    TopNav.setLocation(TopNav.DSS_HOME, "administration");

    $scope.envName = $stateParams.envName;

    $scope.uiState = {
        performChangesOnSave: true,
        upgradeAllPackages: true,
        updateResources: true,
        forceRebuildEnv: false,
        active : 'info'
    }

    $scope.getSingleVersion = function(codeEnv) {
        return null;
    };

    $scope.actionAfterDeletion = function() {
        $state.go("admin.codeenvs-design.list");
    };

    $scope.canBeUpdated = function() {
        return $scope.codeEnv && $scope.codeEnv.canUpdateCodeEnv && ['DESIGN_MANAGED', 'PLUGIN_MANAGED', 'AUTOMATION_SINGLE', 'AUTOMATION_VERSIONED', 'DSS_INTERNAL'].indexOf($scope.codeEnv.deploymentMode) >= 0;
    };

    $scope.getPackagesThatWereRequiredAndInstalledButAreNotRequiredAnymore = function(packageSystem){
        if (!$scope.codeEnv || !$scope.previousPackagesSetForRemovalWarning[packageSystem]) return [];

        const newSet = CodeEnvService.getPackagesThatAreRequired($scope.codeEnv, packageSystem);

        return [...CodeEnvService.difference($scope.previousPackagesSetForRemovalWarning[packageSystem], newSet)];
    }

    $scope.isPackageRequired = function(packageSystem, packageName) {
        let required = "";
        if (packageSystem == 'pip') {
            required = $scope.codeEnv.specPackageList;
        } else if (packageSystem == 'conda') {
            required = $scope.codeEnv.specCondaEnvironment;
        }
        return CodeEnvService.getPackageNames(required).has(packageName);
    }

    $scope.isPackageInstalled = function(packageSystem, packageName) {
        let installed = "";
        if (packageSystem == 'pip') {
            installed = $scope.codeEnv.actualPackageList;
        } else if (packageSystem == 'conda') {
            installed = $scope.codeEnv.actualCondaEnvironment;
        }
        return CodeEnvService.getPackageNames(installed).has(packageName);
    }

    $scope.isInternalCodeEnv = function() {
        return $scope.codeEnv && ($scope.codeEnv.deploymentMode === 'DSS_INTERNAL');
    }

    $scope.hasOptedOutOfReferenceSpec = function() {
        return $scope.isInternalCodeEnv() && $scope.codeEnv.desc && ($scope.codeEnv.desc.useReferenceSpec === false)
    }

    $scope.useReferenceSpec = function() {
        return $scope.isInternalCodeEnv() && $scope.codeEnv.desc && ($scope.codeEnv.desc.useReferenceSpec === true)
    }

    $scope.specIsDirty = function() {
        if (!$scope.codeEnv) return false;
        var currentSpec = CodeEnvService.makeCurrentSpec($scope.codeEnv);
        return !angular.equals(currentSpec, $scope.previousSpec);
    };
    checkChangesBeforeLeaving($scope, $scope.specIsDirty);

    $scope.previousSpec = {
    }
    $scope.previousPackagesSetForRemovalWarning = {'pip': new Set(), 'conda': new Set()};

    $scope.refreshMandatoryPackageList = function(){
        if(!$scope.codeEnv) return;

        if(!$scope.codeEnv.desc.installCorePackages && !$scope.codeEnv.desc.installJupyterSupport){
            $scope.codeEnv.mandatoryPackageList = "";
            $scope.codeEnv.mandatoryCondaEnvironment = "";
            return;
        }

        DataikuAPI.codeenvs.getMandatoryPackages($scope.envLang, $scope.codeEnv.desc).then(({data}) => {
           if($scope.codeEnv.conda){
              $scope.codeEnv.mandatoryCondaEnvironment = data;
           }else{
              $scope.codeEnv.mandatoryPackageList = data;
           }
        });
    }

    CodeEnvService.refreshEnv($scope).then(() => $scope.uiState.forceRebuildEnv = $scope.isInternalCodeEnv());

    $scope.updateEnv = function() {
        CodeEnvService.updateEnv(
            $scope, 
            $stateParams.envName, 
            $scope.uiState.upgradeAllPackages, 
            $scope.uiState.updateResources, 
            $scope.uiState.forceRebuildEnv, 
            $scope.envLang, 
            null
        );
    };

    $scope.saveAndMaybePerformChanges = function(performChangesOnSave){
        DataikuAPI.admin.codeenvs.design.save($scope.envLang, $stateParams.envName, $scope.codeEnv).success(function(data) {
            CodeEnvService.refreshEnv($scope);
            if (performChangesOnSave) {
                $scope.updateEnv();
            }
        }).error(setErrorInScope.bind($scope));
    }

    $scope.fetchNonManagedEnvDetails = function(){
        DataikuAPI.admin.codeenvs.design.fetchNonManagedEnvDetails($scope.envLang, $stateParams.envName).success(function(data) {
            $scope.nonManagedEnvDetails = data;
        }).error(setErrorInScope.bind($scope));
    }

    $scope.installJupyterSupport = function(){
        DataikuAPI.admin.codeenvs.design.installJupyterSupport($scope.envLang, $stateParams.envName).success(function(data) {
            FutureProgressModal.show($scope, data, "Env update").then(function(result){
                if (result) { // undefined in case of abort
                    Dialogs.infoMessagesDisplayOnly($scope, "Update result", result.messages, result.futureLog);
                }
                CodeEnvService.refreshEnv($scope);
            })
        }).error(setErrorInScope.bind($scope));
    }

    $scope.removeJupyterSupport = function(){
        DataikuAPI.admin.codeenvs.design.removeJupyterSupport($scope.envLang, $stateParams.envName).success(function(data) {
            FutureProgressModal.show($scope, data, "Env update").then(function(result){
                if (result) { // undefined in case of abort
                    Dialogs.infoMessagesDisplayOnly($scope, "Update result", result.messages, result.futureLog);
                }
                CodeEnvService.refreshEnv($scope);
            })
        }).error(setErrorInScope.bind($scope));
    }

    $scope.specPackageListEditorOptionsPip = $scope.codeMirrorSettingService.get("text/plain", {onLoad: function(cm){$scope.codeMirrorPip = cm;}});
    $scope.specPackageListEditorOptionsConda = $scope.codeMirrorSettingService.get("text/plain", {onLoad: function(cm){$scope.codeMirrorConda = cm;}});
    $scope.codeEnvResourcesEditorOptions = $scope.codeMirrorSettingService.get("text/x-python");
    $scope.getLog = DataikuAPI.admin.codeenvs.design.getLog;
    $scope.downloadURL = function(envLang, envName, logName) {
        return "/dip/api/code-envs/design/stream-log?envLang=" + envLang + "&envName=" + encodeURIComponent(envName) + "&logName=" + encodeURIComponent(logName);
    }
});

app.controller("AdminCodeEnvsDesignPythonEditController", function($scope, $controller, $stateParams, CodeEnvsDesignService, Dialogs, DataikuAPI, CreateModalFromTemplate, $timeout) {
    $scope.envLang = "PYTHON";
    $controller("_AdminCodeEnvsDesignEditController", { $scope });

    // Retrieve a handle on the editor for the spec packages (pip).
    let codeMirrorPip = null;
    $scope.specPackageListEditorOptionsPip = $scope.codeMirrorSettingService.get("text/plain", {
        onLoad: (codeMirror) => {
            codeMirrorPip = codeMirror;
        },
    });

    // Get a handle on the editor for the spec packages (conda).
    let codeMirrorConda = null;
    $scope.specPackageListEditorOptionsConda = $scope.codeMirrorSettingService.get("text/plain", {
        onLoad: (codeMirror) => {
            codeMirrorConda = codeMirror;
        },
    });

    // Fetch the package presets.
    $scope.packageListTypes = [];
    $scope.packageListValues = {};

    // The code environment is not yet loaded when this code is executed.
    CodeEnvsDesignService.getUserAccessibleSettings().then(({data}) => {
        $scope.enableCodeEnvResources = data.enableCodeEnvResources;
    })

    $scope.insertAdditionalPackages = (editorType) => {
        const newScope = $scope.$new();

        const editorSettings = $scope.codeMirrorSettingService.get('text/plain');
        newScope.insertReadOnlyOptions = {
            ...editorSettings,
            readOnly: true,
            lineNumbers: false,
            foldGutter: false,
        };

        CreateModalFromTemplate("/templates/admin/code-envs/design/add-additional-packages-modal.html", newScope, null, (scope) => {
            CodeEnvsDesignService.fetchPackagePresets(newScope.codeEnv, newScope).then(() => {
                newScope.selectedPackages = newScope.packageListTypes[0][0];
            });

            // We must define this function here (rather than in newScope
            // itself) because scope.dismiss() is not available at this stage.
            // It is added when the actual modal scope is built from newScope.
            scope.insertPackages = () => {
                const selectedContent = scope.packageListValues[scope.selectedPackages];
                CodeEnvsDesignService.replacePackageListEditorContent(selectedContent, editorType, codeMirrorPip, codeMirrorConda);

                scope.dismiss();
            };
        });
    };

    $scope.openChangePythonInterpreterModal = function(codeEnv){
        CreateModalFromTemplate("/templates/admin/code-envs/design/change-python-modal.html", $scope, "AdminCodeEnvsDesignChangePythonController", function (newScope, newDOMElt) {
            newScope.oldEnv = codeEnv;
            newScope.previousPython = codeEnv.desc.pythonInterpreter;
            newScope.newEnv = angular.copy(codeEnv.desc);
            newScope.newEnv.pythonInterpreter = "PYTHON39";
            newScope.newEnv.envLang = codeEnv.envLang;
            newScope.newEnv.deploymentMode = codeEnv.deploymentMode;
            newScope.newEnv.specPackageList = newScope.oldEnv.specPackageList;
            newScope.newEnv.updateContainerRuntimeAdditions = true;
            newScope.newEnv.removeContainerRuntimeAdditions = true;
            newScope.nameChangedByUser = false;
            newDOMElt.on('keydown', 'input', function(e) {
                newScope.nameChangedByUser = true;
            })
        })
    }

});

app.controller("AdminCodeEnvsDesignREditController", function($scope, $controller) {
    $scope.envLang = "R";
    $controller("_AdminCodeEnvsDesignEditController", { $scope });
});

app.service("CodeEnvsDesignService", (DataikuAPI, $timeout) => {
        const svc = {};

        svc.getDefaultPythonEnv = () => {
            return {
                   envLang: "PYTHON",
                   deploymentMode: "DESIGN_MANAGED",
                   pythonInterpreter: "PYTHON39",
                   conda: false,
                   installCorePackages: true,
                   corePackagesSet : "PANDAS10",
                   installJupyterSupport: true
               };
        }
        svc.getDefaultPythonEnvNoCorePackage = () => {
            return {
                   envLang: "PYTHON",
                   deploymentMode: "DESIGN_MANAGED",
                   pythonInterpreter: "PYTHON39",
                   conda: false,
                   installCorePackages: true,
                   installJupyterSupport: true
               };
        }
        svc.getDefaultREnv = () => {
            return {
                envLang: "R",
                deploymentMode: "DESIGN_MANAGED",
                conda: false,
                installCorePackages: true,
                installJupyterSupport: true
               };

        }

        svc.CRANEditHelpString = "Specify the packages you want:\n\n"+
                                 "* one row per package\n"+
                                 "* each row is a pair of package name and minimal package version (optional)\n"+
                                 "\n"+
                                 "The version is only a minimal version. It is not supported to specify an explicity version.\n\n"+
                                 "Examples of package specifications:\n\n"+
                                 "* RJSONIO,0.13\n"+
                                 "* dplyr,";
        svc.rCondaSpecEditHelpString = "Specify the packages you want:\n\n"+
                                       "* one row per package\n"+
                                       "* each row is a Conda package match specification ( [link](https://conda.io/docs/user-guide/tasks/build-packages/package-spec.html#package-match-specifications) )\n"+
                                       "\n"+
                                       "Examples of package specifications:\n\n"+
                                       "* r-irkernel>=0.7";
        svc.pipRequirementsEditHelpString = "Specify the packages you want:\n\n"+
                                            "* one row per package\n"+
                                            "* each row is a PIP package specification ( [link](https://setuptools.readthedocs.io/en/latest/pkg_resources.html#requirement-objects) )\n"+
                                            "\n"+
                                            "Examples of package specifications:\n\n"+
                                            "* pandas==0.20.3\n"+
                                            "* numpy>=0.19";
        svc.pyCondaSpecEditHelpString = "Specify the packages you want:\n\n"+
                                        "* one row per package\n"+
                                        "* each row is a Conda package match specification ( [link](https://conda.io/docs/user-guide/tasks/build-packages/package-spec.html#package-match-specifications) )\n"+
                                        "\n"+
                                        "Examples of package specifications:\n\n"+
                                        "* pandas=0.20.3\n"+
                                        "* numpy>=0.19";

        svc.fetchPackagePresets = (codeEnv, $scope) => {
            const { pythonInterpreter, conda, installCorePackages, corePackagesSet } = codeEnv.desc;
            let pythonCorePackageSet = corePackagesSet;
            if (installCorePackages === false) {
                pythonCorePackageSet = null;
            }
            return DataikuAPI.admin.codeenvs.design.listPythonPackagePresets(pythonInterpreter, conda, pythonCorePackageSet)
                .success((presets) => {
                    $scope.packageListTypes = []
                    $scope.packageListValues = {}
                    presets.forEach((preset) => {
                        const { id, description, packages } = preset;
                        $scope.packageListTypes.push([id, description]);
                        $scope.packageListValues[id] = ("# "+description+"\n").concat(packages.join("\n"));
                    });
                })
                .error(setErrorInScope.bind($scope));
        }

        svc.replacePackageListEditorContent = (content, editorType, codeMirrorPip, codeMirrorConda) => {
            const codeMirror = (editorType === "pip") ? codeMirrorPip : codeMirrorConda;

            // Move cursor to end of editor content
            const lastLine = codeMirror.lastLine();
            const lastChar = codeMirror.getLine(lastLine).length;
            if (lastLine > 0) {
                codeMirror.replaceRange('\n\n', {line: lastLine, ch: lastChar});
                codeMirror.setCursor(lastLine + 2, 0);
            } else {
                codeMirror.setCursor(lastLine, lastChar);
            }

            // Timeout to make sure of an angular safe apply.
            $timeout(() => {
                codeMirror.replaceSelection(content, "end");
            });
            codeMirror.focus();
        }

        svc.getUserAccessibleSettings = function() {
            return DataikuAPI.codeenvs.getUserAccessibleCodeEnvSettings();
        }
        return svc;
    }
);

}());

;
(function() {
'use strict';

var app = angular.module('dataiku.admin.codeenvs.automation', []);


app.controller("AdminCodeEnvsAutomationController", function($scope, $rootScope, TopNav, DataikuAPI, Dialogs, FutureProgressModal, CreateModalFromTemplate, ActivityIndicator, Deprecation) {
    $scope.canCreateCodeEnv = $rootScope.mayCreateCodeEnvs;

    $scope.isDeployer = function() {
        return !$scope.appConfig.projectsModuleEnabled;
    }

    $scope.openDeleteEnvModal = function(codeEnv){
        var newScope = $scope.$new();
        newScope.codeEnv = codeEnv;
        // modal appears when usages are ready
        CreateModalFromTemplate("/templates/admin/code-envs/common/delete-env-modal.html", newScope, "AdminCodeEnvsAutomationDeleteController")
    }

    $scope.isPythonDeprecated = function (pythonInterpreter) {
        return Deprecation.isPythonDeprecated(pythonInterpreter);
    };

    $scope.getEnvDiagnostic = function(envLang, envName) {
        ActivityIndicator.success("Generating code env diagnostic ...");
        downloadURL(DataikuAPI.admin.codeenvs.automation.getDiagnosticURL(envLang, envName));
    };
});

app.controller("AdminCodeEnvsAutomationListController", function($scope, $controller, TopNav, DataikuAPI, Dialogs, CreateModalFromTemplate, $state) {
    $controller("AdminCodeEnvsAutomationController", {$scope:$scope});
    TopNav.setLocation(TopNav.DSS_HOME, "administration");

    var buildKernelSpecField = function(env) {
        if (env.currentVersion) {
            env.kernelSpecName = env.currentVersion.kernelSpecName;
        } else if (env.noVersion) {
            env.kernelSpecName = env.noVersion.kernelSpecName;
        } else if (env.versions) {
            var names = [];
            env.versions.forEach(function(ver) {
                angular.forEach(ver.kernelSpecNames, function(v, k) {names.push(k);});
            });
            env.kernelSpecName = names.join(', ');
        }
    };
    $scope.refreshList = function() {
        return DataikuAPI.admin.codeenvs.automation.list().success(function(data) {
            for (const codeEnv of data) {
                codeEnv["languageDisplayName"] = codeEnv["pythonInterpreter"] || codeEnv["envLang"]
            }
            $scope.codeEnvs = data;
            $scope.codeEnvs.forEach(function(env) {buildKernelSpecField(env);});
        }).error(setErrorInScope.bind($scope));
    };
    $scope.refreshList();

    const YOU_DONT_REALLY_WANT_TO_CREATE =
            "Manually creating code envs in automation nodes is <strong>not recommended</strong>. The recommended way to manage " +
            "code envs in automation is to let bundle preload take care of it: simply preload and activate bundles " + 
            "and the required code envs will be automatically managed.<br /><strong>Manually created code envs may not be entirely "+
            "functional</strong>.";

    const YOU_DONT_REALLY_WANT_TO_IMPORT =
            "Manually importing code envs in automation nodes is <strong>not recommended</strong>. The recommended way to manage " +
            "code envs in automation is to let bundle preload take care of it: simply preload and activate bundles " + 
            "and the required code envs will be automatically managed.<br /><strong>Manually imported code envs may not be entirely "+
            "functional</strong>.";

    $scope.openNewPythonEnvModal = function(){
        if ($scope.isDeployer()) {
            CreateModalFromTemplate("/templates/admin/code-envs/automation/new-python-env-modal.html", $scope, "AdminCodeEnvsAutomationNewPythonController");
        } else {
            Dialogs.confirm($scope, "Really create a new Python env?", YOU_DONT_REALLY_WANT_TO_CREATE).then(function(){
                CreateModalFromTemplate("/templates/admin/code-envs/automation/new-python-env-modal.html", $scope, "AdminCodeEnvsAutomationNewPythonController");
            });
        }
    }
    $scope.openNewREnvModal = function(){
        Dialogs.confirm($scope, "Really create a new R env?", YOU_DONT_REALLY_WANT_TO_CREATE).then(function(){
            CreateModalFromTemplate("/templates/admin/code-envs/automation/new-R-env-modal.html", $scope, "AdminCodeEnvsAutomationNewRController");
        });
    }
    $scope.openImportEnvModal = function(){
        if ($scope.isDeployer()) {
            CreateModalFromTemplate("/templates/admin/code-envs/automation/import-env-modal.html", $scope, "AdminCodeEnvsAutomationImportController");
        } else {
            Dialogs.confirm($scope, "Really import a code env?", YOU_DONT_REALLY_WANT_TO_IMPORT).then(function() {
                CreateModalFromTemplate("/templates/admin/code-envs/automation/import-env-modal.html", $scope, "AdminCodeEnvsAutomationImportController");
            });
        }
    }
    $scope.actionAfterDeletion = function() {
        $scope.refreshList();
    };
    $scope.goToEditIfExists = function(envName) {
        const env = $scope.codeEnvs.find(e => e.envName === envName);
        if(env && env.envLang === 'R') {
            $state.go("admin.codeenvs-automation.r-edit", { envName });
        } else if(env && env.envLang === 'PYTHON'){
            $state.go("admin.codeenvs-automation.python-edit", { envName });
        }
    };
});

app.controller("AdminCodeEnvsAutomationDeleteController", function($scope, TopNav, DataikuAPI, Dialogs, FutureProgressModal, $q) {
    $scope.delete = function() {
        var parentScope = $scope.$parent;
        DataikuAPI.admin.codeenvs.automation.delete($scope.codeEnv.envLang, $scope.codeEnv.envName).success(function(data){
            $scope.dismiss();
            FutureProgressModal.show(parentScope, data, "Env deletion").then(function(result){
                const infoModalClosed = result
                    ? Dialogs.infoMessagesDisplayOnly(parentScope, "Deletion result", result.messages, result.futureLog)
                    : $q.resolve();
                infoModalClosed.then(() => $scope.actionAfterDeletion());
            });
        }).error(setErrorInScope.bind($scope));

    };
});

app.controller("AdminCodeEnvsAutomationNewPythonController", function($scope, TopNav, DataikuAPI, Dialogs, FutureProgressModal, $q, Deprecation, CodeEnvService) {
    $scope.isDeployer = function() {
        return !$scope.appConfig.projectsModuleEnabled;
    }

    $scope.newEnv = {
            deploymentMode: "AUTOMATION_SINGLE",
            pythonInterpreter: "PYTHON39",
            conda: false,
            installCorePackages: true,
            // corePackagesSet : "PANDAS10", // let the backend decide
            installJupyterSupport: !$scope.isDeployer()
    };

    $scope.deploymentModes = $scope.isDeployer() ? [
        ["AUTOMATION_SINGLE", "Managed, non versioned"],
    ] : [
        ["AUTOMATION_VERSIONED", "Managed and versioned (recommended)"],
        ["AUTOMATION_SINGLE", "Managed, non versioned"],
        ["AUTOMATION_NON_MANAGED_PATH", "Externally-managed"],
        ["EXTERNAL_CONDA_NAMED", "Named external Conda env"]
    ];

    $scope.$watch("newEnv.conda", function(nv) {
        if (nv === true) {
            $scope.pythonInterpreters = CodeEnvService.pythonCondaInterpreters;
        } else if (nv === false) {
            $scope.pythonInterpreters = CodeEnvService.pythonInterpreters;
            CodeEnvService.getPythonInterpreters().then((interpreters) => {$scope.pythonInterpreters = interpreters});
        }
    });
    $scope.$watch("newEnv.deploymentMode", $scope.isPythonDeprecated);

    $scope.create = function(){
        var parentScope = $scope.$parent.$parent;
        DataikuAPI.admin.codeenvs.automation.create("PYTHON", $scope.newEnv).success(function(data){
            $scope.dismiss();
            FutureProgressModal.show(parentScope, data, "Env creation").then(function(result){
                const modalClosed = result
                    ? Dialogs.infoMessagesDisplayOnly(parentScope, "Creation result", result.messages, result.futureLog, undefined, 'static', false)
                    : $q.resolve();

                const refreshed = parentScope.refreshList();

                $q.all([modalClosed, refreshed]).then(() => {
                    parentScope.goToEditIfExists(result && result.envName);
                });
            });
        }).error(setErrorInScope.bind($scope));
    }
    $scope.isPythonDeprecated = function() {
        return $scope.newEnv && ($scope.newEnv.deploymentMode === "AUTOMATION_SINGLE") && Deprecation.isPythonDeprecated($scope.newEnv.pythonInterpreter);
    }
});

app.controller("AdminCodeEnvsAutomationNewRController", function($scope, TopNav, DataikuAPI, Dialogs, FutureProgressModal, $q) {
    $scope.newEnv = {
            deploymentMode: "AUTOMATION_SINGLE",
            conda: false,
            installCorePackages: true,
            installJupyterSupport: true
        };

    $scope.deploymentModes = [
                              ["AUTOMATION_VERSIONED", "Managed and versioned (recommended)"], // versioned is only created/modified by bundles
                              ["AUTOMATION_SINGLE", "Managed, non versioned"],
                              ["AUTOMATION_NON_MANAGED_PATH", "Externally-managed"],
                              ["EXTERNAL_CONDA_NAMED", "Named external Conda env"]
                          ];

    $scope.create = function(){
        var parentScope = $scope.$parent.$parent;
        DataikuAPI.admin.codeenvs.automation.create("R", $scope.newEnv).success(function(data){
            $scope.dismiss();
            FutureProgressModal.show(parentScope, data, "Env creation").then(function(result){
                const modalClosed = result
                    ? Dialogs.infoMessagesDisplayOnly(parentScope, "Creation result", result.messages, result.futureLog, undefined, 'static', false)
                    : $q.resolve();

                const refreshed = parentScope.refreshList();

                $q.all([modalClosed, refreshed]).then(() => {
                    parentScope.goToEditIfExists(result && result.envName);
                });
            });
        }).error(setErrorInScope.bind($scope));
    }
});

app.controller("AdminCodeEnvsAutomationImportController", function($scope, $state, $stateParams, Assert, TopNav, DataikuAPI, FutureProgressModal, Dialogs, Logs, $q) {
    $scope.newEnv = {}

    $scope.import = function() {
        Assert.trueish($scope.newEnv.file, "No code env file");

        const parentScope = $scope.$parent.$parent;
        DataikuAPI.admin.codeenvs.automation.import($scope.newEnv.file).then(function(data) {
            $scope.dismiss();
            FutureProgressModal.show(parentScope, JSON.parse(data), "Env import", undefined, 'static', false).then(function(result) {
                const modalClosed = result
                    ? Dialogs.infoMessagesDisplayOnly(parentScope, "Creation result", result.messages, result.futureLog, undefined, 'static', false)
                    : $q.resolve();

                const refreshed = parentScope.refreshList();

                $q.all([modalClosed, refreshed]).then(() => {
                    parentScope.goToEditIfExists(result && result.envName);
                });
            });
        }, function(payload) {
            setErrorInScope.bind($scope)(JSON.parse(payload.response), payload.status, function(h) {return payload.getResponseHeader(h)});
        });
    }
});

app.controller("AdminCodeEnvsAutomationImportVersionController", function($scope, $state, $stateParams, Assert, TopNav, DataikuAPI, FutureProgressModal, Dialogs, Logs) {
    $scope.newEnv = {}

    $scope.import = function() {
        Assert.trueish($scope.newEnv.file, "No code env file");

        const parentScope = $scope.$parent.$parent;
        DataikuAPI.admin.codeenvs.automation.importVersion($scope.newEnv.file, $scope.envLang, $scope.envName).then(function(data) {
            $scope.dismiss();
            FutureProgressModal.show(parentScope, JSON.parse(data), "Env import", undefined, 'static', false).then(function(result) {
                if (result) { // undefined in case of abort
                    Dialogs.infoMessagesDisplayOnly(parentScope, "Import result", result.messages, result.futureLog);
                    $scope.addImportedVersion(result.version);
                }
            });
        }, function(payload) {
            setErrorInScope.bind($scope)(JSON.parse(payload.response), payload.status, function(h) {return payload.getResponseHeader(h)});
        });
    }
});

app.controller("_AdminCodeEnvsAutomationEditController", function($scope, $controller, $state, $stateParams, TopNav, DataikuAPI, FutureProgressModal, Dialogs, Logs, CreateModalFromTemplate, $q) {
    $controller("AdminCodeEnvsAutomationController", {$scope:$scope});
    TopNav.setLocation(TopNav.DSS_HOME, "administration");

    $scope.envName = $stateParams.envName;

    $scope.uiState = {
        active : 'info',
        upgradeAllPackages: true,
        updateResources: true,
        forceRebuildEnv: false
     };

     $scope.actionAfterDeletion = function() {
         $state.go("admin.codeenvs-automation.list");
     };

     $scope.canBeUpdated = function() {
         return $scope.codeEnv && $scope.codeEnv.canUpdateCodeEnv && ['DESIGN_MANAGED', 'PLUGIN_MANAGED', 'AUTOMATION_SINGLE', 'DSS_INTERNAL'].indexOf($scope.codeEnv.deploymentMode) >= 0;
     };
     $scope.canVersionBeUpdated = function(versionId) {
         return ['AUTOMATION_VERSIONED'].indexOf($scope.codeEnv.deploymentMode) >= 0 || $scope.canBeUpdated();
     };

     $scope.getSingleVersion = function(codeEnv) {
         if (codeEnv && codeEnv.currentVersion) {
             return codeEnv.currentVersion.versionId;
         } else if (codeEnv && codeEnv.noVersion) {
             return codeEnv.noVersion.versionId;
         } else {
             return null;
         }
     };

     var makeDiffedEnvSettings = function(envSettings) {
        return {
             inheritGlobalSettings: envSettings.inheritGlobalSettings,
             condaCreateExtraOptions: envSettings.condaCreateExtraOptions,
             condaInstallExtraOptions: envSettings.condaInstallExtraOptions,
             pipInstallExtraOptions: envSettings.pipInstallExtraOptions,
             virtualenvCreateExtraOptions: envSettings.virtualenvCreateExtraOptions
        };
     };
     var makeDiffedDesc = function(desc) {
         return {
             yarnPythonBin: desc.yarnPythonBin,
             yarnRBin: desc.yarnRBin,
             dockerImageResources: desc.dockerImageResources,
             updateResourcesApiNode: desc.updateResourcesApiNode,
             corePackagesSet: desc.corePackagesSet,
             envSettings: makeDiffedEnvSettings(desc.envSettings),
             predefinedContainerHooks: angular.copy(desc.predefinedContainerHooks),
             dockerfileAtStart: desc.dockerfileAtStart,
             dockerfileBeforePackages: desc.dockerfileBeforePackages,
             dockerfileAfterCondaPackages: desc.dockerfileAfterCondaPackages,
             dockerfileAfterPackages: desc.dockerfileAfterPackages,
             dockerfileAtEnd: desc.dockerfileAtEnd,
             containerCacheBustingLocation: desc.containerCacheBustingLocation
         };
     };
     var makeDiffedVersion = function(version) {
         return {
             specCondaEnvironment: version.specCondaEnvironment,
             specPackageList: version.specPackageList,
             desc: makeDiffedDesc(version.desc),
             allContainerConfs: version.allContainerConfs,
             containerConfs: version.containerConfs,
             allSparkKubernetesConfs: version.allSparkKubernetesConfs,
             sparkKubernetesConfs: version.sparkKubernetesConfs,
             resourcesInitScript: version.resourcesInitScript
          };
     };
     var makeDiffedSpec = function(codeEnv) {
         var spec = {};
         if (codeEnv) {
             spec.desc = codeEnv.desc;
             spec.externalCondaEnvName = codeEnv.externalCondaEnvName;
             if (codeEnv.currentVersion) {
                 spec.currentVersion = makeDiffedVersion(codeEnv.currentVersion);
             }
             if (codeEnv.noVersion) {
                 spec.noVersion = makeDiffedVersion(codeEnv.noVersion);
             }
             if (codeEnv.versions) {
                 spec.versions = codeEnv.versions.map(function(v) {return makeDiffedVersion(v);});
             }
             spec.permissions = angular.copy(codeEnv.permissions);
             spec.envSettings = angular.copy(codeEnv.envSettings);
             spec.usableByAll = codeEnv.usableByAll;
             spec.owner = codeEnv.owner;
         }
         return spec;
     };

     const getSingleDesc = function(codeEnv) {
        if (codeEnv && codeEnv.currentVersion) {
            return codeEnv.currentVersion.desc;
        } else if (codeEnv && codeEnv.noVersion) {
            return codeEnv.noVersion.desc;
        } else {
            return null;
        }
     }

     $scope.isInternalCodeEnv = function() {
        return $scope.codeEnv && ($scope.codeEnv.deploymentMode === 'DSS_INTERNAL');
     }

     $scope.hasOptedOutOfReferenceSpec = function() {
        const currentDesc = getSingleDesc($scope.codeEnv);
        return $scope.isInternalCodeEnv() && currentDesc && (currentDesc.useReferenceSpec === false);
    }

    $scope.useReferenceSpec = function() {
        const currentDesc = getSingleDesc($scope.codeEnv);
        return $scope.isInternalCodeEnv() && currentDesc && (currentDesc.useReferenceSpec === true);
    }

     $scope.specIsDirty = function() {
         if (!$scope.codeEnv) return false;
         var currentSpec = makeDiffedSpec($scope.codeEnv);
         return !angular.equals(currentSpec, $scope.previousSpec);
     };
     $scope.versionSpecIsDirty = function(versionId) {
         if (!$scope.codeEnv) return false;
         var idx = -1;
         $scope.codeEnv.versions.forEach(function(v, i) {if (v.versionId == versionId) {idx = i;}});
         if (idx < 0) {
             return false;
         } else {
             var currentSpec = makeDiffedVersion($scope.codeEnv.versions[idx]);
             return !angular.equals(currentSpec, $scope.previousSpec.versions[idx]);
         }
     };
    checkChangesBeforeLeaving($scope, $scope.specIsDirty);

    $scope.previousSpec = makeDiffedSpec($scope.codeEnv);

    var listLogs = function(){
        return DataikuAPI.admin.codeenvs.automation.listLogs($scope.envLang, $stateParams.envName).then(function(response) {
            $scope.logs = response.data;
        }).catch(setErrorInScope.bind($scope));
    };

    var refreshEnv = function(){
        const getEnvPromise = DataikuAPI.admin.codeenvs.automation.get($scope.envLang, $stateParams.envName).then(function(response) {
            $scope.codeEnv = response.data;
            $scope.previousSpec = makeDiffedSpec($scope.codeEnv);
        }).catch(setErrorInScope.bind($scope));
        const listLogsPromise = listLogs();
        return $q.all([getEnvPromise, listLogsPromise]);
    }
    var refreshEnvVersion = function(versionId){
        DataikuAPI.admin.codeenvs.automation.getVersion($scope.envLang, $stateParams.envName, versionId).success(function(data) {
            var idx = -1;
            $scope.codeEnv.versions.forEach(function(v, i) {if (v.versionId == versionId) {idx = i;}});
            if (idx >= 0) {
                $scope.codeEnv.versions[idx] = data;
                $scope.previousSpec.versions[idx] = makeDiffedVersion(data);
            }
        }).error(setErrorInScope.bind($scope));
    }
    refreshEnv().then(() => $scope.uiState.forceRebuildEnv = $scope.isInternalCodeEnv());

    $scope.fetchNonManagedEnvDetails = function(){
        DataikuAPI.admin.codeenvs.automation.fetchNonManagedEnvDetails($scope.envLang, $stateParams.envName).success(function(data) {
            $scope.nonManagedEnvDetails = data;
        }).error(setErrorInScope.bind($scope));
    }

    $scope.installJupyterSupport = function(versionId){
        DataikuAPI.admin.codeenvs.automation.installJupyterSupport($scope.envLang, $stateParams.envName, versionId).success(function(data) {
            FutureProgressModal.show($scope, data, "Env update").then(function(result){
                if (result) { // undefined in case of abort
                    Dialogs.infoMessagesDisplayOnly($scope, "Update result", result.messages, result.futureLog);
                }
                if (versionId) {
                    refreshEnvVersion(versionId);
                } else {
                    refreshEnv();
                }
            })
        }).error(setErrorInScope.bind($scope));
    }

    $scope.removeJupyterSupport = function(versionId) {
        DataikuAPI.admin.codeenvs.automation.removeJupyterSupport($scope.envLang, $stateParams.envName, versionId).success(function(data) {
            FutureProgressModal.show($scope, data, "Env update").then(function(result){
                if (result) { // undefined in case of abort
                    Dialogs.infoMessagesDisplayOnly($scope, "Update result", result.messages, result.futureLog);
                }
                if (versionId) {
                    refreshEnvVersion(versionId);
                } else {
                    refreshEnv();
                }
            })
        }).error(setErrorInScope.bind($scope));
    }

    $scope.updateEnvVersion = function(upgradeAllPackages, forceRebuildEnv, updateResources, versionToUpdate) {
        var updateSettings = {
            upgradeAllPackages: upgradeAllPackages,
            forceRebuildEnv: forceRebuildEnv,
            updateResources: updateResources,
            versionToUpdate: versionToUpdate
        }
        DataikuAPI.admin.codeenvs.automation.update($scope.envLang, $stateParams.envName, updateSettings).success(function(data) {
            FutureProgressModal.show($scope, data, "Env update", undefined, 'static', false).then(function(result){
                if (result) { // undefined in case of abort
                    Dialogs.infoMessagesDisplayOnly($scope, "Update result", result.messages, result.futureLog);
                }
                if (versionToUpdate) {
                    refreshEnvVersion(versionToUpdate);
                } else {
                    refreshEnv();
                }
                const broadcastEventParams = versionToUpdate ? {envName: $scope.codeEnv.envName, versionId: versionToUpdate} : {envName: $scope.codeEnv.envName}
                $scope.$broadcast('refreshCodeEnvResources', broadcastEventParams);
            });
        }).error(setErrorInScope.bind($scope));
    }

    // used only in save-update-buttons.html (whenever env is not versioned - behaviour shared with design node):
    // => "Update" button (via direct call to updateEnv() with no arguments)
    // => "Save and update" button (via saveAndMaybePerformChanges(true) - see below)
    $scope.updateEnv = () => $scope.updateEnvVersion(
        $scope.uiState.upgradeAllPackages,
        $scope.uiState.forceRebuildEnv,
        $scope.uiState.updateResources
    )

    $scope.saveAndMaybePerformChanges = function(performChangesOnSave){
        DataikuAPI.admin.codeenvs.automation.save($scope.envLang, $stateParams.envName, $scope.codeEnv).success(function(data) {
            refreshEnv();
            if (performChangesOnSave) {
                // only reachable for non-versioned env so OK to use global update settings
                $scope.updateEnv();
            }
        }).error(setErrorInScope.bind($scope));
    }

    $scope.saveVersionAndMaybePerformChanges = function(performChangesOnSave, upgradeAllPackages, forceRebuildEnv, updateResources, version) {
        DataikuAPI.admin.codeenvs.automation.saveVersion($scope.envLang, $stateParams.envName, version.versionId, version).success(function(data) {
            refreshEnvVersion(version.versionId);
            if (performChangesOnSave) {
                $scope.updateEnvVersion(upgradeAllPackages, forceRebuildEnv, updateResources, version.versionId);
            }
        }).error(setErrorInScope.bind($scope));
    }

    $scope.setContainerConfForAllVersions = function(origVersion) {
        $scope.codeEnv.versions.forEach(function(v) {
            v.desc.allContainerConfs = origVersion.desc.allContainerConfs;
            v.desc.containerConfs = origVersion.desc.containerConfs;
            v.desc.allSparkKubernetesConfs = origVersion.desc.allSparkKubernetesConfs;
            v.desc.sparkKubernetesConfs = origVersion.desc.sparkKubernetesConfs;
        });
    }

    $scope.setResourcesConfForAllVersions = function(origVersion) {
        $scope.codeEnv.versions.forEach(function(v) {
            v.resourcesInitScript = origVersion.resourcesInitScript;
        });
    }

    $scope.getLog = DataikuAPI.admin.codeenvs.automation.getLog;
    $scope.downloadURL = function(envLang, envName, logName) {
        return "/dip/api/code-envs/automation/stream-log?envLang=" + envLang + "&envName=" + encodeURIComponent(envName) + "&logName=" + encodeURIComponent(logName);
    }
    $scope.codeEnvResourcesEditorOptions = $scope.codeMirrorSettingService.get("text/x-python");

    $scope.openImportEnvVersionModal = function(){
        CreateModalFromTemplate("/templates/admin/code-envs/automation/import-env-version-modal.html", $scope, "AdminCodeEnvsAutomationImportVersionController", function(newScope) {
            newScope.envName = $stateParams.envName;
            newScope.envLang = $scope.envLang;
            newScope.addImportedVersion = function(version) {
                if (version == null) return; // aborted? failed?
                // put new version first (bc it's the most recent)
                $scope.codeEnv.versions.splice(0, 0, version);
                $scope.previousSpec.versions.splice(0, 0, makeDiffedVersion(version));
            };
        });
    }
});

app.controller("AdminCodeEnvsAutomationPythonEditController", function($scope, $controller,$state, $stateParams, TopNav, DataikuAPI, FutureProgressModal, Dialogs) {
    $scope.envLang = "PYTHON";
    $controller("_AdminCodeEnvsAutomationEditController", {$scope:$scope});

    DataikuAPI.codeenvs.getUserAccessibleCodeEnvSettings().then(({data}) => {
        $scope.enableCodeEnvResources = data.enableCodeEnvResources;
    });
});

app.controller("AdminCodeEnvsAutomationREditController", function($scope, $controller,$state, $stateParams, TopNav, DataikuAPI, FutureProgressModal, Dialogs) {
    $scope.envLang = "R";
    $controller("_AdminCodeEnvsAutomationEditController", {$scope:$scope});
});

app.directive('pythonVersion', function(DataikuAPI, $state, $stateParams, $rootScope, Deprecation) {
    return {
        restrict : 'A',
        templateUrl : '/templates/admin/code-envs/automation/python-version.html',
        scope : {
                version : '=pythonVersion',
                updateVersion : '&' ,
                saveVersion : '&' ,
                versionSpecIsDirty : '&',
                installJupyterSupport : '&',
                removeJupyterSupport : '&',
                editable : '=',
                withSaveUpdate : '=',
                canVersionBeUpdated : '&'
        },
        link : function($scope, element, attrs) {
            $scope.appConfig = $rootScope.appConfig;
            $scope.uiState = {
                upgradeAllPackages: false,
                updateResources: true,
                forceRebuildEnv: false
            }
            $scope.updateVersionEnv = function() {
                $scope.updateVersion()(
                    $scope.uiState.upgradeAllPackages,
                    $scope.uiState.forceRebuildEnv,
                    $scope.uiState.updateResources,
                    $scope.version.versionId
                );
            }
            $scope.saveVersionEnv = function(performChangesOnSave) {
                $scope.saveVersion()(
                    performChangesOnSave, 
                    $scope.uiState.upgradeAllPackages,
                    $scope.uiState.forceRebuildEnv,
                    $scope.uiState.updateResources,
                    $scope.version
                );
            }
            $scope.installJupyterSupportVersion = function() {
                $scope.installJupyterSupport()($scope.version.versionId);
            }
            $scope.specIsDirty = function() {
                return $scope.versionSpecIsDirty()($scope.version.versionId);
            }
            $scope.canBeUpdated = function() {
                return $scope.canVersionBeUpdated()($scope.version.versionId);
            }
            $scope.removeJupyterSupportVersion = function() {
                $scope.removeJupyterSupport()($scope.version.versionId);
            }
            $scope.refreshMandatoryPackageList = function() {
                if(!$scope.version) return;

                if(!$scope.version.desc.installCorePackages && !$scope.codeEnv.version.installJupyterSupport){
                    $scope.version.mandatoryPackageList = "";
                    $scope.version.mandatoryCondaEnvironment = "";
                    return;
                }

                DataikuAPI.codeenvs.getMandatoryPackages("PYTHON", $scope.version.desc).then(({data}) => {
                    if($scope.version.conda){
                        $scope.version.mandatoryCondaEnvironment = data;
                    }else{
                        $scope.version.mandatoryPackageList = data;
                    }
                });
            }
            
            $scope.isPythonDeprecated = Deprecation.isPythonDeprecated;
        }
    };
});

app.directive('rVersion', function(DataikuAPI, $state, $stateParams, $rootScope) {
    return {
        restrict : 'A',
        templateUrl : '/templates/admin/code-envs/automation/R-version.html',
        scope : {
                version : '=rVersion',
                updateVersion : '&',
                saveVersion : '&' ,
                versionSpecIsDirty : '&',
                installJupyterSupport : '&',
                removeJupyterSupport : '&',
                editable : '=',
                withSaveUpdate : '=',
                canVersionBeUpdated : '&'
        },
        link : function($scope, element, attrs) {
            $scope.appConfig = $rootScope.appConfig;
            $scope.uiState = {
                upgradeAllPackages: false,
                updateResources: true,
                forceRebuildEnv: false
            }
            $scope.updateVersionEnv = function() {
                $scope.updateVersion()(
                    $scope.uiState.upgradeAllPackages,
                    $scope.uiState.forceRebuildEnv,
                    $scope.uiState.updateResources,
                    $scope.version.versionId
                );
            }
            $scope.saveVersionEnv = function(performChangesOnSave) {
                $scope.saveVersion()(
                    performChangesOnSave, 
                    $scope.uiState.upgradeAllPackages,
                    $scope.uiState.forceRebuildEnv,
                    $scope.uiState.updateResources,
                    $scope.version
                );
            }
            $scope.installJupyterSupportVersion = function() {
                $scope.installJupyterSupport()($scope.version.versionId);
            }
            $scope.specIsDirty = function() {
                return $scope.versionSpecIsDirty()($scope.version.versionId);
            }
            $scope.canBeUpdated = function() {
                return $scope.canVersionBeUpdated()($scope.version.versionId);
            }
            $scope.removeJupyterSupportVersion = function() {
                $scope.removeJupyterSupport()($scope.version.versionId);
            }
        }
    };
});

app.directive('containerVersion', function(DataikuAPI, $state, $stateParams, $rootScope, $timeout) {
    return {
        restrict : 'A',
        templateUrl : '/templates/admin/code-envs/automation/container-version.html',
        scope : {
            version : '=containerVersion',
            updateVersion : '&' ,
            saveVersion : '&' ,
            versionSpecIsDirty : '&',
            withSaveUpdate : '=',
            canVersionBeUpdated : '&',
            setForAllVersions: '&',
            envLang: '=',
            deploymentMode: '='
        },
        controller : function($scope) {
            $scope.appConfig = $rootScope.appConfig;
            $scope.uiState = {
                upgradeAllPackages: false,
                updateResources: true,
                forceRebuildEnv: false
            }
            $scope.updateVersionEnv = function() {
                $scope.updateVersion()(
                    $scope.uiState.upgradeAllPackages,
                    $scope.uiState.forceRebuildEnv,
                    $scope.uiState.updateResources,
                    $scope.version.versionId
                );
            }
            $scope.saveVersionEnv = function(performChangesOnSave) {
                $scope.saveVersion()(
                    performChangesOnSave, 
                    $scope.uiState.upgradeAllPackages,
                    $scope.uiState.forceRebuildEnv,
                    $scope.uiState.updateResources,
                    $scope.version
                );
            }
            $scope.specIsDirty = function() {
                return $scope.versionSpecIsDirty()($scope.version.versionId);
            }
            $scope.canBeUpdated = function() {
                return $scope.canVersionBeUpdated()($scope.version.versionId);
            }
            $scope.setThisForAllVersions = function() {
                $timeout(function() {
                    $scope.setForAllVersions()($scope.version);
                });
            }
        }
    };
});


app.directive('resourcesVersion', ['$rootScope', '$timeout', function($rootScope, $timeout) {
    return {
        restrict : 'A',
        templateUrl : '/templates/admin/code-envs/automation/resources-version.html',
        scope : {
            version : '=resourcesVersion',
            envName: '=',
            updateVersion : '&' ,
            saveVersion : '&' ,
            versionSpecIsDirty : '&',
            withSaveUpdate : '=',
            canVersionBeUpdated : '&',
            setForAllVersions: '&',
            codeEnvResourcesEditorOptions: '=editorOptions'
        },
        link : function($scope) {
            $scope.appConfig = $rootScope.appConfig;
            $scope.uiState = {
                upgradeAllPackages: false,
                updateResources: true,
                forceRebuildEnv: false
            }
            $scope.updateVersionEnv = function() {
                $scope.updateVersion()(
                    $scope.uiState.upgradeAllPackages,
                    $scope.uiState.forceRebuildEnv,
                    $scope.uiState.updateResources,
                    $scope.version.versionId
                );
            }
            $scope.saveVersionEnv = function(performChangesOnSave) {
                $scope.saveVersion()(
                    performChangesOnSave, 
                    $scope.uiState.upgradeAllPackages,
                    $scope.uiState.forceRebuildEnv,
                    $scope.uiState.updateResources,
                    $scope.version
                );
            }
            $scope.specIsDirty = function() {
                return $scope.versionSpecIsDirty()($scope.version.versionId);
            }
            $scope.canBeUpdated = function() {
                return $scope.canVersionBeUpdated()($scope.version.versionId);
            }
            $scope.setThisForAllVersions = function() {
                $timeout(function() {
                    $scope.setForAllVersions()($scope.version);
                });
            }
        }
    };
}]);


}());

;
(function(){
'use strict';

var app = angular.module('dataiku.admin.monitoring', []);

app.controller("AdminMonitoringSummaryController", function($scope, $rootScope, $state, DataikuAPI, $filter, $anchorScroll, $timeout){
    $scope.refresh = function refresh(){
        $scope.isLoading = true;
        DataikuAPI.admin.monitoring.getGlobalUsageSummary().success(function(data){
            $scope.globalSummary = data;
            $scope.globalSummary.datasets.allByTypeArray = $filter("toKVArray")($scope.globalSummary.datasets.allByType);
            $scope.globalSummary.recipes.byTypeArray = $filter("toKVArray")($scope.globalSummary.recipes.byType);
            $timeout(() => $anchorScroll());
            $scope.isLoading = false;
        }).error(() => {
            setErrorInScope.bind($scope);
            $scope.isLoading = false;
        });
    }
    $scope.isLoading = false;
});

app.controller("AdminMonitoringWebAppBackendsController", function($scope, $rootScope, $state, DataikuAPI, ActivityIndicator) {
    $scope.refreshList = function(){
        DataikuAPI.webapps.listAllBackendsStates().success(function(data){
            $scope.backends = data;
        }).error(setErrorInScope.bind($scope));
    }

    $scope.stopBackend = function(backend){
        DataikuAPI.webapps.stopBackend({projectKey:backend.projectKey, id:backend.webAppId}).success(function(data){
            ActivityIndicator.success("Backend stopped")
        }).error(setErrorInScope.bind($scope));
    }

    $scope.restartBackend = function(backend){
        DataikuAPI.webapps.restartBackend({projectKey:backend.projectKey, id:backend.webAppId}).success(function(data){
            ActivityIndicator.success("Backend start command sent")
        }).error(setErrorInScope.bind($scope));
    }
    $scope.refreshList();
});

app.controller("AdminMonitoringIntegrationsController", function($scope, DataikuAPI, ActivityIndicator, Dialogs, WT1) {
    $scope.hidePreviewColumn = true;
    $scope.noTags = true;
    $scope.noStar = true;
    $scope.noWatch = true;
    $scope.massDelete = true;
    $scope.massIntegrations = true;
    $scope.noDelete = true;

    $scope.sortBy = [
        { value: 'projectKey', label: 'Project name' },
        { value: 'integrationName', label: 'Integration type' },
        { value: 'integrationActive', label: 'Active' }
    ];
    $scope.selection = $.extend({
        filterQuery: {
            userQuery: '',
            interest: {
                starred: '',
            },
        },
        filterParams: {
            userQueryTargets: ["projectKey", "name", "integrationName", "integrationProperties"],
            propertyRules: { },
        },
        orderQuery: "projectKey",
        orderReversed: false,
    }, $scope.selection || {});
    $scope.sortCookieKey = 'project-integrations';
    $scope.maxItems = 100;

    $scope.list = () => {
        WT1.event("refresh-project-integrations-list");
        DataikuAPI.admin.monitoring.getProjectsIntegrations().success((data) => {
            $scope.integrations = data;
            $scope.listItems = [];
            data.forEach(item => {
                item.integrations.forEach(integration => {
                    $scope.listItems.push({
                        ...integration,
                        integrationName: formatIntegrationName(integration.hook.type),
                        integrationActive: integration.active,
                        integrationDetails: integration.hook.configuration.webhookUrl || undefined,
                        integrationProperties: formatIntegrationProperties(integration),
                        projectKey: item.projectKey
                    });
                });
            });
        }).error(setErrorInScope.bind($scope));
    };
    $scope.list();

    $scope.toggleActive = function(item) {
        WT1.event("integration-save-active");
        DataikuAPI.admin.monitoring.saveProjectIntegration(item.projectKey, item).success(function(data){
            ActivityIndicator.success("Saved");
        }).error(setErrorInScope.bind($scope));
    };

    $scope.deleteIntegration = (item) => {
        WT1.event("integration-delete");
        DataikuAPI.admin.monitoring.deleteProjectIntegration(item.projectKey, item).success(() => {
            ActivityIndicator.success("Saved");
        }).error(setErrorInScope.bind($scope));
    };

    $scope.massDeletion = (items) => {
        if(items.length < 1) return;
        Dialogs.confirm($scope, "Confirm deletion", "Are you sure you want to delete the selected integrations?").then(function() {
            items.forEach((item) => {
                item.active = status;
                $scope.deleteIntegration(item);
                $scope.listItems = $scope.listItems.filter(s => s !== item)
            })
        });
    };

    $scope.allIntegrations = function(objects) {
        if (!objects) return;
        return objects.map(o => o.active).reduce(function(a,b){return a&&b;},true);
    };

    $scope.autoIntegrationsObjects = function(autoIntegrationsStatus, objects) {
        objects.forEach(function(object) {
            if (object.active === autoIntegrationsStatus) return;
            object.active = autoIntegrationsStatus;
            $scope.toggleActive(object);
        })
    };

    const formatIntegrationProperties = (integration) => {
        if(integration.hook.type === "github") {
            return "Repository: " + integration.hook.configuration.repository;
        }

        const labels = [];
        if(integration.hook.configuration.selection.commits)                     { labels.push("Git commits"); }
        if(integration.hook.configuration.selection.discussions)                 { labels.push("Discussions messages"); }
        if(integration.hook.configuration.selection.jobEnds)                     { labels.push("Build jobs ends"); }
        if(integration.hook.configuration.selection.jobStarts)                   { labels.push("Build jobs beginnings"); }
        if(integration.hook.configuration.selection.mlEnds)                      { labels.push("Analysis ML training ends"); }
        if(integration.hook.configuration.selection.mlStarts)                    { labels.push("Analysis ML training beginnings"); }
        if(integration.hook.configuration.selection.scenarioEnds)                { labels.push("Scenario ends"); }
        if(integration.hook.configuration.selection.scenarioStarts)              { labels.push("Scenario starts"); }
        if(integration.hook.configuration.selection.timelineEditionItems)        { labels.push("Objects editions"); }
        if(integration.hook.configuration.selection.timelineItemsExceptEditions) { labels.push("Objects creation / deletion / ..."); }
        if(integration.hook.configuration.selection.watchStar)                   { labels.push("Watch / Star"); }
        return labels.length > 0 ? "Sends on " + labels.join(", ") : "";
    };

    const  formatIntegrationName = (type) => {
        const typeMapping = {
            "msft-teams-project": "Microsoft Teams",
            "google-chat-project": "Google Chat",
            "github": "Github",
            "slack-project": "Slack"
        };
        return typeMapping[type] || type;
    }
});


app.controller("AdminMonitoringClusterTasksController", function($scope, $rootScope, $state, DataikuAPI, Fn, $filter){

	$scope.uiState = {}

    DataikuAPI.admin.connections.list().success(function(data) {
        var array = $filter("toArray")(data);
        var hasHDFS = array.filter(Fn.compose(Fn.prop("type"), Fn.eq("HDFS"))).length > 0;
        $scope.connections = array.filter(function(x){
            return x.type != "HDFS";
        }).map(function(x){
            return { "name"  :x.name , type  : x.type , "id" : x.name }
        });
        if (hasHDFS) {
            $scope.connections.push({
                "name" : "Hadoop",
                "type" : "Hadoop",
                "id" : "HADOOP"
            })
        }
    }).error(setErrorInScope.bind($scope));

    $scope.fetchData = function fetchData(connectionId){
        DataikuAPI.admin.monitoring.getConnectionTasksHistory(connectionId).success(function(data){
            $scope.connectionData = data;

            data.lastTasks.forEach(function(t) {
                t.elapsedTime = t.endTime - t.startTime;
            })

            $scope.projectData = [];
            data.perProject.forEach(function(p) {
                p.types.forEach(function(t, i){
                    $scope.projectData.push(angular.extend(t, {projectKey : p.project, l : p.types.length}));
                });
            });

            $scope.userData = [];
            data.perUser.forEach(function(p) {
                p.types.forEach(function(t, i){
                    $scope.userData.push(angular.extend(t, {initiator : p.user, l : p.types.length}));
                });
            });

        }).error(setErrorInScope.bind($scope));
    }
});

app.controller("AdminMonitoringBackgroundTasksController", function ($scope, DataikuAPI, $rootScope, Dialogs, ProgressStackMessageBuilder) {
    $rootScope.$emit("futureModalOpen");
    $scope.Math = window.Math; // for the display of the running time
    $scope.isLoading = false;
    function isScenarioFuture(future) {
        try {
            return future.payload.action == 'run_scenario';
        } catch (e) { /* Nothing for now */ }
        return false;
    }
    $scope.refreshList = function() {
        $scope.isLoading = true;
        $scope.running = {scenarios:[], futures:[], jobs:[], notebooks:[], clusterKernels:[]};
        DataikuAPI.running.listAll().success(function(data) {
            $scope.running = data;
            //scenario have normal futures, but we put them in another tab
            $scope.running.scenarios = $scope.running.futures.filter(function(f){return isScenarioFuture(f);});
            $scope.running.futures = $scope.running.futures.filter(function(f){return !isScenarioFuture(f);});
            $scope.isLoading = false;
        }).error(() => {
            setErrorInScope.bind($scope)
            $scope.isLoading = false;
        });
    };

    $scope.abortFuture = function(jobId) {
        DataikuAPI.futures.abort(jobId).success(function(data) {
            $scope.refreshList();
        }).error(setErrorInScope.bind($scope));
    };

    $scope.abortNotebook = function(jobId) {
        DataikuAPI.jupyterNotebooks.unload(jobId).success(function(data) {
            $scope.refreshList();
        }).error(setErrorInScope.bind($scope));
    };

    $scope.abortJob = function(projectKey, jobId) {
        DataikuAPI.flow.jobs.abort(projectKey, jobId).success(function(data) {
            $scope.refreshList();
        }).error(setErrorInScope.bind($scope));
    };

    $scope.refreshList();
    $scope.$on('$destroy', function() {
        $rootScope.$emit("futureModalClose");
    });
});


app.controller("AdminMonitoringConnectionDataController", function($scope, $rootScope, $state, DataikuAPI, Fn, $filter, CreateModalFromTemplate, FutureProgressModal, InfoMessagesModal, Dialogs, DatasetsService){

    $scope.uiState = {}

    DataikuAPI.admin.connections.list().success(function(data) {
        var array = $filter("toArray")(data);
        $scope.connections = array.map(function(x){
            return { "name"  :x.name , type  : x.type , "id" : x.name }
        });
    }).error(setErrorInScope.bind($scope));

    $scope.fetchData = function fetchData(connectionId){
        DataikuAPI.admin.monitoring.connectionData.get(connectionId).success(function(data){
            $scope.connectionData = data;
        }).error(setErrorInScope.bind($scope));
    }

    $scope.focusOnProject = function(connectionId, projectKey) {
         DataikuAPI.admin.monitoring.connectionData.getForProject(connectionId, projectKey).success(function(data){
            $scope.projectData = data;
        }).error(setErrorInScope.bind($scope));
    }
    $scope.clearProjectData = function(){
        $scope.projectData = null;
    }

    $scope.updateForProject = function(connectionId, projectKey){
        CreateModalFromTemplate("/templates/admin/monitoring/connection-data-update-confirm.html", $scope, null, function(newScope){
            newScope.settings = {
                computeRecords : false,
                forceRecompute : false
            }
            newScope.go = function(){
                DataikuAPI.admin.monitoring.connectionData.updateForProject(
                    connectionId, projectKey, newScope.settings.computeRecords, newScope.settings.forceRecompute).success(function(data){
                    FutureProgressModal.show($scope, data, "Datasets update").then(function(result){
                        Dialogs.infoMessagesDisplayOnly($scope, "Update result", result);
                        $scope.fetchData(connectionId);
                    })
                    newScope.dismiss();
                });
            }
        });
    }

    $scope.updateSingleDataset = function(connectionId, projectKey, datasetName) {
        CreateModalFromTemplate("/templates/admin/monitoring/connection-data-update-confirm.html", $scope, null, function(newScope){
            newScope.settings = {
                computeRecords : false,
                forceRecompute : false
            }
            newScope.go = function(){
                DataikuAPI.admin.monitoring.connectionData.updateForDataset(
                    projectKey, datasetName, newScope.settings.computeRecords, newScope.settings.forceRecompute).success(function(data){
                    FutureProgressModal.show($scope, data, "Dataset update").then(function(result){
                        InfoMessagesModal.showIfNeeded($scope, result, "Update result");
                        $scope.focusOnProject(connectionId, projectKey);
                    })
                    newScope.dismiss();
                });
            }
        });
    };

    $scope.clearDataset = function(connectionId, projectKey, datasetName) {
        DatasetsService.clear($scope, projectKey, datasetName).then(function() {
            $scope.focusOnProject(connectionId, projectKey);
        });
    };


});


})();
;
(function () {
    "use strict";
    var app = angular.module("dataiku.admin.alertMessageBanner", []);
    const ALERT_MESSAGE_BANNER_ID_KEY = "dss.admin.ALERT_MESSAGE_BANNER_ID_KEY";
    app.component("alertMessageBanner", {
        template: `
            <div ng-if="$ctrl.display && !$root.appConfig.unattendedMode" class="alert alert-warning alert-message-banner children-horizontal-spacing-2x" role="alert">
                <div class="alert-message-banner__header children-horizontal-spacing-2x">
                    <i class="dku-icon-warning-fill-16"></i>
                    <blockquote class="alert-message-banner__title" from-markdown="$ctrl.alertBanner.message"></blockquote>
                </div>
                <button class="btn btn--outline btn--secondary" ng-click="$ctrl.dismiss()">Dismiss</button>
            </div>
        `,
        controller: function ($rootScope, LocalStorage, Notification, Logger, DataikuAPI) {
            this.display = false;
            this.alertBanner = undefined;

            const handleLocalStorageAndDisplay = (alertMessageBanner) => {
                const storedAlertMessageBannerId = LocalStorage.get(ALERT_MESSAGE_BANNER_ID_KEY);
                if (alertMessageBanner && storedAlertMessageBannerId !== alertMessageBanner.id) {
                    LocalStorage.clear(ALERT_MESSAGE_BANNER_ID_KEY);
                    // don't show to the alert banner author
                    if (md5($rootScope.appConfig.login) !== alertMessageBanner.authorId) {
                        this.display = true;
                    }
                } else {
                    this.display = false;
                }
            }

            this.$onInit = function () {
                try {
                    this.alertBanner = $rootScope.appConfig.alertBanner;
                    handleLocalStorageAndDisplay(this.alertBanner);
                } catch (error) {
                    Logger.error("Error handling initialisation of alertMessageBanner.", error)
                    this.display = false;
                }
            };

            Notification.registerEvent("admin-message-banner-changed", (_, message) => {
                this.alertBanner = message.alertBanner;
                handleLocalStorageAndDisplay(message.alertBanner);
            });

            this.dismiss = function () {
                LocalStorage.set(ALERT_MESSAGE_BANNER_ID_KEY, this.alertBanner.id);
                this.display = false;

                // Ignore errors returned by this call: we're just notifying the backend so that it can add a message in audit log
                DataikuAPI.profile.notifyAlertBannerDismissed(this.alertBanner.id);
            };
        },
    });

    app.component("alertMessageBannerForm", {
        bindings: {
            alertBanner: "=",
        },
        template: `
            <form class="dkuform-horizontal">
                <h2 id="alert-message-banner-form">Alert Banner</h2>
                <div class="control-group">
                    <label for="alert-banner-message" class="control-label">Message</label>
                    <div class="controls">
                        <textarea id="alert-banner-message" style="height: 5rem;" ng-model="$ctrl.workingAlertBannerMessage" ng-change="$ctrl.onMessageChange()" />
                        <span class="help-inline">Text of the alert banner. Supports Markdown.</span>
                    </div>
                    <input type="hidden" ng-model="$ctrl.alertBanner.id">
                    <input type="hidden" ng-model="$ctrl.alertBanner.authorId">
                </div>
            </form>
        `,
        controller: function ($rootScope) {
            let previousContent;
            // To avoid updating parent scope alert banner in an uncontrolled way
            this.workingAlertBannerMessage;

            // because $onInit `alertBanner` may be undefined
            this.$doCheck = () => {
                if (this.alertBanner !== undefined && previousContent === undefined) {
                    previousContent = angular.copy(this.alertBanner);
                    this.workingAlertBannerMessage = angular.copy(this.alertBanner.message);
                }
            };

            /**
             * Triggered on message change or message paste
             */
            this.onMessageChange = function () {
                // we don't want to send an empty alert to users
                if (this.workingAlertBannerMessage === "") {
                    this.alertBanner = undefined;
                } else if (previousContent && previousContent.message !== this.workingAlertBannerMessage || previousContent === undefined) {
                    this.alertBanner = {};
                    this.alertBanner.message = this.workingAlertBannerMessage;
                    this.alertBanner.id = md5(this.workingAlertBannerMessage);
                    this.alertBanner.authorId = md5($rootScope.appConfig.login);
                } else {
                    this.alertBanner = previousContent;
                }

            };
        },
    });
})();

;
(function () {
    "use strict";
    var app = angular.module("dataiku.admin.globalApiKeyInput", []);
    /* A <span> that display details about an encrypted api key field, and allow to create and bind a new one. Put it between some <label> and a <span class="help-inline"> */
    app.component("globalApiKeyInput", {
        bindings: {
            apiKey: '=',
            newKeyLabel: '<',       // When creating a new key, the label to use
            newKeyDescription: '<', // When creating a new key, the description to use
        },
        template: `
        <span class="global-api-key-input">
            <input id="apikey"
                type="password"
                ng-model="$ctrl.apiKey"/>
            <span class="overlay-disablable-item-container">
                <span toggle="tooltip" container="body" data-placement="right"
                    title="{{apiKeyDescription}}" >
                    <i ng-if="!apiKeyWarning" class="ng-scope icon-info-sign text-info"></i>
                    <i ng-if="apiKeyWarning" class="ng-scope icon-dku-warning text-warning"></i>
                </span>
                <button class="btn btn--secondary"
                        title="New Global API key"
                        ng-click="createGlobalApiKey()">
                        <i class="icon-plus" ></i>
                </button>
                <span class="overlay-disabled-item"
                    ng-if="!isInstanceAdmin"
                    toggle="tooltip"
                    container="body"
                    title="Only instance administrators can check or create an API key">
                </span>
            </span>
        </div>
        `,
    
        controller: function($scope, $rootScope, $timeout, $filter, DataikuAPI, CreateModalFromTemplate, ClipboardUtils) {
            const $ctrl = this;
            $scope.isInstanceAdmin = $rootScope.appConfig.admin;
            var getKeyDebounceTimeout;
        
            $scope.$watch("$ctrl.apiKey", function(nv) {
                $scope.apiKeyWarning = false;
                $scope.apiKeyDescription = "";
                if (nv && $scope.isInstanceAdmin) {
                    if (getKeyDebounceTimeout) {
                        $timeout.cancel(getKeyDebounceTimeout);
                    }
            
                    getKeyDebounceTimeout = $timeout(function() {
                        DataikuAPI.admin.publicApi.getGlobalKeyDetails(nv).then(function ({data}) {
                            $scope.apiKeyDescription = "Key Label: " + data.label + ", Created on: "+ $filter("friendlyDate")(data.createdOn, 'd MMMM yyyy');
                        }).catch(function (err) {
                            $scope.apiKeyDescription = err.data.detailedMessage;
                            $scope.apiKeyWarning = true;
                        });
                    }, 500); // 500ms delay
                } else {
                    $scope.apiKeyWarning = true;
                    $scope.apiKeyDescription = "No key defined";
                }
            });
    
            $scope.createGlobalApiKey = function() {
                let apiKey = {
                    label : $ctrl.newKeyLabel,
                    description : $ctrl.newKeyDescription,
                    globalPermissions : {admin: true}
                };
                DataikuAPI.admin.publicApi.createGlobalKey(apiKey).success(function (data) {
                    $scope.apiKeyWarning = false;
                    $ctrl.apiKey = data.key;
                    CreateModalFromTemplate("/templates/admin/security/new-api-key-modal.html", $scope, null, function(newScope) {
                        newScope.hashedApiKeysEnabled = $rootScope.appConfig.hashedApiKeysEnabled;
                        newScope.key = data;

                        newScope.copyKeyToClipboard = function() {
                            ClipboardUtils.copyToClipboard(data.key, 'Copied to clipboard.');
                        }
        
                        newScope.viewQRCode = function() {
                            CreateModalFromTemplate("/templates/admin/security/api-key-qrcode-modal.html", $scope, null, function (newScope) {
                                newScope.apiKeyQRCode = JSON.stringify({
                                    k : data.key,
                                    u : $rootScope.appConfig.dssExternalURL
                                });
                            });
                        };
                    });
                }).error(setErrorInScope.bind($scope));
            };
        }
    
    });
})();
    
;
(function() {
'use strict';

const app = angular.module('dataiku.plugins', ['dataiku.filters', 'dataiku.plugindev']);

app.filter('extractGitRefGroup', function() {
    return function(input) {
        return input.split('/', 2)[1] === 'heads' ? "Branch" : "Tag";
    }
});


app.filter('extractGitRefName', function() {
    var namePattern = /(?:.+?\/){2}(.+)$/;
    return function(input) {
        let match;
        if (match = namePattern.exec(input)) {
            return match[1];
        } else {
            return input;
        }
    }
});


app.directive('pluginContributionList', function() {
    return {
        restrict : 'E',
        templateUrl : '/templates/plugins/modals/plugin-contribution-list.html',
        scope : {
            pluginContent : '='
        }
    };
});


app.directive('checkNewPluginIdUnique', function() {
    return {
        require: 'ngModel',
        scope : true,
        link: function(scope, elem, attrs, ngModel) {
            function apply_validation(value) {
                ngModel.$setValidity('uniqueness', true);
                // It is fake, but other check will get it.
                if (value == null || value.length === 0) return value;
                var valid = true;
                if(scope.uniquePluginIds) {
                	valid = scope.uniquePluginIds.indexOf(value) < 0;
                }
                ngModel.$setValidity('uniqueness', valid);
                return value;
            }
            //For DOM -> model validation
            ngModel.$parsers.unshift(apply_validation);

            //For model -> DOM validation
            ngModel.$formatters.unshift(function(value) {
                apply_validation(value);
                return value;
            });
        }
    };
});


app.controller("PluginController", function(
    $scope,
    $controller,
    $state,
    $stateParams,
    DataikuAPI,
    CreateModalFromTemplate,
    SpinnerService,
    TopNav,
    Assert,
    FutureWatcher,
    WT1
){


    TopNav.setLocation(TopNav.DSS_HOME, 'plugin');

    $scope.pluginsUIState = $scope.pluginsUIState || {};

    $controller("PlugindevCommonController", { $scope: $scope });
    $scope.isInstallingOrUpdatingPlugin = function () {
        return $state.includes('plugin.installation') || $state.includes("plugin.installationfromgit") ||
               $state.includes('plugin.update') || $state.includes('plugin.upload') || $state.includes('plugin.upload.update') ||  $state.includes("plugin.updatefromgit")
    };


    $scope.pluginCanBeUninstalled = function() {
        if (!$scope.pluginData) return false;
        return ($scope.pluginData.installedDesc.origin !== 'BUILTIN')
            && ($scope.pluginData.installedDesc.origin !== 'DEV');
    }
    $scope.pluginCanBeMovedToDev = function() {
        if (!$scope.pluginData) return false;
        return $scope.pluginData.installedDesc.origin != 'BUILTIN';
    };

    $scope.pluginCanBeUpdated = function() {
        if (!$scope.pluginData || !$scope.pluginData.storeDesc) return false;
        return $scope.pluginData.storeDesc.storeFlags.downloadable === true;
    };

    $scope.moveToDev = function() {
        CreateModalFromTemplate("/templates/plugins/modals/move-plugin-to-dev.html", $scope, null, function(modalScope) {
            modalScope.go = function() {
                const pluginId = $scope.pluginData.installedDesc.desc.id;
                const pluginVersion = $scope.pluginData.installedDesc.desc.version;
                SpinnerService.lockOnPromise(DataikuAPI.plugins.moveToDev(pluginId)
                    .success(function (data) {
                        if (!data.success){
                            WT1.event("plugin-move-to-dev", { pluginId : pluginId, pluginVersion: pluginVersion, success: "FAILED"});
                            $scope.installationError = data.installationError;
                        } else {
                            WT1.event("plugin-move-to-dev", { pluginId : pluginId, pluginVersion: pluginVersion, success: "DONE" });
                            $state.transitionTo('plugindev.definition', {pluginId});
                        }
                    }).error(setErrorInScope.bind($scope)));
            };
        });
    };

    $scope.getPlugin = function() {
        DataikuAPI.plugins.get($stateParams.pluginId).then(
            function (data) {
                $scope.pluginData = data.data;
                $scope.initContentTypeList($scope.pluginData);
            },
            setErrorInScope.bind($scope)
        );
    };

    $scope.previewUninstallPlugin = function() {
        Assert.trueish($scope.pluginData.installedDesc.origin !== 'BUILTIN', "Plugin is BUILTIN");
        Assert.trueish($scope.pluginData.installedDesc.origin !== 'DEV', "Plugin is DEV");
        var handlePluginDeleted = function(pluginId, pluginVersion) {
            WT1.event("plugin-delete", { pluginId : pluginId, pluginVersion: pluginVersion });
            $state.transitionTo('plugins.installed');
        }
        var handlePluginDeletionFailed = function(data, status, headers) {
            $scope.state = "FAILED";
            $scope.failure = {
                message: getErrorDetails(data, status, headers).detailedMessage
            }
        }

        CreateModalFromTemplate("/templates/plugins/modals/uninstall-plugin-confirm.html", $scope, null, function(newScope) {
            const pluginId = $scope.pluginData.installedDesc.desc.id;
            const pluginVersion = $scope.pluginData.installedDesc.desc.version;

            DataikuAPI.plugins.prepareDelete(pluginId).success(function(usageStatistics) {
                newScope.pluginId = pluginId
                newScope.pluginName = $scope.pluginData.installedDesc.desc.meta.label;
                newScope.usageStatistics = usageStatistics;
                newScope.confirmPluginUninstall = function() {
                    DataikuAPI.plugins.delete(pluginId, true).success(function(initialResponse) {
                        if (initialResponse && initialResponse.jobId && !initialResponse.hasResult) {                        
                            FutureWatcher.watchJobId(initialResponse.jobId).success(function() {
                                handlePluginDeleted(pluginId, pluginVersion);
                            }).error(handlePluginDeletionFailed);
                        } else {
                            handlePluginDeleted(pluginId, pluginVersion);
                        }
                    }).error(handlePluginDeletionFailed);
                }
            });
        });
    };

    $scope.validatePluginEnv = function() {
        $scope.pluginEnvUpToDate = true;
    };

    $scope.invalidatePluginEnv = function() {
        $scope.pluginEnvUpToDate = false;
    };

    if ($scope.isInstallingOrUpdatingPlugin()) {
        $scope.pluginLabel = $stateParams.pluginId;
    }

    if (!$scope.isInstallingOrUpdatingPlugin()) {
        $scope.getPlugin();
    }
});


app.controller("PluginSummaryController", function ($scope, $filter) {

    $scope.filterQuery = { userQuery: '' };
    $scope.filteredContent = {};

    function filterContent(pluginInstallDesc) {
        let filteredContent = {};
        let types = $scope.getComponentsTypeList(pluginInstallDesc);
        types.forEach(function(type) {
            let filteredComponents = $filter('filter')(pluginInstallDesc.content[type], $scope.filterQuery.userQuery);
            if (filteredComponents.length) {
                filteredContent[type] = filteredComponents;
            }
        });
        // Add feature flags as "fake components"
        if (pluginInstallDesc.desc.featureFlags) {
            const matchingFeatureFlag = $filter('filter')(pluginInstallDesc.desc.featureFlags, $scope.filterQuery.userQuery);
            if (matchingFeatureFlag.length > 0) {
                filteredContent['featureFlags'] = $filter('filter')(pluginInstallDesc.desc.featureFlags, $scope.filterQuery.userQuery);
                // Put in the same format as other components for simpler templates
                filteredContent['featureFlags'] = filteredContent['featureFlags'].map(featureFlag => ({ id: featureFlag }));
            }
        }
        return filteredContent;
    }

    function filterContentOnChange() {
        let pluginData = $scope.pluginData;

        if (pluginData && pluginData.installedDesc.content) {
            $scope.filteredContent = filterContent(pluginData.installedDesc);
        } else {
            $scope.filteredContent = {};
        }
    }

    $scope.$watch('pluginData', filterContentOnChange, true);

    $scope.$watch('filterQuery.userQuery', filterContentOnChange, true);

    $scope.getComponentsTypeListFiltered = function() {
        return Object.keys($scope.filteredContent);
    };
});


app.controller("PluginSettingsController", function ($scope, PluginConfigUtils, DataikuAPI, WT1, $stateParams, CreateModalFromTemplate, Dialogs) {

    $scope.pluginsUIState = $scope.pluginsUIState || {};
    $scope.pluginsUIState.settingsPane = $stateParams.selectedTab || 'parameters';

    $scope.hooks = {};

    $scope.setPluginSettings = function (settings) {
        $scope.originalPluginSettings = settings;
        $scope.pluginSettings = angular.copy($scope.originalPluginSettings);
        if ($scope.hooks.settingsSet) {
            $scope.hooks.settingsSet();
        }
    }

    function refreshPluginDesc() {

        if (!$stateParams.pluginId || $scope.installed) {
            return;
        }
        DataikuAPI.plugins.get($stateParams.pluginId, $scope.projectKey).success(function(data) {
            $scope.installed = data.installedDesc;

            if ($scope.installed.desc.params && data.settings.config) {
                PluginConfigUtils.setDefaultValues($scope.installed.desc.params, data.settings.config);
            }
            $scope.setPluginSettings(data.settings);
        }).error(setErrorInScope.bind($scope));
    }

    refreshPluginDesc();

    // Provide a project key to save params at project-level. Defaulty saving at global level
    $scope.savePluginSettings = function(projectKey) {
        DataikuAPI.plugins.saveSettings($scope.pluginData.installedDesc.desc.id, projectKey, $scope.pluginSettings).success(function(data) {

            if (data.error) {
                Dialogs.infoMessagesDisplayOnly($scope, "Update result", data);
            } else {
                // make sure dirtyPluginSettings says it's ok to avoid checkChangesBeforeLeaving complaining
                $scope.originalPluginSettings = angular.copy($scope.pluginSettings);
                $scope.pluginData.settings = $scope.originalPluginSettings;
                WT1.event("plugin-settings-changed", { pluginId : $scope.pluginData.installedDesc.desc.id });
            }
        }).error(setErrorInScope.bind($scope));

    };

    $scope.getParameterSetDesc = function(type) {
        return $scope.installed.customParameterSets.filter(function(parameterSetDesc) {return parameterSetDesc.elementType == type;})[0];
    };

    $scope.getAppTemplateDesc = function(type) {
        return $scope.installed.customAppTemplates.filter(function(appTemplateDesc) {return appTemplateDesc.elementType == type;})[0];
    };

    $scope.deletePreset = function(preset) {
        let index = $scope.pluginSettings.presets.indexOf(preset);
        if (index >= 0) {
            $scope.pluginSettings.presets.splice(index, 1);
        }
    };

    $scope.createPreset = function() {
        CreateModalFromTemplate("/templates/plugins/modals/new-preset.html", $scope, "NewPresetController");
    };

    $scope.isPluginSettingsFormsInvalid = function() {
        return $scope.pluginSettingsForms.$invalid;
    }

    $scope.dirtyPluginSettings = function() {
        return ($scope.originalPluginSettings !== null && !angular.equals($scope.originalPluginSettings, $scope.pluginSettings));
    };

    $scope.$watch("pluginData", function(nv) {
        if (nv && $scope.pluginData && $scope.pluginData.installedDesc.desc && $scope.pluginData.installedDesc.desc.params) {
            PluginConfigUtils.setDefaultValues($scope.pluginData.installedDesc.desc.params, $scope.pluginData.settings.config);
            $scope.pluginSettings = angular.copy($scope.pluginData.settings);
        }
    });


    $scope.presetsByParameterSet = {};
    $scope.hooks.settingsSet = function() {
        $scope.presetsByParameterSet = {};
        $scope.pluginSettings.parameterSets.forEach(function(parameterSet) {$scope.presetsByParameterSet[parameterSet.name] = [];});
        $scope.pluginSettings.presets.forEach(function(preset) {
            let parameterSet = $scope.pluginSettings.parameterSets.filter(function(parameterSet) {return parameterSet.type == preset.type;})[0];
            if (parameterSet) {
                $scope.presetsByParameterSet[parameterSet.name].push(preset);
            }
        });
    };

    checkChangesBeforeLeaving($scope, $scope.dirtyPluginSettings);
});


app.controller("PluginUsagesController", function ($scope, DataikuAPI, StateUtils) {
    $scope.getUsages = function(projectKey) {
        DataikuAPI.plugins.getUsages($scope.pluginData.installedDesc.desc.id, projectKey).success(function(data) {
            $scope.pluginUsages = data;
            $scope.pluginUsages.columns = ['Kind', 'Component', 'Type', 'Project', 'Object'];
            $scope.pluginUsages.columnWidths = [50, 100, 50, 100, 100];
            $scope.pluginUsages.shownHeight = 10 + 25 * (1 + Math.min(10, $scope.pluginUsages.usages.length));
        }).error(setErrorInScope.bind($scope));
    };

    $scope.computeUsageLink = function(usage) {
        return StateUtils.href.dssObject(usage.objectType.toUpperCase(), usage.objectId, usage.projectKey);
    }
});


app.directive('pluginRequirements', function(DataikuAPI, $rootScope, Dialogs, MonoFuture, WT1) {
    return {
        restrict : 'A',
        templateUrl : '/templates/plugins/plugin-requirements.html',
        scope : {
            pluginDesc: '=',
            settings: '=',
            onValid: '=',
            onInvalid: '='
        },
        link : function($scope, element, attrs) {
            $scope.appConfig = $rootScope.appConfig;

            $scope.checkValidity = function() {
                if (!$scope.onValid && !$scope.onInvalid) {
                    return;
                }

                const codeEnvOk = !$scope.pluginDesc.frontendRequirements.codeEnvLanguage || $scope.settings.codeEnvName !== undefined;
                const pythonDepsOK = $scope.pluginDesc.frontendRequirements.pythonPackages.length === 0 || $scope.pluginDesc.frontendRequirements.pythonInstalled;
                const rDepsOk = $scope.pluginDesc.frontendRequirements.rPackages.length === 0 || $scope.pluginDesc.frontendRequirements.rInstalled;
                const customInstallOk = !$scope.pluginDesc.frontendRequirements.installScriptCommand || $scope.pluginDesc.frontendRequirements.customInstalled;

                if (codeEnvOk && pythonDepsOK && rDepsOk && customInstallOk) {
                    $scope.onValid && $scope.onValid();
                } else {
                    $scope.onInvalid && $scope.onInvalid();
                }
            }

            $scope.useCodeEnv = function(envName) {
                DataikuAPI.plugins.useCodeEnv($scope.pluginDesc.desc.id, envName).success(function(data) {
                    WT1.event("plugin-settings-changed", { pluginId : $scope.pluginId });
                    $scope.settings.codeEnvName = envName;
                    $scope.checkValidity();
                }).error(function() {
                    setErrorInScope.apply($scope, arguments);
                    $scope.onInvalid && $scope.onInvalid();
                });
            };

            $scope.codeEnvs = [];
            if ($scope.pluginDesc.codeEnvLang) {
                // otherwise none of this is needed
                $scope.listCodeEnvs = function() {
                    DataikuAPI.codeenvs.listForPlugins($scope.pluginDesc.desc.id).success(function(data) {
                        $scope.codeEnvs = data;
                        $scope.checkValidity();
                    }).error(function() {
                        setErrorInScope.apply($scope, arguments);
                        $scope.onInvalid && $scope.onInvalid();
                    });
                };
                $scope.listCodeEnvs();
            }


            $scope.installingFuture = null;
            function go(type){
                $scope.failure = null;
                $scope.installationLog = null;
                $scope.installationResult = null;
                MonoFuture($scope).wrap(DataikuAPI.plugins.installRequirements)($scope.pluginDesc.desc.id, type).success(function(data) {
                    $scope.state = data.result.success ? "DONE" : "FAILED";
                    WT1.event("plugin-requirement-install", {success : $scope.state, type: type});
                    $scope.installationResult = data.result;
                    $scope.installationLog = data.log;
                    $scope.installingFuture = null;

                    if (data.result.success) {
                        $scope.pluginDesc.frontendRequirements.pythonInstalled = $scope.pluginDesc.frontendRequirements.pythonInstalled || type == 'PYTHON';
                        $scope.pluginDesc.frontendRequirements.rInstalled = $scope.pluginDesc.frontendRequirements.rInstalled || type == 'R';
                        $scope.pluginDesc.frontendRequirements.customInstalled = $scope.pluginDesc.frontendRequirements.customInstalled || type == 'CUSTOM_SCRIPT';
                        $scope.checkValidity();
                    } else {
                        $scope.onInvalid && $scope.onInvalid();
                    }
                }).update(function(data) {
                    $scope.state = "RUNNING";
                    $scope.installingFuture = data;
                    $scope.installationLog = data.log;
                }).error(function (data, status, headers) {
                    $scope.state = "FAILED";
                    if (data.aborted) {
                        $scope.failure = {
                            message: "Aborted"
                        }
                    } else if (data.hasResult) {
                        $scope.installationResult = data.result;
                    } else {
                        $scope.failure = {
                            message: "Unexpected error"
                        }
                    }
                    $scope.installingFuture = null;
                    $scope.onInvalid && $scope.onInvalid();
                });
            }

            $scope.abort = function() {
                $scope.state = "FAILED";
                $scope.failure = {
                    message: "Aborted"
                }
                DataikuAPI.futures.abort($scope.installingFuture.jobId);
                $scope.onInvalid && $scope.onInvalid();
            };

            $scope.installRequirements = function(type) {
                var message = '';
                if ($scope.pluginDesc.frontendRequirements.disclaimer) {
                    message = $scope.pluginDesc.frontendRequirements.disclaimer;
                } else {
                    let envType = '';
                    if (type == 'PYTHON') envType = 'the Python environment on ';
                    if (type == 'R') envType = 'the R environment on ';
                    if (type == 'CUSTOM_SCRIPT') envType = '';
                    message = 'This operation will alter the setup of ' + envType + 'the machine running the DSS server, and cannot be reverted.';
                }
                Dialogs.confirmDisclaimer($scope,'Dependencies installation', 'Are you sure you want to install these dependencies?', message).then(function() {
                    go(type);
                });
            };

            $scope.checkValidity();
        }
    };
});


app.directive('pluginCodeEnv', function(DataikuAPI, Dialogs, WT1, FutureProgressModal, $rootScope, MonoFuture, CodeEnvService) {
    return {
        restrict : 'A',
        templateUrl : '/templates/plugins/modals/plugin-code-env.html',
        scope : {
            pluginDesc : '=',
            settings : '=',
            onValid: '=',
            onInvalid: '='
        },
        link : function($scope, element, attrs) {
            $scope.uiState = {
                state: "DISPLAY"
            }

            $scope.addLicInfo = $rootScope.addLicInfo;

            $scope.newEnv = {
                deploymentMode: 'PLUGIN_MANAGED',
                pythonInterpreter: 'PYTHON39', // should get overriden by getPluginDefaultAvailableInterpreter later on
                allContainerConfs: false, containerConfs : [],
                allSparkKubernetesConfs: false, sparkKubernetesConfs : [],
                rebuildDependentCodeStudioTemplates: "NONE" // we don't expect plugins envs to be used in Code Studios
            };

            $scope.defaultInterpreterFound = false;

            const getPluginDefaultAvailableInterpreter = (codeEnvLang) => {
                if (codeEnvLang !== 'PYTHON'){
                    return;
                }
                return DataikuAPI.codeenvs.getPluginDefaultAvailableInterpreter($scope.pluginDesc.desc.id).success(data => {
                    $scope.newEnv.pythonInterpreter = data;
                    $scope.defaultInterpreterFound = true;
                }).error(function() {
                    // Do not break UI if call failed
                    $scope.defaultInterpreterFound = true;
                    setErrorInScope.apply($scope, arguments);
                });
            }
            $scope.$watch("pluginDesc.codeEnvLang", getPluginDefaultAvailableInterpreter);

            $scope.useCodeEnv = function(envName, nextState) {
                const handleError = (...args) => {
                    setErrorInScope.apply($scope, args);
                    $scope.onInvalid && $scope.onInvalid();
                }
                const handleSuccess = () => {
                    $scope.settings.codeEnvName = envName;
                    $scope.uiState.state = nextState || 'DISPLAY';
                    $scope.onValid && $scope.onValid();
                }
                DataikuAPI.plugins.useCodeEnv($scope.pluginDesc.desc.id, envName).success(function(data) {
                    WT1.event("plugin-settings-changed", { pluginId : $scope.pluginDesc.desc.id });
                    // the data in the response might be a future result, it may happen if async stuff is needed (e.g. rebuild cde image)
                    if (data && data.jobId){
                        FutureProgressModal.show($scope, data, "Rebuilding container image", undefined, 'static', false, true).then(function(result){
                            Dialogs.infoMessagesDisplayOnly($scope, "Build result", result, result.futureLog, undefined, 'static', false);
                            // even if rebuilding container image fails it should not be considered as an error in the UI, what matters is the sync set operation
                            handleSuccess();
                        });
                    } else {
                        handleSuccess();
                    }
                }).error(handleError);
            };

            $scope.containerNames = [];
            DataikuAPI.containers.listNames(null, "USER_CODE").success(function(data){
                $scope.containerNames = data;
            }).error(function () {
                $scope.onInvalid && $scope.onInvalid();
                setErrorInScope.apply($scope, arguments);
            });
            $scope.sparkKubernetesNames = [];
            DataikuAPI.containers.listSparkNames().success(function(data){
                $scope.sparkKubernetesNames = data;
            }).error(function () {
                $scope.onInvalid && $scope.onInvalid();
                setErrorInScope.apply($scope, arguments);
            });

            $scope.codeEnvs = [];
            $scope.listCodeEnvs = function() {
                DataikuAPI.codeenvs.listForPlugins($scope.pluginDesc.desc.id).success(function(data) {
                    $scope.codeEnvs = data;
                    if (!$scope.settings.codeEnvName && $scope.codeEnvs.length) {
                        $scope.uiState.state = "SELECT";
                    } else if (!$scope.settings.codeEnvName) {
                        $scope.uiState.state = "CREATE";
                    }
                }).error(function () {
                    $scope.onInvalid && $scope.onInvalid();
                    setErrorInScope.apply($scope, arguments);
                });
            };
            $scope.listCodeEnvs();

            $scope.isCurrentSelectedEnvUpToDate = function() {
                if (!$scope.settings.codeEnvName) return true;
                var env = $scope.codeEnvs.filter(function(e) {return e.envName == $scope.settings.codeEnvName;})[0];
                if (env == null) return true;
                return env.isUptodate;
            };

            $scope.codeEnvDeploymentModes = [
                ['PLUGIN_MANAGED', "Managed by DSS (recommended)"],
                ['PLUGIN_NON_MANAGED', "Managed manually"]
            ];
            $scope.possiblePythonInterpreters = [];
            if ($scope.pluginDesc.codeEnvSpec) {
                const codeEnvSpec = $scope.pluginDesc.codeEnvSpec;
                if (!codeEnvSpec.forceConda) {
                    $scope.possiblePythonInterpreters = $scope.possiblePythonInterpreters.concat(['CUSTOM']);
                }
                if (codeEnvSpec.acceptedPythonInterpreters && codeEnvSpec.acceptedPythonInterpreters.length > 0) {
                    $scope.possiblePythonInterpreters = $scope.possiblePythonInterpreters.concat(codeEnvSpec.acceptedPythonInterpreters);
                    $scope.newEnv.pythonInterpreter = codeEnvSpec.acceptedPythonInterpreters[0];
                }
            }

            CodeEnvService.getPythonInterpreters().then(function(enrichedInterpreters) {
                const allowed = new Set($scope.possiblePythonInterpreters);
                $scope.enrichedPossiblePythonInterpreters = enrichedInterpreters.filter(i => allowed.has(i[0]));
            });   
            
            $scope.buildNewCodeEnv = function(newEnv) {
                DataikuAPI.codeenvs.createForPlugin($scope.pluginDesc.desc.id, newEnv, true).success(function(data) {
                    FutureProgressModal.show($scope, data, "Environment creation", undefined, 'static', false, true).then(function(result){
                        Dialogs.infoMessagesDisplayOnly($scope, "Creation result", result.messages, result.futureLog, undefined, 'static', false);
                        $scope.listCodeEnvs();
                        // what matters is that the env was properly created, others errors do not really matter
                        const isCodeEnvImported =
                            result.messages
                            && result.messages.messages
                            && result.messages.messages.findIndex(message => message.code === "INFO_CODEENV_IMPORT_OK") > -1;
                        if (result.messages.error && !isCodeEnvImported) {
                            $scope.onInvalid && $scope.onInvalid();
                        } else {
                            if (result.envName){
                                $scope.settings.codeEnvName = result.envName;
                                $scope.uiState.state = 'DISPLAY';
                            }
                            $scope.onValid && $scope.onValid();
                        }
                    });
                }).error(function () {
                    $scope.onInvalid && $scope.onInvalid();
                    setErrorInScope.apply($scope, arguments);
                });
            };
            $scope.updateCodeEnv = function(envName) {
                DataikuAPI.codeenvs.updateForPlugin($scope.pluginDesc.desc.id, envName).success(function(data) {
                    FutureProgressModal.show($scope, data, "Environment update").then(function(result){
                        Dialogs.infoMessagesDisplayOnly($scope, "Update result", result.messages, result.futureLog);
                        $scope.listCodeEnvs();
                    });
                }).error(function () {
                    $scope.onInvalid && $scope.onInvalid();
                    setErrorInScope.apply($scope, arguments);
                });
            };
        }
    };
});


app.controller("PluginsExploreController", function ($rootScope, $scope, $controller, DataikuAPI, $state, Assert, CreateModalFromTemplate, WT1, TopNav, FutureWatcher, FutureProgressModal, Dialogs, RequestCenterService) {
    $controller("PlugindevCommonController", { $scope: $scope });

    TopNav.setLocation(TopNav.DSS_HOME, 'plugins');

    $scope.pluginsUIState = {
        filteredStorePlugins: {},
        filteredInstalledPlugins: {},
        filteredDevelopmentPlugins: {},
        searchQuery: '',
        storeTags: new Map(),
        storeTagsQuery: [],
        storeInstallationStatusQuery: [],
        installedTags: new Map(),
        installedTagsQuery: [],
        developmentTags: new Map(),
        developmentTagsQuery: [],
        showAllStoreTags: false,
        showAllInstalledTags: false,
        showAllDevelopmentTags: false,
        storeSupportLevelQuery: [],
        installedSupportLevelQuery: [],
        storeInstallationStatusCount: new Map(),
        storeSupportLevelsCount: new Map(),
        installedSupportLevelsCount: new Map()
    };

    $scope.pluginsUIState.supportLevels = [
        {
            value: 'SUPPORTED',
            label: 'Supported',
            icon: 'icon-dku-supported'
        },
        {
            value: 'TIER2_SUPPORT',
            label: 'Tier 2 Support',
            icon: 'icon-dku-half-supported'
        }
    ];

    $scope.uploadedPlugin = {
        isUpdate: false
    };

    $scope.clonePlugin = {
        devMode: false,
        bootstrapMode: "GIT_CLONE",
        path: null,
        customCheckout: true,
    };

    function toggleShowAllTags(tab) {
        $scope.pluginsUIState['showAll' + tab + 'Tags'] = !$scope.pluginsUIState['showAll' + tab + 'Tags'];
    }

    function extractPluginIdFromUrl(pluginUri) {
        if (!pluginUri) {
            return "custom-plugin";
        }

        // If it's from our github, getting the plugin name
        const sshPattern = /git@github\.com:dataiku\/dss-plugin-(.*)\.git/;
        const httpPattern = /https:\/\/github\.com\/dataiku\/dss-plugin-(.*)\.git/;
        const sshMatch = pluginUri.match(sshPattern);
        const httpMatch = pluginUri.match(httpPattern);

        if (sshMatch) {
            return sshMatch[1];
        } else if (httpMatch) {
            return httpMatch[1];
        }
        return "custom-plugin";
    }

    $scope.toggleShowAllStoreTags = toggleShowAllTags.bind(this, 'Store');
    $scope.toggleShowAllInstalledTags = toggleShowAllTags.bind(this, 'Installed');
    $scope.toggleShowAllDevelopmentTags = toggleShowAllTags.bind(this, 'Development');

    /*
     *  Accessible either from the preview installation modal or directly from a plugin card in the store
     */
    $scope.installPlugin = function(pluginToInstall, isUpdate) {

        if ($rootScope.appConfig.admin && pluginToInstall) {

            if (isUpdate === true) {
                $state.transitionTo('plugin.update', {
                    pluginId: pluginToInstall.id,
                    pluginVersion: pluginToInstall.storeVersion
                });
            } else {
                $state.transitionTo('plugin.installation', {
                    pluginId: pluginToInstall.id,
                    pluginVersion: pluginToInstall.storeVersion
                });
            }
            $scope.dismiss && $scope.dismiss();
        }
    };

    $scope.triggerRestart = function() {
        DataikuAPI.plugins.triggerRestart().success(function(data) {
            // 'disconnected' will be shown while the backend restarts
        }).error(function() {
            // this is expected, if the backend dies fast enough and doesn't send the response back
        });
    };

    $scope.getStoreSupportLevelCount = function(supportLevelValue) {
        return $scope.pluginsUIState.storeSupportLevelsCount.get(supportLevelValue) || 0;
    }

    $scope.getInstalledSupportLevelCount = function(supportLevelValue) {
        return $scope.pluginsUIState.installedSupportLevelsCount.get(supportLevelValue) || 0;
    }

    $scope.toggleStoreInstallationStatusFilter = function(installationStatus, event) {
        const installationStatusQuery = $scope.pluginsUIState.storeInstallationStatusQuery;
        const statusPosition = installationStatusQuery.indexOf(installationStatus);

        if (statusPosition > -1) {
            installationStatusQuery.splice(statusPosition, 1);
        } else {
            if (installationStatusQuery.length > 0) {
                installationStatusQuery.splice(0, 1);
                event.preventDefault();
            } else {
                installationStatusQuery.push(installationStatus);
            }
        }
    }

    $scope.isStoreInstallationStatusSelected = function(installationStatus) {
        return $scope.pluginsUIState.storeInstallationStatusQuery.includes(installationStatus);
    }

    $scope.getStoreInstallationStatusCount = function(installationStatus) {
            return $scope.pluginsUIState.storeInstallationStatusCount.get(installationStatus) || 0;
        }


    function toggleSupportFilter(supportLevel, tab) {
        const supportLevelQuery = $scope.pluginsUIState[tab + 'SupportLevelQuery'];
        let supportLevelPosition = supportLevelQuery.indexOf(supportLevel);

        if (supportLevelPosition > -1) {
            supportLevelQuery.splice(supportLevelPosition, 1);
        } else {
            supportLevelQuery.push(supportLevel);
        }
    }

    $scope.toggleStoreSupportFilter = function(supportLevel) {
        toggleSupportFilter(supportLevel, 'store');
    }

    $scope.toggleInstalledSupportFilter = function(supportLevel) {
        toggleSupportFilter(supportLevel, 'installed');
    }

    $scope.isStoreSupportLevelSelected = function (supportName) {
        return $scope.pluginsUIState.storeSupportLevelQuery.includes(supportName);
    }

    $scope.isInstalledSupportLevelSelected = function (supportName) {
        return $scope.pluginsUIState.installedSupportLevelQuery.includes(supportName);
    }

    function toggleTagQuery(tagName, tab) {
        const tagsQuery = $scope.pluginsUIState[tab + 'TagsQuery'];
        const tagPosition = tagsQuery.indexOf(tagName);

        if (tagPosition > -1) {
            tagsQuery.splice(tagPosition, 1);
        } else {
            tagsQuery.push(tagName);
        }
    }

    $scope.toggleStoreTagQuery = function(tagName) {
        toggleTagQuery(tagName, 'store');
    };

    $scope.toggleInstalledTagQuery = function(tagName) {
        toggleTagQuery(tagName, 'installed');
    };

    $scope.toggleDevelopmentTagQuery = function(tagName) {
        toggleTagQuery(tagName, 'development');
    };

    $scope.isStoreTagSelected = function (tagName) {
        return $scope.pluginsUIState.storeTagsQuery.includes(tagName);
    }

    $scope.isInstalledTagSelected = function (tagName) {
        return $scope.pluginsUIState.installedTagsQuery.includes(tagName);
    }

    $scope.isDevelopmentTagSelected = function (tagName) {
        return $scope.pluginsUIState.developmentTagsQuery.includes(tagName);
    }

    $scope.resetStoreInstallationStatusQuery = function() {
        $scope.pluginsUIState.storeInstallationStatusQuery = [];
    }

    function resetTagsQuery(tab) {
        $scope.pluginsUIState[tab + 'TagsQuery'] = [];
    }

    $scope.resetStoreTagsQuery = function() {
        resetTagsQuery('store');
    };

    $scope.resetInstalledTagsQuery = function() {
        resetTagsQuery('installed');
    };

    $scope.resetDevelopmentTagsQuery = function() {
        resetTagsQuery('development');
    };

    function resetSupportLevelQuery(tab) {
        $scope.pluginsUIState[tab + 'SupportLevelQuery'] = [];
    }

    $scope.resetStoreSupportLevelQuery = function() {
        resetSupportLevelQuery('store');
    };

    $scope.resetInstalledSupportLevelQuery = function() {
        resetSupportLevelQuery('installed');
    };

    $scope.resetStoreQuery = function() {
        $scope.resetStoreTagsQuery();
        $scope.resetStoreSupportLevelQuery();
        $scope.pluginsUIState.searchQuery = '';
    };

    $scope.resetInstalledQuery = function() {
        $scope.resetInstalledTagsQuery();
        $scope.resetInstalledSupportLevelQuery();
        $scope.pluginsUIState.searchQuery = '';
    };

    $scope.resetDevelopmentQuery = function() {
        $scope.resetDevelopmentTagsQuery();
        $scope.pluginsUIState.searchQuery = '';
    };

    $scope.reloadAllPlugins = function() {
        Dialogs.confirmSimple($scope, 'Reload all plugins?').then(function() {
            DataikuAPI.plugindev.reloadAll()
                .success(_ => $scope.refreshList())
                .error(setErrorInScope.bind($scope));
        });
    };

    function filterPluginsByInstallationStatus() {
        return function(plugin) {
            const installationStatusList = $scope.pluginsUIState.storeInstallationStatusQuery;
            const hasUserFiltered = installationStatusList && installationStatusList.length > 0;

            // No filter checks: show the plugin.
            if (!hasUserFiltered) {
                return true;
            }

            // Else check if the plugin has one of the checked installation status.
            return installationStatusList.includes(plugin.installed);
        }
    }

    $scope.filterStorePluginsByInstallationStatus = filterPluginsByInstallationStatus.bind(this);

    function filterPluginsBySupportLevel(tab, supportKey) {
        return function(plugin) {
            const supportList = $scope.pluginsUIState[tab + 'SupportLevelQuery'];
            const pluginSupportLevel = resolveValue(plugin, supportKey); // from utils.js
            const hasUserFiltered = supportList && supportList.length > 0;
            const hasSupportLevel = pluginSupportLevel && pluginSupportLevel.length >= 0;

            // No filter checks: show the plugin.
            if (!hasUserFiltered) {
                return true;
            }

            // Plugin has no "support" field (and some support level filters are checked): don't show.
            if (!hasSupportLevel) {
                return false;
            }

            // Else check if the plugin has one of the checked support filter.
            return supportList.includes(pluginSupportLevel);
        };
    }

    $scope.filterStorePluginsBySupportLevel = filterPluginsBySupportLevel.bind(this, 'store', 'storeDesc.meta.supportLevel');
    $scope.filterInstalledPluginsBySupportLevel = filterPluginsBySupportLevel.bind(this, 'installed', 'installedDesc.desc.meta.supportLevel');

    function filterPluginsByTags(tab, tagsKey) {
        return function(plugin) {

            const tagsList = $scope.pluginsUIState[tab + 'TagsQuery'];
            const pluginTags = resolveValue(plugin, tagsKey);
            const hasUserFiltered = tagsList && tagsList.length > 0;
            const hasTags = pluginTags && pluginTags.length > 0;

            // No tags checks : show the plugin
            if (!hasUserFiltered) {
                return true;
            }

            // Plugin has no tag (and some tags are checked) : don't show
            if (!hasTags) {
                return false;
            }

            // Else check if the plugin has one of the checked tags
            let containsATag = false;

            pluginTags.forEach(tag => {
                // Prevent duplicates like "Time series" and "Time Series"
                tag = tag.split(' ').map(word => { return word.charAt(0).toUpperCase() + word.slice(1); }).join(' ');

                if (tagsList.includes(tag)) {
                    containsATag = true;
                }
            });

            return containsATag;
        };
    }

    $scope.filterStorePluginsByTags = filterPluginsByTags.bind(this, 'store', 'storeDesc.meta.tags');
    $scope.filterInstalledPluginsByTags = filterPluginsByTags.bind(this, 'installed', 'installedDesc.desc.meta.tags');
    $scope.filterDevelopmentPluginsByTags = filterPluginsByTags.bind(this, 'development', 'installedDesc.desc.meta.tags');

    $scope.hasNoResultsForStoreQuery = function() {
        const isListEmpty = $scope.pluginsUIState.filteredStorePlugins && $scope.pluginsUIState.filteredStorePlugins.length === 0;
        const hasSearched = $scope.pluginsUIState.searchQuery && $scope.pluginsUIState.searchQuery.length;
        const hasFilteredTags = $scope.pluginsUIState.storeTagsQuery && $scope.pluginsUIState.storeTagsQuery.length;
        const hasFilteredSupport = $scope.pluginsUIState.storeSupportLevelQuery && $scope.pluginsUIState.storeSupportLevelQuery.length;
        const hasFiltered = hasSearched || hasFilteredTags || hasFilteredSupport;

        return isListEmpty && hasFiltered;
    }

    $scope.hasNoResultsForInstalledQuery = function() {
        const isListEmpty = $scope.pluginsUIState.filteredInstalledPlugins && $scope.pluginsUIState.filteredInstalledPlugins.length === 0;
        const hasSearched = $scope.pluginsUIState.searchQuery && $scope.pluginsUIState.searchQuery.length;
        const hasFilteredTags = $scope.pluginsUIState.installedTagsQuery && $scope.pluginsUIState.installedTagsQuery.length;
        const hasFilteredSupport = $scope.pluginsUIState.installedSupportLevelQuery && $scope.pluginsUIState.installedSupportLevelQuery.length;
        const hasFiltered = hasSearched || hasFilteredTags || hasFilteredSupport;

        return isListEmpty && hasFiltered;
    }

    $scope.computePluginOrigin = function(plugin) {
        let origin = '';

        if (plugin.inStore === true) {
            origin = 'Store';
        } else if (plugin.installedDesc.origin) {
            if (plugin.installedDesc.origin === 'INSTALLED') {
                if (plugin.installedDesc.gitState.enabled === true) {
                    origin = 'Git repository';
                } else {
                    origin = 'Uploaded';
                }
            } else if (plugin.installedDesc.origin === 'DEV') {
                origin = 'Local dev';
            } else {
                origin = plugin.installedDesc.origin;
            }
        } else {
            origin = plugin.installedDesc.origin;
        }

        return origin;
    };

    $scope.refreshTags = function() {
        $scope.pluginsUIState.storeTags.clear()
        $scope.pluginsUIState.installedTags.clear()
        $scope.pluginsUIState.developmentTags.clear()
        $scope.pluginsUIState.storeSupportLevelsCount.clear();
        $scope.pluginsUIState.installedSupportLevelsCount.clear();
        $scope.pluginsUIState.storeInstallationStatusCount.clear();

        $scope.pluginsList.plugins.forEach(plugin => {
            if (plugin.storeDesc) {
                if ($scope.pluginsUIState.storeInstallationStatusCount.has(plugin.installed)) {
                    $scope.pluginsUIState.storeInstallationStatusCount.set(plugin.installed, $scope.pluginsUIState.storeInstallationStatusCount.get(plugin.installed) + 1);
                } else {
                    $scope.pluginsUIState.storeInstallationStatusCount.set(plugin.installed, 1);
                }

                const supportLevelValue = plugin.storeDesc.meta && plugin.storeDesc.meta.supportLevel;
                if (supportLevelValue) {
                    if ($scope.pluginsUIState.storeSupportLevelsCount.has(supportLevelValue)) {
                        $scope.pluginsUIState.storeSupportLevelsCount.set(supportLevelValue, $scope.pluginsUIState.storeSupportLevelsCount.get(supportLevelValue) + 1);
                    } else {
                        $scope.pluginsUIState.storeSupportLevelsCount.set(supportLevelValue, 1);
                    }
                }

                plugin.storeDesc.meta && plugin.storeDesc.meta.tags && plugin.storeDesc.meta.tags.forEach(tag => {
                    // Prevent duplicates like "Time series" and "Time Series"
                    tag = tag.split(' ').map(word => { return word.charAt(0).toUpperCase() + word.slice(1); }).join(' ');

                    if ($scope.pluginsUIState.storeTags.has(tag)) {
                        $scope.pluginsUIState.storeTags.set(tag, $scope.pluginsUIState.storeTags.get(tag) + 1);
                    } else {
                        $scope.pluginsUIState.storeTags.set(tag, 1);
                    }
                });
            }

            if (plugin.installedDesc) {

                const supportLevelValue = plugin.installedDesc.desc.meta && plugin.installedDesc.desc.meta.supportLevel;

                if (supportLevelValue) {
                    if ($scope.pluginsUIState.installedSupportLevelsCount.has(supportLevelValue)) {
                        $scope.pluginsUIState.installedSupportLevelsCount.set(supportLevelValue, $scope.pluginsUIState.installedSupportLevelsCount.get(supportLevelValue) + 1);
                    } else {
                        $scope.pluginsUIState.installedSupportLevelsCount.set(supportLevelValue, 1);
                    }
                }

                plugin.installedDesc.desc.meta && plugin.installedDesc.desc.meta.tags && plugin.installedDesc.desc.meta.tags.forEach(tag => {
                    // Prevent duplicates like "Time series" and "Time Series"
                    tag = tag.split(' ').map(word => { return word.charAt(0).toUpperCase() + word.slice(1); }).join(' ');

                    if ($scope.pluginsUIState.installedTags.has(tag)) {
                        $scope.pluginsUIState.installedTags.set(tag, $scope.pluginsUIState.installedTags.get(tag) + 1);
                    } else {
                        $scope.pluginsUIState.installedTags.set(tag, 1);
                    }

                    if (plugin.installedDesc.origin === 'DEV') {
                        if ($scope.pluginsUIState.developmentTags.has(tag)) {
                            $scope.pluginsUIState.developmentTags.set(tag, $scope.pluginsUIState.developmentTags.get(tag) + 1);
                        } else {
                            $scope.pluginsUIState.developmentTags.set(tag, 1);
                        }
                    }
                });
            }
        });

        // Sort by descending quantity
        $scope.pluginsUIState.storeTags = new Map([...$scope.pluginsUIState.storeTags.entries()].sort((a, b) => b[1] - a[1]));
        $scope.pluginsUIState.installedTags = new Map([...$scope.pluginsUIState.installedTags.entries()].sort((a, b) => b[1] - a[1]));
        $scope.pluginsUIState.developmentTags = new Map([...$scope.pluginsUIState.developmentTags.entries()].sort((a, b) => b[1] - a[1]));
    };

    $scope.refreshList = function(forceFetch) {
        if ($scope.pluginsList && forceFetch === false) {
            return;
        }

        DataikuAPI.plugins.list(forceFetch).success(function(data) {
            $scope.pluginsList = data;

            $scope.refreshTags();

            $scope.pluginsUIState.storePluginsCount = 0;
            $scope.pluginsUIState.developedPluginsCount = 0;
            $scope.pluginsUIState.installedPluginsCount = 0;

            $scope.pluginsList.plugins.forEach(plugin => {
                if (plugin.storeDesc) {
                    $scope.pluginsUIState.storePluginsCount++;
                }

                if (plugin.installedDesc) {
                    $scope.pluginsUIState.installedPluginsCount++;
                    if (plugin.installedDesc.origin === 'DEV') {
                        plugin.isDevPlugin = true;
                        $scope.pluginsUIState.developedPluginsCount++;
                    }
                    if ( (plugin.installedDesc.origin !== 'BUILTIN') && (plugin.installedDesc.origin !== 'DEV') ) {
                        plugin.uninstallable = true;
                    }
                }
                if (plugin.installed && plugin.installedDesc && plugin.storeDesc && plugin.storeDesc.storeFlags.downloadable === true && plugin.storeDesc.storeVersion && plugin.installedDesc.desc.version && plugin.storeDesc.storeVersion > plugin.installedDesc.desc.version) {
                    plugin.updateAvailable = true;
                }
            });

            const supportLevelMap = {SUPPORTED: 2, TIER2_SUPPORT: 1, NOT_SUPPORTED: 0, NONE: -1};
            function rank(plugin) {
                const updateAvailable = plugin.updateAvailable ? 1 : 0;
                const notInstalled = plugin.installed ? 0 : 1;
                const desc = plugin.storeDesc||{};
                const meta = desc.meta||{};
                const supportLevel = supportLevelMap[meta.supportLevel||'NONE'];
                return [updateAvailable, notInstalled, supportLevel];
            }
            $scope.pluginsList.plugins.sort((a,b) => rank(a) > rank(b) ? -1 : 1);
        }).error(setErrorInScope.bind($scope));
    };

    $scope.previewInstallStorePlugin = function(plugin) {
        Assert.trueish(plugin.inStore, "Plugin not in store");
        CreateModalFromTemplate("/templates/plugins/modals/plugin-install-preview.html", $scope, null, function(newScope) {
            newScope.attachDownloadTo = $scope;
            newScope.isUpdate = plugin.installed;
            if (plugin.installed) {
                newScope.installedVersion = plugin.installedDesc.desc.version;
            }
            newScope.uiState = { activeTab: 'details' };
            $scope.initContentTypeList(plugin);
            newScope.storePlugin = plugin.storeDesc;
            newScope.contentTypes = Object.keys(newScope.storePlugin.content);
        });
    };

    $scope.previewUninstallPlugin = function(plugin) {
        Assert.trueish(plugin.installedDesc.origin !== 'BUILTIN', "Plugin is BUILTIN");
        Assert.trueish(plugin.installedDesc.origin !== 'DEV', "Plugin is DEV");

        var handlePluginDeleted = function(pluginId, pluginVersion) {
            WT1.event("plugin-delete", { pluginId : pluginId, pluginVersion : pluginVersion });
            $scope.reload();
        }

        var handlePluginDeletionFailed = function(data, status, headers) {
            WT1.event("plugin-delete", { pluginId : plugin.id, pluginVersion: plugin.installedDesc.desc.version });
            $scope.reload();
        }


        CreateModalFromTemplate("/templates/plugins/modals/uninstall-plugin-confirm.html", $scope, null, function(newScope) {
            DataikuAPI.plugins.prepareDelete(plugin.id).success(function(usageStatistics) {
                newScope.pluginId = plugin.installedDesc.desc.id;
                newScope.pluginName = plugin.installedDesc.desc.meta.label;
                newScope.usageStatistics = usageStatistics;

                newScope.confirmPluginUninstall = function() {
                    DataikuAPI.plugins.delete(plugin.id, true).success(function(initialResponse) {
                        if (initialResponse && initialResponse.jobId && !initialResponse.hasResult) {                        
                            FutureWatcher.watchJobId(initialResponse.jobId).success(function() {
                                handlePluginDeleted(plugin.id, plugin.installedDesc.desc.version);
                            }).error(handlePluginDeletionFailed);
                        } else {
                            handlePluginDeleted(plugin.id, plugin.installedDesc.desc.version);
                        }
                    }).error(handlePluginDeletionFailed);
            }
            }).error(setErrorInScope.bind($scope));
    
        });
    };

    /** On the scope passed in, set hasPreviousRequest as boolean and latestRequest if there was one, empty object if not */
    function checkForPreviousPluginRequest(modalScope, errorScope, pluginId) {
        //defaults
        modalScope.hasPreviousRequest = false;
        modalScope.latestRequest = {};

        if (modalScope.appConfig.admin) {
            //Don't need the info, defaults are fine
            return;
        }

        DataikuAPI.requests.getLatestRequestForCurrentUser(pluginId, "PLUGIN", "").then(response => {
            if (response.data.status === "PENDING" || (modalScope.queryRequestType == "INSTALL_PLUGIN" && response.data.status === "APPROVED")) {
                modalScope.hasPreviousRequest = true;
                modalScope.latestRequest = response.data;
            }
        }, error => {
            if (error.status != 404){
                setErrorInScope.bind(errorScope)(error);
            }
        });
    }

    $scope.seePluginStoreDetails = function(plugin) {
        Assert.trueish(plugin.inStore, "Plugin not in store");
        $scope.uiState = { activeTab: 'details' };
        
        let changeInProgress;
        $scope.$on('$stateChangeStart', function () {
            changeInProgress = true;
        });
        $scope.$on('$stateChangeSuccess', function () {
            changeInProgress = false;
        });

        const modal = CreateModalFromTemplate("/templates/plugins/modals/plugin-see-details.html", $scope, null, function(newScope) {
            $state.go('plugins.store', Object.assign({}, $state.params, {pluginid: plugin.id}), { notify: false })
            newScope.attachDownloadTo = $scope;
            newScope.isUpdate = plugin.installed;
            newScope.queryRequestType = plugin.installed ? "UPDATE_PLUGIN" : "INSTALL_PLUGIN";
            if (plugin.installed) {
                newScope.installedVersion = plugin.installedDesc.desc.version;
            }

            $scope.initContentTypeList(plugin);
            $scope.plugin = plugin;
            newScope.storePlugin = plugin.storeDesc;

            newScope.contentTypes = Object.keys(newScope.storePlugin.content);
            newScope.isDevPlugin = plugin.installedDesc && plugin.installedDesc.origin === "DEV";

            newScope.advancedLLMMeshAllowed = $rootScope.appConfig.licensedFeatures.advancedLLMMeshAllowed;
            newScope.openAdvancedLLMMeshModal = function(forbiddenFeature) {
                CreateModalFromTemplate('/templates/llm/advanced-llm-mesh-required.html', $rootScope, null, function (modalScope) {
                    modalScope.uiState = {
                        forbiddenFeature: forbiddenFeature,
                    };
                });
            }

            //sets newScope.hasPreviousRequest and newScope.latestRequest
            checkForPreviousPluginRequest(newScope, $scope, plugin.id);
        });
        modal.catch(() => {
            if (!changeInProgress) {
                $state.go('plugins.store', Object.assign({}, $state.params, {pluginid: null}), { notify: false });
            }
        });
    };

    $scope.requestInstallOrUpdatePlugin = function(plugin) {
        Assert.trueish(plugin.inStore, "Plugin not in store");
        CreateModalFromTemplate("/templates/plugins/modals/plugin-install-request.html", $scope, null, function(newScope) {
            newScope.plugin = plugin;
            newScope.ui = { message: ""};
            newScope.requestType = newScope.plugin.installed ? "update" : "install";
            newScope.storePlugin = newScope.plugin.storeDesc;                                
            newScope.queryRequestType = newScope.plugin.installed ? "UPDATE_PLUGIN" : "INSTALL_PLUGIN"; 

            //sets newScope.hasPreviousRequest and newScope.latestRequest
            checkForPreviousPluginRequest(newScope, $scope, plugin.id);

            //Refresh plugin info in case has just been installed
            DataikuAPI.plugins.list(false).success(function(data) {
                // Updating plugin list ensures recent installations are captured on the store page
                $scope.pluginsList = data;
                const latestPlugin = data.plugins.find(aPlugin => aPlugin.id === plugin.id);                                
                //this will capture if the plugin is now installed
                //(but we keep the request type the same as it reflects the original user intention)
                newScope.plugin = latestPlugin;
            }).error(setErrorInScope.bind($scope));

            newScope.isDevPlugin = plugin.installedDesc && plugin.installedDesc.origin === "DEV";
            newScope.sendRequest = (requestMessage) => {
                DataikuAPI.requests.createPluginRequest(newScope.queryRequestType, newScope.plugin.id, newScope.ui.message).success((data) => {
                   RequestCenterService.WT1Events.onRequestSent("PLUGIN", null, newScope.plugin.id, requestMessage, data.id);
                }).error(setErrorInScope.bind($scope));
                newScope.dismiss();
                $rootScope.$broadcast("dismissModals");
            };
        });
    }

    $scope.newZippedPlugin = function() {
        CreateModalFromTemplate('/templates/plugins/modals/new-plugin-from-desktop.html', $scope);
    };

    $scope.newGitPlugin = function() {
        CreateModalFromTemplate('/templates/plugins/modals/new-plugin-from-git.html', $scope);
    };

    $scope.newDevPlugin = function() {
        CreateModalFromTemplate("/templates/plugins/development/new-devplugin.html", $scope);
    };

    $scope.uploadPlugin = function(){
        if ($scope.uploadedPlugin.file && $scope.uploadedPlugin.file != '') {
            let fileName = $scope.uploadedPlugin.file.name;
            if ($scope.uploadedPlugin.isUpdate) {
                $state.transitionTo('plugin.upload.update', {
                    pluginId: fileName,
                    uploadedPluginFile: $scope.uploadedPlugin.file
                });
            } else {
                $state.transitionTo('plugin.upload', {
                    pluginId: fileName,
                    uploadedPluginFile: $scope.uploadedPlugin.file
                });
            }
        }
    };

    $scope.previewPullPlugin = function(plugin) {
        CreateModalFromTemplate("/templates/plugins/modals/plugin-preview-pull.html", $scope, null, function(newScope) {
            newScope.attachDownloadTo = $scope;
            if (plugin.installed) {
                newScope.installedVersion = plugin.installedDesc.desc.version;
            }
            newScope.gitPlugin = angular.copy(plugin.installedDesc);
            newScope.update = () => {
                $state.transitionTo('plugin.updatefromgit', {
                    uri: newScope.gitPlugin.gitState.repository,
                    checkout: newScope.gitPlugin.gitState.checkout,
                    path: newScope.gitPlugin.gitState.path,
                    pluginName: extractPluginIdFromUrl(newScope.gitPlugin.gitState.repository)
                });
            }
        });
    };

    $scope.cloneAndCreate = function() {

        const pluginName = extractPluginIdFromUrl($scope.clonePlugin.uri);

        if ($scope.clonePlugin.devMode) {
            DataikuAPI.plugindev.create('', $scope.clonePlugin.bootstrapMode, $scope.clonePlugin.uri,
                $scope.clonePlugin.checkout, $scope.clonePlugin.path).success(function(data) {
                FutureProgressModal.show($scope, data, "Creating plugin", undefined, 'static', false, true).then(function(result){
                    if (result) {
                        WT1.event("plugin-dev-create", { pluginId : pluginName });
                        $scope.goToDevPluginDetails(result.details);
                    }
                });
            }).error(setErrorInScope.bind($scope));
        } else {
            $state.transitionTo("plugin.installationfromgit", {
                uri: $scope.clonePlugin.uri,
                checkout: $scope.clonePlugin.checkout,
                path: $scope.clonePlugin.path,
                pluginName: pluginName
            })
        }
    };

    $scope.reload = function(){
        location.reload();
    };

    $scope.refreshPluginLists = function() {
        $scope.refreshList(true);
    };

    $scope.goToPluginDetails = function (pluginId) {
        $state.transitionTo('plugin.summary', { pluginId: pluginId });
    }

    $scope.goToDevPluginDetails = function (pluginId) {
        $state.transitionTo('plugindev.definition', { pluginId: pluginId });
    }

    $scope.goToDevPluginEditor = function (pluginId) {
        $state.transitionTo('plugindev.editor', { pluginId: pluginId });
    }

    $scope.deletePluginAndReloadPage = function (pluginDesc) {
        $scope.deletePlugin(pluginDesc.id, pluginDesc.version, $scope.reload);
    }

    // shortcut to force the list refresh
    Mousetrap.bind("r l", function() {
        $scope.refreshList(true);
    });

    $scope.$on("$destroy", function() {
        Mousetrap.unbind("r l");
    });

    $scope.refreshList(false);

    if ($state.is('plugins')) {
        $state.transitionTo('plugins.store');
    }
});

app.controller("PluginsStoreController", ($scope, $stateParams) => {
    let unwatchPluginsList = undefined;
    const displayPlugin = (pluginId = $stateParams.pluginid) => {
        const pluginFound = $scope.pluginsList.plugins.find(p => p.id === pluginId);
        if (pluginFound) {
            $scope.seePluginStoreDetails(pluginFound);
        }
    }

    if ($stateParams.pluginid) {
        if ($scope.pluginsList === undefined) {
            unwatchPluginsList = $scope.$watch('pluginsList', newVal => {
                if (!newVal) {
                    return;
                }
                unwatchPluginsList();
                unwatchPluginsList = undefined;
                displayPlugin();
            });
        } else {
            displayPlugin();
        }
    }

    $scope.$on("$destroy", () => {
        if (unwatchPluginsList) {
            unwatchPluginsList();
        }
    });
});

app.controller("PluginPreviewPullController", function ($rootScope, $scope, Assert, $state) {
    $scope.error = null;

    $scope.go = function() {

        if (!$rootScope.appConfig.admin) {
            return;
        }

        Assert.trueish($scope.gitPlugin, "Git plugin not ready");

        $state.transitionTo('plugin.updatefromgit', {
            uri: $scope.gitPlugin.gitState.repository,
            checkout: $scope.gitPlugin.gitState.checkout,
            path: $scope.gitPlugin.gitState.path
        });
    };
});

app.controller("PluginInstallationController", function ($scope, DataikuAPI, MonoFuture, Fn, WT1, $state, $stateParams, FutureWatcher, ProgressStackMessageBuilder, Logger) {
    $scope.state = "NOT_STARTED";
    $scope.environmentState = 'NOT_STARTED';
    $scope.pluginId = $stateParams.pluginId;
    $scope.pluginVersion = $stateParams.pluginVersion;
    $scope.isGit = $state.includes("plugin.installationfromgit") || $state.includes("plugin.updatefromgit");
    $scope.isUpdate = $state.includes('plugin.update') || $state.includes('plugin.upload.update') || $state.includes("plugin.updatefromgit");
    $scope.isUploadUpdate = $state.includes('plugin.upload.update');
    $scope.isUpload = $state.includes('plugin.upload') || $scope.isUploadUpdate;
    $scope.isCodeEnvDefined = false;
    let uploadedPluginFile = $stateParams.uploadedPluginFile;

    function go() {
        MonoFuture($scope).wrap(DataikuAPI.plugins.install)($scope.pluginId, $scope.isUpdate).success(function(data) {
            if (data && data.log){
                $scope.installationLog = data.log;
            }
            if (!data.result.success){
                $scope.state = "FAILED";
                WT1.event("plugin-download", { pluginId : $scope.pluginId, pluginVersion: $scope.pluginVersion, success : "FAILED" });
                $scope.failure = {
                    message: data.result.installationError.detailedMessage,
                    isAlreadyInstalled: data.result.installationError.detailedMessage.includes('already installed')
                }
                $scope.installingFuture = null;
                return;
            }
            WT1.event("plugin-download", { pluginId: $scope.pluginId, pluginVersion : $scope.pluginVersion, success : "DONE" });
            const confirmInstallation = () => {
                $scope.state = "DONE";
                $scope.pluginDesc = data.result.pluginDesc;
                $scope.$parent.pluginLabel = data.result.pluginDesc.desc.meta.label;
                $scope.pluginSettings = data.result.settings;
                $scope.needsRestart = data.result.needsRestart
                $scope.isCodeEnvDefined = !!($scope.pluginDesc && ($scope.pluginDesc.frontendRequirements.hasDependencies || $scope.pluginDesc.frontendRequirements.codeEnvLanguage));
                $scope.installingFuture = null;
            }
            if (data.result.success && data.result.messages && data.result.messages.warning) {
                // We display confirmation only if main operation is success but some warnings were found
                $scope.state = "WAITING_CONFIRMATION";
                $scope.installationMessages = data.result.messages;
                $scope.confirmInstallation = confirmInstallation;
                return;
            }
            confirmInstallation();
        }).update(function(data) {
            $scope.state = "RUNNING";
            $scope.installingFuture = data;
            $scope.installationLog = data.log;
        }).error(function (data, status, headers) {
            $scope.state = "FAILED";
            if (data.aborted) {
                $scope.failure = {
                    message: "Aborted"
                }
            } else if (data.hasResult) {
                $scope.failure = {
                    message: data.result.errorMessage
                }
            } else {
                $scope.failure = {
                    message: "Unexpected error"
                }
            }
            $scope.installingFuture = null;
        });
    }

    const handleError = (message) => {
        $scope.state = "FAILED";
        $scope.failure = {
            message: message
        }
    }

    const handleAPIError = (data, status, headers) => {
        handleError(getErrorDetails(data, status, headers).detailedMessage);
    };

    const handleFutureError = (data) => {
        let errorMessage;
        if (data && data.result && data.result.installationError && data.result.installationError.detailedMessage){
            errorMessage = data.result.installationError.detailedMessage;
        } else {
            errorMessage = 'An error occured';
        }
        handleError(errorMessage);
    }

    const handleResult = (eventId, pluginId) => (data) => {
        if (data.aborted) return;
        if (data && data.log){
            $scope.installationLog = data.log;
        }
        if (!data || !data.result || !data.result.success){
            WT1.event(eventId, { pluginId: pluginId, success : "FAILED" });
            handleFutureError(data);
            return;
        }
        WT1.event(eventId, { pluginId: pluginId, success : "DONE" });
        const confirmInstallation = () => {
            $scope.state = "DONE";
            $scope.pluginDesc = data.result.pluginDesc;
            $scope.$parent.pluginLabel = data.result.pluginDesc.desc.meta.label;
            $scope.pluginSettings = data.result.settings;
            $scope.needsRestart = data.result.needsRestart
            $scope.isCodeEnvDefined = !!($scope.pluginDesc && ($scope.pluginDesc.frontendRequirements.hasDependencies || $scope.pluginDesc.frontendRequirements.codeEnvLanguage));
            unregisterPluginIdWatcher();
            $scope.pluginId = data.result.pluginDesc.desc.id;
        }
        if (data.result.success && data.result.messages && data.result.messages.warning) {
            // We display confirmation only if main operation is success but some warnings were found
            $scope.state = "WAITING_CONFIRMATION";
            $scope.installationMessages = data.result.messages;
            $scope.confirmInstallation = confirmInstallation;
            return;
        }
        confirmInstallation();
    }

    function goFromGit() {
        $scope.state = 'RUNNING';

        const isUpdate = $state.includes("plugin.updatefromgit");
        const eventId = isUpdate ? "plugin-update-from-git" : "plugin-clone";

        DataikuAPI.plugins.clonePlugin($stateParams.uri, $stateParams.checkout, $stateParams.path, $scope.isUpdate).success(function(data) {
            FutureWatcher.watchJobId(data.jobId)
                .success(handleResult(eventId, $stateParams.pluginName))
                .update(function (data) {
                    $scope.clonePercentage = ProgressStackMessageBuilder.getPercentage(data.progress);
                    $scope.cloneLabel = ProgressStackMessageBuilder.build(data.progress, true);
                    $scope.installingFuture = data;
                    $scope.installationLog = data.log;
                }).error(handleAPIError);
        }).error(handleAPIError);
    }

    function upload () {
        $scope.state = 'RUNNING';
        DataikuAPI.plugins.uploadPlugin(uploadedPluginFile, $scope.isUploadUpdate).then(function(payload) {
            const data = JSON.parse(payload);
            FutureWatcher.watchJobId(data.jobId)
                .success(handleResult("plugin-upload"))
                .update(function (data) {
                    $scope.installingFuture = data;
                    $scope.installationLog = data.log;
                }).error(handleAPIError);
            }, function(payload) {
                let errorMessage;
                try {
                    const parsedResponse = JSON.parse(payload.response);
                    errorMessage = parsedResponse.detailedMessage;
                } catch (exception) {
                    Logger.error(exception);
                    errorMessage = 'An unknown error ocurred.'
                }
                handleError(errorMessage);
            });
    }

    $scope.abort = function() {
        $scope.state = "FAILED";
        $scope.failure = {
            message: "Aborted"
        }
        DataikuAPI.futures.abort($scope.installingFuture.jobId);
    };

    $scope.skipEnvironmentCreation = function() {
        $scope.state = 'DONE';
        $scope.environmentState = 'SKIPPED';
    }

    $scope.approveEnvironmentCreation = function() {
        $scope.environmentState = 'WAITING_CONFIRMATION';
    }

    $scope.disapproveEnvironmentCreation = function() {
        $scope.environmentState = 'NOT_STARTED';
    }

    $scope.confirmEnvironmentCreation = function() {
        $scope.environmentState = 'DONE';
    }

    $scope.goToPluginPage = function() {
        window.location = $state.href('plugin.summary', { pluginId: $scope.pluginId });
    };

    $scope.$on("$destroy", function(){
        if ($scope.state == "RUNNING") {
            $scope.abort();
        }
    });

    $scope.triggerRestart = function() {
        $state.go('plugin.summary', { pluginId: $scope.pluginId }, { reload: true });
        DataikuAPI.plugins.triggerRestart().success(function(data) {
            // 'disconnected' will be shown while the backend restarts
        }).error(function() {
            // this is expected, if the backend dies fast enough and doesn't send the response back
        });
    };

    let unregisterPluginIdWatcher = $scope.$watch("pluginId", Fn.doIfNv($scope.isGit ? goFromGit : ($scope.isUpload ? upload : go)));
});


app.controller("PluginLearnMoreController", function ($scope) {
    $scope.customDatasets = $scope.appConfig.customDatasets.filter(function(x){
        return x.ownerPluginId == $scope.pluginDesc.id;
    });
    $scope.customRecipes = $scope.appConfig.customCodeRecipes.filter(function(x){
        return x.ownerPluginId == $scope.pluginDesc.id;
    });
});

/* Permissions */
app.directive('pluginPresetSecurityPermissions', function(PermissionsService) {
    return {
        restrict : 'A',
        templateUrl : '/templates/plugins/plugin-preset-security-permissions.html',
        scope : {
            preset  : '='
        },
        link : function($scope, element, attrs) {
            $scope.ui = {};

            $scope.securityPermissionsHooks = {};
            $scope.securityPermissionsHooks.makeNewPerm = function() {
                return {
                    use: false
                };
            };
            $scope.securityPermissionsHooks.fixupPermissionItem = function(p) {
                p.$useDisabled = false;
            };
            $scope.securityPermissionsHooks.fixupWithDefaultPermissionItem = function(p, d) {
                if (d.use || d.$useDisabled) {
                    p.$useDisabled = true;
                }
            };

            $scope.$watch("preset.owner", function() {
                $scope.ui.ownerLogin = $scope.preset.owner;
            });

            // Ownership mgmt
            $scope.$watch("ui.ownerLogin", function() {
                PermissionsService.transferOwnership($scope, $scope.preset, "preset");
            });
        }
    };
});


app.directive('pluginParameterSetSecurityPermissions', function() {
    return {
        restrict : 'A',
        templateUrl : '/templates/admin/plugins/plugin-parameter-set-security-permissions.html',
        scope : {
            parameterSet  : '='
        },
        link : function($scope, element, attrs) {
            $scope.securityPermissionsHooks = {};
            $scope.securityPermissionsHooks.makeNewPerm = function() {
                return {
                    definableAtProjectLevel: false,
                    definableInline: false
                };
            };
            $scope.securityPermissionsHooks.fixupPermissionItem = function(p) {
                p.$definableAtProjectLevelDisabled = false;
                p.$definableInlineDisabled = false;
            };
            $scope.securityPermissionsHooks.fixupWithDefaultPermissionItem = function(p, d) {
                if (d.definableAtProjectLevel || d.$definableAtProjectLevelDisabled) {
                    p.$definableAtProjectLevelDisabled = true;
                }
                if (d.definableInline || d.$definableInlineDisabled) {
                    p.$definableInlineDisabled = true;
                }
            };
        }
    };
});


app.directive('pluginSecurityPermissions', function(DataikuAPI, PermissionsService) {
    return {
        restrict : 'A',
        templateUrl : '/templates/plugins/plugin-security-permissions.html',
        scope : {
            pluginSettings : '='
        },
        link : function($scope, element, attrs) {
            $scope.securityPermissionsHooks = {};
            $scope.securityPermissionsHooks.makeNewPerm = function() {
                return {
                    canViewComponents: true
                };
            };
            $scope.securityPermissionsHooks.fixupPermissionItem = function(p) {
                p.$adminDisabled = false;
            };
            $scope.securityPermissionsHooks.fixupWithDefaultPermissionItem = function(p, d) {
                if (d.admin || d.$adminDisabled) {
                    p.$adminDisabled = true;
                }
            };
        }
    };
});


app.directive('securityPermissionsBase', function(DataikuAPI, $rootScope, PermissionsService) {
    return {
        restrict : 'A',
        scope : false,
        link : function($scope, element, attrs) {
            $scope.appConfig = $rootScope.appConfig;

            function makeNewPerm() {
                $scope.newPerm = $scope.securityPermissionsHooks.makeNewPerm();
            }
            const hooksDeregister = $scope.$watch("securityPermissionsHooks", function() {
                if (!$scope.securityPermissionsHooks) return;
                makeNewPerm();
                hooksDeregister();
            }, false);

            const fixupDefaultPermission = function() {
                if (!$scope[attrs.permissionsBearer]) return;
                /* Handle implied permissions */
                $scope.securityPermissionsHooks.fixupPermissionItem($scope[attrs.permissionsBearer].defaultPermission);
            };
            const fixupPermissions = function() {
                if (!$scope[attrs.permissionsBearer] || !$scope.securityPermissionsHooks) return;
                /* Handle implied permissions */
                $scope[attrs.permissionsBearer].permissions.forEach(function(p) {
                    $scope.securityPermissionsHooks.fixupPermissionItem(p);
                    $scope.securityPermissionsHooks.fixupWithDefaultPermissionItem(p, $scope[attrs.permissionsBearer].defaultPermission);
                });
            };

            DataikuAPI.security.listGroups(false).success(function(allGroups) {
                $scope.allGroups = allGroups;
                DataikuAPI.security.listUsers().success(function(data) {
                    $scope.allUsers = data.sort((a, b) => a.displayName.localeCompare(b.displayName));
                    $scope.allUsersLogin = data.map(user => '@' + user.login);
                }).error(setErrorInScope.bind($scope));
                $scope.unassignedGroups = PermissionsService.buildUnassignedGroups($scope[attrs.permissionsBearer], $scope.allGroups);
            }).error(setErrorInScope.bind($scope));

            $scope.addPermission = function() {
                $scope[attrs.permissionsBearer].permissions.push($scope.newPerm);
                makeNewPerm();
            };

            $scope.$watch(attrs.permissionsBearer + ".permissions", function(nv, ov) {
                if (!nv) return;
                $scope.unassignedGroups = PermissionsService.buildUnassignedGroups($scope[attrs.permissionsBearer], $scope.allGroups);
                fixupPermissions();
            }, true)
            $scope.$watch(attrs.permissionsBearer + ".permissions", function(nv, ov) {
                if (!nv) return;
                $scope.unassignedGroups = PermissionsService.buildUnassignedGroups($scope[attrs.permissionsBearer], $scope.allGroups);
                fixupPermissions();
            }, false)
            $scope.$watch(attrs.permissionsBearer + ".defaultPermission", function(nv, ov) {
                if (!nv) return;
                fixupDefaultPermission();
                fixupPermissions();
            }, true)
            $scope.$watch(attrs.permissionsBearer + ".defaultPermission", function(nv, ov) {
                if (!nv) return;
                fixupDefaultPermission();
                fixupPermissions();
            }, false)
            $scope.unassignedGroups = PermissionsService.buildUnassignedGroups($scope[attrs.permissionsBearer], $scope.allGroups);
            fixupPermissions();
        }
    };
});


/* Presets */
app.controller("NewPresetController", function($scope) {

    $scope.presetTypes = $scope.pluginSettings.accessibleParameterSetDescs.map(function(p) {
    	return {id:p.elementType, label:(p.desc.meta.label || p.elementType)};
    });
    $scope.newPreset = {type:$scope.presetTypes[0].id, name:'Preset ' + ($scope.pluginSettings.presets.length + 1), permissions : [], defaultPermission:{}, visibleByAll : true, usableByAll : false, canAdmin : true, canManage : true};
    $scope.presetNameList = $scope.pluginSettings.presets.map(function(p) {return p.name;});

    $scope.create = function(){
    	$scope.pluginSettings.presets.push($scope.newPreset);
        $scope.dismiss();
    }
});


app.controller("NewPresetInParameterSetController", function($scope) {
    $scope.newPreset = {name:'Preset ' + ($scope.allPresets.length + 1), permissions : [], defaultPermission:{}, canAdmin : true, canManage : true};
    $scope.presetNameList = $scope.presets.map(function(p) {return p.name;});

    $scope.create = function() {
        $scope.presets.push($scope.newPreset);
        $scope.allPresets.push($scope.newPreset);
        $scope.dismiss();
    };
});


app.directive('pluginParameterSet', function(CreateModalFromTemplate) {
    return {
        restrict : 'A',
        templateUrl : '/templates/plugins/plugin-parameter-set.html',
        scope : {
            parameterSet : '=',
            parameterSetDesc : '=',
            pluginDesc : '=',
            presets : '=',
            remove : '=',
            allPresets : '='
        },
        link : function($scope, element, attrs) {
            $scope.$watch('allPresets', function() {
                if (!$scope.allPresets) return;
                $scope.presets = $scope.allPresets.filter(function(p) {return p.type == $scope.parameterSet.type;});
            }, true);
            $scope.deletePresetInParameterSet = function(preset) {
                var idxInPlugin = $scope.allPresets.indexOf(preset);
                if (idxInPlugin >= 0) {
                    $scope.allPresets.splice(idxInPlugin, 1);
                }
                var idx = $scope.presets.indexOf(preset);
                if (idx >= 0) {
                    $scope.presets.splice(idx, 1);
                }
            };
            $scope.createPresetInParameterSet = function() {
                CreateModalFromTemplate("/templates/plugins/modals/new-preset.html", $scope, "NewPresetInParameterSetController", function(newScope) {
                    newScope.inParameterSet = true;
                    newScope.newPreset.type = $scope.parameterSet.type;
                });
            };
        }
    };
});


app.directive('pluginPreset', function(DataikuAPI) {
    return {
        restrict : 'A',
        templateUrl : '/templates/plugins/plugin-preset.html',
        scope : {
            preset : '=',
            parameterSetDesc : '=',
            pluginDesc : '=',
            remove : '='
        },
        link : function($scope) {
            DataikuAPI.plugins.canAdminPlugin($scope.pluginDesc.id).success(function(data) {
                $scope.canAdminPlugin = data;
            }).error(setErrorInScope.bind($scope));

        	$scope.$watch('preset', function() {
        		if (!$scope.preset) return;
        		$scope.preset.config = $scope.preset.config || {};
        		$scope.preset.pluginConfig = $scope.preset.pluginConfig || {};
        	});
        }
    };
});

app.directive('pluginAppTemplate', function(DataikuAPI) {
    return {
        restrict : 'A',
        templateUrl : '/templates/plugins/plugin-app-template.html',
        scope : {
            appTemplate : '=',
            appTemplateDesc : '=',
            pluginDesc : '=',
            remove : '='
        },
        link : function($scope, element, attrs) {
    
            $scope.addConnectionRemapping = function(name) {
                $scope.appTemplate.remapping.connections.push({
                    source: name,
                    target: null
                });
            };
            $scope.addCodeEnvRemapping = function(name) {
                $scope.appTemplate.remapping.codeEnvs.push({
                    source: name,
                    target: null
                });
            };
            $scope.addContainerExecRemapping = function(name) {
                $scope.appTemplate.remapping.containerExecs.push({
                    source: name,
                    target: null
                });
            };
            
            $scope.availableConnections = [];
            DataikuAPI.admin.connections.list().success(function(data) {
                angular.forEach(data, function(c, n) {
                    $scope.availableConnections.push({name:n, type:c.type});
                });
            }).error(setErrorInScope.bind($scope));
            
            $scope.availableCodeEnvs = [{envLang:'PYTHON', envName:'Builtin', builtin:true}, {envLang:'R', envName:'Builtin', builtin:true}];
            DataikuAPI.codeenvs.listNames('PYTHON').success(function(data) {
                data.forEach(function(n) {
                    $scope.availableCodeEnvs.push({envLang:'PYTHON', envName:n, builtin:false});
                });
            }).error(setErrorInScope.bind($scope));
            DataikuAPI.codeenvs.listNames('R').success(function(data) {
                data.forEach(function(n) {
                    $scope.availableCodeEnvs.push({envLang:'R', envName:n, builtin:false});
                });
            }).error(setErrorInScope.bind($scope));

            $scope.availableContainerExecConfs = [];
            DataikuAPI.containers.listNames("DOCKER", null).success(data => {
                data.forEach(function(n) {
                    $scope.availableContainerExecConfs.push({type:'DOCKER', name:n});
                });
            }).error(setErrorInScope.bind($scope));
            DataikuAPI.containers.listNames("KUBERNETES", null).success(data => {
                data.forEach(function(n) {
                    $scope.availableContainerExecConfs.push({type:'KUBERNETES', name:n});
                });
            }).error(setErrorInScope.bind($scope));
        }
    };
});



})();

;
(function() {
'use strict';


const app = angular.module('dataiku.admin.clusters', []);


app.controller("ClusterCoreController", function($scope, CreateModalFromTemplate, KubernetesClusterService) {
    $scope.openDeleteClusterModal = function(clusterId, actionAfterDeletion) {
        var newScope = $scope.$new();
        newScope.clusterId = clusterId;
        newScope.actionAfterDeletion = actionAfterDeletion || function(){};
        CreateModalFromTemplate("/templates/admin/clusters/delete-cluster-modal.html", newScope, "ClusterDeleteController");
    }

    $scope.getStateDisplayString = KubernetesClusterService.getStateDisplayString;

    $scope.getStartStopButtonDisplayString = KubernetesClusterService.getStartStopButtonDisplayString;

    $scope.getStateDisplayClass = function(cluster) {
        var classes = {NONE:'icon-off none', RUNNING:'icon-play running', STARTING:'icon-play starting', STOPPING:'icon-unlink stopping', '': 'icon-question unknown'};
        if (cluster.state == null || cluster.type === 'manual') return classes[''];
        return classes[cluster.state];
    }
    $scope.getStateDisplayClassNoIcon = function(cluster) {
        var classes = {NONE:'none', RUNNING:'running', STARTING:'starting', STOPPING:'stopping', '': 'question unknown'};
        if (cluster.state == null || cluster.type === 'manual') return classes[''];
        return classes[cluster.state];
    }

    $scope.getCreatedByString = function(cluster) {
        if (cluster.origin == null) {
            return null;
        } else if (cluster.origin.type === 'MANUAL') {
            return "Created by " + cluster.origin.identifier;
        } else if (cluster.origin.type === 'SCENARIO') {
            return "Created by scenario " + cluster.origin.scenarioProjectKey + "." + cluster.origin.scenarioId;
        } else {
            return null;
        }
    }
});

app.controller("ClusterDeleteController", function($scope, Assert, DataikuAPI, FutureProgressModal) {
    $scope.uiState = {stop:true};

    DataikuAPI.admin.clusters.getStatus($scope.clusterId).success(function(data) {
        $scope.clusterStatus = data;
    }).error(setErrorInScope.bind($scope));

    $scope.doesntNeedStop = function() {
        return $scope.clusterStatus && $scope.clusterStatus.clusterType != 'manual' && $scope.clusterStatus.state == 'NONE';
    };
    $scope.mayNeedStop = function() {
        return $scope.clusterStatus && $scope.clusterStatus.clusterType != 'manual' && $scope.clusterStatus.state != 'NONE';
    };

    $scope.delete = function() {
        Assert.inScope($scope, 'clusterStatus');
        if ($scope.mayNeedStop() && $scope.uiState.stop) {
            var parentScope = $scope.$parent;
            DataikuAPI.admin.clusters.stop($scope.clusterId, true, false).success(function(data){
                $scope.dismiss();
                FutureProgressModal.show(parentScope, data, "Stop cluster", undefined, 'static', 'false').then(function(result){
                    if (result) { // undefined in case of abort
                        $scope.actionAfterDeletion();
                    }
                });
            }).error(setErrorInScope.bind($scope));
        } else {
            DataikuAPI.admin.clusters.delete($scope.clusterId).success(function(data){
                $scope.dismiss();
                $scope.actionAfterDeletion();
            }).error(setErrorInScope.bind($scope));
        }
    };
});

app.controller("ClustersController", function ($scope, $controller, DataikuAPI, CreateModalFromTemplate, TopNav,
    FutureWatcher, KubernetesClusterService, ActivityIndicator, $rootScope, Dialogs) {
    $controller("ClusterCoreController", {$scope:$scope});

    TopNav.setLocation(TopNav.DSS_HOME, "administration");

    $scope.uiState = {query:null};

    $scope.clusters = [];

    $scope.getK8SClusterData = (cluster) => {
        let handleError = () => {
            cluster.data.state = 'ERROR';
        };
        DataikuAPI.admin.clusters.countNodesAndPods(cluster.id).success((initialResponse) => {
                let setResponse = (response) => {
                    cluster.data.count = response.result;
                    cluster.data.state = 'LOADED';
                };
                if (initialResponse.hasResult) {
                    setResponse(initialResponse);
                } else {
                    FutureWatcher.watchJobId(initialResponse.jobId)
                        .success(setResponse)
                        .error(handleError);
                }
            }).error(handleError);
    };

    $scope.getNodesMetrics = (cluster) => {
        const clusterInScope = $scope.clusters.find(el => el.id === cluster.id);

        let handleError = () => {
            clusterInScope.metrics.state = 'ERROR';
        }

        DataikuAPI.admin.clusters.getNodesMetrics(cluster.id).success((initialResponse) => {
            let parseResponse = (response) => {
                clusterInScope.metrics = response.result.reduce((accumulator, current) => {
                    accumulator.cpuUsage += current.cpuUsage;
                    accumulator.memoryUsage += current.memoryUsage;
                    return accumulator;
                }, {cpuUsage: 0, memoryUsage:0});
                clusterInScope.metrics.cpuUsage = clusterInScope.metrics.cpuUsage / response.result.length | 0;
                clusterInScope.metrics.memoryUsage = clusterInScope.metrics.memoryUsage / response.result.length | 0;
                clusterInScope.metrics.state = 'LOADED';
            };
            if (initialResponse.hasResult) {
                parseResponse(initialResponse);
            } else {
                FutureWatcher.watchJobId(initialResponse.jobId)
                    .success(parseResponse)
                    .error(handleError);
            }
        }).error(handleError);
    };

    $scope.refreshList = function() {
        $scope.refreshing = true;
        DataikuAPI.admin.clusters.list().success(function (data) {
            // sorting to display K8S clusters first
            data.sort((el1, el2) => {
                if (el1.architecture === el2.architecture || el1.architecture !== 'KUBERNETES' && el2.architecture !== 'KUBERNETES') {
                    return 0;
                } else {
                    return el1.architecture === 'KUBERNETES' ? -1 : 1;
                }
            });

            data.forEach(el => {
                el.ui = {};
                el['ui']['typeName'] = KubernetesClusterService.getTypeDisplayName(el.type);
                el['ui']['icon'] = KubernetesClusterService.getIcon(el.architecture, el.type);
                el.metricsServer = 'NOT_INSTALLED';
                if ($scope.isK8sAndIsRunning(el)) {
                    if (el.canUpdateCluster) {
                        el.data = { state: 'LOADING' };
                        el.metrics = { state: 'LOADING' };
                        el.metricsServer = 'CHECKING';
                        $scope.getK8SClusterData(el);

                        const handleError = () => {
                            el.metricsServer = 'ERROR';
                            el.metrics = { state: 'ERROR' }
                        }
                        const parseResponse = (response) => {
                            el.metricsServer = response.result ? 'INSTALLED' : 'NOT_INSTALLED';
                            if (el.metricsServer === 'INSTALLED') {
                                $scope.getNodesMetrics(el);
                            }
                        }

                        DataikuAPI.admin.clusters.hasMetricsServer(el.id).success((response) => {
                            if (response.hasResult) {
                                parseResponse(response);
                            } else {
                                return FutureWatcher.watchJobId(response.jobId)
                                    .success(parseResponse)
                                    .error(handleError)
                            }
                        }).error(handleError);

                    } else {
                        el.metricsServer = 'INSUFFICIENT_PERMISSION';
                        el.data = { state: 'INSUFFICIENT_PERMISSION' };
                    }
                }
            });

            $scope.clusters = data;

            // Populating create cluster options for the empty state page
            if (data.length === 0) {
                const basicClusterCreateOption = {
                    title: 'Add Cluster',
                    action: $scope.createCluster
                }

                if ($scope.isPluginInstalled('eks') || $scope.isPluginInstalled('aks') || $scope.isPluginInstalled('gke')) {
                    $scope.createClusterOptions = [];
                    const addPluginOption = (name) => {
                        if ($scope.isPluginInstalled(name)) {
                            const currentOption = {
                                title: `Create ${name.toUpperCase()} cluster`,
                                action : () => $scope.createPluginCluster(name)
                            }
                            if (!$scope.mainCreateOption) {
                                $scope.mainCreateOption = currentOption;
                            } else {
                                $scope.createClusterOptions.push(currentOption)
                            }
                        }
                    }

                    addPluginOption('eks');
                    addPluginOption('aks');
                    addPluginOption('gke');

                    $scope.createClusterOptions.push(basicClusterCreateOption);

                } else {
                    $scope.mainCreateOption = basicClusterCreateOption;
                }

            }
            $scope.refreshing = false;
        }).error(setErrorInScope.bind($scope));
    };

    $scope.filterClusters = function(clusterObject) {
        if (!$scope.uiState.query) return true;
        const query = $scope.uiState.query.toLowerCase();
        return clusterObject && clusterObject.name.toLowerCase().includes(query)
            || clusterObject.ui.typeName && clusterObject.ui.typeName.toLowerCase().includes(query)
            || clusterObject.architecture && clusterObject.architecture.toLowerCase().includes(query);
    };

    $scope.isK8sAndIsRunning = (cluster) => {
        return cluster.architecture === 'KUBERNETES' && (cluster.state === 'RUNNING' || cluster.type === 'manual')
    }

    $scope.isK8sAndIsStopped = (cluster) => {
        return cluster.architecture === 'KUBERNETES' && cluster.state !== 'RUNNING' && cluster.type !== 'manual';
    }

    $scope.refreshList();

    $scope.createCluster = function() {
        CreateModalFromTemplate("/templates/admin/clusters/new-cluster-modal.html", $scope, "NewClusterController")
    };

    $scope.deleteCluster = (clusterId, architecture, type) => {
        KubernetesClusterService.sendWt1Event('clusters-action-delete', architecture, type);
        $scope.openDeleteClusterModal(clusterId, function() {
            $scope.refreshList();
            KubernetesClusterService.sendWt1Event('clusters-status-deleted', architecture, type);
        });
    };

    $scope.downloadClusterDiagnostic = (clusterId, architecture, type) => {
        ActivityIndicator.success("Preparing cluster diagnosis ...");
        KubernetesClusterService.sendWt1Event('clusters-action-diagdownload', architecture, type);
        downloadURL(DataikuAPI.admin.clusters.getDiagnosisURL(clusterId));
        KubernetesClusterService.sendWt1Event('clusters-status-diagdownloaded', architecture, type);
    };

    $scope.markStoppedCluster = (clusterId, architecture, type) => {
        KubernetesClusterService.sendWt1Event('clusters-action-markstopped', architecture, type);
        DataikuAPI.admin.clusters.markStopped(clusterId).success(function(){
            $scope.refreshList();
            KubernetesClusterService.sendWt1Event('clusters-status-markstopped', architecture, type);
        }).error(setErrorInScope.bind($scope));
    };

    $scope.forceStopCluster = (clusterId, architecture, type) => {
        Dialogs.confirm($scope, "Confirm cluster force stop", "Are you sure you want to force stop the cluster?").then(() => {
            KubernetesClusterService.stopCluster(true, $scope, clusterId, $scope.refreshList, architecture, type);
        }, () => {});
    };

    const getPluginCreateClusterType = function(cloud) {
        return "pycluster_" + cloud + "-clusters_create-" + cloud + "-cluster";
    };

    $scope.isPluginInstalled = function(cloud) {
        return $rootScope.appConfig.customPythonPluginClusters.some(
            pluginCluster => pluginCluster.clusterType === getPluginCreateClusterType(cloud));
    };

    $scope.createPluginCluster = function(cloud) {
        CreateModalFromTemplate("/templates/admin/clusters/new-cluster-modal.html", $scope, "NewClusterController", function(newScope) {
            newScope.title = "Create " + cloud.toUpperCase() + " cluster";
            newScope.newCluster = {type: getPluginCreateClusterType(cloud), architecture: 'KUBERNETES', params: {}}
            newScope.typeAlreadyKnown = true;
        });
    };
});

app.controller("NewClusterController", function($scope, $rootScope, $state, DataikuAPI, KubernetesClusterService) {

    $scope.title = $scope.title || 'Add cluster';
    $scope.newCluster = $scope.newCluster || {type:'manual', params: {}};
    $scope.clusterArchitectures = [{id:'HADOOP', label:'Hadoop'}, {id:'KUBERNETES', label:'K8S'}];
    $scope.clusterTypes = [{id:'manual', label:'Non managed', architecture:'MANUAL'}];
    $rootScope.appConfig.customPythonPluginClusters.forEach(function(t) {
        var plugin = Array.dkuFindFn($rootScope.appConfig.loadedPlugins, function (n) {
            return n.id == t.ownerPluginId;
        });
        // Don't add "create cluster" options from our plugins since those have dedicated buttons
        // Don't add them either if the plugin components should not be visible to the user.
        if (plugin != null &&
            !plugin.hideComponents &&
            !['pycluster_eks-clusters_create-eks-cluster',
              'pycluster_aks-clusters_create-aks-cluster',
              'pycluster_gke-clusters_create-gke-cluster'].includes(t.clusterType)) {
            $scope.clusterTypes.push({id:t.clusterType, label:t.desc.meta.label || t.id, architecture:t.desc.architecture || 'HADOOP'})
        }
    });

    $scope.$watch('newCluster.type', function() {
        if ($scope.newCluster.type && $scope.newCluster.type != 'manual') {
            let clusterType = $scope.clusterTypes.filter(function(t) {return $scope.newCluster.type == t.id;})[0];
            $scope.newCluster.architecture = clusterType ? clusterType.architecture : null;
        }
    });

    $scope.create = function(){
        var parentScope = $scope.$parent.$parent;
        KubernetesClusterService.sendWt1Event('clusters-action-create', $scope.newCluster.architecture, $scope.newCluster.type);
        DataikuAPI.admin.clusters.create($scope.newCluster).success(function(data){
            $scope.dismiss();
            parentScope.refreshList();
            $state.go("admin.clusters.cluster", {clusterId:data.id});
            KubernetesClusterService.sendWt1Event('clusters-status-created', data.architecture, data.type);
        }).error(setErrorInScope.bind($scope));
    }
});

app.controller("ClusterController", function($scope, $controller, $stateParams, Assert, DataikuAPI, $state, TopNav, FutureProgressModal, ActivityIndicator, $q,
    Logs, CreateModalFromTemplate, FutureWatcher, KubernetesClusterService, StateUtils, Dialogs) {
    $controller("ClusterCoreController", {$scope:$scope});

    TopNav.setLocation(TopNav.DSS_HOME, "administration");

    $scope.uiState = {
        active: $state.params.action ? 'actions' : 'info',
        logsQuery: '',
        detailedNodeData: false,
        detailedPodData: false
    };

    $scope.cluster = {};
    $scope.origCluster = {};

    $scope.listLogs = function(){
        DataikuAPI.admin.clusters.listLogs($scope.cluster.id).success(function(data) {
            $scope.logs = data;
        }).error(setErrorInScope.bind($scope));
    };

    $scope.refreshItem = function() {
        DataikuAPI.admin.clusters.get($stateParams.clusterId).success(function(data) {
            $scope.cluster = data;
            $scope.origCluster = angular.copy(data);
            $scope.cluster.params = $scope.cluster.params || {};
            $scope.listLogs();
            $scope.refreshStatus();
        }).error(setErrorInScope.bind($scope));
    };
    $scope.refreshItem();

    $scope.getDisplayType = clusterType => KubernetesClusterService.getTypeDisplayName(clusterType);

    $scope.refreshStatus = function() {
        if (!$scope.cluster.canUpdateCluster) return;
        DataikuAPI.admin.clusters.getStatus($stateParams.clusterId).success(function(data) {
            $scope.clusterStatus = data;
        }).error(setErrorInScope.bind($scope));
    };

    const dssObjectColumns = [
        {type: "dss-user", key: 'dssSubmitter', columnName: 'DSS user'},
        {type: "dss-project", key: 'dssProjectKey', columnName: 'DSS project'},
        {type: "dss-object", key: 'dssExecutionType', columnName: 'DSS object'}
    ];

    const podMonitoringColumns = {
        simple : [
            "ready",
            {type:"pod-status", key:'status'},
            ...dssObjectColumns
        ],
        detailed: [
            "ready",
            {type:"pod-status", key:'status'},
            "cpuRequestMillis", "cpuLimitMillis", "cpuCurrentMillis",
            "memoryRequestMB", "memoryLimitMB", "memoryCurrentMB",
            ...dssObjectColumns
        ]
    };

    const nodeMonitoringColumns = {
        simple : [
            {type:"node-status", key:'status'}
        ],
        detailed: [
            {type:"node-status", key:'status'},
            "cpuCapacityMillis", "cpuCurrentMillis", "cpuLoad",
            "memoryCapacityMB", "memoryCurrentMB", "memoryLoad"
        ]
    };

    function calcPercent(value, max){
        const m = parseFloat(max), v = parseFloat(value);
        return isNaN(m) || isNaN(v) || m==0 ? '' : Math.floor(100 * Math.min(1, v/m)) + '%';
    }

    // the angularjs default comparator use the row index
    // when a value is not defined (?!), making weird behaviors.
    // we want a comparison with 0 / "" as default value.
    // we also handle the sorting of percentage and fractions
    const isNumericalRegex = /^\d+%$|^\d+ \/ \d+$/; //match "52%" or "2 / 4"
    $scope.resourceSort = function(a,b){
        if (a.type === 'string' || b.type === 'string') {
            let va = a.value || '';
            let vb = b.value || '';
            if (isNumericalRegex.test(va) || isNumericalRegex.test(vb)) {
                va = parseFloat(va) || 0;
                vb = parseFloat(vb) || 0;
                return va - vb;
            }
            return va.localeCompare(vb);
        } else if (a.type === 'number' || b.type === 'number') {
            return (a.value || 0) - (b.value || 0);
        }
        return 0;
    }

    function buildDssObjectLink(item) {
        if (!item['dssExecutionType'] || !item['onLocalDssNode']) {
            return '';
        }

        const projectKey = item['dssProjectKey'];
        // From com.dataiku.dip.resourceusage.ComputeResourceUsageContext.ComputeResourceUsageContextType
        switch (item['dssExecutionType']) {
            case 'ANALYSIS_ML_TRAIN':
                return StateUtils.href.analysis(item['analysisId'], projectKey, {tab: 'ml.list'});
            case 'WEBAPP_BACKEND':
                return StateUtils.href.webapp(item['webappId'], projectKey, {tab: 'edit'});
            case 'JUPYTER_NOTEBOOK_KERNEL':
                return StateUtils.href.jupyterNotebook(item['notebookId'], projectKey);
            case 'EDA':
                return StateUtils.href.dataset(item['datasetName'], projectKey, {tab: 'statistics'});
            case 'JOB_ACTIVITY':
                return StateUtils.href.job(projectKey, item['jobId']);
            case 'CONTINUOUS_ACTIVITY':
                return StateUtils.href.continuousActivity(item['activityId'], projectKey);
            case 'API_DEPLOYER_DEPLOYMENT':
                return StateUtils.href.apiDeployer.deployment(item['deploymentId']);
            case 'CODE_STUDIO':
                return StateUtils.href.codeStudio(item['codeStudioId'], projectKey);
            default:
                return '';
        }
    }

    function addLinksToObjects(data) {
        data.forEach(item => {
            item.dssLink = buildDssObjectLink(item);
        });
    }

    $scope.clusterObjects = [
        {
            name: "Pod",
            state: 'NONE',
            cols: podMonitoringColumns.simple,
            api: () => DataikuAPI.admin.clusters.monitoring.getPods(
                $stateParams.clusterId,
                $scope.namespaces.selectedMonitoringNamespaceFilter.filterType,
                $scope.namespaces.selectedMonitoringNamespaceFilter.name,
                $scope.uiState.detailedPodData
            ),
            hasNamespace: true,
            parseClusterObjectData: (data) => {
                data.forEach(pod => {
                    pod.ready = `${pod.readyContainerCount} / ${pod.containerCount}`;
                });
                addLinksToObjects(data);
            }
        },
        {
            name: "Job",
            state: 'NONE',
            cols: dssObjectColumns,
            api: () => DataikuAPI.admin.clusters.monitoring.getJobs(
                $stateParams.clusterId,
                $scope.namespaces.selectedMonitoringNamespaceFilter.filterType,
                $scope.namespaces.selectedMonitoringNamespaceFilter.name
            ),
            parseClusterObjectData: addLinksToObjects,
            hasNamespace: true
        },
        {
            name: "Deployment",
            state: 'NONE',
            cols: dssObjectColumns,
            api: () => DataikuAPI.admin.clusters.monitoring.getDeployments(
                $stateParams.clusterId,
                $scope.namespaces.selectedMonitoringNamespaceFilter.filterType,
                $scope.namespaces.selectedMonitoringNamespaceFilter.name
            ),
            parseClusterObjectData: addLinksToObjects,
            hasNamespace: true
        },
        {
            name: "Service",
            state: 'NONE',
            cols: [
                "type",
                ...dssObjectColumns
            ],
            api: () => DataikuAPI.admin.clusters.monitoring.getServices(
                $stateParams.clusterId,
                $scope.namespaces.selectedMonitoringNamespaceFilter.filterType,
                $scope.namespaces.selectedMonitoringNamespaceFilter.name
            ),
            parseClusterObjectData: addLinksToObjects,
            hasNamespace: true
        },
        {
            name: "Node",
            state: 'NONE',
            cols: nodeMonitoringColumns.simple,
            api: () => DataikuAPI.admin.clusters.monitoring.getNodes(
                $stateParams.clusterId,
                $scope.uiState.detailedNodeData
            ),
            hasNamespace: false,
            parseClusterObjectData: (data) => {
                data.forEach(node => {
                    node.cpuLoad = calcPercent(node.cpuCurrentMillis, node.cpuCapacityMillis);
                    node.memoryLoad = calcPercent(node.memoryCurrentMB, node.memoryCapacityMB);
                });
            }
        }
    ];

    $scope.namespaces = {
        loading: false,
        loaded: false,
        monitoringNamespaceFilterOptions: [
            {name: 'None specified (kubectl default)', filterType: 'NONE'},
            {name: 'All', filterType: 'ALL'},
            {name: 'All but kube-system', filterType: 'ALL_BUT_SYSTEM'} // Default to match what is seen on tile
        ],
        parseClusterObjectData: (data) => {
            $scope.namespaces.monitoringNamespaceFilterOptions =
                $scope.namespaces.monitoringNamespaceFilterOptions.filter(namespace => namespace.filterType !== 'LOADING');
            $scope.namespaces.loaded = true;
            data.forEach(namespace =>
                $scope.namespaces.monitoringNamespaceFilterOptions.push({name: namespace, filterType: 'EXPLICIT'}));
        },
        api: () => DataikuAPI.admin.clusters.monitoring.getNamespaces($stateParams.clusterId)
    }
    $scope.namespaces.selectedMonitoringNamespaceFilter = $scope.namespaces.monitoringNamespaceFilterOptions[2];

    $scope.$watch("uiState.detailedPodData", function(nv, ov) {
        if (nv === true) {
            // Dirty
            $scope.clusterObjects[0].cols = podMonitoringColumns.detailed;
            $scope.refreshMonitoring();
        } else if (nv=== false) {
            $scope.clusterObjects[0].cols = podMonitoringColumns.simple;
        }
    });

    $scope.$watch("uiState.detailedNodeData", function(nv, ov) {
        if (nv) {
            // Dirty
            $scope.clusterObjects[4].cols = nodeMonitoringColumns.detailed;
            $scope.refreshMonitoring();
        } else {
            $scope.clusterObjects[4].cols = nodeMonitoringColumns.simple;
        }
    });

    $scope.getClusterObjectData = (clusterObject) => {
        if (!$scope.cluster.canUpdateCluster) return;
        clusterObject.state = 'NONE';

        const parseResponse = (data) => {
            clusterObject.state = 'LOADED';
            if (clusterObject.parseClusterObjectData) {
                clusterObject.parseClusterObjectData(data);
            }
            clusterObject.data = data;
        }

        let handleError = (data, status, headers) => {
            clusterObject.state = 'ERROR';
            setErrorInScope.bind($scope)(data, status, headers);
        }

        clusterObject.api().success(function(initialResponse) {
            if (initialResponse.hasResult) {
                parseResponse(initialResponse.result);
            } else {
                clusterObject.state = 'LOADING';
                FutureWatcher.watchJobId(initialResponse.jobId)
                    .success(data => parseResponse(data.result))
                    .error(handleError);
            }
        }).error(handleError);
    }

    $scope.goToClusterObjectMonitoring = function(clusterObject) {
        $scope.uiState.activeMonitoring = clusterObject.name
        if (($scope.cluster.type === 'manual' || $scope.cluster.state === 'RUNNING')) {
            $scope.getClusterObjectData(clusterObject);
        }
        KubernetesClusterService.sendWt1Event(`clusters-monitoring-${clusterObject.name}s-viewed`.toLowerCase(),
                                              $scope.cluster.architecture, $scope.cluster.type);
    }

    $scope.refreshMonitoring = function() {
        let clusterObject = $scope.clusterObjects.find((item) => item.name === $scope.uiState.activeMonitoring);
        $scope.getClusterObjectData(clusterObject);
    }

    $scope.getNamespaces = function() {
        if (($scope.cluster.type === 'manual' || $scope.cluster.state === 'RUNNING') && !$scope.namespaces.loading && !$scope.namespaces.loaded) {
            $scope.namespaces.monitoringNamespaceFilterOptions.push({name: 'Loading cluster namespaces...', filterType: 'LOADING'});
            $scope.getClusterObjectData($scope.namespaces);
        }
    }

    $scope.setMonitoringNamespaceFilter = function(nv) {
        $scope.namespaces.selectedMonitoringNamespaceFilter = nv;
        $scope.refreshMonitoring();
    }

    $scope.filterMonitoring = function(clusterObject) {
        if (!$scope.uiState.monitoringQuery) return true;
        const query = $scope.uiState.monitoringQuery.toLowerCase();
        return clusterObject && clusterObject.name.toLowerCase().includes(query)
            || clusterObject.status && clusterObject.status.toLowerCase().includes(query)
            || clusterObject.namespace && clusterObject.namespace.toLowerCase().includes(query)
            || clusterObject.type && clusterObject.type.toLowerCase().includes(query)
            || clusterObject.dssSubmitter && clusterObject.dssSubmitter.toLowerCase().includes(query)
            || clusterObject.dssExecutionType && clusterObject.dssExecutionType.toLowerCase().includes(query)
            || clusterObject.dssProjectKey && clusterObject.dssProjectKey.toLowerCase().includes(query);
    };

    $scope.showObjectDescribe = function(name, objectType, namespace) {
        CreateModalFromTemplate('/templates/admin/clusters/k8s-object-describe-modal.html', $scope, null, function(newScope) {
            newScope.uiState = {
                mode: "describe"
            };
            newScope.objectName = name;
            newScope.objectType = objectType;
            newScope.clusterId = $scope.cluster.id;
            newScope.namespace = namespace;
            newScope.describeObject();
        });
    };

    $scope.clusterIsDirty = function() {
        if (!$scope.cluster || !$scope.origCluster) return false;
        return !angular.equals($scope.cluster, $scope.origCluster);
    };

    checkChangesBeforeLeaving($scope, $scope.clusterIsDirty);

    $scope.saveCluster = function() {
        var deferred = $q.defer();
        if (!$scope.clusterIsDirty()) { // for when it's called with a keystroke or from start button
            deferred.resolve("Saved");
            return deferred.promise;
        }
        KubernetesClusterService.sendWt1Event('clusters-action-save', $scope.cluster.architecture, $scope.cluster.type);
        DataikuAPI.admin.clusters.save(angular.copy($scope.cluster)).success(function(data) {
            $scope.cluster = data;
            $scope.origCluster = angular.copy(data);
            deferred.resolve("Saved");
            KubernetesClusterService.sendWt1Event('clusters-status-saved',
                                                  $scope.cluster.architecture, $scope.cluster.type);
        }).error(function (a,b,c) {
            setErrorInScope.bind($scope)(a,b,c);
            deferred.reject("Not saved");
        });
        return deferred.promise;
    };

    $scope.deleteCluster = function() {
        KubernetesClusterService.sendWt1Event('clusters-action-delete',
                                              $scope.cluster.architecture, $scope.cluster.type);
        $scope.openDeleteClusterModal($scope.cluster.id, function() {
            $state.go("admin.clusters.list");
            KubernetesClusterService.sendWt1Event('clusters-status-deleted',
                                                  $scope.cluster.architecture, $scope.cluster.type);
        });
    };

    const doStartCluster = function(){
        Assert.inScope($scope, 'cluster');
        KubernetesClusterService.sendWt1Event('clusters-action-start',
                                              $scope.cluster.architecture, $scope.cluster.type);
        return DataikuAPI.admin.clusters.start($scope.cluster.id)
            .success(
                data => FutureProgressModal.show($scope, data, "Start cluster", undefined, 'static', 'false')
                .then(function(result){
                    if (result) { // undefined in case of abort
                        $scope.refreshItem();
                        KubernetesClusterService.sendWt1Event('clusters-status-started',
                                                            $scope.cluster.architecture, $scope.cluster.type);
                    }
                })
            )
            .error(setErrorInScope.bind($scope));
    };

    const doStopCluster = function(forceStop) {
        Assert.inScope($scope, 'cluster');
        return KubernetesClusterService.stopCluster(forceStop, $scope, $scope.cluster.id, $scope.refreshItem,
                                                    $scope.cluster.architecture, $scope.cluster.type);
    };

    const doLongRunningOperation = function(operation) {
        $scope.isLongClusterOperationRunning = true;
        operation().finally(() => $scope.isLongClusterOperationRunning = false);
    };

    const wrapLongClusterOperationRunning = function(operationName, needsConfirmation, operation) {
        return () => {
            if ($scope.isLongClusterOperationRunning) {
                return;
            }
            if (needsConfirmation) {
                Dialogs.confirm($scope, "Confirm cluster " + operationName, "Are you sure you want to " + operationName + " the cluster?").then(() => {
                    doLongRunningOperation(operation);
                }, () => {});
            } else {
                doLongRunningOperation(operation);
            }
        };
    };

    $scope.startCluster = wrapLongClusterOperationRunning("start", false, function(){
        return $scope.saveCluster().then(doStartCluster);
    });

    $scope.stopCluster = wrapLongClusterOperationRunning("stop", true, function(){
        return doStopCluster(false);
    });

    $scope.forceStopCluster = wrapLongClusterOperationRunning("force stop", true, function() {
        return doStopCluster(true);
    });

    $scope.markStoppedCluster = function(){
        Assert.inScope($scope, 'cluster');

        KubernetesClusterService.sendWt1Event('clusters-action-markstopped',
                                              $scope.cluster.architecture, $scope.cluster.type);
        DataikuAPI.admin.clusters.markStopped($scope.cluster.id).success(function(){
            $scope.refreshItem();
            KubernetesClusterService.sendWt1Event('clusters-status-markstopped',
                                                  $scope.cluster.architecture, $scope.cluster.type);
        }).error(setErrorInScope.bind($scope));
    };

    $scope.currentLogName = null;
    $scope.currentLog = null;
    $scope.fetchLog = function(logName) {
    	DataikuAPI.admin.clusters.getLog($scope.cluster.id, logName).success(function(data) {
            $scope.currentLogName = logName;
            $scope.currentLog = data;
            KubernetesClusterService.sendWt1Event('clusters-logs-opened',
                                                  $scope.cluster.architecture, $scope.cluster.type);
        }).error(setErrorInScope.bind($scope));
    };
    $scope.streamLog = function(logName) {
    	Logs.downloadCluster($scope.cluster.id, logName);
    };

    $scope.downloadClusterDiagnostic = function() {
        ActivityIndicator.success("Preparing cluster diagnosis ...");
        KubernetesClusterService.sendWt1Event('clusters-action-diagdownload',
                                              $scope.cluster.architecture, $scope.cluster.type);
        downloadURL(DataikuAPI.admin.clusters.getDiagnosisURL($scope.cluster.id));
        KubernetesClusterService.sendWt1Event('clusters-status-diagdownloaded',
                                              $scope.cluster.architecture, $scope.cluster.type);
    };
});

app.controller('K8SObjectDescribeController', function($rootScope, $scope, DataikuAPI, FutureWatcher, Dialogs, Logs) {
    $scope.describeObject = function() {
        DataikuAPI.admin.clusters.monitoring.describeObject($scope.clusterId, $scope.objectType.toLowerCase(), $scope.objectName, $scope.namespace).success(function(initialResponse) {
            if (initialResponse.hasResult) {
                $scope.describeResult = initialResponse.result.out;
                $scope.err = initialResponse.result.err;
                $scope.code = initialResponse.result.rv;
            } else {
                $scope.loading = true;
                FutureWatcher.watchJobId(initialResponse.jobId)
                    .success(function(data) {
                        $scope.loading = false;
                        $scope.describeResult = data.result.out;
                        $scope.err = data.result.err;
                        $scope.code = data.result.rv;
                    }).update(function() {
                        $scope.loading = true;
                    }).error(function(data, status, headers) {
                        $scope.loading = false;
                        setErrorInScope.bind($scope)(data, status, headers);
                    });
            }
        }).error(setErrorInScope.bind($scope));
    };

    $scope.tailPodLog = function(){
        DataikuAPI.admin.clusters.monitoring.tailPodLog($scope.clusterId, $scope.objectName, $scope.namespace).success(function(data){
            $scope.podLogTail = data;
        }).error(setErrorInScope.bind($scope));
    }

    $scope.downloadPodLog = function() {
        if ($scope.objectType !== 'Pod') {
            return;
        }

        Logs.downloadPod($scope.clusterId, $scope.objectName, $scope.namespace);
    };

    $scope.switchToDescribe = function(){
        $scope.uiState.mode = "describe";
        $scope.describeObject();
    }

    $scope.switchToLogs = function(){
        $scope.uiState.mode = "logs";
        $scope.tailPodLog();
    }

    $scope.canDeleteObject = function() {
        return ['Pod', 'Job', 'Service', 'Deployment'].includes($scope.objectType);
    }

    $scope.deleteObject = function() {
        if (!$scope.canDeleteObject()) {
            return;
        }

        Dialogs.confirmPositive($scope, "Delete " + $scope.objectType,
                                "Are you sure you want to delete this {0}?".format($scope.objectType.toLowerCase())).then(function() {
            $scope.deleting = true;
            let handleSuccess = (response) => {
                $scope.refreshMonitoring();
                $scope.deleting = false;
                $scope.deleteResult = response.result.out;
                $scope.err = response.result.err;
                $scope.code = response.result.rv;
            };
            DataikuAPI.admin.clusters.monitoring.deleteObject($scope.clusterId, $scope.objectType.toLowerCase(), $scope.objectName, $scope.namespace).success(function(initialResponse) {
                if (initialResponse.hasResult) {
                    handleSuccess(initialResponse);
                } else {
                    FutureWatcher.watchJobId(initialResponse.jobId)
                        .success(function(data) {
                            handleSuccess(data)
                        }).update(function() {
                            $scope.deleting = true;
                        }).error(function(data, status, headers) {
                            $scope.deleting = false;
                            setErrorInScope.bind($scope)(data, status, headers);
                        });
                }
            }).error(setErrorInScope.bind($scope));
        }, function() {
            // Empty function to prevent error in console
        });
    };
});

app.directive('clusterParamsForm', function(Assert, $rootScope, PluginConfigUtils) {
    return {
        restrict: 'A',
        templateUrl: '/templates/admin/clusters/cluster-params-form.html',
        scope: {
            params : '=',
            clusterType : '='
        },
        link: function($scope, element, attrs) {
            $scope.$watch('clusterType', function() {
                if (!$scope.clusterType) return;
                $scope.loadedDesc = $rootScope.appConfig.customPythonPluginClusters.filter(function(x){
                    return x.clusterType == $scope.clusterType;
                })[0];

                Assert.inScope($scope, 'loadedDesc');

                $scope.desc = $scope.loadedDesc.desc;

                // put default values in place
                $scope.params.config = $scope.params.config || {};
                PluginConfigUtils.setDefaultValues($scope.desc.params, $scope.params.config);

                $scope.pluginDesc = $rootScope.appConfig.loadedPlugins.filter(function(x){
                    return x.id == $scope.loadedDesc.ownerPluginId;
                })[0];
            });
        }
    };
});

app.directive('clusterActionsForm', function($rootScope, DataikuAPI, CreateModalFromTemplate, KubernetesClusterService) {
    return {
        restrict: 'A',
        templateUrl: '/templates/admin/clusters/cluster-actions-form.html',
        scope: {
            params : '=',
            clusterId : '=',
            clusterType : '=',
            clusterArchitecture: '=',
            clusterState: '='
        },
        link: function($scope) {
            var refreshClusterActions = function() {
                $scope.clusterActions = [];

                const pluginsById = $rootScope.appConfig.loadedPlugins.reduce(function (map, obj) {
                    map[obj.id] = obj;
                    return map;
                }, {});

                $rootScope.appConfig.customRunnables.forEach(function(runnable) {
                    if (!runnable.desc.macroRoles) return;

                    const plugin = pluginsById[runnable.ownerPluginId];
                    if (!plugin || plugin.hideComponents) return; // plugin might have been deleted

                    runnable.desc.macroRoles.forEach(function(macroRole) {
                        if (macroRole.type != 'CLUSTER') return;
                        if (macroRole.limitToSamePlugin) {
                            if (!$scope.loadedDesc || runnable.ownerPluginId != $scope.loadedDesc.ownerPluginId) return;
                        }

                        $scope.clusterActions.push({
                            label: runnable.desc.meta.label || runnable.id,
                            icon: runnable.desc.meta.icon || plugin.icon,
                            roleTarget: macroRole.targetParamsKey || macroRole.targetParamsKeys,
                            runnable: runnable
                        });
                    });
                });

                $scope.isManualOrRunning = () =>  {
                    return $scope.clusterType === 'manual' || $scope.clusterState === 'RUNNING';
                }

                if ($scope.clusterArchitecture === 'KUBERNETES') {

                    $scope.k8sActions = [
                        {
                            label: 'Run kubectl command',
                            description: 'Run an arbitrary kubectl command',
                            icon: 'icon-play',
                            hasFilters: false,
                            customCommand: true,
                            check: () => {return $scope.$parent.isDSSAdmin()}
                        },
                        {
                            label: 'Delete finished pods',
                            description: 'Delete succeeded and failed pods',
                            command: 'delete pods --field-selector=status.phase!=Pending,status.phase!=Running,status.phase!=Unknown',
                            icon: 'icon-trash',
                            hasFilters: true,
                            wt1Event: 'clusters-action-deletefinishedpods',
                            controller: 'RunPredefinedK8sActionController',
                            api: DataikuAPI.admin.clusters.deleteFinishedPods,
                            customCommand: false
                        },
                        {
                            label: 'Delete all pods',
                            description: 'Delete all pods',
                            command: 'delete --all pods',
                            icon: 'icon-trash',
                            hasFilters: true,
                            wt1Event: 'clusters-action-deleteallpods',
                            controller: 'RunPredefinedK8sActionController',
                            api: DataikuAPI.admin.clusters.deleteAllPods,
                            customCommand: false
                        },
                        {
                            label: 'Delete finished jobs',
                            description: 'Delete finished jobs',
                            icon: 'icon-trash',
                            hasFilters: true,
                            wt1Event: 'clusters-action-deletefinishedjobs',
                            controller: 'DeleteFinishedJobsController',
                            customCommand: false
                        }
                    ]

                    $scope.showRunK8sAction = function(clusterId, {command, longDescription, label, icon, hasFilters, controller, wt1Event, api, customCommand}) {
                        KubernetesClusterService.sendWt1Event(wt1Event, $scope.clusterArchitecture, $scope.clusterType);
                        let actionController = controller ? controller : 'RunK8sActionController';
                        CreateModalFromTemplate('/templates/admin/clusters/run-k8s-action-modal.html', $scope, actionController, (newScope) => {
                            newScope.customCommand = customCommand;
                            newScope.longDescription = longDescription;
                            newScope.hasFilters = hasFilters;
                            newScope.clusterId = clusterId;
                            newScope.command = command? command: '';
                            newScope.label = label;
                            newScope.icon = icon;
                            newScope.api = api;
                            newScope.fullCommand = 'kubectl ' + newScope.command;
                        });
                    };

                    $scope.getActionByLabel = label => {
                        return $scope.k8sActions.find(action => action.label === label);
                    }

                    if ($rootScope.$state.params && $rootScope.$state.params.action) {
                        let action = $scope.getActionByLabel($rootScope.$state.params.action);
                        $scope.showRunK8sAction(
                            $scope.clusterId,
                            action
                        );
                    }
                }

                $scope.showCreateRunnable = function(clusterId, {roleTarget, runnable}) {
                    CreateModalFromTemplate('/templates/macros/runnable-modal.html', $scope, null, function(newScope) {
                        newScope.runnable = runnable;
                        newScope.targetKey = roleTarget;
                        newScope.targetValue = clusterId;
                        newScope.cluster = true;
                    });
                };
            };
            $scope.$watch('clusterType', function() {
                if (!$scope.clusterType) return;
                $scope.loadedDesc = $rootScope.appConfig.customPythonPluginClusters.filter(function(x){
                    return x.clusterType == $scope.clusterType;
                })[0];

                refreshClusterActions();
            });
        }
    };
});

app.directive('clusterActionTile', function() {
    return {
        restrict: 'E',
        templateUrl: '/templates/admin/clusters/cluster-action-tile.html',
        scope: {
            action : '<',
            clusterId: '<',
            showCallback: '<',
            description: '<?'
        },
        link: function($scope) {
            if ($scope.description === undefined) {
                // for native action description is stored directly in the action
                $scope.description = $scope.action.description;
            }
        }
    };
});

app.service("KubernetesClusterService", (DataikuAPI, FutureWatcher, FutureProgressModal, WT1) => {
        const svc = {};

        const EKS_CREATE_TYPE = "pycluster_eks-clusters_create-eks-cluster";
        const EKS_ATTACH_TYPE = "pycluster_eks-clusters_attach-eks-cluster";

        const AKS_CREATE_TYPE = "pycluster_aks-clusters_create-aks-cluster";
        const AKS_ATTACH_TYPE = "pycluster_aks-clusters_attach-aks-cluster";

        const GKE_CREATE_TYPE = "pycluster_gke-clusters_create-gke-cluster";
        const GKE_ATTACH_TYPE = "pycluster_gke-clusters_attach-gke-cluster";

        const EMR_CREATE_TYPE = "pycluster_emr-clusters_emr-create-cluster";
        const EMR_ATTACH_TYPE = "pycluster_emr-clusters_emr-attach-cluster";

        svc.getTypeDisplayName = (type) => {
            switch(type) {
                case EKS_CREATE_TYPE:
                    return 'EKS (managed)';
                case EKS_ATTACH_TYPE:
                    return 'EKS (attach)'; /* We say "attach" rather than "attached" to avoid confusion with state */
                case AKS_CREATE_TYPE:
                    return 'AKS (managed)';
                case AKS_ATTACH_TYPE:
                    return 'AKS (attach)';
                case GKE_CREATE_TYPE:
                    return 'GKE (managed)';
                case GKE_ATTACH_TYPE:
                    return 'GKE (attach)';
                case EMR_CREATE_TYPE:
                    return 'EMR (managed)';
                case EMR_ATTACH_TYPE:
                    return 'EMR (attach)';
                default:
                    return type;
            }
        }

        svc.getStartStopButtonDisplayString = (cluster, isStart) => {
            switch(cluster.type) {
                case EKS_CREATE_TYPE:
                case AKS_CREATE_TYPE:
                case GKE_CREATE_TYPE:
                case EMR_CREATE_TYPE:
                    return isStart ? "Start" : "Stop";
                case EKS_ATTACH_TYPE:
                case AKS_ATTACH_TYPE:
                case GKE_ATTACH_TYPE:
                case EMR_ATTACH_TYPE:
                    return isStart ? "Attach" :  "Detach";
                default:
                    return isStart ? "Start/Attach" : "Stop/Detach";
            }
        }

        svc.getStateDisplayString = (cluster) => {
            let displayNames = null;
            switch(cluster.type) {
                case EKS_CREATE_TYPE:
                case AKS_CREATE_TYPE:
                case GKE_CREATE_TYPE:
                case EMR_CREATE_TYPE:
                    displayNames = {NONE:'Stopped', RUNNING:'Running', STARTING:'Starting', STOPPING:'Stopping'};
                    break;
                case EKS_ATTACH_TYPE:
                case AKS_ATTACH_TYPE:
                case GKE_ATTACH_TYPE:
                case EMR_ATTACH_TYPE:
                    displayNames = {NONE:'Detached', RUNNING:'Attached', STARTING:'Attaching', STOPPING:'Detaching'};
                    break;
                default:
                    displayNames = {NONE:'Stopped/Detached', RUNNING:'Running/Attached', STARTING:'Starting/Attaching', STOPPING:'Stopping/Detaching'};
                    break;
            }
            if (cluster.state == null) {
                return 'Unknown';
            }
            return displayNames[cluster.state] || 'unknown';
        }

        svc.getIcon = (architecture, type) => {
            if (architecture === 'KUBERNETES') {
                switch(type) {
                    case EKS_CREATE_TYPE:
                    case EKS_ATTACH_TYPE:
                        return 'icon-amazon-elastic-kubernetes';
                    case AKS_CREATE_TYPE:
                    case AKS_ATTACH_TYPE:
                        return 'icon-azure-kubernetes-services';
                    case GKE_CREATE_TYPE:
                    case GKE_ATTACH_TYPE:
                        return 'icon-gcp-kubernetes-engine';
                    default:
                        // icon for K8S is not available yet using GKE instead
                        return 'icon-gcp-kubernetes-engine';
                }
            } else if (architecture === 'HADOOP') {
                // haven't found the hadoop icon, using the hdfs one
                return 'icon-HDFS'
            }
            // this should never happen, to my knowledge we only support HADOOP AND K8S cluster
            return 'icon-question';
        }

        svc.getTypeForWt1 = function(clusterType) {
            if (['manual',
                 'pycluster_eks-clusters_create-eks-cluster', 'pycluster_eks-clusters_attach-eks-cluster',
                 'pycluster_aks-clusters_create-aks-cluster', 'pycluster_aks-clusters_attach-aks-cluster',
                 'pycluster_gke-clusters_create-gke-cluster', 'pycluster_gke-clusters_attach-gke-cluster'].includes(clusterType)) {
                return clusterType;
            } else {
                return 'custom';
            }
        };

        svc.sendWt1Event = function(eventName, architecture, type) {
            WT1.event(eventName, { architecture: architecture, type: svc.getTypeForWt1(type) });
        };

        svc.stopCluster = (forceStop, scope, clusterId, successCallBack, architecture, type) => {
            svc.sendWt1Event(forceStop ? "clusters-action-forcestop" : "clusters-action-stop", architecture, type);
            return DataikuAPI.admin.clusters.stop(clusterId, false, forceStop)
                .success(
                    data => FutureProgressModal.show(scope, data, "Stop cluster", undefined, 'static', 'false')
                    .then(function(result){
                        if (result) { // undefined in case of abort
                            successCallBack();
                            svc.sendWt1Event("clusters-status-stopped", architecture, type);
                        }
                    })
                )
                .error(setErrorInScope.bind(scope));
        }

        return svc;
    }
);

app.controller("BaseK8sActionController", ($scope, FutureWatcher) => {
    $scope.running =false;
    $scope.dryRun = true;
    $scope.hasFilters = true;
    $scope.hasAdditionalOptions = false;

    $scope.hasCommandDryRunFlag = () => {
        return true;
    }

    $scope.buildCommandArgs = () => {
        throw new Error('Method buildCommandArgs should not be called from the base controller');
    }

    $scope.callApi = () => {
        throw new Error('Method callApi should not be called from the base controller');
    }

    $scope.resetCommandResult = () => {
        $scope.output = '';
        $scope.err = '';
        $scope.code = '';
    }

    $scope.run = () => {
        let parseResponse = (response) => {
            if ($scope.fatalAPIError) {
                delete $scope.fatalAPIError;
            }
            $scope.running = false;
            $scope.output = response.result.out;
            $scope.err = response.result.err;
            $scope.code = response.result.rv;
        }

        let handleError = (data, status, headers, config, statusText, xhrStatus) => {
            $scope.running = false;
            setErrorInScope.bind($scope)(data, status, headers, config, statusText, xhrStatus);
        }

        $scope.resetCommandResult();
        $scope.running = true

        $scope.callApi().success((initialResponse) => {
            if (initialResponse.hasResult) {
                parseResponse(initialResponse);
            } else {
                FutureWatcher.watchJobId(initialResponse.jobId)
                    .success(parseResponse)
                    .error(handleError);
            }
        }).error(handleError);
    };

    $scope.extendedResetSetting = () => {
        // does nothing in the base controller
        return;
    }

    $scope.resetSettings =  () => {
        $scope.extendedResetSetting();
        $scope.namespaceFilter = '';
        $scope.labelFilter = '';
        $scope.commandArgs = '';
        $scope.dryRun = true;
        $scope.resetCommandResult();
        $scope.updateCommand();
    };

    $scope.updateCommand = () => {
        $scope.fullCommand = 'kubectl ' + $scope.buildCommandArgs();
    }
});

app.controller("RunK8sActionController", ($scope, $controller, DataikuAPI) => {
    $controller("BaseK8sActionController", {$scope:$scope});

    const commandsWithDryRun = ["create", "run", "expose", "delete", "apply", "annotate", "autoscale", "label", "patch",
        "replace", "scale", "set", "reconcile", "cordon", "drain", "taint", "uncordon", "undo"];
    const DRY_RUN_FLAG =  " --dry-run=client";
    const EXEC_SEPARATOR = " -- ";

    $scope.hasCommandDryRunFlag = () => {
        let commandArgs = $scope.commandArgs || $scope.command || '';
        let commandParts = commandArgs.split(" ");
        return commandParts.some(el => commandsWithDryRun.includes(el));
    }

    $scope.buildCommandArgs = () => {
        let commandArgs = $scope.commandArgs || $scope.command || '';
        commandArgs += $scope.namespaceFilter && $scope.namespaceFilter.trim() !== '' ? ' --namespace ' +  $scope.namespaceFilter : '';
        if ($scope.labelFilter && $scope.labelFilter.trim() !== '') {
            if (commandArgs.match(/\s--all[\s$]/)) {
                commandArgs = commandArgs.replace(/\s--all[\s$]/, ' ');
            }
            commandArgs += ' -l ' +  $scope.labelFilter;
        }
        if ($scope.dryRun && $scope.hasCommandDryRunFlag()) {
            if (commandArgs.includes(EXEC_SEPARATOR)) {
                commandArgs = commandArgs.replace(EXEC_SEPARATOR, `${DRY_RUN_FLAG} ${EXEC_SEPARATOR}`)
            } else {
                commandArgs += DRY_RUN_FLAG;
            }
        }
        return commandArgs;
    }

    $scope.callApi = () => {
        let commandWithFilters = $scope.buildCommandArgs();
        return DataikuAPI.admin.clusters.runKubectl($scope.clusterId, commandWithFilters);
    }
});

app.controller("RunPredefinedK8sActionController", ($scope, $controller) => {
    $controller("RunK8sActionController", {$scope:$scope});

    $scope.callApi = () => {
        return $scope.api($scope.clusterId, $scope.dryRun, $scope.namespaceFilter, $scope.labelFilter);
    }
});

app.controller("DeleteFinishedJobsController", ($scope, $controller, DataikuAPI) => {
    $controller("BaseK8sActionController", {$scope:$scope});
    const COMMAND = "delete jobs $(kubectl  get job -o go-template='{{range $i, $p := .items}}{{range .status.conditions}}{{if TYPE_FILTER}}{{$p.metadata.name}}{{\" \"}}{{end}}{{end}}{{end}}'LABEL_FILTER NAMESPACE_FILTER)NAMESPACE_FILTER";
    const COMPLETE_JOB_FILTER = "(eq .type \"Complete\")";
    const COMPLETE_AND_FAILED_JOB_FILTER = "or (eq .type \"Failed\") (eq .type \"Complete\")";

    $scope.hasAdditionalOptions = true;

    $scope.deleteFailed = false;
    $scope.fullCommand = 'kubectl ' + $scope.command;

    $scope.buildCommandArgs = () => {
        let commandArgs = COMMAND;
        commandArgs = commandArgs.replace("TYPE_FILTER", $scope.deleteFailed ? COMPLETE_AND_FAILED_JOB_FILTER : COMPLETE_JOB_FILTER);
        let namespaceReplacement = $scope.namespaceFilter && $scope.namespaceFilter.trim() !== '' ? " -n " + $scope.namespaceFilter : "";
        commandArgs = commandArgs.replaceAll("NAMESPACE_FILTER", namespaceReplacement);

        let labelReplacement = $scope.labelFilter && $scope.labelFilter.trim() !== '' ? " -l " + $scope.labelFilter : "";
        commandArgs = commandArgs.replaceAll("LABEL_FILTER", labelReplacement);

        if ($scope.dryRun) {
            commandArgs += " --dry-run=client"
        }
        return commandArgs;
    }

    $scope.callApi = () => {
        return DataikuAPI.admin.clusters.deleteFinishedJobs($scope.clusterId, $scope.dryRun, $scope.deleteFailed, $scope.namespaceFilter, $scope.labelFilter);
    }

    $scope.extendedResetSetting = () => {
        $scope.deleteFailed = false;
    }
});

app.directive('hadoopClusterSettingsBlock', function($rootScope) {
    return {
        restrict: 'A',
        templateUrl: '/templates/admin/clusters/fragments/hadoop-cluster-settings-block.html',
        scope: {
            settings : '=',
            mask : '=',
            impersonationEnabled : '='
        },
        link: function($scope, element, attrs) {
            $scope.appConfig = $rootScope.appConfig;
            $scope.addLicInfo = $rootScope.addLicInfo;
        }
    };
});

app.directive('hiveClusterSettingsBlock', function($rootScope, CodeMirrorSettingService) {
    return {
        restrict: 'A',
        templateUrl: '/templates/admin/clusters/fragments/hive-cluster-settings-block.html',
        scope: {
            settings : '=',
            hadoopSettings : '=',
            mask : '=',
            impersonationEnabled : '='
        },
        link: function($scope, element, attrs) {
            $scope.appConfig = $rootScope.appConfig;
            $scope.addLicInfo = $rootScope.addLicInfo;
            $scope.codeMirrorSettingService = CodeMirrorSettingService;

            $scope.copyHadoopSettings = function() {
                if (!$scope.settings) return;
                if (!$scope.hadoopSettings) return;

                var hiveProps = $scope.settings.executionConfigsGenericOverrides;
                var hadoopPropsNames = $scope.hadoopSettings.extraConf.map(function(p) {return p.key;});
                // remove existing properties with the names of those we add (avoid duplicates)
                hadoopPropsNames.forEach(function(k) {
                   var indices = hiveProps.map(function(p, i) {return p.key == k ? i : null;}).filter(function(x) {return x != null;});
                   indices.reverse().forEach(function(i) {hiveProps.splice(i, 1);});
                });
                $scope.hadoopSettings.extraConf.forEach(function(p) {
                    var hp = angular.copy(p);
                    hiveProps.push(hp);
                });
                // to make the list's UI refresh
                $scope.settings.executionConfigsGenericOverrides = [].concat(hiveProps)
            };
        }
    };
});

app.directive('impalaClusterSettingsBlock', function($rootScope) {
    return {
        restrict: 'A',
        templateUrl: '/templates/admin/clusters/fragments/impala-cluster-settings-block.html',
        scope: {
            settings : '=',
            mask : '=',
            impersonationEnabled : '='
        },
        link: function($scope, element, attrs) {
            $scope.appConfig = $rootScope.appConfig;
            $scope.addLicInfo = $rootScope.addLicInfo;
        }
    };
});

app.directive('sparkClusterSettingsBlock', function(DataikuAPI, $rootScope, FutureProgressModal, ActivityIndicator, FeatureFlagsService) {
    return {
        restrict: 'A',
        templateUrl: '/templates/admin/clusters/fragments/spark-cluster-settings-block.html',
        scope: {
            settings : '=',
            hadoopSettings : '=',
            mask : '=',
            impersonationEnabled : '=',
            clusterId : '='
        },
        link: function($scope, element, attrs) {
            $scope.appConfig = $rootScope.appConfig;
            $scope.addLicInfo = $rootScope.addLicInfo;


            $scope.executionEngines = [
                                   {value:null, label:'Disabled'},
                                   {value:'SPARK_SUBMIT', label:'Enabled'}
                                ];

            $scope.copyHadoopSettings = function() {
                if (!$scope.settings) return;
                if (!$scope.hadoopSettings) return;

                var sparkProps = $scope.settings.executionConfigsGenericOverrides;
                var hadoopPropsNames = $scope.hadoopSettings.extraConf.map(function(p) {return p.key;});
                // remove existing properties with the names of those we add (avoid duplicates)
                hadoopPropsNames.forEach(function(k) {
                   var indices = sparkProps.map(function(p, i) {return p.key == 'spark.hadoop.' + k ? i : null;}).filter(function(x) {return x != null;});
                   indices.reverse().forEach(function(i) {sparkProps.splice(i, 1);});
                });
                $scope.hadoopSettings.extraConf.forEach(function(p) {
                    var sp = angular.copy(p);
                    sp.key = 'spark.hadoop.' + p.key;
                    sparkProps.push(sp);
                });
                // to make the list's UI refresh
                $scope.settings.executionConfigsGenericOverrides = [].concat(sparkProps);
            };

            $scope.preloadYarnClusterFiles = function(yarnClusterSettings) {
                DataikuAPI.admin.clusters.preloadYarnClusterFiles(yarnClusterSettings).success(function(data){
                    FutureProgressModal.show($scope, data, "Preload files on cluster", undefined, 'static', 'false').then(function(result) {
                    });
                }).error(setErrorInScope.bind($scope));
            };

            $scope.sparkVersionsCompatible = function(dssVersion, livyVersion) {
                if (dssVersion == null || dssVersion.length == 0) dssVersion = $rootScope.appConfig.sparkVersion;
                if (dssVersion == null || dssVersion.length == 0) return true;
                if (livyVersion == null || livyVersion.length == 0) return true;
                return dssVersion.substring(0,2) == livyVersion.substring(0,2)
            };
            $scope.shouldUseYarnCluster = function(sparkMaster, deployMode) {
                if (sparkMaster == 'yarn' && deployMode == 'cluster') return true;
                if (sparkMaster == 'yarn-cluster') return true;
                return false;
            };
        }
    };
});

app.directive('containerClusterSettingsBlock', function(DataikuAPI, WT1, $rootScope, FutureProgressModal, Dialogs, CodeMirrorSettingService) {
    return {
        restrict: 'A',
        templateUrl: '/templates/admin/clusters/fragments/container-cluster-settings-block.html',
        scope: {
            settings : '=',
            mask : '=',
            impersonationEnabled : '=',
            clusterId : '=',
            k8sClusters: '=',
            clusterDefinedConfig: '='
        },
        link: function($scope, element, attrs) {
            $scope.appConfig = $rootScope.appConfig;
            $scope.addLicInfo = $rootScope.addLicInfo;
            $scope.codeMirrorSettingService = CodeMirrorSettingService;

            $scope.getNewContainerConfig = function() {
                return {
                    type: 'KUBERNETES',
                    usableBy: 'ALL', allowedGroups: [], workloadType: 'ANY',
                    dockerResources: [],
                    kubernetesResources: {
                        memRequestMB: -1, memLimitMB: -1,
                        cpuRequest: -1, cpuLimit: -1,
                        customRequests: [], customLimits: [],
                        hostPathVolumes: []
                    },
                    customSHMValueMB: -1,
                    properties: []
                };
            };

            DataikuAPI.security.listGroups(false)
                .success(data => {
                    if (data) {
                        data.sort();
                    }
                    $scope.allGroups = data;
                })
                .error(setErrorInScope.bind($scope));

            $scope.isBaseImageNameSuspicious = function(baseImage) {
                return /^(?:[\w-_]+\.)+\w+(?::\d+)?\//.test(baseImage);
            };

            var testErrors = {};
            $scope.getTestError = function(config) {
                let s = testErrors[config.name];
                return s != null ? s.fatalAPIError : null;
            };

            $scope.testConf = function(configuration, clusterId) {
                testErrors[configuration.name] = {}; // doesn't need to be a scope for setErrorInScope()
                DataikuAPI.admin.containerExec.testConf(configuration, $scope.clusterDefinedConfig, clusterId || $scope.clusterId, $scope.settings.executionConfigsGenericOverrides).success(function(data){
                    FutureProgressModal.show($scope, data, "Testing container configuration", undefined, 'static', 'false').then(function(result){
                        if (result) {
                            Dialogs.infoMessagesDisplayOnly($scope, "Container test result", result.messages, result.futureLog);
                        }
                    })
                }).error(setErrorInScope.bind(testErrors[configuration.name]));
                WT1.event('container-conf-test');
            }

            $scope.getExtraClusters = function() {
                if (!$scope.k8sClusters) return [];
                return $scope.k8sClusters.filter(function(c) {return (c.id || '__builtin__') != ($scope.clusterId || '__builtin__');});
            };

        }
    };
});

app.directive('clusterSecurityPermissions', function(DataikuAPI, $rootScope, PermissionsService) {
    return {
        restrict : 'A',
        templateUrl : '/templates/admin/clusters/fragments/security-permissions.html',
        replace : true,
        scope : {
                cluster  : '='
        },
        link : function($scope, element, attrs) {
            $scope.appConfig = $rootScope.appConfig;
            $scope.ui = {};

            function makeNewPerm(){
                $scope.newPerm = {
                    update: true,
                    use: true
                }
            }
            makeNewPerm();

            const fixupPermissions = function() {
                if (!$scope.cluster) return;
                /* Handle implied permissions */
                $scope.cluster.permissions.forEach(function(p) {
                    p.$updateDisabled = false;
                    p.$manageUsersDisabled = false;
                    p.$useDisabled = false;

                    if ($scope.cluster.usableByAll) {
                        p.$useDisabled = true;
                    }
                    if (p.update) {
                        p.$useDisabled = true;
                    }
                    if (p.manageUsers) {
                        p.$useDisabled = true;
                        p.$updateDisabled = true;
                    }
                });
            };

            DataikuAPI.security.listGroups(false).success(function(allGroups) {
                if (allGroups) {
                    allGroups.sort();
                }
                $scope.allGroups = allGroups;
                DataikuAPI.security.listUsers().success(function(data) {
                    $scope.allUsers = data.sort((a, b) => a.displayName.localeCompare(b.displayName));
                    $scope.allUsersLogin = $scope.allUsers.map(user => '@' + user.login);
                }).error(setErrorInScope.bind($scope));
                $scope.unassignedGroups = PermissionsService.buildUnassignedGroups($scope.cluster, $scope.allGroups);
            }).error(setErrorInScope.bind($scope));

            $scope.$watch("cluster.owner", function() {
                $scope.ui.ownerLogin = $scope.cluster.owner;
            });

            $scope.addPermission = function() {
                $scope.cluster.permissions.push($scope.newPerm);
                makeNewPerm();
            };

            $scope.$watch("cluster.usableByAll", function(nv, ov) {
                fixupPermissions();
            })
            $scope.$watch("cluster.permissions", function(nv, ov) {
                if (!nv) return;
                $scope.unassignedGroups = PermissionsService.buildUnassignedGroups($scope.cluster, $scope.allGroups);
                fixupPermissions();
            }, true)
            $scope.$watch("cluster.permissions", function(nv, ov) {
                if (!nv) return;
                $scope.unassignedGroups = PermissionsService.buildUnassignedGroups($scope.cluster, $scope.allGroups);
                fixupPermissions();
            }, false)
            $scope.unassignedGroups = PermissionsService.buildUnassignedGroups($scope.cluster, $scope.allGroups);
            fixupPermissions();

            // Ownership mgmt
            $scope.$watch("ui.ownerLogin", function() {
                PermissionsService.transferOwnership($scope, $scope.cluster, "cluster");
            });


        }
    };
});

})();
;
(function () {
    'use strict';

    const app = angular.module('dataiku.admin.charts_dashboards', []);

    app.controller("ChartsAndDashboardsController", function ($scope, $timeout, DSSVisualizationTheming, DefaultDSSVisualizationTheme) {
        $scope.promises.then(function() {
            if (!$scope.generalSettings.selectedDSSVisualizationThemeId) {
                $scope.generalSettings.selectedDSSVisualizationThemeId = DefaultDSSVisualizationTheme.id;
            }
        });

        $scope.handleCurrentThemeIdChange = (id) => {
            $timeout(() => {
                $scope.generalSettings.selectedDSSVisualizationThemeId = id;
            });
        };
    });
}());

;
(function() {
'use strict';

    const app = angular.module('dataiku.plugindev', ['dataiku.plugindev.git']);


    app.controller('PlugindevCommonController', function($rootScope, $scope, DataikuAPI, $controller, $state, $stateParams, $filter, CreateModalFromTemplate, TopNav, FutureWatcher, WT1, FutureProgressModal, Dialogs, AgentCodeTemplates) {
        TopNav.setLocation(TopNav.DSS_HOME, 'plugin');

        const COMPONENT_IDENTIFIER_PLACEHOLDER = 'use-this-kind-of-case';

        if ($scope.appConfig.pluginDevGitMode === 'PLUGIN') {
            $controller('_PlugindevGitController', {$scope: $scope});
        }

        let regularPyAndJavaDescriptors = [
            { key: "python", label: "Python" },
            { key: "java", label: "Java"}
        ];

        let regularPyOnlyDescriptor = [
            { key: "python", label: "Python" }
        ];

        let regularJavaOnlyDescriptor = [
            { key: "java", label: "Java" }
        ];

        let regularJythonOnlyDescriptor = [
            { key: "jython", label: "Python" }
        ];

        let regularPyAndRDescriptors = [
            { key: "python", label: "Python" },
            { key: "r", label: "R"}
        ];
        let regularGenericOnlyDescriptor = [
            { key : "generic", label: "Generic" }
        ];

        let downloadIframe = $('<iframe>').attr('id', 'plugin-downloader');
        downloadIframe[0].onload = function() {
            // If download failed, notify user.
            if (this.contentDocument.URL !== 'about:blank') {
                CreateModalFromTemplate('/templates/plugins/modals/plugin-download-error.html', $scope);
            }
        };

        function getIdentifierHint(typeName = '') {
            return `Unique identifier for the new ${typeName} type. <br> It should not start with the plugin id and must be unique across the ${typeName} components of this plugin.` ;
        }

        $scope.initContentTypeList = function(pluginData=$scope.pluginData) {
            let codeEnvSpec = pluginData.installedDesc && pluginData.installedDesc.codeEnvSpec;
            $scope.contentTypeList = [
                {
                    name: "Code Env",
                    type: "codeEnv",
                    icon: 'icon-cogs',
                    iconColor: 'universe-color more',
                    disabled: codeEnvSpec != null,
                    disabledReason: "Only one code env can exist in a plugin",
                    description: "Create the code environment associated with this plugin, defining required libraries and their versions.",
                    addDescriptors: regularPyAndRDescriptors,
                    addFuncMap: {
                        python: DataikuAPI.plugindev.addPythonCodeEnv,
                        r: DataikuAPI.plugindev.addRCodeEnv
                    }
                },
                {
                    name: "Dataset",
                    type: "customDatasets",
                    icon: "icon-database",
                    iconColor: "universe-color datasets",
                    description: "Create a new type of dataset. This is generally used to fetch data from an external service, for example using an API",
                    addDescriptors: regularPyAndJavaDescriptors,
                    addFuncMap: {
                        java: DataikuAPI.plugindev.addJavaDataset,
                        python: DataikuAPI.plugindev.addPythonDataset
                    },
                    identifierPlaceholder: (() => COMPONENT_IDENTIFIER_PLACEHOLDER),
                    identifierHint: (() => getIdentifierHint('dataset'))
                },
                {
                    name: "Recipe",
                    type: "customCodeRecipes",
                    icon: 'icon-FLOW_recipe_empty',
                    iconColor: 'universe-color recipe',
                    disabled: true,
                    disabledReason: "To create a new plugin recipe, you need to create it from an existing code recipe in a project. Go to the advanced tab > Convert to plugin recipe.",
                    description: "Create a new kind of recipe (Python, R, or Scala)"
                },
                {
                    name: "Macro",
                    type: "customRunnables",
                    icon: 'icon-macro',
                    iconColor: 'universe-color more',
                    description: "Create a new kind of runnable piece. Useful to occasionally launch external or maintenance tasks. Macros can also be run in a scenario.",
                    addDescriptors: regularPyAndJavaDescriptors,
                    addFuncMap: {
                        java: DataikuAPI.plugindev.addJavaRunnable,
                        python: DataikuAPI.plugindev.addPythonRunnable
                    },
                    identifierPlaceholder: (() => COMPONENT_IDENTIFIER_PLACEHOLDER),
                    identifierHint: (() => getIdentifierHint('runnable'))
                },
                {
                    name: "Parameter set",
                    type: "customParameterSets",
                    icon: 'icon-indent-right',
                    iconColor: 'universe-color more',
                    description: "Create a definition for presets in this plugin.",
                    addDescriptors : regularGenericOnlyDescriptor,
                    addFuncMap: {
                        generic: DataikuAPI.plugindev.addParameterSet,
                    },
                    identifierPlaceholder: (() => COMPONENT_IDENTIFIER_PLACEHOLDER),
                    identifierHint: (() => getIdentifierHint('parameter set'))
                },
                {
                    name: "Notebook template",
                    type: function(language = '') {
                        if (language.startsWith("scala")) {
                            return "customScalaNotebookTemplates"
                        }
                        if (language.startsWith("r")) {
                            return "customRNotebookTemplates"
                        }
                        return "customPythonNotebookTemplates";
                    },
                    icons: {
                        'customPreBuiltDatasetNotebookTemplates': { icon: 'icon-dku-nav_notebook', iconColor: 'universe-color notebook'},
                        'customRNotebookTemplates': { icon: 'icon-code_r_recipe', iconColor: 'universe-color notebook'},
                        'customRDatasetNotebookTemplates': { icon: 'icon-code_r_recipe', iconColor: 'universe-color notebook'},
                        'customPythonNotebookTemplates': { icon: 'icon-code_python_recipe', iconColor: 'universe-color notebook'},
                        'customPythonDatasetNotebookTemplates': { icon: 'icon-code_python_recipe', iconColor: 'universe-color notebook'},
                        'customScalaNotebookTemplates': { icon: 'icon-code_spark_scala_recipe', iconColor: 'universe-color notebook'},
                        'customScalaDatasetNotebookTemplates': { icon: 'icon-code_spark_scala_recipe', iconColor: 'universe-color notebook'}
                    },
                    description: "Create a new notebook template.",
                    addDescriptors: [
                        { key: "pythonDataset", label: "Python (create from a dataset)" },
                        { key: "pythonStandalone", label: "Python (create from notebooks list, unrelated to a dataset)" },
                        { key: "pythonDatasetPrebuilt", label: "Python (create from a dataset, in 'predefined' list)" },
                        { key: "rDataset", label: "R (create from a dataset)" },
                        { key: "rStandalone", label: "R (create from notebooks list, unrelated to a dataset)" },
                        { key: "rDatasetPrebuilt", label: "R (create from a dataset, in 'predefined' list)" },
                        { key: "scalaDataset", label: "Scala (create from a dataset)" },
                        { key: "scalaStandalone", label: "Scala (create from notebooks list, unrelated to a dataset)" }
                    ],
                    identifierPlaceholder: (() => COMPONENT_IDENTIFIER_PLACEHOLDER),
                    identifierHint: (() => getIdentifierHint('notebook template')),
                    addFuncMap: {
                        pythonDataset: DataikuAPI.plugindev.addNotebookTemplate.bind(this, "dataset", "python", false),
                        pythonStandalone: DataikuAPI.plugindev.addNotebookTemplate.bind(this, "standalone", "python", false),
                        pythonDatasetPrebuilt: DataikuAPI.plugindev.addNotebookTemplate.bind(this, "dataset", "python", true),
                        rDataset: DataikuAPI.plugindev.addNotebookTemplate.bind(this, "dataset", "r", false),
                        rStandalone: DataikuAPI.plugindev.addNotebookTemplate.bind(this, "standalone", "r", false),
                        rDatasetPrebuilt: DataikuAPI.plugindev.addNotebookTemplate.bind(this, "dataset", "r", true),
                        scalaDataset: DataikuAPI.plugindev.addNotebookTemplate.bind(this, "dataset", "scala", false),
                        scalaStandalone: DataikuAPI.plugindev.addNotebookTemplate.bind(this, "standalone", "scala", false)
                    }
                },
                {
                    name: "RMarkdown report template",
                    type: "customRMarkdownReportTemplates",
                    icon: 'icon-DKU_rmd',
                    iconColor: 'universe-color report',
                    description: "Create a new template for RMarkdown reports",
                    addDescriptors : [
                        { key : "rmarkdown", label: "RMarkdown" }
                    ],
                    addFuncMap: {
                        rmarkdown: DataikuAPI.plugindev.addRMarkdownReportTemplate
                    },
                    identifierPlaceholder : (() => COMPONENT_IDENTIFIER_PLACEHOLDER),
                    identifierHint : (() => getIdentifierHint('RMarkdown report template'))
                },
                 {
                    name: "Webapp",
                    type: "customWebApps",
                    icon: 'icon-eye',
                    iconColor: 'universe-color recipe-visual',
                    disabled: true,
                    disabledReason: "To create a new plugin webapp, you need to create it from an existing webapp in a project. Go to the advanced tab > Convert to plugin webapp.",
                    description: "Create a reusable webapp for custom visualization or interactive screen without code for the end user",
                },
                {
                    name: "Webapp template",
                    type: function(language = '') {
                        if (language.startsWith("bokeh")) {
                            return "customBokehWebAppTemplates"
                        }
                        if (language.startsWith("shiny")) {
                            return "customShinyWebAppTemplates"
                        }
                        if (language.startsWith("dash")) {
                            return "customDashWebAppTemplates"
                        }
                        return "customStandardWebAppTemplates";
                    },
                    icons: {
                        'customBokehWebAppTemplates': { icon: 'icon-bokeh', iconColor: 'universe-color recipe-code' },
                        'customDashWebAppTemplates': { icon: 'icon-dash', iconColor: 'universe-color recipe-code' },
                        'customShinyWebAppTemplates': { icon: 'icon-shiny', iconColor: 'universe-color recipe-code' },
                        'customStandardWebAppTemplates': { icon: 'icon-code', iconColor: 'universe-color recipe-code' }
                    },
                    description: "Create a new template for webapps",
                    addDescriptors: [
                        { key : "standard", label: "Standard webapp (JS/HTML/CSS/Python)" },
                        { key : "bokeh", label: "Bokeh webapp (Python)" },
                        { key : "dash", label: "Dash webapp (Python)" },
                        { key : "shiny", label: "Shiny webapp (R)" }
                    ],
                    addFuncMap: {
                        standard: DataikuAPI.plugindev.addStandardWebAppTemplate,
                        bokeh: DataikuAPI.plugindev.addBokehWebAppTemplate,
                        dash: DataikuAPI.plugindev.addDashWebAppTemplate,
                        shiny: DataikuAPI.plugindev.addShinyWebAppTemplate
                    },
                    identifierPlaceholder: (() => COMPONENT_IDENTIFIER_PLACEHOLDER),
                    identifierHint: (() => getIdentifierHint('template for webapps'))
                },
                {
                    name: "Scenario trigger",
                    type: "customPythonTriggers",
                    icon: 'icon-list',
                    iconColor: 'universe-color scenario',
                    description: "Create a new kind of trigger for scenarios",
                    addDescriptors: regularPyOnlyDescriptor,
                    addFuncMap: {
                        python: DataikuAPI.plugindev.addPythonTrigger
                    },
                    identifierPlaceholder: (() => COMPONENT_IDENTIFIER_PLACEHOLDER),
                    identifierHint: (() => getIdentifierHint('trigger'))
                },
                {
                    name: "Scenario step",
                    type: "customPythonSteps",
                    icon: 'icon-step-forward',
                    iconColor: 'universe-color scenario',
                    description: "Create a new kind of step for scenarios",
                    addDescriptors: regularPyOnlyDescriptor,
                    addFuncMap: {
                        python: DataikuAPI.plugindev.addPythonStep
                    },
                    identifierPlaceholder: (() => COMPONENT_IDENTIFIER_PLACEHOLDER),
                    identifierHint: (() => getIdentifierHint('scenario step'))
                },
                {
                    name: "Metrics probe",
                    type: function(language = '') {
                        if (language.startsWith("sql")) {
                            return "customSQLProbes"
                        }
                        return "customPythonProbes";
                    },
                    icons: {
                        'customSQLProbes': { icon: 'icon-subscript', iconColor: 'universe-color datasets' },
                        'customPythonProbes': { icon: 'icon-superscript', iconColor: 'universe-color datasets' }
                    },
                    description: "Create a new kind of probe to compute metrics, that can be applied on datasets",
                    addDescriptors: [
                        { key: "python", label: "Python" },
                        { key: "sql", label: "SQL"}
                    ],
                    addFuncMap: {
                        python: DataikuAPI.plugindev.addPythonProbe,
                        sql: DataikuAPI.plugindev.addSqlProbe
                    },
                    identifierPlaceholder: (() => COMPONENT_IDENTIFIER_PLACEHOLDER),
                    identifierHint: (() => getIdentifierHint('probe'))
                },
                {
                    name: "Check",
                    type: "customPythonChecks",
                    icon: 'icon-ok',
                    iconColor: 'universe-color dataset',
                    description: "Create a new kind of check, that can be applied on datasets, managed folders, saved model versions, model evaluations or projects",
                    addDescriptors: regularPyOnlyDescriptor,
                    addFuncMap: {
                        python: DataikuAPI.plugindev.addPythonCheck,
                    },
                    identifierPlaceholder: (() => COMPONENT_IDENTIFIER_PLACEHOLDER),
                    identifierHint: (() => getIdentifierHint('check'))
                },
                {
                    name: "Exporter",
                    type: "customExporters",
                    icon: 'icon-dku-download',
                    iconColor: 'universe-color datasets',
                    description: "Create a new option to export dataset out of DSS. This can be export to file (that the user can download) or to custom destinations (like an external API). Exporters have only 'write' support. If you want 'read' support, you need to write a format instead",
                    addDescriptors: regularPyAndJavaDescriptors,
                    addFuncMap: {
                        java: DataikuAPI.plugindev.addJavaExporter,
                        python: DataikuAPI.plugindev.addPythonExporter
                    },
                    identifierPlaceholder: (() => COMPONENT_IDENTIFIER_PLACEHOLDER),
                    identifierHint: (() => getIdentifierHint('exporter'))
                },
                {
                    name: "File format",
                    type: "customFormats",
                    icon: 'icon-file',
                    iconColor: 'universe-color datasets',
                    description: "Create a new supported file format, that DSS uses to read and write on all files-based kinds of datasets (Filesystem, HDFS, S3, ...). Write support is optional",
                    addDescriptors: regularPyAndJavaDescriptors,
                    addFuncMap: {
                        java: DataikuAPI.plugindev.addJavaFormat,
                        python: DataikuAPI.plugindev.addPythonFormat
                    },
                    identifierPlaceholder: (() => COMPONENT_IDENTIFIER_PLACEHOLDER),
                    identifierHint: (() => getIdentifierHint('file format'))
                },
                {
                    name: "Sample dataset",
                    type: "customSampleDatasets",
                    icon: 'icon-dku-tutorial',
                    iconColor: 'universe-color datasets',
                    description: "Create a sample dataset to be used directly in a project",
                    addDescriptors: regularGenericOnlyDescriptor,
                    addFuncMap: {
                        generic: DataikuAPI.plugindev.addSampleDataset
                    },
                    identifierPlaceholder: (() => COMPONENT_IDENTIFIER_PLACEHOLDER),
                    identifierHint: (() => getIdentifierHint('sample dataset'))
                },
                {
                    name: "FS provider",
                    type: "customFileSystemProviders",
                    icon: 'icon-server_file_system_1',
                    iconList: 'universe-color datasets',
                    description: "Create a new kind of files-based system, usable both for dataset (together with a file format) or for managed folders. Examples include cloud storages, file sharing systems, ... Write support is optional.",
                    addDescriptors: regularPyAndJavaDescriptors,
                    addFuncMap: {
                        java: DataikuAPI.plugindev.addJavaFSProvider,
                        python: DataikuAPI.plugindev.addPythonFSProvider
                    },
                    identifierPlaceholder: (() => COMPONENT_IDENTIFIER_PLACEHOLDER),
                    identifierHint: (() => getIdentifierHint('FS provider'))
                },
                {
                    name: "Preparation processor",
                    type: "customJythonProcessors",
                    icon: 'icon-visual_prep_cleanse_recipe',
                    iconColor: 'universe-color recipe-visual',
                    description: "Create a new kind of step for preparation scripts",
                    addDescriptors: regularJythonOnlyDescriptor,
                    addFuncMap: {
                        jython: DataikuAPI.plugindev.addJythonProcessor
                    },
                    identifierPlaceholder: (() => COMPONENT_IDENTIFIER_PLACEHOLDER),
                    identifierHint: (() => getIdentifierHint('processor'))
                },
                {
                    name: "Prediction Algorithm",
                    type: "customPythonPredictionAlgos",
                    icon: 'icon-machine_learning_regression',
                    iconColor: 'universe-color recipe-train',
                    description: "Create a new prediction algorithm",
                    addDescriptors: regularPyOnlyDescriptor,
                    addFuncMap: {
                        python: DataikuAPI.plugindev.addPredictionPythonAlgorithm
                    },
                    identifierPlaceholder: (() => COMPONENT_IDENTIFIER_PLACEHOLDER),
                    identifierHint: (() => getIdentifierHint('prediction algorithm'))
                },
                {
                    name: "LLM Guardrail",
                    type: "customGuardrails",
                    icon: 'dku-icon-shield-check-24',
                    iconColor: 'universe-color more',
                    description: "Create a new guardrail for LLMs",
                    addDescriptors: regularPyOnlyDescriptor,
                    addFuncMap: {
                        python: DataikuAPI.plugindev.addGuardrail
                    },
                    identifierPlaceholder: (() => COMPONENT_IDENTIFIER_PLACEHOLDER),
                    identifierHint: (() => getIdentifierHint('guardrail'))
                },
                {
                    name: "Agent",
                    type: "customAgents",
                    icon: 'dku-icon-ai-agent-plugin-24',
                    iconColor: 'universe-color more',
                    description: "Create a new agent",
                    addDescriptors: regularPyOnlyDescriptor,
                    addFuncMap: {
                        python: ((pluginId, componentId, javaClassName, codeTemplate) => DataikuAPI.plugindev.addAgent(pluginId, componentId, codeTemplate))
                    },
                    codeTemplates: Object.keys(AgentCodeTemplates).map(function(key) {
                        return {
                            key: key,
                            label: AgentCodeTemplates[key].title,
                            description: AgentCodeTemplates[key].description,
                            codeSample: AgentCodeTemplates[key].codeSample,
                        };
                    }),
                    identifierPlaceholder: (() => COMPONENT_IDENTIFIER_PLACEHOLDER),
                    identifierHint: (() => getIdentifierHint('agent'))
                },
                {
                    name: "Agent Tool",
                    type: "customAgentTools",
                    icon: 'dku-icon-tool-wrench-24',
                    iconColor: 'universe-color more',
                    description: "Create a new tool for agents",
                    addDescriptors: regularPyOnlyDescriptor,
                    addFuncMap: {
                        python: DataikuAPI.plugindev.addAgentTool
                    },
                    identifierPlaceholder: (() => COMPONENT_IDENTIFIER_PLACEHOLDER),
                    identifierHint: (() => getIdentifierHint('agent-tool'))
                },
                {
                    name: "Cluster",
                    type: "customPythonClusters",
                    icon: 'icon-sitemap',
                    iconColor: 'universe-color more',
                    description: "(Expert usage) Create a new kind of cluster",
                    addDescriptors: regularPyOnlyDescriptor,
                    addFuncMap: {
                        python: DataikuAPI.plugindev.addPythonCluster
                    },
                    identifierPlaceholder: (() => COMPONENT_IDENTIFIER_PLACEHOLDER),
                    identifierHint: (() => getIdentifierHint('cluster'))
                },
                {
                    name: "Code Studio block",
                    type: "customPythonCodeStudioBlocks",
                    icon: 'icon-code-studio',
                    iconColor: 'universe-color more',
                    description: "(Expert usage) Create a new kind of Code Studio block",
                    addDescriptors: regularPyOnlyDescriptor,
                    addFuncMap: {
                        python: DataikuAPI.plugindev.addPythonCodeStudioBlock
                    },
                    identifierPlaceholder: (() => COMPONENT_IDENTIFIER_PLACEHOLDER),
                    identifierHint: (() => getIdentifierHint('codeStudioBlock'))
                },
                {
                    name: "Custom Fields",
                    type: "customFields",
                    icon: 'icon-list-ol',
                    iconColor: 'universe-color more',
                    description: "Create new custom fields",
                    addDescriptors : [{key: "json", label: "json"}],
                    addFuncMap: {
                        json: DataikuAPI.plugindev.addCustomFields
                    },
                    identifierPlaceholder : (() => COMPONENT_IDENTIFIER_PLACEHOLDER),
                    identifierHint : (() => getIdentifierHint('custom fields'))
                },
                {
                    name: "Custom Policy Hooks",
                    type: "customPolicyHooks",
                    icon: 'icon-legal',
                    iconColor: 'universe-color more',
                    description: "Create new custom policy hooks",
                    addDescriptors : [{key: "java", label: "Java"}],
                    addFuncMap: {
                        java: DataikuAPI.plugindev.addJavaPolicyHooks
                    },
                    identifierPlaceholder : (() => COMPONENT_IDENTIFIER_PLACEHOLDER),
                    identifierHint : (() => getIdentifierHint('custom policy hooks'))
                },
                {
                    name: "Custom User Supplier",
                    type: "customUserSupplier",
                    icon: 'icon-group',
                    iconColor: 'universe-color more',
                    description: "Create new custom user supplier",
                    addDescriptors : [{key: "java", label: "Java"}],
                    addFuncMap: {
                        java: DataikuAPI.plugindev.addJavaUserSupplier
                    },
                    identifierPlaceholder : (() => COMPONENT_IDENTIFIER_PLACEHOLDER),
                    identifierHint : (() => getIdentifierHint('custom user supplier'))
                },
                {
                    name: "Custom User Authenticator",
                    type: "customUserAuthenticator",
                    icon: 'icon-signin',
                    iconColor: 'universe-color more',
                    description: "Create new custom user authenticator",
                    addDescriptors : [{key: "java", label: "Java"}],
                    addFuncMap: {
                        java: DataikuAPI.plugindev.addJavaUserAuthenticator
                    },
                    identifierPlaceholder : (() => COMPONENT_IDENTIFIER_PLACEHOLDER),
                    identifierHint : (() => getIdentifierHint('custom user authenticator'))
                },
                {
                    name: "Custom User Authenticator and Supplier",
                    type: "customUserAuthenticatorAndSupplier",
                    icon: 'icon-signin',
                    iconColor: 'universe-color more',
                    description: "Create new custom user authenticator and supplier",
                    addDescriptors : [{key: "java", label: "Java"}],
                    addFuncMap: {
                        java: DataikuAPI.plugindev.addJavaUserAuthenticatorAndSupplier
                    },
                    identifierPlaceholder : (() => COMPONENT_IDENTIFIER_PLACEHOLDER),
                    identifierHint : (() => getIdentifierHint('custom user authenticator and supplier'))
                },
                {
                    name: "Dataiku Application Template",
                    type: "customAppTemplates",
                    icon: 'icon-tasks',
                    iconColor: 'universe-color more',
                    disabled: true,
                    disabledReason: "To create a new Dataiku application template, you need to create it from an existing application template project.",
                    description: "Create a new Dataiku application template"
                },
                {
                    name: "Exposition",
                    type: "customExpositions",
                    icon: 'icon-external-link',
                    iconColor: 'universe-color more',
                    description: "(Expert usage) Create a new exposition to expose webapps or API services running in containers.",
                    addDescriptors: regularJavaOnlyDescriptor,
                    addFuncMap: {
                        java: DataikuAPI.plugindev.addJavaExposition
                    },
                    identifierPlaceholder: (() => COMPONENT_IDENTIFIER_PLACEHOLDER),
                    identifierHint: (() => getIdentifierHint('exposition'))
                },
                {
                    name: "Custom Java LLM",
                    type: "customJavaLLMs",
                    icon: 'dku-icon-puzzle-piece-24',
                    iconColor: 'universe-color more',
                    description: "Create a new custom LLM",
                    addDescriptors: regularJavaOnlyDescriptor,
                    disabled: true,
                    disabledReason: "Advanced usage - Dataiku support required"
                },
                {
                    name: "Custom Python LLM",
                    type: "customPythonLLMs",
                    icon: 'dku-icon-puzzle-piece-24',
                    iconColor: 'universe-color more',
                    description: "Create a new custom LLM",
                    addDescriptors: regularPyOnlyDescriptor,
                    addFuncMap: {
                        python: ((pluginId, componentId, javaClassName, codeTemplate) => DataikuAPI.plugindev.addCustomLLM(pluginId, componentId, codeTemplate.key))
                    },
                    codeTemplates: [
                        {
                            key: "text-completion",
                            label: "Text completion",
                            description: "Text completion sample",
                        },
                        {
                            key: "embedding",
                            label: "Embedding",
                            description: "Embedding sample",
                        },
                        {
                            key: "image-generation",
                            label: "Image generation",
                            description: "Image generation sample",
                        }
                    ],
                    identifierPlaceholder: (() => COMPONENT_IDENTIFIER_PLACEHOLDER),
                    identifierHint: (() => getIdentifierHint('custom LLM'))
                }
            ];

            if ($scope.appConfig.licensedFeatures.projectStandardsAllowed) {
                $scope.contentTypeList.push({
                    name: "Project Standards check spec",
                    type: "customPythonProjectStandardsCheckSpecs",
                    icon: 'icon-code',
                    iconColor: 'universe-color more',
                    description: "(Expert usage) Create a new kind of Project Standards Check Spec",
                    addDescriptors: regularPyOnlyDescriptor,
                    addFuncMap: {
                        python: DataikuAPI.plugindev.addPythonProjectStandardsCheckSpec
                    },
                    identifierPlaceholder: (() => COMPONENT_IDENTIFIER_PLACEHOLDER),
                    identifierHint: (() => getIdentifierHint('Project Standards check spec'))
                });
            }
        };

        $scope.getComponentsTypeList = function(pluginInstallDesc) {
            if (!pluginInstallDesc || !pluginInstallDesc.content) {
                return [];
            }
            var componentsTypeList = [];
            Object.keys(pluginInstallDesc.content).forEach(function(contentType) {
                var contentList = pluginInstallDesc.content[contentType];
                if (contentList instanceof Array && contentList.length > 0) {
                    componentsTypeList.push(contentType);
                }
            });
            return componentsTypeList;
        };

        $scope.getComponentIcon = function(componentType, component = {}, iconSize = 20) {
            if (typeof componentType === 'function') {
                componentType = componentType();
            }

            let componentIcon = component.meta && component.meta.icon;
            let componentIconColor;

            if (componentType === 'customJavaPolicyHooks') {
                componentType = 'customPolicyHooks';
            }

            if (componentType === 'customJavaUserSupplier') {
                componentType = 'customUserSupplier';
            }

            if (componentType === 'customJavaUserAuthenticator') {
                componentType = 'customUserAuthenticator';
            }

            if (componentType === 'customJavaUserAuthenticatorAndSupplier') {
                componentType = 'customUserAuthenticatorAndSupplier';
            }

            if (componentType === 'javaPreparationProcessors') {
                componentType = 'customJythonProcessors';
            }

            if (componentType === 'javaFormulaFunctions') {
                return 'universe-color analysis icon-beaker';
            }

            if (componentType === 'tutorials') {
                return 'universe-color more icon-dku-tutorial';
            }

            if (componentType === 'customParameterSets') {
                return 'universe-color more icon-indent-right';
            }

            if (componentType === 'featureFlags') {
                return 'universe-color more icon-dkubird';
            }

            for (let index = 0; index < $scope.contentTypeList.length; index++) {
                let currentType = $scope.contentTypeList[index].type;
                if (typeof currentType === 'function') {
                    let currentTypes = $scope.contentTypeList[index].icons;
                    if (!currentTypes) { continue; }
                    if (currentTypes[componentType]) {
                        componentIcon = componentIcon || currentTypes[componentType].icon;
                        componentIconColor = currentTypes[componentType].iconColor;
                        break;
                    }
                } else {
                    let currentType = $scope.contentTypeList[index].type;
                    if (currentType === componentType) {
                        componentIcon = componentIcon || $scope.contentTypeList[index].icon;
                        componentIconColor = $scope.contentTypeList[index].iconColor;
                        break;
                    }
                }
            }

            return componentIconColor + ' ' + $filter('toModernIcon')(componentIcon, iconSize);
        };

        $scope.computeNbComponents = function(pluginInstallDesc) {
            if (!pluginInstallDesc || !pluginInstallDesc.content) {
                return 0;
            }
            var nbComponents = 0;
            Object.keys(pluginInstallDesc.content).forEach(function(contentType) {
                var contentList = pluginInstallDesc.content[contentType];
                if (contentList instanceof Array) {
                    nbComponents += contentList.length;
                }
            });
            if (pluginInstallDesc.desc && pluginInstallDesc.desc.featureFlags) {
                nbComponents += pluginInstallDesc.desc.featureFlags.length;
            }
            return nbComponents;
        };

        $scope.getPlugin = function() {
            if ($scope.appConfig.pluginDevGitMode === 'PLUGIN') {
                $scope.getGitFullStatus();
            }

            return DataikuAPI.plugindev.get($stateParams.pluginId).then(
                function (data) {
                    $scope.pluginData = data.data;
                    $scope.initContentTypeList();
                    //updateCodeEnvs();
                },
                setErrorInScope.bind($scope)
            );
        };

        $scope.downloadPlugin = function(pluginId = $stateParams.pluginId) {
            let url = '/dip/api/plugins/dev/download?pluginId=' + pluginId;
            downloadIframe.attr('src', url);
            $('body').append(downloadIframe);
        };

        $scope.deletePlugin = function(pluginId, pluginVersion, callback) {
            CreateModalFromTemplate("/templates/plugins/development/delete-plugin-confirm-dialog.html", $scope, null, function(newScope) {

                var handlePluginDeleted = function() {
                    WT1.event("plugin-delete", { pluginId : newScope.pluginName, pluginVersion : newScope.pluginVersion });
                    if (callback === undefined) {
                        $state.transitionTo('plugins.installed');
                    } else {
                        callback();
                    }
                }

                newScope.pluginName = pluginId || $scope.pluginData.installedDesc.desc.id;
                newScope.pluginVersion = pluginVersion || $scope.pluginData.installedDesc.desc.version;
                DataikuAPI.plugins.prepareDelete(newScope.pluginName)
                .success(function(usageStatistics) {
                    newScope.usageStatistics = usageStatistics;
                }).error(setErrorInScope.bind($scope));
                newScope.confirmPluginDeletion = function() {
                    DataikuAPI.plugindev.delete(newScope.pluginName, true).success(function(initialResponse) {
                        if (initialResponse && initialResponse.jobId && !initialResponse.hasResult) {                        
                            FutureWatcher.watchJobId(initialResponse.jobId)
                            .success(handlePluginDeleted)
                            .error(function(data, status, headers) {
                                setErrorInScope.bind($scope)(data, status, headers);
                            });
                        } else {
                            handlePluginDeleted();
                        }
                    }).error(setErrorInScope.bind($scope));
                }
            });
        };

        $scope.newComponentPopin = function() {
            CreateModalFromTemplate("/templates/plugins/development/new-component-modal.html", $scope, "NewComponentModalController");
        };

        $scope.rebuildImage = () => {
            // Get the latest default options (rather than ones got at controller init) to minimize risks of race conditions :man_shrugging:
            DataikuAPI.admin.containerExec.buildBaseImage("CDE_PLUGINS").then(({data}) => {
                FutureProgressModal.show($scope, data, "Building image").then(function(result){
                    if (result) {
                        Dialogs.infoMessagesDisplayOnly($scope, "Image built", result, result.futureLog, true);
                        return $scope.getPlugin();
                    }
                });
            }).catch(setErrorInScope.bind($scope));
        }

        const _togglePlugin = value => () => {
            const settings = angular.copy($scope.pluginData.settings);
            settings.excludedFromCDE = value;
            return DataikuAPI.plugins.saveSettings($stateParams.pluginId, $stateParams.projectKey, settings).then(data => {
                if (data.error) {
                    Dialogs.infoMessagesDisplayOnly($scope, "Update result", data);
                } else {
                    return $scope.rebuildImage();
                }
            }).catch(setErrorInScope.bind($scope));
        }
        $scope.removePluginFromImage = _togglePlugin(true)
        $scope.addPluginToImage = _togglePlugin(false)
    });


    /**
     * @ngdoc directive
     * @name pluginEditCallbacks
     * @description
     *   This directive is composed on the scope above FolderEditController.
     *   It is responsible for setting up the callbacks needed to get/set/list
     *   files in the plugin folder
     */
    app.directive('pluginEditCallbacks', function(DataikuAPI, $stateParams, Dialogs, $state) {
        return {
            scope: false,
            restrict: 'A',
            link: {
                pre : function($scope, $element, attrs) {
                    $scope.folderEditCallbacks = {
                        list: function() {
                            return DataikuAPI.plugindev.listContents($stateParams.pluginId);
                        },
                        get: function(content, sendAnyway) {
                            return DataikuAPI.plugindev.getContent($stateParams.pluginId, content.path, sendAnyway);
                        },
                        previewImageURL: function(content) {
                            return '/dip/api/plugins/dev/preview-image?pluginId=' + $stateParams.pluginId + '&path=' + encodeURIComponent(content.path) + '&contentType=' + encodeURIComponent(content.mimeType);
                        },
                        set: function(content) {
                            return DataikuAPI.plugindev.setContent($stateParams.pluginId, content.path, content.data);
                        },
                        validate: function(contentMap) {
                            return DataikuAPI.plugindev.validate($stateParams.pluginId, contentMap);
                        },
                        setAll: function(contentMap) {
                            return DataikuAPI.plugindev.setContentMultiple($stateParams.pluginId, contentMap);
                        },
                        create: function(path, isFolder) {
                            return DataikuAPI.plugindev.createContent($stateParams.pluginId, path, isFolder);
                        },
                        delete: function(content) {
                            return DataikuAPI.plugindev.deleteContent($stateParams.pluginId, content.path);
                        },
                        decompress: function(content) {
                            return DataikuAPI.plugindev.decompressContent($stateParams.pluginId, content.path);
                        },
                        rename: function(content, newName) {
                            return DataikuAPI.plugindev.renameContent($stateParams.pluginId, content.path, newName);
                        },
                        checkUpload: function(contentPath, paths) {
                            return DataikuAPI.plugindev.checkUploadContent($stateParams.pluginId, contentPath, paths);
                        },
                        upload: function(contentPath, file, callback) {
                            return DataikuAPI.plugindev.uploadContent($stateParams.pluginId, contentPath, file, callback);
                        },
                        move: function(content, to) {
                            return DataikuAPI.plugindev.moveContent($stateParams.pluginId, content.path, (to ? to.path : ''));
                        },
                        copy: function(content) {
                            return DataikuAPI.plugindev.copyContent($stateParams.pluginId, content.path);
                        },
                        downloadURL: function(content) {
                            return '/dip/api/plugins/dev/download-content?pluginId=' + $stateParams.pluginId + '&path=' + encodeURIComponent(content.path);
                        }
                    };
                    $scope.folderEditSaveWarning = 'You have unsaved changes to a plugin file, are you sure you want to leave?';
                    $scope.description = $stateParams.pluginId;
                    $scope.headerDescription = "Plugin Content"
                    $scope.rootDescription = '[plugin root]';
                    $scope.localStorageId = $stateParams.pluginId;
                }
            }
        };
    });

    app.filter("humanContentType", function() {
        var fromCamelCaseToHuman = function(str) {
            if (str == 'featureFlags') {
                return 'feature';
            }
            var humanStr = "";
            var upperCase = str.match(/[A-Z]/);
            while (upperCase) {
                humanStr += str.substring(0, upperCase.index) + " ";
                str = upperCase[0].toLowerCase() + str.substring(upperCase.index + 1);
                upperCase = str.match(/[A-Z]/);
            }
            humanStr += str;
            return humanStr;
        }

        return function(contentType) {

            let humanContentType;

            humanContentType = fromCamelCaseToHuman(contentType);

            if (contentType == "customRunnables") {
                humanContentType = "Macros";
            }

            humanContentType = humanContentType.replace("custom ", "");
            if (humanContentType === "code recipes") {
                humanContentType = "Recipes";
            }
            if (humanContentType[humanContentType.length - 1] == 's') {
                humanContentType = humanContentType.substring(0, humanContentType.length - 1);
            }
            return humanContentType;
        }
    });

    app.controller("PlugindevCreateController", function($scope, $element, DataikuAPI, _SummaryHelper, Dialogs, $state,
        WT1, TopNav, SpinnerService, FutureProgressModal, StateUtils, PluginsService, MonoFuture) {

    	_SummaryHelper.addEditBehaviour($scope, $element);

        $scope.desc = {
            id: '',
            bootstrapMode: 'EMPTY',
            gitRepository: '',
            gitCheckout: '',
            gitPath: ''
        };

        $scope.pattern = PluginsService.namingConvention;

        $scope.bootstrap = function() {
            MonoFuture($scope).wrap(DataikuAPI.plugindev.create)($scope.desc.id, $scope.desc.bootstrapMode,
                $scope.desc.gitRepository, $scope.desc.gitCheckout, $scope.desc.gitPath).success(function (data) {
                FutureProgressModal.show($scope, data, "Creating plugin").then(function(result){
                    if (result) {
                        WT1.event("plugin-dev-create");
                        $scope.dismiss();
                        DataikuAPI.plugindev.git.getFullStatus(result.details).finally(StateUtils.go.pluginDefinition(result.details));
                    }
                });
            }).error(setErrorInScope.bind($scope));
        };
    });

    app.controller("PlugindevEditionController", function($scope, DataikuAPI, $state, $stateParams, CreateModalFromTemplate, Dialogs, FutureProgressModal, TopNav, $controller, $filter, WT1, $timeout) {
        $controller('PlugindevCommonController', {$scope: $scope});
        $controller('PlugindevDefinitionController', {$scope});

        $scope.uiState = {
            envName : null,
            newEnvDeploymentMode : 'PLUGIN_MANAGED',
            state: $state
        };

        $scope.reloadPlugin = function() {
            return DataikuAPI.plugindev.reload($stateParams.pluginId).then(
                function(data) {
                    $scope.$broadcast("pluginReload");
                    return $scope.getPlugin();
                },
                setErrorInScope.bind($scope)
            );
        };

        $scope.modalCreateBranch = function(wantedBranch) {
            CreateModalFromTemplate("/templates/plugins/development/git/create-branch-modal.html", $scope, "PlugindevCreateBranchController", function(newScope){
                newScope.targetBranchName = wantedBranch || "";
            });
        };

        $scope.createBranchFromCommit = function(commitId) {
            CreateModalFromTemplate("/templates/plugins/development/git/create-branch-modal.html", $scope, "PlugindevCreateBranchController", function (newScope) {
                newScope.targetBranchName = "";
                newScope.commitId = commitId;
            });
        };

        $scope.focusBranchSearchInput = function() {
            $timeout(function() {
                angular.element('#branch-search-input').focus();
            }, 100);
        };

        $scope.reloadPlugin().then(function() {
            if ($scope.customCodeRecipeIdToOpen && $scope.pluginData && $scope.pluginData.installedDesc && $scope.pluginData.installedDesc.customCodeRecipes) {
                $scope.pluginData.installedDesc.customCodeRecipes.forEach(function(customCodeRecipe) {
                    if (customCodeRecipe.id == $scope.customCodeRecipeIdToOpen) {
                        $scope.openPluginContentInEditor(customCodeRecipe, 'customCodeRecipes');
                    }
                });
                $scope.customCodeRecipeIdToOpen = null;
            }
        });
    });

    app.controller("PlugindevDefinitionController", function($scope, DataikuAPI, StateUtils, CreateModalFromTemplate, TopNav, Dialogs, $filter, $stateParams) {
        /* Components */
        $scope.deleteComponent = function(event, content, contentType) {
            let contentName = content.id;
            event.stopPropagation();
            if (contentType) {
                contentType = $filter('humanContentType')(contentType).toLowerCase();
                if (contentType[contentType.length - 1] == 's') {
                    contentType = contentType.substring(0, contentType.length - 1);
                }
                contentName = contentType + " " + contentName;
            }

            var message = 'Are you sure you want to delete ' + contentName + ' ?';
            Dialogs.confirm($scope,'Delete ' + content.id, message).then(function() {
                DataikuAPI.plugindev.deleteContent($scope.pluginData.installedDesc.desc.id, content.folderName).success(function(data) {
                    $scope.reloadPlugin();
                }).error(setErrorInScope.bind($scope));
            });
        };

        $scope.createCodeEnvPopin = function() {
            CreateModalFromTemplate("/templates/plugins/development/code-env-creation-modal.html", $scope, "NewCodeEnvController");
        };

        $scope.removeCodeEnv = function() {
            DataikuAPI.plugindev.removeCodeEnv($scope.pluginData.installedDesc.desc.id).success(function() {
                $scope.reloadPlugin();
            }).error(setErrorInScope.bind($scope));
        }

        const FILE_TO_OPEN_MAP = {
            "customDatasets": "connector.json",
            "customCodeRecipes": "recipe.json",
            "customExporters": function(folderName) {
                if (folderName.startsWith('java')) {
                    return "jexporter.json";
                }
                return "exporter.py";
            },
            "customFormats": function(folderName) {
                if (folderName.startsWith('java')) {
                    return "jformat.json";
                }
                return "format.py";
            },
            "customPythonChecks": "check.py",
            "customPythonProbes": "probe.py",
            "customSQLProbes": "probe.sql",
            "customPythonSteps": "step.py",
            "customPythonClusters": "cluster.py",
            "customPythonCodeStudioTemplates": "codeStudio.py",
            "customPythonCodeStudioBlocks": "codeStudioBlock.py",
            "customPythonProjectStandardsCheckSpecs": "project_standards_check_spec.py",
            "customPythonTriggers": "trigger.py",
            "customRunnables": function(folderName) {
                if (folderName.startsWith('java')) {
                    return "runnable.json";
                }
                return "runnable.py";
            },
            "customWebApps": "webapp.json",
            "customStandardWebAppTemplates": "app.js",
            "customBokehWebAppTemplates": "backend.py",
            "customDashWebAppTemplates": "backend.py",
            "customShinyWebAppTemplates": "ui.R",
            "customRMarkdownReportTemplates": "script.Rmd",
            "customPythonNotebookTemplates": "notebook.ipynb",
            "customPythonDatasetNotebookTemplates": "notebook.ipynb",
            "customRNotebookTemplates": "notebook.ipynb",
            "customRDatasetNotebookTemplates": "notebook.ipynb",
            "customScalaNotebookTemplates": "notebook.ipynb",
            "customScalaDatasetNotebookTemplates": "notebook.ipynb",
            "customPreBuiltDatasetNotebookTemplates": "notebook.ipynb",
            "customJythonProcessors": "processor.py",
            "customFileSystemProviders": function(folderName) {
                if (folderName.startsWith('java')) {
                    return "fs-provider.json";
                }
                return "fs-provider.py";
            },
            "codeEnv": "desc.json",
            "customPythonPredictionAlgos": "algo.json",
            "customParameterSets": "preset.json",
            "customFields": "custom-fields.json",
            "customJavaPolicyHooks": function(folderName) {
                if (folderName.startsWith('java')) {
                    return "policy-hook.json";
                }
            },
            "customJavaUserSupplier": function(folderName) {
                if (folderName.startsWith('java')) {
                    return "custom-user-supplier.json";
                }
            },
            "customJavaUserAuthenticator": function(folderName) {
                if (folderName.startsWith('java')) {
                    return "custom-user-authenticator.json";
                }
            },
            "customJavaUserAuthenticatorAndSupplier": function(folderName) {
                 if (folderName.startsWith('java')) {
                     return "custom-user-authenticator-and-supplier.json";
                 }
            },
            "customAppTemplates": "app.json",
            "customExpositions": "exposition.json",
            "customAgents": "agent.py",
            "customAgentTools": "tool.py",
            "customGuardrails": "guardrail.py",
            "customSampleDatasets": "dataset.json",
            "customPythonLLMs": "llm.py"
        };

        $scope.hasFileToOpen = function(contentType) {
            return !!FILE_TO_OPEN_MAP[contentType];
        };

        $scope.openPluginContentInEditor = function(content, contentType) {
            var fileToOpen = FILE_TO_OPEN_MAP[contentType]
            if (fileToOpen) {
                if (typeof(fileToOpen) === "function") {
                    fileToOpen = fileToOpen(content.folderName);
                }
                fileToOpen = content.folderName + "/" + fileToOpen;
                openContentInEditor(fileToOpen);
            }
        };

        $scope.$on('PLUGIN_DEV_LIST:openCustomRecipeInEditor', function(event, id) {
            $scope.customCodeRecipeIdToOpen = id;
        });

        var openContentInEditor = function(path) {
            StateUtils.go.pluginEditor($stateParams.pluginId, path);
        };

        $scope.openPluginDescInEditor = function() {
            openContentInEditor("plugin.json");
        };

        /*
         * Filtering
         */

        $scope.filterQuery = {userQuery: ''};
        $scope.filteredContent = {};

        function filterContent(pluginInstallDesc) {
            let filteredContent = {};
            let types = $scope.getComponentsTypeList(pluginInstallDesc);
            types.forEach(function(type) {
                let filteredComponents = $filter('filter')(pluginInstallDesc.content[type], $scope.filterQuery.userQuery);
                if (filteredComponents.length) {
                    filteredContent[type] = filteredComponents;
                }
            });
            // Add feature flags as "fake components"
            if (pluginInstallDesc.desc.featureFlags) {
                const matchingFeatureFlag = $filter('filter')(pluginInstallDesc.desc.featureFlags, $scope.filterQuery.userQuery);
                if (matchingFeatureFlag.length > 0) {
                    filteredContent['featureFlags'] = $filter('filter')(pluginInstallDesc.desc.featureFlags, $scope.filterQuery.userQuery);
                    // Put in the same format as other components for simpler templates
                    filteredContent['featureFlags'] = filteredContent['featureFlags'].map(featureFlag => ({ id: featureFlag }));
                }
            }
            return filteredContent;
        }

        function filterContentOnChange() {
            if ($scope.pluginData && $scope.pluginData.installedDesc && $scope.pluginData.installedDesc.content) {
                $scope.filteredContent = filterContent($scope.pluginData.installedDesc);
            } else {
                $scope.filteredContent = {};
            }
        }

        $scope.$watch('pluginData', filterContentOnChange, true);
        $scope.$watch('filterQuery.userQuery', filterContentOnChange, true);

        $scope.getComponentsTypeListFiltered = function() {
            return Object.keys($scope.filteredContent);
        };

        $scope.validatePluginEnv = function() {
            $scope.pluginEnvUpToDate = true;
        }

        $scope.invalidatePluginEnv = function() {
            $scope.pluginEnvUpToDate = false;
        }
    });

    app.controller("PlugindevEditorController", function($scope, $stateParams) {
        $scope.filePath = $stateParams.filePath || '';
    });


    app.controller("PlugindevHistoryController", function($scope, $stateParams, DataikuAPI, $timeout, TopNav) {

        const PAGE_SIZE = 20;

        $scope.loadMore = function () {
            if ($scope.hasMore && !$scope.loading) {
                $scope.loading = true;
                DataikuAPI.plugindev.git.getLog($stateParams.pluginId, $scope.nextCommit, PAGE_SIZE).success(function (data) {
                    $scope.logEntries = ($scope.logEntries || []).concat(data.logEntries);
                    $scope.nextCommit = data.nextCommit;
                    if (!$scope.nextCommit) {
                        $scope.hasMore = false;
                    }
                    $scope.loading = false;
                }).error(function(e) {
                    $scope.loading = false;
                    setErrorInScope.bind($scope);
                });
            }
        };

        $scope.loadLogFromStart = function() {
            $scope.nextCommit = null;
            $scope.logEntries = [];
            $scope.hasMore = true;
            $scope.loadMore();
        };

        // $timeout here allows us to trigger this API call only once the plugin has been properly reloaded
        $timeout(() => {
            $scope.loadLogFromStart();
        });
    });

    function computeDefaultLanguage(contentType, previousLanguage) {
        const noPreviousLanguageDefined = previousLanguage === null;
        const previousLanguageInvalidForNewComponentType = !(previousLanguage in contentType.addFuncMap);
        const useDefaultLanguage = noPreviousLanguageDefined || previousLanguageInvalidForNewComponentType;
        if (useDefaultLanguage) {
            return contentType.addDescriptors[0].key;
        }
        return previousLanguage;
    }

    app.controller("NewComponentModalController", function($scope, $controller, $element, $timeout, $state, WT1, DKUtils,
        PluginsService, ListFilter) {
        $controller('PlugindevCommonController', {$scope});
        $controller('PlugindevDefinitionController', {$scope});

        $scope.pattern = PluginsService.namingConvention;

        $scope.newComponent = {
            contentType: null,
            contentLanguage: null,
            id: '',
            javaClassNameForPlugin: '',
            codeTemplate: null,
        };

        function resetNewComponentSettings(contentType) {
            $scope.newComponent.contentType = contentType;
            $scope.newComponent.contentLanguage =
                computeDefaultLanguage(contentType, $scope.newComponent.contentLanguage);
            // Keep current id and class name if already chosen
        }

        $scope.selectContentType = function(contentType) {
            if (contentType && contentType.disabled) return; // nice try :)
            resetNewComponentSettings(contentType);
            $timeout(function() {
                $element.find('.language-select').selectpicker('refresh');
            });
        };

        $scope.isJava = function() {
            return $scope.newComponent.contentLanguage === 'java';
        };

        $scope.requiresLanguage = function (contentType) {
            return contentType && !(contentType.addDescriptors.length === 1 && contentType.addDescriptors[0].key === 'generic');
        };

        $scope.requiresCodeTemplate = function(contentType) {
            return contentType && contentType.codeTemplates !== undefined && contentType.codeTemplates.length > 0;
        };

        $scope.create = function() {
            if ($scope.isFormValid()) {
                const addThingFunc = $scope.newComponent.contentType.addFuncMap[$scope.newComponent.contentLanguage];
                addThingFunc($scope.pluginData.installedDesc.desc.id,
                    $scope.newComponent.id,
                    $scope.newComponent.javaClassNameForPlugin,
                    $scope.newComponent.codeTemplate
                ).success(function(data) {
                    WT1.event("plugin-dev-add-" + $scope.newComponent.contentType.name + "-" + $scope.newComponent.contentLanguage);
                    $scope.dismiss();
                    $scope.getPlugin().then(function() {
                        let folderName = data.pathToFiles.substring($scope.pluginData.baseFolderPath.length);
                        if (folderName.startsWith("/")) {
                            folderName = folderName.substring(1);
                        }
                        let componentType = $scope.newComponent.contentType.type;
                        if (typeof($scope.newComponent.contentType.type) === "function") {
                            componentType = $scope.newComponent.contentType.type($scope.newComponent.contentLanguage);
                        }
                        if ($state.$current.name == "plugindev.editor") {
                            DKUtils.reloadState();
                        } else {
                            $scope.openPluginContentInEditor({folderName:folderName}, componentType);
                        }
                        $scope.reloadPlugin();
                    });
                }).error(setErrorInScope.bind($scope));
            }
        };

        $scope.validateComponentId = () => {
            return PluginsService.validateComponentId($scope.newComponent.id, $scope.pluginData.installedDesc.desc.id, $scope.filteredContent[$scope.newComponent.contentType.type] || []);
        }

        $scope.getComponentIdWarningMessages = () => {
            if (!$scope.newComponent.contentType) return null;
            const warnings = $scope.validateComponentId();
            if (warnings.length == 0) {
                return null;
            }
            if (warnings.length == 1) {
                return warnings[0];
            }
            return warnings.map(w => "• " + w).join("<br>");
        }

        $scope.isFormValid = function() {
            if (!$scope.newComponent.contentType) return false;
            const hasAddFunc = $scope.newComponent.contentLanguage && $scope.newComponent.contentType.addFuncMap[$scope.newComponent.contentLanguage];
            const hasTemplate = !$scope.requiresCodeTemplate($scope.newComponent.contentType) || $scope.newComponent.codeTemplate;
            const hasClassName = !$scope.isJava() || ($scope.newComponent.javaClassNameForPlugin && $scope.newComponent.javaClassNameForPlugin.length > 0);
            return $scope.validateComponentId().length == 0 && hasAddFunc && hasTemplate && hasClassName;
        };

        $scope.refreshComponentSearch = function() {
            let components = $scope.contentTypeList;
            components = ListFilter.filter(components, $scope.componentSearchQuery);
            $scope.filteredContentTypeList = components;
        }

        $scope.$watch("componentSearchQuery", $scope.refreshComponentSearch);
    });

    app.controller("NewCodeEnvController", function($scope, $controller, WT1) {
        $controller('PlugindevCommonController', { $scope: $scope });
        $controller('PlugindevDefinitionController', {$scope});
        $scope.codeEnvData = $scope.contentTypeList.find(contentType => { return contentType.type === 'codeEnv'});
        $scope.uiState = {contentLanguage: computeDefaultLanguage($scope.codeEnvData, null)};

        $scope.create = function() {
            if ($scope.isFormValid()) {
                const addThingFunc = $scope.codeEnvData.addFuncMap[$scope.uiState.contentLanguage];
                addThingFunc($scope.pluginData.installedDesc.desc.id, undefined, undefined, $scope.forceConda).success(function(data) {
                    WT1.event("plugin-dev-create-code-env-" + $scope.uiState.contentLanguage);
                    $scope.dismiss();
                    $scope.getPlugin().then(function() {
                        let folderName = data.pathToFiles.substring($scope.pluginData.baseFolderPath.length);
                        if (folderName.startsWith("/")) {
                            folderName = folderName.substring(1);
                        }
                        $scope.openPluginContentInEditor({folderName:folderName}, 'codeEnv');
                        $scope.reloadPlugin();
                    });
                }).error(setErrorInScope.bind($scope));
            }
        };

        $scope.isFormValid = function() {
            return $scope.uiState.contentLanguage && $scope.codeEnvData && $scope.codeEnvData.addFuncMap[$scope.uiState.contentLanguage];
        };
    });

    /*
     * to add new modules to the existing app, some hacking around angular is needed. And
     * this hacking needs to happen in a config block, so that we have access to the
     * providers.
     * see http://benohead.com/angularjs-requirejs-dynamic-loading-and-pluggable-views/ for
     * explanations.
     */
    app.config(['$controllerProvider', '$compileProvider', '$filterProvider', '$provide', '$injector',
        function ($controllerProvider, $compileProvider, $filterProvider, $provide, $injector) {
            // only offer one granularity: module (no injecting just a controller, for ex)
            app.registerModule = function (moduleName) {
                var module = angular.module(moduleName);

                if (module.requires) {
                    // recurse if needed
                    for (var i = 0; i < module.requires.length; i++) {
                        app.registerModule(module.requires[i]);
                    }
                }

                var providers = {
                        $controllerProvider: $controllerProvider,
                        $compileProvider: $compileProvider,
                        $filterProvider: $filterProvider,
                        $provide: $provide
                    };

                angular.forEach(module._invokeQueue, function(invokeArgs) {
                    var provider = providers[invokeArgs[0]];
                    provider[invokeArgs[1]].apply(provider, invokeArgs[2]);
                });
                angular.forEach(module._configBlocks, function (fn) {
                    $injector.invoke(fn);
                });
                angular.forEach(module._runBlocks, function (fn) {
                    $injector.invoke(fn);
                });
            };
        }
    ]);

    app.service('CustomUISetup', function(PluginConfigUtils, DataikuAPI, $q, $stateParams) {
        return {
            setupCallPythonDo : function($scope, errorScope, pluginId, componentId, config, side) {
                // for custom ui: communication with the backend and holding session id
                $scope.uiInteraction = {pluginId:pluginId, componentId:componentId, sessionId:null}
                // This function is called when fetching data for custom forms.
                // See in the documentation: Fetching data for custom forms.
                $scope.callPythonDo = function(payload) {
                    var deferred = $q.defer();
                    DataikuAPI.plugins.callPythonDo($scope.uiInteraction.sessionId, $scope.uiInteraction.pluginId, $scope.uiInteraction.componentId, config, payload, $scope.recipeConfig, $stateParams.projectKey, $stateParams.clusterId, side).success(function(data) {
                        $scope.uiInteraction.sessionId = data.sessionId;
                        deferred.resolve(data.data);
                    }).error(function(data, status, headers, config, statusText, xhrStatus) {
                        setErrorInScope.bind(errorScope)(data, status, headers, config, statusText, xhrStatus);
                        deferred.reject("Failed to get test result for ui");
                    });
                    return deferred.promise;
                };
            }
        };
    });

    app.directive('customTemplateWithCallPythonDo', function(CustomUISetup) {
        return {
            restrict: 'E',
            templateUrl: 'templates/plugins/development/custom-template.html',
            link: function($scope, element, attrs) {
                CustomUISetup.setupCallPythonDo(
                    $scope,
                    $scope.$eval(attrs.errorScope),
                    $scope.$eval(attrs.pluginId),
                    $scope.$eval(attrs.componentId),
                    $scope.$eval(attrs.config),
                    attrs.side
                );
            }
        };
    });

    app.directive('customParamsForm', function(PluginConfigUtils, DataikuAPI, $q, $stateParams, CustomUISetup) {
        return {
            restrict: 'EA',
            scope: {
                pluginDesc: '=',
                componentId: '=',
                desc: '=',
                config: '=',
                columnsPerInputRole: '=', // propagate for form elements (for plugin recipes)
                recipeConfig: '=',
                viewMode: '<' // only supported for auto-config-form
            },
            templateUrl: '/templates/plugins/development/custom-form.html',
            link: function($scope, element, attrs) {
                var setupDone = false;
                var updateSetup = function() {
                    if ($scope.desc == null || setupDone) {
                        // nothing to setup, just skip
                    } else {
                        if ($scope.desc.paramsModule) {
                            app.registerModule($scope.desc.paramsModule);
                        }
                        if ($scope.desc.paramsTemplate) {
                            $scope.baseTemplateUrl = "/plugins/" + $scope.pluginDesc.id + "/resource/";
                            $scope.templateUrl = $scope.baseTemplateUrl + $scope.desc.paramsTemplate;
                        }
                        setupDone = true;
                    }
                }
                updateSetup();
                $scope.$watch('desc', updateSetup);
            }
        };
    });

    app.directive('autoconfigForm', function(Debounce) {
        return {
            restrict: 'EA',
            replace: false,
            scope: {
                params: '=',
                pluginId: '=',
                componentId: '=',
                model: '=',
                rootModel: '=?',
                columnsPerInputRole: '=', // propagate for form elements (for plugin recipes)
                recipeConfig: '=',
                chart: '=',
                side: '@',
                activeDragDrop: '=',
                validity: '=',
                errorScope: '=',
                qaSelectorPrefix: '@?',
                viewMode: '=',
                isList: '=',
                objectListElementIndex: '='
            },
            templateUrl: '/templates/plugins/development/autoconfig-form.html',
            link: function ($scope) {
                if ($scope.rootModel === undefined) {
                    if ($scope.model === undefined) {
                        $scope.$watch($scope.model, () => {
                            if ($scope.rootModel === undefined) {
                                $scope.rootModel = $scope.model;
                            }
                        });
                    } else {
                        $scope.rootModel = $scope.model;
                    }
                }
                $scope.qaSelectorPrefix = $scope.qaSelectorPrefix || 'data-qa-autoconfig-form-element';
                $scope.getQaSelector = function (paramId) {
                    return `${$scope.qaSelectorPrefix}-${paramId}`;
                };


                /*
                    A trigger params is a parameter on which depends a dynamic params. ie: if A_T changes value, D_D needs to be updated
                    A_T, B_T, C_T are trigger params
                    D_D, E_D are dynamic params defined as:
                    D_D is triggered by A_T, B_T, C_T
                    E_D is triggered by C_T
                    F_D is triggered by the model
                */
                $scope.callbacksByDynamicParamId = {};  // D_D -> $D_D.myFunction, E_D -> $E_D.myFunction, F_D -> $F_D.myFunction
                $scope.dynamicParamsByTriggerParamId = {};  // A_T -> [D_D] , B_T -> [D_D], C_T -> [D_D, E_D]
                $scope.dynamicParamIdTriggeredByModelChange = [];  // [F_D]
                
                $scope.isInitializedWithDynamicParams = function() {
                    return Object.keys($scope.dynamicParamsByTriggerParamId).length !== 0 || $scope.dynamicParamIdTriggeredByModelChange.length > 0;
                }

                $scope.registerDynamicParamCallback = function (dynamicParamDesc, reloadCustomChoicesCallback) {
                    let dynamicParamToRegister = dynamicParamDesc.name;
                    $scope.callbacksByDynamicParamId[dynamicParamToRegister] = reloadCustomChoicesCallback;

                    if (!dynamicParamDesc.triggerParameters || dynamicParamDesc.triggerParameters.length === 0) {
                        // Param ${dynamicParamToRegister} is registering a callback for event on the entire model
                        $scope.dynamicParamIdTriggeredByModelChange.push(dynamicParamToRegister);
                    } else {
                        dynamicParamDesc.triggerParameters.forEach(triggerParamId => {
                            //Param ${dynamicParamToRegister} is registering a callback for event on  a specific param ${triggerParamId}
                            if ($scope.dynamicParamsByTriggerParamId[triggerParamId] === undefined) {
                                $scope.dynamicParamsByTriggerParamId[triggerParamId] = []
                            }
                            $scope.dynamicParamsByTriggerParamId[triggerParamId].push(dynamicParamToRegister);
                        })
                    }
                };

                //Aggregate all the trigger params into a set => will allow us to setup a unique watch per trigger param
                $scope.triggerParameterIdsToWatch = new Set(); // A_T, B_T, C_T

                if ($scope.params) {
                    for (const param of $scope.params) {
                        if (param.getChoicesFromPython && !param.disableAutoReload && param.triggerParameters) {
                            param.triggerParameters.forEach(triggerParam => $scope.triggerParameterIdsToWatch.add(triggerParam));
                        }
                    }
                }

                //As we will debounce the user input, we will queue here all the dynamic params that needs to be called back
                $scope.dynamicParamsNeededToBeCalled = new Set()

                for(const triggerParameterId of $scope.triggerParameterIdsToWatch) {
                    $scope.$watch(`model.${triggerParameterId}`, () => {
                        if ($scope.isInitializedWithDynamicParams()) {
                            //We add in `dynamicParamsNeededToBeCalled` all the dynamic params that are listening to this `triggerParameterId`
                            $scope.dynamicParamsByTriggerParamId[triggerParameterId].forEach(dynamicParameterId => {
                                //Register dynamic param ${dynamicParameterId} to be trigger at the next debounce
                                $scope.dynamicParamsNeededToBeCalled.add(dynamicParameterId)
                            })
                        }
                    }, true)
                }

                $scope.$watch('model', Debounce().withDelay(1000, 1000).wrap(() => {
                    if ($scope.isInitializedWithDynamicParams()) {
                        //At this point, dynamicParamsNeededToBeCalled contains all the dynamic params that were listening to specific trigger params
                        //To this list, we add all the dynamic params that are listening to the entire model
                        $scope.dynamicParamIdTriggeredByModelChange.forEach(dynamicParameterId => $scope.dynamicParamsNeededToBeCalled.add(dynamicParameterId))

                        //We trigger the callback for each of the dynamic params
                        $scope.dynamicParamsNeededToBeCalled.forEach(pythonParamId => $scope.callbacksByDynamicParamId[pythonParamId]())
                        //We clear our queue of dynamic params
                        $scope.dynamicParamsNeededToBeCalled.clear()
                    }
                }), true) 
            }
        };
    });

    function initChartWebAppBar(isLeftBar) {
        return function (scope) {
            var loadedType = null;
            scope.$watch('chart.def.$loadedDesc', function () { // all the other values are set at the same time
                if (!scope.chart || !scope.chart.def) return;
                if (scope.chart.def.$loadedDesc != null && scope.chart.def.$loadedDesc.webappType !== loadedType) {
                    scope.config = scope.chart.def.webAppConfig;
                    if (isLeftBar) {
                        scope.optionsFolds.webapp = true;
                    }

                    loadedType = scope.chart.def.$loadedDesc.webappType;

                    scope.loadedDesc = scope.chart.def.$loadedDesc;
                    scope.pluginChartDesc = scope.chart.def.$pluginChartDesc;
                    scope.pluginDesc = scope.chart.def.$pluginDesc;
                    scope.componentId = scope.chart.def.$loadedDesc.id;

                    const module = isLeftBar ? scope.pluginChartDesc.leftBarModule : scope.pluginChartDesc.topBarModule;
                    if (module) {
                        app.registerModule(module);
                    }
                    const template = isLeftBar ? scope.pluginChartDesc.leftBarTemplate : scope.pluginChartDesc.topBarTemplate;
                    if (template) {
                        scope.baseTemplateUrl = '/plugins/' + scope.pluginDesc.id + '/resource/';
                        scope.templateUrl = scope.baseTemplateUrl + template;
                    } else {
                        scope.baseTemplateUrl = null;
                        scope.templateUrl = null;
                    }
                }
            });
        };
    }

    const initChartLeftBarWebApp = initChartWebAppBar(true);
    const initChartRightBarWebApp = initChartWebAppBar(false);

    app.controller("WebAppChartLeftBarController", function($scope) {
        initChartLeftBarWebApp($scope);
    });

    app.controller("WebAppChartTopBarController", function($scope) {
        initChartRightBarWebApp($scope);
    });

    app.directive('customAdminParamsForm', function(PluginConfigUtils, DataikuAPI, $q, $stateParams) {
        return {
            restrict: 'EA',
            scope: {
                pluginDesc: '=',
                componentId: '=',
                desc: '=',
                config: '=',
                columnsPerInputRole: '=', // propagate for form elements (for plugin recipes)
                recipeConfig: '='
            },
            templateUrl: '/templates/plugins/development/custom-admin-form.html',
            link: function($scope, element, attrs) {
            }
        };
    });

    app.directive('pluginSettingsAlert', function($state) {
        return {
            restrict: 'EA',
            scope: {
                componentType :'@',
                appConfig: '=',
                hasSettings: '=',
                pluginDesc: '='
            },
            templateUrl: '/templates/plugins/development/plugin-settings-alert.html',
            link : function($scope, element, attrs) {
                if ($scope.pluginDesc) {
                    $scope.pluginLink = $scope.pluginDesc.isDev ? "plugindev.settings({pluginId: '" + $scope.pluginDesc.id + "'})" : "plugin.settings({pluginId: '" + $scope.pluginDesc.id + "'})";
                } else {
                    $scope.pluginLink = "plugins.installed";
                }
            }
        };
    });

})();

;
(function() {
'use strict';

const app = angular.module('dataiku.plugindev.git',  ['dataiku.git']);


app.controller("_PlugindevGitController", function($scope, DataikuAPI, $state, $stateParams, CreateModalFromTemplate,
                                                   Dialogs, FutureProgressModal, DKUtils, $filter, WT1, FullGitSupportService) {
    $scope.getGitFullStatus = function(cb) {
        return FullGitSupportService.getFullStatus($scope,
                    DataikuAPI.plugindev.git.getFullStatus($stateParams.pluginId),
                    cb);
    };

    $scope.modalRemoveRemote = function() {
        const callback = function(remoteName) {
            WT1.event("plugindev-git-remove-remote", {pluginId: $stateParams.pluginId});
            DataikuAPI.plugindev.git.removeRemote($stateParams.pluginId, remoteName).then(function() {
                $scope.getGitFullStatus();
                $scope.getGitBranches();
            }, setErrorInScope.bind($scope));
        };
        FullGitSupportService.removeRemote($scope, callback);
    };

    $scope.getGitBranches = function () {
         return FullGitSupportService.getBranches($scope, DataikuAPI.plugindev.git.listBranches($stateParams.pluginId));
    };

    $scope.filterBranches = function (query) {
        $scope.gitBranchesFiltered = $filter("filter")($scope.gitBranches, query);
    };

    $scope.formatTrackingCount = function(count) {
        return count != null ? count : "-";
    };

    $scope.postSaveCallback = function() {
        // We want to update the tracking count after the save when autocommit is enabled
        if ($scope.appConfig.pluginDevExplicitCommit === false && $scope.appConfig.pluginDevGitMode === 'PLUGIN') {
            $scope.getGitFullStatus();
        }
    };

    $scope.modalFetch = function() {
        WT1.event("plugindev-git-fetch", {pluginId: $stateParams.pluginId});
        FullGitSupportService.fetch($scope, DataikuAPI.plugindev.git.fetch($stateParams.pluginId));
    };

    $scope.modalPull = function() {
        WT1.event("plugindev-git-pull", {pluginId: $stateParams.pluginId});
        FullGitSupportService.pull($scope, DataikuAPI.plugindev.git.pull($stateParams.pluginId));
    };

    $scope.modalPush = function() {
        WT1.event("plugindev-git-push", {pluginId: $stateParams.pluginId});
        FullGitSupportService.push($scope, DataikuAPI.plugindev.git.push($stateParams.pluginId));
    };

    $scope.modalAddOrEditRemote = function() {
        const callback = function(remoteName, newURL) {
            WT1.event("plugindev-git-set-remote", {pluginId: $stateParams.pluginId});
            DataikuAPI.plugindev.git.setRemote($stateParams.pluginId, remoteName, newURL).then(function() {
                $scope.getGitFullStatus();
            }, setErrorInScope.bind($scope));
        };
        FullGitSupportService.editRemote($scope, callback);
    };

    $scope.switchToBranch = function(branchName) {
        WT1.event("plugindev-git-switch-branch", {pluginId: $stateParams.pluginId});
        FullGitSupportService.switchToBranch($scope, DataikuAPI.plugindev.git.switchBranch($stateParams.pluginId, branchName));
    };

    $scope.modalDeleteLocalBranches = function() {
        const callback = function(modalScope, branchesToDelete, deleteOptions) {
            WT1.event("plugindev-git-delete-branches", {pluginId: $stateParams.pluginId});
            DataikuAPI.plugindev.git.deleteBranches($stateParams.pluginId, branchesToDelete, deleteOptions).then(function() {
                $state.reload();
                modalScope.dismiss();
            }, setErrorInScope.bind(modalScope));
        };
        FullGitSupportService.deleteBranches($scope, callback);
    };

    $scope.needsExplicitCommit = function(){
        return $scope.appConfig.pluginDevExplicitCommit;
    };

    $scope.modalCommit = function() {
        CreateModalFromTemplate("/templates/plugins/development/git/commit-modal.html", $scope, "PlugindevCommitController");
    };

    $scope.getResetModes = function() {
        let modes = [];

        if ($scope.appConfig.pluginDevExplicitCommit)
            modes.push('HEAD');

        if ($scope.gitStatus.hasRemoteOrigin && $scope.gitStatus.hasTrackingCount)
            modes.push('UPSTREAM');

        return modes;
    };

    $scope.modalReset = function() {
        CreateModalFromTemplate("/templates/plugins/development/git/reset-modal.html", $scope, "PlugindevResetController");
    };

    $scope.$on('pluginReload',function() {
        $scope.getGitBranches();
    });

    $scope.canChangeRemote = true;
    $scope.canChangeBranch = true;
    $scope.canUpdateContent = true;
});


app.controller("PlugindevCreateBranchController", function($scope, $stateParams, DataikuAPI, $state) {
    $scope.createBranch = function() {
        DataikuAPI.plugindev.git.createBranch($stateParams.pluginId, $scope.targetBranchName, $scope.commitId).then(function() {
            $state.reload();
            $scope.dismiss();
        }, setErrorInScope.bind($scope));
    };
});

app.controller("PlugindevCommitController", function($scope, $stateParams, $filter, DataikuAPI, ActivityIndicator, $timeout, WT1) {
    DataikuAPI.plugindev.git.prepareCommit($stateParams.pluginId).then(function(resp) {
        $scope.preparationData = resp.data;
    }, setErrorInScope.bind($scope));

    $scope.uiState = {
        activeTab: 'message',
        message: ''
    };

    $timeout(() => {
        // Magic happens here: if commitEditorOptions is defined too early, the textarea won't properly autofocus
        $scope.commitEditorOptions = {
            mode : 'text/plain',
            lineNumbers : false,
            matchBrackets : false,
            autofocus: true,
            onLoad : function(cm) {$scope.codeMirror = cm;}
        };
    }, 100);


    $scope.gitCommit = function() {
        WT1.event("plugindev-git-commit", {pluginId: $stateParams.pluginId});
        DataikuAPI.plugindev.git.commit($stateParams.pluginId, $scope.uiState.message).then(function() {
                ActivityIndicator.success('Changes successfully committed.');
                $scope.dismiss();
                $scope.getGitFullStatus();
            },
            setErrorInScope.bind($scope));
    };
});


app.controller("PlugindevResetController", function($scope, $filter, $stateParams, DataikuAPI, ActivityIndicator, Dialogs, $state, WT1) {
    $scope.resetStrategy = $scope.getResetModes()[0];

    $scope.setStrategy = function(strategy) {
        if ($scope.getResetModes().includes(strategy)) {
            $scope.resetStrategy = strategy;
        }
    };

    $scope.gitReset = function() {
        const resetToUpstream = () => DataikuAPI.plugindev.git.resetToUpstream($stateParams.pluginId);
        const resetToHead = () => DataikuAPI.plugindev.git.resetToHead($stateParams.pluginId);
        const resetAPICall = $scope.resetStrategy === 'HEAD' ? resetToHead : resetToUpstream;
        WT1.event("plugindev-git-reset", {pluginId: $stateParams.pluginId, resetStrategy: $scope.resetStrategy});

        resetAPICall().then(function () {
                ActivityIndicator.success('Reset succeeded.');
                $state.reload();
                $scope.dismiss();
            },
            setErrorInScope.bind($scope));
    };
});


app.directive("pluginGitLog", function($controller, DataikuAPI, $stateParams) {
    return {
        templateUrl: "/templates/git/git-log.html",
        scope: {
            logEntries: '=',
            lastStatus: '=',
            objectRevertable: '=',
            objectRef: '=',
            projectRevertable: '=',
            commitRevertable: '=',
            noCommitDiff: '=',
            noAuthorLink: '=',
            createBranchFromCommit: '='
        },
        link: function ($scope, WT1, element) {
            const pluginGitAPI = {
                getRevisionsDiff: (commitFrom, commitTo) => DataikuAPI.plugindev.git.getRevisionsDiff($stateParams.pluginId, commitFrom, commitTo),
                getCommitDiff: (commitId) => DataikuAPI.plugindev.git.getCommitDiff($stateParams.pluginId, commitId),
                // eslint-disable-next-line no-console
                revertObjectToRevision: () => console.warn("`revertObjectToRevision` should not be fired on a plugin"),  // NOSONAR: OK to use console.
                revertProjectToRevision: (hash) => DataikuAPI.plugindev.git.revertPluginToRevision($stateParams.pluginId, hash),
                revertSingleCommit: (hash) => DataikuAPI.plugindev.git.revertSingleCommit($stateParams.pluginId, hash),
                createBranchFromCommit: $scope.createBranchFromCommit,
                removeTag: (tagName) => DataikuAPI.plugindev.git.removeTag($stateParams.pluginId, tagName),
                addTag: (tagRef, tagName, tagMessage) => DataikuAPI.plugindev.git.addTag($stateParams.pluginId, tagRef, tagName, tagMessage)
            };
            $controller('_gitLogControllerBase', {$scope: $scope, element: element, DataikuGitAPI: pluginGitAPI, objectType: "plugin"});
        }
    }
});


app.directive('branchPopup', function ($stateParams, DataikuAPI,$rootScope,$timeout,CreateModalFromTemplate,Dialogs) {
    return {
        controller: function ($scope) {
        },
        link:function (scope, element, attr) {
        },
        templateUrl: '/templates/plugins/development/git/branch-popup.html"'
    };
});


})();

;
(function(){
'use strict';

    const app = angular.module('dataiku.folder_edit', []);

    // Global definition for text used in menus
    let moveText = 'Move...';
    let duplicateText = 'Duplicate';
    let renameText = 'Rename...';
    let deleteText = 'Delete...';
    let createFileText = 'Create file...';
    let createFolderText = 'Create folder...';
    let uploadText = 'Upload file...';
    let downloadText = 'Download file...';
    let commitText = 'Commit and push...';
    let pullText = 'Reset from remote HEAD';
    let editText = 'Edit Git reference...';
    let unlinkText = 'Unlink remote repository...';
    let closeOtherTabText = 'Close other tabs';
    let customizeEditorSettingsText = 'Customize editor settings';
    let reloadLocalFileText = 'Reload local files';
    let manageReferenceText = 'Manage repositories...';
    let importFromGitText = 'Import from Git...';
    let updateAllText = 'Reset all from remote HEAD...';
    let commitAllText = 'Commit and push all...';
    let editInCodeStudioText = 'Edit in Code Studio...';

    // Global definition for icon used in menus
    let moveIcon = '<i class="icon-fixed-width dku-icon-arrow-right-16 text-icon"> </i>';
    let duplicateIcon = '<i class="icon-fixed-width dku-icon-copy-16 text-icon"> </i>';
    let renameIcon = '<i class="icon-fixed-width dku-icon-edit-note-16 text-icon"> </i>';
    let deleteIcon = '<i class="icon-fixed-width dku-icon-trash-16 text-icon"> </i>';
    let createFileIcon = '<i class="icon-fixed-width dku-icon-file-16 text-icon"> </i>';
    let createFolderIcon = '<i class="icon-fixed-width dku-icon-folder-open-16 text-icon"> </i>';
    let uploadIcon = '<i class="icon-fixed-width dku-icon-file-upload-16 text-icon"> </i>';
    let downloadIcon = '<i class="icon-fixed-width dku-icon-file-download-16 text-icon"> </i>';
    let commitIcon = '<i class="icon-fixed-width dku-icon-git-push-16 text-icon"> </i>';
    let pullIcon = '<i class="icon-fixed-width dku-icon-git-pull-16 text-icon"> </i>';
    let editIcon = '<i class="icon-fixed-width dku-icon-git-edit-16 text-icon"> </i>';
    let editFileIcon = '<i class="icon-fixed-width dku-icon-edit-16 text-icon"> </i>';
    let unlinkIcon = '<i class="icon-fixed-width dku-icon-unlink-16 text-icon"> </i>';
    let closeOtherTabIcon = '<i class="icon-fixed-width dku-icon-dismiss-16 text-icon"> </i>';
    let customizeEditorSettingsIcon = '<i class="icon-fixed-width dku-icon-gear-16 text-icon"> </i>';
    let reloadLocalFileIcon = '<i class="icon-fixed-width dku-icon-arrow-circular-16 text-icon"> </i>';
    let manageReferenceIcon = editIcon;
    let importFromGitIcon = '<i class="icon-fixed-width dku-icon-git-import-16 text-icon"> </i>';
    let updateAllIcon = pullIcon;
    let commitAllIcon = commitIcon;

//' +  + '
    /**
     * @ngdoc directive
     * @name zoneEditCallbacks
     * @description
     *   This directive is composed on the scope above FolderEditController.
     *   It is responsible for setting up the callbacks needed to get/set/list
     *   files in a zone of the admin section.
     */
    app.directive('zoneEditCallbacks', function(DataikuAPI, $stateParams, Dialogs, $state) {
        return {
            scope: false,
            restrict: 'A',
            link: {
                pre: function($scope, $element, attrs) {
                    var zone = attrs.zone;
                    $scope.folderEditCallbacks = {
                        list: function() {
                            return DataikuAPI.admin.folderEdit.listContents(zone);
                        },
                        get: function(content, sendAnyway) {
                            return DataikuAPI.admin.folderEdit.getContent(zone, content.path, sendAnyway);
                        },
                        previewImageURL: function(content) {
                            return '/dip/api/admin/folder-edition/preview-image?type=' + zone + '&path=' + encodeURIComponent(content.path) + '&contentType=' + encodeURIComponent(content.mimeType);
                        },
                        set: function(content) {
                            return DataikuAPI.admin.folderEdit.setContent(zone, content.path, content.data);
                        },
                        // validate: function(contentMap) {
                        //     return DataikuAPI.admin.folderEdit.validate(zone, contentMap);
                        // },
                        setAll: function(contentMap) {
                            return DataikuAPI.admin.folderEdit.setContentMultiple(zone, contentMap);
                        },
                        create: function(path, isFolder) {
                            return DataikuAPI.admin.folderEdit.createContent(zone, path, isFolder);
                        },
                        delete: function(content) {
                            return DataikuAPI.admin.folderEdit.deleteContent(zone, content.path);
                        },
                        decompress: function(content) {
                            return DataikuAPI.admin.folderEdit.decompressContent(zone, content.path);
                        },
                        rename: function(content, newName) {
                            return DataikuAPI.admin.folderEdit.renameContent(zone, content.path, newName);
                        },
                        checkUpload: function(contentPath, paths) {
                            return DataikuAPI.admin.folderEdit.checkUploadContent(zone, contentPath, paths);
                        },
                        upload: function(contentPath, file, callback) {
                            return DataikuAPI.admin.folderEdit.uploadContent(zone, contentPath, file, callback);
                        },
                        move: function(content, to) {
                            return DataikuAPI.admin.folderEdit.moveContent(zone, content.path, (to ? to.path : ''));
                        },
                        copy: function(content) {
                            return DataikuAPI.admin.folderEdit.copyContent(zone, content.path);
                        },
                        downloadURL: function(content) {
                            return '/dip/api/admin/folder-edition/download-content?type=' + zone + '&path=' + encodeURIComponent(content.path);
                        }
                    };
                    $scope.folderEditSaveWarning = 'You have unsaved changes to a file, are you sure you want to leave?';
                    $scope.rootDescription = attrs.rootDescription || '[' + zone + ']';
                    $scope.description =  $state.includes('libedition.libpython') ? 'lib-python' : $state.includes('libedition.libr') ? 'lib-r' : 'local-static';
                    $scope.headerDescription = $state.includes('libedition.localstatic') ? "Web Resources Content" : "Library Content";
                    $scope.localStorageId = $state.includes('libedition.libpython') ? 'lib-python' : $state.includes('libedition.libr') ? 'lib-r' : 'local-static';
                }
            }
        };
    });

    /**
     * @ngdoc directive
     * @name projectZoneEditCallbacks
     * @description
     *   same as zoneEditCallbacks but for the folders inside a project
     */
    app.directive('projectZoneEditCallbacks', function(DataikuAPI, $stateParams, Dialogs, $state, CreateModalFromTemplate, FutureProgressModal, DKUtils, CodeStudiosService) {
        return {
            scope: false,
            restrict: 'A',
            link: {
                pre: function($scope, $element, attrs) {
                    var zone = attrs.zone;
                    var projectKey = $stateParams.projectKey;
                    $scope.folderEditCallbacks = {
                        list: function() {
                            return DataikuAPI.projects.folderEdit.listContents(projectKey, zone);
                        },
                        get: function(content, sendAnyway) {
                            return DataikuAPI.projects.folderEdit.getContent(projectKey, zone, content.path, sendAnyway);
                        },
                        previewImageURL: function(content) {
                            return DataikuAPI.projects.folderEdit.previewImageURL(projectKey, zone, content.path, content.mimeType);
                        },
                        set: function(content, gitLib) {
                            $scope.folderEditCallbacks.setDirty(gitLib);
                            return DataikuAPI.projects.folderEdit.setContent(projectKey, zone, content.path, content.data);
                        },
                        // validate: function(contentMap) {
                        //     return DataikuAPI.projects.folderEdit.validate(projectKey, zone, contentMap);
                        // },
                        setAll: function(contentMap, gitLibs) {
                            DataikuAPI.projects.git.setAllDirty(projectKey, gitLibs);
                            return DataikuAPI.projects.folderEdit.setContentMultiple(projectKey, zone, contentMap);
                        },
                        setDirty: function(gitLib) {
                            if (gitLib && gitLib.length !== 0) {
                                DataikuAPI.projects.git.setDirty(projectKey, gitLib);
                            }
                        },
                        getDirty: function (gitLib) {
                            return DataikuAPI.projects.git.getDirty(projectKey, gitLib);
                        },
                        getAllDirty: function () {
                            return DataikuAPI.projects.git.getDirty(projectKey);
                        },
                        create: function(path, isFolder) {
                            return DataikuAPI.projects.folderEdit.createContent(projectKey, zone, path, isFolder);
                        },
                        delete: function(content) {
                            return DataikuAPI.projects.folderEdit.deleteContent(projectKey, zone, content.path);
                        },
                        decompress: function(content) {
                            return DataikuAPI.projects.folderEdit.decompressContent(projectKey, zone, content.path);
                        },
                        rename: function(content, newName) {
                            return DataikuAPI.projects.folderEdit.renameContent(projectKey, zone, content.path, newName);
                        },
                        checkUpload: function(contentPath, paths) {
                            return DataikuAPI.projects.folderEdit.checkUploadContent(projectKey, zone, contentPath, paths);
                        },
                        upload: function(contentPath, file, callback) {
                            return DataikuAPI.projects.folderEdit.uploadContent(projectKey, zone, contentPath, file, callback);
                        },
                        move: function(content, to) {
                            return DataikuAPI.projects.folderEdit.moveContent(projectKey, zone, content.path, (to ? to.path : ''));
                        },
                        copy: function(content) {
                            return DataikuAPI.projects.folderEdit.copyContent(projectKey, zone, content.path);
                        },
                        downloadURL: function(content) {
                            return DataikuAPI.projects.folderEdit.downloadURL(projectKey, zone, content.path);
                        },
                        editInCodeStudio: function(content) {
                            return CodeStudiosService.editFileInCodeStudio($scope,'project_lib_versioned', content.path);
                        }
                    };

                    $scope.folderEditSaveWarning = 'You have unsaved changes to a file, are you sure you want to leave?';
                    $scope.description =  "lib";
                    $scope.headerDescription = $state.includes('projects.project.libedition.localstatic') ? "Web Resources Content" : "Library Content"
                    $scope.localStorageId = "lib" + "-" + projectKey;
                }
            }
        };
    });

    /**
     * @ngdoc directive
     * @name resourcesProjectZoneEditCallbacks
     * @description
     *   same as zoneEditCallbacks but for the folders inside a project
     */
    app.directive('resourcesProjectZoneEditCallbacks', function(DataikuAPI, $stateParams, Dialogs, $state, CreateModalFromTemplate, FutureProgressModal, DKUtils, CodeStudiosService) {
        return {
            scope: false,
            restrict: 'A',
            link: {
                pre: function($scope, $element, attrs) {
                    var zone = attrs.zone;
                    var projectKey = $stateParams.projectKey;
                    $scope.folderEditCallbacks = {
                        list: function() {
                            return DataikuAPI.projects.resourcesFolderEdit.listContents(projectKey, zone);
                        },
                        get: function(content, sendAnyway) {
                            return DataikuAPI.projects.resourcesFolderEdit.getContent(projectKey, zone, content.path, sendAnyway);
                        },
                        previewImageURL: function(content) {
                            return DataikuAPI.projects.resourcesFolderEdit.previewImageURL(projectKey, zone, content.path, content.mimeType);
                        },
                        set: function(content) {
                            return DataikuAPI.projects.resourcesFolderEdit.setContent(projectKey, zone, content.path, content.data);
                        },
                        // validate: function(contentMap) {
                        //     return DataikuAPI.projects.resourcesFolderEdit.validate(projectKey, zone, contentMap);
                        // },
                        setAll: function(contentMap) {
                            return DataikuAPI.projects.resourcesFolderEdit.setContentMultiple(projectKey, zone, contentMap);
                        },
                        create: function(path, isFolder) {
                            return DataikuAPI.projects.resourcesFolderEdit.createContent(projectKey, zone, path, isFolder);
                        },
                        delete: function(content) {
                            return DataikuAPI.projects.resourcesFolderEdit.deleteContent(projectKey, zone, content.path);
                        },
                        decompress: function(content) {
                            return DataikuAPI.projects.resourcesFolderEdit.decompressContent(projectKey, zone, content.path);
                        },
                        rename: function(content, newName) {
                            return DataikuAPI.projects.resourcesFolderEdit.renameContent(projectKey, zone, content.path, newName);
                        },
                        checkUpload: function(contentPath, paths) {
                            return DataikuAPI.projects.resourcesFolderEdit.checkUploadContent(projectKey, zone, contentPath, paths);
                        },
                        upload: function(contentPath, file, callback) {
                            return DataikuAPI.projects.resourcesFolderEdit.uploadContent(projectKey, zone, contentPath, file, callback);
                        },
                        move: function(content, to) {
                            return DataikuAPI.projects.resourcesFolderEdit.moveContent(projectKey, zone, content.path, (to ? to.path : ''));
                        },
                        copy: function(content) {
                            return DataikuAPI.projects.resourcesFolderEdit.copyContent(projectKey, zone, content.path);
                        },
                        downloadURL: function(content) {
                            return DataikuAPI.projects.resourcesFolderEdit.downloadURL(projectKey, zone, content.path);
                        },
                        editInCodeStudio: function(content) {
                            return CodeStudiosService.editFileInCodeStudio($scope,'project_lib_resources', content.path);
                        }
                    };

                    $scope.isInProjectResources = true;
                    $scope.folderEditSaveWarning = 'You have unsaved changes to a file, are you sure you want to leave?';
                    $scope.description =  "lib";
                    $scope.headerDescription = $state.includes('projects.project.libedition.localstatic') ? "Web Resources Content" : "Library Content"
                    $scope.localStorageId = "lib" + "-" + projectKey;
                }
            }
        };
    });

    /**
     * @ngdoc directive
     * @name codeStudioZoneEditCallbacks
     * @description
     *   same as zoneEditCallbacks but for the folders of a CodeStudio
     */
    app.directive('codeStudioZoneEditCallbacks', function(DataikuAPI, $stateParams, Dialogs, $state, CreateModalFromTemplate, FutureProgressModal, DKUtils, CodeStudiosService) {
        return {
            scope: false,
            restrict: 'A',
            link: {
                pre: function($scope, $element, attrs) {
                    var zone = attrs.zone;
                    var projectKey = $stateParams.projectKey;
                    var codeStudioObjectId = $stateParams.codeStudioObjectId;
                    $scope.folderEditCallbacks = {
                        list: function() {
                            return DataikuAPI.codeStudioObjects.folderEdit.listContents(projectKey, codeStudioObjectId, zone);
                        },
                        get: function(content, sendAnyway) {
                            return DataikuAPI.codeStudioObjects.folderEdit.getContent(projectKey, codeStudioObjectId, zone, content.path, sendAnyway);
                        },
                        previewImageURL: function(content) {
                            return '/dip/api/code-studio-objects/folder-edition/preview-image?projectKey=' + projectKey + '&codeStudioObjectId=' + codeStudioObjectId + '&type=' + zone + '&path=' + encodeURIComponent(content.path) + '&contentType=' + encodeURIComponent(content.mimeType);
                        },
                        set: function(content) {
                            return DataikuAPI.codeStudioObjects.folderEdit.setContent(projectKey, codeStudioObjectId, zone, content.path, content.data);
                        },
                        // validate: function(contentMap) {
                        //     return DataikuAPI.codeStudioObjects.folderEdit.validate(projectKey, codeStudioObjectId, zone, contentMap);
                        // },
                        setAll: function(contentMap) {
                            return DataikuAPI.codeStudioObjects.folderEdit.setContentMultiple(projectKey, codeStudioObjectId, zone, contentMap);
                        },
                        create: function(path, isFolder) {
                            return DataikuAPI.codeStudioObjects.folderEdit.createContent(projectKey, codeStudioObjectId, zone, path, isFolder);
                        },
                        delete: function(content) {
                            return DataikuAPI.codeStudioObjects.folderEdit.deleteContent(projectKey, codeStudioObjectId, zone, content.path);
                        },
                        decompress: function(content) {
                            return DataikuAPI.codeStudioObjects.folderEdit.decompressContent(projectKey, codeStudioObjectId, zone, content.path);
                        },
                        rename: function(content, newName) {
                            return DataikuAPI.codeStudioObjects.folderEdit.renameContent(projectKey, codeStudioObjectId, zone, content.path, newName);
                        },
                        checkUpload: function(contentPath, paths) {
                            return DataikuAPI.codeStudioObjects.folderEdit.checkUploadContent(projectKey, codeStudioObjectId, zone, contentPath, paths);
                        },
                        upload: function(contentPath, file, callback) {
                            return DataikuAPI.codeStudioObjects.folderEdit.uploadContent(projectKey, codeStudioObjectId, zone, contentPath, file, callback);
                        },
                        move: function(content, to) {
                            return DataikuAPI.codeStudioObjects.folderEdit.moveContent(projectKey, codeStudioObjectId, zone, content.path, (to ? to.path : ''));
                        },
                        copy: function(content) {
                            return DataikuAPI.codeStudioObjects.folderEdit.copyContent(projectKey, codeStudioObjectId, zone, content.path);
                        },
                        downloadURL: function(content) {
                            return DataikuAPI.codeStudioObjects.folderEdit.downloadURL(projectKey, codeStudioObjectId, zone, content.path);
                        }
                    };

                    $scope.folderEditSaveWarning = 'You have unsaved changes to a file, are you sure you want to leave?';
                    $scope.description =  "files";
                    $scope.headerDescription = "Files"
                    $scope.localStorageId = zone + "-" + projectKey + "-" + codeStudioObjectId;
                }
            }
        };
    });

    /**
     * @ngdoc directive
     * @name codeStudioTemplateResourcesCallbacks
     * @description
     *   same as zoneEditCallbacks but for the resources of a CodeStudio template
     */
    app.directive('codeStudioTemplateResourcesCallbacks', function(DataikuAPI, $stateParams, Dialogs, $state, CreateModalFromTemplate, FutureProgressModal, DKUtils, CodeStudiosService) {
        return {
            scope: false,
            restrict: 'A',
            link: {
                pre: function($scope, $element, attrs) {
                    const codeStudioTemplateId = $stateParams.codeStudioTemplateId;
                    $scope.folderEditCallbacks = {
                        list: function() {
                            return DataikuAPI.codeStudioTemplates.resources.listContents(codeStudioTemplateId);
                        },
                        get: function(content, sendAnyway) {
                            return DataikuAPI.codeStudioTemplates.resources.getContent(codeStudioTemplateId, content.path, sendAnyway);
                        },
                        previewImageURL: function(content) {
                            return '/dip/api/code-studio-templates/resources/preview-image?codeStudioTemplateId=' + codeStudioTemplateId + '&path=' + encodeURIComponent(content.path) + '&contentType=' + encodeURIComponent(content.mimeType);
                        },
                        set: function(content) {
                            return DataikuAPI.codeStudioTemplates.resources.setContent( codeStudioTemplateId, content.path, content.data);
                        },
                        // validate: function(contentMap) {
                        //     return DataikuAPI.codeStudioTemplates.resources.validate( codeStudioTemplateId, contentMap);
                        // },
                        setAll: function(contentMap) {
                            return DataikuAPI.codeStudioTemplates.resources.setContentMultiple( codeStudioTemplateId, contentMap);
                        },
                        create: function(path, isFolder) {
                            return DataikuAPI.codeStudioTemplates.resources.createContent( codeStudioTemplateId, path, isFolder);
                        },
                        delete: function(content) {
                            return DataikuAPI.codeStudioTemplates.resources.deleteContent( codeStudioTemplateId, content.path);
                        },
                        decompress: function(content) {
                            return DataikuAPI.codeStudioTemplates.resources.decompressContent( codeStudioTemplateId, content.path);
                        },
                        rename: function(content, newName) {
                            return DataikuAPI.codeStudioTemplates.resources.renameContent( codeStudioTemplateId, content.path, newName);
                        },
                        checkUpload: function(contentPath, paths) {
                            return DataikuAPI.codeStudioTemplates.resources.checkUploadContent( codeStudioTemplateId, contentPath, paths);
                        },
                        upload: function(contentPath, file, callback) {
                            return DataikuAPI.codeStudioTemplates.resources.uploadContent( codeStudioTemplateId, contentPath, file, callback);
                        },
                        move: function(content, to) {
                            return DataikuAPI.codeStudioTemplates.resources.moveContent( codeStudioTemplateId, content.path, (to ? to.path : ''));
                        },
                        copy: function(content) {
                            return DataikuAPI.codeStudioTemplates.resources.copyContent( codeStudioTemplateId, content.path);
                        },
                        downloadURL: function(content) {
                            return DataikuAPI.codeStudioTemplates.resources.downloadURL( codeStudioTemplateId, content.path);
                        }
                    };

                    $scope.folderEditSaveWarning = 'You have unsaved changes to a file, are you sure you want to leave?';
                    $scope.description =  "files";
                    $scope.headerDescription = "Resource files"
                    $scope.localStorageId = "codestudio-templates-" + codeStudioTemplateId + "-resources";
                }
            }
        };
    });

    /**
     * @ngdoc directive
     * @name userZoneEditCallbacks
     * @description
     *   same as zoneEditCallbacks but for the folders of a user
     */
    app.directive('userZoneEditCallbacks', function(DataikuAPI, $stateParams, Dialogs, $state, CreateModalFromTemplate, FutureProgressModal, DKUtils) {
        return {
            scope: false,
            restrict: 'A',
            link: {
                pre: function($scope, $element, attrs) {
                    var zone = attrs.zone;
                    $scope.folderEditCallbacks = {
                        list: function() {
                            return DataikuAPI.profile.folderEdit.listContents(zone);
                        },
                        get: function(content, sendAnyway) {
                            return DataikuAPI.profile.folderEdit.getContent(zone, content.path, sendAnyway);
                        },
                        previewImageURL: function(content) {
                            return DataikuAPI.profile.folderEdit.previewImageURL(zone, content.path, content.mimeType);
                        },
                        set: function(content) {
                            return DataikuAPI.profile.folderEdit.setContent(zone, content.path, content.data);
                        },
                        // validate: function(contentMap) {
                        //     return DataikuAPI.profile.folderEdit.validate(zone, contentMap);
                        // },
                        setAll: function(contentMap) {
                            return DataikuAPI.profile.folderEdit.setContentMultiple(zone, contentMap);
                        },
                        create: function(path, isFolder) {
                            return DataikuAPI.profile.folderEdit.createContent(zone, path, isFolder);
                        },
                        delete: function(content) {
                            return DataikuAPI.profile.folderEdit.deleteContent(zone, content.path);
                        },
                        decompress: function(content) {
                            return DataikuAPI.profile.folderEdit.decompressContent(zone, content.path);
                        },
                        rename: function(content, newName) {
                            return DataikuAPI.profile.folderEdit.renameContent(zone, content.path, newName);
                        },
                        checkUpload: function(contentPath, paths) {
                            return DataikuAPI.profile.folderEdit.checkUploadContent(zone, contentPath, paths);
                        },
                        upload: function(contentPath, file, callback) {
                            return DataikuAPI.profile.folderEdit.uploadContent(zone, contentPath, file, callback);
                        },
                        move: function(content, to) {
                            return DataikuAPI.profile.folderEdit.moveContent(zone, content.path, (to ? to.path : ''));
                        },
                        copy: function(content) {
                            return DataikuAPI.profile.folderEdit.copyContent(zone, content.path);
                        },
                        downloadURL: function(content) {
                            return DataikuAPI.profile.folderEdit.downloadURL(zone, content.path);
                        }
                    };

                    $scope.folderEditSaveWarning = 'You have unsaved changes to a file, are you sure you want to leave?';
                    $scope.description =  "files";
                    $scope.headerDescription = "Files"
                    $scope.localStorageId = zone;
                }
            }
        };
    });

    /**
     * @ngdoc directive
     * @name projectZoneGitRefCallbacks
     * @description
     *   This directive is composed on the scope above FolderEditController.
     *   It is responsible for setting up the callbacks needed to get/set/list/rm git references.
     */
    app.directive('projectZoneGitRefCallbacks', function(DataikuAPI, $stateParams, WT1) {
        return {
            scope : false,
            restrict : 'A',
            link : {
                pre : function($scope) {
                    $scope.gitRefCallbacks = {
                        set: function (gitRef, gitRefPath, addPythonPath) {
                            WT1.event("project-libs-git-refs-save");
                            return DataikuAPI.git.setProjectGitRef($stateParams.projectKey, gitRef, gitRefPath, addPythonPath);
                        },
                        rm: function (gitRefPath, deleteDirectory) {
                            WT1.event("project-libs-git-refs-rm", {"delete-directory": deleteDirectory});
                            return DataikuAPI.git.rmProjectGitRef($stateParams.projectKey, gitRefPath, deleteDirectory);
                        },
                        pullOne: function (gitRefPath) {
                            WT1.event("project-libs-git-refs-pull-one");
                            return DataikuAPI.git.pullProjectGitRef($stateParams.projectKey, gitRefPath);
                        },
                        pullAll: function () {
                            WT1.event("project-libs-git-refs-pull-all");
                            return DataikuAPI.git.pullProjectGitRefs($stateParams.projectKey);
                        },
                        pushOne: function (commitMessage, gitRefPath) {
                            WT1.event("project-libs-git-ref-push-one");
                            return DataikuAPI.git.pushProjectGitRefs($stateParams.projectKey, commitMessage, gitRefPath);
                        },
                        pushAll: function (commitMessage) {
                            WT1.event("project-libs-git-ref-push-all");
                            return DataikuAPI.git.pushProjectGitRefs($stateParams.projectKey, commitMessage);
                        },
                        revertAllFiles: function(gitLib) {
                            WT1.event("project-libs-git-ref-revert-all");
                            return DataikuAPI.git.revertAllFiles($stateParams.projectKey, gitLib);
                        },
                        markAsResolved: function(fileName, gitLibPath) {
                            WT1.event("project-libs-git-ref-mark-resolved");
                            return DataikuAPI.git.markAsResolved($stateParams.projectKey, fileName, gitLibPath);
                        }
                    }
                }
            }
        };
    });

    app.controller('TopLevelFolderEditionController', function($scope, DataikuAPI, $state, $stateParams, CreateModalFromTemplate, TopNav) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");
        TopNav.setNoItem();

        $scope.pythonEmptyCta = {
            title: "No shared python code on this " + $scope.wl.productLongName + " instance.",
            text: "Create your own libraries or helpers and share them within all the " + $scope.wl.productShortName + " instance. The contents of 'lib-python' are accessible to python recipes and notebooks just like regular python libraries.",
            image: "static/dataiku/images/empty-states/libraries/empty_state_libraries_large.svg",
            btnAction: "create",
            btnLabel: "Create your first shared python file"
        }

        $scope.rEmptyCta = {
            title: "No shared R code on this " + $scope.wl.productLongName + " instance.",
            text: "Create your own libraries and share them within all the " + $scope.wl.productShortName + " instance. The contents of 'lib-r' are accessible to R recipes and notebooks just like regular R libraries.",
            image: "static/dataiku/images/empty-states/libraries/empty_state_libraries_large.svg",
            btnAction: "create",
            btnLabel: "Create your first shared R file"
        }
    });

    app.controller('TopLevelLocalStaticEditorController', function($scope, DataikuAPI, $state, $stateParams, CreateModalFromTemplate, TopNav, $rootScope) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");
        TopNav.setNoItem();

        $scope.emptyCta = {
            title: "No static web resources on this " + $scope.wl.productLongName + " instance.",
            text: "Create and upload your static web resources and use them for your webapps within all the " + $scope.wl.productShortName + " instance. Right click on local-static root folder and create or upload your files.",
            image: "static/dataiku/images/empty-states/libraries/empty_state_libraries_large.svg",
            btnAction: "upload",
            btnLabel: "Upload your first resource"
        }
    });

    app.controller('ProjectFolderVersionedEditionController', function($scope, DataikuAPI, $state, $stateParams, CreateModalFromTemplate, TopNav, $rootScope) {
        TopNav.setLocation(TopNav.TOP_NOTEBOOKS, 'libraries', TopNav.TABS_NONE, null);
        TopNav.setNoItem();

        $scope.projectNoTabsCta = {
            title: "Share and reuse code with project Libraries",
            text: "Develop Libraries to reuse and share code in Python or R Recipes and Notebooks across your project.",
            image: "static/dataiku/images/empty-states/libraries/empty_state_libraries_large.svg",
            knowledgeBankLink: "/code/shared/concept-project-libraries",
        };
    });

    app.controller('ProjectFolderVersionedHistoryController', function($scope, $stateParams, TopNav) {
        TopNav.setLocation(TopNav.TOP_NOTEBOOKS, 'libraries', TopNav.TABS_NONE, null);
        TopNav.setNoItem();
    });

    app.controller('ProjectFolderResourcesEditionController', function($scope, DataikuAPI, $state, $stateParams, CreateModalFromTemplate, TopNav, $rootScope) {
        TopNav.setLocation(TopNav.TOP_NOTEBOOKS, 'libraries', TopNav.TABS_NONE, null);
        TopNav.setNoItem();

        $scope.projectNoTabsCta = {
            title: "Project resources",
            text: "Create your own resources and share them within the project."
        };
    });

    app.directive('folderContentEditor', function(DataikuAPI, $stateParams, Dialogs, $state, CreateModalFromTemplate, $q,
                                                  $timeout, LocalStorage, $rootScope, openDkuPopin, Logger, CodeMirrorSettingService,
                                                  FutureProgressModal, DKUtils, CodeStudiosService) {
        return {
            scope: true,
            restrict: 'A',
            templateUrl: '/templates/plugins/development/fragments/folder-content-editor.html',
            link: function($scope, $element, attrs) {
                $scope.editorOptions = null;
                $scope.uiState = {
                    foldRoot: false
                };
                // Enable editor by default. If user does not have write access, error will be displayed on save/commit
                $scope.canEdit = true;
                // In some cases, the project key is not available in the state params (e.g. in the plugin development page)
                if ($stateParams.projectKey) {
                    DataikuAPI.projects.getSummary($stateParams.projectKey).success(data => {
                        $scope.canEdit = data.object.canWriteProjectContent;
                    });
                }
                /* Attributes with inherited scope */
                $scope.emptyCta = $scope.$eval(attrs.emptyCta);
                $scope.noTabsCta = $scope.$eval(attrs.noTabsCta);
                $scope.canCommit = $scope.$eval(attrs.canCommit);
                $scope.commitFn = $scope[attrs.commitFn];
                const postSaveCallback = () => {
                    if (attrs.postSaveCallback) {
                        $scope[attrs.postSaveCallback]();
                    }
                };

                const initialPath = attrs.initialPath || $stateParams.initialPath;
                if (initialPath) {
                    $scope.openOnLoad = initialPath;
                }

                /*
                 * Listing plugin content
                 */

                $scope.listContents = function() {
                    return $scope.folderEditCallbacks.list().success(data => {
                        const recGetExpandedState = function(content, map) {
                            if (content.children != null) {
                                map[content.path] = content.expanded || false;
                                content.children.forEach(subContent => recGetExpandedState(subContent, map));
                            }
                        };

                        const recSetExpandedState = (content, map) => {
                            if (content.children != null) {
                                content.expanded = map[content.path] || false;
                            }
                        };
                        const recSetDepth = function(content, depth) {
                            content.depth = depth;
                        };

                        const recSetGit = (externalLibs, content, gitSubPath, gitRefPath) => {
                            if (gitSubPath || content.path in externalLibs.gitReferences) {
                                content.fromGit = true;
                                if (gitSubPath && gitRefPath) {
                                    const name = content.path.substring(gitRefPath.length + 1);
                                    content.gitLib = gitRefPath;
                                    const gitRef = externalLibs.gitReferences[gitRefPath];
                                    if (gitRef && gitRef.conflictingFiles != null && gitRef.conflictingFiles.includes(name)) {
                                        content.isInConflict = true;
                                    }
                                    if (gitRef && gitRef.resolvedFiles && gitRef.resolvedFiles.includes(name)) {
                                        content.isResolved = true;
                                    }
                                }
                            }
                            if (externalLibs.pythonPath.includes(content.path)) {
                                content.inPythonPath = true;
                            }
                            if (externalLibs.rsrcPath.includes(content.path)) {
                                content.inRSrcPath = true;
                            }
                            if (content.children != null) {
                                content.children.forEach(childContent => {
                                    recSetGit(externalLibs, childContent, content.fromGit, !gitSubPath ? content.path : gitRefPath);
                                    if (childContent.isInConflict) {
                                        content.isInConflict = true;
                                    }
                                    if (childContent.isResolved) {
                                        content.isResolved = true;
                                    }
                                });
                            }
                        };

                        const handleContent = (content, states, depth) => {
                            recSetExpandedState(content, states);
                            recSetDepth(content, depth);
                            if (content.children != null) {
                                content.children.forEach(childContent => handleContent(childContent, states, depth + 1));
                            }
                        }

                        const oldStates = {};
                        // save the expanded states
                        if ($scope.devContents != null ) {
                            $scope.devContents.forEach(content => recGetExpandedState(content, oldStates));
                        }
                        // set the new state of the contents tree
                        $scope.devContents = data;
                        if ($scope.gitRefCallbacks) {
                            DataikuAPI.git.getProjectExternalLibs($stateParams.projectKey).then(result => {
                                $scope.externalLibs = result.data;
                                $scope.gitReferences = result.data.gitReferences;

                                if ($scope.devContents != null ) {
                                    $scope.devContents.forEach(content => recSetGit($scope.externalLibs, content, false, content.path));
                                }
                            }, setErrorInScope.bind($scope));
                        }
                        if ($scope.devContents != null ) {
                            $scope.devContents.forEach(content => handleContent(content, oldStates, 1, false, content.path));
                        }
                    }).error(setErrorInScope.bind($scope));
                };

                $scope.listContents().then(() => {
                    openSavedTabs();
                    if ($scope.openOnLoad) {
                        openFileFromExternal($scope.openOnLoad);
                        $scope.openOnLoad = null;
                    }
                });

                $scope.reloadContents = () => {
                    return $scope.listContents().then(() => {
                        $scope.tabsList.forEach($scope.reloadTab);
                        $scope.updateActiveTab($scope.activeTabIndex);
                    });
                }

                $scope.reloadTab = (tab, index) => {
                    const filePaths = searchInDevContents(tab.path);
                    if (filePaths && filePaths.length > 0) {
                        $scope.tabsList[index] = filePaths[filePaths.length - 1];
                        $scope.tabsList[index].needReload = true;
                    }
                }

                $scope.sortFolder = function(content) {
                    content.sort(function(c1, c2) {
                        if (c1.children && !c2.children) {
                            return -1;
                        }
                        if (!c1.children && c2.children) {
                            return 1;
                        }
                        return c1.name > c2.name ? 1 : -1;
                    });
                    return content;
                };

                /*
                 * Opening and closing tabs
                 */

                $scope.originalContentMap = {};
                $scope.editedContentMap = {};
                $scope.activeTabIndex = -1;

                $scope.openFile = function(file) {
                    if (!$scope.canBeDecompressed(file)) {
                        var tabIndex = $scope.addTab(file);
                        $scope.updateActiveTab(tabIndex);
                        if (typeof($scope.unregisterArrowSliderInit) === "function") {
                            $scope.unregisterArrowSliderInit();
                        }
                    }
                }

                $scope.addTab = function(file) {
                    var tabIndex = $scope.tabsList.findIndex(f => f.path === file.path);
                    if (tabIndex === -1) {
                        $scope.tabsList.push(file);
                        tabIndex = $scope.tabsList.length - 1;
                    } else {
                        $scope.tabsList[tabIndex] = file;
                    }
                    return tabIndex;
                };

                $scope.updateActiveTab = function(tabIndex) {
                    if (tabIndex != -1 && tabIndex < $scope.tabsList.length && $scope.tabsList[tabIndex]) {
                        var fileToOpen = $scope.tabsList[tabIndex];
                        //saving scroll position
                        var currentContent = $scope.getCurrentContent();
                        if (currentContent) {
                            saveTabScrollPosition(currentContent.path, $('.CodeMirror-scroll').scrollTop());
                        }
                        //updating tab index
                        $scope.activeTabIndex = tabIndex;
                        saveActiveTab();
                        //scrolling through tabs if necessary
                        $timeout(function() {
                            if ($scope.needSlider()) {
                                slideToTab(fileToOpen.path);
                            }
                        });
                        //replacing editor's content
                        if ($scope.editedContentMap[fileToOpen.path] && !fileToOpen.needReload) {
                            setCurrentContent(fileToOpen);
                        } else {
                            fileToOpen.needReload = false;
                            loadAndSetCurrentContent(fileToOpen, true);
                        }
                    }
                }

                var loadAndSetCurrentContent = function(file, sendAnyway) {
                    $scope.folderEditCallbacks.get(file, sendAnyway).success(function(data){
                        $scope.originalContentMap[file.path] = angular.copy(data);
                        $scope.editedContentMap[file.path] = data;
                        setCurrentContent(file);
                    }).error(setErrorInScope.bind($scope));
                };

                $scope.editActiveTabInCodeStudio = function() {
                    $scope.folderEditCallbacks.editInCodeStudio($scope.getCurrentContent());
                }

                var setCurrentContent = function(file) {
                    var mimeType = selectSyntaxicColoration(file.name, file.mimeType);

                    $scope.editorOptions = CodeMirrorSettingService.get(mimeType, {
                        readOnly: !$scope.canEdit,
                        onLoad: function(codeMirror) {
                            $timeout(function() {
                                codeMirror.scrollTo(0, $scope.getTabScrollPosition(file.path));
                            });
                        }
                    });
                };

                var selectSyntaxicColoration = function(fileName, mimeType) {
                    if (mimeType === 'application/sql' ) {
                        return 'text/x-sql'; // codemirror prefers this one
                    }
                    if (mimeType === 'text/x-python-script') {
                        return 'text/x-python'; // codemirror prefers this one
                    }
                    if (fileName.endsWith('.java')) {
                        return 'text/x-java';
                    }
                    return mimeType
                };

                var refreshSyntaxicColoration = function(fileName, mimeType) {
                    $scope.editorOptions.mode = selectSyntaxicColoration(fileName, mimeType);
                };

                var closeContent = function(file) {
                    var tabIndex = $scope.tabsList.findIndex(f => f.path === file.path);
                    if (tabIndex > -1) {
                        $scope.tabsList.splice(tabIndex, 1);
                        if (tabIndex == $scope.activeTabIndex) {
                            $scope.activeTabIndex = -1;
                            if ($scope.tabsList.length > 0) {
                                var newActiveTabIndex = tabIndex > 0 ? tabIndex - 1 : 0;
                                $scope.updateActiveTab(newActiveTabIndex);
                            }
                        } else if (tabIndex < $scope.activeTabIndex) {
                            $scope.activeTabIndex--;
                        }
                    }
                };

                $scope.closeFile = function(file) {
                    if ($scope.isContentDirty(file)) {
                        CreateModalFromTemplate("/templates/plugins/development/fragments/fileclose-prompt.html", $scope, null, function(newScope) {
                            newScope.closeAndSave = function() {
                                var fileToSave = $scope.editedContentMap[file.path];
                                if (fileToSave) {
                                    $scope.saveContent(fileToSave);
                                }
                                closeContent(file);
                                newScope.dismiss();
                            }
                            newScope.close = function() {
                                closeContent(file);
                                newScope.dismiss();
                            }
                        });
                    } else {
                        closeContent(file);
                    }
                };

                $scope.closeOtherFiles = function(file) {
                    var dirtyFiles = [];
                    var fileToClose = [];
                    $scope.tabsList.forEach(function(f) {
                       if (f.path != file.path) {
                           fileToClose.push(f);
                           if ($scope.isContentDirty(f)) {
                               dirtyFiles.push(f);
                           }
                       }
                    });
                    if (dirtyFiles.length > 0) {
                        Dialogs.confirm($scope,'Discard changes','Are you sure you want to discard your changes?').then(function() {
                            $scope.tabsList = [file];
                            $scope.updateActiveTab(0);
                        }, angular.noop);
                    } else {
                        $scope.tabsList = [file];
                        $scope.updateActiveTab(0);
                    }
                };

                $scope.$watch('tabsList', function(nv, ov) {
                    if (nv && ov && nv.length < ov.length) {
                        var nvPath = nv.map(function(file) {
                            return file.path;
                        })
                        ov.forEach(function(file) {
                            if (nvPath.indexOf(file.path) == -1) {
                                delete $scope.originalContentMap[file.path];
                                delete $scope.editedContentMap[file.path];
                            }
                        });
                        cleanTabScrollPosition();
                    }
                    $scope.tabsMap = {};
                    if (nv) {
                        nv.forEach(function(file) {
                            $scope.tabsMap[file.path] = file;
                        });
                    }
                    saveTabsList();
                }, true);

                var searchInDevContents = function(filePath) {
                    if (filePath == null) return null;
                    var pathFolders = [];
                    var searchRecursively = function(folder) {
                        for (var i = 0; i < folder.length; i++) {
                            var child = folder[i];
                            if (child.children && filePath.startsWith(child.path + "/")) {
                                pathFolders.push(child);
                                return searchRecursively(child.children);
                            } else if (child.path == filePath) {
                                pathFolders.push(child);
                                return pathFolders;
                            }
                        }
                        return null;
                    };
                    return searchRecursively($scope.devContents);
                }

                var openFileFromExternal = function(filePath) {
                    var pathFolders = searchInDevContents(filePath);
                    if (pathFolders && pathFolders.length > 0) {
                        pathFolders.forEach(function(f) {
                            f.expanded = true;
                        });
                        let file = pathFolders[pathFolders.length - 1];
                        // if it's just a directory, pick the first file you find in it
                        if (!file.mimeType) {
                            function recFindFirstFile(x) {
                                if (x.mimeType) return x;
                                if (x.children) {
                                    // don't map+filter so that you get early stop
                                    for (var i = 0; i < x.children.length; i++) {
                                        let y = recFindFirstFile(x.children[i]);
                                        if (y.mimeType) return y;
                                    }
                                }
                                return x;
                            };
                            file = recFindFirstFile(file);
                        }
                        if (file.mimeType) {
                            $scope.openFile(file);
                            $timeout(function() {
                                $scope.focusedFile = file;
                            });
                        }
                    }
                };

                $scope.getCurrentContent = function() {
                    if ($scope.activeTabIndex > -1 && $scope.tabsList && $scope.activeTabIndex < $scope.tabsList.length) {
                        var currentFile = $scope.tabsList[$scope.activeTabIndex];
                        return currentFile ? $scope.editedContentMap[currentFile.path] : null;
                    }
                    return null;
                };

                $scope.getCurrentGitLib = function() {
                    if ($scope.activeTabIndex > -1 && $scope.activeTabIndex < $scope.tabsList.length) {
                        var currentFile = $scope.tabsList[$scope.activeTabIndex];
                        return currentFile && currentFile.fromGit ? currentFile.gitLib : null;
                    }
                    return null;
                };

                $scope.isContentFromGit = function() {
                    if ($scope.activeTabIndex > -1 && $scope.activeTabIndex < $scope.tabsList.length) {
                        var currentFile = $scope.tabsList[$scope.activeTabIndex];
                        return currentFile ? currentFile.fromGit : false;
                    }
                    return false;
                };
                $scope.isInConflict = function() {
                    if ($scope.activeTabIndex > -1 && $scope.activeTabIndex < $scope.tabsList.length) {
                        var currentFile = $scope.tabsList[$scope.activeTabIndex];
                        return currentFile ? currentFile.isInConflict : false;
                    }
                    return false;
                };
                $scope.isResolved = function() {
                    if ($scope.activeTabIndex > -1 && $scope.activeTabIndex < $scope.tabsList.length) {
                        var currentFile = $scope.tabsList[$scope.activeTabIndex];
                        return currentFile ? currentFile.isResolved : false;
                    }
                    return false;
                };
                $scope.isFileIsExternalLibrariesJSON = function() {
                    if ($scope.activeTabIndex > -1 && $scope.activeTabIndex < $scope.tabsList.length) {
                        var currentFile = $scope.tabsList[$scope.activeTabIndex];
                        return currentFile ? (currentFile.path==currentFile.name && currentFile.name == "external-libraries.json") : false;
                    }
                    return false;
                };

                $scope.isFileInPublicDirectory = function() {
                    if(!$scope.isInProjectResources) return false;
                    if ($scope.activeTabIndex > -1 && $scope.activeTabIndex < $scope.tabsList.length) {
                        var currentFile = $scope.tabsList[$scope.activeTabIndex];
                        return currentFile && currentFile.path.startsWith('local-static/');
                    }
                    return false;
                };

                $scope.isContentDirty = function(file) {
                    var isFileOpen = file && file.path && $scope.originalContentMap[file.path] && $scope.editedContentMap[file.path];
                    return isFileOpen && $scope.originalContentMap[file.path].data != $scope.editedContentMap[file.path].data;
                };

                $scope.hasDirtyContent = function() {
                    if ($scope.tabsList) {
                        for (var i=0; i<$scope.tabsList.length; i++) {
                            var file = $scope.tabsList[i];
                            if ($scope.isContentDirty(file)) {
                                return true;
                            }
                        }
                    }
                    return false;
                };

                $scope.reloadFileUponConflict = function(filePath) {
                    var element = searchInDevContents(filePath);
                    element[element.length -1].isInConflict = true;
                    $scope.openFile(element[element.length -1]);
                };

                /*
                 * Tab and Folder Explorer Menu
                 */

                $scope.focusOnFile = function(filePath) {
                    var pathFolders = searchInDevContents(filePath);
                    if (pathFolders && pathFolders.length > 0) {
                        pathFolders.forEach(function(content) {
                            content.expanded = true;
                        });
                        $timeout(function() {
                            $scope.focusedFile = pathFolders[pathFolders.length - 1];
                        });
                    }
                };

                var slideToTab = function(path) {
                    $scope.$broadcast('slideToId', "#tabs-frame", "#tabs-slider", path);
                };

                $scope.openTabMenu = function(element, $event) {
                    var template = '<ul class="dropdown-menu">'
                    +    '<li><a ng-disabled="!canEdit" ng-click="moveContent(element)">' + moveIcon + moveText + '</a></li>'
                    +    '<li ng-disabled="!canEdit" ng-show="canDuplicateContent"><a ng-disabled="!canEdit" ng-click="duplicateContent(element)">' + duplicateIcon + duplicateText + '</a></li>'
                    +    '<li><a ng-disabled="!canEdit" ng-click="renameContent(element)">' + renameIcon + renameText + '</a></li>'
                    +    '<li><a ng-click="closeOtherFiles(element)">' + closeOtherTabIcon + closeOtherTabText + '</a></li>'
                    +    '<li><a ng-disabled="!canEdit" ng-click="deleteContent(element)" style="border-top: 1px #eee solid;">' + deleteIcon + deleteText + '</a></li>'
                    +'</ul>'

                    openRightClickMenu(element, $event, template);
                };

                $scope.openFileMenu = function (element, $event) {
                    let template = '<ul class="dropdown-menu">';

                    template += '<li><a ng-disabled="!canEdit" ng-click="moveContent(element)">' + moveIcon + moveText + '</a></li>'
                        + '<li ng-show="canDuplicateContent"><a ng-disabled="!canEdit" ng-click="duplicateContent(element)">' + duplicateIcon + duplicateText + '</a></li>'
                        + ($scope.folderEditCallbacks.editInCodeStudio && $rootScope.projectSummary.canWriteProjectContent ? '<li><a ng-disabled="!canEdit" ng-click="editInCodeStudio(element)">' + editFileIcon + editInCodeStudioText + '</a></li>' : '')
                        + '<li><a ng-disabled="!canEdit" ng-click="renameContent(element)">' + renameIcon + renameText +'</a></li>'
                        + '<li><a ng-disabled="!canEdit" ng-click="deleteContent(element)">' + deleteIcon + deleteText + '</a></li>'
                        + ($scope.folderEditCallbacks.downloadURL ? '<li><a ng-click="downloadElement(element)">' + downloadIcon + downloadText + '</a></li>' : '')
                        + '</ul>';
                    openRightClickMenu(element, $event, template);
                }

                $scope.openFolderMenu = function(element, $event) {
                    const folderLocalActions = '<li><a ng-disabled="!canEdit" ng-click="addInElement(element.path, false, element)">' + createFileIcon + createFileText + '</a></li>'
                        +	'<li><a ng-disabled="!canEdit" ng-click="addInElement(element.path, true, element)">' + createFolderIcon + createFolderText + '</a></li>'
                        +	'<li><a ng-disabled="!canEdit" ng-click="moveContent(element)">' + moveIcon + moveText + '</a></li>'
                        +	'<li ng-show="canDuplicateContent"><a ng-disabled="!canEdit" ng-click="duplicateContent(element)">' + duplicateIcon + duplicateText + '</a></li>'
                        +	'<li ng-show="!element.fromGit"><a  ng-disabled="!canEdit" ng-click="renameContent(element)">' + renameIcon + renameText + '</a></li>'
                        +   '<li><a ng-disabled="!canEdit" ng-click="uploadElement(element.path)">' + uploadIcon + uploadText + '</a></li>'
                        +   ($scope.folderEditCallbacks.downloadURL ? '<li><a ng-click="downloadElement(element)">' + downloadIcon + downloadText + '</a></li>' : '')
                        +   '<li><a ng-disabled="!canEdit" ng-click="deleteContent(element)">' + deleteIcon + deleteText + '</a></li>';

                    let template = '<ul class="dropdown-menu">';

                    if (element.fromGit) {
                        // This folder has been imported through Git references

                        if (element.path in $scope.gitReferences) {
                           template += '<li><a ng-click="gitRefActions.pushModal(element.path)" ng-disabled="isInConflict()">' + commitIcon + commitText + '</a></li>'
                           + '<li><a ng-click="gitRefActions.pullModal(element.path)">' + pullIcon + pullText + '</a></li>'
                               + '<li><a ng-click="gitRefActions.setModal(gitReferences[element.path], element.path)">' + editIcon + editText + '</a></li>'
                               + '<li><a ng-click="gitRefActions.untrackModal(element.path)">' + unlinkIcon + unlinkText + '</a></li>'
                               + '<li class="divider"></li>';
                        }
                    }

                    template += folderLocalActions + '</ul>';

                    openRightClickMenu(element, $event, template);
                };

                $scope.openRootMenu = function($event) {
                    var template = '<ul class="dropdown-menu">'
                        + '<li><a ui-sref="profile.my.settings({\'#\':\'code_editor\'})" target="_blank">' + customizeEditorSettingsIcon + customizeEditorSettingsText + '</a></li>'
                        +   '<li><a ng-click="listContents(\'\')">' + reloadLocalFileIcon + reloadLocalFileText + '</a></li>'
                        +'</ul>'

                    openRightClickMenu(null, $event, template);
                };

                $scope.openAddMenu = function($event) {
                    var template = '<ul class="dropdown-menu">'
                        +   '<li><a ng-disabled="!canEdit" ng-if="$state.$current.name == \'plugindev.editor\'" ng-click="newComponentPopin()">Create component</a></li>'
                        +   '<li><a ng-disabled="!canEdit" ng-click="addInElement(\'\', false)" id="qa_plugindev_folder-file-add-btn">' + createFileIcon + createFileText + '</a></li>'
                        +   '<li><a ng-disabled="!canEdit" ng-click="addInElement(\'\', true)">' + createFolderIcon + createFolderText + '</a></li>'
                        +   '<li><a ng-disabled="!canEdit" ng-click="uploadElement(\'\')">' + uploadIcon + uploadText + '</a></li>'
                        +   ($scope.folderEditCallbacks.downloadURL ? '<li><a ng-click="downloadElement({path:\'\'})">' + downloadIcon + downloadText + '</a></li>' : '')
                        +'</ul>'

                    openRightClickMenu(null, $event, template);
                };

                $scope.openGitMenu = function($event) {
                    var template = '<ul class="dropdown-menu">'
                        +	'<li ng-disabled="!canEdit" ng-show="gitRefCallbacks"><a ng-click="gitRefActions.setModal()">' + importFromGitIcon + importFromGitText + '</a></li>'
                        +	'<li ng-disabled="!canEdit" ng-show="gitRefCallbacks"><a ng-click="gitRefActions.listModal()">' + manageReferenceIcon + manageReferenceText + '</a></li>'
                        +	'<li ng-disabled="!canEdit" ng-show="gitRefCallbacks"><a ng-click="gitRefActions.pullModal()">' + updateAllIcon + updateAllText + '</a></li>'
                        +	'<li ng-disabled="!canEdit" ng-show="gitRefCallbacks"><a ng-click="gitRefActions.pushModal()">' + commitAllIcon + commitAllText + '</a></li>'
                        +'</ul>'

                    openRightClickMenu(null, $event, template);
                };

                var openRightClickMenu = function(element, $event, template) {
                    var callback = function(newScope) {
                        newScope.element = element;
                    };
                    openMenu($event, template, callback);
                };

                var openMenu = function($event, template, callback) {
                    function isElsewhere(elt, e) {
                        return $(e.target).parents('.plugindev-tab-menu').length == 0;
                    }
                    var dkuPopinOptions = {
                        template: template,
                        isElsewhere: isElsewhere,
                        callback: callback
                    };
                    openDkuPopin($scope, $event, dkuPopinOptions);
                };

                /*
                 * Tabs persistency
                 */

                var getFolderEditLocalStorage = function() {
                    var allFolderEditLocalStorage = LocalStorage.get("dss.folderedit");
                    if (!allFolderEditLocalStorage) {
                        allFolderEditLocalStorage = {};
                    }
                    var folderEditLocalStorage = allFolderEditLocalStorage[$scope.localStorageId];
                    if (!folderEditLocalStorage) {
                        folderEditLocalStorage = {"tabsList":[]};
                    }
                    if (!folderEditLocalStorage.tabsList) {
                        folderEditLocalStorage.tabsList = [];
                    }
                    return folderEditLocalStorage;
                }

                var setFolderEditLocalStorage = function(folderEditLocalStorage) {
                    var allFolderEditLocalStorage = LocalStorage.get("dss.folderedit");
                    if (!allFolderEditLocalStorage) {
                        allFolderEditLocalStorage = {};
                    }
                    allFolderEditLocalStorage[$scope.localStorageId] = folderEditLocalStorage;
                    LocalStorage.set("dss.folderedit", allFolderEditLocalStorage);
                }

                var saveTabsList = function() {
                    if ($scope.tabsList) {
                        var folderEditLocalStorage = getFolderEditLocalStorage();
                        folderEditLocalStorage.tabsList = $scope.tabsList.map(function(file) {
                            return file.path;
                        });
                        setFolderEditLocalStorage(folderEditLocalStorage);
                    }
                };

                var saveActiveTab = function() {
                    var folderEditLocalStorage = getFolderEditLocalStorage();
                    folderEditLocalStorage.activeTab = $scope.tabsList[$scope.activeTabIndex].path;
                    setFolderEditLocalStorage(folderEditLocalStorage);
                }

                var saveTabScrollPosition = function(path, scroll) {
                    var folderEditLocalStorage = getFolderEditLocalStorage();
                    if (!folderEditLocalStorage.scrollPositions) {
                        folderEditLocalStorage.scrollPositions = {};
                    }
                    folderEditLocalStorage.scrollPositions[path] = scroll;
                    setFolderEditLocalStorage(folderEditLocalStorage);
                }

                var cleanTabScrollPosition = function() {
                    var folderEditLocalStorage = getFolderEditLocalStorage();
                    var scrollPositions = folderEditLocalStorage.scrollPositions;
                    if (scrollPositions) {
                        var tabsPathList = $scope.tabsList.map(function(f) {
                            return f.path;
                        });
                        Object.keys(scrollPositions).forEach(function(path) {
                            if (tabsPathList.indexOf(path) == -1) {
                                delete scrollPositions[path];
                            }
                        });
                    }
                    setFolderEditLocalStorage(folderEditLocalStorage);
                }

                $scope.getTabScrollPosition = function(path) {
                    var folderEditLocalStorage = getFolderEditLocalStorage();
                    return folderEditLocalStorage.scrollPositions && folderEditLocalStorage.scrollPositions[path] ? folderEditLocalStorage.scrollPositions[path] : 0;
                }

                var openSavedTabs = function() {
                    var folderEditLocalStorage = getFolderEditLocalStorage();
                    var activeTab = null;
                    $scope.tabsList = [];
                    folderEditLocalStorage.tabsList.forEach(function (filePath) {
                        var objPath = searchInDevContents(filePath);
                        if (objPath) {
                            var file = objPath[objPath.length - 1];
                            $scope.addTab(file);
                            if (filePath == folderEditLocalStorage.activeTab) {
                                activeTab = file;
                            }
                        }
                    });
                    var fileToOpen = folderEditLocalStorage.activeTab && activeTab ? activeTab : $scope.tabsList[$scope.tabsList.length - 1];
                    if (fileToOpen) {
                        $scope.focusOnFile(fileToOpen.path);
                        $scope.openFile(fileToOpen);
                        $scope.unregisterArrowSliderInit = $scope.$on("DKU_ARROW_SLIDER:arrow_slider_initialized", function() {
                            slideToTab(fileToOpen.path);
                        });
                    }
                }

                /*
                 * CRUD
                 */

                const updateGitRefsAfterSave = function(cond) {
                    if ($scope.gitRefCallbacks) {
                        $scope.listContents();
                    }
                };

                $scope.saveCurrentContent = function() {
                    var currentContent = $scope.getCurrentContent();
                    $scope.saveContent(currentContent, $scope.getCurrentGitLib());
                };

                $scope.editInCodeStudio = function(element) {
                    $scope.folderEditCallbacks.editInCodeStudio(element);
                }

                function refreshFileSize(filePath) {
                    const leaf = (searchInDevContents(filePath) || []).pop();
                    if (leaf && $scope.originalContentMap[filePath] && angular.isString($scope.originalContentMap[filePath].data)) {
                        leaf.size = $scope.originalContentMap[filePath].data.length;
                    }
                }

                function doSaveContent(content, gitLib) {
                    $scope.folderEditCallbacks.set(content, gitLib).success(function(data){
                        $scope.originalContentMap[content.path] = angular.copy(content);
                        refreshFileSize(content.path);
                        postSaveCallback();
                        updateGitRefsAfterSave(content.path === 'external-libraries.json');
                        const dirtyFiles = {
                            [content.path]: content.data
                        };
                        reloadPluginIfNeeded(dirtyFiles);
                    }).error(setErrorInScope.bind($scope));
                }

                function validateAndSaveContent(content, gitLib) {
                    if (!$scope.folderEditCallbacks.validate) {
                        doSaveContent(content, gitLib)
                    } else {
                        const dirtyFiles = {
                            [content.path]: content.data
                        };
                        $scope.folderEditCallbacks.validate(dirtyFiles).success(function(data) {
                            if (!data.anyMessage) {
                                doSaveContent(content, gitLib)
                            } else {
                                CreateModalFromTemplate('/templates/plugins/development/plugin-dev-warning-modal.html', $scope, null, function(modalScope) {
                                    modalScope.messages = data;
                                    modalScope.saveAnyway = function() {
                                        modalScope.dismiss();
                                        doSaveContent(content, gitLib)
                                    }
                                })
                            }
                        }).error(setErrorInScope.bind($scope));
                    }
                }

                $scope.saveContent = function(content, gitLib) {
                    validateAndSaveContent(content, gitLib);
                };

                function doSaveAll(dirtyFiles, lib) {
                    $scope.folderEditCallbacks.setAll(dirtyFiles, lib).success(function(data){
                        Object.keys(dirtyFiles).forEach(function(filePath) {
                            $scope.originalContentMap[filePath].data = dirtyFiles[filePath];
                            refreshFileSize(filePath);
                        });
                        postSaveCallback();
                        updateGitRefsAfterSave('external-libraries.json' in dirtyFiles);
                        reloadPluginIfNeeded(dirtyFiles)
                    }).error(setErrorInScope.bind($scope));
                }

                $scope.saveAll = function() {
                    var dirtyFiles = {};
                    var modifiedLibs = new Set();
                    $scope.tabsList.forEach(function(file) {
                        if ($scope.isContentDirty(file)) {
                            dirtyFiles[file.path] = $scope.editedContentMap[file.path].data;
                            modifiedLibs = modifiedLibs.add(file.gitLib);
                        }
                    });
                    resetErrorInScope($scope);
                    if (!$scope.folderEditCallbacks.validate) {
                        doSaveAll(dirtyFiles, modifiedLibs);
                    } else {
                        $scope.folderEditCallbacks.validate(dirtyFiles).success(function(data) {
                            if (!data.anyMessage) {
                                doSaveAll(dirtyFiles, modifiedLibs);
                            } else {
                                CreateModalFromTemplate('/templates/plugins/development/plugin-dev-warning-modal.html', $scope, null, function(modalScope) {
                                    modalScope.messages = data;
                                    modalScope.saveAnyway = function() {
                                        modalScope.dismiss();
                                        doSaveAll(dirtyFiles, modifiedLibs);
                                    }
                                })
                            }
                        }).error(setErrorInScope.bind($scope));
                    }
                };

                function reloadPluginIfNeeded(dirtyFiles) {
                    let doReload = 'reloadPlugin' in $scope;
                    if (doReload && dirtyFiles) { // not specified in the case of folder rename for example (and then we do want to reload)
                        let hasAnyJson = false;
                        let hasAnyJs = false;
                        for (const f of Object.keys(dirtyFiles)) {
                            if (f.toLowerCase().endsWith('.json')) {
                                hasAnyJson = true;
                                break;
                            }
                            if (f.toLowerCase().endsWith('.js')) {
                                hasAnyJs = true;
                                break;
                            }
                        }
                        if (!hasAnyJson && !hasAnyJs) {
                            doReload = false;
                        }
                    }
                    return doReload ? $scope.reloadPlugin($stateParams.pluginId) : Promise.resolve();
                }

                $scope.deleteContent = function(content) {
                    var isNonEmptyFolder = content.children != null && content.children.length > 0;
                    var message = isNonEmptyFolder ? 'Are you sure you want to delete ' + content.name + ' and all its contents?' : 'Are you sure you want to delete ' + content.name + ' ?';
                    Dialogs.confirm($scope,'Delete ' + ( isNonEmptyFolder ? 'folder' : 'file'), message).then(function() {
                        $scope.folderEditCallbacks.delete(content).success(function(data){
                            if ( ((content.children && content.children.length>0) || content.mimeType) && $scope.folderEditCallbacks.setDirty) {
                                $scope.folderEditCallbacks.setDirty(content.gitLib);
                            }
                           var toClose = [];
                            $scope.tabsList.forEach(function(file) {
                                if (isIncludedOrEqual(file.path, content.path)) {
                                    toClose.push(file);
                                }
                            });
                            toClose.forEach(function(f) {
                                closeContent(f);
                            });
                            if ($scope.folderEditCallbacks.setDirty) {
                                DKUtils.reloadState(); // needed for git-enabled editors
                            } else {
                                $scope.listContents('');
                            }
                        }).error(setErrorInScope.bind($scope));
                    }, angular.noop);
                };

                $scope.renameContent = function(content) {
                    var popinName = content.children ? "Rename folder" : "Rename file";
                    Dialogs.prompt($scope, popinName, 'New name', content.name).then(function(newName) {
                        $scope.folderEditCallbacks.rename(content, newName).success(function(data){
                            if ( ((content.children && content.children.length>0) || content.mimeType) && $scope.folderEditCallbacks.setDirty) {
                                $scope.folderEditCallbacks.setDirty(content.gitLib);
                            }
                            //syntaxic coloration issues
                            if (content.mimeType != data.mimeType) {
                                //refreshing code mirror syntaxic coloration if we renamed current content
                                if ($scope.getCurrentContent() && $scope.getCurrentContent().path == content.path) {
                                    refreshSyntaxicColoration(content.name, data.mimeType);
                                }
                                //necessary if file renamed is among tabs (but not active one), otherwise syntaxic coloration won't be updated when we come back to this tab.
                                if ($scope.tabsMap[content.path]) {
                                    propagatingMimeTypeChange(content.path, data.mimeType);
                                }
                            }
                            // necessary, otherwise it fails when clicking on save after a move
                            propagatingPathChange(content.path, data.path);
                            reloadPluginIfNeeded().then(() => {
                                if ($scope.folderEditCallbacks.setDirty) {
                                    DKUtils.reloadState(); // needed for git-enabled editors
                                } else {
                                    $scope.listContents('');
                                }
                            });
                        }).error(setErrorInScope.bind($scope));
                    }, angular.noop);
                };

                $scope.moveContent = function(content) {
                    CreateModalFromTemplate("/templates/plugins/development/fragments/filemove-prompt.html", $scope, "MoveContentModalController", function(newScope) {
                        newScope.devContents = angular.copy($scope.devContents); // so that the stat is disconnected from the main display of the hierarchy
                        newScope.toMove = content;
                        newScope.doMove = function(to) {
                            $scope.folderEditCallbacks.move(content, to).success(function(data){
                                if ( ((content.children && content.children.length>0) || content.mimeType) && $scope.folderEditCallbacks.setDirty) {
                                    $scope.folderEditCallbacks.setDirty(content.gitLib);
                                }
                                // necessary, otherwise it fails when clicking on save after a move
                                propagatingPathChange(content.path, data.path);
                                reloadPluginIfNeeded().then(() => {
                                    if ($scope.folderEditCallbacks.setDirty) {
                                        DKUtils.reloadState(); // needed for git-enabled editors
                                    } else {
                                        $scope.listContents('');
                                    }
                                });
                            }).error(setErrorInScope.bind($scope));
                        };
                    });
                };

                $scope.canDuplicateContent = $scope.folderEditCallbacks.copy != null;

                $scope.duplicateContent = function(content) {
                    if ( ((content.children && content.children.length>0) || content.mimeType) && $scope.folderEditCallbacks.setDirty) {
                        $scope.folderEditCallbacks.setDirty(content.gitLib);
                    }
                    $scope.folderEditCallbacks.copy(content).success(function(data){
                        if ($scope.folderEditCallbacks.setDirty) {
                            DKUtils.reloadState(); // needed for git-enabled editors
                        } else {
                            $scope.listContents('');
                        }
                    }).error(setErrorInScope.bind($scope));
                }

                const propagatingPathChange = function(oldPath, newPath) {
                    // In tabslist
                    $scope.tabsList.forEach(function(file) {
                        if (isIncludedOrEqual(file.path, oldPath)) {
                            const oldFilePath = file.path;
                            const newFilePath = file.path.replace(oldPath, newPath);
                            const newName = newFilePath.split('/').at(-1); // file name may changed
                            file.path = newFilePath;
                            file.name = newName;
                            //updating originalContentMap
                            const originalContent = $scope.originalContentMap[oldFilePath];
                            if (originalContent) {
                                originalContent.path = newFilePath;
                                originalContent.name = newName;
                                delete $scope.originalContentMap[oldFilePath];
                                $scope.originalContentMap[newFilePath] = originalContent;
                            }
                            //updating editedContentMap
                            const editedContent = $scope.editedContentMap[oldFilePath];
                            if (editedContent) {
                                editedContent.path = newFilePath;
                                editedContent.name = newName;
                                delete $scope.editedContentMap[oldFilePath];
                                $scope.editedContentMap[newFilePath] = editedContent;
                            }
                        }
                    });
                    // In localStorage
                    const folderEditLocalStorage = getFolderEditLocalStorage();
                    for (let i=0; i<folderEditLocalStorage.tabsList.length; i++) {
                        const filePath = folderEditLocalStorage.tabsList[i];
                        if (isIncludedOrEqual(filePath, oldPath)) {
                            folderEditLocalStorage.tabsList[i] = filePath.replace(oldPath, newPath);
                        }
                    }
                    if (folderEditLocalStorage.activeTab && isIncludedOrEqual(folderEditLocalStorage.activeTab, oldPath)) {
                        folderEditLocalStorage.activeTab = folderEditLocalStorage.activeTab.replace(oldPath, newPath);
                    }
                    setFolderEditLocalStorage(folderEditLocalStorage);
                    // In devContents
                    const lastPathElementRegExp = new RegExp('/[^/]*$'); //no short form coz eslint doesn't like slash escape
                    const oldParent = (searchInDevContents(oldPath.replace(lastPathElementRegExp, '')) || []).pop();
                    if (oldParent && oldParent.children) {
                        //cut leaf from oldPath
                        const childidx = oldParent.children.findIndex(child => child.path === oldPath);
                        if (childidx !== -1) {
                            const leaf = oldParent.children.splice(childidx, 1)[0];
                            //paste leaf to newPath
                            const newParent = (searchInDevContents(newPath.replace(lastPathElementRegExp, '')) || []).pop();
                            if (newParent && newParent.children) {
                                function updatePath(leaf) {
                                    if (leaf) {
                                        leaf.path = leaf.path.replace(oldPath, newPath);
                                        leaf.name = leaf.path.split('/').at(-1);
                                        if (leaf.children) leaf.children.forEach(updatePath);
                                    }
                                }
                                updatePath(leaf);
                                newParent.children.push(leaf);
                            }
                        }
                    }
                };

                var propagatingMimeTypeChange = function(path, mimeType) {
                    for (var i = 0; i<$scope.tabsList.length; i++) {
                        var file = $scope.tabsList[i];
                        if (file.path == path) {
                            file.mimeType = mimeType;
                            break;
                        }
                    }
                }

                var decompressibleMimes = ['application/zip', 'application/x-bzip', 'application/x-bzip2', 'application/x-gzip', 'application/x-tar', 'application/gzip', 'application/bzip', 'application/bzip2', 'application/x-compressed-tar'];
                $scope.canBeDecompressed = function(content) {
                    return content && content.mimeType && decompressibleMimes.indexOf(content.mimeType) >= 0;
                };

                $scope.computeFileIconClass = function(file) {
                    if (file.fromGit) {
                        return 'dku-icon-git-file-16';
                    } else if ($scope.canBeDecompressed(file)) {
                        return 'dku-icon-file-zip-16';
                    } else if ($scope.isImage(file)) {
                        return 'dku-icon-image-16';
                    } else {
                        return 'dku-icon-file-text-16';
                    }
                };

                $scope.decompressContent = function(content) {
                    $scope.folderEditCallbacks.decompress(content).success(function(data){
                        $scope.listContents();
                    }).error(setErrorInScope.bind($scope));
                };

                $scope.addInElement = function(contentPath, isFolder, element) {
                    CreateModalFromTemplate("/templates/plugins/development/fragments/filename-prompt.html", $scope, null, function(newScope) {
                        newScope.isFolder = isFolder;
                        newScope.doCreation = function(fileName) {
                            $scope.folderEditCallbacks.create(contentPath + '/' + fileName, isFolder).success(function(data){
                                if (!isFolder && element && element.fromGit) {
                                    if (element.gitLib) {
                                        $scope.folderEditCallbacks.setDirty(element.gitLib);
                                    } else {
                                        $scope.folderEditCallbacks.setDirty(element.name)
                                    }
                                }
                                $scope.listContents().success(function() {
                                    if (!isFolder) {
                                        var newElPath = contentPath && contentPath.length > 0 ? contentPath + '/' + fileName : fileName;
                                        var newElement = searchInDevContents(newElPath);
                                        if (newElement) {
                                            $scope.openFile(newElement[newElement.length - 1]);
                                        }
                                    }
                                });
                            }).error(setErrorInScope.bind($scope));
                        };
                    });
                };

                var openFirstUpload = function(contentPath, firstUpload) {
                    $scope.listContents().success(function() {
                        var firstUploadPath = contentPath && contentPath.length > 0 ? contentPath + "/" + firstUpload.name : firstUpload.name;
                        var firstUploadPathObj = searchInDevContents(firstUploadPath);
                        if (firstUploadPathObj) {
                            $scope.openFile(firstUploadPathObj[firstUploadPathObj.length - 1]);
                        }
                    });
                };
                $scope.uploadElement = function(contentPath) {
                    CreateModalFromTemplate("/templates/plugins/development/fragments/upload-prompt.html", $scope, "UploadContentModalController", function(newScope) {
                        newScope.folderEditCallbacks = $scope.folderEditCallbacks;
                        newScope.openFirstUpload = openFirstUpload;
                        newScope.contentPath = contentPath;
                    });
                };

                $scope.downloadElement = function(content) {
                    let downloadUrl = $scope.folderEditCallbacks.downloadURL ? $scope.folderEditCallbacks.downloadURL(content) : undefined;
                    if (downloadUrl) {
                        downloadURL(downloadUrl);
                    } else {
                        Logger.warn("No download possible for " + content.path);
                    }
                };


                checkChangesBeforeLeaving($scope,  $scope.hasDirtyContent, $scope.folderEditSaveWarning);
                /*
                 * Git references actions
                 */

                $scope.gitRefActions = {
                    setModal: function (gitRef, gitRefPath) {
                        CreateModalFromTemplate("/templates/plugins/development/fragments/git-ref-prompt.html", $scope, "GitReferenceSetController", newScope => {
                            if (gitRef && gitRefPath) {
                                newScope.gitRef = gitRef;
                                newScope.gitRefPath = gitRefPath;
                                newScope.isEditingGitRef = true;
                            }
                            newScope.onSetCallback = () => { $scope.listContents(); };
                        });
                    },
                    listModal: function () {
                        $scope.listContents().then(() => {
                            CreateModalFromTemplate("/templates/plugins/development/fragments/git-ref-list.html", $scope, undefined, newScope => {
                                newScope.gitURL = url => {
                                    url = url.replace(/\.git(#.*)?$/, '')
                                    if (url.startsWith("git")) {
                                        url = url.replace(':', '/');
                                        url = url.replace('git@', "https://");
                                    }
                                    return url;
                                }
                            });
                        }, setErrorInScope.bind($scope));
                    },
                    isDirty: function (gitRefPath) {
                        return gitRefPath ? $scope.folderEditCallbacks.getDirty(gitRefPath) : $scope.folderEditCallbacks.getAllDirty();
                    },
                    doPull: function (gitRefPath) {
                        const pullGitRefAPI = gitRefPath ?
                            $scope.gitRefCallbacks.pullOne(gitRefPath) :
                            $scope.gitRefCallbacks.pullAll();
                        pullGitRefAPI.then(pullResult => {
                            FutureProgressModal.show($scope, pullResult.data, "Updating").then(futureResult => {
                                if (futureResult) {
                                    Dialogs.infoMessagesDisplayOnly($scope, "Update result", futureResult).then(() => {
                                        // If at least one of the pulls succeeded, we want to reload the state as files might have changed
                                        if (futureResult.messages.some(message => message['severity'] === 'SUCCESS')) {
                                            DKUtils.reloadState();
                                        }
                                    });
                                }
                            }, setErrorInScope.bind($scope));
                        }, setErrorInScope.bind($scope));
                    },
                    pullModal: function (gitRefPath) {
                        const state = $scope.gitRefActions.isDirty(gitRefPath);
                        state.then((dirtyState) => {
                            if (dirtyState.data) {
                                Dialogs.confirm($scope, pullText, 'Some files are unpushed, are you sure you want to '+pullText ,
                                    {btnConfirm: "Reset"})
                                    .then(()=> {$scope.gitRefActions.doPull (gitRefPath)}, angular.noop)
                            } else {
                                $scope.gitRefActions.doPull(gitRefPath);
                            }
                        });
                    },
                    rmModal: function (gitRefPath) {
                        Dialogs.confirm($scope, 'Remove Git reference', 'Are you sure you want to remove this Git reference and the associated folder?', {btnConfirm: "Remove"}).then(() => {
                            $scope.gitRefCallbacks.rm(gitRefPath, true).then(() => {
                                DKUtils.reloadState();
                            }, setErrorInScope.bind($scope))
                        }, angular.noop);
                    },
                    untrackModal: function (gitRefPath) {
                        Dialogs.confirm($scope, 'Untrack Git reference', 'Are you sure you want to untrack this Git reference and keep the associated folder?', {btnConfirm: "Untrack"}).then(() => {
                            $scope.gitRefCallbacks.rm(gitRefPath, false).then(() => {
                                DKUtils.reloadState();
                            }, setErrorInScope.bind($scope))
                        }, angular.noop);
                    },
                    pushModalWithCommitMessage: function (gitRefPath, commitMessage) {
                        const pushGitRefAPI = gitRefPath ?
                            $scope.gitRefCallbacks.pushOne(commitMessage, gitRefPath) :
                            $scope.gitRefCallbacks.pushAll(commitMessage);

                            pushGitRefAPI.then(pushResult => {
                            const title = pushResult.config.params.gitReferencePath ? "Pushing library " + pushResult.config.params.gitReferencePath : "Pushing all " + Object.keys($scope.gitReferences).length + " libraries";
                            FutureProgressModal.show($scope, pushResult.data, title).then(futureResult => {
                                if (futureResult) {
                                    const toDisplay = Object.assign({}, futureResult, { messages: futureResult.messages.filter(message => (message['code']  !== 'INFO_GIT_PUSH_CONFLICTING_FILE')) });
                                    Dialogs.infoMessagesDisplayOnly($scope, "Push result", toDisplay).then(() => {
                                        // If at least one of the pushs succeeded, we want to reload the state as files might have changed
                                        if (futureResult.messages.some(message => (message['severity'] === 'SUCCESS') || (message['code'] === 'INFO_GIT_PUSH_CONFLICTING_FILE'))) {
                                            if (gitRefPath) { //If only one library is push, we open all conflicting files
                                                const conflictingFiles = futureResult.messages.filter(message => message['code']  === 'INFO_GIT_PUSH_CONFLICTING_FILE');
                                                conflictingFiles.forEach(file => {
                                                    $scope.reloadFileUponConflict(gitRefPath + "/" + file['details']);
                                                });
                                            }
                                        }
                                    });
                                    $scope.reloadContents();
                                }
                            }, setErrorInScope.bind($scope));
                        }, setErrorInScope.bind($scope));
                    },
                    push:function (gitRefPath = null) {
                        let standardMessage = "Library update from DSS";
                        var options = {type: 'textarea', btnConfirm: "Commit"};
                        Dialogs.prompt($scope, "Enter your commit message", "Commit message", standardMessage, options).then(function (newMessage) {
                            $scope.gitRefActions.pushModalWithCommitMessage(gitRefPath, newMessage);
                        }, angular.noop)
                    },
                    pushModal: function(gitRefPath = null){
                        if ($scope.hasDirtyContent()) {
                            var options = { btnConfirm: "Save", positive: true};
                            Dialogs.confirm($scope, "Unsaved files", "You have unsaved files. Do you want to save them before?", options).then(function() {
                                $scope.saveAll();
                                $scope.gitRefActions.push(gitRefPath)
                            }, angular.noop);
                        } else {
                            $scope.gitRefActions.push(gitRefPath);
                        }
                    },
                    makeResolve: function(currentFile) {
                        var options = { btnConfirm: "Mark", positive: true};
                        Dialogs.confirm($scope, "Mark as resolved", "<p>Do you really want to mark this file as resolved?</p><p>You won't be able to mark as unresolved.</p>", options).then(function(){
                            $scope.gitRefCallbacks.markAsResolved(currentFile.path, currentFile.gitLib).then(result => {
                                currentFile.isResolved = true;
                                currentFile.isInConflict = false;
                                $scope.reloadContents();
                           }, error => Dialogs.displaySerializedError($scope, error));
                       }, angular.noop);
                    },
                    markAsResolved: function() {
                        if ($scope.activeTabIndex > -1 && $scope.activeTabIndex < $scope.tabsList.length) {
                            var currentFile = $scope.tabsList[$scope.activeTabIndex];
                            if ($scope.isContentDirty(currentFile)) {
                                var options = { btnConfirm: "Save and mark", positive: true};
                                Dialogs.confirm($scope, 'Save file',
                                "Your file has been modified. Do you want to save it and mark as resolved?<br/>" +
                                "You won't be able to mark as unresolved.", options
                                ).then(function() {
                                    $scope.saveCurrentContent();
                                    $scope.gitRefCallbacks.markAsResolved(currentFile.path, currentFile.gitLib).then(() => {
                                        currentFile.isResolved = true;
                                        currentFile.isInConflict = false;
                                        $scope.reloadContents();
                                    }, error => Dialogs.displaySerializedError($scope, error));
                                }, angular.noop);
                            } else {
                                $scope.gitRefActions.makeResolve(currentFile);
                            }
                       } else {
                            Dialogs.error($scope, "Mark as resolved", "Unable to find this file");
                       }
                    },
                    restoreLibToPreviousVersion: function() {
                        if ($scope.activeTabIndex > -1 && $scope.activeTabIndex < $scope.tabsList.length) {
                            var currentFile = $scope.tabsList[$scope.activeTabIndex];
                            var options = { btnConfirm: "Restore"};
                            Dialogs.confirm($scope, "Restore the library to your previous version", "Do you really want to rollback to the previous version. This will replace the current (conflicting) files with the files you were editing before trying to push your library.", options).then(function(){
                                $scope.gitRefCallbacks.revertAllFiles(currentFile.gitLib).then(result => {
                                    if (result) {
                                        $scope.reloadContents();
                                    }
                                }, error => Dialogs.displaySerializedError($scope, error));
                            }, angular.noop);
                        } else {
                            Dialogs.error($scope, "Restore to previous version", "Unable to find this file");
                        }
                    },
                };


                /*
                 * UI Utils
                 */

                $scope.getMarginFromDepth = function(depth) {
                    return (depth + 1)*15 + 10;
                }

                $scope.getCarretLeftPosition = function(depth) {
                    return $scope.getMarginFromDepth(depth - 1);
                }

                $scope.containsFolder = function(element) {
                    if (!element.children) {
                        return false;
                    }
                    for (var i=0; i<element.children.length; i++) {
                        var e = element.children[i];
                        if (typeof(e.children) !== "undefined") {
                            return true;
                        }
                    }
                    return false;
                }

                $scope.isImage = function(element) {
                    return element.mimeType.startsWith('image');
                }

                $scope.emptyCtaBtnAction = function() {
                    if ($scope.emptyCta && $scope.emptyCta.btnAction) {
                        switch ($scope.emptyCta.btnAction) {
                            case 'create':
                                $scope.addInElement('', false);
                                break;
                            case 'upload':
                                $scope.uploadElement('');
                                break;
                            default:
                                return false;
                        }
                    }
                }


                /*
                 * Utils
                 */

                var isIncludedOrEqual = function(path, path2) {
                    return path.startsWith(path2 + "/") || path == path2;
                }
            }
        };
    });

    app.controller("NewFileModalController", function($scope, DataikuAPI, $state, $stateParams, WT1){
        $scope.fileName = null;
        $scope.create = function() {
            $scope.doCreation($scope.fileName);
            $scope.dismiss();
        };
    });

    app.controller("GitReferenceSetController", function($scope, $stateParams, DKUtils, DataikuAPI, ActivityIndicator, CreateModalFromTemplate, SpinnerService) {
        $scope.gitRef = $scope.gitRef || {
            remote: '',
            remotePath: '',
            checkout: ''
        };
        $scope.gitRefPath = $scope.gitRefPath || '';
        $scope.addPythonPath = true;
        $scope.isGitRefPathUnique = true;

        $scope.$watch("gitRefPath", function(gitRefPath) {
            if (!$scope.isEditingGitRef) {
                $scope.isGitRefPathUnique = !gitRefPath || !(gitRefPath in $scope.gitReferences);
            }
        });

        $scope.setGitRef = function() {
            $scope.gitRefCallbacks.set($scope.gitRef, $scope.gitRefPath, $scope.addPythonPath).then(result => {
                ActivityIndicator.success('Git reference successfully set.');
                if ($scope.onSetCallback) {
                    $scope.onSetCallback();
                }
                $scope.gitRefPath = result.data.refPath;
                $scope.gitRefActions.pullModal($scope.gitRefPath);
            }, setErrorInScope.bind($scope));
        };
    });

    app.controller("MoveContentModalController", function($scope) {
        $scope.uiState = {moveToTop : false, destination : null};

        $scope.changeDestination = function(to) {
            $scope.uiState.destination = to;
            var recClearMoveToHere = function(l) {
                l.forEach(function(e) {
                    if (e != to) {
                        e.moveToHere = false;
                        if (e.children) {
                            recClearMoveToHere(e.children);
                        }
                    }
                });
            };
            recClearMoveToHere($scope.devContents);
            if (to != null) {
                $scope.uiState.moveToTop = false;
                to.moveToHere = true;
            } else {
                $scope.uiState.moveToTop = true;
            }
        };

        $scope.hasNowhereToGo = function() {
            if ($scope.uiState.moveToTop) {
                return false;
            } else if ($scope.uiState.destination && $scope.uiState.destination.moveToHere) {
                return false;
            } else {
                return true;
            }
        }

        $scope.move = function() {
            if ($scope.uiState.moveToTop) {
                $scope.doMove(null);
            } else if ($scope.uiState.destination && $scope.uiState.destination.moveToHere) {
                $scope.doMove($scope.uiState.destination);
            }
            $scope.dismiss();
        };
    });

    app.controller("UploadContentModalController", function($scope, DataikuAPI, $state, $stateParams, WT1, Logger) {
        $scope.toUpload = [];

        $scope.selectedCount = function() {
            return $scope.toUpload.filter(function(f) {return f.$selected;}).length;
        };
        $scope.startedCount = function() {
            return $scope.toUpload.filter(function(f) {return f.started;}).length;
        };
        $scope.doneCount = function() {
            return $scope.toUpload.filter(function(f) {return f.done;}).length;
        };

        var getPathForFileToUpload = function(fileToUpload) {
            return fileToUpload.name;
        };
        var isAlreadyListed = function(filePath) {
            var found = false;
            $scope.toUpload.forEach(function(u) {
                found |= filePath == u.name;
            });
            return found;
        };
        $scope.uploadFiles = function(files) {
            var filePaths = [];
            var newFiles = [];
            for (var i = 0, len = files.length; i < len ; i++) { // no forEach() on the files :(
                var file = files[i];
                var filePath = getPathForFileToUpload(file);
                if (!isAlreadyListed(filePath)) {
                    filePaths.push(filePath);
                    newFiles.push(file);
                }
            }
            $scope.doCheckUpload(filePaths).success(function(data) {
                for (var i = 0, len = newFiles.length; i < len ; i++) { // no forEach() on the files :(
                    var file = newFiles[i];
                    var feasability = data.feasabilities[i];
                    $scope.toUpload.push({file:file, name:getPathForFileToUpload(file), feasability:feasability, $selected:feasability.canUpload});
                }
            }).error(setErrorInScope.bind($scope));
        };

        $scope.upload = function() {
            $scope.doUpload($scope.toUpload.filter(function(f) {return f.$selected;}));
        };

        $scope.goToFirstUploaded = function() {
            var succeeded = $scope.toUpload.filter(function(f) {return f.succeeded != null;})[0];
            if (succeeded != null) {
                $scope.openFirstUpload($scope.contentPath, succeeded);
            }
            $scope.dismiss();
        };
        var checkUploadCompletion = function() {
            if ($scope.startedCount() == 1 && $scope.startedCount() == $scope.doneCount()) {
                var succeeded = $scope.toUpload.filter(function(f) {return f.succeeded != null;})[0];
                if (succeeded != null) {
                    $scope.goToFirstUploaded();
                }
            }
        };
        $scope.doUpload = function(filesToUpload) {
            filesToUpload.forEach(function(fileToUpload) {
                fileToUpload.started = true;
                $scope.folderEditCallbacks.upload($scope.contentPath, fileToUpload.file, function (e) {
                    if (e.lengthComputable) {
                        $scope.$apply(function () {
                            fileToUpload.progress = Math.round(e.loaded * 100 / e.total);
                        });
                    }
              }).then(function (data) {
                  Logger.info("file " + fileToUpload.name + "uploaded", data);
                  fileToUpload.done = true;
                  fileToUpload.succeeded = JSON.parse(data);
                  checkUploadCompletion();
              }, function(payload){
                  Logger.info("file " + fileToUpload.name + "could not be uploaded", payload);
                  fileToUpload.done = true;
                  fileToUpload.failed = getErrorDetails(JSON.parse(payload.response), payload.status, function(h){return payload.getResponseHeader(h)}, payload.statusText);
                  fileToUpload.failed.html = getErrorHTMLFromDetails(fileToUpload.failed);
                  checkUploadCompletion();
              });
            });
        };
        $scope.doCheckUpload = function(filePaths) {
            return $scope.folderEditCallbacks.checkUpload($scope.contentPath, filePaths);
        };
    });
})();

;
(function() {
'use strict';

const app = angular.module('dataiku.taggableobjects', []);

// Keep in sync with Java TaggableType enum
const TAGGABLE_TYPES = [
    'PROJECT',
    'FLOW_ZONE',
    'DATASET',
    'MANAGED_FOLDER',
    'STREAMING_ENDPOINT',
    'LABELING_TASK',
    'RECIPE',
    'SQL_NOTEBOOK',
    'SEARCH_NOTEBOOK',
    'JUPYTER_NOTEBOOK',
    'ANALYSIS',
    'STATISTICS_WORKSHEET',
    'SAVED_MODEL',
    'MODEL_EVALUATION_STORE',
    'MODEL_COMPARISON',
    'PROMPT_STUDIO',
    'RETRIEVABLE_KNOWLEDGE',
    'AGENT_TOOL',
    'SCENARIO',
    'DASHBOARD',
    'INSIGHT',
    'WEB_APP',
    'CODE_STUDIO',
    'REPORT',
    'ARTICLE',
    'LAMBDA_SERVICE',
    'WORKSPACE',
    'DATA_COLLECTION'
];

app.constant("TAGGABLE_TYPES", TAGGABLE_TYPES);

// Can be exposed from one project to another
// FLOW_QUICK_SHAREABLE_TYPES can also be exposed
const FLOW_EXPOSABLE_TYPES = [
    'JUPYTER_NOTEBOOK',
    'WEB_APP',
    'REPORT',
    'SCENARIO'
];

// Quick shareable types
const FLOW_QUICK_SHAREABLE_TYPES = [
    'DATASET',
    'MANAGED_FOLDER',
    'SAVED_MODEL',
    'MODEL_EVALUATION_STORE'
];

app.constant("FLOW_QUICK_SHAREABLE_TYPES", FLOW_QUICK_SHAREABLE_TYPES);
app.constant("FLOW_EXPOSABLE_TYPES", [...FLOW_QUICK_SHAREABLE_TYPES, 'RETRIEVABLE_KNOWLEDGE', ...FLOW_EXPOSABLE_TYPES]);
app.constant("ALL_EXPOSABLE_TYPES", [...FLOW_QUICK_SHAREABLE_TYPES, ...FLOW_EXPOSABLE_TYPES, 'ARTICLE']);

// Computable in the flow (as outputs of recipes)
app.constant("FLOW_COMPUTABLE_TYPES", [
    'DATASET',
    'MANAGED_FOLDER',
    'SAVED_MODEL',
    'MODEL_EVALUATION_STORE',
    'STREAMING_ENDPOINT',
    'RETRIEVABLE_KNOWLEDGE'
]);

// Publishable as insights on dashboards
app.constant("PUBLISHABLE_TYPES", [
    'DATASET',
    'JUPYTER_NOTEBOOK',
    'SAVED_MODEL',
    'MANAGED_FOLDER',
    'SCENARIO',
    'WEB_APP',
    'REPORT',
    'CODE_STUDIO'
]);

app.constant('PIPELINEABILITY_ACTIONS', {
    changeSQL: 'action-change-sql-pipelineability',
    changeSpark: 'action-change-spark-pipelineability'
});


function isHDFSAbleType(type) {
    return ['HDFS', 'S3', 'GCS', 'Azure'].includes(type);
}

/*
There are essentially 4 representations for taggable objects in the JS code:
- {type, projectKey, id [,displayName]}, the equivalent of java TaggableObjectRef (so it's good to call those variables tor or something similar)
- the actual serialized taggable object (they can be called using they actual types or 'to' or something similar)
- the graph node (that use in particular a specific id system) (good to call them 'node' or something similar)
- the list items, as used in list pages (good to call them 'listItem' or something similar)

This service helps juggling with them
*/
app.service("TaggableObjectsUtils", function($state, $stateParams) {
    const svc = this;

    // Returns the items common taggableType or 'TAGGABLE_OBJECT' if they don't have the same
    // Items can be an array of taggable type or an array of arbitrary objects that 'typeFieldAccessor' can map to taggable types
    this.getCommonType = function(items, typeFieldAccessor) {
        let commonType = null;
        for (let i = 0; i < items.length; ++i) {
            let itemType = typeFieldAccessor ? typeFieldAccessor(items[i]) : items[i];
            if (!commonType) {
                commonType = itemType;
            } else if (commonType != itemType) {
                return 'TAGGABLE_OBJECT';
            }
        }
        return commonType || 'TAGGABLE_OBJECT';
    };

    this.fromNodeType = function(nodeType) {
        if (!nodeType) return;

        if (nodeType.includes('DATASET')) {
            return 'DATASET';
        } else if (nodeType.includes('LAMBDA_SERVICE')) {
            return 'LAMBDA_SERVICE';
        } else if (nodeType.includes('SAVEDMODEL') || nodeType.includes('SAVED_MODEL')) {
            return 'SAVED_MODEL';
        } else if (nodeType.includes('MODELEVALUATIONSTORE') || nodeType.includes('MODEL_EVALUATION_STORE')) {
            return 'MODEL_EVALUATION_STORE';
        } else if (nodeType.includes('RETRIEVABLE_KNOWLEDGE')) {
            return 'RETRIEVABLE_KNOWLEDGE';
        } else if (nodeType.includes('MODEL_COMPARISON')) {
            return 'MODEL_COMPARISON';
        } else if (nodeType.includes('PROMPT_STUDIO')) {
            return 'PROMPT_STUDIO';
        } else if (nodeType.includes('AGENT_TOOL')) {
            return 'AGENT_TOOL';
        } else if (nodeType.includes('STREAMING_ENDPOINT')) {
            return 'STREAMING_ENDPOINT';
        } else if (nodeType.includes('MANAGED_FOLDER')) {
            return 'MANAGED_FOLDER';
        } else if (nodeType == 'RECIPE') {
            return 'RECIPE';
        } else if (nodeType == 'LABELING_TASK') {
            return 'LABELING_TASK';
        } else if (nodeType == 'PROJECT') {
            return 'PROJECT';
        } else if (nodeType == 'JUPYTER_NOTEBOOK') {
            return 'JUPYTER_NOTEBOOK';
        } else if (nodeType == 'SQL_NOTEBOOK') {
            return 'SQL_NOTEBOOK';
        } else if (nodeType == 'SEARCH_NOTEBOOK') {
            return 'SEARCH_NOTEBOOK';
        } else if (nodeType == 'DASHBOARD') {
            return 'DASHBOARD';
        } else if (nodeType == 'INSIGHT') {
            return 'INSIGHT';
        } else if (nodeType == 'SCENARIO') {
            return 'SCENARIO';
        } else if (nodeType == 'ANALYSIS') {
            return 'ANALYSIS';
        } else if (nodeType == 'WEB_APP') {
            return 'WEB_APP';
        } else if (nodeType == 'CODE_STUDIO') {
            return 'CODE_STUDIO';
        } else if (nodeType == 'REPORT') {
            return 'REPORT';
        } else if (nodeType == 'NOTEBOOK') {
            return 'NOTEBOOK';
        } else if (nodeType == 'ZONE') {
            return 'FLOW_ZONE';
        } else if (nodeType == 'ARTICLE') {
            return 'ARTICLE';
        }

        throw new Error("Unhandled nodeType");
    };

    this.fromNode = function(node) {
        return {
            type: svc.fromNodeType(node.nodeType),
            projectKey: node.projectKey,
            id: node.name,
            displayName: node.description || node.name,
            subType: node.datasetType || node.recipeType || node.savedModelType || node.folderType,
            tags: node.tags
        };
    };

    this.fromObjectItem = function(objectItem) {
        return {
            type: svc.fromNodeType(objectItem.nodeType),
            projectKey: objectItem.projectKey,
            id: objectItem.id || objectItem.name,
            displayName: objectItem.description || objectItem.name,
            subType: objectItem.datasetType || objectItem.recipeType || objectItem.savedModelType,
            tags: objectItem.tags,
            workspaceKey: objectItem.workspaceKey,
        };
    };

    this.fromListItem = function(listItem) {
        return {
            type: svc.taggableTypeFromAngularState(listItem),
            projectKey: $stateParams.projectKey,
            id: listItem.id,
            displayName: listItem.name,
            subType: listItem.type,
            tags: listItem.tags,
            workspaceKey: listItem.workspaceKey,
        };
    };

    this.taggableTypeFromAngularState = function(listItem) {
        const stateName = $state.current.name;

        // in the special case of notebooks, we cannot retrieve the taggable type only from angular state
        // then we use the item to get the language to determine the proper taggle type
        if (listItem && listItem.type && stateName.startsWith('projects.project.notebooks')) {
            switch(listItem.type) {
                case "SQL":
                    return 'SQL_NOTEBOOK';
                case "SEARCH":
                    return 'SEARCH_NOTEBOOK';
                default:
                    return 'JUPYTER_NOTEBOOK';
            }
        }
        if (listItem && stateName.startsWith('projects.project.continuous-activities')) {
            return 'CONTINUOUS_ACTIVITY';
        }

        if (stateName.startsWith('projects.project.datasets') || stateName.startsWith('projects.project.foreigndatasets')) {
            return 'DATASET';
        } else if (stateName.startsWith('projects.project.streaming-endpoints')) {
            return 'STREAMING_ENDPOINT';
        }  else if (stateName.startsWith('projects.project.labelingtasks')) {
            return 'LABELING_TASK';
        } else if (stateName.startsWith('projects.project.managedfolders')) {
            return 'MANAGED_FOLDER';
        } else if (stateName.startsWith('projects.project.savedmodels') || stateName.startsWith('projects.project.genai')) {
            return 'SAVED_MODEL';
        } else if (stateName.startsWith('projects.project.modelevaluationstores')) {
            return 'MODEL_EVALUATION_STORE';
        } else if (stateName.startsWith('projects.project.modelcomparisons')) {
            return 'MODEL_COMPARISON';
        } else if (stateName.startsWith('projects.project.promptstudios')) {
            return 'PROMPT_STUDIO';
        } else if (stateName.startsWith('projects.project.agenttools')) {
            return 'AGENT_TOOL';
        } else if (stateName.startsWith('projects.project.recipes')) {
            return 'RECIPE';
        } else if (stateName.startsWith('projects.project.analyses')) {
            return 'ANALYSIS';
        } else if (stateName.startsWith('projects.project.scenarios')) {
            return 'SCENARIO';
        } else if (stateName.startsWith('projects.project.webapps') || stateName.startsWith('projects.project.dataikuanswers')) {
            return 'WEB_APP';
        } else if (stateName.startsWith('projects.project.code-studios')) {
            return 'CODE_STUDIO';
        } else if (stateName.startsWith('projects.project.reports')) {
            return 'REPORT';
        } else if (stateName.startsWith('projects.project.dashboards.insights')) {
            return 'INSIGHT';
        } else if (stateName.startsWith('projects.project.dashboards')) {
            return 'DASHBOARD';
        } else if (stateName.startsWith('projects.project.lambdaservices')) {
            return 'LAMBDA_SERVICE';
        } else if (stateName.startsWith('projects.project.continuous-activities')) {
            throw new Error("Cannot get continuous activity taggable type from angular state");
        } else if (stateName.startsWith('projects.project.notebooks')) {
            throw new Error("Cannot get notebook taggable type from angular state");
        }
        //Note that we never return 'PROJECT'
        throw new Error("Failed to get taggable type from angular state");
    };

    this.isComputable = function(tor) {
        if (!tor) return;
        return tor.type == 'DATASET' || tor.type == 'MANAGED_FOLDER' || tor.type == 'SAVED_MODEL' || tor.type == 'MODEL_EVALUATION_STORE' || tor.type == 'STREAMING_ENDPOINT';
    };

    this.isLocal = function(tor) {
        if (!tor || !$stateParams.projectKey) return;
        return tor.projectKey == $stateParams.projectKey;
    };
    this.isHDFSAbleType = isHDFSAbleType;

    this.humanReadableObjectType = function(objectType) {
        if (!objectType) return;
        switch(objectType) {
        case "MANAGED_FOLDER":
            return "folder";
        case "SAVED_MODEL":
            return "model";
        case "MODEL_EVALUATION_STORE":
            return "evaluation store";
        case "LAMBDA_SERVICE":
            return "API service";
        default:
            return objectType.toLowerCase().replace('_', ' ');
        }
    }
});


app.service("TaggableObjectsService", function($stateParams, $rootScope, $q, DataikuAPI, CreateModalFromTemplate, Dialogs, Logger, QuickSharingWT1EventsService) {
    const THUMBNAIL_WIDTH = 220;
    const THUMBNAIL_HEIGHT = 138;

    /* deletionRequests should be {type: <taggableType> , projectKey: ... , id: ...} */
    this.delete = function(deletionRequests, customMassDeleteSelected) {
        let deferred = $q.defer();

        CreateModalFromTemplate("/templates/taggable-objects/delete-modal.html", $rootScope, "DeleteTaggableObjectsModalController", function(modalScope) {
            modalScope.nrObjectsToDelete=0;
            modalScope.nrObjectsToUnshare=0;

            //show dialog in delete and reconnect mode - set to true if any of deletionRequests are for a delete and reconnect (either they all are or none of them) 
            modalScope.deleteAndReconnect = false;
            deletionRequests.forEach(function(dr) {
                dr.options =  {...dr.options, ...{dropData: false}} ;
                dr.projectKey == $stateParams.projectKey ? modalScope.nrObjectsToDelete+=1 : modalScope.nrObjectsToUnshare+=1;
                modalScope.deleteAndReconnect |= dr.options.deleteAndReconnect;
            });

            modalScope.computedImpact = {};
            modalScope.deletionRequests = deletionRequests;
            modalScope.currentProject = $stateParams.projectKey;
        }).then(function() {
            deletionRequests = deletionRequests.filter(it => it.type != "FLOW_ZONE" || it.id != "default"); // do not delete default zone
            deletionRequests.forEach(function(item) {
                if (item.type == 'JUPYTER_NOTEBOOK' && item.activeSessions) {
                    item.activeSessions.forEach(function(session) {
                        DataikuAPI.jupyterNotebooks.unload(session.sessionId);
                    });
                    delete item.activeSessions;
                }

            });

            let tmpDeletionRequests = [], unsharingRequests = [];
            for (const delRequest of deletionRequests) {
                delRequest.projectKey == $stateParams.projectKey ? tmpDeletionRequests.push(delRequest) : unsharingRequests.push(delRequest);
            }

            const deleteCall = customMassDeleteSelected || DataikuAPI.taggableObjects.delete;

            deleteCall(deletionRequests, $stateParams.projectKey)
                .success(function(errors) {
                    $rootScope.$emit('flowItemAddedOrRemoved', deletionRequests);

                    deletionRequests.forEach(function(req) {
                        // It there is an authorization error on one unshare, no object is unshared (even if we would have had the auth to unshare some)
                        // it's possible in theory that we could have other error cases that partially unshare, but it's super-edge case (backend IOErrors...)
                        if (!errors.error && req.projectKey !== $stateParams.projectKey) {
                            DataikuAPI.projects.getObjectExposition(req.projectKey, req.type, req.id).success((objectExposition) => {
                                const object = {
                                    projectKey: req.projectKey,
                                    quickSharingEnabled: objectExposition.quickSharingEnabled,
                                    type: req.type,
                                    localName: req.id,
                                };

                                QuickSharingWT1EventsService.onRuleRemoved(object, $stateParams.projectKey);
                                if(objectExposition.rules.length === 0 && !objectExposition.quickSharingEnabled) {
                                    QuickSharingWT1EventsService.onSharingDisabled({
                                        ...object,
                                        // we need WT1 to report the number of rules BEFORE disabling, but objectExposition is retrieved after, so we fake the deleted one.
                                        rules: [{ targetProject: $stateParams.projectKey }],
                                    });
                                }
                            });
                        }

                        // HistoryService.notifyRemoved({
                        //     type: "DATASET",
                        //     id: req.name,
                        //     projectKey: req.projectKey
                        // },
                        // $scope.computedImpact.data
                        // );
                    });

                    DataikuAPI.flow.zones.list($stateParams.projectKey).then(data => {
                        if (data.data.length == 1 && data.data[0].id == "default") {
                            DataikuAPI.flow.zones.delete($stateParams.projectKey, "default");
                        }
                    });
                    Dialogs.infoMessagesDisplayOnly($rootScope, "Cannot delete or unshare", errors);
                    deferred.resolve();
                })
                .error(function() {
                    deferred.reject.apply(this, arguments);
                })
        });
        return deferred.promise;
    };

    /**
     * Update the thumbnail data if necessary using the provided selector
     * @param {Object}   tor              - the taggable object reference, must contains a valid versionTag
     * @param {string}   selector         - a query selector string corresponding to the element to capture
     * @param {Document} [sourceDocument] - optional, the document where the element is located (e.g iframe document)
     * @returns a promise resolved with true when the thumbnail as been successfully updated, false if no update was necessary
     */
    this.checkAndUpdateThumbnailData = (tor, selector, sourceDocument = document) => {
        const deferred = $q.defer();

        DataikuAPI.images.getObjectThumbnailInfo(tor.projectKey, tor.type, tor.id).success((imageInfo) => {
            if (tor.versionTag && tor.versionTag.lastModifiedOn > imageInfo.lastModified) {
                Logger.debug(`Thumbnail for object ${tor.type} is outdated, generating a new one`);

                const containerToUseAsThumbnail = sourceDocument.querySelector(selector);
                if (containerToUseAsThumbnail) {

                    // Ensure all images are already loaded
                    const loadImagePromises = [];
                    containerToUseAsThumbnail.querySelectorAll('img').forEach(img => {
                        if (!img.complete) {
                            loadImagePromises.push($q((resolve) => {
                                img.onload = resolve;
                                img.onerror = resolve;
                            }));
                        }
                    });

                    $q.all(loadImagePromises)
                        .then(() => this.generateThumbnailData(selector, sourceDocument))
                        .then(thumbnailData => {
                            DataikuAPI.images.uploadCapturedObjectThumbnail(tor.projectKey, tor.type, tor.id, thumbnailData)
                                .success(() => deferred.resolve(true))
                                .error(deferred.reject.bind(deferred));
                        })
                        .catch(deferred.reject.bind(deferred));
                } else {
                    deferred.reject('Failed to generate thumbnail, html element not found');
                }
            } else {
                deferred.resolve(false);
            }
        });

        return deferred.promise;
    };

    /**
     * Generate a thumbnail for the specified element
     * @param {string}   selector         - a query selector string corresponding to the element to capture
     * @param {Document} [sourceDocument] - optional, the document where the element is located (e.g iframe document)
     * @returns a promise resolved with the thumbnail data
     */
    this.generateThumbnailData = (selector, sourceDocument = document) => {
        const container = sourceDocument.querySelector(selector);
        const rect = container.getBoundingClientRect();
        const width = rect.width;
        const height = width * THUMBNAIL_WIDTH / THUMBNAIL_HEIGHT;
        const scale = THUMBNAIL_WIDTH / width;
        const maxBounds = {
            x: rect.x + width,
            y: rect.y + height
        };

        // Remove scroll offset
        for (let node = container; node !== null; node = node.parentElement) {
            maxBounds.x += node.scrollLeft;
            maxBounds.y += node.scrollTop;
        }

        return html2canvas_latest(container, {
            foreignObjectRendering: true,
            width: width / scale,
            height: height / scale,
            scale,
            ignoreElements: element => {
                // Ignore out of bound elements
                const bounds = element.getBoundingClientRect();
                if (bounds.x > maxBounds.x || bounds.y > maxBounds.y) {
                    return true;
                }
                // Ignore elements not affiliated to the container
                return !element.contains(container) && !container.contains(element);
            },
            onclone: cloneDocument => {
                let node = cloneDocument.querySelector(selector);
                // Reset the position so the container is at 0, 0
                while (node) {
                    if (node.style) {
                        node.scrollTop = 0;
                        node.scrollLeft = 0;
                        node.style.position = 'absolute';
                        node.style.top = 0;
                        node.style.left = 0;
                        node.style.margin = 'initial';
                        node.style.padding = 'initial';
                    }
                    node = node.parentNode;
                }
                // Use images from the source document to avoid blank images or cors error
                const images = cloneDocument.querySelectorAll('img');
                const promises = [];
                if (images.length) {
                    const imageCanvas = document.createElement('canvas');
                    const imageCtx = imageCanvas.getContext('2d');
                    cloneDocument.querySelectorAll('img').forEach(img => {
                        img.crossOrigin = 'anonymous'; // Enable CORS without credentials
                        promises.push(new Promise((resolve) => {
                            img.onload = () => {
                                if (img && img.naturalWidth) {
                                    imageCanvas.width = img.naturalWidth;
                                    imageCanvas.height = img.naturalHeight;
                                    imageCtx.clearRect(0, 0, img.naturalWidth, img.naturalHeight);
                                    imageCtx.drawImage(img, 0, 0);
                                    img.src = imageCanvas.toDataURL();
                                }
                                img.onload = null;
                                resolve();
                            };
                            img.onerror = () => {
                                img.remove();
                                resolve();
                            }
                        }));
                    });
                }
                // Tell html2canvas to wait for image loading
                return Promise.all(promises);
            }
        }).then(canvas => {
            // Thumbnail has the desired size but the canvas is too large
            // we crop to the thumbnail size
            const thumbnail = document.createElement('canvas');
            thumbnail.width = THUMBNAIL_WIDTH;
            thumbnail.height = THUMBNAIL_HEIGHT;
            
            const imageData = canvas.getContext('2d').getImageData(0, 0, THUMBNAIL_WIDTH, THUMBNAIL_HEIGHT);
            thumbnail.getContext('2d').putImageData(imageData, 0, 0);
            return thumbnail.toDataURL();
        })
    };
});


app.filter('niceTaggableType', function($filter, $translate) {
    //Make sure to keep in sync with its angular version
    const dict = {
        'MANAGED_FOLDER': 'folder',
        'LOCAL_MANAGED_FOLDER': 'folder',
        'SAVED_MODEL': 'model',
        'LOCAL_SAVEDMODEL': 'model',
        'MODEL_EVALUATION_STORE': 'evaluation store',
        'LOCAL_MODEL_EVALUATION_STORE': 'evaluation store',
        'MODEL_COMPARISON': 'model comparison',
        'LAMBDA_SERVICE': 'API service',
        'SQL_NOTEBOOK': 'SQL notebook',
        'SEARCH_NOTEBOOK': 'Search notebook',
        'RETRIEVABLE_KNOWLEDGE': 'knowledge bank',
        'TAGGABLE_OBJECT': 'item'
    };
    const plurals = {
        'ANALYSIS': 'analyses'
    };

    return function(input, count = 1) {
        if (!input) return input;
        const key = "TAGGABLE_TYPE." + input;
        const usePlural = count > 1 || (count === 0 && ($translate && $translate.proposedLanguage() || $translate.use() || "en") === 'en');
        if (usePlural) {
            // First try to look for special plural case
            const pluralKey = key + ".PLURAL";
            const pluralResult = $translate.instant(pluralKey);
            if (pluralResult !== pluralKey) {
                return pluralResult;
            }
        }
        const result = $translate.instant(key);
        if (result === key) {
            // Translation not found, use default
            return $filter('plurify')(dict[input] || input.toLowerCase().replace('_', ' '), count, plurals[input]);
        } else {
            return usePlural ? result + 's' : result;
        }
    };
});

app.filter("taggableObjectRef", function() {
    return function(input) {
        if (!input || !input.type) return "";
        switch (input.type) {
            case 'PROJECT': return "Project " + input.projectKey;
            case 'DATASET': return "Dataset " + input.projectKey + "." + input.id;
            case 'RECIPE': return "Recipe " + input.projectKey + "." + input.id;
            default:
                // Pure laziness
                return input.id;
        }
    }
});


app.controller('_TaggableObjectPageRightColumnActions', function($scope, $rootScope, $controller, $state, TaggableObjectsUtils, CreateModalFromTemplate, $timeout, ActivityIndicator) {

    $controller('TaggableObjectPageMassActionsCallbacks', {$scope: $scope});

    $scope.getSelectedNodes = function() {
        return [$scope.selection.selectedObject];
    };

    $scope.getSelectedTaggableObjectRefs = function() {
        return $scope.getSelectedNodes().map(TaggableObjectsUtils.fromObjectItem);
    };

    const safeRenamingObjectTypes = ["AGENT_TOOL", "RETRIEVABLE_KNOWLEDGE"];
    $scope.safeIsDirty = function() {
        if (safeRenamingObjectTypes.includes($scope.objectType)) {
            if ($scope.isDirty != null) {
                return $scope.isDirty();
            } else if($scope.dirtySettings) {
                return $scope.dirtySettings();
            }
        }
        return false;
    };

    $scope.renameTaggableObject = function () {
        CreateModalFromTemplate("/templates/taggable-objects/rename-modal.html", $scope, null, function(newScope) {
            const currentName = $rootScope.topNav.item.data.name;
            newScope.objectName = currentName;
            newScope.uiState = { newName: currentName };
            newScope.go = function() {
                $scope.renameObjectAndSave(newScope.uiState.newName).success(() => {
                    ActivityIndicator.success("Saved");
                    $state.reload();
                }).error(setErrorInScope.bind($scope));
                newScope.dismiss();
            }
        });
    };

});


app.controller("DeleteTaggableObjectsModalController", function($scope, $state, $stateParams, Assert, DataikuAPI, WT1, TaggableObjectsUtils, FLOW_COMPUTABLE_TYPES, PUBLISHABLE_TYPES) {

    $scope.$watch("deletionRequests", function(newDeletionRequests) {
        if (newDeletionRequests == null) {
            return;
        }

        Assert.inScope($scope, 'deletionRequests');

        $scope.commonTaggableType = TaggableObjectsUtils.getCommonType($scope.deletionRequests, function(x) {return x.type;});

        // Flow computables, publishable items and dashboards deletion can have impact
        let typesWithImpact = PUBLISHABLE_TYPES.concat(FLOW_COMPUTABLE_TYPES).concat('DASHBOARD').concat('LABELING_TASK').concat('RETRIEVABLE_KNOWLEDGE');
        let computeImpact = $scope.deletionRequests.filter(function(x) {return typesWithImpact.indexOf(x.type) > -1;}).length > 0;

        if (computeImpact) {
            DataikuAPI.taggableObjects.computeDeletionImpact($scope.deletionRequests, $stateParams.projectKey)
            .success(function(data) {
                $scope.computedImpact.data = data;
                $scope.nrDistinctUnsharedItemTypes = new Set($scope.computedImpact.data.toUnshare.map(obj => obj.type)).size;
                $scope.nbActions = 0;
                $scope.nbActionsNotRequested = 0;
                for(let k in data.availableOptions) {
                    
                    // Avoid counting actions on shared objects
                    let isShared = false
                    for (const o of data.toUnshare) {
                        let objectName = o.projectKey + "." + o.id;
                        isShared |= objectName === k;
                    }

                    if (!isShared) {
                        let optns = data.availableOptions[k];
                        let enabled = false;
                        for(let j in optns) {
                            enabled |= 'isDropDataDangerous' !== j && optns[j];
                        }
                        if(enabled) {
                            $scope.nbActions++;
                            if (!$scope.deletionRequests.map(dr => (dr.projectKey + "." + dr.id)).includes(k)) {
                                $scope.nbActionsNotRequested++;
                            }
                        }
                    }
                }
                // dashboards have two additional actions
                $scope.deletionRequests.forEach(function(dr) {
                    switch (dr.type) {
                    case 'DASHBOARD':
                        $scope.nbActions += 2;
                        $scope.hasAnyDashboard = true;
                        break;
                    case 'DATASET':
                        dr.options.dropData = $scope.appConfig.dropDataWithDatasetDeleteEnabled;
                        dr.options.dropMetastoreTable = $scope.appConfig.dropDataWithDatasetDeleteEnabled;
                        break;
                    case 'MANAGED_FOLDER':
                        dr.options.dropData = $scope.appConfig.dropDataWithDatasetDeleteEnabled;
                        break;
                    case 'LABELING_TASK':
                        dr.options.dropData = $scope.appConfig.dropDataWithDatasetDeleteEnabled;
                        dr.options.dropData = false;
                        break;
                    }
                });
                let deletedLabelingTasks = angular.copy($scope.computedImpact.data.deletedLabelingTasks);
                deletedLabelingTasks.forEach(function(lt) {
                    lt.displayName = lt.name + " (" + lt.id + ")";
                    lt.options = $scope.computedImpact.data.availableOptions[lt.projectKey + "." + lt.id];
                    lt.type = "LABELING_TASK";
                    });
                $scope.allDeletedObjects = $scope.deletionRequests.concat(deletedLabelingTasks);
            })
            .error(setErrorInScope.bind($scope));
        } else {
            $scope.computedImpact.data = {
                ok: true
            };
        }

        $scope.confirm = function() {
            WT1.event("taggable-items-delete-many", {state: $state.current, numberOfItems: $scope.deletionRequests.length});
            $scope.resolveModal();
            $scope.dismiss();
        };

        $scope.isZoneDeleted = newDeletionRequests.some(deletionRequest => deletionRequest.type === "FLOW_ZONE");
    });
});

// WARNING Keep the switch in sync with other _XXX_MassActionsCallbacks controllers (flow, taggable objects pages, list pages)
app.controller('TaggableObjectPageMassActionsCallbacks', function($scope, $rootScope, $state, DKUtils, PIPELINEABILITY_ACTIONS) {

    $scope.onAction = function(action) { //NOSONAR
        switch (action) {
            case 'action-delete':
                $state.go("projects.project.flow");
                break;
            case 'action-tag':
                break;
            case 'action-watch':
            case 'action-star':
                $rootScope.$emit('userInterestsUpdated');
                break;
            case 'action-clear':
                DKUtils.reloadState();
                break;
            case 'action-build':
                break;
            case 'action-change-connection':
                DKUtils.reloadState();
                break;
            case 'action-update-status':
            case 'action-set-auto-count-of-records':
            case 'action-set-virtualizable':
            case 'action-add-to-scenario':
            case 'action-share':
                break;
            case 'action-unshare':
                $state.go("projects.project.flow");
                break;
            case 'action-change-recipes-engines':
            case 'action-change-spark-config':
            case PIPELINEABILITY_ACTIONS.changeSpark:
            case PIPELINEABILITY_ACTIONS.changeSQL:
            case 'action-change-impala-write-mode':
            case 'action-change-hive-engine':
            case 'action-change-spark-engine':
                //Should not be possible on this page
                break;
            case 'action-convert-to-hive':
            case 'action-convert-to-impala':
                DKUtils.reloadState();
                break;
            case 'action-change-python-env':
            case 'action-change-r-env':
                DKUtils.reloadState();
                break;
            case 'action-code-studio-state-change':
                break;
            default:
                break;
        }
    }
});

// WARNING Keep the switch in sync with other _XXX_MassActionsCallbacks controllers (flow, taggable objects pages, list pages)
app.controller('ListMassActionsCallbacks', function($scope, PIPELINEABILITY_ACTIONS) {

    $scope.onAction = function(action) { //NOSONAR
        switch (action) {
            case 'action-delete':
            case 'action-tag':
            case 'action-watch':
            case 'action-star':
            case 'action-clear':
            case 'action-build':
            case 'action-change-connection':
            case 'action-update-status':
            case 'action-set-auto-count-of-records':
            case 'action-set-virtualizable':
            case 'action-add-to-scenario':
            case 'action-share':
            case 'action-unshare':
            case 'action-change-recipes-engines':
            case 'action-change-spark-config':
            case PIPELINEABILITY_ACTIONS.changeSQL:
            case PIPELINEABILITY_ACTIONS.changeSpark:
            case 'action-change-impala-write-mode':
            case 'action-change-hive-engine':
            case 'action-change-spark-engine':
            case 'action-convert-to-hive':
            case 'action-convert-to-impala':
            case 'action-change-python-env':
            case 'action-change-r-env':
            case 'action-code-studio-state-change':
                $scope.list();
                break;
            default:
                break;
        }
    }
});


app.controller('_TaggableObjectsListPageCommon', function($controller, $scope, $filter, $state, $stateParams, $rootScope,
    DataikuAPI, WT1, Fn,
    TaggableObjectsService, TaggableObjectsUtils, TaggingService, InterestsService) {

    $controller('ListMassActionsCallbacks', {$scope: $scope});
    $controller('_TaggableObjectsMassActions', {$scope: $scope});

    let loadMoreLock = false;
    $scope.loadMoreItems = function() {
        if(!loadMoreLock && $scope.listItems && $scope.maxItems < $scope.listItems.length) {
            $scope.maxItems += 20;
            loadMoreLock = true;
            setTimeout(function() {loadMoreLock=false;}, 300);
        }
    };

    $scope.selectTag = function(event, tag) {
        if (!$scope.selection || !$scope.selection.filterQuery || !$scope.selection.filterQuery.tags) return;
        let index = $scope.selection.filterQuery.tags.indexOf(tag);
        if (index >= 0) {
            $scope.selection.filterQuery.tags.splice(index, 1);
        } else {
            $scope.selection.filterQuery.tags.push(tag);
        }
    };

    $scope.restoreOriginalSelection = function() {
        if (!$scope.selection || !$scope.selection.filteredSelectedObjects) {
            return
        }
        $scope.selection.filteredSelectedObjects.forEach(obj => {
            let i = $scope.listItems.find(item => item.id === obj.id);
            if (i) {
                i.$selected = true;
            }
        });
    }

    $scope.$on('tagSelectedInList', function(e, tag) {
        $scope.selectTag($scope.selection.filterQuery,tag);
        e.stopPropagation();
    });

    $scope.$on('selectedIndex', function(e, index) {
        // an index has been selected, we unselect the multiselect
        $scope.$broadcast('clearMultiSelect');
    });

    $scope.$on('projectTagsUpdated', function (e, args) {
        if (args.refreshFlowFilters) $scope.list();
    });

    const updateSelectedObjectMetadata = function (metaData) {
        const o = $scope.selection.selectedObject;
        if (o && metaData) {
            o.shortDesc = metaData.shortDesc;
            o.description = metaData.description;
            o.tags = angular.copy(metaData.tags);
            o.checklists = angular.copy(metaData.checklists);
            o.customFields = angular.copy(metaData.customFields);
        }
    };

    const unregisterObjectMetaDataRefresh = $rootScope.$on("objectMetaDataRefresh", (ev, metaData) => {
        updateSelectedObjectMetadata(metaData);
    });

    const unregisterObjectMetaDataChanged = $rootScope.$on('objectMetaDataChanged', (ev, metaData) => {
        updateSelectedObjectMetadata(metaData);
    });

    $scope.allStarred = function(listItems) {
        if (!listItems || !listItems.length) return true;
        return listItems.map(x => !!(x && x.interest && x.interest.starred)).reduce((a,b) => a && b);
    };

    $scope.allWatching = function(listItems) {
        if (!listItems || !listItems.length) return true;

        return listItems
            .map(it => it.interest && it.interest.watching)
            .every($scope.isWatching);
    };

    $scope.watchObject = function(watch, item) {
        InterestsService.watch($scope, [TaggableObjectsUtils.fromListItem(item)], watch).then($scope.list); //GRUIK, list is not that necessary
    };

    $scope.starObject = function(star, item) {
        InterestsService.star($scope, [TaggableObjectsUtils.fromListItem(item)], star).then($scope.list); //GRUIK, list is not that necessary
    };

    $scope.toggleFilterStarred = function() {
        let fq = $scope.selection.filterQuery;
        fq.interest.starred = (fq.interest.starred === '' ? 'true' : '');
    };

    /* Default list call */
    $scope.list = function() {
        $scope.listHeads($stateParams.projectKey, $scope.tagFilter).success(function(data) {
            $scope.filteredOut = data.filteredOut;
            $scope.listItems = data.items;
            $scope.restoreOriginalSelection();
            if ($scope.listHeadHook && typeof($scope.listHeadHook.list) === "function") {
                $scope.listHeadHook.list($scope.listItems);
            }
        }).error(setErrorInScope.bind($scope));
    };

    $scope.getSelectedListItems = function() {
        if($scope.selection.selectedObjects && $scope.selection.selectedObjects.length) {
            return $scope.selection.selectedObjects;
        } else if ($scope.selection.selectedObject) {
            return [$scope.selection.selectedObject];
        } else {
            return [];
        }
    };

    $scope.getSelectedTaggableObjectRefs = function() {
        return $scope.getSelectedListItems().map(TaggableObjectsUtils.fromListItem);
    };

    $scope.$on("$destroy", () => {
        unregisterObjectMetaDataRefresh();
        unregisterObjectMetaDataChanged();
    });

    try {
        $scope.listItemType = TaggableObjectsUtils.taggableTypeFromAngularState();
    } catch (e) {
        // eslint-disable-next-line no-console
        console.info("Cannot set the taggable type on this list page"); // It will be the case for notebooks in particular since there is no "notebook" taggableType
    }
});


app.service('TaggableObjectsCapabilities', function(
    $stateParams,
    AI_EXPLANATION_MODAL_MODES,
    RecipesCapabilities,
    TaggableObjectsUtils,
    CreateModalFromTemplate
) {
    $.extend(this, RecipesCapabilities);

    this.canChangeConnection = function(tor) {
        if (tor.nodeType) {
            //This is a node, not a taggableObjectReference
            tor = TaggableObjectsUtils.fromNode(tor);
        }
        if ($stateParams.projectKey && tor.projectKey != $stateParams.projectKey) {
            return false;
        }
        if (tor.type == 'DATASET') {
            if (tor.subType && ['Inline', 'UploadedFiles'].includes(tor.subType)) {
                return false;
            }
            return true;
        } else if (tor.type == 'MANAGED_FOLDER') {
            return true;
        }
        return false;
    };
    this.canSyncMetastore = function(tor) {
        if (tor.nodeType) {
            //This is a node, not a taggableObjectReference
            tor = TaggableObjectsUtils.fromNode(tor);
        }
        if ($stateParams.projectKey && tor.projectKey != $stateParams.projectKey) {
            return false;
        }
        if (tor.type == 'DATASET') {
            if (tor.subType && isHDFSAbleType(tor.subType)) {
                return true;
            }
        }
        return false;
    };

    this.explainObject = function() {
        const scope = this;
        CreateModalFromTemplate(
            "/static/dataiku/ai-explanations/explanation-modal/explanation-modal.html",
            scope,
            "AIExplanationModalController",
            function(newScope) {
                newScope.objectType = TaggableObjectsUtils.fromNodeType(
                    scope.selection.confirmedItem.nodeType
                );
                newScope.object = scope.selection.selectedObject;
                newScope.mode = AI_EXPLANATION_MODAL_MODES.EXPLAIN;
            }
        );
    };
});


// Move the service functionalities to the scope
app.controller('_TaggableObjectsCapabilities', function($scope, TaggableObjectsCapabilities) {
    $.extend($scope, TaggableObjectsCapabilities);
});


app.controller('_TaggableObjectsMassActions', function($scope, $state, $rootScope, $q, $stateParams, WT1, Logger, Dialogs, DataikuAPI, CreateModalFromTemplate,
    TaggableObjectsService, TaggingService,
    ImpalaService, HiveService, SparkService, ComputablesService, DatasetsService, DatasetConnectionChangeService, SubFlowCopyService, RecipesEnginesService, ExposedObjectsService,
    GlobalProjectActions, RecipeDescService, FlowGraphSelection, FlowTool,
    InterestsService, InterestWording, WatchInterestState, TaggableObjectsUtils,
    CodeEnvsService, ToolBridgeService, PipelineService, PIPELINEABILITY_ACTIONS) {
    //Expects a $scope.getSelectedTaggableObjectRefs()

    function onAction(action) {
        return function(data) {
            if ($scope.onAction) {
                $scope.onAction(action);
            } else {
                Logger.warn('No mass action callbacks handler');
            }
            return data;
        };
    }

    // Generic mass actions

    $scope.deleteSelected = function(items = $scope.getSelectedTaggableObjectRefs(), onSuccess = onAction('action-delete')) {
        WT1.event("action-delete", {state: $state.current.name, items: items.length});
        return TaggableObjectsService.delete(items, $scope.customMassDeleteSelected).then(onSuccess);
    };

    $scope.startApplyTagging = function(selection = $scope.getSelectedTaggableObjectRefs()) {
        WT1.event("action-tag", {state: $state.current.name, items: selection.length});
        TaggingService.startApplyTagging(selection).then(onAction('action-tag'));
    };

    $scope.copyAllSelected = function() {
        // Note that we want the nodes here, not the refs
        WT1.event("action-copy-all", {state: $state.current.name, preselectedNodes: $scope.getSelectedNodes().length});
        $scope.startTool('COPY', {preselectedNodes: $scope.getSelectedNodes().map(n => n.id)});
    };

    $scope.watchObjects = function(watch) {
        InterestsService.watch($scope, $scope.getSelectedTaggableObjectRefs(), watch).then(onAction('action-watch'));
    };

    $scope.starObjects = function(star) {
        InterestsService.star($scope, $scope.getSelectedTaggableObjectRefs(), star).then(onAction('action-star'));
    };

    $scope.isWatching = WatchInterestState.isWatching;
    $scope.actionLabels = { ...InterestWording.labels };
    $scope.actionTooltips = { ...InterestWording.tooltips };

    $scope.shareObjectsInWorkspace = (items = $scope.getSelectedTaggableObjectRefs().map(tor => ({ reference: tor}))) => {
        CreateModalFromTemplate('/templates/dialogs/share-in-workspace.html', $scope, undefined, (newScope) => {
            newScope.init(items);
        });
    }

    $scope.addToDataCollection = (items = $scope.getSelectedTaggableObjectRefs()) => {
        // we need rootScope, because when resizing screen when several objects are selected, right panel is re-rendered, which kills the current $scope and break the modal
        CreateModalFromTemplate('/templates/dialogs/add-to-data-collection.html', $rootScope, undefined, (newScope) => {
            newScope.init(items);
        });
    }

    // Computables

    $scope.resynchronizeMetastore = function() {
        WT1.event("action-sync-metastore", {state: $state.current.name, items: $scope.getSelectedTaggableObjectRefs().length});
        HiveService.resynchronizeMetastore($scope.getSelectedTaggableObjectRefs());
    };

    $scope.resynchronizeDataset = function() {
        WT1.event("action-sync-dataset", {state: $state.current.name, items: $scope.getSelectedTaggableObjectRefs().length});
        HiveService.resynchronizeDataset($scope.getSelectedTaggableObjectRefs());
    };

    $scope.clearSelected = function() {
        WT1.event("action-clear", {state: $state.current.name, items: $scope.getSelectedTaggableObjectRefs().length});
        ComputablesService.clear($scope, $scope.getSelectedTaggableObjectRefs()).then(onAction('action-clear'));
    };

    $scope.buildSelected = function() {
        WT1.event("action-build", {state: $state.current.name, items: $scope.getSelectedTaggableObjectRefs().length});
        $scope.buildSelectedComputables($scope.getSelectedTaggableObjectRefs()).then(onAction('action-build'));
    }

    $scope.changeSelectedItemsConnections = function() {
        WT1.event("action-change-connection", {state: $state.current.name, items: $scope.getSelectedTaggableObjectRefs().length});
        DatasetConnectionChangeService.start($scope.getSelectedTaggableObjectRefs()).then(onAction('action-change-connection'));
    };

    $scope.updateStatuses = function() {
        const items = $scope.getSelectedTaggableObjectRefs().filter(it => it.type == 'DATASET');
        WT1.event("action-update-status", {state: $state.current.name, items: items.length});
        DatasetsService.refreshSummaries($scope, items)
            .then(onAction('action-update-status'))
            .then(function(result) {
                if(result.anyMessage) {
                    Dialogs.infoMessagesDisplayOnly($scope, "Datasets statuses update results", result);
                }
            }, setErrorInScope.bind($scope))
    };

    $scope.startSetAutoCountOfRecords = function() {
        const items = $scope.getSelectedTaggableObjectRefs().filter(it => it.type == 'DATASET');
        WT1.event("action-set-auto-count-of-records", {state: $state.current.name, items: items.length});
        DatasetsService.startSetAutoCountOfRecords(items).then(onAction('action-set-auto-count-of-records'));
    };

    $scope.setAutoCountOfRecords = function(autoCountOfRecords) {
        const items = $scope.getSelectedTaggableObjectRefs().filter(it => it.type == 'DATASET');
        WT1.event("action-set-auto-count-of-records2", {state: $state.current.name, items: items.length});
        DatasetsService.setAutoCountOfRecords(items, autoCountOfRecords).then(onAction('action-set-auto-count-of-records'));
    };

    $scope.setVirtualizable = function(virtualizable) {
        WT1.event("action-set-virtualizable", {state: $state.current.name, items: $scope.getSelectedTaggableObjectRefs().length});
        DatasetsService.setVirtualizable($scope, $scope.getSelectedTaggableObjectRefs(), virtualizable).then(onAction('action-set-virtualizable'));
    };

    $scope.addSelectedToScenario = function() {
        WT1.event("action-add-to-scenario", {state: $state.current.name, items: $scope.getSelectedTaggableObjectRefs().length});

        CreateModalFromTemplate('/templates/scenarios/add-to-scenario-modal.html', $scope, 'AddToScenarioModalController', function(modalScope) {
        /*empty?*/}).then(onAction('action-add-to-scenario'));
    };

    $scope.exposeSelected = function() {
        WT1.event("action-share", {state: $state.current.name, items: $scope.getSelectedTaggableObjectRefs().length});
        ExposedObjectsService.exposeObjects($scope.getSelectedTaggableObjectRefs()).then(onAction('action-share'));
    };

    $scope.unshare = function() {
        WT1.event("action-unshare", {state: $state.current.name, items: $scope.getSelectedTaggableObjectRefs().length});
        var items = $scope.getSelectedTaggableObjectRefs();
        var deletionRequests = items.map(item => ({type: item.type, projectKey: item.projectKey, id: item.id, displayName: item.displayName}))

        ExposedObjectsService.unshare(deletionRequests).then(onAction('action-unshare'));
    };

    $scope.isZoneInput = function() {
        return ($scope.selection.selectedObject.usedByZones.length && $scope.selection.selectedObject.usedByZones[0] != $scope.selection.selectedObject.ownerZone);
    };

    // Recipes

    $scope.changeSelectedRecipesEngines = function() {
        WT1.event("action-change-recipes-engines", {state: $state.current.name, items: $scope.getSelectedTaggableObjectRefs().length});
        RecipesEnginesService.startChangeEngine($scope.getSelectedTaggableObjectRefs()).then(onAction('action-change-recipes-engines'));
    };

    $scope.changeSelectedSparkConfig = function() {
        WT1.event("action-change-spark-config", {state: $state.current.name, items: $scope.getSelectedTaggableObjectRefs().length});
        SparkService.startChangeSparkConfig($scope.getSelectedTaggableObjectRefs()).then(onAction('action-change-spark-config'));
    };

    $scope.changeSelectedPipelineability = function(pipelineActionType) {
        WT1.event(pipelineActionType, {state: $state.current.name, items: $scope.getSelectedTaggableObjectRefs().length});
        const pipelineType = pipelineActionType === PIPELINEABILITY_ACTIONS.changeSpark ? 'SPARK' : 'SQL';
        PipelineService
            .startChangePipelineability($scope.getSelectedTaggableObjectRefs(), pipelineType)
            .then(onAction(pipelineActionType));
    };

    $scope.changeSelectedSparkPipelineability = function() {
        $scope.changeSelectedPipelineability(PIPELINEABILITY_ACTIONS.changeSpark);
    };

    $scope.changeSelectedSqlPipelineability = function() {
        $scope.changeSelectedPipelineability(PIPELINEABILITY_ACTIONS.changeSQL);
    };

    $scope.changeSelectedImpalaWriteMode = function() {
        WT1.event("action-change-impala-write-mode", {state: $state.current.name, items: $scope.getSelectedTaggableObjectRefs().length});
        ImpalaService.startChangeWriteMode($scope.getSelectedTaggableObjectRefs()).then(onAction('action-change-impala-write-mode')).then(function() {
            ToolBridgeService.emitRefreshView('ImpalaWriteModeView');
        });
    };

    $scope.changeSelectedHiveEngine = function() {
        WT1.event("action-change-hive-engine", {state: $state.current.name, items: $scope.getSelectedTaggableObjectRefs().length});
        HiveService.startChangeHiveEngine($scope.getSelectedTaggableObjectRefs()).then(onAction('action-change-hive-engine')).then(function() {
            ToolBridgeService.emitRefreshView('HiveModeView');
        });
    };

    $scope.convertSelectedToHive = function() {
        WT1.event("action-convert-to-hive", {state: $state.current.name, items: $scope.getSelectedTaggableObjectRefs().length});
        ImpalaService.convertToHive($scope.getSelectedTaggableObjectRefs()).then(onAction('action-convert-to-hive')).then(function() {
            ToolBridgeService.emitRefreshView('HiveModeView');
        });
    };

    $scope.convertSelectedToImpala = function() {
        WT1.event("action-convert-to-impala", {state: $state.current.name, items: $scope.getSelectedTaggableObjectRefs().length});
        HiveService.convertToImpala($scope.getSelectedTaggableObjectRefs()).then(onAction('action-convert-to-impala')).then(function() {
            ToolBridgeService.emitRefreshView('ImpalaWriteModeView');
        });
    };

    $scope.changePythonEnvSelection = function() {
        WT1.event("action-change-python-env", {state: $state.current.name, items: $scope.getSelectedTaggableObjectRefs().length});
        CodeEnvsService.startChangeCodeEnv($scope.getSelectedTaggableObjectRefs(), 'PYTHON', $scope).then(onAction('action-change-python-env'))
    };

    $scope.changeREnvSelection = function() {
        WT1.event("action-change-r-env", {state: $state.current.name, items: $scope.getSelectedTaggableObjectRefs().length});
        CodeEnvsService.startChangeCodeEnv($scope.getSelectedTaggableObjectRefs(), 'R', $scope).then(onAction('action-change-r-env'))
    };

    $scope.computeActionsUsability = function() {
        const usability = {
            recipes: { // quick hack: prefill with actions that will be available even without datasets
                python: {ok: true},
                r: {ok: true},
                shell: {ok: true},
                julia: {ok: true},
            },
            things: {},
            selectablePlugins: []
        };

        if ($scope.isAllEvaluationStores || $scope.isAllModels) {
            usability.recipes.r.ok = false;
            usability.recipes.shell.ok = false;
            usability.recipes.julia.ok = false;
        }

        function mergeUsability(itemUsability) {
            const rus = usability.recipes;
            $.each(itemUsability.recipes, function(recipeType, ru) {
                if (!rus[recipeType]) {
                    rus[recipeType] = angular.copy({ok: true, details: {enableStatus: 'OK'} });
                }
                if (!rus[recipeType].ok) {
                    return;
                }
                if (!ru.ok) {
                    rus[recipeType].ok = false;
                    rus[recipeType].reason = ru.reason;
                }
            });


            const tus = usability.things;
            $.each(itemUsability.things, function(thing, tu) {
                if (!tus[thing]) {
                    tus[thing] = angular.copy({ok: true, details: {enableStatus: 'OK'} });
                }
                if (!tus[thing].ok) {
                    return;
                }
                if (!tu.ok) {
                    tus[thing].ok = false;
                    tus[thing].reason = tu.reason;
                }
            });
        }

        $scope.getSelectedTaggableObjectRefs().forEach(function(item) {
            if (item.type == 'DATASET') {
                const datasetRef = {
                    projectKey: item.projectKey,
                    name: item.id,
                    type: item.subType
                };
                mergeUsability(GlobalProjectActions.getAllStatusForDataset(datasetRef))
            }
        });
        return usability;
    }
    $scope.startContinuous = function(item) {
        WT1.event("start-continuous")
        CreateModalFromTemplate("/templates/continuous-activities/start-continuous-activity-modal.html", $scope, "StartContinuousActivityController", function(newScope) {
            newScope.recipeId = item.name;
        }).then(function(loopParams) {
            DataikuAPI.continuousActivities.start($stateParams.projectKey, item.name, loopParams).success(function(data){
                onAction('action-build');
            }).error(setErrorInScope.bind($scope));
        });
    }
    $scope.stopContinuous = function(item) {
        WT1.event("stop-continuous")
        DataikuAPI.continuousActivities.stop($stateParams.projectKey, item.name).success(function(data){
            onAction('action-build');
        }).error(setErrorInScope.bind($scope));
    }
    
    $scope.startAllContinuous = function(objects) {
        WT1.event("start-continuous")
        CreateModalFromTemplate("/templates/continuous-activities/start-continuous-activity-modal.html", $scope, "StartContinuousActivityController", function(newScope) {
            newScope.recipeId = objects[0].name;
        }).then(function(loopParams) {
            let promises = objects.map(function(object) {
                return DataikuAPI.continuousActivities.start($stateParams.projectKey, object.name, loopParams)
            });
            $q.all(promises).then(function (values) {
                onAction('action-build');
            });
        });
    };
    $scope.stopAllContinuous = function(objects) {
        WT1.event("stop-continuous")
        let promises = objects.map(function(object) {
            return DataikuAPI.continuousActivities.stop($stateParams.projectKey, object.name)
        });
        $q.all(promises).then(function (values) {
            onAction('action-build');
        });
    };
    $scope.refreshAllContinuous = function(objects) {
        // grab the state of all continuous activities in the project, not just the selected ones
        // but the nodes objects are in the objects parameter
        DataikuAPI.continuousActivities.getStates($stateParams.projectKey).success(function(data) {
            // compute the {beingBuilt,continuousActivityDone} of each activity
            let states = {};
            data.activities.forEach(function(activity) {
                let beingBuilt = activity.desiredState == 'STARTED';
                let continuousActivityDone = beingBuilt && ((activity.mainLoopState || {}).futureInfo || {}).hasResult
                states[activity.recipeId] = {beingBuilt: beingBuilt, continuousActivityDone: continuousActivityDone}
            });
            
            var changes = 0;
            objects.forEach(function(o) {
                let continuousActivityDone = (states[o.name] || {}).continuousActivityDone || false;
                if (o.continuousActivityDone != continuousActivityDone) {
                    o.continuousActivityDone = continuousActivityDone;
                    changes += 1;
                }                    
            });
            if (changes > 0) {
                $rootScope.$broadcast("graphRendered");
            }
        }).error(setErrorInScope.bind($scope));
    };
    $scope.startAllCodeStudios = function( codeStudios){
        let refreshList = () => onAction('action-code-studio-state-change')();
        const stoppedCodeStudios = codeStudios.filter(k => k.uiState.state !== 'RUNNING');
        if (stoppedCodeStudios.length > 0) {
            Promise
                .all(stoppedCodeStudios.map(async k => $scope.restart(k.id)))
                .then(refreshList, refreshList);
        }
    };
    $scope.stopAllCodeStudios = function( codeStudios){
        let refreshList = () => onAction('action-code-studio-state-change')();
        const runningCodeStudios = codeStudios.filter(k => k.uiState.state === 'RUNNING');
        if (runningCodeStudios.length > 0) {
            Dialogs.confirmInfoMessages($scope, 'Confirm ' + runningCodeStudios.length +' Code Studios Shutdown', {}, "You will disconnect current users of these Code Studios (if any).", false)
                .then(() => {
                    Promise
                        .all(runningCodeStudios.map(async k => $scope.stop(k.id)))
                        .then(refreshList, refreshList);
                });
        }
    };
    
});


app.directive('contributorsList', function($state) {
    return {
        templateUrl: '/templates/contributors-list.html',
        scope: {
            timeline: '=tl'
        },
        link: function(scope) {
            scope.$state = $state;
        }
    };
});


app.service("_SummaryHelper", function(
    $stateParams,
    AI_EXPLANATION_MODAL_MODES,
    InterestsService,
    DataikuAPI,
    CreateModalFromTemplate,
    Dialogs,
    WT1,
    InterestWording,
    WatchInterestState
) {
    const svc = this;

    this.addEditBehaviour = function($scope, element) {

        $scope.state = {
            currentEditing : null,
            name : { editing: false, newVal: null, selector : ".name-edit-zone"},
            shortDesc : { editing: false, newVal: null, selector : ".shortdesc-edit-zone"},
            description : { editing: false, newVal: null, selector : ".desc-edit-zone"},
            tags: {editing : false, newVal : null},
            projectStatus: {editing : false, newVal : null},

            checklistTitle : { editing : null, newVal : null }
        };

        function cancelAllEdits () {
            if ($scope.state.checklistTitle.editing) {
                $scope.cancelChecklistTitleEdit();
            }
            if ($scope.state.currentEditing) {
                $scope.cancelFieldEdit();
            }
            if ($scope.state.tags.editing) {
                $scope.cancelEditTags();
            }
        }

        $scope.startEditChecklistTitle = function(checklist) {
            cancelAllEdits();
            checklist.editingTitle = true;
            $scope.state.checklistTitle.editing = checklist;
            $scope.state.checklistTitle.newVal = checklist.title;
            window.setTimeout(function() {
                $(".checklist-title", element).on("click.editField", function(e) {
                    e.stopPropagation();
                });
                $("html").on("click.editField", function(event) {
                    $scope.$apply(function() {$scope.cancelChecklistTitleEdit()});
                })
            }, 0)
        };

        $scope.validateChecklistTitleEdit = function() {
            $scope.state.checklistTitle.editing.title = $scope.state.checklistTitle.newVal;
            $scope.cancelChecklistTitleEdit();

            $scope.$emit("objectSummaryEdited");
        };

        $scope.cancelChecklistTitleEdit = function() {
            $scope.state.checklistTitle.editing.editingTitle = false;
            $scope.state.checklistTitle.editing = null;
            $(".checklist-title", element).off("click.editField");
            $("html").off("click.editField");
        };

        $scope.startFieldEdit = function(field, allowed) {
            if (allowed === false) return;
            cancelAllEdits();
            const fstate = $scope.state[field];
            fstate.editing = true;
            if (fstate.newVal == null) {
                fstate.newVal = $scope.object[field];
            }
            $scope.state.currentEditing = field;

            window.setTimeout(function() {
                fstate.suppressClick = false;
                $(fstate.selector, element).on("mousedown.editField", function(e) {
                    fstate.suppressClick = true;
                });
                $("html").on("mouseup.editField", function(event) {
                    const filterCMHints = function(node) {
                        return node.className == 'CodeMirror-hints';
                    };
                    const filterCMModal = function(node) {
                    return Array.prototype.indexOf.call(node.classList || [],'codemirror-editor-modal') >= 0;  //often this is a DOMTokenList not an array
                    };
                    const filterBSSelect = function(node) {
                        return Array.prototype.indexOf.call(node.classList || [],'bootstrap-select') >= 0;
                    };
                    const filterObjectSelector = function(node) {
                        return Array.prototype.indexOf.call(node.classList || [],'dss-object-selector-popover') >= 0;
                    };
                    const filterNGDropdownPanel = function(node) {
                        return Array.prototype.indexOf.call(node.classList || [], 'ng-dropdown-panel') >= 0;
                    };
                    const path = event.originalEvent && (event.originalEvent.path || (event.originalEvent.composedPath && event.originalEvent.composedPath()));
                    const isEventFromCMHints = path && path.filter(filterCMHints).length > 0;
                    const isEventFromCMModal = path && path.filter(filterCMModal).length > 0;
                    const isEventFromBSSelect = path && path.filter(filterBSSelect).length > 0;
                    const isEventFromObjectSelector = path && path.filter(filterObjectSelector).length > 0;
                    const isEventFromNGDropdownPanel = path && path.filter(filterNGDropdownPanel).length > 0;
                    if (
                        fstate.suppressClick ||
                        isEventFromCMHints ||
                        isEventFromCMModal ||
                        isEventFromBSSelect ||
                        isEventFromObjectSelector ||
                        isEventFromNGDropdownPanel
                    ) {
                        fstate.suppressClick = null;
                    } else {
                        $scope.$apply(function() {$scope.cancelFieldEdit()});
                    }
                })
                $scope.$broadcast('elastic:adjust');
            }, 0)
        };

        $scope.validateFieldEdit = function($event) {
            if ($event) {
                $event.preventDefault();
            }
            $scope.object[$scope.state.currentEditing] = $scope.state[$scope.state.currentEditing].newVal;
            $scope.state[$scope.state.currentEditing].newVal = null;
            $scope.$emit("objectSummaryEdited", $scope.state.currentEditing);
            $scope.cancelFieldEdit();
        };

        $scope.validateFieldEditNotUndefined = function() {
            if ($scope.state[$scope.state.currentEditing].newVal != undefined) {
                $scope.validateFieldEdit();
            }
        }

        $scope.cancelFieldEdit = function() {
            const field = $scope.state.currentEditing;
            if (field) {
                const fstate = $scope.state[field];
                fstate.editing = false;
                fstate.newVal = null;
                $(fstate.selector, element).off("mousedown.editField");
                $("html").off("mouseup.editField");
            }
        };

        $scope.startEditTags = function() {
            cancelAllEdits();
            if ($scope.state.tags.newVal == null) {
                $scope.state.tags.newVal = angular.copy($scope.object.tags);
            }
            $scope.state.tags.editing = true;
        };

        $scope.validateEditTags = function() {
            $scope.$broadcast("tagFieldAddTag", function() {
                $scope.object.tags = angular.copy($scope.state.tags.newVal);
                $scope.state.tags.newVal = null;
                $scope.state.tags.editing = false;
                $scope.$emit("objectSummaryEdited");
            });
        };

        $scope.cancelEditTags = function() {
            $scope.state.tags.newVal = null;
            $scope.state.tags.editing = false;
        };

        $scope.addChecklist = function(index) {
            WT1.event("add-checklist", {objectType: $scope.objectType});
            const nChecklists = $scope.object.checklists.checklists.length;
            $scope.object.checklists.checklists.push({
                id: Math.floor(Math.random()*16777215).toString(16), //16777215 == ffffff
                title:"Todo list"+(nChecklists ? ' '+(nChecklists+1) : ''),
                items: [],
                $newlyCreated : true
            });
        };

        $scope.deleteChecklist = function(index) {
            Dialogs.confirmSimple($scope, "Delete checklist").then(function() {
                $scope.object.checklists.checklists.splice(index, 1);
                $scope.$emit("objectSummaryEdited");
            });
        };

        $scope.$on("checklistEdited", function() {
            $scope.$emit("objectSummaryEdited");
        });

        $scope.editProjectStatus = function(projectStatus) {
            $scope.state.projectStatus.newVal = projectStatus;
            $scope.validateFieldEdit();
        };

        $scope.generateDescriptionForObject = function() {
            return CreateModalFromTemplate(
                "/static/dataiku/ai-explanations/explanation-modal/explanation-modal.html",
                $scope,
                "AIExplanationModalController",
                function(newScope) {
                    // newScope.objectType and newScope.object are inherited
                    newScope.mode = AI_EXPLANATION_MODAL_MODES.GENERATE;
                }
            );
        };
    };

    this.addInterestsManagementBehaviour = function($scope) {
        function getTaggableObjects() {
            return [{
                type: $scope.objectType,
                projectKey: $stateParams.projectKey,
                id: $scope.getObjectId(),
                workspaceKey: $stateParams.workspaceKey,
            }];
        }

        $scope.toggleWatch = function (event) {
            const currentState = $scope.data.interest.watching;
            const nextState = (currentState === WatchInterestState.values.YES || currentState === WatchInterestState.values.SHALLOW) 
                              ? WatchInterestState.values.ENO 
                              : WatchInterestState.values.YES;
            $scope.watchObject(nextState, event);
        };
        

        $scope.watchObject = function(watch, event) {
            return InterestsService.watch($scope, getTaggableObjects(), watch)
                .success(function() {
                    InterestsService.getInterest(
                        $scope,
                        $scope.$root.appConfig.user,
                        $scope.objectType,
                        $stateParams.projectKey,
                        $scope.getObjectId(),
                        $stateParams.workspaceKey 
                    ).success(function(newInterest) {
                        $scope.data.interest.nbWatching = newInterest.nbWatching;
                        $scope.data.interest.watching = newInterest.watching;

                        if (event) $scope.refreshTooltip(event);
                    });
                });
        };
        

        $scope.starObject = function(star, event) {
            return InterestsService.star($scope, getTaggableObjects(), star)
                .success(function () {
                    InterestsService.getInterest(
                        $scope,
                        $scope.$root.appConfig.user,
                        $scope.objectType,
                        $stateParams.projectKey,
                        $scope.getObjectId(),
                        $stateParams.workspaceKey
                    ).success(function (newInterest) {
                        $scope.data.interest.nbStarred = newInterest.nbStarred;
                        $scope.data.interest.starred = newInterest.starred;
        
                        if (event) $scope.refreshTooltip(event);
                    });
                });
        };
        
        
        $scope.isWatching = WatchInterestState.isWatching;
        $scope.actionLabels = { ...InterestWording.labels };

        $scope.showWatchingUsers = function () {
            WT1.event("list-interested-users", {type: 'watches'});
            DataikuAPI.interests.listWatchingUsers($scope.objectType, $stateParams.projectKey, $scope.getObjectId(), $stateParams.workspaceKey).success(function(users) {
                const modalScope = $scope.$new();
                modalScope.usersList = users;
                modalScope.icon = "icon-eye-open";
                modalScope.title = users.length + (users.length > 1? " users are " : " user is ") + "watching this object";
                CreateModalFromTemplate("/templates/interested-users.html", modalScope);
            }).error(setErrorInScope.bind($scope));
        };

        $scope.showUsersWithStar = function () {
            WT1.event("list-interested-users", {type: 'stars'});
            DataikuAPI.interests.listUsersWithStar($scope.objectType, $stateParams.projectKey, $scope.getObjectId(), $stateParams.workspaceKey).success(function(users) {
                const modalScope = $scope.$new();
                modalScope.usersList = users;
                modalScope.icon = "icon-star";
                modalScope.title = users.length + " user"+(users.length>1?'s':'')+" starred this object";
                CreateModalFromTemplate("/templates/interested-users.html", modalScope);
            }).error(setErrorInScope.bind($scope));
        };    

        //TODO : hesitated with adding this directly to the tooltip directive... can change
        $scope.refreshTooltip = function(event) {
            const el = angular.element(event.currentTarget);
            el.tooltip('hide')
              .attr('data-original-title', el.attr('title'))
              .tooltip('show');
        };
        
        
    };
});

app.directive('editableSummary', function(DatasetsService, DataikuAPI, $stateParams, $rootScope, TopNav, Dialogs, _SummaryHelper) {
    return {
        templateUrl : '/templates/editable-summary.html',
        scope: {
            object : '=',
            objectType : '@',
            getTags : '=',
            insightMode : '=',
            nameEditable : '=?',
            saveCallback : '=',
            editable : '=?',
            objectInterest : '=?',
            tagColor : '='
        },
        link : function($scope, element, attrs) {
            $scope.appConfig = $rootScope.appConfig;

            if ($scope.nameEditable == undefined) $scope.nameEditable = true;
            $scope.getObjectId = function () {
                if ($scope.objectType == "PROJECT") {
                    return $stateParams.projectKey;
                } else if ($scope.object) {
                    if ($scope.objectType === "DATASET" || $scope.objectType === "DATASET_CONTENT" || $scope.objectType === "RECIPE") {
                        return $scope.object.name;
                    } else {
                        return $scope.object.id;
                    }
                }
                return null;
            };

            _SummaryHelper.addEditBehaviour($scope, element);
            _SummaryHelper.addInterestsManagementBehaviour($scope);

            $scope.image = {};
            $scope.pattern = attrs['pattern'];

            $scope.$watch("object", function (nv, ov) {
                if (!nv) return;
                if ($scope.objectType == "PROJECT" || $scope.objectType == "INSIGHT") {
                    $scope.display_image = true;
                    // $scope.image_src = '/dip/api/image/get-image?size=50x50&projectKey=' + $stateParams.projectKey + '&type=' + $scope.objectType + '&id=' + $scope.getObjectId();
                    $scope.totem = {
                        projectKey: $stateParams.projectKey,
                        objectType: $scope.objectType,
                        id: $scope.getObjectId()
                    }
                } else {
                    $scope.display_image = false;
                }
            });

            $scope.saveCustomFields = function(customFields) {
                $scope.$emit('customFieldsSummaryEdited', customFields);
            };

            $rootScope.$on('customFieldsSaved', function(event, item, newCustomFields) {
                if (TopNav.sameItem(TopNav.getItem(), item)) {
                    $scope.object.customFields = newCustomFields;
                }
            });
        }
    };
});

app.directive('editableProjectSummary', function(
    $rootScope,
    $stateParams,
    AI_EXPLANATION_MODAL_MODES,
    AIExplanationService,
    TopNav,
    _SummaryHelper
) {
    return {
        scope : false,
        link : function($scope, element, attrs) {
            $scope.getObjectId = function() {
                return $scope.object && $scope.object.projectKey;
            };

            _SummaryHelper.addEditBehaviour($scope, element);
            _SummaryHelper.addInterestsManagementBehaviour($scope);

            $scope.$stateParams = $stateParams;
            $scope.currentBranch = "master";

            $scope.$watch("projectSummary", function(nv) {
                $scope.object = nv;
                $scope.objectType = 'PROJECT';
            });

            $scope.$watch("projectCurrentBranch", function(nv) {
                $scope.currentBranch = nv ? nv : "master";
            });

            $scope.saveCustomFields = function(customFields) {
                $scope.$emit('customFieldsSummaryEdited', customFields);
            };

            $rootScope.$on('customFieldsSaved', function(event, item, newCustomFields) {
                if (TopNav.sameItem(TopNav.getItem(), item)) {
                    $scope.object.customFields = newCustomFields;
                }
            });

            $scope.generateDescription = function() {
                $scope.generateDescriptionForObject().then(function(description) {
                    $scope.object.description = description;
                    $scope.$emit("objectSummaryEdited");
                });
            };
        }
    };
});

app.controller('CustomFieldsEditModalController', function($scope, $rootScope, PluginConfigUtils) {
    let customFieldsMapFlattenList = [];
    $scope.uiState = { cfComponentIdx: 0 };

    function populateCustomFields() {
        $scope.uiState.customFields = angular.copy($scope.objectCustomFields);
        PluginConfigUtils.setDefaultValues(customFieldsMapFlattenList, $scope.uiState.customFields);
        $scope.uiState.cfComponentIdx = $scope.editingTabIndex === undefined || $scope.editingTabIndex === null ? 0 : $scope.editingTabIndex;
    }

    $scope.$watch('objectType', function() {
        if ($scope.objectType) {
            $scope.customFieldsMap = $rootScope.appConfig.customFieldsMap[$scope.objectType];
            customFieldsMapFlattenList = [];
            $scope.customFieldsMap.forEach(ref => customFieldsMapFlattenList = customFieldsMapFlattenList.concat(ref.customFields));
            if ($scope.objectCustomFields) {
                populateCustomFields();
            }
        }
    });
    $scope.$watch('objectCustomFields', populateCustomFields);
    $scope.save = function() {
        $scope.resolveModal($scope.uiState.customFields);
    };

    populateCustomFields();
});

app.directive('customFieldsPopup', function() {
    return {
        templateUrl: '/templates/taggable-objects/custom-fields-popup.html',
        scope: {
            customFields: '=',
            customFieldsMap: '='
        }
    };
});

app.directive('customFieldsEditForm', function($rootScope) {
    return {
        templateUrl : '/templates/taggable-objects/custom-fields-edit-form.html',
        scope: {
            customFields: '=',
            objectType: '=',
            componentIndex: '='
        },
        link : function($scope, element, attrs) {
            $scope.$watch('componentIndex', function() {
                if ($scope.componentIndex >= 0) {
                    $scope.customFieldsMap = [$rootScope.appConfig.customFieldsMap[$scope.objectType][$scope.componentIndex]];
                } else {
                    $scope.customFieldsMap = $rootScope.appConfig.customFieldsMap[$scope.objectType];
                }
            });
        }
    };
});

// TO DO : Delete this directive when the Summary tab of all DSS objects are moved to the standardizedSidePanel directive
app.directive('customFieldsInSummary', function($rootScope, Logger, PluginConfigUtils) {
    return {
        templateUrl : '/templates/taggable-objects/custom-fields-summary.html',
        scope: {
            customFields: '=',
            objectType: '=',
            saveFn: '='
        },
        link : function($scope, element, attrs) {
            $scope.customFieldsMap = $rootScope.appConfig.customFieldsMap[$scope.objectType];
            let customFieldsMapFlattenList = [];
            $scope.customFieldsMap.forEach(ref => customFieldsMapFlattenList = customFieldsMapFlattenList.concat(ref.customFields));
            $scope.ui = {
                customFields: angular.copy($scope.customFields),
                editing: false
            };
            PluginConfigUtils.setDefaultValues(customFieldsMapFlattenList, $scope.ui.customFields);
            $scope.editCF = function() {
                if (!$scope.saveFn) {
                    Logger.warn("There is no save function attached to the custom fields editable summary");
                    $scope.ui.editing = false;
                } else {
                    $scope.ui.editing = true;
                }
            };
            $scope.discardCF = function() {
                if (!$scope.saveFn) {
                    Logger.warn("There is no save function attached to the custom fields editable summary");
                    $scope.ui.editing = false;
                } else {
                    $scope.ui.customFields = angular.copy($scope.customFields);
                    PluginConfigUtils.setDefaultValues(customFieldsMapFlattenList, $scope.ui.customFields);
                    $scope.ui.editing = false;
                }
            };
            $scope.saveCF = function() {
                if (!$scope.saveFn) {
                    Logger.warn("There is no save function attached to the custom fields editable summary");
                    $scope.ui.editing = false;
                } else {
                    $scope.saveFn(angular.copy($scope.ui.customFields));
                    $scope.ui.editing = false;
                }
            };
            $scope.$watch('customFields', function() {
                $scope.ui.customFields = angular.copy($scope.customFields);
                PluginConfigUtils.setDefaultValues(customFieldsMapFlattenList, $scope.ui.customFields);
            });
        }
    };
});

app.directive('customFieldsInSidePanel', function($rootScope, Logger, PluginConfigUtils, TopNav) {
    return {
        templateUrl : '/templates/taggable-objects/custom-fields-sidepanel.html',
        scope: {
            customFields: '=',
            objectType: '=',
            saveFn: '=',
            editCustomFields: '=',
            editable: '=',
        },
        link : function($scope, element, attrs) {
            $scope.customFieldsMap = $rootScope.appConfig.customFieldsMap[$scope.objectType];
            let customFieldsMapFlattenList = [];
            $scope.customFieldsMap.forEach(ref => customFieldsMapFlattenList = customFieldsMapFlattenList.concat(ref.customFields));
            $scope.ui = {
                customFields: angular.copy($scope.customFields),
            };
            PluginConfigUtils.setDefaultValues(customFieldsMapFlattenList, $scope.ui.customFields);

            $scope.canWriteProject = () => $scope.editable;

            $scope.editCF = function() {
                if (!$scope.saveFn) {
                    Logger.warn("There is no save function attached to the custom fields editable summary");
                }
            };
            $scope.discardCF = function() {
                if (!$scope.saveFn) {
                    Logger.warn("There is no save function attached to the custom fields editable summary");
                } else {
                    $scope.ui.customFields = angular.copy($scope.customFields);
                    PluginConfigUtils.setDefaultValues(customFieldsMapFlattenList, $scope.ui.customFields);
                }
            };
            $scope.saveCF = function() {
                if (!$scope.saveFn) {
                    Logger.warn("There is no save function attached to the custom fields editable summary");
                } else {
                    $scope.saveFn(angular.copy($scope.ui.customFields));
                }
            };
            $scope.$watch('customFields', function() {
                $scope.ui.customFields = angular.copy($scope.customFields);
                PluginConfigUtils.setDefaultValues(customFieldsMapFlattenList, $scope.ui.customFields);
            });

            $rootScope.$on('customFieldsSaved', function(event, item, newCustomFields) {
                if (TopNav.sameItem(TopNav.getItem(), item)) {
                    $scope.ui.customFields = newCustomFields;
                }
            });
        }
    };
});

})();

;
(function() {
'use strict';

const app = angular.module('dataiku.taggableobjects');


app.service('TaggingService', function($rootScope, $stateParams, $timeout, $q, DataikuAPI, CreateModalFromTemplate, TaggableObjectsUtils) {
    const svc = this;
    let projectTags = {};
    let projectTagsList = undefined;
    let projectTagsUsageMap = undefined;
    let globalTagsCategories = undefined;
    let globalTags = {};


    var setList = function (tagsObj) {
        projectTagsList = [];

        Object.keys(tagsObj).forEach(function(tagTitle, index) {
            projectTagsList.push({'title': tagTitle,  color: tagsObj[tagTitle].color});
        });
        return svc.sortTagList(projectTagsList);
    }

    this.getTagWithUsage = function (tag, items) {
        tag.usage=0;
        items.forEach(i => {if (i.tags && i.tags.indexOf(tag.title)>=0) tag.usage++;});
        tag.initialState = tag.newState = (tag.usage==0 ? 0 : (tag.usage==items.length ? 2 : 1));
        return tag;
    }

    this.sortTagList = function(list) {
        return list.sort((a,b) => a.title.localeCompare(b.title));
    }

    this.getGlobalTags = function(objectType) {
        if (!objectType || objectType === 'TAGGABLE_OBJECT') {
            return globalTags;
        }
        const globalTagsForType = {};
        Object.keys(globalTags).forEach((key) => {
            if (svc.shouldGlobalTagApply(globalTags[key].appliesTo, objectType)) {
                globalTagsForType[key] = globalTags[key];
            }
        });
        return globalTagsForType;
    }

    const fetchGlobalTagsDeferred = $q.defer();
    this.fetchGlobalTags = function(forceFetch) {
        if (globalTagsCategories && !forceFetch) {
            return fetchGlobalTagsDeferred.promise;
        }
        globalTagsCategories = globalTagsCategories || {};
        DataikuAPI.globalTags.getGlobalTagsInfo().success(function(data) {
            Object.keys(data.globalTags).forEach(function(k) {
                let category = data.globalTags[k].globalTagsCategory;
                Object.assign(data.globalTags[k], {appliesTo : data.globalTagsCategories[category]});
            });
            globalTagsCategories = data.globalTagsCategories;
            globalTags = data.globalTags;
            fetchGlobalTagsDeferred.resolve()
        }).error(setErrorInScope.bind($rootScope));
        return fetchGlobalTagsDeferred.promise;
    }

    this.setProjectTags = function(tags) {
        projectTags = fillTagsMap(tags);
        projectTagsList = undefined;
    };

    this.getProjectTags = function() {
        return projectTags;
    };

    this.getProjectTagsList = function(objectType) {
        if (projectTagsList==undefined) setList(Object.assign(this.getProjectTags(), globalTags));
        return projectTagsList;
    }

    this.getProjectTagsUsageMap = function () {
        const deferred = $q.defer();
        DataikuAPI.taggableObjects.listTagsUsage($stateParams.projectKey, {}, "nospinner").success(function(data) {
            projectTagsUsageMap = data;
            deferred.resolve(projectTagsUsageMap);
        });
        return deferred.promise;
    }

    this.startApplyTagging = function(selectedItems) {
        return CreateModalFromTemplate('/templates/apply-tags-modal.html', $rootScope, 'ApplyTaggingController', function(modalScope) {
            modalScope.selectedItems = selectedItems;
            modalScope.itemsType = TaggableObjectsUtils.getCommonType(selectedItems, it => it.type);
            modalScope.tagsSorted = svc.getTagsSorted([{...projectTags, ...globalTags}], t => svc.getTagWithUsage(t, selectedItems), modalScope.itemsType);
        });
    };

    this.applyTagging = function(request) {
        return DataikuAPI.taggableObjects.applyTagging($stateParams.projectKey, request)
            .success(function() {
                $rootScope.$broadcast('taggableObjectTagsChanged');
            }).error(setErrorInScope.bind($rootScope));
    };

    this.getTagColor = function(tag) {
        return globalTags[tag] ? globalTags[tag].color : projectTags[tag] ? projectTags[tag].color : svc.getDefaultColor(tag);
    };

    this.getGlobalTagCategory = function(tag, objectType) {
        if (globalTags[tag]) {
            let category = globalTags[tag].globalTagsCategory;
            if (objectType == "TAGGABLE_OBJECT" || svc.shouldGlobalTagApply(globalTagsCategories[category],objectType)) {
                return category;
            }
            return false;
        }
        return  null;
    };

    var pushTagsToList = function (tags, list) {
        Object.keys(tags).forEach(function(tagTitle, index) {
            const t = tags[tagTitle];
            const titleLower = tagTitle.toLowerCase();
            list.push({tag: t, title: tagTitle, titleLower: titleLower});
        });
    }

    this.shouldGlobalTagApply = function(appliesTo, objectType) {
        return objectType === "TAGGABLE_OBJECT" || appliesTo.includes(objectType);
    };

    this.getTagsSorted = function(tagLists, fTagMapper, itemsType) {
        const list = [];

        tagLists.forEach(l => pushTagsToList(l, list));

        const titleLookup = {};
        const filteredList = list.filter(it => (!it.tag.appliesTo || !itemsType || (itemsType && svc.shouldGlobalTagApply(it.tag.appliesTo, itemsType))));
        //sort global tags last and if both are global tags, sort by localeCompare
        let sortedList = filteredList.sort((a,b) => (!!a.tag.globalTagsCategory - !!b.tag.globalTagsCategory) || a.title.localeCompare(b.title));
        sortedList.forEach((item, index) => {
            titleLookup[item.titleLower] = index;
        });
        if (fTagMapper) sortedList = sortedList.map(fTagMapper)
        return {list: sortedList, titleLookup: titleLookup}
    }

    this.applyManageTagsChangesToTagList = function (tagsList, newAllTags, keepOrphans) { //rename any tags in-edit
        const newTags = [];
        tagsList.forEach( tag => {
            if (newAllTags[tag]) {
                if (newAllTags[tag].updatedTagName) {
                    newTags.push(newAllTags[tag].updatedTagName);
                }
                else {
                    newTags.push(tag);
                }
            }
            else if (keepOrphans) {
                newTags.push(tag);
            }
        })
        return newTags;
    }

    const COLORS = [
        "#1ac2ab",
        "#0f6d82",
        "#FFD83D",
        "#de1ea5",
        "#90a8b7",
        "#28aadd",
        "#00a55a",
        "#94be8e",
        "#d66b9b",
        "#77bec2",
        "#123883",
        "#a088bd"
    ];

    this.getDefaultColor = function(tag) {
        const hash = tag.split('').reduce(function(a,b) {a=((a<<5)-a)+b.charCodeAt(0);return a&a;},0); //NOSONAR a&a basically means floor, who cares if it is unclear here?
        return COLORS[Math.abs(hash) % 12];
    }

    function fillTagsMap(tags) {
        for (let tag in tags) {
            if (globalTags[tag]) { //update projectTags with globalTags informations
                tags[tag] = globalTags[tag];
            }
            if (!tags[tag].color) {
                tags[tag].color = svc.getDefaultColor(tag);
            }
            const col = d3.rgb(tags[tag].color);
            tags[tag].fadedColor = 'rgba('+col.r+','+col.g+','+col.b+', 0.35)';
        }
        return tags;
    }

    this.fillTagsMapFromArray = function(a) {
        const tagMap = {}
        a.map((tag) => tagMap[tag] = {});
        return fillTagsMap(tagMap);
    }

    function  argsForUpdateBcast(refreshFlowFilters, checkFilterQuery, updateGraphTags) {
        const args = {}
        args.refreshFlowFilters = refreshFlowFilters;
        args.checkFilterQuery = checkFilterQuery;
        args.updateGraphTags = updateGraphTags;
        return args;
    }

    this.bcastTagUpdate = function (minReloadReqd, updateGraphTags) {
        $rootScope.$broadcast('projectTagsUpdated', argsForUpdateBcast(!minReloadReqd, minReloadReqd, updateGraphTags));
    }

    this.update = function(minReloadReqd) {
        return DataikuAPI.taggableObjects.listTags($stateParams.projectKey).success(function(response) {
            svc.setProjectTags(response.tags);
            svc.bcastTagUpdate(minReloadReqd, false);
        });
    };

    this.saveToBackend = function(newTags) {
        return DataikuAPI.taggableObjects.setTags($stateParams.projectKey, newTags).then(function(response) {
            svc.setProjectTags(response.tags);
        });
    };
});


app.filter('tagToColor', function(TaggingService) {
    return function(tag) {
        return TaggingService.getTagColor(tag);
    }
});

const TRISTATES = {
    OFF: 0,
    PART: 1,
    ON: 2,
    ROTATE: 3
}

app.controller('ApplyTaggingController', function($scope, WT1, TaggingService, translate, CreateModalFromTemplate, $timeout) {
    /*  expect initialisation via TaggingService.startApplyTagging:
     *  scope.selectedItems, scope.itemsType, scope.tagsSorted
    */

    var pushOperation = function(operations, allTags, mode) {
        const reqdState = (mode=='ADD' ? TRISTATES.ON : TRISTATES.OFF);
        const tags = allTags.filter(t => t.newState!=t.initialState && t.newState==reqdState).map(t => t.title);
        if (tags.length > 0) {
            WT1.event("tagging-apply", {tags: tags.length, elements: $scope.selectedItems.length, add: (mode=='ADD')});
            operations.push({tags: tags, mode: mode});
        }
    }

    $scope.uiState = {newTag: ""};
    $scope.translate = translate;

    $scope.doSave

    // actually run the apply tagging api call. may be overridden in some contexts (it is for mass tagging projects)
    // whether this default is fine for a given object type depends on whether the backend taggable service supports saving this type of object
    $scope.applyTagging = (request) => {
        return TaggingService.applyTagging(request)
            .then($scope.resolveModal)
            .catch(setErrorInScope.bind($scope));
    }

    // may be overridden where it doesn't make sense
    $scope.showManageTagsInProjectSettingsMessage = true;

    $scope.save = function() {
        const request = {elements: $scope.selectedItems, operations: [] };
        ['ADD', 'REMOVE'].forEach(mode => pushOperation(request.operations, $scope.tagsSorted.list, mode));

        if (request.operations.length > 0) {
            $scope.applyTagging(request);
        }
        else {
            $scope.resolveModal();
        }

    }

    $scope.isChanged = function(tag) {
        return (tag.initialState != tag.newState);
    }

    $scope.createTagOnEnter = function(event) {
        if (event.which === 13) { // create tag when enter key is pressed
            $scope.onAddTag();
        }
    }

    $scope.usageText = function (tag) {
        switch (tag.newState) {
            case TRISTATES.OFF: return "-";
            case TRISTATES.PART: return tag.usage + "/" + $scope.selectedItems.length;
            case TRISTATES.ON: return $scope.selectedItems.length + "/" + $scope.selectedItems.length;
            default:
                break;
        }
    }

    $scope.hasTags = function() {
        return $scope.tagsSorted.filteredList.length > 0;
    }

    const addNewTagToList = function (title, newState, color) {
        $scope.tagsSorted.list.push({
            title: title,
            tag: {color: color ? color : TaggingService.getDefaultColor(title)},
            initialState: TRISTATES.OFF,
            newState: newState});

        $scope.tagsSorted.titleLookup[title.toLowerCase()] = "new";
    }

    const isNewlyAdded = function (title) {
        return $scope.tagsSorted.titleLookup[title.toLowerCase()] == "new"
    }

    $scope.onAddTag = function(){
        if ($scope.canCreateTag()) {
            addNewTagToList($scope.uiState.newTag, TRISTATES.ON);
            TaggingService.sortTagList($scope.tagsSorted.list);
            $scope.uiState.newTag = "";
        }
    };

    function filterTagListByInput(input = $scope.uiState.newTag) {
        if (!input) return $scope.tagsSorted.list;
        return $scope.tagsSorted.list.filter(tag => tag.title.toLowerCase().startsWith(input.toLowerCase()));
    };

    $scope.$watch('uiState.newTag', (nv) => {
        TaggingService.sortTagList($scope.tagsSorted.list);
        $scope.tagsSorted.filteredList = filterTagListByInput(nv);
    });

    $scope.canCreateTag = function() {
        const titleLower = $scope.uiState.newTag.toLowerCase();
        return titleLower && !$scope.tagsSorted.titleLookup.hasOwnProperty(titleLower);
    };

    $scope.rotateCheckbox = function(item) {
        item.newState = TRISTATES.ROTATE;
    }
});

app.directive('checkboxTristate', function () {
    return {
        template: `<input type="checkbox" name="{{id}}" id="{{id}}" ng-click="rotate()" aria-label="{{alabel}}" > `,
        scope: {
            triState: "=ngModel",  // 0=off 1=indeterminate 2=checked
            initialState: "=",
            id: "=",
            alabel: "="
        },
        link: function (scope, $element) {
            const chkbx = $element.children()[0];

            scope.onSet = function (newState) {
                scope.triState = newState;
                switch (newState) {
                    case TRISTATES.PART:  {
                        chkbx.checked = false;
                        chkbx.indeterminate = true;
                        break;
                    }
                    default: {
                        chkbx.checked = !!newState;
                        chkbx.indeterminate = false;
                        break;
                    }

                }
            }

            var incrementValue = function (v) {
                let newState = (v+1) % 3;
                if (newState==TRISTATES.PART && scope.initialState!=TRISTATES.PART) newState = TRISTATES.ON; //you can only return to indeterminate state
                return newState;
            }
            scope.rotate = function (event) {
                const newState = incrementValue(scope.triState);
                scope.onSet(newState);
            }

            scope.$watch("triState", function(nv, ov) {
                if (nv==TRISTATES.ROTATE) {
                    nv = incrementValue(ov);
                }
                scope.onSet(nv);
            });

            scope.onSet(scope.triState);
        }
    }
});

app.directive('addTagInput', function() {
    return {
        template: `<form class="common-styles-only tag-form noflex horizontal-flex" >
                        <input class="flex" type="text"
                        ng-model="newTag"
                        ng-class="{'has-error': !validator()}"
                        placeholder="Create new tag"
                        ng-keydown="onAddKeydown($event)"
                        aria-label="Create new tag" />

                    <button type="button"
                        class="btn btn--primary tags-settings-btn noflex"
                        ng-disabled="uiState.newTag.length == 0 || !validator()"
                        ng-click="addTag($event)">Add</button>
                </form>`,
        scope:   {
            newTag: '=ngModel',
            validator: '&',
            onAddTag: '&'
        },
        link: function (scope, $element){

            function eatEvent(e) {
                e.stopPropagation();
                e.preventDefault();
            }
            scope.onAddKeydown = function (e) {
                if (e.keyCode == 13) { // enter
                    scope.onAddTag();
                    eatEvent(e);
                }
            };

            scope.addTag = function(e) {
                if (scope.validator()) scope.onAddTag();
                eatEvent(e);
            }
        }
    }
});

app.directive('tagsList', function(translate, Assert, CreateModalFromTemplate, TaggingService) {
    return {
        template: `<div class="tagsList">
            <ul class="tags vertical-flex">
                <li ng-repeat="tag in tags">
                    <span ng-if="isTagSelected(tag)" class="tag selected" ng-click="addTag(tag)" style="color:white; background-color:#{{tag.color.substring(1)}}" >
                        <span class="bullet" style="background-color:white;"> </span>
                        <span ui-global-tag="tag.title" object-type="objectType"/>
                    </span>
                    <span ng-if="!isTagSelected(tag)" class="tag" ng-click="addTag(tag)">
                        <span class="bullet" style="background-color:{{tag.color}};"> </span>
                        <span ui-global-tag="tag.title" object-type="objectType"/>
                    </span>
                </li>
                <li ng-if="noTagAvailable()">
                    <span class="tag disabled" translate="GLOBAL.TAGS.NO_TAGS_AVAILABLE">No tags available</span>
                </li>
            </ul>
            <button class="btn btn--contained btn--tag-list" ng-click="manageTags()" ng-if="$parent.canWriteProject()" translate="GLOBAL.TAGS.MANAGE_TAGS">Manage tags</button>
        </div>`,
        scope: {
            selected: '=tagsListSelected',
            objectType: '='
        },
        link: function(scope, element) {
            scope.tags = TaggingService.getProjectTagsList(scope.objectType);
            Assert.trueish(scope.tags, 'no tags list');

            scope.noTagAvailable = function() {
                return scope.tags.length == 0;
            };

            scope.$on('projectTagsUpdated', () => {
                scope.tags = TaggingService.getProjectTagsList(scope.objectType);
            });

            scope.isTagSelected = function(tag) {
                return scope.selected.indexOf(tag.title) > -1;
            }

            scope.manageTags = function() {
                CreateModalFromTemplate("/templates/widgets/edit-tags-modal.html", scope, null, function(modalScope) {
                    modalScope.translate = translate;
                    modalScope.tagsDirty = angular.copy(TaggingService.getProjectTags());

                    modalScope.save = function() {
                        scope.selected = TaggingService.applyManageTagsChangesToTagList(scope.selected, modalScope.tagsDirty);
                        TaggingService.saveToBackend(modalScope.tagsDirty)
                            .then(modalScope.resolveModal)
                            .catch(setErrorInScope.bind(scope));
                    };
                    modalScope.cancel = function() {modalScope.dismiss();};
                });
            };

            scope.addTag = function(tag) {
                scope.$emit('tagSelectedInList', tag.title);
            };
        }
    };
});


app.directive('tagsListEditor', function($timeout, TaggingService, translate){
    return {
        scope: {
            tags: '=tagsListEditor'
        },
        replace: true,

        template: `<div class="tags-settings vertical-flex h100" style="position:relative" >
        <div class="tag-edit-filter" ng-class="{'tag-edit-filter--focus': filterFocused}">
            <i class="icon-dku-search"></i>
            <input ng-model="uiState.newTag" name="tagEditorInput" type="search" ng-keydown="createTagOnEnter($event)" auto-focus="true" tabindex="0"
                ng-focus="filterFocused = true" ng-blur="filterFocused = false" autocomplete="off"
                placeholder="{{::translate('GLOBAL.TAGS.FILTER_OR_CREATE_TAGS', 'Filter tags or create a new tag')}}" class="tag-edit-filter__input" aria-label="Selected tags"/>
        </div>
        <div ng-if="hasTag()" class="tag-help-text">
            <span class="tag-help-text__name" translate="GLOBAL.TAGS.TAG_NAME">Tag name</span>
            <span ng-if="totalObjects > 0" class="tag-help-text__usage" translate="GLOBAL.TAGS.TAG_USAGE">Tag usage</span>
        </div>
        <div class="tags">
            <editable-list ng-if="hasTag()" ng-model="tagsSorted.list" class="tags" disable-create-on-enter="true" skip-to-next-focusable="true"
                transcope="{ uiState: uiState, canRenameTag: canRenameTag, updateTag: updateTag, onRemoveTag: onRemoveTag, onRestoreTag: onRestoreTag}"
                disable-remove="true" disable-add="true" full-width-list="true" has-divider="false">

                <div class="tag-row tag-row--editable-list horizontal-flex"
                    scroll-to-me="{{uiState.scrollToTagIdx==$index}}">

                    <div class="tag-row-item flex horizontal-flex">
                        <span class="tag-color noflex" style="background-color:{{it.tag.color}};" colorpicker colorpicker-with-input="true" ng-model="it.tag.color" aria-role="button" aria-label="Tag color">
                            <i class="icon-tint"></i>
                        </span>
                        <span ng-if="!it.tag.globalTagsCategory" class="tag-edit flex common-styles-only horizontal-flex">
                            <editable-list-input type="text" class="tag-input flex" ng-init="it.updatedTagName = it.tag.updatedTagName || it.title" ng-model="it.updatedTagName" on-key-up-callback="updateTag(it)" aria-label="Tag name" required="true" unique="true"/>
                        </span>
                        <span ng-if="it.tag.globalTagsCategory" class="tag-title flex">
                            <span ui-global-tag="it.tag.updatedTagName || it.title" object-type="'TAGGABLE_OBJECT'"></span>
                        </span>
                        <span ng-show="$parent.$parent.totalObjects > 0" class="tag-usage noflex">{{$parent.$parent.tagsUsage[it.title] || '-'}}</span>
                    </div>
                    <button ng-if="!it.tag.globalTagsCategory" class="noflex btn btn--text btn--danger btn--dku-icon btn--icon editable-list__delete m0" ng-click="onRemoveTag($event, it.title)"><i class="dku-icon-trash-16"></i></button>
                    <div ng-if="it.tag.globalTagsCategory" class="editable-list__icon">
                        <i class="noflex dku-icon-info-circle-fill-16" aria-role="button" aria-label="Restore tag: {{it.title}}"></i>
                    </div>
                </div>
            </editable-list>
            <div ng-if="canCreateTag()" class="tags">
                <div ng-if="canCreateTag()" class="tag-row horizontal-flex tags-settings__create" ng-click="onAddNewTag(e)" ng-keyup="$event.keyCode == 13 && onAddNewTag(e)" tabindex="0">
                    <i class="icon-plus flex-no-grow"></i>
                    <span class="flex" translate="GLOBAL.TAGS.CREATE_TAG" translate-values="{name:uiState.newTag}">Create &laquo;{{uiState.newTag}}&raquo;</span>
                    <code class="dku-tiny-text-sb text-weak tags-settings__create-shortcut" translate="GLOBAL.TAGS.CREATE_TAG.ENTER">Enter</code>
                    <span class="return flex-no-grow tags-settings-btn mright4">&crarr;</span>
                </div>
            </div>
        </div>
        <div ng-if="!hasTag() && !canCreateTag()" class="noflex no-tag-yet"><p translate="GLOBAL.TAGS.NO_TAGS_AVAILABLE">No tags available</p></div>

    </div>`,

        link: function(scope, element, attrs){
            scope.uiState = {
                originalTagName: "",
                updatedTagName: "",
                editTagIdx : undefined,
                newTag: "",
                scrollToTagIdx : undefined
            };

            scope.totalObjects = 0;
            scope.translate = translate;

            var ui = scope.uiState;

            var eatEvent = function(e) {
                if (e) {
                    e.stopPropagation();
                    e.preventDefault();
                }
            };

            scope.updateTag = function(item) {
                if (item.title === item.updatedTagName && !item.isEdited) {
                    return;
                }
                ui.originalTagName = item.title;
                ui.updatedTagName = item.updatedTagName || "";
                const tag = scope.tags[ui.originalTagName];
                tag.updatedTagName = ui.updatedTagName;
                tag.isEdited = true;
            }

            scope.createTagOnEnter = function(event) {
                if (event.which === 13) { // create tag when enter key is pressed
                    scope.onAddNewTag(event);
                }
            };

            scope.$watch('uiState.newTag', updateSortedTags);

            function filterTagsByInput(input) {
                const allTags = Object.assign({}, scope.tags, scope.globalTags);
                if (!input) {
                    return allTags;
                }
                const filteredTags = {};
                if (allTags) {
                    Object.keys(allTags).forEach((tagTitle) => {
                        const lowerTitle = tagTitle.toLowerCase();
                        if (lowerTitle.startsWith(input)) {
                            filteredTags[tagTitle] = allTags[tagTitle];
                        }
                    });
                }
                return filteredTags;
            }

            function updateSortedTags() {
                const input = ui.newTag.toLowerCase();
                const filteredTags = filterTagsByInput(input);
                scope.tagsSorted = TaggingService.getTagsSorted([filteredTags]);
            }

            var calcTagUsageForCurrentProject = function() {
                scope.tagsUsage = {};
                TaggingService.getProjectTagsUsageMap().then(usageMap => {
                    const tagsUsage = {};
                    scope.totalObjects = 0;
                    Object.keys(usageMap).forEach((objName) => {
                        scope.totalObjects++;
                        const tags = usageMap[objName];
                        tags.forEach(tagName => {
                            if (!tagsUsage[tagName]) tagsUsage[tagName]=0;
                            tagsUsage[tagName]++;
                        });
                    });
                    Object.keys(scope.tags).forEach(function(t) {
                       if (isNaN(tagsUsage[t])) tagsUsage[t] = 0;
                    });
                    scope.tagsUsage = angular.copy(tagsUsage);
                });
            };

            scope.$watch("tags", function(nv, ov) {
                if (!angular.equals(nv, ov)) {
                    getGlobalTags();
                    updateSortedTags();
                    calcTagUsageForCurrentProject();
                }
            });

            function getGlobalTags() {
                scope.globalTags = TaggingService.getGlobalTags();
                scope.$parent.hasGlobalTags = Object.keys(scope.globalTags).length > 0;
            };
            getGlobalTags();

            scope.onAddNewTag = function(e) {
                if (scope.canCreateTag()) {
                    scope.tags[ui.newTag] = {color: TaggingService.getDefaultColor(ui.newTag), usage:0, isNew: true};
                    updateSortedTags();
                    ui.scrollToTagIdx = scope.tagsSorted.titleLookup[ui.newTag.toLowerCase()];
                    ui.newTag = "";
                }
                updateSortedTags();
                eatEvent(e);
            };

            scope.canCreateTag = function() {
                const titleLower = ui.newTag.toLowerCase();
                return titleLower && !scope.tagsSorted.titleLookup.hasOwnProperty(titleLower);
            };

            scope.onRemoveTag = function(e, tag) {
                ui.updatedTagName = "";
                delete scope.tags[tag];
                updateSortedTags();
                eatEvent(e);
            };

            scope.canRenameTag = function() {
                const toLower = ui.updatedTagName.toLowerCase();
                if (ui.originalTagName.toLowerCase() == toLower) return true;
                return toLower && !scope.tagsSorted.titleLookup.hasOwnProperty(toLower);
            };

            scope.hasTag = function() {
                return !$.isEmptyObject(scope.tags) || !$.isEmptyObject(scope.globalTags);
            };

            updateSortedTags();
            calcTagUsageForCurrentProject();
            element.find('.tag-edit-filter__input').focus();
        }
    };
});

})();

;
(function() {
'use strict';

const app = angular.module('dataiku.collab.timeline', []);


app.directive("objectTimelineWithPost", function(DataikuAPI) {
    return {
        templateUrl: '/templates/widgets/object-timeline-with-post.html',
        restrict: "A",
        scope: {
            objectType: '=',
            projectKey: '=',
            objectId: '=',
            initialTimeline: '=',
            initialFetch: '@',
            fetchTimelinePromiseFn: '=?'
        },
        link: function($scope) {
            $scope.refreshTimeline = function () {
                if ($scope.fetchTimelinePromiseFn) {
                    $scope.fetchTimelinePromiseFn().success(function (data) {
                        $scope.timeline = data;
                    }).error(setErrorInScope.bind($scope));
                } else {
                    DataikuAPI.timelines.getForObject($scope.projectKey, $scope.objectType, $scope.objectId).success(function (data) {
                        $scope.timeline = data;
                    }).error(setErrorInScope.bind($scope));
                }
            };
            $scope.$watch("initialTimeline", function(nv, ov) {
                if (nv) $scope.timeline = $scope.initialTimeline;
            });
            $scope.$watch("objectId", (nv, ov) => {
                if (nv === ov) return;
                $scope.refreshTimeline();
            });
            if ($scope.initialFetch && $scope.objectId) {
                $scope.refreshTimeline();
            }
        }
    }
});


app.service("TimelineItemUtils", function($state) {
    const svc = this;
    this.isAboutTags = function(evt) {
        return evt.details.addedTags || evt.details.removedTags;
    };
    this.isAboutTasks = function(evt) {
        return evt.details.totalTasks != null;
    };
    this.isAboutDescriptions = function(evt) {
        return evt.details.descriptionEdited || evt.details.shortDescEdited;
    };
    this.isAboutTagsOnly = function(evt) {
        return svc.isAboutTags(evt) && !svc.isAboutTasks(evt) && !svc.isAboutDescriptions(evt);
    };
    this.isAboutTasksOnly = function(evt) {
        return !svc.isAboutTags(evt) && svc.isAboutTasks(evt) && !svc.isAboutDescriptions(evt);
    };
    this.isMoreComplex = function(evt) {
        return !svc.isAboutTasksOnly(evt) && !svc.isAboutTagsOnly(evt)
    };
    this.goToProjectRevisionPage = function(evt) {
        $state.go('projects.project.version-control', {commitId: evt.details.revision, branch: evt.details.branchName});
    }
    this.goToMergeRequest = function(evt) {
        $state.go('projects.project.version-control-merge', {mergeRequestId: evt.details.request_id});
    }
});


app.directive('timeline', function($filter, $state) {
    return {
        templateUrl: '/templates/timeline.html',
        scope: {
            timeline: '=timeline',
            context: '@',
            reverse: '@',
            hook: '='
        },
        link: function(scope, element) {
            function humanReadableObjectType (objectType) {
                if (!objectType) return;
                switch(objectType) {
                case "MANAGED_FOLDER":
                    return "folder";
                case "SAVED_MODEL":
                    return "model";
                case "MODEL_EVALUATION_STORE":
                    return "evaluation store";
                case "LAMBDA_SERVICE":
                    return "API service";
                default:
                    return objectType.toLowerCase().replace('_', ' ');
                }
            }

            let maxItems = 15;

            function update() {
                if (!scope.timeline) {return}
                angular.forEach(scope.timeline.items, function(item) {
                    item.day = $filter('friendlyDate')(item.time);
                    item.humanReadableObjectType = scope.hook && scope.hook.humanReadableObjectType ? scope.hook.humanReadableObjectType(humanReadableObjectType)(item.objectType) : humanReadableObjectType(item.objectType);
                    item.details = item.details || {};
                    item.details.objectDisplayName = item.details.objectDisplayName || item.objectId;
                });

                const displayedItems = scope.timeline.items ? scope.timeline.items.slice() : [];

                scope.orderedItems = $filter('orderBy')(displayedItems, (scope.reverse?'-time':'time'));
                scope.orderedItems = scope.orderedItems.slice(0, maxItems);

                /* Insert days separators */
                scope.orderedItemsWithDays = [];
                scope.orderedItems.forEach(function(x, i) {
                    if (i === 0) {
                        scope.orderedItemsWithDays.push({isSeparator: true, day : x.day});
                        scope.orderedItemsWithDays.push(x);
                    } else if (x.day === scope.orderedItems[i-1].day) {
                        scope.orderedItemsWithDays.push(x);
                    } else {
                        scope.orderedItemsWithDays.push({isSeparator: true, day : x.day});
                        scope.orderedItemsWithDays.push(x);
                    }
                });
            }

            scope.$state = $state;
            scope.$watch('timeline', function(nv, ov) {
                if(nv) update();
            });

            scope.scroll = function() {
                maxItems += 20;
                update();
            }
        }
    };
});


app.directive('timelineItem', function(TimelineItemUtils) {
    return {
        templateUrl: '/templates/timeline-item.html',
        link: function(scope) {
            scope.TimelineItemUtils = TimelineItemUtils;
        }
    };
});


app.directive('discussionMentionItem', function() {
    return {
        templateUrl: '/templates/discussion-mention-item.html',
    };
});

app.directive('discussionReplyItem', function() {
    return {
        templateUrl: '/templates/discussion-reply-item.html',
    };
});

app.directive('discussionCloseItem', function() {
    return {
        templateUrl: '/templates/discussion-close-item.html',
    };
});

app.directive('commitMentionItem', function() {
    return {
        templateUrl: '/templates/commit-mention-item.html',
    };
});

app.directive('requestItem', function() {
    return {
        templateUrl: '/templates/request-item.html',
    };
});

app.directive('requestGrantedItem', function() {
    return {
        templateUrl: '/templates/request-granted-item.html',
    };
});

app.directive('timelineTaskItem', function(DataikuAPI) {
    return {
        templateUrl: '/templates/task-notification-item.html',
        link: function(scope) {
            scope.downloadExport = function(exportId) {
                downloadURL(DataikuAPI.exports.getDownloadURL(exportId));
            };
        }
    };
});


})();
;
(function() {
'use strict';

const app = angular.module('dataiku.collab.discussions', []);


app.controller('DiscussionsWidgetController', function($scope, $state, $location, $stateParams, $rootScope, $timeout, $filter, Assert, DataikuAPI, WT1, TopNav, Notification, Dialogs, ActivityIndicator, Debounce) {
    let currentItem;

    $scope.appConfig = $rootScope.appConfig;
    $scope.uiState = {};
    $scope.discussionsContext = $stateParams.workspaceKey ? 'workspace' : 'project';

    /**** UI ****/
    $scope.discussionWidgetPage = function() {
        if (!$scope.discussions) {
            return 'LIST'; //loading
        } else if ($scope.uiState.creatingNewConv) {
            return 'CREATION';
        } else if (!$scope.uiState.creatingNewConv && !$scope.uiState.selectedConv) {
            return 'LIST';
        } else {
            return 'DISCUSSION';
        }
    };

    $scope.onSearchBarEnter = function($event) {
        $event.target.blur();
    };

    $scope.getNewRepliesLabel = function() {
        if (!$scope.uiState.selectedConv) {
            return;
        }
        const newRepliesCount = $scope.getNumberOfNewReplies($scope.uiState.selectedConv, $scope.uiState.cachedUserReadTime || 0);
        return newRepliesCount + ' new repl' + (newRepliesCount > 1 ? 'ies' : 'y');
    };

    $scope.getNumberOfNewReplies = function(discussion, fromTime) {
        if (!discussion) {
            return;
        }
        return discussion.replies.filter(reply => reply.time > fromTime).length;
    };

    $scope.getDisabledReason = function() {
        if (!$scope.uiState.selectedConv && !($scope.uiState.newConvTopic || '').length) {
            return 'The topic cannot be empty!';
        }
        if (!$scope.uiState.selectedConv && ($scope.uiState.newConvTopic || '').length > 200) {
            return 'The topic cannot be longer than 200 characters!';
        }
        if (!($scope.uiState.newReply || '').length) {
            return 'The message cannot be empty!';
        }
        if (($scope.uiState.newReply || '').length > 10000) {
            return 'The message cannot be longer than 10000 characters!';
        }
        return 'Press Enter to insert a new line. And press Ctrl+Enter to send your message.';
    };

    $scope.viewNewReplies = function(scrollToNewReplies) {
        $scope.uiState.displayToastNewReplies = false;
        $scope.uiState.invalidateCachedUserReadTime = true;
        if (scrollToNewReplies) {
            $timeout(function() {
                const el = $('.discussions-widget-list-replies');
                const newReplies = $('.discussions-widget-newreplies');
                if (!newReplies || !el.get(0)) {
                    return;
                }
                const scrollPos = newReplies.length ? el.scrollTop() + newReplies.position().top : el.get(0).scrollHeight;
                el.scrollTop(scrollPos);
            }, 200);
        }

        if ($scope.uiState.selectedConv && (($scope.uiState.selectedConv.users[$rootScope.appConfig.login] || {}).lastReadTime || 0) <= $scope.uiState.selectedConv.lastReplyTime) {
            DataikuAPI.discussions.ack(currentItem.projectKey, currentItem.type, currentItem.id, $scope.uiState.selectedConv.id, currentItem.workspaceKey)
                .error(setErrorInScope.bind($scope));
        }
    };

    $scope.openDiscussion = function(discussion, userAction) {
        $scope.uiState.selectedConv = discussion;
        // handle new replies when discussion is opened
        if ($scope.uiState.selectedConv) {
            // refresh cached user read time when:
            // - flag invalidate is true
            // - action comes from user
            // - no new reply from other users (basically exclude same user replies)
            const newLastReadTime = (($scope.uiState.selectedConv.users[$rootScope.appConfig.login] || {}).lastReadTime || 0);
            const hasLastReadTimeChanged = newLastReadTime != $scope.uiState.cachedUserReadTime;
            const hasNoPeerNewReplies = !$scope.uiState.selectedConv.replies.filter(reply => reply.time > newLastReadTime && reply.author != $rootScope.appConfig.login).length;
            if ($scope.uiState.invalidateCachedUserReadTime || userAction || hasNoPeerNewReplies) {
                $scope.uiState.cachedUserReadTime = newLastReadTime;
                $scope.uiState.invalidateCachedUserReadTime = false;
            }
            // display "view new replies" when action does not comes from user and there are new replies from other users
            if (userAction || hasNoPeerNewReplies) {
                $scope.viewNewReplies(userAction || hasLastReadTimeChanged);
            } else {
                $scope.uiState.displayToastNewReplies = true;
                const convListWidget = $('.discussions-widget-list-replies');
                const newRepliesLine = $('.discussions-widget-newreplies');
                if (convListWidget.size() && newRepliesLine.size()) {
                    $scope.uiState.showToastNewReplies = newRepliesLine.offset().top > convListWidget.offset().top + convListWidget.height() - 30;
                }
            }
        }
    };

    $scope.resetInputs = function() {
        delete $scope.uiState.newConvTopic;
        delete $scope.uiState.newReply;
        delete $scope.uiState.editingTopic;
    };

    $scope.getDiscussionParticipants = function(discussion) {
        const MAX_PARTICIPANT_LIST_LENGTH = 30;
        let participantListLength = 0;
        const participants = [];
        const displayedParticipants = [];
        for (const login in discussion.users) {
            if (discussion.users[login].lastReplyTime > 0) {
                participants.push(discussion.users[login]);
            }
        }
        participants.sort((a, b) => b.lastReplyTime - a.lastReplyTime);
        for (let i = 0; i < participants.length; i++) {
            const escapedDisplayName = $filter('escapeHtml')(participants[i].displayName || 'Unknown user');
            if (participantListLength + escapedDisplayName.length + 2 <= MAX_PARTICIPANT_LIST_LENGTH) {
                displayedParticipants.push(escapedDisplayName);
                participantListLength += escapedDisplayName.length + 2;
            } else {
                break;
            }
        }
        const othersCount = participants.length - displayedParticipants.length;
        let participantsListStr = displayedParticipants.join('<small>, </small>');
        if (othersCount > 0) {
            if (displayedParticipants.length == 0) {
                participantsListStr += participants.length + '<small> participant' + (participants.length > 1 ? 's' : '') + '</small>';
            } else {
                const hiddenParticipantCount = participants.length - displayedParticipants.length;
                participantsListStr += '<small> and </small>' + hiddenParticipantCount + '<small> other' + (hiddenParticipantCount > 1 ? 's' : '') + '</small>';
            }
        }
        return participantsListStr;
    };

    $scope.getDiscussionParticipantsList = function(discussion) {
        const arr = [];
        angular.forEach(discussion.users, function(value, key) {
            arr.push(angular.extend(value, {login: key}));
        });
        arr.sort((a, b) => b.lastReplyTime - a.lastReplyTime);
        return arr;
    };

    $scope.scrollChanged = function(userAction) {
        $timeout(function() {
            const convListWidget = $('.discussions-widget-list-replies');
            const newRepliesLine = $('.discussions-widget-newreplies');
            if (convListWidget.size() && newRepliesLine.size()) {
                $scope.uiState.showToastNewReplies = newRepliesLine.offset().top > convListWidget.offset().top + convListWidget.height() - 30;
                if (userAction && !$scope.uiState.showToastNewReplies && $scope.uiState.displayToastNewReplies) {
                    $scope.viewNewReplies(false);
                }
            }
        });
    };

    /**** Actions ****/
    $scope.closeDiscussion = function(close) {
        if (!$scope.uiState.selectedConv) {
            return;
        }
        DataikuAPI.discussions.close(currentItem.projectKey, currentItem.type, currentItem.id, $scope.uiState.selectedConv.id, close, currentItem.workspaceKey)
            .success(function() {
                WT1.event("discussion-close", {close: close, state: $state.current.name});
                broadcastDiscussionCountChange();
            })
            .error(setErrorInScope.bind($scope));
    };

    $scope.editTopic = function() {
        if (!$scope.uiState.selectedConv || !$scope.uiState.newConvTopic) {
            return;
        }
        WT1.event("discussion-edit-topic", {state: $state.current.name});
        DataikuAPI.discussions.save(currentItem.projectKey, currentItem.type, currentItem.id, $scope.uiState.selectedConv.id, $scope.uiState.newConvTopic, currentItem.workspaceKey)
            .success(() => $scope.uiState.editingTopic = false)
            .error(setErrorInScope.bind($scope));
        $scope.resetInputs();
    };

    $scope.resetEditing = function() {
        delete $scope.uiState.replyEditing;
        delete $scope.uiState.replyEditedText;
    }

    $scope.editReply = function() {
        const validEditedText = ($scope.uiState.replyEditedText || '').length > 0 && ($scope.uiState.replyEditedText || '').length <= 10000;
        if (!$scope.uiState.selectedConv || !($scope.uiState.selectedConv.replies[$scope.uiState.replyEditing] || {}).id || !validEditedText) {
            return;
        }
        WT1.event("discussion-edit-reply", {state: $state.current.name});
        DataikuAPI.discussions.reply(currentItem.projectKey, currentItem.type, currentItem.id, $scope.uiState.selectedConv.id, $scope.uiState.replyEditedText, $scope.uiState.selectedConv.replies[$scope.uiState.replyEditing].id, currentItem.workspaceKey)
            .success(function() {
                $scope.resetEditing();
            })
            .error(setErrorInScope.bind($scope));
        $scope.resetInputs();
    };

    $scope.replyDiscussion = function() {
        const replyTopic = $scope.uiState.newConvTopic || '';
        const replyContent = $scope.uiState.newReply;
        if (!replyContent || (!replyTopic && !$scope.uiState.selectedConv)) {
            return;
        }
        WT1.event("discussion-reply", {
                state: $state.current.name,
                'number_of_discussions': $scope.discussions.length,
                'number_of_replies': $scope.discussions.map(c => c.replies.length).reduce((a,b) => a+b, 0)
        });
        if ($scope.uiState.selectedConv) {
            DataikuAPI.discussions.reply(currentItem.projectKey, currentItem.type, currentItem.id, $scope.uiState.selectedConv.id, replyContent, null, currentItem.workspaceKey)
                .error(setErrorInScope.bind($scope));
        } else {
            DataikuAPI.discussions.create(currentItem.projectKey, currentItem.type, currentItem.id, replyTopic, replyContent, currentItem.workspaceKey)
                .success(function(data) {
                    $scope.uiState.forceSelectedConvId = data.id;
                    broadcastDiscussionCountChange();
                })
                .error(setErrorInScope.bind($scope));
        }
        $scope.resetInputs();
        // ugly hack to remove tooltips
        $timeout(() => { $('body > .tooltip').remove(); });
    };

    $scope.deleteDiscussion = function() {
        if (!$scope.uiState.selectedConv) {
            return;
        }
        Dialogs.confirm($scope, 'Delete discussion', 'Warning: deleting ' + ($scope.uiState.selectedConv.topic ? ('discussion "' + $scope.uiState.selectedConv.topic + '"') : 'this discussion') + ' will erase permanently its whole content including all the replies. This operation is irreversible. Do you want to continue?').then(function() {
            DataikuAPI.discussions.delete(currentItem.projectKey, currentItem.type, currentItem.id, $scope.uiState.selectedConv.id, currentItem.workspaceKey)
                .success(function() {
                    WT1.event("discussion-delete", {state: $state.current.name});
                    delete $scope.uiState.selectedConv;
                    broadcastDiscussionCountChange();
                })
                .error(setErrorInScope.bind($scope));
        });
    };

    function getDiscussionById(id) {
        Assert.inScope($scope, 'discussions');
        return $scope.discussions.find(conv => conv.id == id) || null;
    }

    function refreshDiscussions() {
        const discussionIdFromStateParams = $stateParams.discussionId;
        if (discussionIdFromStateParams) {
            // Clear from stateParams so that it does not stick around if we move directly to another taggable object...
            $state.go('.', {'#': $location.hash(), discussionId: null}, {notify: false, location: 'replace'});
        }
        DataikuAPI.discussions.getForObject(currentItem.projectKey, currentItem.type, currentItem.id, currentItem.workspaceKey)
            .success(function(data) {
                $scope.discussions = data.discussions;
                if ($scope.discussions && $scope.discussions.length > 0 && !$scope.discussions.find(discu => discu.closedOn == 0)) {
                    $scope.uiState.showClosed = true;
                }
                const userActionWhenSingleDiscussion = $scope.discussionId && !$scope.uiState.selectedConv;
                if ($scope.discussionId) {
                    $scope.uiState.forceSelectedConvId = $scope.discussionId();
                }
                if ($scope.uiState.forceSelectedConvId) {
                    $scope.uiState.creatingNewConv = false;
                    $scope.uiState.selectedConv = getDiscussionById($scope.uiState.forceSelectedConvId);
                    delete $scope.uiState.forceSelectedConvId;
                }
                if ($scope.uiState.selectedConv) {
                    $scope.openDiscussion(getDiscussionById($scope.uiState.selectedConv.id), userActionWhenSingleDiscussion);
                } else if (discussionIdFromStateParams && !$scope.discussionId) {
                    const discussion = getDiscussionById(discussionIdFromStateParams);
                    if (discussion) {
                        $scope.openDiscussion(discussion, true);
                    } else {
                        ActivityIndicator.error("Discussion "+discussionIdFromStateParams+" not found");
                    }
                }
                $scope.scrollChanged(false);
            })
            .error(setErrorInScope.bind($scope));
    }

    function updateLastReadByUsers(evt, message) {
        if (!$scope.discussions) {
            return;
        }
        if (!message || !(message.details || {}).time || !message.user) {
            return;
        }
        const conv = $scope.discussions.find(conv => conv.id == message.discussionId);
        if (conv) {
            if (!conv.users[message.user]) {
                conv.users[message.user] = {
                    login: message.user,
                    displayName: message.details.userDisplayName
                };
            }
            conv.users[message.user].lastReadTime = message.details.time;
        }
    }

    function selectedItemUpdated() {
        const value = $scope.selectedItem();
        const hasSelectedItem = !!value;
        const hasSelectedItemChanged = hasSelectedItem && !angular.equals(currentItem, value);
        const hasSingleDiscussionChanged = $scope.discussionId && !angular.equals($scope.discussionId(), $scope.uiState.forceSelectedConvId);
        if (hasSelectedItem && (hasSelectedItemChanged || hasSingleDiscussionChanged)) {
            currentItem = value;
            $scope.uiState = {forceSelectedConvId: $scope.uiState.forceSelectedConvId || ($scope.uiState.selectedConv || {}).id};
            if ((currentItem.projectKey || currentItem.workspaceKey) && currentItem.type && currentItem.id) {
                refreshDiscussions();
            }
        }
    }

    function broadcastDiscussionCountChange() {
        if ($state.is("projects.project.flow")) {
            $rootScope.$broadcast('discussionCountChanged');
        }
    }

    /*** Init ***/
    if ($scope.watchObject && $scope.selectedItem) {
        // current item is defined through directive attribute
        let debounceFn = Debounce().withDelay(20, 200).withScope($scope).wrap(selectedItemUpdated);
        $scope.$watch('watchObject', debounceFn, true);
    } else {
        // current item is retrieved from TopNav
        currentItem = angular.copy(TopNav.getItem());
        currentItem.projectKey = $stateParams.projectKey;
        currentItem.workspaceKey = $stateParams.workspaceKey;
        refreshDiscussions();
    }

    const replyListenerDestroyer = Notification.registerEvent('discussion-reply', refreshDiscussions);
    const deleteListenerDestroyer = Notification.registerEvent('discussion-delete', refreshDiscussions);
    const updateListenerDestroyer = Notification.registerEvent('discussion-update', refreshDiscussions);
    const closeListenerDestroyer = Notification.registerEvent('discussion-close', refreshDiscussions);
    const ackListenerDestroyer = Notification.registerEvent('discussion-ack', updateLastReadByUsers);

    $scope.$on('$destroy', function() {
        replyListenerDestroyer();
        deleteListenerDestroyer();
        updateListenerDestroyer();
        closeListenerDestroyer();
        ackListenerDestroyer();
    });

});


app.directive('discussionsWidget', function() {
    return {
        restrict: 'AE',
        templateUrl: '/templates/widgets/discussions-widget-content.html',
        scope: {
            selectedItem: '&',
            watchObject: '='
        }
    };
});


app.directive('discussionsWidgetSingle', function() {
    return {
        restrict: 'AE',
        templateUrl: '/templates/widgets/discussions-widget-single.html',
        scope: {
            selectedItem: '&',
            discussionId: '&',
            watchObject: '='
        }
    };
});


app.directive('discussionsButton', function($rootScope, $compile, $q, $templateCache, $http, $state, $stateParams, Assert, DataikuAPI, TopNav, Notification, Debounce) {
    return {
        restrict: 'AE',
        templateUrl: '/templates/widgets/discussions-widget-button.html',
        scope: {
            selectedItem: '&',
            watchObject: '='
        },
        replace: true,
        link: function($scope, element, attrs) {
            let currentItem;

            $scope.discussionCounts = null;

            $scope.uiState = {};
            $scope.uiState.loadRequested = false;
            $scope.uiState.displayed = false;
            $scope.uiState.maximized = false;

            $scope.titleForNumberOfDiscussions = function() {
                let title;
                const counts = $scope.discussionCounts;

                if (!counts) return; //Not ready
                if (counts.open) {
                    title = counts.open +
                        (counts.unread == counts.open && counts.unread > 0 ? ' unread' :'') +
                        ' discussion' +
                        (counts.open == 1 ? '' : 's');
                } else if (counts.total) {
                    title = 'No open discussion';
                } else {
                    title = 'No discussion';
                }
                let objectTypeName = $scope.uiState.currentObjectType.replace('_', ' ');
                if (objectTypeName == 'project') {
                    objectTypeName = "project's home page"; // let's not be confusing
                }
                title += ' on this ' + objectTypeName;

                if(counts.unread != counts.open && counts.unread > 0) {
                    title += ' (' + counts.unread + ' unread)';
                }
                return title;
            };

            function openDiscussionsWidget() {
                if (!$scope.uiState.loadRequested) {
                    loadWidget();
                }
                $scope.uiState.displayed = true;
                $scope.uiState.maximized = true;
            }

            $scope.closeDiscussionsWidget = function() {
                $scope.uiState.displayed = false;
            };

            $scope.toggleDiscussionsWidget = function() {
                if ($scope.uiState.displayed && $scope.uiState.maximized) {
                    $scope.closeDiscussionsWidget();
                } else {
                    openDiscussionsWidget();
                }
            };

            $scope.toggleMaximized = function() {
                $scope.uiState.maximized = !$scope.uiState.maximized;
            };

            function refreshCounts() {
                DataikuAPI.discussions.getCounts(currentItem.projectKey, currentItem.type, currentItem.id, currentItem.workspaceKey)
                    .success(function(data) {
                        $scope.discussionCounts = data;
                        if (!$scope.uiState.displayed && $scope.discussionCounts.unread > 0) {
                            if (!$scope.uiState.loadRequested) {
                                loadWidget();
                            }
                            $scope.uiState.displayed = true;
                            $scope.uiState.maximized = false;
                        }
                    })
                    .error(setErrorInScope.bind($scope));
            }

            function loadWidget() {
                Assert.trueish(!$scope.uiState.loadRequested, 'loadWidget called twice');
                $scope.uiState.loadRequested = true;
                const location = '/templates/widgets/discussions-widget-popover.html'
                $q.when($templateCache.get(location) || $http.get(location, {cache: true}))
                    .then(function(template) {
                        // we need to make sure the item didn't change or the scope didn't get destroyed
                        // between the API call and when we load the widget (or we will display it on the wrong page)
                        const selectedItem = $scope.watchObject && $scope.selectedItem ? $scope.selectedItem() : TopNav.getItem();
                        if (!TopNav.sameItem(currentItem, selectedItem)) return;
                        
                        if (angular.isArray(template)) {
                            template = template[1];
                        } else if (angular.isObject(template)) {
                            template = template.data;
                        }
                        const widgetEl = $(template);
                        $compile(widgetEl)($scope);
                        $('.main-view').append(widgetEl);
                        $scope.$on('$destroy', function() {
                            widgetEl.remove();
                        });
                    });
            }

            function selectedItemUpdated() {
                const value = $scope.selectedItem();
                const hasSelectedItem = !!value;
                const hasSelectedItemChanged = hasSelectedItem && !angular.equals(currentItem, value);
                const hasSingleDiscussionChanged = $scope.discussionId && !angular.equals($scope.discussionId(), $scope.uiState.forceSelectedConvId);
                if (hasSelectedItem && (hasSelectedItemChanged || hasSingleDiscussionChanged)) {
                    currentItem = value;
                    $scope.uiState.currentObjectType = value.type.toLowerCase().replace('[^a-zA-Z]', ' ');
                    refreshCounts();
                }
            }

            if ($scope.watchObject && $scope.selectedItem) {
                // current item is defined through directive attribute
                let debounceFn = Debounce().withDelay(20, 200).withScope($scope).wrap(selectedItemUpdated);
                $scope.$watch('watchObject', debounceFn, true);
            } else {
                // current item is retrieved from TopNav
                currentItem = angular.copy(TopNav.getItem());
                currentItem.projectKey = $stateParams.projectKey;
                currentItem.workspaceKey = $stateParams.workspaceKey;
                if (currentItem.type) {
                    $scope.uiState.currentObjectType = currentItem.type.toLowerCase().replace('[^a-zA-Z]', ' ');
                }
                refreshCounts();
            }

            const replyListenerDestroyer = Notification.registerEvent('discussion-reply', refreshCounts);
            const deleteListenerDestroyer = Notification.registerEvent('discussion-delete', refreshCounts);
            const closeListenerDestroyer = Notification.registerEvent('discussion-close', refreshCounts);
            const ackListenerDestroyer = Notification.registerEvent('discussion-ack', function(evtType, message) {
                if ($scope.discussionCounts && $scope.discussionCounts.unread && message.user == $rootScope.appConfig.login) {
                    refreshCounts()
                }
            });
            $scope.$on('$destroy', function() {
                replyListenerDestroyer();
                deleteListenerDestroyer();
                closeListenerDestroyer();
                ackListenerDestroyer();
            });

            if ($stateParams.discussionId) {
                openDiscussionsWidget();
            }
        }
    };
});


app.service('Discussions', function($rootScope, Notification, MessengerUtils, StateUtils, UserImageUrl) {
    function userAvatar(userLogin, size) {
        if (!userLogin) return "";
        const imageUrl = UserImageUrl(userLogin, size);
        const sizeClass = size ? "size-" + sanitize(size) : "size-fit";
        return `<img class="user-avatar ${sizeClass}" src="${imageUrl}" />`;
    }

    function userLink(userLogin, innerHTML) {
        return '<a href="/profile/'+escape(userLogin)+'/" class="link-std">'+ innerHTML + '</a>';
    }

    function dssObjectLink(event, innerHTML) {
        const tor = {type: event.objectType, id: event.objectId, projectKey: event.projectKey, workspaceKey: event.workspaceKey};
        const link = StateUtils.href.taggableObject(tor, {moveToTargetProject: false, discussionId: event.discussionId});
        return '<a href="'+link+'" class="link-std">'+innerHTML+'</a>';
    }

    function discussionIsOpen(id) {
        const discussionScope = angular.element($('.discussions-widget-list-replies')).scope();
        return discussionScope && discussionScope.uiState && discussionScope.uiState.selectedConv.id == id;
    }

    const replyListenerDestroyer = Notification.registerEvent('discussion-reply',function(evt, message) {
        if (message.user == $rootScope.appConfig.login) {
            return; // Hopefully the user knows that they wrote
        }
        if (!message.newReply) {
            return; // A reply was edited, don't notify...
        }
        if (message.mentionedUsers && message.mentionedUsers.includes($rootScope.appConfig.login)) {
            return; // The current user is mentioned in this message, there will be a specific notification for that, don't duplicate
        }
        if (discussionIsOpen(message.discussionId)) {
            return;
        }

        MessengerUtils.post({
            message: userLink(message.user, sanitize(message.details.userDisplayName || message.user))
                + (message.creation ? " created a discussion on " : " added a reply on ")
                + dssObjectLink(message, sanitize(message.details.objectDisplayName))
                + ":"
                + '<span class="messenger-comment">'
                + sanitize(message.text.substr(0,400))
                + (message.text.length > 400 ? '[...]' : '')
                + '</span>'
                ,
            icon: userAvatar(message.user),
            hideAfter: 5,
            showCloseButton: true,
            id: message.user+'connected',
            type: 'no-severity'
        });
    });

    const mentionListenerDestroyer = Notification.registerEvent("discussion-mention", function(evt, message) {
        MessengerUtils.post({
            message: userLink(message.author, sanitize(message.details.authorDisplayName || message.author))
                + " mentioned you in a discussion on "
                + dssObjectLink(message, sanitize(message.details.objectDisplayName))
                + ":"
                + '<span class="messenger-comment">'
                + sanitize(message.message.substr(0,400))
                + (message.message.length > 400 ? '[...]' : '')
                + '</span>'
                ,
            icon: userAvatar(message.author),
            type: 'no-severity',
            showCloseButton: true
        });
    });

    const closeListenerDestroyer = Notification.registerEvent("discussion-close", function(evt, message) {
        if (message.user == $rootScope.appConfig.login) {
            return;
        }
        MessengerUtils.post({
            message: userLink(message.user, sanitize(message.details.userDisplayName || message.user))
                + " has " + (message.details.closed ? "resolved" : "reopened") + " a discussion on "
                + dssObjectLink(message, sanitize(message.details.objectDisplayName))
                ,
            icon: userAvatar(message.user),
            type: 'no-severity',
            showCloseButton: true
        });
    });

    $rootScope.$on('$destroy', function() {
        replyListenerDestroyer();
        mentionListenerDestroyer();
        closeListenerDestroyer();
    });
});


})();
;
(function() {
'use strict';

const app = angular.module('dataiku.collab.wikis', []);

app.constant('WIKI_TAXONOMY_KEY', 'dss.wiki.taxonomy');

app.service('WikiUtilsService', function () {
    const svc = this;

    svc.getParentId = function(id, node, taxonomy) {
        const children = node ? node.children : taxonomy;

        if (!children) {
            return null;
        }
        const nodeId = node ? node.id : '';
        for (const child of children) {
            if (child.id === id) {
                return nodeId;
            }
            const foundParent = svc.getParentId(id, child);
            if (foundParent) {
                return foundParent;
            }
        }
        return null;
    };

    svc.getArticleNodeById = function(articleId, node, taxonomy) {
        const children = node && node.children ? node.children : taxonomy;
        for (const child of children) {
            if (child.id === articleId) {
                return child;
            }
            const foundNode = svc.getArticleNodeById(articleId, child);
            if (foundNode) {
                return foundNode;
            }
        }
        return null;
    };

    svc.addArticlesToList = function(nodes, articlesIds) {
    