(function() {
'use strict';

const app = angular.module('dataiku.promptstudios', ['dataiku.services', 'dataiku.filters']);

app.factory("PromptUtils", function(Logger, CreateModalFromTemplate, $filter, $dkuSanitize, COLOR_PALETTES, $stateParams, DataikuAPI, WT1, SmartId) {
    const svc = {};

    svc.getTemperatureRange = function(llm) {
        if (!llm) {
            Logger.warn("No selected LLM");
            return { min: undefined, max: undefined, step: undefined };
        }
        return llm.temperatureRange;
    };

    svc.getTopKRange = function(llm) {
        if (!llm) {
            Logger.warn("No selected LLM");
            return { min: undefined, max: undefined, step: undefined };
        }
        return llm.topKRange;
    };

    svc.prefixToInputType = function(prefix, supportedPrefixes) {
        if (prefix === 'image:' && supportedPrefixes[prefix]) {
            return 'IMAGE';
        }

        return 'TEXT';
    };

    svc.updateInputCheck = function(prompt, supportsImageInputs) {  // for advanced mode prompts only
        if (!prompt) return;

        if (prompt.textPromptTemplate == null && prompt.textPromptSystemTemplate == null) {
            prompt.textPromptTemplateInputs = [];
            return;
        }

        const fullText = (prompt.textPromptTemplate || '') + (prompt.textPromptSystemTemplate || '');
        if (fullText === '') {
            prompt.textPromptTemplateInputs = [];
            return;
        }

        const oldInputs = angular.copy(prompt.textPromptTemplateInputs);
        const pattern = /{{([^{}]+)}}/g;
        const matchedInputNames = new Set([...fullText.matchAll(pattern)].map(match => match[1]));
        const inputArray = (prompt.textPromptTemplateInputs || []);
        const supportedPrefixes = this.getSupportedPrefixes(supportsImageInputs);
        inputArray.forEach(function(input) {
            if (!matchedInputNames.has(input.name)) {
                input.toDelete = true;
            } else {
                // update type if LLM was changed and new one doesn't/does support image inputs
                input.type = svc.getPromptInputFromPlaceholder(input.name, supportedPrefixes).type;
                matchedInputNames.delete(input.name);
            }
        });
        const missingNewInputs = Array.from(matchedInputNames).map(name => this.getPromptInputFromPlaceholder(name, supportedPrefixes));
        prompt.textPromptTemplateInputs = inputArray.filter(input => !input.toDelete).concat(missingNewInputs);
        // return true if inputs changed
        return !angular.equals(oldInputs, prompt.textPromptTemplateInputs);
    };

    svc.createNewPrompt = function(mode, llmId, llmSettings, inputSource, datasetRef, templateId) {
        const newPromptStudioPrompt = {};
        newPromptStudioPrompt.inlinePromptTemplateQueries = [];
        newPromptStudioPrompt.tags = [];
        newPromptStudioPrompt.llmId = llmId;
        newPromptStudioPrompt.llmSettings = Object.assign({ temperature: undefined, topK: undefined, topP: undefined, maxOutputTokens: undefined, stopSequences: [] }, llmSettings);
        newPromptStudioPrompt.nbRows = 8;

        const newPrompt = {};
        newPrompt.promptTemplateInputs = [];
        newPrompt.textPromptTemplateInputs = [];
        newPrompt.structuredPromptExamples = [];
        newPrompt.promptTemplateQueriesSource = inputSource;
        newPrompt.promptMode = mode;
        newPrompt.placeholder = '';

        if (datasetRef) {
            newPromptStudioPrompt.dataset = datasetRef;
        }

        if (newPrompt.promptMode === "PROMPT_TEMPLATE_TEXT") {
            newPrompt.textPromptTemplate = "Do something about {{input}}";
            newPrompt.textPromptSystemTemplate = "";
            newPrompt.textPromptTemplateInputs = [{
                name: "input",
                type: 'TEXT'
            }];

            WT1.event("prompt-studio-new-prompt-modal-new-prompt-template", {
                llmId: llmId
            });

        } else if (newPrompt.promptMode === "PROMPT_TEMPLATE_STRUCTURED") {
            WT1.event("prompt-studio-new-prompt-modal-new-from-template", {
                templateId,
                llmId: llmId
            });

            switch(templateId) {
                case "sample-blank":
                    newPrompt.promptTemplateInputs = [{ name: "input", type: "TEXT" }];
                    newPrompt.structuredPromptPrefix = "";
                    newPrompt.placeholder = "Explain here what the model must do";
                    break;
                case "sample-extract-technical-specifications":
                    newPrompt.promptTemplateInputs = [{ name: "Product description", type: "TEXT" }];
                    newPrompt.structuredPromptPrefix = "Extract the key information from the below product reviews in JSON format";
                    break;
                case "sample-summarize-product-review":
                    newPrompt.promptTemplateInputs = [
                        { name: "Review text", type: "TEXT" },
                        { name: "Review title", type: "TEXT" },
                    ];
                    newPrompt.structuredPromptPrefix = "Summarize the below product reviews in two sentences. Use only the information from the review text and title.";
                    newPrompt.structuredPromptExamples =[
                        {
                            inputs: ["When I went to the mailbox, I was super sad to see that the packaging was damaged when it arrived. It was really messy. That's disappointing. However, once I opened it, it was good. Instructions seemed clear, but it still took me long to get running. Results are okay",
                                    "Package was a mess"],
                            output: "The packaging was damaged, but the product gives satisfying results. Ease of use could be improved"
                        }
                    ];
                    break;
                case "sample-support-requests-classification":
                    newPrompt.promptTemplateInputs = [{ name: "Support request text", type: "TEXT" }];
                    newPrompt.structuredPromptPrefix = "Given the following support request from a customer, classify it among the following topics:\n\n* Pricing request\n* Payment problem\n* Information request before sale\n* Delivery issue\n* Defective product\n* Complaint against company personnel\n* Insulting/Abusive customer\n* Unknown/Other\n\nOnly reply with the category";
                    break;
                case "sample-contract-analysis":
                    newPrompt.promptTemplateInputs = [{ name: "Contract text", type: "TEXT" }];
                    newPrompt.structuredPromptPrefix = "Read the provided software master license agreement. Then indicate:\n\n1) What is the general liability cap\n2) What is the governing law\n3) What are the termination for convenience terms";
                    break;
                case "sample-align-language-level":
                    newPrompt.promptTemplateInputs = [{ name: "Text to rewrite", type: "TEXT" }];
                    newPrompt.structuredPromptPrefix = "Rewrite the following text as professional-level language. Correct improper English. Do not try to summarize or otherwise change the meaning.";
                    newPrompt.structuredPromptExamples = [
                        {
                            inputs: ["U know your in a bad place when u see cockroaches. Urgh. I'll never 4 ever wuld go there again. The food was meh, and the service ... sigh"],
                            output: "There were cockroaches in the place. Would never return. The food was mediocre. Service was not great."
                        },
                        {
                            inputs: ["We had a lovely time in this restaurant. The waiter was nice and service was fast. Food was tasty. The price was a bit high but worth it"],
                            output: "We had a lovely time in this restaurant. The waiter was nice and service was fast. Food was tasty. The price was a bit high but worth it"
                        }
                    ];
                    break;
                case "sample-chain-of-thought":
                    newPrompt.promptTemplateInputs = [{ name: "Reasoning problem", type: "TEXT" }];
                    newPrompt.structuredPromptPrefix = "Here are some reasoning problems. Think step by step, before giving your final answer.";
                    newPrompt.structuredPromptExamples = [
                        {
                            inputs: ["A coin is heads up. Ka turns the coin over. Sherrie turns the coin over. Is the coin still heads up?"],
                            output: "The coin was turned over by both Ka and Sherrie. So the coin was turned over two times, which is an even number. The coin started heads up, and a coin only has two sides, so after an even number of turns, it will be back to heads up. So the answer is yes."
                        },
                        {
                            inputs: ["A person is traveling at 20 km/h and reaches his destination in 2.5 hours, can you find the traveled distance?"],
                            output: "The distance that the person traveled would be 20 km/hr * 2.5 hrs = 50 km. The answer is 50 km."
                        },
                        {
                            inputs: ["There were nine computers in the server room. Five more computers were installed each day, from Monday to Thursday. How many computers are now in the server room?"],
                            output: "There were originally 9 computers. For each of four days, five more computers were added. So 5 * 4 = 20 computers were added. 9 + 20 is 29. The answer is 29."
                        }
                    ];
                    break;
            }
        } else if (newPrompt.promptMode === "RAW_PROMPT") {
            WT1.event("prompt-studio-new-prompt-modal-new-raw-prompt", {
                llmId: llmId
            });
            newPrompt.rawPromptText = "Explain in simple terms what Generative AI is and why prompts matter for it";
        } else if (newPrompt.promptMode === "CHAT") {
            WT1.event("prompt-studio-new-prompt-modal-new-chat", {
                llmId: llmId
            });
            newPrompt.chatSystemMessage = '';
            newPrompt.chatMessages = {};
        }
        else {
            throw("Unknown prompt mode: " + newPrompt.promptMode);
        }
        newPrompt.resultValidation = { expectedFormat: 'NONE' };
        newPromptStudioPrompt.prompt = newPrompt;

        return newPromptStudioPrompt;
    };

    svc.getPromptInputFromPlaceholder = function(name, supportedPrefixes) {
        const index = name.indexOf(':');
        const prefix = name.substring(0, index + 1);
        return {
            name,
            type: this.prefixToInputType(prefix, supportedPrefixes)
        };
    };

    svc.getSupportedPrefixes = function(supportsImageInputs) {
        return {
            'image:': supportsImageInputs
        }
    }

    svc.getInputs = function(prompt) {
        if (prompt.promptMode === 'PROMPT_TEMPLATE_STRUCTURED') {
            return prompt.promptTemplateInputs;
        }
        if (prompt.promptMode === 'PROMPT_TEMPLATE_TEXT') {
            if (!prompt.textPromptTemplateInputs) { // for older prompts, in case textPromptTemplateInputs did not exist
                prompt.textPromptTemplateInputs = prompt.promptTemplateInputs;
            }
            return prompt.textPromptTemplateInputs;
        }
        return []; // should not happen
    };

    svc.hasNoUserMessage = function(prompt) {
        switch (prompt.promptMode) {
            case "PROMPT_TEMPLATE_STRUCTURED":
                return !this.getInputs(prompt).length;
            case "PROMPT_TEMPLATE_TEXT":
                return !this.getInputs(prompt).length && !prompt.textPromptTemplate;
            case "RAW_PROMPT":
                return !prompt.rawPromptText;
            default:
                return true;
        }
    }

    svc.updateInputsOnLLMChange = function(prompt, llm) {
        if (prompt.promptMode === 'PROMPT_TEMPLATE_TEXT') {
            this.updateInputCheck(prompt, llm.supportsImageInputs);
        } else if (prompt.promptMode === 'PROMPT_TEMPLATE_STRUCTURED') {
            if (!llm.supportsImageInputs) {
                this.getInputs(prompt).forEach(input => {
                    input.type = 'TEXT';
                });
            }
        }
    }

    svc.hasImageInput = function(prompt) {
        return this.getInputs(prompt).some(input => input.type === 'IMAGE');
    }

    svc.getInlinePromptInputMap = function(prompt, templateQuery) {
        const inputNames = svc.getInputs(prompt).map(input => input.name);
        const inputs = inputNames.map(column => templateQuery.data[column])
        return {
            inputNames,
            inputs
        };
    };

    svc.formatOutput = function(response) {
        return response.formattedOutput || response.rawLLMOutput;
    };

    svc.openRawPromptModal = function(scope, prompt, llmId, rawInputs) {
        function mergeAdjacentTextMessages(allMessages) {
            if (allMessages.length <= 1) return allMessages;

            const mergedMessages = [allMessages[0]];

            for (let i = 1; i < allMessages.length; i++) {
                let currentMessage = allMessages[i];
                let lastMergedMessage = mergedMessages[mergedMessages.length -1];
                if (lastMergedMessage.role !== currentMessage.role || lastMergedMessage.type !== currentMessage.type || currentMessage.type === "IMAGE") {
                    mergedMessages.push(currentMessage);
                } else {
                    lastMergedMessage.content += lastMergedMessage.inline ? "" : "\n";
                    lastMergedMessage.content += currentMessage.content;
                    lastMergedMessage.inline = currentMessage.inline;
                }
            }
            return mergedMessages;
        }

        function addRagSystemPrompt(rag, messages) {
            messages.push({
                "role": "system",
                "content": rag.ragllmSettings.contextMessage,
                "type": "TEXT"
            });
            return messages;
        }

        function addRagSourcesMessage(messages) {
            messages.push({
                "role": "user",
                "content": "{{rag_sources}}",
                "type": "TEXT"
            });
            return messages;
        }

        function createImageMessage(path) {
            return { type: "IMAGE", "url": svc.getImageUrl(path, prompt.imageFolderId) };
        }

        function createMessages(rag) {
            let messages = [];
            if (prompt.promptMode === "PROMPT_TEMPLATE_STRUCTURED") {
                if (prompt.structuredPromptPrefix) {
                    messages.push({"role": "system", "content": prompt.structuredPromptPrefix, "type": "TEXT"});
                }
                if (prompt.structuredPromptExamples && prompt.structuredPromptExamples.length > 0) {
                    for (let example of prompt.structuredPromptExamples) {
                        let messageContent = example.inputs.map((e, i) => prompt.promptTemplateInputs[i].name + ': ' + e).join('\n');
                        messages.push({"role": "User", "content": messageContent, "type": "TEXT"});
                        messages.push({"role": "Assistant", "content": example.output, "type": "TEXT"});
                    }
                }
                if (rag) {
                    messages = addRagSystemPrompt(rag, messages);
                    messages = addRagSourcesMessage(messages);
                }
                prompt.promptTemplateInputs.forEach((input, index) => {
                    if (input.datasetColumnName in rawInputs) {
                        if (input.type === "IMAGE") {
                            messages.push(createImageMessage(rawInputs[input.datasetColumnName]));
                        } else {
                            messages.push({
                                "role": "user",
                                "content": input.name + ': ' + (rawInputs[input.datasetColumnName] ? rawInputs[input.datasetColumnName] : ""),
                                "type": "TEXT"
                            });
                        }
                    } else {
                        if (prompt.promptTemplateQueriesSource === "INLINE" && rawInputs[input.name] !== undefined){
                            messages.push({"role": "user", "content": input.name + ': ' + rawInputs[input.name], "type": "TEXT" });
                        }
                        else {
                            messages.push({"role": "user", "content": `${input.name}: {value for input #${index + 1} (${input.name})}`, "type": "TEXT" });
                        }
                    }
                });
            } else if (prompt.promptMode === "PROMPT_TEMPLATE_TEXT") {
                if (prompt.textPromptSystemTemplate) {
                    messages.push({"role": "system", "content": prompt.textPromptSystemTemplate, "type": "TEXT"});
                }
                if (rag) {
                    messages = addRagSystemPrompt(rag, messages);
                    messages = addRagSourcesMessage(messages);
                }
                if (Object.values(rawInputs).length === 0) {
                    messages.push({"role": "user", content: prompt.textPromptTemplate, "type": "TEXT"})
                } else {
                    const mappedInputs = prompt.textPromptTemplateInputs.reduce((acc, obj) => ({ ...acc, ["{{" + obj.name + "}}"]: obj }), {});
                    let textSplit = prompt.textPromptTemplate.split(/({{[^}]+}})/g); // splits a list into component parts

                    for (let item of textSplit) {
                        if (item in mappedInputs) {
                            if (mappedInputs[item].type === "IMAGE") {
                                messages.push(createImageMessage(rawInputs[mappedInputs[item].datasetColumnName]))
                            } else {
                                let content;
                                if (prompt.promptTemplateQueriesSource === "INLINE") {
                                    content = rawInputs[mappedInputs[item].name] === null ? "" : rawInputs[mappedInputs[item].name]
                                } else {
                                    content = rawInputs[mappedInputs[item].datasetColumnName] === null ? "" : rawInputs[mappedInputs[item].datasetColumnName];
                                }

                                messages.push({
                                    "role": "user",
                                    "content": content,
                                    "type": "TEXT",
                                    "inline": true
                                })
                            }
                        } else {
                            messages.push({ "role": "user", content: item, "type": "TEXT", "inline": true })
                        }
                    }
                }
            } else if (prompt.promptMode === "RAW_PROMPT") {
                if (rag) {
                    messages = addRagSystemPrompt(rag, messages);
                    messages = addRagSourcesMessage(messages);
                }
                messages.push({"role": "user", "content": prompt.rawPromptText, "type": "TEXT"});
            }
            return messages;
        }

        const llmIdParts = llmId.split(":");
        const isRAG = llmIdParts[0] === "retrieval-augmented-llm";
        if (isRAG) {
            DataikuAPI.savedmodels.getFullInfo($stateParams.projectKey, SmartId.create(llmIdParts[1], $stateParams.projectKey)).success(function(res) {
                const ragActiveVersion = res.model.inlineVersions.find(inlineVersion => inlineVersion.versionId === res.model.activeVersion);
                CreateModalFromTemplate("/templates/promptstudios/prompt-studio-see-raw-prompt.html", scope, null, function(newScope) {
                    newScope.messages = mergeAdjacentTextMessages(createMessages(ragActiveVersion));
                });
            });
        } else {
            CreateModalFromTemplate("/templates/promptstudios/prompt-studio-see-raw-prompt.html", scope, null, function(newScope) {
                newScope.messages = mergeAdjacentTextMessages(createMessages(undefined));
            });
        }
    };

    svc.getPromptText = function(prompt) {
        let fullText = '';
        let firstMessage;
        let parentMessage;

        switch (prompt.promptMode) {
            case 'PROMPT_TEMPLATE_TEXT':
                fullText = prompt.textPromptSystemTemplate || '';
                fullText += "\n\n"
                fullText += prompt.textPromptTemplate || '';
                break;
            case 'PROMPT_TEMPLATE_STRUCTURED':
                fullText = prompt.structuredPromptPrefix || '';
                break;
            case 'CHAT':
                // use first user message
                parentMessage = Object.values(prompt.chatMessages).find(message => !message.parentId);
                if(parentMessage){
                    firstMessage = Object.values(prompt.chatMessages).find(message => message.parentId == parentMessage.id && message.version === 0);
                }
                fullText = firstMessage && firstMessage.message && firstMessage.message.content ? firstMessage.message.content : '';
                break;
            case 'RAW_PROMPT':
                fullText = prompt.rawPromptText || '';
                break;
        }

        return fullText;
    };

    svc.getLLMIcon = function(llmType, size = '') {
        switch(llmType) {
            case "CUSTOM":
                return "dku-icon-puzzle-piece-" + size;
            case "MOSAICML":
                return "dku-icon-mosaicml-" + size;
            case "OPENAI":
                return "dku-icon-openai-" + size;
            case "AZURE_OPENAI_DEPLOYMENT":
            case "AZURE_OPENAI_MODEL":
            case "AZURE_LLM":
                    return "dku-icon-microsoft-azure-" + size;
            case "VERTEX":
                return "dku-icon-google-vertex-" + size;
            case "COHERE":
                return "dku-icon-cohere-" + size;
            case "MISTRALAI":
                return "dku-icon-mistral-" + size;
            case "ANTHROPIC":
                return "dku-icon-anthropic-" + size;
            case "BEDROCK":
                return "dku-icon-amazon-aws-bedrock-" + size;
            case "HUGGINGFACE_API":
            case "HUGGINGFACE_TRANSFORMER_LOCAL":
                return "dku-icon-huggingface-" + size;
            case "SAVED_MODEL_AGENT":
                return "dku-icon-ai-agent-" + size;
            case "TOOLS_USING_AGENT":
                return "dku-icon-ai-agent-visual-" + size;
            case "SAVED_MODEL_FINETUNED_OPENAI":
            case "SAVED_MODEL_FINETUNED_AZURE_OPENAI":
            case "SAVED_MODEL_FINETUNED_VERTEX":
            case "SAVED_MODEL_FINETUNED_HUGGINGFACE_TRANSFORMER":
                return "dku-icon-saved-model-fine-tuning-" + size;
            case "RETRIEVAL_AUGMENTED":
                return "dku-icon-llm-augmented-" + size;
            case "SAGEMAKER_GENERICLLM":
                return "dku-icon-amazon-sagemaker-" + size;
            case "SNOWFLAKE_CORTEX":
                return "dku-icon-snowflake-" + size;
            case "DATABRICKS":
                return "dku-icon-databricks-" + size;
            default:
                return "dku-icon-question-circle-outline-" + size;
        }
    };

    svc.getLLMColorStyle = function(llmId, availableLLMs) {
        const index = availableLLMs.findIndex(llm => llm.id === llmId);
        const backgroundColor = index >= 0 ? COLOR_PALETTES.highContrast[index % COLOR_PALETTES.highContrast.length] : '#BBB';
        return {
            color: $filter('colorContrast')(backgroundColor),
            backgroundColor
        };
    };

    svc.getMainInputNames = function(inputs) {
        return (inputs || []).map(input => input.name);
    }

    svc.getContentType = function(path) {
        let contentType = 'image/jpeg';
        if (path.endsWith('.png')) {
            contentType = 'image/png';
        }
        return contentType;
    }

    svc.getImageUrl = function(path, folderId) {
        if (!path && !folderId) return null;

        return `/dip/api/managedfolder/preview-image?contextProjectKey=${encodeURIComponent($stateParams.projectKey)}&projectKey=${encodeURIComponent($stateParams.projectKey)}&odbId=${encodeURIComponent(folderId)}&itemPath=${encodeURIComponent(path)}&contentType=${encodeURIComponent(svc.getContentType(path))}`;
    }

    svc.getPromptOutputClass = function(response) {
        let status = '';
        if (!response || !response.validationStatus) {
            status = 'not-run'
        } else if (response.error) {
            status = 'invalid'
        } else if (response.validationStatus === 'VALID') {
            status = 'valid';
        } else if (response.validationStatus === 'INVALID') {
            status = 'invalid';
        } else if (response.validationStatus === 'NOT_PERFORMED') {
            status = 'no-validation';
        }

        return 'prompt-studio-test-case__output--' + status;
    }

    svc.getChatOutputClass = function(message) {
        let status = '';
        if (message.error) {
            status = 'invalid'
        } else {
            status = 'valid';
        }
        return 'prompt-chat__assistant-message--' + status;
    }

    svc.addStatusClassToResponses = function(responses) {
        if (responses && responses.length) {
            responses.forEach(response => response.statusClass = svc.getPromptOutputClass(response));
        }
    }

    svc.buildCurrentChatBranch = function(messages, lastMessageId){
        let currentBranch = [];
        if (!messages || Object.keys(messages).length === 0) {
            return currentBranch;
        }
        let currentMessage = messages[lastMessageId];
        while (currentMessage && currentMessage.parentId) {
            currentBranch.push(currentMessage);
            try {
                if (currentMessage.message.content){
                    currentMessage.message.$markdownContent = $dkuSanitize(marked(currentMessage.message.content, {
                        breaks: true
                    }));
                }
            } catch(e) {
                Logger.error('Error parsing markdown HTML, switching to plain text', e);
                currentMessage.message.$markdownContent = null;
            }
            currentMessage = messages[currentMessage.parentId];
        }
        let previousAssistantMessage;
        currentBranch =  currentBranch.reverse();
        Object.entries(currentBranch).forEach(([key, message]) => {
            if (message.message && message.message.role === 'assistant'){
                if (previousAssistantMessage){
                    message.$changedSettings = svc.compareLlmSettings(message, previousAssistantMessage);
                }
                previousAssistantMessage = message;
            }
        });
        return currentBranch;
    };
        
    svc.compareLlmSettings = function(newMessage, oldMessage){
        let changes = [];
        const compareAndPush = (newVal, oldVal, label) => {
            if (newVal !== oldVal) {
                if(["System message", "Response Format"].includes(label)){
                    changes.push(`${label} has changed`);
                }
                else{
                    changes.push(`${label} has changed from ${oldVal ?? "not defined"} to ${newVal ?? "not defined"}`);
                }
            }
        };
        
        compareAndPush(newMessage.llmStructuredRef.id, oldMessage.llmStructuredRef.id, "Model");
        compareAndPush(newMessage.systemMessage, oldMessage.systemMessage, "System message");
        const settings = {
            "temperature": "Temperature",
            "topK": "TopK",
            "topP": "TopP",
            "maxOutputTokens": "Max Output Tokens",
            "frequencyPenalty": "Frequency Penalty",
            "presencePenalty": "Presence Penalty",
        };
        
        Object.entries(settings).forEach(([key, label]) => {
            compareAndPush(newMessage.completionSettings?.[key], oldMessage.completionSettings?.[key], label);
        });
        compareAndPush(newMessage.completionSettings.responseFormat?.type, oldMessage.completionSettings.responseFormat?.type, "Response Format");
        return changes;
    };

    svc.enrichChatMessages = function(messages) {
        const enrichedMessages = angular.copy(messages);
        Object.entries(enrichedMessages).forEach(([key, message]) => {
            message.statusClass = svc.getChatOutputClass(message);
            const parentId = message.parentId;
            if (parentId) {
                if (!enrichedMessages[parentId].childrenIds) {
                    enrichedMessages[parentId].childrenIds = [];
                }
                enrichedMessages[parentId].childrenIds[message.version] = message.id;
            }
        });
        return enrichedMessages;
    };

    return svc;
});

// Keep in sync with LLMType (PromptStudio.java)
// Also used in settings_features.js (pretrainedModelSelector)
app.filter("niceLLMType", function() {
    return function(rawType) {
        switch(rawType) {
        case 'CUSTOM':
            return 'Custom LLM';
        case 'DATABRICKS':
            return 'Databricks Mosaic AI';
        case 'BEDROCK':
            return 'AWS Bedrock';
        case 'OPENAI':
            return 'OpenAI';
        case 'AZURE_OPENAI_MODEL':
            return 'Azure OpenAI Model';
        case 'AZURE_OPENAI_DEPLOYMENT':
            return 'Azure OpenAI Deployment';
        case 'VERTEX':
            return 'Vertex Generative AI';
        case 'COHERE':
            return 'Cohere';
        case 'MISTRALAI':
            return 'Mistral AI';
        case 'MOSAICML':
            return 'MosaicML';
        case 'ANTHROPIC':
            return 'Anthropic';
        case 'HUGGINGFACE_TRANSFORMER_LOCAL':
            return 'Hugging Face local models';
        case 'HUGGINGFACE_API':
            return 'Hugging Face API';
        case 'SAVED_MODEL_AGENT':
            return 'Agent';
        case 'SAVED_MODEL_FINETUNED_OPENAI':
            return 'Fine-tuned OpenAI Saved Model';
        case 'SAVED_MODEL_FINETUNED_AZURE_OPENAI':
            return 'Fine-tuned Azure OpenAI Saved Model';
        case 'SAVED_MODEL_FINETUNED_VERTEX':
            return 'Fine-tuned Vertex Generative AI Saved Model';
        case 'SAVED_MODEL_FINETUNED_HUGGINGFACE_TRANSFORMER':
            return 'Fine-tuned Hugging Face transformer Saved Model';
        case 'SAVED_MODEL_FINETUNED_BEDROCK':
            return 'Fine-tuned AWS Bedrock Saved Model';
        case "RETRIEVAL_AUGMENTED":
            return "Retrieval-Augmented";
        case "SAGEMAKER_GENERICLLM":
            return "SageMaker endpoint";
        case "AZURE_LLM":
            return "Azure LLM endpoint";
        case "SNOWFLAKE_CORTEX":
            return "Snowflake Cortex";
        default:
            return rawType;
        }
    }
});

app.controller('NewPromptModalController', function($scope, Logger, TaggingService, PromptUtils, WT1) {
    WT1.event("prompt-studio-new-prompt-modal-open");

    $scope.uiState = {
        selectedTemplate: { id: "sample-blank", name: "Blank template", icon: "dku-icon-edit-24" },
    };

    $scope.promptTypes = [
        {
            id: 'PROMPT_ENGINEERING', desc: 'Design and test prompts', title: 'Prompt engineering', promptModes:
                [
                    { id: 'PROMPT_TEMPLATE_STRUCTURED', icon: 'dku-icon-text-field-20', desc: 'Specify the task, input details, and optionally examples. Dataiku will generate the final prompt for you.', title: 'Managed mode' },
                    { id: 'PROMPT_TEMPLATE_TEXT', icon: 'dku-icon-code-20', desc: 'Create the prompt using placeholders, giving you complete control over the final prompt sent to the LLM.', title: 'Advanced mode' }
                ]
        },
        {
            id: 'TEST_CONVERSATION', desc: 'Cannot be turned into a recipe', title: 'Test conversation', promptModes:
                [
                    { id: 'CHAT', icon: 'dku-icon-comment-20', desc: 'Run multi-turn chat sessions with several messages.', title: 'Chat' },
                    { id: 'RAW_PROMPT', icon: 'dku-icon-edit-20', desc: 'Write a message and get a response.', title: 'Prompt without inputs' },
                ]
        },
    ];

    $scope.selectMode = function(mode) {
        $scope.uiState.promptMode = mode;
        $scope.uiState.source = (mode === 'RAW_PROMPT' || mode === 'CHAT') ? '' : 'DATASET';
        $scope.uiState.datasetRef = '';

        if ($scope.activePromptStudioPrompt) {
            /* Shortcut: reuse LLM from active prompt if there is already one */
            $scope.uiState.llmId = $scope.activePromptStudioPrompt.llmId;

            if (mode !== 'RAW_PROMPT' && mode !== 'CHAT') {
                $scope.uiState.datasetRef = $scope.activePromptStudioPrompt.dataset;
                $scope.uiState.source = $scope.activePromptStudioPrompt.prompt.promptTemplateQueriesSource || $scope.uiState.source;
            }
        }
    }

    $scope.selectPromptTemplate = function (sample) {
        $scope.uiState.selectedTemplate = $scope.templates.find(template => template.name == sample);
    };

    //TODO - Before merging Chat feature banch remove the icons from this list if we still don't use them
    $scope.templates = [
        { id: "sample-blank", name: "Blank template" },
        { id: "sample-extract-technical-specifications", name: "Extract technical specifications" },
        { id: "sample-summarize-product-review", name: "Analyze product reviews" },
        { id: "sample-support-requests-classification", name: "Classify support requests" },
        { id: "sample-contract-analysis", name: "Analyze contract text" },
        { id: "sample-align-language-level", name: "Improve text level language" },
        { id: "sample-chain-of-thought", name: "Chain-of-thought prompt" },
    ];

    $scope.accept = function() {
        const newPromptStudioPrompt = PromptUtils.createNewPrompt($scope.uiState.promptMode, $scope.uiState.llmId, {}, $scope.uiState.source, $scope.uiState.datasetRef, $scope.uiState.selectedTemplate.id);
        Logger.info("Adding prompt", newPromptStudioPrompt);

        $scope.addPrompt(newPromptStudioPrompt);
        $scope.dismiss();
    };

    $scope.createButtonDisabledReason = function() {
        return !$scope.uiState.promptMode;
    };
});

app.controller('PromptStudioController', function ($scope, $state, $stateParams, $q, $timeout, WT1, DataikuAPI, PromptUtils, TopNav, DatasetUtils, FutureProgressModal,
    FutureWatcher, Debounce, CreateModalFromTemplate, TaggingService, Dialogs, localStorageService, CodeMirrorSettingService, Logger, ClipboardUtils, $dkuSanitize, $rootScope) {
    const BASE_KEY = 'promptstudio.prompt.' + $stateParams.promptStudioId;
    const LOCAL_STORAGE_KEY_MENU = BASE_KEY + '.menuExpanded';
    const LOCAL_STORAGE_KEY_ACTIVE_PROMPT = BASE_KEY + '.active-prompt-id';

    $scope.uiState = {};
    $scope.isMenuExpanded = JSON.parse(localStorageService.get(LOCAL_STORAGE_KEY_MENU) || true);

    $scope.getInputs = PromptUtils.getInputs;
    $scope.formatOutput = PromptUtils.formatOutput;
    $scope.getPromptText = PromptUtils.getPromptText;
    $scope.getLLMColorStyle = PromptUtils.getLLMColorStyle;
    $scope.getMainInputNames = PromptUtils.getMainInputNames;

    $scope.outputClass = '';
    $scope.chatSessionMap = {};

    let savedStudio;
    $scope.setPromptStudio = function(promptStudio) {
        $scope.promptStudio = promptStudio;
        savedStudio = angular.copy(promptStudio);
        $scope.tagsMap = retrieveTagMap();
    };
    $scope.promptDatasetPreview = null;
    $scope.forceDatasetPreviewUpdate = false;

    $scope.promptStudioIsDirty = function() {
        return !angular.equals(savedStudio, $scope.promptStudio);
    };

    $scope.promptIsDirty = function(promptStudioPrompt) {
        if (!savedStudio) {
            return;
        }

        const originalPrompt = savedStudio.prompts.find(savedPrompt => savedPrompt.id === promptStudioPrompt.id);
        return !angular.equals(originalPrompt, promptStudioPrompt);
    };

    checkChangesBeforeLeaving($scope, $scope.promptStudioIsDirty, "Changes were made since the last run and will be lost, are you sure you want to leave ?");

    $scope.updateLlmId = function() {
        $scope.$broadcast('update-llm-id');
    };

    $scope.setActivePromptStudioPrompt = function(promptStudioPrompt, runId) {
        $scope.uiState.comparisonModeActive = false;

        $scope.activePromptStudioPrompt = promptStudioPrompt;
        $scope.activeRunId = runId; // for history
        $scope.activeResponse = null;
        $scope.activeResponseDataset = null;
        $scope.promptDatasetPreview = null;
        $scope.isPromptDirty = false;

        if (!$scope.activePromptStudioPrompt) {
            localStorageService.remove(LOCAL_STORAGE_KEY_ACTIVE_PROMPT);
            $state.go('projects.project.promptstudios.promptstudio.prompt', {'promptId': null}, {'location': 'replace', 'notify': false});
            $scope.openNewPromptModal();
            return;
        }
        localStorageService.set(LOCAL_STORAGE_KEY_ACTIVE_PROMPT, $scope.activePromptStudioPrompt.id);
        $state.go('projects.project.promptstudios.promptstudio.prompt', {'promptId': $scope.activePromptStudioPrompt.id}, {'location': 'replace', 'notify': false});

        DataikuAPI.promptStudios.getPromptLastResponse($stateParams.projectKey, $stateParams.promptStudioId, $scope.activePromptStudioPrompt.id).success(function(data){
            /* Only set as last response if it's valid */
            if (data.promptId) {
                $scope.setActiveResponse(data);
                // check if prompt doesn't match run
                $scope.isPromptDirty = $scope.promptIsDirty($scope.activePromptStudioPrompt);
            }

            if ($scope.activePromptStudioPrompt.prompt.promptTemplateQueriesSource === 'DATASET' && (!data.promptId || data.querySource === 'INLINE' || $scope.isPromptDirty)) {
                $scope.getPromptDatasetPreview();
            }
        }).error(setErrorInScope.bind($scope));
    };

    $scope.promptStats = {};
    $scope.setActiveResponse = function(response) {
        $scope.activeResponse = response;
        $scope.activeResponseDataset = response.querySource === 'DATASET' ? $scope.activePromptStudioPrompt.dataset : null;
        $scope.promptDatasetPreview = null;
        $scope.isPromptDirty = false;
        $scope.promptStats[response.promptId] = response.stats;
        if ($scope.activeResponse && $scope.activeResponse.responses) {
            PromptUtils.addStatusClassToResponses($scope.activeResponse.responses);
        }
        if ($scope.activePromptStudioPrompt.prompt.promptMode === 'CHAT' && !($scope.chatSessionMap[$scope.activePromptStudioPrompt.id] || {}).currentlyStreaming) {
            if (response.responses && response.responses.length === 1) {
                prepareActiveChatSession(response.responses[0]);
                scrollChat($scope.activePromptStudioPrompt.id, true);
            } else {
                throw("For Chat mode there should always be one response only.");
            }
        }
    };

    $scope.getPromptDatasetPreview = function() {
        $scope.uiState.invalidPrompt = false;

        if (!$scope.activePromptStudioPrompt.llmId || !$scope.activePromptStudioPrompt.dataset || $scope.activePromptStudioPrompt.prompt.promptTemplateQueriesSource !== 'DATASET') {
            $scope.uiState.invalidPrompt = true;
            return;
        }

        $scope.forceDatasetPreviewUpdate = false;
        $scope.uiState.loadingPrompt = true;

        DataikuAPI.promptStudios.getPromptDatasetPreview($scope.promptStudio, $scope.activePromptStudioPrompt).then(function({ data }) {
            FutureWatcher.watchJobId(data.jobId).then(function({ data }) {
                if (!data.result) return;
                $scope.setPromptDatasetPreview(data.result);
                $scope.uiState.loadingPrompt = false;
            }).catch(err => {
                $scope.uiState.loadingPrompt = false;
                setErrorInScope.bind($scope)(err);
            });
        }).catch(err => {
            $scope.uiState.loadingPrompt = false;
            setErrorInScope.bind($scope)(err);
        });
    }

    $scope.setPromptDatasetPreview = function(preview) {
        $scope.promptDatasetPreview = angular.copy(preview);
        $scope.promptDatasetPreview.hasOutput = false;
        PromptUtils.addStatusClassToResponses($scope.promptDatasetPreview.responses);

        if ($scope.activeResponse && $scope.activeResponse.responses) {
            $scope.promptDatasetPreview.hasOutput = true;
            $scope.promptDatasetPreview.responses = $scope.promptDatasetPreview.responses.map((response, index) => ({
                ...$scope.activeResponse.responses[index],
                mainInputs: response.mainInputs
            }));
        }
    }

    $scope.fillPromptData = function(prompt) {
        DataikuAPI.promptStudios.getPromptLastResponse($stateParams.projectKey, $stateParams.promptStudioId, prompt.id).success(function(data) {
            /* Only set as last response if it's valid */
            $scope.promptStats[data.promptId] = data.stats;

            // add chat messages to each prompt
            if (prompt.prompt && prompt.prompt.promptMode === 'CHAT' && Object.keys(prompt.prompt.chatMessages).length === 0 && data.responses && data.responses.length) {
                prompt.prompt.chatMessages = data.responses[0].chatMessages;
                savedStudio = angular.copy($scope.promptStudio);
            }
        }).error(setErrorInScope.bind($scope));
    };

    $scope.onPromptStarChange = function(prompt, starred) {
        prompt.starred = starred;
        $scope.save();
    };

    $scope.save = function(callback) {
        DataikuAPI.promptStudios.save($scope.promptStudio).success(function(data) {
            callback && callback(data);
            savedStudio = angular.copy($scope.promptStudio);
        }).error(setErrorInScope.bind($scope));
    };

    $scope.showRawPrompt = function() {
        PromptUtils.openRawPromptModal($scope, $scope.activePromptStudioPrompt.prompt, $scope.activePromptStudioPrompt.llmId, {});
    };

    function loadPromptStudio() {
        DataikuAPI.pretrainedModels.listAvailableLLMs($stateParams.projectKey, "GENERIC_COMPLETION").success(function(data){
            $scope.availableLLMs = data.identifiers;
            $scope.$watch("activePromptStudioPrompt.llmId", function(nv, ov) {
                if (!$scope.activePromptStudioPrompt) return;
                $scope.activeLLM = $scope.availableLLMs.find(llm => llm.id === $scope.activePromptStudioPrompt.llmId);
                $scope.activeLLMIcon = {
                    name: PromptUtils.getLLMIcon(($scope.activeLLM || {}).type),
                    style: PromptUtils.getLLMColorStyle($scope.activePromptStudioPrompt.llmId, $scope.availableLLMs)
                };
            });

            WT1.event("prompt-studio-llms-load", {
                llmsCount: data.identifiers.length,
            });
        }).error(setErrorInScope.bind($scope));

        DatasetUtils.listDatasetsUsabilityForAny($stateParams.projectKey).success(data => {
            // Move the usable flag where it's going to be read
            data.forEach(x => {
                x.usable = x.usableAsInput;
                x.usableReason = x.inputReason;
            });
            $scope.availableDatasets = data;
        }).error(setErrorInScope.bind($scope));


        DataikuAPI.promptStudios.get($stateParams.projectKey, $stateParams.promptStudioId).success(function(data) {
            $scope.setPromptStudio(data);

            if (!$scope.promptStudio.prompts.length) {
                $scope.openNewPromptModal();
            }

            TopNav.setItem(TopNav.ITEM_PROMPT_STUDIO, $stateParams.promptStudioId, {
                name : $scope.promptStudio.name
            });
            TopNav.setPageTitle($scope.promptStudio.name + " - Prompt Studio");

            if (!$stateParams.promptId) {
                $stateParams.promptId = localStorageService.get(LOCAL_STORAGE_KEY_ACTIVE_PROMPT);
            }
            const activePromptStudioPrompt = $scope.promptStudio.prompts.find(prompt => prompt.id === $stateParams.promptId) || $scope.promptStudio.prompts[0];
            if (activePromptStudioPrompt) {
                $scope.setActivePromptStudioPrompt(activePromptStudioPrompt);
            }

            WT1.event("prompt-studio-load", {
                promptsCount: $scope.promptStudio.prompts.length,
                activeLLM: activePromptStudioPrompt && activePromptStudioPrompt.llmId
            });

            for (const prompt of $scope.promptStudio.prompts) {
                $scope.fillPromptData(prompt);
            }

        }).error(setErrorInScope.bind($scope));
    };

    $scope.addPrompt = function(prompt) {
        prompt.id = (Math.random().toString(36) + '00000000000000000').slice(2, 12);
        $scope.promptStudio.prompts.push(prompt);
        $scope.setActivePromptStudioPrompt($scope.promptStudio.prompts[$scope.promptStudio.prompts.length - 1]);
    };

    $scope.openNewPromptModal = function() {
        CreateModalFromTemplate("/templates/promptstudios/new-prompt-modal.html", $scope);
    };

    /****** Action icons ******/
    $scope.duplicateActivePrompt = function() {
        const copiedPrompt = angular.copy($scope.activePromptStudioPrompt);
        copiedPrompt.$selected = false;
        $scope.addPrompt(copiedPrompt);
        $scope.save();

        WT1.event("prompt-studio-prompt-duplicate");
    };

    $scope.showActivePromptHistory = function() {
        DataikuAPI.promptStudios.getPromptHistory($stateParams.projectKey, $stateParams.promptStudioId, $scope.activePromptStudioPrompt.id).success(function(data) {

            WT1.event("prompt-studio-prompt-history-open", {
                historyLength: data.entries.length
            });

            CreateModalFromTemplate("/templates/promptstudios/prompt-history.html", $scope, null, function(modalScope) {
                modalScope.history = data;

                modalScope.setActiveRun = function(runEntry) {
                    modalScope.activeRunEntry = runEntry;
                    modalScope.activeRunEntryResponse = null;
                    DataikuAPI.promptStudios.getPromptHistoryResponse($stateParams.projectKey, $stateParams.promptStudioId, $scope.activePromptStudioPrompt.id, runEntry.runId).success(function(data) {
                        modalScope.activeRunEntryResponse = data;
                        PromptUtils.addStatusClassToResponses((modalScope.activeRunEntryResponse || []).responses);
                    }).error(setErrorInScope.bind(modalScope));
                }

                modalScope.revertToActiveRun = function() {
                    WT1.event("prompt-studio-prompt-history-revert-to-past-run");
                    DataikuAPI.promptStudios.revertPrompt($stateParams.projectKey, $scope.promptStudio, $scope.activePromptStudioPrompt.id, modalScope.activeRunEntry.runId).success(function(data) {
                        $scope.setPromptStudio(data);
                        // ensure we use object directly from $scope.promptStudio.prompts
                        const promptStudioPrompt = $scope.promptStudio.prompts.find(prompt => prompt.id === $scope.activePromptStudioPrompt.id);
                        $scope.setActivePromptStudioPrompt(promptStudioPrompt, modalScope.activeRunEntry.runId);
                        modalScope.dismiss();
                    }).error(setErrorInScope.bind(modalScope));
                }
                modalScope.createPromptFromActiveRun = function() {
                    WT1.event("prompt-studio-prompt-history-create-prompt-from-past-run");
                    $scope.addPrompt(angular.copy(modalScope.activeRunEntry.promptStudioPrompt));
                    modalScope.dismiss();
                }
            })
        }).error(setErrorInScope.bind($scope));
    };

    $scope.openConversationHistoryModal = function () {
        CreateModalFromTemplate("/templates/promptstudios/conversation-history-modal.html", $scope, null, function (modalScope) {
            modalScope.uiState = {
                activeTab: "json",
            };

            modalScope.pythonReadOnlyOptions = $scope.codeMirrorSettingService.get('text/x-python');
            modalScope.pythonReadOnlyOptions["readOnly"] = "nocursor";
            modalScope.jsonReadOnlyOptions = $scope.codeMirrorSettingService.get('application/json');
            modalScope.jsonReadOnlyOptions["readOnly"] = "nocursor";

            function getJsonMessages() {
                let messages = PromptUtils.buildCurrentChatBranch($scope.activePromptStudioPrompt.prompt.chatMessages, $scope.activePromptStudioPrompt.prompt.$lastMessageId)
                    .filter(item => !item.error)
                    .map(item => {
                        const {$markdownContent, ...rest} = item.message;
                        return rest;
                    });
                if ($scope.activePromptStudioPrompt.prompt.chatSystemMessage) {
                    let systemMessage = {
                        "role": "system",
                        "content": $scope.activePromptStudioPrompt.prompt.chatSystemMessage
                    };
                    messages.unshift(systemMessage);
                }
                return messages;
            };

            function getPythonMessages() {
                let messages = getJsonMessages();

                let pythonMessages = "import dataiku\n";
                pythonMessages += `LLM_ID = "${$scope.activeLLM.id}"\n`;
                pythonMessages += `llm = dataiku.api_client().get_default_project().get_llm(LLM_ID)\n`;
                pythonMessages += `# Create and run a completion query with your chat history\n`;
                pythonMessages += `chat = llm.new_completion()\n\n`;

                if ($scope.activePromptStudioPrompt.llmSettings['temperature']) {
                    pythonMessages += `chat.settings["temperature"] = ${$scope.activePromptStudioPrompt.llmSettings['temperature']}\n`
                }
                if ($scope.activePromptStudioPrompt.llmSettings['topK']) {
                    pythonMessages += `chat.settings["topK"] = ${$scope.activePromptStudioPrompt.llmSettings['topK']}\n`
                }
                if ($scope.activePromptStudioPrompt.llmSettings['topP']) {
                    pythonMessages += `chat.settings["topP"] = ${$scope.activePromptStudioPrompt.llmSettings['topP']}\n`
                }
                if ($scope.activePromptStudioPrompt.llmSettings['maxOutputTokens']) {
                    pythonMessages += `chat.settings["maxOutputTokens"] = ${$scope.activePromptStudioPrompt.llmSettings['maxOutputTokens']}\n`
                }
                if ($scope.activePromptStudioPrompt.llmSettings['stopSequences'] && $scope.activePromptStudioPrompt.llmSettings['stopSequences'].length > 0) {
                    pythonMessages += `chat.settings["stopSequences"] = ${$scope.activePromptStudioPrompt.llmSettings['stopSequences']}\n`
                }
                if ($scope.activePromptStudioPrompt.llmSettings['presencePenalty']) {
                    pythonMessages += `chat.settings["presencePenalty"] = ${$scope.activePromptStudioPrompt.llmSettings['presencePenalty']}\n`
                }
                if ($scope.activePromptStudioPrompt.llmSettings['frequencyPenalty']) {
                    pythonMessages += `chat.settings["frequencyPenalty"] = ${$scope.activePromptStudioPrompt.llmSettings['frequencyPenalty']}\n`
                }
                if ($scope.activePromptStudioPrompt.llmSettings['responseFormat']) {
                    pythonMessages += `chat.with_json_output()\n`
                }

                pythonMessages += "\n";
                messages.forEach((message) => {
                    pythonMessages += `chat.with_message(${JSON.stringify(message["content"])}, role=${JSON.stringify(message["role"])})\n`;
                });

                pythonMessages += "\n";
                pythonMessages += `response = chat.execute()`;
                return pythonMessages;
            }

            modalScope.jsonMessages = getJsonMessages();
            modalScope.pythonMessages = getPythonMessages();

            modalScope.copyConversationHistory = function () {
                if (modalScope.uiState.activeTab === "json") {
                    ClipboardUtils.copyToClipboard(JSON.stringify(modalScope.jsonMessages, null, 2), "Conversation copied to clipboard");
                }
                if (modalScope.uiState.activeTab === "python") {
                    ClipboardUtils.copyToClipboard(modalScope.pythonMessages, "Conversation copied to clipboard");
                }
            }
        });
    };

    $scope.removeActivePrompt = function() {
        if (!$scope.activePromptStudioPrompt) return;

        Dialogs.confirmAlert($scope, "Delete current prompt", "Are you sure you want to delete this prompt?").then(function() {
            WT1.event("prompt-studio-prompt-delete");
            $scope.promptStudio.prompts = $scope.promptStudio.prompts.filter(prompt => prompt.id !== $scope.activePromptStudioPrompt.id);
            $scope.setActivePromptStudioPrompt($scope.promptStudio.prompts[0]);
            $scope.save();
        }, function() {
            // modal is closed
        });
    };

    $scope.hasImageInput = function() {
        return PromptUtils.hasImageInput($scope.activePromptStudioPrompt.prompt);
    }

    /* *********************** Initialization code **************** */

    TopNav.setLocation(TopNav.TOP_PROMPT_STUDIOS, 'prompt studios', TopNav.TABS_NONE, null);
    TopNav.setItem(TopNav.ITEM_PROMPT_STUDIO, $stateParams.promptStudioId);

    loadPromptStudio();

    $scope.editSettings = function() {
        WT1.event("prompt-studio-prompt-settings-edit");

        CreateModalFromTemplate("/templates/promptstudios/prompt-settings-modal.html", $scope, null, function(modalScope) {
            modalScope.uiState = {
                activeTab: "settings",
                llmSettings: angular.copy($scope.activePromptStudioPrompt.llmSettings),
                validationSettings: angular.copy($scope.activePromptStudioPrompt.prompt.resultValidation),
                guardrailsPipelineSettings: angular.copy($scope.activePromptStudioPrompt.prompt.guardrailsPipelineSettings),
                streamingDisabled: $scope.activePromptStudioPrompt.prompt.streamingDisabled,
            };

            if (!modalScope.uiState.guardrailsPipelineSettings) {
                modalScope.uiState.guardrailsPipelineSettings = {}
            }
            if (!modalScope.uiState.guardrailsPipelineSettings.guardrails) {
                modalScope.uiState.guardrailsPipelineSettings.guardrails = []
            }

            modalScope.forms = {};

            modalScope.saveSettings = function() {
                $scope.activePromptStudioPrompt.llmSettings = modalScope.uiState.llmSettings;
                $scope.activePromptStudioPrompt.prompt.resultValidation = modalScope.uiState.validationSettings;
                $scope.activePromptStudioPrompt.prompt.guardrailsPipelineSettings = modalScope.uiState.guardrailsPipelineSettings;
                $scope.activePromptStudioPrompt.prompt.streamingDisabled = modalScope.uiState.streamingDisabled;
                $scope.save(modalScope.dismiss);
            };
        });
    }

    function retrieveTagMap() {
        const promptStudioTags = $scope.promptStudio.prompts.map(prompt => prompt.tags).reduce(function(acc, cur) {
            return acc.concat(cur);
        }, []);
        return TaggingService.fillTagsMapFromArray(promptStudioTags);
    }

    $scope.getPromptStudioTags = function() {
        if (!$scope.promptStudio) return;

        const deferred = $q.defer();
        deferred.resolve(retrieveTagMap());
        return getRewrappedPromise(deferred);
    };

    $scope.$watchCollection('activePromptStudioPrompt.tags', function(nv) {
        if (!nv) return;

        $scope.tagsMap = retrieveTagMap();
    });

    $scope.exportAsRecipe = function() {
        CreateModalFromTemplate("/templates/recipes/nlp/edit-in-recipe.html", $scope, "ExportAsPromptRecipeModalController", function(newScope) {
        });
    };

    /**
     * @return A message saying why the active prompt cannot be saved as a recipe, or return falsy if the active prompt can be saved as a recipe.
     */
    $scope.exportAsRecipeDisabledReason = function() {
        if (!$scope.activePromptStudioPrompt || !$scope.activePromptStudioPrompt.prompt) {
            return "No active prompt";
        }
        if ($scope.activePromptStudioPrompt.prompt.promptMode === 'RAW_PROMPT') {
            return "Single-shot prompts cannot be saved as a recipe";
        }
        if ($scope.activePromptStudioPrompt.prompt.promptMode === 'CHAT') {
            return "Chat cannot be saved as a recipe";
        }
        if ($scope.activePromptStudioPrompt.prompt.promptTemplateQueriesSource !== 'DATASET') {
            return "Prompt must use an input dataset to be saved as a recipe"
        }
    };

    $scope.getPromptIcon = function(promptStudioPrompt, size) {
        const llm = ($scope.availableLLMs || []).find(l => promptStudioPrompt && l.id === promptStudioPrompt.llmId);
        return PromptUtils.getLLMIcon(llm ? llm.type : '', size);
    };

    $scope.addExample = function() {  // for structured prompts only
        $scope.activePromptStudioPrompt.prompt.structuredPromptExamples.push({
            inputs: Array(PromptUtils.getInputs($scope.activePromptStudioPrompt.prompt).length).fill(''),
            output: '',
        });
    };

    $scope.addFilledExample = function(inputs, output) {  // for structured prompts only
        $scope.activePromptStudioPrompt.prompt.structuredPromptExamples.push({
            inputs: PromptUtils.getInputs($scope.activePromptStudioPrompt.prompt).map((_, index) => inputs[index]),
            output: output || '',
        });

        $timeout(() => document.getElementById('prompt-example-output-' + ($scope.activePromptStudioPrompt.prompt.structuredPromptExamples.length - 1)).focus());
    };

    // for written test cases
    $scope.addFilledExampleInline = function(query, response) {
        const { inputs } = PromptUtils.getInlinePromptInputMap($scope.activePromptStudioPrompt.prompt, query);

        $scope.addFilledExample(inputs, PromptUtils.formatOutput(response))
    };

    $scope.getDisplayedLLMOutput = function (response) {
        if (!response) {
            return null;
        }
        if (response.error) {
            return response.llmError;
        }
        if (response.validationStatus === 'INVALID') {
            return response.formattedOutput || response.rawLLMOutput;
        }
        if (['NOT_PERFORMED', 'VALID'].includes(response.validationStatus)) {
            return response.formattedOutput;
        }
        return null;
    };

    $scope.viewRowRawPrompt = function(row, columns, originalPrompt) {
        const rawInputs = {}
        for (let i = 0; i < row.length; i++) {
            rawInputs[columns[i]] = row[i];
        }
        PromptUtils.openRawPromptModal($scope, originalPrompt.prompt, originalPrompt.llmId, rawInputs);
    };

    // for rows that have already been run and not altered by prompt change
    $scope.viewRowRawPromptFromRun = function(row, response) {
        DataikuAPI.promptStudios.getPromptHistory($stateParams.projectKey, $stateParams.promptStudioId, $scope.activePromptStudioPrompt.id).then(function({data}) {
            const runId = response.runId;
            const entry = data.entries.find(entry => entry.runId === runId);

            if (entry) {
                return $scope.viewRowRawPrompt(row, PromptUtils.getMainInputNames(response.mainPromptTemplateInputs), entry.promptStudioPrompt);
            }
        }).catch(setErrorInScope.bind($scope));
    };

    // for written test cases
    $scope.viewRowRawPromptInline = function(query) {
        const { inputNames, inputs } = PromptUtils.getInlinePromptInputMap($scope.activePromptStudioPrompt.prompt, query);

        $scope.viewRowRawPrompt(inputs, inputNames, $scope.activePromptStudioPrompt);
    }

    $scope.runActivePromptIfAbleTo = function() {
        if ($scope.activePromptStudioPrompt.prompt.promptMode !== 'CHAT' && !$scope.runButtonDisabledReason()) {
            $scope.runActivePrompt();
        }
    };

    $scope.runActivePrompt = function() {
        const before = new Date().getTime();
        DataikuAPI.promptStudios.startExecutePrompt($scope.promptStudio, $scope.activePromptStudioPrompt).success(function(data) {
            FutureProgressModal.show($scope, data, "Computing your prompt").then(function(result) {
                if (!result) return; // user abort

                const after = new Date().getTime();

                WT1.event("prompt-studio-prompt-execute-done", {
                    totalTimeMS: after-before,
                    llmId: $scope.activePromptStudioPrompt.llmId,
                    promptMode: $scope.activePromptStudioPrompt.prompt.promptMode,
                    llmCompletionSettings: $scope.activePromptStudioPrompt.llmSettings,
                    querySource: result.querySource,
                    estimatedCostPer1KRecords: result.stats.estimatedCostPer1KRecords,
                    testedRecords: result.stats.testedRecords,
                    averagePromptTokens: result.responses.map(r => r.promptTokens).reduce((a,b) => a+b, 0) / result.stats.testedRecords,
                    averageCompletionTokens: result.responses.map(r => r.completionTokens).reduce((a,b) => a+b, 0) / result.stats.testedRecords
                });

                $scope.setActiveResponse(result);

                savedStudio = angular.copy($scope.promptStudio);
            });
        }).error(setErrorInScope.bind($scope));
    };

    $scope.runActiveChat = function(promptId, isRerun) {
        $scope.chatSessionMap[promptId] = {
            streamedMessage: '',
            currentlyStreaming: true,
            useMarkdown: true,
            streamingSupported: true,
            streamingNotSupportedMessage: "",
            abortController:  new AbortController(),
            isRerun
        };

        const CHAR_LIMIT_FOR_MARKDOWN = 500;
        const THROTTLE_MIN_LENGTH = 10000;
        let previousLength = 0;
        let rawText = '';
        const chunkCallback = chunk => {
            scrollChat(promptId, false);

            const event = chunk['event'];
            const data = chunk['data'];
            switch(event) {
                case 'completion-chunk': {
                    if (data.text) {
                        const text = data.text;
                        rawText += text;
                        let currentLength = rawText.length;
                        const convertToMarkdown = currentLength < THROTTLE_MIN_LENGTH || (currentLength > THROTTLE_MIN_LENGTH && currentLength > previousLength + CHAR_LIMIT_FOR_MARKDOWN);

                        // throttle text streaming if the string is too long
                        if ($scope.chatSessionMap[promptId].useMarkdown && convertToMarkdown) {
                            // if there is an error parsing the HTML (e.g., LLM is spitting out garbage text), stop using markdown
                            try {
                                $scope.chatSessionMap[promptId].streamedMessage = $dkuSanitize(marked(rawText, {
                                    breaks: true
                                }));
                                $scope.$apply();
                                previousLength = currentLength;
                            } catch(e) {
                                Logger.error('Error parsing markdown HTML, switching to plain text', e);
                                $scope.chatSessionMap[promptId].useMarkdown = false;
                            }
                        }

                        if (!$scope.chatSessionMap[promptId].useMarkdown) {
                            $scope.chatSessionMap[promptId].streamedMessage = rawText;
                            $scope.$apply();
                        }

                        scrollChat(promptId, false);
                    }
                    break;
                } case 'completion-response': {
                    $scope.chatSessionMap[promptId] = {
                        streamedMessage: '',
                        currentlyStreaming: false,
                        isRerun: false
                    };

                    // only add the message if we are actually still on the same session
                    if ($scope.activePromptStudioPrompt.id === promptId) {
                        prepareActiveChatSession(data.responses[0]);
                        scrollChat(promptId, false);
                    }
                    break;
                } case 'no-streaming': {
                    $scope.chatSessionMap[promptId].streamingSupported = false;
                    $scope.chatSessionMap[promptId].streamingNotSupportedMessage = data["text"];
                    $scope.$apply();
                    // TODO : provide info to the user that the current settings don't allow for streaming. Waiting for a Figma proposal
                    break;
                } default: {
                    Logger.error('Unknown chunk event: ' + event);
                }
            }
        };

        $timeout(() => {
            scrollChat(promptId, true);
        });

        DataikuAPI.promptStudios.streamedCompletion($scope.promptStudio, $scope.activePromptStudioPrompt, chunkCallback, $scope.chatSessionMap[promptId].abortController).then(function(response) {
            savedStudio = angular.copy($scope.promptStudio);
        }).catch(setErrorInScope.bind($scope));
    };

    $scope.runButtonDisabledReason = function() {
        if (!$scope.activePromptStudioPrompt.llmId || !$scope.activeLLM) {
            return "Select a LLM";
        }
        if ($scope.activePromptStudioPrompt.prompt.promptTemplateQueriesSource === 'DATASET' && !$scope.activePromptStudioPrompt.dataset) {
            return "Select a dataset";
        }
        if ($scope.activePromptStudioPrompt.prompt.promptTemplateQueriesSource === 'INLINE' && !($scope.activePromptStudioPrompt.inlinePromptTemplateQueries || []).length) {
            return "Add queries before running your prompt";
        }
        if ($scope.activePromptStudioPrompt.prompt.promptMode === 'RAW_PROMPT' && !$scope.activeLLM.promptDriven) {
            return "Select a LLM that supports single-shot prompts";
        }
        if ($scope.activePromptStudioPrompt.prompt.promptMode === 'PROMPT_TEMPLATE_STRUCTURED' && $scope.activePromptStudioPrompt.prompt.structuredPromptPrefix === '') {
            return "Write a prompt";
        }
        if ($scope.activeLLM.type === 'RETRIEVAL_AUGMENTED' && PromptUtils.hasNoUserMessage($scope.activePromptStudioPrompt.prompt)) {
            return "Add an input to run a RAG prompt";
        }
        return '';
    };

    $scope.clearActiveAnswer = function() {
        Dialogs.confirm($scope, "Delete last response", "Are you sure you want to delete the last response for this prompt?").then(function() {
            DataikuAPI.promptStudios.clearLastResponse($stateParams.projectKey, $scope.promptStudio.id, $scope.activePromptStudioPrompt.id).success(function() {
                $scope.activeResponse = null;
                $scope.activeResponseDataset = null;
                delete $scope.promptStats[$scope.activePromptStudioPrompt.id];
                $scope.getPromptDatasetPreview();
            }).error(setErrorInScope.bind($scope));
        }, function() {
            // modal is closed
        });
    };

    $scope.getPrompt = id => {
        return $scope.promptStudio.prompts.find(p => p.id === id);
    };

    $scope.allPromptsAreSelected = function() {
        if (!$scope.promptStudio || !$scope.promptStudio.prompts.length) return;

        return $scope.promptStudio.prompts.every(p => p.$selected);
    };

    $scope.numberOfPromptsSelected = function () {
        if ($scope.promptStudio && $scope.promptStudio.prompts) {
            return $scope.promptStudio.prompts.filter(p => p.$selected).length;
        } else {
            return 0;
        }
    };

    $scope.cannotCompareReason = function() {
        if ($scope.numberOfPromptsSelected() < 2) {
            return "Cannot compare: Needs at least 2 prompts selected";
        }
        const selectedPrompts = $scope.promptStudio.prompts.filter(p => p.$selected);

        const hasChatPrompt = selectedPrompts.some(p => p.prompt.promptMode === 'CHAT');
        if (hasChatPrompt) {
            return 'Cannot compare: Chat not supported'
        }

        const hasPromptWithoutInputs = selectedPrompts.some(p => p.prompt.promptMode === "RAW_PROMPT");
        const hasPromptWithInputs = selectedPrompts.some(p => p.prompt.promptMode !== "RAW_PROMPT");
        if (hasPromptWithInputs && hasPromptWithoutInputs) {
            return "Cannot compare a prompt without inputs and a prompt with inputs.";
        }

        const firstNoDatasetPrompt = selectedPrompts.findIndex(p => p.prompt.promptTemplateQueriesSource === "DATASET" && !p.dataset);
        if (firstNoDatasetPrompt !== -1) {
            return "Cannot compare: Selected prompt " + firstNoDatasetPrompt + " has no input dataset selected";
        }
        const allInlineInputs = selectedPrompts.filter(p => p.prompt.promptTemplateQueriesSource === "INLINE").flatMap(p => p.prompt.promptTemplateInputs.map(input => input.name));
        const selectedPromptInputs = selectedPrompts.map(p => p.prompt.promptTemplateInputs.map(input => input.name))
        const firstMissingInput = allInlineInputs.findIndex(inputName => !selectedPromptInputs.every(inputs => inputs.includes(inputName)))
        if (firstMissingInput !== -1) {
            return "Cannot compare: Not all prompts have an input column named " + allInlineInputs[firstMissingInput];
        }
        return '';
    };

    $scope.massSelect = () => {
        if ($scope.allPromptsAreSelected()) {
            $scope.promptStudio.prompts.forEach(p => p.$selected = false);
        } else {
            $scope.promptStudio.prompts.forEach(p => p.$selected = true);
        }
    };

    $scope.compareSelectedPrompts = function() {
        $scope.save(compareSelectedPrompts);
    };

    function compareSelectedPrompts() {
        const promptsToCompare = $scope.promptStudio.prompts.filter(p => p.$selected);

        const before = new Date().getTime();
        DataikuAPI.promptStudios.startExecuteComparison($scope.promptStudio, promptsToCompare.map(p => p.id)).success(function(data) {
            FutureProgressModal.show($scope, data, "Computing prompt comparison").then(function(result) {
                $scope.uiState.comparisonModeActive = true;
                $scope.activeComparisonResponse = result;
                ($scope.activeComparisonResponse || {}).promptsResponses.forEach(comparisonResponse => PromptUtils.addStatusClassToResponses(comparisonResponse.responses));

                const after = new Date().getTime();

                WT1.event("prompt-studio-comparison-execute-done", {
                    totalTimeMS: after-before,
                    comparedPrompts: promptsToCompare.length
                });

            });
        }).error(setErrorInScope.bind($scope));
    }

    $scope.changeDataset = function(dataset, queriesSource) {
        $scope.activePromptStudioPrompt.dataset = dataset || $scope.activePromptStudioPrompt.dataset;
        $scope.activePromptStudioPrompt.prompt.promptTemplateQueriesSource = queriesSource;
        $scope.forceDatasetPreview();
    }

    $scope.toggleMenu = (isOpen) => {
        $scope.isMenuExpanded = isOpen;
        localStorageService.set(LOCAL_STORAGE_KEY_MENU, isOpen);
    }

    $scope.updateSampleSettings = (settings) => {
        $scope.activePromptStudioPrompt.nbRows = settings.nbRows;
        $scope.getPromptDatasetPreview();
    };

    $scope.forceDatasetPreview = () => $scope.forceDatasetPreviewUpdate = true;

    $scope.sendChatMessage = (newMessage, isRerun) => {
        $scope.activePromptStudioPrompt.prompt.lastUserMessage = newMessage;
        $scope.runActiveChat($scope.activePromptStudioPrompt.id, isRerun);
    };

    $scope.stopChatStreaming = () => {
        let promptId =  $scope.activePromptStudioPrompt.id;
        $scope.chatSessionMap[promptId].abortController.abort();
        $timeout(function(){
            DataikuAPI.promptStudios.getPromptLastResponse($stateParams.projectKey, $stateParams.promptStudioId, promptId).success(function(data) {
                prepareActiveChatSession(data.responses[0]);
                scrollChat(promptId, false);
                $scope.chatSessionMap[promptId] = {
                    streamedMessage: '',
                    currentlyStreaming: false
                };
            }).error(setErrorInScope.bind($scope));
        },600)
    }

    $scope.forkMessage = (messages, userMessage) => {
        const sourcePromptId = $scope.activePromptStudioPrompt.id;
        const copiedPrompt = angular.copy($scope.activePromptStudioPrompt);
        copiedPrompt.$selected = false;
        copiedPrompt.$selected = false;
        copiedPrompt.prompt.chatMessages = {};
        $scope.addPrompt(copiedPrompt);

        DataikuAPI.promptStudios.forkSession($scope.promptStudio, copiedPrompt, sourcePromptId, userMessage.id).then(function({ data }) {
            copiedPrompt.prompt.chatMessages = data.responses[0].chatMessages;
            copiedPrompt.prompt.enrichedChatMessages = PromptUtils.enrichChatMessages(copiedPrompt.prompt.chatMessages);
            copiedPrompt.prompt.$lastMessageId = userMessage.parentId;
            $scope.chatSessionMap[copiedPrompt.id] = $scope.chatSessionMap[copiedPrompt.id] || {};
            $scope.chatSessionMap[copiedPrompt.id].defaultMessage = userMessage.message.content;
        }).catch(setErrorInScope.bind($scope));
    };

    $scope.$watch('[activePromptStudioPrompt, activeRunId]', Debounce().withDelay(200, 200).wrap(function([newPrompt, newRunId], [oldPrompt, oldRunId]) {
        // do not mark as stale when switching between prompts, loading page, history or if number of sample rows changes
        if (oldPrompt === undefined || newPrompt?.id !== oldPrompt.id || newRunId !== oldRunId || newPrompt.nbRows !== oldPrompt.nbRows) return;

        if (shouldFetchDatasetPreview(newPrompt, oldPrompt)) {
            $scope.isPromptDirty = true;
            $scope.getPromptDatasetPreview();
        }
    }), true);

    function shouldFetchDatasetPreview(nv, ov) {
        return $scope.forceDatasetPreviewUpdate ||
            (!$scope.promptDatasetPreview && !angular.equals(ov.prompt, nv.prompt)); // if the prompt has changed but we are still showing the last run
    }

    $scope.$watch('activeLLM', function(nv, ov) {
        // if we go from an LLM that supports vision to one that doesn't and vice versa, we should update the inputs
        if (nv && ov && nv.supportsImageInputs !== ov.supportsImageInputs) {
            PromptUtils.updateInputsOnLLMChange($scope.activePromptStudioPrompt.prompt, $scope.activeLLM);

            // update dataset preview inputs
            PromptUtils.getInputs($scope.activePromptStudioPrompt.prompt).forEach((input, index) => {
                if ($scope.promptDatasetPreview && $scope.promptDatasetPreview.mainPromptTemplateInputs) {
                    $scope.promptDatasetPreview.mainPromptTemplateInputs[index].type = input.type;
                }
            });
        }
    }, true);

    function scrollChat(promptId, forceScroll) {
        // only scroll if the active session is streaming
        if ($scope.activePromptStudioPrompt.id === promptId) {
            const containerEl = document.getElementById('prompt-chat');
            if (containerEl) {
                const SCROLL_BUFFER = 50; // don't autoscroll if more than SCROLL_BUFFER pixels from bottom
                // only scroll to bottom if forceScroll is true or if we are already at the bottom
                if (forceScroll || (containerEl.scrollTop >= containerEl.scrollHeight - containerEl.offsetHeight - SCROLL_BUFFER)) {
                    containerEl.scrollTop = containerEl.scrollHeight;
                }
            }
        }
    }

    function prepareActiveChatSession(response) {
        // $lastMessageId is prefixed with '$' so it's ignored during angular.equals comparison between current and saved promptStudio
        $scope.activePromptStudioPrompt.prompt.$lastMessageId = response.lastMessageId;
        $scope.activePromptStudioPrompt.prompt.chatMessages = response.chatMessages;
        $scope.activePromptStudioPrompt.prompt.enrichedChatMessages = PromptUtils.enrichChatMessages(response.chatMessages);

        savedStudio = angular.copy($scope.promptStudio);
    }
});

app.controller("ExportAsPromptRecipeModalController", function($scope, $stateParams, $state, DataikuAPI, WT1) {
    $scope.recipeSettings = {};

    DataikuAPI.flow.recipes.list($stateParams.projectKey).then(function({data}) {
        $scope.promptRecipes = data.filter(recipe => recipe.type === "prompt");
        $scope.recipeSettings.create = !$scope.promptRecipes.length;
    }).catch(setErrorInScope.bind($scope));

    WT1.event("prompt-studio-prompt-save-as-recipe-start");

    $scope.confirm = function() {
        if ($scope.recipeSettings.create) {
            $scope.save(function() { createNewRecipe() });
        } else {
            $scope.save(function() { overwriteRecipe() });
        }
    };

    function overwriteRecipe() {
        DataikuAPI.promptStudios.overwriteRecipe($stateParams.projectKey, $scope.recipeSettings.recipeName, $scope.promptStudio.id, $scope.activePromptStudioPrompt.id)
        .then(function() {
            $state.go("projects.project.recipes.recipe", { projectKey: $stateParams.projectKey, recipeName: $scope.recipeSettings.recipeName });
        }).catch(setErrorInScope.bind($scope));
    }

    function createNewRecipe() {
        DataikuAPI.promptStudios.getRecipePayloadFromPrompt($stateParams.projectKey, $scope.promptStudio.id, $scope.activePromptStudioPrompt.id).then(function({data}) {
            const recipePayload = data;
            $scope.showCreatePromptModal($scope.activePromptStudioPrompt.dataset, $scope.activePromptStudioPrompt.prompt.imageFolderId, recipePayload, $scope.getRelevantZoneId($scope.$stateParams.zoneId));
        }).catch(setErrorInScope.bind($scope));
    };
});


app.controller("PromptStudiosListController", function($scope, $controller, $stateParams, DataikuAPI, $state, TopNav, CreateModalFromTemplate) {
    $controller('_TaggableObjectsListPageCommon', {$scope: $scope});

    $scope.sortBy = [
        { value: 'name', label: 'Name' },
        { value: '-lastModifiedOn', label: 'Last modified'}
    ];

    $scope.selection = $.extend({
        filterQuery: {
            userQuery: '',
            tags: [],
            interest: {
                starred: '',
            },
            inputDatasetSmartName: []
        },
        filterParams: {
            userQueryTargets: ["name", "tags"],
            propertyRules: {tag: 'tags'},
        },
        orderQuery: "-lastModifiedOn",
        orderReversed: false
    }, $scope.selection || {});

    $scope.list = function() {
        DataikuAPI.promptStudios.listHeads($stateParams.projectKey).success(function(data) {
            $scope.listItems = data;
            $scope.restoreOriginalSelection();
        }).error(setErrorInScope.bind($scope));
    };

    TopNav.setLocation(TopNav.TOP_PROMPT_STUDIOS, TopNav.LEFT_PROMPT_STUDIOS, TopNav.TABS_NONE, null);
    TopNav.setNoItem();
    $scope.list();

    /* Tags handling */

    $scope.$on('selectedIndex', function(e, index){
        // an index has been selected, we unselect the multiselect
        $scope.$broadcast('clearMultiSelect');
    });

    /* Specific actions */
    $scope.goToItem = function(data) {
        $state.go("projects.project.promptstudios.promptstudio", {projectKey : $stateParams.projectKey, promptStudioId : data.id});
    };

    $scope.newPromptStudio = function() {
        CreateModalFromTemplate('/templates/promptstudios/new-prompt-studio-modal.html', $scope, 'NewPromptStudioController');
    };
});

app.controller('NewPromptStudioController', function($scope, $state, $stateParams, DataikuAPI, StringUtils) {
    $scope.newPromptStudio = {
        name: ''
    };

    if ($scope.listItems) {
        setDefaultPromptStudioName($scope.listItems);
    } else {
        DataikuAPI.promptStudios.listHeads($stateParams.projectKey).success(function(data) {
            setDefaultPromptStudioName(data);
        }).error(setErrorInScope.bind($scope));
    }

    $scope.create = function() {
        resetErrorInScope($scope);
        DataikuAPI.promptStudios.create($stateParams.projectKey, $scope.newPromptStudio.name).success(function(ps) {
            $scope.dismiss();
            $state.go("projects.project.promptstudios.promptstudio", {promptStudioId: ps.id})
        }).error(setErrorInScope.bind($scope));
    };

    function setDefaultPromptStudioName(list) {
        if (!$scope.newPromptStudio.name) {
            $scope.newPromptStudio.name = StringUtils.transmogrify('Prompt Studio', list.map(item => item.name), undefined, 1, true);
        }
    }
});

app.component('promptTile', {
    bindings: {
        promptStudioPrompt: '<',
        stats: '<',
        runOn: '<',
        runBy: '<',
        starrable: '<',
        onStarChange: '<',
        availableLlms: '<',
        tagsMap: '<',
        showFullTitle: '<',
        showValidation: '<'
    },
    templateUrl: '/templates/promptstudios/prompt-tile.html',
    controller: function($filter, $scope, PromptUtils) {
        const $ctrl = this;
        $ctrl.MAX_PROMPT_LENGTH = 200;

        function updateLLM() {
            if (!$ctrl.availableLlms || !$ctrl.promptStudioPrompt) {
                return;
            }
            $ctrl.llm = $ctrl.availableLlms.find(l => l.id == $ctrl.promptStudioPrompt.llmId);
            $ctrl.iconForLLM = PromptUtils.getLLMIcon($ctrl.llm ? $ctrl.llm.type : '', 12);
        }

        $ctrl.$onChanges = updateLLM;

        $scope.$on("update-llm-id", updateLLM);

        $ctrl.getFormattedPrompt = () => {
            const fullText = $ctrl.getFullPrompt();
            return fullText.length > $ctrl.MAX_PROMPT_LENGTH ? fullText.substring(0, $ctrl.MAX_PROMPT_LENGTH / 2) + '…' + fullText.substring(fullText.length - $ctrl.MAX_PROMPT_LENGTH / 2, fullText.length) : fullText;
        };

        $ctrl.getFullPrompt = () => PromptUtils.getPromptText($ctrl.promptStudioPrompt.prompt);

        $ctrl.getLLMColorStyle = () => PromptUtils.getLLMColorStyle(($ctrl.promptStudioPrompt || {}).llmId, $ctrl.availableLlms);
    }
});


app.component('promptResponseSummary', {
    bindings: {
        responseStats: '<',
        costPrefix: '<',
        showValidation: '<?'
    },
    templateUrl: '/templates/promptstudios/prompt-response-summary.html',
    controller: function() {
        const $ctrl = this;
        $ctrl.showValidation = $ctrl.showValidation === undefined ? true : $ctrl.showValidation;
        $ctrl.nbOkRecordsWithoutValidation = function() {
            return $ctrl.responseStats.testedRecords - $ctrl.responseStats.validRecords - $ctrl.responseStats.invalidRecords - $ctrl.responseStats.failedRecords;
        };
    }
});


app.component('promptInputDefinition', {
    bindings: {
        prompt: '<',
        dataset: '<',
        disableFields: '<?',  // prevent to add/delete/update input fields
        onColumnChange: '&',
        activeLlm: '<?',
        disableImageInputs: '<?'
    },
    templateUrl: '/templates/promptstudios/prompt-input-definition.html',
    controller: function($stateParams, SmartId, DataikuAPI, PromptUtils) {
        const $ctrl = this;
        $ctrl.getInputs = PromptUtils.getInputs;
        $ctrl.getInputLabel = () => {
            if ($ctrl.prompt.promptMode == 'PROMPT_TEMPLATE_TEXT') {
                return 'Name';
            }
            if ($ctrl.prompt.promptMode == 'PROMPT_TEMPLATE_STRUCTURED') {
                return 'Description';
            }
        };

        $ctrl.inputDatasetColumns = [];
        function updateInputDatasetColumns() {
            if (!$ctrl.dataset || !$ctrl.prompt) return;

            const dataset = SmartId.resolve($ctrl.dataset, $stateParams.projectKey);
            DataikuAPI.datasets.get(dataset.projectKey, dataset.id, $stateParams.projectKey).then(response => {
                const schema = response.data.schema;
                $ctrl.inputDatasetColumns = schema.columns.map(column => column.name);
                PromptUtils.getInputs($ctrl.prompt).forEach(function(input) {
                    if (!$ctrl.inputDatasetColumns.includes(input.datasetColumnName)) {
                        input.datasetColumnName = undefined;
                    }
                });
            });
        };

        $ctrl.$onChanges = function() {
            updateInputDatasetColumns();
        }

        $ctrl.addInput = function(type) {  // for structured prompts only
            PromptUtils.getInputs($ctrl.prompt).push({
                name: '',
                datasetColumnName: undefined,
                type
            });
            $ctrl.prompt.structuredPromptExamples.forEach(example => {
                example.inputs.push('');
            });
            $ctrl.onColumnChange();
        };

        $ctrl.deleteInput = function(index) {  // for structured prompts only
            PromptUtils.getInputs($ctrl.prompt).splice(index, 1);
            $ctrl.prompt.structuredPromptExamples.forEach(example => {
                example.inputs.splice(index, 1);
            });
            // trigger column change as column has been deleted
            $ctrl.onColumnChange();
        };
    }
});

app.component('promptStudioOutput', {
    bindings: {
        response: '<',
        getLlmOutput: '&',
    },
    templateUrl: '/templates/promptstudios/prompt-studio-output.html'
});

app.component('promptStudioResponseActions', {
    bindings: {
        prompt: '<',
        response: '<',
        singleResponse: '<',
        onViewPrompt: '&',
        onAddExample: '&',
        onDelete: '&',
        getLlmOutput: '&'
    },
    templateUrl: '/templates/promptstudios/prompt-studio-response-actions.html',
    controller: function(PromptUtils, ClipboardUtils) {
        const $ctrl = this;
        
        $ctrl.canCreateExampleFromResponse = function() {
            // inline prompts inputs should always align with response
            if ($ctrl.prompt.promptTemplateQueriesSource === 'INLINE') {
                return true;
            }

            const promptInputNames = PromptUtils.getInputs($ctrl.prompt).map(input => input.datasetColumnName);

            // use == to compare null/undefined
            return $ctrl.response && $ctrl.response.mainPromptTemplateInputs && $ctrl.response.mainPromptTemplateInputs.every((input, index) => input.name == promptInputNames[index]);
        };

        $ctrl.canViewRawPrompt = function() {
            // can view raw prompt if prompt hasn't been run yet or if runId exists (legacy responses don't have a runId)
            return !$ctrl.response.runOn || $ctrl.response.runId;
        };

        $ctrl.copyTraceToClipboard = function() {
            ClipboardUtils.copyToClipboard(JSON.stringify($ctrl.singleResponse.fullTrace, null, 2), "Trace copied to clipboard");
        };
        $ctrl.copyResponseToClipboard = function() {
            ClipboardUtils.copyToClipboard($ctrl.getLlmOutput($ctrl.singleResponse), "Response copied to clipboard");
        };
    }
});


app.component('promptResultValidation', {
    bindings: {
        validationSettings: '<',
        form: '<',
        activeLlm: '<',
        promptMode: '<',
        streamingDisabled: '=',
    },
    templateUrl: '/templates/promptstudios/prompt-result-validation.html',
    controller: function() {
        const $ctrl = this;
        $ctrl.changeFormValidity = function(valid) {
            if (!$ctrl.form) return;

            $ctrl.form.$setValidity('editable-list', valid);
        };

        $ctrl.isOpenAILLM = function() {
            if (!$ctrl.activeLlm) return;

            return $ctrl.activeLlm.type === 'OPENAI';
        };
    }
});

app.component('promptStudioSources', {
    bindings: {
        sources: '<',
        expanded: '<',
    },
    templateUrl: '/templates/promptstudios/prompt-studio-sources.html',
    controller: function() {
        const $ctrl = this;
        $ctrl.displayedSources = [];

        $ctrl.$onInit = function() {
            if($ctrl.expanded !== false) {
                $ctrl.expanded = true;
            }
        };

        $ctrl.$onChanges = function(changes) {
            if (changes.sources) {
                if($ctrl.sources){
                    $ctrl.displayedSources = $ctrl.sources.map(source => {
                        const displayedSource = angular.copy(source);

                        if (source.excerpt && source.excerpt.type === 'IMAGE_REF') {
                            if (source.excerpt.images && source.excerpt.images.length) {
                                // assume excerpt uses only one folder
                                const fullFolderId = source.excerpt.images[0].fullFolderId;
                                const folder = fullFolderId.split('.');
                                displayedSource.excerpt.fullFolderId = fullFolderId;
                                displayedSource.excerpt.projectKey = folder[0];
                                displayedSource.excerpt.folderId = folder[1];
                                displayedSource.excerpt.imagePaths = source.excerpt.images.map(image => image['path']);
                            }
                        }

                        return displayedSource;
                    });
                } else{ // sources are not available anymore, emptying displayedSources
                    $ctrl.displayedSources = [];
                }
            }
        };
    }
});

app.component('promptLlmCompletionSettings', {
    bindings: {
        completionSettings: '<',
        activeLlm: '<',
    },
    templateUrl: '/templates/promptstudios/prompt-llm-completion-settings.html',
    controller: function($scope, PromptUtils) {
        const $ctrl = this;
        $ctrl.$onChanges = function(changes) {
            if (changes.activeLlm) {
                $ctrl.temperatureRange = PromptUtils.getTemperatureRange($ctrl.activeLlm);
                $ctrl.topKRange = PromptUtils.getTopKRange($ctrl.activeLlm);
            }

            if (changes.completionSettings) {
                $ctrl.jsonMode = $ctrl.completionSettings.responseFormat && $ctrl.completionSettings.responseFormat.type === 'json';
            }
        };

        $ctrl.setJsonMode = function() {
            if ($ctrl.completionSettings) {
                if ($ctrl.jsonMode) {
                    $ctrl.completionSettings.responseFormat = {type: 'json'};
                } else {
                    $ctrl.completionSettings.responseFormat = null;
                }
            }
        };
    }
});


app.component('promptReusableTemplate', {
    bindings: {
        prompt: '<',
        llm: '<',
        disablePromptSwitch: '<',
        disableImageInputs: '<',
        onPromptInputChange: '&',
    },
    require: {
        apiErrorContext: "^apiErrorContext"
    },
    templateUrl: '/templates/promptstudios/prompt-template.html',
    controller: function($scope, $stateParams, PromptUtils, WT1, Debounce) {
        const $ctrl = this;
        $ctrl.switchMode = function(mode) {
            if (!$ctrl.prompt) return;

            WT1.event("prompt-studio-prompt-mode-switch", {targetMode: mode});
            $ctrl.prompt.promptMode = mode;
        };

        $ctrl.updateInputCheck = Debounce().withDelay(300, 300).wrap(() => {
            const inputsChanged = PromptUtils.updateInputCheck($ctrl.prompt, $ctrl.llm.supportsImageInputs && !$ctrl.disableImageInputs);

            if (inputsChanged && $ctrl.onPromptInputChange) {
                $ctrl.onPromptInputChange();
            }
        });

        $ctrl.showRawPrompt = function() {
            PromptUtils.openRawPromptModal($scope, $ctrl.prompt, $ctrl.llm.id, {});
        };

        $ctrl.addTextToMessage = function(text) {
            const userMessageEl = document.getElementById('prompt-studio-task__user-message');
            userMessageEl.focus();

            const [start, end] = [userMessageEl.selectionStart, userMessageEl.selectionEnd];
            userMessageEl.setRangeText(text, start, end, 'end');

            angular.element(userMessageEl).change();
        };
    }
});

app.component('promptSamplingSettings', {
    bindings: {
        settings: '<',
        updateSettings: '&',
    },
    templateUrl: '/templates/promptstudios/prompt-sampling-settings.html',
    controller: function() {
        const $ctrl = this;
        let oldSettings = {};

        $ctrl.$onInit = () => {
            $ctrl.settings = Object.assign({}, {
                nbRows: 8
            }, $ctrl.settings);
            oldSettings = angular.copy($ctrl.settings);
        }

        $ctrl.$onChanges = () => oldSettings = angular.copy($ctrl.settings);

        $ctrl.update = () => $ctrl.updateSettings({settings: $ctrl.settings});

        $ctrl.isDirty = () => !angular.equals(oldSettings, $ctrl.settings);
    }
});

app.directive('promptStudioRightColumnSummary', function($controller, $state, $stateParams, $rootScope, DataikuAPI, CreateModalFromTemplate, QuickView,
    ActiveProjectKey, ActivityIndicator, TopNav, WT1, GlobalProjectActions) {

    return {
        templateUrl :'/templates/promptstudios/right-column-summary.html',

        link : function(scope) {
            $controller('_TaggableObjectsMassActions', {$scope: scope});

            scope.$stateParams = $stateParams;
            scope.QuickView = QuickView;

            scope.getSmartName = function (projectKey, name) {
                if (projectKey == ActiveProjectKey.get()) {
                    return name;
                } else {
                    return projectKey + '.' + name;
                }
            }

            scope.refreshData = function() {
                const projectKey = scope.selection.selectedObject.projectKey;
                const name = scope.selection.selectedObject.name;

                DataikuAPI.promptStudios.getFullInfo(ActiveProjectKey.get(), scope.selection.selectedObject.id).then(function({data}){
                    if (!scope.selection.selectedObject || scope.selection.selectedObject.projectKey != projectKey || scope.selection.selectedObject.name != name) {
                        return;
                    }
                    data.realName = data.name;
                    scope.promptStudioData = data;
                    scope.promptStudio = data.promptStudio;
                    scope.selection.selectedObject.interest = data.interest;
                    //scope.objectAuthorizations = data.objectAuthorizations;
                }).catch(setErrorInScope.bind(scope));
            };

            scope.copy = function() {
                if (!scope.promptStudio) return;

                DataikuAPI.promptStudios.copy(ActiveProjectKey.get(), scope.promptStudio.id).then(function({data}) {
                    if ($state.includes('projects.project.promptstudios.promptstudio')) {
                        $state.transitionTo("projects.project.promptstudios.promptstudio",{
                            projectKey: $stateParams.projectKey,
                            promptStudioId: data.id
                        });
                    } else {
                        scope.list();
                    }
                }).catch(setErrorInScope.bind(scope));
            };

            function save() {
                return DataikuAPI.promptStudios.save(scope.promptStudio, {summaryOnly: true})
                .success(function() {
                    ActivityIndicator.success("Saved");
                }).error(setErrorInScope.bind(scope));
            }

            scope.$on("objectSummaryEdited", save);

            scope.$watch("selection.selectedObject",function() {
                if (scope.selection.selectedObject != scope.selection.confirmedItem) {
                    scope.promptStudio = null;
                    scope.objectTimeline = null;
                }
            });

            scope.$watch("selection.confirmedItem", function(nv) {
                if (!nv) return;
                if (!nv.projectKey) {
                    nv.projectKey = ActiveProjectKey.get();
                }
                scope.refreshData();
            });

            scope.deletePromptStudio = function() {
                GlobalProjectActions.deleteTaggableObject(scope, 'PROMPT_STUDIO', scope.selection.selectedObject.id, scope.selection.selectedObject.name);
            };


            /** Custom fields from plugins **/
            scope.editCustomFields = function() {
                if (!scope.promptStudio) {
                    return;
                }
                let modalScope = angular.extend(scope, {objectType: 'PROMPT_STUDIO', objectName: scope.promptStudio.name, objectCustomFields: scope.promptStudio.customFields});
                CreateModalFromTemplate("/templates/taggable-objects/custom-fields-edit-modal.html", modalScope).then(function(newCustomFields) {
                    WT1.event('custom-fields-save', {objectType: 'PROMPT_STUDIO'});
                    const oldCustomFields = angular.copy(scope.promptStudio.customFields);
                    scope.promptStudio.customFields = newCustomFields;
                    return save().then(function() {
                            $rootScope.$broadcast('customFieldsSaved', TopNav.getItem(), scope.promptStudio.customFields);
                        }, function() {
                            scope.promptStudio.customFields = oldCustomFields;
                        });
                });
            };

            const customFieldsListener = $rootScope.$on('customFieldsSaved', scope.refreshData);
            scope.$on("$destroy", customFieldsListener);
        }
    }
});

app.component('promptDatasetSelector', {
    bindings: {
        dataset: '<',
        queriesSource: '<',
        datasets: '<',
        onChange: '&'
    },
    template: `
        <div class="prompt-studio-task-input__dataset-selector" dataset-selector="$ctrl.inputSource" on-dataset-selector-change="$ctrl.onDatasetSelectorChange()" popover-placement="right" available-datasets="$ctrl.availableDatasets"></div>
    `,
    controller: function() {
        const $ctrl = this;
        const INLINE_ID = '!!INLINE_PROMPT'; // id of object to be passed to dataset-selector as a "NON_DATASET"
        $ctrl.inputSource = '';
        $ctrl.availableDatasets = [];

        $ctrl.onDatasetSelectorChange = () => {
            if ($ctrl.inputSource === INLINE_ID) {
                $ctrl.onChange({ queriesSource: 'INLINE' });
            } else {
                $ctrl.onChange({ queriesSource: 'DATASET', dataset: $ctrl.inputSource });
            }
        }

        $ctrl.$onChanges = function(changes) {
            if (changes.queriesSource || changes.dataset) {
                if ($ctrl.queriesSource === 'INLINE') {
                    $ctrl.inputSource = INLINE_ID;
                } else {
                    $ctrl.inputSource = $ctrl.dataset;
                }
            }

            if (changes.datasets) {
                $ctrl.availableDatasets = ($ctrl.datasets || []).concat([{
                    name: 'Written test cases',
                    id: INLINE_ID,
                    type: 'NON_DATASET',
                    icon: 'dku-icon-list-bulleted-24',
                    usable: true,
                    header: true
                }]);
            }
        };
    }
});

app.component('promptImages', {
    bindings: {
        paths: '<',
        folderId: '<',
        showLargeImages: '<'
    },
    template: `
        <span class="prompt-images" ng-class="{ 'prompt-images--large': $ctrl.showLargeImages }">
            <span class="prompt-image" ng-repeat="path in $ctrl.displayedPaths" ng-click="$ctrl.openImage($index)">
                <prompt-image url="$ctrl.getImageUrl(path)" image-class="'prompt-image__image'" show-preview-on-hover="true" show-large-image="$ctrl.showLargeImages">
            </span><span ng-if="$ctrl.hiddenCount > 0" class="prompt-image prompt-image--extra">+ {{ $ctrl.hiddenCount }}</span>
        </span>
    `,
    controller: function($scope, $stateParams, CreateModalFromTemplate, $timeout, PromptUtils) {
        const $ctrl = this;
        const MAX_DISPLAYED_IMAGES = 10;

        $ctrl.$onChanges = function(changes) {
            if (changes.paths) {
                const paths = $ctrl.paths || [];
                $ctrl.displayedPaths = paths.slice(0, MAX_DISPLAYED_IMAGES);
                $ctrl.hiddenCount = paths.length - MAX_DISPLAYED_IMAGES;
            }
        };

        $ctrl.getImageUrl = (path) => PromptUtils.getImageUrl(path, $ctrl.folderId);

        $ctrl.openImage = function(index) {
            const scope = $scope.$new();
            scope.paths = $ctrl.paths;
            scope.selectedIndex = index;
            scope.getImageUrl = $ctrl.getImageUrl;
            scope.showLargeImages = $ctrl.showLargeImages;
            CreateModalFromTemplate("/templates/promptstudios/prompt-image-modal.html", scope, null, function(newScope) {
                newScope.onItemChange = function(newIndex) {
                    $timeout(() => newScope.selectedIndex = newIndex);
                }
            });
        };
    }
});

app.component('promptImage', {
    bindings: {
        url: '<',
        imageClass: '<',
        errorIconClass: '<',
        showPreviewOnHover: '<',
        showLargeImage: '<',
    },
    templateUrl: '/templates/promptstudios/prompt-image.html',
    controller: function() {
        const $ctrl = this;
        $ctrl.errorText = 'Image not found';

        $ctrl.onError = function() {
            $ctrl.hasError = true;
        }

        $ctrl.$onChanges = function(changes) {
            if (changes.url) {
                $ctrl.hasError = false;
            }
        };
    }
});

app.component('promptStudioChat', {
    bindings: {
        messages: '<',
        lastMessageId: '=',
        streamData: '<',
        promptId: '<',
        availableLlms: '<',
        activeLlm: '<',
        activeLlmIcon: '<',
        defaultMessage: '<',
        onSendMessage: '&',
        onForkMessage: '&',
        onStopStreaming: '&',
    },
    templateUrl: '/templates/promptstudios/prompt-studio-chat.html',
    controller: function($scope, $timeout, $rootScope, PromptUtils, ClipboardUtils) {
        const $ctrl = this;
        const entryElId = 'prompt-chat-entry';

        $ctrl.message = '';
        $ctrl.messageBranch = [];
        $ctrl.iconSize = 16;
        $ctrl.editedMessage = '';
        $ctrl.editedMessageId = '';
        
        $ctrl.$onChanges = (changes) => {
            if (changes.messages) {
                $ctrl.messageBranch = PromptUtils.buildCurrentChatBranch($ctrl.messages, $ctrl.lastMessageId);
                document.getElementById(entryElId).focus();
                $timeout(() => {
                    const containerEl = document.getElementById('prompt-chat');
                    containerEl.scrollTop = containerEl.scrollHeight;
                });
            }
            if (changes.promptId) {
                clearMessage();
            }
            if (changes.defaultMessage && changes.defaultMessage.currentValue) {
                $ctrl.message = $ctrl.defaultMessage;
                $timeout(() => {
                    $ctrl.setMessageField(document.getElementById(entryElId));
                });
            }
        };

        // used for auto textarea resizing
        $ctrl.setMessageField = (el) => {
            el.parentNode.dataset.replicatedValue = el.value;
        };

        $ctrl.sendMessage = () => {
            const newMessage = {
                parentId: $ctrl.lastMessageId,
                id: generateRandomId(7),
                runBy: $rootScope.appConfig.login,
                message: {
                    role: 'user',
                    content : $ctrl.message
                }
            };
            $ctrl.messageBranch.push(newMessage);
            $ctrl.streamParentId = $ctrl.lastMessageId;
            $ctrl.onSendMessage({ newMessage });
            clearMessage();
        };

        $ctrl.sendEditedMessage = (parentId) => {
            const newMessage = {
                parentId,
                id: generateRandomId(7),
                runBy: $rootScope.appConfig.login,
                message: {
                    role: 'user',
                    content : $ctrl.editedMessage
                }
            };
            $ctrl.messageBranch = PromptUtils.buildCurrentChatBranch($ctrl.messages, parentId);
            $ctrl.messageBranch.push(newMessage);
            $ctrl.streamParentId = parentId;
            $ctrl.onSendMessage({ newMessage });
            $ctrl.clearEditedMessage();
        };

        $ctrl.forkMessage = (userMessage) => {
            const messageHistory = PromptUtils.buildCurrentChatBranch($ctrl.messages, userMessage.parentId);
            $ctrl.onForkMessage({
                messageHistory,
                userMessage
            });
        }

        $ctrl.rerunResponse = (message) => {
            const parentId = message.parentId;
            const newMessage = $ctrl.messages[parentId];
            $ctrl.messageBranch = PromptUtils.buildCurrentChatBranch($ctrl.messages, parentId);
            $ctrl.onSendMessage({ newMessage, isRerun: true });
        };

        $ctrl.clearEditedMessage = () => {
            $ctrl.editedMessage = '';
            $ctrl.editedMessageId = '';
        };

        $ctrl.cannotSendMessage = () => {
            return $ctrl.cannotRunChat() || $ctrl.message.trim() === '' || $ctrl.hasError() || $ctrl.editedMessageId;
        };

        $ctrl.hasError = () => {
            return $ctrl.messageBranch.some(x => x.error);
        };

        $ctrl.cannotSendEditedMessage = () => {
            return $ctrl.cannotRunChat() || $ctrl.editedMessage.trim() === '';
        };

        $ctrl.cannotRerunMessage = () => {
            return $ctrl.cannotRunChat();
        };

        $ctrl.isStreaming = () => {
            return $ctrl.streamData && $ctrl.streamData.currentlyStreaming;
        };

        $ctrl.cannotRunChat = () => {
            return !$ctrl.activeLlm || $ctrl.isStreaming();
        };

        $ctrl.switchBranch = (message, position) => {
            const parent = $ctrl.messages[message.parentId];
            let nextMesssage = $ctrl.messages[parent.childrenIds[position]];
            while (nextMesssage.childrenIds && nextMesssage.childrenIds.length) {
                nextMesssage = $ctrl.messages[nextMesssage.childrenIds[0]];
            }
            $ctrl.lastMessageId = nextMesssage.id;
            $ctrl.messageBranch = PromptUtils.buildCurrentChatBranch($ctrl.messages, $ctrl.lastMessageId);
        };

        $ctrl.onKeydown = (event, sendEdited, parentId) => {
            if (event.key === 'Enter') {
                if (!event.shiftKey) {
                    event.preventDefault();
                    if (sendEdited) {
                        !$ctrl.cannotSendEditedMessage() && $ctrl.sendEditedMessage(parentId);
                    } else {
                        !$ctrl.cannotSendMessage() && $ctrl.sendMessage();
                    }
                }
            }
        };

        // note: this function assumes there is only one message part
        $ctrl.editMessage = (message) => {
            $ctrl.editedMessageId = message.id;
            $ctrl.editedMessage = message.message.content;
            $timeout(() => {
                const editEl = document.getElementById('prompt-chat-edit-entry');
                editEl.focus();
                $ctrl.setMessageField(editEl);
            });
        };

        $ctrl.getLLMIconName = (llmStructuredRef) => {
            return PromptUtils.getLLMIcon(llmStructuredRef.type, $ctrl.iconSize);
        };

        $ctrl.getLLMIconStyle = (llmStructuredRef) => {
             return PromptUtils.getLLMColorStyle(llmStructuredRef.id, $ctrl.availableLlms);
        };
        $ctrl.getLLMFriendlyName = (llmStructuredRef) => {
            return $ctrl.availableLlms.find(llm => llm.id === llmStructuredRef.id)?.friendlyName;
        };

        $ctrl.copyTraceToClipboard = function(message) {
            ClipboardUtils.copyToClipboard(JSON.stringify(message.fullTrace, null, 2), "Trace copied to clipboard");
        };
        $ctrl.copyMessageToClipboard = function(message) {
            ClipboardUtils.copyToClipboard(message.error ? message.llmError : message.message.content, "Message copied to clipboard");
        };

        function clearMessage() {
            const entryEl = document.getElementById(entryElId);
            $ctrl.message = '';
            entryEl.parentNode.dataset.replicatedValue = '';
        }
    }
});

app.component('promptStudioChatBranchSwitch', {
    bindings: {
        numChildren: '<',
        message: '<',
        isStreaming: '<',
        onSwitchBranch: '&',
    },
    templateUrl: '/templates/promptstudios/prompt-studio-chat-branch-switch.html',
    controller: function() {
        const $ctrl = this;

        $ctrl.switchBranch = (position) => {
            $ctrl.onSwitchBranch({ message: $ctrl.message, position });
        };

        $ctrl.canSwitchBack = () => {
            return !$ctrl.isStreaming && $ctrl.message.version !== 0;
        };

        $ctrl.canSwitchForward = () => {
            return !$ctrl.isStreaming && $ctrl.message.version < $ctrl.numChildren - 1;
        };
    }
})

app.controller("PromptStudioPageRightColumnActions", function($controller, $scope, $rootScope, DataikuAPI, $stateParams, ActiveProjectKey) {

    $controller('_TaggableObjectPageRightColumnActions', {$scope: $scope});

    $scope.selection = {};

    DataikuAPI.promptStudios.get(ActiveProjectKey.get(), $stateParams.promptStudioId).success((data) => {
        $scope.selection = {
            selectedObject : data, confirmedItem : data,
        };
        $scope.selection.selectedObject.nodeType = "PROMPT_STUDIO";
    }).error(setErrorInScope.bind($scope));

    $scope.renameObjectAndSave = function(newName) {
        $scope.selection.selectedObject.name = newName;
        return DataikuAPI.promptStudios.save($scope.selection.selectedObject);
    };

    /** Interests (star, watch) **/
    function updateUserInterests() {
        DataikuAPI.interests.getForObject($rootScope.appConfig.login, "PROMPT_STUDIO", ActiveProjectKey.get(), $scope.selection.selectedObject.id).success(function(data) {
            $scope.selection.selectedObject.interest = data;
            $scope.promptStudioData.interest = data;
        }).error(setErrorInScope.bind($scope));
    }

    const interestsListener = $rootScope.$on('userInterestsUpdated', updateUserInterests);
    $scope.$on("$destroy", interestsListener);
});

})();
