(function(){
'use strict';

const app = angular.module('dataiku.retrievableknowledge', []);



/* ************************************ List / Right column  *************************** */

app.directive('retrievableKnowledgeRightColumnSummary', function($controller, $state, $stateParams, $rootScope, FlowGraphSelection,
    DataikuAPI, SmartId, QuickView, ActiveProjectKey, ActivityIndicator, FlowBuildService, AnyLoc) {

    return {
        templateUrl :'/templates/retrievable-knowledge/right-column-summary.html',

        link : function(scope) {
            $controller('_TaggableObjectsMassActions', {$scope: scope});

            scope.$stateParams = $stateParams;
            scope.QuickView = QuickView;

            /* Auto save when summary is modified */
            scope.$on("objectSummaryEdited", function(){
                DataikuAPI.retrievableknowledge.save(scope.retrievableKnowledge, {summaryOnly: true}).success(function(data) {
                    ActivityIndicator.success("Saved");
                }).error(setErrorInScope.bind(scope));
            });

            scope.getSmartName = function (projectKey, name) {
                if (projectKey == ActiveProjectKey.get()) {
                    return name;
                } else {
                    return projectKey + '.' + name;
                }
            }

            scope.isOnObjectPage = function() {
                return $state.includes('projects.project.retrievableknowledges.retrievableknowledge');
            }


            scope.refreshData = function() {
                const projectKey = scope.selection.selectedObject.projectKey;
                const name = scope.selection.selectedObject.name;
                scope.canAccessObject = false;

                DataikuAPI.retrievableknowledge.getFullInfo(ActiveProjectKey.get(), SmartId.create(name, projectKey)).then(function({data}){
                    if (!scope.selection.selectedObject || scope.selection.selectedObject.projectKey != projectKey
                        || scope.selection.selectedObject.name != name) {
                        return; // too late!
                    }
                    scope.retrievableKnowledgeFullInfo = data;
                    scope.retrievableKnowledge = data.retrievableKnowledge;
                    scope.retrievableKnowledge.zone = (scope.selection.selectedObject.usedByZones || [])[0] || scope.selection.selectedObject.ownerZone;
                    scope.selection.selectedObject.interest = data.interest;
                    scope.isLocalRetrievableKnowledge = projectKey == ActiveProjectKey.get();
                    scope.objectAuthorizations = data.objectAuthorizations;
                    scope.canAccessObject = true;
                }).catch(setErrorInScope.bind(scope));
            };

            scope.$watch("selection.selectedObject",function(nv) {
                if(scope.selection.selectedObject != scope.selection.confirmedItem) {
                    scope.retrievableKnowledge = null;
                    scope.objectTimeline = null;
                }
            });

            function updateUserInterests() {
                DataikuAPI.interests.getForObject($rootScope.appConfig.login, "RETRIEVABLE_KNOWLEDGE", ActiveProjectKey.get(), scope.selection.selectedObject.name).success(function(data) {
                    scope.selection.selectedObject.interest = data;
                    scope.retrievableKnowledgeFullInfo.interest = data;
                }).error(setErrorInScope.bind(scope));
            }

            const interestsListener = $rootScope.$on('userInterestsUpdated', updateUserInterests);
            scope.$on("$destroy", interestsListener);

            scope.buildRetrievableKnowledge = function() {
                FlowBuildService.openSingleComputableBuildModalFromObjectTypeAndLoc(scope, "RETRIEVABLE_KNOWLEDGE",
                            AnyLoc.makeLoc(scope.retrievableKnowledge.projectKey, scope.retrievableKnowledge.id));
            };

            scope.$watch("selection.confirmedItem", function(nv, ov) {
                if (!nv) {
                    return;
                }
                if (!nv.projectKey) {
                    nv.projectKey = ActiveProjectKey.get();
                }
                scope.refreshData();
            });

            scope.zoomToOtherZoneNode = function(zoneId) {
                const otherNodeId = scope.selection.selectedObject.id.replace(/zone__.+?__retrievableknowledge/, "zone__" + zoneId + "__retrievableknowledge");
                if ($stateParams.zoneId) {
                    $state.go('projects.project.flow', Object.assign({}, $stateParams, { zoneId: zoneId }))
                }
                else {
                    scope.zoomGraph(otherNodeId);
                    FlowGraphSelection.clearSelection();
                    FlowGraphSelection.onItemClick(scope.nodesGraph.nodes[otherNodeId], null);
                }
            }
        }
    }
});

app.controller("RetrievableKnowledgePageRightColumnActions", function($controller, $scope, $rootScope, DataikuAPI, $stateParams, ActiveProjectKey) {

    $controller('_TaggableObjectPageRightColumnActions', {$scope: $scope});

    $scope.selection = {};

    DataikuAPI.retrievableknowledge.get(ActiveProjectKey.get(), $stateParams.retrievableKnowledgeId).success((data) => {
        data.description = data.shortDesc;
        data.nodeType = 'LOCAL_RETRIEVABLE_KNOWLEDGE';
        data.realName = data.name;
        data.name = data.id;
        data.interest = {};

        $scope.selection = {
            selectedObject : data, confirmedItem : data,
        };
    }).error(setErrorInScope.bind($scope));

    $scope.renameObjectAndSave = function(newName) {
        $scope.selection.selectedObject.name = newName;
        return DataikuAPI.retrievableknowledge.save($scope.selection.selectedObject);
    };
});


app.controller("RetrievableKnowledgeListController", function($scope, $controller, $stateParams, DataikuAPI, $state, TopNav, WT1) {
    $controller('_TaggableObjectsListPageCommon', {$scope: $scope});

    $scope.sortBy = [
        { value: 'realName', label: 'Name' },
        { value: '-lastModifiedOn', label: 'Last modified'}
    ];

    $scope.selection = $.extend({
        filterQuery: {
            userQuery: '',
            tags: [],
            interest: {
                starred: '',
            },
            inputDatasetSmartName: []
        },
        filterParams: {
            userQueryTargets: ["realName", "tags"],
            propertyRules: {tag: 'tags'},
        },
        orderQuery: "-lastModifiedOn",
        orderReversed: false
    }, $scope.selection || {});

    $scope.maxItems = 20;

    $scope.list = function() {
        DataikuAPI.retrievableKnowledge.listHeads($stateParams.projectKey).success(function(data) {
            // dirty things to handle the discrepancy between the types of selected objects
            // which can have info displayed in the right panel
            data.forEach(mes => {
                mes.realName = mes.name;
                mes.name = mes.id;
            });
            $scope.listItems = data;
            $scope.restoreOriginalSelection();
        }).error(setErrorInScope.bind($scope));
    };

    TopNav.setLocation(TopNav.TOP_RETRIEVABLE_KNOWLEDGE, TopNav.LEFT_RETRIEVABLE_KNOWLEDGE, TopNav.TABS_NONE, null);
    TopNav.setNoItem();
    $scope.list();

    /* Tags handling */

    $scope.$on('selectedIndex', function(e, index){
        // an index has been selected, we unselect the multiselect
        $scope.$broadcast('clearMultiSelect');
    });

    /* Specific actions */
    $scope.goToItem = function(data) {
        $state.go("projects.project.retrievableknowledges.retrievableknowledge.usage", {projectKey : $stateParams.projectKey, retrievableKnowledgeId: data.id});
    }
});

app.controller("RetrievableKnowledgeController", function($scope, DataikuAPI, $stateParams,
            WT1, ActiveProjectKey, ActivityIndicator, TopNav, RetrievableKnowledgeUtils, FlowBuildService, AnyLoc) {

    $scope.uiState = {};

    DataikuAPI.retrievableknowledge.getFullInfo(ActiveProjectKey.get(), $stateParams.retrievableKnowledgeId).success(function(data){
        $scope.rkFullInfo = data;
        $scope.retrievableKnowledge = data.retrievableKnowledge;
        RetrievableKnowledgeUtils.updateIndexName($scope.retrievableKnowledge);
        $scope.savedSettings = angular.copy($scope.retrievableKnowledge);
        TopNav.setItem(TopNav.ITEM_RETRIEVABLE_KNOWLEDGE, $stateParams.retrievableKnowledgeId, {name: $scope.retrievableKnowledge.name});

        WT1.event("llm-mesh-knowledge-bank-load", {
            retrieverType: data.retrievableKnowledge.retrieverType,
            vectorStoreType: data.retrievableKnowledge.vectorStoreType,
            embeddingLLMId: data.retrievableKnowledge.embeddingLLMId,
            // Todo @augmentedllm clean WT1 llmsExposedWithCount
            //llmsExposedWithCount: data.retrievableKnowledge.llmsExposedWith.length
        });
    }).error(setErrorInScope.bind($scope));

    $scope.save = function(disableSuccessIndicator) {
        if(RetrievableKnowledgeUtils.hasConnection($scope.retrievableKnowledge) && !$scope.retrievableKnowledge.connection) {
           ActivityIndicator.error("You must select a connection");
           return;
        }

        if(RetrievableKnowledgeUtils.hasIndex($scope.retrievableKnowledge) && !$scope.retrievableKnowledge.indexName) {
            ActivityIndicator.error("You must provide an index name");
            return;
        }

        const settings = angular.copy($scope.retrievableKnowledge);
        return DataikuAPI.retrievableknowledge.save($scope.retrievableKnowledge).success(function(data) {
            $scope.savedSettings = settings;
            if (!disableSuccessIndicator) {
                ActivityIndicator.success("Saved");
            }
        }).error(setErrorInScope.bind($scope));
    };

    $scope.dirtySettings = function() {
        return !angular.equals($scope.savedSettings, $scope.retrievableKnowledge);
    };

    $scope.openBuildKbModal = function () {
      FlowBuildService.openSingleComputableBuildModalFromObjectTypeAndLoc(
        $scope,
        "RETRIEVABLE_KNOWLEDGE",
        AnyLoc.makeLoc($stateParams.projectKey, $scope.retrievableKnowledge.id),
        {
          redirectToJobPage: true,
        }
      );
    };

    const allowedTransitions = [
        'projects.project.retrievableknowledges.retrievableknowledge.query',
        'projects.project.retrievableknowledges.retrievableknowledge.usage',
        'projects.project.retrievableknowledges.retrievableknowledge.settings'
    ];

    checkChangesBeforeLeaving($scope, $scope.dirtySettings, null, allowedTransitions);
});


/* ************************************ Settings *************************** */
app.constant("DOCUMENT_SPLITTING_METHOD_MAP", {
    'CHARACTERS_BASED': 'Character count',
    'NONE': 'Do not split'
});

app.constant("VECTOR_STORE_UPDATE_METHOD_MAP", {
    'SMART_OVERWRITE': {isSmart: true, title: 'Smart Sync', description: "Insert new rows. Update modified rows. Remove rows from the Knowledge Bank that are not in the dataset."},
    'SMART_APPEND': {isSmart: true, title: 'Upsert', description: "Insert new rows. Update modified rows. No deletion is performed."},
    'OVERWRITE': {isSmart: false, title: 'Overwrite', description: "Completely rebuild the Knowledge Bank."},
    'APPEND': {isSmart: false, title: 'Append', description: "Append input rows to the Knowledge Bank. Can create duplicates."},
});

app.constant("EMBED_DOCS_VECTOR_STORE_UPDATE_METHOD_MAP", { // TODO keep in sync with above constant
    'SMART_OVERWRITE': {isSmart: true, title: 'Smart Sync', description: "Process new documents, fully reprocess modified ones, and remove those missing from the input managed folder."},
    'SMART_APPEND': {isSmart: true, title: 'Upsert', description: "Process new documents and fully reprocess modified ones without performing deletions."},
    'OVERWRITE': {isSmart: false, title: 'Overwrite', description: "Completely rebuild the Knowledge Bank."},
    'APPEND': {isSmart: false, title: 'Append', description: "Append documents to the Knowledge Bank. Can create duplicates."},
});

app.constant("VECTOR_STORE_TYPE_MAP", {
    CHROMA: 'ChromaDB',
    FAISS: 'FAISS',
    PINECONE: 'Pinecone',
    AZURE_AI_SEARCH: 'Azure AI Search',
    VERTEX_AI_GCS_BASED: 'Vertex AI - GCS based',
    ELASTICSEARCH: 'Elasticsearch',
    QDRANT_LOCAL: 'Qdrant (local)'
});

app.constant("VECTOR_STORE_HYBRID_SUPPORT", [
    'AZURE_AI_SEARCH', 'ELASTICSEARCH'
]);

app.component("printDocumentSourcesRadio", {
    bindings: {
        llmExposedWith: '='
    },
    controller: function () {
        this.$onInit = () => {
            if (this.llmExposedWith.includeContentInSources) {
                this.sourcesToPrint = "METADATA_AND_CONTENT";
            } else {
                this.sourcesToPrint = "METADATA";
            }
            this.metadataOnlyText = this.llmExposedWith.retrievalSource==='CUSTOM'?
                "Metadata only (excluding retrieval content)" :
                "Metadata only"
        };

        this.updateRagLlm = () => {
            if (this.sourcesToPrint === "METADATA") {
                this.llmExposedWith.includeContentInSources = false;
            } else if (this.sourcesToPrint === "METADATA_AND_CONTENT") {
                this.llmExposedWith.includeContentInSources = true;
            }
        };
    },
    templateUrl: "/templates/retrievable-knowledge/print-document-sources-radio.html"
});


app.component("sourceOutputFormat", {
    bindings: {
        llmExposedWith: '='
    },
    controller: function () {
        this.$onInit = () => {

            if (this.llmExposedWith?.printSources) {
                if (this.llmExposedWith.outputFormat === "TEXT") {
                    this.outputFormat = "TEXT";
                } else if (this.llmExposedWith.outputFormat === "JSON") {
                    this.outputFormat = "JSON";
                } else if (this.llmExposedWith.outputFormat === "SEPARATED") {
                    this.outputFormat = "SEPARATED";
                }
            } else {
                this.outputFormat = "NOTHING";
            }
        };

        this.updateRagLlm = () => {
            if (this.outputFormat === "NOTHING") {
                this.llmExposedWith.printSources = false;
            } else if (this.outputFormat === "TEXT") {
                this.llmExposedWith.printSources = true;
                this.llmExposedWith.outputFormat = "TEXT";
            } else if (this.outputFormat === "JSON") {
                this.llmExposedWith.printSources = true;
                this.llmExposedWith.outputFormat = "JSON";
            } else if (this.outputFormat === "SEPARATED") {
                this.llmExposedWith.printSources = true;
                this.llmExposedWith.outputFormat = "SEPARATED";
            }
        };
    },
    template: `
<label class="control-label" for="{{'sourceOutputFormat-' + $index}}">Legacy source output format</label>
<div class="controls" >
    <select id="{{'sourceOutputFormat-' + $index}}" 
            ng-change="$ctrl.updateRagLlm()" 
            dku-bs-select 
            layout="list"
            ng-model="$ctrl.outputFormat"
            options-descriptions="[
                'Completion and sources are concatenated in a single text in the output.', 
                'Completion and sources are put in a structured json in the output.', 
                'Completion is put as text in the output. Sources are put in a separated field.', 
                'Sources are not output.']" >
        <option value="TEXT">Plain text</option>
        <option value="JSON">JSON</option>
        <option value="SEPARATED">Separated (Recommended)</option>
        <option value="NOTHING">Do not print</option>
    </select>
</div>
`
});


app.controller("_RetrievableKnowledgeCommonController", function($scope, $stateParams, DataikuAPI, VECTOR_STORE_UPDATE_METHOD_MAP) {
    $scope.VECTOR_STORE_UPDATE_METHOD_MAP = VECTOR_STORE_UPDATE_METHOD_MAP;
    $scope.recipeInfoLoaded = false;

    $scope.checkCodeEnvCompat = () => {
        if (!$scope || !$scope.retrievableKnowledge) {
            $scope.showCodeEnvVersionWarning = false;
            return;
        }

        if ($scope.retrievableKnowledge.vectorStoreType !== "PINECONE") {
            $scope.showCodeEnvVersionWarning = false;
            return;
        }

        DataikuAPI.retrievableknowledge.checkPineconeCompatibility($stateParams.projectKey, $scope.retrievableKnowledge.connection, $scope.retrievableKnowledge.envSelection).then(function({data}) {
            if (data.compatible) {
                $scope.showCodeEnvVersionWarning = false;
            } else {
                $scope.showCodeEnvVersionWarning = true;
                $scope.codeEnvVersionWarningMessage = data.reasons[0];
            }
        }).catch(setErrorInScope.bind($scope));
    }

    const deregister = $scope.$watch("retrievableKnowledge", function(nv, ov) {
        if (!nv) return;

        DataikuAPI.retrievableknowledge.getCurrentVersionInfo($stateParams.projectKey, $scope.retrievableKnowledge.id)
            .then(function({data}) {
                const { embeddingRecipeParams, rkAtVersion, isBuilt } = data;
                $scope.embeddingRecipeDesc = embeddingRecipeParams;
                $scope.rkAtEmbedding = rkAtVersion;
                $scope.isRkBuilt = isBuilt;

                if (embeddingRecipeParams) {
                    $scope.possibleSourceIDColumns = embeddingRecipeParams.metadataColumns.map((mc) => mc.column);
                } else {
                    $scope.possibleSourceIDColumns = [];
                }
                $scope.recipeInfoLoaded = true;
            })
            .catch(setErrorInScope.bind($scope));

        DataikuAPI.pretrainedModels.listAvailableLLMs($stateParams.projectKey, "GENERIC_COMPLETION").then(function({data}) {
            $scope.availableAugmentableLLMs = data.identifiers.filter(id => id.type !== "RETRIEVAL_AUGMENTED");
        }).catch(setErrorInScope.bind($scope));

        DataikuAPI.pretrainedModels.listAvailableLLMs($stateParams.projectKey, "TEXT_EMBEDDING_EXTRACTION").then(function({data}) {
            $scope.availableEmbeddingModels = data.identifiers;
        }).catch(setErrorInScope.bind($scope));

        $scope.checkCodeEnvCompat();
        deregister();
    });
});

app.component('rrfHelpMessage', {
    bindings: {
    },
    template: `
<div class="help-inline">
    Uses Elasticsearch implementation of
    <external-link
            href="https://www.elastic.co/guide/en/elasticsearch/reference/current/rrf.html">Reciprocal Ranking Fusion (RRF)
    </external-link>
    &nbsp This feature requires an Elastic Enterprise subscription; please verify your subscription type with Elastic.
</div>
`
});

app.component('semanticRankerHelpMessage', {
    template: `
    <div class="help-inline">
        Uses Azure AI proprietary
        <external-link
                href="https://learn.microsoft.com/en-us/azure/search/semantic-search-overview">Semantic Ranker
        </external-link>
    </div>
    `
});

app.component('retrievalContentSettings', {
    bindings: {
        retrievableKnowledge: '<',
        embeddingRecipeDesc: '<', // optional - used for a helpful UI message
        availableAugmentableLLMs: '<', // optional - used for warning when non-multimodal LLM selected with multimodal retrieval column
        metadataColumns: '<',
        llmExposedWith: '='
    },
    templateUrl: '/templates/retrievable-knowledge/retrieval-content-settings.html',
    controller: function() {
        const $ctrl = this;

        $ctrl.hasMetadataColumns = function() {
            return $ctrl.metadataColumns && $ctrl.metadataColumns.length > 0;
        };

        $ctrl.getEmbeddingRetrievalText = function () {
            //Change embedding title as the embed column is automatically generated for KB from embed doc recipe - we don't want to display this automatic name
            if ($ctrl.retrievableKnowledge?.multimodalColumn) {
                return "Content embedded";
            }

            if ($ctrl.embeddingRecipeDesc) {
                return "Same column as embedding: " + $ctrl.embeddingRecipeDesc.knowledgeColumn;
            }

            // regular embedding, but no embedding recipe -> built using the Python API
            // use a generic name in this case
            return "Content embedded";
        };

        $ctrl.prefillRetrievalColumn = (ragllmSettings) => {
            if (ragllmSettings.retrievalSource === "CUSTOM" && !ragllmSettings.retrievalColumn) {
                ragllmSettings.retrievalColumn = $ctrl.metadataColumns[0];
            }
        };

        $ctrl.disableTextGuardrails = function (ragllmSettings) {
            ragllmSettings.ragSpecificGuardrails.faithfulnessSettings.enabled = false;
            ragllmSettings.ragSpecificGuardrails.relevancySettings.enabled = false;
        };

        $ctrl.disableMultimodalGuardrails = function (ragllmSettings) {
            ragllmSettings.ragSpecificGuardrails.multimodalFaithfulnessSettings.enabled = false;
            ragllmSettings.ragSpecificGuardrails.multimodalRelevancySettings.enabled = false;
        };

        $ctrl.shouldAlertRetrievalColumn = (ragllmSettings) => {
            if (ragllmSettings.retrievalSource === "EMBEDDING" || ragllmSettings.retrievalSource === "MULTIMODAL" || !ragllmSettings.retrievalColumn) {
                return false
            }
            return !$ctrl.metadataColumns.includes(ragllmSettings.retrievalColumn);
        };

        $ctrl.shouldAlertMultimodalColumn = (ragllmSettings) => {
            if (ragllmSettings.retrievalSource !== "MULTIMODAL" || !$ctrl.availableAugmentableLLMs) {
                return false
            }
            const selectedAugmentedLLM =  $ctrl.availableAugmentableLLMs.filter(llm => llm.id == ragllmSettings.llmId);
            if(!selectedAugmentedLLM || !selectedAugmentedLLM.length){
                return false;
            }
            return !selectedAugmentedLLM[0].supportsImageInputs; // Non-vlm augmented model is selected to use the multimodal column at retrieval => can leads to errors when using it
        };

    }
});

app.component('retrievalParamsSettings', {
    bindings: {
        retrievableKnowledge: '<',
        defaultPrompt: '<', // optional - used for setting the default custom prompt for RA LLMs but not yet for tools
        llmExposedWith: '='
    },
    templateUrl: '/templates/retrievable-knowledge/retrieval-params-settings.html',
    controller: function(VECTOR_STORE_HYBRID_SUPPORT, DataikuAPI, SemanticVersionService) {
        const $ctrl = this;

        $ctrl.$onInit = () => {
            if (!$ctrl.llmExposedWith.searchType) {
                $ctrl.llmExposedWith.searchType = "SIMILARITY";
            }
            if (!$ctrl.llmExposedWith.similarityThreshold) {
                $ctrl.llmExposedWith.similarityThreshold = 0.5;
            }

            $ctrl.elasticSearchData = null
            setSearchTypeDescriptions();
        };

        $ctrl.$onChanges = (changes) => {
            if (changes.retrievableKnowledge && !(changes.retrievableKnowledge.previousValue)) {
                if ($ctrl.retrievableKnowledge && $ctrl.retrievableKnowledge.vectorStoreType === "ELASTICSEARCH" && $ctrl.retrievableKnowledge.connection) {
                    DataikuAPI.admin.connections.get($ctrl.retrievableKnowledge.connection).success(function(connectionData) {
                        DataikuAPI.admin.connections.testElasticSearch(connectionData, null).success(function(testResponse) {
                            $ctrl.elasticSearchData = testResponse;
                            setSearchTypeDescriptions();
                        });
                    }).error(setErrorInScope.bind($ctrl));
                }
            }
        };

        $ctrl.diversityDocsUnavailableReason = function() {
            if($ctrl.retrievableKnowledge && $ctrl.retrievableKnowledge.vectorStoreType && $ctrl.retrievableKnowledge.vectorStoreType === 'AZURE_AI_SEARCH'){
                return "This option isn't available for the selected vector store type (Azure AI Search)";
            }
            return "";
        };

        $ctrl.hybridSearchSupported = function() {
            return $ctrl.retrievableKnowledge && VECTOR_STORE_HYBRID_SUPPORT.includes($ctrl.retrievableKnowledge.vectorStoreType);
        };

        $ctrl.hybridSearchUnavailableReason = function () {
            if (!$ctrl.hybridSearchSupported()) {
                return "Hybrid search is not supported for this vector store type";
            }
            if ($ctrl.retrievableKnowledge.vectorStoreType === "ELASTICSEARCH") {
                if (!$ctrl.elasticSearchData || !$ctrl.elasticSearchData.connectionOK) {
                    // No connection info yet, so we cannot determine if hybrid search is available
                    return "";
                }
                if ($ctrl.elasticSearchData.distributionName !== "ElasticSearch") {
                    $ctrl.llmExposedWith.useAdvancedReranking = false;
                    return "Hybrid search is only available for ElasticSearch distribution";
                }
                if (SemanticVersionService.compareVersions($ctrl.elasticSearchData.version, "8.4") <= 0) {
                    $ctrl.llmExposedWith.useAdvancedReranking = false;
                    return "Hybrid search is only available for ElasticSearch version 8.4 and above";
                }
            }
            return "";
        };

        function setSearchTypeDescriptions() {
            const similarityDescription = "Retrieve the n closest documents by similarity score.";
            const similarityThresholdDescription = "Retrieve all documents below the given similarity score threshold.";
            let diversityDocsDescription = "Helps the selected documents to cover a wider range of information while still matching the query. Uses MMR reranking.";
            let hybridDescription = "Combine both a similarity search and a keyword search to retrieve more relevant documents.";
            const diversityDocsUnavailableReason = $ctrl.diversityDocsUnavailableReason();
            const hybridSearchUnavailableReason = $ctrl.hybridSearchUnavailableReason();
            if (diversityDocsUnavailableReason) {
                diversityDocsDescription += '<div class="alert alert-warning alert-in-dropdown" style="margin: 10px 0 0 0;"><i class="icon-dku-warning"></i>' + diversityDocsUnavailableReason + '</div>';
            }
            if (hybridSearchUnavailableReason) {
                hybridDescription += '<div class="alert alert-warning alert-in-dropdown" style="margin: 10px 0 0 0;"><i class="icon-dku-warning"></i>' + hybridSearchUnavailableReason + '</div>';
            }
            $ctrl.diversityDocsUnavailable = !!diversityDocsUnavailableReason;
            $ctrl.hybridSearchUnavailable = !!hybridSearchUnavailableReason;
            $ctrl.searchTypeDescriptions = [similarityDescription, similarityThresholdDescription, diversityDocsDescription, hybridDescription]
        };
    }
});

app.component('mmrSearchSettings', {
    bindings: {
        llmExposedWith: '=',
        vectorStoreType: '<',
    },
    templateUrl: '/templates/retrievable-knowledge/mmr-search-settings.html',
    controller: function() {
        const $ctrl = this;
    }
});

app.component('hybridSearchSettings', {
    bindings: {
        llmExposedWith: '=',
        vectorStoreType: '<',
        connection: '<',
        elasticSearchData: '<'
    },
    templateUrl: '/templates/retrievable-knowledge/hybrid-search-settings.html',
    controller: function(DataikuAPI, VECTOR_STORE_HYBRID_SUPPORT, SemanticVersionService) {
        const $ctrl = this

        $ctrl.hybridSearchSupported = function () {
            return VECTOR_STORE_HYBRID_SUPPORT.includes($ctrl.vectorStoreType);
        }

        $ctrl.$onChanges = (changes) => {
            if (!$ctrl.hybridSearchSupported()) {
                $ctrl.llmExposedWith.useAdvancedReranking = false;
            }
        };

        $ctrl.advancedRerankingUnavailableReason = function () {
            if ($ctrl.vectorStoreType === "ELASTICSEARCH") {
                if (!$ctrl.elasticSearchData || !$ctrl.elasticSearchData.connectionOK) {
                    // No connection info yet, so we cannot determine if rrf is available
                    return "";
                }
                if (SemanticVersionService.compareVersions($ctrl.elasticSearchData.version, "8.16") <= 0) {
                    $ctrl.llmExposedWith.useAdvancedReranking = false;
                    return "RRF option is only available for ElasticSearch version 8.16 and above";
                }
            }
            return "";
        }
    }
});

app.component('searchInputStrategySettings', {
    bindings: {
        llmExposedWith: '=',
        defaultRewritePrompt: '=',
    },
    templateUrl: '/templates/retrievable-knowledge/search-input-strategy-settings.html',
    controller: function(DataikuAPI) {
        const $ctrl = this

        $ctrl.searchRewriteIsEnabled = function() {
            return $ctrl.llmExposedWith.searchInputStrategySettings?.strategy == "REWRITE_QUERY";
        };
    }
});

    app.constant("SNIPPET_FORMATS", [
        {key: 'TEXT', label: 'Text', helpText: 'The snippet is plain text'},
        {key: 'MARKDOWN', label: 'Markdown', helpText: 'The snippet is in Markdown format'},
        {key: 'HTML', label: 'HTML', helpText: 'The snippet is in HTML format'}
    ])

    app.component('metadataSelectField', {
        bindings: {
            label: '@',
            column: '=',
            format: '=',
            fieldType: '<',
            helpText: '@',
            possibleColumns: '<',
        },
        controller: function (SNIPPET_FORMATS) {
            const $ctrl = this;
            $ctrl.$onInit = function () {
                $ctrl.SNIPPET_FORMATS = SNIPPET_FORMATS;
                $ctrl.FORMAT_DESCRIPTIONS = SNIPPET_FORMATS.map(format => format.helpText);
            }

        },
        template: `
    <div class="control-group m0">
      <label class="control-label">{{ $ctrl.label }}</label>
      <div class="controls horizontal-flex" >
        <ng2-typeahead
            [(value)]="$ctrl.column"
            [suggestions]="$ctrl.possibleColumns"
            placeholder="Select metadata">
        </ng2-typeahead>
        <select
            class="mleft4"
            ng-if="$ctrl.fieldType === 'snippetMetadata'"
            layout="list"
            dku-bs-select
            ng-model="$ctrl.format"
            options-descriptions="$ctrl.FORMAT_DESCRIPTIONS" >
            <option ng-repeat="format in $ctrl.SNIPPET_FORMATS" value="{{format.key}}">{{format.label}}</option>
        </select>
      </div>
    </div>
  `
    });

    app.component('additionalInfoSourcesDefaultMetadata', {
        bindings: {
            model: '=',
            metadataColumns: '<',
        },
        controller: function () {
            const $ctrl = this;

            $ctrl.$onInit = function () {
                $ctrl.metadataItems = $ctrl.model.metadataInSources.map(col => ({name: col}))
                $ctrl.metadataToAdd = $ctrl.getMetadataToAdd();
            }

            $ctrl.updateDefaultMetadata = function () {
                $ctrl.model.metadataInSources = $ctrl.metadataItems
                    .map(item => {
                        item.name = (item.name || '').trim();
                        return item.name;
                    })
                    .filter(Boolean);
                $ctrl.metadataToAdd.splice(0, $ctrl.metadataToAdd.length, ...$ctrl.getMetadataToAdd().map(item => item.name));
                $ctrl.duplicatedMetadata = $ctrl.model.metadataInSources
                    .filter((name, i, all) => all.indexOf(name) !== i);
            }

            $ctrl.addDefault = function () {
                $ctrl.metadataItems = [...$ctrl.metadataItems, {name: ""}];
            }
            $ctrl.getMetadataToAdd = function () {
                if (!$ctrl.metadataColumns || $ctrl.metadataColumns.length === 0) {
                    return [];
                }
                const existingMetadata = new Set($ctrl.metadataItems.map(item => item.name));
                return $ctrl.metadataColumns
                    .filter(col => col !== undefined && !existingMetadata.has(col)).map(col => ({name: col}));
            }
            $ctrl.fillAllDefaultMetadata = function () {
                $ctrl.metadataItems = [...$ctrl.metadataItems, ...$ctrl.getMetadataToAdd()];
                $ctrl.updateDefaultMetadata()
            }

        },
        template: `

    <div class="control-group">
        <label class="control-label">Standard</label>
        <div class="controls">
            <editable-list ng-model="$ctrl.metadataItems"
                no-change-on-add="true"
                transcope="{
                    ctrl: $ctrl
                }"
                disable-add="true"
                template="{name: ''}"
                on-change="$ctrl.updateDefaultMetadata()"
            >
                <ng2-typeahead
                    [(value)]="it.name"
                    [suggestions]="ctrl.metadataToAdd"
                    placeholder="Select metadata">
                </ng2-typeahead>
            </editable-list>
            <div data-block="newItem" class="dropdown"">
                <button
                    type="button"
                    data-qa-add-button
                    ng-click="$ctrl.addDefault()"
                    class="btn btn--primary btn--text editable-list__add-label btn--dku-icon" >
                    <i class="dku-icon-plus-16"></i>Add Metadata
                </button>
                <span class="retrieval-augmented-llm-button_separator" ng-if="$ctrl.metadataToAdd.length > 0"></span>
                <button class="btn m0 btn--text btn--primary btn--icon btn--dku-icon" data-toggle="dropdown" dropdown-position="fixed" ng-if="$ctrl.metadataToAdd.length > 0">
                    <i class="dku-icon-caret-down-16"></i>
                </button>
                <ul class="dropdown-menu" ng-if="$ctrl.metadataToAdd.length > 0">
                    <li><a ng-click="$ctrl.fillAllDefaultMetadata()" ng-if="$ctrl.metadataToAdd.length > 0">
                            Add all metadata
                        </a>
                    </li>
                </ul>
            </div>
            <div class="alert alert-warning mtop8 mbot0 horizontal-flex" ng-if="$ctrl.duplicatedMetadata.length">
            <div class="padright16"><i class="icon-dku-warning" /></div>
              You have the following metadata {{$ctrl.duplicatedMetadata.length > 1? 's': ''}} declared more than once:
              <strong>&nbsp{{$ctrl.duplicatedMetadata.join(',&nbsp')}}</strong>
            </div>
        </div>
    </div>
    `
    });


    app.constant("METADATA_FIELDS", [
        {name: 'titleMetadata', label: 'Title', helpText: 'Title to display'},
        {name: 'urlMetadata', label: 'URL', helpText: 'URL to make a link from the source (if relevant)'},
        {
            name: 'thumbnailURLMetadata',
            label: 'Thumbnail URL',
            helpText: 'URL to an image to display a thumbnail for the source (if relevant)'
        },
        {
            name: 'snippetMetadata',
            label: 'Snippet',
            helpText: 'Snippet text to display. If left empty, the retrieval text is used instead'
        }
    ]);
    app.component('additionalInfoSourcesFormattingConfig', {
        bindings: {
            model: '=',
            metadataColumns: '<',
        },
        controller: function ($timeout, METADATA_FIELDS) {
            const $ctrl = this;

            $ctrl.$onInit = function () {
                $ctrl.METADATA_FIELDS = METADATA_FIELDS;
                $ctrl.visibleFields = METADATA_FIELDS.filter(field => $ctrl.model[field.name] !== undefined);
                if ($ctrl.model['snippetFormat'] === undefined) {
                    $ctrl.model.snippetFormat = 'TEXT'; // Default snippet format
                }
            }

            $ctrl.addField = function (field) {
                if (!$ctrl.visibleFields.some(f => f.name === field.name)) {
                    $ctrl.visibleFields = [...$ctrl.visibleFields, field];
                }
            }

            $ctrl.onDeleteField = function () {
                $ctrl.getRemainingFields().map(field => $ctrl.model[field.name] = undefined)
            }

            $ctrl.getRemainingFields = function () {
                const usedNames = new Set($ctrl.visibleFields.map(f => f.name));
                return $ctrl.METADATA_FIELDS.filter(f => !usedNames.has(f.name));
            };
        },
        template: `
    <p>Add metadata to the LLM output. Choose standard to include it as is, or select a role (e.g., Title, URL) to format it accordingly</p>
    <additional-info-sources-default-metadata
        model="$ctrl.model"
        metadata-columns="$ctrl.metadataColumns">
    </additional-info-sources-default-metadata>
    <div class="mtop16"></div>
    <div class="control-group" ng-if="$ctrl.visibleFields.length > 0">
        <label class="control-label">With role</label>
    </div>
    <editable-list
        ng-model="$ctrl.visibleFields"
        on-remove="$ctrl.onDeleteField()"
        disable-add="true"
        transcope="{
            ctrl: $ctrl
        }">
        <metadata-select-field
            label="{{it.label}}"
            column="ctrl.model[it.name]"
            format="ctrl.model.snippetFormat"
            field-type="it.name"
            help-text="{{it.helpText}}"
            possible-columns="ctrl.metadataColumns">
        </metadata-select-field>
    </editable-list>
    <div class="control-group">
        <label ng-if="$ctrl.visibleFields.length==0" class="control-label">With role</label>
        <div class="controls">
            <div data-block="newItem" class="dropdown"">
                <button
                    type="button"
                    data-qa-add-button
                    data-toggle="dropdown"
                    dropdown-position="fixed"
                    class="btn btn--primary btn--text editable-list__add-label btn--dku-icon" >
                    <i class="dku-icon-plus-16"></i>Add Role<i class="dku-icon-caret-down-16 mleft4"></i>
                </button>
                <ul class="dropdown-menu detailed-dropdown-menu">
                    <li ng-repeat="field in $ctrl.getRemainingFields()" class="detailed-dropdown-menu__item" ng-click="$ctrl.addField(field)">
                        <div class="detailed-dropdown-menu__item-info">
                            <div class="detailed-dropdown-menu__item-title">{{field.label}}</div>
                            <div class="detailed-dropdown-menu__item-description">{{field.helpText}}</div>
                        </div>
                    </li>
                    <li disabled="true" ng-repeat="field in $ctrl.visibleFields" class="detailed-dropdown-menu__item disabled">
                        <div disabled="true" class="detailed-dropdown-menu__item-info">
                            <div disabled="true" class="detailed-dropdown-menu__item-title">{{field.label}}</div>
                            <div disabled="true" class="detailed-dropdown-menu__item-description">{{field.helpText}}</div>
                        </div>
                    </li>
                </ul>
            </div>
        </div>
    </div>
    `
    });

app.controller("RetrievableKnowledgeUsageController", function($scope, $rootScope, TopNav, $controller, CreateModalFromTemplate, PluginsService, ActivityIndicator, $state, DataikuAPI, ActiveProjectKey, $stateParams) {
    $controller('_RetrievableKnowledgeCommonController', {$scope: $scope});

    TopNav.setLocation(TopNav.TOP_RETRIEVABLE_KNOWLEDGE, TopNav.LEFT_RETRIEVABLE_KNOWLEDGE, TopNav.TABS_RETRIEVABLE_KNOWLEDGE, "usage");

    $scope.answersDocRef = $rootScope.versionDocRoot + "generative-ai/answers.html";

    $scope.openKBSearchToolModal = function () {
        CreateModalFromTemplate('/templates/retrievable-knowledge/kb-search-agent-tool.html', $scope, 'KBSearchToolModalController');
    };

    $scope.openTestInPythonNotebookModal = function () {
        CreateModalFromTemplate('/templates/retrievable-knowledge/test-in-python-notebook.html', $scope, 'TestInPythonNotebookModalController');
    };

    $scope.openUseAnswersWebappModal = function () {
        PluginsService.checkInstalledStatus('document-question-answering').then((isPluginInstalled) => {
            if (isPluginInstalled) {
                CreateModalFromTemplate('/templates/llm/use-answers-webapp.html', $scope, 'UseAnswersWebappModalController');
            } else {
                ActivityIndicator.error('The Answers plugin is not installed. Please refer to the documentation.');
            }
        });
    };

    $scope.showCreateRetrievalAugmentedLLMModal = function (preselectedInput) {
        CreateModalFromTemplate('/templates/savedmodels/retrieval-augmented-llm/create-retrieval-augmented-llm-modal.html', $scope, 'CreateRetrievalAugmentedLLMModalController', function (newScope) {
            newScope.input.preselectedInput = [{
                id: $scope.retrievableKnowledge.id,
                displayName: $scope.retrievableKnowledge.name,
                type: 'RETRIEVABLE_KNOWLEDGE',
            }];
        });
    };

    DataikuAPI.retrievableknowledge
        .getRetrievalAugmentedLLMList(ActiveProjectKey.get(), $stateParams.retrievableKnowledgeId)
        .success(function (data) {
            $scope.retrievalAugmentedLLMs = data;
        })
        .error(setErrorInScope.bind($scope));

    $scope.goToRetrievalAugmentedLLMActiveVersion = function (retrievalAugmentedLLM) {
        $state.go('projects.project.savedmodels.savedmodel.retrievalaugmentedllm.design', {
            fullModelId: `S-${$stateParams.projectKey}-${retrievalAugmentedLLM.id}-${retrievalAugmentedLLM.activeVersion}`,
            smId: retrievalAugmentedLLM.id,
        });
    };

});


app.controller("RetrievableKnowledgeSettingsController", function($scope, $rootScope, DataikuAPI, CreateModalFromTemplate, $state, $stateParams, TopNav, $controller,
    ActivityIndicator, ComputableSchemaRecipeSave, ActiveProjectKey, DOCUMENT_SPLITTING_METHOD_MAP, Dialogs) {
    $controller('_RetrievableKnowledgeCommonController', {$scope: $scope});
    TopNav.setLocation(TopNav.TOP_RETRIEVABLE_KNOWLEDGE, TopNav.LEFT_RETRIEVABLE_KNOWLEDGE, TopNav.TABS_RETRIEVABLE_KNOWLEDGE, "settings");

    $scope.showCodeEnvVersionWarning = false;
    $scope.codeEnvVersionWarningMessage = "";

    $scope.userFriendlyDocSplittingMethod = function(method) {
        return DOCUMENT_SPLITTING_METHOD_MAP[method];
    };
    $scope.clear = function() {
        $scope.save();
        return CreateModalFromTemplate("/templates/retrievable-knowledge/clear-modal.html", $scope, null, function(modalScope) {
            modalScope.confirm = function() {
                DataikuAPI.retrievableknowledge.clear(ActiveProjectKey.get(), $stateParams.retrievableKnowledgeId).then(function ({data}) {
                    modalScope.dismiss();
                    Dialogs.infoMessagesDisplayOnly($scope, "Clear result", data);
                }).catch(setErrorInScope.bind($scope));
            }
        });
    }

    DataikuAPI.pretrainedModels.listAvailableLLMs($stateParams.projectKey, "TEXT_EMBEDDING_EXTRACTION").then(function({data}) {
        $scope.availableEmbeddingLLMs = data.identifiers;
    }).catch(setErrorInScope.bind($scope));

    $scope.$watch("retrievableKnowledge.envSelection.envMode", function() {
        $scope.checkCodeEnvCompat();
    });

    $scope.$watch("retrievableKnowledge.envSelection.envName",  function() {
        $scope.checkCodeEnvCompat();
    });

});

app.controller('KBSearchToolModalController', function($scope, $state, $stateParams, DataikuAPI, StringUtils, $rootScope, AgentToolService, WT1) {
    $scope.newAgentTool = {
        name: '',
        id: '',
        type: 'VectorStoreSearch',
        quickTestQuery: {"input": AgentToolService.getAgentToolQuickTestQueryForType('VectorStoreSearch'), "context": {}}
    };

    $scope.create = function() {
        DataikuAPI.agentTools.createFromKB($stateParams.projectKey, $scope.newAgentTool, $stateParams.retrievableKnowledgeId).success(function(data) {
            WT1.event(
                'agent-tool-create', {
                    agentToolType: 'VectorStoreSearch',
                });
            $state.go("projects.project.agenttools.agenttool", {agentToolId: data.id})
            $scope.dismiss();
        }).error(setErrorInScope.bind($scope));
    }
});

app.controller('TestInPythonNotebookModalController', function ($scope, $stateParams, $state, DataikuAPI) {
    $scope.notebook = {
        name: `KB Test Notebook - ${$scope.retrievableKnowledge.name}`,
    };

    $scope.editInStudio = function () {
        const templateDesc = {
            language: 'python',
            type: 'RETRIEVABLE_KNOWLEDGE',
            id: '01-dku-test-kb-notebook',
            origin: 'BUILTIN',
        };

        $scope.save(true).then(function () {
            DataikuAPI.jupyterNotebooks
                .newNotebookForKB($stateParams.projectKey, $scope.notebook.name, $scope.retrievableKnowledge.id, templateDesc)
                .then(function ({ data }) {
                    $state.go("projects.project.notebooks.jupyter_notebook", {notebookId : data.name})
                })
                .catch(setErrorInScope.bind($scope));
        });
    };
});

app.controller('UseAnswersWebappModalController', function ($scope, $stateParams, $state, DataikuAPI, WebAppsService, PluginConfigUtils) {
    $scope.webapp = {
        name: `Answers RAG - ${$scope.retrievableKnowledge.name}`,
    };

    const answersWebAppId = 'webapp_document-question-answering_document-intelligence-explorer';

    $scope.createAnswersWebapp = function () {
        const webAppLoadedDesc = WebAppsService.getWebAppLoadedDesc(answersWebAppId);
        const config = {
            retrieval_mode: 'kb',
            knowledge_bank_id: $scope.retrievableKnowledge.id,
        };
        PluginConfigUtils.setDefaultValues(webAppLoadedDesc.desc.params, config);
        DataikuAPI.webapps
            .create($stateParams.projectKey, $scope.webapp.name, answersWebAppId, null, config)
            .success(function (result) {
                $state.go('projects.project.webapps.webapp.edit', { projectKey: $stateParams.projectKey, webAppId: result.webAppId });
            })
            .catch(setErrorInScope.bind($scope));
    };
});

app.component('vectorStoreSelector', {
    bindings: {
        retrievableKnowledge: '=',
        rkAtEmbedding: '<',
        disableIndexNaming: '<'
    },
    templateUrl: '/templates/retrievable-knowledge/vector-store-selector.html',
    controller: function ($scope, RetrievableKnowledgeUtils, VECTOR_STORE_TYPE_MAP, DataikuAPI) {
        const $ctrl = this;
        $scope.VECTOR_STORE_TYPE_MAP = VECTOR_STORE_TYPE_MAP;

        $scope.updateIndexName = RetrievableKnowledgeUtils.updateIndexName;
        $scope.hasConnection = RetrievableKnowledgeUtils.hasConnection;
        $scope.hasIndex = RetrievableKnowledgeUtils.hasIndex;

        $scope.availableVectorStoreConnectionsList = [];
        DataikuAPI.retrievableknowledge.getVectorStoreConnections().then(function({data}) {
            $scope.availableVectorStoreConnectionsList = data;
        }).catch(setErrorInScope.bind($scope));

        $scope.availableVectorStoreConnections = function () {
            if (!$scope.availableVectorStoreConnectionsList) return [];
            return $scope.availableVectorStoreConnectionsList.filter((conn) => conn.vectorStoreType === $ctrl.retrievableKnowledge.vectorStoreType);
        };
    },
});

app.service('RetrievableKnowledgeUtils', function () {
    function hasConnection(retrievableKnowledge) {
        return retrievableKnowledge && ['PINECONE', 'ELASTICSEARCH', 'AZURE_AI_SEARCH', 'VERTEX_AI_GCS_BASED'].indexOf(retrievableKnowledge.vectorStoreType) >= 0;
    }

    function hasIndex(retrievableKnowledge) {
        return retrievableKnowledge && ['ELASTICSEARCH', 'AZURE_AI_SEARCH', 'VERTEX_AI_GCS_BASED'].indexOf(retrievableKnowledge.vectorStoreType) >= 0;
    }

    function updateIndexName(retrievableKnowledge, disableIndexNaming) {
        if (hasIndex(retrievableKnowledge) && !retrievableKnowledge.indexName && !disableIndexNaming) {
            retrievableKnowledge.indexName = '${projectKey}_kb_' + retrievableKnowledge.id;
        }
    }

    return { hasConnection, hasIndex, updateIndexName };
});

app.component("guardrailItem", {
    bindings: {
        "guardrailSetting": "=", // The setting object, see the FaithfulnessSettings java class for example
        "displayName": "<",
        "shortDescription": "<",
        "disabled": "<",
        "isMultimodal": "<",
    },
    template: `
<div class="control-group">
    <label class="guardrail-label control-label" for="{{ $ctrl.displayName + '-group' }}">{{ ($ctrl.isMultimodal ? 'Multimodal ' : '') + $ctrl.displayName }}
        <i class="dku-icon-question-circle-outline-16 dibvat detailed-metrics-table__metrics-info" style="color: #666666;" toggle="tooltip-right" title="{{ $ctrl.shortDescription }}"></i>
    </label>
    <div class="controls flex-row-fluid">
        <label class="dku-toggle">
            <input type="checkbox" ng-model="$ctrl.guardrailSetting.enabled" disabled-if-message="$ctrl.disabled ? 'Disabled' : null">
            <span></span>
        </label>
        <div class="mright4">
            <ng-container ng-show="$ctrl.guardrailSetting.enabled && !$ctrl.isMultimodal">
                <input id="{{ $ctrl.displayName + '-group' }}" type="range"
                    min="0" max="1" step="0.01"
                    ng-model="$ctrl.guardrailSetting.threshold">
            </ng-container>
        </div>
        <input type="number" class="input-fit-content" step="0.01" min="0" ng-show="$ctrl.guardrailSetting.enabled && !$ctrl.isMultimodal"
            max="1" ng-model="$ctrl.guardrailSetting.threshold"/>
    </div>
    <div class="controls guardrail-action-control" ng-show="$ctrl.guardrailSetting.enabled">
        <label>
            <basic-select
                items="$ctrl.handlingOptions"
                ng-model="$ctrl.guardrailSetting.handling"
                bind-value="handlingValue"
                bind-label="handlingDisplayValue">
                </basic-select>
            <div class="help-inline" ng-show="!$ctrl.isMultimodal">Action for scores below threshold</div>
            <div class="help-inline" ng-show="$ctrl.isMultimodal">{{ 'Action for low (score=0) ' + $ctrl.displayName }}</div>
        </label>
        <label>
            <textarea id="{{'answerOverwrite-' + $index}}" ng-show="$ctrl.guardrailSetting.handling == 'OVERWRITE_ANSWER'"
                        class="textarea--full-width"
                        ng-model="$ctrl.guardrailSetting.answerOverwrite"/>
        </label>
    </div>
</div>
`,
    controller: function() {
        const ctrl = this;
        ctrl.handlingOptions = [
            {
                "handlingValue": "FAIL",
                "handlingDisplayValue": "Fail"
            },
            {
                "handlingValue": "OVERWRITE_ANSWER",
                "handlingDisplayValue": "Overwrite answer"
            },
        ]
    }
});

})();
