(function(){
'use strict';
const app = angular.module('dataiku.agents', ['dataiku.ml.report', 'dataiku.lambda']);

app.controller("_AgentSavedModelController", function($scope, $rootScope, $state, $controller, $stateParams, DataikuAPI, ActiveProjectKey, FullModelLikeIdUtils, $q, DKUtils, Dialogs, WT1) {
    $controller("_SavedModelReportController", {$scope});

    DataikuAPI.savedmodels.llmCommon.getStatus(ActiveProjectKey.get(), $stateParams.smId).success(function(data){
        $scope.fillVersionSelectorStuff(data);
    });

    const currentVersionId = FullModelLikeIdUtils.parse($stateParams.fullModelId).versionId;
    $scope.currentVersionId = currentVersionId;

    $scope.$watch("savedModel", function(nv) {
        if (!nv) return;

        $scope.currentVersionIdx = nv.inlineVersions.findIndex(v => v.versionId === currentVersionId);
        if ($scope.currentVersionIdx < 0) {
            return;
        }
        $scope.currentlySavedInlineModel = angular.copy(nv.inlineVersions[$scope.currentVersionIdx]);
        $scope.currentVersionData = nv.inlineVersions[$scope.currentVersionIdx];
    });

    $scope.save = function() {
        if (!$scope.isDirty()) {
            return $q.when('agent not dirty');
        }

        const deferred = $q.defer();
        const saveAfterConflictCheck = function() {
            DataikuAPI.savedmodels.agents.save($scope.savedModel, $scope.currentVersionId).success(function(data) {
                WT1.event(
                    'agent-save', {
                        savedModelType: data.savedModelType,
                    });
                $scope.savedModel = data;
                $scope.currentlySavedInlineModel = angular.copy($scope.savedModel.inlineVersions[$scope.currentVersionIdx]);
                deferred.resolve('agent saved');
            }, () => deferred.reject());
        };

        DataikuAPI.savedmodels.agents.checkSaveConflict($scope.savedModel).success(function(conflictResult) {
            if (!conflictResult.canBeSaved) {
                Dialogs.openConflictDialog($scope, conflictResult).then(
                    function(resolutionMethod) {
                        if (resolutionMethod === 'erase') {
                            saveAfterConflictCheck();
                        } else if (resolutionMethod === 'ignore') {
                            deferred.reject();
                            DKUtils.reloadState();
                        }
                    }
                );
            } else {
                saveAfterConflictCheck();
            }
        }).error(setErrorInScope.bind($scope));

        return deferred.promise;
    };
    $rootScope.saveAgentModel = $scope.save;

    $scope.saveIgnoringQuickTestQuery = function() {
        if (!$scope.isDirty()) {
            return $q.when('agent not dirty');
        }

        let frankenVersion = angular.copy($scope.currentlySavedInlineModel);
        frankenVersion["quickTestQuery"] = $scope.savedModel.inlineVersions[$scope.currentVersionIdx]["quickTestQuery"];
        if (angular.equals(frankenVersion, $scope.savedModel.inlineVersions[$scope.currentVersionIdx])) {
            return $q.when('only difference is the quicktest query, not saving');
        }

        return $scope.save();
    };

    $scope.isDirty = function() {
        if (!$scope.savedModel) return false;
        return !angular.equals($scope.savedModel.inlineVersions[$scope.currentVersionIdx], $scope.currentlySavedInlineModel);
    };
    $rootScope.agentModelIsDirty = $scope.isDirty;

    checkChangesBeforeLeaving($scope, $scope.isDirty);
});

app.controller("AgentSavedModelDesignController", function($scope, $rootScope, $state, $stateParams, $controller, ActivityIndicator, CodeBasedEditorUtils,
                                                           CreateModalFromTemplate, DataikuAPI, PluginConfigUtils, PluginsService, TopNav, AgentDefaultQuickTestQuery) {
    $controller("_AgentSavedModelController", {$scope});

    $scope.editorOptions = CodeBasedEditorUtils.editorOptions('text/x-python', $scope, true);
    CodeBasedEditorUtils.registerBroadcastSelectionHandler($scope.editorOptions);
    CodeBasedEditorUtils.setRecipeScript($scope.script);

    $scope.$watch("savedModel", (nv) => {
        if (!nv) return;

        if ($scope.savedModel.savedModelType === "PLUGIN_AGENT") {
            $scope.loadedDesc =  $rootScope.appConfig.customAgents.find(x => x.desc.id === $scope.currentVersionData.pluginAgentType);
            $scope.desc = $scope.loadedDesc.desc;
            $scope.pluginDesc = PluginsService.getOwnerPluginDesc($scope.loadedDesc);

            if ($scope.pluginDesc && $scope.desc && $scope.desc.params) {
                PluginConfigUtils.setDefaultValues($scope.desc.params, $scope.currentVersionData.pluginAgentConfig);
            }
        }

        if ($scope.savedModel.savedModelType === "TOOLS_USING_AGENT") {
            $scope.onToolLLMChange = function() {
                if ($scope.currentVersionData.toolsUsingAgentSettings.llmId && $scope.availableLLMs) {
                    $scope.activeLLM = $scope.availableLLMs.find(l => l.id === $scope.currentVersionData.toolsUsingAgentSettings.llmId);
                }
            };

            DataikuAPI.agentTools.listAvailable($stateParams.projectKey).success(function(data) {
                $scope.availableTools = data;
            }).error(setErrorInScope.bind($scope));

            DataikuAPI.codeenvs.checkAgentsCodeEnv($stateParams.projectKey)
                .then(function ({data}) {
                    $scope.codeEnvType = data["codeEnvType"];
                    $scope.codeEnvError = data["codeEnvError"];
                }).catch(setErrorInScope.bind($scope));
        }

        DataikuAPI.pretrainedModels.listAvailableLLMs($stateParams.projectKey, "GENERIC_COMPLETION").success(function(data) {
            $scope.availableLLMs = data.identifiers;

            if ($scope.savedModel.savedModelType === "TOOLS_USING_AGENT") {
                // Don't allow selecting myself as the LLM for this agent, to avoid infinite loop
                $scope.availableLLMs = $scope.availableLLMs.filter(llm => !llm.id.startsWith("agent:" + $scope.savedModel.id));
                $scope.onToolLLMChange();
            }
        }).error(setErrorInScope.bind($scope));

        if ($scope.currentVersionData && !$scope.currentVersionData.quickTestQuery) {
            $scope.currentVersionData.quickTestQuery = AgentDefaultQuickTestQuery;
            $scope.savedModel.inlineVersions[$scope.currentVersionIdx].quickTestQuery = $scope.currentVersionData.quickTestQuery;
            $scope.save();
        }

        $scope.internalCodeEnvsHRef = function() {
            if ($scope.appConfig.isAutomation) {
                return $state.href("admin.codeenvs-automation.internal");
            } else {
                return $state.href("admin.codeenvs-design.internal");
            }
        }

        if (!$scope.noSetLoc) {
            if ($scope.savedModel.savedModelType === "PLUGIN_AGENT"){
                TopNav.setLocation(TopNav.TOP_GENAI_MODELS, TopNav.LEFT_GENAI_MODELS, "PLUGIN_AGENT-SAVED_MODEL-VERSION", "design");
            } else{
                TopNav.setLocation(TopNav.TOP_GENAI_MODELS, TopNav.LEFT_GENAI_MODELS, "AGENT-SAVED_MODEL-VERSION", "design");
            }
        }
    });

    $scope.createNewTool = function ($event) {
        $event.preventDefault();
        const stateName = 'projects.project.agenttools.list';
        if ($scope.isDirty()) {
            $scope.save().then(function () {
                ActivityIndicator.success("Saved");
                $scope.$state.go(stateName, { createTool: true });
            }).error(setErrorInScope.bind($scope));
        } else {
            $scope.$state.go(stateName, { createTool: true });
        }
    };

    $scope.showToolDescriptionModal = function(usedTool){
        DataikuAPI.agentTools.get($scope.savedModel.projectKey, usedTool.toolRef).success(agentTool => {
            DataikuAPI.agentTools.getDescriptor(agentTool.projectKey, agentTool.id).success(descriptor => {
                $scope.toolDescriptionModal = {
                    // remove the global description that was appended by the back-end in get-descriptor to show it in another field
                    internalDescription: descriptor.description.replace("\n\n" + agentTool.additionalDescriptionForLLM, ""),
                    globalDescription: agentTool.additionalDescriptionForLLM,
                    agentSpecificDescription: usedTool.additionalDescription || "",
                }
                CreateModalFromTemplate('/templates/savedmodels/agents/agent-tool-description-modal.html', $scope);
            }).error(setErrorInScope.bind($scope));
        }).error(setErrorInScope.bind($scope));
    }
});

app.controller("AgentSavedModelSettingsController", function($scope, $controller, $stateParams, DataikuAPI, TopNav) {
    $controller("_AgentSavedModelController", { $scope });

    $scope.$watch("savedModel", (nv) => {
        if (!nv) return;

        switch ($scope.savedModel.savedModelType) {
            case "PYTHON_AGENT":
                $scope.currentAgentSettings = $scope.currentVersionData.pythonAgentSettings;
                break;
            case "PLUGIN_AGENT":
                $scope.currentAgentSettings = $scope.currentVersionData.pluginAgentSettings;
                break;
            case "TOOLS_USING_AGENT":
                $scope.currentAgentSettings = $scope.currentVersionData.toolsUsingAgentSettings;
                break;
        }

        if ($scope.savedModel.savedModelType === "TOOLS_USING_AGENT") {
            DataikuAPI.pretrainedModels.listAvailableLLMs($stateParams.projectKey, "GENERIC_COMPLETION").success(function(data) {
                if ($scope.currentAgentSettings.llmId && data.identifiers) {
                    $scope.activeLLM = data.identifiers.find(l => l.id === $scope.currentAgentSettings.llmId);
                }
            }).error(setErrorInScope.bind($scope));
        }

        if (!$scope.noSetLoc) {
            if ($scope.savedModel.savedModelType === "PLUGIN_AGENT"){
                TopNav.setLocation(TopNav.TOP_GENAI_MODELS, TopNav.LEFT_GENAI_MODELS, "PLUGIN_AGENT-SAVED_MODEL-VERSION", "settings");
            } else{
                TopNav.setLocation(TopNav.TOP_GENAI_MODELS, TopNav.LEFT_GENAI_MODELS, "AGENT-SAVED_MODEL-VERSION", "settings");
            }
        }
    });
});

app.controller("AgentVersionHistoryController", function($scope, $stateParams, $state, $controller, TopNav) {
    $controller("_AgentSavedModelController", {$scope});
    $scope.redirectCallback = function() {
        $state.go('projects.project.savedmodels.savedmodel.versions', {projectKey:$stateParams.projectKey, savedModelId:$scope.savedModel.id});
    };
    TopNav.setLocation(TopNav.TOP_GENAI_MODELS, TopNav.LEFT_GENAI_MODELS, "AGENT-SAVED_MODEL-VERSION", "history");
});


app.controller("AgentSavedModelLogsController", function($scope, $stateParams, $controller, DataikuAPI, TopNav) {
    $controller("_AgentSavedModelController", { $scope });

    $scope.listLogs = function() {
        DataikuAPI.savedmodels.agents.listLogs($stateParams.projectKey, $scope.savedModel.id, $scope.currentVersionId).success(function(data) {
            $scope.agentLogs = data;
        }).error(setErrorInScope.bind($scope));
    }

    $scope.logGet = DataikuAPI.savedmodels.agents.getLog;
    $scope.logDownloadURL = function(projectKey, savedModelId, versionId, logName) {
        return "/dip/api/savedmodels/agents/stream-log?projectKey=" + projectKey + "&smId=" + encodeURIComponent(savedModelId) + "&versionId=" + encodeURIComponent(versionId) + "&logName=" + encodeURIComponent(logName);
    }

    $scope.$watch("savedModel", (nv) => {
        if (!nv) return;

        $scope.listLogs();

        if (!$scope.noSetLoc) {
            if ($scope.savedModel.savedModelType === "PLUGIN_AGENT"){
                TopNav.setLocation(TopNav.TOP_GENAI_MODELS, TopNav.LEFT_GENAI_MODELS, "PLUGIN_AGENT-SAVED_MODEL-VERSION", "logs");
            } else{
                TopNav.setLocation(TopNav.TOP_GENAI_MODELS, TopNav.LEFT_GENAI_MODELS, "AGENT-SAVED_MODEL-VERSION", "logs");
            }
        }
    });
});

})();