(function() {
    'use strict';

    var app = angular.module('dataiku.shaker.table', ['dataiku.filters', 'platypus.utils', 'dataiku.constants']);

    const BIN_UNCOLORED = 0;
    const BIN_EMPTY = -1;
    //const BIN_HAPAX = -2;
    //const BIN_INVALID = -3;
    const EVEN = " even";
    const ODD = " odd";

    const binCache = new Map([
        ["CA1-1", 1],
        ["CA1-2", 2],
        ["CA1-3", 3],
        ["CA1-4", 4],
        ["CA1-5", 5],
        ["CA1-6", 6],
        ["CA1-7", 7],
        ["CA1-8", 8],
        ["CA1-9", 9],
        ["CA1-10", 10],
        ["CA1-11", 11],
        ["CA1--1", -1],
        ["CA1--2", -2],
        ["CB-1", 1],
        ["CB-2", 2],
        ["CB--3", -3],
        ["CN-1", 1],
        ["CN-2", 2],
        ["CN-3", 3],
        ["CN-4", 4],
        ["CN-5", 5],
        ["CN--1", -1],
        ["CN--3", -3],
        ["CG-1", 1],
        ["CG-2", 2],
        ["CG--3", -3],
    ]);

    app.service('ShakerTableModel', function() {
        return function(tableData, scope, maxPageWidth, maxPageHeight) {
            var PAGE_WIDTH = !maxPageWidth ? Math.pow(2,5) : maxPageWidth;
            var PAGE_HEIGHT = !maxPageHeight ? Math.pow(2,6) : maxPageHeight;

            var pageFromData = function(I,J,data) {
                return function(i,j)  {
                    var offset = (i-I)*data.nbCols + (j-J);
                    return {
                        content: data.content[offset],
                        highlightedIndices: data.highlightedIndices && data.highlightedIndices[offset],
                        status: data.status[offset],
                        colorBin : data.colorBin ? data.colorBin[offset] : null,
                        origRowIdx : data.origRowIdx[i-I],
                        rowId: i,
                        colId: j,
                    };
                };
            };

            tableData = $.extend(
                new fattable.PagedAsyncTableModel(),
                tableData,
            {
                getHeader: function(j, cb) {
                    // Here we fork the scope for each header to append
                    // a new header property.
                    var newScope = scope.$new(false);
                    newScope.header = tableData.headers[j];
                    cb(newScope);
                }
            ,
                searchHeader: function(q) {
                    q = q.toLowerCase();
                    var results = [];
                    for (let i = 0; i < tableData.headers.length; i++) {
                        let header = tableData.headers[i];
                        if (header.name.toLowerCase() == q) {
                            results.push(i);
                        }
                    }
                    for (let i = 0; i < tableData.headers.length; i++) {
                        let header = tableData.headers[i];
                        if ((header.name.toLowerCase().indexOf(q) != -1) && (results.indexOf(i) == -1) ) {
                            results.push(i);
                        }
                    }
                    return results;

                }
            ,
                hasHeader:  function() {
                    return true;  // we are synchronous for headers.
                }
            ,
                fetchCellPage: function(pageName, cb) {
                    var coords =  JSON.parse(pageName);
                    var I = coords[0];
                    var J = coords[1];
                    var nbRequestedRows = Math.min(this.totalKeptRows - I, PAGE_HEIGHT);
                    var nbRequestedCols = Math.min(this.allColumnNames.length - J, PAGE_WIDTH);
                    var promise = scope.getTableChunk(I, nbRequestedRows, J, nbRequestedCols);
                    promise.then(function(resp) {
                        var page = pageFromData(I,J,resp);
                        cb(page);
                    });
                }
            ,
                cellPageName: function(i,j) {
                    return JSON.stringify([Math.floor(i / PAGE_HEIGHT) * PAGE_HEIGHT, Math.floor(j / PAGE_WIDTH) * PAGE_WIDTH]);
                }
            });

            // populate the page cache with the initial data.
            var initialPage = pageFromData(0,0, tableData.initialChunk);
            var initialPageName = tableData.cellPageName(0,0);
            tableData.pageCache.set(initialPageName, initialPage);
            tableData.PAGE_WIDTH = PAGE_WIDTH;
            tableData.PAGE_HEIGHT = PAGE_HEIGHT;
            return tableData;
        }
    });


    app.service('computeColumnWidths', function() {
        return function(sampleData, headers, minColumnWidth, hasAnyFilterOnColumn, columnWidthsByName, columnUseScientificNotationByName, reset = false) {
            // Upper bounds for a cell/col containing only capital M: { header = 99, body = 95 }
            // Lower bound wih only small l: {header = 2.9, body = 2.6 }

            // Seems reasonable: 7 / 7.5
            const CELL_LETTER_WIDTH = 7;
            const HEADER_LETTER_WIDTH = 7.5;

            const CELL_MARGIN = 15;
            const HEADER_MARGIN = 15;
            const MAX_WIDTH = 300;
            const FILTER_FLAG_WIDTH = 20;

            let columnWidthsByIndex = [];
            const nbCols = headers.length;

            for (var colId = 0; colId < nbCols; colId++) {
                const header = headers[colId];
                const columnName = header.name;
                let columnWidth;

                if (!reset) {
                    columnWidth = columnWidthsByName[columnName];
                }

                if (!(Number.isInteger(columnWidth)) && columnUseScientificNotationByName[columnName] !== true) {
                    // when the column contains doubles, we rewrite the server-computed width in order to contain full numbers (no scientific notation) from 1e-15 to 9.999e15
                    let ncharsToShow = header.selectedType && ['DoubleMeaning', 'FrenchDoubleMeaning'].includes(header.selectedType.name) ? Math.max(17, header.ncharsToShow) : header.ncharsToShow;
                    let cellColumnWidth =  Math.ceil(ncharsToShow * CELL_LETTER_WIDTH + CELL_MARGIN);
                    let colColumnWidth =  Math.ceil(header.name.length * HEADER_LETTER_WIDTH + HEADER_MARGIN);
                    columnWidth = Math.max(colColumnWidth, cellColumnWidth);
                    columnWidth = fattable.bound(columnWidth, minColumnWidth, MAX_WIDTH);

                    if ((hasAnyFilterOnColumn === undefined) || hasAnyFilterOnColumn(header.name)) {
                        columnWidth += FILTER_FLAG_WIDTH;
                    }

                    columnWidthsByName[columnName] = columnWidth;
                }

                columnWidthsByIndex.push(columnWidth);
            }
            return [ columnWidthsByIndex, columnWidthsByName ];
        };
    });

    app.directive('fattable', function(ShakerTableModel, computeColumnWidths, ColorUtils, ContextualMenu, CreateModalFromDOMElement, CreateModalFromTemplate,
            $filter, $templateCache, $q, $http, $timeout, $rootScope, $stateParams, $compile,Debounce, ShakerProcessorsUtils, ShakerSuggestionsEngine, Logger, WT1, FatTouchableService, FatDraggableService, FatResizableService,
            ClipboardUtils, PrettyPrintDoubleService, ChartFilters, DashboardFilters, DashboardUtils, DKU_NO_VALUE, FeatureFlagsService, findAppliedColoringGroup, ConditionalFormattingEditorService) {
        const insertHighlightTags = (content, indices) => {
            const contents = []
            let sliceStart = 0;
            for (let i = 0; i < indices.length; i++) {
                const start = indices[i][0];
                const end = indices[i][1];
                if (sliceStart < start) {
                    contents.push(sanitize(content.slice(sliceStart, start)));
                }

                contents.push("<mark>");
                contents.push(sanitize(content.slice(start, end)));
                contents.push("</mark>");

                // there is another highlighted match after that one, get its start to slice
                const sliceEnd = i < indices.length - 1 ? indices[i + 1][0] : content.length;
                contents.push(sanitize(content.slice(end, sliceEnd)));
                sliceStart = sliceEnd;
            }
            return contents.join("");
        }


        // Fattable delegates filling cells / columns header
        // with content to this object.
        function ShakerTablePainter(scope) {

            const hasHighlights = (cell) => {
                return scope.shaker.coloring.highlightSearchMatches && cell.highlightedIndices;
            }

            const applyHighlights = (contentWithColor) => {
                let contentWithColorAndHighlight = contentWithColor;

                if (scope.shaker.coloring.highlightWhitespaces) {
                    contentWithColorAndHighlight = contentWithColorAndHighlight
                            .replace(/^(\s*)/, "<span class='ls'>$1</span>")
                            .replace(/(\s*)$/, "<span class='ts'>$1</span>")
                            .replace(/(\s\s+)/g, "<span class='ms'>$1</span>");
                }

                return contentWithColorAndHighlight;
            }

            const isValidHTTPUrl = (potentialUrl) => {
                let url;
                
                try {
                  url = new URL(potentialUrl);
                } catch (_) {
                  return false;  
                }
              
                return url.protocol === "http:" || url.protocol === "https:";
              }

            const getElForLink = (el, content, cell) => {
                const linkEl = document.createElement('a');
                const sanitizedContent = sanitize(content);
                linkEl.href = content;
                linkEl.target = '_blank';
                linkEl.rel = 'noopener noreferrer';

                if (scope.shaker.coloring) {
                    let contentWithColor = hasHighlights(cell) ? insertHighlightTags(content, cell.highlightedIndices) : sanitizedContent;
                    let contentWithColorAndHighlight = applyHighlights(contentWithColor);
                    linkEl.innerHTML = contentWithColorAndHighlight;
                    $(el).html(linkEl);
                } else {
                    linkEl.textContent = sanitizedContent;
                    el.appendChild(linkEl);
                }

                return el;
            }

            const getElForText = (el, content, cell) => {
                el.textContent = content;
                if (scope.shaker.coloring) {
                    let contentWithColor = hasHighlights(cell) ? insertHighlightTags(content, cell.highlightedIndices) : sanitize(content);
                    let contentWithColorAndHighlight = applyHighlights(contentWithColor);

                    $(el).html(contentWithColorAndHighlight);
                }
                return el;
            }

            return $.extend(new fattable.Painter(), {

                setupHeader: function(el) {
                    el.setAttribute("column-header", "header");
                    el.setAttribute("data-page-tour", "column-header");
                    el.setAttribute("ng-class", "{'columnHeader': true, 'filtered': hasAnyFilterOnColumn(column.name)}");
                 }
            ,
                fillHeader: function(el, headerScope)  {
                    var $el = $(el);
                    this.destroyFormerHeaderScope(el);
                    el.scopeToDestroy = headerScope;
                    $el.empty();
                    $compile($el)(headerScope);
                }
            ,
                destroyFormerHeaderScope: function(el) {
                    if (el.scopeToDestroy !== undefined) {
                        el.scopeToDestroy.$destroy();
                        el.scopeToDestroy = undefined;
                    }
                }
            ,
                fillCellPending: function(el,cell) {
                    el.textContent = "Wait...";
                    el.className = "PENDING"
                }
            ,
                fillCell: function(el, cell)  {
                    const coloring = scope.shaker.coloring;

                    const MAX_TITLE_LENGTH = 980;
                    el.dataset.rowId = cell.rowId;
                    // `cell.colId` represents the id of the column after removing unselected columns
                    el.dataset.colId = cell.colId;

                    const columnLabel = scope.tableModel.headers[cell.colId].name;
                    const viewMoreContentLabel = `...\n\nShift + v to ${scope.isCellKO(cell.status) || !scope.columnSupportsSpecialPreview(columnLabel) ? "view complete cell value" : "preview" }.`;


                    // optionally rewrite the cell content when meaning is double (only use scientific notation if exponent is >15)
                    if(scope.tableModel.headers[cell.colId].selectedType && ['DoubleMeaning', 'FrenchDoubleMeaning'].includes(scope.tableModel.headers[cell.colId].selectedType.name) &&
                    scope.shaker.columnUseScientificNotationByName[columnLabel] !== true) {
                        cell.content = PrettyPrintDoubleService.avoidScientificNotation(cell.content);
                    }

                    el.title = cell.content && cell.content.length > MAX_TITLE_LENGTH ? cell.content.slice(0, MAX_TITLE_LENGTH) + viewMoreContentLabel : cell.content;

                    if (cell.colorBin !== null) {
                        let coloringGroup = null;
                        if (coloring.scheme === "COLORING_GROUPS") {
                            coloringGroup = findAppliedColoringGroup(coloring.coloringGroups, columnLabel);
                        }

                        // Also checks if the based on column has not been deleted
                        let isBasedOnAnotherColumnScope =
                            coloringGroup && coloringGroup.scope === "ALL_COLUMNS_BASED_ON_ANOTHER_COLUMN";

                        let isColoredByRules =
                            coloring.scheme === "INDIVIDUAL_COLUMNS_RULES" ||
                            coloring.scheme === "SINGLE_COLUMN_RULES" ||
                            (coloringGroup && coloringGroup.scheme === "RULES");

                        if (isColoredByRules) {
                            // Alternates cell style every row even if no conditional formatting rules applies
                            el.className = cell.status + [EVEN, ODD][cell.rowId % 2];
                            // Resets the element style to avoid side effects from a previous fillCell
                            el.style.backgroundColor = "";
                            el.style.color = "";

                            if (cell.colorBin !== undefined && cell.colorBin !== BIN_EMPTY) {
                                // Collects the filter descriptions from the group of rules associated to the current column
                                let rulesGroupFilterDescs = null;

                                if (coloring.scheme === "COLORING_GROUPS") {
                                    if (
                                        (coloringGroup.scope === "COLUMNS" || isBasedOnAnotherColumnScope) &&
                                        coloringGroup.rulesGroup.filterDescs.length > 0
                                    ) {
                                        rulesGroupFilterDescs = coloringGroup.rulesGroup.filterDescs;
                                    }
                                }
                                // Legacy logic
                                else if (coloring.scheme === "INDIVIDUAL_COLUMNS_RULES") {
                                    const columnRules = angular.copy(
                                        scope.shaker.coloring.individualColumnsRules.filter(
                                            r => r.column === columnLabel
                                        )
                                    );

                                    if (columnRules[0] && columnRules[0].rulesDesc) {
                                        rulesGroupFilterDescs = columnRules[0].rulesDesc;
                                    }
                                }
                                // Legacy logic
                                else if (coloring.scheme === "SINGLE_COLUMN_RULES") {
                                    const columnRules = angular.copy(
                                        scope.shaker.coloring.individualColumnsRules.filter(
                                            r => r.column === coloring.singleColumn
                                        )
                                    );

                                    if (columnRules[0] && columnRules[0].rulesDesc) {
                                        rulesGroupFilterDescs = columnRules[0].rulesDesc;
                                    }
                                }

                                if (rulesGroupFilterDescs) {
                                    let rules = rulesGroupFilterDescs;
                                    if (
                                        cell.colorBin >= 0 &&
                                        cell.colorBin < rules.length &&
                                        "uiData" in rules[cell.colorBin]
                                    ) {
                                        if ("color" in rules[cell.colorBin].uiData) {
                                            if (rules[cell.colorBin].enabled === true) {
                                                el.className =
                                                    el.className + " " + rules[cell.colorBin].uiData.color.styleClass;
                                            }
                                            if (el.className.split(" ").includes("shaker-color-rule--custom")) {
                                                // An empty string represents a "none" color.
                                                // Because the css class above is set to even/odd, there is nothing else
                                                // to do when the color is "none".
                                                el.style.backgroundColor =
                                                    rules[cell.colorBin].uiData.color.styleCustomBackgroundColor;
                                                el.style.color = rules[cell.colorBin].uiData.color.styleCustomFontColor;
                                            }
                                        }
                                    }
                                }
                            }
                        } else {
                            // Reset styles to avoid unexpected behavior with previous cells colors
                            el.style.color = "";
                            el.style.backgroundColor = "";

                            const colorBinFloor = Math.floor(cell.colorBin);

                            const isColorScale = coloringGroup && coloringGroup.scheme === "COLOR_SCALE";
                            const hasValidColorScale =
                                isColorScale &&
                                coloringGroup.colorScaleDef &&
                                coloringGroup.colorScaleDef.sample &&
                                colorBinFloor > 0;

                            if (hasValidColorScale) {
                                if (colorBinFloor <= coloringGroup.colorScaleDef.sample.length) {
                                    // Interpolates cell colors for numbers only
                                    if (
                                        cell.status &&
                                        cell.status.indexOf("CN") !== -1 &&
                                        colorBinFloor < coloringGroup.colorScaleDef.sample.length
                                    ) {
                                        el.style.backgroundColor = ColorUtils.getLerpRGB(
                                            coloringGroup.colorScaleDef.sample[colorBinFloor - 1],
                                            coloringGroup.colorScaleDef.sample[colorBinFloor],
                                            cell.colorBin - colorBinFloor
                                        );
                                    } else {
                                        el.style.backgroundColor =
                                            coloringGroup.colorScaleDef.sample[colorBinFloor - 1];
                                    }
                                    el.style.color = ColorUtils.getFontContrastColor(el.style.backgroundColor);

                                    el.className = cell.status + "-" + colorBinFloor + " shaker-color-scale";
                                } else {
                                    // Alternates white/grey
                                    el.className = cell.status + " VU " + [EVEN, ODD][cell.rowId % 2];
                                }
                            } else {
                                if (colorBinFloor === BIN_UNCOLORED) {
                                    el.className = cell.status + [EVEN, ODD][cell.rowId % 2];
                                } else if (colorBinFloor > 0 && cell.status && cell.status.indexOf("CN") !== -1) {
                                    // Interpolates cell colors for numbers only
                                    const sample = ConditionalFormattingEditorService.getScalePaletteForNumber();
                                    el.style.backgroundColor = ColorUtils.getLerpRGB(
                                        sample[colorBinFloor - 1],
                                        sample[colorBinFloor],
                                        cell.colorBin - colorBinFloor
                                    );
                                    el.style.color = ColorUtils.getFontContrastColor(el.style.backgroundColor);

                                    el.className = cell.status + "-" + colorBinFloor + " shaker-color-scale";
                                } else {
                                    el.className = cell.status + "-" + colorBinFloor + [EVEN, ODD][cell.rowId % 2];
                                }
                            }
                        }
                    } else {
                        el.className = cell.status + [EVEN, ODD][cell.rowId % 2];
                    }
                    if (scope.searchableDataset) {
                        if (!el.className.split(" ").includes("VU")) { // does not include class name if already here
                            el.className += " VU";  // Valid unit when hovering cell, not set in the backend
                        }
                        const isColumnMenuDisabledFn = scope.shaker.$isColumnMenuDisabled;
                        if (isColumnMenuDisabledFn && isColumnMenuDisabledFn(columnLabel)) {
                            el.className += " IG"; // IG = ignore, not part of the schema
                        }
                    }

                    if (scope.shakerState.lockedHighlighting.indexOf(cell.rowId) >=0) {
                        el.className += " FH";
                    }
                    if (scope.shakerState.selectedRow === cell.rowId) {
                        el.className += " SELECTED";
                    }
                    if(!cell.content){
                        el.textContent = "";
                        return;
                    }

                    // highlight selections
                    var lastDisplayedRow = scope.shakerTable.firstVisibleRow + scope.shakerTable.nbRowsVisible;
                    const content = cell.content.replace(/(\r\n|\n)/g, "¶");
                    const shouldDisplayLink = $rootScope.appConfig.dataTableLinksEnabled && scope.shaker.origin !== "PREPARE_RECIPE" && scope.tableModel.headers[cell.colId].meaningLabel === 'URL' && isValidHTTPUrl(content);

                    if (shouldDisplayLink) {
                        el = getElForLink(el, content, cell);
                    } else {
                        el = getElForText(el, content, cell);
                    }

                    el.appendChild(document.createElement('div'));
                }
            ,
                setupCell: function(el) {
                    el.oncontextmenu = function(evt) {
                        var row = el.dataset.rowId;
                        var col = el.dataset.colId;
                        scope.showCellPopup(el, row, col, evt);
                        return false;
                    };
                }
            ,
                cleanUpCell: function(cellDiv) {
                    $(cellDiv).remove();
                }
            ,
                cleanUpheader: function(headerDiv) {
                    this.destroyFormerHeaderScope(headerDiv);
                    $(headerDiv).remove();
                }
            });
        }
        

        return {
            restrict: 'A',
            scope: true,
            link: function(scope, element, attrs) {

                const getSelectedCell = function(hasHighlightMatches, elt) {
                    if (hasHighlightMatches && elt.tagName === scope.highlightTagName) {
                        // on a searchable dataset with and a highlighted word, so get parent
                        return elt.parentElement;
                    }
                    return elt;
                }

                var currentMousePos;

                $(element).mousemove(function(evt) {
                    currentMousePos = {
                        x : evt.clientX,
                        y: evt.clientY
                    }
                });

                var tableDataExpr = attrs.fattableData;

                { // bind "c" to "scroll to column"

                    var shown = false;

                    //<input type="text" class="form-control" ng-model="selectedState" ng-options="state for state in states" placeholder="Enter state" bs-typeahead>
                    scope.openSearchBox = function() {
                        shown=true;
                        const newScope = scope.$new();
                        const controller = function() {
                            newScope.searchHeaderName = function(query) {
                                const columnIds = scope.tableModel.searchHeader(query);
                                return columnIds.map(i => scope.tableModel.headers[i].name);
                            }
                            newScope.move = function(query) {
                                const columnIds = scope.tableModel.searchHeader(query);
                                if (columnIds.length > 0) {
                                    const columnSelected = columnIds[0];
                                    scope.shakerTable.goTo(undefined, columnSelected);
                                }
                            }
                            $("body").addClass("fattable-searchbox-modal");
                            $("#fattable-search").focus();
                        };
                        CreateModalFromTemplate('/templates/shaker/search-column.html', newScope, controller, function(modalScope) {
                            $(".modal").one("hide", function() {
                                shown = false;
                                $("body").removeClass("fattable-searchbox-modal");
                            });
                            modalScope.onSubmit = function(e) {
                                modalScope.dismiss();
                            }
                            modalScope.$on('typeahead-updated', modalScope.onSubmit);
                        });
                    };

                    var $window = $(window);

                    var keyCodes = {
                        tab: 9,
                        pageup: 33,
                        pagedown: 34,
                        left: 37,
                        up: 38,
                        right: 39,
                        down: 40
                    };

                    if (!DashboardUtils.isInDashboard()) {
                        Mousetrap.bind("c", function() {
                            if (!shown) {
                                scope.hidePopups();
                                scope.openSearchBox();
                            }
                        });
                    }

                    $window.on("keydown.fattable", function(e){
                        if (["INPUT", "SELECT", "TEXTAREA"].indexOf(e.target.tagName) == -1) {
                            var move = function(dx,dy) {
                                var scrollBar = scope.shakerTable.scroll;
                                var x = scrollBar.scrollLeft + dx;
                                var y = scrollBar.scrollTop + dy;
                                scrollBar.setScrollXY(x,y);
                            };

                            var smallJump = 20;
                            var bigJump = smallJump * 7;
                            switch(e.keyCode) {
                                case keyCodes.up:
                                    move(0, -smallJump);
                                    break;
                                case keyCodes.down:
                                    move(0, smallJump);
                                    break;
                                case keyCodes.left:
                                    move(-smallJump, 0);
                                    break;
                                case keyCodes.right:
                                    move(smallJump, 0);
                                    break;
                                case keyCodes.pagedown:
                                    move(0, bigJump);
                                    break;
                                case keyCodes.pageup:
                                    move(0, -bigJump);
                                    break;
                            }
                        }
                    });

                    scope.$on('scrollToColumn', function(e, columnName) {
                        var c = scope.tableModel.searchHeader(columnName)[0];
                        if (c >= 0) {
                            scope.shakerTable.goTo(undefined, c);
                        }
                    });

                    scope.$on('$destroy', function() {
                        $(window).off("keydown.fattable");
                    });

                }


                // binding cell click events.
                {
                    var $el = $(element);
                    var $currentSelectCell = null;
                    $el.off(".shakerTable");
                    var getRow = function($el) {
                        var rowId = $el[0].dataset.rowId;
                        return $el.siblings("[data-row-id='"+ rowId+ "']");
                    };

                    if (scope.shakerCellClickable) {
                        $el.on("click.shakerTable", ".fattable-body-container > div > div", function(evt) {
                            const rowIdx = parseInt(scope.shakerState.hoverredRow);
                            const colIdx = parseInt(scope.shakerState.hoverredCol);
                            scope.selectRow(rowIdx);
                            $rootScope.$broadcast('shakerCellClick', scope.getRowCells(rowIdx), colIdx, attrs.fatId);
                        });
                    }

                    $el.on("mousedown.shakerTable", ".fattable-body-container > div > div", function(evt) {
                        // Prevent a bit selection of more than one cell.
                        if ($currentSelectCell != null) {
                            $currentSelectCell.parent().find(".selectable").removeClass("selectable");
                            $currentSelectCell.parent().removeClass("inselection");
                        }
                        $currentSelectCell = $(getSelectedCell(scope.searchableDataset, $(evt.target)[0]));
                        $currentSelectCell.addClass("selectable");
                        $currentSelectCell.parent().addClass("inselection");
                    });

                    $el.on("mouseup.shakerTable", ".fattable-body-container > div", function(evt) {
                        if (evt.button != 1) {
                            if (!scope.isCellPopupVisible() ) {
                                var target = evt.target;
                                if ($currentSelectCell != null) {
                                    target = getSelectedCell(scope.searchableDataset, target);
                                    var row = target?.dataset?.rowId;
                                    var col = target?.dataset?.colId;
                                    if (row && col && $currentSelectCell[0] == target) {
                                        scope.showCellPopup(target, row, col, evt);
                                        // If the event bubbles up to body,
                                        // it will trigger hidePopup.
                                        evt.stopPropagation();
                                    }
                                }
                                $currentSelectCell = null;
                            }
                        }
                    });

                    const hoverRow = (target) => {
                        getRow(target).addClass('H');
                        scope.shakerState.hoverredRow = target[0].dataset.rowId;
                        scope.shakerState.hoverredCol = target[0].dataset.colId;
                    }

                    const blurRow = (target) => {
                        getRow(target).removeClass('H');
                        scope.shakerState.hoverredRow = null;
                        scope.shakerState.hoverredCol = null;
                    }

                    $el.on("mouseenter.shakerTable", ".fattable-body-container > div > div", function(evt) {
                        hoverRow($(evt.target));
                    });

                    $el.on("mouseleave.shakerTable", ".fattable-body-container > div > div", function(evt) {
                        blurRow($(evt.target));
                    });

                    $el.on("mouseenter.shakerTable", ".fattable-body-container  > div > div > a", function(evt) {
                        hoverRow($(evt.target).parent());
                    });

                    $el.on("mouseleave.shakerTable", ".fattable-body-container  > div > div > a", function(evt) {
                        blurRow($(evt.target).parent());
                    });
                }

                const CELL_PREVIEW_TYPE = {
                    CELL_VALUE_PREVIEW : "CELL_VALUE_PREVIEW",
                    GEO_PREVIEW : "GEO_PREVIEW",
                    IMAGE_PREVIEW : "IMAGE_PREVIEW"
                };

                // setuping the cell popup.
                const popupContent = $('<div><div class="popover-content shaker-cell-popover"></div></div>');
                $("body").append(popupContent);

                const cvPopupContent = $('<div><div class="popover-content"></div></div>');
                cvPopupContent.canClose = true;
                cvPopupContent.setPopupClosable = () => { cvPopupContent.canClose = true; };
                cvPopupContent.setPopupNotClosable = () => { cvPopupContent.canClose = false; };
                $("body").append(cvPopupContent);

                function showPopupUsing(template, customScope, cellPreviewType) {
                    var newDOMElt = $('<div>');
                    newDOMElt.html(template);
                    $timeout(function() {
                        $compile(newDOMElt)(customScope);
                        cvPopupContent.find(".popover-content").empty().append(newDOMElt);
                        $timeout(function() {
                            cvPopupContent.css("display", "block");
                            cvPopupContent[0].className = "shaker-cell-popover popover ";
                            switch(cellPreviewType) {
                                case CELL_PREVIEW_TYPE.GEO_PREVIEW:
                                    cvPopupContent[0].className += "shaker-cell-popover--geo-preview ";
                                    break;
                                case CELL_PREVIEW_TYPE.IMAGE_PREVIEW:
                                    cvPopupContent[0].className += "shaker-cell-popover--image-preview ";
                            }
                            cvPopupContent.on('click', function(e){
                                if(! $(e.target).closest('a,input,button,select,textarea').length){
                                    e.stopPropagation();
                                }
                            });
                        });
                    });
                }

                var $doc = $(document);

                scope.isCellPopupVisible = function() {
                    return popupContent.css("display") != "none";
                };

                scope.hidePopup = function(popup, className) {
                    if(popup.canClose === undefined || popup.canClose) {
                        var formerPopupScope = popup.find(`.${className} > div`).first().scope();
                        if ((formerPopupScope != undefined) && (formerPopupScope !== scope)) {
                            formerPopupScope.$destroy();
                        }
                        popup.css("display", "none");
                    }
                };

                scope.hidePopups = function(){
                    scope.hidePopup(popupContent, "popover-content");
                    scope.hidePopup(cvPopupContent, "popover-content")
                    $doc.unbind("click.shaker.cellPopup");
                };

                scope.isCellKO = function(cellStatus) {
                    if(cellStatus === null) return false; // in case of wrong status allow the option (column coloring for instance)
                    return cellStatus !== undefined
                        && (cellStatus.startsWith("I") || cellStatus.startsWith("E") || cellStatus.startsWith("F"));
                };

                function columnSupportsGeoPreview(columnLabel) {
                    const columnHeaders = scope.table.headers.filter(column => column.name === columnLabel);
                    if(columnHeaders.length > 1 || columnHeaders.length === 0) {
                        return false;
                    }

                    const columnHeader = columnHeaders[0];
                    if (!columnHeader.selectedType) return false;
                    const meaning = columnHeader.selectedType.name;

                    return meaning === "GeometryMeaning" || meaning === "GeoPoint";
                };

                function columnSupportsImagePreview(columnLabel) {
                    if (!scope.imageViewSettings || !scope.imageViewSettings.enabled) {
                        return false;
                    }
                    if (scope.imageViewSettings.pathColumn === columnLabel) {
                        return true;
                    }
                    return false;
                };

                function getPreviewType(columnName) {
                    if (columnSupportsGeoPreview(columnName)) {
                        return CELL_PREVIEW_TYPE.GEO_PREVIEW;
                    } else if (columnSupportsImagePreview(columnName)) {
                        return CELL_PREVIEW_TYPE.IMAGE_PREVIEW;
                    }
                    return CELL_PREVIEW_TYPE.CELL_VALUE_PREVIEW;
                }

                scope.columnSupportsSpecialPreview = function(columnName) {
                    return getPreviewType(columnName) !== CELL_PREVIEW_TYPE.CELL_VALUE_PREVIEW;
                }

                scope.toggleRowHighlight = function(rowIdx) {
                    var arr = scope.shakerState.lockedHighlighting;
                    if (arr.indexOf(rowIdx) >=0){
                        arr.splice(arr.indexOf(rowIdx), 1);
                    } else {
                        arr.push(rowIdx);
                    }
                    scope.shakerTable.refreshAllContent(true);
                }

                /**
                 * only one row can be selected at time, null is used to deselect the current row
                 * @param {*} rowIdx number | null
                 */
                scope.selectRow = function(rowIdx) {
                    scope.shakerState.selectedRow = rowIdx;
                    scope.shakerTable.refreshAllContent(true);
                }

                scope.changeSelectedRow = function(direction) {
                    let currentlySelectedRow = scope.shakerState.selectedRow;
                    if (!isNaN(currentlySelectedRow)) {
                      if (direction === "previous" && currentlySelectedRow > 0) {
                        currentlySelectedRow = currentlySelectedRow - 1;
                      } else if (direction === "next" && currentlySelectedRow < scope.shakerTable.nbRows - 1) {
                        currentlySelectedRow = currentlySelectedRow + 1;
                      }
                      // if we do not apply the whole table will rerender with a blink.
                      scope.$apply(function () {
                        scope.selectRow(currentlySelectedRow);
                      });
                      scope.scrollToLine(currentlySelectedRow);
                      $rootScope.$broadcast("compareCellValueNewRowSelected", scope.getRowCells(currentlySelectedRow), attrs.fatId);
                    }
                }

                scope.copyRowAsJSON = async function(rowIdx) {
                    function getColumnSchema(column) {
                        if (scope.shaker.origin === "DATASET_EXPLORE") {
                            return column.datasetSchemaColumn;
                        } else if (scope.shaker.origin === "PREPARE_RECIPE" && column.recipeSchemaColumn) {
                            return column.recipeSchemaColumn.column;
                        }
                    }

                    function getCellPromise(rowIdx, colIdx) {
                        return new Promise((resolve) => {
                            scope.tableModel.getCell(rowIdx, colIdx, resolve);
                        });
                    }

                    function smartCast(colType, colValue) {
                        switch (colType) {
                            case "tinyint":
                            case "smallint":
                            case "int":
                            case "bigint":
                                return Number.parseInt(colValue);
                            case "float":
                            case "double":
                                return Number.parseFloat(colValue);
                            default:
                                return colValue;
                            }
                    }



                    const colTypes = scope.table.headers.reduce((obj, column) => {
                        const colSchema = getColumnSchema(column);
                        obj[column.name] = colSchema ? colSchema.type : null;
                        return obj;
                      }, {});

                    const columnNames = scope.tableModel.allColumnNames;
                    const columnIndices = [...Array(columnNames.length).keys()];
                    const row = {};

                    await Promise.all(columnIndices.map(colIdx => getCellPromise(rowIdx, colIdx)))
                        .then((cells) => {
                            for (const [index, cell] of cells.entries()) {
                                const columnName = columnNames[index];
                                row[columnName] = smartCast(colTypes[columnName], cell.content);
                            }
                        });

                    ClipboardUtils.copyToClipboard(JSON.stringify(row, null, 2), `Row copied to clipboard.`);
                };

                scope.getRowCells = async function(rowIdx) {
                    function getCellPromise(rowIdx, colIdx) {
                        return new Promise((resolve) => {
                            scope.tableModel.getCell(rowIdx, colIdx, resolve);
                        });
                    }

                    const columnNames = scope.tableModel.allColumnNames;
                    const columnIndices = [...Array(columnNames.length).keys()];

                    return Promise.all(columnIndices.map(colIdx => getCellPromise(rowIdx, colIdx)));
                }

                if (!DashboardUtils.isInDashboard()) {
                    Mousetrap.bind("shift+h", function(){
                        var rowIdx = scope.shakerState.hoverredRow;
                        if (!rowIdx) return;
                        rowIdx = parseInt(rowIdx);
                        scope.$apply(function(){
                            scope.toggleRowHighlight(rowIdx);
                        });
                    }, 'keyup');
    
                     Mousetrap.bind("shift+v", function(){
                        if (!scope.shakerState.hoverredRow) return;
                        const rowIdx = parseInt(scope.shakerState.hoverredRow);
                        const colIdx = parseInt(scope.shakerState.hoverredCol);
                        scope.$apply(function(){
                            scope.hidePopups();
                            scope.showCVCellPopup(rowIdx, colIdx);
                        });
                    }, 'keyup');
    
                    Mousetrap.bind("shift+j", function(){
                        var rowIdx = scope.shakerState.hoverredRow;
                        if (!rowIdx) return;
                        rowIdx = parseInt(rowIdx);
                        scope.$apply(function(){
                            scope.copyRowAsJSON(rowIdx);
                        });
                    }, 'keyup');
    
                    Mousetrap.bind("shift+down", function (){
                        scope.changeSelectedRow('next');
                        return false;
                    });
    
                    Mousetrap.bind("shift+up", function (){
                        scope.changeSelectedRow('previous');
                        return false;
                    });
                }

                scope.$on('$destroy', function() {
                    scope.hidePopups();
                    $(document).off('.shaker');
                    popupContent.remove();
                    Mousetrap.unbind('c');
                    Mousetrap.unbind('shift+h', 'keyup');
                    Mousetrap.unbind('shift+v', 'keyup');
                    Mousetrap.unbind('shift+j', 'keyup');
                    Mousetrap.unbind('shift+down');
                    Mousetrap.unbind('shift+up');
                });

                scope.showCVCellPopup2 = function(cellValue, column, sanitizedHTMLContent, cellPreviewType, mousePosition) {
                    ContextualMenu.prototype.closeAny();

                    // no popup displayed for empty cells
                    if(!cellValue || !column) return;

                    // computing new custom scope
                    const newScope = scope.$new();
                    newScope.cellValue = cellValue;
                    const placePopup = function () {
                        $timeout(() => {
                            const placement = getPlacementForMouse(mousePosition, cvPopupContent, mousePosition.left, mousePosition.top);
                            cvPopupContent.css(placement.css);
                            cvPopupContent[0].className += placement.clazzes.join(" ");
                            });
                        };
                    cvPopupContent.onPopupCreated = placePopup;
                    newScope.popup = cvPopupContent;
                    placePopup(); // first placement to avoid lag
                    let template;
                    switch(cellPreviewType) {
                        case CELL_PREVIEW_TYPE.CELL_VALUE_PREVIEW : {
                            newScope.sanitizedHtmlContent = sanitizedHTMLContent;
                            newScope.column = column;
                            if (scope.searchableDataset) {
                                newScope.highlightTagName = scope.highlightTagName;
                                newScope.searchValue = (content) => {
                                    scope.searchInteractiveFilter(column.name, content);
                                };
                            }
                            template = `
                                <cell-value-popup-component 
                                    cell-value="cellValue"
                                    sanitized-html-content="sanitizedHtmlContent"
                                    search-value="searchValue"
                                    column-meaning="column.selectedType.name"
                                    popup-container="popup">
                                </cell-value-popup-component>`;
                                break;
                            }
                        case CELL_PREVIEW_TYPE.GEO_PREVIEW : {
                            template = `
                                <geo-preview-map-content 
                                    cell-value="cellValue" 
                                    popup-container="popup">
                                </geo-preview-map-content>`;
                                break;
                            }
                        case CELL_PREVIEW_TYPE.IMAGE_PREVIEW:
                            template = `
                              <image-preview-component
                                image-path="cellValue"
                                image-view-settings="imageViewSettings"
                                popup-container="popup">
                              </image-preview-component>`;
                            break;
                        default :
                            throw new Error(`Unknown value for CELL_PREVIEW_TYPE: ${cellPreviewType}`);
                        }
                        showPopupUsing(template, newScope, cellPreviewType);
                }

                scope.showCVCellPopup = function(row, col) {
                    scope.tableModel.getCell(row, col, function(cellData) {
                        const currentMousePosition = {
                            top: currentMousePos.y,
                            left: currentMousePos.x
                        }
                        const columnName = scope.table.headers[col].name;
                        const cellValue = cellData.content;
                        const cellPreviewType = scope.isCellKO(cellData.status) ? CELL_PREVIEW_TYPE.CELL_VALUE_PREVIEW : getPreviewType(columnName);
                        const hasHighlight = scope.shaker.coloring.highlightSearchMatches && cellData.highlightedIndices;
                        const sanitizedHTMLContent = hasHighlight ? insertHighlightTags(cellValue, cellData.highlightedIndices) : sanitize(cellValue);  // html content has already been sanitized in insertHighlightTags()
                        scope.showCVCellPopup2(cellValue, scope.table.headers[col], sanitizedHTMLContent, cellPreviewType, currentMousePosition);
                    });
                }



                scope.showCellPopup = function(elt, row, col, evt) {
                    const rowIndex = Number.parseInt(row);
                    const colIndex = Number.parseInt(col);
                    if (isNaN(rowIndex) || isNaN(colIndex)) {
                        Logger.warn("Invalid cell index while showing cell popup")
                        return;
                    }
                    ContextualMenu.prototype.closeAny();
                    scope.hidePopups();
                    // TODO eventually get rid of this.
                    // terrible monkey patching
                    {
                        var parent = popupContent.parent();
                        if (popupContent.parent().length == 0) {
                            // why is not under body anymore!?
                            $("body").append(popupContent);
                            WT1.event("shaker-cell-popup-content-disappear");
                            Logger.error("POPUP CONTENT DISAPPEARED. MONKEY PATCH AT WORK");
                        }
                    }
                    // end of terrible monkey...
                    // explicit activation of popup menu
                    // if neither shakerWritable nor shakerReadOnlyActions are set, do not show the popup
                    if (!scope.shakerWritable && !scope.shakerReadOnlyActions) return;
                    scope.tableModel.getCell(rowIndex, colIndex, function(cellData) {
                        var cellValue = cellData.content;
                        var req = {
                            cellValue: cellValue,
                            type: "cell",
                            row: rowIndex,
                            column: scope.table.headers[colIndex].name,
                            cellStatus: cellData.status
                        };
                        const currentMousePosition = {
                            top: currentMousePos.y,
                            left: currentMousePos.x
                        };
                        const hasHighlight = scope.shaker.coloring && scope.shaker.coloring.highlightSearchMatches && cellData.highlightedIndices;
                        elt = getSelectedCell(hasHighlight, elt);
                        let highlightTagName = null;
                        if (hasHighlight) {
                            highlightTagName = scope.highlightTagName;  // search highlight (ElasticSearch)
                        } else if (scope.shaker.coloring && scope.shaker.coloring.highlightWhitespaces) {
                            highlightTagName = "span"; // tag inserted to highlight whitespace: <span class="ms">...</span>
                        }
                        const selection = getSelectionInElement(elt, highlightTagName);
                        if (selection != null) {
                            req.type = "content";
                            req.content = selection.content;
                            req.startOffset = selection.startOffset;
                            req.endOffset = selection.endOffset;
                        }

                        var templateUrl = "/templates/shaker/suggestions-popup.html";
                        $q.when($templateCache.get(templateUrl) || $http.get(templateUrl, {
                            cache: true
                        })).then(function(template) {
                            if(angular.isArray(template)) {
                                template = template[1];
                            } else if(angular.isObject(template)) {
                                template = template.data;
                            }
                            var newDOMElt = $('<div>');
                            newDOMElt.html(template);
                            $timeout(function() {
                                var newScope = scope.$new();
                                newScope.req = req;
                                var invalidCell = cellData.status && cellData.status.indexOf("I") == 0;
                                const appConfig = scope.appConfig || scope.$root.appConfig;
                                newScope.columnData = ShakerSuggestionsEngine.computeColumnSuggestions(scope.table.headers[col], CreateModalFromDOMElement, CreateModalFromTemplate, true, invalidCell, appConfig);
                                newScope.cellData = ShakerSuggestionsEngine.computeCellSuggestions(scope.table.headers[col], cellValue, cellData.status, CreateModalFromDOMElement, appConfig);
                                if(cellValue == null){
                                    cellValue = "";
                                }
                                if (newScope.shakerWritable && req.type == "content") {
                                    newScope.contentData = ShakerSuggestionsEngine.computeContentSuggestions(scope.table.headers[col], cellValue, req.content,
                                                cellData.status, CreateModalFromTemplate, req.startOffset, req.endOffset);
                                }
                                newScope.executeSuggestion = function(sugg) {
                                    sugg.action(scope);
                                };
                                newScope.showCellValue = function(){
                                    const sanitizedHTMLContent = hasHighlight ? elt.innerHTML : null;  // html content has already been sanitized in insertHighlightTags()
                                    scope.showCVCellPopup2(cellValue, scope.table.headers[col], sanitizedHTMLContent, CELL_PREVIEW_TYPE.CELL_VALUE_PREVIEW, currentMousePosition);
                                }
                                newScope.showSpecialPreview = function() {
                                    scope.showCVCellPopup2(cellValue, scope.table.headers[col], null, getPreviewType(scope.table.headers[col].name), currentMousePosition);
                                }
                                newScope.getPreviewIcon = function(columnName) {
                                    switch(getPreviewType(columnName)) {
                                        case CELL_PREVIEW_TYPE.GEO_PREVIEW:
                                            return "dku-icon-globe-16";
                                        case CELL_PREVIEW_TYPE.IMAGE_PREVIEW:
                                            return "dku-icon-image-16";
                                        default:
                                            return "";
                                    }
                                }

                                newScope.triggerCompareColumnValues = function(rowIdx) {
                                    scope.selectRow(rowIdx);
                                    $rootScope.$broadcast('triggerCompareColumnValues', scope.getRowCells(rowIdx), cellData.colId);
                                }

                                newScope.copyCellValue = function()  {
                                    ClipboardUtils.copyToClipboard(cellValue, `Copied to clipboard.`);
                                }
                                newScope.searchValue = function(content) {
                                    scope.searchInteractiveFilter(req.column, content);
                                }
                                newScope.getStepDescription = function(a,b) {
                                    return ShakerProcessorsUtils.getStepDescription(null, a,b);
                                };
                                newScope.filter = function(val, matchingMode) {
                                    if(!val) {
                                        val = '';
                                    }
                                    var v = {};
                                    v[val] = true;
                                    if (scope.table.headers[col].selectedType) {
                                        scope.addColumnFilter(scope.table.headers[col].name, v, matchingMode,
                                            scope.table.headers[col].selectedType.name, scope.table.headers[col].isDouble);
                                    }
                                };
                                newScope.enableFilter = !DashboardUtils.isInDashboard() && !scope.searchableDataset;
                                newScope.canCrossFilter = function(value) {
                                    if (DashboardUtils.isInDashboard() && DashboardFilters.canCrossFilter($stateParams.pageId)) {
                                        const isDouble = scope.table.headers[col].isDouble;
                                        const columnType = ['Date', 'DateOnly', 'DatetimeNoTz'].indexOf(scope.table.headers[col].selectedType.name) >= 0 ? 'DATE' : (isDouble ? 'NUMERICAL' : 'ALPHANUM');
                                        // In case of an empty cell in a date or numerical column we do not allow the include only
                                        // - On SQL, we currently do not support alphanum filter for numeric columns
                                        // - TODO For date, build a date part filter, filtering on 'No value'
                                        return !(value == null && (columnType === 'DATE' || columnType === 'NUMERICAL'));
                                    }
                                    return false;
                                };

                                function computeCrossFilterData(value) {
                                    const isDouble = scope.table.headers[col].isDouble;
                                    const columnType = ['Date', 'DateOnly', 'DatetimeNoTz'].indexOf(scope.table.headers[col].selectedType.name) >= 0 ? 'DATE' : (isDouble ? 'NUMERICAL' : 'ALPHANUM');
                                    let axisElt = { label: value };

                                    if (value == null) {
                                        axisElt = { label: DKU_NO_VALUE };
                                    } else if (columnType === 'DATE') {
                                        // The getTime method always uses UTC for time representation
                                        const timestamp = new Date(value).getTime();
                                        if (isNaN(timestamp)) {
                                            Logger.error("Invalid timestamp");
                                            return;
                                        }
                                        axisElt = { min: timestamp, max: timestamp, tsValue: timestamp, label: value };
                                    } else if (columnType === 'NUMERICAL') {
                                        const numericalValue = Number(value);
                                        axisElt = { min: numericalValue, max: numericalValue, sortValue: numericalValue, label: value };
                                    }
                                    const column = scope.table.headers[col].name;

                                    return {axisElt, column, columnType};
                                }

                                newScope.includeOnly = function(value) {
                                    const {axisElt, column, columnType} = computeCrossFilterData(value);
                                    const filter = ChartFilters.createFilter({
                                        column,
                                        columnType,
                                        isAGlobalFilter: true,
                                        excludeOtherValues: true,
                                        useMinimalUi: true,
                                        includeEmptyValues: false
                                    }, {
                                        axisElt
                                    });
                                    $rootScope.$emit('crossFiltersAdded', {
                                        filters: [filter],
                                        wt1Args: {
                                            from: 'dataset_table',
                                            action: 'contextual_menu',
                                            filterType: 'include'
                                        }
                                    });
                                };
                                newScope.exclude = function(value) {
                                    const {axisElt, column, columnType} = computeCrossFilterData(value);
                                    const filter = ChartFilters.getExcludeNDFilter([{
                                        dimension: {column, type: columnType },
                                        axisElt,
                                        value: axisElt.label
                                    }]);
                                    filter.isAGlobalFilter = true;
                                    $rootScope.$emit('crossFiltersAdded', {
                                        filters: [filter],
                                        wt1Args: {
                                            from: 'dataset_table',
                                            action: 'contextual_menu',
                                            filterType: 'exclude'
                                        }
                                    });
                                };

                                $compile(newDOMElt)(newScope);
                                popupContent.find(".popover-content").empty().append(newDOMElt);
                                $timeout(function() {
                                    var placement = getPlacement2(elt, popupContent, evt);
                                    newScope.popupPlacement = placement;

                                    popupContent.css("display", "block");
                                    popupContent.css(placement.css);
                                    var popupClassNames = "shaker-cell-popover popover ";
                                    popupClassNames += placement.clazzes.join(" ");
                                    popupContent[0].className = popupClassNames;
                                    popupContent.on('click', function(e){
                                        if(! $(e.target).closest('a,input,button,select,textarea').length){
                                            e.stopPropagation();
                                        }
                                    });
                                }, 0);
                            });
                        });
                    });
                };

                scope.shakerTable = null;

                var ratioX = 0;
                var ratioY = 0;
                scope.setNewTable = function(tableData) {
                    if (scope.shakerTable) {
                        if (scope.shakerTable.scroll) {
                            ratioX = scope.shakerTable.scroll.scrollLeft / scope.shakerTable.W;
                            ratioY = scope.shakerTable.scroll.scrollTop  / scope.shakerTable.H;
                        } else {
                            // we're in the middle of refreshing, so use the last saved values of ratioX and ratioY
                            // and anyway scrollLeft and scrollTop haven't been regenerated yet
                        }
                        scope.shakerTable.cleanUp();
                    } else {
                        ratioX = 0;
                        ratioY = 0;
                    }
                    if (scope.tableScope) {
                        scope.tableScope.$destroy();
                        scope.shakerTable.onScroll = null;
                    }
                    scope.tableScope = scope.$new();
                    let maxPageHeight = null;
                    let maxPageWidth = null;
                    if (scope.setPageHeightWidthFromChunk) {
                        maxPageHeight = tableData.initialChunk.nbRows;
                        maxPageWidth = tableData.initialChunk.nbCols;
                    }
                    scope.tableModel = ShakerTableModel(tableData, scope.tableScope, maxPageWidth, maxPageHeight);

                    // Absolute minimum for "Decimal (FR format)"
                    var minColumnWidth = 100;
                    var headerHeight = 63;
                    /* Space for schema */
                    if (!scope.shaker.$headerOptions) {
                        if (scope.shaker.origin == "PREPARE_RECIPE" || scope.shaker.origin == "DATASET_EXPLORE") {
                            headerHeight += 19;
                        }
                        if (scope.shakerState.hasAnyComment) {
                            headerHeight += 15;
                        }
                        if (scope.shakerState.hasAnyCustomFields) {
                            headerHeight += 19;
                        }
                    } else {
                        headerHeight += scope.shakerState.hasAnyComment ? -4 : 3;

                        if (!scope.shaker.$headerOptions.showName) {
                            headerHeight -= scope.shakerState.hasAnyComment ? 28 : 34;
                        }

                        if (!scope.shaker.$headerOptions.showStorageType) {
                            headerHeight -= 19;
                        }

                        if (scope.shaker.$headerOptions.showMeaning) {
                            headerHeight += 19;
                        }

                        if (scope.shakerState.hasAnyComment && scope.shaker.$headerOptions.showDescription) {
                            headerHeight += 19;
                        }

                        if (scope.shakerState.hasAnyCustomFields && scope.shaker.$headerOptions.showCustomFields) {
                            headerHeight += 19;
                        }

                        if (!scope.shaker.$headerOptions.showProgressBar) {
                            headerHeight -= 11;
                        }

                        var unwatch = scope.$watch("shaker.$headerOptions", function(nv, ov) {
                            if (!nv || nv == ov) return;
                            unwatch();
                            scope.setNewTable(tableData);
                        }, true);
                    }

                    var ROW_HEIGHT = 27;

                    scope.shaker.columnWidthsByName = scope.shaker.columnWidthsByName || {};
                    scope.shaker.columnUseScientificNotationByName = scope.shaker.columnUseScientificNotationByName || {};
                    [ tableData.columnWidthsByIndex, scope.shaker.columnWidthsByName ] = computeColumnWidths(scope.tableModel.initialChunk, scope.tableModel.headers, minColumnWidth, scope.hasAnyFilterOnColumn, scope.shaker.columnWidthsByName, scope.shaker.columnUseScientificNotationByName);

                    scope.shakerTable = fattable({
                        "container": element[0],
                        "model": scope.tableModel,
                        "nbRows": scope.tableModel.totalKeptRows,
                        "headerHeight": headerHeight,
                        "rowHeight":  ROW_HEIGHT,
                        "columnWidths": tableData.columnWidthsByIndex,
                        "painter": ShakerTablePainter(scope.tableScope),
                        "autoSetup": false
                    });

                    scope.shakerTable.onScroll = function(x,y) {
                        scope.hidePopups();
                    }

                    // we save the scroll state, as a ratio.
                    // we scroll back to the position we were at.
                    var newX = (scope.shakerTable.W *ratioX) | 0;
                    var newY = (scope.shakerTable.H *ratioY) | 0;


                    var leftTopCorner = scope.shakerTable.leftTopCornerFromXY(newX, newY);
                    var I = leftTopCorner[0];
                    var J = leftTopCorner[1];

                    var requested = 0;
                    var shakerTable = scope.shakerTable;
                    // A lib like async would have been nice here.
                    // only draw the table if all the
                    // pages are ready.
                    var everythingDone = function() {
                        if (requested == 0) {
                            // we check that the shaker has not
                            // been replaced.
                            if (shakerTable === scope.shakerTable) {
                                scope.shakerTable.setup();
                                scope.shakerTable.scroll.setScrollXY(newX, newY);
                                if (typeof scope.refreshTableDone === 'function') {
                                    scope.refreshTableDone();
                                }
                            }
                        }

                        if (isTouchDevice()) {
                            if (typeof(scope.unsetTouchable) === "function") {
                                scope.unsetTouchable();
                            }
                            scope.unsetTouchable = FatTouchableService.setTouchable(scope, element, scope.shakerTable);
                        }

                        if (attrs.fatDraggable !== undefined) {
                            scope.isDraggable = true;

                            // fatDraggable callback for placeholder shaping : use the whole table height instead of header only
                            scope.onPlaceholderUpdate = function(dimensions) {
                                let table = scope.shakerTable.container;
                                if (table) {
                                    dimensions.height = table.getBoundingClientRect().height;
                                }
                            };

                            FatDraggableService.setDraggable({
                                element: scope.shakerTable.container,
                                onDrop: scope.reorderColumnCallback,
                                onPlaceholderUpdate: scope.onPlaceholderUpdate,
                                scrollBar: scope.shakerTable.scroll,
                                classNamesToIgnore: ['icon-sort-by-attributes', 'sort-indication', 'filter-indication', 'pull-right', 'fat-resizable__handler']
                            })
                        }

                        if (attrs.fatResizable !== undefined) {
                            scope.isResizable = true;

                            let table = scope.shakerTable.container;
                            let barHeight;
                            if (table) {
                                barHeight = table.getBoundingClientRect().height;
                            }

                            FatResizableService.setResizable({
                                element: scope.shakerTable.container,
                                barHeight: barHeight,
                                onDrop: function(resizeData) {
                                    tableData.columnWidthsByIndex[resizeData.index] = resizeData.width;
                                    scope.shakerHooks.updateColumnWidth(resizeData.name, Math.round(resizeData.width));
                                }
                            })
                        }
                    }
                    for (var i=I; i<I+scope.shakerTable.nbRowsVisible; i+=scope.tableModel.PAGE_HEIGHT) {
                        for (var j=J; j<J+scope.shakerTable.nbColsVisible; j+=scope.tableModel.PAGE_WIDTH){
                            if (!scope.tableModel.hasCell(i,j)) {
                                requested += 1;
                                scope.tableModel.getCell(i,j, function() {
                                    requested -= 1;
                                    everythingDone();
                                });
                            }
                        }
                    }
                    everythingDone();
                }

                // we only resize at the end of the resizing.
                // == when the user has been idle for 200ms.
                var formerScrollLeft = 0;
                var formerScrollTop = 0;
                var debouncedResizingHandler = Debounce().withScope(scope).withDelay(200,200).wrap(function() {
                    if (scope.shakerTable !== null) {
                        // check whether we really need to resize the
                        // the table. See #1851
                        var widthChanged = (scope.shakerTable.w != scope.shakerTable.container.offsetWidth);
                        var heightChanged = (scope.shakerTable.h != scope.shakerTable.container.offsetHeight - scope.shakerTable.headerHeight);
                        if (widthChanged || heightChanged) {
                            scope.shakerTable.setup();
                            scope.shakerTable.scroll.setScrollXY(formerScrollLeft, formerScrollTop);
                        }
                    }
                });
                var wrappedDebouncedResizingHandler = function() {
                    if (scope.shakerTable && scope.shakerTable.scroll) {
                        var scrollBar = scope.shakerTable.scroll;
                        formerScrollLeft = scrollBar.scrollLeft;
                        formerScrollTop = scrollBar.scrollTop;
                    } else {
                        // a table is being refreshed, keep the last known values of the scroll position
                    }
                    debouncedResizingHandler();
                };

                scope.$on('scrollToLine', function(e, lineNum) {
                    scope.scrollToLine(lineNum);
                });

                scope.scrollToLine = function(lineNum){
                    var table = scope.shakerTable;
                    if (table && table.scroll) {
                        var nbRowsVisible = table.h / table.rowHeight; // we need the float value
                        var firstVisibleRow = table.scroll.scrollTop / table.rowHeight; // we need the float value
                        var x = table.scroll.scrollLeft;
                        if (lineNum == -1) {
                            let y = table.nbRows * table.rowHeight;
                            table.scroll.setScrollXY(x, y);
                        } else if (lineNum <= firstVisibleRow) {
                            let y = Math.max(lineNum, 0) * table.rowHeight;
                            table.scroll.setScrollXY(x,y);
                        } else if (lineNum >= firstVisibleRow + nbRowsVisible - 1) {
                            let y = (Math.min(lineNum, table.nbRows) + 1) * table.rowHeight - table.h;
                            table.scroll.setScrollXY(x,y);
                        }
                    }
                }

                scope.$on("compareColumnValuesChangeRow", function (_event, direction, compareColumnId) {
                    if (compareColumnId && compareColumnId !== attrs.fatId) {
                        return; // This event is no destined to us
                    }
                    scope.changeSelectedRow(direction)
                });

                scope.$on('reflow',wrappedDebouncedResizingHandler);
                $(window).on("resize.shakerTable",wrappedDebouncedResizingHandler);
                scope.$on('resize', wrappedDebouncedResizingHandler);
                scope.$on('visualIfResize_', wrappedDebouncedResizingHandler);
                scope.$on("compareColumnValuesViewResize", (event, payload) => {
                    if (payload === "close") {
                        scope.selectRow(null);
                    }
                    wrappedDebouncedResizingHandler();
                });

                $doc.bind("click.shakerTable", scope.hidePopups);
                scope.$on("$destroy", function() {
                    scope.$broadcast("shakerIsGettingDestroyed");
                    $(window).off('.shakerTable');
                    $doc.off('.shakerTable');
                    if (scope.shakerTable) scope.shakerTable.cleanUp();
                    if (scope.tableScope) {
                        scope.tableScope.$destroy();
                    }
                    /* I'm not 100% clear on why we need this but experimentally,
                     * this helps avoid some leaks ... */
                    scope.shakerTable = undefined;
                    scope.tableModel = undefined;
                });

                scope.$on("forcedShakerTableResizing", wrappedDebouncedResizingHandler);

                scope.$watch("shaker.coloring.highlightWhitespaces", function(nv){
                    if (nv === undefined) return;
                    const tableData = scope.$eval(tableDataExpr);
                    if (tableData) {
                        scope.setNewTable(tableData);
                    }
                });

                scope.$watch(tableDataExpr, function(tableData) {

                    var curScope = undefined;
                    scope.hidePopups();
                    if (tableData) {
                        scope.setNewTable(tableData);
                    }
                });

            }
        }
    });

    app.directive('columnHeader', function($controller, CreateModalFromDOMElement, CreateModalFromTemplate, ContextualMenu, $state, DataikuAPI, WT1,
                                           ShakerSuggestionsEngine, $window, translate, $stateParams, findAppliedColoringGroup, ConditionalFormattingEditorService) {
        return {
            restrict: 'A',
            replace: false,
            templateUrl: '/templates/shaker/column_header.html',
            scope: true,
            link: function(scope, element, attrs) {

                scope.$on("shakerIsGettingDestroyed", function(){
                    /* Since fattable does not use jQuery to remove its elements,
                     * we need to use jQuery ourselves to remove our children (and
                     * ourselves).
                     * Doing that will ensure that the jQuery data cache is cleared
                     * (it's only cleared when it's jQuery that removes the element)
                     * Without that, since Angular has used jQuery().data() to retrieve
                     * some stuff in the element, the jQuery data cache will always
                     * contain the scope and ultimately the element, leading to massive
                     * DOM leaks
                     */
                    element.empty();
                    element.remove();
                })

                scope.storageTypes = [
                    ['string', 'String'],
                    ['int', 'Integer'],
                    ['double', 'Double'],
                    ['float', 'Float'],
                    ['tinyint', 'Tiny int (8 bits)'],
                    ['smallint', 'Small int (16 bits)'],
                    ['bigint', 'Big int (64 bits)'],
                    ['boolean', 'Boolean'],
                    ['date', 'Datetime with tz'],
                    ['dateonly', 'Date only'],
                    ['datetimenotz', 'Datetime no tz'],
                    ['geopoint', "Geo Point"],
                    ['geometry', "Geometry / Geography"],
                    ['array', "Array"],
                    ['object', "Complex object"],
                    ['map', "Map"]
                ];

                // We avoid using a simple bootstrap dropdown
                // because we want to avoid having the hidden menus
                // DOM polluting our DOM tree.

                scope.anyMenuShown = false;

                scope.menusState = {
                    name: false,
                    meaning: false,
                    type : false,
                    color : false
                }

                scope.menu = new ContextualMenu({
                    template: "/templates/shaker/column-header-contextual-menu.html",
                    cssClass : "column-header-dropdown-menu",
                    scope: scope,
                    contextual: false,
                     onOpen: function() {
                        scope.menusState.name = true;
                    },
                    onClose: function() {
                        scope.menusState.name = false;
                    },
                    enableClick: true
                });

                scope.meaningMenu = new ContextualMenu({
                    template: "/templates/shaker/edit-meaning-contextual-menu.html",
                    cssClass : "column-header-meanings-menu",
                    scope: scope,
                    contextual: false,
                     onOpen: function() {
                        scope.menusState.meaning = true;
                    },
                    onClose: function() {
                        scope.menusState.meaning = false;
                    }
                });

                scope.datasetStorageTypeMenu = new ContextualMenu({
                    template: "/templates/shaker/edit-storagetype-contextual-menu.html",
                    cssClass : "column-header-types-menu",
                    scope: scope,
                    contextual: false,
                     onOpen: function() {
                        scope.menusState.type = true;
                    },
                    onClose: function() {
                        scope.menusState.type = false;
                    }
                });
                scope.colorMenu = new ContextualMenu({
                    template: "/templates/shaker/column-num-color-contextual-menu.html",
                    cssClass : "column-colors-menu",
                    scope: scope,
                    contextual: false,
                     onOpen: function() {
                        scope.menusState.color = true;
                    },
                    onClose: function() {
                        scope.menusState.color = false;
                    }
                });

                scope.toggleHeaderMenu = function() {

                    if (!scope.menusState.name) {
                         element.parent().append(element); //< do not remove this!
                        // It puts the element at the end, and put the menu
                        // over the siblings
                        // The former z-index machinery is broken by the use of css transform.
                        scope.menu.openAlignedWithElement(element.find(".name"), function() {}, true, true);
                    } else {
                        scope.menu.closeAny();
                    }
                };

                scope.toggleMeaningMenu = function() {
                    if (!scope.menusState.meaning) {
                        element.parent().append(element); //< do not remove this!
                        scope.meaningMenu.openAlignedWithElement(element.find(".meaning"), function() {}, true, true);
                    } else {
                        scope.meaningMenu.closeAny();
                    }
                };
                scope.toggleStorageTypeMenu = function() {
                    if (!scope.menusState.type) {
                        element.parent().append(element); //< do not remove this!
                        scope.datasetStorageTypeMenu.openAlignedWithElement(element.find(".storage-type"), function() {}, true, true);
                    } else {
                        scope.datasetStorageTypeMenu.closeAny();
                    }
                };
                 scope.toggleColorMenu = function() {
                    if (!scope.menusState.color) {
                        element.parent().append(element); //< do not remove this!
                        scope.colorMenu.openAlignedWithElement(element.find(".progress:visible"), function() {}, true, true);
                    } else {
                        scope.colorMenu.closeAny();
                    }
                };

                /**
                 * Returns the rules (filter descriptions) targeting a column. If none are found, returns null.
                 */
                function getRulesGroupFilterDescs (columnName) {
                    const coloring = scope.shaker.coloring;

                    if (coloring.scheme === "COLORING_GROUPS") {
                        const coloringGroup = findAppliedColoringGroup(coloring.coloringGroups, columnName);
                        if (coloringGroup && coloringGroup.scheme === "RULES") {
                            if (coloringGroup.scope === "COLUMNS" || coloringGroup.scope === "ALL_COLUMNS_BASED_ON_ANOTHER_COLUMN") {
                                return coloringGroup.rulesGroup.filterDescs;
                            }
                        }
                    }
                    // Legacy
                    else if (coloring.scheme === "INDIVIDUAL_COLUMNS_RULES" || coloring.scheme === "SINGLE_COLUMN_RULES") {
                        const columnRules = angular.copy(coloring.individualColumnsRules.filter(columnRule => columnRule.column === columnName));
                        // Takes the first one, and it is the list of FilterDesc `rulesDesc` that is returned as the "column rules"
                        if (columnRules && columnRules[0] && columnRules[0].rulesDesc) {
                            return columnRules[0].rulesDesc;
                        }
                    }

                    return null;
                };

                scope.ruleEnabledWithColorInfo = function(rules, ruleIndex) {
                    return rules && ruleIndex !== -1
                        && ruleIndex < rules.length
                        && rules[ruleIndex]
                        && rules[ruleIndex].enabled === true
                        && rules[ruleIndex].uiData
                        && rules[ruleIndex].uiData.color
                        && rules[ruleIndex].uiData.color.styleClass;
                }

                const isDefaultFontColor = function(color) {
                    return color === "#000000" || color === "#FFFFFF";
                }

                const isDefaultBackgroundColor = function(color) {
                    return color !== undefined && (color === "#F2F2F2" || color === "rgb(242,242,242)");
                }

                const isNoneColor = function(color) {
                    return color === "";
                }

                /**
                 * Computes the style for a header progress bar chunk of a column on which conditional formatting rules are applied
                 * for a specific rule.
                 * If the rule is not defined with a custom color, returns an empty style.
                 */
                scope.getProgressBarChunkColorStyle = function (columnName, ruleIndex) {
                    const rules = getRulesGroupFilterDescs(columnName);

                    if (
                        scope.ruleEnabledWithColorInfo(rules, ruleIndex) &&
                        "shaker-color-rule--custom" === rules[ruleIndex].uiData.color.styleClass
                    ) {
                        const color = rules[ruleIndex].uiData.color;

                        if (isDefaultFontColor(color.styleCustomFontColor) || isNoneColor(color.styleCustomFontColor)) {
                            if (isDefaultBackgroundColor(color.styleCustomBackgroundColor) || isNoneColor(color.styleCustomBackgroundColor)) {
                                return {
                                    "background-color": "none",
                                };
                            }

                            // Only displays the non default/none-color (background) color
                            return {
                                "background-color": color.styleCustomBackgroundColor,
                            };
                        } else if (isDefaultBackgroundColor(color.styleCustomBackgroundColor) || isNoneColor(color.styleCustomBackgroundColor)) {
                            // Only displays the non default/none-color (font) color
                            return { "background-color": color.styleCustomFontColor };
                        } else {
                            // Displays the background (top) and font (bottom) colors
                            return {
                                "background-color": color.styleCustomBackgroundColor,
                                "border-bottom": "5px solid " + color.styleCustomFontColor,
                            };
                        }
                    }

                    return {};
                };

                function groupBasedOnAnotherColumn(group, columnName) {
                    return (
                        group.scope === "ALL_COLUMNS_BASED_ON_ANOTHER_COLUMN" && group.basedOnColumnName === columnName
                    );
                }

                function extractBin(cls) {
                    if (binCache.has(cls)) {
                        // Check if bin is already in cache
                        return binCache.get(cls);
                    }
                    // Compute the bin and store it in cache
                    // Matches the possible bins. Bins are defined in TableColoringService.getColorBin
                    const matches = cls.match(/-?\d+/g);
                    if (!matches) return null;
                    let lastMatch = matches[matches.length - 1];

                    if (!cls.includes("--")) {
                        const parts = lastMatch.split("-"); // Splits around '-'
                        lastMatch = parts[parts.length - 1]; // Takes the part after the last '-'
                    }
                    binCache.set(cls, parseInt(lastMatch, 10));
                    return lastMatch;
                }

                scope.hasLegacyColorScheme = function () {
                    return scope.shaker.coloring.scheme !== "COLORING_GROUPS";
                };

                /**
                 * Computes the style for a header progress bar chunk of a column on which conditional formatting scale is applied
                 * If the scale is not defined (should never happen), returns an empty style.
                 */
                scope.getProgressBarChunkColorStyleForScale = function (columnName, cls) {
                    if (scope.hasLegacyColorScheme()) {
                        if (cls.indexOf("CN") !== -1) {
                            const bin = extractBin(cls);
                            const scalePalette = ConditionalFormattingEditorService.getScalePaletteForNumber();
                            if (0 < bin && bin < scalePalette.length) {
                                return {
                                    background:
                                        "linear-gradient(to right," +
                                        scalePalette[bin - 1] +
                                        "," +
                                        scalePalette[bin] +
                                        ")",
                                };
                            }
                        }
                        return "";
                    }
                    const coloringGroup = scope.shaker.coloring.coloringGroups.findLast(
                        group =>
                            group.enabled &&
                            (group.targetedColumnNames.includes(columnName) ||
                                groupBasedOnAnotherColumn(group, columnName))
                    );

                    if (
                        coloringGroup &&
                        coloringGroup.scheme === "COLOR_SCALE" &&
                        coloringGroup.colorScaleDef &&
                        coloringGroup.colorScaleDef.sample
                    ) {
                        const bin = extractBin(cls);
                        if (0 < bin && bin <= coloringGroup.colorScaleDef.sample.length) {
                            if (
                                cls.indexOf("CN") !== -1 &&
                                bin < coloringGroup.colorScaleDef.sample.length
                            ) {
                                return {
                                    background:
                                        "linear-gradient(to right," +
                                        coloringGroup.colorScaleDef.sample[bin - 1] +
                                        "," +
                                        coloringGroup.colorScaleDef.sample[bin] +
                                        ")",
                                };
                            } else {
                                return { "background-color": coloringGroup.colorScaleDef.sample[bin - 1] };
                            }
                        }
                    }
                    return { "background-color": "none" };
                };

                /**
                 * Computes the class name for a header progress bar chunk of a column on which conditional formatting rules are applied
                 * for a specific rule.
                 */
                scope.getProgressBarChunkColorClass = function (columnName, ruleIndex) {
                    const rules = getRulesGroupFilterDescs(columnName);

                    if (scope.ruleEnabledWithColorInfo(rules, ruleIndex)) {
                        // Non-custom color
                        if (
                            rules[ruleIndex].uiData.color.styleClass.includes("text") ||
                            rules[ruleIndex].uiData.color.styleClass.includes("background")
                        ) {
                            return rules[ruleIndex].uiData.color.styleClass + "__progress-bar";
                        }

                        // Custom color
                        return rules[ruleIndex].uiData.color.styleClass;
                    }

                    return "";
                };

                scope.column = scope.header;
                scope.columnIndex = scope.columns.indexOf(scope.column.name);

                scope.isType = function(x) {
                    return this.column.selectedType && this.column.selectedType.name == x;
                };

                scope.possibleMeanings = $.map(scope.column.possibleTypes, function(t) {
                    return t.name;
                });


                    // scope.unprobableTypes = [];
                    // for (var tIdx in scope.types) {
                    //     if ($.inArray(scope.types[tIdx], scope.possibleTypes) == -1) {
                    //         scope.unprobableTypes.push(scope.types[tIdx]);
                    //     }
                    // }

                    // Column have changed, need to update layout -
                    // We only do it for the last column of the layout
                    if (scope.header && scope.table && scope.table.headers && scope.table.headers.length &&
                        scope.column  === scope.table.headers[scope.table.headers.length - 1]) {
                        scope.$emit('updateFixedTableColumns');
                    }
                // });

                if (scope.shakerWritable) {
                    var s = ShakerSuggestionsEngine.computeColumnSuggestions(scope.column, CreateModalFromDOMElement, CreateModalFromTemplate,
                                undefined, undefined, scope.appConfig);
                    scope.suggestions = s[0];
                    scope.moreSuggestions = s[1];
                } else {
                    scope.suggestions = [];
                }

                if (scope.isRecipe){
                    scope.setStorageType = function(newType) {
                        scope.recipeOutputSchema.columns[scope.column.name].column.type = newType;
                        scope.recipeOutputSchema.columns[scope.column.name].persistent = true;
                        scope.schemaDirtiness.dirty = true;
                    };
                }

                scope.executeSuggestion = function(sugg) {
                    sugg.action(scope);
                };
                scope.hasSuggestions = function() {
                    return Object.keys(scope.suggestions).length > 0;
                };
                scope.hasMoreSuggestions = function() {
                    return Object.keys(scope.moreSuggestions).length > 0;
                };

                scope.hasInvalidData = function() {
                    return scope.column.selectedType && scope.column.selectedType.nbNOK > 0;
                };
                scope.hasEmptyData = function() {
                    return scope.column.selectedType && scope.column.selectedType.nbEmpty > 0;
                };

                scope.setColumnMeaning = function(newMeaning) {
                    scope.shakerHooks.setColumnMeaning(scope.column, newMeaning);
                };

                scope.editColumnUDM = function(){
                    CreateModalFromTemplate("/templates/meanings/column-edit-udm.html", scope, null, function(newScope){
                        newScope.initModal(scope.column.name, scope.setColumnMeaning);
                    });
                }

                scope.setColumnStorageType = function(newType){
                    var schemaColumn = null;

                    if (scope.shaker.origin == "DATASET_EXPLORE") {
                        schemaColumn = scope.column.datasetSchemaColumn;
                    } else if (scope.shaker.origin == "PREPARE_RECIPE") {
                        if (scope.column.recipeSchemaColumn) {
                            schemaColumn = scope.column.recipeSchemaColumn.column;
                        } else {
                            return; // ghost column, added by a stray filter for ex
                        }
                    } else {
                        throw Error("Can't set storage type here origin=" + scope.shaker.origin);
                    }
                    var impact = scope.shakerHooks.getSetColumnStorageTypeImpact(scope.column, newType);
                    if (impact != null) {
                        var doSetStorageType = function(data) {
                            if (data.justDoIt) {
                                scope.shakerHooks.setColumnStorageType(scope.column, newType, null);
                            } else {
                                CreateModalFromTemplate("/templates/shaker/storage-type-change-warning-modal.html", scope, null, function(newScope){
                                    newScope.ok = function() {
                                        newScope.dismiss();
                                        scope.shakerHooks.setColumnStorageType(scope.column, newType, newScope.extraActions.filter(function(a) {return a.selected;}).map(function(a) {return a.id;}));
                                    };
                                    newScope.warnings = data.warnings;
                                    newScope.extraActions = data.extraActions;
                                });
                            }
                        };
                        if (impact.success) {
                            impact.success(doSetStorageType).error(setErrorInScope.bind(scope));
                        } else {
                            impact.then(doSetStorageType);
                        }
                    }
                }
                scope.editThisColumnDetails = function() {
                    var schemaColumn = null;

                    if (scope.shaker.origin == "DATASET_EXPLORE") {
                        schemaColumn = scope.column.datasetSchemaColumn;
                    } else if (scope.shaker.origin == "PREPARE_RECIPE") {
                        if (scope.column.recipeSchemaColumn) {
                            schemaColumn = scope.column.recipeSchemaColumn.column;
                        } else {
                            return; // ghost column, added by a stray filter for ex
                        }
                    } else {
                        schemaColumn = angular.extend({}, scope.shaker.analysisColumnData[scope.column.name], {name: scope.column.name});
                        if (!schemaColumn) {
                            schemaColumn = {name: scope.column.name}
                            scope.shaker.analysisColumnData[scope.column.name] = schemaColumn;
                        }
                    }
                    scope.editColumnDetails(schemaColumn);
                }

                scope.goToImageView = function() {
                    scope.shakerState.activeView = 'images';
                };

                scope.isImageViewSpecialColumn = function() {
                    if (!scope.imageViewSettings || !scope.imageViewSettings.enabled) {
                        return false;
                    }

                    if (scope.imageViewSettings.pathColumn === scope.header.name) {
                        return true;
                    }

                    if (scope.imageViewSettings.annotationParams && scope.imageViewSettings.annotationParams.enabled && scope.imageViewSettings.annotationParams.annotationColumn === scope.header.name) {
                        return true;
                    }

                    return false;
                };

                scope.getModelType = function() {
                    if (scope.imageViewSettings && scope.imageViewSettings.enabled && scope.imageViewSettings.annotationParams && scope.imageViewSettings.annotationParams.enabled && scope.imageViewSettings.annotationParams.annotationColumn === scope.header.name) {
                        return scope.imageViewSettings.annotationParams.annotationType;
                    }
                    if (['Date', 'DateOnly', 'DatetimeNoTz'].indexOf(scope.header.selectedType.name) >= 0) {
                        return 'Temporal';
                    }
                    return 'Default';
                };


                scope.setFilterEmpty = function() {
                    if (!scope.column.selectedType) return;
                    scope.addValidityFilter(scope.column.name, scope.column.selectedType.name, "empty");
                };
                scope.setFilterOK = function() {
                    if (!scope.column.selectedType) return;
                    scope.addValidityFilter(scope.column.name, scope.column.selectedType.name, "ok");
                };
                scope.setFilterNOK = function() {
                    if (!scope.column.selectedType) return;
                    scope.addValidityFilter(scope.column.name, scope.column.selectedType.name, "nok");
                };

                scope.createColumnFilter = function() {
                    if (!scope.column.selectedType) return;
                    scope.addColumnFilter(scope.column.name, {}, 'full_string', scope.column.selectedType.name, scope.column.isDouble);
                };

                scope.deleteColumn = function() {
                    scope.addStepNoPreview("ColumnsSelector", {
                        "keep": false,
                        "appliesTo": "SINGLE_COLUMN",
                        "columns": [ scope.column.name ]
                    });
                    scope.mergeLastColumnDeleters();
                    scope.autoSaveForceRefresh();
                };

                scope.renameColumn = function() {
                    CreateModalFromDOMElement("#rename-column-box", scope, "RenameColumnController", function(newScope) {
                        newScope.setColumn(scope.column.name);
                    });
                };

                scope.datasetInsightLoaded = false;
                scope.callbackDatasetLoaded = function() {
                    if (typeof scope.refreshTableDone === 'function') {
                        scope.refreshTableDone();
                    }
                    scope.datasetInsightLoaded = true;
                }

                scope.moveColumn = function() {
                    CreateModalFromDOMElement("#move-column-box", scope, "MoveColumnController", function(newScope) {
                        newScope.setColumn(scope.column.name);
                    });
                };



                scope.createPredictionModelOnColumn = function(column, datasetName, mode = "auto", predictionType = null) {
                    if (scope.analysisCoreParams){ // In an analysis, we do not create a new analysis to create the ML task
                        $controller('AnalysisNewMLTaskController', { $scope: scope });
                    } else { // otherwise we create a new analysis
                        $controller('DatasetLabController', { $scope: scope});
                    }

                    if(mode === "deephub-computer-vision") {
                        scope.newPrediction(column, datasetName, mode, predictionType, scope.imageViewSettings.managedFolderSmartId);
                    } else {
                        scope.newPrediction(column, datasetName, mode, predictionType);
                    }
                };

                scope.inWorkspace = function() {
                    return $stateParams.workspaceKey !== undefined;
                }

                const getOutputDatasetOrSmartName = (scope, datasetSmartName) => {
                    if (scope && scope.recipe && scope.recipe.outputs && scope.recipe.outputs.main &&
                        Array.isArray(scope.recipe.outputs.main.items) && scope.recipe.outputs.main.items.length > 0) {
                        
                        return scope.recipe.outputs.main.items[0].ref;
                    }
                    return datasetSmartName;
                }

                scope.buildDataLineageRoute = function(column, datasetSmartName) {
                    // Use output dataset for prepare recipes
                    return $state.href('datalineage.graph', {contextProjectKey: $stateParams.projectKey, smartName: getOutputDatasetOrSmartName(scope, datasetSmartName), columnName: column});
                };

                scope.selectedShakerOrigin = function(shakerOrigin) {
                    switch (shakerOrigin) {
                        case "DATASET_EXPLORE":
                            return "dataset"
                        case "PREPARE_RECIPE":
                            return "prepare-recipe"
                        case "ANALYSIS":
                            return "analysis"
                        default:
                            return undefined
                    }
                }

                scope.buildDataLineageRouteWT1Event = function(datasetSmartName) {
                    WT1.tryEvent("data-lineage-access", () => ({
                        from: `${scope.selectedShakerOrigin(scope.shaker.origin)}-column-header-menu`,
                        dataseth: md5($stateParams.projectKey + "." + getOutputDatasetOrSmartName(scope, datasetSmartName))
                    }));
                };

                // For interactive search filter menu
                let isInteractiveSearchFilterMenuVisible = false;
                scope.interactiveSearchFilterMenu = new ContextualMenu({
                    template: "/templates/shaker/column-interactive-search-filter-panel.html",
                    cssClass : "ff-contextual-menu",
                    scope: scope,
                    contextual: false,
                    handleKeyboard: false,
                    onOpen: function() {
                        isInteractiveSearchFilterMenuVisible = true;
                    },
                    onClose: function() {
                        isInteractiveSearchFilterMenuVisible = false;
                    },
                    enableClick: true
                });

                scope.showInteractiveSearchFilterMenu = function() {
                    let openAtX = $(element).offset().left;
                    if (openAtX > $($window).width()/2) {
                        openAtX += $(element).outerWidth();
                    }
                    scope.interactiveSearchFilterMenu.openAtXY(openAtX, $(element).offset().top + $(element).height(), function() {}, false, true); // NOSONAR: OK to have empty method
                };

                scope.toggleInteractiveSearchFilterMenu = function() {
                    if (scope.isInteractiveSearchFilterMenuVisible) {
                        scope.hideInteractiveSearchFilterMenu();
                    } else {
                        scope.showInteractiveSearchFilterMenu();
                    }
                };

                scope.hideInteractiveSearchFilterMenu = function() {
                    scope.interactiveSearchFilterMenu.closeAny();
                };
            }
        };
    });

app.controller("InteractiveSearchFilterPanelController", function($scope, DateUtilsService) {
    $scope.facetUiState = {timezoneDateRangeModel: "UTC", textFilters: []};

    $scope.submitDateRange = function(columnName) {
        const from = $scope.facetUiState.fromDateRangeModel;
        const to = $scope.facetUiState.toDateRangeModel;
        const tz = $scope.facetUiState.timezoneDateRangeModel;

        const fromDate = from != null ? DateUtilsService.convertDateFromTimezone(from, tz) : null;
        const toDate = to != null ? DateUtilsService.convertDateFromTimezone(to, tz) : null;
        $scope.addDateRangeToInteractiveFilter(columnName, fromDate, toDate);
        $scope.hideInteractiveSearchFilterMenu();
    }

    $scope.submitTextFilter = function(columnName) {
        $scope.appendTextValuesToInteractiveFilter(columnName, $scope.facetUiState.textFilters);
        $scope.hideInteractiveSearchFilterMenu();
    }

    $scope.isValidDateRange = function() {
        if (!$scope.facetUiState.toDateRangeModel && !$scope.facetUiState.fromDateRangeModel) {
            return false;
        }
        if ($scope.facetUiState.toDateRangeModel && $scope.facetUiState.fromDateRangeModel && $scope.facetUiState.fromDateRangeModel > $scope.facetUiState.toDateRangeModel) {
            return false;
        }

        return true;
    }
});

app.controller("ShakerEditColumnDetailsController", function($scope, $controller, DataikuAPI, $state, Debounce, $stateParams, categoricalPalette, ContextualMenu, CreateModalFromTemplate){
    $scope.column = null;

    $scope.uiState = {};

    $scope.setColumn = function(column) {
        $scope.column = column;
    }

    $scope.save = function() {
        if ($scope.column.customFields && Object.keys($scope.column.customFields).length == 0) {
            delete $scope.column.customFields;
        }
        const prevColumnComment = $scope.dataset?.schema?.columns?.find(item => item.name === $scope.column.name)?.comment || null;
        if (areStringsMeaningfullyDifferent(prevColumnComment, $scope.column.comment)) {
            $scope.column.isColumnEdited = true;
        }
        $scope.shakerHooks.updateColumnDetails($scope.column);
        $scope.dismiss();
    };


    $scope.openMeaningMenu = function($event, column) {
            $scope.meaningMenu.openAtXY($event.pageX, $event.pageY);
            $scope.meaningColumn = column;
    };

    $scope.setColumnMeaning = function(meaningId) {
        $scope.meaningColumn.meaning = meaningId;
        $(".code-edit-schema-box").css("display", "block");
    };

    $scope.editColumnUDM = function() {
        CreateModalFromTemplate("/templates/meanings/column-edit-udm.html", $scope, null, function(newScope) {
            newScope.initModal($scope.meaningColumn.name, $scope.setColumnMeaning);
        });
    };

    $scope.meaningMenu = new ContextualMenu({
        template: "/templates/shaker/edit-meaning-contextual-menu.html",
        cssClass : "column-header-meanings-menu pull-right",
        scope: $scope,
        contextual: false,
        onOpen: function() {},
        onClose: function() {}
    });

    //Here the values "", null & undefined are treated in the same way as non-meaningful
    function areStringsMeaningfullyDifferent(str1, str2) {
        // If both are null, undefined, or empty return false
        if ((str1 ?? "") === "" && (str2 ?? "") === "") {
            return false;
        }
        return str1 !== str2;
    }
});

})();
