(function() {
    'use strict';
    /** @typedef {import('../../types').ChartDef} ChartDef */
    /** @typedef {import('../../types').GeneratedSources.PivotTableTensorRPivotTableRequestesponse} PivotTableRequest */

    const app = angular.module('dataiku.charts');

    /**
     * Service responsible for computing pivot requests by chart type compatible with the get-pivot-response api.
     * (!) This file previously was in static/dataiku/js/simple_report/chart_compute_requests.js
     */
    app.service('ChartRequestComputer', function(ChartDimension, LoggerProvider, BinnedXYUtils, _GeometryCommon, ChartColorUtils, ChartStoreFactory, CHART_TYPES, CHART_VARIANTS, ChartUADimension, MultiplotUtils, ChartUsableColumns, ChartCustomMeasures, ReferenceLines, ColumnAvailability, ChartFeatures, ColorMode) {
        const Logger = LoggerProvider.getLogger('chartrequest');

        /* Helpers */

        const has = function(array) {
            return array && array.length > 0;
        };

        const makeAggregatedAxis = function(dim, chartDef, isInteractiveDateDimension = false, isMainInteractiveDateAxis = false) {
            dim = angular.copy(dim);
            const axis = {
                column: dim.column,
                type: dim.type,
                sortPrune: {},
                name: dim.name,
                isRBND: dim.isRBND
            };

            if (dim.type == 'NUMERICAL' && dim.numParams.mode != 'TREAT_AS_ALPHANUM') {
                axis.numParams = dim.numParams;
            } else if (dim.type == 'NUMERICAL' && dim.numParams.mode == 'TREAT_AS_ALPHANUM') {
                dim.type = 'ALPHANUM';
                axis.type = dim.type;
            } else if (dim.type === 'DATE' && dim.dateParams.mode != 'TREAT_AS_ALPHANUM') {
                axis.dateParams = ChartDimension.buildDateParamsForAxis(dim, chartDef.type, isMainInteractiveDateAxis);
            } else if (dim.type === 'DATE' && dim.dateParams.mode == 'TREAT_AS_ALPHANUM') {
                dim.type = 'ALPHANUM';
                axis.type = dim.type;
            }

            if (dim.type == 'ALPHANUM' || ChartDimension.isUngroupedNumerical(dim)) {
                axis.sortPrune.maxValues = dim.maxValues;
            }

            axis.sortPrune.sortType = dim.sort.type;
            if (dim.sort.type == 'AGGREGATION') {
                axis.sortPrune.aggregationSortId = dim.sort.measureIdx;
                axis.sortPrune.sortAscending = dim.sort.sortAscending;
            } else if (dim.sort.type == 'COUNT') {
                axis.sortPrune.sortAscending = dim.sort.sortAscending;
            } else {
                axis.sortPrune.sortAscending = true; // NATURAL => ASC !
            }

            axis.sortPrune.generateOthersCategory = dim.generateOthersCategory;
            axis.sortPrune.filters = dim.filters;

            if (!ChartFeatures.supportSorting(dim, chartDef, !isMainInteractiveDateAxis)) {
                axis.sortPrune = { sortType: 'NATURAL', sortAscending: true };
            }

            return axis;
        };

        const measureToAggregation = function(measure, id) {
            const ret = {
                id: id,
                column: measure.column,
                type: measure.type,
                'function': measure.function,
                'computeMode': measure.computeMode || 'NORMAL',
                'computeModeDim': measure.computeModeDim,
                customFunction: measure.customFunction,
                percentile: measure.percentile
            };
            return ret;
        };


        /**
         * Populates the request.aggregations array with all relevant measures used across the chart.
         *
         * Important note:
         * The tensor `counts` returned by the backend depends on the full list of requested aggregations.
         * As a result, we assume that a non-zero count means the bin is visible somewhere in the chart context.
         *
         * This means all aggregations added to `request.aggregations` must be strongly related to each other.
         * If unrelated aggregations are mixed in, the resulting counts may incorrectly suggest visibility
         * and lead to misleading or incorrect visual behavior.
         *
         * Always ensure that any added aggregation is relevant to the chart's current grouping context.
         */
        const addAggregations = function(request, chartDef, colorMeasures = []) {
            request.aggregations = chartDef.genericMeasures.map(measureToAggregation)
                .concat(chartDef.tooltipMeasures.map(measureToAggregation))
                .concat(colorMeasures)
                .concat(ChartFeatures.canDisplayTotalValues(chartDef) && chartDef.stackedColumnsOptions
                    && chartDef.stackedColumnsOptions.totalsInChartDisplayOptions
                    && chartDef.stackedColumnsOptions.totalsInChartDisplayOptions.additionalMeasures
                    ? chartDef.stackedColumnsOptions.totalsInChartDisplayOptions.additionalMeasures.map(measureToAggregation) : [])
                .concat(_.flatten(chartDef.genericMeasures.map(m =>
                    m.valuesInChartDisplayOptions && m.valuesInChartDisplayOptions.additionalMeasures
                        ? m.valuesInChartDisplayOptions.additionalMeasures.map(measureToAggregation)
                        : [])));
            request.count = true;
        };

        const addUa = function(request, ua, id) {
            const ret = {
                id: id,
                column: ua.column,
                type: ua.type
            };

            if (ua.treatAsAlphanum && ua.type == 'NUMERICAL') {
                ret.type = 'ALPHANUM';
            } else if (ua.type == 'DATE') {
                ret.dateMode = ua.dateMode;
            }
            request.columns.push(ret);
        };

        /**
         * Adds filters to the given request object from the chart definition filters.
         *
         * @param {PivotTableRequest}  request
         * @param {ChartDef}           chartDef
         */
        const addFilters = function(request, chartDef) {
            request.filters = angular.copy(chartDef.filters).map((filter) => {
                if (filter.selectedValues) {
                    filter.selectedValues = Object.keys(filter.selectedValues);
                }
                if (filter.excludedValues) {
                    filter.excludedValues = Object.keys(filter.excludedValues);
                }

                return filter;
            });
        };

        const clipLeaflet = function(chartSpecific, request) {
            if (chartSpecific.leafletMap) {
                const bounds = chartSpecific.leafletMap.getBounds();
                request.minLon = bounds.getWest();
                request.maxLon = bounds.getEast();
                request.minLat = bounds.getSouth();
                request.maxLat = bounds.getNorth();
            }
        };

        const getScatterAxis = function(dim) {
            const ret = {
                column: dim.column,
                type: dim.type
            };

            if (dim.treatAsAlphanum && dim.type == 'NUMERICAL') {
                ret.type = 'ALPHANUM';
            }
            if (ret.type == 'ALPHANUM') {
                ret.sortPrune = {
                    sortType: dim.sortBy || 'COUNT'
                };
                if (dim.sortBy == 'COUNT') {
                    ret.sortPrune.sortAscending = false;
                } else {
                    ret.sortPrune.sortAscending = true;
                }
            } else if (ret.type == 'DATE') {
                ret.dateMode = dim.dateMode;
            }
            return ret;
        };

        /* Handling of chart types */

        const computeScatter = function(chartDef, chartSpecific) {
            Logger.info('Compute scatter request for', chartDef);
            const request = {};
            request.type = 'SCATTER_NON_AGGREGATED';

            request.maxRows = !_.isNil(chartDef.scatterOptions.numberOfRecords) ? chartDef.scatterOptions.numberOfRecords : 1000000;
            request.xAxis = getScatterAxis(chartDef.uaXDimension[0]);
            request.yAxis = getScatterAxis(chartDef.uaYDimension[0]);
            request.columns = [];

            if (has(chartDef.uaSize)) {
                addUa(request, chartDef.uaSize[0], 'size');
            }

            if (has(chartDef.uaColor)) {
                addUa(request, chartDef.uaColor[0], 'color');
            }

            if (has(chartDef.uaShape)) {
                chartDef.uaShape[0].type = 'ALPHANUM';
                addUa(request, chartDef.uaShape[0], 'shape');
            }

            chartDef.uaTooltip.forEach(function(ua, idx) {
                addUa(request, ua, 'tooltip_' + idx);
            });

            addFilters(request, chartDef);
            addSideRequests(chartDef, request, [chartDef.uaXDimension[0], chartDef.uaYDimension[0]], chartSpecific, false);
            return request;
        };

        const computeScatterMultiplePairs = function(chartDef, chartSpecific) {
            Logger.info('Compute scatter request for', chartDef);
            const request = {};
            request.type = 'SCATTER_MULTIPLE_PAIRS_NON_AGGREGATED';

            request.maxRows = !_.isNil(chartDef.scatterMPOptions.numberOfRecords) ? chartDef.scatterMPOptions.numberOfRecords : 1000000;
            request.axisPairs = [];
            request.columns = [];

            const displayablePairs = ChartUADimension.getDisplayableDimensionPairs(chartDef.uaDimensionPair);
            const usedDimensions = [];

            displayablePairs.forEach(function(pair) {
                const uaXDimension = ChartUADimension.getPairUaXDimension(displayablePairs, pair);
                const xAxis = getScatterAxis(uaXDimension[0]);
                const yAxis = getScatterAxis(pair.uaYDimension[0]);
                const axisPair = { xAxis, yAxis };

                request.axisPairs.push(axisPair);

                usedDimensions.push(uaXDimension[0]);
                usedDimensions.push(pair.uaYDimension[0]);
            });


            chartDef.uaTooltip.forEach(function(ua, idx) {
                addUa(request, ua, 'tooltip_' + idx);
            });

            addFilters(request, chartDef);
            addSideRequests(chartDef, request, usedDimensions, chartSpecific, false);
            return request;
        };

        const computeAdminMap = function(chartDef, chartSpecific) {
            const request = {};
            const dim0 = chartDef.geometry[0];

            request.type = 'AGGREGATED_GEO_ADMIN';
            request.geoColumn = dim0.column;

            if (!dim0.adminLevel) {
                dim0.adminLevel = 2;
            }
            request.adminLevel = dim0.adminLevel;
            request.filled = chartDef.variant == 'filled_map';

            clipLeaflet(chartSpecific, request);

            request.aggregations = [];
            if (has(chartDef.colorMeasure)) {
                const a = measureToAggregation(chartDef.colorMeasure[0]);
                a.id = 'color';
                request.aggregations.push(a);
            }
            if (has(chartDef.sizeMeasure)) {
                const a = measureToAggregation(chartDef.sizeMeasure[0]);
                a.id = 'size';
                request.aggregations.push(a);
            }

            request.aggregations = request.aggregations.concat(chartDef.tooltipMeasures.map(measureToAggregation));

            // post-prune limit is handled backend-side for admin-map

            request.count = true;

            addFilters(request, chartDef);
            return request;
        };

        const computeGridMap = function(chartDef, chartSpecific) {
            const request = {};
            const dim0 = chartDef.geometry[0];

            request.type = 'AGGREGATED_GEO_GRID';
            request.geoColumn = dim0.column;

            clipLeaflet(chartSpecific, request);

            request.lonRadiusDeg = chartDef.mapGridOptions.gridLonDeg;
            request.latRadiusDeg = chartDef.mapGridOptions.gridLatDeg;

            request.aggregations = [];
            if (has(chartDef.colorMeasure)) {
                const a = measureToAggregation(chartDef.colorMeasure[0]);
                a.id = 'color';
                request.aggregations.push(a);
            }
            if (has(chartDef.sizeMeasure)) {
                const a = measureToAggregation(chartDef.sizeMeasure[0]);
                a.id = 'size';
                request.aggregations.push(a);
            }

            chartDef.tooltipMeasures.forEach(function(measure, idx) {
                const a = measureToAggregation(measure);
                a.id = 'tooltip_' + idx;
                request.aggregations.push(a);
            });

            // There is no post-prune limit for grid-map

            request.count = true;

            addFilters(request, chartDef);
            return request;
        };

        const _handleStandardSubchartsAndAnimation = function(chartDef, request) {
            if (has(chartDef.facetDimension)) {
                request.axes.push(makeAggregatedAxis(chartDef.facetDimension[0], chartDef));

                /* Safety limit: refuse to draw more than 200 subcharts */
                request.postPruneLimits.push({ limit: 200, axesToCheck: [request.axes.length - 1], template: 'Too many subcharts: %d, limit %d' });
            }

            if (has(chartDef.animationDimension)) {
                request.axes.push(makeAggregatedAxis(chartDef.animationDimension[0], chartDef));

                /* Safety limit: refuse to have more than 1000 elements in animation */
                request.postPruneLimits.push({ limit: 1000, axesToCheck: [request.axes.length - 1], template: 'Too many animation steps: %d, limit %d' });
            }
        };

        const getSideRequestsFromMeasures = function(chartDef, measures) {
            const sideRequests = [];
            const withFilters = measures.filter(m => !m.ignoreExistingFilters);
            const withoutFilters = measures.filter(m => m.ignoreExistingFilters);

            withFilters.length && sideRequests.push(computeKpi({ ...chartDef, genericMeasures: withFilters }));
            withoutFilters.length && sideRequests.push(computeKpi({ ...chartDef, genericMeasures: withoutFilters }, false));
            return sideRequests;
        };

        const addRequestsFromSource = function(chartDef, request, source_key, source, usedColumns, customMeasures, allMeasures) {
            const DATASET_COLUMN = 'DatasetColumn';
            const CUSTOM_AGG = 'CustomAggregation';

            if (source && !source.length) {
                source = [source];
            }

            const sourceMeasures = source && source.filter(measure => [DATASET_COLUMN, CUSTOM_AGG].includes(measure.sourceType))
                .map(measure => {
                    const measureMeta = ColumnAvailability.getDynamicMeasureMeta(measure, allMeasures, usedColumns, customMeasures);
                    return { measure, measureMeta };
                })
                .filter(({ measureMeta }) => measureMeta.index !== -1)
                .map(({ measure, measureMeta }) => ({
                    ...measureMeta.measure,
                    function: measure.sourceType === CUSTOM_AGG ? 'CUSTOM' : measure.aggregation,
                    percentile: measure.percentile,
                    ignoreExistingFilters: measure.ignoreExistingFilters
                })) || [];

            if (sourceMeasures.length) {
                request.sideRequests = {
                    ...request.sideRequests,
                    [source_key]: getSideRequestsFromMeasures(chartDef, sourceMeasures)
                };
            }
        };

        const addSideRequests = function(chartDef, request, usedColumns, chartSpecific) {
            const customMeasures = ChartCustomMeasures.getMeasuresLikeCustomMeasures(chartSpecific.datasetProjectKey, chartSpecific.datasetName, chartSpecific.context);
            const allMeasures = ChartUsableColumns.getUsableColumns(chartSpecific.datasetProjectKey, chartSpecific.datasetName, chartSpecific.context);

            const sideRequestsSources = {
                'REFERENCE_LINES': chartDef.referenceLines,
                'GAUGE_MAX': chartDef.gaugeOptions.max,
                'GAUGE_MIN': chartDef.gaugeOptions.min,
                'GAUGE_TARGETS': chartDef.gaugeOptions.targets,
                'GAUGE_RANGES': chartDef.gaugeOptions.ranges
            };

            Object.entries(sideRequestsSources).forEach(([source_key, source]) => addRequestsFromSource(chartDef, request, source_key, source, usedColumns, customMeasures, allMeasures));
        };

        const computeStdAggregated = function(chartDef, chartSpecific, computeSubTotals = false) {
            const request = { type: 'AGGREGATED_ND', axes: [], postPruneLimits: [] };

            /*
             * Global limit on global number of elements in the tensor.
             * This is a pure safety limit to avoid the risk of allocating too much JS memory to parse the tensor
             * or to draw a too crazy number of SVG elements.
             * There are additional limits on the first axis and on subcharts/animation where relevant
             */
            request.postPruneLimits.push({ limit: 100000 });

            let interactiveDateDimension;
            if (has(chartDef.genericDimension0)) {
                const dimension = chartDef.genericDimension0[0];
                if (ChartDimension.isCandidateForInteractivity(dimension)) {
                    interactiveDateDimension = dimension;
                }
                request.axes.push(makeAggregatedAxis(dimension, chartDef, Boolean(interactiveDateDimension), true));

                /*
                 * In the vast majority of aggregated charts, the axis 0 is the horizontal axis and non-scrolling.
                 * It does not make sense to have much more elements on it than the potential number of pixels
                 * on the screen.
                 * Exception is stacked_bars, for which axis 0 is vertical and scrolling, so can go above
                 */
                if (chartDef.type != CHART_TYPES.STACKED_BARS) {
                    request.postPruneLimits.push({ axesToCheck: [0], limit: 10000, template: 'Too many elements in axis: %d, limit %d' });
                }
            }

            if (has(chartDef.genericDimension1)) {
                const dimension = chartDef.genericDimension1[0];
                const isSameInteractiveDateDimension = interactiveDateDimension && interactiveDateDimension.column === dimension.column;
                request.axes.push(makeAggregatedAxis(dimension, chartDef, isSameInteractiveDateDimension, false));
            }

            _handleStandardSubchartsAndAnimation(chartDef, request);

            // Should not happen ?
            if (chartDef.genericMeasures.length === 0) {
                throw new Error('To finish your chart, please select what you want to display and drop it in the \'Show\' section');
            }

            addAggregations(request, chartDef);
            addFilters(request, chartDef);

            if (interactiveDateDimension && chartSpecific.zoomUtils && !chartSpecific.zoomUtils.disableZoomFiltering) {
                request.filters.push(ChartDimension.buildZoomRuntimeFilter(interactiveDateDimension, chartSpecific.zoomUtils));
            }

            if (chartSpecific.zoomUtils && chartSpecific.zoomUtils.sequenceId) {
                request.sequenceId = chartSpecific.zoomUtils.sequenceId;
            }
            request.computeSubTotals = computeSubTotals;

            addSideRequests(chartDef, request, chartDef.genericMeasures, chartSpecific);

            return request;
        };

        const computeHierarchical = function(chartDef) {
            const request = { type: 'AGGREGATED_ND', axes: [], postPruneLimits: [] };

            if (has(chartDef.xDimension)) {
                request.axes.push(...chartDef.xDimension.map(xDim => makeAggregatedAxis(xDim, chartDef)));
            }

            if (has(chartDef.yDimension)) {
                request.axes.push(...chartDef.yDimension.map(yDim => makeAggregatedAxis(yDim, chartDef)));
            }

            request.aggregations = [];
            let colorAggs = [];
            if ((!ChartFeatures.canHaveConditionalFormatting(chartDef.type) || chartDef.colorMode === ColorMode.UNIQUE_SCALE)
                && has(chartDef.colorMeasure)) {
                colorAggs = measureToAggregation(chartDef.colorMeasure[0], 'color');
            } else if (chartDef.colorMode === ColorMode.COLOR_GROUPS) {
                colorAggs = chartDef.colorGroups
                    .filter(group => has(group.colorMeasure))
                    .map((group, i) => measureToAggregation(group.colorMeasure[0], `color_${i}`))
                ;
            }

            addAggregations(request, chartDef, colorAggs);

            addFilters(request, chartDef);
            request.computeSubTotals = true;

            const { store, id } = ChartStoreFactory.getOrCreate(chartDef.$chartStoreId);
            chartDef.$chartStoreId = id;
            const requestOptions = store.getRequestOptions();

            if (!requestOptions.removePivotTableLimitations) {
                /* Global limit on global number of elements in the post-prune tensor. */
                request.postPruneLimits.push({ limit: 200000 });
            }

            return request;
        };

        const computeBinnedXY = function(chartDef, width, height) {
            const request = { type: 'AGGREGATED_ND', aggregations: [], postPruneLimits: [] };

            /*
             * Global limit on global number of elements in the tensor.
             * This is a pure safety limit to avoid the risk of allocating too much JS memory to parse the tensor
             * or to draw a too crazy number of SVG elements.
             */
            request.postPruneLimits.push({ limit: 100000 });

            request.axes = [makeAggregatedAxis(chartDef.xDimension[0], chartDef), request.yAxis = makeAggregatedAxis(chartDef.yDimension[0], chartDef)];

            if (has(chartDef.sizeMeasure)) {
                request.aggregations.push(measureToAggregation(chartDef.sizeMeasure[0], 'size'));
            }
            if (has(chartDef.colorMeasure)) {
                request.aggregations.push(measureToAggregation(chartDef.colorMeasure[0], 'color'));
            }

            chartDef.tooltipMeasures.forEach(function(measure) {
                request.aggregations.push(measureToAggregation(measure, 'tooltip'));
            });

            _handleStandardSubchartsAndAnimation(chartDef, request);

            addFilters(request, chartDef);

            if (chartDef.variant == CHART_VARIANTS.binnedXYHexagon) {
                request.hexbin = true;
                const margins = { top: 10, right: 50, bottom: 50, left: 50 };
                const chartWidth = width - margins.left - margins.right;
                const chartHeight = height - margins.top - margins.bottom;
                const radius = BinnedXYUtils.getRadius(chartDef, chartWidth, chartHeight);
                request.hexbinXHexagons = Math.floor(chartWidth / (2 * Math.cos(Math.PI / 6) * radius));
                request.hexbinYHexagons = Math.floor(chartHeight / (1.5 * radius));
                request.$expectedVizWidth = chartWidth;
                request.$expectedVizHeight = chartHeight;
            }

            return request;
        };

        const computeGroupedXY = function(chartDef) {
            const request = {
                type: 'AGGREGATED_ND',
                axes: [makeAggregatedAxis(chartDef.groupDimension[0], chartDef)],
                postPruneLimits: []
            };

            /*
             * Global limit on global number of elements in the tensor.
             * This is a pure safety limit to avoid the risk of allocating too much JS memory to parse the tensor
             * or to draw a too crazy number of SVG elements.
             */
            request.postPruneLimits.push({ limit: 100000 });

            _handleStandardSubchartsAndAnimation(chartDef, request);

            request.aggregations = [
                measureToAggregation(chartDef.xMeasure[0]),
                measureToAggregation(chartDef.yMeasure[0])
            ];
            if (has(chartDef.sizeMeasure)) {
                request.aggregations.push(measureToAggregation(chartDef.sizeMeasure[0], 'size'));
            }
            if (has(chartDef.colorMeasure)) {
                request.aggregations.push(measureToAggregation(chartDef.colorMeasure[0], 'color'));
            }

            addFilters(request, chartDef);

            return request;
        };

        const computeLift = function(chartDef) {
            const request = {
                type: 'AGGREGATED_ND',
                axes: [makeAggregatedAxis(chartDef.groupDimension[0], chartDef)],
                postPruneLimits: []
            };

            /* For lift, it does not make sense to have more elements overall than horizontal pixels */
            request.postPruneLimits.push({ limit: 10000 });

            _handleStandardSubchartsAndAnimation(chartDef, request);

            request.aggregations = [
                measureToAggregation(chartDef.xMeasure[0]),
                measureToAggregation(chartDef.yMeasure[0])
            ];

            addFilters(request, chartDef);

            return request;
        };

        const computeBoxplots = function(chartDef) {
            const request = {
                type: 'BOXPLOTS',
                axes: []
            };
            if (has(chartDef.boxplotBreakdownDim)) {
                request.axes.push(makeAggregatedAxis(chartDef.boxplotBreakdownDim[0], chartDef));
            }
            if (has(chartDef.boxplotBreakdownDim) && has(chartDef.genericDimension1)) {
                request.axes.push(makeAggregatedAxis(chartDef.genericDimension1[0], chartDef));
            }
            request.column = {
                column: chartDef.boxplotValue[0].column,
                type: chartDef.boxplotValue[0].type
            };
            addFilters(request, chartDef);

            // post-prune limit is handled backend-side for boxplots

            return request;
        };

        const computeDensity2D = function(chartDef) {
            const request = {};
            request.type = 'DENSITY_2D';
            request.xColumn = chartDef.xDimension[0].column;
            request.yColumn = chartDef.yDimension[0].column;
            addFilters(request, chartDef);

            // No post-prune-limit for density-2D

            return request;
        };

        const computeKpi = function(chartDef, withFilters = true) {
            const request = {};
            request.type = 'NO_PIVOT_AGGREGATED';

            const colorAggs = chartDef.colorGroups
                .filter(group => has(group.colorMeasure))
                .map((group, i) => measureToAggregation(group.colorMeasure[0], `color_${i}`))
            ;

            addAggregations(request, chartDef, colorAggs);
            withFilters && addFilters(request, chartDef);

            // No post-prune-limit for density-2D

            return request;
        };

        const computeGauge = function(chartDef, chartSpecific) {
            const request = {};
            request.type = 'NO_PIVOT_AGGREGATED';
            addAggregations(request, chartDef);

            addFilters(request, chartDef);
            let rangesAndTargetsList = [];
            if (chartDef.gaugeOptions.targets) {
                rangesAndTargetsList = rangesAndTargetsList.concat(chartDef.gaugeOptions.targets);
            }
            if (chartDef.gaugeOptions.ranges) {
                rangesAndTargetsList = rangesAndTargetsList.concat(chartDef.gaugeOptions.ranges);
            }

            const usedDimensions = [
                chartDef.gaugeOptions.min,
                chartDef.gaugeOptions.max,
                ...rangesAndTargetsList
            ];
            addSideRequests(chartDef, request, usedDimensions, chartSpecific, false);

            return request;
        };

        const computeScatterMap = function(chartDef, chartSpecific) {
            const request = {};

            request.type = 'MAP_SCATTER_NON_AGGREGATED';
            request.maxRows = 100000;

            request.geoColumn = chartDef.geometry[0].column;

            clipLeaflet(chartSpecific, request);

            request.columns = [];
            if (has(chartDef.uaSize)) {
                addUa(request, chartDef.uaSize[0], 'size');
            }
            if (has(chartDef.uaColor)) {
                addUa(request, chartDef.uaColor[0], 'color');
            }
            chartDef.uaTooltip.forEach(function(ua, idx) {
                addUa(request, ua, 'tooltip_' + idx);
            });

            addFilters(request, chartDef);
            return request;
        };

        const computeDensityHeatMap = function(chartDef, chartSpecific) {
            // Temporary solution until definition of a new request.type
            return computeScatterMap(chartDef, chartSpecific);
        };

        const computeGeometryMap = function(chartDef, chartSpecific) {
            const request = {};

            request.type = 'RAW_GEOMETRY';
            request.maxRows = 100000;

            request.geoColumns = [];

            request.columns = [];

            const displayedGeoLayers = _GeometryCommon.getDisplayableLayers(chartDef.geoLayers);

            displayedGeoLayers.forEach(function(geoLayer, index) {
                const geometry = geoLayer.geometry[0];
                const geoColumn = { column: geometry.column };
                if (geometry.aggregationFunction) {
                    geoColumn.aggregationFunction = geometry.aggregationFunction;
                }
                request.geoColumns.push(geoColumn);
                if ((has(geoLayer.uaColor)) && (geoLayer.uaColor[0].column)) {
                    addUa(request, geoLayer.uaColor[0], ChartColorUtils.getPaletteName(index));
                }
            });
            chartDef.uaTooltip.forEach(function(ua, idx) {
                addUa(request, ua, 'tooltip_' + idx);
            });

            addFilters(request, chartDef);
            return request;
        };

        const computeWebapp = function(chartDef, chartSpecific) {
            const request = {};
            request.type = 'WEBAPP';
            request.columns = [];
            addFilters(request, chartDef);
            return request;
        };

        const computeFilters = function(chartDef) {
            const request = {
                type: 'FILTERS',
                columns: []
            };
            addFilters(request, chartDef);
            return request;
        };

        const svc = {
            addFilters,

            /**
             * Computes the corresponding pivot request from the given chart def
             *
             * @param {ChartDef} chartDef
             * @param {number} width
             * @param {number} height
             * @param {Record<string, unknown>} chartSpecific
             *
             * @return {PivotTableRequest.java}
             */
            compute: function(chartDef, width, height, chartSpecific) {
                // If the chart is facetted, the height the charts is different from the original height
                if (chartDef.facetDimension && chartDef.facetDimension.length) {
                    height = chartDef.chartHeight;
                }

                switch (chartDef.type) {

                    case CHART_TYPES.GROUPED_COLUMNS:
                    case CHART_TYPES.STACKED_COLUMNS:
                    case CHART_TYPES.STACKED_BARS:
                    case CHART_TYPES.LINES:
                    case CHART_TYPES.STACKED_AREA:
                    case CHART_TYPES.RADAR:
                    case CHART_TYPES.PIE:
                        return computeStdAggregated(chartDef, chartSpecific);

                    case CHART_TYPES.MULTI_COLUMNS_LINES: {
                        const computeSubTotals = MultiplotUtils.shouldSubtotalsBeCalculated(chartDef);
                        const modifiedChartDef = MultiplotUtils.shouldTakeColorDimensionInAccount(chartDef) ? chartDef : { ...chartDef, genericDimension1: [] };
                        return computeStdAggregated(modifiedChartDef, chartSpecific, computeSubTotals);
                    }
                    case CHART_TYPES.PIVOT_TABLE:
                    case CHART_TYPES.SANKEY:
                    case CHART_TYPES.TREEMAP:
                        return computeHierarchical(chartDef);

                    case CHART_TYPES.BINNED_XY:
                        return computeBinnedXY(chartDef, width, height);

                    case CHART_TYPES.GROUPED_XY:
                        return computeGroupedXY(chartDef);

                    case CHART_TYPES.SCATTER:
                        return computeScatter(chartDef, chartSpecific);

                    case CHART_TYPES.SCATTER_MULTIPLE_PAIRS:
                        return computeScatterMultiplePairs(chartDef, chartSpecific);

                    case CHART_TYPES.ADMINISTRATIVE_MAP:
                        return computeAdminMap(chartDef, chartSpecific);

                    case CHART_TYPES.GRID_MAP:
                        return computeGridMap(chartDef, chartSpecific);

                    case CHART_TYPES.SCATTER_MAP:
                        return computeDensityHeatMap(chartDef, chartSpecific);

                    case CHART_TYPES.DENSITY_HEAT_MAP:
                        return computeScatterMap(chartDef, chartSpecific);

                    case CHART_TYPES.GEOMETRY_MAP:
                        return computeGeometryMap(chartDef, chartSpecific);

                    case CHART_TYPES.BOXPLOTS:
                        return computeBoxplots(chartDef);

                    case CHART_TYPES.DENSITY_2D:
                        return computeDensity2D(chartDef);

                    case CHART_TYPES.GAUGE:
                        return computeGauge(chartDef, chartSpecific);

                    case CHART_TYPES.KPI:
                        return computeKpi(chartDef);

                    case CHART_TYPES.LIFT:
                        var req = computeLift(chartDef);
                        req.diminishingReturns = true;
                        return req;

                    case CHART_TYPES.WEBAPP:
                        return computeWebapp(chartDef, chartSpecific);

                    case CHART_TYPES.FILTERS:
                        return computeFilters(chartDef);

                    default:
                        Logger.error('Unhandled chart type', chartDef);
                        throw new Error('unknown chart type', chartDef);
                }
            }
        };
        return svc;
    });

})();
