(function() {
    'use strict';

    angular.module('dataiku.charts')
        .factory('GroupedXYChart', GroupedXYChart);

    // (!) This service previously was in static/dataiku/js/simple_report/scatter/aggr_scatter.js
    function GroupedXYChart(ChartManager, ChartDataWrapperFactory, ColorUtils, ChartDataUtils, SVGUtils, ChartAxesUtils, ChartYAxisPosition) {
        return function($container, chartDef, chartHandler, axesDef, data) {
            const chartData = ChartDataWrapperFactory.chartTensorDataWrapper(data, axesDef),
                facetLabels = chartData.getAxisLabels('facet') || [null]; // We'll through the next loop only once if the chart is not facetted

            const xMeasure = 0,
                yMeasure = 1,
                sizeMeasure = (chartDef.sizeMeasure.length) ? 2 : -1,
                colorMeasure = (chartDef.colorMeasure.length && chartDef.sizeMeasure.length) ? 3 : (chartDef.colorMeasure.length ? 2 : -1),
                color = ColorUtils.toRgba(chartDef.colorOptions.singleColor, chartDef.colorOptions.transparency);

            let sizeScale;

            if (sizeMeasure >= 0) {
                sizeScale = d3.scale.sqrt().domain(ChartDataUtils.getMeasureExtent(chartData, sizeMeasure, true)).range([Math.min(10, chartDef.bubblesOptions.defaultRadius), Math.max(10, chartDef.bubblesOptions.defaultRadius)]);
                chartDef.sizeMeasure.$mIdx = sizeMeasure;
            }

            if (colorMeasure >= 0) {
                chartDef.colorMeasure.$mIdx = colorMeasure;
            }

            const drawFrame = function(frameIdx, chartBase) {
                chartData.fixAxis('animation', frameIdx);
                facetLabels.forEach(function(facetLabel, f) {
                    const g = d3.select(chartBase.$svgs.eq(f).find('g.chart').get(0));
                    GroupedXYChartDrawer(g, chartDef, chartHandler, chartData.fixAxis('facet', f), chartBase, f);
                });

                // Signal to the callee handler that the chart has been successfully loaded. Dashboards use it to determine when all insights are completely loaded.
                if (typeof (chartHandler.loadedCallback) === 'function') {
                    chartHandler.loadedCallback();
                }
            };

            const yAxisID = ChartAxesUtils.computeYAxisID(ChartYAxisPosition.LEFT);

            ChartManager.initChart(chartDef, chartHandler, chartData, $container, drawFrame,
                {
                    x: { type: 'MEASURE', measureIdx: xMeasure, measure: chartDef.xMeasure, customExtent: chartDef.xAxisFormatting.customExtent },
                    [yAxisID]: { id: yAxisID, type: 'MEASURE', measureIdx: yMeasure, measure: chartDef.yMeasure, customExtent: ChartAxesUtils.getYAxisCustomExtent(chartDef.yAxesFormatting, yAxisID), position: ChartYAxisPosition.LEFT }
                },
                { type: 'MEASURE', measureIdx: colorMeasure, measure: chartDef.colorMeasure, withRgba: true });

            function GroupedXYChartDrawer(g, chartDef, chartHandler, chartData, chartBase, f) {

                function getBubbleColor(d) {
                    if (colorMeasure >= 0) {
                        return chartBase.colorScale(chartData.aggr(colorMeasure).get({ group: d.$i }));
                    } else {
                        return color;
                    }
                }

                const bubbles = g.selectAll('circle').data(chartData.getAxisLabels('group'), function(d, i) {
                    return i;
                });
                bubbles.enter().append('circle')
                    .attr('fill', getBubbleColor)
                    .each(function(d, i) {
                        d.$i = i;
                        chartBase.tooltips.addTooltipHandlers(this, { group: i, facet: f }, getBubbleColor(d, i));
                        chartBase.contextualMenu.addContextualMenuHandler(this, { group: i, facet: f });
                    });

                bubbles
                    .filter(function(d) {
                        return chartData.getCount({ group: d.$i }) === 0;
                    })
                    .transition()
                    .duration((chartDef.animationFrameDuration || 3000) / 2)
                    .attr('opacity', 0)
                    .attr('cx', null)
                    .attr('cy', null)
                    .attr('r', null);

                bubbles
                    .filter(function(d) {
                        return chartData.getCount({ group: d.$i }) > 0;
                    })
                    .transition()
                    .duration((chartDef.animationFrameDuration || 3000) / 2)
                    .attr('opacity', 1)
                    .attr('cx', function(d) {
                        return chartBase.xAxis.scale()(chartData.aggr(xMeasure).get({ group: d.$i }));
                    })
                    .attr('cy', function(d) {
                        return chartBase.yAxes[0].scale()(chartData.aggr(yMeasure).get({ group: d.$i }));
                    })
                    .attr('r', function(d) {
                        if (sizeScale) {
                            return sizeScale(chartData.aggr(sizeMeasure).get({ group: d.$i }));
                        } else {
                            return chartDef.bubblesOptions.defaultRadius;
                        }
                    })
                    .attr('fill', getBubbleColor);

                // Clip paths to prevent bubbles from overlapping axis when user chose a custom range which results in the bar being cropped (out of visible range)
                ChartAxesUtils.isCroppedChart(chartDef) && SVGUtils.clipPaths(chartBase, g, bubbles);

            }
        };
    }
})();
