// @ts-check
(function() {
    'use strict';
    /** @typedef {import("../../../types").AgGridRow} AgGridRow */
    /** @typedef {import("../../../types").DisplayTotalsToolPanelParams} DisplayTotalsToolPanelParams */
    /** @typedef {import("../../../types").GeneratedSources.ChartDef.PivotTableDisplayTotals} PivotTableDisplayTotals */

    /** @typedef {import("../../../../../../../../../../server/src/frontend/node_modules/ag-grid-community").IToolPanelComp} IToolPanelComp */
    /** @typedef {import("../../../../../../../../../../server/src/frontend/node_modules/ag-grid-community").ColDef} ColDef */
    /** @typedef {import("../../../../../../../../../../server/src/frontend/node_modules/ag-grid-community").ColGroupDef} ColGroupDef */
    /** @typedef {import("../../../../../../../../../../server/src/frontend/node_modules/ag-grid-community").GridApi} GridApi */
    /** @typedef {import("../../../../../../../../../../server/src/frontend/node_modules/ag-grid-community").ColumnApi} ColumnApi */

    angular.module('dataiku.charts')
        .factory('DisplayTotalsToolPanel', function() {

            const DISPLAY_TOTALS_TOOL_PANEL_LABEL = 'Display totals';
            const TOTALS_ID = 'toggle-totals';
            const TOTALS_ROWS_ID = 'toggle-total-rows';
            const TOTALS_COLUMNS_ID = 'toggle-total-columns';
            const GRAND_TOTALS_ID = 'toggle-grand-totals';
            const GRAND_TOTALS_ROWS_ID = 'toggle-grand-total-rows';
            const GRAND_TOTALS_COLUMNS_ID = 'toggle-grand-total-columns';

            /**
             * @implements IToolPanelComp
             */
            class DisplayTotalsToolPanel {

                constructor() {
                    /** @type {ColumnApi} */
                    this.columnApi = null;
                    /** @type {GridApi} */
                    this.gridApi = null;
                    /** @type {AgGridRow[]} */
                    this.rowData;
                    /** @type {string} */
                    this.subtotalColumnType;
                    /** @type {AgGridRow[]} */
                    this.subtotalRowData;
                    /** @type {AgGridRow} */
                    this.grandTotalRowData;
                    /** @type {ColDef & ColGroupDef} */
                    this.grandTotalColumnDef;
                    /** @type {boolean} */
                    this.hasSubtotalRows;
                    /** @type {boolean} */
                    this.hasSubtotalColumns;
                    /** @type {boolean} */
                    this.hasGrandTotalRow;
                    /** @type {boolean} */
                    this.hasGrandTotalColumn;
                    /** @type {Record<string, boolean>} */
                    this.disableState;
                    /** @type {Record<string, boolean>} */
                    this.checkedState;
                    /** @type {PivotTableDisplayTotals} */
                    this.pivotDisplayTotals;
                    /** @type {() => void} */
                    this.recomputeColorScale;

                    this._actions = {};
                    this._actions[TOTALS_ID] = { label: 'Totals', callback: () => this.toggleTotals(), childActions: [TOTALS_ROWS_ID, TOTALS_COLUMNS_ID] };
                    this._actions[TOTALS_ROWS_ID] = { label: 'Rows', callback: () => this.toggleTotalRows(), parentId: TOTALS_ID };
                    this._actions[TOTALS_COLUMNS_ID] = { label: 'Columns', callback: () => this.toggleTotalColumns(), parentId: TOTALS_ID };
                    this._actions[GRAND_TOTALS_ID] = { label: 'Grand Total', callback: () => this.toggleGrandTotals(), childActions: [GRAND_TOTALS_ROWS_ID, GRAND_TOTALS_COLUMNS_ID] };
                    this._actions[GRAND_TOTALS_ROWS_ID] = { label: 'Row', callback: () => this.toggleGrandTotalRows(), parentId: GRAND_TOTALS_ID };
                    this._actions[GRAND_TOTALS_COLUMNS_ID] = { label: 'Column', callback: () => this.toggleGrandTotalColumns(), parentId: GRAND_TOTALS_ID };
                    this.$rootElement;
                }


                refresh() {
                }

                destroy() {
                    delete this.rowData;
                    delete this.subtotalRowData;
                    delete this.grandTotalRowData;
                    this.clearClickListeners();
                }

                setCheckboxState(id, newState) {
                    this.checkedState[id] = newState;
                    if (this._actions[id].childActions) {
                        this._actions[id].childActions.map(childId => this.checkedState[childId] = newState);
                    } else {
                        const parentId = this._actions[id].parentId;
                        let switchParentState = true;
                        this._actions[parentId].childActions.map(childId => {
                            if (this.checkedState[childId] !== newState && !this.disableState[childId]) {
                                switchParentState = false;
                            }
                        });
                        if (switchParentState) {
                            this.checkedState[parentId] = newState;
                        }
                    }
                    this.renderTemplate();
                }

                toggleTotals() {
                    const shouldDisplay = !this.checkedState[TOTALS_ID];
                    if (shouldDisplay !== this.checkedState[TOTALS_ROWS_ID] && !this.disableState[TOTALS_ROWS_ID]) {
                        this.toggleTotalRows(true);
                    }
                    if (shouldDisplay !== this.checkedState[TOTALS_COLUMNS_ID] && !this.disableState[TOTALS_COLUMNS_ID]) {
                        this.toggleTotalColumns(true);
                    }
                    this.updateRows();
                }

                updateRows() {
                    this.recomputeColorScale();
                    this.gridApi.redrawRows();
                }

                toggleTotalRows(skipRowUpdate = false) {
                    const shouldDisplay = !this.checkedState[TOTALS_ROWS_ID];

                    if (shouldDisplay) {
                        const rowData = [...this.rowData, ...this.subtotalRowData];

                        if (this.hasGrandTotalRow && this.checkedState[GRAND_TOTALS_ROWS_ID]) {
                            rowData.push(this.grandTotalRowData);
                        }
                        this.gridApi.setRowData(rowData);
                    } else {
                        const rowData = [...this.rowData];

                        if (this.hasGrandTotalRow && this.checkedState[GRAND_TOTALS_ROWS_ID]) {
                            rowData.push(this.grandTotalRowData);
                        }
                        this.gridApi.setRowData(rowData);
                    }
                    this.pivotDisplayTotals.subTotals.rows = shouldDisplay;
                    this.setCheckboxState(TOTALS_ROWS_ID, shouldDisplay);
                    if (!skipRowUpdate) {
                        this.updateRows();
                    }
                }

                toggleTotalColumns(skipRowUpdate = false) {
                    const shouldDisplay = !this.checkedState[TOTALS_COLUMNS_ID];
                    let columnGroupShow = 'closed';
                    if (shouldDisplay) {
                        columnGroupShow = undefined;
                    }
                    this.columnApi.getAllColumns()
                        .filter(col => col.getUserProvidedColDef().type === this.subtotalColumnType)
                        .forEach(col => {
                            col.setColDef({ ...col.getColDef(), columnGroupShow }, { colId: col.getColId() });
                        });

                    this.gridApi.setColumnDefs(this.gridApi.getColumnDefs());
                    this.pivotDisplayTotals.subTotals.columns = shouldDisplay;
                    this.setCheckboxState(TOTALS_COLUMNS_ID, shouldDisplay);
                    if (!skipRowUpdate) {
                        this.updateRows();
                    }
                }

                toggleGrandTotals() {
                    const display = !this.checkedState[GRAND_TOTALS_ID];
                    if (display !== this.checkedState[GRAND_TOTALS_ROWS_ID] && !this.disableState[GRAND_TOTALS_ROWS_ID]) {
                        this.toggleGrandTotalRows();
                    }
                    if (display !== this.checkedState[GRAND_TOTALS_COLUMNS_ID] && !this.disableState[GRAND_TOTALS_COLUMNS_ID]) {
                        this.toggleGrandTotalColumns();
                    }
                }

                toggleGrandTotalRows() {
                    const shouldDisplay = !this.checkedState[GRAND_TOTALS_ROWS_ID];

                    if (shouldDisplay) {
                        this.gridApi.applyTransaction({ add: [this.grandTotalRowData] });
                    } else {
                        const grandTotalRowNode = this.gridApi.getRowNode(this.grandTotalRowData.id);
                        this.gridApi.applyTransaction({ remove: [grandTotalRowNode] });
                    }

                    this.setCheckboxState(GRAND_TOTALS_ROWS_ID, shouldDisplay);
                    this.pivotDisplayTotals.grandTotal.row = shouldDisplay;
                }

                toggleGrandTotalColumns() {
                    const shouldDisplay = !this.checkedState[GRAND_TOTALS_COLUMNS_ID];
                    if (this.grandTotalColumnDef.children) {
                        // @ts-ignore
                        this.grandTotalColumnDef.children.forEach(childCol => this.columnApi.setColumnVisible(childCol.field, shouldDisplay));
                    }
                    this.columnApi.setColumnVisible(this.grandTotalColumnDef.field, shouldDisplay);
                    this.pivotDisplayTotals.grandTotal.column = shouldDisplay;
                    this.setCheckboxState(GRAND_TOTALS_COLUMNS_ID, shouldDisplay);
                }

                /**
                 * Returns disable state
                 * @param {boolean} hasSubtotalRows
                 * @param {boolean} hasSubtotalColumns
                 * @param {boolean} hasGrandTotalRow
                 * @param {boolean} hasGrandTotalColumn
                 * @return {Record<string, boolean>}
                 */
                getDisableState(hasSubtotalRows, hasSubtotalColumns, hasGrandTotalRow, hasGrandTotalColumn) {
                    /** @type {Record<string, boolean>} */
                    const disableState = {};
                    disableState[TOTALS_ID] = !hasSubtotalRows && !hasSubtotalColumns;
                    disableState[TOTALS_ROWS_ID] = !hasSubtotalRows;
                    disableState[TOTALS_COLUMNS_ID] = !hasSubtotalColumns;
                    disableState[GRAND_TOTALS_ID] = !hasGrandTotalRow && !hasGrandTotalColumn;
                    disableState[GRAND_TOTALS_ROWS_ID] = !hasGrandTotalRow;
                    disableState[GRAND_TOTALS_COLUMNS_ID] = !hasGrandTotalColumn;
                    return disableState;
                }

                /**
                 * Returns init checked state, all true by default
                 * @param {boolean} hasSubtotalRows
                 * @param {boolean} hasSubtotalColumns
                 * @param {boolean} hasGrandTotalRow
                 * @param {boolean} hasGrandTotalColumn
                 * @return {Record<string, boolean>}
                 */
                getInitCheckedState(hasSubtotalRows, hasSubtotalColumns, hasGrandTotalRow, hasGrandTotalColumn) {
                    /** @type {Record<string, boolean>} */
                    const checkedState = {};
                    checkedState[TOTALS_ID] = (!hasSubtotalRows || this.pivotDisplayTotals.subTotals.rows) && (!hasSubtotalColumns || this.pivotDisplayTotals.subTotals.columns);
                    checkedState[TOTALS_ROWS_ID] = this.pivotDisplayTotals.subTotals.rows;
                    checkedState[TOTALS_COLUMNS_ID] = this.pivotDisplayTotals.subTotals.columns;
                    checkedState[GRAND_TOTALS_ID] = (!hasGrandTotalRow || this.pivotDisplayTotals.grandTotal.row) && (!hasGrandTotalColumn || this.pivotDisplayTotals.grandTotal.column);
                    checkedState[GRAND_TOTALS_ROWS_ID] = this.pivotDisplayTotals.grandTotal.row;
                    checkedState[GRAND_TOTALS_COLUMNS_ID] = this.pivotDisplayTotals.grandTotal.column;
                    return checkedState;
                }

                /**
                 * @param {DisplayTotalsToolPanelParams} params
                 */
                init(params) {
                    this.setParams(params);
                    this.eGui = document.createElement('div');
                    this.eGui.classList.add('expand-collapse-tool-panel');
                    this.params = params.api;
                    const renderPanel = () => {
                        this.renderTemplate();
                        this.setToolPanelButtonIcon();
                    };
                    params.api.addEventListener('gridReady', renderPanel);
                }
                /**
                 * @param {DisplayTotalsToolPanelParams} params
                 */
                setParams(params) {
                    this.pivotDisplayTotals = params.pivotDisplayTotals;
                    this.checkedState = this.getInitCheckedState(
                        params.hasSubtotalRows,
                        params.hasSubtotalColumns,
                        params.hasGrandTotalRow,
                        params.hasGrandTotalColumn
                    );
                    this.disableState = this.getDisableState(
                        params.hasSubtotalRows,
                        params.hasSubtotalColumns,
                        params.hasGrandTotalRow,
                        params.hasGrandTotalColumn
                    );
                    this.subtotalColumnType = params.subtotalColumnType;
                    this.$rootElement = params.$rootElement;
                    this.gridApi = params.api;
                    this.columnApi = params.columnApi;
                    this.rowData = params.rowData;
                    this.subtotalRowData = params.subtotalRowData;
                    this.grandTotalRowData = params.grandTotalRowData;
                    this.grandTotalColumnDef = params.grandTotalColumnDef;
                    this.hasSubtotalRows = params.hasSubtotalRows;
                    this.hasSubtotalColumns = params.hasSubtotalColumns;
                    this.hasGrandTotalRow = params.hasGrandTotalRow;
                    this.hasGrandTotalColumn = params.hasGrandTotalColumn;
                    this.recomputeColorScale = params.recomputeColorScale;
                }

                /**
                 * @param {DisplayTotalsToolPanelParams} params
                 */
                customRefresh(params) {
                    this.setParams(params);
                    this.renderTemplate();
                }

                getGui() {
                    return this.eGui;
                }

                setListeners() {
                    this.clearClickListeners();
                    for (const id in this._actions) {
                        if (Object.hasOwnProperty.call(this._actions, id)) {
                            const action = this._actions[id];
                            this.$rootElement.find(`.${this.getActionClassFromId(id)}`).on('change', (e) => {
                                e.stopPropagation();
                                action.callback();
                            });
                        }
                    }
                }

                setToolPanelButtonIcon() {
                    /** @type HTMLElement[] */
                    // @ts-ignore
                    const toolPanelButtons = this.$rootElement.find('.ag-side-button').toArray();
                    const displayTotalsButton = toolPanelButtons.filter(element => element.textContent.includes(DISPLAY_TOTALS_TOOL_PANEL_LABEL))[0];
                    const iconElement = displayTotalsButton.querySelector('.ag-icon');
                    iconElement.classList.remove('ag-icon', 'ag-icon-');
                    iconElement.classList.add('dku-icon-plus-16');
                }

                clearClickListeners() {
                    Object
                        .keys(this._actions).forEach(id => this.$rootElement.find(`.${this.getActionClassFromId(id)}`).off('click'));
                }

                /**
                 * @param {string} id
                 * @returns
                 */
                getActionClassFromId(id) {
                    return `pivot-table-display-totals-tool-panel__input-checkbox-${id}`;
                }

                renderTemplate() {
                    this.eGui.innerHTML = Object
                        .keys(this._actions)
                        .filter(id => this._actions[id].childActions !== undefined)
                        .map((id) => `
                            <ul class="pivot-table-display-totals-tool-panel__list-item ${this.getActionClassFromId(id)}">
                                <div class="df aic">
                                    <div class="pivot-table-display-totals-tool-panel__chevron-icon">
                                        <i class="dku-icon-chevron-right-12 dibvam"></i>
                                    </div>
                                    <div>
                                        <input class="pivot-table-display-totals-tool-panel__input-checkbox" type="checkbox" ${this.checkedState[id] && !this.disableState[id] ? 'checked' : ''} ${this.disableState[id] ? 'disabled' : ''}>
                                            ${this._actions[id].label}
                                        </input>
                                    </div>
                                </div>
                                ${this._actions[id].childActions.map(childId => `
                                <li class="pivot-table-display-totals-tool-panel__item ${this.getActionClassFromId(childId)}">
                                    <input class="pivot-table-display-totals-tool-panel__input-checkbox" type="checkbox" ${this.checkedState[childId] && !this.disableState[childId] ? 'checked' : ''}  ${this.disableState[childId] ? 'disabled' : ''}>
                                        ${this._actions[childId].label}
                                    </input>
                                </li>
                                `).join('')}
                            </ul>`)
                        .join('');
                    this.setListeners();
                }
            }
            DisplayTotalsToolPanel.label = DISPLAY_TOTALS_TOOL_PANEL_LABEL;
            DisplayTotalsToolPanel.id = 'displayTotals';
            return DisplayTotalsToolPanel;
        });
})();
