// @ts-check
(function() {
    'use strict';
    /** @typedef {import("../../../types").ChartDef} ChartDef */
    /** @typedef {import("../../../types").GeneratedSources.MeasureDef} MeasureDef */
    /** @typedef {import("../../../types").GeneratedSources.DimensionDef} DimensionDef */

    angular.module('dataiku.charts')
        .factory('PivotTableUtils', function(CHART_VARIANTS, StringUtils, DashboardUtils, ClipboardUtils, ChartFormattingPaneSections, $state) {
            const PIVOT_TABLE_FONT = '13px';
            const PIVOT_TABLE_CELL_OFFSET = 16;
            const FORMAT_HEADERS_LABEL = 'Format table headers';
            const FORMAT_HEADERS_ICON = 'dku-icon-paint-bucket-16';
            const COPY_TO_CLIPBOARD_LABEL = 'Copy to clipboard';
            const COPY_TO_CLIPBOARD_ICON = 'dku-icon-copy-step-16';
            const getHideHeaderLabel = (type) => {
                return `Hide ${type} headers`;
            };
            const HIDE_HEADER_ICON = 'dku-icon-eye-off-16';
            const COLUMN_MAX_WIDTH = 280;
            const canvas = document.createElement('canvas');
            /*
             * Necessary due to an issue with the HTML2DRenderingContext's measureText method,
             * which returns a slightly different value after page reload leading to the column labels sometimes being cropped
             */
            const PIVOT_TABLE_SAFETY_OFFSET = 2;

            return {
                COLUMN_MAIN_HEADER_CLASSNAME: 'pivot-table-column-main-header',
                COLUMN_SUBHEADER_CLASSNAME: 'pivot-table-column-subheader',
                ROW_MAIN_HEADER_CLASSNAME: 'pivot-table-row-main-header',
                ROW_SUBHEADER_CLASSNAME: 'pivot-table-row-subheader',

                /**
                 * @param   {ChartDef}  chartDef
                 * @return  {boolean}   true if at least one field have been placed as row
                 */
                hasRows(chartDef) {
                    return chartDef.yDimension.length > 0;
                },

                /**
                 * @param   {ChartDef}  chartDef
                 * @return  {boolean}   true if at least one field have been placed as column
                 */
                hasColumns(chartDef) {
                    return chartDef.xDimension.length > 0;
                },

                /**
                 * @param   {ChartDef}  chartDef
                 * @return  {boolean}   true if the pivot table was a colored pivot table (from before DSS 11)
                 */
                isColoredTable(chartDef) {
                    return chartDef.variant == CHART_VARIANTS.colored;
                },

                /**
                 * Uses canvas.measureText to measure Pivot table text width needed by cell.
                 * @param {String} text The text to be rendered.
                 * @return {number} text width
                 */
                getTextWidth(text, fontSize = PIVOT_TABLE_FONT, fontFamily = 'SourceSansPro') {
                    const width = StringUtils.getTextWidth(text, `bold ${fontSize} ${fontFamily}, SourceSansPro`, canvas);
                    return Math.min((width + PIVOT_TABLE_CELL_OFFSET + PIVOT_TABLE_SAFETY_OFFSET), COLUMN_MAX_WIDTH);
                },

                /**
                 * Checks if given coords have a cell value for any of the given measures
                 * @param {*} chartData
                 * @param {number[]} aggrIdx indexes of aggregations displayed in a cell
                 * @param {{[key: string]: number}} coords
                 * @param {boolean} displayEmptyValues
                 * @returns {boolean}
                 */
                hasCellValueForCoords(chartData, aggrIdx, coords, displayEmptyValues) {
                    return displayEmptyValues || aggrIdx.some(idx => {
                        return chartData.getNonNullCount(coords, idx) > 0;
                    });
                },

                getColumnHeaderActions(tableFormatting, cellValue, mainHeader, callbacks) {
                    const actions = [];

                    actions.push({
                        label: COPY_TO_CLIPBOARD_LABEL,
                        icon: COPY_TO_CLIPBOARD_ICON,
                        onMouseDown: () => {
                            const selectedText = (getSelection() || '').toString();
                            ClipboardUtils.copyToClipboard(selectedText === '' ? cellValue : selectedText);
                        }
                    });

                    if (DashboardUtils.isInDashboard()) {
                        return actions;
                    }

                    actions.unshift({
                        label: getHideHeaderLabel('column'),
                        icon: HIDE_HEADER_ICON,
                        onMouseDown: () => {
                            tableFormatting.showColumnHeaders = false;
                        }
                    });

                    if (mainHeader) {
                        actions.unshift(
                            {
                                label: getHideHeaderLabel('column main'),
                                icon: HIDE_HEADER_ICON,
                                onMouseDown: () => {
                                    tableFormatting.showColumnMainHeaders = false;
                                }
                            }
                        );
                    }

                    const isReadOnly = $state.current.name === 'projects.project.dashboards.insights.insight.view';

                    if (!isReadOnly) {
                        actions.unshift({
                            label: FORMAT_HEADERS_LABEL,
                            icon: FORMAT_HEADERS_ICON,
                            onMouseDown: () => {
                                callbacks.openFormatTableSection();
                            }
                        });
                    }

                    return actions;
                },

                getRowHeaderContextualMenuActions(tableFormatting, canHideHeaders, callbacks, cellValue, mainHeader) {
                    const actions = [];

                    actions.push({
                        label: COPY_TO_CLIPBOARD_LABEL,
                        icon: COPY_TO_CLIPBOARD_ICON,
                        onMouseDown: () => {
                            const selectedText = (getSelection() || '').toString();
                            ClipboardUtils.copyToClipboard(selectedText === '' ? cellValue : selectedText);
                        }
                    });

                    if (DashboardUtils.isInDashboard()) {
                        return actions;
                    }

                    const areRowHeadersFrozen = tableFormatting.freezeRowHeaders;

                    actions.unshift({
                        label: areRowHeadersFrozen ? 'Unfreeze row headers' : 'Freeze row headers',
                        icon: areRowHeadersFrozen ? 'dku-icon-filter-position-top-16' : 'dku-icon-filter-position-left-16',
                        onMouseDown: () => {
                            callbacks.toggleFreezeHeaders();
                        }
                    });

                    if (canHideHeaders) {
                        actions.unshift({
                            label: getHideHeaderLabel('row'),
                            icon: HIDE_HEADER_ICON,
                            onMouseDown: () => {
                                callbacks.toggleShowHeaders();
                            }
                        });
                    }

                    if (mainHeader) {
                        actions.unshift({
                            label: getHideHeaderLabel('row main'),
                            icon: HIDE_HEADER_ICON,
                            onMouseDown: () => {
                                callbacks.toggleShowMainHeaders();
                            }
                        });
                    }

                    const isReadOnly = $state.current.name === 'projects.project.dashboards.insights.insight.view';

                    if (!isReadOnly) {
                        actions.unshift({
                            label: FORMAT_HEADERS_LABEL,
                            icon: FORMAT_HEADERS_ICON,
                            onMouseDown: () => {
                                callbacks.openFormatTableSection();
                            }
                        });
                    }

                    return actions;
                },

                getToggleRowHeadersCallbacks(chartDef, openSection) {
                    return {
                        toggleShowHeaders: () => {
                            chartDef.pivotTableOptions.tableFormatting.showRowHeaders = !chartDef.pivotTableOptions.tableFormatting.showRowHeaders;
                        },
                        toggleShowMainHeaders: () => {
                            chartDef.pivotTableOptions.tableFormatting.showRowMainHeaders = !chartDef.pivotTableOptions.tableFormatting.showRowMainHeaders;
                        },
                        toggleFreezeHeaders: () => {
                            chartDef.pivotTableOptions.tableFormatting.freezeRowHeaders = !chartDef.pivotTableOptions.tableFormatting.freezeRowHeaders;
                        },
                        openFormatTableSection: () => {
                            openSection(ChartFormattingPaneSections.TABLE);
                        }
                    };
                }
            };
        });
})();
