(function() {
    'use strict';

    describe('TimeseriesForecastingCustomTrainTestFoldsUtils service', function() {
        let TimeseriesForecastingCustomTrainTestFoldsUtils;
        beforeEach(module('pascalprecht.translate'));
        beforeEach(module('dataiku.filters'));
        beforeEach(module('dataiku.ml.report'));
        beforeEach(module('dataiku.analysis.mlcore'));

        beforeEach(inject(function(_TimeseriesForecastingCustomTrainTestFoldsUtils_) {
            TimeseriesForecastingCustomTrainTestFoldsUtils = _TimeseriesForecastingCustomTrainTestFoldsUtils_;
        }));

        describe('validateCustomTrainTestFold', function() {
            it('should return undefined if the fold is valid', function() {
                const timestepParams = {
                    timeunit: "DAY",
                    endOfWeekDay: 1,
                    unitAlignment: 1
                };
                const predictionLength = 10;
                const interval = {
                    train: [new Date('2020-02-01T00:00:00.000'), new Date('2020-05-01T00:00:00.000')],
                    test: [new Date('2020-05-02T00:00:00.000'), new Date('2020-05-12T00:00:00.000')]
                }
                expect(TimeseriesForecastingCustomTrainTestFoldsUtils.validateCustomTrainTestFold(timestepParams, interval, predictionLength)).toBe(undefined);
            });
            it('should return an error message when the train start is after train end', function() {
                const timestepParams = {
                    timeunit: "DAY",
                    endOfWeekDay: 1,
                    unitAlignment: 1
                };
                const testSize = 10;
                const predictionLength = 5;
                const interval = {
                    train: [new Date('2020-02-02T00:00:00.000'), new Date('2020-02-01T00:00:00.000')],
                    test: [new Date('2020-03-04T00:00:00.000'), new Date('2020-03-15T00:00:00.000')]
                }
                expect(TimeseriesForecastingCustomTrainTestFoldsUtils.validateCustomTrainTestFold(timestepParams, interval, predictionLength)).toBe("Train interval start date must be before the end date");
            });
            it('should return an error message when the test start is after test end', function() {
                const timestepParams = {
                    timeunit: "DAY",
                    endOfWeekDay: 1,
                    unitAlignment: 1
                };
                const predictionLength = 10;
                const interval = {
                    train: [new Date('2020-02-01T00:00:00.000'), new Date('2020-05-01T00:00:00.000')],
                    test: [new Date('2020-05-13T00:00:00.000'), new Date('2020-05-12T00:00:00.000')]
                }
                expect(TimeseriesForecastingCustomTrainTestFoldsUtils.validateCustomTrainTestFold(timestepParams, interval, predictionLength)).toBe("Test interval start date must be before the end date");
            });
            it('should return an error message when the test start is before train end', function() {
                const timestepParams = {
                    timeunit: "DAY",
                    endOfWeekDay: 1,
                    unitAlignment: 1
                };
                const predictionLength = 10;
                const interval = {
                    train: [new Date('2020-02-01T00:00:00.000'), new Date('2020-05-03T00:00:00.000')],
                    test: [new Date('2020-05-02T00:00:00.000'), new Date('2020-05-12T00:00:00.000')]
                }
                expect(TimeseriesForecastingCustomTrainTestFoldsUtils.validateCustomTrainTestFold(timestepParams, interval, predictionLength)).toBe("Test interval start date must be after the end of the train interval");
            });
            it('should return an error message when the business day is not aligned', function() {
                const timestepParams = {
                    timeunit: "BUSINESS_DAY",
                    endOfWeekDay: 1,
                    unitAlignment: 1
                };
                const predictionLength = 10;
                let interval = {
                    train: [new Date('2020-02-01T00:00:00.000'), new Date('2020-05-01T00:00:00.000')],
                    test: [new Date('2020-05-02T00:00:00.000'), new Date('2020-05-12T00:00:00.000')]
                }
                expect(TimeseriesForecastingCustomTrainTestFoldsUtils.validateCustomTrainTestFold(timestepParams, interval, predictionLength)).toBe("Train start is not a business day.");
                interval = {
                    train: [new Date('2020-02-03T00:00:00.000'), new Date('2020-05-02T00:00:00.000')],
                    test: [new Date('2020-05-03T00:00:00.000'), new Date('2020-05-13T00:00:00.000')]
                }
                expect(TimeseriesForecastingCustomTrainTestFoldsUtils.validateCustomTrainTestFold(timestepParams, interval, predictionLength)).toBe("Train end is not a business day.");
                interval = {
                    train: [new Date('2020-02-03T00:00:00.000'), new Date('2020-05-01T00:00:00.000')],
                    test: [new Date('2020-05-03T00:00:00.000'), new Date('2020-05-16T00:00:00.000')]
                }
                expect(TimeseriesForecastingCustomTrainTestFoldsUtils.validateCustomTrainTestFold(timestepParams, interval, predictionLength)).toBe("Test start is not a business day.");
                interval = {
                    train: [new Date('2020-02-03T00:00:00.000'), new Date('2020-05-01T00:00:00.000')],
                    test: [new Date('2020-05-04T00:00:00.000'), new Date('2020-05-16T00:00:00.000')]
                }
                expect(TimeseriesForecastingCustomTrainTestFoldsUtils.validateCustomTrainTestFold(timestepParams, interval, predictionLength)).toBe("Test end is not a business day.");
                interval = {
                    train: [new Date('2020-02-03T00:00:00.000'), new Date('2020-05-01T00:00:00.000')],
                    test: [new Date('2020-05-04T00:00:00.000'), new Date('2020-05-18T00:00:00.000')]
                }
                expect(TimeseriesForecastingCustomTrainTestFoldsUtils.validateCustomTrainTestFold(timestepParams, interval, predictionLength)).toBe(undefined);
            });

            it('should return an error message when the weekday is not aligned', function() {
                const timestepParams = {
                    timeunit: "WEEK",
                    endOfWeekDay: 4,
                    unitAlignment: 1
                };
                const predictionLength = 10;
                let interval = {
                    train: [new Date('2020-02-01T00:00:00.000'), new Date('2020-05-01T00:00:00.000')],
                    test: [new Date('2020-05-02T00:00:00.000'), new Date('2020-05-12T00:00:00.000')]
                }
                expect(TimeseriesForecastingCustomTrainTestFoldsUtils.validateCustomTrainTestFold(timestepParams, interval, predictionLength)).toBe("Train start is not a Wednesday.");
                interval = {
                    train: [new Date('2020-02-05T00:00:00.000'), new Date('2020-05-01T00:00:00.000')],
                    test: [new Date('2020-05-03T00:00:00.000'), new Date('2020-05-13T00:00:00.000')]
                }
                expect(TimeseriesForecastingCustomTrainTestFoldsUtils.validateCustomTrainTestFold(timestepParams, interval, predictionLength)).toBe("Train end is not a Wednesday.");
                interval = {
                    train: [new Date('2020-02-05T00:00:00.000'), new Date('2020-05-06T00:00:00.000')],
                    test: [new Date('2020-05-12T00:00:00.000'), new Date('2020-09-17T00:00:00.000')]
                }
                expect(TimeseriesForecastingCustomTrainTestFoldsUtils.validateCustomTrainTestFold(timestepParams, interval, predictionLength)).toBe("Test start is not a Wednesday.");
                interval = {
                    train: [new Date('2020-02-05T00:00:00.000'), new Date('2020-05-06T00:00:00.000')],
                    test: [new Date('2020-05-13T00:00:00.000'), new Date('2020-09-15T00:00:00.000')]
                }
                expect(TimeseriesForecastingCustomTrainTestFoldsUtils.validateCustomTrainTestFold(timestepParams, interval, predictionLength)).toBe("Test end is not a Wednesday.");
                interval = {
                    train: [new Date('2020-02-05T00:00:00.000'), new Date('2020-05-06T00:00:00.000')],
                    test: [new Date('2020-05-13T00:00:00.000'), new Date('2020-09-16T00:00:00.000')]
                }
                expect(TimeseriesForecastingCustomTrainTestFoldsUtils.validateCustomTrainTestFold(timestepParams, interval, predictionLength)).toBe(undefined);
            });

            it('should return an error message when the quarter is not aligned', function() {
                const timestepParams = {
                    timeunit: "QUARTER",
                    endOfWeekDay: 4,
                    unitAlignment: 2
                };
                const predictionLength = 3;
                let interval = {
                    train: [new Date('2018-01-01T00:00:00.000'), new Date('2020-05-01T00:00:00.000')],
                    test: [new Date('2020-05-02T00:00:00.000'), new Date('2020-05-12T00:00:00.000')]
                }
                expect(TimeseriesForecastingCustomTrainTestFoldsUtils.validateCustomTrainTestFold(timestepParams, interval, predictionLength)).toBe("Train start is not one of Feb, May, Aug, Nov.");
                interval = {
                    train: [new Date('2018-02-01T00:00:00.000'), new Date('2020-06-01T00:00:00.000')],
                    test: [new Date('2020-07-03T00:00:00.000'), new Date('2022-05-13T00:00:00.000')]
                }
                expect(TimeseriesForecastingCustomTrainTestFoldsUtils.validateCustomTrainTestFold(timestepParams, interval, predictionLength)).toBe("Train end is not one of Feb, May, Aug, Nov.");
                interval = {
                    train: [new Date('2018-02-01T00:00:00.000'), new Date('2020-05-01T00:00:00.000')],
                    test: [new Date('2020-07-03T00:00:00.000'), new Date('2022-05-13T00:00:00.000')]
                }
                expect(TimeseriesForecastingCustomTrainTestFoldsUtils.validateCustomTrainTestFold(timestepParams, interval, predictionLength)).toBe("Test start is not one of Feb, May, Aug, Nov.");
                interval = {
                    train: [new Date('2018-02-01T00:00:00.000'), new Date('2020-05-01T00:00:00.000')],
                    test: [new Date('2020-08-03T00:00:00.000'), new Date('2022-04-13T00:00:00.000')]
                }
                expect(TimeseriesForecastingCustomTrainTestFoldsUtils.validateCustomTrainTestFold(timestepParams, interval, predictionLength)).toBe("Test end is not one of Feb, May, Aug, Nov.");
                interval = {
                    train: [new Date('2018-02-01T00:00:00.000'), new Date('2020-05-01T00:00:00.000')],
                    test: [new Date('2020-08-03T00:00:00.000'), new Date('2022-08-13T00:00:00.000')]
                }
                expect(TimeseriesForecastingCustomTrainTestFoldsUtils.validateCustomTrainTestFold(timestepParams, interval, predictionLength)).toBe(undefined);
            });

            it('should return an error message when the half year is not aligned', function() {
                const timestepParams = {
                    timeunit: "HALF_YEAR",
                    endOfWeekDay: 4,
                    unitAlignment: 4
                };
                const predictionLength = 3;
                let interval = {
                    train: [new Date('2016-01-01T00:00:00.000'), new Date('2020-05-01T00:00:00.000')],
                    test: [new Date('2020-05-02T00:00:00.000'), new Date('2020-05-12T00:00:00.000')]
                }
                expect(TimeseriesForecastingCustomTrainTestFoldsUtils.validateCustomTrainTestFold(timestepParams, interval, predictionLength)).toBe("Train start is not one of April and October.");
                interval = {
                    train: [new Date('2016-04-01T00:00:00.000'), new Date('2020-06-01T00:00:00.000')],
                    test: [new Date('2020-07-03T00:00:00.000'), new Date('2022-05-13T00:00:00.000')]
                }
                expect(TimeseriesForecastingCustomTrainTestFoldsUtils.validateCustomTrainTestFold(timestepParams, interval, predictionLength)).toBe("Train end is not one of April and October.");
                interval = {
                    train: [new Date('2016-04-01T00:00:00.000'), new Date('2020-10-01T00:00:00.000')],
                    test: [new Date('2021-03-03T00:00:00.000'), new Date('2022-05-13T00:00:00.000')]
                }
                expect(TimeseriesForecastingCustomTrainTestFoldsUtils.validateCustomTrainTestFold(timestepParams, interval, predictionLength)).toBe("Test start is not one of April and October.");
                interval = {
                    train: [new Date('2016-04-01T00:00:00.000'), new Date('2020-10-01T00:00:00.000')],
                    test: [new Date('2021-04-03T00:00:00.000'), new Date('2022-05-13T00:00:00.000')]
                }
                expect(TimeseriesForecastingCustomTrainTestFoldsUtils.validateCustomTrainTestFold(timestepParams, interval, predictionLength)).toBe("Test end is not one of April and October.");
                interval = {
                    train: [new Date('2016-04-01T00:00:00.000'), new Date('2020-10-01T00:00:00.000')],
                    test: [new Date('2021-04-03T00:00:00.000'), new Date('2022-10-13T00:00:00.000')]
                }
                expect(TimeseriesForecastingCustomTrainTestFoldsUtils.validateCustomTrainTestFold(timestepParams, interval, predictionLength)).toBe(undefined);
            });

            it('should return an error message if the test set is too small', function() {
                const timestepParams = {
                    timeunit: "HALF_YEAR",
                    endOfWeekDay: 4,
                    unitAlignment: 4
                };
                const predictionLength = 3;
                const interval = {
                    train: [new Date('2016-04-01T00:00:00.000'), new Date('2020-10-01T00:00:00.000')],
                    test: [new Date('2021-04-03T00:00:00.000'), new Date('2022-04-13T00:00:00.000')]
                }
                expect(TimeseriesForecastingCustomTrainTestFoldsUtils.validateCustomTrainTestFold(timestepParams, interval, predictionLength)).toBe("Test interval is too small (Minimum of 3 half years)");
            });

            
            
        });
    });
})();