/* global describe, it, inject, expect, module, beforeEach */

describe('chart type change handler', () => {

    beforeEach(() => {
        module('dataiku.charts');
		module('dataiku.directives.simple_report');
        module('dataiku.mock');
		module(function ($provide) { // Get rid of dependencies not needed by tests
			$provide.value('WebAppsService', {});
			$provide.value('PluginConfigUtils', {});
			$provide.value('Assert', {});
            $provide.value('ChartDataUtils', {});
            $provide.value('ChartFilters', {});
            $provide.value('RegressionTypes', {});
			$provide.value('AxisTicksConfigMode', {});
            $provide.value('ValuesInChartOverlappingStrategy', {});
            $provide.value('ValuesInChartPlacementMode', {});
			$provide.value('PolygonSources', {});
			$provide.value('LineStyleTypes', {});
			$provide.value('LinkColorVariants', {});
			$provide.value('AxisTicksFormatting', {});
			$provide.value('AxisTitleFormatting', {});
			$provide.value('DSSVisualizationThemeUtils', {});
			$provide.value('ChartLabels', { CHART_ERROR_MESSAGES: {}});
		});
    });

	function makeEmpty() {
		return {
			genericDimension0: [],
			genericDimension1: [],
			xDimension: [],
			yDimension: [],
			groupDimension: [],
			genericMeasures: [],
			uaSize: [],
			uaColor: [],
			uaShape: [],
			sizeMeasure: [],
			colorMeasure: [],
			tooltipMeasures: []
		};
	}

	function make1D2MStackedColumns() {
		const chartSpec = makeEmpty();

		chartSpec.genericDimension0 = [{ column: 'dcol0', type: 'ALPHANUM', isA: 'dimension' }];
		chartSpec.genericMeasures = [
			{ column: 'dmes0', type: 'ALPHANUM', 'function': 'AVG'},
			{ column: 'dmes1', type: 'NUMERICAL', 'function': 'AVG'},
			{ column: 'dmes2', type: 'NUMERICAL', 'function': 'AVG'}
		];
		chartSpec.uaTooltip = [];
		chartSpec.boxplotValue = [];
		chartSpec.uaXDimension = [];
		chartSpec.uaYDimension = [];
		chartSpec.animationDimension = [];
		chartSpec.facetDimension = [];

		return chartSpec;
	}

	function makeAlphanumResultChart() {
		const chartSpec = makeEmpty();

		chartSpec.genericDimension0 = [{ column: 'dcol0', type: 'ALPHANUM', isA: 'dimension' }];
		chartSpec.genericMeasures = [
			{ column: 'dmes5', type: 'NUMERICAL', 'function': 'AVG'},
			{ column: 'dmes6', type: 'CUSTOM', 'function': 'CUSTOM', 'inferredType': 'NUMERICAL'},
			{ column: 'dmes6', type: 'CUSTOM', 'function': 'CUSTOM', 'inferredType': 'ALPHANUM'}
		];
		return chartSpec;
	}

	function deleteUnrelevantProperties(chartSpec) {
		chartSpec.genericDimension0 && chartSpec.genericDimension0[0] && delete chartSpec.genericDimension0[0].chartDefAttributeName;
		chartSpec.uaXDimension && chartSpec.uaXDimension[0] && delete chartSpec.uaXDimension[0].chartDefAttributeName;
		chartSpec.uaYDimension && chartSpec.uaYDimension[0] && delete chartSpec.uaYDimension[0].chartDefAttributeName;
		chartSpec.xDimension && chartSpec.xDimension[0] && delete chartSpec.xDimension[0].chartDefAttributeName;
		
		if (chartSpec.genericMeasures) {
			chartSpec.genericMeasures[0] && delete chartSpec.genericMeasures[0].chartDefAttributeName;
			chartSpec.genericMeasures[1] && delete chartSpec.genericMeasures[1].chartDefAttributeName;
			chartSpec.genericMeasures[2] && delete chartSpec.genericMeasures[2].chartDefAttributeName;
		}

		chartSpec.uaColor && chartSpec.uaColor[0] && delete chartSpec.uaColor[0].chartDefAttributeName;
		chartSpec.uaSize && chartSpec.uaSize[0] && delete chartSpec.uaSize[0].chartDefAttributeName;
		
	}

	it('can change std to std', function() {
		inject(function(ChartTypeChangeHandler) {
			const chartSpec = make1D2MStackedColumns();
			const originalChartSpec = angular.copy(chartSpec);
		
			ChartTypeChangeHandler.onChartTypeChange(chartSpec, 'stacked_area', 'STANDARD');
			deleteUnrelevantProperties(chartSpec);

			expect(chartSpec.genericDimension0).toEqual(originalChartSpec.genericDimension0);
			expect(chartSpec.genericMeasures).toEqual(originalChartSpec.genericMeasures);
		});
	});

	it('can change std to scatter', function() {
		inject(function(ChartTypeChangeHandler) {
			const chartSpec = make1D2MStackedColumns();
			const originalChartSpec = angular.copy(chartSpec);

			ChartTypeChangeHandler.onChartTypeChange(chartSpec, 'scatter', 'STANDARD');
			deleteUnrelevantProperties(chartSpec);

			expect(chartSpec.uaXDimension).toEqual(originalChartSpec.genericDimension0);
			expect(chartSpec.uaYDimension).toEqual([]);
		});
	});

	it('can change weird mixed to scatter', function() {
		inject(function(ChartTypeChangeHandler) {
			const chartSpec = make1D2MStackedColumns();
			chartSpec.xDimension = [{ column: 'xcol0', type: 'ALPHANUM' }];
			const originalChartSpec = angular.copy(chartSpec);

			ChartTypeChangeHandler.onChartTypeChange(chartSpec, 'scatter', 'STANDARD');
			deleteUnrelevantProperties(chartSpec);

			expect(chartSpec.uaXDimension).toEqual(originalChartSpec.genericDimension0);
			expect(chartSpec.uaYDimension).toEqual(originalChartSpec.xDimension);
		});
	});

	it('only assign accepted types for scatter ua dimensions', function() {
		inject(function(ChartTypeChangeHandler) {
			const chartSpec = make1D2MStackedColumns();
			const originalChartSpec = angular.copy(chartSpec);
			ChartTypeChangeHandler.onChartTypeChange(chartSpec, 'scatter', 'STANDARD');
			deleteUnrelevantProperties(chartSpec);

			// first accepted size dimension is correctly assigned to uaSize first
			expect(chartSpec.uaSize[0].column).toEqual(originalChartSpec.genericMeasures[1].column);
			// then to uaColor (whatever the type)
			expect(chartSpec.uaColor[0].column).toEqual(originalChartSpec.genericMeasures[2].column);
		});
	});

	it('can remove alphanum custom measure if needed', function() {
		inject(function(ChartTypeChangeHandler) {
			const chartSpec = makeAlphanumResultChart();
			const originalChartSpec = angular.copy(chartSpec);

			ChartTypeChangeHandler.onChartTypeChange(chartSpec, 'stacked_area', 'STANDARD');
			deleteUnrelevantProperties(chartSpec);

			expect(chartSpec.genericDimension0).toEqual(originalChartSpec.genericDimension0);
			expect(chartSpec.genericMeasures).toEqual([
				{ column: 'dmes5', type: 'NUMERICAL', 'function': 'AVG'},
				{ column: 'dmes6', type: 'CUSTOM', 'function': 'CUSTOM', 'inferredType': 'NUMERICAL'}
			]);
		});
	});

});
