import json
import logging
import os
import sys
import shutil
import tempfile
import tarfile

from dataiku.base.utils import tar_extractall
from dataiku.core.intercom import backend_api_post_call
from dataiku.core.intercom import has_a_jek
from dataiku.core.intercom import jek_api_post_call

logger = logging.getLogger(__name__)

TAR_BUFFER_SIZE = 10 * 16 * 1024


def fetch_from_backend(resources_file_paths):
    """
    (Advanced) Fetch specific resources files from backend if not already in current resources
    directory. Intended for containerized execution, no-op when running in backend.
    :param list resources_file_paths: List of (relative) resources file paths to fetch.
    :return dict: Dictionary where keys are the file paths to fetch and values are True/False
                  whether the files are available (fetched or already up-to-date) or not.
    """
    if not isinstance(resources_file_paths, (list, tuple, set)):
        raise TypeError("resources_file_paths should be a list, tuple or set")

    # Make sure we have no duplicate paths
    resources_file_paths = list(set(resources_file_paths))

    # Initialize result
    fetch_result = {file_path: False for file_path in resources_file_paths}

    if os.environ.get("DKU_CONTAINER_EXEC"):
        # Get execution_id for API call
        with open("/home/dataiku/execution.json", "r") as f:
            execution = json.load(f)
            execution_id = execution["id"]

        env_folder = os.path.realpath(sys.prefix)
        env_name = None
        if os.path.isfile(env_folder + "/dku-codeenv-ref.json"):
            with open(env_folder + "/dku-codeenv-ref.json", "r") as e:
                env_ref = json.load(e)
                env_name = env_ref.get("name", None)
            available_envs = execution.get("codeEnvs", [])
            if not any(env_name == e.get("name", None) for e in available_envs):
                raise ValueError("code env '{}' not available in execution '{}'".format(env_name, execution_id))
        resources_folder = env_folder + "/resources"

        for file_path in resources_file_paths:
            target_path = os.path.realpath(os.path.join(resources_folder, file_path))
            if os.path.commonprefix([target_path, resources_folder]) != resources_folder:
                raise ValueError("'{}' is not within '{}'".format(target_path, resources_folder))  # DT: ok
            parent_directory = os.path.abspath(os.path.join(target_path, os.pardir))
            if not os.path.exists(parent_directory):
                os.makedirs(parent_directory)

            if os.path.isdir(target_path):
                shutil.rmtree(target_path)
            elif os.path.isfile(target_path):
                os.remove(target_path)

            fetch_result[file_path] = _fetch_file_or_dir(file_path, target_path, execution_id, env_name)

    return fetch_result


def _fetch_file_or_dir(path, target, execution_id, env_name):
    resp = _api_stream_call('containers/get-file', data={"executionId": execution_id, "fileKind": "CODE_ENV_RESOURCE", "path": path, "codeEnvName": env_name, "compress": False})

    if resp.status_code == 200:
        if resp.headers.get("Content-Type", "") == "application/x-tar":
            # Directory
            target = os.path.abspath(target)
            parent_dir = os.path.dirname(target)
            # existence of 'parent_dir' was already ensured by the caller
            tmp_dir = tempfile.mkdtemp(dir=parent_dir)

            with tarfile.open(fileobj=resp.raw, mode='r|', bufsize=TAR_BUFFER_SIZE) as tar:
                tar_extractall(tar, tmp_dir)

            # non-existence of the 'target' was already ensured by the caller
            os.renames(tmp_dir, target)
        else:
            # File
            with open('_dku_fetch_file', 'wb') as f:
                for chunk in resp.iter_content(chunk_size=4096):
                    f.write(chunk)
            shutil.move('_dku_fetch_file', target)
        return True
    elif resp.status_code != 404:
        logger.warning("Error fetching %s, HTTP status %d: %s", path, resp.status_code, resp.text)
        return False
    else:
        logger.warning("Could not fetch %s, not found", path)
        return False


def _api_stream_call(path, data=None, **kwargs):
    if has_a_jek():
        return jek_api_post_call(path, data, stream=True, **kwargs)
    else:
        return backend_api_post_call(path, data, stream=True, **kwargs)
