#! /bin/sh -e

MYDIR=`dirname $0`
MYDIR=`cd $MYDIR && pwd -P`

cd $MYDIR

CONTEXT_DIR="$1"
CODE_ENV_IMAGE_NAME="$2"
PACKAGE_IMAGE_NAME="$3"
PACKAGE_IMAGE_TAG="$4"
BASE_IMAGE_NAME="$5" # might already include registry and tag
FORCE_REBUILD="$6"
IS_DATAIKU_CLOUD="$7"
USER_REGISTRY="$8"

if [ "$IS_DATAIKU_CLOUD" = "yes" ]; then
  echo "Building image for Dataiku Cloud, will push to $USER_REGISTRY"
  CODE_ENV_IMAGE_NAME=$USER_REGISTRY/$CODE_ENV_IMAGE_NAME
  PACKAGE_IMAGE_NAME=$USER_REGISTRY/$PACKAGE_IMAGE_NAME
fi

echo "Content of context dir: $CONTEXT_DIR"
(cd $CONTEXT_DIR && find  . | grep -v ".git")

CODE_ENV_IMAGE_TAG=`cd $CONTEXT_DIR && find ./tmp/codeenvs -type f -exec md5sum {} \; | sort -k 2 | md5sum | awk '{print $1}'`

BASE_HASH_TAG_NAME="com.dataiku.api-deployer.base-image-hash"

if [ "$IS_DATAIKU_CLOUD" = "yes" ]; then
    CURRENT_BASE_HASH=$(crane manifest $BASE_IMAGE_NAME | jq -r '.config.digest // ""')
else
    CURRENT_BASE_HASH=$(docker inspect --format='{{.Id}}' $BASE_IMAGE_NAME)
fi

if [ "$FORCE_REBUILD" = "yes" ]; then
    if [ ! "$IS_DATAIKU_CLOUD" = "yes" ]; then # On dataiku cloud, don't delete the images: we don't manage our clients' repos
        echo "Removing docker images for code envs and API service"
        docker image rm -f $CODE_ENV_IMAGE_NAME:$CODE_ENV_IMAGE_TAG
        docker image rm -f $PACKAGE_IMAGE_NAME:$PACKAGE_IMAGE_TAG # that's... dumb ? Because $PACKAGE_IMAGE_TAG includes the date/time ?
    fi
    BUILD_CODENV=1
else
    if [ "$IS_DATAIKU_CLOUD" = "yes" ]; then
        set +e # depending on the cloud vendor/registry, this will fail if repo or image does not exist
        IMG_INFO=$(crane ls $CODE_ENV_IMAGE_NAME | grep $CODE_ENV_IMAGE_TAG)
        set -e
    else
        IMG_INFO=$(docker images -q $CODE_ENV_IMAGE_NAME:$CODE_ENV_IMAGE_TAG 2> /dev/null)
    fi

    if [ -z "$IMG_INFO" ]; then
        echo "Code env image does not exist, it will be built."
        BUILD_CODENV=1
    else
        if [ "$IS_DATAIKU_CLOUD" = "yes" ]; then
            RECORDED_BASE_HASH=$(crane config $CODE_ENV_IMAGE_NAME:$CODE_ENV_IMAGE_TAG | jq -r ".config.Labels.\"$BASE_HASH_TAG_NAME\" // \"\"")
        else
            RECORDED_BASE_HASH=$(docker inspect --format "{{ index .Config.Labels \"$BASE_HASH_TAG_NAME\" }}" $CODE_ENV_IMAGE_NAME:$CODE_ENV_IMAGE_TAG)
        fi

        if [ "$RECORDED_BASE_HASH" != "$CURRENT_BASE_HASH" ]; then
            # code env image was built from a different base image, let's rebuild it
            echo "Existing code env image was built from a base image with the same name and tag, but having a different hash. Forcing rebuild."
            BUILD_CODENV=1
        else
            echo "Configured base image has the same hash as the one used to build the code env image. Not forcing rebuild of code env image."
        fi
    fi
fi

if [ ! -z "$BUILD_CODENV" ]; then
    echo "Building code env image ctx=$CONTEXT_DIR CODE_ENV_IMAGE_NAME=$CODE_ENV_IMAGE_NAME CODE_ENV_IMAGE_TAG=$CODE_ENV_IMAGE_TAG BASE_IMAGE_NAME=$BASE_IMAGE_NAME"
    echo "FROM $BASE_IMAGE_NAME" > "$CONTEXT_DIR/Dockerfile"
    echo "LABEL $BASE_HASH_TAG_NAME=$CURRENT_BASE_HASH" >> "$CONTEXT_DIR/Dockerfile"
    echo "USER root" >> "$CONTEXT_DIR/Dockerfile"
    echo "WORKDIR /opt/dataiku" >> "$CONTEXT_DIR/Dockerfile"

    cat "$CONTEXT_DIR/Docker.code_envs_build" >> "$CONTEXT_DIR/Dockerfile"
    sed -e "s/__DKU_UID__/${DKU_UID:-500}/" Dockerfile-fragment-code-env >> "$CONTEXT_DIR/Dockerfile"
    echo "USER ${DKU_UID:-500}" >> "$CONTEXT_DIR/Dockerfile"
    echo "Will build docker image with code environments with Docker file:"
    echo "--- Beginning of Dockerfile ---"
    cat "$CONTEXT_DIR/Dockerfile"
    echo "--- End of Dockerfile ---"
    set -x
    docker build -t "$CODE_ENV_IMAGE_NAME:$CODE_ENV_IMAGE_TAG" "$CONTEXT_DIR"
    set +x
else
    echo "Using base code env image $CODE_ENV_IMAGE_NAME:$CODE_ENV_IMAGE_TAG."
fi

echo "Building API service image ctx=$CONTEXT_DIR PACKAGE_IMAGE_NAME=$PACKAGE_IMAGE_NAME PACKAGE_IMAGE_TAG=$PACKAGE_IMAGE_TAG CODE_ENV_IMAGE_NAME=$CODE_ENV_IMAGE_NAME CODE_ENV_IMAGE_TAG=$CODE_ENV_IMAGE_TAG"

cp docker-entrypoint.sh "$CONTEXT_DIR"

echo "FROM $CODE_ENV_IMAGE_NAME:$CODE_ENV_IMAGE_TAG" > "$CONTEXT_DIR/Dockerfile"
cat "$CONTEXT_DIR/Docker.additional_init_deployment_type" >> "$CONTEXT_DIR/Dockerfile"
cat Dockerfile-fragment-model >> "$CONTEXT_DIR/Dockerfile"
cat "$CONTEXT_DIR/Docker.additional_finalize_deployment_type" >> "$CONTEXT_DIR/Dockerfile"

echo "Will build docker image with model with Docker file:"
echo "--- Beginning of Dockerfile ---"
cat "$CONTEXT_DIR/Dockerfile"
echo "--- End of Dockerfile ---"
set -x
docker build -t "$PACKAGE_IMAGE_NAME:$PACKAGE_IMAGE_TAG" "$CONTEXT_DIR"
set +x
