/*
 * Usage
 * node export.js magicHeadlessAuth scriptConfigFilePaht
 * sandboxing outputDirectory exportPageWidth exportPageHeight steps...
 *
 * Arguments
 * - magicHeadlessAuth (String): API Key generated by DSS that allows authentification.
 * - scriptConfigFilePath (String): Filesystem path to retrieve the configuration file.
 *
 * The config file contains a json with:
 * - browserSandBoxing (String): 'default' to sandbox the browser (default), 'no-sandbox to allow non-sandboxed browser.
 * - outputDirectory (String): path to the output directory.
 * - width (Integer): will be used to set initial width of viewport.
 * - height (Integer): will be used to set initial height of viewport.
 * - puppeteerInstructions (Object): list of puppeteer placeholder to resolve
 */

'use strict';

const fs = require('fs');
const utils = require('./utils');
const log = require('./log');

const magicHeadlessAuth = process.argv[2].toString();

const scriptConfigFile = process.argv[3].toString();
log.info("Reading Dashboard export script configuration from " + scriptConfigFile);
const config = JSON.parse(fs.readFileSync(scriptConfigFile));
log.info(JSON.stringify(config));

const enforceSandboxing = config.browserSandBoxing;
const pageDefaultTimeout = config.pageDefaultTimeout;
const outputDirectory = config.outputDirectory;
const exportPageWidth = parseInt(config.width);
const exportPageHeight = parseInt(config.height);
const defaultBottomMargin = 20; // we have a margin in the bottom of some page
const deviceScaleFactor = 2; // Enable screenshots to have high ppi (retina-like resolution)
const dssErrorElementNotFound = "[DSSError: cannot find the information related to this placeholder.]";
const defaultAdditionalWaitingMs = 500;
const CSS_PLACEHOLDER_PUPPETEER = "[#@#]";  // Keep in sync with PuppeteerConfig.CSS_PLACEHOLDER_PUPPETEER

// ========================
// Entry point
// ========================
try {
    log.info("Charts export script started.");

    // Parsing the command line to build the list of elements to export
    let elementsToExtract = [];

    for (let i = 0; i < config.puppeteerInstructions.length; i += 1) {
        const puppeteerConfigName = config.puppeteerInstructions[i].puppeteerConfigName;
        const placeholderName = config.puppeteerInstructions[i].placeholderName;
        const url = config.puppeteerInstructions[i].url;
        const stepsAsArray = config.puppeteerInstructions[i].steps;

        let steps = [];
        for (let s = 0; s < stepsAsArray.length; s += 1) {
            let a = stepsAsArray[s];
            steps.push({
                type: a.type,
                cssSelector: a.cssSelector,
                waitForAnimations: a.waitForAnimations,
                loadedStateField: a.loadedStateField,
                bottomMargin: a.bottomMargin,
                hasParameter: a.hasParameter,
                parameterValue: a.parameterValue
            });
        }

        const objectToExtract = { id: puppeteerConfigName, name: placeholderName, url: url, steps: steps };
        elementsToExtract.push(objectToExtract);
    }

    let options = {};
    utils.createBrowser(enforceSandboxing, options).then(function(browser) {
        return exportElements(browser, elementsToExtract).then(function(result) {
            log.info("Closing browser");
            return browser.close();
        });
    }).catch(function(err) {
        utils.exit(utils.ERR_GENERIC, "Error while running export script", err);
    }).then(exportDone);
} catch (err) {
    utils.exit(utils.ERR_GENERIC, "Error while running export script", err);
}

function exportDone() {
    log.info("Done exporting DOM elements");
    process.exit(0);
}

/**
 * Exports the supplied elements
 *
 * @return {Promise.<Void>}
 */
function exportElements(browser, elements) {
    let result = Promise.resolve();
    for (let i = 0; i < elements.length; i++) {
        let element = elements[i];
        result = result.then(function() {
            return loadElementPage(browser, element).then(function(page) {
                incrementProgress();
                return buildAndRunSteps(page, element).catch(function() {
                    log.info("Failed to export element " + element.id);
                    return page.close().then(function() {
                        return Promise.reject();
                    });
                }).then(function() {
                    log.info("Successfully exported element " + element.id);
                    return page.close();
                });
            });
        }).catch(function() {
            // If an error occurred during the element extraction we still want to extract the other elements so we return a successful promise
            return Promise.resolve();
        });
    }
    return result;
}

/**
 * Create a new browser page and navigate to the specified element
 *
 * @return {Promise.<Page>}
 */
function loadElementPage(browser, element) {
    return utils.newBrowserPage(browser, exportPageWidth, exportPageHeight, magicHeadlessAuth, pageDefaultTimeout, deviceScaleFactor).then(function(page) {
        page.on('console', logPupeteerConsoleMessage);
        return utils.navigateTo(page, element.url).then(function() {
            log.info("Navigated to url: " + element.url);
            return page;
        });
    });
}

function deepCopySteps(steps) {
    return JSON.parse(JSON.stringify(steps));
}

/**
 * Run additional operations before running the steps
 */
function buildAndRunSteps(page, element) {
    return computeAdditionalSteps(page, element).then(function() {
        return runElementSteps(page, element);
    });
}

/**
 * Returns the first click step in the array if there is no screenshot or extract steps before the click step, else return null.
 */
function getLeadingClickStep(steps) {
    if (Array.isArray(steps)) {
        for (let i = 0; i < steps.length; i++) {
            const step = steps[i];
            if (step.type === "RESET_LOADED_STATE" || step.type === "EXECUTE_PREPARE_EXPORT_FUNCTION") {
                continue; // Ignore these kind of steps
            }
            if (step.type === "CLICK") {
                if (step.hasParameter) continue;    // Ignore parametrized clicks as they don't need to be repeated
                return step;
            } else {
                return null;
            }
        }
    }
    return null;
}
/**
 * If the first step is a click on a CSS selector which has multiple matches in the page
 * we repeat the whole sequence of steps for each of them
 */
function computeAdditionalSteps(page, element) {
    const clickStep = getLeadingClickStep(element.steps);
    if (!clickStep) {
        return Promise.resolve();
    }

    let selector = clickStep.cssSelector;
    let clickStepIndex = element.steps.indexOf(clickStep);

    return page.waitForSelector(selector, { timeout: utils.ELEMENT_CONTENT_TIMEOUT }).catch(function(err) {
        stepFailed(element.name, selector, err);
        captureError(dssErrorElementNotFound, element.id + ".0.0.txt");
        return Promise.reject();
    }).then(function() {
        return page.$$(selector).then(function(elementHandles) {
            let initialSteps = element.steps;
            let newSteps = [];

            for (let i = 0; i < elementHandles.length; i++) {
                newSteps = newSteps.concat(deepCopySteps(initialSteps));
                // When you click on something, sometime it will refresh everything
                // so we have to keep the order and request again the element when we want to click on it.
                newSteps[clickStepIndex + (i * initialSteps.length)].elementHandleNumber = i;
            }
            element.steps = newSteps;
            return Promise.resolve();
        });
    });
}

function runElementSteps(page, element) {
    log.info(`Started element extraction with puppeteer config ${element.id}`);
    return runElementStep(page, element, 0);
}

function runElementStep(page, element, stepIndex) {
    if (stepIndex >= element.steps.length) {
        return Promise.resolve();
    }

    let elementName = element.name;
    let step = element.steps[stepIndex];
    let selector = step.cssSelector;
    let stepType = step.type;
    let filename = element.id + "." + stepIndex;

    return escapeParameterValue(page, step).then(function(cleanParameterValue) {
        if (cleanParameterValue) {
            selector = selector.replace(CSS_PLACEHOLDER_PUPPETEER, cleanParameterValue);
        }
        return page.waitForSelector(selector, { timeout: utils.ELEMENT_CONTENT_TIMEOUT });
    }).then(function() {
        if (stepType === "CLICK") {
            return clickOnElement(page, step.elementHandleNumber, selector, step.waitForAnimations);
        } else if (stepType === "SCREENSHOT") {
            return captureScreenshot(page, elementName, selector, filename, step.waitForAnimations, step.loadedStateField, step.bottomMargin);
        } else if (stepType === "RESET_LOADED_STATE") {
            return resetLoadedState(page, elementName, selector, step.loadedStateField);
        } else if (stepType === "EXECUTE_PREPARE_EXPORT_FUNCTION") {
            return executePrepareExportFunction(page, elementName, selector, step.loadedStateField);
        } else if (stepType === "TEXT_EXTRACTION") {
            return captureText(page, elementName, selector, filename + ".0.txt", step.waitForAnimations, step.loadedStateField);
        } else if (stepType === "JSON_EXTRACTION") {
            return captureJson(page, elementName, selector, filename + ".0.json", step.waitForAnimations, step.loadedStateField);
        } else if (stepType === "REFRESH") {
            return refresh(page);
        }
    }).catch(function(err) {
        stepFailed(elementName, selector, err);
        captureError(dssErrorElementNotFound, filename + ".0.txt");
        return Promise.reject();
    }).then(function() {
        return runElementStep(page, element, stepIndex + 1);
    });
}

function escapeParameterValue(page, step) {
    if (step.hasParameter) {
        // Use CSS.escape from browser as it's not installed in Node.js
        return page.evaluate((val) => CSS.escape(val), step.parameterValue);
    } else {
        return Promise.resolve()
    }
}

function clickOnElement(page, elementHandleNumber, selector, waitForAnimations) {
    console.info("Click on item #" + elementHandleNumber);

    const additionalWaiting = waitForAnimations ? defaultAdditionalWaitingMs : 0;

    return page.evaluate(function() {
        // Open dropdown menus
        return document.querySelectorAll(".dropdown-menu.open").forEach(elt => elt.style.display = "block");
    }).then(function() {
        // Wait for selector to be visible
        return page.waitForSelector(selector, { hidden: false, visible: true, timeout: utils.ELEMENT_CONTENT_TIMEOUT });
    }).then(function() {
        return utils.timeout(additionalWaiting);
    }).then(function() {
        // Query selector
        return page.$$(selector);
    }).then(function(elementHandles) {
        // Click on element
        let handle;
        if (0 <= elementHandleNumber && elementHandleNumber < elementHandles.length) {
            handle = elementHandles[elementHandleNumber];
        } else if (elementHandles.length > 0) {
            handle = elementHandles[0];
        }
        if (handle) {
             // First try to click with puppeteer
            return handle.click().catch(function() {
                // Fallback to click with javascript
                return handle.evaluate(element => element.click());
            });
        }
    }).then(function() {
        return page.evaluate(function() {
            // Close dropdown menus
            return document.querySelectorAll(".dropdown-menu.open").forEach(elt => elt.style.display = "none");
        });
    }).then(function() {
        // If the selection is an option of a dropdown menu, wait for it to be hidden
        if (selector.match(/.*dropdown.* li .text$/)) {
            return page.waitForSelector(selector, { hidden: true, visible: false, timeout: utils.ELEMENT_CONTENT_TIMEOUT });
        }
    });
}

function captureScreenshot(page, elementName, cssSelector, filename, waitForAnimations, loadedStateField, requestedBottomMargin) {
    // Add timeout to wait for the end of animation present in some charts.
    const additionalWaiting = waitForAnimations ? defaultAdditionalWaitingMs : 0;
    const bottomMargin = (requestedBottomMargin === undefined || requestedBottomMargin === -1) ? defaultBottomMargin : requestedBottomMargin;

    return utils.waitForElementContentToLoad(page, cssSelector, loadedStateField).catch(function(logMessage) {
        log.info("The content of element " + cssSelector + " couldn't be loaded.");
        stepFailed(elementName, cssSelector, logMessage);
    }).then(function() {
        return utils.timeout(additionalWaiting);
    }).then(function() {
        return page.evaluate(function(cssSelector) {
            let domElement = document.querySelector(cssSelector);
            if (!domElement) {
                stepFailed(elementName, cssSelector);
                return false;
            }

            // Screenshots should have a black text on a white background as it's the most common style for documents
            if (domElement.style) {
                domElement.style.backgroundColor = "white";
                domElement.style.color = "black";
                domElement.style.animation = "none";
                domElement.style.transition = "none";
            }

            // The intercom button shouldn't pollute the screenshots
            let intercomBubble = document.querySelector(".intercom-lightweight-app");
            if (intercomBubble && intercomBubble.style) {
                intercomBubble.style.display = "none";
            }

            domElement.scrollIntoView();

            const rect = domElement.getBoundingClientRect();
            return {
                x: Number(rect.x.toFixed(0)),
                y: Number(rect.y.toFixed(0)),
                width: rect.width,
                height: rect.height
            };
        }, cssSelector).then(function(rect) {
            if (rect) {
                log.info("Taking screenshots...");
                log.info(`Image location from ${rect.y} to ${rect.y + rect.height}.`);
                const screenshotCount = Math.ceil(rect.height / (exportPageHeight - rect.y - bottomMargin));
                log.info("Taking " + screenshotCount + " screenshots of " + exportPageHeight + " pixels max.");

                function scrollAndScreenshot(screenshotIndex) {
                    // First scroll
                    const scrollValue = exportPageHeight - rect.y - bottomMargin;
                    log.info(`Scrolling and taking screenshot ${screenshotIndex} with scrollValue=${scrollValue}`);
                    return doScroll(page, cssSelector, screenshotIndex, scrollValue).then(function(actualScroll) {
                        // Then take a screenshot and save it
                        return doScreenshot(page, actualScroll, screenshotIndex, screenshotCount, rect, bottomMargin, filename)
                    });
                }

                var screenshots = [];
                var promise = scrollAndScreenshot(0);
                screenshots.push(promise);
                for (let screenshotIndex = 1; screenshotIndex < screenshotCount; screenshotIndex++) {
                    promise = promise.then(function() {
                        return scrollAndScreenshot(screenshotIndex)
                    });
                    screenshots.push(promise);
                }
                return Promise.all(screenshots);
            }
        });
    });
}


function refresh(page) {
    return page.reload({ waitUntil: ["networkidle0", "domcontentloaded"] });
}

function resetLoadedState(page, elementName, cssSelector, loadedStateField) {
    log.info("Waiting for element to reset... to be ready.");
    return utils.waitForElementContentToLoad(page, cssSelector, loadedStateField).catch(function(logMessage) {
        log.info("The content of element " + cssSelector + " couldn't be loaded.");
        stepFailed(elementName, cssSelector, logMessage);
    }).then(function() {
        return utils.resetElementContentToLoad(page, cssSelector, loadedStateField);
    });
}

function executePrepareExportFunction(page, elementName, cssSelector, loadedStateField) {
    log.info(`Preparing execution of function puppeteerPrepareForExport on selector ${cssSelector}`);
    return utils.waitForElementContentToLoad(page, cssSelector, loadedStateField).catch(function(logMessage) {
        log.info(`The content of element ${cssSelector} couldn't be loaded.`);
        stepFailed(elementName, cssSelector, logMessage);
    }).then(function() {
        return page.evaluate(function(cssSelector) {
            let domElement = document.querySelector(cssSelector);
            if (!domElement) {
                stepFailed(elementName, cssSelector);
                return false;
            }
            angular.element(domElement).scope().puppeteerPrepareForExport();
        }, cssSelector);
    });
}

function doScroll(page, cssSelector, imageIndex, scrollValue) {
    return page.evaluate(function(cssSelector, imageIndex, scrollValue) {
        if (imageIndex > 0) {
            let domElement = document.querySelector(cssSelector);
            // Scroll the current element or scroll its parent when this does not works
            // This method need to be in the scope of the evaluate
            function forceScroll(element, scrollValue) {
                if (!element) {
                    return 0;
                }

                let scroll = element.scrollTop;
                element.scrollTop += scrollValue;
                if (scroll === element.scrollTop && typeof element.parentElement !== 'undefined') {
                    // Try to be luckier on parent
                    return forceScroll(element.parentElement, scrollValue);
                } else {
                    return element.scrollTop - scroll;
                }
            }

            return forceScroll(domElement, scrollValue);
        } else { // No scroll
            return 0;
        }
    }, cssSelector, imageIndex, scrollValue)
}

function doScreenshot(page, actualScroll, screenshotIndex, screenshotCount, rect, bottomMargin, filename) {
    log.info(`doScreenshot(actualScroll=${actualScroll}, screenshotIndex=${screenshotIndex}, screenshotCount=${screenshotCount}, rect=${JSON.stringify(rect)}, filename=${filename})`);
    let screenshotY;
    let screenshotHeight;
    if (screenshotCount === 1) {
        // Single screenshot image
        screenshotY = rect.y;
        screenshotHeight = rect.height;
    } else if (screenshotIndex < (screenshotCount - 1)) {
        // Multiples screenshots image (all screenshots but the last one)
        screenshotY = rect.y;
        screenshotHeight = exportPageHeight - rect.y - bottomMargin;
    } else {
        // Multiples screenshots image (the last screenshot)
        const capturedHeight = exportPageHeight - rect.y - bottomMargin;
        log.info(` >> Final screenshot, capturedHeight=${capturedHeight}`);
        screenshotY = rect.y + capturedHeight - actualScroll;
        screenshotHeight = rect.height % capturedHeight;
    }
    log.info(` >> screenshotY=${screenshotY}, screenshotHeight=${screenshotHeight}`);

    // If the width or the height of the screenshot is 0, there is no need to take a screenshot
    if (rect.width === 0 || screenshotHeight === 0) {
        log.info("No need to take a screenshot");
        return Promise.resolve();
    }

    const path = outputDirectory + "/" + filename + "." + screenshotIndex + ".png";

    log.info(`Saving screenshot in ${path}`);
    return page.screenshot({
        path: path,
        type: "png",
        clip: {
            x: rect.x,
            y: screenshotY,
            width: rect.width,
            height: screenshotHeight
        }
    });
}

function captureText(page, elementName, cssSelector, filename, waitForAnimations, loadedStateField) {
    // Add timeout to wait for the end of animation present in some pages.
    const additionalWaiting = waitForAnimations ? defaultAdditionalWaitingMs : 0;

    return utils.waitForElementContentToLoad(page, cssSelector, loadedStateField).catch(function(logMessage) {
        log.info("The content of element " + cssSelector + " couldn't be loaded.");
        stepFailed(elementName, cssSelector, logMessage);
    }).then(function() {
        return utils.timeout(additionalWaiting);
    }).then(function() {
        return page.evaluate(function(elementName, cssSelector, stepFailed) {

            let domElement, getValue;
            // For corner cases
            if ((elementName === "design.other_algorithms_search_strategy.table" || elementName === "design.other_algorithms_search_strategy.image")
                    && cssSelector === "[mdg__design-algorithms__algorithm-title]") {
                domElement = document.querySelector(cssSelector + " input");
                getValue = true;
            }

            // For normal cases
            if (!domElement) {
                domElement = document.querySelector(cssSelector);
                getValue = false;
            }

            if (!domElement) {
                stepFailed(elementName, cssSelector);
                return ""; // Rather than displaying an ugly error in the output template, let's not show anything
            }

            // Add an empty line before each <h1> to better seperate text sections in the final output
            const BEFORE_H1_PLACEHOLDER = "[#@#]";
            domElement.querySelectorAll("h1").forEach((titleNode) => {
                let emptyNode = document.createElement("span");
                emptyNode.innerHTML = BEFORE_H1_PLACEHOLDER;
                let parentNode = titleNode.parentNode;
                parentNode.insertBefore(emptyNode, titleNode);
            });

            // HTML lists aren't printed with the innerText method so we need to create lists manually
            domElement.querySelectorAll("li").forEach((element) => {
                element.innerText = "- " + element.innerText;
            });

            let extractedText;
            // For corner cases
            if (getValue) {
                extractedText = domElement.value;
            }
            // For normal cases
            else {
                // We don't want ugly spaces at the beginning and the end of the string
                extractedText = domElement.innerText.split(/\r?\n/).map(x => x.trim()).join('\n');
            }
            return extractedText;

        }, elementName, cssSelector, stepFailed).then(function(innerText) {
            log.info("Saving template text...");
            writeFile(filename, innerText);
        });
    });
}

/**
 * Method that will retrieve complex elements from the UI as a table (formatted inside a json object).
 */
function captureJson(page, elementName, cssSelector, filename, waitForAnimations, loadedStateField) {
    // Add timeout to wait for the end of animation present in some pages.
    const additionalWaiting = waitForAnimations ? defaultAdditionalWaitingMs : 0;

    return utils.waitForElementContentToLoad(page, cssSelector, loadedStateField).catch(function(logMessage) {
        log.info("The content of element " + cssSelector + " couldn't be loaded.");
        stepFailed(elementName, cssSelector, logMessage);
    }).then(function() {
        return utils.timeout(additionalWaiting);
    }).then(function() {
        console.log("elementName: " + elementName);
        if (elementName === "design.chosen_algorithm_search_strategy.table"
            || elementName === "design.other_algorithms_search_strategy.table") {
            // Algorithms are a particular case.
            return captureAlgorithm(page, elementName, cssSelector, filename);
        } else {
            return page.evaluate(function(cssSelector) {
                const domElement = document.querySelector(cssSelector);
                if (!domElement) {
                    stepFailed(elementName, cssSelector);
                    return ""; // Rather than displaying an ugly error in the output template, let's not show anything
                }

                // Start extracting the data element by element and generate the finalJson
                var finalJson = [];
                domElement.querySelectorAll("tr").forEach((lineElement) => {
                    var currentElement = [];
                    for (var i = 0; i < lineElement.children.length; i++) {
                        // Dot not create empty first column or full empty line
                        if (currentElement.length != 0 || lineElement.children[i].innerText.trim() !== "") {
                            currentElement.push(lineElement.children[i].innerText.trim());
                        }
                        // Check for tooltips inside table
                        lineElement.children[i].querySelectorAll(':scope > i[data-original-title]:not([exclude-from-export])')
                            .forEach(icon => currentElement.push(icon.getAttribute("data-original-title")));
                    }
                    finalJson.push(currentElement);
                });

                return JSON.stringify(finalJson, null, 2);

            }, cssSelector).then(function(innerText) {

                log.info("Saving json...");
                writeFile(filename, innerText);

            });
        }
    });
}

/**
 * Method that will retrieve forms as json object. It will be used to retrieve the algorithm parameters.
 */
function captureAlgorithm(page, elementName, cssSelector, filename) {
    return page.evaluate(function(cssSelector) {
        const domElement = document.querySelector(cssSelector);
        if (!domElement) {
            stepFailed(elementName, selector);
            return ""; // Rather than displaying an ugly error in the output template, let's not show anything
        }

        // HTML lists aren't printed with the innerText method so we need to create lists manually
        domElement.querySelectorAll("li").forEach((element) => {
            element.innerText = "- " + element.innerText;
        });

        // clear the help
        domElement.querySelectorAll(".help-inline").forEach((element) => {
            element.parentNode.removeChild(element);
        });

        // Start extracting the data element by element and generate the finalJson
        var finalJson = [];
        domElement.querySelectorAll(".control-group").forEach((pairElement) => {
            // Support only visible element.
            if (window.getComputedStyle(pairElement).display !== 'none' && window.getComputedStyle(pairElement.parentNode).display !== 'none') {
                var currentElementLabel;
                if (pairElement.querySelector(".control-label") !== null) {
                    currentElementLabel = pairElement.querySelector(".control-label").innerText;
                } else {
                    currentElementLabel = "Python code";
                }
                var currentElementValues = [];
                pairElement.querySelectorAll("input").forEach((element) => {
                    if (element.type === 'checkbox') {
                        if (element.checked) {
                            if (element.labels[0] && element.labels[0].innerText && element.labels[0].innerText.trim() !== '') {
                                currentElementValues.push(element.labels[0].innerText.trim() + ": Yes");
                            } else {
                                currentElementValues.push("Yes");
                            }
                        } else {
                            if (element.labels[0] && element.labels[0].innerText && element.labels[0].innerText.trim() !== '') {
                                currentElementValues.push(element.labels[0].innerText.trim() + ": No");
                            } else {
                                currentElementValues.push("No");
                            }
                        }
                    } else if (element.type === 'radio' && element.checked) {
                        currentElementValues.push(element.parentNode.innerText.trim());
                    } else if (element.type === 'number' && element.value.trim() !== '') {
                        if (element.parentNode.classList.contains('pull-left')) {
                            currentElementValues.push(element.parentNode.innerText.trim() + ': ' + element.value.trim());
                        } else {
                            currentElementValues.push(element.value.trim());
                        }
                    } else if (element.type === 'text' && element.value.trim() !== '') {
                        currentElementValues.push(element.value.trim());
                    }
                });

                // For fields with only one boolean value, don't repeat the label in the value (e.g. NPTS "use a seasonal model")
                if (currentElementValues.length === 1) {
                    let elementValue = currentElementValues[0];
                    if (elementValue.endsWith(": Yes")) {
                        elementValue = "Yes";
                    } else if (elementValue.endsWith(": No")) {
                        elementValue = "No";
                    }
                    currentElementValues[0] = elementValue;
                }

                if (currentElementValues.length !== 0) {
                    // For Log/Uniform distribution the currentElementValues should contains the number associated to the distribution
                    pairElement.querySelectorAll("span").forEach((element) => {
                        if (element.hasAttribute('data-original-title')) {
                            currentElementValues.push(element.getAttribute('data-original-title'));
                        }
                    });
                }

                if (currentElementValues.length === 0) {
                    // Retrieve text of Custom code
                    pairElement.querySelectorAll("pre.CodeMirror-line").forEach((element) => {
                        currentElementValues.push(element.innerText);
                    });
                }

                // On custom code, we want to avoid retrieving button
                if (currentElementValues.length === 0) {
                    pairElement.querySelectorAll("button").forEach((element) => {
                        if (element.innerText.trim() !== '') {
                            currentElementValues.push(element.innerText.trim());
                        }
                    });
                }

                if (currentElementValues.length === 0) {
                    pairElement.querySelectorAll("div.tag").forEach((element) => {
                        if (element.innerText.trim() !== '') {
                            currentElementValues.push(element.innerText.trim());
                        }
                    });
                }

                var currentElement = [];
                currentElement.push(currentElementLabel);
                currentElement.push(currentElementValues.join('\n'));
                finalJson.push(currentElement);
            }
        });
        return JSON.stringify(finalJson, null, 2);
    }, cssSelector).then(function(innerText) {
        log.info("Saving json...");
        writeFile(filename, innerText);
    });
}

function stepFailed(elementName, selector, err = undefined) {
    let message = "Warning: The placeholder {{" + elementName + "}} failed to retrieve information from the web interface. (Error on \"" + selector + "\")";
    log.info(message);
    if (err !== undefined) {
        log.error(err) // Print the stack trace
    }
}

function captureError(msg, filename) {
    log.info("Saving DSSError...");
    writeFile(filename, "");  // Rather than displaying an ugly error in the output template, let's not show anything
}

function writeFile(filename, content) {
    const path = outputDirectory + "/" + filename;
    try {
        fs.writeFileSync(path, content);
        log.info("Saved in " + path);
    } catch (err) {
        utils.exit(utils.ERR_GENERIC, "Unable to create file", err);
    }
}

/**
 * The Java backend continuously watch the logs of all its processes and print them as soon as it receives one,
 * meaning that a console.log in this NodeJS script will also trigger a console.log in the Java backend.
 *
 * However, when the headless browser page logs text, it is logged inside the headless browser process and not in this NodeJS script,
 * in order to do so we have to subscribe to the onConsole event of the headless browser page
 */
function logPupeteerConsoleMessage(msg) {
    const location = (typeof msg.location === "function") ? msg.location() : { url: "unavailable", lineNumber: 0, columnNumber: 0 };
    const filename = (location === undefined || location.url === undefined) ? "unavailable" : (location.url.substring(location.url.lastIndexOf("/") + 1, location.url.length));
    let text = msg.text();
    if (text.startsWith("%c")) {
        text = text.substring(2);
    }
    if (text.includes(" color: #")) {
        text = text.substring(0, text.lastIndexOf(" color: #"));
    }
    console.info("[WebBrowser/" + msg.type().toUpperCase() + "] " + text + " - " + filename + ":" + location.lineNumber + ":" + location.columnNumber);
}

function incrementProgress() {
    log.info("Processing export step")
}
