/**
 * This script use exportToolbox object that expose a set of functions needed for this feature:
 * ...
 *
 * To access this object, a dummy span has been added with id flow-export-toolbox-anchor so we can access the scope.
 *
 * If you want to do modifications or understand the code better, exportToolbox object is defined in project flow directive.
 *
 * Usage
 * node export-flow.js magicHeadlessAuth scriptConfigFile
 *
 * Arguments
 * - magicHeadlessAuth (String): API Key generated by DSS that allows authentication.
 * - scriptConfigFile (String): Path to the JSON file containing the arguments of the script (see FlowExportScriptRunner.Config Java class for details)
 */

'use strict';

const puppeteer = require('puppeteer');
const fs = require('fs');
const utils = require('./utils');
const log = require('./log');

// ========================
// Entry point
// ========================
const magicHeadlessAuth = process.argv[2].toString();

const scriptConfigFile = process.argv[3].toString();
log.info("Reading Flow export script configuration from " + scriptConfigFile);
const config = JSON.parse(fs.readFileSync(scriptConfigFile));
log.info(JSON.stringify(config));
const enforceSandboxing = config.browserSandBoxing;
const pageDefaultTimeout = config.pageDefaultTimeout;
const outputDirectory = config.outputDirectory;
const fileType = config.fileType;
const exportPageWidth = parseInt(config.width);
const exportPageHeight = parseInt(config.height);
const tileScale = parseFloat(config.tileScale);
const flowUrl = config.flowUrl;
const drawZones = config.drawZones;
const collapsedZones = config.collapsedZones;

const exportObjectType = flowUrl.includes("data-lineage/export") ? "Data Lineage" : "Flow";

try {
    log.info(`${exportObjectType} export script started`);
    utils.createBrowser(enforceSandboxing).then(function (browser) {
        return exportFlow(browser).then(function (result) {
            log.info("Closing browser");
            return browser.close();
        });
    }).then(function () {
        log.info(`${exportObjectType} export script completed`);
    }).catch(function (err) {
        utils.exit(utils.ERR_GENERIC, "Error while running export script", err);
    });
} catch (err) {
    utils.exit(utils.ERR_GENERIC, "Error while running export script", err);
}

/**
 * Export the flow
 *
 * @param browser returned by the method createBrowser.
 * @return {Promise.<Void>}
 */
function exportFlow(browser) {
    return loadFlowPage(browser).then(function (page) {
        return getToolbox(page).then(function (toolbox) {
            log.info(`Exporting ${flowUrl.includes("data-lineage/export") ? "data lineage graph" : "project flow"}`);
            return getGraphBoundaries(page, toolbox).then(function (graphBounds) {
                log.info("Retrieved graphBounds > " + JSON.stringify(graphBounds));
                let sheetRatio = exportPageWidth / exportPageHeight;
                let graphRatio = graphBounds.width / graphBounds.height;
                log.info("Computed sheetRatio:" + sheetRatio + " / graphRatio:" + graphRatio);

                let scaleFactor = tileScale;
                if (scaleFactor === 0) {
                    log.info("Determining best scale factor... ");
                    const targetFactor = 1.0;
                    const xFactor = graphBounds.width / exportPageWidth;
                    const yFactor = graphBounds.height / exportPageHeight;
                    log.info("xFactor=" + xFactor + " / yFactor=" + yFactor);
                    scaleFactor = Math.max(1, Math.ceil(Math.max(xFactor, yFactor) / targetFactor));
                    log.info("Computed scaleFactor=" + scaleFactor);
                } else {
                    log.info("Supplied scaleFactor= " + scaleFactor);
                }
                if (scaleFactor === 1) {
                    return adjustViewBox(page, toolbox, graphBounds.x, graphBounds.y, graphBounds.width, graphBounds.height).then(function () {
                        return utils.captureScreen(page, fileType, getToolbox, outputDirectory, `${flowUrl.includes("data-lineage/export") ? "Data_Lineage" : "Flow"}`);
                    });
                } else {
                    let graphSheetWidth;
                    let graphSheetHeight;
                    if (sheetRatio < graphRatio) {
                        // Dominant width
                        graphSheetWidth = graphBounds.width / scaleFactor;
                        graphSheetHeight = graphSheetWidth / sheetRatio;
                    } else {
                        // Dominant height
                        graphSheetHeight = graphBounds.height / scaleFactor;
                        graphSheetWidth = graphSheetHeight * sheetRatio;
                    }
                    if (tileScale > 1 && (graphSheetHeight < 200 || graphSheetWidth < 200)) {
                        utils.exit(utils.ERR_INVALID_TILE_SCALE_VALUE, `Invalid tile scale for this ${exportObjectType} graph: ` + (tileScale * 100) + "%. Try again with a smaller tile scale value.");
                    }
                    return captureMultiplePages(page, toolbox, graphSheetWidth, graphSheetHeight, graphBounds);
                }
            });
        }).then(function () {
            log.info("Closing page");
            return page.close();
        }).then(function () {
            // Beware: Do not modify or remove the following line as it is used by the backend to report on the progress of the script.
            log.info(`Successfully exported ${exportObjectType.toLocaleLowerCase()}`);
        }).catch(function (err) {
            utils.exit(utils.ERR_GENERIC, `Export of ${exportObjectType.toLocaleLowerCase()} failed`, err);
        });
    });
}

/**
 * Create a new browser page and navigate to the flow url
 *
 * @return {Promise.<Page>}
 */
function loadFlowPage(browser) {
    return utils.newBrowserPage(browser, exportPageWidth, exportPageHeight, magicHeadlessAuth, pageDefaultTimeout).then(function (page) {
        return utils.navigateTo(page, flowUrl).then(function () {
            return setLocalStorage(page).then(function () {
                return utils.waitForPageToLoad(page, getToolbox, 0).then(function () {
                    return getToolbox(page).then(function (toolbox) {
                        return removeDecorations(page, toolbox).then(function () {
                            return configureZones(page, toolbox, drawZones, collapsedZones).then(function () {
                                log.info(`Waiting for ${exportObjectType.toLocaleLowerCase()} to redraw itself`);
                                return utils.waitForPageToLoad(page, getToolbox, 2500).then(function () {
                                    log.info(`New browser page for ${exportObjectType.toLocaleLowerCase()} created`);
                                    return page;
                                });
                            });
                        });
                    });
                });
            });
        });
    });
}

/**
 * Iterates through all parts of the graph and capture each part separately into its own image/pdf.
 *
 * @return {Promise.<Void>}
 */
function captureMultiplePages(page, toolbox, graphSheetWidth, graphSheetHeight, graphBounds) {
    // Defines the various parts and their bounds
    log.info(`Capturing ${exportObjectType.toLocaleLowerCase()} graph with scrolling > graphSheetWidth:` + graphSheetWidth + ", graphSheetHeight:" + graphSheetHeight + ", graphBounds:" + JSON.stringify(graphBounds));
    let graphParts = [];
    const xCount = Math.max(1, Math.ceil(graphBounds.width / graphSheetWidth));
    const yCount = Math.max(1, Math.ceil(graphBounds.height / graphSheetHeight));
    for (let xIndex = 0; xIndex < xCount; xIndex++) {
        for (let yIndex = 0; yIndex < yCount; yIndex++) {
            let imagePart = {
                x: graphBounds.x + xIndex * graphSheetWidth,
                y: graphBounds.y + yIndex * graphSheetHeight,
                width: graphSheetWidth,
                height: graphSheetHeight,
                xIndex: xIndex,
                yIndex: yIndex
            };
            graphParts.push(imagePart);
        }
    }
    log.info("Capturing the following parts:" + JSON.stringify(graphParts));

    // Chain the promises to execute them in order
    if (graphParts.length === 0) {
        return Promise.all([]);
    }
    let promise = captureGraphPart(page, toolbox, graphParts[0]);
    for (let i = 1; i <= graphParts.length; i++) {
        promise = promise.then(function () {
            if (i < graphParts.length) {
                return captureGraphPart(page, toolbox, graphParts[i]);
            }
        });
    }
    return promise;
}

function captureGraphPart(page, toolbox, graphPart) {
    log.info("Capturing graph part " + JSON.stringify(graphPart));
    return adjustViewBox(page, toolbox, graphPart.x, graphPart.y, graphPart.width, graphPart.height).then(function (graphBounds) {
        const filename = `${flowUrl.includes("data-lineage/export") ? "Data_Lineage" : "Flow"}` + "_Part-" + graphPart.xIndex + "-" + graphPart.yIndex;
        return utils.captureScreen(page, fileType, getToolbox, outputDirectory, filename);
    });
}

/**
 * Gets the boundaries of the grah.
 *
 * @param page Browser page
 * @param toolbox Flow toolbox JS handle
 * @param Promise<Boundary>
 */
function getGraphBoundaries(page, toolbox) {
    return page.evaluate(function (toolbox) {
        return toolbox.getGraphBoundaries();
    }, toolbox);
}

/**
 * Sets the graph' SVG viewbox to the specified values.
 *
 * @param page Browser page
 * @param toolbox Flow toolbox JS handle
 * @param x x-coordinate of the viewbox
 * @param y y-coordinate of the viewbox
 * @param w width of the viewbox
 * @param h height of the viewbox
 * @return {Promise.<Void>}
 */
function adjustViewBox(page, toolbox, x, y, w, h) {
    return page.evaluate(function (toolbox, x, y, w, h) {
        return toolbox.adjustViewBox(x, y, w, h);
    }, toolbox, x, y, w, h);
}

/**
 * Collapses the specified zones
 *
 * @param page Browser page
 * @param toolbox Flow toolbox JS handle
 * @param drawZones true to show zones, false to hide them
 * @param collapsedZones array of string, each string containing the ID of a collapsed zone
 * @return {Promise.<Void>}
 */
function configureZones(page, toolbox, drawZones, collapsedZones) {
    return page.evaluate(function (toolbox, drawZones, collapsedZones) {
        return toolbox.configureZones(drawZones, collapsedZones);
    }, toolbox, drawZones, collapsedZones);
}

/**
 * Hides DSS header & navigation, right menu and widgets printed above the flow.
 *
 * @param page Browser page
 * @param toolbox Flow toolbox JS handle
 * @return {Promise.<Void>}
 */
function removeDecorations(page, toolbox) {
    return page.evaluate(function (toolbox) {
        return toolbox.removeDecorations();
    }, toolbox);
}

/**
 * Retrieve the Flow toolbox (defined in project_flow.js)
 *
 * @param page Browser page handle
 * @return {*} toolbox JS handle
 */
function getToolbox(page) {
    return page.evaluateHandle(function () {
        let toolboxAnchor = document.querySelector('#flow-export-toolbox-anchor');
        return angular.element(toolboxAnchor).scope().exportToolbox;
    });
}

function getCurrentUser(page) {
  return page.evaluate(() => {
    if (window.angular) {
      const rootElement = document.body;
      const injector = angular.element(rootElement).injector();
      if (injector) {
        const $rootScope = injector.get("$rootScope");
        if ($rootScope.appConfig && $rootScope.appConfig.login) {
          return $rootScope.appConfig.login;
        }
      }
    }
    return null;
  });
}

function setLocalStorage(page) {
  const searchQuery = config.searchQuery;
  const projectKey = parseProjectKeyFromUrl();
  if (!projectKey || !searchQuery) {
    return Promise.resolve();
  }
  return getCurrentUser(page)
    .then((user) => {
      if (!user) {
        return null;
      }
      const localStorageValue = JSON.stringify({ [user]: { [projectKey]: searchQuery } });
      return { "dss.omnibox.queries": localStorageValue };
    })
    .then(function (localStorage) {
        if (localStorage) {
            return utils.setLocalStorage(page, localStorage);
        } else {
            return Promise.resolve()
        }
    });
}

function parseProjectKeyFromUrl() {
  const url = new URL(flowUrl);
  const pathname = url.pathname;
  const segments = pathname.split("/");
  const projectsIndex = segments.indexOf("projects");
  if (projectsIndex !== -1 && segments.length > projectsIndex + 1) {
    return segments[projectsIndex + 1];
  }
  return null;
}
