from dataiku.runnables import Runnable
import os, json
import datetime, time, logging
import dataiku

class MyRunnable(Runnable):
    def __init__(self, project_key, config, plugin_config):
        self.project_key = project_key
        self.config = config

    def get_progress_target(self):
        return (100, 'NONE')

    def get_last_activity(self, client, app_instance): 
        project_key = app_instance['projectKey']
        last_commit_activity = app_instance.get('lastCommitTime', app_instance.get('lastCommit', {}).get('time', 0))
        print("App instance '{}' last commit".format(project_key), last_commit_activity)
        
        project = client.get_project(project_key)
        last_scenario_run_time = 0
        for scenario in project.list_scenarios(as_type="objects"):
            for last_run in scenario.get_last_runs(limit=1):
                current_time = last_run.run.get('end', last_run.run.get('start', 0))
                if (last_scenario_run_time == 0 or current_time > last_scenario_run_time):
                    last_scenario_run_time = current_time
            
        print("App instance '{}' last scenario run".format(project_key), last_scenario_run_time)

        last_job_run_time = 0
        for job in project.list_jobs():
            end_time = job.get('endTime', job.get('startTime', 0))
            if (last_job_run_time == 0 or end_time > last_job_run_time):
                last_job_run_time = end_time

        print("App instance '{}' last job run".format(project_key), last_job_run_time)

        return max(last_commit_activity, last_scenario_run_time, last_job_run_time)

    def run(self, progress_callback):
        perform_deletion = self.config.get("performDeletion", False)
        maximum_age = int(self.config.get("age", 15))
        maximum_timestamp = int(time.mktime((datetime.datetime.now() - datetime.timedelta(days=maximum_age)).timetuple())) * 1000
        requested_app_id = self.config.get("appId", None)
        
        client = dataiku.api_client()
        
        if requested_app_id is None or len(requested_app_id) == 0:
            app_ids = [a['appId'] for a in client.list_apps()] + ['__ORPHANS__']
        else:
            app_ids = [requested_app_id]
        
        to_delete = []
        for app_id in app_ids:
            for app_instance in client.get_app(app_id).list_instances():
                print(json.dumps(app_instance, indent=2))
                app_instance_id=app_instance['projectKey']
                try:
                    last_activity = self.get_last_activity(client, app_instance)
                    if last_activity < maximum_timestamp:
                        print("App instance '{}' last activity '{}' is lower to requested maximum timestamp '{}' => App will be deleted".format(app_instance_id, last_activity, maximum_timestamp) )
                        to_delete.append([app_id, app_instance['projectKey']])
                    else:
                        print("App instance '{}' last activity '{}' is higher to requested maximum timestamp '{}' => App won't be deleted".format(app_instance_id, last_activity, maximum_timestamp) )
                except Exception as e:
                    logging.warn("Unable to compute the last activity for App instance '{}', app won't be deleted by precaution. Reason: {}".format(app_instance_id, str(e)))
        print('-'*20)
        print(json.dumps(to_delete))
        print('-'*20)

        headers = ['App', 'Instance',  ('Success' if perform_deletion else '')]
        report_rows = ['<tr>%s</tr>' % ''.join(['<th>%s</th>' % header for header in headers])]
        deleted_total = 0
        done = 0
        for app_id, project_key in to_delete:
            deletion_status = None
            try:
                if perform_deletion:
                    client.get_project(project_key).delete()
                deletion_status = 'Y' if perform_deletion else ''
            except Exception as e:
                deletion_status = str(e)
                
            deleted_total += 1
            cells = [app_id, project_key, deletion_status]
            report_rows.append('<tr>%s</tr>' % ''.join(['<td>%s</td>' % cell for cell in cells]))
            
            done += 1
            progress_callback((done * 100) / len(to_delete))
    
        if perform_deletion:
            html = '<div><div>Deleted %s instances from %i apps.</div>'  % (len(to_delete), len(app_ids))
        else:
            html = '<div><div>Will delete %s instances from %i apps.</div>'  % (len(to_delete), len(app_ids))
        
        if len(to_delete) > 0:
            html += '<table class="table table-striped">%s</table>' % (''.join(report_rows))
    
        html += "</div>"
        return html

