import os, json, re, logging, shutil
from dataiku.code_studio import CodeStudioBlock, get_dataiku_user_uid

from block_utils import LibLocationPathReplacer, generate_python_codenv, get_block_python_codenv_path, sniff_jupyter_block, python_version

from distutils.version import LooseVersion

class VoilaCodeStudioBlock(CodeStudioBlock):
    def __init__(self, config, plugin_config):
        self.config = config
        self.plugin_config = plugin_config

    _ENTRYPOINT_FILE = "voila-entrypoint.sh"
        
    def _get_entrypoint_path(self):
        entrypoint_path = self.config.get("startScript", "/opt/dataiku")
        if entrypoint_path.endswith("/") or not entrypoint_path.endswith(".sh"):
            entrypoint_path = os.path.join(entrypoint_path, self._ENTRYPOINT_FILE)
        return entrypoint_path

    def _get_port(self):
        return self.config.get("port", 8866)
        
    def build_spec(self, spec, env, template):
        port = self._get_port()
        entrypoint_path = self._get_entrypoint_path()
        start_file = self.config.get("startFile", "__CODE_STUDIO_VERSIONED__/voila/app.ipynb")
        settings_path = self.config.get("settingsPath", "__CODE_STUDIO_VERSIONED__/voila")
        enable_xsrf = self.config.get("enableXSRF", False)

        # replace the lib locations in settings_path and open_in_path
        replacer = LibLocationPathReplacer(spec)
        start_file = replacer.replace_variable_by_path(start_file)
        settings_path = replacer.replace_variable_by_path(settings_path)

        # get code env stuff
        generated_code_env_mode = self.config.get("generatedCodeEnvMode", "voila-jupyterlab")
        older_than_py39 = python_version(self.config.get("pythonVersion", "python3.9")) < python_version("python3.9")
        # the package sets try to stay close to the jupyterlab package set we use
        if generated_code_env_mode == 'voila-jupyterlab':
            voila_packages = 'anyio==3.7.1 argon2-cffi==23.1.0 argon2-cffi-bindings==21.2.0 attrs==23.1.0 Babel==2.13.1 backcall==0.2.0 beautifulsoup4==4.12.2 bleach==6.0.0 certifi==2023.11.17 cffi==1.15.1 charset-normalizer==3.3.2 decorator==5.1.1 defusedxml==0.7.1 entrypoints==0.4 exceptiongroup==1.2.0 fastjsonschema==2.19.0 idna==3.5 importlib-metadata==6.7.0 importlib-resources==5.12.0 ipykernel==4.8.2 ipython==7.34.0 ipython-genutils==0.2.0 ipywidgets==7.6.5 jedi==0.19.1 Jinja2==3.1.2 json5==0.9.14 jsonschema==4.17.3 jupyter-client==6.1.12 jupyter_core==4.12.0 jupyter-server==1.24.0 jupyterlab-pygments==0.2.2 jupyterlab_server==2.24.0 jupyterlab-widgets==3.0.9 lxml==4.9.3 MarkupSafe==2.1.3 matplotlib-inline==0.1.6 mistune==0.8.4 nbclassic==1.0.0 nbclient==0.5.13 nbconvert==6.5.4 nbformat==5.8.0 nest-asyncio==1.5.8 notebook==6.5.6 notebook_shim==0.2.3 numpy==1.21.6 packaging==23.2 pandas==1.1.5 pandocfilters==1.5.0 parso==0.8.3 pexpect==4.8.0 pickleshare==0.7.5 pip==23.0.1 pkgutil_resolve_name==1.3.10 prometheus-client==0.17.1 prompt-toolkit==3.0.41 ptyprocess==0.7.0 pycparser==2.21 Pygments==2.17.2 pyrsistent==0.19.3 python-dateutil>=2.8.2,<3 pyzmq==23.2.1 requests==2.31.0 Send2Trash==1.8.2 setuptools==47.1.0 simplegeneric==0.8.1 six==1.16.0 sniffio==1.3.0 soupsieve==2.4.1 terminado==0.17.1 tinycss2==1.2.1 tornado==6.1 traitlets==5.3.0 typing_extensions==4.7.1 urllib3==1.26.18 voila==0.3.7 wcwidth==0.2.12 webencodings==0.5.1 websocket-client==1.6.1 websockets==11.0.3 widgetsnbextension==3.5.2 zipp==3.15.0'
        elif generated_code_env_mode == 'voila-vuetify':
            voila_packages = 'anyio==3.7.1 argon2-cffi==23.1.0 argon2-cffi-bindings==21.2.0 attrs==23.1.0 Babel==2.13.1 backcall==0.2.0 beautifulsoup4==4.12.2 bleach==6.0.0 certifi==2023.11.17 cffi==1.15.1 charset-normalizer==3.3.2 decorator==5.1.1 defusedxml==0.7.1 entrypoints==0.4 exceptiongroup==1.2.0 fastjsonschema==2.19.0 idna==3.5 importlib-metadata==6.7.0 importlib-resources==5.12.0 ipykernel==4.8.2 ipython==7.34.0 ipython-genutils==0.2.0 ipyvue==1.10.1 ipyvuetify==1.8.10 ipywidgets==7.6.5 jedi==0.19.1 Jinja2==3.1.2 json5==0.9.14 jsonschema==4.17.3 jupyter-client==6.1.12 jupyter_core==4.12.0 jupyter-server==1.24.0 jupyterlab-pygments==0.2.2 jupyterlab_server==2.24.0 jupyterlab-widgets==3.0.9 lxml==4.9.3 MarkupSafe==2.1.3 matplotlib-inline==0.1.6 mistune==0.8.4 nbclassic==1.0.0 nbclient==0.5.13 nbconvert==6.5.4 nbformat==5.8.0 nest-asyncio==1.5.8 notebook==6.5.6 notebook_shim==0.2.3 numpy==1.21.6 packaging==23.2 pandas==1.1.5 pandocfilters==1.5.0 parso==0.8.3 pexpect==4.8.0 pickleshare==0.7.5 pip==23.0.1 pkgutil_resolve_name==1.3.10 prometheus-client==0.17.1 prompt-toolkit==3.0.41 ptyprocess==0.7.0 pycparser==2.21 Pygments==2.17.2 pyrsistent==0.19.3 python-dateutil>=2.8.2,<3 pyzmq==23.2.1 requests==2.31.0 Send2Trash==1.8.2 setuptools==47.1.0 simplegeneric==0.8.1 six==1.16.0 sniffio==1.3.0 soupsieve==2.4.1 terminado==0.17.1 tinycss2==1.2.1 tornado==6.1 traitlets==5.3.0 typing_extensions==4.7.1 urllib3==1.26.18 voila==0.3.7 voila-vuetify==0.6.0 wcwidth==0.2.12 webencodings==0.5.1 websocket-client==1.6.1 websockets==11.0.3 widgetsnbextension==3.5.2 zipp==3.15.0'
            if len(self.config.get("template", "")) == 0:
                self.config["template"] = "vuetify-base"
        elif generated_code_env_mode == 'voila-material':
            voila_packages = 'anyio==3.7.1 argon2-cffi==23.1.0 argon2-cffi-bindings==21.2.0 attrs==23.1.0 Babel==2.13.1 backcall==0.2.0 beautifulsoup4==4.12.2 bleach==6.0.0 certifi==2023.11.17 cffi==1.15.1 charset-normalizer==3.3.2 decorator==5.1.1 defusedxml==0.7.1 entrypoints==0.4 exceptiongroup==1.2.0 fastjsonschema==2.19.0 idna==3.5 importlib-metadata==6.7.0 importlib-resources==5.12.0 ipykernel==4.8.2 ipython==7.34.0 ipython-genutils==0.2.0 ipywidgets==7.6.5 jedi==0.19.1 Jinja2==3.1.2 json5==0.9.14 jsonschema==4.17.3 jupyter-client==6.1.12 jupyter_core==4.12.0 jupyter-server==1.24.0 jupyterlab-pygments==0.2.2 jupyterlab_server==2.24.0 jupyterlab-widgets==3.0.9 lxml==4.9.3 MarkupSafe==2.1.3 matplotlib-inline==0.1.6 mistune==0.8.4 nbclassic==1.0.0 nbclient==0.5.13 nbconvert==6.5.4 nbformat==5.8.0 nest-asyncio==1.5.8 notebook==6.5.6 notebook_shim==0.2.3 numpy==1.21.6 packaging==23.2 pandas==1.1.5 pandocfilters==1.5.0 parso==0.8.3 pexpect==4.8.0 pickleshare==0.7.5 pip==23.0.1 pkgutil_resolve_name==1.3.10 prometheus-client==0.17.1 prompt-toolkit==3.0.41 ptyprocess==0.7.0 pycparser==2.21 Pygments==2.17.2 pyrsistent==0.19.3 python-dateutil>=2.8.2,<3 pyzmq==23.2.1 requests==2.31.0 Send2Trash==1.8.2 setuptools==47.1.0 simplegeneric==0.8.1 six==1.16.0 sniffio==1.3.0 soupsieve==2.4.1 terminado==0.17.1 tinycss2==1.2.1 tornado==6.1 traitlets==5.3.0 typing_extensions==4.7.1 urllib3==1.26.18 voila==0.3.7 voila-material==0.4.3  wcwidth==0.2.12 webencodings==0.5.1 websocket-client==1.6.1 websockets==11.0.3 widgetsnbextension==3.5.2 zipp==3.15.0'
            if len(self.config.get("template", "")) == 0:
                self.config["template"] = "material"
        elif generated_code_env_mode == 'voila-gridstack':
            voila_packages = 'anyio==3.7.1 argon2-cffi==23.1.0 argon2-cffi-bindings==21.2.0 attrs==23.1.0 Babel==2.13.1 backcall==0.2.0 beautifulsoup4==4.12.2 bleach==6.0.0 certifi==2023.11.17 cffi==1.15.1 charset-normalizer==3.3.2 decorator==5.1.1 defusedxml==0.7.1 entrypoints==0.4 exceptiongroup==1.2.0 fastjsonschema==2.19.0 idna==3.5 importlib-metadata==6.7.0 importlib-resources==5.12.0 ipykernel==4.8.2 ipython==7.34.0 ipython-genutils==0.2.0 ipywidgets==7.6.5 jedi==0.19.1 Jinja2==3.1.2 json5==0.9.14 jsonschema==4.17.3 jupyter-client==6.1.12 jupyter_core==4.12.0 jupyter-server==1.24.0 jupyterlab-pygments==0.2.2 jupyterlab_server==2.24.0 jupyterlab-widgets==3.0.9 lxml==4.9.3 MarkupSafe==2.1.3 matplotlib-inline==0.1.6 mistune==0.8.4 nbclassic==1.0.0 nbclient==0.5.13 nbconvert==6.5.4 nbformat==5.8.0 nest-asyncio==1.5.8 notebook==6.5.6 notebook_shim==0.2.3 numpy==1.21.6 packaging==23.2 pandas==1.1.5 pandocfilters==1.5.0 parso==0.8.3 pexpect==4.8.0 pickleshare==0.7.5 pip==23.0.1 pkgutil_resolve_name==1.3.10 prometheus-client==0.17.1 prompt-toolkit==3.0.41 ptyprocess==0.7.0 pycparser==2.21 Pygments==2.17.2 pyrsistent==0.19.3 python-dateutil>=2.8.2,<3 pyzmq==23.2.1 requests==2.31.0 Send2Trash==1.8.2 setuptools==47.1.0 simplegeneric==0.8.1 six==1.16.0 sniffio==1.3.0 soupsieve==2.4.1 terminado==0.17.1 tinycss2==1.2.1 tornado==6.1 traitlets==5.3.0 typing_extensions==4.7.1 urllib3==1.26.18 voila==0.3.7 voila-gridstack==0.3.1 wcwidth==0.2.12 webencodings==0.5.1 websocket-client==1.6.1 websockets==11.0.3 widgetsnbextension==3.5.2 zipp==3.15.0'
            if len(self.config.get("template", "")) == 0:
                self.config["template"] = "gridstack"
        elif generated_code_env_mode == 'voila-latest':
            if older_than_py39:
                # voila is not compatible with ipywidgets<8 starting from voila 0.4.0, so latest will always need a recent ipywidgets
                voila_packages = 'voila "ipywidgets>=8" "pandas>=1.1,<1.2" "numpy<1.24" python-dateutil>=2.8.2,<3 "urllib3<2" "requests<3" decorator==5.1.1 ipykernel==4.8.2 "ipython<8" ipython_genutils==0.2.0 jupyter_client==6.1.12 jupyter_core==4.12.0 pexpect==4.8.0 pickleshare==0.7.5 ptyprocess==0.7.0 pyzmq==23.2.1 simplegeneric==0.8.1 tornado==6.1 traitlets==5.3.0'
            else:
                voila_packages = 'voila "ipywidgets>=8" "pandas>=1.1,<1.2" "numpy<1.24" "python-dateutil>=2.8.2,<3" "urllib3<2" "requests<3" "decorator==5.1.1" "ipykernel==6.23.3" "ipython>=8.12,<8.13" "ipython_genutils==0.2.0" "jupyter_client<9,>=7.4.4" "jupyter_core==5.7.2" "pexpect==4.8.0" "pickleshare==0.7.5" "ptyprocess==0.7.0" "pyzmq>=24" "simplegeneric==0.8.1" "tornado>=6.3,<6.4" "traitlets==5.9.0"'
        elif generated_code_env_mode == 'voila-custom':
            voila_version = self.config.get("voilaVersion", "0.3.7")
            voila_packages = 'voila==%s urllib3<2 ipykernel==4.8.2' % voila_version
            if 'ipywidgets' not in voila_packages:
                voila_packages = voila_packages + ' ipywidgets==7.6.5'
        else:
            voila_packages = 'voila'
        generate_codenv, pyenv_path = generate_python_codenv("VOILA", self.config, template, voila_packages, "/opt/dataiku/python-code-envs/pyenv-voila", "python3.9", env.get("globalCodeEnvsExtraSettings"))

        # check if there's a jupyterlab we should work together with, to get the same kernels
        jupyter_block = sniff_jupyter_block(template)
        if jupyter_block is None:
            add_jupyter_kernel = ''
        else:
            jupyter_pyenv_path = get_block_python_codenv_path("JUPYTERLAB", self.config, template, "/opt/dataiku/pyenv-jupyter")
            add_jupyter_kernel = """
source {jupyter_pyenv_path}/bin/activate
python3 -m ipykernel install --prefix {pyenv_path}/ --name jupyter-venv --display-name "Jupyterlab"
deactivate
""".format(pyenv_path=pyenv_path, jupyter_pyenv_path=jupyter_pyenv_path)
            
        additional_args = []
        template = self.config.get("template", '')
        if len(template) > 0:
            additional_args = additional_args + ['--template=%s' % template]
        template_resources = self.config.get("templateResources", '')
        if len(template_resources) > 0:
            additional_args = additional_args + ['--VoilaConfiguration.resources=%s' % template_resources]
        theme = self.config.get("theme", '')
        if len(theme) > 0:
            additional_args = additional_args + ['--theme=%s' % theme]
        extra_args = self.config.get('extraArguments', [])
        for extra_arg in extra_args:
            flag = extra_arg.split("=")[0]
            already_in_additional_args = [a for a in additional_args if a.startswith(flag)]
            if len(already_in_additional_args) > 0:
                raise Exception("Can't have '%s' as extra argument, because it's already present by default: '%s'" % (extra_arg, " ".join(already_in_additional_args)))
        additional_args = additional_args + extra_args
            
        # add the entrypoint script in the buildir
        entrypoint_script = """
#! /bin/bash

USER=dataiku
HOME=/home/dataiku

unset JUPYTER_DATA_DIR
unset IPYTHONDIR
unset JUPYTER_RUNTIME_DIR
unset JUPYTER_CONFIG_DIR

# Register user kernels
mkdir -p {pyenv_path}/share/jupyter/kernels/
if [ -d /opt/dataiku/python-code-envs ]; then
    for i in $(ls /opt/dataiku/python-code-envs); do
        source /opt/dataiku/python-code-envs/$i/bin/activate
        python3 -m ipykernel install --prefix {pyenv_path}/ --name py-dku-venv-$i --display-name "DSS Codeenv - $i"
        deactivate
    done
fi
{add_jupyter_kernel}


mkdir -p {settings_path}/.voila

if [ $DKU_CODE_STUDIO_IS_PUBLIC_PORT_{port} = "1" ]; then
    export BIND_ADDR=0.0.0.0
else
    export BIND_ADDR=127.0.0.1
fi

mkdir -p /home/dataiku/.voila
export BASE_URL=$(eval echo "$DKU_CODE_STUDIO_BROWSER_PATH_{port}")

# START VOILA

# cd'ing in order to pick the custom config
cd {settings_path}
LC_ALL=en_US.utf8  {pyenv_path}/bin/python -m voila --Voila.ip="$BIND_ADDR" --debug --no-browser --Voila.tornado_settings="{{'allow_origin': '*', 'disable_check_xsrf': {disable_xsrf} }}" --Voila.base_url="$BASE_URL/" --port={port} {start_file} {additional_args}
""".format(pyenv_path=pyenv_path, start_file=start_file, settings_path=settings_path, port=port, 
           disable_xsrf="False" if enable_xsrf else "True",
           additional_args=' '.join(additional_args), add_jupyter_kernel=add_jupyter_kernel)
        with open(os.path.join(env["buildDir"], self._ENTRYPOINT_FILE), "wb") as f:
            f.write(entrypoint_script.encode("utf8"))
            
        # the dockerfile addition
        spec["dockerfile"] = spec.get("dockerfile", "") + """

##### VOILA BLOCK #####

USER root
WORKDIR /opt/dataiku

{generate_codenv}

# entrypoint.sh
COPY {entrypoint_file} {entrypoint_path}
RUN chown dataiku:root {entrypoint_path} && chmod +x {entrypoint_path}

# USER dataiku
USER {uid}
WORKDIR /home/dataiku
""".format(pyenv_path=pyenv_path, start_file=start_file, port=port,
           entrypoint_path=entrypoint_path, entrypoint_file=self._ENTRYPOINT_FILE,
           generate_codenv=generate_codenv,
           uid=get_dataiku_user_uid())
        return spec

    def build_launch(self, spec, env):
        if env['launchedFrom'] == 'WEBAPP' and not self.config.get("useInWebapps", False):
            return spec
        port = self._get_port()
        spec['entrypoints'] = spec.get('entrypoints', []) + [self._get_entrypoint_path()]
        readiness_probe_url = "http://localhost:" + str(port) + "${baseUrlPort" + str(port) + "}/voila/static/voila.js" # baseUrlPort should be replaced by actual url in BlockBasedCodeStudioMeta/buildYaml
        if spec.get('readinessProbeUrl', "") == "":
            spec['readinessProbeUrl'] = readiness_probe_url
        exposed_port = {
            "label": "Voila",
            "proxiedUrlSuffix": "$uri$is_args$args",
            "exposeHtml": True,
            "port": port,
            "readinessProbeUrl": readiness_probe_url
        }
        spec['exposedPorts'] = spec.get('exposedPorts', []) + [exposed_port]
        return spec
        
    def build_creation(self, spec, env):
        tmp_dir = env["tmpDir"]
        file_name = "voila.app.ipynb"
        generated_code_env_mode = self.config.get("generatedCodeEnvMode", "voila-jupyterlab")
        if generated_code_env_mode == 'voila-vuetify':
            src_path = os.path.join(os.environ["DKU_CUSTOM_RESOURCE_FOLDER"], 'voila-vuetify-app.ipynb')
        elif generated_code_env_mode == 'voila-custom':
            src_path = os.path.join(os.environ["DKU_CUSTOM_RESOURCE_FOLDER"], 'voila-custom-app.ipynb')
        else:
            src_path = os.path.join(os.environ["DKU_CUSTOM_RESOURCE_FOLDER"], 'voila-app.ipynb')
        dst_path = os.path.join(tmp_dir, file_name)
        shutil.copyfile(src_path, dst_path)
        spec["codeStudioVersionedFiles"].append({"key":file_name, "value": "voila/app.ipynb"})
        return spec
        
