import time
from enum import Enum
from typing import Any, Dict, Optional, TYPE_CHECKING

import boto3.session
from langchain_community.vectorstores import OpenSearchVectorSearch
from opensearchpy import RequestsHttpConnection, AWSV4SignerAuth

from dataiku.core.vector_stores.dku_vector_store import DkuRemoteVectorStore, logger, RecordManagerCleanupMode
from dataiku.core.vector_stores.elasticsearch_vector_store import ElasticSearchVectorStoreDocumentFilter
from dataiku.langchain.dataset_loader import VectorStoreLoader
from dataiku.llm.types import RetrievableKnowledge
from dataikuapi.dss.admin import DSSConnection
from dataikuapi.dss.langchain import DKUEmbeddings

if TYPE_CHECKING:
    from langchain_core.vectorstores import VectorStore

class AWSServiceType(str, Enum):
    OPENSEARCH_SERVERLESS = 'aoss'
    OPENSEARCH_HOSTED = 'es'


class OpenSearchVectorStore(DkuRemoteVectorStore):

    # Keep in sync with ElasticSearchConnection class on java side
    DEFAULT_WAIT_DELAY_SECONDS: float = 20  # DEFAULT_REFRESH_WAIT_DELAY_MS in java
    DEFAULT_MAX_WAIT_DELAY_SECONDS: float = 90  # DEFAULT_REFRESH_WAIT_MAX_DELAY_MS in java

    def __init__(self, kb: RetrievableKnowledge, exec_folder: str):
        self.opensearch_url: Optional[str] = None
        self.auth_kwargs: Dict = {}
        self.is_amazon_opensearch_serverless = False
        self.wait_delay_s = self.DEFAULT_WAIT_DELAY_SECONDS
        self.max_wait_delay_s = self.DEFAULT_MAX_WAIT_DELAY_SECONDS
        super(OpenSearchVectorStore, self).__init__(kb, exec_folder, bulk_size=500)  # OpenSearch default limit bulk size
        self.document_filter = ElasticSearchVectorStoreDocumentFilter(self.metadata_column_type)

    def set_index_name(self, index_name: str) -> None:
        self.index_name = index_name.lower()

    def init_connection(self, connection: DSSConnection) -> None:
        connection_params = connection.get_info()['params']
        self.set_wait_properties(connection_params)
        auth_type = connection_params['authType']
        self.auth_kwargs = {}
        scheme = "http"
        if connection_params['ssl']:
            scheme = "https"
            if connection_params['trustAnySSLCertificate']:
                self.auth_kwargs['verify_certs'] = False

        self.opensearch_url = "{}://{}:{}".format(scheme, connection_params['host'], connection_params['port'])
        if auth_type == "NONE":
            pass
        elif auth_type == "PASSWORD":
            self.auth_kwargs['http_auth'] = (connection_params['username'], connection_params['password'])
        elif auth_type == "OAUTH2_APP":
            raise NotImplementedError("We do not support OAuth v2.0 for Knowledge Banks")
        elif auth_type in ["AWS_KEYPAIR", "AWS_ENVIRONMENT", "AWS_STS", "AWS_CUSTOM"]:
            # always use sessions provided on Java side to ensure that we have the same behaviour
            credentials = connection.get_info()['resolvedAWSCredential']
            session = boto3.session.Session(
                aws_access_key_id=credentials['accessKey'],
                aws_secret_access_key=credentials['secretKey'],
                aws_session_token=credentials.get('sessionToken')
            ).get_credentials()
            self.set_aws_connection(session, connection_params)
        else:
            raise ValueError("Unknown authentication type: " + auth_type)

    def set_aws_connection(self, session: boto3.session.Session, connection_params: Dict) -> None:
        service_name = AWSServiceType[connection_params['aws']['service']].value
        self.auth_kwargs['http_auth'] = AWSV4SignerAuth(session, connection_params['aws']['region'], service_name)
        self.auth_kwargs['connection_class'] = RequestsHttpConnection
        self.is_amazon_opensearch_serverless = service_name == "aoss"
        if self.is_amazon_opensearch_serverless:
            # AOSS fails without _large enough_ read http timeout, so let's use a larger value based on wait_delay_s
            self.auth_kwargs['timeout'] = self.wait_delay_s * 5

    def set_wait_properties(self, connection_params: Dict) -> None:
        for property in connection_params['dkuProperties']:
            if property['name'] == "dku.connection.elasticsearch.refreshWaitDelayMs":
                self.wait_delay_s = int(property['value']) / 1000.0
                logger.info("Set custom wait delay: {}s".format(self.wait_delay_s))
            elif property['name'] == "dku.connection.elasticsearch.refreshMaxWaitDelayMs":
                self.max_wait_delay_s = int(property['value']) / 1000.0
                logger.info("Set custom max wait delay: {}s".format(self.max_wait_delay_s))

    def get_db(self, embeddings: Optional[DKUEmbeddings], allow_creation: bool = False, **kwargs: Any) -> OpenSearchVectorSearch:
        # todo should check if index already exist to raise an error if allow_creation=false first (langchain always create it by default if unfound)
        db = OpenSearchVectorSearch(
            self.opensearch_url,
            self.index_name,
            embeddings,
            **self.auth_kwargs
        )
        return db

    def clear_index(self) -> None:
        db = self.get_db(None)
        if db.index_exists(self.index_name):
            db.delete_index(self.index_name)
        # see also custom AOSS handling on java side in ElasticSearchIndex and ElasticSearchOutput classes
        if self.is_amazon_opensearch_serverless:
            # AOSS needs time so that the deletion of the index does not race with indexing
            time.sleep(self.wait_delay_s)
        logger.info("Cleared OpenSearch index {}".format(self.index_name))

    def ensure_documents_are_indexed(self) -> None:
        if self.is_amazon_opensearch_serverless:
            logger.info("Waiting for documents to be indexed...")
            time.sleep(self.max_wait_delay_s)

    def _index_documents(self, documents_loader: VectorStoreLoader, vectorstore_db: "VectorStore", cleanup_mode: RecordManagerCleanupMode, source_id_key: str, batch_size: Optional[int] = None) -> None:
        if self.is_amazon_opensearch_serverless:
            raise Exception("AWS OpenSearch Serverless does not support smart sync or upsert modes")
        super(OpenSearchVectorStore, self)._index_documents(documents_loader, vectorstore_db, cleanup_mode, source_id_key, batch_size)



