import logging

class SparkLikeDialect(object):
    def __init__(self):
        self._to_dss_types_map = None
        
    def _get_to_dss_types_map(self):
        if self._to_dss_types_map is None:
            self._to_dss_types_map = {}
        return self._to_dss_types_map
        
    def allow_empty_schema_after_catalog(self):
        return False
        
    def identifier_quote_char(self):
        """Get the character used to quote identifiers"""
        return '"'
    
    def quote_identifier(self, identifier):
        """Quote an identifier"""
        quote = self.identifier_quote_char()
        return quote + identifier.replace(quote, quote * 2) + quote if identifier is not None else ''
        
    def unquote_identifier(self, identifier):
        """Unquote an identifier"""
        quote = self.identifier_quote_char()
        if identifier is not None and quote in identifier:
            return identifier[1:-1].replace(quote * 2, quote)
        else:
            return identifier
        
    def _column_name_to_sql_column(self, identifier):
        return identifier # override with the right "col(value)" or similar
    
    def _python_literal_to_sql_literal(self, value, column_type, original_type=None):
        return value # override with the right "lit(value)" or similar
    
    def python_string_to_sql_literal(self, value, column_type, original_type=None):
        """Build an expression to represent a string value"""
        literal = None
        # cast in python as appropriate
        if value is None:
            literal = None
        elif column_type == 'boolean':
            literal = bool(value)
        elif column_type in ['tinyint', 'smallint', 'int', 'bigint']:
            literal = int(value)
        elif column_type in ['float', 'double']:
            literal = float(value)
        else:
            literal = str(value)
        return self._python_literal_to_sql_literal(literal, column_type, original_type)
        
    
    def _get_components_from_df_schema(self, df_schema):
        return [], {}
        
    def get_dss_schema_from_df_schema(self, df_schema):
        """
        A simple function that returns a DSS schema from a dataframe
        """
        logging.debug("Computing DSS schema corresponding to dataframe schema: %s" % df_schema)
        column_names, column_fields = self._get_components_from_df_schema(df_schema)
        if len(set(column_names)) != len(list(column_names)):
            raise Exception("DSS doesn't support dataframes containing multiple columns with the same name.")

        dss_schema = []
        for field_name in column_names:
            field = column_fields[field_name]
            t = self._get_dss_type_from_df_datatype(field["datatype"])
            if isinstance(t, dict):
                sc = t
            else:
                sc = { 'type': t }
            sc['name'] = field["name"]
            dss_schema.append(sc)
        logging.debug("Computed schema from dataframe: %s" % dss_schema)
        return dss_schema

    def _get_datatype_name_from_df_datatype(self, datatype):
        return str(datatype)

    def _get_dss_type_from_df_datatype(self, datatype):
        # returns a type (=str) or a schemaColumn (=dict) without the name
        datatype_name = self._get_datatype_name_from_df_datatype(datatype)
        return self._get_to_dss_types_map().get(datatype_name, "string")

