(function() {
'use strict';

const app = angular.module('dataiku.catalog');


const FACET_FIELDS_DISPLAY_NAMES = Object.freeze({
    'projectName': 'Project',
    'type_raw': 'Type',
    'numColumns': 'Columns',
    'usedIn': 'Used in',
    'user': 'Contributors',
    'storedAs': 'Stored as',
    'projectKey.raw': 'Project',
    'tag.raw': 'Tags',
    'catalog.raw': 'DB Catalog',
    'connection.raw': 'Connection',
    'schema.raw': 'DB Schema',
    'partitioned': 'Partitioned',
    'isInDataCollection': 'In Collection',
    'closed': 'Discussion status',
    'dataSteward': 'Data Steward',
    'objectType': 'On object of type',
});


const DISPLAY_NAMES = Object.freeze({
    recipeTypes: {
        'shaker': 'Prepare'
    },
    notebookTypes: {
        'SQL_NOTEBOOK': 'SQL',
        'SEARCH_NOTEBOOK': 'Search',
        'JUPYTER_NOTEBOOK': 'Jupyter',
        'IPYTHON_NOTEBOOK': 'Jupyter' //Legacy
    },
    notebookLanguages: {
        'ES_QUERY_STRING': 'ElasticSearch Query String',
        'SQL': 'SQL',
        'python2': 'Python 2',
        'python3': 'Python 3',
        'ir': 'R',
        'toree': 'Scala'
    }
});

/**
     * CatalogItemService makes a set of simple re-useable functions for displaying/formatting catalog-items
     * available to other controllers e.g. personal home page.
     */
    app.service('CatalogItemService', function( StateUtils, $location, Navigator, $filter, $state, FLOW_COMPUTABLE_TYPES) {
        const svc = this;

        svc.getLink = function (_type, _source, discussionId) {
            if (!_type || !_source) return;

            switch (_type) {
                case 'article':
                    return StateUtils.href.dssObject(_type.toUpperCase(), _source.name, _source.projectKey);

                case 'dashboard':
                    return StateUtils.href.dashboard(_source.id, _source.projectKey, {name: _source.name});

                case 'column':
                    return $state.href(
                        'projects.project.datasets.dataset.explore',
                        {projectKey: _source.projectKey, datasetName: _source.dataset}
                    );

                case 'comment':
                    return StateUtils.href.dssObject(
                        _source.objectType.toUpperCase(),
                        _source.objectId,
                        _source.projectKey,
                        { moveToTargetProject: true }
                    );
                case 'table':
                    return $state.href(
                        'external-table',
                        {
                            connection: _source.virtualConnection ? _source.virtualConnection : _source.connection,
                            catalog: _source.catalog,
                            schema: _source.schema,
                            table: _source.name
                        }
                    );
                case 'discussion': {
                    const obj = {
                        type: _source.objectType.toUpperCase(),
                        id: _source.objectId,
                        workspaceKey: _source.workspaceKey,
                        projectKey: _source.projectKey
                    };
                    return StateUtils.href.taggableObject(obj, {moveToTargetProject: true, discussionId: _source.discussionId});
                }
                case 'flow_zone':
                    return StateUtils.href.flowZone(_source.id, _source.projectKey);
                case 'data_collection':
                    return StateUtils.href.dataCollection(_source.id)
                case 'chart':
                    return StateUtils.href.chart(_source.objectType, _source.projectKey, _source.objectId, _source.index)
                default:
                    return StateUtils.href.dssObject(
                        svc.indexableTypeToTaggableType(_type, _source.type_raw).toUpperCase(),
                        _source.hasOwnProperty('id') ? _source.id : _source.name,
                        _source.projectKey,
                        { moveToTargetProject: true }
                    )
            }
        };

        svc.indexableTypeToTaggableType = function(indexableType, rawType) {
            if (!indexableType) return;

            switch(indexableType.toLowerCase()) {
                case 'notebook':
                    return rawType;
                case 'knowledge_bank':
                    return 'retrievable_knowledge';
                default:
                    return indexableType;
            }
        };

        svc.goToItem = function (_type, _source) {
            _type = svc.indexableTypeToTaggableType(_type, _source.type_raw);
            $location.path(svc.getLink(_type, _source));
        };

        svc.hasNavigator = function (_type) {
            return ['dataset', 'recipe', 'saved_model', 'model_evaluation_store', 'managed_folder', 'streaming_endpoint', 'labeling_task', 'knowledge_bank'].indexOf(_type) > -1;
        };

        svc.openNavigator = function (_type, _source) {
            return function () {
                _type = svc.indexableTypeToTaggableType(_type, _source.type_raw);
                Navigator.show(_source.projectKey, _type.toUpperCase(), _source.id);
            };
        };

        svc.getFlowLink = function (_type, _source, _contextProjectKey) {
            if (!_type || !_source) return;
            _type = svc.indexableTypeToTaggableType(_type, _source.type_raw);
            return StateUtils.href.flowLinkFromProps(_type, _source.projectKey, _source.id, _contextProjectKey)
        };

        svc.hasFlowLink = function (_type) {
            _type = svc.indexableTypeToTaggableType(_type);
            return FLOW_COMPUTABLE_TYPES.includes(_type.toUpperCase()) || _type === 'recipe' || _type === 'labeling_task';
        };

        svc.itemToIcon = function (_type, _source, inList, size) {
            if (!_type || !_source) {
                return;
            }

            switch (_type) {
                case 'dataset':
                    return inList ? $filter('typeToIcon')(_source.type_raw, size) : $filter('datasetTypeToIcon')(_source.type_raw, size);
                case 'notebook':
                    return $filter('typeToIcon')(_source.type_raw, size);
                case 'web_app':
                    return $filter('subTypeToIcon')(_source.subtype || _source.type_raw, _type, size);
                case 'saved_model': {
                    return $filter('savedModelSubtypeToIcon')(_source.taskType, _source.backendType, _source.predictionType, _source.savedModelType, _source.proxyModelProtocol || _source.externalSavedModelType, size);
                }
                default:
                    return $filter('subTypeToIcon')(_source.type_raw, _type, size);
            }
        };

        svc.itemToColor = function (_type, _source) {
            if (!_type || !_source) return;
            if (_type == 'insight') {
                return $filter('insightTypeToColor')(_source.type_raw) + ' insight-icon';
            } else if (_type == 'recipe') {
                return $filter('recipeTypeToColorClass')(_source.type_raw);
            } else if (_type == 'saved_model') {
                return $filter('savedModelTypeToClassColor')(_source.savedModelType);
            } else if (_type == 'chart') {
                return $filter('chartTypeToColor')(_source.objectType);
            } else {
                return _type.toLowerCase();
            }
        };
    });



app.controller('CatalogItemsController', function($controller, $scope, $injector, $state, $stateParams, $route,
    $q, $location, $filter, $timeout,
    DataikuAPI, CreateModalFromTemplate, WT1, TopNav, DatasetsService, StateUtils, Debounce,
    CatalogUtils, Navigator, DashboardUtils, CachedAPICalls, CatalogItemService, TypeMappingService, TaggingService) {

    const overridenCatalogUtils = {
        getHash: CatalogUtils.getHash,
        getLink: CatalogUtils.getLink,
        parseHash: function($scope, hash) {
            CatalogUtils.parseHash($scope, hash);
            if (!$scope.query.facets['scope']) {
                $scope.query.facets['scope'] = $stateParams.scope ? $stateParams.scope : ['dss'];
            }
            if (!$scope.query.facets['_type'] && $stateParams._type) {
                $scope.query.facets['_type'] = $stateParams._type;
            }
        }
    };

    $controller("_CatalogControllerBase", {
        $scope: $scope,
        searchEndpoint: DataikuAPI.catalog.search,
        CatalogUtils: overridenCatalogUtils
    });

    $scope.getCatalogScope = () => $scope;

    $scope.tableComparator = function(one, other) {
        return one && other && one._id === other._id;
    };

    $scope.getNames = function(tables) {
        return tables.map(el => el._source.name);
    };

    $scope.getImportData = function (tables) {
        const selectedTables = tables || $scope.selection.selectedObjects;

        const tableKeys = selectedTables.map(t => ({
            connectionName: t._source.connection,
            catalog: t._source.catalog,
            schema: t._source.schema,
            name: t._source.name
        }));
        return {
            wt1Context: {
                from: 'search-dss'
            },
            workflowType : "KEYS",
            tableKeys : tableKeys
        };
    };

    $scope.importTables = function (tables, zoneId) {
        const selectedTables = tables && !angular.isArray(tables) ? [tables] : tables;
        if ($stateParams.projectKey) {
            $state.go('projects.project.tablesimport', {
                projectKey: $stateParams.projectKey,
                importData : JSON.stringify($scope.getImportData(selectedTables)),
                zoneId
            });
        } else {
            let newScope;
            if (selectedTables) {
                newScope = $scope.$new();
                newScope.getImportData = () => $scope.getImportData(selectedTables)
            } else {
                newScope = $scope;
            }
            CreateModalFromTemplate("/templates/datasets/tables-import-project-selection-modal.html", newScope, "TablesImportProjectSelectionModalController");
        }
    };

    $scope.goToConnectionsExplorer = function() {
        if ($state.includes('projects.project')) {
            $state.go('projects.project.datacatalog.database-explorer');
        } else {
            $state.go('homeV2.data-catalog.database-explorer');
        }
    };

    $scope.hasUnindexedConnections = function() {
        const iau = $scope.indexedAndUnindexed;
        if (!iau) {
            return;
        }
        return (iau.possiblyUnscannedHive || iau.unindexedButIndexableConnections > 0) &&
            ($scope.query.facets.scope[0] === "external" || $scope.query.facets.scope[0] === "all" );
    };

    const parentResetSearch = $scope.resetSearch;
    $scope.resetSearch = function() {
        const facetsScope = $scope.query.facets.scope;
        parentResetSearch();
        $scope.query.facets.scope = facetsScope || ['dss'];
    };

    $scope.isItemSelectable = function(item) {
        return item._type === 'table';
    };

    $scope.clickShowSelected = function() {
        $scope.showSelectedOnly = !$scope.showSelectedOnly;
    };

    $scope.isFunction = angular.isFunction;

    // Init

    $scope.showSelectedOnly = false;

    $scope.showAllTypes = false;

    $scope.projectKey = $stateParams.projectKey;
    $scope.itemsPage = true;
    $scope.displayItemTypeFacet = !$state.current.name.includes('datacatalog');

    $scope.mapItemToIcon = function(item, size) {
        if (item.key === 'chart') {
            // TODO : remove this hack when implementing [sc-243801]
            return 'dku-icon-chart-' + size; // this icon is not mapped yet so let's do something dirty for now, otherwise it will have impact on the whole application
        }
        return TypeMappingService.mapTypeToIcon(item.key, size);
    }

    $scope.locations = [
        {name: 'DSS', id: 'dss'},
        {name: 'External tables', id: 'external'}
    ];

    $scope.sortBy = [
        {
            label: 'Relevance',
            value: '_score'
        },
        {
            label: 'Type',
            value: i => i._type + i._source.type_raw
        },
        {
            label: 'Creation',
            value: i => i._source.createdOn
        },
        {
            label: 'Last modification',
            value: i => i._source.lastModifiedOn
        }
    ];


    const projectNames = {};

    $scope.types = [];
    $scope.tagMaps = {};
    $scope.users = {};
    $scope.projects = {}; // Map of UIProject indexed by project key

    // Watches

    $scope.$watch('selection.selectedObjects.length', function(nv, ov) {
        if (ov > 0 && nv === 0) {
            $scope.showSelectedOnly = false;
        }
    });

    function init() {
        DataikuAPI.connections.countIndexedAndUnindexed()
            .success(function(response) {
                $scope.indexedAndUnindexed = response;
            })
            .error(setErrorInScope.bind($scope));

        DataikuAPI.security.listUsers()
            .success(function(response) {
                angular.forEach(response, function(user) {
                    $scope.users[user.login] = user.displayName;
                })
            })
            .error(setErrorInScope.bind($scope));

        DataikuAPI.projects.list(true)
            .success(function(response) {
                angular.forEach(response, function(project) {
                    projectNames[project.projectKey] = project.name;
                    $scope.projects[project.projectKey] = project;
                })
            })
            .error(setErrorInScope.bind($scope));
    }

    // Make sure that active facets are displayed even if their doc_count is 0
    function addFacets(aggs) {
        for (const field in aggs) {
            angular.forEach($scope.query.facets[field], function(value) {
                if (field == '_type' && value == 'all') return;
                if (!aggs[field].agg.buckets.filter(function(bucket) {
                        return bucket.key == value;
                    }).length) {
                    aggs[field].agg.buckets.push({key: value, doc_count: 0});
                }
            });
        }
    }

    $scope.getLink = function (input, discussionId) {
        if (!input) return;
        return CatalogItemService.getLink(input._type, input._source, discussionId);
    };

    $scope.goToItem = function(item) {
        if (!item) return;
        return CatalogItemService.goToItem(item._type, item._source);
    };

    $scope.hasNavigator = function(item) {
        if (!item) return;
        return CatalogItemService.hasNavigator(item._type);
    };

    $scope.openNavigator = function(item) {
        if (!item) return;
        return CatalogItemService.openNavigator(item._type, item._source);
    };

    $scope.getFlowLink = function(input) {
        if (!input) return;
        return CatalogItemService.getFlowLink(input._type, input._source);
    };

    $scope.hasFlowLink = function(input) {
        if (!input) return;
        return CatalogItemService.hasFlowLink(input._type, input._source);
    };

    $scope.formatFacetField = function(field) {
        return FACET_FIELDS_DISPLAY_NAMES[field] || $filter('capitalize')(field);
    };

    $scope.formatFacetValue = function(value, facet) {
        switch (facet) {
            case 'closed':
                return value ? 'Closed' : 'Open';
            case 'projectKey.raw':
                return projectNames[value] || value;
            case '_type':
                return $filter('capitalize')(value.replace(/_/g, " "));
            case 'user':
            case 'owner':
            case 'dataSteward':
                return $scope.users[value] ? `${$scope.users[value]} (${value})` : value;
            case 'language':
                return DISPLAY_NAMES.notebookLanguages[value] || value;
            case 'connection.raw':
                return $filter('connectionNameFormatter')(value);
            case 'type_raw':
                if (!$scope.query.facets._type) {
                    return value;
                }
                switch ($scope.query.facets._type[0]) {
                    case 'dataset':
                        return $filter('capitalize')(value);
                    case 'recipe':
                        return DISPLAY_NAMES.recipeTypes[value] || $filter('capitalize')(value.replace(/_/g, " "));
                    case 'notebook':
                        return DISPLAY_NAMES.notebookTypes[value];
                    case 'saved_model':
                        return $filter('capitalize')(value);
                    case 'model_evaluation_store':
                        return $filter('capitalize')(value);
                    case 'labeling_task':
                        return $filter('capitalizeWord')($filter('niceConst')(value))
                    case 'insight':
                        return DashboardUtils.getInsightHandler(value).name || 'Unknown';
                    default:
                        return value;
                }
            case 'partitioned':
            case 'isInDataCollection':
                return value === 0 ? 'No' : 'Yes';
        }
        return value;
    };

    // Override to allow to search for project by both name and key in the project list facet
    $scope.facetValueMatching = function(field) {
        return function(search) {
            search = (search || '').toLowerCase();
            return function(item) {
                if (!search || !search.length) {
                    return true;
                }
                if (item.key.toLowerCase().includes(search)) {
                    return true;
                } else if (field === "projectKey.raw") {
                    return (projectNames[item.key] || '').toLowerCase().includes(search);
                } else if (field === "user" || field === "owner" || field === "dataSteward") {
                    return ($scope.users[item.key] || '').toLowerCase().includes(search);
                }
                return false;
            };
        };
    };

    $scope.itemToIcon = function(item, inList, size) {
        if (!item) return;
        return CatalogItemService.itemToIcon(item._type, item._source, inList, size);
    };

    $scope.formatItemName = function(item, inList) {
        if (item._type == 'discussion') {
            const src = item._source;
            // Comes from _source, encode HTML entities in order to display attributes like <stuff
            const topic = (item.highlight && item.highlight['discussions.topic']) ? item.highlight['discussions.topic'][0] : ($filter('escapeHtml')(((src.discussions && src.discussions.length && src.discussions[0].topic) ? src.discussions[0].topic : "Unnamed discussion")));
            const title = topic + " <small>on " + src.objectType.replace('_', ' ') + "</small> " + $filter('escapeHtml')(src.objectName);
            return title;
        }

        if (item._type == 'chart') {
            const src = item._source;
            // Comes from _source, encode HTML entities in order to display attributes like <stuff
            const name = $scope.encodeItemName(item) || 'Untitled chart'; // TODO : change done near release. We should factorize the untitled behavior to all types
            const title = name + " <small>on " + src.objectType.replace('_', ' ') + "</small> " + $scope.encodeItemObjectName(item);
            return title;
        }
        return $scope.encodeItemName(item);
    };

    $scope.encodeItemName = function(item) {
        if (item.highlight) {
            if (item.highlight.name && item.highlight.name.length) {
                return sanitizeHighlighted(item.highlight.name[0]);
            }
            if (item.highlight['name.raw'] && item.highlight['name.raw'].length) {
                return sanitizeHighlighted(item.highlight['name.raw'][0]);
            }
        }
        // Comes from _source, encode HTML entities in order to display attributes like <stuff
        return $filter('encodeHTML')(item._source.name);
    };

    $scope.encodeItemObjectName = function(item) {
        if (item.highlight && item.highlight['objectName'] && item.highlight['objectName'].length) {
            return sanitizeHighlighted(item.highlight['objectName'][0]);
        }
        // Comes from _source, encode HTML entities in order to display attributes like <stuff
        return $filter('encodeHTML')(item._source.objectName);
    };

    $scope.getCommentLink = function(item) {
        const tor = {
            type: item._source.objectType.toUpperCase(),
            id: item._source.objectId,
            projectKey: item._source.projectKey
        };
        return StateUtils.href.taggableObject(tor, {moveToTargetProject: false});
    };

    $scope.highlightedAttachmentsList = function(item) {
        if (!item.hasOwnProperty('highlightedAttachments')) {
            item.highlightedAttachments = [];
            let strippedAttachments = [];
            if (item.highlight && item.highlight['attachments.displayName']) {
                strippedAttachments = item.highlight['attachments.displayName'].map(str => str.replace(/<\/?em>/g, ''));
            }
            angular.forEach(item._source.attachments, function(attachment) {
                const index = strippedAttachments.indexOf(attachment.name);
                if (index == -1) {
                    item.highlightedAttachments.push(attachment);
                } else {
                    item.highlightedAttachments.push({
                        displayName: item.highlight['attachments.displayName'][index],
                        type: attachment.type
                    });
                }
            });
        };

        return item.highlightedAttachments;
    };

    function getTagHighlights(highlight) {
        if (!highlight) {
            return [];
        }
        return [
            ...(highlight["tag.plaintext"] || []),
            ...(highlight.tag || []),
        ];
    }

    // this returns the list of what is in the highlighted results, without any <em> tag. may contain hallucinated tags.
    $scope.highlightedTagList = function(item) {
        if (!item.hasOwnProperty('highlightedTags')) {
            item.highlightedTags = getTagHighlights(item.highlight).map(
                str => str.replace(/<\/?em>/g, '')
            );
            item.tagsMap = TaggingService.fillTagsMapFromArray(item._source.tag || []);
        }

        return item.highlightedTags;
    };

    // highlighter may return fragments that don't match any tag (because it returns fragments of the concatenation of all tags)
    // this returns the list of what is in the highlighted results AND is confirmed to be a real tag. Still contains the <em> tags
    // this may miss some highlights, but guarantees there is no tag hallucination.
    $scope.realHighlightedTagList = function(item) {
        if (!item.hasOwnProperty('realHighlightedTag')) {
            const htags = $scope.highlightedTagList(item);
            const withEmBlocks = getTagHighlights(item.highlight);
            const actuallyExistingTags = new Set(item._source.tag || []);
            const realHighlightedTag = new Set();

            htags.forEach((candidate, idx) => {
                if(actuallyExistingTags.has(candidate)) {
                    realHighlightedTag.add(withEmBlocks[idx]);
                }
            });
            item.realHighlightedTag = [...realHighlightedTag];
        }

        return item.realHighlightedTag;
    };

    $scope.itemCount = function() {
        let type = "all";
        if ($scope.query.facets._type && $scope.query.facets._type.length) {
            type = $scope.query.facets._type[0].replace('_', ' ');
        }
        const plural = $scope.results && $scope.results.hits.total > 1;
        if (type == 'analysis') {
            type = plural ? 'analyses' : 'analysis';
        } else if (type == 'all') {
            type = 'item' + (plural ? 's' : '');
        } else {
            type += plural ? 's' : '';
        }
        return '<strong>' + ($scope.results ? $scope.results.hits.total : 0) + '</strong> ' + type;
    };

    init();
});

app.service('CatalogRefreshUtils', function($state, StateUtils) {
    function breakFlowComputableNameIntoParts(name) {
        const parts = name.split('.');
        if (parts.length === 1) return {name: name};
        return {
            projectKey: parts[0],
            name: parts.slice(1).join(''),
            fullName: name
        };
    }

    function reformatNames(flowComputables) {
        if (!flowComputables) return [];
        return flowComputables.map(e => Object.assign(e, breakFlowComputableNameIntoParts(e['name'])));
    }

    function highlightedColumnList(item, columns) {
        const highlightedColumns = [];
        let strippedCols = [];
        if (item.highlight && item.highlight['column']) {
            strippedCols = item.highlight['column'].map(str => str.replace(/<\/?em>/g, ''));
        }
        angular.forEach(columns, function(column) {
            const index = strippedCols.indexOf(column.name);
            if (index === -1) {
                highlightedColumns.push(column);
            } else {
                highlightedColumns.push({
                    name: item.highlight['column'][index],
                    type: column.type
                });
            }
        });

        return highlightedColumns;
    }

    function computeDatasetInputsOutputsByType(item) {
        item.splitOutputs = reformatNames(item._source.recursiveOutputs);
        item.splitOutputsByType = {};
        item.splitInputsByType = {};
        const obt = item.splitOutputsByType;
        const ibt = item.splitInputsByType;
        reformatNames(item._source.recursiveOutputs).forEach(out => {
            if (!obt[out.type]) {
                obt[out.type] = [];
            }
            obt[out.type].push(out);
        });
        reformatNames(item._source.recursiveInputs).forEach(out => {
            if (!ibt[out.type]) {
                ibt[out.type] = [];
            }
            ibt[out.type].push(out);
        });
    }

    function getLinkForElement(currentProjectKey, element) {
        const projectKey = element.projectKey || currentProjectKey;
        switch (element.type) {
            case "DATASET":
                if (element.projectKey && element.projectKey !== currentProjectKey) {
                    return $state.href("projects.project.foreigndatasets.dataset.explore", {
                        datasetFullName: element.datasetFullName,
                        projectKey: currentProjectKey
                    });
                } else {
                    return $state.href("projects.project.datasets.dataset.explore", {
                        datasetName: element.name, projectKey: projectKey
                    });
                }
            case "SAVED_MODEL":
                return $state.href("projects.project.savedmodels.savedmodel.versions", {
                    smId: element.name,
                    projectKey: projectKey
                });
            case "MODEL_EVALUATION_STORE":
                return StateUtils.href.modelEvaluationStore(element.name, projectKey);
            case "MANAGED_FOLDER":
                return $state.href("projects.project.managedfolders.managedfolder.view", {
                    odbId: element.name,
                    projectKey: projectKey
                });
            case "STREAMING_ENDPOINT":
                return $state.href("projects.project.streaming-endpoints.streaming-endpoint.settings", {
                    streamingEndpointId: element.name,
                    projectKey: projectKey
                });
            default:
                throw "Incorrect or missing flow computable type";
        }
    }

    return {
        breakFlowComputableNameIntoParts,
        reformatNames,
        highlightedColumnList,
        computeDatasetInputsOutputsByType,
        getLinkForElement
    };
});

app.component('relatedByType', {
    template: `
        <div class="accordion" ng-repeat="(type, elements) in $ctrl.elementsByType">
            <h4 class="accordion-title" ng-click="show=!show">
                <i ng-class="{'dku-icon-chevron-up-16':show, 'dku-icon-chevron-down-16':!show}"></i>
                {{$ctrl.baseString}} {{type | niceTaggableType : elements.length}}<span ng-show="elements.length > 1"> ({{elements.length}})</span>
            </h4>

            <ul ng-show="show">
                <li ng-repeat="input in elements" class="mx-textellipsis horizontal-flex">
                    <span class="flex mx-textellipsis">
                        <a href="{{getLinkForElement($ctrl.baseItemProjectKey, input)}}">{{ input.name }}</a>
                    </span>
                    <span ng-if="input.projectKey != $ctrl.baseItemProjectKey" class="info noflex">{{input.projectKey}}</span>
                </li>
            </ul>
        </div>
        `,
    bindings: {
        elementsByType: '<',
        baseItemProjectKey: '<',
        baseString: '<'
    },
    controller: function($scope, CatalogRefreshUtils) {
        $scope.getLinkForElement = CatalogRefreshUtils.getLinkForElement;
    }
});


app.controller('CatalogRefreshController', function($scope, $rootScope, DataikuAPI, CachedAPICalls, $sce,
                                                    CatalogItemService, ActivityIndicator, ExposedObjectsService,
                                                    AppConfig, FLOW_COMPUTABLE_TYPES, CatalogRefreshUtils,
                                                    QuickSharingWT1EventsService) {
    $scope.getLinkForElement = CatalogRefreshUtils.getLinkForElement;

    $scope.$watch("formatted_items", function() {
        if ($scope.formatted_items && $scope.selected && $scope.selected.item) {
            for (let i = 0; i < $scope.formatted_items.length; i++) {
                if ($scope.formatted_items[i]._id == $scope.selected.item._id) {
                    $scope.selected.index = i;
                    $scope.selected.item = $scope.formatted_items[i];
                    return;
                }
            }
            $scope.selected.index = null;
            $scope.selected.item = null;
        }
    });

    $scope.$watch("selected.item", function(nv, ov) {
        if (!$scope.selected || !$scope.selected.item) return;
        const item = $scope.selected.item;
        $scope.exposeDisabled = true;


        const related = nv._source.rawRelatedItems;
        if (related) {
            nv.related = {projects:[],datasets:[],recipes:[]};
            related.projects.forEach(p => {
                nv.related.projects.push({key:p.key, name: p.name});
                p.datasets.forEach(d => {
                    nv.related.datasets.push($.extend({}, {projectKey: p.key}, d));
                    d.recipes.forEach(r => {
                        nv.related.recipes.push($.extend({}, {projectKey: p.key}, r));
                    });
                });
            });
        }

        if (item._type === 'insight') {
            $scope.insightData = null;
            DataikuAPI.dashboards.insights.get(item._source.projectKey, item._source.id)
                .success(function(response) {})
                .error(setErrorInScope.bind($scope));
        }

        else if (item._type === 'dataset') {
            CatalogRefreshUtils.computeDatasetInputsOutputsByType(item);
        }

        else if (item._type === 'recipe' || item._type === "labeling_task") {
            item.splitOutputs = CatalogRefreshUtils.reformatNames(item._source.outputs);
            item.splitInputs = CatalogRefreshUtils.reformatNames(item._source.inputs);
        }

        $scope.navigatorFn = CatalogItemService.hasNavigator(item._type) ? CatalogItemService.openNavigator(item._type, item._source) : false;

        const taggableType = (CatalogItemService.indexableTypeToTaggableType(item._type, item._source.type_raw) || '').toUpperCase();
        item.highlightedPartitionColumns = CatalogRefreshUtils.highlightedColumnList(item, item._source.partitioning);

        const exposable = FLOW_COMPUTABLE_TYPES.includes(taggableType);
        const targetProjectDefined = !!$scope.projectKey;
        const selectedItemSource = item._source;

        const exposeObjectToTargetProjectFn = () => {
            ExposedObjectsService.doExposeSingleObject(
                taggableType,
                selectedItemSource.projectKey,
                selectedItemSource.id,
                $scope.projectKey,
                selectedItemSource.name,
                selectedItemSource.isQuicklyShareable,
                true
            ).error(setErrorInScope.bind($scope.getCatalogScope()));
        };
        const exposeObjectToProjectsFn = (canManageExposedElements) =>
            () => ExposedObjectsService.openSingleObjectUseModal(taggableType, selectedItemSource.id, selectedItemSource.name, selectedItemSource.projectKey, canManageExposedElements ,{from: 'search-dss'});

        const requestSharingFn =
            () => ExposedObjectsService.requestSharing(taggableType, selectedItemSource.id, selectedItemSource.name, selectedItemSource.projectKey, $scope.projectKey, {from: 'search-dss'})

        $scope.exposeObjectFn = null;
        $scope.exposeLabel = null;
        $scope.importLbl = 'Use';
        $scope.exposeIcon = 'icon-dku-share';

        function checkObjectExpositionInsideProject(projectKey, objectType, objectId) {
            DataikuAPI.projects.getObjectAuthorizations(projectKey, objectType, objectId).success((data) => {
                if (data.isQuicklyShareable) {
                    $scope.exposeDisabled = !data.canReadObject;
                    if (data.canReadObject) {
                        $scope.exposeLabel = 'Use in project ' + $scope.projectKey;
                        $scope.importLbl = 'Use';
                    } else {
                        $scope.exposeLabel = 'No permission to share object';
                    }
                } else {
                    $scope.exposeDisabled = !data.canManageExposedElements && !data.isObjectSharingRequestEnabled;
                    if (data.canManageExposedElements) {
                        $scope.exposeLabel = 'Use in project ' + $scope.projectKey;
                        $scope.importLbl = 'Use';
                    } else if (data.isObjectSharingRequestEnabled) {
                        $scope.exposeObjectFn = requestSharingFn;
                        $scope.exposeIcon = "icon-lock";
                        $scope.exposeLabel = 'Request to use in project ' + $scope.projectKey;
                        $scope.importLbl = 'Request';
                    } else {
                        $scope.exposeLabel = 'No permission to share object';
                    }
                }
            }).error(function (response, status, headers) {
                $scope.exposeDisabled = true;
                $scope.exposeLabel = 'No permission to share object';
                setErrorInScope.bind($scope.getCatalogScope())(response, status, headers);
            })
        }

        const checkObjectExposition= function(projectKey, objectType, objectId) {
            DataikuAPI.projects.getObjectAuthorizations(projectKey, objectType, objectId).success((data) => {
                $scope.exposeObjectFn = exposeObjectToProjectsFn(data.canManageExposedElements);
                if (data.isQuicklyShareable) {
                    $scope.exposeDisabled = !data.canReadObject;
                    if (data.canReadObject) {
                        $scope.exposeLabel = 'Use in another project';
                    } else {
                        $scope.exposeLabel = 'No permission to share object';
                    }
                } else {
                    $scope.exposeDisabled = !data.canManageExposedElements && !data.isObjectSharingRequestEnabled;
                    if (data.canManageExposedElements) {
                        $scope.exposeLabel = 'Use in another project';
                    } else if (data.isObjectSharingRequestEnabled) {
                        $scope.exposeObjectFn = requestSharingFn;
                        $scope.exposeIcon = "icon-lock";
                        $scope.exposeLabel = 'Request to use in another project';
                        $scope.importLbl = 'Request';
                    } else {
                        $scope.exposeLabel = 'No permission to share object';
                    }
                }
            }).error(function (response, status, headers) {
                $scope.exposeDisabled = true;
                $scope.exposeLabel = 'No permission to share object';
                setErrorInScope.bind($scope.getCatalogScope())(response, status, headers);
            })
        }

        if (exposable) {
            if (targetProjectDefined) {
                $scope.exposeObjectFn = exposeObjectToTargetProjectFn;
                if ($scope.projectKey === selectedItemSource.projectKey) {
                    $scope.exposeLabel = 'Target project must be different from source';
                    $scope.exposeDisabled = true;
                } else if (selectedItemSource.usedIn && selectedItemSource.usedIn.includes($scope.projectKey)) {
                    $scope.exposeLabel = 'Object already shared';
                    $scope.exposeDisabled = true;
                } else {
                    checkObjectExpositionInsideProject(item._source.projectKey, taggableType, item._source.id);
                }
            } else {
                $scope.exposeObjectFn = exposeObjectToProjectsFn(false); // this is just a temp value while actual authorization is fetched in checkObjectExposition
                checkObjectExposition(item._source.projectKey, taggableType, item._source.id);
            }
        }

        // Filter "Related final" section
        if (item.splitOutputsByType) {
            Object.keys(item.splitOutputsByType).forEach(key => {
                let outputsByType = item.splitOutputsByType[key].filter(input => {
                    const targetProject = $scope.projects[input.projectKey];
                    return targetProject && targetProject.canReadProjectContent;
                });
                if (outputsByType.length > 0) {
                    item.splitOutputsByType[key] = outputsByType;
                } else {
                    delete item.splitOutputsByType[key];
                }
            });
        }

        // Filter "Exposed in projects" section
        if (item && item._source) {
            if (item._source.usedIn && item._source.usedIn.length) {
                const sourceProject = $scope.projects[item._source.projectKey];
                const canReadSourceProject = sourceProject && sourceProject.canReadProjectContent;
                if (canReadSourceProject) {
                    item._source.usedInReadable = item._source.usedIn;
                    item._source.usedInHiddenCount = 0;
                } else {
                    item._source.usedInReadable = item._source.usedIn.filter(projectKey => {
                        const targetProject = $scope.projects[projectKey];
                        return targetProject && targetProject.canReadProjectContent;
                    });
                    item._source.usedInHiddenCount = item._source.usedIn.length - item._source.usedInReadable.length;
                }
            } else {
                item._source.usedInReadable = undefined;
                item._source.usedInHiddenCount = 0;
            }
        }
    });

    $scope.$watch("selected.item.highlight.description[0]", function(nv, ov) {
        if (!nv) return;
        let description = nv.replace(/<em>/g, "((STARTEM))").replace(/<\/em>/g, "((ENDEM))");
        CachedAPICalls.emojisTable.then(function(emojisTable) {
            marked.setOptions({
                emoji: function(emoji) {
                    return emoji in emojisTable ? emojisTable[emoji] : (':' + emoji + ':');
                }
            });
            description = marked(description);
            $scope.selected.item.$highlightedDescription = description.replace(/\(\(STARTEM\)\)/g, '<em class="highlight">').replace(/\(\(ENDEM\)\)/g, "</em>");
        });
    });

    $scope.$watch("selected.item.highlight.shortDesc[0]", function(nv, ov) {
        if (!nv) return;
        let shortDesc = nv.replace(/<em>/g, "((STARTEM))").replace(/<\/em>/g, "((ENDEM))");
        CachedAPICalls.emojisTable.then(function(emojisTable) {
            marked.setOptions({
                emoji: function(emoji) {
                    return emoji in emojisTable ? emojisTable[emoji] : (':' + emoji + ':');
                }
            });
            shortDesc = marked(shortDesc);
            $scope.selected.item.$highlightedShortDesc = shortDesc.replace(/\(\(STARTEM\)\)/g, '<em class="highlight">').replace(/\(\(ENDEM\)\)/g, "</em>");
        });
    });
});


app.directive('itemHighlight', function() { // item-highlight
    return {
        scope: {
            highlights: '=itemHighlight',//item.highlight.hasOwnProperty('projects') // item['projects']
            label: '@',
            plural: '@'
        },
        template: `
        <p ng-if="highlights" >
            <span>{{label | plurify : highlights.length : plural}}:</span>
            <span ng-if="highlights.length > 4">
                <span>{{highlights.length}} matching</span>
            </span>
            <span ng-if="highlights.length < 5" ng-repeat="highlight in highlights track by $index">
                <span ng-bind-html="highlight | sanitizeHighlighted"></span>
                <span class="separator" ng-show="!$last"> • </span>
            </span>
        </p>`
    };
});


})();
