(function() {
    'use strict';

    angular.module('dataiku.dashboards').component('dashboardFilterPanel', {
        bindings: {
            page: '<', // DashboardPage
            filters: '<', // FrontendChartFilter[]
            filtersParams: '<',
            editable: '<', // boolean
            enableCrossFilters: '<', // boolean
            isDeactivated: '<', // boolean
            direction: '<', // FiltersPanelDirection
            isTileLocked: '<', // boolean
            isTileSelected: '<', // boolean
            theme: '<', // DSSVisualizationTheme
            filtersChange: '&', // ({ $filters }) => void
            filtersParamsChange: '&', // ({ $filtersParams }) => void
            raiseError: '&', // ({ $errorData }) => void,
            isDeactivatedChange: '&', // ({ $isDeactivated }) => void
            isTileLockedChange: '&' // () => void
        },
        templateUrl: '/static/dataiku/js/dashboards/components/dashboard-filter-panel/dashboard-filter-panel.component.html',
        controller: function($stateParams, $timeout, $scope, $rootScope, MonoFuture, CrossFiltersService, DashboardFilters, $location, ActiveProjectKey, WT1, DatasetUtils, ChartFilters, DashboardFiltersUrlParams, ClipboardUtils, ActivityIndicator, DataikuAPI, FiltersPanelPosition, FiltersPanelDirection, ChartsStaticData, ChartDataUtils, DashboardFilterPanelService, ChartFilterUtils) {
            const ctrl = this;
            const activeProjectKey = ActiveProjectKey.get();
            const DEFAULT_FILTERS_OPTIONS = { hasSamplingParamsChanged:  false, disableLoadingIndicator: false, canPropagageBeforeFacetUpdate: false, needAggregationRequest: true };

            let summary;
            let shaker;
            let dataSpec;
            let isDestroyed = false;
            const filtersPivotRequestMonoFuture = MonoFuture();


            ctrl.filtersApi;
            ctrl.canAllFiltersBeCleared;
            ctrl.usableColumns = [];
            ctrl.error = null;
            ctrl.validity = { valid: true };
            ctrl.filterFacets = [];
            ctrl.datasetLoadingError = false;
            ctrl.loading;
            ctrl.globalSummary = '';
            ctrl.hasAllColumnsSelected = false;
            ctrl.canModerateDashboards = $rootScope.projectSummary && $rootScope.projectSummary.canModerateDashboards;
            ctrl.hasInitializedFilters = false;
            ctrl.dataset = null;
            ctrl.schema = null;
            ctrl.areFiltersBeingCreated = false;
            ctrl.areFiltersBeingReordered = false;
            ctrl.sampleIsWholeDataset = null;
            ctrl.filterFacetsLoadingState = null;
            ctrl.filterFacetsLoadingError = null;
            ctrl.filterFacetsLoadingErrorMessage = '';
            ctrl.displayRefreshCacheButton = false;

            ctrl.FiltersPanelPosition = FiltersPanelPosition;
            ctrl.FiltersPanelDirection = FiltersPanelDirection;
            ctrl.warningBadge = ChartsStaticData.WARNING_BADGE;

            ctrl.$onChanges = function(changes) {
                if (changes.filters) {
                    if (ctrl.filters == null) {
                        ctrl.filters = [];
                    }
                    if (ctrl.filters.length === 0) {
                        ctrl.sampleIsWholeDataset = null;
                    }
                    synchronizeSelectedColumns();
                }

                if (changes.filtersParams) {
                    if (ctrl.filtersParams == null) {
                        ctrl.filtersParams = {};
                    }
                    const previousFiltersParams = changes.filtersParams.previousValue;
                    const currentFiltersParams = changes.filtersParams.currentValue;

                    if (!ctrl.filtersParams || !ctrl.filtersParams.datasetSmartName) {
                        ctrl.usableColumns = [];
                        ctrl.availableColumns = [];
                        ctrl.hasInitializedFilters = true;
                        return;
                    }

                    const datasetChanged = hasDatasetChanged(previousFiltersParams, currentFiltersParams);

                    if (changes.filtersParams.isFirstChange() || datasetChanged) {
                        ctrl.loading = true;
                        handleDatasetChange(ctrl.filtersParams.datasetSmartName).finally(() => {
                            try {
                                // we execute handleFiltersChange even if handleDatasetChange fails to display the error message in the filter panel.
                                ctrl.handleFiltersChange(ctrl.filters, { hasSamplingParamsChanged: !changes.filtersParams.isFirstChange(), disableLoadingIndicator: false, canPropagageBeforeFacetUpdate: false });
                            } catch {
                                // we ignore
                            }

                            $timeout(() => {
                                ctrl.samplingParams = DashboardFilters.getSamplingProps(ctrl.filtersParams);
                                ctrl.hasInitializedFilters = true;
                                ctrl.loading = false;
                            });
                        });
                    } else if (!changes.filtersParams.isFirstChange() && hasSamplingParamsChanged(previousFiltersParams, currentFiltersParams)) {
                        handleSamplingParamsChange(previousFiltersParams, currentFiltersParams);
                    }
                }
            };

            ctrl.$onDestroy = function() {
                unregisterCrossFilterEvent();
                filtersPivotRequestMonoFuture.destroy();
                isDestroyed = true;
            };

            ctrl.handleFiltersChange = (newFilters, requestOptions = DEFAULT_FILTERS_OPTIONS) => {
                // Check if component has been destroyed before executing the promise has method is included in promise chains
                if (isDestroyed) {
                    return;
                }
                const options = { ...DEFAULT_FILTERS_OPTIONS, ...requestOptions };
                if (requestOptions.needAggregationRequest == null) {
                    options.needAggregationRequest = true;
                }
                if (options.canPropagageBeforeFacetUpdate && options.needAggregationRequest) {
                    saveFiltersAndPropagateToDashboard(newFilters);
                }
                if (options.needAggregationRequest === false) {
                    saveFiltersInternally(newFilters);
                    return;
                }
                return getFilterFacets(newFilters, options.disableLoadingIndicator)
                    .then((filterFacets) => {
                        ctrl.filterFacets = filterFacets;

                        if (filterFacets == null) {
                            return;
                        }

                        const updatedFilters = newFilters.map((filter, index) => {
                            /*
                             * Cross filtering is only available in view mode and reordering is only available in edit mode, so the facet indexes always match the filter indexes.
                             */
                            if (filter.useMinimalUi) {
                                return filter;
                            }
                            return ChartFilters.updateFilterWithFacet(filter, filterFacets[index], options.hasSamplingParamsChanged);
                        });

                        if (ChartFilters.areFilterListsEqual(updatedFilters, ctrl.filters)) {
                            return;
                        }

                        if (!options.canPropagageBeforeFacetUpdate) {
                            saveFiltersAndPropagateToDashboard(updatedFilters);
                        } else {
                            saveFiltersInternally(updatedFilters);
                        }
                    });
            };


            ctrl.getPivotResponseOptions = function() {
                return {
                    projectKey: activeProjectKey,
                    dataSpec,
                    requestedSampleId: summary.requiredSampleId
                };
            };

            ctrl.handleCanAllFilteredBeClearedChange = function(canAllFiltersBeCleared) {
                ctrl.canAllFiltersBeCleared = canAllFiltersBeCleared;
            };

            ctrl.toggleFilterPanelActivation = function(isDeactivated) {
                ctrl.isDeactivated = isDeactivated;
                ctrl.globalSummary = ctrl.isDeactivated ? 'Filters tile disabled' : ChartFilters.getAllFiltersSummary(ctrl.filters);
                ctrl.isDeactivatedChange({ $isDeactivated: ctrl.isDeactivated });
            };

            ctrl.switchView = () => {
                const areFiltersBeingCreated = !ctrl.areFiltersBeingCreated;
                if (areFiltersBeingCreated) {
                    WT1.tryEvent('dashboard-filters-create-open', () => ({}));
                }
                $timeout(() => ctrl.areFiltersBeingCreated = areFiltersBeingCreated);
            };

            ctrl.handleCancelSelection = () => {
                WT1.tryEvent('dashboard-filters-create-cancel', () => ({}));
                ctrl.switchView();
            };

            ctrl.handleSelectColumns = (columns) => {
                WT1.tryEvent('dashboard-filters-create-success', () => ({
                    previouslyExistingFiltersCount: (ctrl.filters || []).length,
                    filtersCreatedCount: (ctrl.filters || []).length + (columns || []).length
                }));

                addSelectedColumns(columns);
                ctrl.switchView();
            };

            ctrl.handleSelectDataset = ({ sourceObject, sourceType, selectedTile }) => {
                let engineType;
                if (sourceType === DashboardFilters.filtersSourceTypes.DATASET) {
                    engineType = getSourceEngineType(sourceObject.object);
                } else {
                    engineType = selectedTile.engineType || getSourceEngineType(sourceObject.object);
                }

                const datasetSmartName = sourceObject.smartId;

                ctrl.filtersParamsChange({
                    $filtersParams: {
                        ...(ctrl.filtersParams || {}),
                        datasetSmartName,
                        engineType,
                        refreshableSelection: getDefaultRefreshableSelection()
                    }
                });
            };

            ctrl.copyUrl = function() {
                try {
                    const baseUrl = $location.absUrl().split('?')[0];
                    const filtersQueryStringValue = DashboardFiltersUrlParams.getFiltersQueryStringValue(ctrl.filters, ctrl.isDeactivated, ctrl.filterFacets);
                    const queryParams = { ...$location.search(), pageFilters: filtersQueryStringValue };
                    const url = `${baseUrl}?${Object.entries(queryParams).map(([key, value]) => `${key}=${encodeURIComponent(value)}`).join('&')}`;
                    /*
                     * Max URL size (defined in the nginx configuration) is 256K.
                     * A character is encoded on 1B, so the URL can be at most 256,000 characters long.
                     */
                    if (url.length > 256000) {
                        WT1.tryEvent('dashboard-filters-url-copy-error', () => ({
                            reason: 'EXCEEDS_MAX_URL_LENGTH',
                            urlLength: url.length
                        }));
                        throw 'query is too long';
                    }
                    ClipboardUtils.copyToClipboard(url);
                    WT1.tryEvent('dashboard-filters-url-copy', () => ({}));
                } catch (error) {
                    ActivityIndicator.error(`Filters cannot be copied to URL (${error})`, 5000);
                }
            };

            ctrl.emitPanelPositionChange = function(panelPosition) {
                ctrl.filtersParamsChange({ $filtersParams: { ...ctrl.filtersParams, panelPosition } });
            };

            ctrl.expandAll = function() {
                $timeout(() => ctrl.filters = ctrl.filters.map((filter) => ({ ...filter, folded: false })));
            };

            ctrl.collapseAll = function() {
                $timeout(() => ctrl.filters = ctrl.filters.map((filter) => ({ ...filter, folded: true })));
            };

            ctrl.clearAll = function() {
                if (!ctrl.filtersApi) {
                    return;
                }
                ctrl.filtersApi.clearAllFilters();
            };

            ctrl.removeAll = function() {
                ctrl.handleFiltersChange([]);
            };

            ctrl.handleAreFiltersBeingReorderedChange = function(areFiltersBeingReordered) {
                $timeout(() => ctrl.areFiltersBeingReordered = areFiltersBeingReordered);
            };

            ctrl.getEmptyStateText = () => {
                if (ctrl.enableCrossFilters) {
                    if (ctrl.canModerateDashboards) {
                        return 'Add filters from charts and datasets, or edit the dashboard to add default filters.';
                    } else {
                        return 'Add filters from charts and datasets.';
                    }
                } else {
                    if (ctrl.canModerateDashboards) {
                        return 'Cross-filtering is disabled. Enable it or add default filters.';
                    } else {
                        return 'Cross-filtering is disabled. Ask a dashboard owner to enable it or add default filters.';
                    }
                }
            };

            ctrl.shouldDisplayEmptyState = () => {
                return ctrl.filters.length === 0 && !ctrl.editable;
            };

            ctrl.openSamplingModal = function() {
                DashboardFilterPanelService.openSamplingModal(ctrl.dataset, ctrl.schema, ctrl.samplingParams)
                    .then(samplingParams => ctrl.filtersParamsChange({ $filtersParams: { ...ctrl.filtersParams, ...samplingParams } }));
            };

            ctrl.refreshSelection = function() {
                ctrl.filtersParams.refreshableSelection._refreshTrigger = Date.now();
                handleSamplingParamsChange(ctrl.filtersParams, ctrl.filtersParams, ctrl.canModerateDashboards);
            };

            function saveFiltersAndPropagateToDashboard(newFilters) {
                DashboardFilters.setGlobalFilters(newFilters, $stateParams.pageId);
                ctrl.filtersChange({ $filters: newFilters });
                if (newFilters.length === 0) {
                    ctrl.filtersParamsChange({ $filtersParams: { panelPosition: ctrl.filtersParams.panelPosition } });
                }
            }

            function saveFiltersInternally(newFilters) {
                // Useful for dashboard export to retrieve last filter version
                DashboardFilters.setGlobalFilters(newFilters, $stateParams.pageId);
                ctrl.filters = newFilters;
            }

            function handleSamplingParamsChange(previousSamplingParams, currentSamplingParams, saveDashboard = false) {
                WT1.tryEvent('dashboard-filters-sample-setting-update', () => ({
                    engineType: currentSamplingParams.engineType,
                    reuseFromExplore: false,
                    samplingMethod: currentSamplingParams.refreshableSelection.selection.samplingMethod,
                    recordsNumber: currentSamplingParams.refreshableSelection.selection.maxRecords,
                    targetRatio: currentSamplingParams.refreshableSelection.selection.targetRatio,
                    filtersNumber:
                    currentSamplingParams.refreshableSelection.selection.filter &&
                    currentSamplingParams.refreshableSelection.selection.filter.enabled &&
                    currentSamplingParams.refreshableSelection.selection.filter.uiData &&
                    currentSamplingParams.refreshableSelection.selection.filter.uiData.conditions
                        ? currentSamplingParams.refreshableSelection.selection.filter.uiData.conditions.length
                        : 0
                }));

                const engineHasChanged = currentSamplingParams.engineType !== (previousSamplingParams || {}).engineType;
                updateDatasetAndColumnsInfo(currentSamplingParams.datasetSmartName, currentSamplingParams.engineType, currentSamplingParams.refreshableSelection)
                    .then(() =>{
                        ctrl.samplingParams = DashboardFilters.getSamplingProps(ctrl.filtersParams);
                        if (engineHasChanged) {
                            migrateFiltersOnNewEngine(ctrl.filters).then(migratedFilters => ctrl.handleFiltersChange(migratedFilters, { hasSamplingParamsChanged: true, disableLoadingIndicator: false, canPropagageBeforeFacetUpdate: false }));
                        } else {
                            ctrl.handleFiltersChange(ctrl.filters, { hasSamplingParamsChanged: true, disableLoadingIndicator: false, canPropagageBeforeFacetUpdate: false });
                        }
                    })
                    .finally(()=> saveDashboard && $rootScope.$broadcast('saveDashboard'));
            };

            const unregisterCrossFilterEvent = $rootScope.$on('crossFiltersAdded', function(_, { filters: crossFilters, wt1Args }) {
                const usableColumns = summary ? summary.usableColumns : [];
                const { filters, hasChanged } = CrossFiltersService.applyCrossFiltersToFilters(crossFilters, ctrl.filters, ctrl.filterFacets, usableColumns);
                if (hasChanged) {
                    ctrl.handleFiltersChange(filters, { hasSamplingParamsChanged: false, disableLoadingIndicator: false, canPropagageBeforeFacetUpdate: true });
                    WT1.tryEvent('dashboards-create-cross-filter', () => ({
                        ...(wt1Args || {}),
                        columnTypes: (crossFilters || []).map(filter => filter.columnType)
                    }));
                }
            });

            function migrateFiltersOnNewEngine(filters) {
                const resolvedDataset = resolveDatasetFullName(ctrl.filtersParams.datasetSmartName, activeProjectKey);
                return DashboardFilters.hasCaseInsensitiveColumnNames(resolvedDataset.projectKey, resolvedDataset.datasetName, activeProjectKey).then(hasCaseInsensitiveColumnNames => {
                    return filters.map(filter => {
                        const { column, columnType } = ctrl.usableColumns.find(col => {
                            const filterColumnName = hasCaseInsensitiveColumnNames ? filter.column.toLowerCase() : filter.column;
                            const currentColumnName = hasCaseInsensitiveColumnNames ? col.column.toLowerCase() : col.column;
                            return filterColumnName === currentColumnName;
                        });
                        if (filter.columnType === columnType) {
                            return { ...filter, column, label : column };
                        }
                        return ChartFilters.createFilter({
                            column,
                            columnType,
                            isAGlobalFilter: true,
                            id: filter.id
                        });
                    });
                });
            }

            function getFilterFacets(filters, disableLoadingIndicator = false) {
                const alreadyUsedColumns = new Set(filters.map(({ column }) => column));
                ctrl.availableColumns = ctrl.usableColumns.filter(({ column }) => !alreadyUsedColumns.has(column));
                ctrl.filterFacetsLoadingError = null;
                ctrl.filterFacetsLoadingErrorMessage = '';

                ctrl.globalSummary = ChartFilters.getAllFiltersSummary(filters || []);
                if (!filters || !filters.length) {
                    // no filters, prevent calling executeFiltersPivotRequest and initialize empty data directly
                    return Promise.resolve([]);
                }

                // If there are only cross filters or custom filters, there is no facet to update
                const numberOfCrossFilters = filters.filter(filter => ChartFilterUtils.isCrossFilter(filter)).length;
                const numberOfCustomFilters = filters.filter(filter => ChartFilterUtils.isCustomFilter(filter)).length;
                if ((numberOfCrossFilters + numberOfCustomFilters) === filters.length) {
                    // Create fake facets for customFilters so we remove the loading indicator
                    return Promise.resolve(numberOfCustomFilters > 0 ? Array.from({ length: numberOfCustomFilters }, (_, index) => ({ filterIdx: index, values: [] })) : []);
                }

                if (!disableLoadingIndicator) {
                    ctrl.loading = true;
                }

                return DashboardFilters.executeFiltersPivotRequest(
                    activeProjectKey, summary || {}, filters, dataSpec, filtersPivotRequestMonoFuture
                )
                    .update(data => {
                        ctrl.filterFacetsLoadingState = data;
                    })
                    .then((data) => {
                        ctrl.filterFacetsLoadingState = data;
                        const pivotResponse = data.data.result.pivotResponse;
                        ctrl.sampleIsWholeDataset = pivotResponse.sampleMetadata ? pivotResponse.sampleMetadata.sampleIsWholeDataset : true;

                        const sampleUIData = ChartDataUtils.getSampleMetadataAndSummaryMessage(pivotResponse, 'Click to open sampling settings', 'Click to open sampling settings');
                        ctrl.samplingSummaryMessage = ctrl.editable ? sampleUIData.clickableSummaryMessage : sampleUIData.summaryMessage;

                        return pivotResponse.filterFacets;
                    })
                    .catch(({ data, status, headers, config, statusText }) => {
                        ctrl.filterFacetsLoadingError = { data, status, headers, config, statusText };
                        if (ChartDataUtils.isPivotRequestAborted(data)) {
                            // Manually aborted => do not report as error)
                            ctrl.filterFacetsLoadingErrorMessage = 'Filter panel construction was aborted, open the Sampling & Engine setting to update its configuration.';

                        } else {
                            ctrl.apiErrorAlert = ctrl.filterFacetsLoadingError.data;
                            ctrl.isCredentialError = ctrl.apiErrorAlert && ctrl.apiErrorAlert.code && (ctrl.apiErrorAlert.code==='ERR_CONNECTION_OAUTH2_REFRESH_TOKEN_FLOW_FAIL' || ctrl.apiErrorAlert.code==='ERR_CONNECTION_NO_CREDENTIALS');
                            ctrl.filterFacetsLoadingErrorMessage = ctrl.apiErrorAlert.message;
                            ctrl.displayRefreshCacheButton = 'com.dataiku.dip.pivot.backend.model.CorruptedDataException' === ctrl.apiErrorAlert.errorType;
                        }
                    }).finally(() => {
                        ctrl.filterFacetsLoadingState = null;
                        ctrl.loading = false;
                    });
            }

            // Clear column selection and apply tileParams one
            function synchronizeSelectedColumns() {
                if (ctrl.selectedColumns == null) {
                    ctrl.selectedColumns = new Set();
                }
                ctrl.selectedColumns.clear();
                ctrl.filters.forEach((filter) => {
                    ctrl.selectedColumns.add(filter.column);
                });
                ctrl.hasAllColumnsSelected = ctrl.usableColumns.length && ctrl.selectedColumns.size === ctrl.usableColumns.length;
            }

            function getSourceEngineType(filtersSource) {
                return filtersSource && DatasetUtils.isSQL(filtersSource) ? 'SQL' : 'LINO';
            }

            function addSelectedColumns(columns) {
                const newFilters = [...ctrl.filters];
                for (const column of columns) {
                    const columnType = ctrl.usableColumns.find(current => current.column === column).type;
                    const filter = ChartFilters.createFilter({
                        column,
                        columnType,
                        isAGlobalFilter: true
                    });
                    newFilters.push(filter);
                }
                // Save filters immediately so that they appear in the panel and then get their facets.
                saveFiltersInternally(newFilters);
                ctrl.handleFiltersChange(newFilters, { hasSamplingParamsChanged:  false, disableLoadingIndicator: true, canPropagageBeforeFacetUpdate: false });
            }

            function hasDatasetChanged(previousFiltersParams, currentFiltersParams) {
                const previousDatasetSmartName = previousFiltersParams && previousFiltersParams.datasetSmartName;
                const currentDatasetSmartName = currentFiltersParams && currentFiltersParams.datasetSmartName;
                return previousDatasetSmartName !== currentDatasetSmartName;
            }

            function hasSamplingParamsChanged(previousFiltersParams, currentFiltersParams) {
                // As we might abort the cache building process, we might need need to trigger the sampling change even if the sampling is the same when clicking the "Apply" button
                if (DashboardFilters.getSamplingRefreshTrigger(previousFiltersParams) !== DashboardFilters.getSamplingRefreshTrigger(currentFiltersParams)) {
                    return true;
                }
                const previousSamplingParams = previousFiltersParams && DashboardFilters.getSamplingProps(previousFiltersParams);
                const currentSamplingParams = currentFiltersParams && DashboardFilters.getSamplingProps(currentFiltersParams);
                return !angular.equals(previousSamplingParams, currentSamplingParams);
            }

            function handleDatasetChange(datasetSmartName) {
                // The active project key and the dataset project key can be different if the dataset has been shared from another project.
                const resolvedDataset = resolveDatasetFullName(datasetSmartName, activeProjectKey);

                const promises = [
                    updateDatasetAndColumnsInfo(datasetSmartName, ctrl.filtersParams.engineType, ctrl.filtersParams.refreshableSelection),
                    DataikuAPI.datasets.get(resolvedDataset.projectKey, resolvedDataset.datasetName, activeProjectKey)
                        .error(() => {
                            ctrl.datasetLoadingError = true;
                        })
                        .success(function(data) {
                            ctrl.dataset = data;
                            ctrl.schema = data.schema;
                        })
                ];
                return Promise.all(promises);
            }

            function updateDatasetAndColumnsInfo(datasetSmartName, engineType, refreshableSelection) {
                if (!datasetSmartName) {
                    shaker = null;
                    dataSpec = null;
                    summary = null;
                    ctrl.usableColumns = [];
                    return Promise.resolve();
                }
                return DashboardFilters.getShaker(activeProjectKey, datasetSmartName)
                    .then(_shaker => {
                        shaker = _shaker;
                        dataSpec = DashboardFilters.getDataSpec(activeProjectKey, datasetSmartName, engineType, refreshableSelection, shaker);
                        return DashboardFilters.fetchColumnsSummary(activeProjectKey, dataSpec, ctrl.filtersParams);
                    })
                    .then(({ summary: _summary, usableColumns: _usableColumns }) => {
                        summary = _summary;
                        ctrl.usableColumns = _usableColumns;
                    });
            }

            function getDefaultRefreshableSelection() {
                return {
                    _refreshTrigger: Date.now(),
                    selection: {
                        samplingMethod: 'FULL'
                    }
                };
            }
        }
    });
})();
